/**
 * @file WapdmgrAccessPointProxyManager.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

#include "ProxyTypes.h"
#include "WBLPortsDefines.h"
#include "WapdmgrDefines.h"
#include "WapdmgrAccessPointProxyManager.h"

namespace org 
{
namespace bosch 
{

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", WapdmgrAccessPointProxyManager, Info);

WapdmgrAccessPointProxyManager::WapdmgrAccessPointProxyManager()
{
   LOG_INFO(" WapdmgrAccessPointProxyManager() entered ");

}

WapdmgrAccessPointProxyManager::~WapdmgrAccessPointProxyManager()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   //LOG_INFO(" ~WapdmgrAccessPointProxyManager() entered ");
}

void WapdmgrAccessPointProxyManager::createProxy(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::createProxy(): busType=%d busName=%20s objPath=%s",
         busType, busName.c_str(), objPath.c_str());

   ProxyMetadata metadata;
   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   ::boost::shared_ptr< WapdmgrAPPropertiesProxy > apPropertiesProxy;

   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   if (false == _proxyAP.isProxyAvailable(accessPointProxy, metadata))
   {
      accessPointProxy = WapdmgrAccesspointProxy::createProxy(
            sWapdmgrAccessPointPortName, busName, objPath, busType, *this);
      _proxyAP.addProxyInstance(metadata, accessPointProxy);

      apPropertiesProxy = WapdmgrAPPropertiesProxy::createProxy(
            sWapdmgrAccessPointPortName, busName,objPath,busType, *this);
      _proxyAPProperties.addProxyInstance(metadata, apPropertiesProxy);
   }

}

void WapdmgrAccessPointProxyManager::createDHCPDNSConfigurationProxy(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::createDHCPDNSConfigurationProxy(): busType=%d busName=%20s objPath=%s",
         busType, busName.c_str(), objPath.c_str());

   ProxyMetadata metadata;

   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrDHCPDNSConfigurationProxy > proxyDHCPDNSConfiguration;

   proxyDHCPDNSConfiguration = WapdmgrDHCPDNSConfigurationProxy::createProxy(
         sWapdmgrDHCPDNSConfigurationPortName, busName, objPath, busType, *this);

   _proxyDHCPDNSConfiguration.addProxyInstance(metadata, proxyDHCPDNSConfiguration);

}

void WapdmgrAccessPointProxyManager::createTetheringSettingsProxy(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::createTetheringSettingsProxy(): busType=%d busName=%20s objPath=%s",
         busType, busName.c_str(), objPath.c_str());

   ProxyMetadata metadata;

   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrTetheringSettingsProxy > proxyTetheringSettings;

   proxyTetheringSettings = WapdmgrTetheringSettingsProxy::createProxy(
         sWapdmgrTetheringSettingsPortName, busName, objPath, busType, *this);

   _proxyTetheringSettings.addProxyInstance(metadata, proxyTetheringSettings);

}

void WapdmgrAccessPointProxyManager::destroyProxy()
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::destroyProxy() entered ");

   for (unsigned int i = 0; i < _proxyAP.getNumberOfProxyInstances(); ++i)
   {
      ::boost::shared_ptr< WapdmgrAccesspointProxy >& accessPointProxy = _proxyAP[i].proxy;

      if (accessPointProxy)
      {
         accessPointProxy->sendDeregisterAll();
      }
   }
   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _proxyAP.resetAllProxiesAndClear();
   _proxyCbManager.removeAllCallbackIfs();
}

bool WapdmgrAccessPointProxyManager::isProxyServiceAvailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType)
{
   return (_proxyAP.isProxyServiceAvailable(busName, objPath, busType));
}

void WapdmgrAccessPointProxyManager::setCallbackIf(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      ProxyUser user, IWapdmgrAccessPointNotifCallbackIf* callbackIf)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::setCallbackIf(): user = %d ", user);

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;
   metadata.user = user;

   if (NULL != callbackIf)
   {
      _proxyCbManager.addCallbackIf(metadata, callbackIf);
   }
   else
   {
      _proxyCbManager.removeCallbackIf(metadata);
   }
}

act_t WapdmgrAccessPointProxyManager::sendSetSSIDRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrSetSSIDCbIf& callbackIf, const ::std::vector< uint8 >& ssid,
        bool utf_ssid)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendSetSSIDRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendSetSSIDRequest(callbackIf, ssid, utf_ssid)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendDeAuthenticateStationRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrDeAuthenticateStationCbIf& callbackIf, const ::std::string& macAddress)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendDeAuthenticateStationRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendDeAuthenticateStationRequest(callbackIf, macAddress)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendBlacklistStationRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrBlacklistStationCbIf& callbackIf, const ::std::string& macAddress,
        bool allow)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendBlacklistStationRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendBlacklistStationRequest(callbackIf, macAddress, allow)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendSetOperatingChannelsRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrSetOperatingChannelsCbIf& callbackIf, const ::std::vector< uint16 >& channels)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendSetOperatingChannelsRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendSetOperatingChannelsRequest(callbackIf, channels)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendStartWPSPushButtonRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrStartWPSPushButtonCbIf& callbackIf)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendStartWPSPushButtonRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendStartWPSPushButtonRequest(callbackIf)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendCancelWPSRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrCancelWPSCbIf& callbackIf)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendCancelWPSRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendCancelWPSRequest(callbackIf)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendGetAssociatedStationsRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrGetAssociatedStationsCbIf& callbackIf)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendGetAssociatedStationsRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendGetAssociatedStationsRequest(callbackIf)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendRegisterVendorServiceRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrRegisterVendorServiceCbIf& callbackIf, const ::std::vector< ::std::vector< uint8 > >& iEs)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendRegisterVendorServiceRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendRegisterVendorServiceRequest(callbackIf, iEs)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendUnregisterVendorServiceRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrUnregisterVendorServiceCbIf& callbackIf, const ::std::vector< ::std::vector< uint8 > >& iEs)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendUnregisterVendorServiceRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendUnregisterVendorServiceRequest(callbackIf, iEs)) :(bool) (DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendGetDnsDhcpConfRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrGetDnsDhcpConfCbIf& callbackIf)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendGetDnsDhcpConfRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendGetDnsDhcpConfRequest(callbackIf)) : (bool) (DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendRestoreDefaultAPSettingsRequest(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrRestoreDefaultAPSettingsCbIf& callbackIf)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendRestoreFactorySettingsRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                   (bool)(accessPointProxy->sendRestoreDefaultAPSettingsRequest(callbackIf)) : (bool) (DEFAULT_ACT);
}

bool WapdmgrAccessPointProxyManager::sendPoweredSet(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      WapdmgrAPPoweredCbIf& callbackIf, bool powered)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendPoweredSet() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                      (bool)(accessPointProxy->sendPoweredSet(callbackIf, powered)) : (bool)(DEFAULT_ACT);
}

bool WapdmgrAccessPointProxyManager::sendHiddenSet(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      WapdmgrAPHiddenCbIf& callbackIf, bool hidden)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendHiddenSet() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                      (bool)(accessPointProxy->sendHiddenSet(callbackIf, hidden)) : (bool)(DEFAULT_ACT);
}


bool WapdmgrAccessPointProxyManager::sendTetheringSet(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      WapdmgrTetheringCbIf& callbackIf, bool tethering)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendTetheringSet() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrTetheringSettingsProxy > proxyTetheringSettings;
   return (true == _proxyTetheringSettings.isProxyAvailable(proxyTetheringSettings, metadata)) ?
                      (bool)(proxyTetheringSettings->sendTetheringSet(callbackIf, tethering)) : (bool)(DEFAULT_ACT);
}

bool WapdmgrAccessPointProxyManager::sendMaxStationsAllowedSet(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      WapdmgrAPMaxStationsAllowedCbIf& callbackIf, uint16 noOfStations)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendMaxStationsAllowedSet() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                      (bool)(accessPointProxy->sendMaximumStationsAllowedSet(callbackIf, noOfStations)) : (bool)(DEFAULT_ACT);
}

bool WapdmgrAccessPointProxyManager::sendSecuritySet(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      WapdmgrAPSecurityCbIf& callbackIf, const ::std::string& security)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendSecuritySet() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                      (bool)(accessPointProxy->sendSecuritySet(callbackIf, security)) : (bool)(DEFAULT_ACT);
}

bool WapdmgrAccessPointProxyManager::sendPassphraseSet(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      WapdmgrAPPassphraseCbIf& callbackIf, const ::std::string& passphrase)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendPassphraseSet() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                      (bool)(accessPointProxy->sendPassphraseSet(callbackIf, passphrase)) : (bool)(DEFAULT_ACT);
}

bool WapdmgrAccessPointProxyManager::sendCountryCodeSet(const ::std::string& busName,
        const ::std::string& objPath, const ::DBusBusType busType,
        WapdmgrAPCountryCodeCbIf& callbackIf, const ::std::string& country_code)
{
    ProxyMetadata metadata;
    ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;

    LOG_INFO("Access point obj path: %s country_code to be set: %s",
            objPath.c_str(), country_code.c_str());

    metadata.busName = busName;
    metadata.objPath = objPath;
    metadata.busType = busType;

    if (_proxyAP.isProxyAvailable(accessPointProxy, metadata))
        return (bool) accessPointProxy->sendCountryCodeSet(callbackIf, country_code);
    return (bool) (DEFAULT_ACT);
}

bool WapdmgrAccessPointProxyManager::sendAccessNetworkTypeSet(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      WapdmgrAccessNetworkTypeCallbackIF& callbackIf, uint16 networkType)
{
    ProxyMetadata metadata;
    ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;

    LOG_INFO("Access point obj path: %s Network Type to set: %d",
            objPath.c_str(), networkType);

    metadata.busName = busName;
    metadata.objPath = objPath;
    metadata.busType = busType;

    if (_proxyAP.isProxyAvailable(accessPointProxy, metadata))
        return (bool) accessPointProxy->sendAccessNetworkTypeSet(callbackIf, networkType);
    return (bool) (DEFAULT_ACT);

}

act_t WapdmgrAccessPointProxyManager::sendSupportedChannelsGet(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      WapdmgrAPSupportedChannelsCbIf& callbackIf)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendSupportedChannelsGet() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAccesspointProxy > accessPointProxy;
   return (true == _proxyAP.isProxyAvailable(accessPointProxy, metadata)) ?
                  (bool)(accessPointProxy->sendSupportedChannelsGet(callbackIf)) : (bool)(DEFAULT_ACT);
}

act_t WapdmgrAccessPointProxyManager::sendGetAllProperties(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,WapdmgrAPGetAllCallbackIf& callbackIf)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::sendGetAllProperties() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< WapdmgrAPPropertiesProxy > apPropertiesProxy;

   return (true == _proxyAPProperties.isProxyAvailable(apPropertiesProxy, metadata)) ?
         (bool) apPropertiesProxy->sendGetAllRequest(callbackIf, "org.bosch.wapdman.Accesspoint") : (bool) DEFAULT_ACT;
}

// ServiceAvailableIF implementation
void WapdmgrAccessPointProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::onAvailable() entered ");

   for (unsigned int i = 0;i < _proxyAP.getNumberOfProxyInstances();++i)
   {
      ::boost::shared_ptr< WapdmgrAccesspointProxy >& accessPointProxy = _proxyAP[i].proxy;
      if ((accessPointProxy) && (proxy == accessPointProxy))
      {
         _proxyAP[i].isServiceAvailable = true;
         registerAccesspointProperties(accessPointProxy,stateChange);
      } //if ((accessPointProxy) && (proxy == accessPointProxy))
      ::boost::shared_ptr< WapdmgrDHCPDNSConfigurationProxy>& proxyDHCP = _proxyDHCPDNSConfiguration[i].proxy;
      if ((proxyDHCP) && (proxy == proxyDHCP))
      {
         _proxyDHCPDNSConfiguration[i].isServiceAvailable = true;
        registerDHCPProperties(proxyDHCP,stateChange);
      } //if ((proxyDHCP) && (proxy == proxyDHCP))
      ::boost::shared_ptr< WapdmgrTetheringSettingsProxy>& proxyTethering = _proxyTetheringSettings[i].proxy;
      if ((proxyTethering) && (proxy == proxyTethering))
      {
         _proxyTetheringSettings[i].isServiceAvailable = true;
        registerTetheringSettings(proxyTethering,stateChange);
      } //if ((proxyTethering) && (proxy == proxyTethering))

   }
}

void WapdmgrAccessPointProxyManager::registerTetheringSettings(
      ::boost::shared_ptr<WapdmgrTetheringSettingsProxy>& proxyTethering,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::registerTetheringSettings() entered ");
   (void)stateChange;
   const ::std::string& busName = proxyTethering->getDBusBusName();
   const ::std::string& objectPath = proxyTethering->getDBusObjectPath();
   ::DBusBusType busType = proxyTethering->getBusType();

   std::vector<IWapdmgrAccessPointNotifCallbackIf*> callbacksList =
         _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
   for (std::vector<IWapdmgrAccessPointNotifCallbackIf*>::iterator it = callbacksList.begin();
         it != callbacksList.end(); ++it)
   {
      proxyTethering->sendTetheringRegister(**it);
      proxyTethering->sendTetheringGet(**it);
   }
}

void WapdmgrAccessPointProxyManager::registerDHCPProperties(
      ::boost::shared_ptr<WapdmgrDHCPDNSConfigurationProxy>& proxyDHCP,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::registerDHCPProperties() entered ");
   (void)stateChange;
   const ::std::string& busName = proxyDHCP->getDBusBusName();
   const ::std::string& objectPath = proxyDHCP->getDBusObjectPath();
   ::DBusBusType busType = proxyDHCP->getBusType();

   std::vector<IWapdmgrAccessPointNotifCallbackIf*> callbacksList =
         _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
   for (std::vector<IWapdmgrAccessPointNotifCallbackIf*>::iterator it = callbacksList.begin();
         it != callbacksList.end(); ++it)
   {
      proxyDHCP->sendIPv4PoolRegister(**it);
      proxyDHCP->sendIPv4PoolGet(**it);
   }
}

void WapdmgrAccessPointProxyManager::registerAccesspointProperties(
      ::boost::shared_ptr<WapdmgrAccesspointProxy>& accessPointProxy,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::registerAccesspointProperties() entered ");

   const ::std::string& busName = accessPointProxy->getDBusBusName();
   const ::std::string& objectPath = accessPointProxy->getDBusObjectPath();
   ::DBusBusType busType = accessPointProxy->getBusType();

   std::vector<IWapdmgrAccessPointNotifCallbackIf*> callbacksList =
         _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
   for (std::vector<IWapdmgrAccessPointNotifCallbackIf*>::iterator it = callbacksList.begin();
         it != callbacksList.end(); ++it)
   {
      accessPointProxy->sendAssociatedStationsChangedRegister(**it);
      accessPointProxy->sendAssociationRequestRejectedRegister(**it);
      accessPointProxy->sendSSIDRegister(**it);
      accessPointProxy->sendMacAddressRegister(**it);
      accessPointProxy->sendInterfaceRegister(**it);
      accessPointProxy->sendPoweredRegister(**it);
      accessPointProxy->sendHiddenRegister(**it);
      accessPointProxy->sendSecurityRegister(**it);
      accessPointProxy->sendPassphraseRegister(**it);
      accessPointProxy->sendSupportedChannelsRegister(**it);
      accessPointProxy->sendCurrentOperatingChannelRegister(**it);
      accessPointProxy->sendMaximumStationsAllowedRegister(**it);
      accessPointProxy->sendPowerStateRegister(**it);
      accessPointProxy->sendPowerFailureReasonRegister(**it);
      accessPointProxy->sendCountryCodeRegister(**it);
      accessPointProxy->sendAccessNetworkTypeRegister(**it);

      //TODO - check if below properties are required
      //accessPointProxy->sendBlackListedStationsRegister(**it);
      //accessPointProxy->sendHWFeaturesRegister(**it);

      (*it)->onWapdmgrAccessPointServiceAvailable(busName, objectPath, busType,
            stateChange.getPreviousState(), stateChange.getCurrentState());
   }
}
void WapdmgrAccessPointProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" WapdmgrAccessPointProxyManager::onUnavailable() entered ");

   for (unsigned int i = 0; i < _proxyAP.getNumberOfProxyInstances(); ++i)
   {
      ::boost::shared_ptr< WapdmgrAccesspointProxy >& accessPointProxy = _proxyAP[i].proxy;
      if ((accessPointProxy) && (proxy == accessPointProxy))
      {
         accessPointProxy->sendDeregisterAll();

         _proxyAP[i].isServiceAvailable = false;

         const ::std::string& busName = accessPointProxy->getDBusBusName();
         const ::std::string& objectPath = accessPointProxy->getDBusObjectPath();
         ::DBusBusType busType = accessPointProxy->getBusType();

         std::vector<IWapdmgrAccessPointNotifCallbackIf*> callbacksList =
               _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
         for (std::vector<IWapdmgrAccessPointNotifCallbackIf*>::iterator it = callbacksList.begin();
               it != callbacksList.end(); ++it)
         {
            (*it)->onWapdmgrAccessPointServiceUnavailable(busName, objectPath, busType,
                  stateChange.getPreviousState(), stateChange.getCurrentState());
         } //for (std::vector<IWapdmgrAccessPointNotifCallbackIf*>::iterator it = callbacksList.begin();...)
      } //if ((accessPointProxy) && (proxy == accessPointProxy))
   }
}


} // namespace bosch
} // namespace org

/** @} */
