/**
 * @file IWapdmgrAccessPointProxyIf.h
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

#ifndef _I_WAPDMGR_ACCESS_POINT_PROXY_IF_H
#define _I_WAPDMGR_ACCESS_POINT_PROXY_IF_H

#include <string>

#include "ProxyTypes.h"
#include "WapdmgrAccessPointTypes.h"
#include "IWapdmgrAccessPointNotifCallbackIf.h"

namespace org 
{
namespace bosch 
{
// forward declarations

class IWapdmgrAccessPointProxyIf
{
public:

   virtual ~IWapdmgrAccessPointProxyIf() {}

   virtual void createProxy(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType) = 0;

   virtual void destroyProxy() = 0;

   virtual void createDHCPDNSConfigurationProxy(const ::std::string& busName,
               const ::std::string& objPath, const ::DBusBusType busType) = 0;

   virtual void createTetheringSettingsProxy(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType) = 0;

   virtual bool isProxyServiceAvailable(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType) = 0;

   /**
    * Set callback handler.
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] user: user
    * @param[in] callbackIf: interface to receive notifications
    */
   virtual void setCallbackIf(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         ProxyUser user, IWapdmgrAccessPointNotifCallbackIf* callbackIf) = 0;

   /**
    * Send method request to send a SetSSID request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] ssid: ssid value
    * @param[in] utf_ssid: indicates whether the SSID is to be interpreted using UTF-8 encoding
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSetSSIDRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrSetSSIDCbIf& callbackIf, const ::std::vector< uint8 >& ssid,
         bool utf_ssid) = 0;

   /**
    * Send method request to send a DeauthenticateStation request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] macAddress: MacAddress of station
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDeAuthenticateStationRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrDeAuthenticateStationCbIf& callbackIf, const ::std::string& macAddress) = 0;

   /**
    * Send method request to send a BlacklistStation request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] macAddress: MacAddress of station
    * @param[in] allow: indicates whether to Blacklist or white list a specific station.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBlacklistStationRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrBlacklistStationCbIf& callbackIf, const ::std::string& macAddress,
         bool allow) = 0;

   /**
    * Send method request to send a SetOperatingChannels request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] channels: list of channels to be preferred
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSetOperatingChannelsRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrSetOperatingChannelsCbIf& callbackIf, const ::std::vector< uint16 >& channels) = 0;

   /**
    * Send method request to send a StartWPSPushButton request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendStartWPSPushButtonRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrStartWPSPushButtonCbIf& callbackIf) = 0;

   /**
    * Send method request to send a CancelWPS request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCancelWPSRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrCancelWPSCbIf& callbackIf) = 0;

   /**
    * Send method request to send a GetAssociatedStations request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetAssociatedStationsRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrGetAssociatedStationsCbIf& callbackIf) = 0;

   /**
    * Send method request to send a RegisterVendorService request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] iEs: IE elements
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRegisterVendorServiceRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrRegisterVendorServiceCbIf& callbackIf, const ::std::vector< ::std::vector< uint8 > >& iEs) = 0;

   /**
    * Send method request to send a UnregisterVendorService request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] iEs: IE elements
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendUnregisterVendorServiceRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrUnregisterVendorServiceCbIf& callbackIf, const ::std::vector< ::std::vector< uint8 > >& iEs) = 0;

   /**
    * Send method request to send a GetDnsDhcpConf request
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetDnsDhcpConfRequest(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrGetDnsDhcpConfCbIf& callbackIf) = 0;

   /**
    * Send method request to set Powered property
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] powered: powered state
    *
    * @return = bool
    */
   virtual bool sendPoweredSet(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrAPPoweredCbIf& callbackIf, bool powered) = 0;

   /**
    * Send method request to set Hidden property
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] hidden: hidden state
    *
    * @return = bool
    */
   virtual bool sendHiddenSet(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrAPHiddenCbIf& callbackIf, bool hidden) = 0;

   /**
    * Send method request to set Security property
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] security: security type
    *
    * @return = bool
    */
   virtual bool sendSecuritySet(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrAPSecurityCbIf& callbackIf, const ::std::string& security) = 0;

   /**
    * Send method request to set Security property
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] passphrase: passphrase value
    *
    * @return = bool
    */
   virtual bool sendPassphraseSet(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrAPPassphraseCbIf& callbackIf, const ::std::string& passphrase) = 0;

   /**
    * Send method request to get SupportedChannels property
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    *
    * @return = bool
    */
   virtual act_t sendSupportedChannelsGet(const ::std::string& busName,
        const ::std::string& objPath,const ::DBusBusType busType,
        WapdmgrAPSupportedChannelsCbIf& callbackIf) = 0;

   /**
    * Send method request to get GetAll properties
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    *
    * @return = bool
    */
   virtual act_t sendGetAllProperties(const ::std::string& busName,
         const ::std::string& objPath,const ::DBusBusType busType,
         WapdmgrAPGetAllCallbackIf& callbackIf) = 0;
		 
   /**
    * Send method request to restore factory settings for AP
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    *
    * @return = bool
    */
   virtual act_t sendRestoreDefaultAPSettingsRequest(const ::std::string& busName, const ::std::string& objPath,
         const ::DBusBusType busType, WapdmgrRestoreDefaultAPSettingsCbIf& callbackIf) = 0;

   /**
   * Send method request to regulatory change
   *
   * @param[in] busName: bus name
   * @param[in] objPath: object path
   * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
   * @param[in] callbackIf: interface to receive response
   * @param[in] country_code: A valid ISO/IEC 3166-1 alpha2 country code
   *
   * @return = bool
   */
   virtual bool sendCountryCodeSet(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrAPCountryCodeCbIf& callbackIf, const ::std::string& country_code) = 0;

   /**
    * Send method request to set Tethering property
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] Tethering: Tethering state
    *
    * @return = bool
    */
   virtual bool sendTetheringSet(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrTetheringCbIf& callbackIf, bool tethering) = 0;

   /**
    * Send method request to set MaxStationsAllowed property
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] noOfStations: No of Stations allowed for this AP
    *
    * @return = bool
    */
   virtual bool sendMaxStationsAllowedSet(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrAPMaxStationsAllowedCbIf& callbackIf, uint16 noOfStations) = 0;

   /**
    * Send method request to set AccessNetworkType property
    *
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] callbackIf: interface to receive response
    * @param[in] noOfStations: No of Stations allowed for this AP
    *
    * @return = bool
    */
   virtual bool sendAccessNetworkTypeSet(const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType,
         WapdmgrAccessNetworkTypeCallbackIF& callbackIf, uint16 networkType) = 0;

};

} // namespace bosch
} // namespace org

#endif //_I_WAPDMGR_ACCESS_POINT_PROXY_IF_H

/** @} */
