/**
 * @file LCMClient.cpp
 * @author RBEI/ECO32 Jiji Anna Jacob
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_business_logic
 * @brief Clienthandler to LCM for WiFiState handling
 *
 * @{
 */

#include "LCMClient.h"
#include "WBLDefines.h"
#include "WBLASFComponent.h"
#include "IEEE80211APClient.h"
#include "IEEE80211STAClient.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", LCMClient, Debug);

LCMClient::LCMClient() :
      _poLCMProxy(nullptr), _currState(LCM_WIFISTATE_UNKNOWN),_prevState(LCM_WIFISTATE_OFF)
{
   LOG_INFO("LCMClient:Constructor");
   DBusProxyFactory* proxyFactory = DBusProxyFactory::getInstance();
   _poLCMProxy = (proxyFactory) ? (proxyFactory->getLCMProxyIf()) : (nullptr);
   if (_poLCMProxy)
   {
      _poLCMProxy->createProxy("LCMPort");
   }
   isRestricted = false;
   _wblWifiState = WIFI_STATE_REQUEST_IDLE;
} //LCMClient::LCMClient

LCMClient::~LCMClient()
{
   try
   {
      LOG_INFO("LCMClient:Destructor");
      if (_poLCMProxy)
      {
         _poLCMProxy->destroyProxy();
         RELEASE_MEM(_poLCMProxy);
      }
   }catch(...){}

} //LCMClient::~LCMClient

void LCMClient::onLCMProxyAvailable()
{
   LOG_INFO("LCMClient:onLCMProxyAvailable");
   if (_poLCMProxy)
   {
      _poLCMProxy->registerProperties(*this);
      _poLCMProxy->getWiFiState(*this);
   }

} //LCMClient::onLCMProxyAvailable

void LCMClient::onLCMProxyUnavailable()
{
   LOG_INFO("LCMClient:onLCMProxyUnavailable");
   if (_poLCMProxy)
   {
      _poLCMProxy->deregisterProperties();
   }

} //LCMClient::onLCMProxyAvailable

LCMWiFiState LCMClient::getCurrentWiFiState()
{
   LOG_INFO("LCMClient:getCurrentWiFiState %s", WIFISTATE_TO_STR(_currState));
   return _currState;

} //LCMClient::getCurrentWiFiState

void LCMClient::onWifiStateError(const LCMDBusProxy& proxy, const WiFiStateError& error)
{
   (void) proxy;
   (void) error;
   LOG_ERROR("LCMClient:onWifiStateError");

} //LCMClient::onWifiStateError


void LCMClient::onWifiStateUpdate(const LCMDBusProxy& proxy, const WiFiStateUpdate& update)
{
   WBL_ASSERT_AND_EXIT(!(proxy && update));
   _prevState = _currState;
   const uint8 currState = static_cast < uint8 >(update->getWifiState());
   _currState = (LCMWiFiState) currState;

   LOG_INFO("LCMClient::onWifiStateUpdate:PreviousState = %s", WIFISTATE_TO_STR(_prevState));
   LOG_INFO("LCMClient::onWifiStateUpdate:CurrentState = %s", WIFISTATE_TO_STR(_currState));

   processWifiStateEvent();


} //LCMClient::onWifiStateUpdate

void LCMClient::onWifiDeactivateRequest(const WifiSetUpItem& setUpItem)
{
   LOG_INFO("LCMClient:onWifiDeactivateRequest");

   WBLPowerOffReason powerOffType;
   powerOffType =  (_currState == LCM_WIFISTATE_OFF || _currState == LCM_WIFISTATE_BLOCK) ?
            WIFI_POWER_OFF_BLOCK_WIFISTATE_REQUEST : WIFI_POWER_OFF_WIFISTATE_REQUEST;
   const ::boost::shared_ptr<DeActivateSetupMsg> deactivateMsg(new DeActivateSetupMsg);
   deactivateMsg->setMsgID(WBL_MSG_DEACTIVATE_SETUP_REQ);
   deactivateMsg->setWifiMode(setUpItem.property.mode);
   deactivateMsg->setRequestType(powerOffType);
   sendDeactivateRequest(deactivateMsg);
   _wblWifiState = WIFI_STATE_REQUEST_BLOCK_IN_PROGRESS;

} //LCMClient::onWifiDeactivateRequest

void LCMClient::onWifiUnblockRequest(const WifiSetUpItem& setUpItem)
{

   bool bRet;
   WifiMode currMode = setUpItem.property.mode;
   LOG_INFO("onWifiUnblockRequest:Mode:%s", WIFI_MODE_TO_STR(currMode));
   bRet = setPowered(setUpItem, true);
   if(bRet)
   {
      _wblWifiState =   WIFI_STATE_REQUEST_UNBLOCK_IN_PROGRESS;
   }
   else
   {
      LOG_INFO("onWifiUnblockRequest:Power ON Request failed. Retry upon service availability");
      _wblWifiState = WIFI_STATE_REQUEST_FAILED;
   }


} //LCMClient::onWifiUnblockRequest

bool LCMClient::isStateMachineBusy(WifiMode currMode)
{
   bool isBusy = false;
   SMManagerFactory* smFactory = SMManagerFactory::getInstance();
   tICmdStateMachineManagerSptr smManagerIf = (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
   if (smManagerIf)
   {
      isBusy = smManagerIf->isSMBusy(currMode);
   }

   LOG_INFO("LCMClient::isStateMachineBusy:%s",BOOL_TO_STR(isBusy));
   return isBusy;

} //LCMClient::isStateMachineBusy

bool LCMClient::getActiveSetUps(WifiSetUpList& wifiSetups)const
{
   bool isValid = false;
   ConflictManagerFactory* conflictMgrFactory = ConflictManagerFactory::getInstance();
   ICmdConflictManager* conflictMgrIf =
                 (conflictMgrFactory) ? (conflictMgrFactory->getConflictManagerRequestIf()) : (nullptr);
   if (conflictMgrIf)
   {
      isValid = conflictMgrIf->getWiFiSetUps(wifiSetups);
   }
   LOG_INFO("LCMClient::getActiveSetUps:%s",BOOL_TO_STR(isValid));
   return isValid;

} //LCMClient::getActiveSetUps

bool LCMClient::setPowered(const WifiSetUpItem& setUpItem, const bool isPowered)
{
   LOG_INFO("LCMClient::setPowered:type = %s,mode = %s,objPath = %s",
           BOOL_TO_STR(isPowered), WIFI_MODE_TO_STR(setUpItem.property.mode),
           setUpItem.nativeObjPath.c_str());
   IEEE80211ClientFactory* clientFactory = IEEE80211ClientFactory::getInstance();
   bool bRet = false;
   switch (setUpItem.property.mode)
   {
      case WIFI_MODE_STA:
      {
         IEEE80211STAClientIf* staClientIf = (clientFactory) ? (clientFactory->getSTAClientIf()) : (nullptr);
         if (staClientIf)
         {
            bRet = staClientIf->setPowered(setUpItem.nativeObjPath, isPowered);

         }
      }
      break;

      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         IEEE80211APClientIf* apClientIf = (clientFactory) ? (clientFactory->getAPClientIf()) : (nullptr);
         if (apClientIf)
         {
           bRet = apClientIf->setPowered(setUpItem.nativeObjPath, isPowered);
           if(setUpItem.property.apSetup.type == WBL_AP_TYPE_CARPLAY) //If CPW APType Register Vendor Elements
           {
               bRet = apClientIf->updateRegisterVendorService(setUpItem.nativeObjPath);
           }
         }

      }
      break;

      default:
      {
         LOG_ERROR("setPowered:Unhandled case");
      }
      break;
   }
   return bRet;
} //LCMClient::setPowered

void LCMClient::sendDeactivateRequest(const ::boost::shared_ptr < DeActivateSetupMsg >& msg)
{
   LOG_INFO("LCMClient::sendDeactivateRequest");

   ICmdConflictManager* conflictManagerIf =
         ConflictManagerFactory::getInstance()->getConflictManagerRequestIf();

   if (NULL != conflictManagerIf)
   {
      conflictManagerIf->onProcessRequest(msg);
   }
} //LCMClient::sendSMRequest

bool LCMClient::setTechnologyRestrictions(bool isRestrict)
{
   LOG_INFO("LCMClient::setTechnologyRestrictions");
   isRestricted = isRestrict;
   handleWifiStateUpdate();
   return true;
} //LCMClient::setTechnologyRestrictions

void LCMClient::processWifiStateEvent()
{
   LOG_INFO("LCMClient::processWifiStateEvent WBLWifiState:%d",_wblWifiState);
   switch (_wblWifiState)
   {
      case WIFI_STATE_REQUEST_IDLE:
      case WIFI_STATE_REQUEST_BLOCK:
      case WIFI_STATE_REQUEST_UNBLOCK:
      case WIFI_STATE_REQUEST_FAILED:
      {
         handleWifiStateUpdate();
      }
      break;
      case WIFI_STATE_REQUEST_BLOCK_IN_PROGRESS:
      case WIFI_STATE_REQUEST_UNBLOCK_IN_PROGRESS:
      default:
      {
         //Nothing to do. Wait for Completion
         LOG_INFO("LCMClient::processWifiStateEvent WBLWifiState Event Already in Progress");
      }
      break;
   }
}

bool LCMClient::checkUnBlockCompleted(WifiSetUpList &wifiSetups)
{
   LOG_INFO("LCMClient:checkUnBlockCompleted");
   bool bRet =true;
   for (auto iter = wifiSetups.begin(); iter != wifiSetups.end(); ++iter)
   {
     bool poweredSetting;
     IDBManager* dbmgrIf = DBManagerFactory::getInstance()->getDBManagerIf();
     if (dbmgrIf)
     {
        poweredSetting = dbmgrIf->readPoweredSetting(iter->property.mode, iter->setupObjPath);
        LOG_DEBUG("LCMClient::checkUnBlockCompleted:poweredSetting [%d]= %s", iter->property.mode,
              BOOL_TO_STR(poweredSetting));
        if (iter->property.isPowered != poweredSetting)
        {
           bRet = false;
           break;
        }
     }
   }
   return bRet;
}

bool LCMClient::checkBlockCompleted(WifiSetUpList &wifiSetups)
{
   LOG_INFO("LCMClient:checkBlockCompleted");
   bool bRet =true;
   for (auto iter = wifiSetups.begin(); iter != wifiSetups.end(); ++iter)
   {
      if(validateWifiBlock(*iter))
      {
         bRet = false;
         break;
      }
   }
   LOG_DEBUG("LCMClient:checkBlockCompleted: %s",BOOL_TO_STR(bRet));
   return bRet;
}
void LCMClient::notifyActiveWiFiSetups(WifiSetUpList wifiSetups)
{
   LOG_INFO("LCMClient:notifyActiveWiFiSetups");

   if (_wblWifiState == WIFI_STATE_REQUEST_BLOCK_IN_PROGRESS)
   {
      if (checkBlockCompleted(wifiSetups))
      {
         _wblWifiState = WIFI_STATE_REQUEST_BLOCK;
         processWifiStateEvent();
      }
   }
   else if (_wblWifiState == WIFI_STATE_REQUEST_UNBLOCK_IN_PROGRESS)
   {
      if (checkUnBlockCompleted(wifiSetups))
      {
         _wblWifiState = WIFI_STATE_REQUEST_UNBLOCK;
         processWifiStateEvent();
      }
   }
}

bool LCMClient::validateWifiBlock(const WifiSetUpItem& setUpItem)
{
   LOG_INFO("LCMClient:validateWifiBlock");
   bool isBlockRequired = false;

   WifiMode currMode = setUpItem.property.mode;
   LOG_INFO("LCMClient::validateWifiBlock:Mode %s", WIFI_MODE_TO_STR(currMode));

   if ((currMode == WIFI_MODE_AP1 || currMode == WIFI_MODE_AP2)
         && (WBL_AP_TYPE_CARPLAY == setUpItem.property.apSetup.type) && isRestricted)
   {
      LOG_DEBUG("LCMClient::onWifiDeactivateRequest:AP is of Carplay Type. So Wifi OFF/BLOCK is not Allowed");
      isBlockRequired = false;
   }
   else if (setUpItem.property.isPowered)
   {
      isBlockRequired = true;
   }

   LOG_DEBUG("LCMClient::validateWifiBlock:Mode %s: DeactivateRequired:%s", WIFI_MODE_TO_STR(currMode),BOOL_TO_STR(isBlockRequired));
   return isBlockRequired;
}

void LCMClient::handleWifiStateUpdate()
{
   LOG_INFO("LCMClient:handleWifiStateUpdate");
   switch (_currState)
     {
        case LCM_WIFISTATE_OFF:
        case LCM_WIFISTATE_BLOCK:
        case LCM_WIFISTATE_NORMAL_SWITCH_OFF_WIFI:
        {
           WifiSetUpList wifiSetups;
           if (getActiveSetUps(wifiSetups))
           {
              for (auto iter = wifiSetups.begin(); iter != wifiSetups.end(); ++iter)
              {
                 if(validateWifiBlock(*iter))
                 {
                    onWifiDeactivateRequest(*iter);
                 }
              }
           }
        }
        break;
        case LCM_WIFISTATE_NORMAL:
        {
           WifiSetUpList wifiSetups;

           if (getActiveSetUps(wifiSetups))
           {
              for (auto iter = wifiSetups.begin(); iter != wifiSetups.end(); ++iter)
              {
                 bool poweredSetting;
                 IDBManager* dbmgrIf = DBManagerFactory::getInstance()->getDBManagerIf();
                 if (dbmgrIf)
                 {
                    poweredSetting = dbmgrIf->readPoweredSetting(iter->property.mode, iter->setupObjPath);
                    LOG_DEBUG("LCMClient::handleWifiStateUpdate:poweredSetting [%d]= %s", iter->property.mode,
                          BOOL_TO_STR(poweredSetting));
                    if ((!iter->property.isPowered) && poweredSetting)
                    {
                       onWifiUnblockRequest(*iter);
                    }
                 }
              }
           }
        }
        break;
        default:
        {
           LOG_ERROR("handleWifiStateUpdate:Unhandled case");
        }
        break;
     }
}//LCMClient::handleWifiStateUpdate()

void LCMClient::notifySTASserviceAvailability(bool isAvailable)
{
   LOG_INFO("LCMClient:notifySTASserviceAvailability:%s",BOOL_TO_STR(isAvailable));
   if(isAvailable)
      handleWifiStateUpdate();
}

void LCMClient::notifyAPServiceAvailability(bool isAvailable)
{
   LOG_INFO("LCMClient:notifyAPServiceAvailability:%s",BOOL_TO_STR(isAvailable));
   if(isAvailable)
      handleWifiStateUpdate();
}

void LCMClient::onDeactivateResponse(const ::boost::shared_ptr < DeActivateSetupMsg >& msg)
{
   LOG_INFO("LCMClient:onDeactivateResponse");
   if(msg->getRequestType() == WIFI_POWER_OFF_BLOCK_WIFISTATE_REQUEST ||
         msg->getRequestType() == WIFI_POWER_OFF_WIFISTATE_REQUEST)
   {
      _wblWifiState = WIFI_STATE_REQUEST_BLOCK;
   }
}

void LCMClient::onDeactivateError(const ::boost::shared_ptr < DeActivateSetupMsg >& msg)
{
   LOG_INFO("LCMClient:onDeactivateError");
   if(msg->getRequestType() == WIFI_POWER_OFF_BLOCK_WIFISTATE_REQUEST ||
            msg->getRequestType() == WIFI_POWER_OFF_WIFISTATE_REQUEST)
   {
      _wblWifiState = WIFI_STATE_REQUEST_FAILED;
   }

}

} //namespace bosch
} //namespace org

/** @} */
