/**
 * @file ConnmanTechnologyProxyManager.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */


#include "ProxyTypes.h"
#include "WBLPortsDefines.h"
#include "ConnmanTechnologyProxyManager.h"

namespace org 
{
namespace bosch 
{

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", ConnmanTechnologyProxyManager, Info);

ConnmanTechnologyProxyManager::ConnmanTechnologyProxyManager()
{
   LOG_INFO(" ConnmanTechnologyProxyManager() entered ");
}

ConnmanTechnologyProxyManager::~ConnmanTechnologyProxyManager()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   // LOG_INFO(" ~ConnmanTechnologyProxyManager() entered ");
}

void ConnmanTechnologyProxyManager::createProxy(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType)
{
   LOG_INFO(" ConnmanTechnologyProxyManager::createProxy(): busType=%d busName=%20s objPath=%s",
         busType, busName.c_str(), objPath.c_str());

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< ConnmanTechnologyProxy > technologyProxy;
   if(false == _proxyManager.isProxyAvailable(technologyProxy, metadata))
   {
      technologyProxy = ConnmanTechnologyProxy::createProxy(
            sConnmanTechnologyPortName, busName, objPath, busType, *this);
      _proxyManager.addProxyInstance(metadata, technologyProxy);
   }
}

void ConnmanTechnologyProxyManager::destroyProxy()
{
   LOG_INFO(" ConnmanTechnologyProxyManager::destroyProxy() entered ");

   for (unsigned int i = 0; i < _proxyManager.getNumberOfProxyInstances(); ++i)
   {
      ::boost::shared_ptr< ConnmanTechnologyProxy >& technologyProxy = _proxyManager[i].proxy;

      if (technologyProxy)
      {
         technologyProxy->sendDeregisterAll();
      }
   }
   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _proxyManager.resetAllProxiesAndClear();
   _proxyCbManager.removeAllCallbackIfs();
}

bool ConnmanTechnologyProxyManager::isProxyServiceAvailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType)
{
   return (_proxyManager.isProxyServiceAvailable(busName, objPath, busType));
}

void ConnmanTechnologyProxyManager::setCallbackIf(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      ProxyUser user, IConnmanTechnologyNotifCallbackIf* callbackIf)
{
   LOG_INFO(" ConnmanTechnologyProxyManager::setCallbackIf() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;
   metadata.user = user;

   if (callbackIf)
   {
      _proxyCbManager.addCallbackIf(metadata, callbackIf);
   }
   else
   {
      _proxyCbManager.removeCallbackIf(metadata);
   }
}

bool ConnmanTechnologyProxyManager::sendSetPropertyRequest(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      ConnmanSetPropertyCbIf& callbackIf, const ::std::string& name,
      const ::asf::dbus::DBusVariant& value)
{
   LOG_INFO(" ConnmanTechnologyProxyManager::sendSetPropertyRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< ConnmanTechnologyProxy > technologyProxy;
   if (true == _proxyManager.isProxyAvailable(technologyProxy, metadata))
   {
      technologyProxy->sendSetPropertyRequest(callbackIf, name, value);
      return true;
   }
   return false;
}

// ServiceAvailableIF implementation
void ConnmanTechnologyProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" ConnmanTechnologyProxyManager::onAvailable() entered ");

   for (unsigned int i = 0; i < _proxyManager.getNumberOfProxyInstances(); ++i)
   {
      ::boost::shared_ptr< ConnmanTechnologyProxy >& technologyProxy = _proxyManager[i].proxy;
      if ((technologyProxy) && (proxy == technologyProxy))
      {
         _proxyManager[i].isServiceAvailable = true;

         const ::std::string& busName = technologyProxy->getDBusBusName();
         const ::std::string& objectPath = technologyProxy->getDBusObjectPath();
         ::DBusBusType busType = technologyProxy->getBusType();

         std::vector<IConnmanTechnologyNotifCallbackIf*> callbacksList =
               _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
         for (std::vector<IConnmanTechnologyNotifCallbackIf*>::iterator it = callbacksList.begin();
               it != callbacksList.end(); ++it)
         {
            technologyProxy->sendPropertyChangedRegister(**it);
            (*it)->onConnmanTechnologyServiceAvailable(busName, objectPath, busType,
                  stateChange.getPreviousState(), stateChange.getCurrentState());
         } //for (std::vector<IConnmanTechnologyNotifCallbackIf*>::iterator it = callbacksList.begin();...)
      } //if ((technologyProxy) && (proxy == technologyProxy))
   }
}

void ConnmanTechnologyProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" ConnmanTechnologyProxyManager::onUnavailable() entered ");

   for (unsigned int i = 0; i < _proxyManager.getNumberOfProxyInstances(); ++i)
   {
      ::boost::shared_ptr< ConnmanTechnologyProxy >& technologyProxy = _proxyManager[i].proxy;
      if ((technologyProxy) && (proxy == technologyProxy))
      {
         technologyProxy->sendDeregisterAll();

         _proxyManager[i].isServiceAvailable = false;

         const ::std::string& busName = technologyProxy->getDBusBusName();
         const ::std::string& objectPath = technologyProxy->getDBusObjectPath();
         ::DBusBusType busType = technologyProxy->getBusType();

         std::vector<IConnmanTechnologyNotifCallbackIf*> callbacksList =
               _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
         for (std::vector<IConnmanTechnologyNotifCallbackIf*>::iterator it = callbacksList.begin();
               it != callbacksList.end(); ++it)
         {
            (*it)->onConnmanTechnologyServiceUnavailable(busName, objectPath, busType,
                  stateChange.getPreviousState(), stateChange.getCurrentState());
         } //for (std::vector<IConnmanTechnologyNotifCallbackIf*>::iterator it = callbacksList.begin();...)
      } //if ((technologyProxy) && (proxy == technologyProxy))
   }
}


act_t ConnmanTechnologyProxyManager::sendGetPropertiesRequest(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType, ConnmanGetPropertiesCbIf& callbackIf)
{
   LOG_INFO(" WapdmgrManagerProxyManager::sendGetPropertiesRequest() entered ");

   ProxyMetadata metadata;
   metadata.busName = busName;
   metadata.objPath = objPath;
   metadata.busType = busType;

   ::boost::shared_ptr< ConnmanTechnologyProxy > technologyProxy;
   return (true == _proxyManager.isProxyAvailable(technologyProxy, metadata)) ?
                  (bool)(technologyProxy->sendGetPropertiesRequest(callbackIf)) : (bool)(DEFAULT_ACT);

}
} // namespace bosch
} // namespace org

/** @} */
