/**
 * @file ProxyCallbacksManager.h
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */
 
#ifndef _PROXY_CALLBACKS_MANAGER_H
#define _PROXY_CALLBACKS_MANAGER_H

#include <string>
#include <map>
#include <utility>
#include <vector>

#include "ProxyMetadata.h"

namespace org 
{
namespace bosch 
{

//Forward declaration
//typedef ::std::pair< ProxyMetadata, int >  tProxyKey;

template<typename tProxyCbIf>
class ProxyCallbacksManager
{
   public:
      ProxyCallbacksManager() {}

      virtual ~ProxyCallbacksManager() {}

      void addCallbackIf(const ProxyMetadata& metaData, tProxyCbIf* callbackIf)
      {
         if (callbackIf)
         {
            _proxyCallbacks[metaData] = callbackIf;
         }
      }

      void addCallbackIf(const ::std::string& busName, const ::std::string& objPath,
            const ::DBusBusType busType, ProxyUser user, tProxyCbIf* callbackIf)
      {
         ProxyMetadata metadata;
         metadata.busName = busName;
         metadata.objPath = objPath;
         metadata.busType = busType;
         metadata.user = user;
         addCallbackIf(metadata, callbackIf);
      }

      void removeCallbackIf(const ProxyMetadata& metaData)
      {
         typename ::std::map< ProxyMetadata, tProxyCbIf* >::iterator it = _proxyCallbacks.find(metaData);

         if(_proxyCallbacks.end() == it)
         {
            // entry is already erased => can happen
         }
         else
         {
            // entry is not erased
            _proxyCallbacks.erase(it);
         }
      }

      void removeCallbackIf(const ::std::string& busName, const ::std::string& objPath,
            const ::DBusBusType busType, ProxyUser user)
      {
         ProxyMetadata metadata;
         metadata.busName = busName;
         metadata.objPath = objPath;
         metadata.busType = busType;
         metadata.user = user;
         removeCallbackIf(metadata);
      }

      bool bIsCallbackAvailable(const ProxyMetadata &metaData, const tProxyCbIf *cb)
      {
          typename ::std::map< ProxyMetadata, tProxyCbIf* >::iterator it = _proxyCallbacks.begin();
          for(; it != _proxyCallbacks.end(); ++it)
             if (metaData == it->first && cb == it->second)
            	 return true;
          return false;
      }

      tProxyCbIf* getCallbackIf(const ProxyMetadata& metaData)
      {
         typename ::std::map< ProxyMetadata, tProxyCbIf* >::iterator it = _proxyCallbacks.find(metaData);

         return ((_proxyCallbacks.end() != it) ? (it->second) : (NULL));
      }

      tProxyCbIf* getCallbackIf(const ::std::string& busName, const ::std::string& objPath,
            const ::DBusBusType busType, ProxyUser user)
      {
         ProxyMetadata metaData;
         metaData.busName = busName;
         metaData.objPath = objPath;
         metaData.busType = busType;
         metaData.user = user;

         return getCallbackIf(metaData);
      }

      std::vector<tProxyCbIf*> getAllCallbackIf(ProxyMetadata metaData)
      {
         metaData.user = e8PROXY_USER_UNKNOWN;
         std::vector<tProxyCbIf*> callbacksList;

         typename ::std::map< ProxyMetadata, tProxyCbIf* >::iterator it = _proxyCallbacks.begin();
         for(; it != _proxyCallbacks.end(); ++it)
         {
            ProxyMetadata metadataInstance;
            metadataInstance = (it->first);
            metadataInstance.user = e8PROXY_USER_UNKNOWN;
            if (metadataInstance == metaData)
            {
               callbacksList.push_back(it->second);
            }
         }
         return callbacksList;
      }

      std::vector<tProxyCbIf*> getAllCallbackIf(const ::std::string& busName, const ::std::string& objPath,
            const ::DBusBusType busType)
      {
         std::vector<tProxyCbIf*> callbacksList;

         typename ::std::map< ProxyMetadata, tProxyCbIf* >::iterator it = _proxyCallbacks.begin();
         for(; it != _proxyCallbacks.end(); ++it)
         {
            const ProxyMetadata& metadata = it->first;
            if ((metadata.busName == busName) && (metadata.objPath == objPath) && (metadata.busType == busType))
            {
               callbacksList.push_back(it->second);
            }
         }
         return callbacksList;
      }

      void removeAllCallbackIfs()
      {
         _proxyCallbacks.clear();
      }

   private:
      ProxyCallbacksManager(const ProxyCallbacksManager& ref);
      ProxyCallbacksManager& operator=(const ProxyCallbacksManager& ref);

      ::std::map< ProxyMetadata, tProxyCbIf* > _proxyCallbacks;

};


} // namespace bosch
} // namespace org


#endif //_PROXY_CALLBACKS_MANAGER_H
