/**
 * @file GenericDbusParser.h
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

#ifndef _GENERIC_DBUS_PARSER_H
#define _GENERIC_DBUS_PARSER_H

#include <string>

#include "asf/dbus/DBusVariant.h"

#include "DbusIfTypes.h"

namespace org 
{
namespace bosch 
{

// class forward declarations
class IDbusProperty2EnumMapping;

/**
 * DBUS property parser class.
 */
class GenericDbusParser
{
public:
   GenericDbusParser();
   virtual ~GenericDbusParser();

   /**
    * Parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] matchingInterfaces: interfaces matching the given properties
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    * @param[in] mapping: reference to mapping instance
    */
   void parseProperties(::std::vector<DbusVariantProperty>& outProperties,
         const ::std::vector< int >& matchingInterfaces,
         ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
          const ::DBusBusType busType, const ::std::string& busName,
          const ::std::string& objPath, const ::std::string& interfaceName,
          const IDbusProperty2EnumMapping* mapping, const bool parseDictItemsAsProperties = true) const;

   /**
    * Parse function.
    *
    * @param[in] data: introspection data
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    * @param[in] mapping: reference to mapping instance
    */
   void parseIntrospection(const ::std::string& data, const ::DBusBusType busType,
         const ::std::string& busName, const ::std::string& objPath,
         const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[in] addedObjectPath: added object path
    * @param[in] inInterfaces: map with interfaces (mutable)
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    * @param[in] mapping: reference to mapping instance
    */
   void parseInterfacesAdded(const ::std::string& addedObjectPath,
         ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
           const ::DBusBusType busType, const ::std::string& busName,
           const ::std::string& objPath, const ::std::string& interfaceName,
           const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[in] removedObjectPath: removed object path
    * @param[in] inInterfaces: vector with interfaces
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    * @param[in] mapping: reference to mapping instance
    */
   void parseInterfacesRemoved(const ::std::string& removedObjectPath,
         const ::std::vector< ::std::string >& inInterfaces, const ::DBusBusType busType,
         const ::std::string& busName, const ::std::string& objPath,
         const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] propertyName: property to be found/parsed
    * @param[in] interface: related interface
    * @param[in] mapping: reference to mapping instance
    *
    * @return = true: property found,
    * @return = false: property not found
    */
   bool findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList,
         ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
          const ::std::string& propertyName, const int interface,
          const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] propertyValue: received property value (mutable)
    * @param[in] propertyName: received property name
    * @param[in] interface: related interface
    * @param[in] mapping: reference to mapping instance
    */
   void parsePropertyValue(::std::vector<DbusVariantProperty>& outPropertyList,
         ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName,
          const int interface, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Adds an item to dictionary.
    *
    * @param[out] dictionary: dictionary
    * @param[in] key: key
    * @param[in] value: value
    */
   void addDictionaryItem(::std::map< ::std::string, ::asf::dbus::DBusVariant >& dictionary,
         const ::std::string& key, const DbusVariant& value) const;

   /**
    * Set traces enabled flag.
    *
    * @param[in] enabled: traces enabled flag
    */
   inline void setTraces(const bool enabled) { _tracesEnabled = enabled; }

   /**
    * Get traces enabled flag.
    *
    * @return = traces enabled flag
    */
   inline bool getTraces(void) const { return _tracesEnabled; }

private:
   bool _tracesEnabled; /**< traces enabled flag */

   /**
    * Parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] matchingInterfaces: interfaces matching the given properties
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] mapping: reference to mapping instance
    */
   void parsePropertiesInternal(::std::vector<DbusVariantProperty>& outProperties,
         const ::std::vector< int >& matchingInterfaces,
         ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
          const IDbusProperty2EnumMapping* mapping, const bool parseDictItemsAsProperties = true) const;

   /**
    * Internal parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] interfaceList: interfaces matching the given properties
    * @param[in] iter: iterator
    * @param[in] signature: signature
    * @param[in] property: property
    * @param[in] counter: property counter
    * @param[in] depth: depth
    * @param[in] isDict: dictionary marker
    * @param[in] dictKey: dictionary key
    */
   void parseProperty(::std::vector<DbusVariantProperty>& outProperties,
         const ::std::vector<const ::std::map< ::std::string, int >*>& interfaceList,
         ::DBusMessageIter* iter, const ::std::string& signature,
          const ::std::string& property, const unsigned int counter,
          const unsigned int depth, const bool isDict,
          const ::std::string& dictKey, const bool parseDictItemsAsProperties = true) const;

   DECLARE_CLASS_LOGGER();
};

} //namespace bosch
} //namespace org

#endif //_GENERIC_DBUS_PARSER_H

/** @} */
