/**
 * @file DbusParser.h
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

#ifndef _DBUS_PARSER_H
#define _DBUS_PARSER_H

#include <string>

#include "asf/dbus/DBusVariant.h"

#include "DbusIfTypes.h"
#include "GenericDbusParser.h"

namespace org
{
namespace bosch
{

/**
 * DBUS property parser class for WBL.
 */
template < typename T >
class DbusParser
{
public:
   DbusParser();
   virtual ~DbusParser();

   /**
    * Parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] matchingInterfaces: interfaces matching the given properties
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseProperties(::std::vector<DbusVariantProperty>& outProperties,
         const ::std::vector< int >& matchingInterfaces,
         ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
         const ::DBusBusType busType, const ::std::string& busName,
         const ::std::string& objPath, const ::std::string& interfaceName,
         const bool parseDictItemsAsProperties = true) const;

   /**
    * Parse function.
    *
    * @param[in] data: introspection data
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseIntrospection(const ::std::string& data,
         const ::DBusBusType busType, const ::std::string& busName,
         const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[in] addedObjectPath: added object path
    * @param[in] inInterfaces: map with interfaces (mutable)
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseInterfacesAdded(const ::std::string& addedObjectPath,
         ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
         const ::DBusBusType busType, const ::std::string& busName,
         const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[in] removedObjectPath: removed object path
    * @param[in] inInterfaces: vector with interfaces
    * @param[in] busType: bus type (DBUS_BUS_SYSTEM or DBUS_BUS_SESSION)
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseInterfacesRemoved(const ::std::string& removedObjectPath,
         const ::std::vector< ::std::string >& inInterfaces,
         const ::DBusBusType busType, const ::std::string& busName,
         const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] propertyName: property to be found/parsed
    * @param[in] interface: related interface
    *
    * @return = true: property found,
    * @return = false: property not found
    */
   bool findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList,
         ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
          const ::std::string& propertyName, const int interface) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] propertyValue: received property value (mutable)
    * @param[in] propertyName: received property name
    * @param[in] interface: related interface
    */
   void parseProperty(::std::vector<DbusVariantProperty>& outPropertyList,
         ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName,
          const int interface) const;

   /**
    * Convert function.
    *
    * @param[in] interface: interface
    *
    * @return = interface name
    */
   const ::std::string& getInterfaceName(const uint interface) const;

   /**
    * Set traces enabled flag.
    *
    * @param[in] enabled: traces enabled flag
    */
   void setTraces(const bool enabled);

   /**
    * Get traces enabled flag.
    *
    * @return = traces enabled flag
    */
   inline bool getTraces(void) const { return _tracesEnabled; }

protected:
   /**
    * Get property to enum mapping list. To be implemented by child class.
    *
    * @param[out] interfaceList: property to enum mapping list matching the given interfaces
    * @param[in] matchingInterfaces: interfaces to be checked
    */
   virtual void getInterfaces(::std::vector<const ::std::map< ::std::string, int >*>& interfaceList, const ::std::vector< int >& matchingInterfaces) const;

private:

   bool _tracesEnabled; /**< traces enabled flag */
   T  _interfaces;
   GenericDbusParser _dbusParser;

};

template < typename T >
DbusParser<T>::DbusParser()
{
   _tracesEnabled = false;
   _dbusParser.setTraces(_tracesEnabled);
}

template < typename T >
DbusParser<T>::~DbusParser()
{
}

template < typename T >
void DbusParser<T>::parseProperties(::std::vector<DbusVariantProperty>& outProperties,
      const ::std::vector< int >& matchingInterfaces,
      ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
      const ::DBusBusType busType, const ::std::string& busName,
      const ::std::string& objPath, const ::std::string& interfaceName,
      const bool parseDictItemsAsProperties) const
{
   _dbusParser.parseProperties(outProperties, matchingInterfaces, inProperties,
         busType, busName, objPath, interfaceName, &_interfaces, parseDictItemsAsProperties);
}

template < typename T >
void DbusParser<T>::parseIntrospection(const ::std::string& data,
      const ::DBusBusType busType, const ::std::string& busName,
      const ::std::string& objPath, const ::std::string& interfaceName) const
{
   _dbusParser.parseIntrospection(data, busType, busName, objPath, interfaceName, &_interfaces);
}

template < typename T >
void DbusParser<T>::parseInterfacesAdded(const ::std::string& addedObjectPath,
      ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
      const ::DBusBusType busType, const ::std::string& busName,
      const ::std::string& objPath, const ::std::string& interfaceName) const
{
   _dbusParser.parseInterfacesAdded(addedObjectPath, inInterfaces, busType, busName, objPath, interfaceName, &_interfaces);
}

template < typename T >
void DbusParser<T>::parseInterfacesRemoved(const ::std::string& removedObjectPath,
      const ::std::vector< ::std::string >& inInterfaces,
      const ::DBusBusType busType, const ::std::string& busName,
      const ::std::string& objPath, const ::std::string& interfaceName) const
{
   _dbusParser.parseInterfacesRemoved(removedObjectPath, inInterfaces, busType, busName, objPath, interfaceName, &_interfaces);
}

template < typename T >
bool DbusParser<T>::findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList,
      ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
       const ::std::string& propertyName, const int interface) const
{
   return _dbusParser.findAndParseProperty(outPropertyList, inProperties, propertyName, interface, &_interfaces);
}

template < typename T >
void DbusParser<T>::parseProperty(::std::vector<DbusVariantProperty>& outPropertyList,
      ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName, const int interface) const
{
   _dbusParser.parsePropertyValue(outPropertyList, propertyValue, propertyName, interface, &_interfaces);
}

template < typename T >
const ::std::string& DbusParser<T>::getInterfaceName(const uint interface) const
{
   return _interfaces.getInterfaceName(interface);
}

template < typename T >
void DbusParser<T>::setTraces(const bool enabled)
{
   _tracesEnabled = enabled;
   _dbusParser.setTraces(_tracesEnabled);
}

template < typename T >
void DbusParser<T>::getInterfaces(::std::vector<const ::std::map< ::std::string, int >*>& interfaceList, const ::std::vector< int >& matchingInterfaces) const
{
   _interfaces.getInterfaces(interfaceList, matchingInterfaces);
}



} //bosch
} //org

#endif //_DBUS_PARSER_H

/** @} */
