/**
 * @file WifiMlmeUpdates.cpp
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief Main file for Wifi_Business_Logic.
 *
 *
 * @{
 */

#include <errno.h>
#include "WpaSupplicantClient.h"
#include "WifiMlmeUpdates.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cWifiMlmeUpdates, Debug);

cWifiMlmeNotifier::cWifiMlmeNotifier()
{
}

cWifiMlmeNotifier::~cWifiMlmeNotifier()
{
}

int cWifiMlmeNotifier::registerForMlmeUpdates(cWifiMlmeConnectionObserver *client)
{
	std::vector<cWifiMlmeConnectionObserver *>::iterator it;

	if (!client)
		return -EINVAL;

	for (it = _mlmeObservers.begin(); it < _mlmeObservers.end(); it++)
		if (client == *it)
			return -EALREADY;
	_mlmeObservers.push_back (client);
	return 0;
}

int cWifiMlmeNotifier::unRegisterForMlmeUpdates(cWifiMlmeConnectionObserver *client)
{
	bool exist = false;
	std::vector<cWifiMlmeConnectionObserver *>::iterator it;

	if (!client)
		return -EINVAL;

	for (it = _mlmeObservers.begin(); it < _mlmeObservers.end(); it++)
		if (client == *it) {
			exist = true;
			break;
		}

	if (exist) {
		_mlmeObservers.erase(it);
		return 0;
	}

	return -ENOENT;
}

void cWifiMlmeNotifier::updateMlmeStatus(bool connected, const ::std::string &ifname,
				const ::std::string &associatedBss,const unsigned int &ifindex,const ::std::string &macaddr)
{
	std::vector<cWifiMlmeConnectionObserver *>::iterator it;

	if (!ifname.empty() && !associatedBss.empty()) {
		for (it = _mlmeObservers.begin(); it < _mlmeObservers.end(); it++) {
			if (connected) {
				(*it)->MlmeConnected(ifname, associatedBss,ifindex,macaddr);
			} else {
				(*it)->MlmeDisConnected(ifname, associatedBss);
			}
		}
	}
}

cWifiMlmeUpdates::cWifiMlmeUpdates()
{
}

cWifiMlmeConnectionDetails *cWifiMlmeUpdates::getMlmeConnectionDetails(unsigned int wiphyid,
		unsigned int ifindex, const ::std::string &ifname)
{
	std::vector<cWifiMlmeConnectionDetails>::iterator it;

	for (it = _mlmeStatus.begin(); it < _mlmeStatus.end(); it++)
		if ((*it)._wiphyid == wiphyid && (*it)._ifindex == ifindex &&
				!(*it)._ifname.compare(ifname))
			return &(*it);
	return NULL;
}

int cWifiMlmeUpdates::addMlmeConnectionDetails(unsigned int wiphyid, unsigned int ifindex,
		const ::std::string &ifname, const ::std::string &mac)
{
	int iRet;
	cWifiMlmeConnectionDetails client;

	if (getMlmeConnectionDetails(wiphyid, ifindex, ifname))
		return -EALREADY;

	client._ifindex = ifindex;
	client._ifname = ifname;
	client._macaddr = mac;
	client._wiphyid = wiphyid;
	iRet = cWpaSupplicantClient::getInstance()->getBssFromMacAddr(mac, client._currentBss);
	if (iRet < 0) {
		LOG_ERROR ("Failed to get the object path from wpa_supplicant for the asoociated BSS: %s [%s/%d]",
				mac.c_str(), strerror(-iRet), -iRet);
	}

	_mlmeStatus.push_back(client);
	return iRet;
}

int cWifiMlmeUpdates::removeMlmeConnectionDetails(unsigned int wiphyid, unsigned int ifindex,
		const ::std::string &ifname)
{
	bool available = false;
	std::vector<cWifiMlmeConnectionDetails>::iterator it;

	for (it = _mlmeStatus.begin(); it < _mlmeStatus.end(); it++)
		if ((*it)._wiphyid == wiphyid && (*it)._ifindex == ifindex &&
				!(*it)._ifname.compare(ifname)) {
			available = true;
			break;
		}

	if (available) {
		_mlmeStatus.erase(it);
		return 0;
	}

	return -ENOENT;
}

void cWifiMlmeUpdates::mlmeConnected(const cMlmeEventConnect &cevent)
{
	int iRet;
	cWifiMlmeConnectionDetails *client;
	bool notify = true;

	if (cevent.getIfname().empty())
		return;

	LOG_INFO("MLME Connection status, radio: %u ifindex: %u ifname: %s "
			"timedout: %s bss: %s status: %u", cevent.getIeee80211index(), cevent.getIfindex(),
			cevent.getIfname().c_str(), cevent.getTimedout() ? "Y" :"N",
			cevent.getTimedout() ? "None" :	cevent.getBssid().c_str(),
			cevent.getStatuscode());

	if (cevent.getTimedout() || cevent.getStatuscode() != 0 ||
			cevent.getBssid().empty())
		return;

	if (getMlmeConnectionDetails(cevent.getIeee80211index(), cevent.getIfindex(), cevent.getIfname()))
		removeMlmeConnectionDetails(cevent.getIeee80211index(), cevent.getIfindex(), cevent.getIfname());

	iRet = addMlmeConnectionDetails(cevent.getIeee80211index(), cevent.getIfindex(),
			cevent.getIfname(), cevent.getBssid());
	if (iRet < 0) {
		LOG_ERROR ("Failed to add the mlme connection status of [wiphyid: %u] "
				"[ifname: %s]: %s [%s/%d]",cevent.getIeee80211index(),
				cevent.getIfname().c_str(), cevent.getBssid().c_str(), strerror(-iRet), -iRet);
		notify = false;
	}

	client = getMlmeConnectionDetails(cevent.getIeee80211index(), cevent.getIfindex(), cevent.getIfname());
	if (client) {
		if (notify) {
			updateMlmeStatus(true, client->_ifname, client->_currentBss, client->_ifindex, client->_macaddr);
		} else {
			client->_isDue = true;
			LOG_INFO("Connection information for this interface %s is due to Regulatory core",
					client->_ifname.c_str());
		}
	}
}

void cWifiMlmeUpdates::mlmeDisConnected(const cMlmeEventDisconnect &devent)
{
	int iRet;
	cWifiMlmeConnectionDetails *client;
	bool notify = true;

	if (devent.getIfname().empty())
		return;

	LOG_INFO("MLME Disconnection status, radio: %u ifindex: %u ifname: %s",
			devent.getIeee80211index(), devent.getIfindex(),
			devent.getIfname().c_str());

	client = getMlmeConnectionDetails(devent.getIeee80211index(), devent.getIfindex(), devent.getIfname());
	if (client) {
		if (client->_isDue) {
			LOG_INFO ("Connection status due for the radio %s to bss %s",
					client->_ifname.c_str(), client->_macaddr.c_str());
			iRet = cWpaSupplicantClient::getInstance()->getBssFromMacAddr(client->_macaddr, client->_currentBss);
			if (iRet < 0) {
				LOG_ERROR ("Failed to get the object path from wpa_supplicant for the asoociated BSS: %s [%s/%d]",
						client->_macaddr.c_str(), strerror(-iRet), -iRet);
				notify = false;
			} else {
				updateMlmeStatus(true, client->_ifname, client->_currentBss, client->_ifindex,client->_macaddr);
				client->_isDue = false;
			}
		}
		if (notify)
			updateMlmeStatus(false, client->_ifname, client->_currentBss, client->_ifindex,client->_macaddr);
		removeMlmeConnectionDetails(devent.getIeee80211index(), devent.getIfindex(), devent.getIfname());
	} else {
		LOG_ERROR ("There is no entry for the mlme connection status for [wiphyid: %u] [ifname: %s]",
				devent.getIeee80211index(), devent.getIfname().c_str());
	}
}

cWifiMlmeUpdates::~cWifiMlmeUpdates()
{
}

	}
}

/** @} */
