/**
 * @file WatchDog.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2018 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include "DBManagerFactory.h"
#include "WBLUtils.h"
#include "WatchDog.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cWatchDog, Debug);
DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cRegulatoryWatchDog, Debug);

void cWatchDog::notify(void *data)
{
	for (auto it = _callbacks.begin(); it != _callbacks.end(); ++it)
		(*it)->watchDogExpired(data);
}

int cWatchDog::subscribe(cWatchDogCallbackIf *cb)
{
	int err = -EINVAL;

	if (cb) {
		if (::std::find(_callbacks.begin(), _callbacks.end(), cb) == _callbacks.end())
			_callbacks.push_back (cb);
		else { err = -EALREADY; }
	}

	return  err;
}

int cWatchDog::unsubscribe(cWatchDogCallbackIf *cb)
{
	auto it = ::std::find(_callbacks.begin(), _callbacks.end(), cb);
	if (it == _callbacks.end())
		return -ENOENT;

	_callbacks.erase(it);
	return 0;
}

const std::string cWatchDog::watchDogTypeToString(const wttype_t type)
{
	std::string stype;

	switch (type) {
	case eWatchDog_Type_Once:
		stype = "once";
		break;
	case eWatchDog_Type_Periodic:
		stype = "Periodic";
		break;
	default:
		break;
	}
	return stype;
}

bool cWatchDog::createTimer()
{
	bool create = false;

	if (_interval > 0) {
		_watchDog = new ::asf::core::Timer(*this, _interval,
				(size_t) (_type == eWatchDog_Type_Once ? 1 : 0));
		LOG_INFO("Created watch dog timer with _interval: %u [%p] [type: %s]",
				_interval, _watchDog, watchDogTypeToString(_type).c_str());
		create = true;
	}

	return create;
}

int cWatchDog::setTimeInterval(const unsigned int interval)
{
	if (_interval != interval) {
		_interval = interval;
		return 0;
	}
	return -EALREADY;
}

inline unsigned int cWatchDog::getWatchDogInterval()
{
	return _interval;
}

cWatchDog::cWatchDog(const wttype_t type) :
		_type(type), _interval(0), _watchDog(nullptr)
{

}

cWatchDog::~cWatchDog()
{
	destroyTimer();
}

void cWatchDog::onExpired(::asf::core::Timer& timer,
		::boost::shared_ptr< ::asf::core::TimerPayload > data)
{
	LOG_INFO("OnExpired event for watch dog timer: %p "
			"[expected: %p] [repeat count: %u] Reason: \"%d\" "
			"Act: \"%u\"",
			&timer, _watchDog, data->getRepeatCount(),
			data->getReason(), data->getAct());

	if (_watchDog == &timer)
		expired(data);
}

int cWatchDog::startWatchDog()
{
	if (!_watchDog)
		return -ENOTCONN;
	if (!_watchDog->isStopped())
		return -EALREADY;

	_watchDog->start();
	return 0;
}

int cWatchDog::stopWatchDog()
{
	if (!_watchDog)
		return -ENOTCONN;
	if (_watchDog->isStopped())
		return -EALREADY;

	_watchDog->stop();
	return 0;
}

void cWatchDog::destroyTimer()
{
	if (_watchDog) {
		stopWatchDog();
		delete _watchDog;
		_watchDog = nullptr;
	}
}

int cRegulatoryWatchDog::getWatchDogConfigurations(const ::std::string &key,
		unsigned int &data)
{
	int iRet;
	::std::string group("WatchDogConfigurations");

	iRet = DBManagerFactory::getInstance()->getDBManagerIf()->getRegulatoryWatchDogInfo(group, key, &data);
	if (iRet < 0) {
		LOG_INFO("There is no data for key \"%s\" in DB", key.c_str());
	}

	return iRet;
}

int cRegulatoryWatchDog::setWatchDogConfigurations(const ::std::string &key, unsigned int &data)
{
	int iRet;
	::std::string group("WatchDogConfigurations");

	iRet = DBManagerFactory::getInstance()->getDBManagerIf()->setRegulatoryWatchDogInfo(group, key, data);
	if (iRet < 0) {
		LOG_INFO("There is no data for key \"%s\" in DB", key.c_str());
	}
	return iRet;
}

void cRegulatoryWatchDog::removeOldKeys()
{
	int iRet;
	::std::string group("Global"), key("WatchDogTimeLeft");

	iRet = DBManagerFactory::getInstance()->getDBManagerIf()->removeKey(group, key);
	if (iRet < 0)
		LOG_INFO("Failed to remove the key \"%s\" in DB", key.c_str());
}

cRegulatoryWatchDog::cRegulatoryWatchDog() : cWatchDog(eWatchDog_Type_Periodic)
{
	unsigned int interval = 0;
	bool result = false;

	_timeLeft = 0;
	memset (&_startTime, 0, sizeof (struct timespec));

	(void) removeOldKeys();
	if (0 != getWatchDogConfigurations("Interval", interval))
		interval = REGULATORY_DOMAIN_WATCH_DOG_TIMER_MIN;

	LOG_INFO("Loaded Time interval for regulatory watchdog: \"%u\"", interval);

	if (0 != getWatchDogConfigurations("TimeToWatch", _timeToWatch))
		_timeToWatch = REGULATORY_DOMAIN_WATCH_DOG_TIMER_MAX;

	LOG_INFO("Loaded Time to watch for regulatory watchdog: \"%u\"", _timeToWatch);

	if (_timeToWatch % interval) {
		interval = REGULATORY_DOMAIN_WATCH_DOG_TIMER_MIN;
		_timeToWatch = REGULATORY_DOMAIN_WATCH_DOG_TIMER_MAX;
	}

	if (0 != getWatchDogConfigurations("TimeLeft", _timeLeft))
		_timeLeft = _timeToWatch;

	LOG_INFO("Loaded Time left for regulatory watchdog: \"%u\"", _timeLeft);

	if (_timeLeft % interval)
		_timeLeft = _timeToWatch;

	(void) setTimeInterval(interval);
	result = createTimer();

	LOG_INFO("%s timer with interval: \"%u\", timetowatch: \"%u\", time left: \"%u\"",
			result == true ? "Successfully created" : "Failed to create",
					interval, _timeToWatch, _timeLeft);
}

cRegulatoryWatchDog::~cRegulatoryWatchDog()
{
   try
   {
      stop(false);
   }catch(...){}
}

int cRegulatoryWatchDog::start ()
{
	int iRet;

	iRet = startWatchDog();
	if (0 == iRet) {
		iRet = cUtils::getMonotonicTime(_startTime);
		if (0 == iRet) {
			LOG_INFO("Regulatory watch dog started at %lu.%06lu seconds",
					_startTime.tv_sec, _startTime.tv_nsec);
		}
	} else {
		LOG_ERROR("Failed to start the regulatory watch dog: %s/%d", strerror (-iRet), -iRet);
	}

	return iRet;
}

int cRegulatoryWatchDog::stop(bool reset)
{
	int iRet;

	iRet = stopWatchDog();
	if (0 == iRet) {
		iRet = cUtils::getMonotonicTime(_startTime);
		if (0 == iRet) {
			LOG_INFO("Regulatory watch dog stopped at %lu.%06lu seconds",
					_startTime.tv_sec, _startTime.tv_nsec);
		}
	} else {
		LOG_ERROR("Failed to stop the regulatory watch dog: %s/%d", strerror (-iRet), -iRet);
	}

	if (reset) {
		_timeLeft = _timeToWatch;
		(void) setWatchDogConfigurations("TimeLeft", _timeLeft);
	}

	return iRet;
}

void cRegulatoryWatchDog::expired(::boost::shared_ptr< ::asf::core::TimerPayload > data)
{
	(void) data;
	::std::string group("Global"), cc("00");
	bool updatecc = false;

	LOG_INFO("Watch Dog Time Left: \"%u\" _interval: \"%u\"",
			_timeLeft - getWatchDogInterval(), getWatchDogInterval());

	_timeLeft = _timeLeft - getWatchDogInterval();
	if (!_timeLeft) {
		_timeLeft = _timeToWatch;
		updatecc = true;
	}

	if (updatecc)
		notify(static_cast<void *>(&cc));

	(void) setWatchDogConfigurations("TimeLeft", _timeLeft);
}

	}
}

/** @} */
