/**
 * @file RegulatoryMCCMethod.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2018 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include "RegulatoryMCCMethod.h"
#include "MCCClient.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cRegulatoryMCCMethod, Debug);

cRegulatoryMCCMethod::cRegulatoryMCCMethod() :
		cRegulatoryMethod(REGULATORY_METHOD_MCC, REGULATORY_METHOD_PRIO_DEFAULT,
				eCC_SOURCE_MCC, CC_TYPE_NUMERIC)
{
	_cc_availability = false;
	_notify = false;
	_mccClient = new cMCCClient();
	if (_mccClient) {
		_mccClient->updateServiceAvailability(true);
		_mccClient->registerCountryChangeNotification(this);
	}
}

cRegulatoryMCCMethod::cRegulatoryMCCMethod(cRegulatoryChangeNotifier *observer) :
		cRegulatoryMethod(REGULATORY_METHOD_MCC, REGULATORY_METHOD_PRIO_DEFAULT,
				eCC_SOURCE_MCC, CC_TYPE_NUMERIC)
{
	int iRet;

	_cc_availability = false;
	_notify = false;
	_mccClient = new cMCCClient();
	if (_mccClient) {
		_mccClient->updateServiceAvailability(true);
		_mccClient->registerCountryChangeNotification(this);
	}

	if (observer) {
		iRet = Register(observer);
		if (iRet < 0)
			LOG_ERROR ("Failed to register the observer[%p]: %s/%d", observer,
					strerror(-iRet), -iRet);
	}
}

cRegulatoryMCCMethod::~cRegulatoryMCCMethod()
{
	if (_mccClient)
		_mccClient->unRegisterCountryChangeNotification(this);
	delete _mccClient;
}

void cRegulatoryMCCMethod::setCountryCodeInfo (const cc_availability_t serAvail,
		const ::std::string &cc, const uint16_t numeric)
{
	cc_availability_t old = _serviceAvailability;

	LOG_INFO("Old: %s ServiceAvailability: %s, country code: %s numeric: %u",
			SERVICE_UNAVAILABLE == old ? "N" : "Y",
			SERVICE_UNAVAILABLE == serAvail ? "N" : "Y", cc.c_str(), numeric);

	_serviceAvailability = serAvail;
	_numeric_cc = numeric;

	if (old != _serviceAvailability)
		checkServiceAvailability();

	if (SERVICE_AVAILABLE == _serviceAvailability && _mccClient)
		_mccClient->setCountryCodeInfo(numeric);
}

void cRegulatoryMCCMethod::checkServiceAvailability()
{
	bool old = _availability;

	LOG_INFO("Old: \"%s\", _serviceAvailability: \"%s\", _cc_availability: \"%s\"",
			old == true ? "Y" : "N", SERVICE_AVAILABLE == _serviceAvailability ? "Y" : "N",
					_cc_availability == true ? "Y" : "N");

	_availability = false;
	if (SERVICE_AVAILABLE == _serviceAvailability &&
			_cc_availability == true)
		_availability = true;

	if (old != _availability)
		notifyServiceAvailability();
}

int cRegulatoryMCCMethod::start()
{
	if (!_notify) {
		_notify = true;

		/* Usually we dont need this if condition, as
		 * core would have called start only if the service
		 * is available but just to be sure */
		if (_availability)
			_mccClient->updateCountryCodeInfo();
	}

	return 0;
}

void cRegulatoryMCCMethod::countryChangeAvailability(bool availability)
{
	LOG_INFO ("Change in country change update service availability: %s "
			"previous: %s", (availability == true) ? "Y" : "N",
					(_cc_availability == true) ? "Y" : "N");

	if (_cc_availability != availability) {
		_cc_availability = availability;
		checkServiceAvailability();
	}
}

int cRegulatoryMCCMethod::stop()
{
	if (_notify)
		_notify = false;

	return 0;
}

void cRegulatoryMCCMethod::countryChangeUpdate(const ::std::string &country)
{
	LOG_INFO ("MCC Client Update :%s Notify: %s", country.c_str(),
			(_notify == true) ? "YES" : "NO");

	if (true == _notify && !country.empty())
		notifyRegChange(country);
}

	}
}

/** @} */
