/**
 * @file RegulatoryBin.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <limits.h>
#include <string.h>
#include "KDSConfiguration.h"
#include "RegulatoryBin.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cRegulatoryBin, Debug);

static const ::std::string regBinFolder = "/usr/lib/crda";
//static const ::std::string customRegBinFolder = "/usr/local/lib/crda";
static const ::std::string customRegBinPath = "/var/opt/bosch/dynamic/connectivity/bt_module/regulatory.bin";

cRegulatoryBin::cRegulatoryBin()
{
}

cRegulatoryBin::~cRegulatoryBin()
{
}

int cRegulatoryBin::getRegulatoryBin(bool fcc, const ::std::string searchPath,
		::std::string &regBin)
{
    DIR *dir;
    struct dirent *ent;
    ::std::string file;

    if (searchPath.empty())
    	return -EINVAL;

    dir = opendir(searchPath.c_str());
    if (!dir)
    	return -errno;

    regBin.clear();
    while ((ent = readdir (dir))) {

    	file = ent->d_name;
        if (!file.compare(".") || !file.compare(".."))
            continue;

        if (fcc && (file.find("fcc") != std::string::npos) &&
        		(file.find("non-fcc") == std::string::npos)) {
        	regBin = file;
        	break;
        }
        else if (!fcc && (file.find("non-fcc") != std::string::npos)) {
        	regBin = file;
        	break;
        }
    }

    closedir(dir);
    if (regBin.empty())
    	return -ENOENT;
    return 0;
}

int cRegulatoryBin::createSymLink(const ::std::string src,
		const ::std::string dest)
{
	int ret;

	if (src.empty() || dest.empty())
		return -EINVAL;

	LOG_INFO("ln -s %s %s", src.c_str(), dest.c_str());

	ret = symlink(src.c_str(), dest.c_str());
	if (ret < 0) {
		LOG_ERROR("ln -s %s %s failed: %s/%d", src.c_str(),
				dest.c_str(), strerror(errno), errno);
		ret = -errno;
	}

	return ret;
}

int cRegulatoryBin::removeFile(const ::std::string &file)
{
	int ret;

	if (file.empty())
		return -EINVAL;

	LOG_INFO("Removing %s", file.c_str());

	ret = unlink(file.c_str());
	if (ret < 0) {
		LOG_ERROR("Failed to remove %s: %s/%d", file.c_str(),
				strerror(errno), errno);
		ret = -errno;
	}

	return ret;
}

int cRegulatoryBin::setupSymlink()
{
	int ret;
	struct stat sb;
	bool fcc = false, create = false;
	::std::string regBinFile,
	 	 regBinPath = regBinFolder;
	char realfile [PATH_MAX];

	fcc = cKDSConfiguration::getInstance()->isWiFiFCC();
	ret = getRegulatoryBin(fcc, regBinFolder, regBinFile);
	if (ret < 0) {
		LOG_ERROR("Failed to get the regulatory bin file : %s/%d",
				strerror(-ret), -ret);
		return ret;
	}

	regBinPath.append("/");
	regBinPath.append(regBinFile);
	LOG_INFO("Regulatory Bin File for this device is identified as: %s",
			regBinPath.c_str());

	ret = lstat(customRegBinPath.c_str(), &sb);
	if (ret < 0) {
		LOG_ERROR("Regulatory bin custom dir %s lookup failed: %s/%d",
				customRegBinPath.c_str(), strerror(errno), errno);
		create = true;
		goto create_symlink;
	}

	/* Remove the file if it is not a symlink */
	if (!S_ISLNK(sb.st_mode)) {
		removeFile(customRegBinPath);
		create = true;
	}

	memset(realfile, 0 , sizeof(realfile));

	if (!create) {

		ret = static_cast<int>(readlink(customRegBinPath.c_str(), realfile, sizeof(realfile)));
        LOG_INFO ("%s is pointing to %s", customRegBinPath.c_str(), realfile);

        if (ret < 0) {
        	LOG_ERROR("Failed to perform readlink on: %s [%s/%d]",
        			customRegBinPath.c_str(), strerror(errno), -errno);
        	create = true;
        } else {
        	if (regBinPath.compare(realfile))
        		create = true;
        }

        if (create)
        	removeFile(customRegBinPath);
	}

create_symlink:
	if (create)
		return createSymLink(regBinPath, customRegBinPath);
	return 0;
}

	}
}

/** @} */
