/**
 * @file Regulatory80211dMethod.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include "Regulatory80211dMethod.h"
#include "WpaSupplicantClient.h"
#include "KDSConfiguration.h"

#define WBL_BSS_LOWEST_HEADCOUNT_FOR_REGULATORY_DECISIONS		1

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cRegulatory80211dMethod, Debug);

cRegulatory80211dMethod::cRegulatory80211dMethod() :
		cRegulatoryMethod(REGULATORY_METHOD_80211D, REGULATORY_METHOD_PRIO_LOW,
				eCC_SOURCE_IEEE80211D, CC_TYPE_ALPHA2)
{
	int iRet;

	_notify = false;
	_registered = false;

	iRet = cWpaSupplicantClient::getInstance()->Subscribe(this);
	if (iRet < 0) {
		LOG_ERROR("Failed to subscribe to the wpa_supplicant service availability");
	}
}

cRegulatory80211dMethod::cRegulatory80211dMethod(cRegulatoryChangeNotifier *observer) :
		cRegulatoryMethod(REGULATORY_METHOD_80211D, REGULATORY_METHOD_PRIO_LOW,
				eCC_SOURCE_IEEE80211D, CC_TYPE_ALPHA2)
{
	int iRet;

	_notify = false;
	_registered = false;

	iRet = cWpaSupplicantClient::getInstance()->Subscribe(this);
	if (iRet < 0) {
		LOG_ERROR("Failed to subscribe to the wpa_supplicant service availability");
	}

	if (observer) {
		iRet = Register(observer);
		if (iRet < 0)
			LOG_ERROR ("Failed to register the observer[%p]: %s/%d", observer,
					strerror(-iRet), -iRet);
	}
}

cRegulatory80211dMethod::~cRegulatory80211dMethod()
{
   try
   {
      cWpaSupplicantClient::getInstance()->UnSubscribe(this);
   }catch(...){}
}

int cRegulatory80211dMethod::start()
{
	int iRet = 0;

	if (!_registered) {
		iRet = cWpaSupplicantClient::getInstance()->registerBSSNotification(this);
		if (iRet < 0) {
			LOG_ERROR("Failed to register to the BSS notifications: %s/%d",
					strerror(-iRet), iRet);
		} else {
			_registered = true;
		}
	}

	if (!_notify)
		_notify = true;

	return iRet;
}

int cRegulatory80211dMethod::stop()
{
	int iRet = 0;

	if (_registered) {
		iRet = cWpaSupplicantClient::getInstance()->unRegisterBSSNotification(this);
		if (iRet < 0) {
			LOG_ERROR("Failed to Unregister to the BSS notifications: %s/%d",
					strerror(-iRet), iRet);
		} else {
			_registered = false;
		}
	}

	if (_notify)
		_notify = false;

	return iRet;
}

cCountryGroup *cRegulatory80211dMethod::getCountry(::std::vector<cCountryGroup> &BSSs,
		const ::std::string &country)
{
	::std::vector<cCountryGroup>::iterator it;

	for (it = BSSs.begin(); it < BSSs.end(); it++)
		if (!(*it)._country.compare(country))
			return &(*it);
	return NULL;
}

void cRegulatory80211dMethod::dumpCountryDetails(::std::vector<cCountryGroup> &BSSs)
{
	::std::vector<cCountryGroup>::iterator it;

	LOG_INFO ("Country group size: %ld", BSSs.size());
	for (it = BSSs.begin(); it < BSSs.end(); it++)
		LOG_INFO("Country: %s [BSS Count: %d]", (*it)._country.c_str(),
				(*it)._count);
}

void cRegulatory80211dMethod::notifyRegChannelAvailabilty(const eRegChannelType_t type,
			const bool availability)
{
	LOG_INFO ("Regulatory Channel type: %s, Availability: %s",
			(REGULATORY_CHANNEL_WPA_SUPPLICANT == type) ? "wpa_supplicant" : "Unknown",
					availability ? "AVAILABLE" : "UNAVAILABLE");

	if (REGULATORY_CHANNEL_WPA_SUPPLICANT == type) {
		if (_availability != availability) {
			_availability = availability;
			notifyServiceAvailability();
		}
	}
}

void cRegulatory80211dMethod::updateCountryChange(::std::vector<cCountryGroup> &BSSs)
{
	bool fcc, no80211d = false;
	unsigned int maxCount = 0;
	::std::string country = "00";
	::std::vector<cCountryGroup>::iterator it;

	fcc = cKDSConfiguration::getInstance()->isWiFiFCC();

#if 0
	if (fcc) {
		LOG_INFO("Module is configured for FCC, 80211D based survey is not enabled");
		return;
	}
#endif

	LOG_INFO("Module is configured for %s, 80211D based survey is enabled",
			fcc ? "FCC" : "NON-FCC");

	if (!BSSs.size())
		no80211d = true;
	else if (BSSs.size() == 1) {
		maxCount = BSSs.at(0)._count;
		country = BSSs.at(0)._country;
#if 0
		if (fcc) {
			if (country.compare("US"))
				country = "00";
		}
#endif
	} else {

#if 0
		if (fcc) {
			country = "00";
		} else {
#endif
			/* statistics based on the larger group */
			::std::sort(BSSs.begin(), BSSs.end());
			maxCount = BSSs.back()._count;
			country = BSSs.back()._country;
			BSSs.pop_back();

			for (it = BSSs.begin(); it < BSSs.end(); ++it) {
				if ((*it)._count == maxCount) {
					country = "00";
					break;
				}
			}
#if 0
		}
#endif
	}

	if (!no80211d) {
#if 0
		if (!fcc) {
#endif
			if (maxCount < WBL_BSS_LOWEST_HEADCOUNT_FOR_REGULATORY_DECISIONS)
				return;
#if 0
		}
#endif
	} else {
#if 0
		if (!fcc)
#endif
			return;
	}

	LOG_INFO ("Regulatory Change Notification: %s, Notify: %s", country.c_str(),
			(true == _notify) ? "YES" : "NO");
	if (!country.empty() && (true == _notify))
		notifyRegChange(country);
}

void cRegulatory80211dMethod::BSSNotification(eBssEvent_t event, const ::std::string &objPath,
			const cBasicServiceSet &bss)
{
   	(void) bss;

	cCountryGroup *country;
	::std::vector< cCountryGroup > BSSs;
	::std::vector< cBasicServiceSet > bsses;
	::std::vector< cBasicServiceSet >::iterator it;

	LOG_INFO ("Event:%s, object: %s", BssEventToString(event).c_str(),
			objPath.c_str());

	bsses = cWpaSupplicantClient::getInstance()->getBSSes();
	for (it = bsses.begin(); it < bsses.end(); it++) {
		if ((*it).Get80211d()) {
			country = getCountry(BSSs, (*it).GetCountry());
			if (!country) {
				cCountryGroup obj;

				obj._country = (*it).GetCountry();
				obj._count++;
				obj._BSSs.push_back((*it).GetObjPath());
				BSSs.push_back(obj);
			} else {
				country->_count++;
				country->_BSSs.push_back((*it).GetObjPath());
			}
		}
	}

	dumpCountryDetails(BSSs);
	updateCountryChange(BSSs);
}

	}
}

/** @} */
