/**
 * @file DBusClient.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include <errno.h>
#include "DbusIfTypes.h"
#include "DBusProxyFactory.h"
#include "DBusClient.h"
#include "IDBusProxyIf.h"
#include "WBLDefines.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/Regulation", cDBusClient, Debug);

static const std::string sDBusBusName = "org.freedesktop.DBus";
static const std::string sDBusObjPath = "/org/freedesktop/DBus";

cDBusClient::cDBusClient()
{
	int iRet;
	cIDBusProxyIf *dbusproxy;

	_sesBusServiceSvailability = false;
	dbusproxy = DBusProxyFactory::getInstance()->getDBusProxyIf();
	iRet = dbusproxy->createProxy("wpa_supplicant_bss_port", sDBusBusName, sDBusObjPath,
			DBUS_BUS_SESSION, PROXY_TYPE_WIRED_AT_RUNTIME);
	if (iRet == 0) {
		iRet = dbusproxy->setCallbackIf(sDBusBusName, sDBusObjPath, DBUS_BUS_SESSION, this);
		if (iRet < 0) {
			LOG_ERROR ("Failed to add the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
					sDBusObjPath.c_str(), DBUS_BUS_SESSION, strerror(-iRet), -iRet);
		}
	} else {
		LOG_ERROR ("Failed to create proxy to object: [objPath: %s] [BusType: %d] %s/%d",
				sDBusObjPath.c_str(), DBUS_BUS_SESSION, strerror(-iRet), -iRet);
	}
}

cDBusClient::~cDBusClient()
{
   try
   {
      int iRet;
      cIDBusProxyIf *dbusproxy;

      dbusproxy = DBusProxyFactory::getInstance()->getDBusProxyIf();
      iRet = dbusproxy->destroyProxy(PROXY_DESTROY_ONE, sDBusBusName, DBUS_BUS_SESSION, sDBusObjPath);
      if (iRet == 0) {
         iRet = dbusproxy->setCallbackIf(sDBusBusName, sDBusObjPath, DBUS_BUS_SESSION, nullptr);
         if (iRet < 0) {
            LOG_ERROR ("Failed to remove the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
                  sDBusObjPath.c_str(), DBUS_BUS_SESSION, strerror(-iRet), -iRet);
         }
      } else {
         LOG_ERROR ("Failed to destroy proxy to object: [objPath: %s] [BusType: %d] %s/%d",
               sDBusObjPath.c_str(), DBUS_BUS_SESSION, strerror(-iRet), -iRet);
      }
   }catch(...){}
}

int cDBusClient::registerNameOwnerNotification(cDBusWatcher *listener)
{
	int iRet;
	std::vector<cDBusWatcher *>::iterator it;

	/* Currently only session bus is explored */
	if (!listener ||
			listener->_bustype != DBUS_BUS_SESSION)
		return -EINVAL;

	LOG_INFO ("Registering the Name Owner Observer: %p", listener);

	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++)
		if (listener == *it)
			return -EALREADY;

	_NameChangeObservers.push_back (listener);
	iRet = getUniqueName(listener->_busname, listener->_bustype, listener);
	if (iRet < 0) {
		LOG_ERROR ("Failed to get the unique name for: %s [error: %s/%d]",
				listener->_busname.c_str(), strerror(-iRet), -iRet);
	}
	return 0;
}

int cDBusClient::unRegisterNameOwnerNotification(cDBusWatcher *listener)
{
	bool exist = false;
	std::vector< cDBusWatcher *>::iterator it;

	if (!listener)
		return -EINVAL;

	LOG_INFO ("UnRegistering the Name Owner Observer: %p", listener);

	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++)
		if (listener == *it) {
			exist = true;
			break;
		}

	if (exist) {
		_NameChangeObservers.erase(it);
		return 0;
	}

	return -ENOENT;
}

int cDBusClient::currentSearch()
{
	bool search = false;
	std::vector< cDBusWatcher *>::iterator it;

	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++) {
		if ((*it)->_searchstate == DBUS_SEARCH_SEARCH_INITIATED) {
			search = true;
			break;
		}
	}

	return ((search == true) ? -EINPROGRESS : 0);
}

int cDBusClient::getUniqueName (const ::std::string &busname, const ::DBusBusType busType,
		cDBusWatcher *listener)
{
	act_t token;
	int iRet = 0;
	cIDBusProxyIf *dbusproxy;

	if (!_sesBusServiceSvailability ||
			busname.empty() || !listener)
		return -EINVAL;

	iRet = currentSearch();
	if (iRet < 0)
		return iRet;

	dbusproxy = DBusProxyFactory::getInstance()->getDBusProxyIf();
	token = dbusproxy->sendGetNameOwnerRequest(sDBusBusName, sDBusObjPath, busType,
			*this, busname);
	if (token == DEFAULT_ACT) {
		LOG_ERROR ("Failed to initiate Get Name Owner request for: %s",
				busname.c_str());
		return -ENOTCONN;
	}

	listener->_searchstate = DBUS_SEARCH_SEARCH_INITIATED;
	return iRet;
}

void cDBusClient::searchQueuedRequests()
{
	act_t token;
	bool search = false;
	cIDBusProxyIf *dbusproxy;
	std::vector< cDBusWatcher *>::iterator it;

	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++) {
		if ((*it)->_searchstate == DBUS_SEARCH_INVAL) {
			search = true;
			break;
		}
	}

	if (search) {
		dbusproxy = DBusProxyFactory::getInstance()->getDBusProxyIf();
		token = dbusproxy->sendGetNameOwnerRequest(sDBusBusName, sDBusObjPath,
				DBUS_BUS_SESSION, *this, (*it)->_busname);
		if (token == DEFAULT_ACT) {
			LOG_ERROR ("Failed to initiate Get Name Owner request for: %s",
					(*it)->_busname.c_str());
		} else {
			LOG_ERROR ("Search: %p", *it);
			(*it)->_searchstate = DBUS_SEARCH_SEARCH_INITIATED;
		}
	}
}

void cDBusClient::onDBusServiceAvailable(const ::std::string& busName,
		const ::std::string& objPath, const ::DBusBusType busType,
		const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState)
{
   (void) busName;
   (void) objPath;
   (void) previousState;
   (void) currentState;

	if (busType == DBUS_BUS_SESSION)
		_sesBusServiceSvailability = true;
	searchQueuedRequests();
}

void cDBusClient::onDBusServiceUnavailable(const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType,
		const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState)
{
   (void) previousState;
   (void) currentState;

   std::vector< cDBusWatcher *>::iterator it;

	LOG_INFO ("Service Unavailable for: %s [objpath : %s/%d]", busName.c_str(),
			objPath.c_str(), busType);

	if (busType == DBUS_BUS_SESSION)
		_sesBusServiceSvailability = false;
	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++) {
		if ((*it)->_searchstate != DBUS_SEARCH_INVAL)
			(*it)->_searchstate = DBUS_SEARCH_INVAL;
	}
}

void cDBusClient::onGetNameOwnerError(const ::boost::shared_ptr< cDBusProxy >& proxy,
		const ::boost::shared_ptr< cGetNameOwnerError >& error)
{
	::std::string uniquename;
	std::vector< cDBusWatcher *>::iterator it;

	LOG_ERROR ("On Get NameOwner Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());

	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++) {
		if ((*it)->_searchstate == DBUS_SEARCH_SEARCH_INITIATED) {
			LOG_INFO("Notifying the listener on get name owner failure: %p", (*it));
			(*it)->updateNameOwnerChanges(false, (*it)->_busname, (*it)->_uniquename, proxy->getBusType());
			(*it)->_searchstate = DBUS_SEARCH_SEARCH_DONE;
		}
	}

	searchQueuedRequests();
}

void cDBusClient::onNameOwnerChangedError(const ::boost::shared_ptr< cDBusProxy > &proxy,
		const ::boost::shared_ptr< cNameOwnerChangedError > &error)
{
	LOG_ERROR ("On NameOwner Changed Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

void cDBusClient::onGetNameOwnerResponse(const ::boost::shared_ptr< cDBusProxy >& proxy,
		const ::boost::shared_ptr< cGetNameOwnerResponse >& response)
{
	::std::string uniquename;
	std::vector< cDBusWatcher *>::iterator it;

    LOG_INFO ("Get NameOwner Response: %s [interface: %s] [bustype: %d]", proxy->getDBusObjectPath().c_str(),
    		proxy->getInterfaceName().c_str(), proxy->getBusType());

    uniquename = response->getUniqueid();
    LOG_INFO ("Unique Name : %s", uniquename.c_str());
	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++) {
		if ((*it)->_searchstate == DBUS_SEARCH_SEARCH_INITIATED) {
			LOG_INFO("Notifying the listener on bus name appearance: %p", (*it));
			(*it)->_uniquename = uniquename;
			(*it)->updateNameOwnerChanges(true, (*it)->_busname, uniquename, proxy->getBusType());
			(*it)->_searchstate = DBUS_SEARCH_SEARCH_DONE;
		}
	}

	searchQueuedRequests();
}

void cDBusClient::onNameOwnerChangedSignal(const ::boost::shared_ptr< cDBusProxy >& proxy,
		const ::boost::shared_ptr< cNameOwnerChangedSignal >& signal)
{
	bool added = false, changed = false;
	::std::string busname, newowner, oldowner;
	std::vector< cDBusWatcher *>::iterator it;

    LOG_INFO ("NameOwner Changed Signal: %s [interface: %s] [bustype: %d]", proxy->getDBusObjectPath().c_str(),
    		proxy->getInterfaceName().c_str(), proxy->getBusType());

    busname = signal->getName();
    newowner = signal->getNew();
    oldowner = signal->getOld();

    LOG_INFO("BusName: %s, newowner: %s, oldowner: %s", busname.c_str(),
    		newowner.c_str(), oldowner.c_str());

	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++) {
		if (!(*it)->_busname.compare(busname)) {
			LOG_INFO("Existing Unique name: %s", (*it)->_uniquename.c_str());
			if (oldowner.empty() && !newowner.empty()) {
				if ((*it)->_uniquename.compare(newowner)) {
					added = true;
					changed = true;
					(*it)->_uniquename = newowner;
				}
			}
			else if (newowner.empty() && !(*it)->_uniquename.empty() &&
					!oldowner.empty() && !(*it)->_uniquename.compare(oldowner)) {
				(*it)->_uniquename.clear();
				changed = true;
			}

			if (changed)
				(*it)->updateNameOwnerChanges(added, (*it)->_busname, (*it)->_uniquename, proxy->getBusType());

			if ((*it)->_searchstate != DBUS_SEARCH_SEARCH_DONE)
				(*it)->_searchstate = DBUS_SEARCH_SEARCH_DONE;
		}
	}
}

void cDBusClient::nameOwnerDisappeared(const ::std::string &busname, const ::DBusBusType busType)
{
	std::vector< cDBusWatcher *>::iterator it;

	LOG_INFO ("NameOwner Disappeared: %s", busname.c_str());

	for (it = _NameChangeObservers.begin(); it < _NameChangeObservers.end(); it++) {
		if (!(*it)->_busname.compare(busname)) {
			(*it)->_uniquename.clear();
			(*it)->updateNameOwnerChanges(false, (*it)->_busname, (*it)->_uniquename, busType);
			if ((*it)->_searchstate != DBUS_SEARCH_SEARCH_DONE)
				(*it)->_searchstate = DBUS_SEARCH_SEARCH_DONE;
		}
	}
}

	}
}

/** @} */
