/**
 * @file WifiNetworkInterface.h
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 *
 * @{
 */

#ifndef _WIFI_NETWORK_INTERFACE_H
#define _WIFI_NETWORK_INTERFACE_H

#include <string>
#include <string.h>
#include <errno.h>
#include <linux/if_ether.h>
#include "WBLUtils.h"

namespace org {
    namespace bosch {

        class cWifiNetworkInterface {
            ::std::string _smacaddress,
             _ifname;
            uint8_t _rmacaddress[ETH_ALEN];
            uint32_t _index;

        public:
            cWifiNetworkInterface() {
                memset(_rmacaddress, 0 , sizeof (_rmacaddress));
                _index = -1;
            }

            cWifiNetworkInterface(const ::std::string &ifname,
                    const uint8_t *addr, uint32_t index) {
                char macaddress[18];

                _index = index;
                _ifname = ifname;
                memset(_rmacaddress, 0, sizeof (_rmacaddress));

                if (addr) {
                    memcpy(_rmacaddress, addr, ETH_ALEN);
                    memset(macaddress, 0, sizeof(macaddress));
                    snprintf(macaddress, sizeof(macaddress),
                            "%02x:%02x:%02x:%02x:%02x:%02x",
                            addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);
                    _smacaddress = macaddress;
                }
            }

            cWifiNetworkInterface(const ::std::string &ifname,
                    const ::std::string &addr, uint32_t index) {

                size_t i = 0, temp = 0;
                char address[13];

                _index = index;
                _ifname = ifname;
                if (!addr.empty()) {
                    memset(_rmacaddress, 0, ETH_ALEN);
                    memset(address, 0, sizeof (address));
                    for ( ; i < addr.length(); ++i)
                        if (addr.c_str()[i] != ':')
                            address[temp++] = addr.c_str()[i];
                    cUtils::convertHexToBin(address, (char *)_rmacaddress, ETH_ALEN);
                    _smacaddress = addr;
                }
            }

            bool operator==(cWifiNetworkInterface &iface) {
                if (_index == iface.getIndex() &&
                        !_ifname.compare(iface.getIfName()) &&
                        !_smacaddress.compare(iface.getMacAddress()))
                    return true;
                return false;
            }

            cWifiNetworkInterface &operator=(cWifiNetworkInterface &iface) {
                if(this == &iface)
                    return *this;
                _index = iface.getIndex();
                _ifname = iface.getIfName();
                _smacaddress = iface.getMacAddress();
                memset(_rmacaddress, 0, sizeof (_rmacaddress));
                memcpy(_rmacaddress, iface.getRMacAddress(), ETH_ALEN);
                return *this;
            }

            ~cWifiNetworkInterface() { }

            uint8_t *getRMacAddress() {
                return _rmacaddress;
            }

            int setIfname(const ::std::string &name) {
                if (!_ifname.compare(name))
                    return -EALREADY;
                _ifname = name;
                return 0;
            }

            const ::std::string &getIfName() {
                return _ifname;
            }

            int setIndex(uint32_t index) {
                if (_index == index)
                    return -EALREADY;
                _index = index;
                return 0;
            }

            uint32_t &getIndex() {
                return _index;
            }

            ::std::string &getMacAddress() {
                return _smacaddress;
            }

            int setMacAddress(uint8_t *address) {
                char macaddress[18];

                if (!address)
                    return -EINVAL;

                memset(_rmacaddress, 0, sizeof (_rmacaddress));
                memcpy(_rmacaddress, address, ETH_ALEN);
                memset(macaddress, 0, sizeof(macaddress));
                snprintf(macaddress, sizeof(macaddress),
                        "%02x:%02x:%02x:%02x:%02x:%02x",
                        address[0], address[1], address[2],
                        address[3], address[4], address[5]);
                _smacaddress = macaddress;
                return 0;
            }

            int setMacAddress(const ::std::string addr) {

                size_t index = 0, temp = 0;
                char address[13];

                if (!_smacaddress.compare(addr))
                    return -EALREADY;

                memset(_rmacaddress, 0, ETH_ALEN);
                memset(address, 0, sizeof (address));

                for ( ; index < addr.length(); ++index)
                    if (addr.c_str()[index] != ':')
                        address[temp++] = addr.c_str()[index];

                cUtils::convertHexToBin(address, (char *)_rmacaddress, ETH_ALEN);
                _smacaddress = addr;
                return 0;
            }

        };

    }
}

#endif //_WIFI_NETWORK_INTERFACE_H

/** @} */
