/**
 * @file RegulatoryMethod.h
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#ifndef _REGULATORY_METHOD_H
#define _REGULATORY_METHOD_H

#include <errno.h>
#include <vector>
#include "DBManagerFactory.h"
#include "CountryCodeClientInfo.h"

namespace org {
    namespace bosch {

        typedef enum {
            REGULATORY_METHOD_GNSS = 1,
            REGULATORY_METHOD_MCC,
            REGULATORY_METHOD_80211D,
            REGULATORY_METHOD_MAX
        } eRegulatoryMethod_t;

        typedef enum {
            REGULATORY_METHOD_PRIO_LOW = -100,
            REGULATORY_METHOD_PRIO_DEFAULT = 0,
            REGULATORY_METHOD_PRIO_HIGH = 100
        } eRegMethodPriority_t;

        class cCountryChangeUpdate {
        public:
            cCountryChangeUpdate() { }
            virtual ~cCountryChangeUpdate() { }
            virtual void countryChangeUpdate(const ::std::string &country) = 0;
            virtual void countryChangeAvailability(bool availability) = 0;
        };

        class cRegulatoryChangeNotifier {
        public:
            cRegulatoryChangeNotifier() {}
            virtual ~cRegulatoryChangeNotifier() {}
            virtual int notifyRegChange(const std::string &country) = 0;
            virtual void serviceAvailability(const eRegulatoryMethod_t method,
                    const bool available) = 0;
            std::string regMethodToString(const eRegulatoryMethod_t method) {
                switch(method) {
                case REGULATORY_METHOD_GNSS:
                    return "REGULATORY_METHOD_GNSS";
                case REGULATORY_METHOD_MCC:
                    return "REGULATORY_METHOD_MCC";
                case REGULATORY_METHOD_80211D:
                    return "REGULATORY_METHOD_80211D";
                case REGULATORY_METHOD_MAX:
                    return "UNKNOWN";
                }
                return "UNKNOWN";
            }

            eRegulatoryMethod_t srcToRegMethod(const ::std::string &src) {
                eRegulatoryMethod_t method = REGULATORY_METHOD_MAX;

                if (!src.compare("MCC") || !src.compare("mcc"))
                    method = REGULATORY_METHOD_MCC;
                else if (!src.compare("GNSS") || !src.compare("gnss"))
                    method = REGULATORY_METHOD_GNSS;
                else if (!src.compare("80211D") || !src.compare("80211d"))
                    method = REGULATORY_METHOD_80211D;

                return method;
            }
        };

        class cRegulatoryMethod : public cCountryCodeClientInfo {
        private:
            eRegulatoryMethod_t _method;
            eRegMethodPriority_t _priority;
            std::vector<cRegulatoryChangeNotifier *> _observers;
            cRegulatoryMethod(const cRegulatoryMethod& ref);
            cRegulatoryMethod& operator =(const cRegulatoryMethod& ref);
        protected:
            bool _availability;
        public:
            eRegulatoryMethod_t getRegMethod() {
                return _method;
            }
            bool getServiceAvailability() {
                return _availability;
            }
            int getMethodPriority() {
                return _priority;
            }
            static bool RegPrioComparison (cRegulatoryMethod *method1,
                    cRegulatoryMethod *method2) {
                return method1->getMethodPriority() > method2->getMethodPriority();
            }
            explicit cRegulatoryMethod(const eRegulatoryMethod_t method,
                    const eRegMethodPriority_t priority,
                    const cc_source_t src,
                    const uint32_t type) : cCountryCodeClientInfo (src, type) {
                _priority = priority;
                _availability = false;
                _method = method;
            }

            int Register(cRegulatoryChangeNotifier *observer) {
                std::vector<cRegulatoryChangeNotifier *>::iterator it;
                for (it = _observers.begin(); it < _observers.end(); it++) {
                    if (observer == *it)
                        return -EALREADY;
                }
                _observers.push_back (observer);
                /* Update the current status for the registering client */
                notifyServiceAvailability();
                return 0;
            }
            int UnRegister(cRegulatoryChangeNotifier *observer) {
                std::vector<cRegulatoryChangeNotifier *>::iterator it;
                for (it = _observers.begin(); it < _observers.end(); it++) {
                    if (observer == *it) {
                        _observers.erase (it);
                        return 0;
                    }
                }
                return -ENOENT;
            }
            void notifyRegChange(const ::std::string &country) {
                std::vector<cRegulatoryChangeNotifier *>::iterator it;
                for (it = _observers.begin(); it < _observers.end(); ++it)
                    (*it)->notifyRegChange (country);
            }
            void notifyServiceAvailability() {
                std::vector<cRegulatoryChangeNotifier *>::iterator it;
                for (it = _observers.begin(); it < _observers.end(); ++it)
                    (*it)->serviceAvailability (_method, _availability);
            }
            virtual ~cRegulatoryMethod() { }
            virtual int start() = 0;
            virtual int stop() = 0;
        };

    }
}

#endif //_REGULATORY_METHOD_H

/** @} */
