/**
 * @file DBusClient.h
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */
#ifndef _DBUS_CLIENT_H
#define _DBUS_CLIENT_H

#include "asf/core/Logger.h"
#include "DBusTypes.h"
#include "IDBusNotifCallbackIf.h"
#include "ProxyTypes.h"
#include "GenericSingleton.h"

namespace org {
	namespace bosch {

class cDBusWatcher;

typedef enum {
	DBUS_SEARCH_INVAL,
	DBUS_SEARCH_SEARCH_INITIATED,
	DBUS_SEARCH_SEARCH_DONE
} eBusSearch_t;

class cNameOwnerChangedObserver {
public:
	cNameOwnerChangedObserver() { }
	virtual ~cNameOwnerChangedObserver() { }
	virtual void nameAppeared(const ::std::string &busname,
			const ::std::string &uniquename) = 0;
	virtual void nameDisappeared(const ::std::string &busname) = 0;
};

class cDBusClient final:
		public cIDBusNotifCallbackIf,
		public cGetNameOwnerCallbackIF,
		public GenericSingleton<cDBusClient> {

public:
	cDBusClient();
	virtual ~cDBusClient();

	virtual void onDBusServiceAvailable(const ::std::string &busName,
			const ::std::string &objPath, const ::DBusBusType busType,
			const ::asf::core::ServiceState previousState,
			const ::asf::core::ServiceState currentState) override;
	virtual void onDBusServiceUnavailable(const ::std::string &busName,
			const ::std::string &objPath, const ::DBusBusType busType,
			const ::asf::core::ServiceState previousState,
			const ::asf::core::ServiceState currentState) override;

	virtual void onGetNameOwnerError(const ::boost::shared_ptr< cDBusProxy >& proxy,
			const ::boost::shared_ptr< cGetNameOwnerError >& error) override;
	virtual void onGetNameOwnerResponse(const ::boost::shared_ptr< cDBusProxy >& proxy,
			const ::boost::shared_ptr< cGetNameOwnerResponse >& response) override;

	virtual void onNameOwnerChangedError(const ::boost::shared_ptr< cDBusProxy > &proxy,
			const ::boost::shared_ptr< cNameOwnerChangedError > &error) override;
	virtual void onNameOwnerChangedSignal(const ::boost::shared_ptr< cDBusProxy >& proxy,
				const ::boost::shared_ptr< cNameOwnerChangedSignal >& signal) override;

	int registerNameOwnerNotification(cDBusWatcher *listener);
	int unRegisterNameOwnerNotification(cDBusWatcher *listener);

	void nameOwnerDisappeared(const ::std::string &busname, const ::DBusBusType busType);

private:

	void searchQueuedRequests();
	int getUniqueName (const ::std::string &busname, const ::DBusBusType busType,
			cDBusWatcher *listener);
	int currentSearch();

	bool _sesBusServiceSvailability;
	::std::vector< cDBusWatcher* > _NameChangeObservers;

	friend class GenericSingleton<cDBusClient>;
	DECLARE_CLASS_LOGGER();
};

class cDBusWatcher {
	friend class cDBusClient;
	::std::vector< cNameOwnerChangedObserver* > _observers;
protected:
	eBusSearch_t _searchstate;
	::std::string _uniquename;
public:
	::std::string _clientname;
	::std::string _busname;
	::DBusBusType _bustype;
	explicit cDBusWatcher(const ::std::string &client, const ::std::string &busname,
			const ::DBusBusType busType) {
		_searchstate = DBUS_SEARCH_INVAL;
		_clientname = client;
		_busname = busname;
		_bustype = busType;
		(void) cDBusClient::getInstance()->registerNameOwnerNotification(this);
	}
	int subscribeNameOwnerChanges(cNameOwnerChangedObserver *listener) {

		if (!listener)
			return -EINVAL;

		if (::std::find(_observers.begin(), _observers.end(),
				listener) != _observers.end())
			return -EALREADY;

		_observers.push_back (listener);
		return 0;
	}

	int unSubscribeNameOwnerChanges(cNameOwnerChangedObserver *listener) {
		std::vector< cNameOwnerChangedObserver *>::iterator it;

		if (!listener)
			return -EINVAL;

		it = ::std::find(_observers.begin(), _observers.end(), listener);
		if (it == _observers.end())
			return -ENOENT;
		else _observers.erase(it);

		return 0;
	}
	void updateNameOwnerChanges(bool added, const ::std::string &busName,
			const ::std::string &uName, const ::DBusBusType _bustype) {
		(void) _bustype;
		std::vector< cNameOwnerChangedObserver *>::iterator it;
		for (it = _observers.begin(); it != _observers.end(); ++it) {
			if (added)
				(*it)->nameAppeared(busName, uName);
			else {
				(*it)->nameDisappeared(busName);
			}
		}
	}
	virtual ~cDBusWatcher() {
	   try
	   {
	      (void) cDBusClient::getInstance()->unRegisterNameOwnerNotification(this);
	   }catch(...){}
	}
};

	}
}

#endif //_DBUS_CLIENT_H

/** @} */
