/**
 * @file BasicServiceSet.h
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#ifndef _BASIC_SERVICE_SET_H
#define _BASIC_SERVICE_SET_H

#include <string>
#include <errno.h>
#include <dbus/dbus.h>
#include "WBLUtils.h"

namespace org {
	namespace bosch {

#define IEEE80211_NETWORK_KEYMGMT_NONE				BIT(1)
#define IEEE80211_NETWORK_KEYMGMT_IEEE8021X			BIT(2)
#define IEEE80211_NETWORK_KEYMGMT_WPA_NONE			BIT(3)
#define IEEE80211_NETWORK_KEYMGMT_WPA_PSK			BIT(4)
#define IEEE80211_NETWORK_KEYMGMT_WPA_FT_PSK		BIT(5)
#define IEEE80211_NETWORK_KEYMGMT_WPA_PSK_256		BIT(6)
#define IEEE80211_NETWORK_KEYMGMT_WPA_EAP			BIT(7)
#define IEEE80211_NETWORK_KEYMGMT_WPA_FT_EAP		BIT(8)
#define IEEE80211_NETWORK_KEYMGMT_WPA_EAP_256		BIT(9)

#define IEEE80211_NETWORK_CAP_ESS		0x0001
#define IEEE80211_NETWORK_CAP_IBSS		0x0002
#define IEEE80211_NETWORK_CAP_PRIVACY	0x0010

#define IEEE80211_NETWORK_SECURITY_TYPE_IEEE8021X	(IEEE80211_NETWORK_KEYMGMT_WPA_EAP \
		| IEEE80211_NETWORK_KEYMGMT_WPA_FT_EAP	\
		| IEEE80211_NETWORK_KEYMGMT_WPA_EAP_256)

#define IEEE80211_NETWORK_SECURITY_TYPE_PSK			(IEEE80211_NETWORK_KEYMGMT_WPA_PSK \
		| IEEE80211_NETWORK_KEYMGMT_WPA_FT_PSK	\
		| IEEE80211_NETWORK_KEYMGMT_WPA_PSK_256)

#define BSS_IES_SET							BIT(1)
#define BSS_SSID_SET						BIT(2)
#define BSS_FREQUENCY_SET					BIT(3)
#define BSS_PRIVACY_SET						BIT(4)
#define BSS_MODE_SET						BIT(5)
#define BSS_WPA_KEYMGMT_SET					BIT(6)
#define BSS_RSN_KEYMGMT_SET					BIT(7)
#define BSS_BSSID_SET						BIT(8)

typedef enum {
	EVENT_BSS_INVAL,
	EVENT_BSS_ADDED,
	EVENT_BSS_CHANGED,
	EVENT_BSS_REMOVED
} eBssEvent_t;

typedef enum {
	IEEE80211_NETWORK_SECURITY_UNKNOWN,
	IEEE80211_NETWORK_SECURITY_NONE,
	IEEE80211_NETWORK_SECURITY_WEP,
	IEEE80211_NETWORK_SECURITY_PSK,
	IEEE80211_NETWORK_SECURITY_8021X
} eIeee80211NetworkSecurity_t;

typedef enum {
	IEEE80211_NETWORK_MODE_UNKNOWN,
	IEEE80211_NETWORK_MODE_INFRASTRUCTURE,
	IEEE80211_NETWORK_MODE_IBSS,
	IEEE80211_NETWORK_MODE_MASTER
} eIeee80211NetworkMode_t;

typedef enum {
	IEEE80211_NETWORK_KEYMGMT_UNKNOWN,
	IEEE80211_NETWORK_KEYMGMT_WPA,
	IEEE80211_NETWORK_KEYMGMT_RSN
} eIeee80211NetworkKeyMgmt_t;

class cBasicServiceSet {
private:
	bool _has80211d;
	::std::string _sCountry;
	::std::string _sObjPath;
	::std::string _sIface;
	::std::string _macaddress;
	::DBusBusType _BusType;
	uint8_t _ssid[33];
	size_t _ssid_length;
	uint16_t _frequency;
	eIeee80211NetworkSecurity_t _security;
	eIeee80211NetworkMode_t _mode;
	uint32_t _wpakeymgmt, _rsnkeymgmt;
	bool _privacy;
public:
	~cBasicServiceSet() {}
	cBasicServiceSet() {
		_has80211d = false;
		_frequency = 0;
		_ssid_length = 0;
		_wpakeymgmt = 0;
		_rsnkeymgmt = 0;
		_privacy = false;
		_security = IEEE80211_NETWORK_SECURITY_UNKNOWN;
		_mode = IEEE80211_NETWORK_MODE_UNKNOWN;
		_BusType = static_cast<::DBusBusType>(2);
		memset(_ssid, 0, sizeof(_ssid));
	}
	bool operator==(const cBasicServiceSet &ref) const {
		if (_BusType == ref._BusType && _sObjPath.compare(ref._sObjPath)
				&& !_sIface.compare(ref._sIface))
			return true;
		return false;
	}
	bool operator!=(const cBasicServiceSet &ref) const {
		if (_BusType == ref._BusType && _sObjPath.compare(ref._sObjPath)
				&& !_sIface.compare(ref._sIface))
			return false;
		return true;
	}
	void computeBssSecurity() {
		if ((_wpakeymgmt & IEEE80211_NETWORK_SECURITY_TYPE_IEEE8021X) ||
				(_rsnkeymgmt & IEEE80211_NETWORK_SECURITY_TYPE_IEEE8021X))
			_security = IEEE80211_NETWORK_SECURITY_8021X;
		else if ((_wpakeymgmt & IEEE80211_NETWORK_SECURITY_TYPE_PSK) ||
				(_rsnkeymgmt & IEEE80211_NETWORK_SECURITY_TYPE_PSK))
			_security = IEEE80211_NETWORK_SECURITY_PSK;
		else if (_privacy)
			_security = IEEE80211_NETWORK_SECURITY_WEP;
		else
			_security = IEEE80211_NETWORK_SECURITY_NONE;
	}
	const eIeee80211NetworkSecurity_t &getSecurityType() const {
		return _security;
	}
	int setKeyMgmt(eIeee80211NetworkKeyMgmt_t type, uint32_t keyMgmt) {

		if (IEEE80211_NETWORK_KEYMGMT_UNKNOWN == type)
			return -EINVAL;
		else if (IEEE80211_NETWORK_KEYMGMT_WPA == type)
			_wpakeymgmt = keyMgmt;
		else if (IEEE80211_NETWORK_KEYMGMT_RSN == type)
			_rsnkeymgmt = keyMgmt;

		return 0;
	}
	uint32_t getKeyMgmt(eIeee80211NetworkKeyMgmt_t type) {
		if (IEEE80211_NETWORK_KEYMGMT_WPA == type)
			return _wpakeymgmt;
		else if (IEEE80211_NETWORK_KEYMGMT_RSN == type)
			return _rsnkeymgmt;
		return 0;
	}
	uint8_t *getSSID() {
		if (_ssid[0] == '\0')
			return NULL;
		return _ssid;
	}
	size_t &getSsidLength() {
		return _ssid_length;
	}
	int setSSID(uint8_t *ssid, size_t length) {
		if (!ssid || length > 32)
			return -EINVAL;
		if (length == _ssid_length &&
				!memcmp(ssid, _ssid, length))
			return -EALREADY;
		memset(_ssid, 0, sizeof(_ssid));
		memcpy(_ssid, ssid, length);
		_ssid_length = length;
		return 0;
	}
	int setFrequency(uint16_t freq) {
		if (_frequency == freq)
			return -EALREADY;
		_frequency = freq;
		return 0;
	}
	const uint16_t &getFrequency() {
		return _frequency;
	}
	::std::string convertSSIDToHexString() const {
		int i = 0;
		::std::string ssid;
		const uint8_t *temp = _ssid;
		char arr[3];
		for ( ; temp[i]; i++) {
			memset(arr, 0, sizeof (arr));
			snprintf(arr, sizeof(arr), "%02x", static_cast<char>(temp[i]));
			ssid.append(arr);
		}
		return ssid;
	}
	static const ::std::string bssSecurityToString(eIeee80211NetworkSecurity_t security) {
		switch(security) {
		case IEEE80211_NETWORK_SECURITY_UNKNOWN:
			return "unknown";
		case IEEE80211_NETWORK_SECURITY_NONE:
			return "open";
		case IEEE80211_NETWORK_SECURITY_WEP:
			return "wep";
		case IEEE80211_NETWORK_SECURITY_PSK:
			return "psk";
		case IEEE80211_NETWORK_SECURITY_8021X:
			return "ieee8021x";
		}
		return "unknown";
	}
	static const ::std::string bssModeToString(eIeee80211NetworkMode_t mode) {
		switch(mode) {
		case IEEE80211_NETWORK_MODE_UNKNOWN:
			return "unknown";
		case IEEE80211_NETWORK_MODE_IBSS:
			return "ibss";
		case IEEE80211_NETWORK_MODE_INFRASTRUCTURE:
			return "managed";
		case IEEE80211_NETWORK_MODE_MASTER:
			return "master";
		}
		return "unknown";
	}
	int setMode(const char *mode) {
		if (!mode)
			return -EINVAL;
		if (!strcmp(mode, "infrastructure"))
			_mode = IEEE80211_NETWORK_MODE_INFRASTRUCTURE;
		else if (!strcmp(mode, "ad-hoc"))
			_mode = IEEE80211_NETWORK_MODE_IBSS;
		return 0;
	}
	const eIeee80211NetworkMode_t &getMode() const {
		return _mode;
	}
	int setPrivacy(bool privacy) {
		if (_privacy == privacy)
			return -EALREADY;
		_privacy = privacy;
		return 0;
	}
	const bool &getPrivacy() {
		return _privacy;
	}
	int SetCountry(const ::std::string sCountry) {
		if (!_sCountry.compare(sCountry))
			return -EALREADY;
		_sCountry = sCountry;
		return 0;
	}
	const std::string &GetCountry() {
		return _sCountry;
	}
	int SetObjPath(const ::std::string &objPath) {
		if (!_sObjPath.compare(objPath))
			return -EALREADY;
		_sObjPath = objPath;
		return 0;
	}
	const std::string &GetObjPath() {
		return _sObjPath;
	}
	int SetIfName(const ::std::string &ifName) {
		if (!_sIface.compare(ifName))
			return -EALREADY;
		_sIface = ifName;
		return 0;
	}
	const std::string &GetIfName() {
		return _sIface;
	}
	int SetBusType(const ::DBusBusType type) {
		if (_BusType == type)
			return -EALREADY;
		_BusType = type;
		return 0;
	}
	const ::DBusBusType &GetBusType() {
		return _BusType;
	}
	int Set80211d(const bool b80211d) {
		if (_has80211d == b80211d)
			return -EALREADY;
		_has80211d = b80211d;
		return 0;
	}
	const bool &Get80211d() {
		return _has80211d;
	}
	int setMacAddress(char *address) {
		if (!address)
			return -EINVAL;
		_macaddress = address;
		return 0;
	}
	const ::std::string &getMacAddress() {
		return _macaddress;
	}
};

class cBSSNotifier {
	::std::string _name;
public:
	cBSSNotifier(){}
	virtual ~cBSSNotifier(){}
	::std::string BssEventToString(const eBssEvent_t event) {
		switch (event) {
		case EVENT_BSS_INVAL:
			return "Unknown";
		case EVENT_BSS_ADDED:
			return "BSS_ADDED";
		case EVENT_BSS_CHANGED:
			return "BSS_CHANGED";
		case EVENT_BSS_REMOVED:
			return "BSS_REMOVED";
		}
		return "Unknown";
	}
	virtual void BSSNotification(eBssEvent_t event, const ::std::string &objPath,
			const cBasicServiceSet &bss) = 0;
};

	}
}

#endif //_BASIC_SERVICE_SET_H

/** @} */
