/**
 * @file IEEE80211STAClient.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

#include <glib.h>
#include <wifi_rootdaemon_client.h>

#include "DBusProxyFactory.h"
#include "STAModeClient_Msgs.h"
#include "IEEE80211STAClient.h"
#include "DbusIfTypes.h"
#include "DbusParser.h"
#include "ConnmanDbusInterfaces.h"
#include "ConnmanDefines.h"
#include "WBLTypes.h"
#include "WBLDefines.h"
#include "WBLTypeProperties.h"
#include "DbusHelper.h"
#include "IConnmanServiceProxyIf.h"
#include "WBLPortsDefines.h"
#include "DBManagerFactory.h"

namespace org 
{
namespace bosch 
{

static DbusParser<ConnmanDbusInterfaces> connmanDbusParser;

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/IEEE80211Clients", IEEE80211STAClient, Info);

IEEE80211STAClient::IEEE80211STAClient()
{
   LOG_INFO("IEEE80211STAClient() entered ");

   _busType = static_cast<::DBusBusType>(2);

   //! Create Connman Manager Proxy
   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IConnmanManagerProxyIf* connmanManagerProxyIf = (dbusFactory)?
           (dbusFactory->getConnmanManagerProxyIf()) : (nullptr);
   if (connmanManagerProxyIf)
   {
      connmanManagerProxyIf->setCallbackIf(e8PROXY_USER_STA_CLIENT, this);
      connmanManagerProxyIf->createProxy();
   }
   connmanDbusParser.setTraces(true);
   _restoringSTA = false;
}//IEEE80211STAClient::IEEE80211STAClient

IEEE80211STAClient::~IEEE80211STAClient()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   //LOG_INFO("~IEEE80211STAClient() entered ");
}//IEEE80211STAClient::~IEEE80211STAClient

bool IEEE80211STAClient::setPowered(const ::std::string& objPath, bool powered)
{
   LOG_INFO("IEEE80211STAClient::setPowered() entered: Powered = %d, ObjPath = %s ", powered, objPath.c_str());

   bool isRequestSent = false;
   WBLPowerStateType type = WIFI_STATE_UNKNOWN;
   type = powered? WIFI_STATE_POWERING_ON:WIFI_STATE_POWERING_OFF;
   updatePowerState(objPath,type);

   if (isTechnology(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IConnmanTechnologyProxyIf* connmanTechnologyProxyIf = (dbusFactory)?
              (dbusFactory->getConnmanTechnologyProxyIf()) : (nullptr);
      if ((connmanTechnologyProxyIf) &&
            (connmanTechnologyProxyIf->isProxyServiceAvailable(sConnmanBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         ConnmanDbusInterfaces connmanInterfaces;
         ::std::string poweredName = connmanInterfaces.getTechnologyProperty2String(TECHNOLOGY_POWERED);

         ::asf::dbus::DBusVariant poweredVariant;
         init_variant(poweredVariant, powered);
         isRequestSent = connmanTechnologyProxyIf->sendSetPropertyRequest(sConnmanBusName,
               objPath, DBUS_BUS_SYSTEM, *this, poweredName, poweredVariant);
      }
   } // if (isTechnology(objPath))
   return isRequestSent;
}

void IEEE80211STAClient::getTechnologies(TechnologyList& list)
{
   LOG_INFO("IEEE80211STAClient::getTechnologies() entered ");
   list = _technologyList;
}

void IEEE80211STAClient::onConnmanManagerServiceAvailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      const ServiceState previousState, const ServiceState currentState)
{ 
   LOG_INFO("IEEE80211STAClient::onConnmanManagerServiceAvailable() bustype: %d", busType);
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);

   _busType = busType;

#ifndef WBL_INCLUDE_ONLY_REGULATORY_CHANGES
   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IConnmanManagerProxyIf* connmanManagerProxyIf = (dbusFactory)?
           (dbusFactory->getConnmanManagerProxyIf()) : (nullptr);
   if (connmanManagerProxyIf)
   {
      connmanManagerProxyIf->sendGetTechnologiesRequest(*this);
   }
#endif
} // void IEEE80211STAClient::onConnmanManagerServiceAvailable

void IEEE80211STAClient::onConnmanManagerServiceUnavailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      const ServiceState previousState, const ServiceState currentState)
{
   LOG_INFO("IEEE80211STAClient::onConnmanManagerServiceUnavailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);

   _busType = static_cast<::DBusBusType>(2);
   //! Add code
} // void IEEE80211STAClient::onConnmanManagerServiceUnavailable

void IEEE80211STAClient::onTechnologyAddedError(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
         const ::boost::shared_ptr< ConnmanTechnologyAddedError >& error)
{
   LOG_INFO("IEEE80211STAClient::onTechnologyAddedError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onTechnologyAddedError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onTechnologyAddedError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onTechnologyError(objPath, IEEE80211_MSG_STA_MODE_TECHNOLOGYADDED_IND);

} // void IEEE80211STAClient::onTechnologyAddedError 

void IEEE80211STAClient::onTechnologyAddedSignal(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
      const ::boost::shared_ptr< ConnmanTechnologyAddedSignal >& signal)
{
   LOG_INFO("IEEE80211STAClient::onTechnologyAddedSignal() entered ");
   WBL_ASSERT_AND_EXIT(!(signal && proxy));

   ::std::string objPath = signal->getPath();
   LOG_DEBUG("onTechnologyAddedSignal: objPath = %s ", objPath.c_str());

   TechnologyProperty techProperty;
   parseTechnologyInfo(proxy->getDBusBusName(), proxy->getDBusObjectPath(),
         proxy->getBusType(), proxy->getInterfaceName(), signal->getPropertiesMutable(), techProperty);

   //! Add proxy for wifi technology
   if (0 == techProperty.name.compare("WiFi"))
   {
      addTechnology(objPath, techProperty);
   }
} // void IEEE80211STAClient::onTechnologyAddedSignal

void IEEE80211STAClient::onTechnologyRemovedError(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
      const ::boost::shared_ptr< ConnmanTechnologyRemovedError >& error)
{
   LOG_INFO("IEEE80211STAClient::onTechnologyRemovedError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onTechnologyRemovedError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onTechnologyRemovedError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onTechnologyError(objPath, IEEE80211_MSG_STA_MODE_TECHNOLOGYREMOVED_IND);

} // void IEEE80211STAClient::onTechnologyRemovedError

void IEEE80211STAClient::onTechnologyRemovedSignal(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
      const ::boost::shared_ptr< ConnmanTechnologyRemovedSignal >& signal)
{
   LOG_INFO("IEEE80211STAClient::onTechnologyRemovedSignal() entered ");
   WBL_ASSERT_AND_EXIT(!(signal && proxy));

   removeTechnology(signal->getPath());

} // void IEEE80211STAClient::onTechnologyRemovedSignal

void IEEE80211STAClient::onGetTechnologiesError(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
      const ::boost::shared_ptr< ConnmanGetTechnologiesError >& error)
{
   LOG_INFO("IEEE80211STAClient::onGetTechnologiesError() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(error);
   //! Add code
} // void IEEE80211STAClient::onGetTechnologiesError

void IEEE80211STAClient::onGetTechnologiesResponse(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
      const ::boost::shared_ptr< ConnmanGetTechnologiesResponse >& response)
{
   LOG_INFO("IEEE80211STAClient::onGetTechnologiesResponse() entered ");
   WBL_ASSERT_AND_EXIT(!(response && proxy));

   typedef ::std::vector< ConnmanGetTechResponseStruct > TechStructList;
   TechStructList techStructList = response->getTechnologies();

   for (TechStructList::iterator it = techStructList.begin(); it != techStructList.end(); ++it)
   {
      ::std::string techObjPath = (*it).getElem1();
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > techData = (*it).getElem2Mutable();

      TechnologyProperty techProperty;
      parseTechnologyInfo(proxy->getDBusBusName(), proxy->getDBusObjectPath(),
            proxy->getBusType(), proxy->getInterfaceName(), techData, techProperty);

      if (0 == techProperty.name.compare(sConnmanWifiTechName))
      {
         addTechnology(techObjPath, techProperty);
         break;
      }
   } // for (TechStructList::iterator it =...)
} // void IEEE80211STAClient::onGetTechnologiesResponse

void IEEE80211STAClient::onConnmanTechnologyServiceAvailable(const ::std::string& busName, const ::std::string& objPath,
      const ::DBusBusType busType, const ::asf::core::ServiceState previousState,
      const ::asf::core::ServiceState currentState)
{
   LOG_INFO("IEEE80211STAClient::onConnmanTechnologyServiceAvailable() entered ");
   LOG_INFO("IEEE80211STAClient::objPath %s", objPath.c_str());
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IConnmanTechnologyProxyIf* connmanTechnologyProxyIf = (dbusFactory)?
           (dbusFactory->getConnmanTechnologyProxyIf()) : (nullptr);
   if (connmanTechnologyProxyIf)
   {
      (void) connmanTechnologyProxyIf->sendGetPropertiesRequest(busName, objPath, busType, *this);
   }
   ILCMClient *lcmClientIf = LCMFactory::getInstance()->getLCMClientIf();
   if(lcmClientIf)
   {
     lcmClientIf->notifySTASserviceAvailability(true);
   }
   cWifiConnectionStatus::getInstance()->subscribeWifiConnectionStatus(this);
} // void IEEE80211STAClient::onConnmanTechnologyServiceAvailable

void IEEE80211STAClient::restoreLastModeSettings(const ::std::string& objPath, const TechnologyProperty& property)
{
   LOG_INFO("IEEE80211STAClient::restoreLastModeSettings() entered ");
   if (_restoringSTA)
   {
      if ((!property.isPowered) && (LCM_WIFISTATE_NORMAL == getWifiState()))
      {
         if(!setPowered(objPath, true))
         {
            LOG_DEBUG("IEEE80211STAClient::restoreLastModeSettings() STA Restoration Failed. Retry on Next Update");
            _restoringSTA=true;
         }
      }
      else if(property.isPowered) //Restoration taken care automatically.
      {
         updatePowerState(objPath,WIFI_STATE_POWERED_ON);
      }
   }
}
void IEEE80211STAClient::onConnmanTechnologyServiceUnavailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState)
{
   LOG_INFO("IEEE80211STAClient::onConnmanTechnologyServiceUnavailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);
   cWifiConnectionStatus::getInstance()->unSubscribeWifiConnectionStatus(this);
   ILCMClient *lcmClientIf = LCMFactory::getInstance()->getLCMClientIf();
   if(lcmClientIf)
   {
     lcmClientIf->notifySTASserviceAvailability(false);
   }
   //! Add code
} // void IEEE80211STAClient::onConnmanTechnologyServiceUnavailable

void IEEE80211STAClient::onPropertyChangedError(const ::boost::shared_ptr< ConnmanTechnologyProxy >& proxy,
      const ::boost::shared_ptr< ConnmanPropertyChangedError >& error)
{
   LOG_INFO("IEEE80211STAClient::onPropertyChangedError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onTechnologyRemovedError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onTechnologyRemovedError: Error message - %s ", error->getMessage().c_str());
   //TODO - notify error to observers if callback is triggered only for Powered property

} // void IEEE80211STAClient::onPropertyChangedError

void IEEE80211STAClient::onPropertyChangedSignal(const ::boost::shared_ptr< ConnmanTechnologyProxy >& proxy,
      const ::boost::shared_ptr< ConnmanPropertyChangedSignal >& signal)
{
   LOG_INFO("IEEE80211STAClient::onPropertyChangedSignal() entered ");
   WBL_ASSERT_AND_EXIT(!(signal && proxy));

   ::std::string objPath = proxy->getDBusObjectPath();
   if (true == isTechnology(objPath))
   {
      ::std::string changedPropertyName = signal->getName();
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > changedProperties;
      changedProperties.insert(std::pair< ::std::string, ::asf::dbus::DBusVariant >(
            changedPropertyName, signal->getValue()));

      TechnologyProperty techProperty;
      parseTechnologyInfo(proxy->getDBusBusName(), objPath,
            proxy->getBusType(), proxy->getInterfaceName(), changedProperties, techProperty);

      ConnmanDbusInterfaces connmanInterfaces;
      TechnologyPropertyName propertyName = connmanInterfaces.getTechnologyProperty2Enum(changedPropertyName);
      updateTechnology(objPath, techProperty, propertyName);
   }
} // void IEEE80211STAClient::onPropertyChangedSignal

void IEEE80211STAClient::onSetPropertyError(const ::boost::shared_ptr< ConnmanTechnologyProxy >& proxy,
      const ::boost::shared_ptr< ConnmanSetPropertyError >& error)
{
   LOG_INFO("IEEE80211STAClient::onSetPropertyError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onSetPropertyError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onSetPropertyError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();

   if(error->getName().compare("net.connman.Error.AlreadyEnabled") == 0)
   {
      TechnologyProperty techProperty;
      techProperty.isPowered = true;
      updateTechnology(objPath, techProperty, TECHNOLOGY_POWERED);
   }
   else if(error->getName().compare("net.connman.Error.AlreadyDisabled") == 0)
   {
      TechnologyProperty techProperty;
      techProperty.isPowered = false;
      updateTechnology(objPath, techProperty, TECHNOLOGY_POWERED);
   }
   else
   {
      updatePowerState(objPath,WIFI_STATE_POWER_FAILURE);
      onTechnologyError(objPath, IEEE80211_MSG_STA_MODE_POWERED_IND);
   }
   //! @Note: This callback is assumed to be only for Powered property update, since WBL currently
   //! does not set any other property.

} // void IEEE80211STAClient::onSetPropertyError

void IEEE80211STAClient::onSetPropertyResponse(const ::boost::shared_ptr< ConnmanTechnologyProxy >& proxy,
      const ::boost::shared_ptr< ConnmanSetPropertyResponse >& response)
{
   LOG_INFO("IEEE80211STAClient::onSetPropertyResponse() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(response);
   //! @Note: SetProperty response is not notified to observers since the property update will be sent.

} // void IEEE80211STAClient::onSetPropertyResponse

void IEEE80211STAClient::onGetPropertiesError(const ::boost::shared_ptr<ConnmanTechnologyProxy>& proxy,
            const ::boost::shared_ptr<ConnmanGetPropertiesError>& error)
{
   (void) proxy;
   (void) error;

   LOG_INFO("IEEE80211STAClient::onGetPropertiesError() entered ");

}
void IEEE80211STAClient::onGetPropertiesResponse(const ::boost::shared_ptr<ConnmanTechnologyProxy>& proxy,
      const ::boost::shared_ptr<ConnmanGetPropertiesResponse>& response)
{
   LOG_INFO("IEEE80211STAClient::onGetPropertiesResponse() entered ");
   WBL_ASSERT_AND_EXIT(!(response && proxy));

   ::std::map< ::std::string, ::asf::dbus::DBusVariant > techData = response->getProperties();

   TechnologyProperty techProperty;
   parseTechnologyInfo(proxy->getDBusBusName(), proxy->getDBusObjectPath(),
        proxy->getBusType(), proxy->getInterfaceName(), techData, techProperty);

   if (0 == techProperty.name.compare(sConnmanWifiTechName))
   {
      addTechnology(proxy->getDBusObjectPath(), techProperty);
      if(!_restoringSTA)
         updateTechnology(proxy->getDBusObjectPath(), techProperty, TECHNOLOGY_POWERED);
   }
}

bool IEEE80211STAClient::isTechnology(const ::std::string& objPath) const
{
   bool isValidTechnology = false;
   for (TechnologyList::const_iterator it = _technologyList.begin(); it != _technologyList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         isValidTechnology = true;
         break;
      }
   }
   LOG_DEBUG("IEEE80211STAClient::isTechnology: %s (ObjPath = %s)",
         BOOL_TO_STR(isValidTechnology), objPath.c_str());
   return isValidTechnology;
} // bool IEEE80211STAClient::isTechnology

bool IEEE80211STAClient::getTechnologyInfo(const ::std::string& objPath, TechnologyItem& item)
{

   bool isValidTechnology = false;
   for (TechnologyList::iterator it = _technologyList.begin(); it != _technologyList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         isValidTechnology = true;
         item = (*it);
         break;
      }
   }
   LOG_DEBUG("IEEE80211STAClient::getTechnologyInfo: isValidTechnology = %s (ObjPath = %s)",
         BOOL_TO_STR(isValidTechnology), objPath.c_str());
   return isValidTechnology;
} // bool IEEE80211STAClient::getTechnologyInfo

void IEEE80211STAClient::addTechnology(const ::std::string& objPath, const TechnologyProperty& property)
{
   LOG_INFO("IEEE80211STAClient::addTechnology() entered ");

   bool isNewTechnology = true;

   TechnologyItem newTechItem;
   newTechItem.objPath = objPath;
   newTechItem.property = property;

   for (TechnologyList::iterator it = _technologyList.begin(); it != _technologyList.end(); ++it)
   {
      if (objPath == it->objPath)
      {
         isNewTechnology = false;
         break;
      }
   }
   if (true == isNewTechnology)
   {
      initLastModeSettings(objPath,property);
      addTechnologyProxy(objPath);
      _technologyList.push_back(newTechItem);
      TechnologyAddedMsg msg;
      msg.setTechnologyItem(newTechItem);
      notifyObservers(IEEE80211_EVENT_STA, &msg);
   }
   else
   {
      restoreLastModeSettings(objPath,newTechItem.property);
   }// if (true == isNewTechnology)
} // void IEEE80211STAClient::addTechnology

void IEEE80211STAClient::updateTechnology(const ::std::string& objPath, const TechnologyProperty& property,
      const TechnologyPropertyName& changedPropertyName)
{
   LOG_INFO("IEEE80211STAClient::updateTechnology() entered ");

   TechnologyList::iterator it = _technologyList.begin();
   for (; it != _technologyList.end(); ++it)
   {
      if (objPath == (*it).objPath)
      {
         break;
      } // if (objPath == curTechItem.objPath)
   } // for (TechnologyList::iterator it =...)

   if (_technologyList.end() != it)
   {
      TechnologyChangedMsg msg;
      TechnologyProperty curTechProperty = (*it).property;

      switch (changedPropertyName)
      {
         case TECHNOLOGY_POWERED:
         {
            it->property.isPowered = property.isPowered;
            WBLPowerStateType type;
            type = (property.isPowered) ? WIFI_STATE_POWERED_ON : WIFI_STATE_POWERED_OFF;
            updatePowerState(objPath,type);
            msg.setIEEE80211EventID(IEEE80211_MSG_STA_MODE_POWERED_IND);
         }
         break;
         case TECHNOLOGY_CONNECTED:
         {
            (*it).property.isConnected = property.isConnected;
            msg.setIEEE80211EventID(IEEE80211_MSG_STA_MODE_CONNECTED_IND);

            if (true == property.isConnected)
            {
               uint32 frequency = 255;
               uint16 channel = 255;

               if(true == readFrequency(frequency))
               {
                  LOG_DEBUG("Frequency read success");
                  (*it).property.frequency = frequency;
                  channel = getCurrOperatingChannel(frequency);
                  if(channel > 0)
                  {
                     (*it).property.currChannel = channel;
                  }
               }
            }
         }
         break;
         case TECHNOLOGY_NAME:
         {
            (*it).property.name = property.name;
            msg.setIEEE80211EventID(IEEE80211_MSG_STA_MODE_NAME_IND);
         }
         break;
         case TECHNOLOGY_TYPE:
         {
            (*it).property.type = property.type;
            msg.setIEEE80211EventID(IEEE80211_MSG_STA_MODE_TYPE_IND);
         }
         break;
         default:
            LOG_ERROR(" updateTechnology: Invalid changed property name %d ", changedPropertyName);
         break;
      } //switch (changedPropertyName)

      TechnologyProperty newTechProperty = (*it).property;
      bool notifyTechChange = !(curTechProperty == newTechProperty);
      if (notifyTechChange)
      {
         msg.setTechnologyItem(*it);
         notifyObservers(IEEE80211_EVENT_STA, &msg);
      }
   }
} // void IEEE80211STAClient::updateTechnology

bool IEEE80211STAClient::readFrequency(uint32& frequency)
{
   LOG_INFO("IEEE80211STAClient::readFrequency");
   int skfd = 0;
   struct iwreq wrq;
   bool isSuccess = false;
   ::std::string interface = "wlan0";

   strncpy(wrq.ifr_ifrn.ifrn_name, interface.c_str(),sizeof(wrq.ifr_ifrn.ifrn_name)-1);

   skfd = socket(AF_INET, SOCK_DGRAM, 0);
   if ((ioctl(skfd, SIOCGIWFREQ, &wrq) >= 0))
   {
      frequency = static_cast<int>(getFrequencyinMhz(&(wrq.u.freq)));
      isSuccess = true;
   }
   else
   {
      LOG_DEBUG("ioctl failed and returned errno %s", strerror(errno));
   }
   if(0<=skfd)
   {
      close(skfd);
   }
   return isSuccess;
} //void IEEE80211STAClient::readFrequency

double IEEE80211STAClient::getFrequencyinMhz(iwfreq* in)
{
   int    i;
   double res = (double) in->m;
   for(i = 0; i < in->e; i++)
   res *= 10;
   return(res/MEGA);

} //double IEEE80211STAClient::getFrequencyinMhz

uint16 IEEE80211STAClient::getCurrOperatingChannel(uint32 freq)
{
   uint16 channel = 255;
   if((freq >= WBL_FREQ_2_4_LOWER) && (freq <= WBL_FREQ_2_4_UPPER))
   {
      channel = static_cast<uint16> ((freq - WBL_FREQ_2_4_LOWER) / 5 + 1);
   }
   else if ((freq >= WBL_FREQ_5_LOWER) && (freq <= WBL_FREQ_5_UPPER))
   {
      channel = static_cast<uint16> ((freq - WBL_FREQ_5_LOWER) / 5 + 34);
   }
   LOG_DEBUG("Channel is %d",channel);
   return channel;
} //uint16 IEEE80211STAClient::getCurrOperatingChannel

void IEEE80211STAClient::removeTechnology(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211STAClient::removeTechnology() entered ");

   for (TechnologyList::iterator it = _technologyList.begin(); it != _technologyList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         removeTechnologyProxy(objPath);

         _technologyList.erase(it);

         TechnologyRemovedMsg msg;
         msg.setobjPath(objPath);
         notifyObservers(IEEE80211_EVENT_STA, &msg);
         break;
      } // if ((*it).objPath == objPath)
   } // for (TechnologyList::iterator it =...)
} // void IEEE80211STAClient::removeTechnology

void IEEE80211STAClient::addTechnologyProxy(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211STAClient::addTechnologyProxy() entered ");

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IConnmanTechnologyProxyIf* connmanTechnologyProxyIf = (dbusFactory)?
           (dbusFactory->getConnmanTechnologyProxyIf()) : (nullptr);
   if (connmanTechnologyProxyIf)
   {
      connmanTechnologyProxyIf->setCallbackIf(sConnmanBusName,
            objPath, DBUS_BUS_SYSTEM, e8PROXY_USER_STA_CLIENT, this);
      connmanTechnologyProxyIf->createProxy(sConnmanBusName,
            objPath, DBUS_BUS_SYSTEM);
   }
} // void IEEE80211STAClient::addTechnologyProxy

void IEEE80211STAClient::removeTechnologyProxy(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211STAClient::removeTechnologyProxy() entered ");

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IConnmanTechnologyProxyIf* connmanTechnologyProxyIf = (dbusFactory)?
           (dbusFactory->getConnmanTechnologyProxyIf()) : (nullptr);
   if (connmanTechnologyProxyIf)
   {
      connmanTechnologyProxyIf->setCallbackIf(sConnmanBusName,
            objPath, DBUS_BUS_SYSTEM, e8PROXY_USER_STA_CLIENT, nullptr);
   }
} // void IEEE80211STAClient::removeTechnologyProxy

void IEEE80211STAClient::parseTechnologyInfo(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType, const ::std::string& interfaceName,
      ::std::map< ::std::string, ::asf::dbus::DBusVariant >& data, TechnologyProperty& property) const
{
   LOG_INFO("IEEE80211STAClient::parseTechnologyInfo() entered ");

   ::std::vector<int> matchingInterfaces;
   matchingInterfaces.push_back((int)CONNMAN_IF_TECHNOLOGY);

   ::std::vector< DbusVariantProperty > ifProperties;
   connmanDbusParser.parseProperties(ifProperties, matchingInterfaces, data,
         busType, busName, objPath, interfaceName);

   for(size_t i = 0; i < ifProperties.size(); ++i)
   {
      TechnologyPropertyName propertyName = static_cast<TechnologyPropertyName>(ifProperties[i].propEnum);
      switch(propertyName)
      {
         case TECHNOLOGY_POWERED:
         {
            WBL_ASSERT(VARIANT_BOOL != ifProperties[i].propData.getType());
            property.isPowered = ifProperties[i].propData.getBool();
            property.powerState = (property.isPowered)?WIFI_STATE_POWERED_ON:WIFI_STATE_POWERED_OFF;
            LOG_DEBUG("parseTechnologyInfo: PowerState = %s, isPowered = %d ",
                  wblPowerState.getPowerState2String(property.powerState).c_str(), property.isPowered);
            break;
         }
         case TECHNOLOGY_CONNECTED:
         {
            WBL_ASSERT(VARIANT_BOOL != ifProperties[i].propData.getType());
            property.isConnected = ifProperties[i].propData.getBool();
            LOG_DEBUG("parseTechnologyInfo: isConnected = %d ", property.isConnected);
            break;
         }
         case TECHNOLOGY_NAME:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.name = ifProperties[i].propData.getString();
            LOG_DEBUG("parseTechnologyInfo: name = %s ", property.name.c_str());
            break;
         }
         case TECHNOLOGY_TYPE:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.type = ifProperties[i].propData.getString();
            LOG_DEBUG("parseTechnologyInfo: type = %s ", property.type.c_str());
            break;
         }
         default:
         {
            LOG_ERROR("parseTechnologyInfo: Invalid access point property %d ", propertyName);
            break;
         }
      } // switch(ifProperties[i].propEnum)
   } // for(size_t i = 0; i < ifProperties.size(); i++)
} // void IEEE80211STAClient::parseTechnologyInfo

void IEEE80211STAClient::onTechnologyError(const ::std::string& techObjPath, IEEE80211EventID event,
      IEEE80211ErrorCode error)
{
   TechnologyItem item;
   if (getTechnologyInfo(techObjPath, item))
   {
      switch (event)
      {
         case IEEE80211_MSG_AP_MODE_APADDED_IND:
         {
            TechnologyAddedMsg msg;
            msg.setTechnologyItem(item);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
         case IEEE80211_MSG_AP_MODE_APREMOVED_IND:
         {
            TechnologyRemovedMsg msg;
            msg.setobjPath(techObjPath);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
         case IEEE80211_MSG_STA_MODE_POWERED_IND:
         {
            TechnologyChangedMsg msg;
            msg.setTechnologyItem(item);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_STA, &msg);
         }
         break;
         default:
         {
            TechnologyChangedMsg msg;
            msg.setTechnologyItem(item);
            msg.setIEEE80211EventID(event);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
      }
   }
} // void IEEE80211STAClient::onTechnologyError


/* net.connman.Service implementation required for Regulations */
int IEEE80211STAClient::subscribeIEEE80211STAConnectionStatus(IEEE80211STAConnectionStatusNotifier *listener)
{
   ::std::vector<IEEE80211STAConnectionStatusNotifier *>::iterator it;

   if (!listener)
      return -EINVAL;

   for (it = _ieee80211Stationobservers.begin();
         it < _ieee80211Stationobservers.end(); ++it)
      if (*it == listener)
         return -EALREADY;
   _ieee80211Stationobservers.push_back(listener);
   return 0;
}

int IEEE80211STAClient::unSubscribeIEEE80211STAConnectionStatus(IEEE80211STAConnectionStatusNotifier *listener)
{
   int avail = 0;
   ::std::vector<IEEE80211STAConnectionStatusNotifier *>::iterator it;

   if (!listener)
      return -EINVAL;

   for (it = _ieee80211Stationobservers.begin();
         it < _ieee80211Stationobservers.end(); ++it)
      if (*it == listener) {
         avail = 1;
         break;
      }

   if (!avail)
      return -ENOENT;

   _ieee80211Stationobservers.erase(it);
   return 0;
}

void IEEE80211STAClient::notifyConnectionStatus(const ::std::string &objpath, eIeee80211NetworkStatus_t status)
{
   IEEE80211Network network;
   std::vector<IEEE80211STAConnectionStatusNotifier*>::iterator it;

   if (objpath.empty())
      return;

   network.setObjectPath(objpath);

   LOG_INFO("Connection status change for the network: %s "
         "status: %s",
         network.getHidden() ? "Hidden" : network.getNetworkSsid(),
         IEEE80211STAConnectionStatusNotifier::IEEE80211StatusToString(status).c_str());

   for (it = _ieee80211Stationobservers.begin();
         it < _ieee80211Stationobservers.end(); ++it)
      (*it)->connectionStatusChange(network, status);
}

int IEEE80211STAClient::createConnManServiceProxy(const ::std::string &objpath)
{
   int iRet;
   cIConnManServiceProxyIf *connmanServiceProxy;

   if (objpath.empty())
      return -EINVAL;

   if (_busType == 2)
      return -ENOMEDIUM;

   LOG_INFO("Creating Proxy to the ConnMan Service Object: %s", objpath.c_str());

   connmanServiceProxy = DBusProxyFactory::getInstance()->getConnManServiceIf();
   iRet = connmanServiceProxy->createProxy(sConnManServicePort, sConnmanBusName,
         objpath, _busType, PROXY_TYPE_WIRED_AT_RUNTIME);
   if (iRet == 0) {
      iRet = connmanServiceProxy->setCallbackIf(sConnmanBusName, objpath, _busType, this);
      if (iRet < 0) {
         LOG_ERROR ("Failed to add the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
               objpath.c_str(), _busType, strerror(-iRet), -iRet);
      }
   } else {
      LOG_ERROR ("Failed to create proxy to the interface [%s] "
            "of object: [objPath: %s] [BusType: %d] error: %s/%d",
            sConnManServiceIfName.c_str(), objpath.c_str(),
            _busType, strerror(-iRet), -iRet);
      if (iRet == -EEXIST)
         iRet = 0;
   }

   return iRet;
}

int IEEE80211STAClient::destroyConnManServiceProxy(const ::std::string &objpath)
{
   int iRet;
   cIConnManServiceProxyIf *connmanServiceProxy;

   if (objpath.empty())
      return -EINVAL;

   if (_busType == 2)
      return -ENOMEDIUM;

   LOG_INFO("Destroying Proxy to the ConnMan Service Object: %s", objpath.c_str());

   connmanServiceProxy = DBusProxyFactory::getInstance()->getConnManServiceIf();
   iRet = connmanServiceProxy->destroyProxy(PROXY_DESTROY_ONE, sConnmanBusName, _busType, objpath);
   if (iRet == 0) {
      iRet = connmanServiceProxy->setCallbackIf(sConnmanBusName, objpath, _busType, nullptr);
      if (iRet < 0) {
         LOG_ERROR ("Failed to remove the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
               objpath.c_str(), _busType, strerror(-iRet), -iRet);
      }
   } else {
      LOG_ERROR ("Failed to destroy proxy to the interface [%s] "
            "of object: [objPath: %s] [BusType: %d] error: %s/%d",
            sConnManServiceIfName.c_str(), objpath.c_str(),
            _busType, strerror(-iRet), -iRet);
   }

   return iRet;
}

void IEEE80211STAClient::onDisconnectError(const ::boost::shared_ptr< cConnManServiceProxy >& proxy,
            const ::boost::shared_ptr< cConnManServiceDisconnectError >& error)
{
   LOG_ERROR ("Failed to Disconnect [proxy object: %s, interface: %s] [type: %d]: %s/%s",
         proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
         proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
   notifyConnectionStatus(proxy->getDBusObjectPath(), IEEE80211_NETWORK_DISCONNECT_FAILED);
}

void IEEE80211STAClient::onDisconnectResponse(const ::boost::shared_ptr< cConnManServiceProxy >& proxy,
            const ::boost::shared_ptr< cConnManServiceDisconnectResponse >& response)
{
   (void) response;

   LOG_INFO ("Disconnection Successful [proxy object: %s, interface: %s] [type: %d]",
         proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
         proxy->getBusType());

   notifyConnectionStatus(proxy->getDBusObjectPath(), IEEE80211_NETWORK_DISCONNECTED);
}

void IEEE80211STAClient::onConnManServiceServiceAvailable(const ::std::string &busName,
      const ::std::string &objPath, const ::DBusBusType busType,
      const ::asf::core::ServiceState previousState,
      const ::asf::core::ServiceState currentState)
{
   (void) previousState;
   (void) currentState;

   cIConnManServiceProxyIf *connmanServiceProxy;
   act_t token;

   LOG_INFO("Service Available for the Connman Service: %s [busType: %d] [busname: %s]",
         objPath.c_str(), busType, busName.c_str());

   connmanServiceProxy = DBusProxyFactory::getInstance()->getConnManServiceIf();
   token = connmanServiceProxy->sendDisconnectRequest(sConnmanBusName, objPath, busType, *this);
   if (DEFAULT_ACT == token)
      notifyConnectionStatus(objPath, IEEE80211_NETWORK_DISCONNECT_FAILED);
   else {
      notifyConnectionStatus(objPath, IEEE80211_NETWORK_DISCONNECTING);
   }
}

void IEEE80211STAClient::onConnManServiceServiceUnavailable(const ::std::string &busName,
      const ::std::string &objPath, const ::DBusBusType busType,
      const ::asf::core::ServiceState previousState,
      const ::asf::core::ServiceState currentState)
{
   (void) previousState;
   (void) currentState;

   LOG_INFO("Service UnAvailable for the Connman Service: %s [busType: %d] [busname: %s]",
         objPath.c_str(), busType, busName.c_str());
}

void IEEE80211STAClient::onPropertyChangedError(const ::boost::shared_ptr< cConnManServiceProxy >& proxy,
      const ::boost::shared_ptr< cConnManServicePropertyChangedError >& error)
{
   LOG_DEBUG("Property Changed error for Connman Service: %s [busType: %d] "
         "errorName: %s error Message: %s", proxy->getDBusObjectPath().c_str(),
         proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

void IEEE80211STAClient::onPropertyChangedSignal(const ::boost::shared_ptr< cConnManServiceProxy >& proxy,
      const ::boost::shared_ptr< cConnManServicePropertyChangedSignal >& signal)
{
   ::std::string name;
   const char *state = NULL;
   ::asf::dbus::DBusVariant value;

   name = signal->getName();
   LOG_DEBUG("Property Changed Signal for Connman Service: %s [busType: %d] [Changed Property: %s]",
         proxy->getDBusObjectPath().c_str(), proxy->getBusType(), name.c_str());

   if (name.compare("State"))
      return;

   /* Just for information */
   value = signal->getValue();
   dbus_message_iter_get_basic(value.getReadIterator(), &state);
   LOG_INFO("Changed state for the Network[%s]: %s", proxy->getDBusObjectPath().c_str(), state);
}

void IEEE80211STAClient::initLastModeSettings(::std::string objPath,const TechnologyProperty& property)
{
   LOG_INFO("IEEE80211STAClient::%s ", __func__);
   bool isPoweredSetting = false;

   IDBManager* dbmgrIf = DBManagerFactory::getInstance()->getDBManagerIf();
   if (dbmgrIf)
   {
      isPoweredSetting = dbmgrIf->readPoweredSetting(WIFI_MODE_STA, objPath);
      if ((isPoweredSetting) && (!property.isPowered) && LCM_WIFISTATE_NORMAL == getWifiState())
      {
         TechnologyChangedMsg msg;
         msg.setIEEE80211EventID(IEEE80211_MSG_STA_MODE_RESTORE_START);
         notifyObservers(IEEE80211_EVENT_STA, &msg);
         _restoringSTA = true;
      }
   }
}

void IEEE80211STAClient::updateLastModeSettings(const TechnologyProperty& property)
{
   LOG_INFO("IEEE80211STAClient::%s ", __func__);
   if(property.isPowered ||
         property.powerState == WIFI_STATE_POWERING_ON ||
         property.powerState == WIFI_STATE_POWER_FAILURE )
   {
      TechnologyChangedMsg msg;
      msg.setIEEE80211EventID(IEEE80211_MSG_STA_MODE_RESTORE_COMPLETE);
      notifyObservers(IEEE80211_EVENT_STA, &msg);
      _restoringSTA = false;
   }
}

void IEEE80211STAClient::updatePowerState(const ::std::string& objPath, const WBLPowerStateType type)
{
   LOG_INFO("IEEE80211STAClient::%s ", __func__);
   TechnologyProperty techProperty;
   TechnologyChangedMsg msg;
   TechnologyList::iterator it = _technologyList.begin();
   for (; it != _technologyList.end(); ++it)
   {
     if (objPath == it->objPath)
     {
        break;
     } // if (objPath == curTechItem.objPath)
   } // for (TechnologyList::iterator it =...)

   if (_technologyList.end() != it)
   {
      it->property.powerState = type;
      if(_restoringSTA)
         updateLastModeSettings(it->property);
      msg.setTechnologyItem(*it);
      msg.setIEEE80211EventID(IEEE80211_MSG_STA_MODE_POWERSTATE_IND);
      notifyObservers(IEEE80211_EVENT_STA, &msg);
   }
}

void IEEE80211STAClient::wifiConnectionStatusChanged(const WifiMode mode,const uint32 healthValue)
{
   LOG_INFO("IEEE80211STAClient::%s %s healthValue:%d", __func__,WIFI_MODE_TO_STR(mode),healthValue);
   if(mode != WIFI_MODE_STA)
      return;
   TechnologyChangedMsg msg;
   TechnologyList::iterator it = _technologyList.begin();
   for (; it != _technologyList.end(); ++it)
   {
     if (it->property.isConnected)
     {
        break;
     } // if (objPath == curTechItem.objPath)
   } // for (TechnologyList::iterator it =...)

   if (_technologyList.end() != it)
   {
      it->property.connHealth = healthValue;
      msg.setTechnologyItem(*it);
      msg.setIEEE80211EventID(IEEE80211_MSG_STA_MODE_CONNECTED_IND);
      notifyObservers(IEEE80211_EVENT_STA, &msg);
   }
}

} // namespace bosch
} // namespace org

/** @} */
