/**
 * @file IEEE80211STAClient.h
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */
#ifndef _IEEE_80211_STA_CLIENT_H
#define _IEEE_80211_STA_CLIENT_H

#include <vector>
#include <string>

#include "asf/core/Logger.h"

#include "IConnmanManagerProxyIf.h"
#include "IConnmanTechnologyProxyIf.h"
#include "IEEE80211STAClientIf.h"
#include "STAModeClient_Types.h"
#include "ConnmanDbusTypes.h"
#include "ConnmanServiceTypes.h"
#include "IConnmanServiceNotifCallbackIf.h"
#include "IEEE80211STAConnectionStatus.h"
#include "WifiConnectionStatus.h"

#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/wireless.h>
#include <errno.h>

namespace org 
{
namespace bosch 
{

using namespace ::asf::core;

typedef struct iw_freq iwfreq;

//TO DO:Move from here to WBLDefines.
#define MEGA   1e6
#define WBL_FREQ_2_4_LOWER 2412
#define WBL_FREQ_2_4_UPPER 2484
#define WBL_FREQ_5_LOWER 5170
#define WBL_FREQ_5_UPPER 5825

class IEEE80211STAConnectionStatusNotifier;

class IEEE80211STAClient final:
   public IEEE80211STAClientIf,
   public IConnmanManagerNotifCallbackIf,
   public IConnmanTechnologyNotifCallbackIf,
   public ConnmanGetTechnologiesCbIf,
   public ConnmanSetPropertyCbIf,
   public cIConnManServiceNotifCallbackIf,
   public cConnManServiceDisconnectCallbackIF,
   public cWifiConnectionStatusObserver
{
   public:

      IEEE80211STAClient();

      virtual ~IEEE80211STAClient();

      //! IEEE80211STAClientIf interfaces
      virtual bool setPowered(const ::std::string& objPath, bool powered) override;
      virtual void getTechnologies(TechnologyList& list) override;


      //! IConnmanManagerNotifCallbackIf interfaces
      virtual void onConnmanManagerServiceAvailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ServiceState previousState, const ServiceState currentState) override;
      virtual void onConnmanManagerServiceUnavailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ServiceState previousState, const ServiceState currentState) override;

      virtual void onTechnologyAddedError(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
            const ::boost::shared_ptr< ConnmanTechnologyAddedError >& error) override;
      virtual void onTechnologyAddedSignal(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
            const ::boost::shared_ptr< ConnmanTechnologyAddedSignal >& signal) override;
      virtual void onTechnologyRemovedError(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
            const ::boost::shared_ptr< ConnmanTechnologyRemovedError >& error) override;
      virtual void onTechnologyRemovedSignal(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
            const ::boost::shared_ptr< ConnmanTechnologyRemovedSignal >& signal) override;

      //! Connman Manager response callbacks
      virtual void onGetTechnologiesError(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
            const ::boost::shared_ptr< ConnmanGetTechnologiesError >& error) override;
      virtual void onGetTechnologiesResponse(const ::boost::shared_ptr< ConnmanManagerProxy >& proxy,
            const ::boost::shared_ptr< ConnmanGetTechnologiesResponse >& response) override;

      //! IConnmanTechnologyNotifCallbackIf interfaces
      virtual void onConnmanTechnologyServiceAvailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState) override;
      virtual void onConnmanTechnologyServiceUnavailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState) override;

      virtual void onPropertyChangedError(const ::boost::shared_ptr< ConnmanTechnologyProxy >& proxy,
            const ::boost::shared_ptr< ConnmanPropertyChangedError >& error) override;
      virtual void onPropertyChangedSignal(const ::boost::shared_ptr< ConnmanTechnologyProxy >& proxy,
            const ::boost::shared_ptr< ConnmanPropertyChangedSignal >& signal) override;

      //! Connman Technology response callbacks
      virtual void onSetPropertyError(const ::boost::shared_ptr< ConnmanTechnologyProxy >& proxy,
            const ::boost::shared_ptr< ConnmanSetPropertyError >& error) override;
      virtual void onSetPropertyResponse(const ::boost::shared_ptr< ConnmanTechnologyProxy >& proxy,
            const ::boost::shared_ptr< ConnmanSetPropertyResponse >& response) override;

      virtual void onGetPropertiesError(const ::boost::shared_ptr<ConnmanTechnologyProxy>& proxy,
                  const ::boost::shared_ptr<ConnmanGetPropertiesError>& error) override;
      virtual void onGetPropertiesResponse(const ::boost::shared_ptr<ConnmanTechnologyProxy>& proxy,
            const ::boost::shared_ptr<ConnmanGetPropertiesResponse>& response) override;

      //TODO - implement Technology properties Get callbacks if required

      /* ConnMan Service interfaces -> required by Regulatory core in order to
       * disconnect a network */
      virtual int subscribeIEEE80211STAConnectionStatus(IEEE80211STAConnectionStatusNotifier *listener) override;
      virtual int unSubscribeIEEE80211STAConnectionStatus(IEEE80211STAConnectionStatusNotifier *listener) override;
      virtual int createConnManServiceProxy(const ::std::string &objpath) override;
      virtual int destroyConnManServiceProxy(const ::std::string &objpath) override;
      virtual void onDisconnectError(const ::boost::shared_ptr< cConnManServiceProxy >& proxy,
            const ::boost::shared_ptr< cConnManServiceDisconnectError >& error) override;
      virtual void onDisconnectResponse(const ::boost::shared_ptr< cConnManServiceProxy >& proxy,
            const ::boost::shared_ptr< cConnManServiceDisconnectResponse >& response) override;
      virtual void onConnManServiceServiceAvailable(const ::std::string &busName,
            const ::std::string &objPath, const ::DBusBusType busType,
            const ::asf::core::ServiceState previousState,
            const ::asf::core::ServiceState currentState) override;
      virtual void onConnManServiceServiceUnavailable(const ::std::string &busName,
            const ::std::string &objPath, const ::DBusBusType busType,
            const ::asf::core::ServiceState previousState,
            const ::asf::core::ServiceState currentState) override;
      virtual void onPropertyChangedError(const ::boost::shared_ptr< cConnManServiceProxy >& proxy,
            const ::boost::shared_ptr< cConnManServicePropertyChangedError >& error) override;
      virtual void onPropertyChangedSignal(const ::boost::shared_ptr< cConnManServiceProxy >& proxy,
            const ::boost::shared_ptr< cConnManServicePropertyChangedSignal >& signal) override;

      //cWifiConnectionStatusNotifier updates `Value for the ConnectionStatus
      virtual void wifiConnectionStatusChanged(const WifiMode mode,const uint32 healthValue);

   protected:

   private:

      bool isTechnology(const ::std::string& objPath) const;
      bool getTechnologyInfo(const ::std::string& objPath, TechnologyItem& item);
      void addTechnology(const ::std::string& objPath, const TechnologyProperty& property);
      void updateTechnology(const ::std::string& objPath, const TechnologyProperty& property,
            const TechnologyPropertyName& changedPropertyName);
      void removeTechnology(const ::std::string& objPath);

      void addTechnologyProxy(const ::std::string& objPath);
      void removeTechnologyProxy(const ::std::string& objPath);

      bool readFrequency(uint32& frequency);
      double getFrequencyinMhz(iwfreq* in);
      uint16 getCurrOperatingChannel(uint32 freq);

      void parseTechnologyInfo(const ::std::string& busName, const ::std::string& objPath,
            const ::DBusBusType busType, const ::std::string& interfaceName,
            ::std::map< ::std::string, ::asf::dbus::DBusVariant >& data, TechnologyProperty& property) const;

      void onTechnologyError(const ::std::string& techObjPath, IEEE80211EventID event,
            IEEE80211ErrorCode error = IEEE80211_ERROR_UNKNOWN); //TODO error code

      void notifyConnectionStatus(const ::std::string &objpath, eIeee80211NetworkStatus_t status);

      void initLastModeSettings(::std::string objPath,const TechnologyProperty& property);
      void updateLastModeSettings(const TechnologyProperty& property);
      void updatePowerState(const ::std::string& objPath, const WBLPowerStateType type);
      void restoreLastModeSettings(const ::std::string& objPath,const TechnologyProperty& property);

      TechnologyList _technologyList;

      ::std::vector<IEEE80211STAConnectionStatusNotifier *> _ieee80211Stationobservers;
      ::DBusBusType _busType;

      bool _restoringSTA;

      DECLARE_CLASS_LOGGER();
};

} // namespace bosch
} // namespace org

#endif //_IEEE_80211_STA_CLIENT_H

/** @} */
