
/**
 * @file WifiConnectionStatus
 * @author RBEI/ECO3 Karthikeyan Madeswaran
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/epoll.h>
#include <errno.h>
#include <string.h>

#include <netlink/socket.h>
#include <netlink/genl/genl.h>
#include <netlink/genl/ctrl.h>

#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include "WBLUtils.h"
#include "WifiConnectionStatus.h"
#include "STAModeClient_Msgs.h"
#include "WBLASFComponent.h"

#define CONN_STATUS_CLASS_PATH        "wifi_business_logic/ConnectionStatus"
#define GENL_DRIVER_IFACE     "nl80211"
#define ETH_ALEN 6
#define GOOD 90
#define MEDIUM 80
#define LOW 70
#define CRITICAL 0

#define DEFAULT_BEACON_INTERVAL   100  //Most routers are automatically set to a default of 100 milliseconds
#define CONNECTION_STATUS_WATCH_DOG_TIMER_SEC   1000  //1 second default
#define FILE_PATH_BEACON_RECEIVED_COUNT "/proc/mwlan/wlan0/log"
#define FILE_PATH_AP "/proc/mwlan/uap0/log"


namespace org {
   namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConnectionStatus", cWifiConnectionWatchDog, Debug);
DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConnectionStatus", cWifiConnectionStatus, Debug);
DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConnectionStatus", cWifiHealthCalculator, Info);

static int m_sbeaconReceivedCount = 0;
static uint16_t m_sbeaconInterval = 0;
static int m_sdot11QosReceivedFragmentCount = 0;
static int m_sdot11FCSErrorCount = 0;


cWifiConnectionStatus::cWifiConnectionStatus():cNetLink(NL_SOCKET_TYPE_GENERIC)
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   m_iNl80211id = -1;
   m_ifIndex = 0;
   m_healthIndicatorValue = 0;
   m_sessionCount = 0;
   m_watchDog = new cWifiConnectionWatchDog();
   if(m_watchDog)
      m_watchDog->subscribe(this);
   (void) cWifiMlmeUpdates::getInstance()->registerForMlmeUpdates(this);
   m_bAPPower = false;
   m_currentMode = WIFI_MODE_UNKNOWN;
}

cWifiConnectionStatus::~cWifiConnectionStatus()
{
   try
   {
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   m_iNl80211id = -1;
   m_ifIndex = 0;
   m_healthIndicatorValue = 0;
   if(m_watchDog)
      delete m_watchDog;

   }catch(...){}
}

/**
 * 1) MLME Connected --> Upon the Connection detection through MLME
 * WBL Starts the timer to
 *    Receive Beacon Interval through NLSocket
 *    Receive Beacon Received count through NLSocket or
 *    Alternate way to read beaconReceivedCount value from proc/mlan/wlan0
 * Compute the HealthIndicator Value
 * */
void cWifiConnectionStatus::MlmeConnected(const ::std::string &ifName, const ::std::string &associatedbss,
      const unsigned int &ifIndex,const ::std::string &macaddr)
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   (void) ifName;
   (void) associatedbss;
   LOG_DEBUG("cWifiConnectionStatus::%s ifIndex:%d,macaddr:%s", __func__,ifIndex,macaddr.c_str());
   m_ifIndex = ifIndex;
   m_macaddr = macaddr;
   initiateNLSockets();
   initiateConnectionStatus(WIFI_MODE_STA);
}

void cWifiConnectionStatus::MlmeDisConnected(const ::std::string &ifName, const ::std::string &disconnectedBss)
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   (void) ifName;
   (void) disconnectedBss;
   closeConnectionStatus();
}

void cWifiConnectionStatus::onAsyncResult(const ::boost::shared_ptr< AsyncResult< uint32 > >& result)
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   LOG_INFO("cWifiConnectionStatus::HealthValue: %d",result->getData());
   uint32 healthVal = result->getData();
   updateSessionHealthStatus(healthVal);
}

void cWifiConnectionStatus::updateSessionHealthStatus(const uint32 healthVal)
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);

   if (healthVal <= m_healthIndicatorValue || (!m_healthIndicatorValue))
      m_healthIndicatorValue = healthVal;

   LOG_DEBUG("cWifiConnectionStatus::%s m_healthIndicatorValue:%d sessionCout:%d",
         __func__, m_healthIndicatorValue, m_sessionCount);

   if (0 == m_sessionCount % 15)
   {
      notifyHealthValue(m_healthIndicatorValue);
      m_sessionCount = 0;
      m_healthIndicatorValue = 0;
   }
}
cWifiConnectionWatchDog::cWifiConnectionWatchDog(): cWatchDog(eWatchDog_Type_Periodic)
{
   LOG_INFO("cWifiConnectionWatchDog::%s", __func__);
   unsigned int interval = 0;
   bool result = false;

   memset (&_startTime, 0, sizeof (struct timespec));

   interval = CONNECTION_STATUS_WATCH_DOG_TIMER_SEC;

   LOG_INFO("Loaded Time interval for ConnectionStatus watchdog: \"%u\"", interval);

   (void) setTimeInterval(interval);
   result = createTimer();

   LOG_INFO("%s timer with interval: \"%u\"", result == true ? "Successfully created" : "Failed to create",
         interval);
}

cWifiConnectionWatchDog::~cWifiConnectionWatchDog()
{
   try
   {
      LOG_INFO("cWifiConnectionWatchDog::%s", __func__);
      stop(false);
   }catch(...){}
}

int cWifiConnectionWatchDog::start ()
{
   LOG_INFO("cWifiConnectionWatchDog::%s", __func__);
   int iRet;

   iRet = startWatchDog();
   if (0 == iRet) {
      iRet = cUtils::getMonotonicTime(_startTime);
      if (0 == iRet) {
         LOG_INFO("Connection watch dog started at %lu.%06lu seconds",
               _startTime.tv_sec, _startTime.tv_nsec);
      }
   } else {
      LOG_ERROR("Failed to start the ConnectionStatus watch dog: %s/%d", strerror (-iRet), -iRet);
   }

   return iRet;
}

int cWifiConnectionWatchDog::stop(bool reset)
{
   LOG_INFO("cWifiConnectionWatchDog::%s", __func__);
   int iRet;
   (void) reset;

   iRet = stopWatchDog();
   if (0 == iRet) {
      iRet = cUtils::getMonotonicTime(_startTime);
      if (0 == iRet) {
         LOG_INFO("ConnectionStatus watch dog stopped at %lu.%06lu seconds",
               _startTime.tv_sec, _startTime.tv_nsec);
      }
   } else {
      LOG_ERROR("Failed to stop the ConnectionStatus watch dog: %s/%d", strerror (-iRet), -iRet);
   }
   return iRet;
}

void cWifiConnectionWatchDog::expired(::boost::shared_ptr< ::asf::core::TimerPayload > data)
{
   LOG_INFO("cWifiConnectionWatchDog::%s", __func__);
   (void) data;
   ::std::string cc("Time Out");
   notify(static_cast<void *>(&cc));
}

void cWifiConnectionStatus::initiateNLSockets()
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   int iRet;
   struct nl_sock *pNlSk;
   pNlSk = pGetNlSock();
   if (pNlSk)
   {
      iRet = iSetUpEventHandler();
      LOG_DEBUG("cWifiConnectionStatus::initiateNLSockets iSetUpEventHandler iRet:%d", iRet);
      if (!iRet)
      {
         iRet = genl_ctrl_resolve(pNlSk, GENL_DRIVER_IFACE);
         if (iRet < 0)
         {
            LOG_ERROR("Failed to Nl80211id : %s/%d", nl_geterror(-iRet), -iRet);
         }
         else
         {
            m_iNl80211id = iRet;
         }
         iRet = iGenlRequest(true, NL80211_CMD_GET_SCAN, NLM_F_DUMP, m_ifIndex, iProcessGenlScanEvent);
         if (iRet < 0)
         {
            LOG_ERROR("Failed to request the Scan:NL80211_CMD_GET_SCAN %s/%d", strerror(-iRet), -iRet);
         }
         /* Support from Kernel to receive RECEIVED_RX in not available
          * So we are using another approach of reading it from proc file "/proc/mwlan/wlan0/log".
          * iRet = iGenlRequest(true, NL80211_CMD_GET_STATION, NLM_F_DUMP,macaddr,iProcessGenlScanEvent);
          if (iRet < 0)
          {
          LOG_ERROR("Failed to request the STA information:NL80211_CMD_GET_STATION %s/%d", strerror(-iRet), -iRet);
          }*/
      }
   }
}

int cWifiConnectionStatus::iSetUpEventHandler()
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   struct nl_cb *cb;

   cb = pGetNlCB();
   if (!cb) {
      LOG_ERROR("Netlink callback is not created");
      return -EIO;
   }

   nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, iProcessGenlScanEvent, this);
   return 0;
}

int cWifiConnectionStatus::iProcessGenlScanEvent(struct nl_msg *msg, void *arg)
{
   int iRet = 0;
   struct genlmsghdr *gnlh;
   struct nlattr *tb[NL80211_ATTR_MAX + 1];
   ::asf::core::Logger oLogger(CONN_STATUS_CLASS_PATH, ::asf::core::Logger::Info);

   if (nlmsg_get_proto(msg) != NETLINK_GENERIC)
      return NL_SKIP;
   gnlh = static_cast<struct genlmsghdr *>(nlmsg_data(nlmsg_hdr(msg)));
   oLogger.debug("iProcessGenlScanEvent:: Generic Netlink Event received [event : %s/%d]", __LINE__,
   __FILE__, sConvertGenlEventToString(static_cast<enum nl80211_commands>(gnlh->cmd)).c_str(),
         static_cast<enum nl80211_commands>(gnlh->cmd));

   switch (gnlh->cmd)
   {
      case NL80211_CMD_GET_STATION:
      {
            nla_parse(tb, NL80211_ATTR_MAX, genlmsg_attrdata(gnlh, 0), genlmsg_attrlen(gnlh, 0), nullptr);
            iRet = iProcessStationInfoEvent(tb, arg);
                  if (iRet < 0)
                     oLogger.error("Failed to Get the Station Info: %s/%d",
                           __LINE__, __FILE__, strerror(-iRet), -iRet);
      }
      break;
      case NL80211_CMD_NEW_SCAN_RESULTS:
      {
            oLogger.debug("iProcessGenlScanEvent:NL80211_CMD_NEW_SCAN_RESULTS", __LINE__,__FILE__);
            nla_parse(tb, NL80211_ATTR_MAX, genlmsg_attrdata(gnlh, 0), genlmsg_attrlen(gnlh, 0), nullptr);
            iRet = iProcessScanInfoEvent(tb, arg);
                  if (iRet < 0)
                     oLogger.error("Failed to Get the Beacon Interval: %s/%d",
                           __LINE__, __FILE__, strerror(-iRet), -iRet);
      }
      break;
      default:
      oLogger.debug("Not interested with the event : %d", __LINE__,
            __FILE__, gnlh->cmd);
      break;

   }
   return iRet;
}
void cWifiConnectionStatus::closeNLSockets()
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
}

void cWifiConnectionStatus::watchDogExpired(void *data)
{
   (void) data;
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   m_sessionCount++;
   WBLASFComponent::getCurrentComponent()->asyncClass< uint32 >(*this,ASYNC_CLASS(cWifiHealthCalculator,m_currentMode));
}

const std::string cWifiConnectionStatus::sConvertGenlEventToString(enum nl80211_commands eGeNlEvt)
{
   switch (eGeNlEvt)
   {
      case NL80211_CMD_GET_STATION:
         return "NL80211_CMD_GET_STATION";
      case NL80211_CMD_GET_SCAN:
         return "NL80211_CMD_GET_SCAN";
      case NL80211_CMD_NEW_SCAN_RESULTS:
         return "NL80211_CMD_NEW_SCAN_RESULTS";
      default:
         break;
   }
   return "NL80211_CMD_UNKNOWN";
}

struct nl_sock *cWifiConnectionStatus::createNlSocket()
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   int iRet, realsk;
   struct nl_sock *sk;

   iRet = getSocketType();
   if (iRet < 0)
      return NULL;

   sk = nl_socket_alloc();
   if (!sk)
      return NULL;

   iRet = nl_connect(sk, getSocketType());
   if (iRet < 0)
   {
      LOG_ERROR("Failed to connect to the NL socket: %s/%d", nl_geterror(-iRet), -iRet);
      return NULL;
   }

   realsk = nl_socket_get_fd(sk);
   if (realsk != -1)
      (void) fcntl(realsk, F_SETFL, O_CLOEXEC);

   return sk;
}

int cWifiConnectionStatus::iGenlRequest(bool event_socket, enum nl80211_commands cmd,
      int flags,const unsigned int &ifindex,iEventhandler handler)
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   int iRet;
   struct nl_msg *msg;

   struct nl_sock *sk;

   LOG_INFO("Requesting NL command: %s, new socket: %s", sConvertGenlEventToString(cmd).c_str(),
         event_socket ? "Y" : "N");
   if (event_socket)
   {
      sk = pGetNlSock();
      if (!sk)
         return -ENOTCONN;
   }
   else
   {
      sk = createNlSocket();
      if (!sk)
         return -ENOMEM;
   }

   msg = nlmsg_alloc();
   if (!msg)
   {
      if (!event_socket)
         nl_socket_free(sk);
      return -ENOMEM;
   }
   if ((!genlmsg_put(msg, 0, 0, m_iNl80211id, 0, flags, cmd, 0))
         || (nla_put_u32(msg, NL80211_ATTR_IFINDEX, (uint32_t) ifindex) < 0))
   {
      if (!event_socket)
         nl_socket_free(sk);
      nlmsg_free(msg);
      return -ENOMSG;
   }

   iRet = iGenlSendRecv(sk, msg, handler, this);
   if (iRet < 0)
      LOG_ERROR("Failed to aquire the %d information, error: %s", cmd, nl_geterror(-iRet));

   if (!event_socket)
      nl_socket_free(sk);
   nlmsg_free(msg);
   return iRet;
}

int cWifiConnectionStatus::iProcessStationInfoEvent(struct nlattr **attr, void *arg)
{
   (void) attr;
   (void) arg;
   return 0;
}

int cWifiConnectionStatus::iProcessScanInfoEvent(struct nlattr **attr, void *arg)
{
   ::asf::core::Logger oLogger(CONN_STATUS_CLASS_PATH, ::asf::core::Logger::Info);
   oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent Entered", __LINE__,__FILE__);
   int iRet;
   struct nlattr *bss[NL80211_BSS_MAX + 1];
   static struct nla_policy bss_policy[NL80211_BSS_MAX + 1];
   uint8 bssid[ETH_ALEN];
   char macstring[ETH_ALEN * 3];

   cWifiConnectionStatus *pConnStatus = static_cast<cWifiConnectionStatus *>(arg);
   if (!pConnStatus)
   {
      oLogger.error("cWifiConnectionStatus::Invalid cWifiConnectionStatus instance", __LINE__, __FILE__);
      return -EINVAL;
   }

   bss_policy[NL80211_BSS_BSSID].type= NLA_UNSPEC;
   bss_policy[NL80211_BSS_FREQUENCY].type= NLA_U32;
   bss_policy[NL80211_BSS_TSF].type= NLA_U64;
   bss_policy[NL80211_BSS_BEACON_INTERVAL].type= NLA_U16;
   bss_policy[NL80211_BSS_CAPABILITY].type= NLA_U16;
   bss_policy[NL80211_BSS_INFORMATION_ELEMENTS].type= NLA_UNSPEC;
   bss_policy[NL80211_BSS_SIGNAL_MBM].type= NLA_U32;
   bss_policy[NL80211_BSS_SIGNAL_UNSPEC].type= NLA_U8;
   bss_policy[NL80211_BSS_STATUS].type= NLA_U32;
   bss_policy[NL80211_BSS_SEEN_MS_AGO].type= NLA_U32;
   bss_policy[NL80211_BSS_BEACON_IES].type= NLA_UNSPEC;

   if (!attr[NL80211_ATTR_BSS])
   {
      oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent NL_SKIP while checking ATTRIBS", __LINE__,__FILE__);
      return NL_SKIP;
   }

   iRet = nla_parse_nested(bss, NL80211_BSS_MAX, attr[NL80211_ATTR_BSS], bss_policy);
   if (iRet < 0)
   {
      oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent NL_SKIP while doing nested parse Error Code:%d", __LINE__,__FILE__,iRet);
      return NL_SKIP;
   }

   if (bss[NL80211_BSS_STATUS])
   {
      enum nl80211_bss_status status = static_cast<enum nl80211_bss_status> (nla_get_u32(bss[NL80211_BSS_STATUS]));
      if (bss[NL80211_BSS_BSSID])
      {

         memset(bssid, 0, ETH_ALEN);
         memcpy(bssid, nla_data(bss[NL80211_BSS_BSSID]),ETH_ALEN);
         oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent BSSID:" MACSTR,__LINE__,__FILE__,MAC2STR(bssid));
         cUtils::macToString(macstring,bssid);
         oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent Converted Mac String:%s",__LINE__,__FILE__,macstring);
         if(!pConnStatus->m_macaddr.compare(macstring))
         {
            oLogger.debug(
                  "cWifiConnectionStatus::iProcessScanInfoEvent Received Mac:%s is connected.Read Beacon Interval",
                  __LINE__, __FILE__, macstring);
            if (bss[NL80211_BSS_BEACON_INTERVAL])
            {
               m_sbeaconInterval = nla_get_u16(bss[NL80211_BSS_BEACON_INTERVAL]);
               oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent Beacon Interval: %d", __LINE__,__FILE__,m_sbeaconInterval);
            }
            else
            {
               oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent Failed to receive Beacon Interval", __LINE__,__FILE__);
            }
         }
         else
         {
            oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent Received Mac:%s is not connected. "
                           "So try another scan",__LINE__,__FILE__,macstring);
         }

      }
      else
      {
         oLogger.debug("cWifiConnectionStatus::iProcessScanInfoEvent BSSID NOT ASSOCIATED %d", __LINE__,__FILE__, status);
      }
   }
   return iRet;
}

int cWifiConnectionStatus::iGenlSendRecv(struct nl_sock *sk, struct nl_msg *msg,
      iEventhandler handler, void *data)
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   struct nl_cb *cb;
   int err = -ENOMEM, res = 0;

   if (!sk)
      return -EINVAL;

   if (!msg)
      return -ENODATA;

   cb = nl_cb_clone(pGetNlCB());
   if (!cb)
      return -ENOMEM;

   err = nl_send_auto(sk, msg);
   if (err < 0) {
      nl_cb_put(cb);
      return err;
   }

   nl_cb_err(cb, NL_CB_CUSTOM, (nl_recvmsg_err_cb_t) iGenlErrorHandler, &err);
   nl_cb_set(cb, NL_CB_FINISH, NL_CB_CUSTOM, iGenlFinishHandler, &err);
   nl_cb_set(cb, NL_CB_ACK, NL_CB_CUSTOM, iGenlAckHandler, &err);
   nl_cb_set(cb, NL_CB_SEQ_CHECK, NL_CB_CUSTOM, iGenlSeqCheckHandler, nullptr);

   if (handler)
      nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, handler, data);

   while (err > 0) {
      res = nl_recvmsgs(sk, cb);
      if (res < 0)
         LOG_ERROR("Failed to receive nl80211 msgs : %s [%d]",
               nl_geterror(-res), -res);
   }

   nl_cb_put(cb);
   return res;
}

int cWifiConnectionStatus::iGenlErrorHandler(struct sockaddr_nl *nla,
      struct nlmsgerr *err, void *arg)
{
   (void) nla;
   int *iRet = static_cast<int *>(arg);
   ::asf::core::Logger oLogger(CONN_STATUS_CLASS_PATH, ::asf::core::Logger::Info);

   if (!iRet || !err)
      return NL_STOP;

   *iRet = err->error;
   oLogger.error(
         "Error received for Nl message with sequence no [%d] : %s/%d system error : %s/%d",
         __LINE__, __FILE__, err->msg.nlmsg_seq, nl_geterror(err->error),
         err->error, strerror(-err->error), -err->error);
   return NL_STOP;
}

int cWifiConnectionStatus::iGenlFinishHandler(struct nl_msg *msg, void *arg)
{
   struct nlmsghdr *nlhdr;
   int *iRet = static_cast<int *>(arg);
   ::asf::core::Logger oLogger(CONN_STATUS_CLASS_PATH, ::asf::core::Logger::Info);

   if (!msg || !iRet)
      return NL_SKIP;

   *iRet = 0;
   nlhdr = nlmsg_hdr(msg);
   oLogger.debug("Message with seq no [%d] finished successfully", __LINE__,
         __FILE__, nlhdr->nlmsg_seq);
   return NL_SKIP;
}

int cWifiConnectionStatus::iGenlAckHandler(struct nl_msg *msg, void *arg)
{
   struct nlmsghdr *nlhdr;
   int *iRet = static_cast<int *>(arg);
   ::asf::core::Logger oLogger(CONN_STATUS_CLASS_PATH, ::asf::core::Logger::Info);

   if (!msg || !iRet)
      return NL_STOP;

   *iRet = 0;
   nlhdr = nlmsg_hdr(msg);
   oLogger.debug("Message with seq no [%d] acknowledged", __LINE__,
         __FILE__, nlhdr->nlmsg_seq);
   return NL_STOP;
}

int cWifiConnectionStatus::iGenlSeqCheckHandler(struct nl_msg *msg, void *arg)
{
   (void) arg;
   struct nlmsghdr *nlhdr;
   ::asf::core::Logger oLogger(CONN_STATUS_CLASS_PATH, ::asf::core::Logger::Info);

   if (!msg)
      return NL_OK;

   nlhdr = nlmsg_hdr(msg);
   oLogger.debug("Sequence check for Message with seq no [%d]", __LINE__,
         __FILE__, nlhdr->nlmsg_seq);
   return NL_OK;
}

cWifiHealthCalculator::cWifiHealthCalculator(WifiMode mode)
{
   LOG_INFO("cWifiHealthCalculator::%s for Mode:%s", __func__,
         wblWifiMode.getWifiMode2String(mode).c_str());
   m_healthValue = 0;
   m_mode = mode;
}

cWifiHealthCalculator::~cWifiHealthCalculator()
{
   try
   {
      LOG_INFO("cWifiHealthCalculator::%s", __func__);
      m_healthValue = 0;
   }catch(...){}
}

void cWifiHealthCalculator::run()
{
   LOG_INFO("cWifiHealthCalculator::%s", __func__);
   if(WIFI_MODE_STA == m_mode )
      computeConnectionStatusSTA();
   else if(WIFI_MODE_AP1 == m_mode)
      computeConnectionStatusAP();
   else
      LOG_DEBUG("cWifiHealthCalculator::%s Invalid Mode", __func__);
   setResult(m_healthValue);
}

int cWifiHealthCalculator::readFromProcFile(const char* fileName,const char* readParameter)
{
   LOG_INFO("cWifiHealthCalculator::%s", __func__);
   char readStr[4096];
   ::std::vector<::std::string> splitStrings;
   int count = 0;
   try
   {
      FILE *fp = fopen(fileName,"r");
      if(!fp)
      {
         LOG_ERROR("cWifiHealthCalculator::%s Failed to open:%s Error:%s ",__func__,fileName,strerror(errno));
         return 0;
      }
      while(fgets(readStr,sizeof readStr,fp)!=NULL)
      {
        if (strstr(readStr,readParameter) != NULL)
        {
          boost::split(splitStrings,readStr,boost::is_any_of("="));
          count = atoi(splitStrings.back().c_str());
          LOG_DEBUG("cWifiHealthCalculator::%s Read from %s %s=%d",__func__,fileName,readParameter,count);
          break;
        }
      }
      fclose(fp);
   }catch(...){}
   return count;
}

/**
 * The computation of Healthiness Indicator value for STA performed
 * based on the guidelines mentioned in "TechNote_BT_Paging_Wifi_CoExistence_improvements.docx"
 * by Mr. Musielak Joerg.
 *
 * From the Proc file ("/proc/mwlan/wlan0/log") the "beaconReceivedCount" parameter is received and
 * the healthValue is calculated based on the the beacounReceivedCount on time 't'
 * with the expectedbeacons on time 't'
 *
 */

void cWifiHealthCalculator::computeConnectionStatusSTA()
{
   LOG_INFO("cWifiHealthCalculator::%s", __func__);
   try
   {
      uint32 configTimer = CONNECTION_STATUS_WATCH_DOG_TIMER_SEC;
      uint32 expectedBeacons;
      uint32 currentCount = m_sbeaconReceivedCount;
      m_sbeaconReceivedCount = readFromProcFile(FILE_PATH_BEACON_RECEIVED_COUNT,"beaconReceivedCount");
      if (m_sbeaconReceivedCount > 0 && m_sbeaconInterval > 0)
      {
         expectedBeacons = (configTimer/(m_sbeaconInterval));
         LOG_DEBUG("cWifiHealthCalculator::computeConnectionStatus expectedBeacons:%d", expectedBeacons);
         currentCount = m_sbeaconReceivedCount - currentCount;
         LOG_DEBUG("cWifiHealthCalculator::computeConnectionStatus current beacon count:%d", currentCount);
         m_healthValue = (currentCount*100)/expectedBeacons;
         if(m_healthValue > 100)
            m_healthValue=100;
         LOG_DEBUG("cWifiHealthCalculator::computeConnectionStatus m_healthValue:%d", m_healthValue);
      }
   }catch(...){}
}

/**
 * The computation of Healthiness Indicator value for AP performed
 * based on the guidelines mentioned in "TechNote_BT_Paging_Wifi_CoExistence_improvements.docx"
 * by Mr. Musielak Joerg.
 *
 * From the Proc file ("/proc/mwlan/uap0/log") the "dot11QosReceivedFragmentCount" and
 * "dot11FCSErrorCount" parametes are received and the healthValue is calculated based on the
 * computed fcsRatio. This computation would happen at every time interval 't'
 *
 */

void cWifiHealthCalculator::computeConnectionStatusAP()
{
   LOG_INFO("cWifiHealthCalculator::%s", __func__);
   try
   {
      m_healthValue = 0;
      double deltaQos = m_sdot11QosReceivedFragmentCount;
      double deltaFCS = m_sdot11FCSErrorCount;
      uint32 val = LOW;
      double fcsRatio = 0;
      m_sdot11QosReceivedFragmentCount = readFromProcFile(FILE_PATH_AP,"dot11QosReceivedFragmentCount");
      m_sdot11FCSErrorCount = readFromProcFile(FILE_PATH_AP,"dot11FCSErrorCount");
      deltaQos = m_sdot11QosReceivedFragmentCount - deltaQos;
      deltaFCS = m_sdot11FCSErrorCount - deltaFCS;
      if(deltaQos > 0)
         fcsRatio = deltaFCS / deltaQos;

      if(deltaQos > 10)
      {
         if(fcsRatio <= 0.5)
            val = GOOD;
         else if(fcsRatio <= 1)
            val = MEDIUM;
         else if(fcsRatio <= 2)
            val = LOW;
         else
            val = CRITICAL;
      }
      else if(fcsRatio > 50)
      {
         if(fcsRatio <= 40)
            val = GOOD;
         else if(fcsRatio <= 80)
            val = MEDIUM;
         else if(fcsRatio <= 120)
            val = LOW;
         else
            val = CRITICAL;
      }
      else
      {
         val = GOOD;
      }
      m_healthValue = val;
      LOG_DEBUG("cWifiHealthCalculator::%s deltaQos:%f ,deltaFCS:%f ,fcsRatio:%f,Health:%d",__func__,deltaQos,deltaFCS,fcsRatio,m_healthValue);
   }catch(...){}
}

void cWifiHealthCalculator::interrupt()
{
   LOG_INFO("cWifiHealthCalculator::%s", __func__);
}

void cWifiConnectionStatus::setDefaultValues()
{
   m_sbeaconReceivedCount = 0;
   m_sbeaconInterval = DEFAULT_BEACON_INTERVAL;
   m_healthIndicatorValue = 0;
   m_sdot11QosReceivedFragmentCount = 0;
   m_sdot11FCSErrorCount = 0;
   m_sessionCount = 0;
   m_currentMode = WIFI_MODE_UNKNOWN;
}

void cWifiConnectionStatus::notifyHealthValue(const uint32 healthValue)
{
   LOG_INFO("cWifiConnectionStatus::%s", __func__);
   updateConnectionStatus(m_currentMode,healthValue);
}

void cWifiConnectionStatusNotifier::updateConnectionStatus(const WifiMode mode,const uint32 healthValue)
{
   ::std::vector<cWifiConnectionStatusObserver *>::iterator it;
   for (it = _connStatusObservers.begin();
         it < _connStatusObservers.end(); ++it)
      (*it)->wifiConnectionStatusChanged(mode,healthValue);
}

int cWifiConnectionStatusNotifier::subscribeWifiConnectionStatus(cWifiConnectionStatusObserver *observer)
{
   ::std::vector<cWifiConnectionStatusObserver *>::iterator it;

   if (!observer)
      return -EINVAL;

   for (it = _connStatusObservers.begin();
         it < _connStatusObservers.end(); ++it)
      if (*it == observer)
         return -EALREADY;
   _connStatusObservers.push_back(observer);
   return 0;
}

int cWifiConnectionStatusNotifier::unSubscribeWifiConnectionStatus(cWifiConnectionStatusObserver *observer)
{
   int avail = 0;
   ::std::vector<cWifiConnectionStatusObserver *>::iterator it;

   if (!observer)
      return -EINVAL;

   for (it = _connStatusObservers.begin();
         it < _connStatusObservers.end(); ++it)
      if (*it == observer) {
         avail = 1;
         break;
      }

   if (!avail)
      return -ENOENT;

   _connStatusObservers.erase(it);
   return 0;
}

int cWifiConnectionStatus::iStartWatch()
{
   return 0;
}

void cWifiConnectionStatus::notifyActiveWiFiSetups(WifiSetUpList wifiSetups)
{
   LOG_INFO("cWifiConnectionStatus:%s", __func__);

   /*On notification of ActiveWifi Setups ConnectionStatus should computed
    * if Mode is AP in 2.4GH/STA is Connected and still connection status not updated
    * Computation would be initiated by setting m_currentMode and starting the timer*/

   for (auto iter = wifiSetups.begin(); iter != wifiSetups.end(); ++iter)
   {
      if (iter->property.mode == WIFI_MODE_AP1
            && iter->property.frequency == WBL_FREQ_BAND_2_4_GHZ)
      {
         LOG_DEBUG("cWifiConnectionStatus:%s Previous Power:%s , Current Power:%s",
               __func__, BOOL_TO_STR(m_bAPPower),
               BOOL_TO_STR(iter->property.isPowered));
         if (m_bAPPower != iter->property.isPowered)
         {
            m_bAPPower = iter->property.isPowered;
            if (m_bAPPower)
            {
               initiateConnectionStatus(iter->property.mode);
            }
            else
            {
               closeConnectionStatus();
            }
         }
      }
      else if (iter->property.mode == WIFI_MODE_STA)
      {
         if (iter->property.isConnected && m_currentMode != WIFI_MODE_STA)
         {
            initiateConnectionStatus(iter->property.mode);
         }
         else if(!(iter->property.isConnected) && m_currentMode == WIFI_MODE_STA)
         {
            closeConnectionStatus();
         }

      }
   }
}

void cWifiConnectionStatus::initiateConnectionStatus(const WifiMode mode)
{
   LOG_INFO("cWifiConnectionStatus:%s", __func__);
   setDefaultValues();
   m_currentMode = mode;
   if(m_watchDog)
      m_watchDog->start();
   LOG_DEBUG("cWifiConnectionStatus:%s: %s timer Watchdog started successfully",
            __func__,WIFI_MODE_TO_STR(m_currentMode));
}

void cWifiConnectionStatus::closeConnectionStatus()
{
   LOG_INFO("cWifiConnectionStatus:%s", __func__);
   if(m_watchDog)
      m_watchDog->stop(true);
   setDefaultValues();
}

}
}
/** @} */
