/**
 * @file CryptoSDC.cpp
 * @author RBEI/ECO3 Murali Kakkanavar
 * @copyright (c) 2020 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief encrypt/decrypt persistent data for WBL
 * Currently used to handle LastIntendedMode
 * @{
 */

#include "CryptoSDC.h"
#include "DBDefines.h"
#include "WBLHelper.h"
#include "WBLTypeProperties.h"
#include <string>
#include <unistd.h>
#include <errno.h>
#include <sdc_op_conv.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <glib.h>

namespace org
{
namespace bosch
{
DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Database", CryptoSDC, Debug);

CryptoSDC::CryptoSDC()
{
    LOG_INFO("CryptoSDC::%s", __func__);
    sdcKeyId = SDC_WBL_KEY_ID;
    poSdcSession = NULL;
    sdcType = NULL;

} //CryptoSDC::CryptoSDC

CryptoSDC::~CryptoSDC()
{
    LOG_INFO("CryptoSDC::%s", __func__);
    sdcKeyId = 0;
    poSdcSession = NULL;
    sdcType = NULL;
} //CryptoSDC::~CryptoSDC

bool CryptoSDC::checkSDCErrorCode ( sdc_error_t ret )
{
    bool failed = false;
    LOG_INFO("CryptoSDC::%s, ret - %d", __func__,ret);
    if ( ret != SDC_OK ) {
        failed = true;
    }

    return ( failed );
} //CryptoSDC::checkSDCErrorCode

// ==================== Key-Id availability check: BGN =========================
/* The code will check if an Key-ID/AES-Key pair is available in the
   SDC (Secure Data Container) or try to create it.
   If an AES Key has to be created (typically only in a virgin startup scenario),
   we do so and if this was successful we will delete an already existing
   database or journal file to avoid any inconsistencies.*/
bool CryptoSDC::cryptoUtilRegister()
{
    LOG_INFO("CryptoSDC::%s", __func__);
    sdc_error_t sdcRes = SDC_OK;
    sdc_permissions_t *poPermissions = NULL;
    bool failed = false;

    IF_NOT_FAILED {
        /*As first step we open an SDC session*/
        LOG_INFO("CryptoSDC::%s - sdc_open_session ",__func__);
        sdcRes = sdc_open_session(&poSdcSession);
        failed = checkSDCErrorCode( sdcRes );
    }

    IF_NOT_FAILED {
        /*Allocated default architecture type*/
        LOG_INFO("CryptoSDC::%s - sdc_wrap_unwrap_type_alloc", __func__);
        sdcRes = sdc_wrap_unwrap_type_alloc(&sdcType);
        failed = checkSDCErrorCode( sdcRes );
    }

    IF_NOT_FAILED {
        /*Set required crypto algorithm*/
        LOG_INFO("CryptoSDC::%s - sdc_wrap_unwrap_type_set_algorithm",__func__);
        sdcRes= sdc_wrap_unwrap_type_set_algorithm(
                sdcType, SDC_WRAP_ALG_AES );
        failed = checkSDCErrorCode( sdcRes );
    }

    IF_NOT_FAILED {
        /*To set a block mode operations*/
        LOG_INFO("CryptoSDC::%s - sdc_wrap_unwrap_type_set_block_mode",__func__);
        sdcRes = sdc_wrap_unwrap_type_set_block_mode(
                sdcType, SDC_WRAP_BLK_CCM );
        failed = checkSDCErrorCode( sdcRes );
    }

    /*Probe the key store, if the WBL related <Key-ID, AES-Key> is already
      available/known by the SDC.*/
    IF_NOT_FAILED {
        if(poSdcSession) {
            LOG_INFO("CryptoSDC::%s - sdc_session_load_storage_key, %d",
                    __func__, sdcKeyId);
            sdcRes = sdc_session_load_storage_key(poSdcSession, sdcKeyId);
            failed = checkSDCErrorCode(sdcRes);
        }
    }

    /*
     The WBL Key-ID is not known by the SDC key store, so we seem to be in a
     "virgin" startup of the device.
     We will try to create the needed <Key-ID, AES-Key> "pair" in the SDC
     keystore for the WBL files
     */
    IF_FAILED {
        if(poSdcSession) {
            LOG_INFO("CryptoSDC::%s - sdc_permissions_alloc, %d",
                    __func__, sdcKeyId);
            /*Create a new object of SDC permissions. This has to be freed later*/
            sdcRes = sdc_permissions_alloc(&poPermissions);
            failed = checkSDCErrorCode(sdcRes);
        }
        IF_NOT_FAILED {
            if(poPermissions) {
                LOG_INFO("CryptoSDC::%s - sdc_set_default_permissions_current_gid, %d",
                        __func__, sdcKeyId);
                /*Get default permissions for generate the key*/
                sdcRes = sdc_set_default_permissions_current_gid(poPermissions);
                failed = checkSDCErrorCode(sdcRes);
            }
        }

        IF_NOT_FAILED {
            LOG_INFO("CryptoSDC::%s - sdc_generate_random_storage_key, %d",
                    __func__, sdcKeyId);
            /*Here we create the <Key-ID, AES-Key> "pair" for the WBL files*/
            sdcRes = sdc_generate_random_storage_key(&sdcKeyId, SDC_CREATE_KEY_FIXED_ID,
                    SKEY_LENGTH_BYTES, SDC_PERSISTENT_STORAGE_KEY, poPermissions);
            failed = checkSDCErrorCode(sdcRes);
        }

        IF_NOT_FAILED {
            LOG_INFO("CryptoSDC::%s - sdc_session_load_storage_key, %d",
                    __func__, sdcKeyId);
            /*Here we check a 2nd time, if the WBL related <Key-ID, AES-Key> is
             now available / known by the SDC.*/
            sdcRes = sdc_session_load_storage_key(poSdcSession, sdcKeyId);
            failed = checkSDCErrorCode(sdcRes);
        }

        IF_NOT_FAILED {
            /*
             When we come here we have created a new pair <Key-ID, AES-Key>
             successfully, but this means that any current database will not be
             accessible with any old <Key-ID, AES-Key> information. Therefore we
             will remove the current database.
             Remove the current database, if it exists
             */

            LOG_INFO("CryptoSDC::%s Removing [%s] since not accessible with any "
                    "old <Key-ID, AES-Key>,", __func__, WBL_LIM_FILE_PATH);
            int iRes = 0;  // Temporary result variable

            iRes = unlink(WBL_LIM_FILE_PATH);
            if(iRes < 0)
                LOG_ERROR("CryptoSDC::%s Failed to Remove file [%s]",
                        __func__, WBL_LIM_FILE_PATH);

        }
    }

    /*Free allocated permissions*/
    if (poPermissions)
        sdc_permissions_free(poPermissions);

    return failed;
}

bool CryptoSDC::cryptoUtilUnregister()
{
    /*TBD: This implementation to be checked, infact we need to keep the session*/

    /*Close the open SDC session*/
    LOG_INFO("CryptoSDC:: %s", __func__);
    sdc_error_t sdcRes = SDC_OK;
    bool failed = false;
    if(poSdcSession) {
        sdcRes = sdc_close_session(poSdcSession);
        failed = checkSDCErrorCode(sdcRes);
    }

    LOG_INFO("CryptoSDC::%s - %s", __func__, failed ? "failed" : "success");
    return failed;
}

bool CryptoSDC::cryptoUtilEncrypt(const char *filepathEnc, char* indata,
        size_t length)
{
    bool failed = false;
    uint8_t *bsRaw = NULL,*bsEnc = NULL;
    size_t bsRawSize = 0,bsEncSize = 0;

    FILE *ofp = NULL;
    sdc_error_t sdcRes = SDC_OK;

    /* sanity checks */
    if ( !poSdcSession || !filepathEnc || !indata || (length == 0)) {
        LOG_ERROR ("CryptoSDC::%s - Invalid parameters!\n", __func__);
        return (failed = true);
    }

    bsRaw = (uint8_t*)g_strdup(indata);
    bsRawSize = length;
    /* Open out file in Write mode */
    ofp = fopen (filepathEnc, "w");
    if (!ofp) {
        LOG_ERROR ("CryptoSDC::%s - Failed to open the file %s, error : %s",
                __func__,filepathEnc, strerror(errno));
        return (failed = true);
    }

    /* Encrypt using SDC API */
    IF_NOT_FAILED {
        sdcRes = sdc_wrap_formatted( poSdcSession, sdcType, bsRaw, bsRawSize, &bsEnc,
                &bsEncSize );
        failed = checkSDCErrorCode( sdcRes );
    }

    /*Write encrypted into out file*/
    IF_NOT_FAILED {
        if ( bsEncSize != fwrite( bsEnc, sizeof(uint8_t), bsEncSize, ofp ) ) {
            failed = true;
        } else {
            sync();
        }
    }

    /* Free local buffer */
    if( bsRaw ) {
        free( bsRaw );
        bsRaw = NULL;
    }

    if ( bsEnc ) {
        free ( bsEnc );
        bsEnc = NULL;
    }

    if (fclose (ofp) < 0) {
        LOG_ERROR ("CryptoSDC::%s - Failed to close the file : %s [%s]",
                __func__,filepathEnc, strerror (errno));
        return (failed = true);
    }

    return failed;
}

bool CryptoSDC::cryptoUtilDecrypt(const char *filepathEnc,
        const char *filepathPlain)
{
    bool failed = false;
    uint8_t *bsRaw = NULL,*bsEnc = NULL;
    size_t bsRawSize = 0,bsEncSize = 0;
    int64_t size = 0;
    FILE *ifp = NULL,*ofp = NULL;

    sdc_error_t sdcRes = SDC_OK;

    /* sanity checks */
    if ( !poSdcSession || !filepathPlain || !filepathEnc ) {
        LOG_ERROR ("CryptoSDC::%s -Invalid parameters!\n", __func__);
        return (failed = true);
    }

    /*Open file to be decrypted in 'Read' mode*/
    ifp = fopen (filepathEnc, "r");
    if (!ifp) {
        LOG_ERROR ("CryptoSDC::%s - Failed to open the file \"%s\", error : %s",
                __func__,filepathEnc, strerror (errno));
        return (failed = true);
    }

    /*Open file to write decrypted data*/
    ofp = fopen (filepathPlain, "w+");
    if (!ofp) {
        LOG_ERROR ("CryptoSDC::%s - Failed to open the file \"%s\", error : %s",
                __func__,filepathPlain, strerror(errno));
        fclose (ifp);
        return (failed = true);
    }

    /* get encrypted File Size */
    IF_NOT_FAILED {
        if ( fseek ( ifp, 0 , SEEK_END ) != 0 ) {
            LOG_ERROR ("CryptoSDC::%s - Failed to read file size\n",__func__);
            failed = true;
        } else {
            size = ftell(ifp);
            if ( size < 0 ) {
                failed = true;
            } else {
                bsEncSize = size;
            }
            rewind ( ifp );
        }
        failed |= (bsEncSize > 0) ? false : true;
    }

    /* allocate Buffer to store the extracted data*/
    IF_NOT_FAILED{
        bsEnc = (uint8_t*) malloc (bsEncSize);
        failed = (bsEnc == NULL) ? true : false;
    }

    IF_NOT_FAILED {
        memset (bsEnc, 0, bsEncSize);
        /* read the encrypted File */
        size_t ret = 0;
        ret = fread ( bsEnc, 1, bsEncSize, ifp );
        if ( ret != bsEncSize ) {
            LOG_ERROR (" CryptoSDC::%s - reading file failed\n", __func__);
            free (bsEnc);
            bsEnc = NULL;
            bsEncSize = 0;
            failed = true;
        }
    }

    LOG_INFO ("CryptoSDC::%s Encrypted Data size = [%d]",__func__, bsEncSize);

    /* Decrypt using SDC API */
    IF_NOT_FAILED {
        sdcRes = sdc_unwrap_formatted( poSdcSession, sdcType, bsEnc, bsEncSize,
                &bsRaw, &bsRawSize );
        failed = checkSDCErrorCode( sdcRes );
    }

    /* Write decrypted data into output file */
    IF_NOT_FAILED {
        LOG_INFO ("CryptoSDC::%s Decrypted Data - [%s], size = [%d]",
                __func__, bsRaw, bsRawSize);
        if ( bsRawSize != fwrite( bsRaw, 1, bsRawSize, ofp ) ) {
            failed = true;
        } else {
            sync();
        }
    }

    /* Free local buffer */
    if( bsRaw ) {
        free( bsRaw );
        bsRaw = NULL;
    }

    if ( bsEnc ) {
        free ( bsEnc );
        bsEnc = NULL;
    }

    /* Close files using its file handlers */
    if (fclose (ifp) < 0) {
        LOG_ERROR (" CryptoSDC::%s Failed to close input file : %s [%s]",
                __func__,filepathPlain, strerror (errno));
        fclose(ofp);
        return (failed = true);
    }

    if (fclose (ofp) < 0) {
        LOG_ERROR (" CryptoSDC::%s Failed to output the file : %s [%s]",
                __func__,filepathEnc, strerror (errno));
        return (failed = true);
    }

    return failed;
}


}//namespace bosch
}//namespace org
