/**
 * @file SetupIdleState.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "WBLDefines.h"
#include "StateMachineProperties.h"
#include "IWifiSetupSM.h"
#include "SetupIdleState.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", SetupIdleState, Info);

SetupIdleState::SetupIdleState(IWifiSetupSM* wifiSetupSMIf):
      WifiSetupState(ST_SETUP_IDLE, wifiSetupSMIf)
{
   LOG_INFO("[WifiMode=%d] SetupIdleState::Constructor entered", _wifiMode);
}

SetupIdleState::~SetupIdleState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   //LOG_INFO("[WifiMode=%d] SetupIdleState::Destructor entered", _wifiMode);
}

void SetupIdleState::onEntry(::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupIdleState::onEntry()", _wifiMode);
   WBL_INTENTIONALLY_UNUSED(msg);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   //! If wifi setup interface is unavailable, switch to SetupDisabled state
   const WifiSetUpItem& wifiSetupItem = _wifiSetupSMIf->getWifiSetupInfo();
   if (wifiSetupItem.property.interface.empty())
   {
      _wifiSetupSMIf->setState(ST_SETUP_DISABLED, nullptr);
   }
   //! If wifi setup interface is available, proceed with any ongoing requests
   else
   {
      if (_wifiSetupSMIf->getPrepareSetupMsg())
      {
         _wifiSetupSMIf->setState(ST_PREPARING_SETUP, nullptr);
      }
      else if (_wifiSetupSMIf->getDeactivateSetupMsg())
      {
         _wifiSetupSMIf->setState(ST_DEACTIVATING_SETUP, nullptr);
      }
      else
      {
         LOG_DEBUG("onEntry: No ongoing requests to be processed");
      }
   }
}

void SetupIdleState::onRequest(::boost::shared_ptr< PrepareSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupIdleState::onRequest(PrepareSetup) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   _wifiSetupSMIf->setPrepareSetupMsg(msg);
   _wifiSetupSMIf->setState(ST_PREPARING_SETUP, nullptr);
}

void SetupIdleState::onRequest(::boost::shared_ptr< DeActivateSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupIdleState::onRequest(DeactivateSetup) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   _wifiSetupSMIf->setDeactivateSetupMsg(msg);
   _wifiSetupSMIf->setState(ST_DEACTIVATING_SETUP, nullptr);
}

void SetupIdleState::onRequest(::boost::shared_ptr< DisableSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupIdleState::onRequest(DisableSetup) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   _wifiSetupSMIf->setDisableSetupMsg(msg);
   _wifiSetupSMIf->setState(ST_DISABLING_SETUP, nullptr);
}

void SetupIdleState::onRequest(::boost::shared_ptr< RestoreFactorySettingsMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupIdleState::onRequest(FactoryResetMsg) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   _wifiSetupSMIf->setRestoreFactorySettingsMsg(msg);
   _wifiSetupSMIf->setState(ST_RESETTING_SETUP, nullptr);
}

void SetupIdleState::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode=%d] SetupIdleState::onEvent() entered: EventID = %s, isError = %d",
         _wifiMode, SM_EVENT_TO_STR(eventID), isError);

   if (IS_CURRENT_MODE(setup.property.mode))
   {
      switch (eventID)
      {
         case WIFI_SETUP_EVENT_STA_TECH_REMOVED:
         case WIFI_SETUP_EVENT_AP_REMOVED:
         {
            //! If wifi setup interface is removed, switch to SetupDisabled state
            if (!isError)
            {
               LOG_DEBUG("onEvent: Interface removed. Switching to SetupDisabled state.");
               WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);
               _wifiSetupSMIf->setState(ST_SETUP_DISABLED, nullptr);
            }
         }
         break;
         default:
         {
            //! Forward unhandled events to base class event handler
            WifiSetupState::onEvent(eventID, setup, isError);
         }
         break;
      }
   } // if (IS_CURRENT_MODE(setup.property.mode))
}

} // namespace bosch
} // namespace org

/** @} */
