/**
 * @file SetupDisabledState.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "WBLDefines.h"
#include "StateMachineProperties.h"
#include "IWifiSetupSM.h"
#include "SetupDisabledState.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", SetupDisabledState, Info);

SetupDisabledState::SetupDisabledState(IWifiSetupSM* wifiSetupSMIf):
      WifiSetupState(ST_SETUP_DISABLED, wifiSetupSMIf)
{
   LOG_INFO("[WifiMode=%d] SetupDisabledState::Constructor entered", _wifiMode);
}

SetupDisabledState::~SetupDisabledState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   //LOG_INFO("[WifiMode=%d] SetupDisabledState::Destructor entered", _wifiMode);
}

void SetupDisabledState::onEntry(::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupDisabledState::onEntry()", _wifiMode);
   WBL_INTENTIONALLY_UNUSED(msg);

   //! TODO - below code is commented since interface will still be available in wifi setup,
   //! since wifi setup contains last received state of the setup. 
   //! TBC if below logic is required on entring this state.
   /*WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   //! If wifi setup interface is available, switch to Idle state
   const WifiSetUpItem& wifiSetupItem = _wifiSetupSMIf->getWifiSetupInfo();
   if (!(wifiSetupItem.property.interface.empty()))
   {
      _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
   }*/
}

void SetupDisabledState::onRequest(::boost::shared_ptr< PrepareSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupDisabledState::onRequest(PrepareSetup) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   //! Enable setup only for STA mode
   if (WIFI_MODE_STA == msg->getWifiMode())
   {
      _wifiSetupSMIf->setPrepareSetupMsg(msg);
      _wifiSetupSMIf->setState(ST_ENABLING_SETUP, nullptr);
   }
   else
   {
      _wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_NOTSUPPORTED);
   }
}

void SetupDisabledState::onRequest(::boost::shared_ptr< DeActivateSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupDisabledState::onRequest(DeactivateSetup) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   //! Send success response since setup is disabled
   _wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_NOERROR);
}

void SetupDisabledState::onRequest(::boost::shared_ptr< DisableSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupDisabledState::onRequest(DisableSetup) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   //! Send success response since setup is disabled
   _wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_NOERROR);
}

void SetupDisabledState::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode=%d] SetupDisabledState::onEvent() entered: EventID = %s, isError = %d",
         _wifiMode, SM_EVENT_TO_STR(eventID), isError);

   if (IS_CURRENT_MODE(setup.property.mode))
   {
      switch (eventID)
      {
         case WIFI_SETUP_EVENT_STA_TECH_ADDED:
         case WIFI_SETUP_EVENT_AP_ADDED:
         {
            //! If wifi setup interface is added, switch to Idle state
            if (!isError)
            {
               LOG_DEBUG("onEvent: Interface available. Switching to SetupIdle state.");
               WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);
               _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
            }
         }
         break;
         case WIFI_SETUP_EVENT_STA_TECH_REMOVED:
         case WIFI_SETUP_EVENT_AP_REMOVED:
         {
            //! Nothing to be done. Setup is already disabled.
         }
         break;
         default:
         {
            //! Forward unhandled events to base class event handler
            WifiSetupState::onEvent(eventID, setup, isError);
         }
         break;
      }
   } // if (IS_CURRENT_MODE(setup.property.mode))
}

} // namespace bosch
} // namespace org

/** @} */
