/**
 * @file EnablingSetupState.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string>

#include "WBLDefines.h"
#include "StateMachineProperties.h"
#include "IWifiSetupSM.h"
#include "WBLASFComponent.h"
#include "EnablingSetupState.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", EnablingSetupState, Info);

EnablingSetupState::EnablingSetupState(IWifiSetupSM* wifiSetupSMIf):
      WifiSetupState(ST_ENABLING_SETUP, wifiSetupSMIf)
{
   LOG_INFO("[WifiMode=%d] EnablingSetupState::Constructor entered", _wifiMode);
}

EnablingSetupState::~EnablingSetupState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   // LOG_INFO("[WifiMode=%d] EnablingSetupState::Destructor entered", _wifiMode);
}

void EnablingSetupState::onEntry(::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] EnablingSetupState::onEntry()", _wifiMode);
   WBL_INTENTIONALLY_UNUSED(msg);

   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   if (_wifiSetupSMIf->getPrepareSetupMsg())
   {
      if (!sendEnableRequest())
      {
         LOG_ERROR("onEntry: Could not enable Wifi interface. Switching to SetupError state.");
         switchToErrorState();
      }
   } 
   else
   {
      LOG_ERROR("onEntry: No PrepareSetup request available. Defaulting to SetupIdle state");
      _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
   }
}

void EnablingSetupState::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode=%d] EnablingSetupState::onEvent() entered: EventID = %s, isError = %d",
         _wifiMode, SM_EVENT_TO_STR(eventID), isError);

   if (IS_CURRENT_MODE(setup.property.mode))
   {
      switch (eventID)
      {
         case WIFI_SETUP_EVENT_STA_TECH_ADDED:
         {
            //! If STA interface is added, switch to Idle state. Else switch to Error state.
            (isError) ? (switchToErrorState()) : (onEnableSetupSuccess());
         }
         break;
         case WIFI_SETUP_EVENT_STA_POWERED:
         {
            //! If STA interface is powered ON, switch to Idle state. Else switch to Error state.
            (isError) ? (switchToErrorState()) : (onEnableSetupSuccess());
         }
         break;
         default:
         {
            //! Forward unhandled events to base class event handler
            WifiSetupState::onEvent(eventID, setup, isError);
         }
         break;
      }
   } // if (IS_CURRENT_MODE(setup.property.mode))
}

bool EnablingSetupState::sendEnableRequest()
{
   bool isRequestSent = false;
   WBL_ASSERT_AND_RETURN(!_wifiSetupSMIf, isRequestSent);

   bool isInterfaceAvailable = !((_wifiSetupSMIf->getWifiSetupInfo()).property.interface.empty());
   if (isInterfaceAvailable)
   {
      isRequestSent = true;
      onEnableSetupSuccess();
   }
   else
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_STA:
         {
            /*We cant perform these commands without the support in rootdaemon, therefore just use connman apis to power off the station mode..
            IEEE80211STAClientIf* staClientIf = (NULL != factory) ? (factory->getSTAClientIf()) : (NULL);
            if (NULL != staClientIf)
            {
               isRequestSent = staClientIf->enableInterface();
            }*/
            bool isInterfaceAvailable = isSTAPoweredON();
            LOG_INFO("[WifiMode=%d] EnablingSetupState::sendEnableRequest: isInterfaceAvailable = %s",_wifiMode, BOOL_TO_STR(isInterfaceAvailable));
            if (!isInterfaceAvailable)
            {
               IEEE80211STAClientIf* staClientIf = getSTAClientIf();
               if (staClientIf)
               {
                  const ::std::string& staObjPath = (_wifiSetupSMIf->getWifiSetupInfo()).nativeObjPath;
                  isRequestSent = staClientIf->setPowered(staObjPath,true);
                  LOG_INFO("[WifiMode=%d] EnablingSetupState::sendEnableRequest: isRequestSent = %s",
                        _wifiMode, BOOL_TO_STR(isRequestSent));
               }
            }
            else
            {
               isRequestSent = true;
               onEnableSetupSuccess();
            }
         }
         break;
         default:
         {
            LOG_ERROR("sendEnableRequest: Enabling mode %d not supported", WIFI_MODE_TO_STR(_wifiMode));
            switchToErrorState();
         }
         break;
      }

      LOG_INFO("[WifiMode=%d] EnablingSetupState::sendEnableRequest: isRequestSent = %s",
            _wifiMode, BOOL_TO_STR(isRequestSent));
   }

   return isRequestSent;
}

void EnablingSetupState::onEnableSetupSuccess()
{
   LOG_DEBUG("onEnableSetupSuccess: Wifi mode interface available. Switching to SetupIdle state.");
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
}

bool EnablingSetupState::isSTAPoweredON()
{
   bool bRet = false;
   WifiSetUpList wifiSetups =_wifiSetupSMIf->getAllWifiSetups();
   for(auto it = wifiSetups.begin(); it != wifiSetups.end(); ++it)
   {
      if(WIFI_MODE_STA == it->property.mode)
         bRet = it->property.isPowered;
   }
   return bRet;
}
} // namespace bosch
} // namespace org

/** @} */
