/**
 * @file DbusIfTypes.h
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#ifndef _DBUS_IF_TYPES_H
#define _DBUS_IF_TYPES_H

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>
#include <vector>
#include <map>

#include "asf/core/Types.h"

/**
 * Type definition for wrapper of act (ASF DBUS). To be used for SM handling.
 * HINT: outside of namespace!
 */
typedef uint64_t DbusAct;

namespace org 
{
namespace bosch
{

/**
 * Type definition for DBUS variant data type.
 */
enum DbusVariantDataType
{
   VARIANT_INT8 = 0,
   VARIANT_UINT8,
   VARIANT_INT16,
   VARIANT_UINT16,
   VARIANT_INT32,
   VARIANT_UINT32,
   VARIANT_INT64,
   VARIANT_UINT64,
   VARIANT_ENUM,
   VARIANT_BOOL,
   VARIANT_STRING,
   VARIANT_STRING_ARRAY,
   VARIANT_BYTE_ARRAY,
   VARIANT_UINT16_ARRAY,
   VARIANT_DICT,
   VARIANT_NONE,
   // -------------------------------------------------------------------------
   VARIANT_UNKNOWN /**< end marker, must be the last one */
};

/**
 * Type definition for DBUS basic variant data.
 */
class DbusIntVariant
{
public:
   DbusIntVariant();
   ~DbusIntVariant();
   DbusIntVariant(const DbusIntVariant& ref);
   DbusIntVariant& operator=(const DbusIntVariant& ref);
   bool operator==(const DbusIntVariant& ref) const;
   bool operator!=(const DbusIntVariant& ref) const;

   int8_t _int8Data; /**< int8 data */
   uint8_t _uint8Data; /**< uint8 data */
   int16_t _int16Data; /**< int16 data */
   uint16_t _uint16Data; /**< uint16 data */
   int32_t _int32Data; /**< int32 data */
   uint32_t _uint32Data; /**< uint32 data */
   int64_t _int64Data; /**< int64 data */
   uint64_t _uint64Data; /**< uint64 data */
   int _enumData; /**< enum data */
   bool _boolData; /**< bool data */
   bool _empty; /**< _empty data */
};

//------------------------------------------------------------------------------

/**
 * Type definition for DBUS variant data.
 * HINT: If there is a property containing a dictionary then each entry of the dictionary will be handled as a separate property. TODO: recheck later again
 */
class DbusVariant
{
public:
   DbusVariant();
   DbusVariant(const DbusVariant& ref);
   DbusVariant& operator=(const DbusVariant& ref);
   bool operator==(const DbusVariant& ref) const;
   bool operator!=(const DbusVariant& ref) const;
   virtual ~DbusVariant();
   inline DbusVariantDataType getType(void) const { return _type; }
   inline void setInt8(const int8_t data) { _intData._int8Data = data; _type = VARIANT_INT8; }
   inline void setUInt8(const uint8_t data) { _intData._uint8Data = data; _type = VARIANT_UINT8; }
   inline void setInt16(const int16_t data) { _intData._int16Data = data; _type = VARIANT_INT16; }
   inline void setUInt16(const uint16_t data) { _intData._uint16Data = data; _type = VARIANT_UINT16; }
   inline void setInt32(const int32_t data) { _intData._int32Data = data; _type = VARIANT_INT32; }
   inline void setUInt32(const uint32_t data) { _intData._uint32Data = data; _type = VARIANT_UINT32; }
   inline void setInt64(const int64_t data) { _intData._int64Data = data; _type = VARIANT_INT64; }
   inline void setUInt64(const uint64_t data) { _intData._uint64Data = data; _type = VARIANT_UINT64; }
   inline void setBool(const bool data) { _intData._boolData = data; _type = VARIANT_BOOL; }
   inline void setEnum(const int data) { _intData._enumData = data; _type = VARIANT_ENUM; }
   inline void setString(const ::std::string& data) { _stringData = data; _type = VARIANT_STRING; }
   inline void setStringArray(const ::std::vector< ::std::string >& data) { _stringArray = data; _type = VARIANT_STRING_ARRAY; }
   inline void setByteArray(const ::std::vector<uint8_t> data) { _byteArray = data; _type = VARIANT_BYTE_ARRAY; }
   inline void setIntegerArray(const ::std::vector<uint16_t> data) { _integerArray = data; _type = VARIANT_UINT16_ARRAY; }
   inline void setInt8DictElement(const ::std::string& key, const int8_t value) { _dict[key]._int8Data = value; _type = VARIANT_DICT; }
   inline void setUInt8DictElement(const ::std::string& key, const uint8_t value) { _dict[key]._uint8Data = value; _type = VARIANT_DICT; }
   inline void setInt16DictElement(const ::std::string& key, const int16_t value) { _dict[key]._int16Data = value; _type = VARIANT_DICT; }
   inline void setUInt16DictElement(const ::std::string key, const uint16_t value) { _dict[key]._uint16Data = value; _type = VARIANT_DICT; }
   inline void setInt32DictElement(const ::std::string& key, const int32_t value) { _dict[key]._int32Data = value; _type = VARIANT_DICT; }
   inline void setUInt32DictElement(const ::std::string& key, const uint32_t value) { _dict[key]._uint32Data = value; _type = VARIANT_DICT; }
   inline void setInt64DictElement(const ::std::string& key, const int64_t value) { _dict[key]._int64Data = value; _type = VARIANT_DICT; }
   inline void setUInt64DictElement(const ::std::string& key, const uint64_t value) { _dict[key]._uint64Data = value; _type = VARIANT_DICT; }
   inline void setBoolDictElement(const ::std::string& key, const bool value) { _dict[key]._boolData = value; _type = VARIANT_DICT; }

   inline ::std::vector< ::std::string >& getStringArrayMutable(void) { _type = VARIANT_STRING_ARRAY; return _stringArray; }
   inline void setNone(void) { _intData._empty = true; _type = VARIANT_NONE; }
   int8_t getInt8(void) const;
   uint8_t getUInt8(void) const;
   int16_t getInt16(void) const;
   uint16_t getUInt16(void) const;
   int32_t getInt32(void) const;
   uint32_t getUInt32(void) const;
   int64_t getInt64(void) const;
   uint64_t getUInt64(void) const;
   bool getBool(void) const;
   int getEnum(void) const;
   const ::std::string& getString(void) const;
   const ::std::vector< ::std::string >& getStringArray(void) const;
   const ::std::vector<uint8_t>& getByteArray(void) const;
   const ::std::vector<uint16_t>& getUInt16Array(void) const;
   ::std::map < ::std::string, int8_t > getString2Int8Dict(void) const;
   ::std::map < ::std::string, uint8_t > getString2UInt8Dict(void) const;
   ::std::map < ::std::string, int16_t > getString2Int16Dict(void) const;
   ::std::map < ::std::string, uint16_t > getString2UInt16Dict(void) const;
   ::std::map < ::std::string, int32_t > getString2Int32Dict(void) const;
   ::std::map < ::std::string, uint32_t > getString2UInt32Dict(void) const;
   ::std::map < ::std::string, int64_t > getString2Int64Dict(void) const;
   ::std::map < ::std::string, uint64_t > getString2UInt64Dict(void) const;
   ::std::map < ::std::string, bool > getString2Bool8Dict(void) const;

   bool getNone(void) const;
   void reset(void);

private:
   DbusVariantDataType _type; /**< data type */
   DbusIntVariant _intData; /**< integer */
   ::std::string _stringData; /**< string */
   ::std::vector< ::std::string > _stringArray; /**< string array */
   ::std::vector<uint8_t>  _byteArray;
   ::std::vector<uint16_t>  _integerArray;
   ::std::map < ::std::string, DbusIntVariant > _dict;
};

//------------------------------------------------------------------------------

/**
 * Type definition for DBUS variant property.
 */
class DbusVariantProperty final
{
public:
   DbusVariantProperty();
   DbusVariantProperty(const DbusVariantProperty& ref);
   DbusVariantProperty& operator=(const DbusVariantProperty& ref);
   bool operator==(const DbusVariantProperty& ref) const;
   bool operator!=(const DbusVariantProperty& ref) const;
   virtual ~DbusVariantProperty();
   void reset(void);
   // member with public access
   int propEnum; /**< enum value */
   DbusVariant propData; /**< data */
};

//------------------------------------------------------------------------------

/**
 * Type definition for a DBUS property entry.
 * A DBUS property entry contains a pointer to a string with property name and an enum value.
 */
class DbusPropertyEntry
{
public:
   DbusPropertyEntry();
   DbusPropertyEntry(const DbusPropertyEntry& ref);
   DbusPropertyEntry& operator=(const DbusPropertyEntry& ref);
   bool operator==(const DbusPropertyEntry& ref) const;
   bool operator!=(const DbusPropertyEntry& ref) const;
   bool operator<(const DbusPropertyEntry& ref) const;
   bool operator>(const DbusPropertyEntry& ref) const;
   virtual ~DbusPropertyEntry();
   // member with public access
   const ::std::string* propName; /**< pointer to a string with property name */
   int propEnum; /**< enum value */

protected:
   inline void setPropName(const ::std::string* name) { propName = name; }
   int compare(const DbusPropertyEntry& ref) const;
};

//------------------------------------------------------------------------------

/**
 * Type definition for a DBUS interface entry.
 * A DBUS interface entry contains a pointer to a list with properties and an enum value and a property list.
 * TODO: check for removal
 */
class DbusInterfaceEntry
{
public:
   DbusInterfaceEntry();
   DbusInterfaceEntry(const DbusInterfaceEntry& ref);
   DbusInterfaceEntry& operator=(const DbusInterfaceEntry& ref);
   bool operator==(const DbusInterfaceEntry& ref) const;
   bool operator!=(const DbusInterfaceEntry& ref) const;
   bool operator<(const DbusInterfaceEntry& ref) const;
   bool operator>(const DbusInterfaceEntry& ref) const;
   virtual ~DbusInterfaceEntry();
   // member with public access
   const ::std::string* interfaceName; /**< pointer to a string with interface name */
   int interfaceEnum; /**< enum value */
   const ::std::vector<DbusPropertyEntry>* propertyList; /**< list with properties related to the interface */

protected:
   inline void setInterfaceName(const ::std::string* name) { interfaceName = name; }
   inline void setPropertyList(const ::std::vector<DbusPropertyEntry>* list) { propertyList = list; }
   int compare(const DbusInterfaceEntry& ref) const;
};

//------------------------------------------------------------------------------

} //namespace bosch
} //namespace org

#endif //_DBUS_IF_TYPES_H

/** @} */
