/**
 * @file servicereg.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <string.h>
#include <errno.h>
#include <dbus-cli.h>
#include <log.h>
#include "inc/utils.h"
#include <accesspoint_if.h>
#include <ap.h>
#include <servicereg.h>

#define WLAN_EID_VENDOR_SPECIFIC    221

typedef
enum __validate_ie {
	VALIDATE_NONE,
	VALIDATE_HEADER,
	VALIDATE_LENGTH
} valiateie_t;

typedef
struct __vendor_service {
    enum nl80211_iftype iftype;
    char *IE;
    unsigned long length;
    void *data;
} vendorservice;

typedef
struct __vendor_service_owner {
    char *owner;
    unsigned int watch;
    GSList *vendorservices;
} vendorserviceowner;

static GSList *servicedrivers;
static GHashTable *activeservices;

static void
vendor_service_cleanup (gpointer data)
{
    vendorservice *service = data;
    return_if_fail (service);
    g_free (service->IE);
    g_free (service);
}

static void
vendor_service_owner_cleanup (gpointer data)
{
    vendorserviceowner *owner = data;

    return_if_fail (owner);

    INFO ("Cleaning up owner : %s [watch : %d]", owner->owner, owner->watch);

    g_free (owner->owner);
    if (owner->watch > 0)
        dbus_rm_watch (owner->watch);
    g_slist_free_full (owner->vendorservices, vendor_service_cleanup);
    g_free (owner);
}

static serviceregops*
find_driver (const enum nl80211_iftype iftype)
{
    GSList *temp;
    serviceregops *driver;

    temp = servicedrivers;
    for ( ; temp; temp = temp->data) {
        driver = temp->data;
        if (driver && driver->iftype == iftype) {
            INFO ("Vendor service driver found for %d : %s", iftype, driver->name);
            return driver;
        }
    }

    return NULL;
}

static vendorservice*
vendor_service_find (const vendorserviceowner *owner,
                     const enum nl80211_iftype iftype,
                     const char *IEs,
                     const unsigned long length)
{
    GSList *temp;
    vendorservice *service;

    return_val_if_fail (owner && IEs && length, NULL);

    temp = owner->vendorservices;
    for ( ; temp; temp = temp->next) {
        service = temp->data;
        if (service && service->iftype == iftype && length == service->length &&
                !memcmp (service->IE, IEs, length))
            return service;
    }

    return NULL;
}

static int
vendor_service_is_present (const vendorserviceowner *owner,
                           const enum nl80211_iftype iftype,
                           const char *IEs,
                           const unsigned long length)
{
    GSList *temp;
    vendorservice *service;

    return_val_if_fail (owner && IEs && length, -EINVAL);

    temp = owner->vendorservices;
    for ( ; temp; temp = temp->next) {
        service = temp->data;
        if (service && service->iftype == iftype && length == service->length &&
                !memcmp (service->IE, IEs, length))
            return 0;
    }

    return -ENOENT;
}

static int
vendor_service_is_duplicate (const enum nl80211_iftype iftype,
                             const char *IEs,
                             const unsigned long length)
{
    GSList *temp;
    unsigned int size = 0, diff = 0,
            duplicate = 0;
    GHashTableIter iter;
    gpointer key, value;
    vendorservice *service;
    vendorserviceowner *serviceowner;

    return_val_if_fail (IEs && length, -EINVAL);

    g_hash_table_iter_init (&iter, activeservices);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        serviceowner = value;
        temp = (serviceowner != NULL) ? serviceowner->vendorservices : NULL;
        for ( ; temp; temp = temp->next) {

            size = 0; diff = 0;
            service = temp->data;
            if (service && service->iftype == iftype &&
                    service->length == length) {
                for ( ; size < length; size++) {
                    if (service->IE [size] != IEs [size]) {
                        diff = 1;
                        break;
                    }
                }
            }

            if (!diff) {
                duplicate = 1;
                break;
            }
        }
    }

    if (duplicate)
        return 0;

    return -ENOENT;
}

static long
hex_to_decimal (const char *string,
                int base)
{
    long value;

    return_val_if_fail (string, 0);

    value = strtol (string, NULL, base);
    if ((errno == ERANGE && (value == LONG_MAX || value == LONG_MIN))
            || (errno != 0 && value == 0))
        return 0;

    INFO ("string : %s converted : %ld", string, value);
    return value;
}

static int
vendor_elements_get_length (unsigned long *length,
                            void *data,
                            const enum nl80211_iftype iftype)
{
    GSList *temp;
    GHashTableIter iter;
    gpointer key, value;
    vendorservice *service;
    vendorserviceowner *serviceowner;

    return_val_if_fail (length && activeservices, -EINVAL);

    *length = 0;
    g_hash_table_iter_init (&iter, activeservices);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        serviceowner = value;
        temp = (serviceowner != NULL) ? serviceowner->vendorservices : NULL;
        for ( ; temp; temp = temp->next) {
            service = temp->data;
            if (service && service->iftype == iftype && service->data == data)
                *length += service->length;
        }
    }

    DEBUG ("Length of the vendor elements : %ld", *length);
    return 0;
}

static char*
contruct_vendor_elements (unsigned long *length,
                          void *data,
                          enum nl80211_iftype iftype)
{
    int ret;
    GSList *temp;
    vendorservice *service;
    vendorserviceowner *serviceowner;
    unsigned long offset = 0;
    GHashTableIter iter;
    gpointer key, value;
    char *constructedie = NULL;

    return_val_if_fail (length && activeservices, NULL);

    ret = vendor_elements_get_length (length, data, iftype);
    if (ret < 0) {
        ERROR ("Failed to get the length of the complete vendor "
               "elements : %d/%s", -ret, strerror (-ret));
        return NULL;
    }

    constructedie = g_try_malloc0 (sizeof (*constructedie) * (*length));
    return_val_if_fail (constructedie, NULL);

    g_hash_table_iter_init (&iter, activeservices);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        serviceowner = value;
        temp = (serviceowner != NULL) ? serviceowner->vendorservices : NULL;
        for ( ; temp; temp = temp->next) {
            service = temp->data;
            if (service && service->iftype == iftype && service->data == data) {
                memcpy (constructedie + offset, service->IE, service->length);
                offset += service->length;
            }
        }
    }

    return constructedie;
}

static void
owner_disconnect_func (char *name, void *userdata)
{
    int ret, aptype = 0;
    gboolean contains;
    serviceregops *driver;
    vendorservice *service;
    char *constructedie;
    vendorserviceowner *serviceowner = userdata;
    unsigned long length = 0;
    struct vendor_element *constructed;
    GSList *datalist = NULL, *temp;

    DEBUG ("\"%s\" got disconnected from dbus, data: %p", name, serviceowner);

    return_if_fail (name && serviceowner);

    temp = serviceowner->vendorservices;
    for ( ; temp; temp = temp->next) {
        service = temp->data;
        if (service) {
            if (service->iftype == NL80211_IFTYPE_AP)
                aptype = 1;
            if (!g_slist_find (datalist, service->data))
                datalist = g_slist_append (datalist, service->data);
        }
    }

    INFO ("Length of the data list : %d", g_slist_length (datalist));

    contains = g_hash_table_contains (activeservices, name);
    if (contains) {

        g_hash_table_remove (activeservices, name);
        if (aptype) {

            driver = find_driver (NL80211_IFTYPE_AP);
            if (!driver || !driver->register_service)
                return;

            temp = datalist;
            for ( ; temp; temp = temp->next) {
                constructedie = contruct_vendor_elements (&length, temp->data, NL80211_IFTYPE_AP);
                if (!constructedie)
                    return;

                constructed = g_try_malloc0 (sizeof (*constructed));
                if (!constructed) {
                    g_free (constructedie);
                    return;
                }

                constructed->ies = constructedie;
                constructed->length = length;
                ret = driver->unregister_service (constructed, temp->data, NULL);
                if (ret < 0) {
                    ERROR ("Failed to unregister the vendor element of owner %s : %s", name, constructedie);
                    g_free (constructed->ies);
                    g_free (constructed);
                }
            }
        }
    } else
        vendor_service_owner_cleanup (serviceowner);
}

static int
validate_vendor_element (struct vendor_element *elements,
						 valiateie_t type)
{
	char temp[3];
	long vendorid = 0;

	return_val_if_fail (elements, -EINVAL);

	if ((elements->length & 0x01) || elements->length <= 4)
		return -EINVAL;

	memset (temp, 0, sizeof (temp));
	switch (type) {
		case VALIDATE_HEADER:
            memcpy (temp, elements->ies, 2);
            vendorid = hex_to_decimal (temp, 16);
            if (vendorid != WLAN_EID_VENDOR_SPECIFIC)
                return -EINVAL;
			break;
		case VALIDATE_LENGTH:
			memcpy (temp, elements->ies + 2, 2);
			vendorid = hex_to_decimal (temp, 16);
			if (vendorid != (long) ((elements->length - 4) / 2))
				return -EINVAL;
			break;
		default:
			ERROR ("Invalid Request for validation: %d", type);
			return -EINVAL;
	}

	return 0;
}

int
register_vendor_service (const enum nl80211_iftype iftype,
                         const char *owner,
                         void *vendor_elements,
                         void *userdata,
                         void *invocation)
{
    unsigned int watch;
    int ret = 0, newowner = 0;
    unsigned long length = 0, len;
    struct vendor_element *elements, *constructed;
    serviceregops *driver;
    vendorservice *service;
    vendorserviceowner *serviceowner;
    char *constructedie = NULL, *bindata;
    GSList *list = vendor_elements, *templist;
    gboolean hexstring = FALSE;

    DEBUG ("Owner: %s Iftype: %d length: %d", owner, iftype,
           g_slist_length (list));

    return_val_if_fail (list && invocation, -EINVAL);
    return_val_if_fail (g_slist_length (list), -EINVAL);
    driver = find_driver (iftype);
    return_val_if_fail (driver, -ENODEV);
    return_val_if_fail (driver->register_service, -ENOTSUP);

    templist = list;
    for ( ; templist; templist = templist->next) {

        hexstring = FALSE;
        bindata = NULL; len = 0;
        elements = templist->data;
        if (!elements)
            continue;

        if (NL80211_IFTYPE_AP == iftype) {

            ret = is_hex_string (elements->ies, elements->length);
            if (ret < 0) {
                if (elements->length <= 2 || (unsigned char)elements->ies [0] != WLAN_EID_VENDOR_SPECIFIC ||
                        (unsigned long)elements->ies [1] != (elements->length - 2))
                    return -EINVAL;
            } else {
                ret = validate_vendor_element (elements, VALIDATE_HEADER);
                return_val_if_fail (ret == 0, -EINVAL);
                ret = validate_vendor_element (elements, VALIDATE_LENGTH);
                return_val_if_fail (ret == 0, -EINVAL);
                hexstring = TRUE;
            }

            if (hexstring) {
                len = elements->length / 2;
                bindata = g_try_malloc0 (sizeof (*bindata) * len);
                return_val_if_fail (bindata, -ENOMEM);
                access_point_hex2bin (elements->ies, bindata, elements->length);
            } else {
                bindata = elements->ies;
                len = elements->length;
            }
        }

        ret = vendor_service_is_duplicate (iftype, bindata, len);

        if (hexstring)
            g_free (bindata);

        if (!ret)
            return -EEXIST;
    }

    serviceowner = g_hash_table_lookup (activeservices, owner);
    if (!serviceowner) {
        serviceowner = g_try_malloc0 (sizeof (*serviceowner));
        return_val_if_fail (serviceowner, -ENOMEM);
        watch = dbus_add_service_watch (owner, NULL, owner_disconnect_func, serviceowner);
        if (!watch) {
            ERROR ("Failed to add the dbus watch for the client : %s", owner);
            g_free (serviceowner);
            return -EIO;
        }

        INFO ("Service owner: %p watch id: %d", serviceowner, watch);
        newowner = 1;
        serviceowner->owner = g_strdup (owner);
        serviceowner->watch = watch;
        g_hash_table_replace (activeservices, serviceowner->owner, serviceowner);
    }

    templist = list;
    for ( ; templist; templist = templist->next) {

        elements = templist->data;
        if (!elements)
            continue;

        len = elements->length;
        if (hexstring)
            len = elements->length / 2;

        service = g_try_malloc0 (sizeof (*service));
        if (!service) {
            ret = -ENOMEM;
            continue;
        }

        service->IE = g_try_malloc0 (sizeof (*service->IE) * len);
        if (!service->IE) {
            g_free (service);
            ret = -ENOMEM;
            continue;
        }

        service->iftype = iftype;
        service->data = userdata;

        if (hexstring) {
            service->length = len;
            access_point_hex2bin (elements->ies, service->IE, elements->length);
        } else {
            service->length = elements->length;
            memcpy (service->IE, elements->ies, elements->length);
        }

        serviceowner->vendorservices = g_slist_append (serviceowner->vendorservices, service);
    }

    if (!g_slist_length (serviceowner->vendorservices))
        goto failure;

    constructedie = contruct_vendor_elements (&length, userdata, iftype);
    if (!constructedie) {
        ret = -ENOMEM;
        goto failure;
    }

    constructed = g_try_malloc0 (sizeof (*constructed));
    if (!constructed) {
        g_free (constructedie);
        ret = -ENOMEM;
        goto failure;
    }

    constructed->ies = constructedie;
    constructed->length = length;

    HEXDUMP (constructedie, length);
    ret = driver->register_service (constructed, userdata, invocation);
    if (ret < 0) {
        ERROR ("Failed to register the vendor element : %s", constructedie);
        g_free (constructed->ies);
        g_free (constructed);
        goto failure;
    }

    return 0;

failure:
    if (newowner)
        g_hash_table_remove (activeservices, owner);

    return ret;
}

int
unregister_vendor_service (const enum nl80211_iftype iftype,
                           const char *owner,
                           void *vendor_elements,
                           void *userdata,
                           void *invocation)
{
    int ret;
    unsigned long length = 0, len;
    struct vendor_element *elements, *constructed;
    serviceregops *driver;
    vendorservice *service;
    vendorserviceowner *serviceowner;
    char *constructedie = NULL, *bindata;
    GSList *list = vendor_elements, *templist;
    gboolean hexstring = FALSE;

    DEBUG ("Owner %s Iftype %d length %d", owner,
           iftype, g_slist_length (list));

    return_val_if_fail (list && invocation, -EINVAL);
    return_val_if_fail (g_slist_length (list), -EINVAL);
    serviceowner = g_hash_table_lookup (activeservices, owner);
    return_val_if_fail (serviceowner, -ENOENT);

    INFO ("Service %p watch %d", serviceowner, serviceowner->watch);

    driver = find_driver (iftype);
    return_val_if_fail (driver, -ENODEV);
    return_val_if_fail (driver->unregister_service, -ENOTSUP);

    templist = list;
    for ( ; templist; templist = templist->next) {

        hexstring = FALSE;
        bindata = NULL; len = 0;
        elements = templist->data;
        if (!elements)
            continue;

        if (NL80211_IFTYPE_AP == iftype) {

            ret = is_hex_string (elements->ies, elements->length);
            if (ret < 0) {
                if (elements->length <= 2 || (unsigned char)elements->ies [0] != WLAN_EID_VENDOR_SPECIFIC ||
                        (unsigned long)elements->ies [1] != (elements->length - 2))
                    return -EINVAL;
            } else {
                ret = validate_vendor_element (elements, VALIDATE_HEADER);
                return_val_if_fail (ret == 0, -EINVAL);
                ret = validate_vendor_element (elements, VALIDATE_LENGTH);
                return_val_if_fail (ret == 0, -EINVAL);
                hexstring = TRUE;
            }

            if (hexstring) {
                len = elements->length / 2;
                bindata = g_try_malloc0 (sizeof (*bindata) * len);
                return_val_if_fail (bindata, -ENOMEM);
                access_point_hex2bin (elements->ies, bindata, elements->length);
            } else {
                bindata = elements->ies;
                len = elements->length;
            }
        }

        ret = vendor_service_is_present (serviceowner, iftype, bindata, len);

        if (hexstring)
            g_free (bindata);

        if (ret < 0)
            return ret;
    }

    templist = list;
    for ( ; templist; templist = templist->next) {

        hexstring = FALSE;
        bindata = NULL; len = 0;
        elements = templist->data;
        if (!elements)
            continue;

        if (NL80211_IFTYPE_AP == iftype) {
            ret = is_hex_string (elements->ies, elements->length);
            if (!ret) {
                len = elements->length / 2;
                bindata = g_try_malloc0 (sizeof (*bindata) * len);
                return_val_if_fail (bindata, -ENOMEM);
                access_point_hex2bin (elements->ies, bindata, elements->length);
                hexstring = TRUE;
            } else {
                bindata = elements->ies;
                len = elements->length;
            }
        }

        service = vendor_service_find (serviceowner, iftype, bindata, len);
        if (hexstring)
            g_free (bindata);
        return_val_if_fail (service, -ENOENT);
        serviceowner->vendorservices = g_slist_remove (serviceowner->vendorservices, service);
        vendor_service_cleanup (service);
    }

    constructedie = contruct_vendor_elements (&length, userdata, iftype);
    if (!constructedie && length)
        return -ENOMEM;

    constructed = g_try_malloc0 (sizeof (*constructed));
    if (!constructed) {
        g_free (constructedie);
        return -ENOMEM;
    }

    constructed->ies = constructedie;
    constructed->length = length;
    HEXDUMP (constructedie, length);
    if (!g_slist_length (serviceowner->vendorservices))
        g_hash_table_remove (activeservices, owner);

    ret = driver->unregister_service (constructed, userdata, invocation);
    if (ret < 0) {
        ERROR ("Failed to register the vendor element : %s", constructedie);
        g_free (constructed->ies);
        g_free (constructed);
    }

    return ret;
}

int
vendor_service_driver_register (serviceregops *ops)
{
    GSList *temp;
    int found = 0;
    serviceregops *driver;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("Registering Vendor service driver : %s [%p]",
           ops->name ? ops->name : "UNKNOWN", ops);

    temp = servicedrivers;
    for ( ; temp; temp = temp->next) {
        driver = temp->data;
        if (ops == driver) {
            found = 1;
            break;
        }
    }

    if (!found) {
        servicedrivers = g_slist_append (servicedrivers, ops);
        return 0;
    }

    return -EEXIST;
}

int
vendor_service_driver_unregister (serviceregops *ops)
{
    int found = 0;
    GSList *temp;
    serviceregops *driver;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("Unregistering the vendor service driver : %s [%p]",
           ops->name ? ops->name : "UNKNOWN", ops);

    temp = servicedrivers;
    for ( ; temp; temp = temp->next) {
        driver = temp->data;
        if (ops == driver) {
            found = 1;
            break;
        }
    }

    if (found) {
        servicedrivers = g_slist_remove_link (servicedrivers, temp);
        return 0;
    }

    return -ENOENT;
}

int
servicereg_init ()
{
    DEBUG ("");

    activeservices = g_hash_table_new_full (g_str_hash, g_str_equal, NULL,
                                            vendor_service_owner_cleanup);

    return 0;
}

int
servicereg_deinit ()
{
    DEBUG ("");

    g_hash_table_destroy (activeservices);

    return 0;
}

/** @} */
