/**
 * @file ap.c
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_mw\wifi_ap_direct_manager
 * @brief
 *
 * @{
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>
#include <glib.h>
#include <gio/gio.h>
#include <errno.h>
#include <string.h>
#include <sys/socket.h>
#include <ap.h>
#include <main.h>
#include <tech_manager.h>
#include <accesspoint.h>
#include <log.h>
#include "inc/utils.h"
#include "inc/declarations.h"
#include <database.h>
#include <ippool.h>
#include <ipconfig.h>
#include <server.h>
#include <task.h>
#include <errors.h>
#include <accesspoint_if.h>
#include <associatedstation_if.h>
#include <servicereg.h>
#include <nat.h>
#include <genl.h>
#include <crypto.h>

#define CHANNEL_LENGTH                      128
#define SSID_MAX_LEN                        32

#define WPA_PSK_PASSPHRASE_MIN_LENGTH       8
#define WPA_PSK_PASSPHRASE_MAX_LENGTH       64

#define OPEN_SYSTEM_AUTHENTICATION          (1 << 0)
#define SHARED_KEY_AUTHENTICATION           (1 << 1)

#define HT_CAP_INFO_LDPC_CODING_CAP         (1 << 0)
#define HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET_MINUS	(1 << 1)
#define HT_CAP_INFO_SMPS_MASK               (1 << 2 | 1 << 3)
#define HT_CAP_INFO_SMPS_STATIC             (0)
#define HT_CAP_INFO_SMPS_DYNAMIC            (1 << 2)
#define HT_CAP_INFO_SMPS_DISABLED           (1 << 2 | 1 << 3)
#define HT_CAP_INFO_GREEN_FIELD             (1 << 4)
#define HT_CAP_INFO_SHORT_GI20MHZ           (1 << 5)
#define HT_CAP_INFO_SHORT_GI40MHZ           (1 << 6)
#define HT_CAP_INFO_TX_STBC                 (1 << 7)
#define HT_CAP_INFO_RX_STBC_MASK            (1 << 8 | 1 << 9)
#define HT_CAP_INFO_RX_STBC_1               (1 << 8)
#define HT_CAP_INFO_RX_STBC_12              (1 << 9)
#define HT_CAP_INFO_RX_STBC_123             (1 << 8 | 1 << 9)
#define HT_CAP_INFO_DELAYED_BA              (1 << 10)
#define HT_CAP_INFO_MAX_AMSDU_SIZE          (1 << 11)
#define HT_CAP_INFO_DSSS_CCK40MHZ           (1 << 12)
/* B13 - Reserved (was PSMP support during P802.11n development) */
#define HT_CAP_INFO_40MHZ_INTOLERANT		(1 << 14)
#define HT_CAP_INFO_LSIG_TXOP_PROTECT_SUPPORT	(1 << 15)
#define HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET_PLUS	(1 << 16)

#define WPS_CONFIG_USBA             ((unsigned short) 1 << 0)
#define WPS_CONFIG_ETHERNET         ((unsigned short) 1 << 1)
#define WPS_CONFIG_LABEL            ((unsigned short) 1 << 2)
#define WPS_CONFIG_DISPLAY          ((unsigned short) 1 << 3)
#define WPS_CONFIG_EXT_NFC_TOKEN    ((unsigned short) 1 << 4)
#define WPS_CONFIG_INT_NFC_TOKEN    ((unsigned short) 1 << 5)
#define WPS_CONFIG_NFC_INTERFACE    ((unsigned short) 1 << 6)
#define WPS_CONFIG_PUSHBUTTON       ((unsigned short) 1 << 7)
#define WPS_CONFIG_KEYPAD           ((unsigned short) 1 << 8)
#define WPS_CONFIG_VIRT_PUSHBUTTON  ((unsigned short) 1 << 9)
#define WPS_CONFIG_PHY_PUSHBUTTON   ((unsigned short) 1 << 10)
#define WPS_CONFIG_P2PS             ((unsigned short) 1 << 11)
#define WPS_CONFIG_VIRT_DISPLAY     ((unsigned short) 1 << 12)
#define WPS_CONFIG_PHY_DISPLAY      ((unsigned short) 1 << 13)

#define WPA_KEY_MGMT_PSK            "WPA-PSK"

#define CIPHER_CCMP                 "CCMP"
#define CIPHER_TKIP                 "TKIP"
#define CIPHER_WEP104               "WEP104"
#define CIPHER_WEP40                "WEP40"

#define DHCP_SETTINGS_GROUP_NAME		"DHCP-DNS-Settings"
#define TETHERING_SETTINGS_GROUP_NAME	"TetheringSettings"
#define CONFIG_FILE_FINGERPRINT     	"Conf-FingerPrint"

#define CTRL_INTERFACE_PATH			"/var/run/hostapd"
#define NL80211_DRIVER				"nl80211"
#define RTS_THRESHOLD				2347
#define FRAGMENT_THRESHOLD			2346
#define DTIM_PERIOD					1
#define BEACON_INTERVAL				100
#define MAX_STATIONS_ALLOWED		10		/* Max supported by HW */
#define AP_MAX_INACTIVITY			10
#define CONF_DISABLED				0
#define CONF_ENABLED				1
#define CONF_INVAL					-1

#define AC_VHT_20MHZ				(1 << 0)
#define AC_VHT_40MHZ				(1 << 1)
#define AC_VHT_80MHZ				(1 << 2)

#define IEEE80211_CHANNELS_2_4_LENGTH			14
#define IEEE80211_CHANNELS_5_LENGTH				47
#define IEEE80211_CHANNELS_LENGTH				(IEEE80211_CHANNELS_2_4_LENGTH + IEEE80211_CHANNELS_5_LENGTH)

#define DUMP_DATA_TO_KEYFILE(type, keyfile, group, member, data) \
    g_key_file_set_ ## type (keyfile, group, member, data);
#define REMOVE_DATA_FROM_KEYFILE(keyfile, group, member, error) \
	g_key_file_remove_key(keyfile, group, member, error);
#define GET_DATA_FROM_KEYFILE(type, keyfile, group, member, error) \
    g_key_file_get_ ## type (keyfile, group, member, error);

typedef
struct __ssid_buf
{
    char *ssid;
    size_t length;
} SSID;

typedef
struct __channel_info
{
    unsigned int *data;
    unsigned int length;
} channel;

typedef channel rate;

typedef
struct
{
    ippooltype type;
    char *start;
    char *end;
    char *gateway;
    char **nameservers;
    char **ntpservers;

    /* Pretty clear that no overlapping
     * blocks are supported and thus the
     * allowed pool size shall be of max 253
     * i.e., excluding network, router and
     * broadcast addresses (and obviously
     * revisit if there are requirements to
     * support overlapping blocks) */
    int poolsize;
    unsigned int leaseperiod;
    gboolean dnssupport;
    struct dhcp_server *server;
} dhcp_settings;

/* PFRC - Power Failure Reason Code */
typedef
enum {
    WAPDMAN_PFRC_NONE,
    WAPDMAN_PFRC_UNKNOWN,
    WAPDMAN_PFRC_NO_DRIVER,
    WAPDMAN_PFRC_INVAL_CONF,
    WAPDMAN_PFRC_DB_FAILURE,
    WAPDMAN_PFRC_REGDOM_FAILURE,
    WAPDMAN_PFRC_CONF_HAPD_FAILURE,
    WAPDMAN_PFRC_RM_HAPD_FAILURE,
    WAPDMAN_PFRC_HAPD_REJECTED,
    WAPDMAN_PFRC_DHCPDNS_CONF_FAILURE,
    WAPDMAN_PFRC_DHCPDNS_FAILED_STARTUP,
    WAPDMAN_PFRC_DHCPDNS_FAILED_STOP,
    WAPDMAN_PFRC_NAT_INVALID_CONF,
    WAPDMAN_PFRC_NAT_FAILED_START,
    WAPDMAN_PFRC_NAT_FAILED_STOP
} PowerFailureReason;

typedef
enum {
    /* invalid state */
    WAPDMAN_AP_STATE_INVALID,

    /* AP parameters are initialized */
    WAPDMAN_AP_STATE_INITIALIZED,

    /* Currently Configuring Regulatory domain for this interface */
    WAPDMAN_AP_STATE_CONFIGURING_REGDOM,

    /* Regulatory domain is configured for this interface */
    WAPDMAN_AP_STATE_CONFIGURED_REGDOM,

    /* AP is configured with hostapd */
    WAPDMAN_AP_STATE_CONFIGURING_HOSTAPD,

    /* AP is successfully configured with
     * hostapd */
    WAPDMAN_AP_STATE_CONFIGURED_HOSTAPD,

    /* AP configuration is getting removed
     * from hostapd */
    WAPDMAN_AP_STATE_REMOVING_HOSTAPD,

    /* AP is successfully removed from hostapd */
    WAPDMAN_AP_STATE_REMOVED_HOSTAPD,

    /* DHCP server is getting started */
    WAPDMAN_AP_STATE_STARTING_DHCP_SERVER,

    /* DHCP server is successfully started */
    WAPDMAN_AP_STATE_DHCP_SERVER_STARTED,

    /* Triggered to stop the DHCP server */
    WAPDMAN_AP_STATE_STOPPING_DHCP_SERVER,

    /* DHCP server successfully stopped */
    WAPDMAN_AP_STATE_DHCP_SERVER_STOPPED,

    /* Triggered NAT configuration */
    WAPDMAN_AP_STATE_CONFIGURING_NAT,

    /* NAT configured successfully */
    WAPDMAN_AP_STATE_CONFIGURED_NAT,

    /* Disabling the NAT for this interface */
    WAPDMAN_AP_STATE_DISABLING_NAT,

    /* NAT disabled successfully */
    WAPDMAN_AP_STATE_DISABLED_NAT,

    /* Any of the above states have been failed
     * and the AP is in failed state which may be
     * recoverable or may be not */
    WAPDMAN_AP_STATE_FAILED
} apstate;

typedef enum {
    POWER_INVAL = 0,
    POWERING_ON,
    POWERED_ON,
    POWERING_OFF,
    POWERED_OFF,
    POWER_FAILURE
} powerstate_t;

struct access_point
{
    unsigned int index;
    char *address;

    apstate state;
    powerstate_t powerstate;
    gboolean targetpowerstate;
    gboolean actualpowerstate;
    gboolean interfacedown;
    PowerFailureReason pwrfreason;

    char *interface;
    char *driver;
    char *ctrl_interface;
    unsigned int ctrl_iface_grp;
    SSID *ssid;
    char country_code [3];

    int ieee80211d;
    int ieee80211h;
    int ieee80211w;

    int current_operating_hwmode;
    unsigned int supported_hwmodes;

    unsigned short acs;
    unsigned int operating_channel;
    unsigned int acs_num_scans;
    channel *preferred_channels;
    channel *supported_channels;
    channel *supported_frequencies;

    unsigned short int beacon_int;
    int dtim;
    int max_stations_allowed;
    int rts_threshold;
    int fragm_threshold;

    rate *supported_rates;

    enum {
        ACCEPT_UNLESS_DENIED = 0,
        DENY_UNLESS_ACCEPTED = 1,
        USE_EXTERNAL_RADIUS_AUTH = 2
    } macaddr_acl;
    char *deny_mac_file;
    GList *blacklisted_stations;

    enum {
        BROADCAST_SSID = 0,
        BROADCAST_EMPTY_SSID = 1,
        BROADCAST_CLEAR_SSID = 2
    }ignore_broadcast_ssid;

    int wmm_enabled;

    int ap_max_inactivity;
    int skip_inactivity_poll;
    int disassoc_low_ack;

    int max_listen_interval;
    int uapsd_advertisement_enabled;

    /* HT - High Throughput */
    int ieee80211n;
    int ht_capab;
    int enable_ht;
    int ht_settings;

    /* VHT - Very High Throughput */
    int ieee80211ac;
    int enable_vht;
    int vht_oper_chwidth;
    int vht_oper_centr_freq_seg0_idx;
    int vht_settings;

    char *passphrase;
    securitytype security;

    int rsn_preauth;

    /* WPS */
    enum {
        WPS_DISABLED = 0,
        WPS_ENABLED_NOT_CONFIGURED = 1,
        WPS_ENABLED_CONFIGURED = 2
    } wps_state;
    int eap_server;
    int wps_independent;
    int ap_setup_locked;
    char *uuid;
    char *device_name;
    char *manufacturer;
    char *model_name;
    char *device_type;
    int model_number;
    int serial_number;
    unsigned short int config_methods;

    /* IEEE 802.11u - Interworking */
    int interworking;
    enum {
        PRIVATE_NETWORK = 0,
        PRIVATE_NETWORK_GUEST_ACCESS = 1,
        CHARGEABLE_PUBLIC_NETWORK = 2,
        FREE_PUBLIC_NETWORK = 3,
        PERSONAL_DEVICE_NETWORK = 4,
        EMERGENCY_SERIVCE_NETWORK_ONLY = 5,
        /* 6->13 reserved */
        EXPERIMENTAL = 14,
        WILDCARD = 15
    } access_network_type;
    int internet;
    int asra;
    int esr;
    int uesa;
    enum {
        VENUE_GROUP_UNSPECIFIED = 0,
        VENUE_GROUP_ASSEMBLY = 1,
        VENUE_GROUP_BUSINESS = 2,
        VENUE_GROUP_EDUCATIONAL = 3,
        VENUE_GROUP_FACTORY_AND_INDUSTRIAL = 4,
        VENUE_GROUP_INSTITUTIONAL = 5,
        VENUE_GROUP_MERCANTILE = 6,
        VENUE_GROUP_RESIDENTIAL = 7,
        VENUE_GROUP_STORAGE = 8,
        VENUE_GROUP_UTILITY_AND_MISCELLANEOUS = 9,
        VENUE_GROUP_VEHICULAR = 10,
        VENUE_GROUP_OUTDOOR = 11
        /* 12 - 255 reserved */
    } venue_group;
    /* Only the venue group Vehicular is considered */
    enum {
        VENUE_TYPE_UNSPECIFIED_VEHICLE = 0,
        VENUE_TYPE_AUTOMOBILE_OR_TRUCK = 1,
        VENUE_TYPE_AIRPLANE = 2,
        VENUE_TYPE_BUS = 3,
        VENUE_TYPE_FERRY = 4,
        VENUE_TYPE_SHIP_OR_BOAT = 5,
        VENUE_TYPE_TRAIN = 6,
        VENUE_TYPE_MOTOR_BIKE = 7
        /* 8 - 255 reserved */
    } venue_type;

    GHashTable *stations;
    gboolean removing;
    char *tempfile;
    char *conffile;

    char *vendor_elements;
    char *assocresp_elements;

    dhcp_settings *dhcpsettings;
    char *tethering_device;
    gboolean tethering;

    struct apdbuservice *dbusservice;
    nl80211_scan_state scanstate;

    GList *taskqueue;
};

struct _stations_notify {
    GHashTable *add;
    GHashTable *remove;
};

static GHashTable *access_points = NULL;
static char **configfile = NULL;
static int apdriverstatus = 0;
static struct _stations_notify *stations_notify = NULL;

static int ap_task_wakeup_next (struct access_point *ap);
static int access_point_create_conf_file (struct access_point *ap);
static void ap_add_interface_cb (int result, char *ifname, void *userdata);
static int ap_configure_nat (struct access_point *ap);
static int ap_task_configure_hostapd (struct access_point *ap, struct wapdman_task *task);
static int validate_preferred_channels (struct access_point *ap, int *chann2_4, int *chann5,
                            size_t *len2_4, size_t *len5, char *channels);
static int is_valid_channel (struct access_point *ap, int channel);
static int access_point_load_defaults (struct access_point *ap,
                            GKeyFile **keyfile, char **confpath);

static void
access_point_channel_cleanup (gpointer data)
{
    channel *chn = data;
    return_if_fail (chn);
    g_free (chn->data);
    g_free (chn);
}

static void
access_point_ssid_cleanup (gpointer data)
{
    SSID *ssid = data;
    return_if_fail (ssid);
    g_free (ssid->ssid);
    g_free (ssid);
}

static void
blacklisted_station_cleanup (gpointer data)
{
    g_free (data);
}

static void
access_point_cleanup (gpointer data)
{
    struct access_point *ap = data;

    return_if_fail (ap);

    DEBUG ("Cleaning up the access point: %s", ap->interface);

    if (ap->ssid)
        HEXDUMP (ap->ssid->ssid, ap->ssid->length);

    if (ap->dhcpsettings) {
        g_free (ap->dhcpsettings->start);
        g_free (ap->dhcpsettings->end);
        g_free (ap->dhcpsettings->gateway);
        g_strfreev (ap->dhcpsettings->nameservers);
        g_strfreev (ap->dhcpsettings->ntpservers);
        g_free (ap->dhcpsettings);
    }

    g_free (ap->address);
    g_free (ap->interface);
    g_free (ap->driver);
    g_free (ap->ctrl_interface);
    g_free (ap->deny_mac_file);
    g_free (ap->passphrase);
    g_free (ap->uuid);
    g_free (ap->device_name);
    g_free (ap->manufacturer);
    g_free (ap->model_name);
    g_free (ap->device_type);
    g_free (ap->tempfile);
    g_free (ap->conffile);
    g_free (ap->vendor_elements);
    g_free (ap->assocresp_elements);
    g_list_free_full (ap->blacklisted_stations,
                      blacklisted_station_cleanup);
    access_point_channel_cleanup (ap->supported_channels);
    access_point_channel_cleanup (ap->supported_frequencies);
    access_point_channel_cleanup (ap->preferred_channels);
    access_point_channel_cleanup (ap->supported_rates);
    access_point_ssid_cleanup (ap->ssid);
    g_free (ap->tethering_device);
    g_hash_table_destroy (ap->stations);
    g_free (ap);
}

static void
station_cleanup (gpointer data)
{
    associated_station_cleanup (data);
}

static char*
power_state2str (powerstate_t state)
{
    switch (state) {
    default:
        break;
    case POWERING_ON:
        return ENUMTOSTR (POWERING_ON);
    case POWERED_ON:
        return ENUMTOSTR (POWERED_ON);
    case POWERING_OFF:
        return ENUMTOSTR (POWERING_OFF);
    case POWERED_OFF:
        return ENUMTOSTR (POWERED_OFF);
    case POWER_FAILURE:
        return ENUMTOSTR (POWER_FAILURE);
    }

    return ENUMTOSTR (UNKNOWN);
}

static const char*
power_failure_reason2str (PowerFailureReason reason)
{
    switch (reason) {
    default:
    case WAPDMAN_PFRC_NONE:
    case WAPDMAN_PFRC_UNKNOWN:
        return ENUMTOSTR (PFRC_UNKNOWN);
    case WAPDMAN_PFRC_NO_DRIVER:
        return ENUMTOSTR (PFRC_NO_DRIVER);
    case WAPDMAN_PFRC_INVAL_CONF:
        return ENUMTOSTR (PFRC_INVAL_CONF);
    case WAPDMAN_PFRC_DB_FAILURE:
        return ENUMTOSTR (PFRC_DB_FAILURE);
    case WAPDMAN_PFRC_REGDOM_FAILURE:
        return ENUMTOSTR (REGDOM_FAILURE);
    case WAPDMAN_PFRC_CONF_HAPD_FAILURE:
        return ENUMTOSTR (PFRC_CONF_HAPD_FAILURE);
    case WAPDMAN_PFRC_RM_HAPD_FAILURE:
        return ENUMTOSTR (PFRC_RM_HAPD_FAILURE);
    case WAPDMAN_PFRC_HAPD_REJECTED:
        return ENUMTOSTR (PFRC_HAPD_REJECTED);
    case WAPDMAN_PFRC_DHCPDNS_CONF_FAILURE:
        return ENUMTOSTR (PFRC_DHCPDNS_CONF_FAILURE);
    case WAPDMAN_PFRC_DHCPDNS_FAILED_STARTUP:
        return ENUMTOSTR (PFRC_DHCPDNS_FAILED_STARTUP);
    case WAPDMAN_PFRC_DHCPDNS_FAILED_STOP:
        return ENUMTOSTR (PFRC_DHCPDNS_FAILED_STOP);
    case WAPDMAN_PFRC_NAT_FAILED_START:
        return ENUMTOSTR (PFRC_NAT_FAILED_START);
    case WAPDMAN_PFRC_NAT_FAILED_STOP:
        return ENUMTOSTR (PFRC_NAT_FAILED_STOP);
    case WAPDMAN_PFRC_NAT_INVALID_CONF:
        return ENUMTOSTR (PFRC_NAT_INVAL_CONF);
    }

    return ENUMTOSTR (WAPDMAN_PFRC_UNKNOWN);
}

static const char*
ap_state2string (const apstate state)
{
    switch (state) {
    default:
    case WAPDMAN_AP_STATE_INVALID:
        return ENUMTOSTR (STATE_INVALID);
    case WAPDMAN_AP_STATE_INITIALIZED:
        return ENUMTOSTR (STATE_INITIALIZED);
    case WAPDMAN_AP_STATE_CONFIGURING_REGDOM:
        return ENUMTOSTR (STATE_CONFIGURING_REGDOM);
    case WAPDMAN_AP_STATE_CONFIGURED_REGDOM:
        return ENUMTOSTR (STATE_CONFIGURED_REGDOM);
    case WAPDMAN_AP_STATE_CONFIGURING_HOSTAPD:
        return ENUMTOSTR (STATE_CONFIGURING_HOSTAPD);
    case WAPDMAN_AP_STATE_CONFIGURED_HOSTAPD:
        return ENUMTOSTR (STATE_CONFIGURED_HOSTAPD);
    case WAPDMAN_AP_STATE_REMOVING_HOSTAPD:
        return ENUMTOSTR (STATE_REMOVING_HOSTAPD);
    case WAPDMAN_AP_STATE_REMOVED_HOSTAPD:
        return ENUMTOSTR (STATE_REMOVED_HOSTAPD);
    case WAPDMAN_AP_STATE_STARTING_DHCP_SERVER:
        return ENUMTOSTR (STATE_STARTING_DHCP_SERVER);
    case WAPDMAN_AP_STATE_DHCP_SERVER_STARTED:
        return ENUMTOSTR (STATE_DHCP_SERVER_STARTED);
    case WAPDMAN_AP_STATE_STOPPING_DHCP_SERVER:
        return ENUMTOSTR (STATE_STOPPING_DHCP_SERVER);
    case WAPDMAN_AP_STATE_DHCP_SERVER_STOPPED:
        return ENUMTOSTR (STATE_DHCP_SERVER_STOPPED);
    case WAPDMAN_AP_STATE_CONFIGURING_NAT:
        return ENUMTOSTR (STATE_CONFIGURING_NAT);
    case WAPDMAN_AP_STATE_CONFIGURED_NAT:
        return ENUMTOSTR (STATE_CONFIGURED_NAT);
    case WAPDMAN_AP_STATE_DISABLING_NAT:
        return ENUMTOSTR (STATE_DISABLING_NAT);
    case WAPDMAN_AP_STATE_DISABLED_NAT:
        return ENUMTOSTR (STATE_DISABLED_NAT);
    case WAPDMAN_AP_STATE_FAILED:
        return ENUMTOSTR (STATE_FAILED);
    }

    return ENUMTOSTR (STATE_INVALID);
}

const char*
ap_task2string (aptask task)
{
    switch (task) {
    case TASK_AP_INVALID:
        return ENUMTOSTR (INVALID);
    case TASK_AP_SET_SSID:
        return ENUMTOSTR (SET_SSID);
    case TASK_AP_DEAUTHENTICATE_STATION:
        return ENUMTOSTR (DEAUTHENTICATE_STATION);
    case TASK_AP_BLACKLIST_STATION:
        return ENUMTOSTR (BLACKLIST_STATION);
    case TASK_AP_SET_HWMODE:
        return ENUMTOSTR (SET_HWMODE);
    case TASK_AP_SET_OPERATING_CHANNELS:
        return ENUMTOSTR (SET_OPERATING_CHANNELS);
    case TASK_AP_START_WPS_PUSH_BUTTON:
        return ENUMTOSTR (START_WPS_PUSH_BUTTON);
    case TASK_AP_CANCEL_WPS:
        return ENUMTOSTR (CANCEL_WPS);
    case TASK_AP_REGISTER_VENDOR_IE:
        return ENUMTOSTR (REGISTER_VENDOR_IE);
    case TASK_AP_UNREGISTER_VENDOR_IE:
        return ENUMTOSTR (UNREGISTER_VENDOR_IE);
    case TASK_AP_REGISTER_ASSOCRESP_IE:
        return ENUMTOSTR (REGISTER_ASSOCRESP_IE);
    case TASK_AP_UNREGISTER_ASSOCRESP_IE:
        return ENUMTOSTR (UNREGISTER_ASSOCRESP_IE);
    case TASK_AP_SET_POWERED:
        return ENUMTOSTR (SET_POWERED);
    case TASK_AP_CONFIGURE_HAPD:
        return ENUMTOSTR (CONFIGURE_HAPD);
    case TASK_AP_ENABLE_AP:
        return ENUMTOSTR (ENABLE_AP);
    case TASK_AP_DISABLE_AP:
        return ENUMTOSTR (DISABLE_AP);
    case TASK_AP_RELOAD_AP:
        return ENUMTOSTR (RELOAD_AP);
    case TASK_AP_SET_VISIBILITY:
        return ENUMTOSTR (SET_VISIBILITY);
    case TASK_AP_SET_SECURITY:
        return ENUMTOSTR (SET_SECURITY);
    case TASK_AP_SET_PASSPHRASE:
        return ENUMTOSTR (SET_PASSPHRASE);
    case TASK_AP_SET_BEACON_INTERVAL:
        return ENUMTOSTR (SET_BEACON_INTERVAL);
    case TASK_AP_SET_DTIM:
        return ENUMTOSTR (SET_DTIM);
    case TASK_AP_SET_MAX_STATIONS_ALLOWED:
        return ENUMTOSTR (SET_MAX_STATIONS_ALLOWED);
    case TASK_AP_SET_STATION_INACTIVITY_TIMEOUT:
        return ENUMTOSTR (SET_STATION_INACTIVITY_TIMEOUT);
    case TASK_AP_SET_HT:
        return ENUMTOSTR (SET_HT);
    case TASK_AP_SET_VHT:
        return ENUMTOSTR (SET_VHT);
    case TASK_AP_SET_ACCESS_NETWORK_TYPE:
        return ENUMTOSTR(TASK_AP_SET_ACCESS_NETWORK_TYPE);
    case TASK_AP_SET_FAST_ROAMING:
        return ENUMTOSTR (SET_FAST_ROAMING);
    case TASK_AP_SET_UUID:
        return ENUMTOSTR (SET_UUID);
    case TASK_AP_SET_DEVICENAME:
        return ENUMTOSTR (SET_DEVICENAME);
    case TASK_AP_SET_MANUFACTURER:
        return ENUMTOSTR (SET_MANUFACTURER);
    case TASK_AP_SET_MODELNAME:
        return ENUMTOSTR (SET_MODELNAME);
    case TASK_AP_SET_MODELNUMBER:
        return ENUMTOSTR (SET_MODELNUMBER);
    case TASK_AP_SET_SERIALNUMBER:
        return ENUMTOSTR (SET_SERIALNUMBER);
    case TASK_AP_SET_INTERWORKING:
        return ENUMTOSTR (SET_INTERWORKING);
    case TASK_AP_SET_COUNTRY_CODE:
        return ENUMTOSTR (SET_COUNTRY_CODE);
    case TASK_AP_ADD_DHCP_LEASE:
        return ENUMTOSTR (ADD_DHCP_LEASE);
    case TASK_AP_REMOVE_DHCP_LEASE:
        return ENUMTOSTR (REMOVE_DHCP_LEASE);
    case TASK_AP_SET_DNS_SERVERS:
        return ENUMTOSTR (SET_DNS_SERVERS);
    case TASK_AP_SET_DHCP_POOL:
        return ENUMTOSTR (SET_DHCP_POOL);
    case TASK_AP_DHCP_HOST_CONFIGURATION:
        return ENUMTOSTR (DHCP_HOST_CONFIGURATION);
    case TASK_AP_SET_DNSSERVICE_SUPPORT:
        return ENUMTOSTR (SET_DNSSERVICE_SUPPORT);
    case TASK_AP_SET_DHCPSERVICE_SUPPORT:
        return ENUMTOSTR (SET_DHCPSERVICE_SUPPORT);
    case TASK_AP_SET_DNS_PORT:
        return ENUMTOSTR (SET_DNS_PORT);
    case TASK_AP_START_DHCPDNS_SERVER:
        return ENUMTOSTR (START_DHCPDNS_SERVER);
    case TASK_AP_STOP_DHCPDNS_SERVER:
        return ENUMTOSTR (STOP_DHCPDNS_SERVER);
    case TASK_AP_CLEAR_DATA:
        return ENUMTOSTR (AP_CLEAR_DATA);
    case TASK_AP_RESTORE_DEFAULT_PRIVATE_DATA:
        return ENUMTOSTR (RESTORE_DEF_PRIVATE_DATA);
    case TASK_AP_SET_TETHERING:
        return ENUMTOSTR (AP_SET_TETHERING);
    case TASK_AP_REMOVE_HAPF_CONF:
        return ENUMTOSTR (REMOVE_HAPF_CONF);
    case TASK_AP_RECONFIGURE_ON_REGDOM_CHANGE:
        return ENUMTOSTR (RECONFIGURE_ON_REGDOM_CHANGE);
    }

    return ENUMTOSTR (INVALID);
}

static const char*
ap_security2str (securitytype type)
{
    switch (type) {
    case AP_SECURITY_OPEN:
        return ENUMTOSTR (OPEN);
    case AP_SECURITY_WEP:
        return ENUMTOSTR (WEP);
    case AP_SECURITY_WPA_PSK:
        return ENUMTOSTR (WPA_PSK);
    case AP_SECURITY_WPA2_PSK:
        return ENUMTOSTR (WPA2_PSK);
    case AP_SECURITY_WPA_WPA2_PSK:
        return ENUMTOSTR (WPA_WPA2_PSK);
    default:
        return ENUMTOSTR (UNKNOWN);
    }

    return ENUMTOSTR (UNKNOWN);
}

char*
access_point_get_power_failure_reason (struct access_point *ap)
{
    return_val_if_fail (ap, NULL);

    switch (ap->pwrfreason) {
    case WAPDMAN_PFRC_RM_HAPD_FAILURE:
        return ENUMTOSTR (Failed to remove AP configuration from Hostapd);
    case WAPDMAN_PFRC_HAPD_REJECTED:
        return ENUMTOSTR (Invalid AP Configuration/rejected by Hostapd);
    case WAPDMAN_PFRC_DHCPDNS_CONF_FAILURE:
        return ENUMTOSTR (Invalid DHCP/DNS configuration);
    case WAPDMAN_PFRC_NONE:
        return ENUMTOSTR ();
    case WAPDMAN_PFRC_UNKNOWN:
        return ENUMTOSTR (Unknown);
    case WAPDMAN_PFRC_NO_DRIVER:
        return ENUMTOSTR (No AP Driver found);
    case WAPDMAN_PFRC_INVAL_CONF:
        return ENUMTOSTR (Invalid AP configuration/Required AP parameters are not set);
    case WAPDMAN_PFRC_DB_FAILURE:
        return ENUMTOSTR (DB failure);
    case WAPDMAN_PFRC_REGDOM_FAILURE:
        return ENUMTOSTR (RegDom Update failure);
    case WAPDMAN_PFRC_CONF_HAPD_FAILURE:
        return ENUMTOSTR (Failed to configure AP);
    case WAPDMAN_PFRC_DHCPDNS_FAILED_STARTUP:
        return ENUMTOSTR (Failed to start DHCP/DNS server);
    case WAPDMAN_PFRC_DHCPDNS_FAILED_STOP:
        return ENUMTOSTR (Failed to Stop DHCP/DNS server);
    case WAPDMAN_PFRC_NAT_FAILED_START:
        return ENUMTOSTR (Failed to enable NAT);
    case WAPDMAN_PFRC_NAT_INVALID_CONF:
        return ENUMTOSTR (Invalid NAT Configuration);
    case WAPDMAN_PFRC_NAT_FAILED_STOP:
        return ENUMTOSTR (Failed to disable NAT);
    }

    return NULL;
}

static void
set_power_failure_reason (struct access_point *ap,
                          PowerFailureReason reason)
{
    PowerFailureReason oldreason;

    return_if_fail (ap);

    oldreason = ap->pwrfreason;
    if (oldreason == reason)
        return;

    INFO ("old: %s new: %s", power_failure_reason2str (oldreason),
          power_failure_reason2str (reason));

    ap->pwrfreason = reason;
    access_point_update_property (ap->dbusservice, "PowerFailureReason");
}

static void
access_point_set_power_state (struct access_point *ap,
                              powerstate_t state)
{
    powerstate_t old = POWER_INVAL;

    return_if_fail (ap);

    old = ap->powerstate;
    if (old == state)
        return;

    INFO ("old: %s new: %s", power_state2str (old),
          power_state2str (state));

    ap->powerstate = state;
    access_point_update_property (ap->dbusservice, "PowerState");
}

char*
access_point_get_power_state (struct access_point *ap)
{
    return_val_if_fail (ap, NULL);
    return power_state2str (ap->powerstate);
}

static void
set_ap_state (struct access_point *ap,
              apstate state)
{
    apstate old;

    return_if_fail (ap);
    old = ap->state;
    return_if_fail (old != state);
    ap->state = state;
    INFO ("old: %s new: %s", ap_state2string (old), ap_state2string (state));
}

static int
access_point_hex2num (char c)
{
    if (c >= '0' && c <= '9')
        return c - '0';
    if (c >= 'a' && c <= 'f')
        return c - 'a' + 10;
    if (c >= 'A' && c <= 'F')
        return c - 'A' + 10;

    return -1;
}

static int
access_point_hex2byte (const char *hex)
{
    int a, b;

    a = access_point_hex2num (*hex++);
    if (a < 0)
        return -1;

    b = access_point_hex2num (*hex++);
    if (b < 0)
        return -1;

    return (a << 4) | b;
}

int
access_point_hex2bin (const char *hex,
                      char *dest,
                      size_t len)
{
    size_t i = 0;
    int a;
    const char *ipos = hex;
    char *opos = dest;

    for ( ; i < len; i++) {
        a = access_point_hex2byte (ipos);
        if (a < 0)
            return -1;
        *opos++ = (char) a;
        ipos += 2;
    }

    return 0;
}

static int
access_point_compute_ssid (struct access_point *ap,
                           const char *ssid,
                           char **utf8data,
                           size_t *length)
{
    int ret;
    char *temp;
    size_t len;

    return_val_if_fail (ap && ssid && utf8data
                        && length, -EINVAL);
    return_val_if_fail (!*utf8data, -EEXIST);

    DEBUG ("Access point: %p ssid : %s", ap, ssid);

    len = strlen (ssid);
    if (len > 64 || (len & 0x01))
        return -EINVAL;

    len = strlen (ssid) / 2;
    temp = g_try_malloc0 (len * sizeof (*temp));
    return_val_if_fail (temp, -ENOMEM);

    ret = access_point_hex2bin (ssid, temp, len);
    if (ret < 0) {
        ERROR ("SSID consists of invalid characters");
        g_free (temp);
        return -EINVAL;
    }

    *utf8data = temp;
    *length = len;

    HEXDUMP (temp, len);
    return 0;
}

int
access_point_get_preferred_channels (struct access_point *ap,
                                     char *chanls)
{
    char temp [10];
    char *format = "%d ";
    unsigned int length = 0, i = 0;

    return_val_if_fail (ap && chanls, -EINVAL);
    return_val_if_fail (ap->preferred_channels, -ENOENT);

    memset (temp, 0, sizeof (temp));
    length = ap->preferred_channels->length;
    while (length) {
        if (length == 1)
            format = "%d";
        sprintf (temp, format, ap->preferred_channels->data [i]);
        strcat (chanls, temp);
        length--; i++;
    }

    DEBUG ("Access point: %s preferred channel list: %s", ap->interface, chanls);
    return 0;
}

static int
access_point_fill_security (struct access_point *ap,
                            const char *groupid,
                            GKeyFile *kfile)
{
    size_t len;
    int wpa = 0;
    char wep_key [9], *rsn_pairwise,
            *wpa_pairwise;

    return_val_if_fail (ap && groupid && kfile, -EINVAL);

    if (AP_SECURITY_OPEN != ap->security && !ap->passphrase)
        return -EINVAL;

    if (AP_SECURITY_WEP == ap->security) {

        memset (wep_key, 0, sizeof (wep_key));
        DUMP_DATA_TO_KEYFILE (integer, kfile, groupid, "auth_algs", 2);
        DUMP_DATA_TO_KEYFILE (integer, kfile, groupid, "wep_default_key", 0);
        sprintf (wep_key, "wep_key%d", 0);
        DUMP_DATA_TO_KEYFILE (string, kfile, groupid, wep_key, ap->passphrase);
    }
    else if (AP_SECURITY_WPA_PSK == ap->security || AP_SECURITY_WPA2_PSK == ap->security ||
             AP_SECURITY_WPA_WPA2_PSK == ap->security) {

        len = strlen (ap->passphrase);
        (AP_SECURITY_WPA_PSK == ap->security) ? (wpa = 1) : (AP_SECURITY_WPA2_PSK == ap->security) ? (wpa = 2)
        		: (AP_SECURITY_WPA_WPA2_PSK == ap->security) ? (wpa = 3) : (wpa = 2);

        if (len < 8)
          return -EINVAL;

        DUMP_DATA_TO_KEYFILE (integer, kfile, groupid, "auth_algs", 1);
        DUMP_DATA_TO_KEYFILE (integer, kfile, groupid, "wpa", wpa);
        DUMP_DATA_TO_KEYFILE (string, kfile, groupid, "wpa_key_mgmt", WPA_KEY_MGMT_PSK);

        wpa_pairwise = rsn_pairwise = CIPHER_CCMP;
        if (AP_SECURITY_WPA_PSK == ap->security)
            wpa_pairwise = rsn_pairwise = CIPHER_TKIP;
        DUMP_DATA_TO_KEYFILE (string, kfile, groupid, "wpa_pairwise", wpa_pairwise);
        DUMP_DATA_TO_KEYFILE (string, kfile, groupid, "rsn_pairwise", rsn_pairwise);
        if (len == 64) {
            DUMP_DATA_TO_KEYFILE (string, kfile, groupid, "wpa_psk", ap->passphrase);
        } else
            DUMP_DATA_TO_KEYFILE (string, kfile, groupid, "wpa_passphrase", ap->passphrase);
    }

    return 0;
}

#if 0
static char*
access_point_get_ht_capab (struct access_point *ap)
{
    char *ht_capab = NULL, temp [512];
    int noofbits = 0, local;

    return_val_if_fail (ap, NULL);

    for (local = ap->ht_capab; local; local >>= 1)
        if (local & 0x01)
            ++noofbits;

    memset (temp, 0, sizeof (temp));
    for (local = 0; local < noofbits; local++) {
        if (local > 0)
            strcat (temp, " ");
        if (ap->ht_capab & HT_CAP_INFO_LDPC_CODING_CAP)
            strcat (temp, "[LDPC]");
        if (ap->ht_capab & HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET_MINUS)
            strcat (temp, "[HT40-]");
        if (ap->ht_capab & HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET_PLUS)
            strcat (temp, "[HT40+]");
        if (ap->ht_capab & HT_CAP_INFO_SMPS_STATIC)
            strcat (temp, "[SMPS-STATIC]");
        if (ap->ht_capab & HT_CAP_INFO_SMPS_DYNAMIC)
            strcat (temp, "[SMPS-DYNAMIC]");
        if (ap->ht_capab & HT_CAP_INFO_GREEN_FIELD)
            strcat (temp, "[GF]");
        if (ap->ht_capab & HT_CAP_INFO_SHORT_GI20MHZ)
            strcat (temp, "[SHORT-GI-20]");
        if (ap->ht_capab & HT_CAP_INFO_SHORT_GI40MHZ)
            strcat (temp, "[SHORT-GI-40]");
        if (ap->ht_capab & HT_CAP_INFO_TX_STBC)
            strcat (temp, "[TX-STBC]");
        if (ap->ht_capab & HT_CAP_INFO_RX_STBC_1)
            strcat (temp, "[RX-STBC1]");
        if (ap->ht_capab & HT_CAP_INFO_RX_STBC_12)
            strcat (temp, "[RX-STBC12]");
        if (ap->ht_capab & HT_CAP_INFO_RX_STBC_123)
            strcat (temp, "[RX-STBC123]");
        if (ap->ht_capab & HT_CAP_INFO_DELAYED_BA)
            strcat (temp, "[DELAYED-BA]");
        if (ap->ht_capab & HT_CAP_INFO_MAX_AMSDU_SIZE)
            strcat (temp, "[MAX-AMSDU-7935]");
        if (ap->ht_capab & HT_CAP_INFO_DSSS_CCK40MHZ)
            strcat (temp, "[DSSS_CCK-40]");
        if (ap->ht_capab & HT_CAP_INFO_40MHZ_INTOLERANT)
            strcat (temp, "[40-INTOLERANT]");
        if (ap->ht_capab & HT_CAP_INFO_LSIG_TXOP_PROTECT_SUPPORT)
            strcat (temp, "[LSIG-TXOP-PROT]");
    }

    ht_capab = g_strdup (temp);
    return ht_capab;

}
#endif

static int
access_point_mode_conf (struct access_point *ap,
                        const char *groupid,
                        GKeyFile *keyfile)
{
    int ret;
    char *mode = NULL;
#if 0
    *ht_capab;
#endif

    ret = access_point_get_hwmode (ap, &mode);
    return_val_if_fail (ret == 0, ret);

    if (!g_strcmp0 (mode, "ad")) {
        ERROR ("IEEE 802.11ad is not handled yet!!");
        g_free (mode);
        return -EINVAL;
    }

    DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "hw_mode", mode);
    if (!g_strcmp0 (mode, "b")) {
        g_free (mode);
        return 0;
    }

    if (ap->enable_ht || ap->enable_vht) {
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "ieee80211n", ap->ieee80211n);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "wmm_enabled", ap->wmm_enabled);
    }

#if 0
    if (!g_strcmp0 (mode, "a")) {
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "ieee80211ac", ap->ieee80211ac);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "vht_oper_chwidth", ap->vht_oper_chwidth);
        if (ap->vht_oper_centr_freq_seg0_idx)
            DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "vht_oper_centr_freq_seg0_idx",
                                  ap->vht_oper_centr_freq_seg0_idx);

        ht_capab = access_point_get_ht_capab (ap);
        if (ht_capab)
            DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "ht_capab", ht_capab);
        g_free (ht_capab);
    }
#endif

    g_free (mode);
    return 0;
}

static int
access_point_get_first_channel (struct access_point *ap,
                                gboolean is5ghz,
                                unsigned int *channel)
{
    unsigned int index = 0, chan = 0;

    return_val_if_fail (ap && ap->supported_channels
                        && channel, -EINVAL);

    for ( ; index < ap->supported_channels->length; index++) {
        chan = ap->supported_channels->data [index];
        if (!is5ghz) {
            if (chan >= 1 && chan <= 14) {
                *channel = chan;
                return 0;
            }
        } else {
            if (chan > 36) {
                *channel = chan;
                return 0;
            }
        }
    }

    return -ENOENT;
}

static int
access_point_create_conf (struct access_point *ap,
                          const char *groupid,
                          GKeyFile *keyfile)
{
    int ret;
    char *ssid = NULL;
    char preferredchannels [PATH_MAX];
    unsigned int operating_chan = 0;

    return_val_if_fail (ap && groupid && keyfile && ap->ssid, -EINVAL);

    ret = access_point_mode_conf (ap, groupid, keyfile);
    if (ret < 0) {
        ERROR ("Failed to configure the HW mode for %s: %s/%d", ap->interface,
               strerror (-ret), -ret);
        return ret;
    }

    if (ap->ssid->length == 0)
        return -EINVAL;

    return_val_if_fail (ap->ssid->ssid [0] != '\0', -EINVAL);
    ssid = byte_to_hexstring (ap->ssid->ssid, ap->ssid->length);
    return_val_if_fail (ssid, -EINVAL);

    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "utf8_ssid", 1);
    DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "ssid2", ssid);
    g_free (ssid);

    ret = access_point_fill_security (ap, groupid, keyfile);
    if (ret < 0) {
        ERROR ("Invalid security and passphrase combination: %s/%d", strerror (-ret), -ret);
        return ret;
    }

    if (ap->acs) {
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "channel", 0);
        memset (preferredchannels, 0 , sizeof (preferredchannels));
        ret = access_point_get_preferred_channels (ap, preferredchannels);
        if (!ret)
            DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "chanlist", preferredchannels);
    } else {
        if (!ap->operating_channel) {
            ret = access_point_get_first_channel (ap, ap->current_operating_hwmode == IEEE80211_MODE_A ?
                                                      TRUE : FALSE, &operating_chan);
            if (ret < 0) {
                ERROR ("Invalid channel and HW mode detected: %s/%d", strerror (-ret), -ret);
                return ret;
            } else
                (void) access_point_set_operating_channel (ap, operating_chan);
        }
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "channel", (int) ap->operating_channel);
    }

    DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "interface", ap->interface);
    DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "driver", ap->driver);
    DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "ctrl_interface", ap->ctrl_interface);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "rts_threshold", ap->rts_threshold);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "fragm_threshold", ap->fragm_threshold);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "ieee80211w", ap->ieee80211w);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "dtim_period", ap->dtim);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "max_num_sta", ap->max_stations_allowed);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "macaddr_acl", ap->macaddr_acl);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "beacon_int", ap->beacon_int);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "uapsd_advertisement_enabled", ap->uapsd_advertisement_enabled);

//    if (ap->country_code[0] != '\0') {
//        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "country_code", ap->country_code);
//        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "ieee80211d", ap->ieee80211d);
//        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "ieee80211h", ap->ieee80211h);
//    }

    if (ap->deny_mac_file)
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "deny_mac_file", ap->deny_mac_file);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "ignore_broadcast_ssid", ap->ignore_broadcast_ssid);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "ap_max_inactivity", ap->ap_max_inactivity);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "skip_inactivity_poll", ap->skip_inactivity_poll);
    DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "disassoc_low_ack", ap->disassoc_low_ack);

    if (ap->interworking) {
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "interworking", ap->interworking);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "internet", ap->internet);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "asra", ap->asra);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "esr", ap->esr);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "uesa", ap->uesa);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "venue_group", ap->venue_group);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "venue_type", ap->venue_type);
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "access_network_type", ap->access_network_type);
    }

    if (ap->vendor_elements){
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "vendor_elements", ap->vendor_elements);
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "assocresp_elements", ap->vendor_elements);
    }
    return 0;
}

static int
access_point_dump_servers (char **nameservers,
                           char *group,
                           char *option,
                           GKeyFile *keyfile)
{
    int index = 0;
    char servers [BUFSIZ];
    char **nservers;

    return_val_if_fail (nameservers && option &&
                        group && keyfile, -EINVAL);

    if (!g_strv_length (nameservers))
        return 0;

    memset (servers, 0, sizeof (servers));
    for (nservers = nameservers; *nservers; nservers++, index++) {
        if(strlen (*nservers) < sizeof (servers) - strlen (servers)) {
            strcat (servers, *nservers);
            if (index > 0) {
                strcat (servers, ",");
            }
        }
    }

    DUMP_DATA_TO_KEYFILE (string, keyfile, group, option, servers);
    return 0;
}

static int
access_point_dump_keyfile (struct access_point *ap,
                           const char *groupid,
                           GKeyFile *keyfile)
{
    int ret;
    char *mode = NULL, *ssid = NULL;
    const char *security = NULL;
    char *sha256hash = NULL;
    char preferredchannels [PATH_MAX];
    char dhcprange [128];

    return_val_if_fail (ap && groupid && keyfile && ap->ssid, -EINVAL);

    DUMP_DATA_TO_KEYFILE (boolean, keyfile, groupid, "Enable", ap->targetpowerstate);
    ssid = byte_to_hexstring (ap->ssid->ssid, ap->ssid->length);
    if (NULL != ssid) {
       DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "SSID", ssid);
    } else {
    	REMOVE_DATA_FROM_KEYFILE (keyfile, groupid, "SSID", NULL);
    }
    g_free (ssid);

    ret = access_point_get_hwmode (ap, &mode);
    if (!ret)
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "HWMode", mode);
    g_free (mode);

    if (ap->acs) {
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "Channel", 0);
        memset (preferredchannels, 0 , sizeof (preferredchannels));
        ret = access_point_get_preferred_channels (ap, preferredchannels);
        if (!ret)
            DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "Chanlist", preferredchannels);
    } else {
        if (ap->operating_channel) {
            DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "Channel", (int) ap->operating_channel);
            g_key_file_remove_key (keyfile, groupid, "Chanlist", NULL);
        }
    }

    security = access_point_get_security (ap);
    if (security) {
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "Security", security);
    } else {
        ERROR ("Failed to get the security type for the ap: %s", ap->interface);
    }

    if (NULL != ap->passphrase) {
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "Passphrase", ap->passphrase);
    } else {
        REMOVE_DATA_FROM_KEYFILE (keyfile, groupid, "Passphrase", NULL);
    }

    if (ap->beacon_int != 0)
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "BeaconInt", ap->beacon_int);
    if (ap->uapsd_advertisement_enabled != -1)
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "UAPSDAdv", ap->uapsd_advertisement_enabled);
    if (ap->vendor_elements){
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "vendor_elements", ap->vendor_elements);
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "assocresp_elements", ap->vendor_elements);
    }
    if (ap->dtim != 0 && ap->dtim != -1)
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "DTIMPeriod", ap->dtim);
    if (ap->max_stations_allowed != -1)
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "MaxNumStations", ap->max_stations_allowed);
    DUMP_DATA_TO_KEYFILE (boolean, keyfile, groupid, "Visibility", ap->ignore_broadcast_ssid ? FALSE : TRUE);

    if (ap->interworking)
        DUMP_DATA_TO_KEYFILE (integer, keyfile, groupid, "access_network_type", ap->access_network_type);
    if (ap->country_code[0] != '\0')
        DUMP_DATA_TO_KEYFILE (string, keyfile, groupid, "CountryCode", ap->country_code);

    if (ap->conffile) {
            ret = calculate_sha256_hash (ap->conffile, &sha256hash);
        if (ret == 0) {
            INFO ("Calculated sha256 hash for the conf file [%s]: %s", ap->conffile, sha256hash);
            DUMP_DATA_TO_KEYFILE (string, keyfile, CONFIG_FILE_FINGERPRINT, "ConfSHA256", sha256hash);
            g_free (sha256hash);
        } else {
            ERROR ("Failed to caculate the sha256 digest for the conf file "
                   "[%s]: %s/%d", ap->conffile, strerror (-ret), -ret);
        }
    }

    if (ap->dhcpsettings) {
        memset (dhcprange, 0, sizeof (dhcprange));
        DUMP_DATA_TO_KEYFILE (boolean, keyfile, DHCP_SETTINGS_GROUP_NAME, "DNSSupport", ap->dhcpsettings->dnssupport);
        if (ap->dhcpsettings->poolsize != -1)
            DUMP_DATA_TO_KEYFILE (integer, keyfile, DHCP_SETTINGS_GROUP_NAME, "PoolSize", ap->dhcpsettings->poolsize);
        if (ap->dhcpsettings->leaseperiod != 0)
            DUMP_DATA_TO_KEYFILE (integer, keyfile, DHCP_SETTINGS_GROUP_NAME, "LeasePeriod", (int) ap->dhcpsettings->leaseperiod);
        if (ap->dhcpsettings->gateway != 0)
            DUMP_DATA_TO_KEYFILE (string, keyfile, DHCP_SETTINGS_GROUP_NAME, "Gateway", ap->dhcpsettings->gateway);

        access_point_dump_servers (ap->dhcpsettings->nameservers, DHCP_SETTINGS_GROUP_NAME, "NameServers", keyfile);
        access_point_dump_servers (ap->dhcpsettings->ntpservers, DHCP_SETTINGS_GROUP_NAME, "NtpServers", keyfile);

        if (ap->dhcpsettings->start) {
               strncat (dhcprange, ap->dhcpsettings->start, sizeof (dhcprange) - strlen (dhcprange) - 1);
               if (ap->dhcpsettings->end) {
                   strncat (dhcprange, ",", sizeof (dhcprange) - strlen (dhcprange) - 1);
                   strncat (dhcprange, ap->dhcpsettings->end, sizeof (dhcprange)- strlen (dhcprange) - 1);
               }
            DUMP_DATA_TO_KEYFILE (string, keyfile, DHCP_SETTINGS_GROUP_NAME, "DHCPRange", dhcprange);
        }
    }

    if (ap->tethering_device)
        DUMP_DATA_TO_KEYFILE (string, keyfile, TETHERING_SETTINGS_GROUP_NAME, "Interface",
                              ap->tethering_device);
    if (g_key_file_has_key (keyfile, TETHERING_SETTINGS_GROUP_NAME, "TetheringDevice", NULL))
        REMOVE_DATA_FROM_KEYFILE (keyfile, TETHERING_SETTINGS_GROUP_NAME, "TetheringDevice", NULL);
    DUMP_DATA_TO_KEYFILE (boolean, keyfile, TETHERING_SETTINGS_GROUP_NAME, "Enable", ap->tethering);

    return 0;
}

static int
access_point_save (struct access_point *ap)
{
    int ret;
    GKeyFile *keyfile = NULL;

    return_val_if_fail (ap, -EINVAL);

    DEBUG ("Saving the Access point parameters: %s", ap->interface);

    ret = db_load (ap, &keyfile);
    if (ret < 0) {
        ERROR ("Failure in loading the configuration file for ap: %s [error : %s/%d]",
               ap->interface, strerror (-ret), -ret);
        if (ret != -ENOENT)
            return -EIO;
        keyfile = g_key_file_new ();
        return_val_if_fail (keyfile, -EIO);
    }

    ret = access_point_dump_keyfile (ap, ap->interface, keyfile);
    if (ret < 0) {
        ERROR ("Failed to load the keyfile for storing the ap [\"%s\"]: %s",
               ap->interface, strerror (-ret));
        g_key_file_free (keyfile);
        return ret;
    }

    ret = db_save_ap_conf (ap, keyfile);
    if (ret < 0)
        ERROR ("Failed to save the configuration for ap: %s", ap->interface);

    g_key_file_free (keyfile);
    return ret;
}

int
access_point_set_ssid (struct access_point *ap,
                       char *ssidval,
                       size_t length)
{
    // TODO: Need to check the possibility of implementing this function
    // similar to "access_point_set_passphrase" ie.,without length parameter.
    int ret = 0;

    return_val_if_fail (ap && ssidval && (length != 0), -EINVAL);

    if (ap->ssid) {
        if((length == ap->ssid->length) && (!memcmp(ssidval, ap->ssid->ssid, length)))
            return -EPERM;
    }
    else {
        ap->ssid = g_try_malloc0 (sizeof (*ap->ssid));
        return_val_if_fail (ap->ssid, -ENOMEM);
        ap->ssid->ssid = NULL;
    }

    ap->ssid->length = 0;
    g_free (ap->ssid->ssid);

    ap->ssid->ssid = g_try_malloc0 (sizeof (*ap->ssid->ssid) * length);
    return_val_if_fail (ap->ssid->ssid, -ENOMEM);

    memcpy (ap->ssid->ssid, ssidval, length);
    ap->ssid->length = length;

    ret = access_point_save (ap);

    if (ret < 0)
        ERROR ("Failed to store the AP credentials : %s", strerror (-ret));

    access_point_update_property (ap->dbusservice, "SSID");
    return ret;
}

static void
access_point_set_target_powerstate (struct access_point *ap,
                                    gboolean powered)
{
    gboolean old;

    return_if_fail (ap);

    old = ap->targetpowerstate;
    if (ap->targetpowerstate == powered)
        return;

    INFO ("old: %s, new: %s", old ? "ON" : "OFF",
          powered ? "ON" : "OFF");

    ap->targetpowerstate = powered;
}

int
access_point_set_powered (struct access_point *ap,
                          const gboolean powered)
{
    int ret = 0;

    return_val_if_fail (ap, -EINVAL);

    if (ap->actualpowerstate == powered)
        return -EALREADY;

    ap->actualpowerstate = powered;
    access_point_update_property (ap->dbusservice, "Powered");

    if (ap->actualpowerstate == ap->targetpowerstate) {
        ret = access_point_save (ap);
        if (ret < 0)
            ERROR ("Failed to store the AP credentials: %s/%d",
                    strerror (-ret), -ret);
    }

    return ret;
}

int
access_point_set_vendor_elements (struct access_point *ap,
                                  const char *vendor_elements,
                                  const unsigned long length)
{
    char *vendorie = NULL;

    return_val_if_fail (ap, -EINVAL);
    if (length)
        return_val_if_fail (vendor_elements, -EINVAL);

    g_free (ap->vendor_elements);
    ap->vendor_elements = NULL;

    g_free (ap->assocresp_elements);
    ap->assocresp_elements = NULL;

    if (length) {
        vendorie = byte_to_hexstring (vendor_elements, length);
        if (vendorie)
            ap->vendor_elements = vendorie;
        else {
            ERROR ("Invalid vendor elements i.e., failed to convert to a hex string");
            return -EINVAL;
        }
    }

    return 0;
}

int
access_point_set_passphrase (struct access_point *ap,
                             const char *passphrase)
{
    int ret;

    return_val_if_fail (ap && passphrase, -EINVAL);

    if (ap->passphrase && !g_strcmp0 (passphrase, ap->passphrase))
        return -EALREADY;

    g_free (ap->passphrase);
    ap->passphrase = g_strdup (passphrase);

    ret = access_point_save (ap);
    if (ret < 0)
        ERROR ("Failed to store the AP credentials : %s", strerror (-ret));

    access_point_update_property (ap->dbusservice, "Passphrase");
    return 0;
}

int access_point_set_country (struct access_point *ap,
                              const char *country)
{
    int ret;

    return_val_if_fail (ap && country, -EINVAL);

    DEBUG ("RegDom current: %s changed: %s", ap->country_code, country);
    if (!g_strcmp0 (ap->country_code, country))
        return -EALREADY;

    memcpy (ap->country_code, country, sizeof (ap->country_code) - 1);
    ret = access_point_save (ap);
    if (ret < 0)
        ERROR ("Failed to store the AP credentials: %s", strerror (-ret));

    ret = access_point_update_property (ap->dbusservice, "CountryCode");
    if (ret < 0)
        ERROR ("Failed to update the property: %s", "CountryCode");

    return ret;
}

int
access_point_set_hwmode (struct access_point *ap,
                         const char *mode,
                         gboolean save)
{
    int temp = 0, ret;

    return_val_if_fail (ap && mode, -EINVAL);

    if (!g_strcmp0 (mode, "a"))
        temp |= IEEE80211_MODE_A;
    else if (!g_strcmp0 (mode, "b"))
        temp |= IEEE80211_MODE_B;
    else if (!g_strcmp0 (mode, "g"))
        temp |= IEEE80211_MODE_G;
    else if (!g_strcmp0 (mode, "ad"))
        temp |= IEEE80211_MODE_AD;
    else {
        return -EINVAL;
    }

    DEBUG ("Actual: %x target: %x", ap->current_operating_hwmode, temp);

    if (ap->current_operating_hwmode == temp)
        return -EEXIST;

    ap->current_operating_hwmode = temp;
    ret = access_point_update_property (ap->dbusservice, "CurrentHwMode");
    if (ret < 0)
        ERROR ("Failed to update the property : %s", "CurrentHwMode");

    if (!save)
        return 0;

    ret = access_point_save (ap);
    if (ret < 0)
        ERROR ("Failed to store the HW mode : %s", strerror (-ret));
    return ret;
}

int
access_point_get_current_hwmode (struct access_point *ap,
                                 char **mode)
{
    char *temp = NULL;

    return_val_if_fail (ap && mode, -EINVAL);
    return_val_if_fail (!*mode, -EALREADY);

    switch (ap->current_operating_hwmode) {
    case IEEE80211_MODE_A:
        temp = "Mode A";
        break;
    case IEEE80211_MODE_B:
        temp = "Mode B";
        break;
    case IEEE80211_MODE_AD:
        temp = "Mode AD";
        break;
    default:
    case IEEE80211_MODE_G:
        temp = "Mode G";
        break;
    }

    DEBUG ("Current Operating HW mode : %s", temp);
    *mode = g_strdup (temp);
    return 0;
}

int
access_point_get_hwmode (struct access_point *ap,
                         char **mode)
{
    char *temp = NULL;

    return_val_if_fail (ap && mode, -EINVAL);
    return_val_if_fail (!*mode, -EALREADY);

    switch (ap->current_operating_hwmode) {
    case IEEE80211_MODE_A:
        temp = "a";
        break;
    case IEEE80211_MODE_B:
        temp = "b";
        break;
    case IEEE80211_MODE_AD:
        temp = "ad";
        break;
    default:
    case IEEE80211_MODE_G:
        temp = "g";
        break;
    }

    DEBUG ("Current Operating HW mode of ap: %s [%s]", ap->interface, temp);
    *mode = g_strdup (temp);
    return 0;
}

char*
access_point_get_path (struct access_point *ap)
{
    return_val_if_fail (ap, NULL);
    return access_point_get_objpath (ap->dbusservice);
}

unsigned int
access_point_get_dhcp_pools (struct access_point *ap)
{
    unsigned int size = 0;

    return_val_if_fail (ap, 0);
    if (ap->dhcpsettings && ap->dhcpsettings->server)
        size = 1;

    return size;
}

int
access_point_get_dhcp_data_from_index (struct access_point *ap,
                                       unsigned int index,
                                       const char* key,
                                       void **data)
{
    void *value = NULL;
    struct ippool *pool;

    return_val_if_fail (ap && data, -EINVAL);
    return_val_if_fail (!index, -ENOENT);
    return_val_if_fail (!*data, -EEXIST);

    if (!ap->dhcpsettings ||
            !ap->dhcpsettings->server)
        return -ENODATA;

    pool = dhcp_server_get_ippool (ap->dhcpsettings->server);
    return_val_if_fail (pool, -ENODATA);

    if (g_str_equal (key, "start"))
        value = ippool_get_start (pool);
    else if (g_str_equal (key, "router"))
        value = ippool_get_gateway (pool);
    else if (g_str_equal (key, "end"))
        value = ippool_get_end (pool);
    else if (g_str_equal (key, "netmask"))
        value = ippool_get_subnet (pool);
    else if (g_str_equal (key, "network"))
        value = ippool_get_network (pool);
    else if (g_str_equal (key, "broadcast"))
        value = ippool_get_broadcast (pool);
    else if (g_str_equal (key, "leasetime"))
        value = GUINT_TO_POINTER (ap->dhcpsettings->leaseperiod);

    *data = value;
    return 0;
}

gboolean
access_point_get_tethering (struct access_point *ap)
{
    return_val_if_fail (ap, FALSE);
    return ap->tethering;
}

int
access_point_get_all (GVariant **variant)
{
    int ret = 0;
    GHashTableIter iter;
    gpointer key, value;
    struct access_point *ap;
    char *objectpath = NULL;
    GVariantBuilder *builder, *propbuilder;

    return_val_if_fail (variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    builder = g_variant_builder_new (G_VARIANT_TYPE ("a(oa{sv})"));

    g_hash_table_iter_init (&iter, access_points);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        INFO ("Access Point: %s", (char *) key);
        ap = value;
        if (ap) {
            objectpath = access_point_get_objpath (ap->dbusservice);
            propbuilder = g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));
            ret = access_point_get_properties (ap->dbusservice, propbuilder);
            if (!ret) {
                g_variant_builder_add (builder, "(oa{sv})", objectpath, propbuilder);
                g_variant_builder_unref (propbuilder);
            } else {
                ERROR ("Error occured while acquiring the properties for the ap: %s", ap->interface);
                g_variant_builder_unref (propbuilder);
            }
        }
    }

    *variant = g_variant_builder_end (builder);
    g_variant_builder_unref (builder);

    return ret;
}

static struct access_point*
access_point_get (void *service)
{
    GHashTableIter iter;
    gpointer key, value;
    struct access_point *ap;

    return_val_if_fail (service, NULL);
    g_hash_table_iter_init (&iter, access_points);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        ap = value;
        if (ap && (service == ap->dbusservice))
            return ap;
    }

    return NULL;
}

static void
append_station (gpointer key,
                gpointer value,
                gpointer data)
{
    (void) key;

    char *path;
    int ret;
    GVariantBuilder *stations;
    GVariantBuilder *builder = data;
    struct associated_station* station = value;

    path = associated_station_get_objectpath (station);
    stations = g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));

    ret = associated_station_get_properties (station, stations);
    if (ret < 0)
        ERROR ("Failed to get the properties for station : %s",
               associated_station_get_macaddress (station));

    g_variant_builder_add (builder, "(oa{sv})", path, stations);
    g_variant_builder_unref (stations);
}

int
access_point_get_associatedstations (void *service,
                                     GVariantBuilder *builder)
{
    struct access_point *ap;

    return_val_if_fail (builder, -EINVAL);
    ap = access_point_get (service);
    return_val_if_fail (ap, -ENOENT);
    g_hash_table_foreach (ap->stations, append_station, builder);
    return 0;
}

int
access_point_set_hidden (struct access_point *ap,
                         const gboolean hidden)
{
    int ret;
    gboolean current = FALSE;

    return_val_if_fail (ap, -EINVAL);

    current = (ap->ignore_broadcast_ssid != BROADCAST_SSID) ? TRUE : FALSE;
    if (current == hidden)
        return -EALREADY;

    ap->ignore_broadcast_ssid = hidden ? BROADCAST_EMPTY_SSID : BROADCAST_SSID;
    ret = access_point_save (ap);
    if (ret < 0)
        ERROR ("Failed to store the AP credentials : %s", strerror (-ret));

    access_point_update_property (ap->dbusservice, "Hidden");
    return 0;
}

int
access_point_set_tethering (struct access_point *ap,
                            const gboolean tethering)
{
    int ret;

    return_val_if_fail (ap, -EINVAL);

    if (ap->tethering == tethering)
        return -EALREADY;

    ap->tethering = tethering;
    ret = access_point_save (ap);
    if (ret < 0)
        ERROR ("Failed to store the AP credentials: %s/%d",
               strerror (-ret), -ret);

    access_point_update_property (ap->dbusservice, "Tethering");
    return 0;
}

static int
access_point_set_preferred_channels (struct access_point *ap,
                                     int *channels,
                                     size_t length)
{
    unsigned i = 0;
    channel *prechannels = NULL;

    DEBUG("access point: %s length: %u", ap->interface, length);

    return_val_if_fail (ap && channels, -EINVAL);

    if (!length) {
        access_point_channel_cleanup (ap->preferred_channels);
        ap->preferred_channels = NULL;
        return -ENODATA;
    }

    prechannels = g_try_malloc0 (sizeof (*prechannels));
    if (!prechannels)
        return -ENOMEM;

    prechannels->length = (unsigned int) length;
    prechannels->data = g_try_malloc0 (sizeof (*prechannels->data) * length);
    if (!prechannels->data) {
        g_free (prechannels);
        return -ENOMEM;
    }

    for ( ; i < length; i++)
        prechannels->data [i] = (unsigned int) channels [i];

    access_point_channel_cleanup (ap->preferred_channels);
    ap->preferred_channels = prechannels;
    return 0;
}

int
access_point_set_operating_channel (struct access_point *ap,
                                    const unsigned int channel)
{
    return_val_if_fail (ap, -EINVAL);

    if (ap->operating_channel == channel)
        return -EALREADY;

    ap->operating_channel = channel;
    access_point_update_property (ap->dbusservice, "CurrentOperatingChannel");
    return 0;
}

int
access_point_set_channels (struct access_point *ap,
                           unsigned int *channels,
                           unsigned int length)
{
    int ret, update = 0;
    channel *prechannels;
    unsigned int *preferredchans, index = 0;
    char *curhwmode = NULL;

    return_val_if_fail (ap && channels, -EINVAL);

    if (length <= 0)
        return -EINVAL;

    ret = access_point_get_hwmode (ap, &curhwmode);
    if (ret < 0) {
    	ERROR ("Failed to get the current HWMode: %s/%d", strerror (-ret), -ret);
    	return ret;
    }

    ret = channels_is_5ghz (channels, length);
    if (!ret) {
    	if (g_strcmp0 (curhwmode, "a"))
    		access_point_set_hwmode (ap, "a", FALSE);
    }
    else if (ret == -1) {
    	if (g_strcmp0 (curhwmode, "g"))
    		access_point_set_hwmode (ap, "g", FALSE);
    }

    g_free (curhwmode);

    if (length == 1) {
        ap->acs = 0;
        ap->operating_channel = channels [0];
        update = 1;
    } else {

        preferredchans = g_try_malloc0 (sizeof (*preferredchans) * length);
        return_val_if_fail (preferredchans, -ENOMEM);

        for ( ; index < length; index++)
            preferredchans [index] = channels [index];

        prechannels = g_try_malloc0 (sizeof (*prechannels));
        if (!prechannels) {
            g_free (preferredchans);
            return -ENOMEM;
        }

        prechannels->data = preferredchans;
        prechannels->length = length;

        access_point_channel_cleanup (ap->preferred_channels);

        ap->acs = 1;
        ap->preferred_channels = prechannels;
    }

    ret = access_point_save (ap);
    if (ret < 0)
        ERROR ("Failed to store the AP credentials : %s", strerror (-ret));

    if (update)
        access_point_update_property (ap->dbusservice, "CurrentOperatingChannel");

    return 0;
}

int
access_point_set_beacon_int (struct access_point *ap,
                             unsigned short int beacon_int)
{
    return_val_if_fail (ap, -EINVAL);
    ap->beacon_int = beacon_int;
    return 0;
}

int
access_point_set_dtim_period (struct access_point *ap,
                              const int dtim_period)
{
    return_val_if_fail (ap, -EINVAL);
    ap->dtim = dtim_period;
    return 0;
}

int
access_point_set_max_num_station (struct access_point *ap,
                                  const int max_num_sta)
{
    int ret;

    return_val_if_fail (ap, -EINVAL);

    if (max_num_sta == ap->max_stations_allowed)
        return -EALREADY;

    ap->max_stations_allowed = max_num_sta;
    ret = access_point_save (ap);
    if (ret < 0)
        ERROR ("Failed to store the AP credentials: %s/%d", strerror (-ret),
               -ret);

    access_point_update_property (ap->dbusservice, "MaximumStationsAllowed");
    return 0;
}

int
access_point_set_acc_net_type (struct access_point *ap,
                                  const unsigned int acc_net_type)
{
    int ret;

    return_val_if_fail (ap, -EINVAL);

    if (acc_net_type == ap->access_network_type)
        return -EALREADY;

    if(ap->interworking)
        ap->access_network_type = acc_net_type;

    ret = access_point_save (ap);
    if (ret < 0)
        ERROR ("Failed to store the AP credentials: %s/%d", strerror (-ret),
               -ret);

    access_point_update_property (ap->dbusservice, "AccessNetworkType");
    return 0;
}

char*
access_point_get_address (struct access_point *ap)
{
    return_val_if_fail (ap, NULL);
    return ap->address;
}

char*
access_point_get_interface (struct access_point *ap)
{
    return_val_if_fail (ap, NULL);
    return ap->interface;
}

gboolean
access_point_get_powered (struct access_point *ap)
{
    return_val_if_fail (ap, FALSE);
    return ap->actualpowerstate;
}

gboolean
access_point_get_hidden (struct access_point *ap)
{
    return_val_if_fail (ap, FALSE);
    return ap->ignore_broadcast_ssid ? TRUE : FALSE;
}

const char*
access_point_get_security (struct access_point *ap)
{
    securitytype type;

    return_val_if_fail (ap, NULL);

    type = ap->security;
    switch (type) {
    case AP_SECURITY_OPEN:
        return "open";
    case AP_SECURITY_WEP:
        return "wep";
    case AP_SECURITY_WPA_PSK:
        return "wpa-psk";
    case AP_SECURITY_WPA2_PSK:
        return "wpa2-psk";  
    case AP_SECURITY_WPA_WPA2_PSK:
	default:
        return "wpa/wpa2-psk";
    }
}

char*
access_point_get_passphrase (struct access_point *ap)
{
    return_val_if_fail (ap, NULL);
    return (ap->passphrase != NULL) ? ap->passphrase : "";
}

unsigned int*
access_point_get_supported_channels (struct access_point *ap,
                                     unsigned int *length)
{
    return_val_if_fail (ap && length && ap->supported_channels, NULL);
    *length = ap->supported_channels->length;
    return ap->supported_channels->data;
}

unsigned int*
access_point_get_supported_frequencies (struct access_point *ap,
                                        unsigned int *length)
{
    return_val_if_fail (ap && length && ap->supported_frequencies, NULL);
    *length = ap->supported_frequencies->length;
    return ap->supported_frequencies->data;
}

unsigned int
access_point_get_hwmodes (struct access_point *ap)
{
    return_val_if_fail (ap, IEEE80211_MODE_MAX);
    return ap->supported_hwmodes;
}

int
access_point_get_maxstations (struct access_point *ap)
{
    return_val_if_fail (ap, -1);
    return ap->max_stations_allowed;
}

int
access_point_get_max_allowed_stations (struct access_point *ap)
{
	return_val_if_fail (ap, -1);
	return MAX_STATIONS_ALLOWED;
}

int
access_point_get_ssid (struct access_point *ap,
                       char *ssid,
                       size_t *length)
{
    return_val_if_fail (ap && ssid && length && ap->ssid, -EINVAL);
    memcpy (ssid, ap->ssid->ssid, ap->ssid->length);
    *length = ap->ssid->length;
    return 0;
}


char*
access_point_get_country (struct access_point *ap)
{
    return_val_if_fail (ap, NULL);
    return ap->country_code;
}

unsigned int
access_point_get_accnettype (struct access_point *ap)
{
    return_val_if_fail (ap, 0);
    return ap->access_network_type;
}

unsigned int
access_point_get_operating_channel (struct access_point *ap)
{
    return_val_if_fail (ap, 0);
    return ap->operating_channel;
}

static int
access_point_reconfigure_channels (struct access_point *ap)
{
    char prechannels [PATH_MAX];
    unsigned int operating_chan = 0;
    int ret = 0, temp2_4 [15], temp5 [48];
    size_t length2_4 = 0, length5 = 0;

    return_val_if_fail (ap, -EINVAL);

    memset (temp5, 0, sizeof (temp5));
    memset (temp2_4, 0, sizeof (temp2_4));
    memset (prechannels, 0, sizeof (prechannels));

    if (ap->acs) {
        ret = access_point_get_preferred_channels (ap, prechannels);
        return_val_if_fail (ret == 0, -EINVAL);

        DEBUG ("Preferred channel list: %s", prechannels);
        ret = validate_preferred_channels (ap, temp2_4, temp5, &length2_4,
                                           &length5, prechannels);
        return_val_if_fail (ret == 0, -EINVAL);
        ret = access_point_set_preferred_channels (ap,
                (ap->current_operating_hwmode !=
                IEEE80211_MODE_A) ? temp5 : temp2_4,
                (ap->current_operating_hwmode !=
                IEEE80211_MODE_A) ? length5 : length2_4);
        if (ret != 0) {
            ap->acs = 0;
            (void) access_point_set_operating_channel (ap, operating_chan);
            access_point_set_hwmode (ap, "g", FALSE);
        }
    } else {
        if (is_valid_channel (ap, (int) ap->operating_channel) != 0) {
            ret = access_point_get_first_channel (ap,
                    ap->current_operating_hwmode == IEEE80211_MODE_A ?
                    TRUE : FALSE, &operating_chan);
            (void) access_point_set_operating_channel (ap, operating_chan);
            if (ret != 0)
                access_point_set_hwmode (ap, "g", FALSE);
        }
    }

    return ret;
}

#if 0
static int
access_point_load_blacklisted_stations (struct access_point *ap,
                                        char *deny_file_path)
{
    int ret;
    FILE *fp;
    char station [18];

    return_val_if_fail (ap, -EINVAL);
    return_val_if_fail (deny_file_path, -EINVAL);

    fp = fopen (deny_file_path, "r");
    if (!fp) {
        ERROR ("Failed to load blacklisted stations, file %s "
               "[error : %s]", deny_file_path, strerror (errno));
        return -errno;
    }

    memset (station, 0 , sizeof (station));
    while (fgets (station, sizeof (station), fp)) {
        INFO ("Station \"%s\" is blacklisted", station);
        ap->blacklisted_stations = g_list_append (ap->blacklisted_stations,
                                                  g_strdup (station));
        memset (station, 0 , sizeof (station));
    }

    ret = fclose (fp);
    if (ret < 0)
        ERROR ("Failed to close \"%s\", error : %s", deny_file_path,
               strerror (errno));

    ap->deny_mac_file = g_strdup (deny_file_path);
    return ret;
}
#endif

struct access_point*
get_apfrom_iface (const char *ifname)
{
    GHashTableIter iter;
    gpointer key, value;

    return_val_if_fail (ifname, NULL);
    return_val_if_fail (access_points, NULL);

    g_hash_table_iter_init (&iter, access_points);
    while (g_hash_table_iter_next (&iter, &key, &value))
        if (!g_strcmp0 (ifname, (char *)key))
            return value;

    return NULL;
}

static void
ap_request_cb (int result,
               char *ifname,
               void *userdata)
{
    int completed = 0, ret, opcode;
    struct access_point *ap;
    aptask taskid = TASK_AP_INVALID;
    struct wapdman_task *task = userdata;
    aptask nexttask;

    return_if_fail (ifname && task);

    DEBUG ("Accesspoint: \"%s\" task: \"%p\" id: \"%s\" result: \"%d\"",
           ifname, task, ap_task2string ((aptask) task_get_opcode (task)),
           result);

    if (result < 0)
        result = -ERROR_INTERNAL_FAILURE;

    ap = get_apfrom_iface (ifname);
    if (!ap) {
        CRITICAL ("No AP instance found for the AP: %s", ifname);
        return;
    }

    taskid = (aptask) task_get_opcode (task);
    switch (taskid) {

    case TASK_AP_SET_POWERED:
    {
        switch (ap->state) {

        case WAPDMAN_AP_STATE_CONFIGURING_HOSTAPD:
        {
            ap_add_interface_cb (result, ifname, ap);
            if (ap->state != WAPDMAN_AP_STATE_CONFIGURED_NAT)
                result = -ERROR_INTERNAL_FAILURE;
            completed = 1;
        }
        break;
        default:
        {
            opcode = task_get_next_subtask (task);
            if (result < 0 || opcode < 0) {
                if (result < 0)
                    ERROR ("Failed to configure the access point: %s",
                           ap->interface);
                completed = 1;
                break;
            }

            nexttask = (aptask) opcode;
            INFO ("Task: \"%p\" Next task: %s", task, ap_task2string (nexttask));
            switch (nexttask) {
            case TASK_AP_CONFIGURE_HAPD:
            {
                access_point_set_power_state (ap, POWERING_ON);
                ret = ap_task_configure_hostapd (ap, task);
                if (ret < 0) {
                    completed = 1;
                    result = -ERROR_INTERNAL_FAILURE;
                }
            }
            break;
            default:
                completed = 1;
                break;
            }
        }
        break;
        }
    }
    break;

    case TASK_AP_SET_PASSPHRASE:
    case TASK_AP_SET_VISIBILITY:
    case TASK_AP_SET_SSID:
    case TASK_AP_SET_ACCESS_NETWORK_TYPE:
    {
        opcode = task_get_next_subtask (task);
        if (result < 0 || opcode < 0) {
            if (result < 0)
                ERROR ("Failed to set the requested SSID");
            completed = 1;
            break;
        }

        nexttask = (aptask) opcode;
        switch (nexttask) {
        case TASK_AP_DISABLE_AP:
        {
            ret = accesspoint_disable (ap->interface, ap_request_cb, task);
            if (ret < 0) {
                ERROR ("Task: \"%p\" Failed to disable the AP", task);
                completed = 1;
                result = ret;
                break;
            }
        }
        break;

        case TASK_AP_ENABLE_AP:
        {
            ret = accesspoint_enable (ap->interface, ap_request_cb, task);
            if (ret < 0) {
                ERROR ("Task: \"%p\" Failed to enable the AP", task);
                completed = 1;
                result = ret;
                break;
            }
        }
        break;

        default:
            ERROR ("Invalid task: %s", ap_task2string (nexttask));
            completed = 1;
            break;
        }

        if (!completed) {
            task_set_ongoingoperation (task, nexttask);
            task_rm_subtask (task, nexttask);
        }
    }
    break;

    case TASK_AP_REGISTER_VENDOR_IE:
    case TASK_AP_UNREGISTER_VENDOR_IE:
    {
        opcode = task_get_next_subtask (task);
        if (result < 0 || opcode < 0) {
            if (result < 0)
                ERROR ("Failed to set Vendor IEs in Re(Association) response");
            completed = 1;
            break;
        }
        nexttask = (aptask) opcode;
        if((TASK_AP_REGISTER_ASSOCRESP_IE == nexttask) ||
                (TASK_AP_UNREGISTER_ASSOCRESP_IE == nexttask))
        {
            gboolean isAssocResp = TRUE;
            struct vendor_element *vendorie = NULL;
            vendorie = task_get_data (task);
            if (!vendorie) {
                completed = 1;
                break;
            }
            ret = accesspoint_set_vendor_ie (ap->interface, vendorie->ies,
                                             vendorie->length, isAssocResp, ap_request_cb,
                                             task);
            if (ret < 0) {
                ERROR ("Failed to request set vendor elements in Re(Association) response: %s/%d [%p]",
                       accesspoint_strerror (ret), -ret, task);
                completed = 1;
                result = ret;
                break;
            }
        }
        else{
            ERROR ("Invalid task: %s", ap_task2string (nexttask));
            completed = 1;
            break;
        }

        if (!completed) {
            task_set_ongoingoperation (task, nexttask);
            task_rm_subtask (task, nexttask);
        }
    }
    break;


    case TASK_AP_SET_OPERATING_CHANNELS:
    {
        struct set_operating_channels *channels;

        opcode = task_get_next_subtask (task);
        if (result < 0 || opcode < 0) {
            if (result < 0)
                ERROR ("Failed to set the requested opearting channels");
            completed = 1;
            break;
        }

        nexttask = (aptask) opcode;
        switch (nexttask) {
        case TASK_AP_SET_OPERATING_CHANNELS:
        {
            channels = task_get_data (task);

            if (channels->hwmode) {
                ret = access_point_set_hwmode (ap, channels->hwmode, FALSE);
                if (ret < 0 && ret != -EEXIST)
                    ERROR ("Failed to update the hwmode to : %s [ap : %s]",
                           channels->hwmode, ap->interface);
                g_free (channels->hwmode);
            }

            ret = accesspoint_set_operating_channel (ap->interface, channels->channels,
                                                     channels->length, ap_request_cb, task);
            if (ret < 0) {
                ERROR ("Failed to set the requested opearting channels");
                completed = 1;
                result = ret;
            }
        }
        break;

        case TASK_AP_DISABLE_AP:
        {
            ret = accesspoint_disable (ap->interface, ap_request_cb, task);
            if (ret < 0) {
                ERROR ("Task: \"%p\" Failed to disable the AP", task);
                completed = 1;
                result = ret;
                break;
            }
        }
        break;

        case TASK_AP_ENABLE_AP:
        {
            ret = accesspoint_enable (ap->interface, ap_request_cb, task);
            if (ret < 0) {
                ERROR ("Task: \"%p\" Failed to enable the AP", task);
                completed = 1;
                result = ret;
                break;
            }
        }
        break;

        default:
            ERROR ("Invalid task: %s", ap_task2string (nexttask));
            completed = 1;
            break;
        }

        if (!completed) {
            task_set_ongoingoperation (task, nexttask);
            task_rm_subtask (task, nexttask);
        }
    }
    break;

    case TASK_AP_CLEAR_DATA:
    {
        completed = 1;
        if (result < 0) {
            set_power_failure_reason (ap, WAPDMAN_PFRC_RM_HAPD_FAILURE);
            set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
            break;
        }

        (void) access_point_set_ssid (ap, "" , 1);
        (void) access_point_set_passphrase (ap, "");
        set_ap_state (ap, WAPDMAN_AP_STATE_REMOVED_HOSTAPD);
    }
    break;

    case TASK_AP_RESTORE_DEFAULT_PRIVATE_DATA:
    {
        completed = 1;
        if (result < 0) {
            set_power_failure_reason (ap, WAPDMAN_PFRC_RM_HAPD_FAILURE);
            set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
            break;
        }

        char *defSSID = NULL, *defPassphrase = NULL;
        GKeyFile *conffile = NULL;
        int ret = access_point_load_defaults (ap, &conffile, NULL);
        if (ret < 0) {
            ERROR ("Failed to load the configuration file: %s/%d",
                   strerror (-ret), -ret);
        }
        else {
            defSSID = g_key_file_get_string(conffile, ap->interface, "SSID", NULL);
            defPassphrase = g_key_file_get_string(conffile, ap->interface, "Passphrase", NULL);
            DEBUG ("Def SSID: %s, Def Passphrase: %s", defSSID, defPassphrase);

            g_key_file_free (conffile);
        }

        if (defSSID) {
           (void) access_point_set_ssid (ap, defSSID , strlen(defSSID));
           free (defSSID);
        }
        else {
           (void) access_point_set_ssid (ap, "" , 1);
        }

        if (defPassphrase) {
           (void) access_point_set_passphrase (ap, defPassphrase);
           free (defPassphrase);
        }
        else {
           (void) access_point_set_passphrase (ap, "");
        }

        set_ap_state (ap, WAPDMAN_AP_STATE_REMOVED_HOSTAPD);
    }
    break;

    case TASK_AP_RECONFIGURE_ON_REGDOM_CHANGE:
    {
        completed = 1;
        if (result < 0) {
            set_power_failure_reason (ap, WAPDMAN_PFRC_RM_HAPD_FAILURE);
            set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
            break;
        }

        (void) access_point_reconfigure_channels (ap);
        set_ap_state (ap, WAPDMAN_AP_STATE_REMOVED_HOSTAPD);
    }
    break;

    default:
        completed = 1;
        break;
    }

    if (completed) {
        access_point_handle_results (ap->dbusservice, task_get_invocation (task),
                                     result, task_get_opcode (task), task_get_data (task));
        (void) task_set_state (task, TASK_STATE_COMPLETED);
        (void) task_remove (&ap->taskqueue, task);
        task_cleanup (task);
        ap_task_wakeup_next (ap);
    }
}

static int
ap_task_enable_nat (struct access_point *ap)
{
    int ret;
    unsigned char prefixlen;
    char *address;

    return_val_if_fail (ap->dhcpsettings &&
                        ap->dhcpsettings->server, -ENODATA);

    address = ippool_get_network
            (dhcp_server_get_ippool (ap->dhcpsettings->server));

    prefixlen = calculate_netmask
            (ippool_get_subnet (dhcp_server_get_ippool
                                (ap->dhcpsettings->server)));
    return_val_if_fail (prefixlen != 32, -EINVAL);

    ret = nat_enable (ap->interface, ap->tethering_device,
                      address,
                      ap->dhcpsettings->gateway,
                      prefixlen);

    if (ret < 0 && ret != -EALREADY) {
        ERROR ("Failed to enable NAT on the AP interface \"%s\": %s/%d",
               ap->interface, strerror (-ret), -ret);
    }

    return ret;
}

static int
ap_task_configure_hostapd (struct access_point *ap,
                           struct wapdman_task *task)
{
    int ret;
    PowerFailureReason eReason;
    unsigned int acs_channels = 0;

    ret = access_point_create_conf_file (ap);
    if (ret < 0) {
        eReason = WAPDMAN_PFRC_INVAL_CONF;
        goto failure;
    }

    if (ap->acs && ap->preferred_channels)
        acs_channels = ap->preferred_channels->length;

    ret = accesspoint_add_interface (ap->interface, ap->tempfile,
                                     ap->acs,
                                     (int) acs_channels,
                                     ap_request_cb, task);
    if (ret < 0) {
        ERROR ("Failed to add \"%s\" to hostapd: %s/%d",
               ap->interface, strerror (-ret), -ret);
        unlink (ap->tempfile);
        g_free (ap->tempfile);
        ap->tempfile = NULL;
        eReason = WAPDMAN_PFRC_CONF_HAPD_FAILURE;
        goto failure;
    }

    set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURING_HOSTAPD);
    task_set_ongoingoperation (task, TASK_AP_CONFIGURE_HAPD);
    task_set_state (task, TASK_STATE_REQUESTED);

    return ret;

failure:
    access_point_set_power_state (ap, POWER_FAILURE);
    set_power_failure_reason (ap, eReason);

    if(WAPDMAN_AP_STATE_INITIALIZED == ap->state ||
            WAPDMAN_AP_STATE_CONFIGURED_REGDOM == ap->state)
        set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);

    return ret;
}

static int
ap_configure_nat (struct access_point *ap)
{
    unsigned char prefixlen;
    int ret;
    char *address;

    return_val_if_fail (ap, -EINVAL);

    prefixlen = calculate_netmask
            (ippool_get_subnet (dhcp_server_get_ippool
                                (ap->dhcpsettings->server)));
    if (prefixlen == 32) {
        /*
         * This should not happen as we validate everything
         * with respect to the ippool in the DHCP server handling
         * i.e., we should have a valid pool */
        set_power_failure_reason (ap, WAPDMAN_PFRC_NAT_INVALID_CONF);
        return -EINVAL;
    }

    address = (char *) ippool_get_network
            (dhcp_server_get_ippool (ap->dhcpsettings->server));
    set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURING_NAT);
    ret = nat_enable (ap->interface, ap->tethering_device,
                      address,
                      ap->dhcpsettings->gateway,
                      prefixlen);
    if (ret < 0) {
        if (ret == -EALREADY)
            return 0;
        set_power_failure_reason (ap, WAPDMAN_PFRC_NAT_FAILED_START);
    }

    return ret;
}

static int
ap_task_wakeup_next (struct access_point *ap)
{
    aptask taskid;
    int ret, failed = 0;
    char *station;
    gboolean isbusy = FALSE;
    struct wapdman_task *task;
    struct set_ssid_data *data;
    struct vendor_element *vendorie;

    return_val_if_fail (ap, -EINVAL);

    isbusy = task_is_ongoing (&ap->taskqueue);
    if (isbusy)
        return 0;

    task = task_process_next (&ap->taskqueue);
    if (!task) {
        DEBUG ("No tasks in the task queue");
        return 0;
    }

    INFO ("Accesspoint: %s Task to be performed next: %p state of the ap: %s",
          ap->interface,
          task, ap_state2string (ap->state));

    taskid = (aptask) task_get_opcode (task);
    switch (taskid) {

    case TASK_AP_SET_COUNTRY_CODE:
    {
        char *country_code = task_get_data (task);
        if (!country_code) {
            failed = 1;
            break;
        }

        ret = accesspoint_set_country_code (ap->interface, country_code,
                                            ap_request_cb, task);
        if (ret < 0) {
            ERROR ("Failed to request Passphrase change: %s [%p]",
                   accesspoint_strerror (ret), task);
            failed = 1;
            break;
        }

        INFO ("Successfully requested regdom change to %s on ap [%s], "
              "task [%p]", country_code, ap->interface, task);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    case TASK_AP_SET_MAX_STATIONS_ALLOWED:
    {
        unsigned int max_num_sta = GPOINTER_TO_UINT (task_get_data (task));
        ret = accesspoint_set_max_num_sta (ap->interface,
                                           max_num_sta,
                                           ap_request_cb, task);
        if (ret < 0) {
            ERROR ("Failed to request max_num_sta change to %u: %s/%d [%p]",
                   max_num_sta, accesspoint_strerror (ret), -ret, task);
            failed = 1;
            break;
        }

        INFO ("Successfully requested a change on max_num_sta to %u on ap [%s], "
              "task [%p]", max_num_sta, ap->interface, task);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    case TASK_AP_SET_TETHERING:
    {
        data = task_get_data (task);
        gboolean target, actual;

        target = GPOINTER_TO_INT (data) ? TRUE : FALSE;
        actual = access_point_get_tethering (ap);

        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);

        if (actual == target) {
            ap_request_cb (0, ap->interface, task);
            break;
        }

        if (target)
            ret = ap_task_enable_nat (ap);
        else { ret = nat_disable (ap->interface, TRUE); }

        ap_request_cb (ret, ap->interface, task);
    }
    break;

    case TASK_AP_SET_POWERED:
    {
        data = task_get_data (task);
        gboolean enable = GPOINTER_TO_INT (data) ? TRUE : FALSE;

        access_point_set_target_powerstate (ap, enable);
        if (ap->actualpowerstate == enable) {
            ap_request_cb (0, ap->interface, task);
            break;
        }

        switch (ap->state) {

        case WAPDMAN_AP_STATE_INITIALIZED:
        case WAPDMAN_AP_STATE_CONFIGURED_REGDOM:
        case WAPDMAN_AP_STATE_REMOVED_HOSTAPD:
        {
            if (!enable) {
                ap_request_cb (0, ap->interface, task);
                break;
            }

            access_point_set_power_state (ap, POWERING_ON);
            ret = ap_task_configure_hostapd (ap, task);
            if (ret < 0)
                failed = 1;
        }
        break;

        case WAPDMAN_AP_STATE_FAILED:
        {
            INFO ("Power failure reason: %s",
                  power_failure_reason2str (ap->pwrfreason));

            /* We dont expect a powered off request here,
             * normally the power off requests should not
             * reach here */
            if (!enable) {
                ap_request_cb (0, ap->interface, task);
                break;
            }

            task_set_state (task, TASK_STATE_REQUESTED);
            switch (ap->pwrfreason) {
            /* We shall not landup in this case because
             * of the default settings */
            case WAPDMAN_PFRC_DHCPDNS_CONF_FAILURE:
            default:
                failed = 1;
                break;

            case WAPDMAN_PFRC_RM_HAPD_FAILURE:
            {
                ret = accesspoint_remove_interface (ap->interface, ap_request_cb, task);
                if (ret < 0) {
                    ERROR("Failed accesspoint_remove_interface: %s/%d [%p]",
                          accesspoint_strerror (ret), -ret, task);
                    failed = 1;
                    break;
                }

                set_ap_state (ap, WAPDMAN_AP_STATE_REMOVING_HOSTAPD);
                task_add_subtask (task, TASK_AP_CONFIGURE_HAPD);
                task_set_ongoingoperation (task, TASK_AP_REMOVE_HAPF_CONF);
                task_set_state (task, TASK_STATE_REQUESTED);
            }
            break;

                /* Give it a try again */
            case WAPDMAN_PFRC_INVAL_CONF:
            case WAPDMAN_PFRC_DB_FAILURE:
            case WAPDMAN_PFRC_HAPD_REJECTED:
            case WAPDMAN_PFRC_CONF_HAPD_FAILURE:
            {
                access_point_set_power_state (ap, POWERING_ON);
                ret = ap_task_configure_hostapd (ap, task);
                if (ret < 0) {
                    failed = 1;
                    set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
                }
            }
            break;

            case WAPDMAN_PFRC_DHCPDNS_FAILED_STARTUP:
            {
                access_point_set_power_state (ap, POWERING_ON);
                set_ap_state (ap, WAPDMAN_AP_STATE_STARTING_DHCP_SERVER);
                ret = dhcpserver_start (ap->dhcpsettings->server,
                                        ap->dhcpsettings->nameservers,
                                        ap->dhcpsettings->ntpservers);
                if (ret < 0) {
                    access_point_set_power_state (ap, POWER_FAILURE);
                    set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
                    failed = 1;
                    break;
                }

                set_ap_state (ap, WAPDMAN_AP_STATE_DHCP_SERVER_STARTED);
            }

                /* by intention, fall through */
            case WAPDMAN_PFRC_NAT_FAILED_START:
            {
                access_point_set_power_state (ap, POWERING_ON);
                set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURING_NAT);
                ret = ap_configure_nat (ap);
                if (ret < 0) {

                    failed = 1;
                    if (ap->pwrfreason == WAPDMAN_PFRC_NAT_FAILED_START) {
                        ret = nat_disable (ap->interface, TRUE);
                        if (ret < 0)
                            ERROR ("Failed to disable the NAT for: %s [%s/%d]",
                                   ap->interface, strerror (-ret), -ret);
                    }
                    access_point_set_power_state (ap, POWER_FAILURE);
                    set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
                    break;
                }

                set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURED_NAT);
                set_power_failure_reason (ap, WAPDMAN_PFRC_NONE);
                access_point_set_power_state (ap, POWERED_ON);
                access_point_set_powered (ap, TRUE);
                ap_request_cb (0, ap->interface, task);
            }
            break;
            }
        }
        break;

        default:
        {
            access_point_set_power_state (ap, enable ?
                                              POWERING_ON : POWERING_OFF);
            if (enable)
                ret = accesspoint_enable (ap->interface, ap_request_cb, task);
            else {
                ret = accesspoint_disable (ap->interface, ap_request_cb, task);
            }

            if (ret < 0) {
                ERROR ("Failed to request \"%s\" [ap : %s] : %s [%p]",
                       enable ? "ENABLE" : "DISABLE",
                       ap->interface, accesspoint_strerror (ret), task);
                access_point_set_power_state (ap, POWER_FAILURE);
                failed = 1;
                break;
            }

            INFO ("Successfully requested to \"%s\" [ap : %s], task [%p]",
                  enable ? "ENABLE" : "DISABLE", ap->interface, task);
            task_set_ongoingoperation (task, taskid);
            task_set_state (task, TASK_STATE_REQUESTED);
        }
        break;

        }
    }
    break;

    case TASK_AP_SET_SSID:
    {
        data = task_get_data (task);
        if (!data) {
            failed = 1;
            break;
        }

        if (ap->actualpowerstate) {
            task_add_subtask (task, TASK_AP_DISABLE_AP);
            task_add_subtask (task, TASK_AP_ENABLE_AP);
        }
        ret = accesspoint_set_ssid (ap->interface, data->ssid, data->length,
        							ap->passphrase, ap_request_cb, task);
        if (ret < 0) {
            ERROR ("Failed to request set ssid: %s/%d [%p]",
                   accesspoint_strerror (ret), -ret, task);
            failed = 1;
            break;
        }

        INFO ("Successfully requested a SSID change on \"%s\", task [%p]",
              ap->interface, task);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    case TASK_AP_SET_VISIBILITY:
    {
        data = task_get_data (task);
        if (ap->actualpowerstate) {
            task_add_subtask (task, TASK_AP_DISABLE_AP);
            task_add_subtask (task, TASK_AP_ENABLE_AP);
        }

        ret = accesspoint_set_hidden (ap->interface, GPOINTER_TO_INT (data),
                                      ap_request_cb, task);
        if (ret < 0) {
            ERROR ("Failed to request set visibility: %s/%d [%p]",
                   accesspoint_strerror (ret), -ret, task);
            failed = 1;
            break;
        }

        INFO ("Successfully requested the visibility request for: %s "
              "task [%p]", ap->interface, task);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    case TASK_AP_DEAUTHENTICATE_STATION:
    {
        station = task_get_data (task);
        if (!station) {
            failed = 1;
            break;
        }

        /* Station is not associated, just fake and
         * wake up next */
        if (is_station_connected (ap, station) < 0) {
            ap_request_cb (0, ap->interface, task);
            break;
        }

        ret = accesspoint_deauthenticate_station (ap->interface, station, ap_request_cb, task);
        if (ret < 0) {
            ERROR ("Failed to request deAuthenticate station: %s [%p]",
                   accesspoint_strerror (ret), task);
            failed = 1;
            break;
        }

        INFO ("Successfully requested to deauthenticate the station [%s] on ap [%s], "
              "task [%p]", ap->interface, station, task);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    case TASK_AP_SET_PASSPHRASE:
    {
        char *passphrase = task_get_data (task);
        if (!passphrase) {
            failed = 1;
            break;
        }

        if (ap->actualpowerstate) {
            task_add_subtask (task, TASK_AP_DISABLE_AP);
            task_add_subtask (task, TASK_AP_ENABLE_AP);
        }

        ret = accesspoint_set_passphrase (ap->interface, passphrase, ap_request_cb, task);
        if (ret < 0) {
            ERROR ("Failed to request Passphrase change: %s/%d [%p]",
                   accesspoint_strerror (ret), -ret, task);
            failed = 1;
            break;
        }

        INFO ("Successfully requested password change [%s] on ap [%s], "
              "task [%p]", ap->interface, passphrase, task);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    case TASK_AP_SET_OPERATING_CHANNELS:
    {
        char *curhwmode = NULL;
        struct set_operating_channels *channels;

        channels = task_get_data (task);
        if (!channels) {
            failed = 1;
            break;
        }

        ret = access_point_get_hwmode (ap, &curhwmode);
        if (ret < 0) {
            ERROR ("Failed to get the current hw mode: %s/%d",
                   strerror (-ret), -ret);
            failed = 1;
            break;
        }

        if (channels->is5ghz) {
            if (g_strcmp0 (curhwmode, "a"))
                channels->hwmode = g_strdup ("a");
        } else {
            if (g_strcmp0 (curhwmode, "g"))
                channels->hwmode = g_strdup ("g");
        }

        if (channels->hwmode) {
            task_add_subtask (task, TASK_AP_SET_OPERATING_CHANNELS);
            task_set_ongoingoperation (task, TASK_AP_SET_HWMODE);
            ret = accesspoint_set_hwmode (ap->interface, channels->hwmode,
                                          ap_request_cb, task);
        } else {
            task_set_ongoingoperation (task, taskid);
            ret = accesspoint_set_operating_channel (ap->interface, channels->channels,
                                                     channels->length, ap_request_cb,
                                                     task);
        }

        g_free (curhwmode);
        if (ret < 0) {
            ERROR ("Failed to set operating channels: %s/%d [%p]",
                   accesspoint_strerror (ret), -ret, task);
            failed = 1;
            break;
        }

        if (ap->actualpowerstate) {
            task_add_subtask (task, TASK_AP_DISABLE_AP);
            task_add_subtask (task, TASK_AP_ENABLE_AP);
        }
        task_set_state (task, TASK_STATE_REQUESTED);
        INFO ("Successfully requested to set the new operating "
              "channels on ap: %s task [%p]", ap->interface, task);
    }
    break;

    case TASK_AP_REGISTER_VENDOR_IE:
    case TASK_AP_UNREGISTER_VENDOR_IE:
    {
        vendorie = task_get_data (task);
        if (!vendorie) {
            failed = 1;
            break;
        }

        gboolean isAssoResp = FALSE;
        if(TASK_AP_REGISTER_VENDOR_IE == taskid)
            task_add_subtask (task, TASK_AP_REGISTER_ASSOCRESP_IE);
        if(TASK_AP_UNREGISTER_VENDOR_IE == taskid)
            task_add_subtask (task, TASK_AP_UNREGISTER_ASSOCRESP_IE);

        ret = accesspoint_set_vendor_ie (ap->interface, vendorie->ies,
                                         vendorie->length, isAssoResp, ap_request_cb,
                                         task);
        if (ret < 0) {
            ERROR ("Failed to request set vendor elements: %s/%d [%p]",
                   accesspoint_strerror (ret), -ret, task);
            failed = 1;
            break;
        }

        INFO ("Successfully requested the vendor element set on \"%s\", "
              "task [%p]", ap->interface, task);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    case TASK_AP_SET_ACCESS_NETWORK_TYPE:
    {
        unsigned int acc_net_type = GPOINTER_TO_UINT (task_get_data (task));
        if (ap->actualpowerstate) {
            task_add_subtask (task, TASK_AP_DISABLE_AP);
            task_add_subtask (task, TASK_AP_ENABLE_AP);
        }
        if(!ap->interworking){
            failed = 1;
            break;
        }

        ret = accesspoint_set_accnettype (ap->interface, acc_net_type,
                                      ap_request_cb, task);
        if (ret < 0) {
            ERROR ("Failed to request set access_network_type: %s/%d [%p]",
                   accesspoint_strerror (ret), -ret, task);
            failed = 1;
            break;
        }

        INFO ("Successfully requested the access_network_type request for: %s "
              "task [%p]", ap->interface, task);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    case TASK_AP_RECONFIGURE_ON_REGDOM_CHANGE:
    case TASK_AP_CLEAR_DATA:
    case TASK_AP_RESTORE_DEFAULT_PRIVATE_DATA:
    {
        if (ap->state != WAPDMAN_AP_STATE_CONFIGURED_NAT) {
            task_set_ongoingoperation (task, taskid);
            task_set_state (task, TASK_STATE_REQUESTED);
            ap_request_cb (0, ap->interface, task);
            break;
        } else {
            ret = accesspoint_remove_interface (ap->interface, ap_request_cb, task);
            if (ret < 0) {
                ERROR("Failed accesspoint_remove_interface: %s/%d [%p]",
                      accesspoint_strerror (ret), -ret, task);
                failed = 1;
                break;
            }
        }
        INFO("Successfully reset on ap: \"%s\" task: \"%p\"", ap->interface, task);
        set_ap_state (ap, WAPDMAN_AP_STATE_REMOVING_HOSTAPD);
        task_set_ongoingoperation (task, taskid);
        task_set_state (task, TASK_STATE_REQUESTED);
    }
    break;

    default:
        break;
    }

    if (failed) {
        ret = task_remove (&ap->taskqueue, task);
        if (ret < 0)
            ERROR ("Failed to remove the task %p from the queue of: \"%s\"",
                   task, ap->interface);
        access_point_handle_results (ap->dbusservice, task_get_invocation (task),
                                     -ERROR_INTERNAL_FAILURE, task_get_opcode (task),
                                     task_get_data (task));
        task_cleanup (task);
        /* process next */
        ap_task_wakeup_next (ap);
    }

    return 0;
}

static int
request_has_same_data (aptask taskid,
                       struct wapdman_task *queuedtask,
                       void *newtaskdata)
{
    int diff = -1;
    void *value = NULL;

    return_val_if_fail (queuedtask && newtaskdata, -EINVAL);

    value = task_get_data (queuedtask);
    switch (taskid) {
    case TASK_AP_UNREGISTER_VENDOR_IE:
    case TASK_AP_REGISTER_VENDOR_IE:
    {
        struct vendor_element *vendorie = value;
        if (vendorie && (vendorie->length == ((struct vendor_element *) newtaskdata)->length) &&
                !memcmp (vendorie->ies, ((struct vendor_element *) newtaskdata)->ies, vendorie->length))
            diff = 0;
    }
    break;

    case TASK_AP_SET_SSID:
    {
        struct set_ssid_data *data = value;
        if (data && data->length == ((struct set_ssid_data *) newtaskdata)->length &&
                !memcmp (data->ssid, ((struct set_ssid_data *) newtaskdata)->ssid, data->length))
            diff = 0;
    }
    break;

    case TASK_AP_DEAUTHENTICATE_STATION:
    {
        char *deauthsta = value;
        if (deauthsta && !g_strcmp0 (deauthsta, (char *) newtaskdata))
            diff = 0;
    }
    break;

    case TASK_AP_SET_POWERED:
    case TASK_AP_SET_TETHERING:
    case TASK_AP_SET_VISIBILITY:
    {
        if (GPOINTER_TO_INT (value) == GPOINTER_TO_INT (newtaskdata))
            diff = 0;
    }
    break;

    case TASK_AP_SET_MAX_STATIONS_ALLOWED:
    {
        if (GPOINTER_TO_UINT (value) == GPOINTER_TO_UINT (newtaskdata))
            diff = 0;
    }
    break;

    case TASK_AP_SET_PASSPHRASE:
    {
        char *passphrase = value;
        if (passphrase && !g_strcmp0 (passphrase, (char *) newtaskdata))
            diff = 0;
    }
    break;

    case TASK_AP_SET_COUNTRY_CODE:
    {
        char *country_code = value;
        if (country_code && !g_strcmp0 (country_code, (char *) newtaskdata))
            diff = 0;
    }
    break;

    case TASK_AP_SET_OPERATING_CHANNELS:
    {
        int match;
        unsigned int len1, len2,
                qchan, rchan, index, iter;
        struct set_operating_channels *queued,
                *requested;

        queued = value;
        requested = newtaskdata;

        if (!queued)
            break;

        index = iter = 0;
        len1 = queued->length;
        len2 = requested->length;

        if (len1 != len2)
            break;

        for ( ; index < len1; index++) {

            match = 0;
            iter = 0;
            qchan = queued->channels [index];

            for ( ; iter < len2; iter++) {
                rchan = requested->channels [iter];
                if (qchan == rchan) {
                    match = 1;
                    break;
                }
            }

            if (!match)
                break;
        }

        if (!match)
            diff = 0;
    }
    break;

    case TASK_AP_SET_ACCESS_NETWORK_TYPE:
    {
        if (GPOINTER_TO_UINT (value) == GPOINTER_TO_UINT (newtaskdata))
            diff = 0;
    }
    break;

    default:
        break;
    }

    return diff;
}

static gboolean
is_allowedto_pileup_requests (aptask taskid)
{
    switch (taskid) {
    default:
        return FALSE;
    case TASK_AP_DEAUTHENTICATE_STATION:
    case TASK_AP_BLACKLIST_STATION:
    case TASK_AP_REGISTER_VENDOR_IE:
    case TASK_AP_UNREGISTER_VENDOR_IE:
    case TASK_AP_REGISTER_ASSOCRESP_IE:
    case TASK_AP_UNREGISTER_ASSOCRESP_IE:
        return TRUE;
    }

    return FALSE;
}

static gboolean
access_point_can_update (aptask task)
{
    switch (task) {
    default:
        return FALSE;
    case TASK_AP_SET_SSID:
    case TASK_AP_BLACKLIST_STATION:
    case TASK_AP_SET_HWMODE:
    case TASK_AP_SET_OPERATING_CHANNELS:
    case TASK_AP_REGISTER_VENDOR_IE:
    case TASK_AP_UNREGISTER_VENDOR_IE:
    case TASK_AP_REGISTER_ASSOCRESP_IE:
    case TASK_AP_UNREGISTER_ASSOCRESP_IE:
    case TASK_AP_SET_VISIBILITY:
    case TASK_AP_SET_SECURITY:
    case TASK_AP_SET_TETHERING:
    case TASK_AP_SET_PASSPHRASE:
    case TASK_AP_SET_BEACON_INTERVAL:
    case TASK_AP_SET_DTIM:
    case TASK_AP_SET_MAX_STATIONS_ALLOWED:
    case TASK_AP_SET_STATION_INACTIVITY_TIMEOUT:
    case TASK_AP_SET_HT:
    case TASK_AP_SET_VHT:
    case TASK_AP_SET_ACCESS_NETWORK_TYPE:
    case TASK_AP_SET_FAST_ROAMING:
    case TASK_AP_SET_UUID:
    case TASK_AP_SET_DEVICENAME:
    case TASK_AP_SET_MANUFACTURER:
    case TASK_AP_SET_MODELNAME:
    case TASK_AP_SET_MODELNUMBER:
    case TASK_AP_SET_SERIALNUMBER:
    case TASK_AP_SET_INTERWORKING:
        return TRUE;
    }

    return FALSE;
}

static void
task_timeout_callback (struct wapdman_task *task,
                       void *data)
{
    int ret;
    struct access_point *ap = data;

    ret = task_remove (&ap->taskqueue, task);
    if (ret < 0)
        ERROR ("Failed to remove the task %p from the task queue of ap: %s",
               task, ap->interface);

    access_point_handle_results (ap->dbusservice, task_get_invocation (task),
                                 -ETIMEDOUT, task_get_opcode (task),
                                 task_get_data (task));
    task_cleanup (task);
}

static int
ap_can_update_internals (struct access_point *ap)
{
    apstate state;
    int can_update = -1;

    return_val_if_fail (ap, -EINVAL);

    state = ap->state;
    if (WAPDMAN_AP_STATE_INITIALIZED == state)
        can_update = 0;
    else if (WAPDMAN_AP_STATE_CONFIGURED_REGDOM == state)
        can_update = 0;
    else if (WAPDMAN_AP_STATE_FAILED == state)
        can_update = 0;
    else if (WAPDMAN_AP_STATE_REMOVING_HOSTAPD == state)
        can_update = 0;
    else if (WAPDMAN_AP_STATE_REMOVED_HOSTAPD == state)
        can_update = 0;

    return can_update;
}

static int
is_valid_ap_state (struct access_point *ap)
{
    int valid = -EINVAL;
    apstate state;

    return_val_if_fail (ap, -EINVAL);

    state = ap->state;
    if (WAPDMAN_AP_STATE_INITIALIZED == state)
        valid = 0;
    else if (WAPDMAN_AP_STATE_CONFIGURED_REGDOM == state)
        valid = 0;
    else if (WAPDMAN_AP_STATE_REMOVED_HOSTAPD == state)
        valid = 0;
    else if (WAPDMAN_AP_STATE_FAILED == state)
        valid = 0;
    return valid;
}

static int
ap_task_get_priority (aptask task)
{
    switch (task) {
    case TASK_AP_RECONFIGURE_ON_REGDOM_CHANGE:
        return G_PRIORITY_HIGH;
    default:
        break;
    }

    return G_PRIORITY_DEFAULT;
}

int
ap_process_request (struct access_point *ap,
                    void *invocation,
                    int id,
                    void *data)
{
    GList *temp;
    int ret, remove;
    struct wapdman_task *task;
    aptask taskid = (aptask) id;
    taskstate state = TASK_STATE_INVALID;

    return_val_if_fail (ap, -EINVAL);

    DEBUG ("Accesspoint: \"%s\" [state: %s] task: \"%s\" data: %p", ap->interface,
           ap_state2string (ap->state), ap_task2string (taskid), data);

    temp = ap->taskqueue;
    for ( ; temp; temp = temp->next) {

        remove = 1;
        task = temp->data;
        if (!task)
            continue;

        if ((aptask) task_get_opcode (task) != taskid)
            continue;

        state = task_get_state (task);
        ret = request_has_same_data (taskid, task, data);
        if (!ret) {
            /* Previous requests with the same data */
            if (state == TASK_STATE_REQUESTED)
                return -EINPROGRESS;
        }

        if (is_allowedto_pileup_requests (taskid))
            remove = 0;

        if (!remove)
            continue;

        if (state == TASK_STATE_REQUESTED)
            continue;

        ret = task_remove (&ap->taskqueue, task);
        if (ret < 0) {
            ERROR ("Failed to remove the task [%p] from the queue", task);
            continue;
        }

        access_point_handle_results (ap->dbusservice, task_get_invocation (task), -ECANCELED,
                                     task_get_opcode (task), task_get_data (task));
        task_cleanup (task);
    }

    /*
     * All the initial required checks has been performed and now the
     * state of the AP must be checked and the task shall be created
     * only if the AP is configured with hostapd. If not update the AP
     * parameters as per the request and therefore same will be configured
     * with hostapd when there is a request to power it on
     * */
    if (ap_can_update_internals (ap) == 0 && access_point_can_update (taskid)) {
        access_point_handle_results (ap->dbusservice, invocation,
                                     is_valid_ap_state (ap) == 0
                                     ? 0 : -ENXIO, taskid, data);
        return 0;
    }

    task = task_create (TASK_TYPE_AP, taskid, ap_task_get_priority (id),
                        invocation, data, ap,
                        task_timeout_callback);
    if (!task) {
        access_point_handle_results (ap->dbusservice, invocation, -ENOMEM, taskid, data);
        return -ENOMEM;
    }

    DEBUG ("A New Task created for: %s [%p]", ap_task2string (taskid), task);

    ret = task_add_toqueue (&ap->taskqueue, task);
    if (ret < 0) {
        ERROR ("Failed to add the task %p to the task queue of ap: %s", task, ap->interface);
        access_point_handle_results (ap->dbusservice, task_get_invocation (task), -ERROR_INTERNAL_FAILURE,
                                     task_get_opcode (task), task_get_data (task));
        task_cleanup (task);
        return -ERROR_INTERNAL_FAILURE;
    }

    if (WAPDMAN_AP_STATE_CONFIGURING_HOSTAPD == ap->state ||
            WAPDMAN_AP_STATE_CONFIGURING_REGDOM == ap->state ||
            ((NL80211_SCAN_STATE_SCAN_STARTED == ap->scanstate) &&
                    (TASK_AP_SET_OPERATING_CHANNELS == taskid ||
                            TASK_AP_SET_POWERED == taskid))) {

        DEBUG ("Currently \"%s\" is in progress, task \"%p\" will be handled "
               "after its completion", NL80211_SCAN_STATE_SCAN_STARTED == ap->scanstate ?
               "scanning" : (WAPDMAN_AP_STATE_CONFIGURING_HOSTAPD == ap->state ?
               "hostapd config " : "RegDom config"), task);
        return 0;
    }

    ret = ap_task_wakeup_next (ap);
    if (ret < 0)
        ERROR ("Failed to process the next request: %s/%d", strerror (-ret), -ret);

    return ret;
}

static int
access_point_load_htcapab (struct access_point *ap,
                           const char *htcapab)
{
    char **keys, **temp;

    return_val_if_fail (ap && htcapab, -EINVAL);

    keys = g_strsplit (htcapab, " ", -1);
    return_val_if_fail (keys, -ENODATA);

    for (temp = keys; *temp; temp++) {

        if (!g_strcmp0 (*temp, "[LDPC]"))
            ap->ht_capab |= HT_CAP_INFO_LDPC_CODING_CAP;
        if (!g_strcmp0 (*temp, "[HT40-]"))
            ap->ht_capab |= HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET_MINUS;
        if (!g_strcmp0 (*temp, "[HT40+]"))
            ap->ht_capab |= HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET_PLUS;
        if (!g_strcmp0 (*temp, "[SMPS-STATIC]")) {
            ap->ht_capab &= ~HT_CAP_INFO_SMPS_MASK;
            ap->ht_capab |= HT_CAP_INFO_SMPS_STATIC;
        }
        if (!g_strcmp0 (*temp, "[SMPS-DYNAMIC]")) {
            ap->ht_capab &= ~HT_CAP_INFO_SMPS_MASK;
            ap->ht_capab |= HT_CAP_INFO_SMPS_DYNAMIC;
        }
        if (!g_strcmp0 (*temp, "[GF]"))
            ap->ht_capab |= HT_CAP_INFO_GREEN_FIELD;
        if (!g_strcmp0 (*temp, "[SHORT-GI-20]"))
            ap->ht_capab |= HT_CAP_INFO_SHORT_GI20MHZ;
        if (!g_strcmp0 (*temp, "[SHORT-GI-40]"))
            ap->ht_capab |= HT_CAP_INFO_SHORT_GI40MHZ;
        if (!g_strcmp0 (*temp, "[TX-STBC]"))
            ap->ht_capab |= HT_CAP_INFO_TX_STBC;
        if (!g_strcmp0 (*temp, "[RX-STBC1]")) {
            ap->ht_capab &= ~HT_CAP_INFO_RX_STBC_MASK;
            ap->ht_capab |= HT_CAP_INFO_RX_STBC_1;
        }
        if (!g_strcmp0 (*temp, "[RX-STBC12]")) {
            ap->ht_capab &= ~HT_CAP_INFO_RX_STBC_MASK;
            ap->ht_capab |= HT_CAP_INFO_RX_STBC_12;
        }
        if (!g_strcmp0 (*temp, "[RX-STBC123]")) {
            ap->ht_capab &= ~HT_CAP_INFO_RX_STBC_MASK;
            ap->ht_capab |= HT_CAP_INFO_RX_STBC_123;
        }
        if (!g_strcmp0 (*temp, "[DELAYED-BA]"))
            ap->ht_capab |= HT_CAP_INFO_DELAYED_BA;
        if (!g_strcmp0 (*temp, "[MAX-AMSDU-7935]"))
            ap->ht_capab |= HT_CAP_INFO_MAX_AMSDU_SIZE;
        if (!g_strcmp0 (*temp, "[DSSS_CCK-40]"))
            ap->ht_capab |= HT_CAP_INFO_DSSS_CCK40MHZ;
        if (!g_strcmp0 (*temp, "[40-INTOLERANT]"))
            ap->ht_capab |= HT_CAP_INFO_40MHZ_INTOLERANT;
        if (!g_strcmp0 (*temp, "[LSIG-TXOP-PROT]"))
            ap->ht_capab |= HT_CAP_INFO_LSIG_TXOP_PROTECT_SUPPORT;
    }

    g_strfreev (keys);
    return 0;
}

#if 0
static int
access_point_validate_wpsconfig_methods (const char *config_methods,
                                         int *methods)
{
    char **temp;
    int err = 0;
    char **keys = NULL;
    unsigned int value = 0;

    return_val_if_fail (config_methods, -EINVAL);
    return_val_if_fail (methods, -EINVAL);

    keys = g_strsplit (config_methods, " ", -1);
    if (!keys)
        return -ENODATA;

    temp = keys;
    while (*temp) {

        if (!g_strcmp0 (*temp, "ethernet"))
            value |= WPS_CONFIG_ETHERNET;
        if (!g_strcmp0 (*temp, "label"))
            value |= WPS_CONFIG_LABEL;
        if (!g_strcmp0 (*temp, "display"))
            value |= WPS_CONFIG_DISPLAY;
        if (!g_strcmp0 (*temp, "ext_nfc_token"))
            value |= WPS_CONFIG_EXT_NFC_TOKEN;
        if (!g_strcmp0 (*temp, "int_nfc_token"))
            value |= WPS_CONFIG_INT_NFC_TOKEN;
        if (!g_strcmp0 (*temp, "nfc_interface"))
            value |= WPS_CONFIG_NFC_INTERFACE;
        if (!g_strcmp0 (*temp, "push_button"))
            value |= WPS_CONFIG_PUSHBUTTON;
        if (!g_strcmp0 (*temp, "keypad"))
            value |= WPS_CONFIG_KEYPAD;
        if (!g_strcmp0 (*temp, "virtual_display"))
            value |= WPS_CONFIG_VIRT_DISPLAY;
        if (!g_strcmp0 (*temp, "physical_display"))
            value |= WPS_CONFIG_PHY_DISPLAY;
        if (!g_strcmp0 (*temp, "virtual_push_button"))
            value |= WPS_CONFIG_VIRT_PUSHBUTTON;
        if (!g_strcmp0 (*temp, "physical_push_button"))
            value |= WPS_CONFIG_PHY_PUSHBUTTON;
        if (!g_strcmp0 (*temp, "p2ps"))
            value |= WPS_CONFIG_P2PS;
        else {
            ERROR ("Invalid config method for WPS : %s",
                   *temp);
            err = -1;
            break;
        }

        temp++;
    }

    if (!err)
        *methods = value;

    g_strfreev (keys);
    return err;
}
#endif

int
is_hexa_decimal (const char *string,
                 size_t length)
{
    char c;
    size_t valid, index;

    return_val_if_fail (string, -EINVAL);

    index = valid = 0;
    for ( ; index < length; index++) {

        valid = 0;
        c = string [index];
        if (c >= '0' && c <= '9')
            valid = 1;
        else if (c >= 'A' && c <= 'F')
            valid = 1;
        else if (c >= 'a' && c <= 'f')
            valid = 1;

        if (!valid)
            break;
    }

    if (!valid)
        return -1;

    return 0;
}

int
is_station_connected (struct access_point *ap,
                      const char *station)
{
    return_val_if_fail (ap && station, -EINVAL);
    return (g_hash_table_lookup (ap->stations, station) != NULL) ?
                0 : -ENOENT;
}

static int
is_valid_channel (struct access_point *ap, int channel)
{
    unsigned int index = 0;

    return_val_if_fail (ap && ap->supported_channels, -EINVAL);

    for ( ; index < ap->supported_channels->length; ++index)
        if (ap->supported_channels->data[index] == (unsigned int) channel)
            return 0;
    return -EINVAL;
}

int
channels_is_5ghz (const unsigned int *channels,
                  unsigned int length)
{
    unsigned int chan, index = 0;
    int invalid = 0;

    return_val_if_fail (channels, -EINVAL);

    for ( ; index < length; index++) {

        chan = channels [index];
        if (chan > 14 && chan <= 165)
            continue;

        invalid = 1;
        break;
    }

    if (invalid)
        return -1;

    return 0;
}

int
check_valid_channels (struct access_point *ap,
                      const u_int16_t *channels,
                      size_t length)
{
    int valid = 0;
    size_t len, index = 0, iter = 0;
    unsigned int chan, match;
    unsigned int *chanls;
    int is2_4 = 0, is_5 = 0;
    int invalid = 0;

    return_val_if_fail (ap && ap->supported_channels
                        && channels, -EINVAL);

    len = ap->supported_channels->length;
    chanls = ap->supported_channels->data;
    return_val_if_fail (chanls, -ENODATA);

    for ( ; index < length; index++) {

        valid = 0;
        iter = 0;

        chan = (unsigned int) channels [index];
        if (chan <= 14)
            is2_4 = 1;
        else if (chan > 14 && chan <= 165)
            is_5 = 1;
        else {
            invalid = 1;
            break;
        }

        for ( ; iter < len; iter++) {
            match = chanls [iter];
            if (match == chan) {
                valid = 1;
                break;
            }
        }

        if (!valid)
            break;
    }

    if (invalid || !valid)
        return -EINVAL;
    else if (is2_4 && is_5)
        return -EINVAL;

    return 0;
}

static int
access_point_compute_security (struct access_point *ap,
                               GKeyFile *keyfile,
                               const char *groupid,
                               const char *security)
{
    int valid = 0;
    size_t len = 0;
    char *passphrase;
    securitytype sec = NUM_AP_SECURITIES;

    return_val_if_fail (ap && keyfile, -EINVAL);
    return_val_if_fail (groupid, -ENOKEY);
    return_val_if_fail (security, -ENODATA);

    if (!g_strcmp0 (security, "none"))
        sec = AP_SECURITY_OPEN;
    else if (!g_strcmp0 (security, "wep"))
        sec = AP_SECURITY_WEP;
    else if (!g_strcmp0 (security, "wpa-psk"))
        sec = AP_SECURITY_WPA_PSK;
    else if (!g_strcmp0 (security, "wpa2-psk"))
        sec = AP_SECURITY_WPA2_PSK;
    else if (!g_strcmp0 (security, "wpa/wpa2-psk"))
        sec = AP_SECURITY_WPA_WPA2_PSK;

    if (sec == NUM_AP_SECURITIES)
        return -EINVAL;

    passphrase = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "Passphrase", NULL);
    if (passphrase)
        len = strlen (passphrase);

    INFO ("Passphrase length: %d", len);
    if (!len && sec != AP_SECURITY_OPEN)
        return -EINVAL;

    if (len && sec == AP_SECURITY_OPEN) {
        g_free (passphrase);
        return -EINVAL;
    }

    if (sec == AP_SECURITY_OPEN)
        valid = 1;
    else if (sec == AP_SECURITY_WEP) {
        if (len == 5 || len == 13 || len == 16)
            valid = 1;
        else if (len == 10 || len == 26 || len == 32)
            if (!is_hexa_decimal (passphrase, len))
                valid = 1;
    }
    else if (sec == AP_SECURITY_WPA_PSK || sec == AP_SECURITY_WPA2_PSK ||
             sec == AP_SECURITY_WPA_WPA2_PSK) {
        if (len >= WPA_PSK_PASSPHRASE_MIN_LENGTH && len < WPA_PSK_PASSPHRASE_MAX_LENGTH)
            valid = 1;
        else if (len == 64)
            if (!is_hexa_decimal (passphrase, len))
                valid = 1;
    }

    if (valid) {
        if (sec != AP_SECURITY_OPEN)
            ap->passphrase = g_strdup (passphrase);
        ap->security = sec;
    } else {
        ERROR ("The given passphrase and security combo is invalid: %s [sec: %s] [len: %d]",
               security, passphrase, len);
        g_free (passphrase);
        return -EINVAL;
    }

    INFO ("Security: %s, Passphrase: %s", ap_security2str (sec), passphrase);
    g_free (passphrase);
    return 0;
}

static gboolean
is_filename_valid (const char *filename)
{
    return_val_if_fail (filename, FALSE);

    /* Current or Parent */
    if (filename[0] == '.')
        return FALSE;

    return g_str_has_suffix (filename, ".conf");
}

static void
access_point_cleanup_filepath (gpointer data)
{
    g_free (data);
}

static int
access_point_load_defaults (struct access_point *ap,
                            GKeyFile **keyfile,
                            char **confpath)
{
    GDir *dir;
    int ret = -ENOKEY;
    const char *filename;
    GKeyFile *kfile = NULL;
    char *groupid, pathname [PATH_MAX], **temp;
    struct stat st;
    GSList *slist = NULL;

    return_val_if_fail (ap && keyfile, -EINVAL);
    return_val_if_fail (!*keyfile, -EEXIST);
    return_val_if_fail (configfile, -ENOENT);

    for (temp = configfile; *temp; temp++) {

        DEBUG ("Reading out the config file(s from dir): %s", *temp);
        ret = stat (*temp, &st);
        if (ret < 0 || (!S_ISDIR (st.st_mode) && !S_ISREG (st.st_mode)))
            continue;

        if (S_ISREG (st.st_mode)) {
            if (is_filename_valid (*temp)) {
                kfile = db_load_key_file (*temp);
                if (kfile) {
                    groupid = g_key_file_get_start_group (kfile);
                    DEBUG ("Configuration file: %s Groupid: %s", *temp, groupid);
                    if (groupid && !g_strcmp0 (groupid, ap->interface))
                        slist = g_slist_append (slist, g_strdup (*temp));
                    g_key_file_free (kfile);
                }
            }
        } else {
            dir = g_dir_open (*temp, 0, NULL);
            if (dir) {
                while ((filename = g_dir_read_name (dir))) {
                    if (is_filename_valid (filename)) {
                        memset (pathname, 0, sizeof (pathname));
                        sprintf (pathname, "%s/%s", *temp, filename);
                        kfile = db_load_key_file (pathname);
                        if (kfile) {
                            groupid = g_key_file_get_start_group (kfile);
                            DEBUG ("Configuration file: %s Groupid: %s", pathname, groupid);
                            if (groupid && !g_strcmp0 (groupid, ap->interface))
                                slist = g_slist_append (slist, g_strdup (pathname));
                            g_key_file_free (kfile);
                        }
                    }
                }
                g_dir_close (dir);
            }
        }
    }

    if (!g_slist_length (slist))
        ret = -ENOKEY;
    else if (g_slist_length (slist) == 1) {
        if (confpath)
            *confpath = g_strdup ((char *) (g_slist_nth (slist, 0)->data));
        *keyfile = db_load_key_file ((char *) (g_slist_nth (slist, 0)->data));
        ret = 0;
    }
    else if (g_slist_length (slist) > 1) {
        DEBUG ("There are too many configuration files supplied for the ap: %s [size : %u]",
               ap->interface, g_slist_length (slist));
        ret = -EINVAL;
    }

    g_slist_free_full (slist, access_point_cleanup_filepath);
    return ret;
}

static int
access_point_load_tethering_settings (struct access_point *ap,
                                      GKeyFile *keyfile)
{
    char *tether_grp = TETHERING_SETTINGS_GROUP_NAME;

    return_val_if_fail (ap, -EINVAL);
    return_val_if_fail (keyfile, 0);

    if (g_key_file_has_key (keyfile, tether_grp, "Enable", NULL))
        ap->tethering = GET_DATA_FROM_KEYFILE (boolean, keyfile, tether_grp, "Enable", NULL);

    ap->tethering_device = GET_DATA_FROM_KEYFILE (string, keyfile, tether_grp, "Interface", NULL);

    return 0;
}

static int
access_point_load_dhcpsettings (struct access_point *ap,
                                GKeyFile *keyfile)
{
    int length = 0;
    int ret = -ENOMEM;
    char **keys = NULL, **temp;
    char *settings, *servers;
    char *dhcpgrp = DHCP_SETTINGS_GROUP_NAME;
    gboolean dhcpgroup = FALSE;

    return_val_if_fail (ap && ap->dhcpsettings, -EINVAL);
    return_val_if_fail (keyfile, 0);

    DEBUG ("Configuring DHCP settings for: %s", ap->interface);

    dhcpgroup = g_key_file_has_group (keyfile, dhcpgrp);
    if (!dhcpgroup)
        goto server_create;

    ap->dhcpsettings->dnssupport = GET_DATA_FROM_KEYFILE (boolean, keyfile, dhcpgrp, "DNSSupport", NULL);
    ap->dhcpsettings->leaseperiod = (unsigned) GET_DATA_FROM_KEYFILE (integer, keyfile, dhcpgrp, "LeasePeriod", NULL);
    ap->dhcpsettings->poolsize = GET_DATA_FROM_KEYFILE (integer, keyfile, dhcpgrp, "PoolSize", NULL);
    ap->dhcpsettings->gateway = GET_DATA_FROM_KEYFILE (string, keyfile, dhcpgrp, "Gateway", NULL)

    servers = GET_DATA_FROM_KEYFILE (string, keyfile, dhcpgrp, "NameServers", NULL);
    if (servers) {
        ap->dhcpsettings->nameservers = g_strsplit (servers, ",", -1);
        g_free (servers);
    }

    servers = GET_DATA_FROM_KEYFILE (string, keyfile, dhcpgrp, "NtpServers", NULL);
    if (servers) {
        ap->dhcpsettings->ntpservers = g_strsplit (servers, ",", -1);
        g_free (servers);
    }

    settings = GET_DATA_FROM_KEYFILE (string, keyfile, dhcpgrp, "DHCPRange", NULL);
    if (!settings)
        goto server_create;

    keys = g_strsplit (settings, ",", -1);
    for (temp = keys; *temp; length++, temp++)
        ;

    if (is_private_ipaddress (keys [0])) {
        ap->dhcpsettings->type = IPPOOL_TYPE_FIXED;
        ap->dhcpsettings->start = g_strdup (keys [0]);
    }
    if (length == 2) {
        if (is_private_ipaddress (keys [1]))
            ap->dhcpsettings->end = g_strdup (keys [1]);
    }

    g_strfreev (keys);
    g_free (settings);

server_create:
    ap->dhcpsettings->server = dhcpserver_create (ap->interface, ap->dhcpsettings->type,
                                                  ap->dhcpsettings->start,
                                                  ap->dhcpsettings->end,
                                                  ap->dhcpsettings->poolsize,
                                                  ap->dhcpsettings->dnssupport,
                                                  ap->dhcpsettings->leaseperiod);
    if (ap->dhcpsettings->server) {
        INFO ("Successfully configured DHCP settings for: %s [%p]",
              ap->interface, ap->dhcpsettings);
        ret = 0;
    }

    return ret;
}

static int
access_point_configure_ssid (struct access_point *ap,
                             const char *value)
{
    int ret;
    char *ssid = NULL;
    size_t length = 0;

    return_val_if_fail (ap && value, -EINVAL);

    ret = is_hex_string (value, strlen (value));
    if (ret < 0) {
        ap->ssid->ssid = g_strdup (value);
        ap->ssid->length = strlen (value);
        ret = 0;
    } else {
        ret = access_point_compute_ssid (ap, value, &ssid, &length);
        if (!ret) {
            ap->ssid->ssid = ssid;
            ap->ssid->length = length;
        } else
            ERROR ("Invalid SSID from configuration file: %s", value);
    }

    return ret;
}

static int
validate_preferred_channels(struct access_point *ap,
							int *chann2_4,
							int *chann5,
							size_t *len2_4,
							size_t *len5,
							char *channels)
{
	unsigned int length = 0;
	gchar **chanls = NULL, **temp;
	int channels2_4 [15], chan, channels5 [48],
            length2_4 = 0, length5 = 0, index = 0;
    size_t l2_4 = 0, l5 = 0;

	return_val_if_fail (ap && chann2_4 && chann5
			&& len2_4 && len5 && channels, -EINVAL);

    chanls = g_strsplit (channels, " ", -1);
    length = g_strv_length (chanls);

    if (!length || length >= IEEE80211_CHANNELS_LENGTH) {
    	g_strfreev (chanls);
    	return -EINVAL;
    }

    memset (channels5, 0, sizeof (channels5));
    memset (channels2_4, 0, sizeof (channels2_4));

    for (temp = chanls; *temp; temp++) {

    	if ((chan = atoi (*temp)) == -1)
    		continue;

    	if (chan >= 1 && chan <= 14)
    		channels2_4 [length2_4++] = chan;
    	else if (chan >= 36 && chan <= 173)
    		channels5 [length5++] = chan;
    }

    for (index = 0; index < length2_4; index++)
    	if (!is_valid_channel (ap, channels2_4 [index]))
    		chann2_4 [l2_4++] = channels2_4 [index];

    for (index = 0; index < length5; index++)
    	if (!is_valid_channel (ap, channels5 [index]))
    		chann5 [l5++] = channels5 [index];

    *len2_4 = l2_4;
    *len5 = l5;
    g_strfreev (chanls);
    return 0;
}

static int
access_point_configure_hw_mode (struct access_point *ap,
                                const char *mode,
                                const char *ht_capab, const char *vht_capab)
{
    int ret = 0, ht_settings = 0;

    return_val_if_fail (ap && mode, -EINVAL);

    ret = access_point_set_hwmode (ap, mode, FALSE);
    if (ret < 0) {
        if (ret == -EEXIST)
            ret = 0;
        else {
            ERROR ("Invalid HW mode supplied [hw_mode: %s]: %s/%d", mode, strerror (-ret), -ret);
            return ret;
        }
    }

    if (!g_strcmp0 (mode, "a")) {
        if (!vht_capab) {
            /* By default VHT80 */
            ap->vht_oper_chwidth = 1;
            ap->vht_oper_centr_freq_seg0_idx = 106;
            ap->vht_settings |= AC_VHT_80MHZ;
            ht_settings = 1;
        } else {
            if (!g_strcmp0 (vht_capab, "VHT20")) {
                ap->vht_oper_chwidth = 0;
                ap->vht_settings |= AC_VHT_20MHZ;
            }
            else if (!g_strcmp0 (vht_capab, "VHT40")) {
                ap->vht_oper_chwidth = 0;
                ap->vht_settings |= AC_VHT_40MHZ;
                ht_settings = 1;
            }
            else if (!g_strcmp0 (vht_capab, "VHT80")) {
                ap->vht_oper_chwidth = 0;
                ap->vht_oper_centr_freq_seg0_idx = 106;
                ap->vht_settings |= AC_VHT_80MHZ;
                ht_settings = 1;
            } else
                ret = -EINVAL;
        }

        if (ht_settings) {
            ret = access_point_load_htcapab (ap, ht_capab);
            if (ret < 0)
                ERROR ("Failed to load the ht settings [%s]: %s/%d", ht_capab,
                       strerror (-ret), -ret);
        }
    }

    return ret;
}

static int
access_point_configure_channel (struct access_point *ap,
                                int channel,
                                char *preferredchannels)
{
    int ret = 0, temp2_4 [15], temp5 [48];
    size_t length2_4 = 0, length5 = 0;

    return_val_if_fail (ap, -EINVAL);

    memset (temp5, 0, sizeof (temp5));
    memset (temp2_4, 0, sizeof (temp2_4));

    if (!channel) {
    	DEBUG ("Preferred channel list: %s", preferredchannels);
    	ret = validate_preferred_channels (ap, temp2_4, temp5,
    			&length2_4, &length5, preferredchannels);
    	if (!ret) {
    		ret = access_point_set_preferred_channels (ap, (length5 != 0) ? temp5 : temp2_4,
    				(length5 != 0) ? length5 : length2_4);
            if (0 == ret) {
    			ap->acs = 1;
    			if (length5 || length2_4)
    				access_point_configure_hw_mode (ap, (length5 != 0) ? "a" : "g", "[HT40-]", NULL);
            } else {
                goto hw_mode;
            }
    	}
    } else {
    	if (!is_valid_channel (ap, channel)) {
    		ap->operating_channel = (unsigned int) channel;
    		if (ap->operating_channel >= 1 && ap->operating_channel <= 14)
    			access_point_set_hwmode (ap, "g", FALSE);
    		else if (ap->operating_channel >= 36 && ap->operating_channel <= 173)
    			access_point_set_hwmode (ap, "a", FALSE);
        } else {
            goto hw_mode;
        }
    }

    return ret;

hw_mode:
    ret = access_point_set_hwmode (ap, "g", FALSE);
    if (ret < 0) {
        if (ret == -EEXIST)
            ret = 0;
        else {
            ERROR ("Invalid HW mode supplied [hw_mode: %s]: %s/%d", "g",
                   strerror (-ret), -ret);
        }
    }

    return ret;
}

static int
access_point_load_dhcp_tether_settings (struct access_point *ap,
                                        GKeyFile *keyfile)
{
    int ret;

    return_val_if_fail (ap, -EINVAL);

    ret = access_point_load_dhcpsettings (ap, keyfile);
    if (ret < 0)
        ERROR ("Failed to load the dhcp and dns settings [%s]: %s/%d", ap->interface,
               strerror (-ret), -ret);

    ret = access_point_load_tethering_settings (ap, keyfile);
    if (ret < 0)
        ERROR ("Failed to load the internet settings for [%s]: %s/%d", ap->interface,
               strerror (-ret), -ret);

    return ret;
}

static int
access_point_load_keys (struct access_point *ap,
                        GKeyFile *keyfile)
{
    char *groupid, *temp = NULL,
            *ht_capab, *vht_capab;
    int channel, invalid = 0;
    gboolean visibility;
    GError *error = NULL;
    char **keys = NULL, **dupl = NULL;

    return_val_if_fail (ap && keyfile, -EINVAL);

    groupid = g_key_file_get_start_group (keyfile);
    return_val_if_fail (groupid, -ENOKEY);
    keys = g_key_file_get_keys (keyfile, groupid, NULL, &error);
    if (!keys) {
        ERROR ("Failed to load the keys for group: \"%s\" error : %s",
               groupid, error->message);
        g_error_free (error);
        return -ENOENT;
    }

    for (dupl = keys; *dupl; dupl++) {

        if (!g_strcmp0 (*dupl, "Enable")) {
            ap->targetpowerstate = GET_DATA_FROM_KEYFILE (boolean, keyfile, groupid, "Enable", NULL);
            if (ap->targetpowerstate)
                access_point_set_power_state (ap, POWERING_ON);
        }
        else if (!g_strcmp0 (*dupl, "CountryCode")) {
            temp = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "CountryCode", NULL);
            if (temp && strlen (temp) == 2)
                memcpy (ap->country_code, temp, 2);
            g_free (temp);
        }
        else if (!g_strcmp0 (*dupl, "SSID")) {
            temp = GET_DATA_FROM_KEYFILE (value, keyfile, groupid, "SSID", NULL);
            if (access_point_configure_ssid (ap, temp) != 0)
                invalid = 1;
            g_free (temp);
        }
        else if (!g_strcmp0 (*dupl, "HWMode")) {

            temp = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "HWMode", NULL);
            vht_capab = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "VHTCapabilities", NULL);
            ht_capab = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "HTCapabilities", NULL);

            if (access_point_configure_hw_mode (ap, temp, ht_capab ? ht_capab : "[HT40-]", vht_capab) != 0)
                invalid = 1;

            g_free (ht_capab);
            g_free (vht_capab);
            g_free (temp);
        }
        else if (!g_strcmp0 (*dupl, "Channel")) {
            channel = GET_DATA_FROM_KEYFILE (integer, keyfile, groupid, "Channel", NULL);
            temp = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "Chanlist", NULL);
            if (access_point_configure_channel (ap, channel, temp) != 0)
                invalid = 1;
            g_free (temp);
        }
        else if (!g_strcmp0 (*dupl, "MaxNumStations")) {
            ap->max_stations_allowed = GET_DATA_FROM_KEYFILE (integer, keyfile, groupid, "MaxNumStations", NULL);
            if (ap->max_stations_allowed > 10)
                invalid = 1;
        }
        else if (!g_strcmp0 (*dupl, "Visibility")) {
            visibility = GET_DATA_FROM_KEYFILE (boolean, keyfile, groupid, "Visibility", NULL);
            (visibility == TRUE) ? (ap->ignore_broadcast_ssid = BROADCAST_SSID) :
                                   (ap->ignore_broadcast_ssid = BROADCAST_EMPTY_SSID);
        }
        else if (!g_strcmp0 (*dupl, "Security")) {
            temp = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "Security", NULL);
            if (access_point_compute_security (ap, keyfile, groupid, temp) != 0)
                invalid = 1;
            g_free (temp);
        }
        else if (!g_strcmp0 (*dupl, "BeaconInt")) {
            ap->beacon_int = (unsigned short int) GET_DATA_FROM_KEYFILE (integer, keyfile, groupid, "BeaconInt", NULL);
        }
        else if (!g_strcmp0 (*dupl, "UAPSDAdv")) {
            ap->uapsd_advertisement_enabled = (unsigned short int) GET_DATA_FROM_KEYFILE (integer, keyfile, groupid, "UAPSDAdv", NULL);
        }
        else if (!g_strcmp0 (*dupl, "vendor_elements")) {
            ap->vendor_elements = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "vendor_elements", NULL);
        }
        else if (!g_strcmp0 (*dupl, "assocresp_elements")) {
            ap->assocresp_elements = GET_DATA_FROM_KEYFILE (string, keyfile, groupid, "assocresp_elements", NULL);
        }
        else if (!g_strcmp0 (*dupl, "DTIMPeriod")){
            ap->dtim = GET_DATA_FROM_KEYFILE (integer, keyfile, groupid, "DTIMPeriod", NULL);
        }
        else if(!g_strcmp0 (*dupl, "access_network_type")){
            ap->access_network_type = GET_DATA_FROM_KEYFILE (integer, keyfile, groupid, "access_network_type", NULL);
        }

        if (invalid) {
            DEBUG ("Invalid value/key combination: %s", *dupl);
            break;
        }
    }

    g_strfreev (keys);
    return (invalid == 1) ? -EINVAL : 0;
}

static int
access_point_validate_conf (struct access_point *ap,
                            const char *fingerprint)
{
    int ret;
    char *sha256hash = NULL;
    GKeyFile *conffile = NULL;

    return_val_if_fail (ap && fingerprint, -EINVAL);

    ret = access_point_load_defaults (ap, &conffile, NULL);
    if (ret < 0) {
        ERROR ("Failed to load the configuration file: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = calculate_sha256_hash (ap->conffile, &sha256hash);
    if (ret < 0) {
        ERROR ("Failed to caculate the sha256 digest for the conf file "
               "[%s]: %s/%d", ap->conffile, strerror (-ret), -ret);
        g_key_file_free (conffile);
        return ret;
    }

    DEBUG ("Stored SHA256: \"%s\", Calculated SHA256: \"%s\"",
           fingerprint, sha256hash);
    g_key_file_free (conffile);
    if (g_strcmp0 (sha256hash, fingerprint)) {
        g_free (sha256hash);
        ret = -EINVAL;
    }

    return ret;
}

static int
access_point_load_from_db (struct access_point *ap)
{
    int ret;
    gboolean fromdb = TRUE, fromconf = TRUE;
    char *fingerprint;
    GKeyFile *dbfile = NULL, *confile = NULL;

    return_val_if_fail (ap, -EINVAL);

    (void) access_point_load_defaults (ap, &confile, &ap->conffile);
    ret = db_load (ap, &dbfile);
    if (ret == 0 && g_key_file_has_group (dbfile, CONFIG_FILE_FINGERPRINT)) {
        fingerprint = GET_DATA_FROM_KEYFILE (string, dbfile, CONFIG_FILE_FINGERPRINT, "ConfSHA256", NULL);
        if (fingerprint && !access_point_validate_conf (ap, fingerprint))
            fromconf = FALSE;
    }

    if (ret < 0) {
        /* No entry found in the db, lets load the configurations */
        DEBUG ("No db file found for the ap: %s", ap->interface);
        fromdb = FALSE;
        if (ret == -ENOENT) {
            if (!ap->conffile) {
                ERROR ("No default configuration file found for ap: \"%s\"",
                       ap->interface);
                return ret;
            }
        } else
            return ret;
    }

    (void) access_point_load_keys (ap, fromdb ? dbfile : confile);
    ret = access_point_load_dhcp_tether_settings (ap, fromconf ? confile : dbfile);

    if (dbfile)
        g_key_file_free (dbfile);
    if (confile)
        g_key_file_free (confile);

    return ret;
}

static int
get_country_code_from_db (struct access_point *ap,
                          char **country_code)
{
    int ret;
    GKeyFile *dbfile = NULL;
    char *groupid;

    return_val_if_fail (ap && country_code, -EINVAL);

    ret = db_load (ap, &dbfile);
    return_val_if_fail (ret == 0, ret);

    groupid = g_key_file_get_start_group (dbfile);
    if (!groupid) {
        g_key_file_free (dbfile);
        return -ENOKEY;
    }

    *country_code = GET_DATA_FROM_KEYFILE (string, dbfile, groupid,
                                           "CountryCode", NULL);
    if (!*country_code)
        ret = -ENOKEY;

    g_key_file_free (dbfile);
    return ret;
}

static int
access_point_get_modes (struct access_point *ap)
{
    int ret;

    return_val_if_fail (ap, -EINVAL);

    ret = accesspoint_get_supported_hwmodes (ap->interface, &ap->supported_hwmodes);
    if (ret < 0)
        ERROR ("Failed to get the supported hwmodes information: %s", accesspoint_strerror (ret));
    return ret;
}

static int
access_point_get_hwinfo (struct access_point *ap,
                         const char *info)
{
    int ret = 0;
    channel *data;

    return_val_if_fail (ap && info, -EINVAL);

    data = g_try_malloc0 (sizeof (*data));
    return_val_if_fail (data, -ENOMEM);

    if (!g_strcmp0 (info, "channels")) {
        ap->supported_channels = data;
        ret = accesspoint_get_supported_channels (ap->interface, &ap->supported_channels->data,
                                                  &ap->supported_channels->length);
    }
    else if (!g_strcmp0 (info, "frequencies")) {
        ap->supported_frequencies = data;
        ret = accesspoint_get_supported_frequencies (ap->interface, &ap->supported_frequencies->data,
                                                     &ap->supported_frequencies->length);
    }
    else if (!g_strcmp0 (info, "rates")) {
        ap->supported_rates = data;
        ret = accesspoint_get_supported_rates (ap->interface, &ap->supported_rates->data,
                                               &ap->supported_rates->length);
    }

    if (ret < 0) {
    	if (!g_strcmp0 (info, "channels"))
    		ap->supported_channels = NULL;
    	else if (!g_strcmp0 (info, "frequencies"))
    		ap->supported_frequencies = NULL;
    	else if (!g_strcmp0 (info, "rates"))
    		ap->supported_rates = NULL;
        if (ret != -ENODATA)
            ERROR ("Failed to get the supported \"%s\" information: %s", info,
                   accesspoint_strerror (ret));
        g_free (data);
    }

    return ret;
}


static void
access_point_initialize_hwinfo (struct access_point *ap)
{
    char *country_code;

    return_if_fail (ap);

    access_point_get_modes (ap);
    access_point_get_hwinfo (ap, "channels");
    access_point_get_hwinfo (ap, "frequencies");
    access_point_get_hwinfo (ap, "rates");

    country_code = accesspoint_get_country_code (ap->interface);
    if (country_code) {
        memcpy (ap->country_code, country_code, 2);
        INFO ("Current Regulatory settings for AP: %s [\"%s\"]", ap->interface,
              ap->country_code);
    }
}

static int
access_point_initialize (struct access_point *ap)
{
    char *alpha2 = NULL;
    dhcp_settings *dhcpsettings;
    int ret;

    return_val_if_fail (ap, -EINVAL);

    DEBUG ("Initializing the accesspoint: %s", ap->interface);

    access_point_ssid_cleanup(ap->ssid);

    ap->ssid = g_try_malloc0 (sizeof (*ap->ssid));
    return_val_if_fail (ap->ssid, -ENOMEM);

    ap->ssid->length = 0;
    ap->ssid->ssid = NULL;

    dhcpsettings = g_try_malloc0 (sizeof (*dhcpsettings));
    if (!dhcpsettings) {
        g_free (ap->ssid);
        return -ENOMEM;
    }

    /*
     * By default, DNS is not supported, we dont take the risk
     * even though the dnsmasq is running with minimal privileges
     * i.e., we forward everything to gateway and the gateway
     * shall be able resolve it (well if it is configured rightly) */
    dhcpsettings->dnssupport = FALSE;
    dhcpsettings->type = IPPOOL_TYPE_DYNAMIC;
    dhcpsettings->leaseperiod = 0;
    dhcpsettings->poolsize = 0;
    ap->dhcpsettings = dhcpsettings;

    access_point_initialize_hwinfo (ap);
    ap->actualpowerstate = ap->targetpowerstate = FALSE;

    ret = get_country_code_from_db (ap, &alpha2);
    if (ret == 0) {
        memcpy (ap->country_code, alpha2, 2);
        g_free (alpha2);
    }

    ap->driver = g_strdup (NL80211_DRIVER);
    ap->ctrl_interface = g_strdup (CTRL_INTERFACE_PATH);
    ap->ieee80211d = ap->ieee80211h = CONF_ENABLED;
    ap->current_operating_hwmode = IEEE80211_MODE_G;
    ap->acs = 0;

    ap->powerstate = POWERED_OFF;

    ap->beacon_int = BEACON_INTERVAL;
    ap->dtim = DTIM_PERIOD;
    ap->max_stations_allowed = MAX_STATIONS_ALLOWED;

    ap->macaddr_acl = ACCEPT_UNLESS_DENIED;
    ap->ignore_broadcast_ssid = BROADCAST_SSID;

    ap->wmm_enabled = CONF_ENABLED;

    ap->ap_max_inactivity = AP_MAX_INACTIVITY;
    ap->skip_inactivity_poll = CONF_DISABLED;
    ap->disassoc_low_ack = CONF_DISABLED;

    ap->rts_threshold = RTS_THRESHOLD;
    ap->fragm_threshold = FRAGMENT_THRESHOLD;
    ap->ieee80211w = CONF_ENABLED;

    ap->max_listen_interval = 65535;
    ap->uapsd_advertisement_enabled = CONF_ENABLED;

    ap->ieee80211n = CONF_ENABLED;
    ap->ht_capab |= HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET_MINUS;
    ap->enable_ht = CONF_ENABLED;

    ap->ieee80211ac = ap->enable_vht = CONF_ENABLED;

    ap->security = AP_SECURITY_WPA2_PSK;
    ap->rsn_preauth = CONF_ENABLED;

    ap->wps_state = WPS_DISABLED;
    ap->wps_independent = ap->ap_setup_locked = 1;

    ap->interworking = CONF_ENABLED;
    ap->access_network_type = PRIVATE_NETWORK;
    ap->internet = CONF_DISABLED;
    ap->asra = CONF_DISABLED;
    ap->esr = CONF_DISABLED;
    ap->uesa = CONF_DISABLED;
    ap->venue_group = VENUE_GROUP_VEHICULAR;
    ap->venue_type = VENUE_TYPE_AUTOMOBILE_OR_TRUCK;

    ap->tethering = (gboolean) CONF_ENABLED;
    ap->scanstate = NL80211_NUM_SCAN_STATES;

    ap->stations = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, station_cleanup);
    return 0;
}

static struct access_point*
access_point_allocate_ap (unsigned int index,
                          const char *ifname,
                          const char *address)
{
    struct access_point *ap;

    ap = g_try_malloc0 (sizeof (*ap));
    return_val_if_fail (ap, NULL);

    ap->index = index;
    ap->interface = g_strdup (ifname);
    ap->address = g_strdup (address);
    ap->ctrl_iface_grp = getgid ();
    return ap;
}

static int
access_point_deallocate (struct access_point **apt)
{
    struct access_point *ap;

    return_val_if_fail (apt, -EINVAL);
    return_val_if_fail (*apt, -EALREADY);

    ap = *apt;

    g_free (ap->interface);
    g_free (ap->address);
    g_free (ap);

    *apt = NULL;
    return 0;
}

static void*
access_point_create (unsigned int index,
                     const char *ifname,
                     const char *address)
{
    int ret;
    struct access_point *ap;

    return_val_if_fail (ifname && address, NULL);

    DEBUG ("Index: \"%u\" Wireless Interface: \"%s\" Address: \"%s\"",
           index, ifname, address);

    ap = g_hash_table_lookup (access_points, ifname);
    return_val_if_fail (!ap, ap);

    ap = access_point_allocate_ap (index, ifname, address);
    return_val_if_fail (ap, NULL);
    ap->ssid = NULL;
    ap->passphrase = NULL;
    ret = access_point_initialize (ap);
    if (ret < 0) {
        ERROR ("Failed to initialize the access point: %s/%d",
               strerror (-ret), -ret);
        access_point_deallocate (&ap);
        return NULL;
    }

    set_ap_state (ap, WAPDMAN_AP_STATE_INITIALIZED);
    g_hash_table_replace (access_points, ap->interface, ap);
    return ap;
}

static int
ap_complete_startup (struct access_point *ap)
{
  int ret;

  return_val_if_fail (ap,-EINVAL);

  set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURED_NAT);
  set_power_failure_reason (ap, WAPDMAN_PFRC_NONE);
  access_point_set_power_state (ap, POWERED_ON);
  access_point_set_powered (ap, TRUE);

  ret = ap_task_wakeup_next (ap);
  if (ret < 0)
      ERROR ("Failed to process the next request: %s/%d",
             strerror (-ret), -ret);

  return ret;
}

static void
ap_add_interface_cb (int result,
                     char *ifname,
                     void *userdata)
{
    int ret;
    struct access_point *ap = userdata;
    PowerFailureReason reason = WAPDMAN_PFRC_NONE;

    return_if_fail (ap && ifname);

    DEBUG ("Result: %d ifname: %s accesspoint: %p", result,
           ifname, ap);

    ret = unlink (ap->tempfile);
    if (ret < 0)
        ERROR ("Failed to remove the temp conf file [%s] error: %d/%s",
               ap->tempfile, ret, strerror (errno));
    g_free (ap->tempfile);
    ap->tempfile = NULL;

    if (result < 0) {
        reason = WAPDMAN_PFRC_HAPD_REJECTED;
        goto failure;
    }

    set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURED_HOSTAPD);
    if (!ap->dhcpsettings) {
        /* should never reach this i.e., we should
         * have a valid settings in all possible ways */
        reason = WAPDMAN_PFRC_DHCPDNS_CONF_FAILURE;
        goto failure;
    }

    set_ap_state (ap, WAPDMAN_AP_STATE_STARTING_DHCP_SERVER);
    ret = dhcpserver_start (ap->dhcpsettings->server,
                            ap->dhcpsettings->nameservers,
                            ap->dhcpsettings->ntpservers);
    if (ret < 0) {
        reason = WAPDMAN_PFRC_DHCPDNS_FAILED_STARTUP;
        goto failure;
    }

    set_ap_state (ap, WAPDMAN_AP_STATE_DHCP_SERVER_STARTED);
    if (ap->tethering) {
        ret = ap_configure_nat (ap);
        if (ret < 0) {
            reason = ((PowerFailureReason) (-ret));
            goto failure;
        }
    }

    ap_complete_startup (ap);
    return;

failure:
    set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
    access_point_set_power_state (ap, POWER_FAILURE);
    set_power_failure_reason (ap, reason);
    access_point_set_powered (ap, FALSE);
}

static int
access_point_create_conf_file (struct access_point *ap)
{
    int ret;
    char **keys = NULL;
    GError *error = NULL;
    GKeyFile *keyfile = NULL;

    return_val_if_fail (ap, -EINVAL);

    DEBUG ("Creating hostapd configuration file for: %s",
           ap->interface);

    keyfile = g_key_file_new ();
    return_val_if_fail (keyfile, -ENOMEM);

    ret = access_point_create_conf (ap, ap->interface, keyfile);
    if (ret < 0) {
        ERROR ("Failed to dump the contents in to a keyfile: %s/%d",
               strerror (-ret), -ret);
        set_power_failure_reason (ap, WAPDMAN_PFRC_INVAL_CONF);
        goto error;
    }

    keys = g_key_file_get_keys (keyfile, ap->interface, NULL, &error);
    if (!keys) {
        ERROR ("Failed to load the keys for groupid \"%s\" error :%s",
               ap->interface, error->message);
        set_power_failure_reason (ap, WAPDMAN_PFRC_DB_FAILURE);
        g_error_free (error);
        ret = -EIO;
        goto error;
    }

    ret = db_create_temp_file (ap, &ap->tempfile, keyfile, keys, ap->interface);
    if (ret < 0) {
        ERROR ("Failed to create a temp file for \"%s\": %s/%d",
               ap->interface, strerror (-ret), -ret);
        set_power_failure_reason (ap, WAPDMAN_PFRC_DB_FAILURE);
    }

error:
    g_key_file_free (keyfile);
    return ret;
}

static int
access_point_reallocate (struct access_point **old)
{
    int ret;
    unsigned int index = 0;
    char *iface, *address;
    struct access_point *ap, *temp = NULL;

    return_val_if_fail (old && *old, -EINVAL);

    ap = *old;
    index = ap->index;
    iface = g_strdup (ap->interface);
    address = g_strdup (ap->address);

    g_hash_table_remove (access_points, ap->interface);
    temp = access_point_allocate_ap (index, iface, address);

    g_free (iface);
    g_free (address);

    return_val_if_fail (temp, -ENOMEM);
    ret = access_point_initialize (temp);
    if (0 == ret) {
        ap->dhcpsettings->server =
                dhcpserver_create (temp->interface, temp->dhcpsettings->type,
                                   temp->dhcpsettings->start,
                                   temp->dhcpsettings->end,
                                   temp->dhcpsettings->poolsize,
                                   temp->dhcpsettings->dnssupport,
                                   temp->dhcpsettings->leaseperiod);
        if (temp->dhcpsettings->server)
            g_hash_table_replace (access_points, ap->interface, temp);
        else {
            ERROR ("Failed to create the dhcp server instance for the ap: %s",
                   temp->interface);
            goto failure;
        }
    } else {
        ERROR ("Failed to initialize the ap: %s", temp->interface);
        goto failure;
    }

    *old = temp;
    return ret;

failure:
    *old = NULL;
    access_point_cleanup (temp);
    return ret;
}

static int
ap_load_defaults_register_dbus (struct access_point *ap)
{
    int ret;

    return_val_if_fail (ap, -EINVAL);

    ret = access_point_load_from_db (ap);
    if (ret < 0) {
        ERROR ("Failed to load the configurations from db/defaults "
               "for: %s [%s/%d]", ap->interface, strerror (-ret), -ret);
        if (ret == -EINVAL || ret == -EACCES) {
            /* Lets cleanup on failure and at least register
             * with the defaults */
            ret = access_point_reallocate (&ap);
            if (ret < 0) {
                CRITICAL ("Failed to reallocate the AP: %s/%d",
                          strerror (-ret), -ret);
                return ret;
            }
        }
    }

    ap->dbusservice = access_point_create_service (get_system_bus(), ap);
    if (!ap->dbusservice)
        ERROR ("Failed to register the access point \"%s\" to dbus",
               ap->interface);

    return ret;
}

static void
ap_change_reg_domain_cb (int result,
                         char *iface,
                         void *data)
{
    struct access_point *ap = data;
    PowerFailureReason reason = WAPDMAN_PFRC_NONE;
    int ret;
    unsigned int acs_channels = 0;

    DEBUG ("Result: %d interface: %s", result, iface);

    /* We proceed to setup the AP even on reg dom failure
     * to see if we could operate with current reg dom */
    set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURED_REGDOM);
    ret = ap_load_defaults_register_dbus (ap);
    if (ret < 0) {
        ERROR ("Failed to load the configurations from db/defaults: %s/%d",
               strerror (-ret), -ret);
        return;
    }

    if (!ap->targetpowerstate) {
        /* Check if there are somer requests to "
         * be processed */
        ret = ap_task_wakeup_next (ap);
        if (ret < 0)
            ERROR ("Failed to process the next request: %s/%d",
                   strerror (-ret), -ret);
        return;
    }

    ret = access_point_create_conf_file (ap);
    if (ret < 0) {
        ERROR ("Failed to create the temp configuration: %s",
               ap->interface);
        reason = WAPDMAN_PFRC_INVAL_CONF;
        goto failure;
    }

    if (ap->acs && ap->preferred_channels)
        acs_channels = ap->preferred_channels->length;
    ret = accesspoint_add_interface (ap->interface, ap->tempfile,
                                     ap->acs,
                                     (int) acs_channels,
                                     ap_add_interface_cb, ap);
    if (ret < 0) {
        ERROR ("Failed to add \"%s\" to hostapd: %s/%d",
               ap->interface,
               accesspoint_strerror (-ret), -ret);
        unlink (ap->tempfile);
        g_free (ap->tempfile);
        ap->tempfile = NULL;
        reason = WAPDMAN_PFRC_CONF_HAPD_FAILURE;
        goto failure;
    }

    set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURING_HOSTAPD);
    return;

failure:
    access_point_set_power_state (ap, POWER_FAILURE);
    set_power_failure_reason (ap, reason);
    set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
}

static int
access_point_start (void *data)
{
    int ret = 0;
    char *alpha2 = NULL;
    struct access_point *ap = data;
    gboolean change_regdom = FALSE;

    return_val_if_fail (ap, -EINVAL);

    alpha2 = accesspoint_get_country_code (ap->interface);

    INFO ("Current Reg Dom for the interface \"%s\": %s, target: %s",
          ap->interface, alpha2 ? alpha2 : "none", ap->country_code);

    /* First we configure the right regulatory domain and
     * proceed to setup the AP */
    if (alpha2 && (0 != g_strcmp0 (alpha2, ap->country_code)))
        change_regdom = TRUE;

    if (!change_regdom) {
        ap_change_reg_domain_cb (0, ap->interface, ap);
        return ret;
    }

    ret = accesspoint_set_country_code (ap->interface, ap->country_code,
                                        ap_change_reg_domain_cb,
                                        ap);
    if (ret < 0) {
        ERROR ("Failed to change the reg dom for \"%s\" to: %s [%s/%d]",
               ap->interface, ap->country_code,
               accesspoint_strerror (-ret), -ret);
        ap_change_reg_domain_cb (0, ap->interface, ap);
    } else
        set_ap_state (ap, WAPDMAN_AP_STATE_CONFIGURING_REGDOM);

    return ret;
}

static int
ap_cleanup_pending_tasks (struct access_point *ap)
{
    int ret;
    GList *temp;
    taskstate state;
    struct wapdman_task *task;

    return_val_if_fail (ap, -EINVAL);

    INFO ("List of pending tasks in the task queue of accesspoint %s: %d",
          ap->interface, g_list_length (ap->taskqueue));

    temp = ap->taskqueue;
    for ( ; temp; temp = temp->next) {
        task = temp->data;
        if (task) {
            INFO ("Task %p is still pending", task);
            state = task_get_state (task);
            if (state == TASK_STATE_REQUESTED)
                INFO ("Task %p is still processing", task);

            ret = task_remove (&ap->taskqueue, task);
            if (ret < 0)
                ERROR ("Failed to remove the task %p from the queue", task);

            access_point_handle_results (ap->dbusservice, task_get_invocation (task),
                                         -ENXIO, task_get_opcode (task), task_get_data (task));
            task_cleanup (task);
        }
    }

    return 0;
}

static void
ap_rm_interface_cb (int result,
                    char *ifname,
                    void *userdata)
{
    int ret;
    struct access_point *ap;

    DEBUG ("Result: %d ifname: %s accesspoint: %p", result,
           ifname, userdata);

    ap = g_hash_table_lookup (access_points, ifname);
    return_if_fail (ap);
    if (result < 0) {
        set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
        set_power_failure_reason (ap, WAPDMAN_PFRC_RM_HAPD_FAILURE);
        return;
    }

    ret = access_point_service_destroy (ap->dbusservice);
    if (ret < 0)
        ERROR ("Failed to unregister and destroy the ap dbus service : %s/%d",
               strerror (-ret), -ret);

    set_ap_state (ap, WAPDMAN_AP_STATE_REMOVED_HOSTAPD);
    set_ap_state (ap, WAPDMAN_AP_STATE_STOPPING_DHCP_SERVER);

    ret = dhcpserver_stop (ap->dhcpsettings->server, NULL);
    if (ret < 0 && ret != -ENOENT && ret != -EINVAL) {
        ERROR ("Failed to stop the DHCP server for %s : %s", ap->interface, strerror (-ret));
        set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
        set_power_failure_reason (ap, WAPDMAN_PFRC_RM_HAPD_FAILURE);
        return;
    }

    set_ap_state (ap, WAPDMAN_AP_STATE_DHCP_SERVER_STOPPED);
    set_power_failure_reason (ap, WAPDMAN_PFRC_NONE);
    g_hash_table_remove (access_points, ifname);
}

static int
access_point_stop (void *data)
{
    int ret;
    struct access_point *ap = data;

    return_val_if_fail (ap, -EINVAL);

    DEBUG ("Stopping Accesspoint: %s", ap->interface);

    ret = ap_cleanup_pending_tasks (ap);
    if (ret < 0)
        ERROR ("Failed to cleanup the task queue: %s/%d", strerror (-ret), -ret);

    ret = accesspoint_remove_interface (ap->interface, ap_rm_interface_cb, ap);
    if (ret < 0) {
        ERROR ("Failed to remove \"%s\" from hostapd %s", ap->interface, accesspoint_strerror (-ret));
        set_power_failure_reason (ap, WAPDMAN_PFRC_RM_HAPD_FAILURE);
        set_ap_state (ap, WAPDMAN_AP_STATE_FAILED);
        return ret;
    }

    set_ap_state (ap, WAPDMAN_AP_STATE_REMOVING_HOSTAPD);
    return ret;
}

static int
access_point_remove (unsigned int index,
                     const char *ifname,
                     const char *address)
{
    gboolean removed = FALSE;
    struct access_point *ap;

    return_val_if_fail (address && ifname, -EINVAL);

    DEBUG ("index: %d ifname: %s address: %s", index, ifname, address);

    ap = g_hash_table_lookup (access_points, ifname);
    return_val_if_fail (ap, -ENOENT);
    if (ap->state != WAPDMAN_AP_STATE_DHCP_SERVER_STOPPED) {
        INFO ("Accesspoint \"%s\" will be removed after hostapd "
              "configuration", ifname);
        return 0;
    }

    removed = g_hash_table_remove (access_points, ifname);
    return (removed == TRUE) ? 0 : -ENOENT;
}

static wirelessdriverops ap_ops = {
    .name = "accesspoint",
    .iftype = NL80211_IFTYPE_AP,
    .add = access_point_create,
    .remove = access_point_remove,
    .start = access_point_start,
    .stop = access_point_stop,
};

static void
handle_ap_system_status (const int status)
{
    DEBUG ("AP Driver Status : %s", status ?
               "AVAILABLE" : "UNAVAILABLE");

    if (apdriverstatus != status)
        apdriverstatus = status;
}

static const char*
station_change_reason2str (const changereason reason)
{
    switch (reason) {
    case STATION_CHANGE_NONE:
        return ENUMTOSTR (NONE);
    case STATION_CHANGE_ADDED:
        return ENUMTOSTR (ADDED);
    case STATION_CHANGE_NOCHANGE:
        return ENUMTOSTR (NOCHANGE);
    case STATION_CHANGE_CHANGED:
        return ENUMTOSTR (CHANGED);
    case STATION_CHANGE_REMOVED:
        return ENUMTOSTR (REMOVED);
    }

    return ENUMTOSTR (UNKNOWN);
}

static void
ap_stations_append_changed (gpointer key,
                            gpointer value,
                            gpointer data)
{
    int add = 0;
    const char *path;
    GSList **changed = data;
    struct station_changed *changedstation;
    struct associated_station *station = value;

    return_if_fail (station);

    DEBUG ("key: %s", (char*)key);

    changedstation = g_try_malloc0 (sizeof (*changedstation));
    return_if_fail (changedstation);
    changedstation->station = station;
    path = associated_station_get_objectpath (station);

    if (g_hash_table_lookup (stations_notify->add, path)) {
        add = 1;
        changedstation->reason = STATION_CHANGE_ADDED;
        associated_station_register_interfaces (station);
    }
    else if (g_hash_table_lookup (stations_notify->remove, path))
        changedstation->reason = STATION_CHANGE_REMOVED;
    else {
        add = 1;
        changedstation->reason = STATION_CHANGE_NOCHANGE;
    }

    DEBUG ("Station \"%s\" is: %s", path, station_change_reason2str (changedstation->reason));

    if (add)
        *changed = g_slist_append (*changed, changedstation);
}

static void
ap_stations_append_removed (gpointer key,
                            gpointer value,
                            gpointer data)
{
    char *path;
    GSList **removed = data;
    struct station_changed *removedstation;
    struct associated_station *station = value;

    return_if_fail (station);

    DEBUG ("key: %s", (char*)key);

    path = associated_station_get_objectpath (station);
    if (g_hash_table_lookup (stations_notify->remove, path)) {
        DEBUG ("Removed station: %s", path);
        removedstation = g_try_malloc0 (sizeof (*removedstation));
        return_if_fail (removedstation);
        removedstation->station = station;
        removedstation->reason = STATION_CHANGE_REMOVED;
        *removed = g_slist_append (*removed, removedstation);
    }
}

static void
ap_changedstations_cleanup (gpointer data)
{
    g_free (data);
}

static gboolean
ap_stations_notify_clients (gpointer data)
{
    struct access_point *ap = data;
    struct associated_station *removed;
    struct station_changed *removedstation;
    GSList *addedstations = NULL,
            *removedstations = NULL;
    int ret;
    GSList *temp;
    char *macaddress;

    return_val_if_fail (ap, FALSE);

    DEBUG ("Accesspoint: %s [%p] added stations: %d removed stations: %d",
           ap->interface, ap, g_hash_table_size (stations_notify->add),
           g_hash_table_size (stations_notify->remove));

    g_hash_table_foreach (ap->stations, ap_stations_append_changed, &addedstations);
    g_hash_table_foreach (ap->stations, ap_stations_append_removed, &removedstations);

    ret = access_point_emit_stationsChanged (addedstations, removedstations, ap->dbusservice);
    if (ret < 0)
        ERROR ("Failed to emit the stations changed signal: %s", strerror (-ret));

    temp = removedstations;
    for ( ; temp; temp = temp->next) {
        removedstation = temp->data;
        if (removedstation) {
            macaddress = associated_station_get_macaddress (removedstation->station);
            if (macaddress) {
                removed = g_hash_table_lookup (ap->stations, macaddress);
                if (removed) {
                    INFO ("Removing the \"%s\" station from the list of associated stations",
                          macaddress);
                    g_hash_table_remove (ap->stations, macaddress);
                }
            }
        }
    }

    g_slist_free_full (addedstations, ap_changedstations_cleanup);
    g_slist_free_full (removedstations, ap_changedstations_cleanup);

    g_hash_table_remove_all (stations_notify->remove);
    g_hash_table_remove_all (stations_notify->add);

    return FALSE;
}

static void
ap_schedule_stations_notify (struct access_point *ap)
{
    ap_stations_notify_clients (ap);
}

static void
ap_stations_schedule_update (struct access_point *ap,
                             struct associated_station *station,
                             gboolean added)
{
    char *path = associated_station_get_objectpath (station);

    if (added) {
        g_hash_table_remove (stations_notify->remove, path);
        g_hash_table_replace (stations_notify->add, path, station);
    } else {
        g_hash_table_remove (stations_notify->add, path);
        g_hash_table_replace (stations_notify->remove, path, station);
    }

    DEBUG ("Station: %p obj_path: %s Update event: %s add: %d remove: %d",
           station, path, added ? "ADDED" : "REMOVED",
           g_hash_table_size (stations_notify->add),
           g_hash_table_size (stations_notify->remove));

    ap_schedule_stations_notify (ap);
}

static void
handle_ap_station_associated (const char *interface,
                              const char *station)
{
    struct access_point *ap;
    struct associated_station *sta;
    char *address = NULL;

    DEBUG ("Accesspoint interface: %s connected station: %s",
           interface, station);

    ap = g_hash_table_lookup (access_points, interface);
    return_if_fail (ap);
    sta = g_hash_table_lookup (ap->stations, station);
    return_if_fail (!sta);
    sta = associated_station_create (get_system_bus(), ap, station, interface);
    return_if_fail (sta);
    address = associated_station_get_macaddress (sta);
    g_hash_table_replace (ap->stations, address, sta);
}

static void
handle_ap_station_disconnected (const char *interface,
                                const char *station)
{
    struct access_point *ap;
    struct associated_station *sta;

    DEBUG ("Accesspoint interface: %s disconnected station: %s",
           interface, station);

    ap = g_hash_table_lookup (access_points, interface);
    return_if_fail (ap);
    sta = g_hash_table_lookup (ap->stations, station);
    return_if_fail (sta);
    ap_stations_schedule_update (ap, sta, FALSE);
}

static int
access_point_teardown_required (struct access_point *ap,
                                const unsigned int *frequencies,
                                const unsigned int *channels,
                                const unsigned int len)
{
    unsigned int index = 0, match = 0;

    return_val_if_fail (ap, -EINVAL);

    if (len > 0 && (!frequencies || !channels))
        return -EINVAL;

    INFO ("AP \"%s\" power state: %s configured channel: %u",
          ap->interface, ap->actualpowerstate ? "ON" : "OFF",
          ap->operating_channel);

    if (!ap->operating_channel)
        return -ECANCELED;

    for ( ; index < len; ++index)
        if (ap->operating_channel == channels [index]) {
            match = 1;
            break;
        }

    return match == 1 ? -ECANCELED : 0;
}

static void
handle_ap_regulatory_change (const char *interface,
                             const unsigned int *frequencies,
                             const unsigned int *channels,
                             const unsigned int len)
{
    (void) frequencies;
    (void) channels;
    (void) len;

    int ret;
    struct access_point *ap;

    DEBUG ("Regulatory change notification for Accesspoint: %s", interface);

    ap = g_hash_table_lookup (access_points, interface);
    return_if_fail (ap);

    access_point_channel_cleanup (ap->supported_channels);
    access_point_channel_cleanup (ap->supported_frequencies);
    access_point_channel_cleanup (ap->supported_rates);

    ap->supported_channels = NULL;
    ap->supported_frequencies = NULL;
    ap->supported_rates = NULL;

    ret = access_point_get_hwinfo (ap, "channels");
    assert_if_fail (ret == 0);
    ret = access_point_get_hwinfo (ap, "frequencies");
    assert_if_fail (ret == 0);
    access_point_update_property (ap->dbusservice, "SupportedChannels");

    ret = access_point_get_hwinfo (ap, "rates");
    assert_if_fail (ret == 0);
    if (!ret)
        access_point_update_property (ap->dbusservice, "HWFeatures");

    /* We have a regulatory domain change. First check whether
     * we are affected i.e., the current operating channel is not
     * supported in the changed reg domain. In such hostapd does
     * not act and kernel does stop the beaconing and all transmissions
     * but the device is kept configured. In such disable the AP,
     * and reconfigure if required. If the current operating channel
     * is also supported in the changed reg domain, then everyone is happy */
    ret = access_point_teardown_required (ap, frequencies, channels, len);
    if (ret != 0)
        return;

    /* It is very important we add these works to AP's work queue for
     * sanity i.e., to prevent the external requests before we reconfigure
     * ourselves */
    ap_process_request (ap, 0, TASK_AP_RECONFIGURE_ON_REGDOM_CHANGE, NULL);
    if (ap->actualpowerstate)
        ap_process_request (ap, 0, TASK_AP_SET_POWERED, GINT_TO_POINTER (1));
}

static void
handle_ap_operatingchannel_changed (const char *interface,
                                    const unsigned int frequency,
                                    const unsigned int channel)
{
    int ret;
    struct access_point *ap;

    DEBUG ("Accesspoint interface: %s changed frequency details: [%u channel: %u]",
           interface, frequency, channel);

    ap = g_hash_table_lookup (access_points, interface);
    return_if_fail (ap);
    ret = access_point_set_operating_channel (ap, channel);
    if (ret < 0)
        ERROR ("Failed to set the operating channel: %s", strerror (-ret));
}

static void
handle_interface_up (const char *interface)
{
    DEBUG ("Accesspoint interface: %s", interface);
}

static void
handle_interface_down (const char *interface)
{
    DEBUG ("Accesspoint interface: %s", interface);
}

static void
handle_ap_started (const char *interface)
{
    int ret;
    struct access_point *ap;

    DEBUG ("Accesspoint interface: %s", interface);

    ap = g_hash_table_lookup (access_points, interface);
    return_if_fail (ap);
    access_point_set_power_state (ap, POWERED_ON);
    ret = access_point_set_powered (ap, TRUE);
    if (ret < 0)
        ERROR ("Failed to set the powered state: %s", strerror (-ret));
}

static void
handle_ap_stopped (const char *interface)
{
    int ret;
    struct access_point *ap;

    DEBUG ("Accesspoint interface: %s", interface);

    ap = g_hash_table_lookup (access_points, interface);
    return_if_fail (ap);
    access_point_set_power_state (ap, POWERED_OFF);
    ret = access_point_set_powered (ap, FALSE);
    if (ret < 0)
        ERROR ("Failed to set the powered state: %s", strerror (-ret));
}

static void
handle_ap_station_rejected (const char *interface,
                            const char *station,
                            stationrejectionreason reason)
{
    DEBUG ("Accesspoint: %s station: %s reason: %d", interface, station, reason);
}

static void
handle_ap_acs_event (const char *interface,
                     acsevent event)
{
    (void) interface;
    (void) event;
}

static void
handle_ap_dfs_event (const char *interface,
                     dfsevent event)
{
    (void) interface;
    (void) event;
}

static void
handle_ap_wps_pin_needed (const char *interface,
                          wpsenrollee *device)
{
    (void) interface;
    (void) device;
}

static void
handle_ap_scan_state (const char *wiphy,
					  const char *interface,
                      int state)
{
    struct access_point *ap = NULL;
    GHashTableIter iter;
    gpointer key, value;
    const char *device;
    nl80211_scan_state estate = state;

    return_if_fail (interface && wiphy);

    DEBUG ("Scan state changed for [%s] exposed by device \"%s\": %s", interface,
           wiphy, genl_scan_state_to_string (estate));

    g_hash_table_iter_init (&iter, access_points);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        continue_if_fail (key);
        device = genl_get_phyname (key);
        if (device && !g_strcmp0 (device, wiphy)) {
            ap = g_hash_table_lookup (access_points, key);
            break;
        }
    }

    return_if_fail (ap);

    if (ap->scanstate != estate){
    	ap->scanstate = estate;
		if (NL80211_SCAN_STATE_SCAN_COMPLETED == ap->scanstate)
			ap_task_wakeup_next(ap);
	}
}

static void
handle_dhcp_lease_added (char *interface,
                         char *ipaddress,
                         char *macaddress,
                         char *hostname)
{
    struct access_point *ap;
    struct associated_station *sta;

    DEBUG ("Accesspoint: %s station: %s ipaddress: %s hostname: %s", interface,
           macaddress, ipaddress, hostname);

    ap = g_hash_table_lookup (access_points, interface);
    return_if_fail (ap);
    sta = g_hash_table_lookup (ap->stations, macaddress);
    return_if_fail (sta);
    associated_station_set_hostname (sta, hostname);
    associated_station_set_ipaddress (sta, ipaddress);
    ap_stations_schedule_update (ap, sta, TRUE);
}

static void
handle_dhcp_lease_updated (char *interface,
                           char *ipaddress,
                           char *macaddress,
                           char *hostname)
{
    struct access_point *ap;
    struct associated_station *sta;

    DEBUG ("Accesspoint: %s station: %s ipaddress: %s hostname: %s", interface,
           macaddress, ipaddress, hostname);

    ap = g_hash_table_lookup (access_points, interface);
    return_if_fail (ap);
    sta = g_hash_table_lookup (ap->stations, macaddress);
    return_if_fail (sta);
    associated_station_set_hostname (sta, hostname);
    associated_station_set_ipaddress (sta, ipaddress);
    ap_stations_schedule_update (ap, sta, TRUE);
}

static void
handle_dhcp_lease_deleted (char *interface,
                           char *ipaddress,
                           char *macaddress,
                           char *hostname)
{
    DEBUG ("interface: %s ipaddress: %s macaddress: %s hostname: %s", interface,
           ipaddress, macaddress, hostname);
}

static void
handle_dhcp_server_stopped (int result,
                            char *ifname,
                            void *data)
{
    ap_request_cb (result, ifname, data);
}

static int
handle_service_registration (void *vendor_element,
                             void *userdata,
                             void *invocation)
{
    int ret;
    struct access_point *ap = userdata;
    struct vendor_element *element = vendor_element;

    return_val_if_fail (ap, -ENOKEY);
    return_val_if_fail (element, -ENODATA);

    ret = ap_process_request (ap, invocation, TASK_AP_REGISTER_VENDOR_IE, element);
    if (ret < 0)
        ERROR ("Failed to register the vendor element : %s [%s]", element->ies, strerror (-ret));

    return ret;
}

static int
handle_service_unregistration (void *vendor_element,
                               void *userdata,
                               void *invocation)
{
    int ret;
    struct access_point *ap = userdata;
    struct vendor_element *element = vendor_element;

    return_val_if_fail (ap, -ENOKEY);
    return_val_if_fail (element, -ENODATA);

    ret = ap_process_request (ap, invocation, TASK_AP_UNREGISTER_VENDOR_IE, element);
    if (ret < 0)
        ERROR ("Failed to unregister the vendor element : %s [%s]", element->ies, strerror (-ret));

    return ret;
}

static dhcplease_ops
ap_dhcpserver_callbacks = {
    .clientname = "accesspoint",
    .dhcp_lease_added = handle_dhcp_lease_added,
    .dhcp_lease_updated = handle_dhcp_lease_updated,
    .dhcp_lease_deleted = handle_dhcp_lease_deleted,
    .dhcp_server_stopped = handle_dhcp_server_stopped,
};

static accesspointcallbacks
ap_callbacks = {
    .system_status = handle_ap_system_status,
    .station_associated = handle_ap_station_associated,
    .station_disconnected = handle_ap_station_disconnected,
    .regulatory_change = handle_ap_regulatory_change,
    .operatingchannel_changed = handle_ap_operatingchannel_changed,
    .accesspoint_started = handle_ap_started,
    .accesspoint_stopped = handle_ap_stopped,
    .station_rejected = handle_ap_station_rejected,
    .acs_event = handle_ap_acs_event,
    .dfs_event = handle_ap_dfs_event,
    .wps_pin_needed = handle_ap_wps_pin_needed,
    .scan_state_change = handle_ap_scan_state,
};

static ipconfig_ops
ap_ipconfig_ops = {
    .name = "accesspoint",
    .dev_up = handle_interface_up,
    .dev_down = handle_interface_down,
};

static serviceregops
ap_servicereg_ops = {
    .name = "accesspoint",
    .iftype = NL80211_IFTYPE_AP,
    .register_service = handle_service_registration,
    .unregister_service = handle_service_unregistration,
};

static int
ap_manage_config_files (char *conffiles)
{
    char **temp = NULL;

    if (!conffiles)
        return 0;

    configfile = g_strsplit (conffiles, ",", -1);
    DEBUG ("Total no. of supplied configuration files: %u", g_strv_length (configfile));
    for (temp = configfile; *temp; temp++)
        DEBUG ("Configuration file: %s", *temp);

    return 0;
}

int
ap_init (char *conffiles)
{
    int ret;

    (void) ap_manage_config_files (conffiles);
    stations_notify = g_try_malloc0 (sizeof (*stations_notify));
    return_val_if_fail (stations_notify, -ENOMEM);
    ret = accesspoint_init (&ap_callbacks);
    if (ret < 0) {
        CRITICAL ("Failed to register the access point callbacks : %s",
                  strerror (-ret));
        goto error;
    }

    ret = dhcp_server_register (&ap_dhcpserver_callbacks);
    if (ret < 0) {
        CRITICAL ("Failed to register the dhcp server callbacks : %s",
                  strerror (-ret));
        accesspoint_deinit();
        goto error;
    }

    ret = wireless_tech_driver_register (&ap_ops);
    if (ret < 0) {
        CRITICAL ("Failed to register to the wireless technology : %s",
                  strerror (-ret));
        dhcp_server_unregister (&ap_dhcpserver_callbacks);
        accesspoint_deinit();
        goto error;
    }

    ret = ipconfig_notifier_register (&ap_ipconfig_ops);
    if (ret < 0) {
        CRITICAL ("Failed to register to the wireless technology : %s",
                  strerror (-ret));
        goto deregister;
    }

    ret = vendor_service_driver_register (&ap_servicereg_ops);
    if (ret < 0) {
        CRITICAL ("Failed to register the vendor service driver : %s",
                  strerror (-ret));
        ipconfig_notifier_unregister (&ap_ipconfig_ops);
        goto deregister;
    }

    access_points = g_hash_table_new_full (g_str_hash, g_str_equal,
                                           NULL, access_point_cleanup);

    stations_notify->remove = g_hash_table_new (g_str_hash, g_str_equal);
    stations_notify->add = g_hash_table_new (g_str_hash, g_str_equal);

    return ret;

deregister:
    wireless_tech_driver_unregister (&ap_ops);
    dhcp_server_unregister (&ap_dhcpserver_callbacks);
    accesspoint_deinit();

error:
    g_free (stations_notify);
    stations_notify = NULL;
    return ret;
}

int
ap_deinit ()
{
    int ret;

    DEBUG ("");

    ret = accesspoint_deinit();
    if (ret < 0)
        CRITICAL ("Failed to unregister the access point callbacks");

    ret = dhcp_server_unregister (&ap_dhcpserver_callbacks);
    if (ret < 0)
        CRITICAL ("Failed to unregister the dhcp server callbacks : %s",
                  strerror (-ret));

    ret = wireless_tech_driver_unregister (&ap_ops);
    if (ret < 0)
        CRITICAL ("Failed to unregister to the wireless technology : %s",
                  strerror (-ret));

    ret = ipconfig_notifier_unregister (&ap_ipconfig_ops);
    if (ret < 0)
        CRITICAL ("Failed to unregister to the ipconfig callbacks : %s",
                  strerror (-ret));

    ret = vendor_service_driver_unregister (&ap_servicereg_ops);
    if (ret < 0)
        CRITICAL ("Failed to unregister to the vendor service driver : %s",
                  strerror (-ret));

    g_hash_table_destroy (access_points);
    g_strfreev (configfile);

    if (stations_notify) {
        g_hash_table_destroy (stations_notify->remove);
        g_hash_table_destroy (stations_notify->add);
        g_free (stations_notify);
    }

    return ret;
}

/** @} */
