/**
 * @file utils.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <syslog.h>
#include <libgen.h>
#include <inttypes.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netlink/addr.h>
#include <netlink/attr.h>
#include <glib.h>
#include "inc/utils.h"
#include <log.h>

char*
byte_to_hexstring (const char *bytestring,
                   size_t length)
{
    size_t index = 0;
    char buf [BUFSIZ], temp [2];

    return_val_if_fail (bytestring && length, NULL);

    memset (buf, 0, sizeof (buf));
    for ( ; index < length; index++) {
        memset (temp, 0, sizeof (temp));
        sprintf (temp, "%02x", bytestring [index]);
        strcat (buf, temp);
    }

    INFO ("converted hex string : %s", buf);
    return g_strdup (buf);
}

int
is_hex_string (const char *str,
			   const unsigned long length)
{
    char byte, cont;
    unsigned long index = 0;

    return_val_if_fail (length && str, -EINVAL);

    for ( ; index < length; index++) {
        cont = 0;
        byte = str [index];
        if ((byte >= '0' && byte <= '9') || (byte >= 'a' && byte <= 'f') ||
                (byte >= 'A' && byte <= 'F'))
            cont = 1;

        if (!cont)
            return -EINVAL;
    }

    return 0;
}

static const char*
clockid_to_string (const clockid_t id)
{
    switch (id) {
    case CLOCK_REALTIME:
        return ENUMTOSTR (REALTIME);
    case CLOCK_MONOTONIC:
        return ENUMTOSTR (MONOTONIC);
    case CLOCK_PROCESS_CPUTIME_ID:
        return ENUMTOSTR (PROCESS_CPUTIME_ID);
    case CLOCK_THREAD_CPUTIME_ID:
        return ENUMTOSTR (THREAD_CPUTIME_ID);
    }

    return ENUMTOSTR (UNKNOWN);
}

static int
get_time (struct timespec *ts,
          clockid_t id)
{
    int ret;
    struct timespec tmsp;

    return_val_if_fail (ts, -EINVAL) ;

    ret = clock_gettime (id, &tmsp);
    if (ret < 0) {
        ERROR ("Failed to get the time, clock type: %s, error: %s/%d",
               clockid_to_string (id), strerror (errno), errno);
    } else {
        ts->tv_sec = tmsp.tv_sec;
        ts->tv_nsec = tmsp.tv_nsec;
    }

    return ret;
}

int
get_monotonic_time (struct timeval *val)
{
    int ret;
    struct timespec ts;

    return_val_if_fail (val, -EINVAL);

    ret = get_time (&ts, CLOCK_MONOTONIC);
    if (!ret) {
       val->tv_sec = ts.tv_sec;
       val->tv_usec = ts.tv_nsec / 1000;
    }

    return ret;
}

int
get_real_time (struct timeval *val)
{
    int ret;
    struct timespec ts;

    return_val_if_fail (val, -EINVAL);

    ret = get_time (&ts, CLOCK_REALTIME);
    if (!ret) {
        val->tv_sec = ts.tv_sec;
        val->tv_usec = ts.tv_nsec / 1000;
    }

    return ret;
}

int
get_time_difference (struct timeval *tv1,
                     struct timeval *tv2,
                     struct timeval *res)
{
    return_val_if_fail (tv1 && tv2 && res,
                        -EINVAL);

    res->tv_sec = tv1->tv_sec - tv2->tv_sec;
    res->tv_usec = tv1->tv_usec - tv2->tv_usec;
    if (res->tv_usec < 0) {
        res->tv_sec--;
        res->tv_usec += 1000000;
    }

    return 0;
}

unsigned char
calculate_netmask (const char *netmask)
{
    in_addr_t mask;
    in_addr_t host;
    unsigned char bits = 0;

    return_val_if_fail (netmask, 32);

    mask = inet_network (netmask);
    host = ~mask;

    /* a valid netmask must be 2^n - 1 */
    if ((host & (host + 1)) != 0)
        return 255;

    for ( ; mask; mask <<= 1)
        ++bits;

    return bits;
}

int
file_exists (const char *filepath,
             filetype type)
{
    int ret;
    struct stat st;

    return_val_if_fail (filepath, -EINVAL);

    if ((ret = stat (filepath, &st)) < 0)
        return -errno;

    switch (type) {
    default:
    case NUM_FILE_TYPES:
        return -EINVAL;
    case FILE_TYPE_BLK:
        return S_ISBLK (st.st_mode);
    case FILE_TYPE_CHR:
        return S_ISCHR (st.st_mode);
    case FILE_TYPE_DIR:
        return S_ISDIR (st.st_mode);
    case FILE_TYPE_FIFO:
        return S_ISFIFO (st.st_mode);
    case FILE_TYPE_LNK:
        return S_ISLNK (st.st_mode);
    case FILE_TYPE_REG:
        return S_ISREG (st.st_mode);
    case FILE_TYPE_SOCK:
        return S_ISSOCK (st.st_mode);
    }

    return -EINVAL;
}

char*
nl_addr2string (void *nla,
                int family)
{
    struct nlattr *attr = nla;
    struct nl_addr *addr = NULL;
    char addr_str [INET6_ADDRSTRLEN];

    return_val_if_fail (attr, NULL);
    return_val_if_fail (family == AF_INET || family == AF_INET6, NULL);

    memset (addr_str, 0, sizeof (addr_str));
    addr = nl_addr_alloc_attr (attr, family);
    return_val_if_fail (addr, NULL);

    nl_addr2str (addr, addr_str, INET6_ADDRSTRLEN);
    nl_addr_put (addr);

    return g_strdup (addr_str);
}

int
calculate_argc (char **argv,
                int *length)
{
    int index = 0;

    return_val_if_fail (argv && length, -EINVAL);

    for ( ; *argv != NULL; argv++, index++)
        ;
    *length = index;
    return 0;
}

int
convert_strtoumax (const char *string,
                   unsigned int *value)
{
    uintmax_t temp;

    return_val_if_fail (string && value, -EINVAL);

    temp = strtoumax (string, NULL, 10);
    if ((errno == ERANGE && (temp == UINTMAX_MAX)) ||
            (errno != 0 && temp == 0))
        return -ERANGE;

    *value = (unsigned int) temp;
    return 0;
}

int
get_family_from_addr (const char *address)
{
    int ret, family;
    struct addrinfo hints;
    struct addrinfo *addr = NULL;

    return_val_if_fail (address, -EINVAL);

    memset (&hints, 0, sizeof (struct addrinfo));

    hints.ai_family = AF_UNSPEC;
    hints.ai_flags = AI_NUMERICHOST;

    ret = getaddrinfo (address, NULL, &hints, &addr);
    if (!ret) {
        family = addr->ai_family;
        freeaddrinfo (addr);
        return family;
    }

    freeaddrinfo (addr);
    return ret;
}

int
nl_add_address (void *data,
                int family,
                int attr,
                const char *ipaddress)
{
    int ret;
    struct nl_addr *addr;
    char *buf;
    size_t len = 0;
    struct nl_msg *msg = data;

    return_val_if_fail (msg && ipaddress, -EINVAL);

    if (AF_INET != family && AF_INET6 != family)
        return -EINVAL;

    if (AF_INET == family)
        len = sizeof (struct in_addr);
    else if (AF_INET6 == family)
        len = sizeof (struct in6_addr);

    buf = g_try_malloc0 (len);
    return_val_if_fail (buf, -ENOMEM);

    ret = inet_pton (family, ipaddress, buf);
    if (ret <= 0) {
        g_free (buf);
        return -EINVAL;
    }

    addr = nl_addr_build (family, (void *) buf, len);
    if (!addr) {
        g_free (buf);
        return -ENOMEM;
    }

    ret = nla_put (msg, attr, (int) nl_addr_get_len (addr),
                   nl_addr_get_binary_addr (addr));
    if (ret < 0)
        ERROR ("Failed to add the attibute [%d] to the message: %s/%d",
               attr, nl_geterror (-ret), -ret);

    g_free (buf);
    nl_addr_put (addr);
    return ret;
}

int
is_valid_regdom (const char *alpha2)
{
    return_val_if_fail (alpha2 && strlen (alpha2) == 2,
                        -EINVAL);

#define IS_UPPER_ALPHA(A) (A >= 65 && A <= 90)
    if ((alpha2 [0] == 48 && alpha2 [1] == 48) ||
            (IS_UPPER_ALPHA (alpha2 [0]) && IS_UPPER_ALPHA (alpha2 [1])))
        return 0;
    return -EINVAL;
#undef  IS_UPPER_ALPHA
}

int
objpath_from_addr (const char *address,
                   char *objpath)
{
    size_t size;

    return_val_if_fail (address && objpath, -EINVAL);

    size = strlen (address);
    memcpy (objpath, address, size);

    for ( ; *objpath; objpath++)
        if (*objpath == ':')
            *objpath = '_';

    return 0;
}


/** @} */
