/**
 * @file dbus-cli.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <string.h>
#include <log.h>
#include <ctype.h>
#include "inc/utils.h"
#include <dbus-cli.h>
#include <org-freedesktop-DBus.h>

#define ORG_FREEDESKTOP_GET_NAME_OWNER          "GetNameOwner"
#define ORG_FREEDESKTOP_NAME_HAS_OWNER          "NameHasOwner"
#define ORG_FREEDESKTOP_NAME_OWNER_CHANGED      "NameOwnerChanged"

#define ORG_FREEDESKTOP_NO_OWNER                "NoOwner"

typedef
struct __client_data
{
    dbus_callback callback;
    void *userdata;
} clientdata;

typedef
struct __dbus_watcher
{
    unsigned int watcherid;
    connect_func connect;
    disconnect_func disconnect;
    void *userdata;
} dbus_watcher;

typedef
struct __dbus_client
{
    char *name;
    char *owner;
    GSList *watchers;
} dbus_client;

static guint watch;
static GDBusProxy *dbusproxy;
static GSList *dbuswatchers;
static unsigned int listenerid;

static void
dbus_cleanup_clientdata (gpointer data)
{
    g_free (data);
}

static void
dbus_client_cb_cleanup (gpointer data)
{
    g_free (data);
}

static void
dbus_watch_client_cleanup (gpointer data)
{
    dbus_client *client = data;

    return_if_fail (client);
    DEBUG ("Client Name: %s Owner: %s",
           client->name ? client->name : "Unknown",
           client->owner ? client->owner : "Unknown");
    g_slist_free_full (client->watchers, dbus_client_cb_cleanup);
    g_free (client->name);
    g_free (client->owner);
    g_free (client);
}

static GDBusProxy*
dbus_create_proxy (const char *name,
                   const char *interface,
                   const char *path,
                   GDBusConnection *connection)
{
    GDBusProxy *proxy = NULL;
    GError *error = NULL;

    return_val_if_fail (name && interface && path && connection, NULL);

    INFO ("DBus Connection: %p Name: %s Interface: %s Path: %s",
          connection, name, interface, path);

    proxy = g_dbus_proxy_new_sync (connection,
                                   G_DBUS_PROXY_FLAGS_NONE,
                                   NULL, name,
                                   path, interface,
                                   NULL, &error);
    if (!proxy) {
        ERROR ("Failed to create proxy for the interface %s, "
               "error message is: %s", interface, error->message);
        g_error_free (error);
        return NULL;
    }

    INFO ("Proxy successfully created for the interface : %s",
          interface);
    return proxy;
}

static void
dbus_signal_emitted (GDBusProxy *proxy,
                     gchar      *sender,
                     gchar      *signalname,
                     GVariant   *value,
                     gpointer    userdata)
{
    (void) proxy;
    (void) userdata;

    int change = -1;
    GSList *watchers, *clients;
    dbus_client *client;
    dbus_watcher *watcher;
    char *name = NULL, *newowner = NULL,
            *oldowner = NULL;

    INFO ("Signal Info: \"%s\" Sender: \"%s\" Signature: \"%s\"", sender,
          signalname, g_variant_get_type_string (value));

    if (g_strcmp0 (signalname, ORG_FREEDESKTOP_NAME_OWNER_CHANGED) ||
            !g_variant_check_format_string (value, "(sss)", FALSE))
        return;

    g_variant_get (value, "(sss)", &name, &newowner, &oldowner);

    INFO ("Unique Name: \"%s\" newowner: \"%s\" oldowner: \"%s\"",
          name ? name : "unknown", newowner, oldowner);

    clients = dbuswatchers;
    for ( ; clients; clients = clients->next) {

        client = clients->data;
        continue_if_fail (client && client->name && !g_strcmp0 (client->name, name));

        if (!client->owner && strlen (oldowner) && !strlen (newowner)) {
            change = 0;
            client->owner = g_strdup (newowner);
        }
        else if (client->owner && !strlen (oldowner) &&
                 !g_strcmp0 (client->owner, newowner) && strlen (newowner)) {
            change = 1;
            g_free (client->owner);
            client->owner = NULL;
        }

        continue_if_fail (change == 0 || change == 1);
        watchers = client->watchers;
        for ( ; watchers; watchers = watchers->next) {

            watcher = watchers->data;
            continue_if_fail (watcher);
            if (!change) {
                if (watcher->connect)
                    watcher->connect (client->name ?
                                          client->name : client->owner,
                                      watcher->userdata);
            }
            else if (change == 1) {
                if (watcher->disconnect)
                    watcher->disconnect (client->name ?
                                             client->name : newowner,
                                         watcher->userdata);
            }
        }
    }
}

static void
dbus_name_appeared (GDBusConnection *connection,
                    const gchar *name,
                    const gchar *name_owner,
                    gpointer user_data)
{
    (void) user_data;

    size_t id;

    INFO ("Name %s is owned by %s", name, name_owner);

    return_if_fail (!dbusproxy);
    dbusproxy = dbus_create_proxy (name, ORG_FREEDESKTOP_DBUS_IFACE_NAME,
                                   ORG_FREEDESKTOP_DBUS_OBJ_PATH, connection);
    return_if_fail (dbusproxy);
    id = g_signal_connect (dbusproxy, "g-signal",
                           G_CALLBACK (dbus_signal_emitted),
                           NULL);
    if (id <= 0) {
        CRITICAL ("Failed to connect the signals for the proxy : %s",
                  ORG_FREEDESKTOP_DBUS_IFACE_NAME);
    }
}

static void
dbus_name_vanished (GDBusConnection *connection,
                    const gchar *name,
                    gpointer user_data)
{
    (void) connection;
    (void) user_data;

    INFO("Name %s does not exist anymore", name);

    return_if_fail (dbusproxy);
    g_object_unref (dbusproxy);
    dbusproxy = NULL;
}

static void
dbus_nameowner_callback (GObject *object,
                         GAsyncResult *res,
                         gpointer user_data)
{
    GVariant *result;
    GDBusProxy *proxy;
    clientdata *data = user_data;
    GError *error = NULL;

    DEBUG ("");

    proxy = (GDBusProxy *) object;
    return_if_fail (proxy);
    result = g_dbus_proxy_call_finish (proxy, res, &error);
    if (error) {
        ERROR ("\"%s\" Call failed : %s",
               ORG_FREEDESKTOP_NAME_HAS_OWNER, error->message);
        g_error_free (error);
        result = g_variant_new_boolean (FALSE);
    }

    if (data && data->callback)
        data->callback (result, data->userdata);

    g_variant_unref (result);
    dbus_cleanup_clientdata (data);
}

int
dbus_has_name_owner (const char *const name,
                     dbus_callback cb,
                     void *userdata)
{
    clientdata *data;

    return_val_if_fail (name && cb, -EINVAL);
    return_val_if_fail (dbusproxy, -ENOTCONN);

    DEBUG ("Checking if there exists a unique name for: %s", name);

    data = g_try_malloc0 (sizeof (*data));
    return_val_if_fail (data, -ENOMEM);

    data->callback = cb;
    data->userdata = userdata;

    g_dbus_proxy_call (dbusproxy,
                       ORG_FREEDESKTOP_NAME_HAS_OWNER,
                       g_variant_new_string (name),
                       G_DBUS_CALL_FLAGS_NONE,
                       -1, NULL,
                       dbus_nameowner_callback,
                       data);

    return 0;
}

static void
dbus_getnameowner_callback (GObject *object,
                            GAsyncResult *res,
                            gpointer user_data)
{
    GVariant *result;
    GDBusProxy *proxy;
    clientdata *data = user_data;
    GError *error = NULL;

    proxy = (GDBusProxy *) object;
    return_if_fail (proxy);

    result = g_dbus_proxy_call_finish (proxy, res, &error);
    if (error) {
        ERROR ("\"%s\" Call failed : %s",
               ORG_FREEDESKTOP_GET_NAME_OWNER, error->message);
        g_error_free (error);
        result = g_variant_new_string (ORG_FREEDESKTOP_NO_OWNER);
    }

    if (data && data->callback)
        data->callback (result, data->userdata);

    g_variant_unref (result);
    dbus_cleanup_clientdata (data);
}


int
dbus_get_name_owner (const char *const name,
                     dbus_callback cb,
                     void *userdata)
{
    clientdata *data;

    return_val_if_fail (name && cb, -EINVAL);
    return_val_if_fail (dbusproxy, -ENOTCONN);

    DEBUG ("Get Name Owner for: %s", name);

    data = g_try_malloc0 (sizeof (*data));
    return_val_if_fail (data, -ENOMEM);

    data->callback = cb;
    data->userdata = userdata;

    g_dbus_proxy_call (dbusproxy,
                       ORG_FREEDESKTOP_GET_NAME_OWNER,
                       g_variant_new_string (name),
                       G_DBUS_CALL_FLAGS_NONE,
                       -1, NULL,
                       dbus_getnameowner_callback,
                       data);

    return 0;
}

static dbus_client*
dbus_get_client (const char *const service,
                 const gboolean isowner)
{
    GSList *temp;
    char *name;
    dbus_client *client;

    return_val_if_fail (service, NULL);

    temp = dbuswatchers;
    for ( ; temp; temp = temp->next) {

        client = temp->data;
        continue_if_fail (client);

        isowner == TRUE ? (name = client->owner) :
                (name = client->name);

        if (!g_strcmp0 (service, name))
            return client;
    }

    return NULL;
}

static void
dbus_client_get_owner_cb (GVariant *result,
                          void *userdata)
{
    const char *owner;
    gsize length = 0;
    dbus_client *client = userdata;
    GSList *watchers;
    dbus_watcher *watcher;

    return_if_fail (client);

    owner = g_variant_get_string (result, &length);

    return_if_fail (length);

    INFO ("Service: %s Unique name: %s", client->name, owner);

    client->owner = g_strdup (owner);
    watchers = client->watchers;
    for ( ; watchers; watchers = watchers->next) {
        watcher = watchers->data;
        if (watcher && watcher->connect)
            watcher->connect (client->name, watcher->userdata);
    }
}

static int
dbus_is_unique_watcher (dbus_client *client,
                        connect_func connect,
                        disconnect_func disconnect)
{
    GSList *watchers;
    dbus_watcher *watcher;

    return_val_if_fail (client, -EINVAL);
    watchers = client->watchers;
    for ( ; watchers; watchers = watchers->next) {
        watcher = watchers->data;
        if (watcher && (watcher->connect == connect || watcher->disconnect == disconnect))
            return -EEXIST;
    }

    return 0;
}

static int
dbus_is_alpha (const char *name)
{
    size_t index = 0, length = 0;

    return_val_if_fail (name, -EINVAL);

    length = strlen (name);
    for ( ; index < length; index++)
        if (!isalpha (name [index]))
            return -EINVAL;

    return 0;
}

static int
dbus_is_numeric (const char *name)
{
    size_t index = 0, length = 0;

    return_val_if_fail (name, -EINVAL);

    length = strlen (name);
    for ( ; index < length; index++)
        if (!isdigit (name [index]))
            return -EINVAL;

    return 0;
}

static int
dbus_is_valid_element (const char *name)
{
    size_t index = 0;

    return_val_if_fail (name, -EINVAL);

    /* Each element must only contain the ASCII
     * characters "[A-Z][a-z][0-9]_-" */
    for ( ; index < strlen (name); index++)
        if (!isalnum (name [index]) || (name [index] != '_'
                && name [index] != '-'))
            return -EINVAL;

    return 0;
}

static int
dbus_is_valid_bus_name (const char *const name)
{
    int valid = 0;
    char temp [256], **strv;
    size_t index = 0, length = 0, invalid = 0;

    /* D-Bus specification states that the length no
     * name (including interface, member) shall exceed
     * the length of 255 */
    return_val_if_fail (name && strlen (name) <= 255, -EINVAL);

    if (name [0] == ':') {
        memset (temp, 0, sizeof (temp));
        for ( ; index < strlen (name); index++)
            if (isalpha (name [index]) || isdigit (name [index]))
                temp [length++] = name [index];
        if (dbus_is_alpha (temp) < 0 && !dbus_is_numeric (temp))
            valid = 1;
    } else {
        /* Bus names must contain at least one '.' (period)
         * character (and thus at least two elements). */
        strv = g_strsplit (name, ".", -1);
        if (g_strv_length (strv) >= 2) {
            for (index = 0; index < g_strv_length (strv); index++)
                if (dbus_is_valid_element (strv [index]) < 0) {
                    invalid = 1;
                    break;
                }
            if (!invalid)
                valid = 1;
        }
        g_strfreev (strv);
    }

    if (valid)
        return 0;
    return -EINVAL;
}

unsigned int
dbus_add_service_watch (const char *const service,
                        connect_func connect,
                        disconnect_func disconnect,
                        void *data)
{
    int ret;
    dbus_client *client;
    dbus_watcher *watcher;
    gboolean newclient = FALSE,
            isowner = FALSE;

    return_val_if_fail (service, 0);
    return_val_if_fail (connect || disconnect, 0);
    return_val_if_fail (!dbus_is_valid_bus_name (service), 0);

    DEBUG ("Service Name: %s connect: %p disconnect: %p data: %p",
           service, connect, disconnect, data);

    if (service [0] == ':')
        isowner = TRUE;

    client = dbus_get_client (service, isowner);
    if (!client) {

        client = g_try_malloc0 (sizeof (*client));
        return_val_if_fail (client, 0);

        if (isowner) {
            client->owner = g_strdup (service);
        } else
            client->name = g_strdup (service);

        dbuswatchers = g_slist_append (dbuswatchers, client);
        newclient = TRUE;
    }

    if (!newclient) {
        ret = dbus_is_unique_watcher (client, connect, disconnect);
        return_val_if_fail (!ret, 0);
    }

    watcher = g_try_malloc0 (sizeof (*watcher));
    if (!watcher)
        goto failure;

    if (!isowner && newclient) {
        ret = dbus_get_name_owner (client->name, dbus_client_get_owner_cb,
                                   client);
        if (ret < 0) {
            ERROR ("Failed to get the unique name for the service : %s "
                   "[error: %s/%d]", client->name, strerror (-ret), -ret);
            goto failure;
        }
    }

    watcher->connect = connect;
    watcher->disconnect = disconnect;
    watcher->userdata = data;
    watcher->watcherid = ++listenerid;

    client->watchers = g_slist_append (client->watchers, watcher);
    if (!newclient && client->owner && watcher->connect)
        watcher->connect (client->name, watcher->userdata);

    return watcher->watcherid;

failure:
    dbus_watch_client_cleanup (client);
    return 0;
}

int
dbus_rm_watch (const unsigned int id)
{
    int found = 0;
    dbus_client *client;
    dbus_watcher *watcher;
    GSList *watchers, *clients;

    return_val_if_fail (id > 0, -EINVAL);

    DEBUG ("Removing client with watch ID: %u", id);

    clients = dbuswatchers;
    for ( ; clients; clients = clients->next) {

        client = clients->data;
        watchers = (client != NULL) ? client->watchers : NULL;
        for ( ; watchers; watchers = watchers->next) {
            watcher = watchers->data;
            if (watcher && watcher->watcherid == id) {
                found = 1;
                break;
            }
        }

        if (found) {
            client->watchers = g_slist_remove (client->watchers, watcher);
            g_free (watcher);
            break;
        }
    }

    if (found && !g_slist_length (client->watchers)) {
        dbuswatchers = g_slist_remove (dbuswatchers, client);
        dbus_watch_client_cleanup (client);
    }

    return found == 1 ? 0 : -ENOENT;
}

int
dbus_rm_service_watch (const char *const service,
                       connect_func connect,
                       disconnect_func disconnect)
{
    dbus_client *client;
    dbus_watcher *watcher;
    gboolean remove = FALSE,
            isowner = FALSE;
    GSList *watchers;

    return_val_if_fail (service, -EINVAL);
    return_val_if_fail (connect || disconnect, -EINVAL);

    DEBUG ("Service Name: %s connect: %p disconnect: %p",
           service, connect, disconnect);

    if (service [0] == ':')
        isowner = TRUE;

    client = dbus_get_client (service, isowner);
    return_val_if_fail (client, -ENOENT);

    watchers = client->watchers;
    for ( ; watchers; watchers = watchers->next) {
        watcher = watchers->data;
        if (watcher && (watcher->connect == connect || watcher->disconnect == disconnect)) {
            remove = TRUE;
            break;
        }
    }

    if (remove) {
        client->watchers = g_slist_remove (client->watchers, watcher);
        g_free (watcher);

        if (!g_slist_length (client->watchers)) {
            dbuswatchers = g_slist_remove (dbuswatchers, client);
            dbus_watch_client_cleanup (client);
        }
    }

    return remove == TRUE ? 0 : -ENOENT;
}

int
dbus_cli_init (void)
{
    DEBUG ("");

    watch = g_bus_watch_name (G_BUS_TYPE_SYSTEM,
                              ORG_FREEDESKTOP_DBUS_BUS_NAME,
                              G_BUS_NAME_WATCHER_FLAGS_NONE,
                              dbus_name_appeared,
                              dbus_name_vanished,
                              NULL, NULL);

    if (!watch)
        return -ENOTCONN;

    return 0;
}

int
dbus_cli_deinit ()
{
    DEBUG ("");

    if (watch > 0)
        g_bus_unwatch_name (watch);

    if (dbusproxy)
        g_object_unref (dbusproxy);

    g_slist_free_full (dbuswatchers,
                       dbus_watch_client_cleanup);
    return 0;
}

/** @} */
