/**
 * @file associatedstation_if.c
 * @author RBEI/ECO3-Karthikeyan Madeswaran
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <ap.h>
#include <errno.h>
#include <string.h>
#include <glib.h>
#include <errors.h>
#include <log.h>
#include "inc/utils.h"
#include "inc/declarations.h"
#include <objectmanager_if.h>
#include <accesspoint_if.h>
#include <associatedstation_if.h>
#include <org-bosch-wapdman-accesspoint-generated.h>
#include <org-bosch-wapdman-associatedstation-generated.h>

struct associated_station
{
    char *macaddress;
    char *path;
    char *interface;
    char *ipaddress;
    char *hostname;
    /* to be extended for QoS */

    gboolean registered;
    struct access_point *ap;
    GDBusConnection *conn;
    AssociatedStation *service;
    OrgFreedesktopDBusProperties *properties;
};

static int
assoc_station_property_get_macaddress (const PropertyTable *property,
                                       GVariant **variant,
                                       void *data)
{
    struct associated_station *station = data;

    return_val_if_fail (station && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    *variant = g_variant_new (property->type, station->macaddress);
    return 0;
}

static int
assoc_station_property_get_objectpath (const PropertyTable *property,
                                       GVariant **variant,
                                       void *data)
{
    char *path = NULL;
    struct associated_station *station = data;

    return_val_if_fail (station && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    path = access_point_get_path (station->ap);
    return_val_if_fail (path, -ENODATA);

    *variant = g_variant_new (property->type, path);
    return 0;
}

static int
assoc_station_property_get_ipaddress (const PropertyTable *property,
                                      GVariant **variant,
                                      void *data)
{
    struct associated_station *station = data;

    return_val_if_fail (station && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    *variant = g_variant_new (property->type, station->ipaddress);
    return 0;
}

static int
assoc_station_property_get_hostname (const PropertyTable *property,
                                     GVariant **variant,
                                     void *data)
{
    struct associated_station *station = data;

    return_val_if_fail (station && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    *variant = g_variant_new (property->type, station->hostname);
    return 0;
}

static int
assoc_station_property_get_interface (const PropertyTable *property,
                                      GVariant **variant,
                                      void *data)
{
    struct associated_station *station = data;

    return_val_if_fail (station && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    *variant = g_variant_new (property->type, station->interface);
    return 0;
}

static const PropertyTable assoc_station_properties [] = {
    { "MacAddress", "s", PROPERTY_FLAG_READONLY,
      NULL, assoc_station_property_get_macaddress },
    { "Accesspoint", "o", PROPERTY_FLAG_READONLY,
      NULL, assoc_station_property_get_objectpath },
    { "Interface", "s", PROPERTY_FLAG_READONLY,
      NULL, assoc_station_property_get_interface },
    { "IPAddress", "s", PROPERTY_FLAG_READONLY,
      NULL, assoc_station_property_get_ipaddress },
    { "Hostname", "s", PROPERTY_FLAG_READONLY,
      NULL, assoc_station_property_get_hostname },
    { NULL }
};

static const PropertyTable dbus_properties [] = {
    { NULL }
};

static InterfaceTable assoc_station_interfaces [] = {
    { WAPDMAN_ASSOCIATEDSTATION_INTERFACE_NAME, assoc_station_properties, NULL},
    { ORG_FREEDESKTOP_PROPERTIES, dbus_properties, NULL},
    { NULL }
};

char*
associated_station_get_macaddress (const struct associated_station *station)
{
    return_val_if_fail (station, NULL);
    return station->macaddress;
}

char*
associated_station_get_objectpath (const struct associated_station *station)
{
    return_val_if_fail (station, NULL);
    return station->path;
}

char*
associated_station_get_ipaddress (const struct associated_station *station)
{
    return_val_if_fail (station, NULL);
    return station->ipaddress;
}

char*
associated_station_get_hostname (const struct associated_station *station)
{
    return_val_if_fail (station, NULL);
    return station->hostname;
}

char*
associated_station_get_interface (const struct associated_station *station)
{
    return_val_if_fail (station, NULL);
    return station->interface;
}

int
associated_station_set_ipaddress (struct associated_station *station,
                                  char *ipaddress)
{
    return_val_if_fail (station && ipaddress, -EINVAL);
    g_free (station->ipaddress);
    station->ipaddress = g_strdup (ipaddress);
    return 0;
}

int
associated_station_set_hostname (struct associated_station *station,
                                 char *hostname)
{
    return_val_if_fail (station && hostname, -EINVAL);
    g_free (station->hostname);
    station->hostname = g_strdup (hostname);
    return 0;
}

static gboolean
associated_station_handle_get (OrgFreedesktopDBusProperties *object,
                               GDBusMethodInvocation *invocation,
                               const char *interfacename,
                               const char *propname,
                               gpointer data)
{
    int found = 0;
    GVariant *value = NULL, *var;
    const PropertyTable *properties;
    const InterfaceTable *interfaces;
    struct associated_station *station = data;

    return_val_if_fail (object && invocation && interfacename
                        && propname && station, FALSE);

    DEBUG ("Station: %s Object: %p invocation: %p interface: \"%s\" "
           "property: \"%s\"", station->path ? station->path : "unknown",
           object, invocation, interfacename, propname);

    interfaces = assoc_station_interfaces;
    for ( ; interfaces && interfaces->name; interfaces++)
        if (!g_strcmp0 (interfacename, interfaces->name)) {
            found = 1;
            break;
        }

    if (!found) {
        wapdman_error_invalid_interface (invocation);
        return TRUE;
    }

    found = 0;
    properties = interfaces->properties;
    for ( ; properties && properties->name; properties++)
        if (!g_strcmp0 (properties->name, propname)) {
            found = 1;
            break;
        }

    if (!found) {
        wapdman_error_invalid_property (invocation);
        return TRUE;
    }

    if (!properties->get) {
        wapdman_error_not_implemented (invocation);
        return TRUE;
    }

    found = properties->get (properties, &value, station);
    if (found < 0) {
        ERROR ("Failed to get the \"%s\" value: %s/%d", propname,
               strerror (-found), -found);
        wapdman_error_no_data (invocation);
        return TRUE;
    }

    var = g_variant_new_variant (value);
    org_freedesktop_dbus_properties_complete_get
            (object, invocation, var);

    return TRUE;
}

static gboolean
associated_station_handle_getall (OrgFreedesktopDBusProperties *object,
                                  GDBusMethodInvocation *invocation,
                                  const char *interfacename,
                                  gpointer data)
{
    int found = 0;
    GVariant *value = NULL;
    GVariantBuilder *builder;
    const PropertyTable *properties;
    const InterfaceTable *interfaces;
    struct associated_station *station = data;

    return_val_if_fail (object && invocation && interfacename && station, FALSE);

    DEBUG ("Station: %s Object: %p invocation: %p interface : \"%s\"",
           station->path ? station->path : "unknown",
           object, invocation, interfacename);

    interfaces = assoc_station_interfaces;
    for ( ; interfaces && interfaces->name; interfaces++)
        if (!g_strcmp0 (interfacename, interfaces->name)) {
            found = 1;
            break;
        }

    if (!found) {
        wapdman_error_invalid_interface (invocation);
        return TRUE;
    }

    builder = g_variant_builder_new (G_VARIANT_TYPE_VARDICT);

    properties = interfaces->properties;
    for ( ; properties && properties->name; properties++, value = NULL) {

        continue_if_fail (properties->type && properties->get);
        found = properties->get (properties, &value, station);
        if (found < 0) {
            ERROR ("Failed to get the property: %s [error: %s/%d]",
                   properties->name, strerror (-found), -found);
            continue;
        }

        g_variant_builder_add (builder, "{sv}", properties->name, value);
    }

    org_freedesktop_dbus_properties_complete_get_all
            (object, invocation, g_variant_builder_end (builder));

    g_variant_builder_unref (builder) ;
    return TRUE;
}

int
associated_station_get_properties (struct associated_station *station,
                                   GVariantBuilder *builder)
{
    int ret = 0;
    GVariant *variant = NULL;
    const PropertyTable *properties;

    return_val_if_fail (station && builder, -EINVAL);

    properties = assoc_station_properties;
    for ( ; properties && properties->name; properties++, variant = NULL) {

        continue_if_fail (properties->type && properties->get);
        ret = properties->get (properties, &variant, station);
        if (ret < 0) {
            ERROR ("Failed to get the property value : %s [error: %s/%d]",
                   properties->name, strerror (-ret), -ret);
            continue;
        }

        g_variant_builder_add (builder, "{sv}", properties->name, variant);
    }

    return ret;
}

int
associated_station_register_interfaces (struct associated_station *station)
{
    int ret = 0;
    const InterfaceTable *interface;

    return_val_if_fail (station, -EINVAL);
    return_val_if_fail (!station->registered, -EALREADY);

    interface = assoc_station_interfaces;
    for ( ; interface && interface->name; interface++) {
        ret = objman_register_interfaces (station->path, interface->name,
                                          interface->properties, station);
        if (ret < 0) {
            ERROR ("Failed to register the interface [\"%s\"]"
                   "for the object : %s [error: %s/%d]", interface->name,
                   station->path, strerror (-ret), -ret);
        }
    }

    if (!ret)
        station->registered = TRUE;

    return ret;
}

static int
associated_station_unregister_interfaces (struct associated_station *station)
{
    int ret = 0;
    const InterfaceTable *interface;

    return_val_if_fail (station, -EINVAL);
    return_val_if_fail (station->registered, -EALREADY);

    interface = assoc_station_interfaces;
    for ( ; interface && interface->name; interface++) {
        ret = objman_unregister_interfaces (station->path, interface->name);
        if (ret < 0) {
            ERROR ("Failed to register the interface [\"%s\"]"
                   "for the object : %s [error: %s/%d]", interface->name,
                   station->path, strerror (-ret), -ret);
        }
    }

    if (!ret)
        station->registered = FALSE;

    return ret;
}

int
associated_station_cleanup (gpointer data)
{
    int ret;
    struct associated_station *station = data;

    return_val_if_fail (station, -EALREADY);

    DEBUG ("Station Cleanup: %s", station->macaddress);

    ret = associated_station_unregister_interfaces (station);
    if (ret < 0) {
        ERROR ("Failed to unregister the station interfaces [%s]: %s",
               station->path, strerror (-ret));
    }

    g_object_unref (station->service);
    g_object_unref (station->properties);

    g_free (station->macaddress);
    g_free (station->path);
    g_free (station->interface);
    g_free (station->ipaddress);
    g_free (station->hostname);
    g_free (station);

    return 0;
}

static int
associated_station_service_init (struct associated_station *station)
{
    int ret = -ENOTCONN;
    GError *error = NULL;
    gboolean expt = FALSE;

    return_val_if_fail (station, -EINVAL);
    return_val_if_fail (!station->service && !station->properties, -EEXIST);

    station->service = associated_station_skeleton_new ();
    station->properties = org_freedesktop_dbus_properties_skeleton_new ();

    DEBUG ("Station service path: %s service skeleton: %p "
           "properties skeleton: %p", station->path, station->service,
           station->properties);

    g_signal_connect (station->properties, "handle-get",
                      G_CALLBACK (associated_station_handle_get),
                      station);

    g_signal_connect (station->properties, "handle-get-all",
                      G_CALLBACK (associated_station_handle_getall),
                      station);

    expt = g_dbus_interface_skeleton_export (G_DBUS_INTERFACE_SKELETON (station->service),
                                             station->conn, station->path, &error);
    if (!expt) {
        ERROR ("Failed to export the associated station interface to the dbus : %s",
               error->message);
        g_error_free (error);
        goto error;
    }

    expt = g_dbus_interface_skeleton_export (G_DBUS_INTERFACE_SKELETON (station->properties),
                                             station->conn, station->path, &error);
    if (!expt) {
        ERROR ("Failed to export the associated station property interface to the dbus : %s",
               error->message);
        g_error_free (error);
        goto error;
    }

    return 0;

error:
    g_object_unref (station->service);
    g_object_unref (station->properties);

    station->service = NULL;
    station->properties = NULL;

    return ret;
}

struct associated_station*
associated_station_create (GDBusConnection *conn,
                           struct access_point *ap,
                           const char *macaddress,
                           const char *interface)
{
    int ret;
    char objpath [128];
    char *apobjpath = NULL;
    struct associated_station *station;

    return_val_if_fail (conn && ap && macaddress && interface, NULL);

    apobjpath = strrchr (access_point_get_path (ap), '/');
    if (!apobjpath) {
        ERROR ("AP Path failed for the associated station: %s [iface: %s]",
               macaddress, interface);
        return NULL;
    }

    apobjpath++;
    memset (objpath, 0, sizeof objpath);

    ret = objpath_from_addr (macaddress, objpath);
    if (ret < 0) {
        ERROR ("Failed to compose the object path from the "
               "address: %s [iface: %s] [errpr: %s/%d]",
               macaddress, interface, strerror (-ret), -ret);
        return NULL;
    }

    DEBUG ("Constructed Objpath station [%s] interface [%s]: %s",
           macaddress, interface, objpath);

    station = g_try_malloc0 (sizeof (*station));
    return_val_if_fail (station, NULL);

    station->ap = ap;
    station->conn = conn;
    station->path = g_strdup_printf ("%s/%s/%s/%s_%s",
                                     WAPDMAN_BUS_PATH,
                                     interface,
                                     apobjpath,
                                     WAPDMAN_ASSOCIATEDSTATION_INTERFACE_PATH,
                                     objpath);

    station->macaddress = g_strdup (macaddress);
    station->interface = g_strdup (interface);

    ret = associated_station_service_init (station);
    if (ret < 0) {
        ERROR ("Failed to initialize the associated station service interface : %s",
               strerror (-ret));
        goto error;
    }

    DEBUG ("Created Station [mac: %s] [iface: %s]: %p", macaddress,
           interface, station);
    return station;

error:
    g_free (station->path);
    g_free (station->macaddress);
    g_free (station->interface);
    g_free (station);

    return NULL;
}

int
associated_station_destroy (struct associated_station *station)
{
    int ret;

    return_val_if_fail (station, -EALREADY);

    DEBUG ("Destruct Station [mac: %s] [iface: %s]: %p",
           station->macaddress, station->interface, station);

    ret = associated_station_unregister_interfaces (station);
    if (ret < 0) {
        ERROR ("Failed to unegister the station interfaces : %s",
               strerror(-ret));
    }

    g_free (station->hostname);
    g_free (station->interface);
    g_free (station->ipaddress);
    g_free (station->macaddress);
    g_free (station->path);

    g_object_unref (station->service);
    g_object_unref (station->properties);

    g_free (station);

    return ret;
}

/** @} */
