/**
 * @file fib.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2018 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <getopt.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <linux/rtnetlink.h>
#include <linux/version.h>
#include <net/if.h>
#include <netlink/socket.h>
#include <netlink/msg.h>
#include <glib.h>
#include <rtnl.h>
#include <log.h>
#include <string.h>
#include <route.h>
#include <nlutils.h>
#include "inc/utils.h"
#include <rtnl.h>
#include <ipconfig.h>

#define IP_ROUTE_ATTR_FAMILY    BIT(0)
#define IP_ROUTE_ATTR_TABLE		BIT(1)
#define IP_ROUTE_ATTR_GATEWAY   BIT(2)
#define IP_ROUTE_ATTR_INDEX 	BIT(3)
#define IP_ROUTE_ATTR_SRC    	BIT(4)
#define IP_ROUTE_ATTR_SLEN    	BIT(5)
#define IP_ROUTE_ATTR_DST    	BIT(6)
#define IP_ROUTE_ATTR_DLEN    	BIT(7)

struct __active_route
{
    char *iifname;
    char *oifname;
    char *gateway;
    unsigned int flags;
    unsigned char family;
    unsigned int table;
    unsigned char tos;
    unsigned char proto;
    unsigned char rt_scope;
    unsigned char type;
    unsigned char pref;
    unsigned int oif;
    unsigned int iif;
    unsigned int prio;
    unsigned int mark;
    unsigned int mask;
    int refcnt;
    unsigned char gw;
    char *src;
    char *dst;
    unsigned int flow;
    unsigned char slen, dlen;
};

typedef
struct _nl_socket
{
    struct nl_sock *sk;
    struct nl_cb *cb;
} route_socket;

typedef
enum _route_opcode
{
    IP_ROUTE_OPCODE_INV,
    IP_ROUTE_OPCODE_ADD,
    IP_ROUTE_OPCODE_DEL
} routeop_t;

typedef
struct _route_parser
{
    char **argv;
    routeop_t opcode;
    int argc;
    int refcnt;
    route_t *route;
    unsigned int options;
} routeparser_t;

static struct option route_options [] = {
    { "add",         no_argument,       NULL, 'A'},
    { "del",         no_argument,       NULL, 'D'},
    { "from",        required_argument, NULL, 's'},
    { "to",          required_argument, NULL, 'd'},
    { "gateway",     required_argument, NULL, 'g'},
    { "family",      required_argument, NULL, 'f'},
    { "index",       required_argument, NULL, 'i'},
    { "table",       required_argument, NULL, 't'},
    { NULL },
};

static GHashTable *active_routes;
static GSList *pending_routes;
static route_socket *routesk;

static void
route_parser_unref_impl (routeparser_t *parser,
                         const char *file,
                         int line,
                         const char *caller);
static routeparser_t*
route_parser_ref_impl (routeparser_t *parser,
                       const char *file,
                       int line,
                       const char *caller);

#define route_parser_ref(parser) route_parser_ref_impl(parser, __FILE__, __LINE__, __func__)
#define route_parser_unref(parser) route_parser_unref_impl(parser, __FILE__, __LINE__, __func__)

static const char*
route_opcode2str (routeop_t cmd)
{
    switch (cmd) {
    case IP_ROUTE_OPCODE_INV:
        return "Invalid";
    case IP_ROUTE_OPCODE_ADD:
        return "Add";
    case IP_ROUTE_OPCODE_DEL:
        return "Delete";
    }

    return "Invalid";
}

static void
route_cleanup (void *data)
{
    route_t *route = data;

    return_if_fail (route);

    g_free (route->iifname);
    g_free (route->src);
    g_free (route->dst);
    g_free (route->gateway);
    g_free (route->oifname);

    g_free (route);
}

static void
delete_routes (gpointer data)
{
    route_unref (data);
}

static void
delete_parser (gpointer data)
{
    route_parser_unref (data);
}

static void
routes_cleanup (gpointer data)
{
    g_list_free_full (data, delete_routes);
}

static void
route_parser_cleanup (void *data)
{
    int index = 0;
    routeparser_t *parser = data;

    return_if_fail (parser);

    route_unref (parser->route);
    for ( ; index < parser->argc; index++)
        g_free (parser->argv [index]);
    g_free (parser);
}


static const char*
route_type2str (int type)
{
    switch (type) {
    case RTN_UNSPEC:
        return "unspecified";
    case RTN_UNICAST:
        return "unicast";
    case RTN_LOCAL:
        return "local";
    case RTN_BROADCAST:
        return "broadcast";
    case RTN_ANYCAST:
        return "anycast";
    case RTN_MULTICAST:
        return "multicast";
    case RTN_BLACKHOLE:
        return "blackhole";
    case RTN_UNREACHABLE:
        return "unreachable";
    case RTN_PROHIBIT:
        return "prohibit";
    case RTN_THROW:
        return "throw";
    case RTN_NAT:
        return "nat";
    case RTN_XRESOLVE:
        return "xresolve";
    default:
        return "unknown";
    }
}

route_t*
route_ref_impl (route_t *route,
                const char *file,
                int line,
                const char *caller)
{
    return_val_if_fail (route, NULL);

    DEBUG ("Route %p ref %d by %s:%d:%s()",
           route, route->refcnt + 1,
           file ? file : "unknown",
           line, caller ? caller : "unknown");

    __sync_fetch_and_add (&route->refcnt, 1);
    return route;
}

void
route_unref_impl (route_t *route,
                  const char *file,
                  int line,
                  const char *caller)
{
    return_if_fail (route);

    DEBUG ("Route %p ref %d by %s:%d:%s()",
           route, route->refcnt - 1,
           file ? file : "unknown",
           line, caller ? caller : "unknown");

    if (__sync_fetch_and_sub (&route->refcnt, 1) == 1)
        route_cleanup (route);
}

static void
route_parser_unref_impl (routeparser_t *parser,
                         const char *file,
                         int line,
                         const char *caller)
{
    return_if_fail (parser);

    DEBUG ("Route Parser %p ref %d by %s:%d:%s()",
           parser, parser->refcnt - 1,
           file ? file : "unknown",
           line, caller ? caller : "unknown");

    if (__sync_fetch_and_sub (&parser->refcnt, 1) == 1)
        route_parser_cleanup (parser);
}

static routeparser_t*
route_parser_ref_impl (routeparser_t *parser,
                       const char *file,
                       int line,
                       const char *caller)
{
    return_val_if_fail (parser, NULL);

    DEBUG ("Router Parser %p ref %d by %s:%d:%s()",
           parser, parser->refcnt + 1,
           file ? file : "unknown",
           line, caller ? caller : "unknown");

    __sync_fetch_and_add (&parser->refcnt, 1);
    return parser;
}

static void
dump_route (route_t *route)
{
    return_if_fail (route);

    INFO (LMLN_FMAT"Dumping Route [%p], "
          LMLN_FMAT"family: \"%u\" "
          LMLN_FMAT"Routing table: \"%u\" "
          LMLN_FMAT"iifname: [\"%u\" \"%s\"] "
          LMLN_FMAT"oifname: [\"%u\" \"%s\"] "
          LMLN_FMAT"Gateway: \"%s\" "
          LMLN_FMAT"src: \"%s/%u\" "
          LMLN_FMAT"dst: \"%s/%u\" "
          LMLN_FMAT"TOS: \"%u\" "
          LMLN_FMAT"priority: \"%u\" "
          LMLN_FMAT"fwmark: \"%u\" "
          LMLN_FMAT"fwmask: \"%u\" "
          LMLN_FMAT"class: \"%u\" "
          LMLN_FMAT"scope: \"%u\" "
          LMLN_FMAT"pref: \"%u\" "
          LMLN_FMAT"flags: \"%02x\" %s%s%s%s"
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
          "%s%s"
#endif
          ,
          route,
          route->family,
          route->table,
          route->iif,
          route->iifname ?
              route->iifname : "unknown",
          route->oif,
          route->oifname ?
              route->oifname : "unknown",
          route->gateway ? route->gateway : "",
          route->src ? route->src : "",
          route->slen,
          route->dst ? route->dst : "",
          route->dlen,
          route->tos,
          route->prio,
          route->mark,
          route->mask,
          route->flow,
          route->rt_scope,
          route->pref,
          route->flags,
          !!(route->flags & RTNH_F_DEAD) ? "(dead) " : "",
          !!(route->flags & RTNH_F_ONLINK) ? "(onlink) " : "",
          !!(route->flags & RTNH_F_PERVASIVE) ? "(pervasive) " : "",
          !!(route->flags & RTM_F_NOTIFY) ? "(notify) " : ""
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
          ,
          !!(route->flags & RTNH_F_OFFLOAD) ? "(offload) " : "",
          !!(route->flags & RTNH_F_LINKDOWN) ? "(link down)" : ""
#endif
          );
}

static int
route_alloc (struct nlmsghdr *hdr,
             int family,
             unsigned int scope,
             gboolean chkdst,
             route_t **active_route)
{
    int ret;
    struct nlattr *tb [RTA_MAX + 1];
    route_t *route;
    char ifname [IFNAMSIZ];
    struct in_addr dst = { INADDR_ANY };
    struct in6_addr dst6 = IN6ADDR_ANY_INIT;
    struct nla_policy route_policy [RTA_MAX + 1] = {
        [RTA_IIF]	= { .type = NLA_U32 },
        [RTA_OIF]	= { .type = NLA_U32 },
        [RTA_PRIORITY]	= { .type = NLA_U32 },
        [RTA_FLOW]	= { .type = NLA_U32 },
        [RTA_CACHEINFO]	= { .minlen = sizeof (struct rta_cacheinfo) },
        [RTA_METRICS]	= { .type = NLA_NESTED },
        [RTA_MULTIPATH]	= { .type = NLA_NESTED },
    };

    return_val_if_fail (hdr, -EINVAL);
    return_val_if_fail (active_route, -EINVAL);
    return_val_if_fail (!*active_route, -EALREADY);

    ret = nlmsg_parse (hdr, sizeof (struct rtmsg), tb, RTA_MAX, route_policy);
    return_val_if_fail (ret == 0, -ENOMSG);

    if (chkdst && tb [RTA_DST]) {
        if (AF_INET == family) {
            memcpy (&dst, RTA_DATA (tb [RTA_DST]), sizeof (struct in_addr));
            if (scope != RT_SCOPE_UNIVERSE &&
                    !(scope == RT_SCOPE_LINK && dst.s_addr == INADDR_ANY))
                return -ENOTSUP;

            if (dst.s_addr != INADDR_ANY)
                return -ENOTSUP;
        }
        else if (AF_INET6 == family) {
            memcpy (&dst6, RTA_DATA (tb [RTA_DST]), sizeof (struct in6_addr));
            if (scope != RT_SCOPE_UNIVERSE &&
                !(scope == RT_SCOPE_LINK &&
                    IN6_IS_ADDR_UNSPECIFIED(&dst)))
                return -ENOTSUP;

            if (!IN6_IS_ADDR_UNSPECIFIED(&dst))
                return -ENOTSUP;
        }
    }

    route = g_try_malloc0 (sizeof (*route));
    return_val_if_fail (route, -ENOMEM);

    route = route_ref (route);
    __sync_synchronize();

    if (tb [RTA_FLOW])
        route->mask = nla_get_u32 (tb [RTA_FLOW]);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
    if (tb [RTA_PREF])
        route->pref = nla_get_u8 (tb [RTA_PREF]);
#endif
    if (tb [RTA_MARK])
        route->mark = nla_get_u32 (tb [RTA_MARK]);
    if (tb [RTA_PRIORITY])
        route->prio = nla_get_u32 (tb [RTA_PRIORITY]);
    if (tb [RTA_IIF]) {
        memset (ifname, 0, sizeof (ifname));
        route->iif = nla_get_u32 (tb [RTA_IIF]);
        if (if_indextoname (route->iif, ifname) != 0)
            route->iifname = g_strdup (ifname);
    }
    if (tb [RTA_OIF]) {
        memset (ifname, 0, sizeof (ifname));
        route->oif = nla_get_u32 (tb [RTA_OIF]);
        if (if_indextoname (route->oif, ifname) != 0)
            route->oifname = g_strdup (ifname);
    }
    if (tb [RTA_TABLE])
        route->table = nla_get_u32 (tb [RTA_TABLE]);
    if (tb [RTA_SRC])
        route->src = nl_addr2string (tb [RTA_SRC], family);
    if (tb [RTA_DST])
        route->dst = nl_addr2string (tb [RTA_DST], family);
    if (tb [RTA_GATEWAY])
        route->gateway = nl_addr2string (tb [RTA_GATEWAY], family);

    *active_route = route;
    return 0;
}

static int
route_changed (void *nlhdr,
               gboolean chkdst,
               route_t **route)
{
    int ret, wlan_table = 0;
    struct rtmsg *rhdr;
    struct nlmsghdr *payload = nlhdr;
    route_t *temp = NULL;

    rhdr = nlmsg_data (payload);
    return_val_if_fail (rhdr, -EINVAL);

    if (rhdr->rtm_table == RT_TABLE_MAIN)
        wlan_table = 1;
    else if (rhdr->rtm_table >= 1 && rhdr->rtm_table <= 100)
        wlan_table = 1;

    DEBUG ("Routing table id: \"%u\" Routing Proto: \"%u\" "
           "Family: \"%u\" RTM Type: \"%u %s\" flags: \"%02x\" scope: %u",
           rhdr->rtm_table, rhdr->rtm_protocol,
           rhdr->rtm_family, rhdr->rtm_type,
           route_type2str (rhdr->rtm_type), rhdr->rtm_flags, rhdr->rtm_scope);

    if (AF_INET != rhdr->rtm_family && AF_INET6 != rhdr->rtm_family)
        return -ENOTSUP;
    if (rhdr->rtm_flags & RTM_F_CLONED)
        return -ENOTSUP;
    if (rhdr->rtm_protocol != RTPROT_BOOT &&
            rhdr->rtm_protocol != RTPROT_KERNEL &&
            rhdr->rtm_protocol != RTPROT_STATIC)
        return -ENOTSUP;
    if (rhdr->rtm_type != RTN_UNICAST)
        return -ENOTSUP;

    if (!wlan_table)
        return -ENOTSUP;

    ret = route_alloc (nlhdr, rhdr->rtm_family, rhdr->rtm_scope, chkdst, route);
    if (ret < 0) {
        ERROR ("Failed to create an internal route entry: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    temp = *route;
    temp->table = rhdr->rtm_table;
    temp->tos = rhdr->rtm_tos;
    temp->flags = rhdr->rtm_flags;
    temp->slen = rhdr->rtm_src_len;
    temp->dlen = rhdr->rtm_dst_len;
    temp->family = rhdr->rtm_family;
    temp->type = rhdr->rtm_type;
    temp->rt_scope = rhdr->rtm_scope;
    temp->proto = rhdr->rtm_protocol;

    dump_route (temp);
    return ret;
}

static int
route_handle_newgw (void *nlhdr)
{
    int ret = 0;
    route_t *route = NULL;
    GList *routes = NULL;
    unsigned int oldlength = 0;

    return_val_if_fail (nlhdr, -EINVAL);

    ret = route_changed (nlhdr, TRUE, &route);
    return_val_if_fail (!ret, ret);

    if (!route->gateway) {
        route_unref (route);
        return 0;
    }

    routes = g_hash_table_lookup (active_routes, GUINT_TO_POINTER (route->table));
    if (routes)
        oldlength = g_list_length (routes);
    routes = g_list_append (routes, route);
    g_hash_table_replace (active_routes, GUINT_TO_POINTER (route->table), routes);

    DEBUG ("Size of the available routes in the table \"%u\" old: %u new: %u",
           route->table, oldlength, g_list_length (routes));

    return 0;
}

static int
find_route (GList *temps,
            route_t *temp,
            route_t **fetch)
{
    GList *list;
    int ret = -ENOENT;
    route_t *route;
    struct in_addr dst = { INADDR_ANY };
    struct in_addr gw = { INADDR_ANY };

    return_val_if_fail (temps && temp, -EINVAL);
    return_val_if_fail (fetch, -EINVAL);
    return_val_if_fail (!*fetch, -EINVAL);

    for (list = temps; list; list = list->next) {

        route = list->data;
        if (route->oif != temp->oif)
            continue;
        if (route->family != temp->family)
            continue;
        if (route->type != temp->type)
            continue;
        if (temp->family == AF_INET) {

            if ((!temp->dst || !g_strcmp0 (temp->dst, "/0")
                 || !g_strcmp0 (temp->dst, "")) && (!temp->gateway ||
                    !g_strcmp0 (temp->gateway, "") || !g_strcmp0 (temp->gateway, "/0")))
                continue;

            /* We could be quite sure that we cache only the
             * gateways, nevertheless */
            if (!route->gateway)
                continue;

            if (temp->gateway) {
                if (temp->gateway && g_strcmp0 (route->gateway, temp->gateway) != 0)
                    continue;
            }
            else if (temp->dst) {
                if (!inet_aton (temp->dst, &dst) || !inet_aton (route->gateway, &gw))
                    continue;

                dst.s_addr = ntohl (inet_addr (temp->dst));
                gw.s_addr = ntohl (inet_addr (route->gateway));

                if (((dst.s_addr >> 8) & 0xff) != ((gw.s_addr >> 8) & 0xff))
                    continue;
                if (((dst.s_addr >> 16) & 0xff) != ((gw.s_addr >> 16) & 0xff))
                    continue;
                if (((dst.s_addr >> 24) & 0xff) != ((gw.s_addr >> 24) & 0xff))
                    continue;
            } else {
                INFO ("No match categories found for route: %p", temp);
                continue;
            }
        }

        DEBUG ("Route Match identified: %p == %p", route, temp);
        *fetch = route;
        ret = 0;
        break;
    }

    return ret;
}

static int
find_route_from_oifname (const char *name,
                         route_t **route)
{
    route_t *temp;
    gpointer key, value;
    GHashTableIter iter;
    GHashTable *table = active_routes;
    GList *list;
    int ret = -ENOENT;

    return_val_if_fail (name && route && !*route, -EINVAL);

    g_hash_table_iter_init (&iter, table);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        DEBUG ("Routing table %u, size: %u", GPOINTER_TO_UINT (key),
               g_list_length (value));

        for (list = value; list; list = list->next) {
            temp = list->data;
            continue_if_fail (temp);
            if (temp->gateway && temp->oifname && !g_strcmp0 (temp->oifname, name)) {
                *route = temp;
                ret = 0;
                break;
            }
        }

        if (0 == ret)
            break;
    }

    return ret;
}

static int
find_routes_from_oifname (const char *name,
                          route_t **route,
                          GSList *tables)
{
    route_t *temp;
    gpointer key, value;
    GHashTableIter iter;
    GHashTable *table = active_routes;
    GList *list;
    GSList *slist = NULL;
    int ret = -ENOENT, available = 0;

    return_val_if_fail (name && route && !*route, -EINVAL);

    g_hash_table_iter_init (&iter, table);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        available = 0;
        for (slist = tables; slist; slist = slist->next) {
            if (GPOINTER_TO_UINT (slist->data) == GPOINTER_TO_UINT (key)) {
                available = 1;
                break;
            }
        }

        /* examine next table */
        if (available)
            continue;

        DEBUG ("Routing table: %u, size: %u", GPOINTER_TO_UINT (key),
               g_list_length (value));

        for (list = value; list; list = list->next) {

            temp = list->data;
            continue_if_fail (temp);

            if (temp->gateway && temp->oifname && !g_strcmp0 (temp->oifname, name)) {
                *route = temp;
                ret = 0;
                break;
            }
        }

        if (0 == ret)
            break;
    }

    return ret;
}

static int
route_handle_delgw (void *nlhdr)
{
    int ret = 0;
    GList *routes;
    route_t *route = NULL, *pentry = NULL;
    unsigned int oldlength = 0, table = 0;
    GSList *tables = NULL, *temp;

    return_val_if_fail (nlhdr, -EINVAL);

    ret = route_changed (nlhdr, FALSE, &route);
    return_val_if_fail (!ret, ret);

    /* It is pretty hard, but believe it. If a route gets deleted from
     * kernel, the kernel just sends an update with the main routing table
     * but not for the other routing tables which had this route entry
     * (which kind of makes sense but not sure if this is improved in the
     * latest kernels). Therefore make sure that we have the entries cleaned
     * up properly. */

    while ((ret = find_routes_from_oifname (route->oifname, &pentry, tables)) == 0) {
        tables = g_slist_append (tables, GUINT_TO_POINTER (pentry->table));
        pentry = NULL;
    }

    /* We have now the routing tables and have to iterate through them
     * to find the match */
    pentry = NULL;
    for (temp = tables; temp; temp = temp->next, pentry = NULL) {

        table = GPOINTER_TO_UINT (temp->data);
        routes = g_hash_table_lookup (active_routes, temp->data);

        INFO ("Examining the routing table: %u routes available: %u",
              table, routes ? g_list_length (routes) : 0);

        while ((ret = find_route (routes, route, &pentry)) == 0) {
            if (routes) {
                oldlength = g_list_length (routes);
                if (g_list_length (routes) == 1) {
                    g_hash_table_remove (active_routes, GUINT_TO_POINTER (table));
                    g_list_free_full (routes, delete_routes);
                    routes = NULL;
                } else {
                    routes = g_list_remove (routes, pentry);
                    g_hash_table_replace (active_routes, GUINT_TO_POINTER (table), routes);
                    route_unref(pentry);
                }
                DEBUG ("Route %p has been deleted from the internal list of "
                       "table: \"%u\", old: %u new: %u", pentry, table,
                       oldlength, g_list_length (routes));
            } else {
                DEBUG ("The route list of table %u does not have the route", table);
                break;
            }
        }
    }

    route_unref (route);
    return ret;
}

static int
route_try_allocate_parser (routeparser_t **parser)
{
    route_t *temp;
    routeparser_t *tparser;

    return_val_if_fail (parser && !*parser, -EINVAL);

    tparser = g_try_malloc0 (sizeof (*tparser));
    return_val_if_fail (tparser, -ENOMEM);

    tparser = route_parser_ref (tparser);
    temp = g_try_malloc0 (sizeof (*temp));
    if (!temp) {
        route_parser_unref (tparser);
        return -ENOMEM;
    }

    temp = route_ref (temp);
    tparser->route = temp;

    *parser = tparser;
    return 0;
}

static int
prepare_route_arguments (const char *rule,
                         routeparser_t **parser)
{
    int ret, index;
    int length = 0;
    char **argv;
    routeparser_t *temp = NULL;

    return_val_if_fail (rule && parser && !*parser, -EINVAL);

    ret = route_try_allocate_parser (&temp);
    return_val_if_fail (ret == 0, -ENOMEM);

    argv = g_strsplit_set (rule, " ", -1);

    ret = calculate_argc (argv, &length);
    if (ret < 0) {
        route_parser_unref (temp);
        return ret;
    }

    temp->argc = length + 1;
    temp->argv = g_try_malloc0 ((unsigned int) (temp->argc + 1)
                                * sizeof(char *));
    if (!temp->argv) {
        route_parser_unref (temp);
        g_strfreev (argv);
        return -ENOMEM;
    }

    temp->argv[0] = g_strdup ("argh");
    for (index = 1; index < temp->argc; index++)
        temp->argv [index] = argv [index-1];

    *parser = temp;
    g_free (argv);
    return ret;
}

static int
parse_route (const char *rstring,
             routeparser_t *rparser)
{
    char **address;
    int ret, cleanup = 0, bk,
            length = 0;
    unsigned int value = 0;
    char oif [IF_NAMESIZE];

    return_val_if_fail (rstring && rparser, -EINVAL);

    opterr = 0;
    optind = 0;

    while ((ret = getopt_long (rparser->argc,
                               rparser->argv,
                               "ADs:d:g:f:i:t:",
                               route_options, NULL)) != -1) {

        DEBUG ("Supplied option: %c argument: %s", ret, optarg);

        switch (ret) {
        case 'A':
            rparser->opcode = IP_ROUTE_OPCODE_ADD;
            break;

        case 'D':
            rparser->opcode = IP_ROUTE_OPCODE_DEL;
            break;

        case 's':
        case 'd':
        {
            bk = ret;
            length = 0;
            value = 0;
            address = g_strsplit_set (optarg, "/", -1);

            ret = calculate_argc (address, &length);
            if (ret != 0 || length > 2) {
                g_strfreev (address);
                cleanup = 1;
                break;
            }

            ret = get_family_from_addr (address [0]);
            if (ret < 0 || (ret != AF_INET &&
                            ret != AF_INET6)) {
                g_strfreev (address);
                cleanup = 1;
                break;
            }

            if (length == 2) {
                ret = convert_strtoumax (address [1], &value);
                if (ret < 0 || value > 255) {
                    g_strfreev (address);
                    cleanup = 1;
                    break;
                }
            }

            switch (bk) {
            case 's':
            {
                rparser->route->src = g_strdup (address [0]);
                rparser->options |= IP_ROUTE_ATTR_SRC;

                if (value) {
                    rparser->route->slen = (unsigned char) value;
                    rparser->options |= IP_ROUTE_ATTR_SLEN;
                }
            }
            break;

            case 'd':
            {
                rparser->route->dst = g_strdup (address [0]);
                rparser->options |= IP_ROUTE_ATTR_DST;

                if (value) {
                    rparser->route->dlen = (unsigned char) value;
                    rparser->options |= IP_ROUTE_ATTR_DLEN;
                }
            }
            break;
            }

            g_strfreev (address);
        }
        break;

        case 'i':
        case 't':
        case 'f':
        {
            bk = ret;
            ret = convert_strtoumax (optarg, &value);
            if (ret < 0) {
                cleanup = 1;
                break;
            }

            switch (bk) {
            case 'f':
            {
                if (AF_INET != value && AF_INET6 != value) {
                    cleanup = 1;
                    break;
                }
                rparser->route->family = (unsigned char) value;
                rparser->options |= IP_ROUTE_ATTR_FAMILY;
            }
            break;

            case 't':
            {
                if (value > 100) {
                    cleanup = 1;
                    break;
                }

                rparser->route->table = value;
                rparser->options |= IP_ROUTE_ATTR_TABLE;
            }
            break;

            case 'i':
            {
                if (!if_indextoname (value, oif)) {
                    cleanup = 1;
                    break;
                }
                rparser->route->oif = value;
                rparser->route->oifname = g_strdup (oif);
                rparser->options |= IP_ROUTE_ATTR_INDEX;
            }
            break;
            }
        }
        break;

        case 'g':
        {
            length = 0;
            address = g_strsplit_set (optarg, "/", -1);

            ret = calculate_argc (address, &length);
            if (ret != 0 || length > 1) {
                g_strfreev (address);
                cleanup = 1;
                break;
            }

            ret = get_family_from_addr (address [0]);
            if (ret < 0 || (ret != AF_INET /*&&
                            ret != AF_INET6*/)) {
                g_strfreev (address);
                cleanup = 1;
                break;
            }

            rparser->route->gateway = g_strdup (address [0]);
            rparser->options |= IP_ROUTE_ATTR_GATEWAY;

            g_strfreev (address);
        }
        break;

        default:
            cleanup = 1;
            break;
        }

        if (cleanup)
            break;
    }

    return (cleanup == 0) ? 0 : -EINVAL;
}

static int
route_build_nlmsg (const routeparser_t *rparse,
                   int cmd,
                   struct nl_msg **nlmsg)
{
    struct nl_msg *msg;
    struct rtmsg rhr;
    int flags = 0;

    return_val_if_fail (rparse && nlmsg && !*nlmsg, -EINVAL);

    if (!(rparse->options & IP_ROUTE_ATTR_TABLE))
        return -EINVAL;

    memset (&rhr, 0, sizeof (struct rtmsg));

    flags = NLM_F_REQUEST | NLM_F_ACK;
    if (RTM_NEWROUTE == cmd)
        flags |= NLM_F_CREATE | NLM_F_EXCL;

    msg = nlmsg_alloc_simple (cmd, flags);
    return_val_if_fail (msg, -ENOMEM);

    rhr.rtm_scope = RT_SCOPE_NOWHERE;
    if (RTM_DELROUTE != cmd) {
        rhr.rtm_protocol = RTPROT_BOOT;
        rhr.rtm_scope = RT_SCOPE_UNIVERSE;
        rhr.rtm_type = RTN_UNICAST;
    }
    rhr.rtm_family = rparse->route->family;

    if (rparse->options & IP_ROUTE_ATTR_SLEN)
        rhr.rtm_src_len = rparse->route->slen;
    if (rparse->options & IP_ROUTE_ATTR_DLEN)
        rhr.rtm_dst_len = rparse->route->dlen;

    if (nlmsg_append (msg, &rhr, sizeof (struct rtmsg), NLMSG_ALIGNTO) < 0) {
        nlmsg_free (msg);
        return -ENOMSG;
    }

    rhr.rtm_table = rparse->route->table
            != RT_TABLE_MAIN ?
                RT_TABLE_UNSPEC : RT_TABLE_MAIN;

    nla_put_u32 (msg, RTA_TABLE, rparse->route->table);

    if (rparse->options & IP_ROUTE_ATTR_INDEX)
        nla_put_u32 (msg, RTA_OIF, rparse->route->oif);

    if (rparse->options & IP_ROUTE_ATTR_GATEWAY)
        nl_add_address (msg, rparse->route->family,
                        RTA_GATEWAY, rparse->route->gateway);

    if (rparse->options & IP_ROUTE_ATTR_SRC)
        nl_add_address (msg, rparse->route->family,
                        RTA_SRC, rparse->route->src);

    if (rparse->options & IP_ROUTE_ATTR_DST)
        nl_add_address (msg, rparse->route->family,
                        RTA_DST, rparse->route->dst);

    *nlmsg = msg;
    return 0;
}

static int
find_matching_route (GList *routes,
                     route_t *match,
                     route_t **fetch)
{
    GList *list;
    route_t *route;

    return_val_if_fail (routes && match && fetch, -EINVAL);
    return_val_if_fail (!*fetch, -EINVAL);

    for (list = routes; list; list = list->next) {
        route = list->data;

        if (route->table != match->table)
            continue;
        if (route->oif != match->oif)
            continue;
        if (g_strcmp0 (route->gateway, match->gateway) != 0)
            continue;

        INFO ("Rule [%p] matches [%p]", route, match);
        *fetch = route;
        return 0;
    }

    return -ENOENT;
}

static int
route_is_pending (const routeparser_t *rparse)
{
    int avail = 0;
    GSList *temp = NULL;
    route_t *m = NULL, *e = NULL;

    return_val_if_fail (rparse && rparse->route, -EINVAL);

    e = rparse->route;
    for (temp = pending_routes; temp; temp = temp->next) {

        m = temp->data;
        continue_if_fail (m);

        if (m->table != e->table)
            continue;
        if (m->oif != e->oif)
            continue;
        if (g_strcmp0 (m->gateway, e->gateway) != 0)
            continue;

        avail = 1;
        break;
    }

    return avail == 1 ? 0 : -ENOENT;
}

static int
remove_pending_route (const routeparser_t *rparse)
{
    int avail = 0;
    GSList *temp = NULL;
    route_t *e = NULL;
    routeparser_t *m = NULL;

    return_val_if_fail (rparse && rparse->route, -EINVAL);

    e = rparse->route;
    for (temp = pending_routes; temp; temp = temp->next) {

        m = temp->data;
        continue_if_fail (m && m->route);

        if (m->route->table != e->table)
            continue;
        if (m->route->oif != e->oif)
            continue;
        if (g_strcmp0 (m->route->gateway, e->gateway) != 0)
            continue;

        avail = 1;
        break;
    }

    if (avail) {
        route_parser_unref (m);
        pending_routes = g_slist_remove_link (pending_routes, temp);
        return 0;
    }

    return -ENOENT;
}

static int
custom_error_handler (struct sockaddr_nl *nla,
                      struct nlmsgerr *err,
                      void *arg)
{
    (void) nla;
    (void) arg;
    int *ret = arg;

    return_val_if_fail (err, NL_STOP);

    DEBUG ("Error received for Nl message with sequence no [%u] : %s/%d system error : %s/%d",
           err->msg.nlmsg_seq, nl_geterror (err->error), err->error, strerror (-err->error),
           -err->error);

    if (ret)
        *ret = err->error;

    return NL_STOP;
}

static int
route_add (const routeparser_t *rparse)
{
    struct nl_msg *msg = NULL;
    route_t *pentry = NULL;
    GList *routes = NULL;
    int cmd = RTM_NEWROUTE, ret, err = 0;
    struct nl_cb *cb = NULL;

    return_val_if_fail (routesk, -ENOTCONN);

    ret = route_is_pending (rparse);
    if (ret == 0) {
        DEBUG ("A route request is already pending with the same details");
        return -EEXIST;
    }

    routes = g_hash_table_lookup (active_routes, GUINT_TO_POINTER (rparse->route->table));
    if (routes) {
        ret = find_matching_route (routes, rparse->route, &pentry);
        if (ret == 0)
            return -EEXIST;
    }

    ret = route_build_nlmsg (rparse, cmd, &msg);
    if (ret < 0) {
        ERROR ("Failed to build the netlink message: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = nl_send_auto_complete (routesk->sk, msg);
    nlmsg_free (msg);
    if (ret < 0) {
        ERROR ("Failed to send the nl msg: %s/%d",
               nl_geterror (-ret), -ret);
        return ret;
    }

    cb = nl_cb_clone (routesk->cb);
    if (cb)
        nl_cb_err (cb, NL_CB_CUSTOM, (nl_recvmsg_err_cb_t) custom_error_handler, &err);

    ret = nl_recvmsgs (routesk->sk, cb ? cb : routesk->cb);
    if (ret < 0) {
        ERROR ("Failed to add the route: %p [%s/%d] [err: %s/%d]",
               msg, strerror (-ret), -ret, strerror (-err), -err);
        ret = err;
    }

    nl_cb_put (cb);
    return ret;
}

static int
route_del (const routeparser_t *rparse)
{
    struct nl_msg *msg = NULL;
    route_t *pentry = NULL;
    GList *routes = NULL;
    int cmd = RTM_DELROUTE, ret;

    return_val_if_fail (routesk, -ENOTCONN);

    ret = remove_pending_route (rparse);
    if (ret == 0) {
        DEBUG ("A route request is already pending but not committed, thus deleted");
        return 0;
    }

    /* This route plugin only has a cache about the gateways
     * but not all the routes and therefore check only if the
     * gateway is available */
    if (rparse->route->gateway) {
        routes = g_hash_table_lookup (active_routes, GUINT_TO_POINTER (rparse->route->table));
        return_val_if_fail (routes, -ENOENT);
        ret = find_matching_route (routes, rparse->route, &pentry);
        return_val_if_fail (ret == 0, -ENOENT);
    }

    ret = route_build_nlmsg (rparse, cmd, &msg);
    if (ret < 0) {
        ERROR ("Failed to build the netlink message: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = nl_send_auto_complete (routesk->sk, msg);
    nlmsg_free (msg);
    if (ret < 0) {
        ERROR ("Failed to send the nl msg: %s/%d",
               nl_geterror (-ret), -ret);
        return ret;
    }

    ret = nl_recvmsgs (routesk->sk, routesk->cb);
    if (ret < 0)
        ERROR ("Failed to del the route: %p", msg);

    return ret;
}

static int
commit_route (routeparser_t *rparse)
{
    int ret = -EINVAL;

    return_val_if_fail (rparse && rparse->route, -EINVAL);

    DEBUG ("Route Action: %s", route_opcode2str (rparse->opcode));

    if (rparse->opcode == IP_ROUTE_OPCODE_ADD) {
        ret = route_add (rparse);
        if (ret == -ENETDOWN)
            pending_routes = g_slist_append (pending_routes, rparse);
    }
    else if (rparse->opcode == IP_ROUTE_OPCODE_DEL)
        ret = route_del (rparse);

    return ret;
}

static int
modify_route (const char *rule)
{
    int ret;
    routeparser_t *parser = NULL;

    DEBUG ("IP Route Rule: %s", rule);

    ret = prepare_route_arguments (rule, &parser);
    if (ret < 0) {
        ERROR ("Failed to prepare the rule arguments: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = parse_route (rule, parser);
    if (ret < 0) {
        ERROR ("Failed to parse the rule [%s]: %s/%d", rule,
               strerror (-ret), -ret);
        route_parser_unref (parser);
        return ret;
    }

    ret = commit_route (parser);
    if (ret < 0)
        ERROR ("Failed to commit the rule [%s]: %s/%d", rule,
               strerror (-ret), -ret);

    if (ret == -ENETDOWN &&
            parser->opcode == IP_ROUTE_OPCODE_ADD) {
        INFO ("Route Parser %p has been added to the pending list", parser);
    } else
        route_parser_unref (parser);
    return ret;
}

int
modify_ip_route (const char *route)
{
    return_val_if_fail (route, -EINVAL);
    return modify_route (route);
}

static int
route_handle_dellink (int index,
                      unsigned short type,
                      unsigned int family,
                      unsigned int flags,
                      unsigned int change,
                      const char *name)
{
    int ret;
    GList *routes = NULL;
    route_t *route = NULL;
    unsigned int oldlength = 0;

    DEBUG ("Index: %d type: %u family: %u flags: %u change: %u name: %s",
           index, type, family, flags, change, name);

    return_val_if_fail (index > 0 && name, -EINVAL);

    while ((ret = find_route_from_oifname (name, &route)) == 0) {
        routes = g_hash_table_lookup (active_routes, GUINT_TO_POINTER (route->table));
        if (routes) {
            oldlength = g_list_length (routes);
            if (g_list_length (routes) == 1) {
                g_hash_table_remove (active_routes, GUINT_TO_POINTER (route->table));
                g_list_free_full (routes, delete_routes);
                routes = NULL;
            } else {
                routes = g_list_remove (routes, route);
                g_hash_table_replace (active_routes, GUINT_TO_POINTER (route->table), routes);
                route_unref (route);
            }
            DEBUG ("Route %p has been deleted from the internal list of "
                   "table: \"%u\", old: %u new: %u", route, route->table,
                   oldlength, g_list_length (routes));
        } else {
            DEBUG ("The route list of table %u does not have the route", route->table);
            break;
        }
        route = NULL;
    }

    return ret;
}

static int
commit_pending_rules (unsigned int index)
{
    int avail = 0, ret;
    GSList *temp = pending_routes;
    routeparser_t *r = NULL;

    for ( ; temp; temp = temp->next) {

        r = temp->data;
        continue_if_fail (r && r->route);

        if (r->route->oif != index)
            continue;

        avail = 1;
        ret = commit_route (r);
        if (ret < 0)
            ERROR ("Failed to commit the route [%p]: %s/%d", r,
                   strerror (-ret), -ret);
        break;
    }

    if (avail) {
        route_parser_unref (r);
        pending_routes = g_slist_remove_link (pending_routes, temp);
        return 0;
    }

    return -ENOENT;
}

static void
route_handle_dev_up (const char *name)
{
    int ret;
    unsigned int index = 0;

    return_if_fail (name);

    DEBUG ("Add pending routes if available for the iface: %s", name);

    index = if_nametoindex (name);
    if (0 != index) {
        while ((ret = commit_pending_rules (index)) == 0)
            ;
    } else {
        ERROR ("Failed to get the index for interface: %s [%s/%d]",
               name, strerror (errno), errno);
    }
}

static rtnlwatch route_ops = {
    .client = "route",
    .newgw = route_handle_newgw,
    .delgw = route_handle_delgw,
    .dellink = route_handle_dellink,
};

static ipconfig_ops devstats = {
    .name = "route",
    .dev_up = route_handle_dev_up,
};

static int
route_error_handler (struct sockaddr_nl *nla,
                     struct nlmsgerr *err,
                     void *arg)
{
    (void) nla;
    (void) arg;

    return_val_if_fail (err, NL_STOP);

    DEBUG ("Error received for Nl message with "
           "sequence no [%u] : %s/%d "
           "system error : %s/%d",
           err->msg.nlmsg_seq,
           nl_geterror (err->error),
           err->error, strerror (-err->error), -err->error);

    return NL_STOP;
}

static int
route_finish_handler (struct nl_msg *msg,
                      void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_SKIP);
    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%u] finished successfully",
           nlhdr->nlmsg_seq);
    return NL_SKIP;
}

static int
route_ack_handler (struct nl_msg *msg,
                   void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_STOP);

    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%u] acknowledged", nlhdr->nlmsg_seq);
    return NL_STOP;
}

static int
route_init_transport (route_socket **sk)
{
    int ret;
    route_socket *rsk;

    return_val_if_fail (sk, -EINVAL);
    return_val_if_fail (!*sk, -EEXIST);

    rsk = g_try_malloc0 (sizeof (*rsk));
    return_val_if_fail (rsk, -ENOMEM);

    rsk->cb = nl_cb_alloc (NL_CB_DEFAULT);
    if (!rsk->cb) {
        g_free (rsk);
        return -ENOMEM;
    }

    ret = nl_socket_init (&rsk->sk, NETLINK_ROUTE, TRUE, 0, 0);
    if (ret < 0) {
        nl_cb_put (rsk->cb);
        g_free (rsk);
        return ret;
    }

    nl_cb_set (rsk->cb, NL_CB_ACK, NL_CB_CUSTOM, route_ack_handler, NULL);
    nl_cb_set (rsk->cb, NL_CB_FINISH, NL_CB_CUSTOM, route_finish_handler, NULL);
    nl_cb_err (rsk->cb, NL_CB_CUSTOM, (nl_recvmsg_err_cb_t) route_error_handler, NULL);

    *sk = rsk;
    return ret;
}

static int
route_deinit_transport (route_socket **sk)
{
    route_socket *socket;

    return_val_if_fail (sk, -EINVAL);
    return_val_if_fail (*sk, -EALREADY);

    socket = *sk;
    nl_cb_put (socket->cb);
    nl_socket_free (socket->sk);

    g_free (socket);
    *sk = NULL;

    return 0;
}

static void
destroy_routes (void *table)
{
   gpointer key, value;
   GHashTableIter iter;
   GHashTable *routes = table;

   return_if_fail (routes);

   g_hash_table_iter_init (&iter, routes);
   while (g_hash_table_iter_next (&iter, &key, &value)) {
      DEBUG ("Clearing the table %u, size: %u", GPOINTER_TO_UINT (key),
            g_list_length (value));
      routes_cleanup (value);
   }
}

int
route_init ()
{
    int ret;

    DEBUG ("");

    ret = route_init_transport (&routesk);
    if (ret < 0) {
        ERROR ("Failed to initialize the fib socket: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    active_routes = g_hash_table_new_full (g_direct_hash,
                                           g_direct_equal,
                                           NULL, NULL);
    pending_routes = NULL;
    ret = rtnl_notifier_register (&route_ops);
    if (ret < 0)
        ERROR ("Failed to register to rtnl watch list: %s/%d",
               strerror (-ret), -ret);

    ret = ipconfig_notifier_register (&devstats);
    if (ret < 0)
        ERROR ("Failed to register to ipconfig watch list: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

int
route_deinit ()
{
    int ret;

    DEBUG ("");

    ret = route_deinit_transport (&routesk);
    if (ret < 0)
        ERROR ("Failed to deinitialize the route socket: %s/%d",
               strerror (-ret), -ret);

    ret = rtnl_notifier_unregister (&route_ops);
    if (ret < 0)
        ERROR ("Failed to unregister from rtnl watch list: %s/%d",
               strerror (-ret), -ret);

    ret = ipconfig_notifier_unregister (&devstats);
    if (ret < 0)
        ERROR ("Failed to unregister from ipconfig watch list: %s/%d",
               strerror (-ret), -ret);

    destroy_routes (active_routes);
    g_hash_table_destroy (active_routes);
    g_slist_free_full (pending_routes, delete_parser);
    return ret;
}

/** @} */
