/**
 * @file ipconfig.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <unistd.h>
#include <stdint.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <glib.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <linux/if.h>
#ifndef IFNAMSIZ
#undef _NET_IF_H
#include <net/if.h>
#endif
#include <linux/rtnetlink.h>
#include <netlink/socket.h>
#include <netlink/netlink.h>
#include <netlink/msg.h>
#include <ipconfig.h>
#include <rtnl.h>
#include <log.h>

struct ip_config
{
    int index;
    unsigned int flags;
    char *ifname;
    ipconfigtype type;
    ipaddress *address;
};

typedef
enum {
    IPCONFIG_EVENT_UNKNOWN,
    IPCONFIG_EVENT_DEV_UP,
    IPCONFIG_EVENT_DEV_DOWN,
    IPCONFIG_EVENT_DEV_LOWER_UP,
    IPCONFIG_EVENT_DEV_LOWER_DOWN,
    IPCONFIG_EVENT_DEV_IP_BOUND,
    IPCONFIG_EVENT_DEV_IP_RELEASED,
    IPCONFIG_EVENT_DEV_ROUTE_SET,
    IPCONFIG_EVENT_DEV_ROUTE_UNSET
} ipevents;

typedef
struct __nl_socket
{
    struct nl_sock *sk;
    struct nl_cb *cb;
} nlsocket;

static GHashTable *ipdevices;
static nlsocket *nl_sk;
static GSList *registeredclients;

const char*
ipconfig_type2str (const ipconfigtype type)
{
    switch (type) {
    case IPCONFIG_TYPE_IPV4:
        return ENUMTOSTR (TYPE_IPV4);
    case IPCONFIG_TYPE_IPV6:
        return ENUMTOSTR (TYPE_IPV6);
    case IPCONFIG_TYPE_UNKNOWN:
    case IPCONFIG_MAX_TYPES:
        return ENUMTOSTR (TYPE_UNKNOWN);
    }

    return ENUMTOSTR (TYPE_UNKNOWN);
}

static const char*
ipconfig_event2str (ipevents event)
{
    switch (event) {
    case IPCONFIG_EVENT_UNKNOWN:
        return ENUMTOSTR (EVENT_UNKNOWN);
    case IPCONFIG_EVENT_DEV_UP:
        return ENUMTOSTR (DEV_UP);
    case IPCONFIG_EVENT_DEV_DOWN:
        return ENUMTOSTR (DEV_DOWN);
    case IPCONFIG_EVENT_DEV_LOWER_UP:
        return ENUMTOSTR (DEV_LOWER_UP);
    case IPCONFIG_EVENT_DEV_LOWER_DOWN:
        return ENUMTOSTR (DEV_LOWER_DOWN);
    case IPCONFIG_EVENT_DEV_IP_BOUND:
        return ENUMTOSTR (DEV_IP_BOUND);
    case IPCONFIG_EVENT_DEV_IP_RELEASED:
        return ENUMTOSTR (DEV_IP_RELEASED);
    case IPCONFIG_EVENT_DEV_ROUTE_SET:
        return ENUMTOSTR (DEV_ROUTE_SET);
    case IPCONFIG_EVENT_DEV_ROUTE_UNSET:
        return ENUMTOSTR (DEV_ROUTE_UNSET);
    }

    return ENUMTOSTR (EVENT_UNKNOWN);
}

static void
ipconfig_address_cleanup (ipaddress **data)
{
    ipaddress *address;

    return_if_fail (data && *data);

    address = *data;
    g_free (address->broadcast);
    g_free (address->gateway);
    g_free (address->local);
    g_free (address->peer);
    g_free (address);

    *data = NULL;
}

static void
ipconfig_cleanup (gpointer data)
{
    struct ip_config *ipconfig = data;

    return_if_fail (ipconfig);

    DEBUG ("Cleaning up the ip config for device : %s",
           ipconfig->ifname);

    ipconfig_address_cleanup (&ipconfig->address);
    g_free (ipconfig->ifname);
    g_free (ipconfig);
}

static int
ipconfig_error_handler (struct sockaddr_nl *nla,
                        struct nlmsgerr *err,
                        void *arg)
{
    (void) nla;
    (void) arg;

    return_val_if_fail (err, NL_STOP);
    DEBUG ("Error received for Nl message with sequence no [%d] : %s/%d "
           "system error : %s/%d", err->msg.nlmsg_seq, nl_geterror (err->error),
           err->error, strerror (-err->error), -err->error);
    return NL_STOP;
}

static int
ipconfig_finish_handler (struct nl_msg *msg,
                         void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_SKIP);
    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%d] finished successfully", nlhdr->nlmsg_seq);
    return NL_SKIP;
}

static int
ipconfig_ack_handler (struct nl_msg *msg,
                      void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_STOP);

    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%d] acknowledged", nlhdr->nlmsg_seq);
    return NL_STOP;
}

static ipconfigtype
ipconfig_get_type_from_addr (const char *address)
{
    int ret;
    struct addrinfo hints;
    struct addrinfo *addr = NULL;
    ipconfigtype type = IPCONFIG_MAX_TYPES;

    return_val_if_fail (address, type);

    memset (&hints, 0, sizeof (struct addrinfo));

    type = IPCONFIG_MAX_TYPES;
    hints.ai_family = AF_UNSPEC;
    hints.ai_flags = AI_NUMERICHOST;

    ret = getaddrinfo (address, NULL, &hints, &addr);
    if (!ret) {
        if (addr->ai_family == AF_INET)
            type = IPCONFIG_TYPE_IPV4;
        else if (addr->ai_family == AF_INET6)
            type = IPCONFIG_TYPE_IPV6;
    }

    freeaddrinfo (addr);
    return type;
}

static int
ipconfig_update_clients (struct ip_config *ipdevice,
                         ipevents event)
{
    GSList *temp;
    ipconfig_ops *ops;

    return_val_if_fail (ipdevice, -EINVAL);

    DEBUG ("%s event : %s", ipdevice->ifname,
           ipconfig_event2str (event));

    if (event == IPCONFIG_EVENT_UNKNOWN)
        return 0;

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {

        ops = temp->data;
        continue_if_fail (ops);

        switch (event) {
        case IPCONFIG_EVENT_DEV_UP:
            if (ops->dev_up)
                ops->dev_up (ipdevice->ifname);
            break;

        case IPCONFIG_EVENT_DEV_DOWN:
            if (ops->dev_down)
                ops->dev_down (ipdevice->ifname);
            break;

        case IPCONFIG_EVENT_DEV_LOWER_UP:
            if (ops->dev_lower_up)
                ops->dev_lower_up (ipdevice->ifname);
            break;

        case IPCONFIG_EVENT_DEV_LOWER_DOWN:
            if (ops->dev_lower_down)
                ops->dev_lower_down (ipdevice->ifname);
            break;

        case IPCONFIG_EVENT_DEV_IP_BOUND:
            if (ops->addr_added)
                ops->addr_added (ipdevice->index, ipdevice->ifname,
                                 ipdevice->type, ipdevice->address ?
                                     ipdevice->address->local : NULL);
            break;
        case IPCONFIG_EVENT_DEV_IP_RELEASED:
            if (ops->addr_removed)
                ops->addr_removed (ipdevice->index, ipdevice->ifname,
                                   ipdevice->type, ipdevice->address ?
                                       ipdevice->address->local : NULL);

        default:
            break;
        }
    }

    return 0;
}

static int
ipconfig_new_addr (unsigned int index,
                   int family,
                   const char *label,
                   const char *local,
                   const char *address,
                   const char *anycast,
                   const char *multicast)
{
    (void) label;
    (void) anycast;
    (void) multicast;
    ipaddress *addr;
    struct ip_config *ipconfig;

    DEBUG ("index: %d address: %s", index, address);

    if (family == AF_INET6) {
        INFO ("AF_INET6 is not handled as of now!!");
        return 0;
    }

    ipconfig = g_hash_table_lookup (ipdevices, GINT_TO_POINTER (index));
    return_val_if_fail (ipconfig, -ENOTSUP);
    addr = g_try_malloc0 (sizeof (*addr));
    return_val_if_fail (addr, -ENOMEM);

    ipconfig_address_cleanup (&ipconfig->address);

    addr->family = family;
    addr->local = g_strdup (local);

    ipconfig->address = addr;
    ipconfig->type = ipconfig_get_type_from_addr (address);

    ipconfig_update_clients (ipconfig, IPCONFIG_EVENT_DEV_IP_BOUND);
    return 0;
}

static int
ipconfig_del_addr (unsigned int index,
                   int family,
                   const char *label,
                   const char *local,
                   const char *address,
                   const char *anycast,
                   const char *multicast)
{
    (void) family;
    (void) label;
    (void) local;
    (void) anycast;
    (void) multicast;

    struct ip_config *ipconfig;

    DEBUG ("index : %d address %s", index, address);

    if (family == AF_INET6) {
    	INFO ("AF_INET6 is not handled as of now!!");
    	return 0;
    }

    ipconfig = g_hash_table_lookup (ipdevices, GINT_TO_POINTER (index));
    return_val_if_fail (ipconfig, -ENOTSUP);
    ipconfig_update_clients (ipconfig, IPCONFIG_EVENT_DEV_IP_RELEASED);
    ipconfig_address_cleanup (&ipconfig->address);
    return 0;
}

static rtnlwatch ipaddrwatch = {
  .client  = "ipconfig",
  .newaddr = ipconfig_new_addr,
  .deladdr = ipconfig_del_addr,
};

int
ipconfig_newlink (int index,
                  unsigned int flags,
                  const char *name)
{
    struct ip_config *ipconfig;

    return_val_if_fail (name, -EINVAL);

    DEBUG ("index %d flags %d name %s", index, flags,
           name);

    ipconfig = g_hash_table_lookup (ipdevices, GINT_TO_POINTER (index));
    if (!ipconfig) {
        ipconfig = g_try_malloc0 (sizeof (*ipconfig));
        return_val_if_fail (ipconfig, -ENOMEM);
        ipconfig->address = g_try_malloc0 (sizeof (ipaddress));
        if (!ipconfig->address) {
            g_free (ipconfig);
            return -ENOMEM;
        }

        g_hash_table_insert (ipdevices, GINT_TO_POINTER (index), ipconfig);
        ipconfig->index = index;
    }

    g_free (ipconfig->ifname);
    ipconfig->ifname = g_strdup (name);

    if (ipconfig->flags == flags)
        return 0;

    if ((ipconfig->flags & IFF_UP) != (flags & IFF_UP)) {
        if (flags & IFF_UP)
            ipconfig_update_clients (ipconfig, IPCONFIG_EVENT_DEV_UP);
        else
            ipconfig_update_clients (ipconfig, IPCONFIG_EVENT_DEV_DOWN);
    }

    if ((ipconfig->flags & (IFF_RUNNING | IFF_LOWER_UP)) !=
                (flags & (IFF_RUNNING | IFF_LOWER_UP))) {
        if ((flags & (IFF_RUNNING | IFF_LOWER_UP)) == (IFF_RUNNING | IFF_LOWER_UP))
            ipconfig_update_clients (ipconfig, IPCONFIG_EVENT_DEV_LOWER_UP);
        else if ((flags & (IFF_RUNNING | IFF_LOWER_UP)) == 0)
            ipconfig_update_clients (ipconfig, IPCONFIG_EVENT_DEV_LOWER_DOWN);
    }

    ipconfig->flags = flags;
    return 0;
}

int
ipconfig_dellink (int index)
{
    gboolean removed = FALSE;

    DEBUG ("index %d", index);
    removed = g_hash_table_remove (ipdevices, GINT_TO_POINTER (index));
    return (removed == TRUE) ? 0 : -ENOENT;
}

static int
ipconfig_add_attr (struct nl_msg *msg,
                   int family,
                   int attr,
                   const char *ipaddress)
{
    int ret;
    struct nl_addr *addr;
    unsigned char buf [sizeof (struct in6_addr)];
    char address [INET6_ADDRSTRLEN];

    return_val_if_fail (msg && ipaddress, -EINVAL);

    memset (buf, 0, sizeof (buf));
    memset (address, 0, sizeof (address));

    ret = inet_pton (family, ipaddress, buf);
    if (ret < 0)
        return -EINVAL;

    addr = nl_addr_build (family, (void *) buf, sizeof (buf));
    return_val_if_fail (addr, -ENOMEM);

    nl_addr2str (addr, address, INET6_ADDRSTRLEN);
    ret = nla_put (msg, attr, (int) nl_addr_get_len (addr),
                   nl_addr_get_binary_addr (addr));
    if (ret < 0)
        ERROR ("Failed to add the attibute [%d] to the message : %s",
               attr, nl_geterror (-ret));

    nl_addr_put (addr);
    return ret;
}

static int
ipconfig_modify_address (int cmd,
                         unsigned char flags,
                         unsigned int index,
                         unsigned char family,
                         const char *address,
                         const char *peer,
                         unsigned char prefixlen,
                         const char *broadcast)
{
    int ret = -ENOMEM;
    struct nl_msg *msg;
    struct ifaddrmsg ifam;

    DEBUG ("cmd %#x flags %#x index %d family %d address %s peer %s "
           "prefixlen %hhu broadcast %s", cmd, flags, index, family,
           address, peer, prefixlen, broadcast);

    if (!nl_sk || !nl_sk->sk)
        return -ENOLINK;

    return_val_if_fail (address && broadcast, -EINVAL);
    if (broadcast && family == AF_INET6)
        return -EOPNOTSUPP;

    memset (&ifam, 0, sizeof (struct ifaddrmsg));

    ifam.ifa_family = family;
    ifam.ifa_index = index;
    ifam.ifa_prefixlen = prefixlen;
    ifam.ifa_flags = flags;
    ifam.ifa_scope = RT_SCOPE_UNIVERSE;

    msg = nlmsg_alloc_simple (cmd, flags);
    return_val_if_fail (msg, -ENOMEM);
    if (nlmsg_append (msg, &ifam, sizeof(ifam), NLMSG_ALIGNTO) < 0) {
        ret = -EMSGSIZE;
        goto error;
    }

    ipconfig_add_attr (msg, family, IFA_ADDRESS, address);
    ipconfig_add_attr (msg, family, IFA_LOCAL, address);
    if (peer)
        ipconfig_add_attr (msg, family, IFA_ADDRESS, peer);
    ipconfig_add_attr (msg, family, IFA_BROADCAST, broadcast);

//    nla_put (msg, IFA_FLAGS , sizeof (unsigned int), &flags);

    ret = nl_send_auto_complete (nl_sk->sk, msg);
    nlmsg_free (msg);
    if (ret < 0)
        return ret;

    ret = nl_recvmsgs (nl_sk->sk, nl_sk->cb);
    if (ret < 0)
        ERROR ("Failed to set the ip address \"%s\" to "
               "the interface : %d", address, index);

    return ret;
error:
    nlmsg_free (msg);
    return ret;
}

int
ipconfig_add_address (unsigned int index,
                      unsigned char family,
                      const char *address,
                      const char *peer,
                      unsigned char prefixlen,
                      const char *broadcast)
{
    int ret;
    unsigned char flags = 0;
    struct ip_config *ipconf;

    DEBUG ("index %d family %d address %s peer %s prefixlen %d broadcast %s",
           index, family, address, peer, prefixlen, broadcast);

    return_val_if_fail (ipdevices, -EOPNOTSUPP);

    ipconf = g_hash_table_lookup (ipdevices, GINT_TO_POINTER (index));
    return_val_if_fail (ipconf, -ENOTSUP);

    if (ipconf->address) {
        DEBUG ("index %d old %s new %s", index, ipconf->address->local, address);
        if (!g_strcmp0 (ipconf->address->local, address))
            return -EEXIST;
    }

    flags = NLM_F_REQUEST /*| NLM_F_REPLACE*/ | NLM_F_ACK;

    ret = ipconfig_modify_address (RTM_NEWADDR, flags,
                                   index, family, address,
                                   peer, prefixlen, broadcast);
    if (ret < 0)
        ERROR ("Failed to add/replace the ip address %s for index [%d]",
               address, index);

    return ret;
}

int
ipconfig_delete_address (unsigned int index,
                         unsigned char family,
                         const char *address,
                         const char *peer,
                         unsigned char prefixlen,
                         const char *broadcast)
{
    int ret;
    struct ip_config *ipconf;

    return_val_if_fail (ipdevices, -EOPNOTSUPP);

    ipconf = g_hash_table_lookup (ipdevices, GINT_TO_POINTER (index));
    return_val_if_fail (ipconf, -ENOTSUP);

    DEBUG ("index %d address %s", index, address);

    if (ipconf->address) {
    	DEBUG ("Present address: %s", ipconf->address->local);
    	if (g_strcmp0 (ipconf->address->local, address))
    		return -ENOENT;
    }

    ret = ipconfig_modify_address (RTM_DELADDR, (unsigned char) 0,
                                   index, family, address,
                                   peer, prefixlen, broadcast);
    if (ret < 0)
        ERROR ("Failed to delete the ip address %s for index [%d]",
               address, index);

    return ret;
}

char*
ipconfig_get_address (unsigned int index)
{
    struct ip_config *config;

    config = g_hash_table_lookup (ipdevices, GUINT_TO_POINTER (index));
    return_val_if_fail (config, NULL);
    return config->address ? config->address->local : NULL;
}

static int
ipconfig_socket_init (struct nl_sock **sock)
{
    int ret = 0;
    int sno;
    struct nl_sock *sk;

    return_val_if_fail (sock && !*sock, -EINVAL);

    sk = nl_socket_alloc ();
    return_val_if_fail (sk, -ENOMEM);
    ret = nl_connect (sk, NETLINK_ROUTE);
    if (ret < 0) {
        ERROR ("Failed to connect to the netlink socket : %s",
               nl_geterror (-ret));
        goto error;
    }

    sno = nl_socket_get_fd (sk);
    ret = fcntl (sno, F_SETFL, O_NONBLOCK | O_CLOEXEC);
    if (ret < 0) {
        ret = -errno;
        goto error;
    }

    INFO ("Socket initialized successfully for : %d [%p]",
          sno, sk);

    *sock = sk;
    return ret;

error:
    nl_socket_free (sk);
    *sock = NULL;
    return ret;
}

static int
ipconfig_nl_init (nlsocket **nlsk)
{
    int ret;
    nlsocket *sk;

    return_val_if_fail (nlsk, -EINVAL);
    return_val_if_fail (!*nlsk, -EEXIST);

    sk = g_try_malloc0 (sizeof (*sk));
    return_val_if_fail (sk, -ENOMEM);
    sk->cb = nl_cb_alloc (NL_CB_DEFAULT);
    if (!sk->cb) {
        ERROR ("Failed to allocate the callback for nl socket");
        ret = -ENOMEM;
        goto error;
    }

    ret = ipconfig_socket_init (&sk->sk);
    if (ret < 0)
        goto cb_cleanup;

    nl_cb_err (sk->cb, NL_CB_CUSTOM,
               (nl_recvmsg_err_cb_t) ipconfig_error_handler,
               NULL);
    nl_cb_set (sk->cb, NL_CB_FINISH, NL_CB_CUSTOM,
               ipconfig_finish_handler, NULL);
    nl_cb_set (sk->cb, NL_CB_ACK, NL_CB_CUSTOM,
               ipconfig_ack_handler, NULL);

    INFO ("Ipconfig socket created successfully for : %d",
          nl_socket_get_fd (sk->sk));
    *nlsk = sk;
    return ret;

cb_cleanup:
    nl_cb_put (sk->cb);
error:
    g_free (sk);
    *nlsk = NULL;
    return ret;
}

static int
ipconfig_nl_deinit (nlsocket **nlsk)
{
    return_val_if_fail (nlsk, -EINVAL);
    return_val_if_fail (*nlsk, -EALREADY);

    nl_cb_put ((*nlsk)->cb);
    nl_socket_free ((*nlsk)->sk);
    g_free (*nlsk);
    *nlsk = NULL;

    return 0;
}

int
ipconfig_notifier_register (ipconfig_ops *ops)
{
    int found = 0;
    GSList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("Registering the client for ipconfig updates : %s [%p]",
           ops->name ? ops->name : "UNKNOWN", ops);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        ipconfig_ops *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (!found) {
        registeredclients = g_slist_append (registeredclients, ops);
        return 0;
    }

    return -EALREADY;
}

int
ipconfig_notifier_unregister (ipconfig_ops *ops)
{
    int found = 0;
    GSList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("UnRegistering the client for ipconfig update : %s [%p]",
           ops->name ? ops->name : "UNKNOWN", ops);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        ipconfig_ops *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (found) {
        registeredclients = g_slist_remove_link (registeredclients,
                                    temp);
        return 0;
    }

    return -ENOENT;
}

int
ipconfig_init ()
{
    int ret;

    DEBUG ("");

    ret = ipconfig_nl_init (&nl_sk);
    if (ret < 0) {
        ERROR ("Failed to intialize the nlsocket : %s",
               strerror (-ret));
        return ret;
    }

    ret = rtnl_notifier_register (&ipaddrwatch);
    if (ret < 0) {
        ERROR ("Failed to register to rtnl addr events : %s",
               strerror (-ret));
        ipconfig_nl_deinit (&nl_sk);
        return ret;
    }

    ipdevices =
            g_hash_table_new_full (g_direct_hash,
                                   g_direct_equal,
                                   NULL,
                                   ipconfig_cleanup);

    return ret;
}

int
ipconfig_deinit ()
{
    int ret;

    DEBUG ("");

    ret = ipconfig_nl_deinit (&nl_sk);
    if (ret < 0)
        ERROR ("Failed to deinitalize the nl socket : %s",
               strerror (-ret));

    ret = rtnl_notifier_unregister (&ipaddrwatch);
    if (ret < 0)
        ERROR ("Failed to unregister from rtnl addr events : %s",
               strerror (-ret));

    g_hash_table_destroy (ipdevices);
    return ret;
}

/** @} */
