/**
 * @file log.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <log.h>
#include <main.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <syslog.h>
#include <libgen.h>
#include <time.h>
#include <sys/time.h>
#include <errno.h>
#include <glib.h>
#include <log_file.h>
#include <log_dlt.h>
#include <log_stdout.h>
#include <log_syslog.h>

static GList *log_utility_list = NULL;
static char *programname = NULL;

static int
log_message (int prio)
{
    int log_level = get_log_level ();

    if (log_level < LOG_LEVEL_0 || log_level > LOG_LEVEL_7)
        return 1;
    else if (log_level == prio)
        return 1;
    return 0;
}

int
timestamp (char *tme)
{
    struct tm tm;
    time_t current_time;
    return_val_if_fail (tme, -1);
    current_time = time(NULL);
    tm = *localtime (&current_time);
    strftime (tme, TIMESTAMP, "%d/%m/%Y %H:%M:%S", &tm);
    return 0;
}

char*
log_get_program_name(void)
{
    return programname;
}

void
debug (int prio,
       const char *format,
       ...)
{
    va_list args;
    va_start (args, format);
    char str[BUFSIZ];
    memset (str, 0, sizeof (str));

    if (!log_message (prio))
        goto end;

    GList *temp;
    logops *ops;

    temp = log_utility_list;
    for (; temp; temp = temp->next) {
        ops = (logops*)temp->data;
        if(ops->util_log) {
            if (vsnprintf (str, sizeof(str), format, args) > 0)
                ops->util_log(prio, str, args);
        }
    }
end:
    va_end(args);
}

void
vdebug (int prio,
        const char    *msg,
        va_list  args)
{
    if (!msg)
        return;
    if (!log_message (prio))
        return;

    GList *temp;
    logops *ops;

    temp = log_utility_list;
    for (; temp; temp = temp->next) {
        ops = (logops*)temp->data;
        if(ops->util_log)
            ops->util_log(prio, msg, args);
    }
}

void
hexdump (int prio,
         const char *data,
         size_t size)
{
    size_t index = 0;
    char str[BUFSIZ], tempstr [4];
    memset (str, 0, sizeof (str));

    if (!log_message (prio))
        return;

    for ( ; index < size; index++) {
        memset (tempstr, 0, sizeof (tempstr));
        sprintf (tempstr, " %02x", data [index]);
        strcat (str, tempstr);
    }
    strcat (str, "Hexdump :");

    GList *temp;
    static va_list args;
    logops *ops;


    temp = log_utility_list;
    for (; temp; temp = temp->next) {
        ops = (logops*)temp->data;
        if(ops->util_log)
            ops->util_log(prio, str, args);
    }

}

int
log_init (char *program_name,
          const int log_level,
          const char *debugfile)
{
    if(!program_name)
        programname = program_name;

    (void)log_level;
	(void)debugfile;
    log_file_init();
    log_syslog_init();
    log_dlt_init();
    log_stdout_init();

    return 0;
}

int
log_deinit ()
{
    if(!programname)
        g_free(programname);

    log_syslog_deinit();
    log_dlt_deinit();
    log_stdout_deinit();
    log_file_deinit();

    return 0;
}

int
log_util_register (logops *util)
{
    int found = 0;
    GList *temp;
    logops *ops;

    return_val_if_fail (util, -EINVAL);

    temp = log_utility_list;
    for ( ; temp; temp = temp->next) {
        ops = temp->data;
        if (ops == util) {
            found = 1;
            break;
        }
    }

    if(!found){
        log_utility_list = g_list_append (log_utility_list, util);
        if(util->util_open)
            util->util_open();
        return 0;
    }

    return -EEXIST;
}

int
log_util_unregister (logops *util)
{
    int found = 0;
    GList *temp;
    logops *ops;

    return_val_if_fail (util, -EINVAL);

    for (temp = log_utility_list; temp; temp = temp->next) {
        ops = (logops*)temp->data;
        if ((ops == util)) {
            found = 1;
            break;
        }
    }

    if(found) {
        log_utility_list = g_list_remove (log_utility_list, util);
        if(util->util_close)
            ops->util_close();
    }

    util = NULL;

    return -ENOENT;
}

/** @} */
