/**
 * @file server.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include <errno.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <glib.h>
#include <server.h>
#include <ippool.h>
#include <ipconfig.h>
#include <log.h>
#include "inc/utils.h"

#define DHCP_LEASE_PERIOD       (7 * 24 * 60 * 60) /* better, 7 days */
#define DHCP_POOL_START         2
#define DHCP_POOL_END           252

struct dhcp_server
{
    unsigned int index;
    char *ifname;
    gboolean dnsservice;
    dhcpserverstate state;
    struct ippool *pool;
    unsigned int pool_size;
    unsigned int lease_period;
};

static GList *clients;
static dhcpserver_ops *driver;
static GHashTable *dhcpservers;

static void
dhcp_server_cleanup (gpointer data)
{
    struct dhcp_server *server = data;

    return_if_fail (server);

    DEBUG ("Cleaning up: %s [%p]",
           server->ifname, server);

    ippool_destroy (server->ifname);
    g_free (server->ifname);
    g_free (server);
}

static const char*
dhcp_server_state2string (dhcpserverstate state)
{
    switch (state) {
    case DHCP_SERVER_STATE_UNKNOWN:
        return ENUMTOSTR (STATE_UNKNOWN);
    case DHCP_SERVER_STATE_INITIALIZED:
        return ENUMTOSTR (STATE_INITIALIZED);
    case DHCP_SERVER_STATE_STARTED:
        return ENUMTOSTR (STATE_STARTED);
    case DHCP_SERVER_STATE_STOPPING:
        return ENUMTOSTR (STATE_STOPPING);
    case DHCP_SERVER_STATE_STOPPED:
        return ENUMTOSTR (STATE_STOPPED);
    case DHCP_SERVER_STATE_START_FAILED:
        return ENUMTOSTR (START_FAILED);
    case DHCP_SERVER_STATE_STOP_FAILED:
        return ENUMTOSTR (STOP_FAILED);
    }

    return ENUMTOSTR (STATE_UNKNOWN);
}

static int
dhcp_server_set_state (struct dhcp_server *server,
                       dhcpserverstate state)
{
    dhcpserverstate old;

    return_val_if_fail (server, -EINVAL);

    old = server->state;
    if (old == state)
        return -EALREADY;

    INFO ("Server: %s [%p] old: %s new: %s",
          server->ifname,
          server,
          dhcp_server_state2string (old),
          dhcp_server_state2string (state));

    server->state = state;
    return 0;
}

static void
dhcp_server_update_clients (const dnsmasqevent event,
                            char *interface,
                            char *ipaddress,
                            char *macaddress,
                            char *hostname)
{
    int update = 0;
    GList *temp;

    return_if_fail (event != NUM_DNSMASQ_EVENTS);

    temp = clients;
    for ( ; temp; temp = temp->next) {

        dhcplease_ops *ops = temp->data;
        continue_if_fail (ops);

        if (!ops->ifname ||
                (ops->ifname && !interface) ||
                (ops->ifname && interface &&
                 !g_strcmp0 (ops->ifname, interface)))
            update = 1;

        continue_if_fail (update == 1);
        switch (event) {
        case DNSMASQ_DHCP_LEASE_ADDED:
            if (ops->dhcp_lease_added)
                ops->dhcp_lease_added (interface, ipaddress,
                                       macaddress, hostname);
            break;
        case DNSMASQ_DHCP_LEASE_UPDATED:
            if (ops->dhcp_lease_updated)
                ops->dhcp_lease_updated (interface, ipaddress,
                                         macaddress, hostname);
            break;
        case DNSMASQ_DHCP_LEASE_DELETED:
            if (ops->dhcp_lease_deleted)
                ops->dhcp_lease_deleted (interface, ipaddress,
                                         macaddress, hostname);
            break;
        default:
            ERROR ("Invalid dnsmasq event: %d", event);
            break;
        }
    }
}

void
dhcp_server_dhcp_event (const dnsmasqevent event,
                        char *interface,
                        char *ipaddress,
                        char *macaddress,
                        char *hostname)
{
    dhcp_server_update_clients (event, interface, ipaddress,
                                macaddress, hostname);
}


static int
validate_ipaddress (const char *address)
{
    int ret;
    struct in_addr ip;

    return_val_if_fail (address, -EINVAL);
    ret = inet_aton (address, &ip);
    return_val_if_fail (ret, -ENXIO);
    return 0;
}

int
dhcp_server_driver_register (dhcpserver_ops *ops)
{
    return_val_if_fail (ops, -EINVAL);

    if (driver == ops)
        return -EEXIST;

    DEBUG ("Registering DHCP-DNS driver: %s",
           ops->clientname ? ops->clientname : "UNKNOWN");

    driver = ops;
    return 0;
}

int
dhcp_server_driver_unregister (dhcpserver_ops *ops)
{
    return_val_if_fail (ops, -EINVAL);

    if (driver != ops)
        return -ENOENT;

    DEBUG ("Unregistering DHCP-DNS driver: %s",
           ops->clientname ? ops->clientname : "UNKNOWN");

    driver = NULL;
    return 0;
}

int
dhcp_server_register (dhcplease_ops *ops)
{
    int found = 0;
    GList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("Registering the client with DHCP Server: %s [%p]",
           ops->clientname ? ops->clientname : "UNKNOWN", ops);

    temp = clients;
    for ( ; temp; temp = temp->next) {
        dhcplease_ops *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (!found) {
        clients = g_list_append (clients, ops);
        return 0;
    }

    return -EALREADY;
}

int
dhcp_server_unregister (dhcplease_ops *ops)
{
    int found = 0;
    GList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("UnRegistering the client with DHCP Server %s [%p]",
           ops->clientname ? ops->clientname : "UNKNOWN", ops);

    temp = clients;
    for ( ; temp; temp = temp->next) {
        dhcplease_ops *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (found) {
        clients = g_list_remove_link (clients, temp);
        return 0;
    }

    return -ENOENT;
}

static char**
validate_nameservers (char **servers)
{
    char **temp;
    GList *valid = NULL, *dlist;
    unsigned int index = 0, length = 0;

    return_val_if_fail (servers, NULL);

    for (temp = servers; *temp; temp++)
        if (!validate_ipaddress (*temp))
            valid = g_list_append (valid, *temp);

    length = g_list_length (valid);
    temp = g_try_malloc0 (sizeof (char *) * (length + 1));
    return_val_if_fail (temp, NULL);

    for (dlist = valid; dlist; dlist = dlist->next)
        temp [index++] = dlist->data;

    return temp;
}

struct dhcp_server*
dhcpserver_create (const char *interface,
                  int type,
                  const char *start_ip,
                  const char *end_ip,
                  int poolsize,
                  gboolean dnssupport,
                  unsigned int lease_time)
{
    int ret;
    unsigned int index;
    struct ippool *pool = NULL;
    struct dhcp_server *server;
    unsigned int start = 0, end = 0;

    return_val_if_fail (interface, NULL);

    INFO ("Network interface: \"%s\" "
          "IP Pool type: \"%s\" "
          "Start IP address: \"%s\" "
          "End IP address: \"%s\" "
          "pool size: \"%d\" "
          "DNS Support: \"%s\" "
          "Lease time period: \"%u\"",
          interface,
          ippool_type2string (type),
          start_ip, end_ip, poolsize,
          dnssupport ? "Y":"N", lease_time);

    /* Lease period shall be a minimum of 2 mins */
    if (lease_time < (2 * 60))
        lease_time = DHCP_LEASE_PERIOD;

    index = if_nametoindex (interface);
    if (!index) {
        ERROR ("Failed to get the index for the interface [%s], "
               "error: %s/%d", interface, strerror (errno), errno);
        return NULL;
    }

    server = g_hash_table_lookup (dhcpservers, GUINT_TO_POINTER (index));
    if (server)
        return server;

    type = (ippooltype) type;
    if (type == IPPOOL_TYPE_DYNAMIC) {

        /* for a dynamic DHCP pool we dont need either
         * of these */
        if (start_ip || end_ip)
            return NULL;

        /* by default a pool size of 250 is returned */
        pool = ippool_create_from_nextblock (interface, DHCP_POOL_START, DHCP_POOL_END);
        return_val_if_fail (pool, NULL);
    } else if (type == IPPOOL_TYPE_FIXED) {

        return_val_if_fail (start_ip, NULL);

        ret = validate_ipaddress (start_ip);
        if (ret < 0)
            return NULL;

        start = ntohl (inet_addr (start_ip));
        ret = validate_ipaddress (end_ip);
        if (!ret)
            end =  ntohl (inet_addr (end_ip));

        /* overlapping blocks */
        if (end && ((start & 0xffffff00) != (end & 0xffffff00))) {
            pool = ippool_create_from_nextblock (interface, DHCP_POOL_START, DHCP_POOL_END);
            return_val_if_fail (pool, NULL);
        } else {
            pool = ippool_create_from_ipaddress (interface, start_ip,
                                                 start & 0x000000ff,
                                                 !end ? DHCP_POOL_END
                                                      : end & 0x000000ff);
            return_val_if_fail (pool, NULL);
        }
    }

    server = g_try_malloc0 (sizeof (*server));
    if (!server) {
        if (pool)
            ippool_destroy (interface);
        return NULL;
    }

    server->index = index;
    server->dnsservice = dnssupport;
    server->pool = pool;
    server->lease_period = lease_time;
    server->ifname = g_strdup (interface);
    server->pool_size = (unsigned int) poolsize;

    INFO ("A new DHCP server created for: %s [%p]",
          interface, server);

    dhcp_server_set_state (server, DHCP_SERVER_STATE_INITIALIZED);
    g_hash_table_replace (dhcpservers, GUINT_TO_POINTER (index), server);

    return server;
}

int
dhcpserver_start (struct dhcp_server *server,
                  char **nameservers,
                  char **ntpservers)
{
    unsigned int index;
    struct dhcp_server *local;
    unsigned char prefixlen = 0;
    int ret;
    const char *gateway, *subnet,
            *startip, *endip, *broadcast;
    char **nmservers = NULL, **ntservers = NULL, *interface;
    dhcpserverstate state = DHCP_SERVER_STATE_STARTED;

    return_val_if_fail (server, -ENODATA);
    return_val_if_fail (driver, -ENOTCONN);
    return_val_if_fail (driver->dhcp_server_start, -EOPNOTSUPP);

    interface = dhcp_server_get_interface (server);

    DEBUG ("DHCP server [%p]: %s state: %s", server,
           interface, dhcp_server_state2string (server->state));

    index = if_nametoindex (interface);
    if (!index) {
        ERROR ("Failed to get the index for the interface [%s], "
               "error: %s/%d", interface, strerror (errno), errno);
        return -ENODEV;
    }

    local = g_hash_table_lookup (dhcpservers, GUINT_TO_POINTER (index));
    return_val_if_fail (local, -ENOENT);
    return_val_if_fail (local == server, -ENOENT);

    if (server->state == DHCP_SERVER_STATE_STARTED)
        return 0;

    if (server->state != DHCP_SERVER_STATE_INITIALIZED &&
            server->state != DHCP_SERVER_STATE_STOPPED &&
            server->state != DHCP_SERVER_STATE_START_FAILED)
        return -EINVAL;

    gateway = ippool_get_gateway (server->pool);
    subnet = ippool_get_subnet (server->pool);
    startip = ippool_get_start (server->pool);
    endip = ippool_get_end (server->pool);
    broadcast = ippool_get_broadcast (server->pool);
    nmservers = validate_nameservers (nameservers);
    ntservers = validate_nameservers (ntpservers);

    prefixlen = calculate_netmask (ippool_get_subnet (server->pool));
    if (prefixlen == 32 || prefixlen == 255)
        prefixlen = 24;

    ret = ipconfig_add_address (index, AF_INET, gateway,
                                NULL, prefixlen, broadcast);
    if (ret < 0) {
        if (ret != -EEXIST) {
            ERROR ("Failed to configure the network interface \"%s\": %s/%d",
                   interface, strerror (-ret), -ret);
            state = DHCP_SERVER_STATE_START_FAILED;
            goto error;
        }
    }

    ret = driver->dhcp_server_start (interface,
                                     gateway,
                                     subnet,
                                     startip, endip,
                                     broadcast,
                                     nmservers,
                                     ntservers,
                                     server->dnsservice,
                                     server->lease_period);
    if (ret < 0) {
        ERROR ("Failed to start the dhcp server for: \"%s\" [%s/%d]",
               interface, strerror (-ret), -ret);
        state = DHCP_SERVER_STATE_START_FAILED;
    }

error:
    dhcp_server_set_state (server, state);
    g_free (nmservers);
    g_free (ntservers);
    return ret;
}

int
dhcpserver_destroy (struct dhcp_server *server)
{
    unsigned int index;
    char *interface;
    int found = 0;
    gboolean removed = FALSE;
    GHashTableIter iter;
    gpointer key, gvalue;

    return_val_if_fail (server, -EINVAL);

    interface = dhcp_server_get_interface (server);
    DEBUG ("Destroy DHCP server [%p]: %s state: %s", server,
           interface, dhcp_server_state2string (server->state));

    index = if_nametoindex (interface);
    if (!index) {

        g_hash_table_iter_init (&iter, dhcpservers);
        while (g_hash_table_iter_next (&iter, &key, &gvalue)) {
            server = gvalue;
            if (server && !g_strcmp0 (interface, server->ifname)) {
                found = 1;
                index = server->index;
                break;
            }
        }

        return_val_if_fail (found, -ENOENT);
    }

    removed = g_hash_table_remove (dhcpservers, GUINT_TO_POINTER (index));
    return removed == TRUE ? 0 : -ENOENT;
}

static void
dhcpserver_cb (char *ifname,
               int result,
               GVariant *value,
               void *userdata)
{
    int found = 0;
    GList *temp;
    unsigned int index;
    struct dhcp_server *server;
    GHashTableIter iter;
    gpointer key, gvalue;

    (void) value;
    (void) userdata;

    return_if_fail (ifname);

    DEBUG ("DHCP server stopped for: %s", ifname);

    index = if_nametoindex (ifname);
    if (!index) {

        g_hash_table_iter_init (&iter, dhcpservers);
        while (g_hash_table_iter_next (&iter, &key, &gvalue)) {
            server = gvalue;
            if (server && !g_strcmp0 (ifname, server->ifname)) {
                found = 1;
                break;
            }
        }

        return_if_fail (found);
    } else
        server = g_hash_table_lookup (dhcpservers, GUINT_TO_POINTER (index));

    return_if_fail (server);
    temp = clients;
    for ( ; temp; temp = temp->next) {
        dhcplease_ops *ops = temp->data;
        if (ops && ops->dhcp_server_stopped)
            ops->dhcp_server_stopped (result, ifname, userdata);
    }

    if (0 == result)
        dhcp_server_set_state (server, DHCP_SERVER_STATE_STOPPED);
}

int
dhcpserver_stop (struct dhcp_server *server,
                 void *data)
{
    int ret, found = 0;
    unsigned int index;
    const char *gateway, *broadcast;
    char *interface;
    GHashTableIter iter;
    gpointer key, value;
    struct dhcp_server *local = NULL;
    dhcpserverstate state = DHCP_SERVER_STATE_STOPPING;

    return_val_if_fail (server, -EINVAL);

    interface = dhcp_server_get_interface (server);
    DEBUG ("Stop DHCP server [%p]: %s state: %s", server,
           interface, dhcp_server_state2string (server->state));

    /* There is a possibility that the device gets
     * unregistered from kernel and therefore this
     * syscall fails */
    index = if_nametoindex (interface);
    if (!index) {

        g_hash_table_iter_init (&iter, dhcpservers);
        while (g_hash_table_iter_next (&iter, &key, &value)) {
            local = value;
            if (local && !g_strcmp0 (interface, local->ifname)) {
                found = 1;
                break;
            }
        }
        return_val_if_fail (found, -ENOENT);
    } else
        local = g_hash_table_lookup (dhcpservers, GUINT_TO_POINTER (index));

    return_val_if_fail (local, -ENOENT);
    return_val_if_fail (local == server, -ENOENT);

    if (server->state == DHCP_SERVER_STATE_STOPPED)
        return 0;

    if (server->state != DHCP_SERVER_STATE_STARTED &&
            server->state != DHCP_SERVER_STATE_STOP_FAILED)
        return -EALREADY;

    return_val_if_fail (driver, -ENOTCONN);
    return_val_if_fail (driver->dhcp_server_stop, -EOPNOTSUPP);

    gateway = ippool_get_gateway (server->pool);
    broadcast = ippool_get_broadcast (server->pool);

    ret = ipconfig_delete_address (index, AF_INET, gateway,
                                   NULL, 24, broadcast);
    if (ret < 0 && ret != -ENOTSUP) {
        ERROR ("Failed to configure the network interface \"%s\": %s/%d",
               interface, strerror (-ret), -ret);
        return ret;
    }

    ret = driver->dhcp_server_stop (interface, dhcpserver_cb, data);
    if (ret < 0) {
        ERROR ("Failed to stop the dhcp server \"%s\": %s/%d", interface,
               strerror (-ret), -ret);
        state = DHCP_SERVER_STATE_STOP_FAILED;
    }

    dhcp_server_set_state (server, state);
    return ret;
}

int
dhcpserver_clear_cache (const char *ifname,
                        dhcp_server_callback cb,
                        void *userdata)
{
    int ret;

    return_val_if_fail (driver, -ENOTCONN);
    return_val_if_fail (driver->dhcp_server_clear_cache, -EOPNOTSUPP);
    ret = driver->dhcp_server_clear_cache (ifname, cb, userdata);
    if (ret < 0)
        ERROR ("Failed to stop the clear the dnsmsaq cache: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

int
dhcpserver_set_servers (const char *ifname,
                        GList *servers,
                        dhcp_server_callback cb,
                        void *userdata)
{
    int ret;
    char *address;
    GList *temp;

    if (!g_list_length (servers))
        return -EINVAL;

    return_val_if_fail (driver, -ENOTCONN);
    return_val_if_fail (driver->dhcp_server_set_servers, -EOPNOTSUPP);

    temp = servers;
    for ( ; temp; temp = temp->next) {
        address = (char *) temp->data;
        continue_if_fail (address);
        ret = validate_ipaddress (address);
        if (ret < 0)
            return ret;
    }

    ret = driver->dhcp_server_set_servers (ifname, servers, cb, userdata);
    if (ret < 0)
        ERROR ("Failed to stop the set servers to dnsmsaq: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

int
dhcpserver_add_lease (const char *ifname,
                      const char *ipaddress,
                      const char *macaddress,
                      const char *hostname,
                      const char *clientid,
                      unsigned int leaseduration,
                      unsigned int iaid,
                      gboolean istemporary,
                      dhcp_server_callback cb,
                      void *userdata)
{
    int ret;

    ret = validate_ipaddress (ipaddress);
    if (ret < 0)
        return ret;

    return_val_if_fail (driver, -ENOTCONN);
    return_val_if_fail (driver->dhcp_server_add_lease, -EOPNOTSUPP);
    ret = driver->dhcp_server_add_lease (ifname, ipaddress, macaddress, hostname,
                                         clientid, leaseduration, iaid, istemporary,
                                         cb, userdata);
    if (ret < 0)
        ERROR ("Failed to add a new lease to the dnsmsaq: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

int
dhcpserver_remove_lease (const char *ifname,
                         const char *ipaddress,
                         dhcp_server_callback cb,
                         void *userdata)
{
    int ret;

    ret = validate_ipaddress (ipaddress);
    if (ret < 0)
        return ret;

    return_val_if_fail (driver, -ENOTCONN);
    return_val_if_fail (driver->dhcp_server_remove_lease, -EOPNOTSUPP);
    ret = driver->dhcp_server_remove_lease (ifname, ipaddress, cb, userdata);
    if (ret < 0)
        ERROR ("Failed to add a remove lease from dnsmsaq: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

struct ippool*
dhcp_server_get_ippool (struct dhcp_server *server)
{
	return_val_if_fail (server, NULL);
	return server->pool;
}

char*
dhcp_server_get_interface (struct dhcp_server *server)
{
    return_val_if_fail (server, NULL);
    return server->ifname;
}

int
dhcp_server_init ()
{
    DEBUG ("");

    dhcpservers = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL,
                                         dhcp_server_cleanup);

    return 0;
}

int
dhcp_server_deinit ()
{
    DEBUG ("");

    driver = NULL;
    g_hash_table_destroy (dhcpservers);
    g_list_free (clients);
    return 0;
}

/** @} */
