/**
 * @file server.h
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#ifndef DHCP_SERVER_H
#define DHCP_SERVER_H

#include <dnsmasq-man.h>


typedef
enum {
    DHCP_SERVER_STATE_UNKNOWN,
    DHCP_SERVER_STATE_INITIALIZED,
    DHCP_SERVER_STATE_STARTED,
    DHCP_SERVER_STATE_STOPPING,
    DHCP_SERVER_STATE_STOPPED,
    DHCP_SERVER_STATE_START_FAILED,
    DHCP_SERVER_STATE_STOP_FAILED
} dhcpserverstate;

/* opaque */
struct dhcp_server;
struct ippool;

/**
 * struct dhcplease_ops - DHCP lease callbacks
 *
 * This struct shall be registered/subscribed by the clients
 * to get the updates on DHCP leases.
 *
 * @clientname - name of the client
 *
 * @ifname - name of the interface. If the client is controlling
 * a single interface, the client could only register for the updates
 * on that particular interface. Clients can leave this option in
 * order to get the lease information on any interface
 *
 * @dhcp_lease_added - A DHCP lease is obtained by a wifi station
 *  @interface: name of the interface
 *  @ipaddress: Offered IP address
 *  @macaddress: Mac address of the client
 *  @hostname: Name of the client
 *
 * @dhcp_lease_updated - An existing DHCP lease is updated i.e., DHCP
 *  Renewal or client has again acquired the same lease. This happens
 *  during the reconnections.
 *  @interface: name of the interface
 *  @ipaddress: Offered IP address
 *  @macaddress: Mac address of the client
 *  @hostname: Name of the client
 *
 * @dhcp_lease_deleted - A DHCP lease is deleted from dnsmasq db.
 *  @interface: name of the interface
 *  @ipaddress: Offered IP address
 *  @macaddress: Mac address of the client
 *  @hostname: Name of the client
 */

typedef
struct __dhcp_lease_callbacks
{
    const char *clientname;
    const char *ifname;

    void (*dhcp_lease_added) (char *interface,
                              char *ipaddress,
                              char *macaddress,
                              char *hostname);
    void (*dhcp_lease_updated) (char *interface,
                                char *ipaddress,
                                char *macaddress,
                                char *hostname);
    void (*dhcp_lease_deleted) (char *interface,
                                char *ipaddress,
                                char *macaddress,
                                char *hostname);
    void (*dhcp_server_stopped) (int result,
                                 char *ifname,
                                 void *data);

} dhcplease_ops;

/**
 * dhcp_server_callback - Callback for an async dhcp server operations
 *  @ifname - interface name
 *  @result - 0 on success and -1 on failure
 *  @ret - GVariant which wraps the actual return value of dbus calls
 *   i.e., if the dbus call is supposed to return a value
 *  @userdata - userdata to be passed along with the result information
 */
typedef
void (*dhcp_server_callback) (char *ifname,
                              int result,
                              GVariant *ret,
                              void *userdata);

/**
 * struct dhcpserver_ops - backend description for DHCP Server
 * configurations
 *
 * This struct is registered by the real DHCP Server driver
 * in order to handle configuration requests related to DHCP and
 * dns.
 *
 * All callbacks except where otherwise noted should return 0
 * on success or a negative error code.
 *
 * @clientname - name of the DHCP & DNS Server behind.
 *
 * @dhcp_server_start - Start the DHCP Server
 *  @ifname: name of the interface
 *  @router: Gateway address to be included in the DHCP lease
 *  @subnet: Subnetwork range
 *  @start_ip: Start IP address of the network
 *  @end_ip: End IP address of the network
 *  @broadcast: Broadcast address of the network
 *  @lease_time: lease time in minutes. IP leases will be offered
 *   to client for that given length of time. The minimum lease
 *   period shall be 2 minutes.
 *
 * @dhcp_server_stop - Stop the DHCP Server
 *  @ifname: name of the interface whose DHCP server shall be
 *   stopped
 *  @cb: callback to be invoked for the result
 *  @userdata: any userdata to be passed along with the result
 *
 * @dhcp_server_clear_cache - Clears the domain name cache.
 *  @ifname: name of the interface
 *  @cb: callback to be invoked for the result
 *  @userdata: any userdata to be passed along with the result
 *
 * @dhcp_server_set_servers - Set of upstream DNS servers to be used
 *  by the DNS manager i.e., dnsmasq. Normally upstream DNS servers
 *  are taken from /etc/resolv.conf
 *  @ifname: name of the interface
 *  @servers: List of upstream DNS servers
 *  @cb: callback to be invoked for the result
 *  @userdata: any userdata to be passed along with the result
 *
 * @dhcp_server_add_lease - Adds or updates a DHCP or DHCPv6 lease to
 *  the internal lease database, as if a client requested and obtained
 *  a lease
 *  @ifname: name of the interface
 *  @ipaddress: Offered IP address
 *  @macaddress: Mac address of the client
 *  @hostname: Name of the client
 *  @clientid: The client identifier (IPv4) or DUID (IPv6)
 *  @iaid: The IAID (Identity association identifier) of the DHCPv6
 *   lease, as a network byte-order unsigned integer. For DHCPv4
 *   leases, this must be set to 0.
 *  @istemporary: A boolean which, if true, indicates that the DHCPv6
 *   lease is for a temporary address (IA_TA). If false, the DHCPv6
 *   lease is for a non-temporary address (IA_NA). For DHCPv4 leases,
 *   this must be set to false.
 *  @cb: callback for the result
 *  @userdata: any userdata to be passed along with the result
 *
 * @dhcp_server_remove_lease - Removes a DHCP or DHCPv6 lease to the
 * internal lease database, as if a client sent a release message to
 * abandon a lease.
 *  @ifname: name of the interface
 *  @ipaddress: IP address representing the IP lease to be removed from
 *   the internal db
 *  @cb: callback to be invoked for the result
 *  @userdata: any userdata to be passed along with the result
 */

typedef
struct __dhcp_server_ops
{
    const char *clientname;
    int (*dhcp_server_start) (const char *ifname,
                              const char *router,
                              const char *subnet,
                              const char *start_ip,
                              const char *end_ip,
                              const char *broadcast,
                              char **nameservers,
                              char **ntpservers,
                              gboolean dnssupport,
                              unsigned int lease_time);
    int (*dhcp_server_stop) (const char *ifname,
                             dhcp_server_callback cb,
                             void *userdata);
    int (*dhcp_server_clear_cache) (const char *ifname,
                                    dhcp_server_callback cb,
                                    void *userdata);
    int (*dhcp_server_set_servers) (const char *ifname,
                                    GList *servers,
                                    dhcp_server_callback cb,
                                    void *userdata);
    int (*dhcp_server_add_lease) (const char *ifname,
                                  const char *ipaddress,
                                  const char *macaddress,
                                  const char *hostname,
                                  const char *clientid,
                                  unsigned int leaseduration,
                                  unsigned int iaid,
                                  gboolean istemporary,
                                  dhcp_server_callback cb,
                                  void *userdata);
    int (*dhcp_server_remove_lease) (const char *interface,
                                     const char *ipaddress,
                                     dhcp_server_callback cb,
                                     void *userdata);

} dhcpserver_ops;

/**
 * dhcp_server_driver_register - Register a DHCP & DNS driver
 * @ops: DHCP & DNS driver to register
 * Returns: 0 on success or a negative error code representing the
 * error
 */
int dhcp_server_driver_register (dhcpserver_ops *ops);

/**
 * dhcp_server_driver_register - Unregister a DHCP & DNS driver
 * @ops: DHCP & DNS driver to unregister
 * Returns: 0 on success or a negative error code representing the
 * error
 */
int dhcp_server_driver_unregister (dhcpserver_ops *ops);


/**
 * dhcp_server_dhcp_event - A DHCP lease event has occurred and DHCP
 * driver will call this. Note: client shall not call this function
 * @event: DHCP Lease event i.e., DchpLeaseAdded/Updated/Deleted
 * @ipaddress: Offered IP address
 * @macaddress: Mac address of the client
 * @hostname: Name of the client
 */
void dhcp_server_dhcp_event (const dnsmasqevent event,
                             char *interface,
                             char *ipaddress,
                             char *macaddress,
                             char *hostname);

struct dhcp_server* dhcpserver_create (const char *interface,
                                       int type,
                                       const char *start_ip,
                                       const char *end_ip,
                                       int poolsize,
                                       gboolean dnssupport,
                                       unsigned int lease_time);

int dhcpserver_destroy (struct dhcp_server *server);

/**
 * @dhcpserver_start - Start the DHCP Server
 *  @ifname: name of the interface
 *  @router: Gateway address to be included in the DHCP lease
 *  @type: Type of the IP lease
 *  @start_ip: Start IP address of the network
 *  @end_ip: End IP address of the network
 *  @poolsize: Pool size of the IP lease
 *  @dnssupport: Resolve DNS within this host or forward everything
 *   to the router
 *  @lease_time: lease time in minutes. IP leases will be offered
 *   to client for that given length of time. The minimum lease
 *   period shall be 2 minutes.
 */
int dhcpserver_start (struct dhcp_server *server,
                      char **nameservers,
                      char **ntpservers);

/**
 * @dhcpserver_stop - Stop the DHCP Server
 *  @ifname: name of the interface whose DHCP server shall be
 *   stopped
 *  @cb: callback to be invoked for the result
 *  @userdata: any userdata to be passed along with the result
 */
int dhcpserver_stop (struct dhcp_server *server, void *data);

/**
 * @dhcpserver_clear_cache - Clears the domain name cache.
 *  @ifname: name of the interface
 *  @cb: callback to be invoked for the result
 *  @userdata: any userdata to be passed along with the result
 */
int dhcpserver_clear_cache (const char *ifname,
                            dhcp_server_callback cb,
                            void *userdata);

/**
 * @dhcpserver_set_servers - Set of upstream DNS servers to be used
 *  by the DNS manager i.e., dnsmasq. Normally upstream DNS servers
 *  are taken from /etc/resolv.conf
 *  @ifname: name of the interface
 *  @servers: List of upstream DNS servers
 *  @cb: callback to be invoked for the result
 *  @userdata: any userdata to be passed along with the result
 */
int dhcpserver_set_servers (const char *ifname,
                            GList *servers,
                            dhcp_server_callback cb,
                            void *userdata);

/**
 * @dhcpserver_add_lease - Adds or updates a DHCP or DHCPv6 lease to
 *  the internal lease database, as if a client requested and obtained
 *  a lease
 *  @ifname: name of the interface
 *  @ipaddress: Offered IP address
 *  @macaddress: Mac address of the client
 *  @hostname: Name of the client
 *  @clientid: The client identifier (IPv4) or DUID (IPv6)
 *  @iaid: The IAID (Identity association identifier) of the DHCPv6
 *   lease, as a network byte-order unsigned integer. For DHCPv4
 *   leases, this must be set to 0.
 *  @istemporary: A boolean which, if true, indicates that the DHCPv6
 *   lease is for a temporary address (IA_TA). If false, the DHCPv6
 *   lease is for a non-temporary address (IA_NA). For DHCPv4 leases,
 *   this must be set to false.
 *  @cb: callback for the result
 *  @userdata: any userdata to be passed along with the result
 */
int dhcpserver_add_lease (const char *ifname,
                          const char *ipaddress,
                          const char *macaddress,
                          const char *hostname,
                          const char *clientid,
                          unsigned int leaseduration,
                          unsigned int iaid,
                          gboolean istemporary,
                          dhcp_server_callback cb,
                          void *userdata);

/**
 * @dhcpserver_remove_lease - Removes a DHCP or DHCPv6 lease to the
 * internal lease database, as if a client sent a release message to
 * abandon a lease.
 *  @ifname: name of the interface
 *  @ipaddress: IP address representing the IP lease to be removed from
 *   the internal db
 *  @cb: callback to be invoked for the result
 *  @userdata: any userdata to be passed along with the result
 */
int dhcpserver_remove_lease (const char *interface,
                             const char *ipaddress,
                             dhcp_server_callback cb,
                             void *userdata);

int dhcp_server_register (dhcplease_ops *ops);
int dhcp_server_unregister (dhcplease_ops *ops);

struct ippool *dhcp_server_get_ippool (struct dhcp_server *server);
char *dhcp_server_get_interface (struct dhcp_server *server);


/**
 * dhcp_server_init - Initialize the dhcp server
 */
int dhcp_server_init ();

/**
 * dhcp_server_deinit - Deinitialize the dhcp server
 */
int dhcp_server_deinit ();

#endif //DHCP_SERVER_H

/** @} */
