/**
 * @file          crypto.c
 * @author        RBEI/ECO32-Jiji Anna Jacob
 * @copyright     (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup    ai_connectivity\wifi_mw
 * @brief
 * Implementation to encrypt/decrypt files for
 * Wifi_AP_Direct_Manager using OpenSSL's cryptographic
 * library - using AES 128 bit encryption and the cfb
 * block cipher mode of operation
 * @{
 */

#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <openssl/aes.h>
#include <openssl/rand.h>
#include <openssl/err.h>
#include <openssl/sha.h>
#include <glib.h>
#include <log.h>

#define HASH_ALGORITHM_RESP_SUCCESS    1

unsigned char indata[AES_BLOCK_SIZE];
unsigned char outdata[AES_BLOCK_SIZE];
unsigned char cKey[16]= "8abfe3b1cdeacfea";

int
generate_rand_key ()
{
    int rc;
    unsigned long err = 0;

    rc = RAND_bytes (cKey, sizeof(cKey));
    if(rc != 1) {
        err = ERR_get_error();
        DEBUG ("Error code : %ld", err);
        return -1;
    }

    return 0;
}

/**
 * encrypt_file
 *@filepath_plain: Path to plain text file
 *@filepath_enc: Path to encrypted file 
 *
 * Function invoked to encrypt file @filepath_plain and save 
 * @filepath_enc using AES-cfb 128 Bit encryption
 * Returns: 0 on success,
 * -errno on failure
 *
 **/
int
encrypt_file (const char *filepath_plain,
              const char *filepath_enc)
{
    FILE *ifp,*ofp;
    int num = 0, ret = 0;
    size_t bytes_read, bytes_written;
    unsigned char indata [AES_BLOCK_SIZE];
    unsigned char outdata [AES_BLOCK_SIZE];
    unsigned char iVector [AES_BLOCK_SIZE];
    /* set where on the 128 bit encrypted block to begin encryption*/
    AES_KEY aesKey;

    return_val_if_fail (filepath_plain && filepath_enc, -EINVAL);

    ifp = fopen (filepath_plain, "r");
    if (!ifp) {
        ERROR ("Failed to open the file %s, error : %s", filepath_plain, strerror(errno));
        return -errno;
    }

    ofp = fopen (filepath_enc, "w");
    if (!ofp) {
        ERROR ("Failed to open the file %s, error : %s", filepath_enc, strerror(errno));
        fclose (ifp);
        return -errno;
    }

    /* set the encryption key */
    AES_set_encrypt_key (cKey, 128, &aesKey);
    /*Set the initialisation vector */
    memset (iVector, 0x01, AES_BLOCK_SIZE);

    while (1) {
        bytes_read = fread (indata, 1, AES_BLOCK_SIZE, ifp);
        AES_cfb128_encrypt (indata, outdata, bytes_read,
                            &aesKey, iVector, &num,
                            AES_ENCRYPT);
        bytes_written = fwrite (outdata, 1, bytes_read, ofp);

        (void) bytes_written;
//        DEBUG ("read %d written %d", bytes_read, bytes_written);
        if (bytes_read < AES_BLOCK_SIZE)
            break;
    }

    if (fclose (ifp) < 0) {
        ERROR ("Failed to close the file : %s [%s]", filepath_plain, strerror (errno));
        ret = -errno;
    }

    if (fclose (ofp) < 0) {
        ERROR ("Failed to close the file : %s [%s]", filepath_enc, strerror (errno));
        ret = -errno;
    }

    return ret;
}
/**
 * decrypt_file
 *@filepath_plain: Path to plain text file
 *@filepath_enc: Path to encrypted file
 *
 * Function invoked to decrypt file @filepath_enc and save
 * @filepath_plain using AES-cfb 128 Bit encryption
 * Returns: 0 on success,
 * -errno on failure
 *
 **/
int
decrypt_file (const char *filepath_enc,
              const char *filepath_plain)
{
    FILE *ifp,*ofp;
    int num = 0, ret = 0;
    size_t bytes_read, bytes_written;
    unsigned char indata[AES_BLOCK_SIZE];
    unsigned char outdata[AES_BLOCK_SIZE];
    unsigned char iVector[AES_BLOCK_SIZE];
    AES_KEY aesKey;
    /* set where on the 128 bit encrypted block to begin decryption*/

    return_val_if_fail (filepath_plain && filepath_enc, -EINVAL);

    ifp = fopen (filepath_enc, "r");
    if (!ifp) {
        ERROR ("Failed to open the file \"%s\", error : %s", filepath_enc, strerror (errno));
        return -errno;
    }

    ofp = fopen (filepath_plain, "w");
    if (!ofp) {
        ERROR ("Failed to open the file \"%s\", error : %s", filepath_plain, strerror(errno));
        fclose (ifp);
        return -errno;
    }

    /* set the encryption key */
    AES_set_encrypt_key(cKey, 128, &aesKey);
    /*Set the initialisation vector */
    memset(iVector, 0x01, AES_BLOCK_SIZE);

    while (1) {
        bytes_read = fread(indata,1,AES_BLOCK_SIZE, ifp);
        AES_cfb128_encrypt(indata, outdata, bytes_read, &aesKey,iVector, &num,
                           AES_DECRYPT);
        bytes_written = fwrite(outdata,1,bytes_read,ofp);

        (void) bytes_written;
//        DEBUG ("read %d written %d", bytes_read, bytes_written);
        if (bytes_read < AES_BLOCK_SIZE)
            break;
    }

    if (fclose (ifp) < 0) {
        ERROR ("Failed to close the file : %s [%s]", filepath_plain, strerror (errno));
        ret = -errno;
    }

    if (fclose (ofp) < 0) {
        ERROR ("Failed to close the file : %s [%s]", filepath_enc, strerror (errno));
        ret = -errno;
    }

    return ret;
}

int
calculate_sha256_hash (const char *file,
                    char **hash_value)
{
    int n, fd;
    SHA256_CTX ctx;
    char buf [512];
    char hash [512], temp [3];
    ssize_t bytes;
    unsigned char out [SHA256_DIGEST_LENGTH + 1];

    return_val_if_fail (file && hash_value &&
                        !*hash_value, -EINVAL);

    fd = open (file, O_CLOEXEC | O_RDONLY);
    if (fd < 0)
        return -errno;

    if (HASH_ALGORITHM_RESP_SUCCESS != SHA256_Init (&ctx))
        goto failure;

    memset (buf, 0, sizeof (buf));
    while ((bytes = read (fd, buf, sizeof (buf))) > 0) {
        if (HASH_ALGORITHM_RESP_SUCCESS != SHA256_Update (&ctx, buf, (size_t) bytes))
            goto failure;
        memset (buf, 0, sizeof (buf));
    }

    memset (out, 0, sizeof (out));
    if (HASH_ALGORITHM_RESP_SUCCESS != SHA256_Final (out, &ctx))
        goto failure;

    memset (hash, 0, sizeof (hash));
    for(n = 0; n < SHA256_DIGEST_LENGTH; n++) {
        memset (temp, 0, sizeof (temp));
        sprintf (temp, "%02x", out [n]);
        strcat (hash, temp);
    }

    *hash_value = g_strdup (hash);
    (void)close (fd);
    return 0;

failure:
    (void)close (fd);
    return -EINVAL;
}
