/**
 * @file nlutils.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <linux/netlink.h>
#include <netlink/netlink.h>
#include <glib.h>
#include <log.h>

#define NETLINK_TYPE_CHECK(type)    (type < 32 && type >= 0)
#define CHECK_SOCKET_OPT(opt)       (opt >= 1 && opt < 48)

static int
nl_configure_socket_buf (struct nl_sock *sk,
                         int sockopt,
                         unsigned int pgs)
{
    long sz;
    int ret, buf = 0, skn;
    socklen_t len;

    return_val_if_fail (sk, -EINVAL);
    return_val_if_fail (CHECK_SOCKET_OPT (sockopt), -EINVAL);

    if (sockopt != SO_RCVBUF && sockopt != SO_SNDBUF)
        return -ENOTSUP;

    len = sizeof (buf);
    sz = sysconf (_SC_PAGESIZE);
    if (sz < 0)
        sz = 4096;

    skn = nl_socket_get_fd (sk);
    return_val_if_fail (skn > 0, -EINVAL);
    (void) getsockopt (skn, SOL_SOCKET, sockopt, &buf, &len);

    if (buf >= (int) (pgs * (unsigned int) sz))
        return 0;

    sz = sz * pgs;
    if (sockopt == SO_RCVBUF)
        ret = nl_socket_set_buffer_size (sk, (int)(sz), 0);
    else if (sockopt == SO_SNDBUF)
        ret = nl_socket_set_buffer_size (sk, 0, (int)(sz));

    if (ret < 0)
        return ret;

    buf = 0;
    (void) getsockopt (skn, SOL_SOCKET, sockopt, &buf, &len);
    if (buf < (int) (sz))
        ret = -EIO;

    return ret;
}

int
nl_socket_init (struct nl_sock **sock,
                int family,
                gboolean blocking,
                unsigned int rbpgs,
                unsigned int sbpgs)
{
    int flags, ret, sno;
    struct nl_sock *sk;

    return_val_if_fail (sock, -EINVAL);
    return_val_if_fail (!*sock, -EEXIST);
    return_val_if_fail (NETLINK_TYPE_CHECK (family), -ENOTSUP);

    sk = nl_socket_alloc ();
    return_val_if_fail (sk, -ENOMEM);

    ret = nl_connect (sk, family);
    if (ret < 0)
        goto error;

    sno = nl_socket_get_fd (sk);
    flags = fcntl (sno, F_GETFL);
    if (flags < 0) {
        ret = -errno;
        goto error;
    }

    flags |= O_CLOEXEC;
    if (!blocking)
        flags |= O_NONBLOCK;

    ret = fcntl (sno, F_SETFL, flags);
    if (ret < 0) {
        ret = -errno;
        goto error;
    }

    if (rbpgs > 0) {
        ret = nl_configure_socket_buf (sk, SO_RCVBUF, rbpgs);
        if (ret < 0)
            goto error;
    }

    if (sbpgs > 0) {
        ret = nl_configure_socket_buf (sk, SO_SNDBUF, sbpgs);
        if (ret < 0)
            goto error;
    }

    INFO ("Initiliazed Nl socket: %d", sno);
    *sock = sk;
    return ret;

error:
    nl_socket_free (sk);
    *sock = NULL;
    return ret;
}

/** @} */
