/**
 * @file main.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <sys/signalfd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>
#include <log.h>
#include <inotifier.h>
#include <session.h>
#include <nat.h>
#include <firewall.h>
#include <iptables.h>
#include <iptables-wrapper.h>
#include <genl.h>
#include <rtnl.h>
#include <device.h>
#include <ipconfig.h>
#include <ippool.h>
#include <server.h>
#include <dnsmasq-man.h>
#include <accesspoint.h>
#include <tech_manager.h>
#include <ap.h>
#include <objectmanager_if.h>
#include <manager_if.h>
#include <dbus-cli.h>
#include <servicereg.h>
#include <fib.h>
#include <route.h>

#define APP_SUMMARY "Wifi access point and p2p (optional) managing daemon"

static char *apdevices = NULL;
static char *p2pdevices = NULL;
static char *configfile = NULL;
static int sys_log_level = -1;
static char *debug_file = NULL;
static gboolean opt_daemon = TRUE;
static gboolean opt_fwconf = TRUE;

static GMainLoop *main_loop = NULL;
static GDBusConnection* system_bus = NULL;
static unsigned int terminated = 0;

static void wapdman_global_init (void);

static GOptionEntry options[] =
{
    { "apdevices", 'a', 0, G_OPTION_ARG_STRING, &apdevices,
      "Specify wifi networking device(s) or interface(s) to be "
      "managed as access point(s)", "DEV1,..." },
    { "p2pdevices", 'p', 0, G_OPTION_ARG_STRING, &p2pdevices,
      "Specify wifi networking device(s) or interface(s) to be "
      "managed as P2P device(s)", "DEV1,..." },
    { "config-file", 'c', 0, G_OPTION_ARG_STRING, &configfile,
      "Specify configuration file(s)/director(y/ies)"
      " for the access point(s)", "CONF1,..." },
    { "loglevel", 'l', 0, G_OPTION_ARG_INT, &sys_log_level,
      "Specify log level i.e., Messages with this priority will "
      "only be logged", "LOGLEVEL" },
    { "logfile", 'f', 0, G_OPTION_ARG_STRING, &debug_file,
      "log output (everything) to the specified file", "LOGFILE" },
    { "nodaemon", 'n', G_OPTION_FLAG_REVERSE, G_OPTION_ARG_NONE,
      &opt_daemon, "Don't fork wapdman to background (if logging to "
      "stdout is necessary)", NULL },
    { "nofirewallconfiguration", 0, G_OPTION_FLAG_REVERSE,
      G_OPTION_ARG_NONE, &opt_fwconf,
      "Don't configure netfilter i.e., useful if static firewall "
      "configuration is preferred", NULL },
    { NULL }
};

GDBusConnection*
get_system_bus(void)
{
    return system_bus;
}

int
get_log_level (void)
{
    return sys_log_level;
}

const char*
get_log_file (void)
{
    return debug_file;
}

gboolean
get_deamon_option (void)
{
    return opt_daemon;
}
gboolean
get_firewall_configuration (void)
{
    return opt_fwconf;
}

static void
on_wapdman_bus_acquired (GDBusConnection *connection,
                         const char *name,
                         gpointer data)
{
    (void) data;

    DEBUG ("Dbus acquired: %s [connection: %p]", name, connection);

    return_if_fail (connection);

    system_bus = connection;
    (void) objman_init (connection);
    (void) manager_init (connection);
}

static void
on_wapdman_name_appeared (GDBusConnection *connection,
                          const char *name,
                          gpointer data)
{
    (void) data;

    DEBUG ("wifi_ap_direct_manager appeared on system bus: %s [connection: %p]",
    		name, connection);
    wapdman_global_init ();
    (void) rtnl_get_information ();
}

static void
on_wapdman_name_vanished (GDBusConnection *connection,
                          const char *name,
                          gpointer data)
{
    (void) connection;
    (void) data;

    CRITICAL ("wifi_ap_direct_manager %s does not exist anymore in system bus", name);
}

static gboolean
signal_handler (GIOChannel *channel,
                GIOCondition cond,
                gpointer data)
{
    (void) data;
    struct signalfd_siginfo si;
    ssize_t result;
    int fd;

    DEBUG ("Channel: %p GIOCondition: %d", channel, cond);
    if (cond & (G_IO_NVAL | G_IO_ERR | G_IO_HUP))
        return FALSE;

    fd = g_io_channel_unix_get_fd (channel);

    result = read (fd, &si, sizeof (si));
    if (result != sizeof (si))
        return FALSE;

    switch (si.ssi_signo) {
    case SIGINT:
    case SIGTERM:
        if (terminated == 0) {
            INFO ("Terminating the main loop of wifi_ap_direct_manager");
            g_main_loop_quit (main_loop);
        }

        terminated = 1;
        break;
    }

    return TRUE;
}

static unsigned int
setup_signalfd (void)
{
    GIOChannel *channel;
    unsigned int source;
    sigset_t mask;
    int fd, ret;

    sigemptyset (&mask);
    sigaddset (&mask, SIGINT);
    sigaddset (&mask, SIGTERM);

    ret = sigprocmask (SIG_BLOCK, &mask, NULL);
    if (ret < 0) {
        ERROR ("Failed to set signal mask");
        return 0;
    }

    fd = signalfd (-1, &mask, SFD_CLOEXEC);
    if (fd < 0) {
        ERROR ("Failed to create signal descriptor");
        return 0;
    }

    channel = g_io_channel_unix_new (fd);

    g_io_channel_set_close_on_unref (channel, TRUE);
    g_io_channel_set_encoding (channel, NULL, NULL);
    g_io_channel_set_buffered (channel, FALSE);

    source = g_io_add_watch (channel, G_IO_IN | G_IO_HUP |
                             G_IO_ERR | G_IO_NVAL, signal_handler, NULL);

    g_io_channel_unref (channel);
    return source;
}

static void
wapdman_global_init (void)
{
    (void) dbus_cli_init ();
    (void) inot_init ();
    (void) nat_init ();
    (void) firewall_init ();
    (void) iptables_init ();
    (void) iptw_init();
    (void) rtnl_init ();
    (void) device_init ();
    (void) genl_init ();
    (void) ipconfig_init ();
    (void) fib_init();
    (void) route_init();
    (void) ippool_init ();
    (void) dhcp_server_init ();
    (void) dnsmasq_init ();
    (void) wireless_tech_init (apdevices, p2pdevices);
    (void) servicereg_init ();
    (void) ap_init (configfile);
}

static void
wapdman_global_deinit (void)
{
    (void) ap_deinit();
    (void) dnsmasq_deinit ();
    (void) dhcp_server_deinit ();
    (void) ippool_deinit ();
    (void) servicereg_deinit ();
    (void) wireless_tech_deinit ();
    (void) accesspoint_deinit ();
    (void) genl_deinit ();
    (void) nat_deinit ();
    (void) iptables_deinit ();
    (void) iptw_deinit ();
    (void) firewall_deinit ();
    (void) fib_deinit();
    (void) route_deinit();
    (void) ipconfig_deinit ();
    (void) device_deinit ();
    (void) rtnl_deinit ();
    (void) inot_deinit ();
    (void) dbus_cli_deinit ();
}

int
main (int argc,
      char *argv[])
{
    int ret;
    unsigned int signal;
    unsigned int dbusid;
    GError *error = NULL;
    GOptionContext *context = NULL;
    gboolean parse = FALSE;

    context = g_option_context_new ("");

    g_option_context_set_summary (context, APP_SUMMARY);
    g_option_context_add_main_entries (context, options, NULL);
    parse = g_option_context_parse (context, &argc, &argv, &error);
    if (!parse) {
        if (error) {
            ERROR ("Failed to parse the options error message: %s\n",
                   error->message);
            g_error_free (error);
        } else
            ERROR ("An unknown error occurred");

        exit (EXIT_FAILURE);
    }

    g_option_context_free (context);

    if (opt_daemon) {
        ret = daemon (0, 0);
        if (ret != 0) {
            ERROR ("Failed to detach as a daemon");
            exit (EXIT_FAILURE);
        }
    }

    umask (0077);

    main_loop = g_main_loop_new (NULL, FALSE);
    signal = setup_signalfd ();

    dbusid = g_bus_own_name (G_BUS_TYPE_SYSTEM,
                             WAPDMAN_BUS_NAME,
                             G_BUS_NAME_OWNER_FLAGS_NONE,
                             on_wapdman_bus_acquired,
                             on_wapdman_name_appeared,
                             on_wapdman_name_vanished,
                             NULL, NULL);

    if (!dbusid)
        exit (EXIT_FAILURE);

    (void) log_init (argv [0], sys_log_level, debug_file);

    INFO ("Starting WiFi_AP_Direct_Manager");

    DEBUG ("****** daemonize: %s              *********",
           opt_daemon ? "Y" : "N");
    DEBUG ("****** firewall configuration: %s *********",
           opt_fwconf ? "Y" : "N");

    INFO ("++++++++++++++RUNNING MAIN LOOP++++++++++++++");
    g_main_loop_run (main_loop);

    INFO ("Stopping WiFi_AP_Direct_Manager");
    wapdman_global_deinit ();

    g_source_remove (signal);
    g_bus_unown_name (dbusid);

    g_free (apdevices);
    g_free (p2pdevices);
    g_free (configfile);
    g_free (debug_file);

    g_main_loop_unref (main_loop);

    return 0;
}

/** @} */
