/**
 * @file genl.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>
#include <string.h>
#include <netlink/socket.h>
#include <netlink/genl/genl.h>
#include <netlink/genl/ctrl.h>
#include <linux/nl80211.h>
#include <linux/if_ether.h>
#include <linux/version.h>
#include <glib.h>
#include <accesspoint.h>
#include <genl.h>
#include <device.h>
#include <tech_manager.h>
#include <log.h>
#include <device.h>
#include <nlutils.h>
#include "inc/utils.h"

/**
 * ISO 3166-1 alpha-2 size (including the null byte)
 */
#define GENL_ALPHA_SIZE     3

/**
 * Channel is disabled in current regulatory domain
 * i.e., this channel shall not used at all
 */
#define IEEE80211_CHANNEL_DISABLED      (1 << 0)

/**
 * no mechanisms that initiate radiation are permitted
 * on this channel, this includes sending probe requests,
 * or modes of operation that require beaconing. i.e., shall
 * not be used
 */
#define IEEE80211_CHANNEL_NO_IR         (1 << 1)

/**
 * Radar detection is mandatory on this channel in
 * current regulatory domain i.e., DFS has to be
 * enabled
 */
#define IEEE80211_CHANNEL_RADAR         (1 << 2)

/**
 * Shall be used for indoor operations only in this regulatory
 * domain. to be avoided since HMI module is labelled as an
 * outdoor device
 */
#define IEEE80211_CHANNEL_INDOOR        (1 << 4)

/**
 * IR operation is allowed on this channel if it's connected
 * concurrently to a BSS on the same channel on the 2 GHz band
 * or to a channel in the same UNII band (on the 5 GHz band),
 * and IEEE80211_CHAN_RADAR is not set. (to be avoided)
 */
#define IEEE80211_CHANNEL_IR_CONCURRENT (1 << 5)

/**
 * DFS Sates for channels
 * The channel can be used, but channel availability check (CAC)
 * must be performed before using it for AP or IBSS
 */
#define IEEE80211_CHANNEL_DFS_USABLE            (1 << 9)

/**
 * A radar has been detected on this channel, it is therefore
 * marked as not available.
 */
#define IEEE80211_CHANNEL_DFS_UNAVAILABLE       (1 << 10)

/**
 * The channel has been CAC checked and is available
 */
#define IEEE80211_CHANNEL_DFS_AVAILABLE         (1 << 11)

/**
 * Helper to print the mac address
 */
#define MACADDRESS(x)  x[0],x[1],x[2],x[3],x[4],x[5]

/**
 * convert KHz to MHz
 */
#define KHZ_TO_MHZ(x)   (x/1000)

#define DISABLE_RADAR_CHANNELS

/**
 *  enum hwmodes - IEEE80211 modes supported
 *  by the hardware
 */
typedef
enum __hw_modes
{
    IEEE80211_HW_MODE_B = 1,
    IEEE80211_HW_MODE_G = 2,
    IEEE80211_HW_MODE_A = 4,
    IEEE80211_HW_MODE_AD = 8,
    NUM_IEEE80211_HW_MODES = 0xFFFFFFFF
} hwmodes;

/**
 * struct channelinfo - channel information
 */
typedef
struct __channel_info
{
    /**
     * channel - IEEE80211 channel number
     */
    unsigned int channel;

    /**
     * freq - frequency in MHz
     */
    unsigned int freq;

    /**
     * flags - various information about this channel
     * i.e., INDOOR/DISABLED/NO_IR/RADAR/DFS..
     */
    unsigned int flags;

    /**
     * usable - Whether this channel can be used for
     * Wi-Fi operation (AP/P2P-GO)
     */
    unsigned int usable;
} channelinfo;

/**
 * struct ieee80211modeinfo - various details of a
 * particular HW mode
 */
typedef
struct __ieee80211_mode_info
{
    /**
     * mode - Hardware mode
     */
    hwmodes mode;

    /**
     * channels - All channels supported by this
     * HW mode
     */
    GList *channels;

    /**
     * rates - All supported rates of this mode
     */
    GList *rates;
} ieee80211modeinfo;

/**
 * struct nl80211interface - Represents a wireless interface
 * added by the wireless device attached to the system and the
 * type of the interface.
 */
typedef
struct __nl80211_interface
{
    /**
     * name - name of the wireless interface
     */
    char *name;

    /**
     * iftype - type of the wireless interface
     */
    enum nl80211_iftype iftype;

    /**
     * index - index of the wireless interface
     */
    unsigned int index;

    /**
     * macstr - MAC address of the wireless interface in
     * string format
     */
    char *macstr;

    /**
     * macstr - MAC address of the wireless interface
     * (raw bytes)
     */
    unsigned char mac [ETH_ALEN];

} nl80211interface;

/**
 * struct wiphyinfo - Represents a wireless card attached to
 * the system i.e., information about a wiphy (wireless device)
 * which includes the operational bands, modes, operating channels
 * and various rates supported
 */
typedef
struct __nl80211_phy_info
{
    /**
     * phyname - name of the wireless device
     */
    char *phyname;

    /**
     * alpha2 - Current regulatory settings
     * */
    char alpha2[GENL_ALPHA_SIZE];

    /**
     * ieee80211index - index of the wireless device
     * assigned by the kernel
     */
    unsigned int ieee80211index;

    /**
     * interfaces - List of interfaces added/exposed by
     * this wireless device to the userspace e.g : wlan0,
     * wlan1
     */
    GHashTable *interfaces;

    /**
     * modes - list of 802.11 modes supported by the
     * wireless device */
    GList *modes;

    /**
     * selfmanaged - whether this wiphy is self managed
     * when it comes to regulatory?
     */
    int selfmanaged;

    /**
     * initator - the initiator of the current regulatory
     * change for this device (if the radio is self managed)
     * or the global initiator if the device falls under
     * the regulatory core of kernel
     */
    enum nl80211_reg_initiator initator;

    /**
     * type - the type of the current regulatory
     * change for this device (if the radio is self managed)
     * or the global initiator if the device falls under
     * the regulatory core of kernel
     */
    enum nl80211_reg_type type;

    /**
     * dfs_region - DFS REGION
     */
    enum nl80211_dfs_regions dfs_region;

} wiphyinfo;

/**
 * struct genlclient - client to NETLINK_GENERIC netlink
 * protocol for monitoring the regulatory changes to update
 * the channel information
 */
typedef
struct __genlclient
{
    /**
     * sk - netlink socket for
     * fmaily : AF_NETLINK, type : SOCK_RAW
     * protocol : NETLINK_GENERIC
     */
    struct nl_sock *sk, *s_rw;

    /**
     * cb - Callback handler for the netlink
     * socket
     */
    struct nl_cb *cb;

    /**
     * iochannel - integrating this socket to the
     * main event loop of the application
     */
    GIOChannel *iochannel;

    /**
     * watch - event source id for adding the iochannel
     * to the default main loop context with default priority
     */
    unsigned int watch;

    /**
     * protofeatures - protocol features (e.g. whether the
     * devices support dumping)
     */
    unsigned int protofeatures;

    /**
     * nl80211id -  nl80211 bus id
     */
    int nl80211id;

    /**
     * alpha2 - global reg domain
     */
    char alpha2 [GENL_ALPHA_SIZE];

    /**
     * initator - the initiator of the current regulatory
     * change for the regulatory core of kernel
     */
    enum nl80211_reg_initiator initator;

    /**
     * type - the type of the current regulatory
     * change for the regulatory core of kernel
     */
    enum nl80211_reg_type type;

    /**
     * dfs_region - DFS REGION
     */
    enum nl80211_dfs_regions dfs_region;

} genlclient;

/**
 * struct resolver - nl80211 family resolver
 * nl80211 family and its identifier
 */
typedef
struct __nl80211_family_resolver
{
    /**
     * group - name of the nl80211 family
     */
    char *group;

    /**
     * result - resolved unique family identifier
     */
    int result;
} resolver;

/**
 * genl_cli - nl80211 client
 */
static genlclient *genl_cli;

/**
 * reg_clients - registered clients for the
 * regulstory change updates
 */
static GList *reg_clients;

/**
 * reg_clients - registered clients for the
 * scan related change updates
 */
static GList *scan_clients;

/**
 * wireless_devices - no. of wireless devices
 * attached to the system
 */
static GHashTable *wireless_devices;

/**
 * nlmsg request queue for serialization
 */
static GSList *genlrequests;

/**
 * Initial request for the complete nl80211
 * dump has been completed
 */
static gboolean nl80211dumped = FALSE;

/**
 * various NL80211 multicast groups (mcgrps)
 * - config
 * - scan
 * - regulatory
 * - mlme
 * - vendor
 *
 * genl_multicast_groups - WiFi_AP_Direct_Manager is
 * interested in
 *
 * - regulatory group
 *      to monitor if there is any regulatory update so that
 *      the channel list gets updated accordingly
 *
 * - scan group
 *      to monitor the scan requests to the wiphys because it
 *      affects the channel selection via ACS as the wiphy
 *      becomes busy during the scan period
 *
 * - Config group
 *      to monitor the wiphys and the network interfaces
 */
static const char*
genl_multicast_groups [] = {

    /**
     * regulatory mcgrp
     */
    "regulatory",

    /**
     * Scan mcgrp
    */
    "scan",

    /**
     * Configuration mcgrp
    */
    "config",

    NULL
};

/* forward decl */
static int genl_global_process_event (struct nl_msg *msg, void *arg);
static int genl_process_next (genlclient *client, unsigned int sequence);
static int genl_queue_requests (const genlclient *const client,
                                enum nl80211_commands cmd,
                                gboolean global, unsigned int wiphyid);
static int genl_get_nl80211_settings (genlclient *client,
                                      enum nl80211_commands cmd,
                                      gboolean global,
                                      unsigned int index);

static const char*
genl_error_to_string (const int err)
{
    switch (err) {
    case NLE_FAILURE:
        return ENUMTOSTR (NLE_FAILURE);
    case NLE_INTR:
        return ENUMTOSTR (NLE_INTR);
    case NLE_BAD_SOCK:
        return ENUMTOSTR (NLE_BAD_SOCK);
    case NLE_AGAIN:
        return ENUMTOSTR (NLE_AGAIN);
    case NLE_NOMEM:
        return ENUMTOSTR (NLE_NOMEM);
    case NLE_EXIST:
        return ENUMTOSTR (NLE_EXIST);
    case NLE_INVAL:
        return ENUMTOSTR (NLE_INVAL);
    case NLE_RANGE:
        return ENUMTOSTR (NLE_RANGE);
    case NLE_MSGSIZE:
        return ENUMTOSTR (NLE_MSGSIZE);
    case NLE_OPNOTSUPP:
        return ENUMTOSTR (NLE_OPNOTSUPP);
    case NLE_AF_NOSUPPORT:
        return ENUMTOSTR (NLE_AF_NOSUPPORT);
    case NLE_OBJ_NOTFOUND:
        return ENUMTOSTR (NLE_OBJ_NOTFOUND);
    case NLE_NOATTR:
        return ENUMTOSTR (NLE_NOATTR);
    case NLE_MISSING_ATTR:
        return ENUMTOSTR (NLE_MISSING_ATTR);
    case NLE_AF_MISMATCH:
        return ENUMTOSTR (NLE_AF_MISMATCH);
    case NLE_SEQ_MISMATCH:
        return ENUMTOSTR (NLE_SEQ_MISMATCH);
    case NLE_MSG_OVERFLOW:
        return ENUMTOSTR (NLE_MSG_OVERFLOW);
    case NLE_MSG_TRUNC:
        return ENUMTOSTR (NLE_MSG_TRUNC);
    case NLE_NOADDR:
        return ENUMTOSTR (NLE_NOADDR);
    case NLE_SRCRT_NOSUPPORT:
        return ENUMTOSTR (NLE_SRCRT_NOSUPPORT);
    case NLE_MSG_TOOSHORT:
        return ENUMTOSTR (NLE_MSG_TOOSHORT);
    case NLE_MSGTYPE_NOSUPPORT:
        return ENUMTOSTR (NLE_MSGTYPE_NOSUPPORT);
    case NLE_OBJ_MISMATCH:
        return ENUMTOSTR (NLE_OBJ_MISMATCH);
    case NLE_NOCACHE:
        return ENUMTOSTR (NLE_NOCACHE);
    case NLE_BUSY:
        return ENUMTOSTR (NLE_BUSY);
    case NLE_PROTO_MISMATCH:
        return ENUMTOSTR (NLE_PROTO_MISMATCH);
    case NLE_NOACCESS:
        return ENUMTOSTR (NLE_NOACCESS);
    case NLE_PERM:
        return ENUMTOSTR (NLE_PERM);
    case NLE_PKTLOC_FILE:
        return ENUMTOSTR (NLE_PKTLOC_FILE);
    case NLE_PARSE_ERR:
        return ENUMTOSTR (NLE_PARSE_ERR);
    case NLE_NODEV:
        return ENUMTOSTR (NLE_NODEV);
    case NLE_IMMUTABLE:
        return ENUMTOSTR (NLE_IMMUTABLE);
    case NLE_DUMP_INTR:
        return ENUMTOSTR (NLE_DUMP_INTR);
    }

    return ENUMTOSTR (UNKNOWN);
}

static int
genl_ack_handler (struct nl_msg *msg,
                  void *arg)
{
    int *err = arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg && arg, NL_STOP);

    *err = 0;
    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%u] acknowledged", nlhdr->nlmsg_seq);
    return NL_STOP;
}

static int
genl_finish_handler (struct nl_msg *msg,
                     void *arg)
{
    int *ret = arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg && arg, NL_SKIP);

    *ret = 0;
    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%u] finished successfully", nlhdr->nlmsg_seq);
    return NL_SKIP;
}

static int
genl_error_handler (struct sockaddr_nl *nla,
                    struct nlmsgerr *err,
                    void *arg)
{
    (void) nla;
    int *ret = arg;

    return_val_if_fail (err && arg, NL_STOP);

    DEBUG ("Error received for Nl message with sequence no [%u]: %s/%d "
           "system error: %s/%d", err->msg.nlmsg_seq, nl_geterror (err->error),
           err->error, strerror (-err->error), -err->error);
    *ret = err->error;
    return NL_STOP;
}

static int
genl_global_ack_handler (struct nl_msg *msg,
                         void *arg)
{
    (void) arg;

    int ret;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_STOP);

    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%u] acknowledged [%p]", nlhdr->nlmsg_seq, arg);
    ret = genl_process_next ((genlclient *) arg, nlhdr->nlmsg_seq);
    if (ret < 0)
        ERROR ("Failed to send the next rtnl request: %s/%d",
               strerror (-ret), -ret);
    return NL_STOP;
}

static const char*
genl_mode_to_string (hwmodes mode)
{
    switch (mode) {
    case NUM_IEEE80211_HW_MODES:
        return ENUMTOSTR (UNKNOWN);
    case IEEE80211_HW_MODE_B:
        return ENUMTOSTR (HW_MODE_B);
    case IEEE80211_HW_MODE_G:
        return ENUMTOSTR (HW_MODE_G);
    case IEEE80211_HW_MODE_A:
        return ENUMTOSTR (HW_MODE_A);
    case IEEE80211_HW_MODE_AD:
        return ENUMTOSTR (HW_MODE_AD);
    }

    return ENUMTOSTR (UNKNOWN);
}

static const char*
genl_dfstate_to_string (unsigned int *flags)
{
    return_val_if_fail (flags, ENUMTOSTR (UNKNOWN));

    if (*flags & NL80211_DFS_USABLE)
        return ENUMTOSTR (DFS_USABLE);
    else if (*flags & NL80211_DFS_AVAILABLE)
        return ENUMTOSTR (DFS_AVAILABLE);
    else if (*flags & NL80211_DFS_UNAVAILABLE)
        return ENUMTOSTR (DFS_UNAVAILABLE);

    return ENUMTOSTR (UNKNOWN);
}

static const char*
genl_nl80211_cmd2string (enum nl80211_commands cmd)
{
    switch (cmd) {
    case NL80211_CMD_REG_CHANGE:
        return ENUMTOSTR (NL80211_CMD_REG_CHANGE);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
    case NL80211_CMD_WIPHY_REG_CHANGE:
        return ENUMTOSTR (NL80211_CMD_WIPHY_REG_CHANGE);
#endif
    case NL80211_CMD_NEW_WIPHY:
        return ENUMTOSTR (NL80211_CMD_NEW_WIPHY);
    case NL80211_CMD_DEL_WIPHY:
        return ENUMTOSTR (NL80211_CMD_DEL_WIPHY);
    case NL80211_CMD_NEW_INTERFACE:
        return ENUMTOSTR (NL80211_CMD_NEW_INTERFACE);
    case NL80211_CMD_DEL_INTERFACE:
        return ENUMTOSTR (NL80211_CMD_DEL_INTERFACE);
    case NL80211_CMD_GET_WIPHY:
        return ENUMTOSTR (NL80211_CMD_GET_WIPHY);
    case NL80211_CMD_GET_INTERFACE:
        return ENUMTOSTR (NL80211_CMD_GET_INTERFACE);
    case NL80211_CMD_GET_REG:
    	return ENUMTOSTR (NL80211_CMD_GET_REG);
    case NL80211_CMD_TRIGGER_SCAN:
        return ENUMTOSTR (NL80211_CMD_TRIGGER_SCAN);
    case NL80211_CMD_NEW_SCAN_RESULTS:
        return ENUMTOSTR (NL80211_CMD_NEW_SCAN_RESULTS);
    case NL80211_CMD_SCAN_ABORTED:
        return ENUMTOSTR (NL80211_CMD_SCAN_ABORTED);
    case NL80211_CMD_START_SCHED_SCAN:
        return ENUMTOSTR (NL80211_CMD_START_SCHED_SCAN);
    case NL80211_CMD_SCHED_SCAN_STOPPED:
        return ENUMTOSTR (NL80211_CMD_SCHED_SCAN_STOPPED);
    case NL80211_CMD_SCHED_SCAN_RESULTS:
        return ENUMTOSTR (NL80211_CMD_SCHED_SCAN_RESULTS);

    default:
        break;
    }

    return ENUMTOSTR (NL80211_CMD_UNKNOWN);
}

static void
genl_dump_channelinfo (gpointer value,
                       gpointer data)
{
    (void) data;

    channelinfo *chan = value;

    return_if_fail (chan);

    INFO ("Frequency: %u MHz "
          "[channel: %03u] "
          "[flags: %02x] "
          "[usable: %s] "
          "[DFS State: %s] "
          "[RADAR: %s] "
          "[INDOOR-Only: %s] "
          "[Disabled: %s] "
          "[NO-IR: %s] "
          "[IR-CONCURRENT: %s]",
          chan->freq,
          chan->channel,
          chan->flags,
          chan->usable ? "Y" : "N",
          genl_dfstate_to_string (&chan->flags),
          !!(chan->flags & IEEE80211_CHANNEL_RADAR) ? "Y" : "N",
          !!(chan->flags & IEEE80211_CHANNEL_INDOOR) ? "Y" : "N",
          !!(chan->flags & IEEE80211_CHANNEL_DISABLED) ? "Y" : "N",
          !!(chan->flags & IEEE80211_CHANNEL_NO_IR) ? "Y" : "N",
          !!(chan->flags & IEEE80211_CHANNEL_IR_CONCURRENT) ? "Y" : "N");
}

static void
genl_dump_rateinfo (gpointer value,
                    gpointer data)
{
    (void) data;

    INFO ("Supported Rate: %u", GPOINTER_TO_UINT (value));
}

static void
genl_dump_modeinfo (gpointer value,
                    gpointer data)
{
    (void) data;

    ieee80211modeinfo *mode = value;

    return_if_fail (mode);

    INFO ("Supported IEEE802.11 HW Mode: %s", genl_mode_to_string (mode->mode));
    g_list_foreach (mode->channels, genl_dump_channelinfo, NULL);
    g_list_foreach (mode->rates, genl_dump_rateinfo, NULL);
}

static void
genl_dump_interfaceinfo (gpointer key,
                         gpointer value,
                         gpointer data)
{
    (void) key;
    (void) data;

    nl80211interface *interface = value;

    return_if_fail (interface);

    INFO ("Net If index: %u "
          "Net If Type: %d "
          "Network interface: \"%s\" "
          "MAC Address: \"%s\"",
          interface->index,
          interface->iftype,
          interface->name ? interface->name : "unknown",
          interface->macstr ? interface->macstr : "unknown");
}

static const char*
genl_reg_change_initiator_to_string (enum nl80211_reg_initiator initiator)
{
    switch (initiator) {
    case NL80211_REGDOM_SET_BY_CORE:
        return ENUMTOSTR (WIRELESS_CORE);
    case NL80211_REGDOM_SET_BY_USER:
        return ENUMTOSTR (SET_BY_USER);
    case NL80211_REGDOM_SET_BY_DRIVER:
        return ENUMTOSTR (SET_BY_DRIVER);
    case NL80211_REGDOM_SET_BY_COUNTRY_IE:
        return ENUMTOSTR (SET_FROM_COUNTRY_IE);
    }

    return ENUMTOSTR (UNKNOWN);
}

static const char*
genl_reg_change_dfsdomain_to_string (enum nl80211_dfs_regions domreg)
{
    switch (domreg) {
    case NL80211_DFS_UNSET:
        return ENUMTOSTR (NL80211_DFS_UNSET);
    case NL80211_DFS_FCC:
        return ENUMTOSTR (NL80211_DFS_FCC);
    case NL80211_DFS_ETSI:
        return ENUMTOSTR (NL80211_DFS_ETSI);
    case NL80211_DFS_JP:
        return ENUMTOSTR (NL80211_DFS_JP);
    }

    return ENUMTOSTR (NL80211_DFS_UNSET);
}

static const char*
genl_reg_change_type_to_string (enum nl80211_reg_type type)
{
    switch (type) {
    case NL80211_REGDOM_TYPE_COUNTRY:
        return ENUMTOSTR (COUNTRY);
    case NL80211_REGDOM_TYPE_WORLD:
        return ENUMTOSTR (WORLD);
    case NL80211_REGDOM_TYPE_CUSTOM_WORLD:
        return ENUMTOSTR (CUSTOM_WORLD);
    case NL80211_REGDOM_TYPE_INTERSECTION:
        return ENUMTOSTR (INTERSECTION);
    }

    return ENUMTOSTR (UNKNOWN);
}

static void
genl_dump_wiphyinfo (gpointer key,
                     gpointer value,
                     gpointer data)
{
    (void) key;
    (void) data;

    wiphyinfo *info = value;

    return_if_fail (info);

    DEBUG ("********************************************************"
           "*********************");
    INFO ("Wireless device: \"%s\" "
          "Self Managed Device: %s "
          "[Reg Settings: %s intiator: %s type: %s]",
          info->phyname ? info->phyname : "unknown",
          info->selfmanaged ? "Y" : "N",
          info->alpha2,
          genl_reg_change_initiator_to_string (info->initator),
          genl_reg_change_type_to_string (info->type));

    g_hash_table_foreach (info->interfaces, genl_dump_interfaceinfo, NULL);
    g_list_foreach (info->modes, genl_dump_modeinfo, NULL);
    DEBUG ("********************************************************"
           "*********************");
}

static int
genl_get_supported_channels_impl (wiphyinfo *dev,
                                  int freq,
                                  unsigned int **frequencies,
                                  unsigned int **channels,
                                  unsigned int *length)
{
    GList *modes, *temp;
    ieee80211modeinfo *info;
    channelinfo *chaninfo;
    GSList *local = NULL, *slist;
    unsigned int len = 0;
    unsigned int data, *value, index = 0;

    return_val_if_fail (dev, -EINVAL);

    modes = dev->modes;
    for ( ; modes; modes = modes->next) {
        info = modes->data;
        temp = (info != NULL) ? info->channels: NULL;
        for ( ; temp; temp = temp->next) {
            chaninfo = temp->data;
            if (chaninfo && chaninfo->usable) {
                (freq == 0) ? (data = chaninfo->channel) :
                              (data = chaninfo->freq);
                if (!g_slist_find (local, GUINT_TO_POINTER (data)))
                    local = g_slist_prepend (local, GUINT_TO_POINTER (data));
            }
        }
    }

    local = g_slist_reverse (local);
    len = g_slist_length (local);

    INFO ("Length of the unique %s: %u", freq ? "frequencies" : "Channels",
          len);

    if (!len) {
        *length = len;
        return -ENODATA;
    }

    value = g_try_malloc0 (len * sizeof (unsigned int));
    return_val_if_fail (value, -ENOMEM);

    slist = local;
    for ( ; slist; slist = slist->next, index++)
        value [index] = GPOINTER_TO_UINT (slist->data);

    freq == 0 ? (*channels = value) : (*frequencies = value);
    *length = len;

    return 0;
}

static void
genl_update_regulatory_change (gboolean global,
                               unsigned int index)
{
    int ret, updated = 0;
    GList *temp;
    wiphyinfo *device;
    gpointer key1, value1, key2, value2;
    GHashTableIter iter1, iter2;
    nl80211interface *interface;
    unsigned int *channels = NULL,
            *frequencies = NULL, length = 0;
    regulatory_ops *regops;

    DEBUG ("Informing Regulatory Change to the clients involved, "
           "global: %s [index: %u]", global ? "Y" : "N", index);

    return_if_fail (wireless_devices);

    temp = reg_clients;
    for ( ; temp; temp = temp->next) {

        regops = temp->data;
        continue_if_fail (regops && regops->reg_changed);

        g_hash_table_iter_init (&iter1, wireless_devices);
        while (g_hash_table_iter_next (&iter1, &key1, &value1)) {

            updated = 0;
            device = value1;
            continue_if_fail (device && device->interfaces);

            if (!global && device->ieee80211index != index)
                continue;

            channels = frequencies = NULL;
            ret = genl_get_supported_channels_impl (device, 0, NULL, &channels, &length);
            if (ret < 0 && ret != -ENODATA)
                continue;

            ret = genl_get_supported_channels_impl (device, 1, &frequencies, NULL, &length);
            if (ret < 0 && ret != -ENODATA) {
                g_free (channels);
                continue;
            }

            g_hash_table_iter_init (&iter2, device->interfaces);
            while (g_hash_table_iter_next (&iter2, &key2, &value2)) {

                interface = value2;
                continue_if_fail (interface);

                if (regops->ifname) {
                    if (interface->name && !g_strcmp0 (regops->ifname,interface->name)) {
                        regops->reg_changed (interface->name, frequencies, channels, length);
                        updated = 1;
                        break;
                    }
                } else {
                    regops->reg_changed (interface->name, frequencies, channels, length);
                }
            }

            g_free (channels);
            g_free (frequencies);

            if (updated)
                break;
        }
    }
}

static void
genl_update_scan_state_changes (wiphyinfo *device,
                                nl80211_scan_state state)
{
    int updated = 0;
    GList *temp;
    gpointer key, value;
    GHashTableIter iter;
    nl80211interface *interface;
    scanstate_ops *sops;

    return_if_fail (device && device->interfaces);

    temp = scan_clients;
    for ( ; temp; temp = temp->next) {

        sops = temp->data;
        continue_if_fail (sops && sops->scan_state_changed);

        g_hash_table_iter_init (&iter, device->interfaces);
        while (g_hash_table_iter_next (&iter, &key, &value)) {

            updated = 0;
            interface = value;
            continue_if_fail (interface && interface->name);

            /* A scan change in the device means that every
             * interface of that particular device are affected,
             * so we notify for the every interface */

            if (sops->ifname) {
                if (interface->name &&
                        !g_strcmp0 (sops->ifname, interface->name)) {
                    sops->scan_state_changed (device->phyname, interface->name,
                                              state);
                    updated = 1;
                }
            } else
                sops->scan_state_changed (device->phyname, interface->name,
                                          state);

            if (updated)
                break;
        }
    }
}

static void
genl_check_orphans ()
{
    GHashTableIter iter1, iter2;
    wiphyinfo *device;
    gpointer key1, value1, key2,
            value2;
    nl80211interface *iface;

    return_if_fail (wireless_devices);

    g_hash_table_iter_init (&iter1, wireless_devices);
    while (g_hash_table_iter_next (&iter1, &key1, &value1)) {

        device = value1;
        continue_if_fail (device && device->interfaces);

        g_hash_table_iter_init (&iter2, device->interfaces);
        while (g_hash_table_iter_next (&iter2, &key2, &value2)) {
            iface = value2;
            continue_if_fail (iface);
            (void) wireless_tech_handle_orphans (iface->name);
        }
    }
}

static int
genl_process_next (genlclient *client,
                   unsigned int sequence)
{
    int ret;
    GSList *list;
    int found = 0;
    struct nl_msg *req;
    struct nlmsghdr *nlhdr;
    struct genlmsghdr *gnlh;

    DEBUG ("Netlink message sequence no: %u", sequence);

    list = genlrequests;
    for ( ; list; list = list->next) {
        req = list->data;
        if (req) {
            nlhdr = nlmsg_hdr (req);
            if (nlhdr->nlmsg_seq ==  sequence) {
                found = 1;
                break;
            }
        }
    }

    if (found) {
        genlrequests = g_slist_remove (genlrequests, req);
        gnlh = nlmsg_data (nlmsg_hdr (req));
        DEBUG ("Removing %s from the request queue",
               genl_nl80211_cmd2string (gnlh->cmd));
        nlmsg_free (req);
    }

    /* Request the next message */
    req = g_slist_nth_data (genlrequests, 0);
    if (!req) {
    	if (FALSE == nl80211dumped) {
    		nl80211dumped = TRUE;
    		g_hash_table_foreach (wireless_devices, genl_dump_wiphyinfo, NULL);
    		genl_update_regulatory_change (TRUE, 0);
    		genl_check_orphans ();
    	}
        return 0;
    }

    return_val_if_fail (client, -EINVAL);

    nlhdr = nlmsg_hdr (req);
    gnlh = nlmsg_data (nlmsg_hdr (req));

    DEBUG ("Requesting [%s] dump from the kernel [seq: %u]",
           genl_nl80211_cmd2string (gnlh->cmd),
           nlhdr->nlmsg_seq);
    ret = nl_send_auto (client->sk, req);
    if (ret < 0)
        ERROR ("Failed to post the nl msg to get %s dump, error: %s/%d",
               genl_nl80211_cmd2string(gnlh->cmd),
               nl_geterror (-ret), -ret);

    return ret;
}

static int
genl_global_finish_handler (struct nl_msg *msg,
                            void *arg)
{
    int ret;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_SKIP);

    nlhdr = nlmsg_hdr (msg);

    DEBUG ("Message with seq no [%u] finished", nlhdr->nlmsg_seq);
    ret = genl_process_next ((genlclient *) arg, nlhdr->nlmsg_seq);
    if (ret < 0)
        ERROR ("Failed to send then next rtnl request: %s/%d",
               strerror (-ret), -ret);

    return NL_STOP;
}

static int
genl_global_error_handler (struct sockaddr_nl *nla,
                           struct nlmsgerr *err,
                           void *arg)
{
    (void) nla;
    (void) arg;

    return_val_if_fail (err, NL_STOP);

    DEBUG ("Error received for Nl message with sequence no [%u]: %s/%d "
           "system error : %s/%d", err->msg.nlmsg_seq, nl_geterror (err->error),
           err->error, strerror (-err->error), -err->error);
    return NL_STOP;
}

static int
genl_seq_check (struct nl_msg *msg,
                void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_OK);

    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Sequence check for Message with seq no [%u]", nlhdr->nlmsg_seq);
    return NL_OK;
}

static int
genl_family_resolver (struct nl_msg *msg,
                      void *data)
{
    resolver *fmres = data;
    struct nlattr *tb[CTRL_ATTR_MAX + 1];
    struct genlmsghdr *gnlh;
    struct nlattr *mcgrp;
    int index;

    return_val_if_fail (msg && fmres, NL_SKIP);

    gnlh = nlmsg_data (nlmsg_hdr (msg));

    nla_parse (tb, CTRL_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
               genlmsg_attrlen (gnlh, 0), NULL);

    /*
     * msg does not have the multicast grp info,
     * skip the message
     */
    return_val_if_fail (tb [CTRL_ATTR_MCAST_GROUPS], NL_SKIP);

    nla_for_each_nested (mcgrp, tb [CTRL_ATTR_MCAST_GROUPS], index) {

        struct nlattr *tb2 [CTRL_ATTR_MCAST_GRP_MAX + 1];

        continue_if_fail (mcgrp);
        nla_parse (tb2, CTRL_ATTR_MCAST_GRP_MAX, nla_data(mcgrp),
                   nla_len (mcgrp), NULL);

        /* no grp name */
        continue_if_fail (tb2 [CTRL_ATTR_MCAST_GRP_NAME]);
        /* no grp id */
        continue_if_fail (tb2 [CTRL_ATTR_MCAST_GRP_ID]);

        if (strncmp (nla_data (tb2 [CTRL_ATTR_MCAST_GRP_NAME]), fmres->group,
                     (size_t) nla_len (tb2 [CTRL_ATTR_MCAST_GRP_NAME])) != 0)
            continue;

        fmres->result = (int) nla_get_u32 (tb2 [CTRL_ATTR_MCAST_GRP_ID]);
        INFO ("Multicast group \"%s\" resolved id: %d", fmres->group,
              fmres->result);
        break;
    };

    return NL_SKIP;
}

static int
genl_send_and_recv (struct nl_sock *sk,
                    struct nl_cb *nlcb,
                    struct nl_msg *msg,
                    int (*handler) (struct nl_msg *, void *),
                    void *data)
{
    struct nl_cb *cb;
    volatile int err = -ENOMEM;
	int res;

    return_val_if_fail (sk && msg && nlcb, -EINVAL);

    cb = nl_cb_clone (nlcb);
    if (!cb)
        goto out;

    err = nl_send_auto (sk, msg);
    if (err < 0)
        goto out;

    nl_cb_err (cb, NL_CB_CUSTOM, (nl_recvmsg_err_cb_t) genl_error_handler, (void*)&err);
    nl_cb_set (cb, NL_CB_FINISH, NL_CB_CUSTOM, genl_finish_handler, (void*)&err);
    nl_cb_set (cb, NL_CB_ACK, NL_CB_CUSTOM, genl_ack_handler, (void*)&err);

    if (handler)
        nl_cb_set (cb, NL_CB_VALID, NL_CB_CUSTOM, handler, data);

    while (err > 0) {
        res = nl_recvmsgs (sk, cb);
        if (res < 0)
            ERROR ("Failed to receive nl80211 msgs : %s [%d]",
                   genl_error_to_string (-res), -res);
    }

 out:
    nl_cb_put(cb);
    return err;
}

static int
genl_get_multicast_id (const genlclient *cli,
                       const char *family,
                       const char *group)
{
    int identifier, res = -1;
    struct nl_msg *msg;
    resolver *fmres;

    return_val_if_fail (cli && family && group, -EINVAL);

    DEBUG ("Family: %s Group: %s", family, group);

    fmres = g_try_malloc0 (sizeof (*fmres));
    return_val_if_fail (fmres, -ENOMEM);

    fmres->group = g_strdup (group);
    /*
     * initialize with no entry and the value
     * will be filled if the mcgrp is there
     */
    fmres->result = -ENOENT;

    msg = nlmsg_alloc ();
    if (!msg) {
        res = -ENOMEM;
        goto failure;
    }

    /* get the genl family ("nlctrl") identifier */
    if ((identifier = genl_ctrl_resolve (cli->sk, "nlctrl")) < 0) {
        ERROR ("Failed to resolve the family name %s to numeric identifier: %s/%d",
               family, nl_geterror (identifier), -identifier);
        nlmsg_free (msg);
        res = identifier;
        goto failure;
    }

    INFO ("Genl family identifier: %d", identifier);

    if (!genlmsg_put (msg, 0, 0, identifier, 0, 0, CTRL_CMD_GETFAMILY, 0) ||
            nla_put_string (msg, CTRL_ATTR_FAMILY_NAME, family)) {
        nlmsg_free(msg);
        res = -ENOMSG;
        goto failure;
    }

    if (!genl_send_and_recv (cli->sk, cli->cb, msg, genl_family_resolver, fmres))
        res = fmres->result;

    nlmsg_free (msg);
failure:
    g_free (fmres->group);
    g_free (fmres);
    return res;
}

static int
genl_add_membership (const genlclient *cli,
                     const char *groups [] )
{
    int ret = 0, i = 0;

    return_val_if_fail (cli && groups, -EINVAL);

    for ( ; groups[i]; i++) {

        DEBUG ("Trying to add membership to the family: %s",
              groups[i]);

        ret = genl_get_multicast_id (cli, "nl80211", groups[i]);
        if (ret < 0) {
            ERROR ("Failed to get the multicast id of family: %s",
                   groups[i]);
            continue;
        }

        ret = nl_socket_add_membership (cli->sk, ret);
        if (ret < 0) {
            ERROR ("Failed to get the membership of family: %s "
                   "[error: %s/%d]", groups[i], nl_geterror (ret), -ret);
        }
    }

    return ret;
}

static int
genl_nl80211_feature_hanlder (struct nl_msg *msg,
                              void *data)
{
    unsigned int *features = data;
    struct nlattr *tb[NL80211_ATTR_MAX + 1];
    struct genlmsghdr *gnlh;

    return_val_if_fail (msg && features, -EINVAL);

    gnlh = nlmsg_data(nlmsg_hdr(msg));
    nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
               genlmsg_attrlen (gnlh, 0), NULL);

    if (tb [NL80211_ATTR_PROTOCOL_FEATURES])
        *features = nla_get_u32 (tb [NL80211_ATTR_PROTOCOL_FEATURES]);

    INFO ("nl80211 supported protocol features: %u", *features);
    return NL_SKIP;
}

static int
genl_get_nl80211_protocol_features (const genlclient *const cli,
                                    unsigned int *features)
{
    int err;
    struct nl_msg *msg;

    return_val_if_fail (features && cli, -EINVAL);

    msg = nlmsg_alloc ();
    return_val_if_fail (msg, -ENOMEM);

    if (!genlmsg_put (msg, 0, 0, cli->nl80211id, 0, 0,
                      NL80211_CMD_GET_PROTOCOL_FEATURES, 0)) {
        nlmsg_free(msg);
        return -ENOENT;
    }

    err = genl_send_and_recv (cli->sk, cli->cb, msg, genl_nl80211_feature_hanlder, features);
    nlmsg_free (msg);
    return err;
}

static void
genl_channelinfo_cleanup (gpointer data)
{
    g_free (data);
}

static void
genl_ifaces_cleanup (gpointer data)
{
    nl80211interface *interface = data;

    return_if_fail (interface);

    g_free (interface->name);
    g_free (interface->macstr);
    g_free (interface);
}

static void
genl_ieee80211mode_cleanup (gpointer data)
{
    ieee80211modeinfo *info = data;

    return_if_fail (info);

    INFO ("Freeing the hw mode: %s", genl_mode_to_string (info->mode));
    g_list_free_full (info->channels, genl_channelinfo_cleanup);
    g_list_free (info->rates);
    g_free (info);
}

static void
genl_phyinfo_cleanup (gpointer data)
{
    wiphyinfo *phy = data;

    return_if_fail (phy);

    INFO ("Freeing up the wiphy device : %s [%p]",
          phy->phyname ? phy->phyname : "unknown", phy);

    g_free (phy->phyname);
    g_list_free_full (phy->modes, genl_ieee80211mode_cleanup);
    g_hash_table_destroy (phy->interfaces);
    g_free (phy);
}

static int
genl_clone_bmode (ieee80211modeinfo *bmode,
                  ieee80211modeinfo *gmode)
{
    channelinfo *channel;
    GList *channels, *rates;
    channelinfo *chan;

    return_val_if_fail (bmode && gmode, -EINVAL);

    rates = gmode->rates;
    channels = gmode->channels;

    /*
     * IEEE802.11 B modes supports the complete list of channels
     * supported by the IEEE802.11 G mode
     */
    for ( ; channels; channels = channels->next) {

        chan = channels->data;
        continue_if_fail (chan);

        channel = g_try_malloc0 (sizeof (*channel));
        continue_if_fail (channel);

        channel->channel = chan->channel;
        channel->flags = chan->flags;
        channel->freq = chan->freq;
        channel->usable = chan->usable;

        bmode->channels = g_list_append (bmode->channels, channel);
    }

    /*
     * IEEE802.11 B mode only supports
     * 1 Mbps, 2 Mbps, 5.5 Mbps, 11 Mbps
     */
    for ( ; rates; rates = rates->next) {

        unsigned int rate = GPOINTER_TO_UINT (rates->data);
        if (rate != 10 && rate != 20 && rate != 55
                && rate != 110)
            continue;

        bmode->rates = g_list_append (bmode->rates, GUINT_TO_POINTER (rate));
    }

    INFO ("B Mode cloned successfully from G Mode");
    return 0;
}

static int
genl_channels_to_mode (wiphyinfo *phy)
{
    int ret = 0;
    GList *modes, *rates;
    channelinfo *first;
    int bmodesupported = 0;
    ieee80211modeinfo *bmode, *gmode;
    unsigned int rate;

    return_val_if_fail (phy, -EINVAL);

    /*
     * no modes supported by the wireless card
     * attached to the system
     */
    if (!g_list_length (phy->modes))
        return -EOPNOTSUPP;

    modes = phy->modes;
    for ( ; modes ; modes = modes->next) {

        ieee80211modeinfo *mode = modes->data;
        continue_if_fail (mode);

        /* no channels supported by this mode */
        continue_if_fail (g_list_length (mode->channels) > 0);

        /*
         * Here is the heuristic for modes from band information
         * (2.4 GHz or 5 GHz) i.e., IEEE80211 B and G mode operates
         * in the 2.4 GHz band IEEE80211 A and IEEE80211 AD modes
         * operates in 5 GHz and 60GHz respectively
         *
         * Kernel gives information on the supported bands, list of
         * channels and various supported rates in each band. We derive
         * different modes from these information
         */
        first = (channelinfo *) (g_list_first (mode->channels)->data);

        /* IEEE80211 B or G mode */
        if (first->freq < 4000) {
            mode->mode = IEEE80211_HW_MODE_B;

            /*
             * Starts with IEEE80211 B mode and if on of the supported
             * rates are more than 11 Mbps where IEEE80211 B does not have
             * the support, we make it as IEEE 80211 G mode and clone IEEE80211 B
             * from G mode with the necessary details
             */
            rates = mode->rates;
            for ( ; rates; rates = rates->next) {

                rate = GPOINTER_TO_UINT (rates->data);
                if (rate > 200) {
                    mode->mode = IEEE80211_HW_MODE_G;
                    gmode = mode;
                    bmodesupported = 1;
                    break;
                }
            }
        }
        else if (first->freq > 50000) {
            mode->mode = IEEE80211_HW_MODE_AD;
        } else {
            mode->mode = IEEE80211_HW_MODE_A;
        }
    }

    if (bmodesupported) {

        INFO ("Cloning B Mode from G Mode");

        bmode = g_try_malloc0 (sizeof (*bmode));
        return_val_if_fail (bmode, -ENOMEM);
        bmode->mode = IEEE80211_HW_MODE_B;
        ret = genl_clone_bmode (bmode, gmode);

        if (!ret) {
            phy->modes = g_list_prepend (phy->modes, bmode);
        } else
            g_free (bmode);
    }

    return ret;
}

static int
genl_freq_to_channel (unsigned int freq,
                      unsigned int *channel)
{
    int ret = -EINVAL;

    return_val_if_fail (channel, ret);

    /* IEEE802.11 B ONLY */
    if ( freq == 2484) {

        *channel = 14;
        ret = 0;
    }
    /* IEEE802.11 G & B MODE */
    else if (freq >= 2412 && freq <= 2472) {

        if ((freq - 2407) % 5)
            return -EINVAL;

        *channel = (freq - 2407) / 5;
        ret = 0;
    }
    /* 4.9 GHz (IEEE 802.11 J) */
    else if (freq >= 4900 && freq < 5000) {

        if ((freq - 4000) % 5)
            return -EINVAL;

        *channel = (freq - 4000) / 5;
        ret = 0;
    }
    /* IEEE802.11 A MODE */
    else if (freq >= 5000 && freq <= 5900) {

        if ((freq - 5000) % 5)
            return -EINVAL;

        *channel = (freq - 5000) / 5;
        ret = 0;
    }
    /* IEEE802.11 AD MODE */
    else if (freq >= 56160 + 2160 * 1 && freq <= 56160 + 2160 * 4) {

        if ((freq - 56160) % 2160)
            return -EINVAL;

        *channel = (freq - 56160) / 2160;
        ret = 0;
    }

    return ret;
}

static channelinfo*
genl_wiphy_supported_channels (struct nlattr **tb,
                               enum nl80211_dfs_regions dfsreg)
{
#ifdef DISABLE_RADAR_CHANNELS
    (void) dfsreg;
#endif
    unsigned int freq;
    unsigned int channel;
    channelinfo *info;

    return_val_if_fail (tb, NULL);

    freq = nla_get_u32 (tb [NL80211_FREQUENCY_ATTR_FREQ]);

    if (genl_freq_to_channel (freq, &channel) < 0)
        return NULL;

    info = g_try_malloc0 (sizeof (*info));
    return_val_if_fail (info, NULL);

    info->freq = freq;
    info->channel = channel;
    info->usable = 1;

    if (tb [NL80211_FREQUENCY_ATTR_DISABLED])
        info->flags |= IEEE80211_CHANNEL_DISABLED;
    if (tb [NL80211_FREQUENCY_ATTR_NO_IR])
        info->flags |= IEEE80211_CHANNEL_NO_IR;
    if (tb [NL80211_FREQUENCY_ATTR_RADAR])
        info->flags |= IEEE80211_CHANNEL_RADAR;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
    if (tb [NL80211_FREQUENCY_ATTR_INDOOR_ONLY])
        info->flags |= IEEE80211_CHANNEL_INDOOR;
    if (tb [NL80211_FREQUENCY_ATTR_GO_CONCURRENT])
        info->flags |= IEEE80211_CHANNEL_IR_CONCURRENT;
#endif

    if (tb [NL80211_FREQUENCY_ATTR_DFS_STATE]) {
        enum nl80211_dfs_state state =
            nla_get_u32 (tb [NL80211_FREQUENCY_ATTR_DFS_STATE]);

        switch (state) {
        case NL80211_DFS_USABLE:
            info->flags |= IEEE80211_CHANNEL_DFS_USABLE;
            break;
        case NL80211_DFS_AVAILABLE:
            info->flags |= IEEE80211_CHANNEL_DFS_AVAILABLE;
            break;
        case NL80211_DFS_UNAVAILABLE:
            info->flags |= IEEE80211_CHANNEL_DFS_UNAVAILABLE;
            break;
        }
    }

    /**
     * If a channel requires RADAR detection but there is no DFS
     * master region set, kernel does not perform a selection
     * for that channel as it is unsure about its details.
     *
     * Genl will only pass the usable channels information
     * to its registered clients
     */
    if (!!(info->flags & IEEE80211_CHANNEL_DISABLED) ||
            !!(info->flags & IEEE80211_CHANNEL_NO_IR) ||
            !!(info->flags & IEEE80211_CHANNEL_INDOOR) ||
            !!(info->flags & IEEE80211_CHANNEL_IR_CONCURRENT) ||
            !!(info->flags & IEEE80211_CHANNEL_DFS_UNAVAILABLE)
            ||
#ifdef DISABLE_RADAR_CHANNELS
            !!(info->flags & IEEE80211_CHANNEL_RADAR)
#else

            (!!(info->flags & IEEE80211_CHANNEL_RADAR) &&
             NL80211_DFS_UNSET == dfsreg)
#endif
            )
        info->usable = 0;

    return info;
}

static int
genl_wiphy_supported_rates (ieee80211modeinfo *mode,
                            struct nlattr *rates)
{
    int i;
    unsigned int r;
    struct nlattr *tb [NL80211_BAND_ATTR_MAX + 1];
    struct nlattr *rate;

    return_val_if_fail (mode && rates, -EINVAL);
    nla_for_each_nested (rate, rates, i) {

        nla_parse (tb, NL80211_BITRATE_ATTR_MAX, nla_data (rate),
                   nla_len (rate), NULL);
        continue_if_fail (tb [NL80211_BITRATE_ATTR_RATE]);
        r = nla_get_u32 (tb [NL80211_BITRATE_ATTR_RATE]);
        mode->rates = g_list_append (mode->rates, GUINT_TO_POINTER (r));
    }

    return 0;
}

static ieee80211modeinfo*
genl_wiphy_band_capabilities (struct nlattr *band,
                              enum nl80211_dfs_regions dfsreg)
{
    int i;
    struct nlattr *tb [NL80211_BAND_ATTR_MAX + 1];
    struct nlattr *tbfreq;
    ieee80211modeinfo *mode;
    channelinfo *info;

    return_val_if_fail (band, NULL);

    mode = g_try_malloc0 (sizeof (*mode));
    return_val_if_fail (mode, NULL);

    mode->mode = NUM_IEEE80211_HW_MODES;
    nla_parse (tb, NL80211_BAND_ATTR_MAX, nla_data (band),
               nla_len (band), NULL);

    (void) genl_wiphy_supported_rates (mode, tb [NL80211_BAND_ATTR_RATES]);

    nla_for_each_nested (tbfreq, tb [NL80211_BAND_ATTR_FREQS], i) {

        struct nlattr *tb2 [NL80211_BAND_ATTR_MAX + 1];

        nla_parse (tb2, NL80211_FREQUENCY_ATTR_MAX,
                   nla_data (tbfreq), nla_len (tbfreq), NULL);

        if (tb2 [NL80211_FREQUENCY_ATTR_FREQ]) {
            info = genl_wiphy_supported_channels (tb2, dfsreg);
            if (info)
                mode->channels = g_list_append (mode->channels, info);
        }
    }

    return mode;
}

static wiphyinfo*
genl_get_wiphy_dev (void *wiphys,
                    const char *name,
                    unsigned int index)
{
    wiphyinfo *device;
    GHashTable *devices = wiphys;

    return_val_if_fail (devices, NULL);

    device = g_hash_table_lookup (devices, GUINT_TO_POINTER (index));
    if (device) {

        /* Renaming possible on the fly?, nevertheless */
        if (name && device->phyname && g_strcmp0 (device->phyname, name)) {
            g_free (device->phyname);
            device->phyname = g_strdup (name);
        }
        else if (name && !device->phyname)
            device->phyname = g_strdup (name);

        return device;
    }

    device = g_try_malloc0 (sizeof (*device));
    return_val_if_fail (device, NULL);

    device->ieee80211index = index;
    device->phyname = g_strdup (name);

    device->interfaces = g_hash_table_new_full (g_direct_hash,
                                                g_direct_equal,
                                                NULL, genl_ifaces_cleanup);

    DEBUG ("New Wiphy, name: %s index: %u [%p]", name, index, device);
    g_hash_table_replace (devices, GUINT_TO_POINTER (index), device);
    return device;
}

static int
genl_wiphy_capabilities_handler (genlclient *client,
                                 struct nl_msg *msg,
                                 void *data,
                                 gboolean removal)
{
    GHashTable *devices = data;
    wiphyinfo *device;
    const char *name;
    struct nlattr *tb [NL80211_ATTR_MAX + 1];
    struct genlmsghdr *gnlh;
    struct nlattr *band;
    int i, ret; unsigned int index;
    ieee80211modeinfo *mode;

    return_val_if_fail (msg && devices, NL_SKIP);

    gnlh = nlmsg_data (nlmsg_hdr (msg));

    nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
               genlmsg_attrlen (gnlh, 0), NULL);

    return_val_if_fail (tb [NL80211_ATTR_WIPHY] &&
                        tb [NL80211_ATTR_WIPHY_NAME], NL_SKIP);

    index = nla_get_u32 (tb [NL80211_ATTR_WIPHY]);
    name = nla_get_string (tb [NL80211_ATTR_WIPHY_NAME]);

    if (removal) {
        if (!g_hash_table_remove (devices, GUINT_TO_POINTER (index)))
            ERROR ("Failed to remove the wiphy \"%s\" with index: %u",
                   name, index);
        return NL_SKIP;
    }

    device = genl_get_wiphy_dev (devices, name, index);
    return_val_if_fail (device, NL_SKIP);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
    if (tb [NL80211_ATTR_WIPHY_SELF_MANAGED_REG])
        device->selfmanaged = nla_get_flag(tb [NL80211_ATTR_WIPHY_SELF_MANAGED_REG]);
#endif

    if (!tb [NL80211_ATTR_WIPHY_BANDS])
        return NL_SKIP;

    if (g_list_length (device->modes) > 0) {
        g_list_free_full (device->modes, genl_ieee80211mode_cleanup);
        device->modes = NULL;
    }

    nla_for_each_nested (band, tb [NL80211_ATTR_WIPHY_BANDS], i) {
        mode = genl_wiphy_band_capabilities (band, device->dfs_region);
        continue_if_fail (mode);
        device->modes = g_list_append (device->modes, mode);
    }

    ret = genl_channels_to_mode (device);
    if (ret < 0)
        g_hash_table_remove (devices, GUINT_TO_POINTER (index));
    else {
        if (device->selfmanaged && !nl80211dumped)
            genl_queue_requests (client, NL80211_CMD_GET_REG, FALSE, device->ieee80211index);

        if (nl80211dumped) {
            g_hash_table_foreach (wireless_devices, genl_dump_wiphyinfo, NULL);
            genl_update_regulatory_change (FALSE, index);
        }
    }

    return NL_SKIP;
}

static nl80211interface*
genl_get_nl80211_interface (wiphyinfo *device,
                            const char *name,
                            const enum nl80211_iftype type,
                            u_int8_t *mac,
                            unsigned int index)
{
    nl80211interface *wirelessif;

    return_val_if_fail (device && name && mac, NULL);

    wirelessif = g_hash_table_lookup (device->interfaces,
                                      GUINT_TO_POINTER (index));
    if (!wirelessif) {
        wirelessif = g_try_malloc0 (sizeof (*wirelessif));
        return_val_if_fail (wirelessif, NULL);

        INFO ("New WLAN interface detected: %s "
              "mac: %02x:%02x:%02x:%02x:%02x:%02x "
              "interface index: %u", name,
              MACADDRESS (mac), index);

        g_hash_table_replace (device->interfaces, GUINT_TO_POINTER (index),
                              wirelessif);
    }

    /* Possible that the names and mac addresses
     * gets renamed, but index is the key */
    g_free (wirelessif->name);
    g_free (wirelessif->macstr);

    wirelessif->iftype = type;
    wirelessif->name = g_strdup (name);
    wirelessif->index = index;
    memcpy (wirelessif->mac, mac, sizeof (wirelessif->mac));
    wirelessif->macstr = g_strdup_printf ("%02x:%02x:%02x:%02x:%02x:%02x",
                                          MACADDRESS (wirelessif->mac));

    return wirelessif;
}

static int
genl_wiphy_interfaces_handler (struct nl_msg *msg,
                               void *data,
                               gboolean removal)
{
    GHashTable *devices = data;
    char *name;
    struct nlattr *tb [NL80211_ATTR_MAX + 1];
    struct genlmsghdr *gnlh;
    unsigned int wiphyiidx, index;
    nl80211interface *wirelessif;
    enum nl80211_iftype type;
    u_int8_t mac [ETH_ALEN];
    wiphyinfo *device;
    gboolean success = FALSE;

    return_val_if_fail (msg && devices, NL_SKIP);

    gnlh = nlmsg_data (nlmsg_hdr (msg));

    nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
               genlmsg_attrlen (gnlh, 0), NULL);

    /* mandatory guys */
    if (!tb [NL80211_ATTR_IFNAME] || !tb [NL80211_ATTR_WIPHY] ||
            !tb [NL80211_ATTR_IFTYPE] || !tb [NL80211_ATTR_IFINDEX] ||
            !tb [NL80211_ATTR_MAC])
        return NL_SKIP;

    memset (mac, 0, sizeof (mac));

    wiphyiidx = nla_get_u32 (tb [NL80211_ATTR_WIPHY]);
    type = nla_get_u32 (tb [NL80211_ATTR_IFTYPE]);
    name = nla_get_string (tb [NL80211_ATTR_IFNAME]);
    index = nla_get_u32 (tb [NL80211_ATTR_IFINDEX]);
    memcpy (mac, nla_data (tb [NL80211_ATTR_MAC]), ETH_ALEN);

    device = genl_get_wiphy_dev (devices, NULL, wiphyiidx);
    return_val_if_fail (device, NL_SKIP);

    if (removal)
        success = g_hash_table_remove (device->interfaces, GUINT_TO_POINTER (index));
    else {
        wirelessif = genl_get_nl80211_interface (device, name, type, mac, index);
        if (wirelessif)
            success = TRUE;
    }

    if (!success)
        ERROR ("Failed to %s the network interface index: %u "
               "type: %u name: %s "
               "mac: %02x:%02x:%02x:%02x:%02x:%02x "
               "to the wiphy: %s",
               !removal ? "add" : "remove",
               index, type, name, MACADDRESS (mac),
               device->phyname ? device->phyname : "unknown");

    return NL_SKIP;
}

static int
genl_queue_requests (const genlclient *const client,
                     enum nl80211_commands cmd,
                     gboolean global,
                     unsigned int wiphyid)
{
    int ret;
    struct nl_msg *msg;
    int flags = 0;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (client, -EINVAL);

    msg = nlmsg_alloc ();
    return_val_if_fail (msg, -ENOMEM);

    /* The mainline kernel supports dumping even for the
     * regulatory settings but not in 3 series of Gen3 platform */
    if (
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,0,0)
        NL80211_CMD_GET_REG != cmd &&
#endif
        client->protofeatures & NL80211_PROTOCOL_FEATURE_SPLIT_WIPHY_DUMP)
        flags = NLM_F_DUMP;

    if (!genlmsg_put (msg, 0, 0, client->nl80211id,
                       0, flags, cmd, 0)) {
        nlmsg_free (msg);
        return -ENOMSG;
    }

    if (!global) {
        ret = nla_put_u32 (msg, NL80211_ATTR_WIPHY, wiphyid);
        if (ret < 0) {
            ERROR ("Failed to add the wiphy index to the message: %s/%d",
                   nl_geterror (-ret), -ret);
            nlmsg_free (msg);
            return -ENOMSG;
        }
    }

    genlrequests = g_slist_append (genlrequests, msg);

    if (g_slist_length (genlrequests) > 1)
        return 0;

    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Requesting %s dump from kernel [seq: %u]", genl_nl80211_cmd2string (cmd),
    		nlhdr->nlmsg_seq);

    ret = nl_send_auto (client->sk, msg);
    if (ret < 0)
        ERROR ("Failed to aquire the %s information, error: %s/%d",
               genl_nl80211_cmd2string (cmd), nl_geterror (ret), -ret);

    return ret;
}

static int
genl_get_80211_information (genlclient *client)
{
    int ret;
    size_t index = 0;
    enum nl80211_commands nlcmds [3] = {
        NL80211_CMD_GET_REG,
        NL80211_CMD_GET_WIPHY,
        NL80211_CMD_GET_INTERFACE
    };

    return_val_if_fail (client, -EINVAL);

    DEBUG ("Requesting nl80211 dump via Genl Client: %p", client);

    for ( ; index < ARRAY_SIZE (nlcmds); ++index) {
        ret = genl_get_nl80211_settings (client, nlcmds [index], TRUE, 0);
        if (ret < 0) {
            ERROR ("Failed to get the %s information: %s/%d",
                   genl_nl80211_cmd2string (nlcmds [index]),
                   strerror (-ret), -ret);
        }
    }

    nl80211dumped = TRUE;
    g_hash_table_foreach (wireless_devices, genl_dump_wiphyinfo, NULL);
    genl_update_regulatory_change (TRUE, 0);
    return ret;
}

static int
genl_update_regulatory_data (genlclient *client,
                             void *wiphys,
                             enum nl80211_reg_initiator initiator,
                             enum nl80211_reg_type type,
                             const char *alpha2)
{
    GHashTableIter iter;
    gpointer key, value;
    GHashTable *devices = wiphys;
    wiphyinfo *device;

    return_val_if_fail (client && devices && alpha2, -EINVAL);

    client->initator = initiator;
    client->type = type;
    memcpy (client->alpha2, alpha2, sizeof (client->alpha2));

    g_hash_table_iter_init (&iter, devices);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        device = value;
        continue_if_fail (device && !device->selfmanaged);

        DEBUG ("Updating the device \"%s\" on the regulatory change",
               device->phyname ? device->phyname : "unknown");

        device->initator = initiator;
        device->type = type;
        memcpy (device->alpha2, alpha2, sizeof (device->alpha2));
    }

    return 0;
}

static int
genl_update_dfs_domain (genlclient *client,
                        void *wiphys,
                        enum nl80211_dfs_regions dfsregion)
{
    GHashTableIter iter;
    gpointer key, value;
    GHashTable *devices = wiphys;
    wiphyinfo *device;

    return_val_if_fail (client && devices, -EINVAL);

    client->dfs_region = dfsregion;
    g_hash_table_iter_init (&iter, devices);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        device = value;
        continue_if_fail (device && !device->selfmanaged);
        device->dfs_region = dfsregion;
    }

    return 0;
}

static int
genl_get_nl80211_settings (genlclient *client,
                           enum nl80211_commands cmd,
                           gboolean global,
                           unsigned int index)
{
    int ret;
    struct nl_msg *msg;
    void *temp;
    int flags = 0;

    DEBUG ("Genl Client: %p Requesting NL command: %s/%d", client,
           genl_nl80211_cmd2string (cmd), cmd);

    msg = nlmsg_alloc ();
    return_val_if_fail (msg, -ENOMEM);

    if (
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,0,0)
       NL80211_CMD_GET_REG != cmd &&
#endif
       client->protofeatures & NL80211_PROTOCOL_FEATURE_SPLIT_WIPHY_DUMP)
        flags = NLM_F_DUMP;

    temp = genlmsg_put (msg, 0, 0, client->nl80211id, 0, flags, cmd, 0);
    if (!temp) {
        ret = -ENOMSG;
        goto failure;
    }

    if (!global) {
        ret = nla_put_u32 (msg, NL80211_ATTR_WIPHY, index);
        if (ret < 0) {
            ERROR ("Failed to add the wiphy index to the message: %s/%d",
                   nl_geterror (-ret), -ret);
            goto failure;
        }
    }

    ret = genl_send_and_recv (client->s_rw, client->cb, msg,
                              genl_global_process_event, client);
    if (ret < 0)
        ERROR ("Failed to acquire the %d information, error: %s/%d", cmd,
                nl_geterror (-ret), -ret);

failure:
    nlmsg_free (msg);
    return ret;
}

static int
genl_process_regulatory_change (genlclient *cli,
                                struct nlattr **attr)
{
    int ret = 0, rem;
    char alpha [GENL_ALPHA_SIZE];
    u_int8_t dfs_domain =
            (u_int8_t) NL80211_DFS_UNSET;
    unsigned int index = 0, flags, start_freq,
            end_freq, max_bw = 0, eirp = 0;
    gboolean global = TRUE;
    wiphyinfo *device;
    struct nlattr *rule;
    struct nla_policy rule_policy [NL80211_REG_RULE_ATTR_MAX + 1] = {
        [NL80211_ATTR_REG_RULE_FLAGS]		= { .type = NLA_U32 },
        [NL80211_ATTR_FREQ_RANGE_START]		= { .type = NLA_U32 },
        [NL80211_ATTR_FREQ_RANGE_END]		= { .type = NLA_U32 },
        [NL80211_ATTR_FREQ_RANGE_MAX_BW]	= { .type = NLA_U32 },
        [NL80211_ATTR_POWER_RULE_MAX_ANT_GAIN] = { .type = NLA_U32 },
        [NL80211_ATTR_POWER_RULE_MAX_EIRP]	= { .type = NLA_U32 },
        [NL80211_ATTR_DFS_CAC_TIME]         = { .type = NLA_U32 },
    };

    return_val_if_fail (cli && attr, -EINVAL);

    /* mandatory */
    if (!attr [NL80211_ATTR_REG_ALPHA2] ||
            !attr [NL80211_ATTR_REG_RULES])
        return -EINVAL;

    if (attr [NL80211_ATTR_WIPHY]) {
        index = nla_get_u32 (attr [NL80211_ATTR_WIPHY]);
        global = FALSE;
    }

    if (attr [NL80211_ATTR_DFS_REGION])
        dfs_domain = nla_get_u8 (attr [NL80211_ATTR_DFS_REGION]);

    memset (alpha, 0, sizeof (alpha));
    strncpy (alpha, nla_get_string (attr [NL80211_ATTR_REG_ALPHA2]), 2);

    INFO ("Device Specific: %s [index: %u] "
          "Country %s: %s",
          !global ? "Y" : "N",
          index, alpha,
          genl_reg_change_dfsdomain_to_string (dfs_domain));

    nla_for_each_nested (rule, attr [NL80211_ATTR_REG_RULES], rem) {

        struct nlattr *tb [NL80211_REG_RULE_ATTR_MAX + 1];
        nla_parse (tb, NL80211_REG_RULE_ATTR_MAX, nla_data (rule),
                   nla_len (rule), rule_policy);

        continue_if_fail (tb [NL80211_ATTR_REG_RULE_FLAGS] &&
                          tb [NL80211_ATTR_FREQ_RANGE_START] &&
                          tb [NL80211_ATTR_FREQ_RANGE_END]);

        flags = nla_get_u32 (tb [NL80211_ATTR_REG_RULE_FLAGS]);
        start_freq = KHZ_TO_MHZ (nla_get_u32 (tb [NL80211_ATTR_FREQ_RANGE_START]));
        end_freq = KHZ_TO_MHZ (nla_get_u32 (tb [NL80211_ATTR_FREQ_RANGE_END]));

        if (tb [NL80211_ATTR_FREQ_RANGE_MAX_BW])
            max_bw = nla_get_u32 (tb [NL80211_ATTR_FREQ_RANGE_MAX_BW]) / 1000;
        if (tb [NL80211_ATTR_POWER_RULE_MAX_EIRP])
            eirp = nla_get_u32 (tb [NL80211_ATTR_POWER_RULE_MAX_EIRP]) / 100;


        INFO ("Wiphy Info: %u-%u @ %u MHz %u mBm%s%s%s%s%s%s%s%s"
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
              "%s%s%s%s%s%s"
#endif
              ,
              start_freq,
              end_freq,
              max_bw,
              eirp,
              !!(flags & NL80211_RRF_NO_OFDM) ? " (no OFDM)" : "",
              !!(flags & NL80211_RRF_NO_CCK) ? " (no CCK)" : "",
              !!(flags & NL80211_RRF_NO_INDOOR) ? " (no indoor)" : "",
              !!(flags & NL80211_RRF_NO_OUTDOOR) ? " (no outdoor)" : "",
              !!(flags & NL80211_RRF_DFS) ? " (DFS)" : "",
              !!(flags & NL80211_RRF_PTP_ONLY) ? " (PTP only)" : "",
              !!(flags & NL80211_RRF_PTMP_ONLY) ? " (PTMP only)" : "",
              !!(flags & NL80211_RRF_NO_IR) ? " (no IR)" : ""
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
              ,
              !!(flags & NL80211_RRF_AUTO_BW) ? " (AUTO-BW)" : "",
              !!(flags & NL80211_RRF_IR_CONCURRENT) ? " (IR-CONCURRENT)" : "",
              !!(flags & NL80211_RRF_NO_HT40MINUS) ? " (NO-HT40MINUS)" : "",
              !!(flags & NL80211_RRF_NO_HT40PLUS) ? " (NO-HT40PLUS)" : "",
              !!(flags & NL80211_RRF_NO_80MHZ) ? " (NO-80MHZ)" : "",
              !!(flags & NL80211_RRF_NO_160MHZ) ? " (NO-160MHZ)": ""
#endif
              );
    }

    if (!global) {
        device = genl_get_wiphy_dev (wireless_devices, NULL, index);
        return_val_if_fail (device, NL_SKIP);
        if (device->selfmanaged) {
            device->dfs_region = dfs_domain;
            if (g_strcmp0 (alpha, device->alpha2))
                memcpy (device->alpha2, alpha, GENL_ALPHA_SIZE);
        } else global = TRUE;
    }

    if (global)
        genl_update_dfs_domain (cli, wireless_devices, dfs_domain);

    if (!nl80211dumped) {
        if (global)
            genl_update_regulatory_data (cli, wireless_devices, 255, 255, alpha);
        return ret;
    }

    return genl_get_nl80211_settings (cli, NL80211_CMD_GET_WIPHY,
                                      global, index);
}

static nl80211_scan_state
genl_get_scan_state_from_nlcmd (enum nl80211_commands cmd)
{
    switch (cmd) {
    case NL80211_CMD_START_SCHED_SCAN:
        return NL80211_SCAN_STATE_SCHED_SCAN_STARTED;
    case NL80211_CMD_TRIGGER_SCAN:
        return NL80211_SCAN_STATE_SCAN_STARTED;
    case NL80211_CMD_NEW_SCAN_RESULTS:
        return NL80211_SCAN_STATE_SCAN_COMPLETED;
    case NL80211_CMD_SCAN_ABORTED:
        return NL80211_SCAN_STATE_SCAN_ABORTED;
    case NL80211_CMD_SCHED_SCAN_STOPPED:
        return NL80211_SCAN_STATE_SCHED_SCAN_STOPPED;
    case NL80211_CMD_SCHED_SCAN_RESULTS:
        return NL80211_SCAN_STATE_SCHED_SCAN_COMPLETED;
    default:
        break;
    }

    return NL80211_NUM_SCAN_STATES;
}

const char*
genl_scan_state_to_string (nl80211_scan_state state)
{
    switch (state) {
    case NL80211_SCAN_STATE_SCHED_SCAN_STARTED:
        return ENUMTOSTR (SCHED_SCAN_STARTED);
    case NL80211_SCAN_STATE_SCAN_STARTED:
        return ENUMTOSTR (SCAN_STARTED);
    case NL80211_SCAN_STATE_SCAN_ABORTED:
        return ENUMTOSTR (SCAN_ABORTED);
    case NL80211_SCAN_STATE_SCAN_COMPLETED:
        return ENUMTOSTR (SCAN_COMPLETED);
    case NL80211_SCAN_STATE_SCHED_SCAN_STOPPED:
        return ENUMTOSTR (SCHED_SCAN_STOPPED);
    case NL80211_SCAN_STATE_SCHED_SCAN_COMPLETED:
        return ENUMTOSTR (SCHED_SCAN_COMPLETED);
    default:
        break;
    }

    return ENUMTOSTR (SCAN_STATE_UNKNOWN);
}

static int
genl_process_scan_changes (genlclient *cli,
                           struct nlattr **attr,
                           nl80211_scan_state state)
{
    GHashTableIter iter;
    gpointer key, value;
    wiphyinfo *device;
    unsigned int wiphyidx = 0,
            ifindex = 0;

    return_val_if_fail (cli && attr && wireless_devices &&
                        state != NL80211_NUM_SCAN_STATES, -EINVAL);
    return_val_if_fail (attr [NL80211_ATTR_WIPHY], -EINVAL);

    wiphyidx = nla_get_u32 (attr [NL80211_ATTR_WIPHY]);
    if (attr [NL80211_ATTR_IFINDEX])
        ifindex = nla_get_u32 (attr [NL80211_ATTR_IFINDEX]);

    g_hash_table_iter_init (&iter, wireless_devices);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        device = value;
        continue_if_fail (device && device->ieee80211index == wiphyidx);

        INFO ("Scan state change for the device: %s "
              "[triggered on: %u/%s] "
              "Scan State: %s",
              device->phyname,
              ifindex,
              !device->interfaces ? "unknown" :
              !g_hash_table_lookup (device->interfaces,
              GUINT_TO_POINTER (ifindex)) ? "unknown" :
              ((nl80211interface *) g_hash_table_lookup (
                   device->interfaces,
              GUINT_TO_POINTER (ifindex)))->name,
              genl_scan_state_to_string (state));

        genl_update_scan_state_changes (device, state);
        break;
    }

    return 0;
}

static int
genl_handle_reg_update (genlclient *client,
                        gboolean global,
                        struct nlattr **tb)
{
    u_int8_t init = 255,
            type = 255;
    unsigned int wiphyid = 0;
    wiphyinfo *device;
    char alpha2 [GENL_ALPHA_SIZE], *temp = NULL;

    return_val_if_fail (client && tb, -EINVAL);
    return_val_if_fail (tb [NL80211_ATTR_REG_TYPE] &&
                        tb [NL80211_ATTR_REG_INITIATOR], -EINVAL);

    if (global && !tb [NL80211_ATTR_WIPHY])
        return -EINVAL;

    memset (alpha2, 0, sizeof (alpha2));
    type = nla_get_u8 (tb [NL80211_ATTR_REG_TYPE]);
    init = nla_get_u8 (tb [NL80211_ATTR_REG_INITIATOR]);

    if (global)
        wiphyid = nla_get_u32 (tb [NL80211_ATTR_WIPHY]);

    switch (type) {
    case NL80211_REGDOM_TYPE_COUNTRY:
        strncpy (alpha2, nla_get_string (tb [NL80211_ATTR_REG_ALPHA2]), 2);
        break;
    default:
    case NL80211_REGDOM_TYPE_WORLD:
        temp = "00";
        break;
    case NL80211_REGDOM_TYPE_CUSTOM_WORLD:
        temp = "99";
        break;
    case NL80211_REGDOM_TYPE_INTERSECTION:
        temp = "98";
        break;
    }

    if (temp)
        memcpy (alpha2, temp, 2);

    INFO ("Device Specific: %s "
          "Device Index: %u "
          "change Initiator: \"%s\" "
          "Change type: \"%s\" "
          "country code: \"%s\"",
          !global ? "Y" : "N",
          wiphyid,
          genl_reg_change_initiator_to_string (init),
          genl_reg_change_type_to_string (type),
          alpha2);

    if (!global) {
        device = genl_get_wiphy_dev (wireless_devices, NULL, wiphyid);
        return_val_if_fail (device, -ENOMEM);
        if (device->selfmanaged) {
            device->initator = init;
            device->type = type;
            memcpy (device->alpha2, alpha2, sizeof (device->alpha2));
            return 0;
        }
    }

    return genl_update_regulatory_data (client, wireless_devices, init,
                                        type, alpha2);
}

static int
genl_global_process_event (struct nl_msg *msg,
                           void *arg)
{
    genlclient *cli = arg;
    struct nlattr *tb [NL80211_ATTR_MAX+ 1];
    struct genlmsghdr *gnlh;
    gboolean removed = FALSE;
    unsigned int index = 0;

    return_val_if_fail (msg && cli, NL_SKIP);

    if (nlmsg_get_proto (msg) != NETLINK_GENERIC)
        return 0;

    gnlh = nlmsg_data (nlmsg_hdr (msg));

    INFO ("Event received for the socket: %d [event: %s]",
          nl_socket_get_fd (cli->sk), genl_nl80211_cmd2string (gnlh->cmd));

    switch (gnlh->cmd) {

    /**
     * When there is a regulatory event, not everything gets
     * exported to userspace as an event (except alpha2,
     * intiator and type). Therefore it is necessary to dump
     * the complete settings in order to get a snapshot of
     * what is maintained by the device and kernel
     */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
    case NL80211_CMD_WIPHY_REG_CHANGE:
        /* just utilizing the same variable (not a
         * global change) */
        removed = TRUE;
#endif
        /* fall through */
    case NL80211_CMD_REG_CHANGE:
        /* These events are no multipart messages
         * and therefore only one request will be queued.
         * Take care if that changes (in kernel) */
        nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
                   genlmsg_attrlen (gnlh, 0), NULL);
        genl_handle_reg_update (cli, removed, tb);
        genl_get_nl80211_settings (cli, NL80211_CMD_GET_REG, removed, index);
        break;

    case NL80211_CMD_GET_REG:
        nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
                   genlmsg_attrlen (gnlh, 0), NULL);
        genl_process_regulatory_change (cli, tb);
        break;

    case NL80211_CMD_DEL_WIPHY:
        removed = TRUE;
        /* fall through */
    case NL80211_CMD_NEW_WIPHY:
        genl_wiphy_capabilities_handler (cli, msg, wireless_devices, removed);
        break;

    case NL80211_CMD_DEL_INTERFACE:
        removed = TRUE;
        /* fall through */
    case NL80211_CMD_NEW_INTERFACE:
        genl_wiphy_interfaces_handler (msg, wireless_devices, removed);
        break;

    case NL80211_CMD_TRIGGER_SCAN:
    case NL80211_CMD_NEW_SCAN_RESULTS:
    case NL80211_CMD_SCAN_ABORTED:
    case NL80211_CMD_START_SCHED_SCAN:
    case NL80211_CMD_SCHED_SCAN_RESULTS:
    case NL80211_CMD_SCHED_SCAN_STOPPED:
        nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
                   genlmsg_attrlen (gnlh, 0), NULL);
        genl_process_scan_changes (cli, tb, genl_get_scan_state_from_nlcmd (
                                       gnlh->cmd));
        break;

    default:
        break;
    }

    return NL_OK;
}

static gboolean
genl_global_event (GIOChannel *chan,
                   GIOCondition cond,
                   gpointer data)
{
    genlclient *cli;
    int res;

    cli = data;
    return_val_if_fail (cli, TRUE);

    /* IO Channel no more valid */
    if (cond & (G_IO_NVAL | G_IO_HUP | G_IO_ERR)) {
        ERROR ("Error occured in GIOChannel %p", chan);
        cli->watch = 0;
        return FALSE;
    }

    res = nl_recvmsgs (cli->sk, cli->cb);
    if (res < 0) {
        ERROR ("nl80211: global message receive failed : "
               "%s [%d]", genl_error_to_string (-res), -res);
    }

    return TRUE;
}

static int
genl_client_initialize (genlclient *client)
{
    int ret = -ENOENT;

    return_val_if_fail (client, -EINVAL);

    /* nl80211 unique identifier */
    client->nl80211id = genl_ctrl_resolve (client->sk, "nl80211");
    if (client->nl80211id < 0) {
        ERROR ("\"nl80211\" generic netlink not found");
        return ret;
    }

    /*
     * Check if the device supports WIPHY_DUMP so that the
     * same could be requested
     */
    ret = genl_get_nl80211_protocol_features (client, &client->protofeatures);
    if (ret < 0)
        ERROR ("Failed to get the procotol features: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

static int
genl_client_init (genlclient **cli)
{
    int err = 0, sk;
    genlclient *client = NULL;

    return_val_if_fail (cli, -EINVAL);
    return_val_if_fail (!*cli, -EALREADY);

    client = g_try_malloc0 (sizeof (*client));
    return_val_if_fail (client, -ENOMEM);

    DEBUG ("Genl client created: %p", client);

    client->cb = nl_cb_alloc (NL_CB_DEFAULT);
    if (!client->cb) {
        ERROR ("Failed to allocate the callback for nl socket");
        err = -ENOMEM;
        goto error;
    }

    err = nl_socket_init (&client->sk, NETLINK_GENERIC, FALSE, 32, 8);
    if (err < 0) {
        ERROR ("Failed to allocate and connect the nlsocket: "
               "%s/%d", strerror (-err), -err);
        nl_cb_put (client->cb);
        goto error;
    }

    err = nl_socket_init (&client->s_rw, NETLINK_GENERIC, TRUE, 8, 8);
    if (err < 0) {
        ERROR ("Failed to allocate and connect the nl write socket: "
               "%s/%d", strerror (-err), -err);
        nl_cb_put (client->cb);
        nl_socket_free (client->sk);
        goto error;
    }

    err = genl_add_membership (client, genl_multicast_groups);
    if (err < 0)
        goto socket_failure;

    err = genl_client_initialize (client);
    if (err < 0) {
        ERROR ("Failed to get the procotol features: %s/%d", strerror (-err), -err);
        goto socket_failure;
    }

    sk = nl_socket_get_fd (client->sk);
    client->iochannel = g_io_channel_unix_new (sk);
    g_io_channel_set_close_on_unref (client->iochannel, TRUE);
    g_io_channel_set_encoding (client->iochannel, NULL, NULL);
    g_io_channel_set_buffered (client->iochannel, FALSE);

    client->watch = g_io_add_watch (client->iochannel,
                                    G_IO_IN | G_IO_NVAL |
                                    G_IO_HUP | G_IO_ERR,
                                    genl_global_event, client);

    nl_cb_err (client->cb, NL_CB_CUSTOM,
               (nl_recvmsg_err_cb_t) genl_global_error_handler, NULL);
    nl_cb_set (client->cb, NL_CB_FINISH, NL_CB_CUSTOM,
               genl_global_finish_handler, client);
    nl_cb_set (client->cb, NL_CB_ACK, NL_CB_CUSTOM,
               genl_global_ack_handler, client);
    nl_cb_set (client->cb, NL_CB_SEQ_CHECK, NL_CB_CUSTOM,
               genl_seq_check, NULL);
    nl_cb_set (client->cb, NL_CB_VALID, NL_CB_CUSTOM,
               genl_global_process_event, client);

    INFO ("Event socket created and the io channel setup "
          "successfully for: %d", sk);

    *cli = client;
    return err;

socket_failure:
    nl_cb_put (client->cb);
    nl_socket_free (client->sk);
    nl_socket_free (client->s_rw);
error:
    g_free (client);
    *cli = NULL;
    return err;
}

static int
genl_client_deinit (genlclient **cli)
{
    genlclient *client;

    return_val_if_fail (cli, -EINVAL);
    return_val_if_fail (*cli, -EALREADY);

    DEBUG ("Genl Client: %p", *cli);

    client = *cli;
    if (client->watch > 0)
        g_source_remove (client->watch);

    g_io_channel_unref (client->iochannel);
    nl_cb_put (client->cb);
    nl_socket_free (client->sk);
    g_free (client);

    *cli = NULL;
    return 0;
}

const char*
genl_get_phyname (const char *ifname)
{
    wiphyinfo *device;
    GHashTableIter iter1, iter2;
    gpointer key1, value1, key2, value2;
    nl80211interface *interface;

    return_val_if_fail (ifname && wireless_devices, NULL);

    g_hash_table_iter_init (&iter1, wireless_devices);
    while (g_hash_table_iter_next (&iter1, &key1, &value1)) {

        device = value1;
        continue_if_fail (device && device->interfaces);

        g_hash_table_iter_init (&iter2, device->interfaces);
        while (g_hash_table_iter_next (&iter2, &key2, &value2)) {
            interface = value2;
            if (interface && interface->name &&
                    !g_strcmp0 (ifname, interface->name))
                return device->phyname;
        }
    }

    return NULL;
}

int
genl_get_wiphyid (const char *ifname,
                  unsigned int *index)
{
    wiphyinfo *device;
    GHashTableIter iter1, iter2;
    gpointer key1, value1, key2, value2;
    nl80211interface *interface;

    return_val_if_fail (ifname && wireless_devices && index,
                        -ENOENT);

    g_hash_table_iter_init (&iter1, wireless_devices);
    while (g_hash_table_iter_next (&iter1, &key1, &value1)) {

        device = value1;
        continue_if_fail (device && device->interfaces);

        g_hash_table_iter_init (&iter2, device->interfaces);
        while (g_hash_table_iter_next (&iter2, &key2, &value2)) {
            interface = value2;
            if (interface && interface->name &&
                    !g_strcmp0 (ifname, interface->name)) {
                *index = device->ieee80211index;
                return 0;
            }
        }
    }

    return -ENOENT;
}

int
genl_interface_supported (const char *ifname)
{
    wiphyinfo *device;
    GHashTableIter iter1, iter2;
    gpointer key1, value1, key2, value2;
    nl80211interface *interface;

    return_val_if_fail (ifname, -EINVAL);
    return_val_if_fail (wireless_devices, -ENODEV);

    g_hash_table_iter_init (&iter1, wireless_devices);
    while (g_hash_table_iter_next (&iter1, &key1, &value1)) {

        device = value1;
        continue_if_fail (device && device->interfaces);

        g_hash_table_iter_init (&iter2, device->interfaces);
        while (g_hash_table_iter_next (&iter2, &key2, &value2)) {
            interface = value2;
            if (interface && interface->name &&
                    !g_strcmp0 (ifname, interface->name))
                return 0;
        }
    }

    return -ENOENT;
}

int
genl_get_supported_channels (const char *ifname,
                             unsigned int **channels,
                             unsigned int *size)
{
    int ret;
    wiphyinfo *dev;
    unsigned int wiphyid = 0;

    return_val_if_fail (ifname, -EINVAL);
    return_val_if_fail (wireless_devices, -ENODEV);

    ret = genl_get_wiphyid (ifname, &wiphyid);
    return_val_if_fail (ret == 0, -ENOENT);
    dev = g_hash_table_lookup (wireless_devices, GUINT_TO_POINTER (wiphyid));
    return_val_if_fail (dev, -ENOKEY);
    return genl_get_supported_channels_impl (dev, 0, NULL, channels, size);
}

int
genl_get_supported_frequencies (const char *ifname,
                                unsigned int **frequencies,
                                unsigned int *size)
{
    int ret;
    unsigned int wiphyid = 0;
    wiphyinfo *dev;

    return_val_if_fail (ifname, -EINVAL);
    return_val_if_fail (wireless_devices, -ENODEV);

    ret = genl_get_wiphyid (ifname, &wiphyid);
    return_val_if_fail (ret == 0, -ENOENT);
    dev = g_hash_table_lookup (wireless_devices, GUINT_TO_POINTER (wiphyid));
    return_val_if_fail (dev, -ENOKEY);
    return genl_get_supported_channels_impl (dev, 1, frequencies, NULL, size);
}

int
genl_get_supported_rates (const char *ifname,
                          unsigned int **rates,
                          unsigned int *length)
{
    int ret;
    unsigned int len = 0, wiphyid = 0;
    unsigned int *value, index = 0;
    wiphyinfo *dev;
    ieee80211modeinfo *info;
    GList *modes, *temp;
    GSList *local = NULL, *data;

    return_val_if_fail (ifname, -EINVAL);
    return_val_if_fail (wireless_devices, -ENODEV);

    ret = genl_get_wiphyid (ifname, &wiphyid);
    return_val_if_fail (ret == 0, -ENOENT);
    dev = g_hash_table_lookup (wireless_devices, GUINT_TO_POINTER (wiphyid));
    return_val_if_fail (dev, -ENODATA);

    modes = dev->modes;
    for ( ; modes; modes = modes->next) {
        info = modes->data;
        temp = (info != NULL) ? info->rates : NULL;
        for ( ; temp; temp = temp->next)
            if (!g_slist_find (local, temp->data))
                local = g_slist_prepend (local, temp->data);
    }

    local = g_slist_reverse (local);
    len = g_slist_length (local);

    DEBUG ("Length of unique rates: %u", len);
    value = g_try_malloc (len * sizeof (*value));
    return_val_if_fail (value, -ENOMEM);

    data = local;
    for ( ; data; data = data->next, index++)
        value [index] = GPOINTER_TO_UINT (data->data);

    *rates = value;
    *length = len;

    return 0;
}

int
genl_get_rates_from_mode (const char *ifname,
                          unsigned int mode,
                          unsigned int **rates,
                          unsigned int *length)
{
    int ret;
    unsigned int len = 0, wiphyid = 0;
    unsigned int *value, index = 0;
    wiphyinfo *dev;
    ieee80211modeinfo *info;
    GList *local = NULL, *modes, *temp = NULL;

    return_val_if_fail (ifname, -EINVAL);
    return_val_if_fail (wireless_devices, -ENODEV);

    ret = genl_get_wiphyid (ifname, &wiphyid);
    return_val_if_fail (ret == 0, -ENOENT);
    dev = g_hash_table_lookup (wireless_devices, GUINT_TO_POINTER (wiphyid));
    return_val_if_fail (dev, -ENODATA);

    modes = dev->modes;
    for ( ; modes; modes = modes->next) {
        info = modes->data;
        if (info && info->mode == mode) {
            temp = info->rates;
            break;
        }
    }

    return_val_if_fail (temp, -ENOTSUP);
    len = g_list_length (temp);
    value =  g_try_malloc0 (len * sizeof (*value));
    return_val_if_fail (value, -ENOMEM);

    local = temp;
    for ( ; local; local = local->next, index++)
        value [index] = GPOINTER_TO_UINT (local->data);

    *rates = value;
    *length = len;

    return 0;
}

int
genl_get_interface_type (const char *ifname)
{
    int ret;
    wiphyinfo *dev;
    unsigned int wiphyid = 0;
    GHashTableIter iter;
    gpointer key, value;
    nl80211interface *interface;

    return_val_if_fail (ifname, -EINVAL);
    return_val_if_fail (wireless_devices, -ENODEV);

    ret = genl_get_wiphyid (ifname, &wiphyid);
    return_val_if_fail (ret == 0, -ENOENT);
    dev = g_hash_table_lookup (wireless_devices, GUINT_TO_POINTER (wiphyid));
    return_val_if_fail (dev && dev->interfaces, -ENODATA);

    g_hash_table_iter_init (&iter, dev->interfaces);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        interface = value;
        if (interface && interface->name &&
                !g_strcmp0 (ifname, interface->name))
            return (int) interface->iftype;
    }

    return -1;
}

char*
genl_get_country_code (const char *ifname)
{
    int ret;
    wiphyinfo *dev;
    unsigned int wiphyid = 0;

    return_val_if_fail (ifname && genl_cli && wireless_devices, NULL);
    ret = genl_get_wiphyid (ifname, &wiphyid);
    return_val_if_fail (ret == 0, NULL);
    dev = g_hash_table_lookup (wireless_devices, GUINT_TO_POINTER (wiphyid));
    return_val_if_fail (dev, NULL);
    return dev->selfmanaged ? dev->alpha2 : genl_cli->alpha2;
}

int
genl_set_country_code (const char *ifname,
                       const char *alpha2)
{
    int ret;
    wiphyinfo *dev = NULL;
    void *temp;
    unsigned int wiphyid = 0;
    struct nl_msg *msg;

    return_val_if_fail (alpha2, -EINVAL);
    return_val_if_fail (genl_cli && wireless_devices, -ENOTCONN);

    ret = is_valid_regdom (alpha2);
    return_val_if_fail (ret == 0, -EINVAL);

    if (ifname) {
        ret = genl_get_wiphyid (ifname, &wiphyid);
        return_val_if_fail (ret == 0, -ENOENT);
        dev = g_hash_table_lookup (wireless_devices, GUINT_TO_POINTER (wiphyid));
        return_val_if_fail (dev, -ENODATA);
    }

    msg = nlmsg_alloc ();
    return_val_if_fail (msg, -ENOMEM);

    temp = genlmsg_put (msg, 0, 0, genl_cli->nl80211id, 0, 0,
                        NL80211_CMD_REQ_SET_REG, 0);
    if (!temp) {
        nlmsg_free (msg);
        return -ENOMSG;
    }

    if (dev && dev->selfmanaged) {
        ret = nla_put_u32 (msg, NL80211_ATTR_WIPHY, wiphyid);
        if (ret < 0)
            goto failure;
    }

    ret = nla_put_string (msg, NL80211_ATTR_REG_ALPHA2, alpha2);
    if (ret < 0)
        goto failure;

    ret = nl_send_auto_complete (genl_cli->s_rw, msg);
    nlmsg_free (msg);
    if (ret < 0)
        return ret;

    ret = nl_recvmsgs (genl_cli->s_rw, genl_cli->cb);
    if (ret < 0)
        ERROR ("Failed to change country code for \"%lu\" to [%s]]: %s/%d",
               wiphyid, alpha2, nl_geterror (-ret), -ret);

    return ret;

failure:
    nlmsg_free (msg);
    return ret;
}

int
genl_get_supported_hwmodes (const char *ifname,
                            unsigned int *modes)
{
    int ret;
    wiphyinfo *dev;
    GList *temp;
    unsigned int supp = 0,
            wiphyid = 0;

    return_val_if_fail (ifname && modes, -EINVAL);
    return_val_if_fail (wireless_devices, -ENODEV);

    ret = genl_get_wiphyid (ifname, &wiphyid);
    return_val_if_fail (ret == 0, -ENOENT);
    dev = g_hash_table_lookup (wireless_devices, GUINT_TO_POINTER (wiphyid));
    return_val_if_fail (dev, -ENODATA);

    temp = dev->modes;
    for ( ; temp; temp = temp->next) {
        ieee80211modeinfo *mode = temp->data;
        continue_if_fail (mode);
        if (mode->mode == IEEE80211_HW_MODE_B)
            supp |= IEEE80211_MODE_B;
        else if (mode->mode == IEEE80211_HW_MODE_G)
            supp |= IEEE80211_MODE_G;
        else if (mode->mode == IEEE80211_HW_MODE_A)
            supp |= IEEE80211_MODE_A;
        else if (mode->mode == IEEE80211_HW_MODE_AD)
            supp |= IEEE80211_MODE_AD;
    }

    *modes = supp;
    return 0;
}

#define CLIENT_REGISTRATION(regis, type, ops, clients)  \
    do {    \
        GList *temp;    \
        int found = 0;  \
        \
        return_val_if_fail (ops, -EINVAL);\
        \
        DEBUG ("%s the client for regulatory updates: %s [%p]",    \
               regis ? "Register" : "UnRegister", \
               ops->clientname ? ops->clientname : "UNKNOWN", \
               ops);    \
        \
        temp = clients; \
        for ( ; temp; temp = temp->next) {  \
            type *regops = temp->data; \
            if (ops == regops) { \
                found = 1;  \
                break;  \
            }   \
        }   \
        \
        if (regis && !found) {   \
            clients = g_list_append (clients, ops); \
            return 0;   \
        }   \
        else if (!regis && found) { \
            clients = g_list_remove_link (clients, temp);   \
            return 0;   \
        }   \
        \
        return regis ? -EALREADY : -ENOENT;    \
    } while (0);

int
genl_regnotifier_register (regulatory_ops *ops)
{
    CLIENT_REGISTRATION (TRUE, regulatory_ops, ops, reg_clients);
}

int
genl_regnotifier_unregister (regulatory_ops *ops)
{
    CLIENT_REGISTRATION (FALSE, regulatory_ops, ops, reg_clients);
}

int
genl_scannotifier_register (scanstate_ops *ops)
{
    CLIENT_REGISTRATION (TRUE, scanstate_ops, ops, scan_clients);
}

int
genl_scannotifier_unregister (scanstate_ops *ops)
{
    CLIENT_REGISTRATION (FALSE, scanstate_ops, ops, scan_clients);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(4,0,0)
static int
genl_handle_device_added (unsigned int index,
                          const char *name,
                          const char *address)
{
    int ret;

    return_val_if_fail (genl_cli, -ENOTCONN);

    DEBUG ("Wireless Device Added: %s [%u]", name ?
               name : "unknown", index);

    ret = genl_get_nl80211_settings (genl_cli, NL80211_CMD_GET_WIPHY, TRUE, 0);
    if (ret < 0) {
        ERROR ("Failed to get the available wifi devices information: %s/%d",
               strerror (-ret), -ret);
    }

    ret = genl_get_nl80211_settings (genl_cli, NL80211_CMD_GET_INTERFACE, TRUE, 0);
    if (ret < 0) {
        ERROR ("Failed to get the available wifi interfaces information: %s/%d",
                strerror (-ret), -ret);
    }

    return ret;
}

static int
genl_handle_device_removed (unsigned int index,
                            const char *name,
                            const char *address)
{
    int ret;

    return_val_if_fail (genl_cli, -ENOTCONN);

    DEBUG ("Wireless Device Removed: %s [%u]", name ?
               name : "unknown", index);

    ret = genl_get_nl80211_settings (genl_cli, NL80211_CMD_GET_WIPHY, TRUE, 0);
    if (ret < 0) {
        ERROR ("Failed to get the available wifi devices information: %s/%d",
               strerror (-ret), -ret);
    }

    ret = genl_get_nl80211_settings (genl_cli, NL80211_CMD_GET_INTERFACE, TRUE, 0);
    if (ret < 0) {
        ERROR ("Failed to get the available wifi interfaces information: %s/%d",
                strerror (-ret), -ret);
    }

    return ret;
}

static deviceops dev_ops = {
  .client  = "Genl",
  .device_added = genl_handle_device_added,
  .device_removed = genl_handle_device_removed,
};
#endif

int
genl_init (void)
{
    int ret;

    DEBUG ("");

    wireless_devices = g_hash_table_new_full (g_direct_hash, g_direct_equal,
                                              NULL, genl_phyinfo_cleanup);

    ret = genl_client_init (&genl_cli);
    if (ret < 0) {
        ERROR ("Failed to initialize the genl client: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    /*
     * 3.14 kernel does not report a new device (and iface) if gets
     * added runtime and therefore it is necessary to request again
     * the kernel to see them */
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,0,0)
    ret = device_notifier_register (&dev_ops);
    if (ret < 0) {
        ERROR ("Failed to register the device ops: %s/%d", strerror (-ret), -ret);
        return ret;
    }
#endif

    ret = genl_get_80211_information (genl_cli);
    if (ret < 0)
        ERROR ("Failed to get the global Regulatory settings information: %s/%d",
                strerror (-ret), -ret);

    return ret;
}

int
genl_deinit (void)
{
    int err;

    DEBUG ("");

    err = genl_client_deinit (&genl_cli);
    if (err < 0)
        ERROR ("Failed to deinitialize the genl client: %s/%d",
               strerror (-err), -err);

#if LINUX_VERSION_CODE < KERNEL_VERSION(4,0,0)
    err = device_notifier_unregister (&dev_ops);
    if (err < 0) {
        ERROR ("Failed to unregister the device ops: %s/%d",
               strerror (-err), -err);
    }
#endif

    if (wireless_devices) {
        g_hash_table_destroy (wireless_devices);
        wireless_devices = NULL;
    }

    g_list_free (reg_clients);
    return err;
}

/** @} */
