/**
 * @file accesspoint.h
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#ifndef ACCESSPOINT_H
#define ACCESSPOINT_H

/**
 * Errors reported by the access point
 * Clients have to check this for a better
 * control and decision making
 */
#define APE_SUCCESS            0
#define APE_SENDRECEIVE_FAILED 1
#define APE_TIMEOUT            2
#define APE_FAILURE            3
#define APE_INVAL              4
#define APE_ALREADY            6
#define APE_SYSNOTREADY        7
#define APE_NOLINK             8
#define APE_NO80211IFACE       9
#define APE_INTERNALFAIL       10
#define APE_NOENT              11
#define APE_NOTSTARTED         12
#define APE_NOTSUP             13
#define APE_BUSY               14
#define APE_IO                 15
#define NUM_APE_MAX            16

/**
 * IEEE80211 modes supported by the
 * Hardware
 */
#define IEEE80211_MODE_B        (1 << 0)
#define IEEE80211_MODE_G        (1 << 1)
#define IEEE80211_MODE_A        (1 << 2)
#define IEEE80211_MODE_AD       (1 << 3)
#define IEEE80211_MODE_MAX      0xFFFFFFFF

/**
 * enum stationrejectionreason - a STA's association
 * gets rejected due to any of the below reasons
 */
typedef
enum __station_rejection_reason
{
    /**
     * STA's association request is rejected as it is a
     * blocked station
     */
    REJECTED_BLOCKED_STATION,

    /**
     * STA's association is rejected as the station table
     * is already full i.e., radio is already serving maximum
     * no. of stations that it is capable/allowed
     */
    REJECTED_MAX_STA_REACHED,
    NUM_REJECTED_REASONS
} stationrejectionreason;

/**
 * enum acsevent - ACS events
 */
typedef
enum __acs_events
{
    /**
     * Automatic channel selection started
     */
    EVENT_ACS_STARTED,

    /**
     * Automatic channel selection completed and a
     * valid channel is picked for wlan operation
     */
    EVENT_ACS_COMPLETED,

    /**
     * Automatic channel selection failed due to
     * various reasons
     */
    EVENT_ACS_FAILED,
    NUM_ACS_EVENTS
} acsevent;

/**
 * enum dfsevent - DFS events
 */
typedef
enum __dfs_events
{
    /**
     * Presence of RADAR detected in the current
     * operating channel
     */
    EVENT_DFS_RADAR_DETECTED,

    /**
     * DFS has detected RADAR and will now switch
     * to a new channel
     */
    EVENT_DFS_NEW_CHANNEL,

    /**
     * DFS Channel Availability Check started i.e., the
     * DFS channel shall not be used till CAC completes
     * and detects there is no presence of radar
     */
    EVENT_DFS_CAC_START,

    /**
     * DFS Channel Availability Check completed and the
     * channel is available
     */
    EVENT_DFS_CAC_COMPLETED,

    /**
     * The Non-Occupancy Period for this channel is
     * over and the channel becomes usable
     */
    EVENT_DFS_NOP_FINISHED,
    NUM_DFS_EVENTS
} dfsevent;

/**
 * enum securitytype - Different security types
 * Note: Enterprise flavour is not supported yet
 */
typedef
enum __security_type
{
    AP_SECURITY_OPEN,
    AP_SECURITY_WEP,
    AP_SECURITY_WPA_PSK,
    AP_SECURITY_WPA2_PSK,
    AP_SECURITY_WPA_WPA2_PSK,
    NUM_AP_SECURITIES
} securitytype;

/**
 * struct wpsenrollee - Information of a wps enrollee
 * device which has requested for association
 */
typedef
struct __wps_enrollee
{
    /**
     * uuid - Universally Unique IDentifier (RFC 4122)
     * of the device
     */
    char *uuid;

    /**
     * macaddress - MAC address of the device
     */
    char *macaddress;

    /**
     * device_name - user friendly description of the
     * device
     */
    char *device_name;

    /**
     * manufacturer - manufacturer of the device
     */
    char *manufacturer;

    /**
     * modelname - model of the device
     */
    char *modelname;

    /**
     * modelnumber - additional device description
     */
    char *modelnumber;

    /**
     * serialnumber - serialnumber of the device
     */
    char *serialnumber;

    /**
     * devicecategory - device type
     */
    char *devicecategory;
} wpsenrollee;


/**
 * struct accesspointcallbacks - accesspoint related callbacks
 *
 * This struct shall be registered/subscribed by the controller
 * of the accesspoint to get the various access point events.
 *
 * @system_status - Informs the presence of hostapd process.
 *  @status - 1/0 (hostapd started/no more)
 *
 * @station_associated - a new station is associated
 *  @interface: access point iterface
 *  @station: the mac address of the associated station
 *
 * @station_disconnected - station is disconnected
 *  @interface: access point iterface
 *  @station: the mac address of the disconnected station
 *
 * @regulatory_change - A regulatory change has happened and there is
 * an update in the channel list
 *  @interface: access point interface
 *  @frequencies: changed list of frequencies (array)
 *  @channels: changed list of channels (array)
 *  @len: length of the list (channels and frequencies)
 *
 * @operatingchannel_changed - The current operating channel is changed
 *  @interface: access point interface
 *  @frequency: current operating frequency
 *  @channel: related channel
 *
 * @accesspoint_started - Access point is started
 *  @interface: access point interface
 *
 * @accesspoint_stopeed - Access point stopped
 *  @interface: access point interface
 *
 * @station_rejected - An association request of a station is rejected
 *  @interface: access point interface
 *  @station: mac address of the rejected station
 *  @reason: rejection reason
 *
 * @acs_event - An ACS update
 *  @interface: access point interface
 *  @event: ACS event, the application could be designed in such
 *  a way that if the ACS event is started, HMI could display a
 *  popup saying that the channel selection is in progress and when
 *  ACS gets completed, the selected channel could be displayed
 *
 * @dfs_event - DFS update
 *  @interface: access point interface
 *  @event: DFS event, the application could be designed in such
 *  a way that if the DFS event like RADAR is fired, HMI could display a
 *  popup saying that the RADAR is detected and a new channel is selected,
 *  and the moved channel could be displayed
 *
 * @wps_pin_needed - A WPS device has requested an association
 *  @interface: access point interface
 *  @device: information on the WPS device
 *
 * * @scan_state_change - Scan state has been changed for this interface
 *  @interface: access point interface
 *  @state: scan state
 */
typedef
struct __accesspoint_callbacks
{
    void (*system_status) (const int status);
    void (*station_associated) (const char *interface,
                                const char *station);
    void (*station_disconnected) (const char *interface,
                                  const char *station);
    void (*regulatory_change) (const char *interface,
                               const unsigned int *frequencies,
                               const unsigned int *channels,
                               const unsigned int len);
    void (*operatingchannel_changed) (const char *interface,
                                      const unsigned int frequency,
                                      const unsigned int channel);
    void (*accesspoint_started) (const char *interface);
    void (*accesspoint_stopped) (const char *interface);
    void (*station_rejected) (const char *interface,
                              const char *station,
                              stationrejectionreason reason);
    void (*acs_event) (const char *interface,
                       acsevent event);
    void (*dfs_event) (const char *interface,
                       dfsevent event);
    void (*wps_pin_needed) (const char *interface,
                            wpsenrollee *device);
    void (*scan_state_change) (const char *wiphy,
							   const char *interface,
                               int state);
} accesspointcallbacks;

/**
 * apresultcb - Callback for an async function call
 * @result - 0 on success and -APE_ERROR on failure
 * @ifname - interface name of the requested access point
 * @userdata - userdata to be passed along with the result information
 */
typedef
void (*apresultcb) (int result,
                    char *ifname,
                    void *userdata);

/**
 * accesspoint_init - Initialize the access point internals i.e., open a control
 * interface to hostapd, connect to netlink protocol for monitoring regulatory
 * updates..
 * @cbacks: Accesspoint callbacks to be registered/subscribed by the controller
 * of access point.
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_init (const accesspointcallbacks *cbacks);

/**
 * accesspoint_deinit - DeInitialize the access point internals i.e., closing the
 * control interface sockets to hostapd, disconnecting from netlink protocol..
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_deinit (void);

/**
 * accesspoint_add_interface - Add an interface to be controlled by hostapd
 * @ifname: access point interface
 * @configfile: access point configuration file
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_add_interface (const char *ifname,
                               const char *configfile,
                               int acs,
                               int num_acs_chans,
                               apresultcb cb,
                               void *userdata);

/**
 * accesspoint_remove_interface - Remove an interface from hostapd
 * @ifname: access point interface
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_remove_interface (const char *ifname,
                                  apresultcb cb,
                                  void *userdata);

/**
 * accesspoint_enable - Start/Enable the access point
 * @ifname: access point interface
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_enable (const char *ifname,
                        apresultcb cb,
                        void *userdata);

/**
 * accesspoint_disable - Stop/Disable the access point
 * @ifname: access point interface
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_disable (const char *ifname,
                         apresultcb cb,
                         void *userdata);

/**
 * accesspoint_reload - Reload the access point with the newly
 * configured details
 * @ifname: access point interface
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_reload (const char *ifname,
                        apresultcb cb,
                        void *userdata);

/**
 * accesspoint_disassociate_station - Disassociate/Disconnect a connected
 * station from the access point
 * @ifname: access point interface
 * @station: Mac address of the station to be disassociated/disconnected
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_disassociate_station (const char *ifname,
                                      const char *station,
                                      apresultcb cb,
                                      void *userdata);

int accesspoint_deauthenticate_station (const char *ifname,
                                        const char *station,
                                        apresultcb cb,
                                        void *userdata);

/**
 * accesspoint_set_passphrase - Change the passphrase of the operating
 * access point.
 * @ifname: access point interface
 * @passphrase: Passphrase shall be ASCII text of length 8..63 characters
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_set_passphrase (const char *ifname,
                                const char *passphrase,
                                apresultcb cb,
                                void *userdata);

/**
 * accesspoint_set_security - Change the security type of the operating
 * access point.
 * @ifname: access point interface
 * @type: Security type
 * @wep_key: Shall be given only in the case of security type WEP (AP_SECURITY_WEP)
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * The key length shall be 5 or 13 or 16 characters
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_set_security (const char *ifname,
                              const securitytype type,
                              const char *wep_key,
                              apresultcb cb,
                              void *userdata);

/**
 * accesspoint_set_operating_channel - Change the operating channel of the access
 * point. Only changing the channel will not change the operating channel but
 * the access point has to re-enabled by calling accesspoint_disable and
 * accesspoint_enable
 *
 * @ifname: access point interface
 * @channel: Preferred channel list or a channel or a special value of 0 where the
 * system will choose the best channel
 * @chanlen: length of the channel list
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 *
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_set_operating_channel (const char *ifname,
                                       const unsigned int channel[],
                                       const unsigned int chanlen,
                                       apresultcb cb,
                                       void *userdata);

int accesspoint_set_hwmode (const char *ifname,
                            const char *mode,
                            apresultcb cb,
                            void *userdata);

int accesspoint_set_hidden (const char *ifname,
                            const int hidden,
                            apresultcb cb,
                            void *userdata);

int accesspoint_set_accnettype (const char *ifname,
                            const unsigned int accnettype,
                            apresultcb cb,
                            void *userdata);

int accesspoint_set_vendor_ie (const char *ifname,
                               const char *vendor_element,
                               const unsigned long length,
                               gboolean isAssocResp,
                               apresultcb cb,
                               void *userdata);

/**
 * accesspoint_set_ssid - Change the SSID of the access point. Only changing
 * the channel will not change the operating channel but the access point has
 * to re-enabled by calling accesspoint_disable and accesspoint_enable
 * @ifname: access point interface
 * @ssid: SSID to be used in IEEE 802.11 management frames
 * @utf8: Whether the SSID is to be interpreted using UTF-8 encoding
 * @cb: callback for the result
 * @userdata: any userdata to be passed along with the result
 * Returns: 0 if successfully posted the request to hostapd or -APE_ERRORS on failure
 */
int accesspoint_set_ssid (const char *ifname,
                          const char *ssid,
                          size_t length,
						  const char *passphrase,
                          apresultcb cb,
                          void *userdata);

/**
 * accesspoint_get_supported_channels - Get the list of supported channels by the
 * access point
 * @ifname: access point interface
 * @channels: List of channels. It is the responsibility of the user to free the
 * channels
 * @size: size of the channel list
 * Returns: 0 on success and -APE_ERRORS on failure
 */
int accesspoint_get_supported_channels (const char *ifname,
                                        unsigned int **channels,
                                        unsigned int *size);

/**
 * accesspoint_get_supported_frequencies - Get the list of supported frequencies
 * by the access point
 * @ifname: access point interface
 * @frequencies: List of frequencies. It is the responsibility of the user to free the
 * frequencies
 * @size: size of the frequency list
 * Returns: 0 on success and -APE_ERRORS on failure
 */
int accesspoint_get_supported_frequencies (const char *ifname,
                                           unsigned int **frequencies,
                                           unsigned int *size);

/**
 * accesspoint_get_supported_hwmodes - Get the IEEE 802.11 modes supported by the
 * access point
 * @ifname: access point interface
 * @modes: IEEE 802.11 modes i.e., a bitwise-or'd IEEE80211 MODES
 * Returns: 0 on success and -APE_ERRORS on failure
 */
int accesspoint_get_supported_hwmodes (const char *ifname,
                                       unsigned int *modes);

/**
 * accesspoint_get_supported_rates - Get the list of supported rates
 * by the access point
 * @ifname: access point interface
 * @rates: List of rates. It is the responsibility of the user to free the
 * rates
 * @len: length of the rates list
 * Returns: 0 on success and -APE_ERRORS on failure
 */
int accesspoint_get_supported_rates (const char *ifname,
                                     unsigned int **rates,
                                     unsigned int *len);

int accesspoint_get_rates_from_mode (const char *ifname,
                                     unsigned int mode,
                                     unsigned int **rates,
                                     unsigned int *len);

int accesspoint_set_country_code (const char *ifname,
                                  const char *alpha2,
                                  apresultcb cb,
                                  void *userdata);

char *accesspoint_get_country_code (const char *ifname);


int accesspoint_set_max_num_sta (const char *ifname,
                                 unsigned int max_num_sta,
                                 apresultcb cb,
                                 void *userdata);

const char *accesspoint_strerror (int error);

#endif //ACCESSPOINT_H

/** @} */
