/*******************************************************************************
* FILE:          FC_PhoneBook_VerticalKeyBoard.cpp
* PROJECT:       GM-NextGen
* SW-COMPONENT:  VerticalKeyboard
*-------------------------------------------------------------------------------
*
* DESCRIPTION:   Implementation of Vertical Keyboard functionality
*
*-------------------------------------------------------------------------------
*
* AUTHOR:  CM-AI-Benthin  (ben2hi)
*
* COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
*
* HISTORY:
* - 22.08.2012: Initial version 
* - 16.10.2012: Performance optimization by comparing characters based on 
*               "SortPrio" retrieved from SortLib, instead of a direct comparison
*               of characters with the SortLib.
* - 07.07.2013: - Thai Non-Latin Keyboard added.
*               - Hangul SearchCharacters changed to the "base characters" (as 
*                 Sorting Library has done a related change and places them now 
*                 "in front" of the related syllables).
*               - LINT cleanup, which results in a bigger change from tU32 to 
*                 tS32 for the "Index" variables: 's32KbdIndex' 's32ItemIndex'
*                 as the index for QList is defined as 'signed int'.
*******************************************************************************/

#include "FC_PhoneBook_VerticalKeyBoard.h"

#include "qstringlist.h"  //  Needed for type 'QStringList'

#define SORTLIB_S_IMPORT_INTERFACE_GENERIC
#include "sortlib_if.h"   //  Needed for using the Global Sorting Library in '::vParseItemList(..)'


QList<VerticalSearchKeyboardListItem>* VerticalSearchKeyboard::showKeyboard(QStringList& rfoItemList, quint32 u32language)
{
   VKBD_FPRINTF(" -VKBD- VerticalSearchKeyboard::showKeyboard: Entered. " );

   QList<VerticalSearchKeyboardListItem>* poVerticalKeyboardList = NULL; 
   bool bNonLatinAvailable = false;
   
   // Create empty 'Search Keyboard Lists' (a Latin one and a Non-Latin one).
   QList<tsKeyboardChar> oLatinSearchKbdList;
   QList<tsKeyboardChar> oNonLatinSearchKbdList;
   
   // Initialize 'Search Keyboard Lists', by filling in the search characters (and the corresponding visible ones).
   bNonLatinAvailable = bInitializeSearchKeyboardLists( oLatinSearchKbdList, oNonLatinSearchKbdList, u32language );

   VKBD_FPRINTF(" -VKBD- VerticalSearchKeyboard::showKeyboard:  SearchKeyboards created. " );

   // Parse the given sorted 'Item List' list (rfoItemList) for "hits" in the 'Search Keyboard Lists'. 
   // (Note: Administrative data within the 'Search Keyboard Lists' like '.bCharIsAvailable', '.u32StartIndex' and '.u32EndIndex'
   //        will be adapted for each SearchCharacter during the following 'parsing'.)
   vParseItemList( rfoItemList, oLatinSearchKbdList );
   
   if( bNonLatinAvailable == true )
   {
      vParseItemList( rfoItemList, oNonLatinSearchKbdList );
   }
      

   // Create the resulting Vertikal Keyboard list dependent on dedicated facts, like "only Non-Latin chars" found, etc.
   poVerticalKeyboardList = poGetVerticalKeyboardList( oLatinSearchKbdList, oNonLatinSearchKbdList, bNonLatinAvailable );


   return poVerticalKeyboardList;
}


void VerticalSearchKeyboard::vParseItemList( QStringList &rfoItemList, QList<tsKeyboardChar> &rfoSearchKbdList )   
{
   quint32 u32ListChar_SortPrio;
   quint32 u32SearchChar_SortPrio;   

   // Notes: 
   // The first entry (index 0) of rfoSearchKbdList contains the '#' sign, which represents any character before the given language alphabet.
   //     rfoSearchKbdList[0].qsKbdChar    = "#"
   //     rfoSearchKbdList[0].qsSearchChar = " " (no search character set as to '#' all characters contribute which are smaller than the first  
   //                                          character of the given lanuage alphabet, e.g. 'a' for Latin or 'alpha' for Greek)
   //
   // The second entry of rfoSearchKbdList contains the start character of a language (e.g. 'a' for Latin or 'alpha' for Greek)
   //     rfoSearchKbdList[1].qsKbdChar    = "A"
   //     rfoSearchKbdList[1].qsSearchChar = "a" (search character)
   //
   // The last entry of rfoSearchKbdList contains the end character of a language (e.g. 'z' for Latin or 'omega' for Greek)
   //     rfoSearchKbdList[n].qsKbdChar    = "Z"
   //     rfoSearchKbdList[n].qsSearchChar = "z"
   
   // Therefore the first relevant 'Search Char' for the Vertical Keyboard is at index 1 and this character is
   // relevant to find out the u32Start-/EndIndex for KeyboardList[0] (i.e. for Keyboard character '#')
   
   // Set the Keyboard start character of a language (e.g. 'a' for Latin or 'alpha' for Greek), which has index = 1
   qint32 s32KbdIndex = 1;
   
   
   // Set the potential start index for the provided sorted rfoItemList
   qint32 s32ItemIndex = 0;
   
   bool bContinueSearch = true; 
   
   // Find the first 'List Item' which has a "valid" first letter
   while( (s32ItemIndex < rfoItemList.size()) && (rfoItemList[s32ItemIndex].isEmpty()) )
   {
      s32ItemIndex++;
   }
   
   VKBD_FPRINTF(" -VKBD-     : Item index, First: s32ItemIndex= %d ", s32ItemIndex);
   
   // Continue search, if the found s32ItemIndex is within the ItemList
   if( s32ItemIndex < rfoItemList.size() )
   {
      // If we come here, we have found a an item in rfoItemList with has a "valid" first letter and can continue the search.
      u32ListChar_SortPrio = sortlib_get_char_sort_prio( rfoItemList[s32ItemIndex].at(0).unicode() );

   // Find 'u32Start/StopIndex' for Vertical Keyboard characters with index 0 to rfoSearchKbdList.size()-2, e.g. '#' to 'y' for Latin keyboard
   while( (bContinueSearch == true) && (s32KbdIndex < rfoSearchKbdList.size()) )
   {
         u32SearchChar_SortPrio = rfoSearchKbdList[s32KbdIndex].u32SearchCharPrio;      

         while( (bContinueSearch == true) && (u32ListChar_SortPrio < u32SearchChar_SortPrio) )  // "qsListChar < qsSearchChar"
         {
            // Check if StartIndex for this Keyboard Char is already set and adapt Start/End Index appropriately
            if( rfoSearchKbdList[s32KbdIndex -1].bCharIsAvailable == false )
            {
               rfoSearchKbdList[s32KbdIndex -1].u32StartIndex = (quint32)s32ItemIndex;
               rfoSearchKbdList[s32KbdIndex -1].u32EndIndex   = (quint32)s32ItemIndex;

               rfoSearchKbdList[s32KbdIndex -1].bCharIsAvailable = true;
            }
            else
            {
               rfoSearchKbdList[s32KbdIndex -1].u32EndIndex   = (quint32)s32ItemIndex;
            }

            // Get next valid character from rfoItemList
            s32ItemIndex++;

            // Check if the next List Item which has a "valid" first letter, otherwise continue with the following index
            while( (s32ItemIndex < rfoItemList.size()) && (rfoItemList[s32ItemIndex].isEmpty()) )
            {
               s32ItemIndex++;
            }

            if( s32ItemIndex < rfoItemList.size() )
            {
               // Set next ListChar SortPrio
               u32ListChar_SortPrio = sortlib_get_char_sort_prio( rfoItemList[s32ItemIndex].at(0).unicode() );
         
               VKBD_FPRINTF(" -VKBD-     : Item index, Next: s32ItemIndex= %d ", s32ItemIndex);
            }
            else
            {
               // rfoItemList does not have more items, i.e. we have to stop the Keyboard search
               bContinueSearch = false;
            }

         }  // end of : while( qsListChar < qsSearchChar )

         s32KbdIndex++;

         VKBD_FPRINTF(" -VKBD- : Keyboard index, Next: s32KbdIndex= %d ", s32KbdIndex);

      }  // end of: while( s32KbdIndex < rfoSearchKbdList.size() -1)


      // Find 'u32Start/StopIndex' for Vertical Keyboard character with index 'rfoSearchKbdList.size()-1', e.g. 'z' for Latin keyboard
      s32KbdIndex  = rfoSearchKbdList.size() -1;
 
      // Note: There is no need to check if "qsListChar >= 'z'" as this is already true for the now following qsListChar items 
      // because of the preceeding search up to "qsListChar < 'z'".
            
      VKBD_FPRINTF(" -VKBD- : Start search for last character in Keyboard, e.g. 'z' for Latin keyboard. ");

      while( s32ItemIndex < rfoItemList.size() )   // "qsListChar >= 'z'"
      {
         VKBD_FPRINTF(" -VKBD- : Search for 'z'.  s32KbdIndex= %d,  s32ItemIndex= %d ", s32KbdIndex, s32ItemIndex);

         // Check if StartIndex for this Keyboard Char is already set and adapt Start/End Index appropriately
         if( rfoSearchKbdList[s32KbdIndex].bCharIsAvailable == false )
         {
            rfoSearchKbdList[s32KbdIndex].u32StartIndex = (quint32)s32ItemIndex; 
            rfoSearchKbdList[s32KbdIndex].u32EndIndex   = (quint32)s32ItemIndex; 
            
            rfoSearchKbdList[s32KbdIndex].bCharIsAvailable = true;
         }
         else
         {
            rfoSearchKbdList[s32KbdIndex].u32EndIndex   = (quint32)s32ItemIndex; 
         }
         
         // Get next valid character from rfoItemList
         s32ItemIndex++;
         

      }  // end of : while( s32ItemIndex < rfoItemList.size() )   // "qsListChar >= 'z'"

      VKBD_FPRINTF(" -VKBD- : End search for 'z'. ");

   }  // end of: if( s32ItemIndex < rfoItemList.size() )    

   
   VKBD_FPRINTF(" -VKBD- Printout Keyboard details: ");
   
   #ifdef DEBUG_FPRINTF_VERT_KEYBOARD
      s32KbdIndex = 0;
      while( s32KbdIndex < rfoSearchKbdList.size() )
      {
         QString qsTmp(rfoSearchKbdList[s32KbdIndex].qsKbdChar);
         VKBD_FPRINTF(" -VKBD-      id= %2d, KeyChar= %3s, Available= %d,   StartIndx= %3d, EndIndex= %3d ",
               s32KbdIndex,
               qsTmp.toUtf8().constData(),
               rfoSearchKbdList[s32KbdIndex].bCharIsAvailable,
               rfoSearchKbdList[s32KbdIndex].u32StartIndex,
               rfoSearchKbdList[s32KbdIndex].u32EndIndex
         );
         s32KbdIndex++;
      }
   #endif

   VKBD_FPRINTF(" -VKBD- END: algorithmic search"); 
   VKBD_FPRINTF(" -VKBD- END: ");  

}


QList<VerticalSearchKeyboardListItem>* VerticalSearchKeyboard::poGetVerticalKeyboardList( QList<tsKeyboardChar> &rfoLatinSearchKbdList, 
                                                                                  QList<tsKeyboardChar> &rfoNonLatinSearchKbdList, 
                                                                                  bool bNonLatinAvailable )
{
   QList<VerticalSearchKeyboardListItem>* poSearchKeyboardList = new QList<VerticalSearchKeyboardListItem>;

   if(poSearchKeyboardList != NULL)
   {

      // Define 'World-Icon'
      QString qsWorldIcon = QChar( 0x2641 );  // U+0x2641 (symbol 'Earth', UTF-8: e2 99 81) 

      // Create a single empty Vertical Keyboard Item
      VerticalSearchKeyboardListItem oKbdListItem;

      qint32 s32KbdIndex = 0;

      // Dedicated Cases, which have to be treated slightly different:
      //
      // - (2.1) Non-Latin Keyboard with "only" Non-Latin entries in Keyboard, 
      //      criterion: '#' (in Non-Latin SearchKbd) is "empty" (rfoNonLatinSearchKbdList[0].bCharIsAvailable == false)
      //      and e.g. for Greek: We have at least one character from 'alpha' to 'psi' available.
      //
      // - (2.2) Non-Latin Keyboard with both, i.e. Non-Latin characters and characters which comes alphabetically before e.g. Greek, available
      //      criterion: '#' (in Non-Latin SearchKbd) is "not empty" (rfoNonLatinSearchKbdList[0].bCharIsAvailable == true)
      //      and e.g. for Greek: We have at least one character from 'alpha' to 'psi' available.
      //
      // - (2.3) Non-Latin Keyboard without any Non-Latin entries in Keyboard, 
      //      criterrion: e.g. for Greek: All characters from 'alpha' to 'psi' are not available ("empty")
      //      (Note: 'psi', i.e. the last but one character of alphabet is relevant here. 'omega' could contain 
      //       hits from other languages, which comes alphabetically after Greek.)   


      if( bNonLatinAvailable == true )
      {
         // Non-Latin Search Keyboard has been created (i.e. a Non-Latin Language has been set).
         // (Note: The Latin Search Keyboard will exist always anyway.)


         bool bCharsAvailableInNonLatinKB = false;
         bool bHashAvailableInNonLatinKB  = false;

         if ( rfoNonLatinSearchKbdList.size() > 0 )  // For safety we check that the VerticalSearchKeyboard contains at least index 0
         {
            // Check if we have really Non-Latin characters available in Non-Latin Search Keyboard
            for( s32KbdIndex = 1;  s32KbdIndex < rfoNonLatinSearchKbdList.size()-1; s32KbdIndex++)  
            {  // Note: KbdIndex range reflects 'alpha' to 'psi' (example for Greek)

               if( rfoNonLatinSearchKbdList[s32KbdIndex].bCharIsAvailable == true )
               {
                  bCharsAvailableInNonLatinKB = true;
                  break;  // early leaving of for loop
               }
            }

            // Check if we have characters available which are alphabetically before the Non-Latin chars, 
            // i.e. if '#' in Non-Latin SearchKbd has setting 'CharIsAvailable == true'
            bHashAvailableInNonLatinKB = (rfoNonLatinSearchKbdList[0].bCharIsAvailable == true);
         }


         if( (bCharsAvailableInNonLatinKB == true) && (bHashAvailableInNonLatinKB == false) )
         { 
            // =========================================================================================================================
            // Case (2.1) Non-Latin Language is set  Only Non-Latin characters are available
            // =========================================================================================================================
            //
            // The resulting Vertical Keyboard shall contain only the Non-Latin characters, e.g. 'alpha' to 'omega',
            // but no '#' and no 'World-Icon'
            for( s32KbdIndex = 1;  s32KbdIndex < rfoNonLatinSearchKbdList.size(); s32KbdIndex++)
            {  // Note: s32KbdIndex = 1 excludes the 'Non-Latin #'
               oKbdListItem.m_sSearchKeyboardString = rfoNonLatinSearchKbdList[s32KbdIndex].qsKbdChar;
               oKbdListItem.m_bStringAvailable      = rfoNonLatinSearchKbdList[s32KbdIndex].bCharIsAvailable;
               oKbdListItem.m_u32StringStartIndex   = rfoNonLatinSearchKbdList[s32KbdIndex].u32StartIndex;
               oKbdListItem.m_u32StringEndIndex     = rfoNonLatinSearchKbdList[s32KbdIndex].u32EndIndex;

               (*poSearchKeyboardList).append(oKbdListItem);
            }

         }
         else if( (bCharsAvailableInNonLatinKB == true) && (bHashAvailableInNonLatinKB == true) )
         {
            // =========================================================================================================================
            // Case (2.2) Non-Latin Language is set  Non-Latin characters in combination with Latin characters or numbers are available
            // =========================================================================================================================
            //
            // The resulting Vertical Keyboard shall contain:
            // 'Latin #', Latin characters, 'World-Icon', 'Non-Latin #', Non-Latin characters

            // Insert the Latin characters to Vertical Keyboard
            for( s32KbdIndex = 0;  s32KbdIndex < rfoLatinSearchKbdList.size(); s32KbdIndex++)
            {  // Note: s32KbdIndex = 0 includes the 'Latin #'
               oKbdListItem.m_sSearchKeyboardString = rfoLatinSearchKbdList[s32KbdIndex].qsKbdChar;
               oKbdListItem.m_bStringAvailable      = rfoLatinSearchKbdList[s32KbdIndex].bCharIsAvailable;
               oKbdListItem.m_u32StringStartIndex   = rfoLatinSearchKbdList[s32KbdIndex].u32StartIndex;
               oKbdListItem.m_u32StringEndIndex     = rfoLatinSearchKbdList[s32KbdIndex].u32EndIndex;

               (*poSearchKeyboardList).append(oKbdListItem);
            }

            // Insert the 'World-Icon'
            oKbdListItem.m_sSearchKeyboardString = qsWorldIcon;
            oKbdListItem.m_bStringAvailable      = false;
            oKbdListItem.m_u32StringStartIndex   = 0;
            oKbdListItem.m_u32StringEndIndex     = 0;

            (*poSearchKeyboardList).append(oKbdListItem);

            // Insert the Non-Latin characters 
            for( s32KbdIndex = 0;  s32KbdIndex < rfoNonLatinSearchKbdList.size(); s32KbdIndex++)
            {  // Note: s32KbdIndex = 0 includes the 'Non-Latin #'
               oKbdListItem.m_sSearchKeyboardString = rfoNonLatinSearchKbdList[s32KbdIndex].qsKbdChar;
               oKbdListItem.m_bStringAvailable      = rfoNonLatinSearchKbdList[s32KbdIndex].bCharIsAvailable;
               oKbdListItem.m_u32StringStartIndex   = rfoNonLatinSearchKbdList[s32KbdIndex].u32StartIndex;
               oKbdListItem.m_u32StringEndIndex     = rfoNonLatinSearchKbdList[s32KbdIndex].u32EndIndex;

               (*poSearchKeyboardList).append(oKbdListItem);
            }         
         }
         else if( (bCharsAvailableInNonLatinKB == false) )
         {
            // =========================================================================================================================
            // Case (2.3) Non-Latin Language is set  But no Non-Latin characters are available
            // =========================================================================================================================
            //
            // The resulting Vertical Keyboard shall contain:
            // 'Latin #', Latin characters (Fall back to Latin Vertical Keyboard)

            // Insert the Latin characters to Vertical Keyboard
            for( s32KbdIndex = 0;  s32KbdIndex < rfoLatinSearchKbdList.size(); s32KbdIndex++)
            {  // Note: s32KbdIndex = 0 includes the 'Latin #'
               oKbdListItem.m_sSearchKeyboardString = rfoLatinSearchKbdList[s32KbdIndex].qsKbdChar;
               oKbdListItem.m_bStringAvailable      = rfoLatinSearchKbdList[s32KbdIndex].bCharIsAvailable;
               oKbdListItem.m_u32StringStartIndex   = rfoLatinSearchKbdList[s32KbdIndex].u32StartIndex;
               oKbdListItem.m_u32StringEndIndex     = rfoLatinSearchKbdList[s32KbdIndex].u32EndIndex;

               (*poSearchKeyboardList).append(oKbdListItem);
            }
         }      

      }  // end of: if( bNonLatinAvailable == true )
      else
      {
         // Only a Latin Search Keyboard has been created (i.e. a Latin Language or Chinese has been set)

         // =========================================================================================================================
         // Case (1) Latin Language is set 
         // =========================================================================================================================
         //
         // The resulting Vertical Keyboard shall contain:
         // 'Latin #', Latin characters  (no World-Icon etc.)            

         // Insert the Latin characters to Vertical Keyboard   
         for( s32KbdIndex = 0;  s32KbdIndex < rfoLatinSearchKbdList.size(); s32KbdIndex++)
         {  // Note: s32KbdIndex = 0 includes the 'Latin #'
            oKbdListItem.m_sSearchKeyboardString = rfoLatinSearchKbdList[s32KbdIndex].qsKbdChar;
            oKbdListItem.m_bStringAvailable      = rfoLatinSearchKbdList[s32KbdIndex].bCharIsAvailable;
            oKbdListItem.m_u32StringStartIndex   = rfoLatinSearchKbdList[s32KbdIndex].u32StartIndex;
            oKbdListItem.m_u32StringEndIndex     = rfoLatinSearchKbdList[s32KbdIndex].u32EndIndex;

            (*poSearchKeyboardList).append(oKbdListItem);
         }
      }

      return(poSearchKeyboardList);
   }
   else
   {
      // Never expected to come here. The memory allocation has failed. I would like to throw an FATAL_M_ASSERT_ALWAYS().
      VKBD_FPRINTF(" -VKBD- ::poGetVerticalKeyboardList : FATAL ERROR: Creation of 'new QList<VerticalSearchKeyboardListItem>' has failed. " );   
      // FATAL_M_ASSERT_ALWAYS(); // A FATAL ASSERT would be helpfull here, but is not supported for this lib (compiler error) ;-)
      return(NULLPTR);
   }
}   


bool VerticalSearchKeyboard::bInitializeSearchKeyboardLists( QList<tsKeyboardChar> &rfoLatinSearchKbdList, QList<tsKeyboardChar> &rfoNonLatinSearchKbdList, quint32 u32language )
{
   VKBD_FPRINTF(" -VKBD- VerticalSearchKeyboard::bInitializeSearchKeyboardLists: Given language is: %d . ", u32language);   
   
   // If 'u32language' requests a 'Non-Latin' language, this return variable shall be set to true
   bool bNonLatinAvailable = false;
   
   // Create a single empty keyboard character
   tsKeyboardChar oKbdChar;

   // Independent of the given language we need the 'rfoLatinSearchKbdList' (e.g. ENGLISH || STANDARD_CHINESE || TRADITIONAL_CHINESE)
   
   // Fill the 'Latin' rfoLatinSearchKbdList with search characters and visible characters.
   // Notes:  qsKbdChar:    visible char in HMI's 'Vertical Keyboard', 
   //         qsSearchChar: used as search "border" for mappping of list items to characters in 'Vertical Keyboard'.
   
   oKbdChar.qsKbdChar    = "#";
   oKbdChar.qsSearchChar = " ";
   rfoLatinSearchKbdList.append(oKbdChar);
      
   oKbdChar.qsKbdChar    = "A";
   oKbdChar.qsSearchChar = "a";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "B";
   oKbdChar.qsSearchChar = "b";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "C";
   oKbdChar.qsSearchChar = "c";
   rfoLatinSearchKbdList.append(oKbdChar);
   
   oKbdChar.qsKbdChar    = "D";
   oKbdChar.qsSearchChar = "d";
   rfoLatinSearchKbdList.append(oKbdChar);
     
   oKbdChar.qsKbdChar    = "E";
   oKbdChar.qsSearchChar = "e";
   rfoLatinSearchKbdList.append(oKbdChar);
   
   oKbdChar.qsKbdChar    = "F";
   oKbdChar.qsSearchChar = "f";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "G";
   oKbdChar.qsSearchChar = "g";
   rfoLatinSearchKbdList.append(oKbdChar);
   
   oKbdChar.qsKbdChar    = "H";
   oKbdChar.qsSearchChar = "h";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "I";
   oKbdChar.qsSearchChar = "i";
   rfoLatinSearchKbdList.append(oKbdChar);
   
   oKbdChar.qsKbdChar    = "J";
   oKbdChar.qsSearchChar = "j";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "K";
   oKbdChar.qsSearchChar = "k";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "L";
   oKbdChar.qsSearchChar = "l";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "M";
   oKbdChar.qsSearchChar = "m";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "N";
   oKbdChar.qsSearchChar = "n";
   rfoLatinSearchKbdList.append(oKbdChar);
   
   oKbdChar.qsKbdChar    = "O";
   oKbdChar.qsSearchChar = "o";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "P";
   oKbdChar.qsSearchChar = "p";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "Q";
   oKbdChar.qsSearchChar = "q";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "R";
   oKbdChar.qsSearchChar = "r";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "S";
   oKbdChar.qsSearchChar = "s";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "T";
   oKbdChar.qsSearchChar = "t";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "U";
   oKbdChar.qsSearchChar = "u";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "V";
   oKbdChar.qsSearchChar = "v";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "W";
   oKbdChar.qsSearchChar = "w";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "X";
   oKbdChar.qsSearchChar = "x";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "Y";
   oKbdChar.qsSearchChar = "y";
   rfoLatinSearchKbdList.append(oKbdChar);

   oKbdChar.qsKbdChar    = "Z";
   oKbdChar.qsSearchChar = "z";
   rfoLatinSearchKbdList.append(oKbdChar);

   VKBD_FPRINTF(" -VKBD- oLatinSearchKbdList   rfoLatinSearchKbdList.length()= %d ", rfoLatinSearchKbdList.length() );
   // Retrieve the sorting prio for the search chars of the Search Keyboard from Sorting Lib
   for( quint16 i=0; i<rfoLatinSearchKbdList.length(); i++ )
   {
      rfoLatinSearchKbdList[i].u32SearchCharPrio = sortlib_get_char_sort_prio( rfoLatinSearchKbdList[i].qsSearchChar.at(0).unicode() );
      VKBD_FPRINTF(" -VKBD- VerticalSearchKeyboard::bInitializeSearchKeyboardLists: rfoLatinSearchKbdList[%02d].u32SearchCharPrio= %d . ", i, rfoLatinSearchKbdList[i].u32SearchCharPrio);   
   }

   // Now we check if we need an additional 'Non-Latin Search Keyboard' and fill it accordingly.
   switch( u32language )
   {
      case GREEK:
      {
         bNonLatinAvailable = true;

         oKbdChar.qsKbdChar    = "#";
         oKbdChar.qsSearchChar = " ";
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(913);
         oKbdChar.qsSearchChar = QChar(945);  // GREEK SMALL LETTER ALPHA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(914);
         oKbdChar.qsSearchChar = QChar(946);  // GREEK SMALL LETTER BETA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(915);
         oKbdChar.qsSearchChar = QChar(947);  // GREEK SMALL LETTER GAMMA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(916);
         oKbdChar.qsSearchChar = QChar(948);  // GREEK SMALL LETTER DELTA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(917);
         oKbdChar.qsSearchChar = QChar(949);  // GREEK SMALL LETTER EPSILON
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(918);
         oKbdChar.qsSearchChar = QChar(950);  // GREEK SMALL LETTER ZETA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(919);
         oKbdChar.qsSearchChar = QChar(951);  // GREEK SMALL LETTER ETA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(920);
         oKbdChar.qsSearchChar = QChar(952);  // GREEK SMALL LETTER THETA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(921);
         oKbdChar.qsSearchChar = QChar(953);  // GREEK SMALL LETTER IOTA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(922);
         oKbdChar.qsSearchChar = QChar(954);  // GREEK SMALL LETTER KAPPA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(923);
         oKbdChar.qsSearchChar = QChar(955);  // GREEK SMALL LETTER LAMDA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(924);
         oKbdChar.qsSearchChar = QChar(956);  // GREEK SMALL LETTER MU
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(925);
         oKbdChar.qsSearchChar = QChar(957);  // GREEK SMALL LETTER NU
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(926);
         oKbdChar.qsSearchChar = QChar(958);  // GREEK SMALL LETTER XI
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(927);
         oKbdChar.qsSearchChar = QChar(959);  // GREEK SMALL LETTER OMICRON
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(928);
         oKbdChar.qsSearchChar = QChar(960);  // GREEK SMALL LETTER PI
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(929);
         oKbdChar.qsSearchChar = QChar(961);  // GREEK SMALL LETTER RHO
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(931);
         oKbdChar.qsSearchChar = QChar(963);  // GREEK SMALL LETTER SIGMA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(932);
         oKbdChar.qsSearchChar = QChar(964);  // GREEK SMALL LETTER TAU
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(933);
         oKbdChar.qsSearchChar = QChar(965);  // GREEK SMALL LETTER UPSILON
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(934);
         oKbdChar.qsSearchChar = QChar(966);  // GREEK SMALL LETTER PHI
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(935);
         oKbdChar.qsSearchChar = QChar(967);  // GREEK SMALL LETTER CHI
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(936);
         oKbdChar.qsSearchChar = QChar(968);  // GREEK SMALL LETTER PSI
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(937);
         oKbdChar.qsSearchChar = QChar(969);  // GREEK SMALL LETTER OMEGA
         rfoNonLatinSearchKbdList.append(oKbdChar);

      }
      break;

      case JAPANESE:  //(Hiragana/Katakana)
      {
         bNonLatinAvailable = true;

         oKbdChar.qsKbdChar    = "#";
         oKbdChar.qsSearchChar = " ";
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3042);
         oKbdChar.qsSearchChar = QChar(0x30A0); //KATAKANA-HIRAGANA DOUBLE HYPHEN
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x3044); //Extension I
               oKbdChar.qsSearchChar = QChar(0x3043); //HIRAGANA LETTER SMALL I
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3046); //Extension U
               oKbdChar.qsSearchChar = QChar(0x3045); //HIRAGANA LETTER SMALL U
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3048); //Extension E
               oKbdChar.qsSearchChar = QChar(0x3047); //HIRAGANA LETTER SMALL E
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x304A); //Extension O
               oKbdChar.qsSearchChar = QChar(0x3049); //HIRAGANA LETTER SMALL O
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End
         
         oKbdChar.qsKbdChar    = QChar(0x304B);
         oKbdChar.qsSearchChar = QChar(0x3095); //HIRAGANA LETTER SMALL KA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x304D); //Extension KI
               oKbdChar.qsSearchChar = QChar(0x304D); //HIRAGANA LETTER KI
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x304F); //Extension KU
               oKbdChar.qsSearchChar = QChar(0x304F); //HIRAGANA LETTER KU
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3051); //Extension KE
               oKbdChar.qsSearchChar = QChar(0x3051); //HIRAGANA LETTER KE
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3053); //Extension KO
               oKbdChar.qsSearchChar = QChar(0x3053); //HIRAGANA LETTER KO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End

         oKbdChar.qsKbdChar    = QChar(0x3055);
         oKbdChar.qsSearchChar = QChar(0x3055); //HIRAGANA LETTER SA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x3057); //Extension SI
               oKbdChar.qsSearchChar = QChar(0x3057); //HIRAGANA LETTER SI
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3059); //Extension SU
               oKbdChar.qsSearchChar = QChar(0x3059); //HIRAGANA LETTER SU
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x305B); //Extension SE
               oKbdChar.qsSearchChar = QChar(0x305B); //HIRAGANA LETTER SE
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x305D); //Extension SO
               oKbdChar.qsSearchChar = QChar(0x305D); //HIRAGANA LETTER SO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End

         oKbdChar.qsKbdChar    = QChar(0x305F);
         oKbdChar.qsSearchChar = QChar(0x305F); //HIRAGANA LETTER TA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x3061); //Extension TI
               oKbdChar.qsSearchChar = QChar(0x3061); //HIRAGANA LETTER TI
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3064); //Extension TU
               oKbdChar.qsSearchChar = QChar(0x3063); //HIRAGANA LETTER SMALL TU
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3066); //Extension TE
               oKbdChar.qsSearchChar = QChar(0x3066); //HIRAGANA LETTER TE
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3068); //Extension TO
               oKbdChar.qsSearchChar = QChar(0x3068); //HIRAGANA LETTER TO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End

         oKbdChar.qsKbdChar    = QChar(0x306A);
         oKbdChar.qsSearchChar = QChar(0x306A); //HIRAGANA LETTER NA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x306B); //Extension NI
               oKbdChar.qsSearchChar = QChar(0x306B); //HIRAGANA LETTER NI
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x306C); //Extension NU
               oKbdChar.qsSearchChar = QChar(0x306C); //HIRAGANA LETTER NU
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x306D); //Extension NE
               oKbdChar.qsSearchChar = QChar(0x306D); //HIRAGANA LETTER NE
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x306E); //Extension NO
               oKbdChar.qsSearchChar = QChar(0x306E); //HIRAGANA LETTER NO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End

         oKbdChar.qsKbdChar    = QChar(0x306F);
         oKbdChar.qsSearchChar = QChar(0x306F); //HIRAGANA LETTER HA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x3072); //Extension HI
               oKbdChar.qsSearchChar = QChar(0x3072); //HIRAGANA LETTER HI
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3075); //Extension HU
               oKbdChar.qsSearchChar = QChar(0x3075); //HIRAGANA LETTER HU
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3078); //Extension HE
               oKbdChar.qsSearchChar = QChar(0x3078); //HIRAGANA LETTER HE
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x307B); //Extension HO
               oKbdChar.qsSearchChar = QChar(0x307B); //HIRAGANA LETTER HO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End

         oKbdChar.qsKbdChar    = QChar(0x307E);
         oKbdChar.qsSearchChar = QChar(0x307E); //HIRAGANA LETTER MA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x307F); //Extension MI
               oKbdChar.qsSearchChar = QChar(0x307F); //HIRAGANA LETTER MI
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3080); //Extension MU
               oKbdChar.qsSearchChar = QChar(0x3080); //HIRAGANA LETTER MU
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3081); //Extension ME
               oKbdChar.qsSearchChar = QChar(0x3081); //HIRAGANA LETTER ME
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3082); //Extension MO
               oKbdChar.qsSearchChar = QChar(0x3082); //HIRAGANA LETTER MO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End

         oKbdChar.qsKbdChar    = QChar(0x3084);
         oKbdChar.qsSearchChar = QChar(0x3083); //HIRAGANA LETTER SMALL YA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x3086); //Extension YU
               oKbdChar.qsSearchChar = QChar(0x3085); //HIRAGANA LETTER SMALL YU
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3088); //Extension YO
               oKbdChar.qsSearchChar = QChar(0x3087); //HIRAGANA LETTER SMALL YO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End

         oKbdChar.qsKbdChar    = QChar(0x3089);
         oKbdChar.qsSearchChar = QChar(0x3089); //HIRAGANA LETTER RA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               oKbdChar.qsKbdChar    = QChar(0x308A); //Extension RI
               oKbdChar.qsSearchChar = QChar(0x308A); //HIRAGANA LETTER RI
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x308B); //Extension RU
               oKbdChar.qsSearchChar = QChar(0x308B); //HIRAGANA LETTER RU
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x308C); //Extension RE
               oKbdChar.qsSearchChar = QChar(0x308C); //HIRAGANA LETTER RE
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x308D); //Extension RO
               oKbdChar.qsSearchChar = QChar(0x308D); //HIRAGANA LETTER RO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End

         oKbdChar.qsKbdChar    = QChar(0x308F);
         oKbdChar.qsSearchChar = QChar(0x308F); //HIRAGANA LETTER WA
         rfoNonLatinSearchKbdList.append(oKbdChar);

               //---Extension Block Start
               // Note: The "potential" extension charachters HIRAGANA LETTER WI and HIRAGANA LETTER WE
               //       are not added here as they are old Hiragana letters and rarely used in current times.
               //       This is agreed with HMI.
               // Side-Note: Activation of WI and WE currently would cause a reset in phone hmi anyway.
               //
               // oKbdChar.qsKbdChar    = QChar(0x3090); //Extension WI
               // oKbdChar.qsSearchChar = QChar(0x3090); //HIRAGANA LETTER WI
               // rfoNonLatinSearchKbdList.append(oKbdChar);
               //
               // oKbdChar.qsKbdChar    = QChar(0x3091); //Extension WE
               // oKbdChar.qsSearchChar = QChar(0x3091); //HIRAGANA LETTER WE
               // rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3092); //Extension WO
               oKbdChar.qsSearchChar = QChar(0x3092); //HIRAGANA LETTER WO
               rfoNonLatinSearchKbdList.append(oKbdChar);
               
               oKbdChar.qsKbdChar    = QChar(0x3093); //Extension N
               oKbdChar.qsSearchChar = QChar(0x3093); //HIRAGANA LETTER N
               rfoNonLatinSearchKbdList.append(oKbdChar);
               //---Extension Block End
      }
      break;

      case RUSSIAN:  // (Cyrillic)
      {
         bNonLatinAvailable = true;
               
         oKbdChar.qsKbdChar    = "#";
         oKbdChar.qsSearchChar = " ";
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0410);
         oKbdChar.qsSearchChar = QChar(0x0430);  // CYRILLIC SMALL LETTER A
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0411);
         oKbdChar.qsSearchChar = QChar(0x0431);  // CYRILLIC SMALL LETTER BE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0412);
         oKbdChar.qsSearchChar = QChar(0x0432);  // CYRILLIC SMALL LETTER VE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0413);
         oKbdChar.qsSearchChar = QChar(0x0433);  // CYRILLIC SMALL LETTER GHE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0414);
         oKbdChar.qsSearchChar = QChar(0x0434);  // CYRILLIC SMALL LETTER DE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0415);
         oKbdChar.qsSearchChar = QChar(0x0435);  // CYRILLIC SMALL LETTER IE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0416);
         oKbdChar.qsSearchChar = QChar(0x0436);  // CYRILLIC SMALL LETTER ZHE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0417);
         oKbdChar.qsSearchChar = QChar(0x0437);  // CYRILLIC SMALL LETTER ZE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0418);
         oKbdChar.qsSearchChar = QChar(0x0438);  // CYRILLIC SMALL LETTER I
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x041A);
         oKbdChar.qsSearchChar = QChar(0x043A);  // CYRILLIC SMALL LETTER KA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x041B);
         oKbdChar.qsSearchChar = QChar(0x043B);  // CYRILLIC SMALL LETTER EL
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x041C);
         oKbdChar.qsSearchChar = QChar(0x043C);  // CYRILLIC SMALL LETTER EM
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x041D);
         oKbdChar.qsSearchChar = QChar(0x043D);  // CYRILLIC SMALL LETTER EN
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x041E);
         oKbdChar.qsSearchChar = QChar(0x043E);  // CYRILLIC SMALL LETTER O
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x041F);
         oKbdChar.qsSearchChar = QChar(0x043F);  // CYRILLIC SMALL LETTER PE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0420);
         oKbdChar.qsSearchChar = QChar(0x0440);  // CYRILLIC SMALL LETTER ER
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0421);
         oKbdChar.qsSearchChar = QChar(0x0441);  // CYRILLIC SMALL LETTER ES
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0422);
         oKbdChar.qsSearchChar = QChar(0x0442);  // CYRILLIC SMALL LETTER TE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0423);
         oKbdChar.qsSearchChar = QChar(0x0443);  // CYRILLIC SMALL LETTER U
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0424);
         oKbdChar.qsSearchChar = QChar(0x0444);  // CYRILLIC SMALL LETTER EF
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0425);
         oKbdChar.qsSearchChar = QChar(0x0445);  // CYRILLIC SMALL LETTER HA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0426);
         oKbdChar.qsSearchChar = QChar(0x0446);  // CYRILLIC SMALL LETTER TSE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0427);
         oKbdChar.qsSearchChar = QChar(0x0447);  // CYRILLIC SMALL LETTER CHE
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0428);
         oKbdChar.qsSearchChar = QChar(0x0448);  // CYRILLIC SMALL LETTER SHA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0429);
         oKbdChar.qsSearchChar = QChar(0x0449);  // CYRILLIC SMALL LETTER SHCHA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x042D);
         oKbdChar.qsSearchChar = QChar(0x044D);  // CYRILLIC SMALL LETTER E
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x042E);
         oKbdChar.qsSearchChar = QChar(0x044E);  // CYRILLIC SMALL LETTER YU
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x042F);
         oKbdChar.qsSearchChar = QChar(0x044F);  // CYRILLIC SMALL LETTER YA
         rfoNonLatinSearchKbdList.append(oKbdChar);
      }
      break;

      case THAI: 
      {
         bNonLatinAvailable = true;
               
         oKbdChar.qsKbdChar    = "#";
         oKbdChar.qsSearchChar = " ";
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E01);
         oKbdChar.qsSearchChar = QChar(0x0E01);  // THAI CHARACTER Ko Kai
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E02);
         oKbdChar.qsSearchChar = QChar(0x0E02);  // THAI CHARACTER Kho Khai
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E08);
         oKbdChar.qsSearchChar = QChar(0x0E08);  // THAI CHARACTER Cho Chan
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E0A);
         oKbdChar.qsSearchChar = QChar(0x0E0A);  // THAI CHARACTER Cho Chang
         rfoNonLatinSearchKbdList.append(oKbdChar);
         
         oKbdChar.qsKbdChar    = QChar(0x0E13);
         oKbdChar.qsSearchChar = QChar(0x0E13);  // THAI CHARACTER No Nen
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E14);
         oKbdChar.qsSearchChar = QChar(0x0E14);  // THAI CHARACTER Do Dek
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E17);
         oKbdChar.qsSearchChar = QChar(0x0E17);  // THAI CHARACTER Tho Thahan
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E19);
         oKbdChar.qsSearchChar = QChar(0x0E19);  // THAI CHARACTER No Nu
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E1A);
         oKbdChar.qsSearchChar = QChar(0x0E1A);  // THAI CHARACTER Bo Baimai
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E1E);
         oKbdChar.qsSearchChar = QChar(0x0E1E);  // THAI CHARACTER Pho Phan
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E21);
         oKbdChar.qsSearchChar = QChar(0x0E21);  // THAI CHARACTER Mo Ma
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E22);
         oKbdChar.qsSearchChar = QChar(0x0E22);  // THAI CHARACTER Yo Yak
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E23);
         oKbdChar.qsSearchChar = QChar(0x0E23);  // THAI CHARACTER Ro Rua
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E27);
         oKbdChar.qsSearchChar = QChar(0x0E27);  // THAI CHARACTER Wo Waen
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E28);
         oKbdChar.qsSearchChar = QChar(0x0E28);  // THAI CHARACTER So Sala
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E2A);
         oKbdChar.qsSearchChar = QChar(0x0E2A);  // THAI CHARACTER So Sua
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0E2D);
         oKbdChar.qsSearchChar = QChar(0x0E2D);  // THAI CHARACTER O Ang
         rfoNonLatinSearchKbdList.append(oKbdChar); 
      }
      break;
          
      case ARABIC:
      {
         bNonLatinAvailable = true;
               
         oKbdChar.qsKbdChar    = "#";
         oKbdChar.qsSearchChar = " ";
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0627);
         oKbdChar.qsSearchChar = QChar(0x0623);  // ARABIC LETTER ALEF WITH HAMZA
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0628);
         oKbdChar.qsSearchChar = QChar(0x0628);  // ARABIC LETTER BEH
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x062A);
         oKbdChar.qsSearchChar = QChar(0x062A);  // ARABIC LETTER TEH
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x062B);
         oKbdChar.qsSearchChar = QChar(0x062B);  // ARABIC LETTER THEH
         rfoNonLatinSearchKbdList.append(oKbdChar);
         
         oKbdChar.qsKbdChar    = QChar(0x062C);
         oKbdChar.qsSearchChar = QChar(0x062C);  // ARABIC LETTER JEEM
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x062D);
         oKbdChar.qsSearchChar = QChar(0x062D);  // ARABIC LETTER HAH
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x062E);
         oKbdChar.qsSearchChar = QChar(0x062E);  // ARABIC LETTER KHAH
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x062F);
         oKbdChar.qsSearchChar = QChar(0x062F);  // ARABIC LETTER DAL
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0630);
         oKbdChar.qsSearchChar = QChar(0x0630);  // ARABIC LETTER THAL
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0631);
         oKbdChar.qsSearchChar = QChar(0x0631);  // ARABIC LETTER REH
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0632);
         oKbdChar.qsSearchChar = QChar(0x0632);  // ARABIC LETTER ZAIN
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0633);
         oKbdChar.qsSearchChar = QChar(0x0633);  // ARABIC LETTER SEEN
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0634);
         oKbdChar.qsSearchChar = QChar(0x0634);  // ARABIC LETTER SHEEN
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0635);
         oKbdChar.qsSearchChar = QChar(0x0635);  // ARABIC LETTER SAD
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0636);
         oKbdChar.qsSearchChar = QChar(0x0636);  // ARABIC LETTER DAD
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0637);
         oKbdChar.qsSearchChar = QChar(0x0637);  // ARABIC LETTER TAH
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0638);
         oKbdChar.qsSearchChar = QChar(0x0638);  // ARABIC LETTER ZAH
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0639);
         oKbdChar.qsSearchChar = QChar(0x0639);  // ARABIC LETTER AIN
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x063A);
         oKbdChar.qsSearchChar = QChar(0x063A);  // ARABIC LETTER GHAIN
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0641);
         oKbdChar.qsSearchChar = QChar(0x0641);  // ARABIC LETTER FEH
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0642);
         oKbdChar.qsSearchChar = QChar(0x0642);  // ARABIC LETTER QAF
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0643);
         oKbdChar.qsSearchChar = QChar(0x0643);  // ARABIC LETTER KAF
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0644);
         oKbdChar.qsSearchChar = QChar(0x0644);  // ARABIC LETTER LAM
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0645);
         oKbdChar.qsSearchChar = QChar(0x0645);  // ARABIC LETTER MEEM
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0646);
         oKbdChar.qsSearchChar = QChar(0x0646);  // ARABIC LETTER NOON
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x0647);
         oKbdChar.qsSearchChar = QChar(0x0647);  // ARABIC LETTER HEH
         rfoNonLatinSearchKbdList.append(oKbdChar);         

         oKbdChar.qsKbdChar    = QChar(0x0648);
         oKbdChar.qsSearchChar = QChar(0x0648);  // ARABIC LETTER WAW
         rfoNonLatinSearchKbdList.append(oKbdChar);   

         oKbdChar.qsKbdChar    = QChar(0x064A);
         oKbdChar.qsSearchChar = QChar(0x064A);  // ARABIC LETTER YEH
         rfoNonLatinSearchKbdList.append(oKbdChar);   
      }
      break;
      
      case KOREAN: // (Hangul)
      {
         bNonLatinAvailable = true;    
              
         oKbdChar.qsKbdChar    = "#";
         oKbdChar.qsSearchChar = " ";
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3131);  // HANGUL KIYEOK
         oKbdChar.qsSearchChar = QChar(0x3131);
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3134);  // HANGUL NIEUN
         oKbdChar.qsSearchChar = QChar(0x3134);
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3137);  // HANGUL TIKEUT
         oKbdChar.qsSearchChar = QChar(0x3137);
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3139);  // HANGUL RIEUL
         oKbdChar.qsSearchChar = QChar(0x3139);
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3141);  // HANGUL MIEUM
         oKbdChar.qsSearchChar = QChar(0x3141);
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3142);  // HANGUL PIEUP
         oKbdChar.qsSearchChar = QChar(0x3142);
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3145);  // HANGUL SIOS
         oKbdChar.qsSearchChar = QChar(0x3145);
         rfoNonLatinSearchKbdList.append(oKbdChar);

         oKbdChar.qsKbdChar    = QChar(0x3147);  // HANGUL IEUNG
         oKbdChar.qsSearchChar = QChar(0x3147);
         rfoNonLatinSearchKbdList.append(oKbdChar);
         
         oKbdChar.qsKbdChar    = QChar(0x3148);  // HANGUL CIEUC
         oKbdChar.qsSearchChar = QChar(0x3148);
         rfoNonLatinSearchKbdList.append(oKbdChar);         
         
         oKbdChar.qsKbdChar    = QChar(0x314A);  // HANGUL CHIEUCH
         oKbdChar.qsSearchChar = QChar(0x314A);
         rfoNonLatinSearchKbdList.append(oKbdChar);    

         oKbdChar.qsKbdChar    = QChar(0x314B);  // HANGUL KHIEUKH
         oKbdChar.qsSearchChar = QChar(0x314B);
         rfoNonLatinSearchKbdList.append(oKbdChar);    

         oKbdChar.qsKbdChar    = QChar(0x314C);  // HANGUL THIEUTH
         oKbdChar.qsSearchChar = QChar(0x314C);
         rfoNonLatinSearchKbdList.append(oKbdChar);   

         oKbdChar.qsKbdChar    = QChar(0x314D);  // HANGUL PHIEUPH
         oKbdChar.qsSearchChar = QChar(0x314D);
         rfoNonLatinSearchKbdList.append(oKbdChar); 

         oKbdChar.qsKbdChar    = QChar(0x314E);  // HANGUL HIEUH
         oKbdChar.qsSearchChar = QChar(0x314E);
         rfoNonLatinSearchKbdList.append(oKbdChar); 
      }
      break;
               
      case NA_ENGLISH:           //  0UL
      case GERMAN:               //  1UL
      case ITALIAN:              //  2UL
      case SWEDISH:              //  3UL
      case FRENCH:               //  4UL
      case SPANISH:              //  5UL
      case DUTCH:                //  6UL
      case PORTUGUESE:           //  7UL
      case NORWEGIAN:            //  8UL
      case FINNISH:              //  9UL
      case DANISH:               // 10UL
      // GREEK                      // 11UL, already handled individually
      // JAPANESE                   // 12UL, already handled individually
      // ARABIC                     // 13UL, already handled individually
      case STANDARD_CHINESE:     // 14UL
      case POLISH:               // 15UL
      case TURKISH:              // 16UL
      // KOREAN                     // 17UL, already handled individually
      case TRADITIONAL_CHINESE:  // 18UL
      case UK_ENGLISH:           // 19UL
      case HUNGARIAN:            // 20UL
      case CZECH:                // 21UL
      case SLOVAK:               // 22UL
      // RUSSIAN                    // 23UL, already handled individually 
      case BRAZILIAN_PORTUGUESE: // 24UL
      // THAI                       // 25UL,    TODO: handle individually    // <---- TODO: Check if a dedicated 'Non-Latin' Search Kbd is needed.
      case BULGARIAN:            // 26UL
      case ROMANIAN:             // 27UL
      case SLOVENIAN:            // 28UL
      case CROATIAN:             // 29UL
      case UKRAINIAN:            // 30UL
      case NA_FRENCH:            // 31UL
      case NA_SPANISH:           // 32UL
      case CANTONESE:            // 33UL
      case UZBEK:            // 34UL
      case SERBIAN:            // 35UL
      case LITHUANIAN:            // 36UL
      case ESTONIAN:            // 37UL
      case LATVIAN:            // 38UL
      case ALBANESE:            // 39UL
      case CHINESE_MANDARIN_SIMPLIFIED:            // 40UL
      case ENGLISH_AUS:            // 41UL
      case ENGLISH_JPN:            // 42UL
      case ENGLISH_PRC:            // 43UL
      case ENGLISH_CANADIAN:            // 44UL
      case ENGLISH_INDIA:            // 45UL
      case FILIPINO:            // 46UL
      case FLEMISH:            // 47UL
      case FRENCH_CANADIAN:            // 48UL
      case HEBREW:            // 49UL
      case HINDI:            // 50UL
      case INDONESIAN:            // 51UL
      case MALAY:            // 52UL
      case PERSIAN:            // 53UL
      case SPANISH_LATIN:            // 54UL
      case VIETNAMESE:            // 55UL
      {
         bNonLatinAvailable = false;  
         // These languages are treated as Latin languges.
         // No action needed here as the 'rfoLatinSearchKbdList' is filled in the begin of this function anyway.
      }
      break;
               
      default:
      {
         // We never expect to come here. 
         // If we come here, we probably have forgotten to implement a 'rfoNonLatinSearchKbdList' for a dedicated language and have to add it.

         bNonLatinAvailable = false;  // Fail-Safe setting.
                  
         VKBD_FPRINTF(" -VKBD- ::bInitializeSearchKeyboardLists : FATAL ERROR: No Non-Latin Search Keyboard created. Language unknown. " );   
         // NORMAL_M_ASSERT_ALWAYS(); // A NORMAL ASSERT would be helpfull here, but is not supported for this lib (compiler error) ;-)
      }
      
   }  // end of: switch( u32language )

   VKBD_FPRINTF(" -VKBD- oLatinSearchKbdList   rfoNonLatinSearchKbdList.length()= %d ", rfoNonLatinSearchKbdList.length() );
   
   // Retrieve the sorting prio for the search chars of the Search Keyboard from Sorting Lib
   if( bNonLatinAvailable == true )
   {
      for( quint16 i=0; i<rfoNonLatinSearchKbdList.length(); i++ )
      {
         rfoNonLatinSearchKbdList[i].u32SearchCharPrio = sortlib_get_char_sort_prio( rfoNonLatinSearchKbdList[i].qsSearchChar.at(0).unicode() );
         VKBD_FPRINTF(" -VKBD- VerticalSearchKeyboard::bInitializeSearchKeyboardLists: rfoLatinSearchKbdList[%02d].u32SearchCharPrio= %d . ", i, rfoNonLatinSearchKbdList[i].u32SearchCharPrio);   
      }
   }
 
   VKBD_FPRINTF(" -VKBD- oLatinSearchKbdList    now filled with %d elements. ", rfoLatinSearchKbdList.size() );  
   VKBD_FPRINTF(" -VKBD- oNonLatinSearchKbdList now filled with %d elements. ", rfoNonLatinSearchKbdList.size() );    
      
   return bNonLatinAvailable;
}   

