/*******************************************************************************
*                                                                               
* FILE:          FC_PhoneBook_main.cpp                                   
*                                                                               
* SW-COMPONENT:  FC_PhoneBook application                                  
*                                                                               
* PROJECT:                                                                      
*                                                                               
* DESCRIPTION:   CCA Application.                                               
*                                                                               
* AUTHOR:        Arun Choudhary                                                               
*                                                                               
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim                          
*                                                                               
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "FC_PhoneBook_main.h"
#include "FC_PhoneBook_Debug.h"
#include "FC_PhoneBook_clienthandler_BluetoothSetings.h"
#include "FC_PhoneBook_service_PhoneBook.h"
#include  "Database/FC_PhoneBook_List.h"
#include  "Database/FC_PhoneBook_Database.h"
#include "FC_PhoneBook_clienthandler_Phone.h"
#include "FC_PhoneBook_FavoritesProvider.h"
#include "FC_PhoneBook_clienthandler_DefSetHandler.h"
#include "Qt_except.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_APPLICATION
#include "trcGenProj/Header/FC_PhoneBook_main.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/
#ifndef CCX_DEMO_CLIENT
static GMainLoop* PhBkmainloop = NULL;
#endif


/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/

// Static self reference to be used in function callbacks.
fc_phonebook_tclApp* fc_phonebook_tclApp::m_poMainAppInstance = NULL;

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclApp* fc_phonebook_tclApp::theServer()
*
* DESCRIPTION: The Static function returns the Object of this class.
*
* PARAMETER: None.
*
* RETURNVALUE: fc_phonebook_tclApp*
*
********************************************************************************/

fc_phonebook_tclApp* fc_phonebook_tclApp::theServer()
{

// CMG3G-7719: The following lines for installing Message Handler is added to address the Prio2 LINT warning - Warning 528.
#if QT_VERSION >= 0x050000
   qInstallMessageHandler(emOutput);
#else
   qInstallMsgHandler(emOutput);
#endif
// End of LINT fix

   if(m_poMainAppInstance == NULL)
   {
      m_poMainAppInstance = OSAL_NEW fc_phonebook_tclApp();
   }
   return m_poMainAppInstance;
}
#endif //CONNECTIVITY_PROCESS_STARTUP_GEN3

/*******************************************************************************
*
* FUNCTION: ahl_fc_phonebook_tclApp::ahl_fc_phonebook_tclApp()
*
* DESCRIPTION: Constructor.
*
*              Initialize static self reference to be used in function
*              callbacks.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phonebook_tclApp::fc_phonebook_tclApp():ahl_tclBaseOneThreadApp()
{
   ETG_TRACE_USR4(("fc_phonebook_tclApp() entered."));
   fc_phonebook_tclApp::m_poMainAppInstance = this;
   /* CMG3G-14313-Coverity Fix*/
   m_poClientHandlerBluetoothSetings = NULLPTR;
   m_poClientHandlerStreamRouter = NULLPTR;
   /* end of Fix */
   m_poClientHandlerPhone = NULLPTR;
   m_poPhoneBook = NULLPTR;
   m_poDefSet = NULLPTR;
   m_poDiaglibService = NULLPTR;
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
   m_hRxThreadId = OSAL_NULL;
#endif
   

}


#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
/*******************************************************************************
* FUNCTION: tVoid fc_phone_tclApp::vInitializeThread()
*
* DESCRIPTION: Initializing a Separate  thread.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*******************************************************************************/
tVoid fc_phonebook_tclApp::vInitializeThread()
{
   tU32 u32ThreadPrio = 0;
   tU32 u32StackSize = 0;

   OSAL_trThreadAttribute rAttr;
   tC8 szThreadName[OSAL_C_U32_MAX_NAMELENGTH] = "\0";

   // Note:Maximum size of thread name limited to 8
   tString tStrTemp = OSAL_szStringCopy( szThreadName, FC_PHBK_THREAD_ID);
   tString tStrDummy = tStrTemp;//Added to remove lint warning.
   tStrTemp = tStrDummy;

   u32ThreadPrio = FC_PHBK_DEFAULT_PRIORITY;
   u32StackSize = FC_PHBK_DEFAULT_STACK_SIZE;

   // Initialize thread parameters
   rAttr.szName = szThreadName;
   rAttr.s32StackSize = (tS32)u32StackSize;
   rAttr.u32Priority = u32ThreadPrio;
   rAttr.pfEntry = (OSAL_tpfThreadEntry)fc_phonebook_tclApp::vStartMainLoop;
   rAttr.pvArg = (tPVoid)this;

   /* Create thread using pthread or OSAL thread creation process*/

   m_hRxThreadId = OSAL_ThreadSpawn(&rAttr);

   if ( m_hRxThreadId == OSAL_ERROR )
   {
      ETG_TRACE_USR4(("fc_phonebook_tclApp::vInitializeThread() -> the thread NOT create \n "));
   }
   else
   {
      ETG_TRACE_USR4(("fc_phonebook_tclApp::vInitializeThread() -> Thread creation successful \n"));
      /*Map call back function as vStartMainLoop*/
   }
}

/*******************************************************************************
 * FUNCTION    : name_appeared_system
 *
 * DESCRIPTION : Gets called when Evolution stack system bus name appeared
 *
 * PARAMETER   : connection - GDBUS connection
 *               name - Name of the bus
 *               name_owner - Owner of the bus
 *               userdata - User data if any
 *
 * RETURNVALUE : None.
 *******************************************************************************/
static void name_appeared_system(GDBusConnection *systemBusConnection, const gchar *name, const gchar *name_owner, gpointer userdata)
{
   ETG_TRACE_USR4(("name_appeared_system entered"));
   (tVoid) name;
   (tVoid) name_owner;
   (tVoid) userdata;

   if (systemBusConnection)
   {
      ETG_TRACE_FATAL(("Got System bus Connection"));
      //Assigning system bus for proxy creation
      AssignSystemBusConnection(systemBusConnection);
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::GDBusConnection is null"));
      return;
   }
}

/*******************************************************************************
 * FUNCTION    : name_vanished_system
 *
 * DESCRIPTION : Gets called when Evolution stack system bus name vanished
 *
 * PARAMETER   : connection - GDBUS connection
 *               name - Name of the bus
 *               user_data - User data if any
 *
 * RETURNVALUE : None.
 *******************************************************************************/
static void name_vanished_system(GDBusConnection *connection, const gchar *name, gpointer user_data)
{

   ETG_TRACE_USR4(("name_vanished_system entered"));
   (tVoid) connection;
   (tVoid) name;
   (tVoid) user_data;

   RemoveSystemBusConnection();

}

/*******************************************************************************
 * FUNCTION    : name_appeared_session
 *
 * DESCRIPTION : Gets called when Evolution stack session bus name appeared
 *
 * PARAMETER   : connection - GDBUS connection
 *               name - Name of the bus
 *               name_owner - Owner of the bus
 *               userdata - User data if any
 *
 * RETURNVALUE : None.
 *******************************************************************************/
static void name_appeared_session(GDBusConnection *sessionBusConnection, const gchar *name, const gchar *name_owner, gpointer userdata)
{
   ETG_TRACE_USR4(("name_appeared_session entered"));
   (tVoid) name;
   (tVoid) name_owner;
   (tVoid) userdata;

   if (sessionBusConnection)
   {
      ETG_TRACE_FATAL(("Got Session bus Connection"));
      //Assigning session bus for proxy creation
      AssignSessionBusConnection(sessionBusConnection);
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::GDBusConnection is null"));
      return;
   }
}

/*******************************************************************************
 * FUNCTION    : name_vanished_session
 *
 * DESCRIPTION : Gets called when Evolution stack session bus name vanished
 *
 * PARAMETER   : connection - GDBUS connection
 *               name - Name of the bus
 *               user_data - User data if any
 *
 * RETURNVALUE : None.
 *******************************************************************************/
static void name_vanished_session(GDBusConnection *connection, const gchar *name, gpointer user_data)
{

   ETG_TRACE_USR4(("name_vanished_session entered"));
   (tVoid) connection;
   (tVoid) name;
   (tVoid) user_data;

   RemoveSessionBusConnection();

}

/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclApp::vStartMainLoop()
*
* DESCRIPTION: Running the main loop in thread
*
* PARAMETER: None.
*
* RETURNVALUE: None

*******************************************************************************/
tVoid fc_phonebook_tclApp::vStartMainLoop()    // LINT Fix -Info 715: Symbol 'pvArg' not referenced
{
   ETG_TRACE_USR4(("vStartMainLoop"));
#ifndef CCX_DEMO_CLIENT
   // Create the main loop instance, uses default context
   PhBkmainloop = g_main_loop_new(NULLPTR, FALSE);

   if (!PhBkmainloop)
   {
      ETG_TRACE_ERR(("g_main_loop_new fails"));
   }

   ETG_TRACE_USR4(("Watching BT stack bus connection"));
   guint ownId;
   ownId = g_bus_watch_name(G_BUS_TYPE_SYSTEM, //Session bus for org.bluez.obex
         "org.bluez", //Bus name
         G_BUS_NAME_WATCHER_FLAGS_AUTO_START,
         name_appeared_system,
         name_vanished_system,
         NULLPTR,
         NULLPTR);

   ETG_TRACE_USR4(("System ownId = %d",ownId));

   ownId = g_bus_watch_name(G_BUS_TYPE_SESSION, //Session bus for org.bluez.obex
         "org.bluez.obex", //Bus name
         G_BUS_NAME_WATCHER_FLAGS_AUTO_START,
         name_appeared_session,
         name_vanished_session,
         NULLPTR,
         NULLPTR);

   ETG_TRACE_USR4(("Session ownId = %d",ownId));

   // Main loop
   g_main_loop_run(PhBkmainloop);


#endif
}
#endif //CONNECTIVITY_PROCESS_STARTUP_GEN3
/*******************************************************************************
*
* FUNCTION: ahl_fc_phonebook_tclApp::~ahl_fc_phonebook_tclApp()
*
* DESCRIPTION: Destructor.
*
*              Invalidate static self reference.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phonebook_tclApp::~fc_phonebook_tclApp()
{
   ETG_TRACE_USR4(("~fc_phonebook_tclApp() entered."));
   m_poMainAppInstance = NULLPTR;
   m_poClientHandlerBluetoothSetings = NULLPTR;
   m_poPhoneBook = NULLPTR;
   m_poClientHandlerPhone = NULLPTR;
   m_poClientHandlerStreamRouter = NULLPTR;
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
   OSAL_s32ThreadDelete(m_hRxThreadId);
   m_hRxThreadId = OSAL_NULL;
#endif //CONNECTIVITY_PROCESS_STARTUP_GEN3
   m_poDiaglibService = NULLPTR;
   m_poDefSet = NULLPTR;

}

/*******************************************************************************
*
* FUNCTION: tBool fc_phonebook_tclApp::bOnInit()
*
* DESCRIPTION: This function is called by the CCA framework before starting any
*              CCA message related communication action. As default this
*              function always returns TRUE. The user has to implement all his
*              local application initialization (create client and service
*              handler, create worker threads, allocate application memory,...).
*              If returns TRUE, initialization could be successfully performed
*              and framework will register application, else, a reset of the
*              system is forced immediately.
*
*              This function is the counterpart of vOnApplicationClose().
*
* PARAMETER: None.
*
* RETURNVALUE: TRUE  = Application successfully initialized.
*              FALSE = Application NOT successfully initialized.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::bOnInit().
*******************************************************************************/
tBool fc_phonebook_tclApp::bOnInit()
{
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
   vInitializeThread();
#endif //CONNECTIVITY_PROCESS_STARTUP_GEN3

   tBool bSuccess = TRUE;

   ETG_TRACE_USR4(("bOnInit() entered."));

   m_poClientHandlerBluetoothSetings
   = OSAL_NEW fc_phonebook_tclClientHandler_BluetoothSetings(this);

   if (!m_poClientHandlerBluetoothSetings)
   {
      bSuccess = FALSE;

      FATAL_M_ASSERT_ALWAYS();
   }

   m_poClientHandlerPhone= OSAL_NEW fc_phonebook_tclClientHandler_Phone(this);
   if (!m_poClientHandlerPhone)
   {
      bSuccess = FALSE;
      FATAL_M_ASSERT_ALWAYS();
   }
   //Note: ClientHandler_Phone should be initiated before service_phonebook

   m_poPhoneBook = fc_phonebook_tclService_PhoneBook::pGetInstance(this);

   if (!m_poPhoneBook)
   {
      bSuccess = FALSE;

      FATAL_M_ASSERT_ALWAYS();
   }
   else
   {
      m_poPhoneBook->vSetBTSettingsClientHandlerInstance(m_poClientHandlerBluetoothSetings);
      if (m_poClientHandlerBluetoothSetings != NULL)
      {
         m_poClientHandlerBluetoothSetings->vUpdate_ClientHandler_PhoneBook(m_poPhoneBook);
      }
      else
      {
         bSuccess = FALSE;
         FATAL_M_ASSERT_ALWAYS();
      }
      m_poPhoneBook->poGetFavoritesProviderInstance()-> RegisterPhoneClientHandlerInstance(m_poClientHandlerPhone);

      //Fix NCG3D-67384
      FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
      if (poFC_PhoneBook_SQLite != NULL)
      {
         poFC_PhoneBook_SQLite->SetCheckTableSchema(TRUE);
      }
      //End fix NCG3D-67384
   }

   m_poDiaglibService = OSAL_NEW diaglib::tclServiceDiaglib(this, 1);
   if (m_poDiaglibService == NULL)
   {
      ETG_TRACE_FATAL(( "Creation Of fc_phonebook_Dialog Lib service  failed "));
   }

   m_poDefSet = OSAL_NEW Phonebook_DefSetHandler(m_poDiaglibService);
   if (!m_poDefSet)
   {
      bSuccess = FALSE;
      ETG_TRACE_FATAL(( "Creation fc_phonebook_DefSet HAndler failed"));
   }
   return bSuccess;
}

/**
 * TODO: workaround for NCG3D-67384
 * Response time from phonebook is more than SPMs  timeout value during vOnLoadSettings(). So SPM triggers fc_phonebook shutdown.
 * To avoid that we are Overriding the (virtual) base class method "vOnNewAppState which will increase the timeout value in SPM.
 */
tVoid fc_phonebook_tclApp::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
{
   ETG_TRACE_USR1(("fc_bluetooth_tclApp : vOnNewAppState entered - u32OldAppState:%d u32AppState-%d", u32OldAppState, u32AppState));

   if ( _tclLoadPair.bIfPairIncluded(u32OldAppState, u32AppState) == TRUE)
   {
       vAppStateChanged (u32AppState, 20000);
   }
   else if (_tclSavePair.bIfPairIncluded(u32OldAppState, u32AppState) == TRUE)
   {
       vAppStateChanged (u32AppState, 10000);
   }

   ahl_tclBaseOneThreadApp::vOnNewAppState(u32OldAppState, u32AppState);
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclApp::vOnApplicationClose()
*
* DESCRIPTION: This function is called by the CCA framework to indicate the
*              imminent shutdown of the application. The user has to implement
*              the de-initialization of all his local application data (destroy
*              client and service handler, destroy worker threads, free
*              application memory, ... ). After this function call the
*              destructor of this application will be called.
*
*              This function is the counterpart of bOnInit().
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnApplicationClose().
*******************************************************************************/
tVoid fc_phonebook_tclApp::vOnApplicationClose()
{
   ETG_TRACE_USR4(("vOnApplicationClose() entered."));

   if (m_poPhoneBook)
   {
      m_poPhoneBook->vOnApplicationClose();
   }

   _BP_TRY_BEGIN
   {
      if (m_poClientHandlerBluetoothSetings != NULL)
      {
         OSAL_DELETE m_poClientHandlerBluetoothSetings;
         m_poClientHandlerBluetoothSetings = NULLPTR;
      }

      if (m_poPhoneBook != NULL)
      {
         fc_phonebook_tclService_PhoneBook::vDestroyPhonebookServiceInstance();
         m_poPhoneBook = NULLPTR;
      }

      if (m_poDiaglibService != NULL)
      {
         OSAL_DELETE m_poDiaglibService;
         m_poDiaglibService = NULLPTR;
      }

      if (m_poDefSet != NULL)
      {
         OSAL_DELETE m_poDefSet;
         m_poDefSet = NULLPTR;
      }
   }
   _BP_CATCH_ALL
   {}
   _BP_CATCH_END

   ahl_tclBaseOneThreadApp::vOnApplicationClose();
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclApp::vOnLoadSettings()
*
* DESCRIPTION: This function is called by the CCA framework to trigger the 
*              loading of last mode settings. The user has to load the
*              previously stored last mode settings from persistent memory.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnLoadSettings().
*******************************************************************************/
tVoid fc_phonebook_tclApp::vOnLoadSettings()
{
   // Start time measurement 
   struct timespec time1 = { 0, 0 };
   struct timespec time2 = { 0, 0 };
   struct timespec diff_time = { 0, 0 };
   clock_gettime(CLOCK_MONOTONIC, &time1);

 
   ETG_TRACE_FATAL ((" -PHBK_APP_STATE_CHANGE- vOnLoadSettings() entered. up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000 ));
   // EM trace added temporarily for tracking of PSARCCB-9855 and related issues:
   ETG_TRACE_ERRMEM((" -PHBK_APP_STATE_CHANGE- vOnLoadSettings() entered. up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000 ));

   if (m_poPhoneBook != NULL)
   {
      m_poPhoneBook->OnLoadSettings();
   }
   else
   {
      ETG_TRACE_ERRMEM((" m_poPhoneBook is NULL "));
   }

   // Evaluation of needed time
   clock_gettime(CLOCK_MONOTONIC, &time2);
   if ((time2.tv_nsec - time1.tv_nsec) < 0)
   {
      diff_time.tv_sec = time2.tv_sec - time1.tv_sec - 1;
      diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
   }
   else 
   {
      diff_time.tv_sec = time2.tv_sec - time1.tv_sec;
      diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
   }
   ETG_TRACE_FATAL ((" -PHBK_APP_STATE_CHANGE- vOnLoadSettings() exited, needed time= [%5d] msec ",
      diff_time.tv_sec * 1000 + diff_time.tv_nsec / 1000000));

   // EM trace added temporarily for tracking of PSARCC21-2206, PSARCC30-49 and related issues:
   ETG_TRACE_ERRMEM((" -PHBK_APP_STATE_CHANGE- vOnLoadSettings() exited, needed time= [%5lu] msec ",
      diff_time.tv_sec * 1000 + diff_time.tv_nsec / 1000000));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclApp::vOnSaveSettings()
*
* DESCRIPTION: This function is called by the CCA framework to trigger the 
*              storage of last mode settings. The user has to store the last
*              mode settings to persistent memory.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnSaveSettings().
*******************************************************************************/
tVoid fc_phonebook_tclApp::vOnSaveSettings()
{
   // Start time measurement 
   struct timespec time1     = {0, 0};
   struct timespec time2     = {0, 0};
   struct timespec diff_time = {0, 0};
   clock_gettime(CLOCK_MONOTONIC, &time1);

   ETG_TRACE_FATAL ((" -PHBK_APP_STATE_CHANGE- vOnSaveSettings() entered, up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000));
   // EM-Trace added temporarily for tracking of PSARCCB-9855 and related issues:
   ETG_TRACE_ERRMEM((" -PHBK_APP_STATE_CHANGE- vOnSaveSettings() entered, up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000));

   // FIX PSARCCB-9855 (06/2016): 
   // Activate the skipping of pending 'PbDlSignal'(s) and and request a 'sched_yield()' 
   // (for the callback 'PbDlData_Signalhandler(..)'.
   // - The aspect to "request a skipping of pending 'PbDlSignal'(s) is needed as the parsing of queued vCards make
   //   no sense when going to OFF. We want the get rid as fast as possible of the pending callbacks.
   // - The aspect to "request a 'sched_yield()'" is necessary, so that we can achieve the mutex 'm_mutex' here as 
   //   fast as possible. 
   //   -- This is needed to execute 'FC_PhoneBook_SQLite::GetSQLiteInstance()->CloseDatabase();' as early as possible,
   //      especially when a phonebook download is running and the dubs thread holds the 'm_mutex' for more than
   //      50% of its processing time and therefore makes it difficult for the entry thread to get scheduled at a time
   //      when the mutex is free.
   /*g_iRequestSchedYield = 1;
     g_iSkipDataSignals   = 1;*/

   // FIX PSARCCB-9855 (06/2016): 
   // We close the database here directly (from entry thread). As 'CloseDatabase()' requires the mutex 'm_mutex',
   // the following code line might wait slightly until the dbus thread has released the mutex.
   FC_PhoneBook_SQLite::GetSQLiteInstance()->CloseDatabase();

   // FIX PSARCCB-9855 (06/2016): 
   // After the entry thread has closed the database no 'sched_yield()' is needed in dbus callback 'PbDlData_Signalhandler(..)'
   // any longer and we withdraw this request, to give the dbus thread the chance to return all pending 'PbDlData_Signalhandler(..)' 
   // as fast as possible without processing (as 'g_iSkipDataSignals' is still set ;-)
   //g_iRequestSchedYield = 0;


   //FIX SUZUKI-20814 - Reset observed on varying the voltage (5.7V to 14V)
   if (m_poPhoneBook)
   {
      m_poPhoneBook->vHandleProxyDereferencing();
   }
   //End of fix

   // Evaluation of needed time
   clock_gettime(CLOCK_MONOTONIC, &time2);
   if( (time2.tv_nsec - time1.tv_nsec) < 0 ) 
   {
      diff_time.tv_sec  = time2.tv_sec  - time1.tv_sec-1;
      diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
   } 
   else 
   {
      diff_time.tv_sec  = time2.tv_sec  - time1.tv_sec;
      diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
   }
   ETG_TRACE_FATAL (( " -PHBK_APP_STATE_CHANGE- vOnSaveSettings() exited, needed time= [%5lu] msec ",
         diff_time.tv_sec *1000 + diff_time.tv_nsec / 1000000 ));

   // EM trace added temporarily for tracking of PSARCCB-9855 and related issues:
   ETG_TRACE_ERRMEM(( " -PHBK_APP_STATE_CHANGE- vOnSaveSettings() exited, needed time= [%5lu] msec ",
         diff_time.tv_sec *1000 + diff_time.tv_nsec / 1000000 ));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclApp::vOnTimer(tU16 u16TimerId)
*
* DESCRIPTION: This function is called by the CCA framework on the expiration
*              of a previously via function bStartTimer() started timer. The
*              expired timer is forwarded to the respective service or
*              client-handler via a call of vProcessTimer().
*
*              The function is called from this applications context and
*              therefore no interthread programming rules must be considered
*              and the application methods and/or member variables can be
*              accessed without using the static self reference
*              'm_poMainAppInstance'.
*
* PARAMETER: [IN] u16TimerId = Identifier of the expired timer.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnTimer().
*******************************************************************************/
tVoid fc_phonebook_tclApp::vOnTimer(tU16 u16TimerId)
{
   ETG_TRACE_USR4(("vOnTimer() entered."));

   switch (u16TimerId)
   {
      case PHONEBOOK_TIMER_ID_1:
      case PHONEBOOK_TIMER_ID_2:
      {
         if (m_poPhoneBook)
         {
            //Fix RTC 294749
            m_poPhoneBook->m_oPbk_DnlManager.PhoneBook_vTimerExpired(u16TimerId);
         }
         else
         {
            NORMAL_M_ASSERT_ALWAYS();
         }
      }
      break;

      default:
      {
      }
   }
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclApp::vOnLoopback
*                 (tU16 u16ServiceID, amt_tclServiceData* poMessage)
*
* DESCRIPTION: This function is called by the CCA framework if a message from
*              one if this applications services is received (so called self
*              or loopback messages). The message is forwarded to the respective
*              service via a call of vOnLoopbackService().
*
* PARAMETER: [IN] u16ServiceID = Identifier of the service from where the
*                                message was sent.
*
*            [IN] poMessage = Message object.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnLoopback().
*******************************************************************************/
tVoid fc_phonebook_tclApp::vOnLoopback(tU16 u16ServiceID, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" vOnLoopback() entered."));

   switch (u16ServiceID)
   {
      case CCA_C_U16_SRV_FB_PHONEBOOK:
      {
         if (m_poPhoneBook)
         {
            m_poPhoneBook->vOnLoopbackService(poMessage);
         }
         else
         {
            ETG_TRACE_FATAL((" m_poPhoneBook NULL!!!! "));
         }
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("vOnLoopback(). Unknown ServiceID %u.", u16ServiceID));
      }
   }
   ETG_TRACE_USR4((" vOnLoopback() exit "));
}

/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclApp::fc_phonebook_set_service (tBool on))
*
* DESCRIPTION: This function is used to enable/disable service of Phonebook
*
* PARAMETER: [IN] tBool on = Boolean variable to enable/disable Service 
*
* RETURNVALUE: None.
*
********************************************************************************/

tVoid fc_phonebook_tclApp::fc_phonebook_set_service(tBool on)
{
   m_poMainAppInstance->m_poPhoneBook->vSetPrivateServiceAvailableAllowed(on);
}
