/*******************************************************************************
*
* FILE:          FC_PhoneBook_Dnl_Manager.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:   
*
* AUTHOR:		 Balasubramani.V
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include <QtCore/QtCore>

#include "FC_PhoneBook_Debug.h"

#include "FC_PhoneBook_Dnl_Manager.h"
#include "FC_PhoneBook_Device_Details.h"

#include "FC_PhoneBook_service_PhoneBook.h"
#include "Database/FC_PhoneBook_Database.h"
#include "Database/FC_PhoneBook_ListChange_Handler.h"
#include "Database/FC_PhoneBook_List.h"
#include "Database/FC_PhoneBook_CallHistory_List.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_APPLICATION
#include "trcGenProj/Header/FC_PhoneBook_Dnl_Manager.cpp.trc.h"
#endif

static quint8 g_s32_download_index[MAX_DEVICE_CONNECTION_SUPPORTED];

QMap<QString, RecentCallInfo> RecentCallStatus::map;
QMap<QString, RecentCallInfo>::Iterator RecentCallStatus::it;
bool RecentCallStatus::bDownloadMissedCall = false;
bool RecentCallStatus::bDownloadIncomingCall = false;
bool RecentCallStatus::bDownloadOutgoingCall = false;

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Dnl_Manager
*
* DESCRIPTION:
*             
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
FC_PhoneBook_Dnl_Manager::FC_PhoneBook_Dnl_Manager()
{
   ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager entered"));

   //Dynamically Decides between fusion and Evolution stack interface.
  
   //CMG3G-14314 fix
      ETG_TRACE_USR4(("Evolution instance"));
      m_pDbus_Cca_ApiList = get_cca_dbus_interface_pb_Evo();
   //end of CMG3G-14314 Fix

   m_NoOfActiveCalls = 0;
   m_poPhonebookService = NULLPTR;
   m_u8Retry_Count = 0;
   m_u8GetCHRetryCount = 0;
   m_bContactDlNeeded = TRUE;
}

/*******************************************************************************
*
* FUNCTION:  ~FC_PhoneBook_Dnl_Manager
*
* DESCRIPTION:
*             
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
FC_PhoneBook_Dnl_Manager::~FC_PhoneBook_Dnl_Manager()
{
   m_poPhonebookService = NULLPTR;
   m_pDbus_Cca_ApiList = NULLPTR;
}


/*******************************************************************************
*
* FUNCTION:  vGetCapabilitiesCallback
*
* DESCRIPTION: Function triggered on GetCapabilities callback from dbus daemon
*             
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vGetCapabilitiesCallback(PhBk_Loopback_Data* LBData)
{
   ETG_TRACE_USR4(("PhoneBook_DlGetCapabilities_Callback entered"));

   FC_Device_Details *deviceObject =
         FC_Device_Details::deviceAddressToObjectInstance(LBData->ucDevAddr);
   quint8 l_u8ConnectionID;

   if (deviceObject != NULL)
   {
      l_u8ConnectionID = deviceObject->getConnectionID();

      if (!LBData->blFailed)
      {
         quint8 u8DnlType = LBData->ucDnlType;
         ETG_TRACE_USR4(("u8DnlType: 0x%X", u8DnlType));

         //Fix RTC 174206
         //Though, some projects are considering SIM contacts , we are checking whether device supports SIM contacts or not.
         //If device doesn't support SIM contacts at all, we better skip that in putting it in download array, so that
         //we can skip unnecessary downloads
         if((deviceObject->m_u8FAVSPDDwnldSupport & PBAP_SIM_DWNLD_SUPPORTED)
               != PBAP_SIM_DWNLD_SUPPORTED)
         {
            u8DnlType = u8DnlType & 0xFE;
         }
         //End fix RTC 174206

#ifdef PBDL_SKIP_SIM_LIST   //CMG3G-8364
         ETG_TRACE_USR4(("Updating SIM support to ZERO."));
         u8DnlType = u8DnlType & 0xFE;//SIM support is set to ZERO.
#endif

         ETG_TRACE_COMP((" -PBDL.S- PbDlGetCapabilities returned: bSuccess= %d, Phone's SupportedLists= 0x%X", !LBData->blFailed, u8DnlType));

         if (u8DnlType == PBAPNotSupported)
         {
            ETG_TRACE_USR4(("PBAP is not supported"));

            deviceObject->setDeviceCapabilities(u8DnlType);
            deviceObject->RetryCnt = 0;
            PhoneBook_FetchingCapabilitiesFailed(deviceObject);
         }
         else
         {
            deviceObject->setDeviceCapabilities(u8DnlType);
            deviceObject->RetryCnt = 0;

            //Called because of BUG ID 208420
            PhoneBook_FetchingCapabilitiesSucess(deviceObject);

            m_pDbus_Cca_ApiList->vPhoneBook_DnlPrepare(l_u8ConnectionID);
         }

         //Stop retry timer if it is active for the GetCapabilities
         PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_GET_CAPABILITIES, l_u8ConnectionID);

         //Updation of download state and automatic download list done here
         //As entry for the device is added to feature support table only after invocation of vUpdatePhoneBookFeatureSupport during GetCap return.
         if (m_poPhonebookService)
         {
            m_poPhonebookService->vPhoneBook_UpdateClientsDownloadOnOffList();
            m_poPhonebookService->vPhoneBook_UpdateClientsAutomaticContactDownloadList();
            m_poPhonebookService->vPhoneBook_UpdatePreferredSortOrderListStatus();
         }
      }
      else
      {
         ETG_TRACE_COMP((" -PBDL.S- PbDlGetCapabilities returned with error: (bSuccess= %d), Error code= %d", !LBData->blFailed, LBData->ucErrorCode));

         if ((PhoneBook_blIsRetryActive(l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE))
               && (!PhoneBook_blIsRetryActive(l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_GET_CAPABILITIES)))
         {
            ETG_TRACE_USR4(("Retrying already active for another message"));
         }
         else
         {
            if (m_u8Retry_Count < NO_OF_RETRIES)
            {
               PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_GET_CAPABILITIES, l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_TYPE_INCREMENTAL);
               ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vGetCapabilitiesCallback : Retry Count: %d", m_u8Retry_Count));
            }
            else
            {
               ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vGetCapabilitiesCallback Max No. of retries reached. Hence stopping retry. "));
               PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_GET_CAPABILITIES, l_u8ConnectionID);

               ETG_TRACE_USR4((" Updating Capabilities as PBAP not supported "));
               deviceObject->setDeviceCapabilities(PBAPNotSupported);
               deviceObject->RetryCnt = 0;
               PhoneBook_FetchingCapabilitiesFailed(deviceObject);

               //Updation of download state and automatic download list done here
               //As entry for the device is added to feature support table only after invocation of vUpdatePhoneBookFeatureSupport during GetCap return.
               if (m_poPhonebookService)
               {
                  m_poPhonebookService->vPhoneBook_UpdateClientsDownloadOnOffList();
                  m_poPhonebookService->vPhoneBook_UpdateClientsAutomaticContactDownloadList();
               }
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL "));
   }
}


/*******************************************************************************
*
* FUNCTION:  vPrepareCallback
*
* DESCRIPTION: Function triggered on Prepare callback from dbus daemon
*             
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vPrepareCallback(PhBk_Loopback_Data* LBData)
{
   ETG_TRACE_USR4(("PhoneBook_Prepare_Callback entered"));
   if(LBData == NULL)
   {
      ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      return;
   }

   quint8 u8DnlType = LBData->ucDnlType;

   FC_Device_Details *deviceObject=
      FC_Device_Details::deviceAddressToObjectInstance(LBData->ucDevAddr);
   if(deviceObject != NULL)
   {
      //Fix RTC 174206
      //Though, some projects are considering SIM contacts , we are checking whether device supports SIM contacts or not.
      //If device doesn't support SIM contacts at all, we better skip that in putting it in download array, so that
      //we can skip unnecessary downloads
      if ((deviceObject->m_u8FAVSPDDwnldSupport & PBAP_SIM_DWNLD_SUPPORTED)
            != PBAP_SIM_DWNLD_SUPPORTED)
      {
         u8DnlType = u8DnlType & 0xFE;
      }
      //End fix RTC 174206

#ifdef PBDL_SKIP_SIM_LIST   //CMG3G-8364
      ETG_TRACE_USR4(("Updating SIM support to ZERO."));
      u8DnlType = u8DnlType & 0xFE;//SIM support is set to ZERO - Bit 0 is SIM support.
#endif

      ETG_TRACE_COMP((" -PBDL.S- PbDlPrepare has returned: bSuccess= %d, SupportedLists= 0x%X ('LBData->ucDnlType') ", (tU8) !LBData->blFailed, u8DnlType));

      quint8 l_u8ConnectionID;

      l_u8ConnectionID = deviceObject->getConnectionID();
      if (!LBData->blFailed)
      {
         ETG_TRACE_USR2(("PBDLCommand = %d", deviceObject->PBDLCommand));

         //FIX GMMY16-21009 Contacts and Recent calls list is shown blank after making an outgoing call from Recent calls list twice (with Samsung Galaxy S3 Android 4.1.2)
         //Only for SUZUKI, PB Support also needs to be sent as FALSE on receiving Prepare response as ZERO.
#ifdef UPDATE_BOTH_PBCHSUPPORT_ON_PREPARE_RESPONSEASZERO
         deviceObject->setDeviceCapabilities(u8DnlType);
#else
         /*FIX GMMY17-4548 When a paired and connected Samsung S5 is disconnected and connected again, CONTACT list is not updated in HMI but in IPC (in case BT access to contacts was not granted on S5 persistently)

          On fresh pairing, if the device does not support download of contacts
          � we could set PhoneBookSupport & ContactListSortView to false.
          Then clients will not trigger CreatePhonebookLists at all.
          On reconnection and subsequent call history updates, if Prepare fails, we would retain the initial PhoneBookSupport & ContactListSortView.*/

         if (FALSE == deviceObject->blIsItFirstTimeDownload() && u8DnlType == 0)//Received response from device is ZERO.
         {
            //In GM/JAC, when call history is updating and when user ignores/rejects the pop-up, PhoneBookSupport is not updated as ZERO.
            //Only CH support is updated as ZERO.
            ETG_TRACE_USR4(("Updating only CH support as ZERO. Retaining PB support."));
            deviceObject->setDeviceCapabilities(deviceObject->getDeviceCapabilities()
                  & 0x03);
         }
         else
         {
            deviceObject->setDeviceCapabilities(u8DnlType);
         }
#endif
         //End of fix GMMY16-21009

         //Update the download command based on the current device capabilities before starting download.
         deviceObject->PBDLCommand = deviceObject->PBDLCommand & u8DnlType;

         //FIX CMG3GB-847 When access is given to phonebook download after an outgoing call, only outgoing call entries are displayed and not combined call history list
         //If CCH is supported, then we have to download CCH, even when download list contains request only incoming/outgoing/missed
         //If user ignores/rejects pop up to download, then CCH Support will be FALSE.
         //Now on a call, we have to download the complete CH list. Else only the particular list will be available after download.
         if ((OutgoingCallHistory == deviceObject->PBDLCommand
               || MissedCallHistory == deviceObject->PBDLCommand
               || IncomingCallHistory == deviceObject->PBDLCommand))
         {
            if ((CombinedCallHistory & u8DnlType) == CombinedCallHistory)
            {
               ETG_TRACE_USR2(("Download state changed to CCH"));
               deviceObject->PBDLCommand = CombinedCallHistory;
            }
         }
         //End of fix

         ETG_TRACE_COMP(("  -PBDL.S- Supported PBDL_command= 0x%X ", deviceObject->PBDLCommand));

         //GMMY15-8739 Contacts are Displayed on the HMI, even though PBAP request was not accepted on the phone device.
         //If PB/CH details of the device is available in Database (due to previous connection and no deletion), since PBDL Prepare returns zero, it should be deleted from DB.
         if (deviceObject->PBDLCommand == 0)
         {
            //Since device does not allow preparing of phonebook set Prepare callback status to false
            deviceObject->setPhBkPrepareCallBackStatus(FALSE);
            ETG_TRACE_USR4(("Deleting Call history lists wrt the device as they are no longer valid"));
            if (FC_PhoneBook_CallHistory_List::GetCurrentInstance())
            {
               //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
               FC_PhoneBook_CallHistory_List::GetCurrentInstance()->bClearCHRecords(deviceObject->getConnectionID());
            }
            SendDummyListChange(deviceObject->getDeviceHandle(), getCurDownloadTypeForConnectionID(l_u8ConnectionID));// Changes for GMMY17-4549

            /*FIX GMMY17-4548 When a paired and connected Samsung S5 is disconnected and connected again, CONTACT list is not updated in HMI but in IPC (in case BT access to contacts was not granted on S5 persistently)*/
            //Sort view is updated after all data signals are received from DBus.
            //Since here download type is ZERO, no further download would happen. Hence update it here itself.
            if ( m_poPhonebookService && m_poPhonebookService->m_pPhoneBookDB)
            {
               tBool
                     b_result =
                           m_poPhonebookService->m_pPhoneBookDB->UpdateSortView(deviceObject->getDeviceHandle());
               deviceObject->setSortListViewSupport(b_result);
            }
            //End of fix
         }
         else
         {
            //Since device allows preparing of phonebook set Prepare callback status to true
            deviceObject->setPhBkPrepareCallBackStatus(TRUE);
         }
         //End of fix

         /*FIX GMMY17-4548 When a paired and connected Samsung S5 is disconnected and connected again, CONTACT list is not updated in HMI but in IPC (in case BT access to contacts was not granted on S5 persistently)*/
         //Device phonebook feature support is sent after PBDL Prepare response is received from device. 
         //Not after get capabilities response.
         if (m_poPhonebookService)
         {
            m_poPhonebookService->vUpdatePhoneBookFeatureSupport(deviceObject);
         }
         //End of fix

         //Stop retry timer if it is active for the Prepare
         PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_PREPARE, l_u8ConnectionID);

         g_s32_download_index[l_u8ConnectionID - 1] = 0;
         PhoneBook_findDownloadType(deviceObject, deviceObject->PBDLCommand);//PBDLCommand is the current request for download. Find the download types based on it.
         PhoneBook_PrintAllDownloadTypes(deviceObject);

         quint8 u8DnlIndex = g_s32_download_index[l_u8ConnectionID - 1];
         if (FC_Device_Details::DOWNLOAD_ST_CH_UPDATE
               != deviceObject->GetDownloadState())
         {
            if ((deviceObject->u8DownloadTypeList[u8DnlIndex]
                  == DownLoadPBLocal)
                  || (deviceObject->u8DownloadTypeList[u8DnlIndex]
                        == DownLoadPBSim)
                  || (deviceObject->u8DownloadTypeList[u8DnlIndex]
                        == DownloadPBSpeedDial)
                  || (deviceObject->u8DownloadTypeList[u8DnlIndex]
                        == DownloadPBFavorites))
            {
               deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS);

               ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_PBK_PROGRESS"));
            }
            else if ((deviceObject->u8DownloadTypeList[u8DnlIndex]
                  == DownloadCombinedCallHistorySimLocal)
                  || (deviceObject->u8DownloadTypeList[u8DnlIndex]
                        == DownloadIncomingCallHistorySimLocal)
                  || (deviceObject->u8DownloadTypeList[u8DnlIndex]
                        == DownloadOutGoingCallHistorySimLocal)
                  || (deviceObject->u8DownloadTypeList[u8DnlIndex]
                        == DownloadMissedCallHistorySimLocal))
            {
               deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS);

               ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_CH_PROGRESS"));
            }
         }

         if (deviceObject->PBDLCommand == 0) //CLONE TICKET-GMMY15-1747-1748:Individual call list is not getting updated(Chevy Impala)
         {
            m_pDbus_Cca_ApiList->vPhoneBook_DnlComplete(deviceObject->getConnectionID());
         }
         else
         {
            vStartDnlForTheGivenType(deviceObject, deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID
                  - 1]++]);
         }
      }
      else
      {
         ETG_TRACE_USR2(("Prepare callback failed. Error code- %d", LBData->ucErrorCode));

         if ((PhoneBook_blIsRetryActive(l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE))
               && (!PhoneBook_blIsRetryActive(l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_PREPARE)))
         {
            ETG_TRACE_USR4(("Retrying already active for another message"));
         }
         else
         {
            if (m_u8Retry_Count < NO_OF_RETRIES)
            {
               PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_PREPARE, l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_TYPE_INCREMENTAL);
               ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vPrepareCallback : Retry Count: %d", m_u8Retry_Count));
            }
            else
            {
               ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vPrepareCallback Max No. of retries reached. Hence stopping retry. "));
               PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_PREPARE, l_u8ConnectionID);
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL "));
   }
}


/*******************************************************************************
*
* FUNCTION:  vDnlStartCallback
*
* DESCRIPTION: Function triggered on Download start callback from dbus daemon
*             
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vDnlStartCallback(PhBk_Loopback_Data* LBData)
{
   ETG_TRACE_USR4(("PhoneBook_Start_Callback entered"));
   if(LBData == NULL)
   {
      ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      return;
   }

   //CMG3G-8447. Lint Fix.
   if (!m_poPhonebookService)
   {
      ETG_TRACE_USR4(("m_poPhonebookService is NULL. Hence returning."));
      return;
   }

   FC_Device_Details *deviceObject=
      FC_Device_Details::deviceAddressToObjectInstance(LBData->ucDevAddr);

   if(deviceObject != NULL)
   {
      quint8 l_u8ConnectionID= deviceObject->getConnectionID();

      if(!LBData->blFailed)
      {
         quint8 u8Status= LBData->ucStatus;

         ETG_TRACE_USR2(("Status- %d", u8Status));

         //CLONE TICKET FIX - GMMY15-1537 Missed call indicator display after reconnection of phone.
         deviceObject->m_newMissedcall = LBData->ucMissedCalls;
         const tU8 l_cu8InvalidMissedCallInfo = 0xFF;
         if((l_cu8InvalidMissedCallInfo != LBData->ucMissedCalls)
            && (LBData->ucMissedCalls > 0))
         {
            ETG_TRACE_USR4(("Missed Call::%u",LBData->ucMissedCalls));
            m_poPhonebookService->
               vMissedCallIndicatorUpdate((tU32)LBData->ucMissedCalls,deviceObject);
         }
         else
         {
            ETG_TRACE_USR2(("Missed Call information provided by Phone is invalid"));
         }

         //Stop retry timer if it is active for the Start PBDL
         PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_START_PBDL,
                              l_u8ConnectionID);
         //Stop the retry timer if already active for CH download
         CallHistory_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_CH_DOWNLOAD,
                                       l_u8ConnectionID);
         if(u8Status == PBDL_SUCCESS)
         {
            ETG_TRACE_USR4(("PBDL_SUCCESS"));

            quint8
                  u8phonebookType =
                        FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(deviceObject->getConnectionID());

            ETG_TRACE_USR4(("u8phonebookType: %d", u8phonebookType));
            if (DownLoadPBSim == u8phonebookType || DownLoadPBLocal
                  == u8phonebookType)
            {
               deviceObject->m_u32InidvidualTypeContactsCount
                     = LBData->u32PhoneNumberCount; //CMG3G-8363
               ETG_TRACE_USR2((" Total number of Contacts available in the connected device is: %u", deviceObject->m_u32InidvidualTypeContactsCount));
            }
            else
            {
               deviceObject->m_u32VCardsAvailableForDwnld = LBData->u32PhoneNumberCount;
               ETG_TRACE_USR2((" Total number of v cards available for next download is: %u", deviceObject->m_u32VCardsAvailableForDwnld));
            }

            //Dynamically Decides between fusion and Evolution stack interface.
            //Fix : CMG3G-12880
#ifdef HANDLE_PROBABLY_LOCKED_SCENARIO
               //If reconnection
               //NCG3D-63714
               if ((FC_PhoneBook_SQLite::GetSQLiteInstance())
                     && (TRUE == FC_PhoneBook_SQLite::GetSQLiteInstance()->bGetDlCompletedOnceValue(deviceObject->getDeviceHandle()))
                     && (0 != FC_PhoneBook_SQLite::GetSQLiteInstance()->GetTotalRecords(deviceObject->getDeviceHandle())))
               {
                  deviceObject->vSetSuspiciousGetSize(static_cast<tBool>((deviceObject->bGetSuspiciousGetSize())
                        & ( (DownLoadPBLocal == u8phonebookType && LBData->u32PhoneNumberCount <= 1)
                              || (DownLoadPBLocal != u8phonebookType && LBData->u32PhoneNumberCount == 0))));
                  ETG_TRACE_USR4((" Current value of deviceObject->bGetSuspiciousGetSize(): %d", deviceObject->bGetSuspiciousGetSize()));

                  vCheckWhetherProbablyLockedDevice(deviceObject->getDeviceHandle());

                  if(TRUE == deviceObject->bIsDeviceProbablyLocked())
                  {
                     if ( (DownLoadPBLocal == u8phonebookType && LBData->u32PhoneNumberCount > 1)
                           || (DownLoadPBLocal != u8phonebookType && LBData->u32PhoneNumberCount > 0))
                     {

                        ETG_TRACE_USR1((" -PBDL.S- Device unlocked by user... "));

                        //Update locked device list by removing the device handle from the list
                        //Device handle will be removed from probably locked devices if any of the lists is not empty
                        //i.e, (CCH,ICH,MCH,OCH,[FAV],[SPD]) > 0 and ME/PB > 1
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->vRemoveFromDeviceLockedList(deviceObject->getDeviceHandle());

                        //Since device unlocked,updating speech checksum and sending list change update of all types
                        FC_PhoneBook_SQLite::GetSQLiteInstance()->SpeechUpdateCheckSumTable(deviceObject->getDeviceHandle());
                        if(FC_PhoneBook_List::pGetInstance()->poGetHashTable())
                        {
                           ListChange_Handler oListChangeHandler;
                           oListChangeHandler.UpdateList(FC_PhoneBook_List::pGetInstance()->poGetHashTable(), enAction_Content_Refreshed,
                                 deviceObject->getDeviceHandle(), 0, FC_PhoneBook_SQLite::ListType_ALL);
                        }

                        //Stop timer
                        ETG_TRACE_USR4((" Stopping retry for download sequence as device is unlocked. "));
                        fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE, deviceObject->getConnectionID());

                        //If user unlocks while checking for pb, call history will be included in the download.
                        //Hence adding it to pending download list, there after it will download.
                        ETG_TRACE_USR4((" Adding to pending list, if user unlocks in the middle of either ch or pb is downloading "));
                        deviceObject->m_u8PendingRecentCallCommandList.append(DownloadCombinedCallHistorySimLocal);

                     }
                  }
               }
#endif
               if (0 != LBData->u32PhoneNumberCount)
               {
#ifdef HANDLE_PROBABLY_LOCKED_SCENARIO
                  if(FALSE == deviceObject->bIsDeviceProbablyLocked())
                  {
#endif
                  //For EVO stack, start Pull-All once PbDlStart callback is success
                  EvoPhonebook_Start_PullAll(deviceObject);
#ifdef HANDLE_PROBABLY_LOCKED_SCENARIO
                  }
                  else
                  {
                     //RTC_880047
                     if(DownLoadPBSim == u8phonebookType || DownLoadPBLocal == u8phonebookType)
                     {
                        ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO"));
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO);
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
                     }
                     else if(DownloadOutGoingCallHistorySimLocal == u8phonebookType || DownloadMissedCallHistorySimLocal == u8phonebookType
                           || DownloadIncomingCallHistorySimLocal == u8phonebookType || DownloadCombinedCallHistorySimLocal == u8phonebookType)
                     {
                        ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS"));
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS);
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
                     }
                     //Continue with further downloads
                     PhoneBook_DlStart_Failed(deviceObject, LBData);
                  }
                  //End fix : CMG3G-12880
#endif
               }
               else if (0 == LBData->u32PhoneNumberCount)
               {
                  //RTC_880047
                  if(DownLoadPBSim == u8phonebookType || DownLoadPBLocal == u8phonebookType)
                  {
                     ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO"));
                     (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO);
                     (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
                  }
                  else if(DownloadOutGoingCallHistorySimLocal == u8phonebookType || DownloadMissedCallHistorySimLocal == u8phonebookType
                        || DownloadIncomingCallHistorySimLocal == u8phonebookType || DownloadCombinedCallHistorySimLocal == u8phonebookType)
                  {
                     ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS"));
                     (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS);
                     (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
                  }
                  //Continue with further downloads on no v-cards to download
                  PhoneBook_DlStart_Failed(deviceObject,LBData);
               }
         }
         else
         {
            ETG_TRACE_USR4(("PBDL Failure"));

            //FIX CMG3GB-439 Download state API does not provide download failure / download complete / Memory limit reached.
            //Adding error case for PBDS and RCDS
            if( most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO == m_poPhonebookService->pGetDownloadStateInstance()->u8GetPBDownloadState() ||
               most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO == m_poPhonebookService->pGetDownloadStateInstance()->u8GetPBDownloadState() )
            {
               ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8PBDS_COMPLETE/FI_EN_E8PBDS_ERROR - Pbdl Complete failed"));
               updatePBErrorStatusToClients();
               PhoneBook_DlStart_Failed(deviceObject,LBData);
            }
            else if( most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS == m_poPhonebookService->pGetDownloadStateInstance()->u8GetRecentCallListDownloadState() )
            {
               ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8RCDS_COMPLETE/FI_EN_E8RCDS_ERROR - Pbdl Complete failed"));
               updateCHErrorStatusToClients();
            }
            //End of fix CMG3GB-439
         }
      }
      else
      {
         ETG_TRACE_USR2(("DnlStart callback failed. Error code- %d", LBData->ucErrorCode));

         if((PhoneBook_blIsRetryActive(l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE)) &&
            (!PhoneBook_blIsRetryActive(l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_START_PBDL)))
         {
            ETG_TRACE_USR4(("Retrying already active for another message"));
         }
         else
         {
            //NCG3D-103980
            if( (deviceObject->downloadType == DownLoadPBLocal) || (deviceObject->downloadType == DownLoadPBSim) )
            {
               if (m_u8Retry_Count < NO_OF_RETRIES)
               {
                  PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_START_PBDL, l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_TYPE_INCREMENTAL);
                  ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vDnlStartCallback : Retry Count: %d", m_u8Retry_Count));
               }
               else
               {
                  ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vDnlStartCallback Max No. of retries reached. Hence stopping retry. "));
                  PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_START_PBDL, l_u8ConnectionID);
               }//NCG3D-103980
            }
            else if(deviceObject->downloadType == DownloadOutGoingCallHistorySimLocal || deviceObject->downloadType == DownloadMissedCallHistorySimLocal
                    || deviceObject->downloadType == DownloadIncomingCallHistorySimLocal || deviceObject->downloadType == DownloadCombinedCallHistorySimLocal)
            {
               if (m_u8GetCHRetryCount < NO_OF_CH_RETRIES)
               {
                  CallHistory_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_CH_DOWNLOAD, l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_TYPE_CONSTANT);
                  ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vDnlStartCallback : CH Retry Count: %d", m_u8GetCHRetryCount));
               }
               else
               {
                  ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vDnlStartCallback Max No. of CH retries reached. Hence stopping retry. "));
                  CallHistory_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_CH_DOWNLOAD, l_u8ConnectionID);
               }
            }
            //end of fix
            //FIX CMG3GB-439 Download state API does not provide download failure / download complete / Memory limit reached.
            //Adding error case for PBDS and RCDS
            if (most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO == m_poPhonebookService->pGetDownloadStateInstance()->u8GetPBDownloadState()
                  || most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO == m_poPhonebookService->pGetDownloadStateInstance()->u8GetPBDownloadState())
            {
               ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8PBDS_COMPLETE/FI_EN_E8PBDS_ERROR - Pbdl Start failed"));
               updatePBErrorStatusToClients();
               //Continue with further downloads on PbDlStart failure (even after retries)
               PhoneBook_DlStart_Failed(deviceObject,LBData);
            }
            else if (most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS == m_poPhonebookService->pGetDownloadStateInstance()->u8GetRecentCallListDownloadState())
            {
               ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8RCDS_COMPLETE/FI_EN_E8RCDS_ERROR - CH Start failed"));
               updateCHErrorStatusToClients();
            }
            //end of fix
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("Device object is NULL "));
   }
}

/*******************************************************************************
 * FUNCTION: vCheckWhetherProbablyLockedDevice
 * DESCRIPTION: This function will check whether device is probably locked or not
 * PARAMETER: [IN]  u8DeviceHandle
 *            [OUT] True/False
 * RETURNVALUE:
 ********************************************************************************/
tVoid FC_PhoneBook_Dnl_Manager::vCheckWhetherProbablyLockedDevice(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" bCheckWhetherProbablyLockedDevice entered with device handle:%u", u8DeviceHandle));

   FC_Device_Details *deviceObject =
         FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if (deviceObject != NULL)
   {
      quint8
            u8phonebookType =
                  FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(deviceObject->getConnectionID());
      ETG_TRACE_USR4(("u8phonebookType: %d", u8phonebookType));
      ETG_TRACE_USR4(("deviceObject->u8DownloadTypeList[g_s32_download_index[deviceObject->getConnectionID() - 1]]  %d", deviceObject->u8DownloadTypeList[g_s32_download_index[deviceObject->getConnectionID() - 1]] ));
      ETG_TRACE_USR4(("deviceObject->bGetSuspiciousGetSize() is: %d", deviceObject->bGetSuspiciousGetSize()));

      if (DownLoadPBLocal == u8phonebookType
            && (deviceObject->u8DownloadTypeList[g_s32_download_index[deviceObject->getConnectionID()
                  - 1]] == 0xFF)
            && (TRUE == deviceObject->bGetSuspiciousGetSize()))
      {

         //Update locked device list by adding the device handle to the list
         //Device handle will be added to probably locked devices all list size are empty and less than or equal to one for PB/ME
         //i.e, (CCH,ICH,MCH,OCH,[FAV],[SPD]) == 0 and ME/PB <=1
         (fc_phonebook_tclService_PhoneBook::pGetInstance())->vAddToDeviceLockedList(deviceObject->getDeviceHandle());
         ETG_TRACE_USR1((" -PBDL.S- Device is currently locked! "));
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object not found for given device handle"));
   }
}

/*******************************************************************************
*
* FUNCTION:  getCurDownloadTypeForConnectionID
*
* DESCRIPTION: static funciton which returns the current download type given 
* the connection ID
*            
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
quint8 FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(quint8 l_u8ConnectionID)
{
   ETG_TRACE_USR2((" getCurDownloadTypeForConnectionID entered with ConnectionID= %d ", l_u8ConnectionID));

   FC_Device_Details* DevDetails = FC_Device_Details::getDeviceObject((INDEX )(l_u8ConnectionID-1));

   if (DevDetails)
   {
      if (TRUE == DevDetails->m_bIsPendingRecentCallDownloading)
      {
         ETG_TRACE_USR4((" Returning DownloadCombinedCallHistorySimLocal "));
         return DownloadCombinedCallHistorySimLocal;
      }
      else if (TRUE == DevDetails->m_bIsPendingManualPBDownloading)
      {
         ETG_TRACE_USR4((" Returning DevDetails->downloadType : %d ", ETG_CENUM(FC_PhoneBookDwldCallType, DevDetails->downloadType)));
         return (FC_PhoneBookDwldCallType) DevDetails->downloadType;
      }
   }

   quint8 u8CurrentDnlType = 0xFF;
   if(DevDetails)
   {
      u8CurrentDnlType = DevDetails->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]-1];
      ETG_TRACE_USR4((" u8CurrentDnlType: %d ", u8CurrentDnlType));
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL "));
   }

   return u8CurrentDnlType;
}


void FC_PhoneBook_Dnl_Manager::vConnectionLost(PhBk_Loopback_Data* LBData)
{
   ETG_TRACE_USR4(("vConnectionLost"));
   if(LBData == NULL)
   {
      ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      return;
   }

   FC_Device_Details *deviceObject=
      FC_Device_Details::deviceAddressToObjectInstance(LBData->ucDevAddr);
   //CMG3G-14313-Coverity Fix
   if(!deviceObject)
   {
      ETG_TRACE_ERR(("deviceObject is NULL. Hence returning."));
      return;
   }
   //End of fix
   PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE,
                        deviceObject->getConnectionID());

   m_pDbus_Cca_ApiList->vPhoneBook_DnlComplete(
      deviceObject->getConnectionID());

}


void FC_PhoneBook_Dnl_Manager::SendDummyListChange(tU8 u8DeviceHandle, int DownloadType)
{
   ETG_TRACE_USR4(("SendDummyListChange with DeviceHandle %d ",u8DeviceHandle));
   //CMG3G-8447. Lint Fix.
   if (!m_poPhonebookService)
   {
      ETG_TRACE_ERR(("m_poPhonebookService is NULL. Hence returning."));
      return;
   }
   if ((DownloadType == DownLoadPBSim) || (DownloadType == DownLoadPBLocal)
         || (DownloadType == DownloadPBSpeedDial) || (DownloadType
         == DownloadPBFavorites))
   {
      const QHash< quint16, FC_PhoneBook_List_Data* >* l_poHashTable;   // Return type made const to avoid LINT Warning 1536 (CMG3G-4848)
      l_poHashTable = m_poPhonebookService->m_pPhoneBookDB->poGetPhonebookList()->
         poGetHashTable();

      ListChange_Handler l_oListChange_Handler;
      l_oListChange_Handler.UpdateList(l_poHashTable, enAction_Dummy,
         u8DeviceHandle,0);
   }
   else
   {
      //TODO: Send update for call histoy
   }
}


/*******************************************************************************
*
* FUNCTION:  vDataCompleteCallback
*
* DESCRIPTION: Function triggered on data complete callback from dbus daemon
*             
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vDataCompleteCallback(PhBk_Loopback_Data* LBData)
{
   ETG_TRACE_USR4(("PhoneBook_DataComplete_Callback entered"));

   if(LBData == NULL)
   {
      ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      return;
   }

   //CMG3G-8447. Lint Fix.
   if (!m_poPhonebookService)
   {
      ETG_TRACE_USR4(("m_poPhonebookService is NULL. Hence returning."));
      return;
   }

   FC_Device_Details *deviceObject=
      FC_Device_Details::deviceAddressToObjectInstance(LBData->ucDevAddr);

   if(deviceObject != NULL)
   {
      tU8 u8ConnectionID = deviceObject->getConnectionID();

      // -bn: The following code part has been moved from 'PbDlData_Signalhandler(..)' to here: BEGIN
      quint8 u8phonebookType =
         FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(u8ConnectionID);

      ETG_TRACE_COMP(( " -PBDL.S- List completed, ListType= %u (via ambiguous 'getCurDownloadTypeForConnectionID(.)'), DownloadStatus= 0x%02X ", u8phonebookType, LBData->ucStatus ));

      tBool l_bCurrentDownloadStatus = false;

      if( ((LBData->ucStatus == 0x00) || (LBData->ucStatus == 0x0C) || (LBData->ucStatus == 0x0B) ))
      {                    
         l_bCurrentDownloadStatus = true;
      }

      tU8 u8NextDownloadIndex = g_s32_download_index[u8ConnectionID-1];

      // Check if the completed download is an ME or SM download
      if( u8phonebookType == DownLoadPBLocal || u8phonebookType == DownLoadPBSim )
      {  // ME or SM list download has completed

         // Check which dedicated download has completed (ME or the SM)
         if( deviceObject->u8DownloadTypeList[u8NextDownloadIndex -1] == DownLoadPBLocal )
         {  // ME list download completed, set MEDownloadStatus
            deviceObject->m_bMEDownloadStatus = l_bCurrentDownloadStatus; 
            ETG_TRACE_COMP(( " -PBDL.S-    ME List Download completed " ));

         }
         else if ( deviceObject->u8DownloadTypeList[u8NextDownloadIndex -1] == DownLoadPBSim )
         {  // SM list download completed, set SMDownloadStatus
            deviceObject->m_bSMDownloadStatus = l_bCurrentDownloadStatus;
            ETG_TRACE_COMP(( " -PBDL.S-    SM List Download completed " ));

         }

         // Check if both lists (ME and SM) are already downloaded in current "download pass".
         // We do this by searching in the DownloadArray, if a "pending" SM list is in the array.
         if( bIs_PBSim_inDownloadArray(deviceObject) == false ) 
         {  // No SM list download pending in current "download pass".

            ETG_TRACE_COMP(( " -PBDL.S- ME/SM List Download completed, now calling 'm_poPhonebookService->vPhoneBook_PBDownloadComplete' " ));

            // ME and SM list download has been completed, so that we can start the further
            // processing for the "combined" (ME/SM) Phonebook list:

            // - Do a checksum update in phonebook database if needed, i.e. when:
            //   -- new contacts have been inserted or
            //   -- old contacts have to be deleted (as they have not been found
            //      again in current download pass, but ME and SM download status
            //      was OK)
            // - Send out ListChanges
            // - Refresh FavoritesOn DownloadComplete
            // For this purpose we use 'm_poPhonebookService->vPhoneBook_PBDownloadComplete(.)'
            // which re-delegates to 'FC_PhoneBook_Database::DownloadCompleted(guint8 u8DeviceHandle)'
            m_poPhonebookService->vPhoneBook_PBDownloadComplete(deviceObject->getDeviceHandle());

            //Fix RTC 159808
#ifndef PBDL_SKIP_SIM_LIST
           if (FALSE == deviceObject->blIsItFirstTimeDownload())
           {
              deviceObject->m_u32ContributingVCardCount
                    = deviceObject->m_u32ContributingVCardCount
                          + deviceObject->m_u32DeviceSimContactsCount;
           }
#endif
           //End Fix 159808

            //FIX CMG3G-3262 Download State property implementation
           //Fix RTC 167747
           //For IVI, If number of download contacts is reached the limit 4000, we are sending pb download state as
           //LIMIT_REACHED in CompareAndInsertIntoDatabase() method. If not, we are sending pb download state as
           //COMPLETE here. This fix is according to the CMG3G-7702.
            if ((deviceObject->m_u32ContributingVCardCount
                  <= PBDL_MAX_CONTACTS_PER_PHONE_TO_STORE)
                  && (most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_CONTACT_INFO
                        == m_poPhonebookService->pGetDownloadStateInstance()->u8GetPBDownloadState()
                  || most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_CONTACT_PHOTO
                        == m_poPhonebookService->pGetDownloadStateInstance()->u8GetPBDownloadState()))
            {
               ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_COMPLETE"));
               m_poPhonebookService->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_COMPLETE);
            }
            //End fix RTC 167747
            m_poPhonebookService->vUpdatePhonebookDownloadProgress(deviceObject->getDeviceHandle(), deviceObject->m_u32ContributingVCardCount);
            ETG_TRACE_USR4(("PhonebookDownloadProgress update sent with Count on Download complete = %d", deviceObject->m_u32ContributingVCardCount));

            m_poPhonebookService->vPhoneBook_UpdateDownloadState();
         }
      }
      else if (u8phonebookType == DownloadPBSpeedDial || u8phonebookType == DownloadPBFavorites)
      {
         //FAV/SPD has completed
         ETG_TRACE_COMP((" -PBDL.S-    FAV/SPD List Download completed "));

         //Note: Currently PBDS_COMPLETE is not sent intermittently during SPD/FAV download.
         //Also list change updates wrt contacts will be triggered after subsequent ME/SM list download is complete.
      }
      else
      {
         // Any CH list has completed.
         ETG_TRACE_COMP(( " -PBDL.S-    Any CH List Download completed " ));

         //FIX CMG3G-3262 Download State property implementation
         //Fix NCG3D-8884 Redial option enables & disables continuously during incoming call popup
         //For separate call list download due to missing call date info, List change and RCDS update need not be sent for every list download.
         //It can be sent once before and once after all three lists are downloaded.
         if (TRUE == deviceObject->Device_DateTime_Support.CHDateTimeSupport
               || (FALSE
                     == deviceObject->Device_DateTime_Support.CHDateTimeSupport
                     && DownloadIncomingCallHistorySimLocal == u8phonebookType))//Since DownloadIncomingCallHistorySimLocal is the last type added to the download array in ConstructDownloadArrayForSeparateCallList.
         {
            if (most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS
                  == m_poPhonebookService->pGetDownloadStateInstance()->u8GetRecentCallListDownloadState())
            {
               ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_COMPLETE"));
               m_poPhonebookService->pGetDownloadStateInstance()->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_COMPLETE);
               m_poPhonebookService->vPhoneBook_UpdateDownloadState();
            }

            // Sent out ListChange for call history list
            vListDownloadComplete(deviceObject);
         }
         //End of fix CMG3G-3262 and NCG3D-8884
      }

      tU16 u16downloadOnOffStatus = deviceObject->getDownloadOnOffStatus();
      if(!(deviceObject->m_u8PendingRecentCallCommandList.isEmpty()) && PB_DL_ON == u16downloadOnOffStatus)
      {
         ETG_TRACE_USR4((" Run any pending CH Item "));
         // Get capabilities is already done.

         tU8 l_u8RecentCallDownloadCommand;
         l_u8RecentCallDownloadCommand
               = deviceObject->m_u8PendingRecentCallCommandList.first();
         deviceObject->m_u8PendingRecentCallCommandList.removeFirst();

         deviceObject->m_bIsPendingRecentCallDownloading = true;
         deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS);
         ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_CH_PROGRESS"));

         m_pDbus_Cca_ApiList->vPhoneBook_DnlStart(u8ConnectionID, l_u8RecentCallDownloadCommand, deviceObject->getDeviceHandle());

      }
      else if (!(deviceObject->m_u8PendingManualPBCommandList.isEmpty()) && PB_DL_ON == u16downloadOnOffStatus)
      {
         deviceObject->m_bIsPendingRecentCallDownloading = false;

         ETG_TRACE_USR4((" Run any pending Manual PB Item"));

         tU8 u8ManualPBDownloadCommand;
         u8ManualPBDownloadCommand
               = deviceObject->m_u8PendingManualPBCommandList.first();
         deviceObject->m_u8PendingManualPBCommandList.removeFirst();

         // Re-initiate the lists: 'NewContactsAdded' and 'InvalidContactHandleList'
         // Create the TMP table for contact update to DB, etc.
         // For once before starting the pending manual download.
         if (m_poPhonebookService && FALSE == deviceObject->m_bIsPendingManualPBDownloading)
         {
            m_poPhonebookService->vPhoneBook_DownloadInitiated(deviceObject);
         }

         deviceObject->m_bIsPendingManualPBDownloading = true;
         deviceObject->downloadType = (FC_PhoneBookDwldCallType)u8ManualPBDownloadCommand;
         deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS);
         ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_PBK_PROGRESS"));

         m_pDbus_Cca_ApiList->vPhoneBook_DnlStart(u8ConnectionID, u8ManualPBDownloadCommand, deviceObject->getDeviceHandle());
      }
      else
      {
         //Reset Pending downloads variables
         deviceObject->m_bIsPendingRecentCallDownloading = false;
         deviceObject->m_bIsPendingManualPBDownloading = false;

         // -bn: The above code part has been moved from 'PbDlData_Signalhandler(..)' to here: END
         // The following code parts are related to the call back evaluation triggered by old PhoneBook_DlComplete(..)

         //HOTFIX 047
         if(m_poPhonebookService && m_poPhonebookService->m_pPhoneBookDB)
         {
            tBool b_result = m_poPhonebookService->m_pPhoneBookDB->UpdateSortView(deviceObject->getDeviceHandle());
            deviceObject->setSortListViewSupport(b_result);
            m_poPhonebookService->vUpdatePhoneBookFeatureSupport(deviceObject);
         }
         ETG_TRACE_USR2(("   PBDLCommand - %d",deviceObject->PBDLCommand));

         ETG_TRACE_USR2(("DataComplete Status- %d", LBData->ucStatus));

         quint8 l_u8ConnectionID = deviceObject->getConnectionID();

         if( (getCurDownloadTypeForConnectionID(l_u8ConnectionID) == DownloadCombinedCallHistorySimLocal)
            && (deviceObject->Device_DateTime_Support.CHDateTimeSupport == false) )
         {
            ETG_TRACE_USR2(("DownloadCombinedCallHistorySimLocal = %d",l_u8ConnectionID));
            
            //FIX GMMY16-21988 "No content available" displayed in recent call list view for a short moment when end the ongoing call 
            /*Delete only those v-cards with no calltype, date and time from DB. 
            (With default call type as FC_PhoneBook_SQLite::callhisttype_combined.)
            Other types will be deleted before insertion of respective lists to DB, after download.

            This is done so that valid v-cards are available till the list is refreshed after download.
            And thereby making sure that "No content available" is not displayed during call history update.*/

            //No need to send list change updates. The same will be sent after the fresh download is complete.
            FC_PhoneBook_CallHistory_List::GetCurrentInstance()->DeleteCallHistoryRecordsGivenCallType(l_u8ConnectionID, FC_PhoneBook_SQLite::callhisttype_combined);
            //End of fix GMMY16-21988

            //Construct array for separate CH download and start download.
            ConstructDownloadArrayForSeparateCallList(
               deviceObject, deviceObject->getDeviceCapabilities());
            deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS);
            vStartDnlForTheGivenType(deviceObject,
               deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]++]);
         }
         else
         {
            if((deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]] != 0xFF))
            {
               if(   (FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS == deviceObject->GetDownloadState())
                  || (FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS == deviceObject->GetDownloadState()) )
               {
                  if(   (deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]] == DownLoadPBLocal)
                     || (deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]] == DownLoadPBSim)
                     || (deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]] == DownloadPBSpeedDial)
                     || (deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]] == DownloadPBFavorites) )
                  {
                     deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS);

                     ETG_TRACE_USR3(("Download State changed - DOWNLOAD_ST_PBK_PROGRESS"));
                  }
                  else
                  {
                     deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS);

                     ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_CH_PROGRESS"));
                  }
               }

               ETG_TRACE_USR4(("CALLING vStartDnlForTheGivenType for other Types"));
               vStartDnlForTheGivenType(deviceObject,
                  deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]++]);
            }
            /*JAC-4996 Phonebook download complete status is received twice from fc_phonebook (First time phone pairing and connection)
            In JAC/SUZUKI, as photo need not be downloaded from device, Pass 2 can be eliminated as it is redundant.*/
#ifdef TWO_PASS_DOWNLOAD
            else if(deviceObject->blIsItFirstTimeDownload())
            {  // When we come here then: deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]] == 0xFF
               //   This means that the 1st pass download (in pairing scenario) has been completed
               //   and we have to start the second pass with ME and SM list only.

               ETG_TRACE_COMP(( " -PBDL.S- First pass download (in pairing scenario) completed " ));

               // Reset the "FirstTimeDownload" flag
               deviceObject->setFirstTimeDownload(false);

               deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS);

               // Re-initiate the lists: 'NewContactsAdded' and 'InvalidContactHandleList'
               // Implementation is in 'FC_PhoneBook_List::PhoneBookDownloadInitiated(.)',
               // Short description:
               //   DeviceDetails->NewContactsAdded.clear();
               //   DeviceDetails->InvalidContactHandleList.clear();
               //   m_sqlite->GetContactHandles(DeviceDetails->getDeviceHandle(),
               //      DeviceDetails->InvalidContactHandleList));
               //   Fix CMG3G-12876
               //   Fill all pb types(FAV/SPD/PB/SIM) count(reported by device) into device object variables,
               //   before going for second pass download
               m_pDbus_Cca_ApiList->vPhoneBook_DnlGetSize(deviceObject);
               //   End Fix CMG3G-12876
               if(m_poPhonebookService)
               {
                  m_poPhonebookService->vPhoneBook_DownloadInitiated(deviceObject);
               }

               // Clear 'deviceObject->u8DownloadTypeList' and insert the ME and SM list, 
               // (depending on which list is supported by the phone (device capabilities)).
               g_s32_download_index[l_u8ConnectionID-1] = 0;
               PhoneBook_findDownloadType(deviceObject, 
                  (PBInLocal | PBInSIM) & deviceObject->getDeviceCapabilities());

               // Start 2nd pass download
               vStartDnlForTheGivenType(deviceObject,
                  deviceObject->u8DownloadTypeList[g_s32_download_index[l_u8ConnectionID-1]++]);
            }
#endif //TWO_PASS_DOWNLOAD
            //End of fix JAC-4996

            else
            {  // Note: (u8DownloadType.. == 0xFF) AND (..blIsItFirstTimeDownload() == FALSE)

               // When we come here, we have finished in the second pass download in paring scenario
               // or we have finished the single pass verification download in reconnection scenario

               ETG_TRACE_COMP(( " -PBDL.S- Download has finished normally, (i.e. without disconnection etc.) " ));

               if (deviceObject->GetDownloadState() != FC_Device_Details::DOWNLOAD_ST_CH_UPDATE)
               {
                  /*FIX GMMY16-782 GMMY16-1106 GMMY15-12623 After contacts downloaded, recent calls list still only shows phone numbers (no names).
                  During call history download, if a record in call history is not found in current PB contact list, first name and last name of the call history record is cleared.
                  This is done to avoid presence of a record with first name and last name, not stored in PB contact list 'downloaded' to the target.

                  Consider scenario where first pass download in pairing scenario is not complate and device was disconnected.
                  Now if device is reconnected call history will be downloaded first.
                  Then only PB contacts, including those which were not downloaded in first pass of pairing scenario, will be downloaded.
                  During call history download, clearing of first name and last name fields as mentioned above will take place.
                  This clearing will include records whose corresponding PB contacts are not yet downloaded from device.
                  This clearing is fine if later after complete download of PB contacts is done, the call history records are updated with first name and last name.
                  But this is not done. Hence syncing call history with completely downloaded PB contacts here.

                  After second pass download in paring scenario or verification download in reconnection scenario is complete check if call history contains any unknown record whose details are available in PB.
                  If yes update the same with PB contact's details*/
                  if (FC_PhoneBook_CallHistory_List::GetCurrentInstance())
                  {
                     FC_PhoneBook_CallHistory_List::GetCurrentInstance()->SyncCallHistoryList(deviceObject->getDeviceHandle());
                  }
                  //End of fix

                  if (FC_PhoneBook_SQLite::GetSQLiteInstance())
                  {
                     // Drop PhoneBook_Tmp Table as current download sequence has completed for contacts download.
                     if ( false == FC_PhoneBook_SQLite::GetSQLiteInstance()->DropTableFromDB(FC_PB_DEVICE_TMP) )
                     {
                        ETG_TRACE_ERR(( " ERROR: Dropping of table 'PhoneBook_Tmp' has failed. " ));
                     }
                  }

                  FC_PhoneBook_List* poFC_PhoneBook_List =
                        FC_PhoneBook_List::pGetInstance();
                  if (poFC_PhoneBook_List != NULL)
                  {
                     if (TRUE
                           == poFC_PhoneBook_List->bUpdateDlCompletedOnceValue(deviceObject->getDeviceHandle(), true))
                     {
                        ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager:: DlCompletedOnceValue Updated Successfully"));
                     }
                     else
                     {
                        ETG_TRACE_ERR(("FC_PhoneBook_Dnl_Manager:: DlCompletedOnceValue Update FAILED"));
                     }
                  }
               }

               m_pDbus_Cca_ApiList->vPhoneBook_DnlComplete(deviceObject->getConnectionID());
               deviceObject->setFirstTimeDownload(false);
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL "));
   }
}

/*******************************************************************************
*
* FUNCTION:  vDnlCompleteCallback
*
* DESCRIPTION: Function triggered on download complete callback from dbus daemon
*             
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vDnlCompleteCallback(PhBk_Loopback_Data* LBData)
{
   ETG_TRACE_USR4(("PhoneBook_DownloadComplete_Callback entered"));
   if(LBData == NULL)
   {
      ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED to come here: LBData == NULL " ));
      return;
   }

   FC_Device_Details *deviceObject =
      FC_Device_Details::deviceAddressToObjectInstance(LBData->ucDevAddr);

   if((deviceObject != NULL) && (m_poPhonebookService != NULL))
   {
      quint8 l_u8ConnectionID= deviceObject->getConnectionID();

      if(!LBData->blFailed)
      {

         if(LBData->ucStatus == PB_DOWNLOAD_COMPLETE_SUCCESS)
         {
            ETG_TRACE_USR4(("PB_DOWNLOAD_COMPLETE_SUCCESS"));
         }
         else if(LBData->ucStatus == PB_DOWNLOAD_COMPLETE_FAILED)
         {
            ETG_TRACE_USR4(("PB_DOWNLOAD_COMPLETE_FAILED"));

            //FIX CMG3GB-439 Download state API does not provide download failure / download complete / Memory limit reached.
            //Adding error case for PBDS and RCDS. If download has not complted and error is received here, then send error.
            if( most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_COMPLETE != m_poPhonebookService->pGetDownloadStateInstance()->u8GetPBDownloadState() )
            {
               updatePBErrorStatusToClients();
            }
            if( most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_COMPLETE != m_poPhonebookService->pGetDownloadStateInstance()->u8GetRecentCallListDownloadState() )
            {
               updateCHErrorStatusToClients();
            }
            //End of fix CMG3GB-439
         }

         //Stop retry timer if it is active for the Download complete
         PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_DNL_COMPLETE,
                              l_u8ConnectionID);

         //Reset the download array as current download sequence is over.
         for(quint8 u8ArrIndex= 0; u8ArrIndex < MAX_PBDL_TYPE_SUPPORTED; u8ArrIndex++)
         {
            deviceObject->u8DownloadTypeList[u8ArrIndex] = 0xFF;
         }
         g_s32_download_index[l_u8ConnectionID - 1] = 0;

         // Print some PB Download parameters of 'deviceObject', to check if they have been reset adequately
         ETG_TRACE_COMP(( " -PBDL.S- PbDlComplete has returned successfully (PBAP session has been closed) " ));
         ETG_TRACE_COMP(( " -PBDL.S-    Number of PendingRecentCalls= %d,  DeviceObject->m_enPBKDownloadState= %d ", 
         deviceObject->m_u8PendingRecentCallCommandList.size(), 
            ETG_CENUM(enDownloadStates,deviceObject->GetDownloadState()) ));

      }
      else
      {
         ETG_TRACE_USR1((" -PBDL.S- PhoneBook_DownloadComplete_Callback returned error. Error code= %d", LBData->ucErrorCode));

         //FIX CMG3GB-439 Download state API does not provide download failure / download complete / Memory limit reached.
         //Adding error case for PBDS and RCDS. If download has not complted and error is received here, then send error. 
         if( most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_COMPLETE != m_poPhonebookService->pGetDownloadStateInstance()->u8GetPBDownloadState() )
         {
            updatePBErrorStatusToClients();
         }
         if( most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_COMPLETE != m_poPhonebookService->pGetDownloadStateInstance()->u8GetRecentCallListDownloadState() )
         {
            updateCHErrorStatusToClients();
         }
         //End of fix CMG3GB-439

         if((PhoneBook_blIsRetryActive(l_u8ConnectionID,
            FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE)) &&
            (!PhoneBook_blIsRetryActive(l_u8ConnectionID,
            FC_PhoneBook_RetryDetails::RETRY_DNL_COMPLETE)))
         {
            ETG_TRACE_USR4(("Retrying already active for another message"));
         }
         else
         {
            if ( (m_u8Retry_Count < NO_OF_RETRIES) && (TRUE == deviceObject->getPhBkPrepareCallBackStatus() ) )
            {
               PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_DNL_COMPLETE, l_u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_TYPE_INCREMENTAL);
               ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vDnlCompleteCallback : Retry Count: %d", m_u8Retry_Count));
            }
            else
            {
               ETG_TRACE_USR4(("FC_PhoneBook_Dnl_Manager::vDnlCompleteCallback : Retries stopped. "));
               PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_DNL_COMPLETE,
                                    l_u8ConnectionID);
            }
         }
      }

      /*FIX CMG3GB-696 Phonebook support value is received as 1 even if user has given cancel for access to contact and call list (after call operation)
      If PbDlGetCapabilities is success, we hold the update and we shall update it in the PbDlPrepare success/failure response.
      If PbDlGetCapabilities fails, update will be sent then itself.
      Also sending updates of Phone book support during call history update.*/
      FC_Device_Details::enDownloadStates CurDnlState=
         deviceObject->GetDownloadState();

      if(   (FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS == CurDnlState)
         || (FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS  == CurDnlState) )
      {
         ETG_TRACE_USR4(("DOWNLOAD_ST_CH_PROGRESS or DOWNLOAD_ST_PBK _PROGRESS"));
         m_poPhonebookService->vUpdatePhoneBookFeatureSupport(deviceObject);
         ETG_TRACE_USR2((" Complete download time - %d ms", dnlTime.elapsed()));
      }
      else if(FC_Device_Details::DOWNLOAD_ST_CH_UPDATE == CurDnlState)
      {
         ETG_TRACE_USR4(("DOWNLOAD_ST_CH_UPDATE"));
         m_poPhonebookService->vUpdatePhoneBookFeatureSupport(deviceObject);
      }
      //End of fix CMG3GB-696

      //Reset cancel requested flag as current download has completed.
      deviceObject->setCancelPBDownload(FALSE);

      deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
      ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_IDLE"));

      // Check if a call history download is pending. This can happen when the CH has been added
      // just to the PendingRecendCallList, but the 'PbDlComplete' (at bt_daemon) was already issued.  
      // In such a case we have to start a new download sequence to update the pending call history type. 
      // Note: The following implementation is "logically copied" from method 
      //       'FC_PhoneBook_Dnl_Manager::vCallStatus', especially from the "PBK or CH Download 
      //       NOT in PROGRESS" cases:
      // The above scenario is applicable for manual contacts download also.
      // Hence adding the check of pending manual PB lists as well.
      FC_Device_Details *pDeviceObject = NULL;
      for (tU8 u8Index = 0; u8Index < NO_OF_CONNECTED_DEVICE; u8Index++)
      {
         pDeviceObject = FC_Device_Details::getDeviceObject((INDEX) u8Index);
         if (pDeviceObject)
         {
            if (pDeviceObject->m_u8PendingRecentCallCommandList.size() > 0
                  && PB_DL_ON == pDeviceObject->getDownloadOnOffStatus())
            {
               ETG_TRACE_USR4((" Run any pending CH Item"));

               tU8 l_u8RecentCallDownloadCommand =
                     pDeviceObject->m_u8PendingRecentCallCommandList.first();
               pDeviceObject->m_u8PendingRecentCallCommandList.removeFirst();

               pDeviceObject->m_bIsPendingRecentCallDownloading = true;

               FC_PhoneBookSupportedCallType e8DnlType = CombinedCallHistory;

               // Map the 'DlListType_as_Number' [e.g. CCH=5, OCH=2] to the related 'DlListType_as_BitMask'
               //     (type: enum FC_PhoneBookSupportedCallType) [e.g. CCH=0x020, OCH=0x04]
               switch (l_u8RecentCallDownloadCommand)
               {
                  case DownloadCombinedCallHistorySimLocal:
                     e8DnlType = CombinedCallHistory;
                  break;
                  case DownloadOutGoingCallHistorySimLocal:
                     e8DnlType = OutgoingCallHistory;
                  break;
                  case DownloadMissedCallHistorySimLocal:
                     e8DnlType = MissedCallHistory;
                  break;
                  case DownloadIncomingCallHistorySimLocal:
                     e8DnlType = IncomingCallHistory;
                  break;
                  default:
                     ETG_TRACE_USR4(("Invalid Download type in l_u8RecentCallDownloadCommand"));
                  break;
               }

               // Start download of the pending CH list
               m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(pDeviceObject, e8DnlType);
               break;
            }
            else if (pDeviceObject->m_u8PendingManualPBCommandList.size() > 0
                  && PB_DL_ON == pDeviceObject->getDownloadOnOffStatus())
            {
               pDeviceObject->m_bIsPendingRecentCallDownloading = false;

               ETG_TRACE_USR4((" Run any pending Manual PB Item"));

               tU8 u8ManualPBDownloadCommand =
                     pDeviceObject->m_u8PendingManualPBCommandList.first();
               pDeviceObject->m_u8PendingManualPBCommandList.removeFirst();

               pDeviceObject->m_bIsPendingManualPBDownloading = true;
               pDeviceObject->downloadType
                     = (FC_PhoneBookDwldCallType) u8ManualPBDownloadCommand;

               // Start download of the pending manual download
			   if (m_poPhonebookService)
               {
                  m_poPhonebookService->vPhoneBook_DownloadInitiated(pDeviceObject);
               }
               m_pDbus_Cca_ApiList->vPhoneBook_StartDlPB(pDeviceObject, u8ManualPBDownloadCommand);
               break;
            }
            else
            {
               pDeviceObject->m_bIsPendingManualPBDownloading = false;
               pDeviceObject->m_bIsPendingRecentCallDownloading = false;
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL "));
   }

   //Process new device connections if any as current download is over.
   if ( m_poPhonebookService && 
         m_poPhonebookService->m_poBTSettings && deviceObject)
   {
      //NCG3D-65982
      //Once the download is is completed, remove the device handle from suspended download list if is applicable
      if((TRUE == deviceObject->bIsDeviceDownloadSuspended()))
      {
         m_poPhonebookService->vRemoveFromSuspendedDownloadList(deviceObject->getDeviceHandle());
      }
      //End fix NCG3D-65982
      m_poPhonebookService->m_poBTSettings->vCheckForNewlyConnectedDevicesOrRetries();
   }
}


/*******************************************************************************
*
* FUNCTION:  vOnDeviceConnection
*
* DESCRIPTION: Function triggered a when a new device is connected
*             
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vNewDeviceConnected(quint8 u8DeviceHandle)
{
   ETG_TRACE_COMP((" -PBDL.S- vNewDeviceConnected entered with DeviceHandle= %d ", u8DeviceHandle));
   FC_Device_Details* DeviceDetails=
      FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if(DeviceDetails)
   {
      DeviceDetails->m_bMEDownloadStatus = false;
      DeviceDetails->m_bSMDownloadStatus = false;  // -bn-PBDL.S: line added

#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
      //Connect PBAP profile via DBus
      m_pDbus_Cca_ApiList->vPhoneBook_onDeviceConnect(DeviceDetails);
#else
      if(m_poPhonebookService)
      {
            if(m_poPhonebookService->m_poBTSettings)
            {
               //Connect PBAP profile via BM APPL
               m_poPhonebookService->m_poBTSettings->vTrigger_ConnectBluetoothProfile_MethodStart(u8DeviceHandle);
            }
      }
#endif
      dnlTime.start();
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL "));
   }
}

/*******************************************************************************
*
* FUNCTION:  vStartDnlTime
*
* DESCRIPTION: Function triggered to start the timer when a new download is started
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vStartDnlTime()
{
   ETG_TRACE_USR4((" FC_PhoneBook_Dnl_Manager::vStartDnlTime entered "));
   dnlTime.start();
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Dnl_Manager::vDeviceDisconnected(quint8 l_u8ConnectionID)
*
* DESCRIPTION: Function triggered when a device is disconnected       
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vDeviceDisconnected(quint8 l_u8ConnectionID)
{
   ETG_TRACE_COMP((" -PBDL.S- vDeviceDisconnected entered with ConnectionID= %d ", l_u8ConnectionID));
   FC_Device_Details* DeviceDetails =
         FC_Device_Details::getDeviceObject((INDEX) (l_u8ConnectionID - 1));
   FC_PhoneBook_List* poFC_PhoneBook_List = FC_PhoneBook_List::pGetInstance();

   if (DeviceDetails)
   {
      if (m_poPhonebookService && m_poPhonebookService->m_poBTSettings)
      {
         if (DeviceDetails->getDeviceHandle()
               == m_poPhonebookService->m_poBTSettings->u8GetDeviceConnectInProgress())
         {
            //Reset DeviceConnectInProgress to ZERO as download got aborted for the device handle.
            m_poPhonebookService->m_poBTSettings->vSetDeviceConnectInProgress(0);
         }
      }

      //Disconnect PBAP profile via DBus
      m_pDbus_Cca_ApiList->vPhoneBook_onDeviceDisconnect(l_u8ConnectionID);

#ifdef CONNECT_PBAP_MAP_VIA_BM_APPL
      /* CMG3G-14314 - Commenting out the call to getBtStack as we are currently using a single stack*/
     // if (NULL != m_poPhonebookService && ALPS_EVOLUTION_BT_STACK == m_poPhonebookService->getBtStack())
      if (m_poPhonebookService)
      {
         //Disconnect PBAP profile via BM APPL
         if (m_poPhonebookService->m_poBTSettings)
         {
            m_poPhonebookService->m_poBTSettings->vTrigger_DisconnectBluetoothProfile_MethodStart(DeviceDetails->getDeviceHandle());
         }
      }
#endif

      switch (DeviceDetails->GetDownloadState())
      {
         case FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS:
            DeviceDetails->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);

            ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_IDLE"));
            ETG_TRACE_COMP((" -PBDL.S- Now calling: DownloadAborted "));

            if (poFC_PhoneBook_List != NULL)
            {
               poFC_PhoneBook_List->DownloadAborted(DeviceDetails->getDeviceHandle());
            }
         break;

         case FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS:
            DeviceDetails->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);

            ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_IDLE"));
            ETG_TRACE_COMP((" -PBDL.S- Now calling: m_poPhonebookService->vPhoneBook_PBDownloadComplete(.) "));

            if (m_poPhonebookService)
            {
               m_poPhonebookService->vPhoneBook_PBDownloadComplete(DeviceDetails->getDeviceHandle());
            }
         break;

         case FC_Device_Details::DOWNLOAD_ST_CH_UPDATE:
            DeviceDetails->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);

            ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_IDLE"));
         break;

         case FC_Device_Details::DOWNLOAD_ST_IDLE:
         case FC_Device_Details::DOWNLOAD_ST_NOT_STARTED:
         default:
         break;
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL "));
   }

   //Process new device connections if any.
   if (m_poPhonebookService && m_poPhonebookService->m_poBTSettings)
   {
      m_poPhonebookService->m_poBTSettings->vCheckForNewlyConnectedDevicesOrRetries();
   }
}


/*******************************************************************************
*
* FUNCTION:  vStartDnlForTheGivenType
*
* DESCRIPTION: This function starts download for a given type of data
*             
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vStartDnlForTheGivenType(FC_Device_Details*
                                                        DevDetails, quint8 pbDownlaodType)
{
   ETG_TRACE_USR4(("vStartDnlForTheGivenType"));
   if(pbDownlaodType != 0xFF)
   {
      ETG_TRACE_COMP((" -PBDL.S- PbDlStart for ListType= %d, (Phone's ListSupport= 0x%X) ", pbDownlaodType, DevDetails->getDeviceCapabilities() ));

      //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
      tU16 u16downloadOnOffStatus = DevDetails->getDownloadOnOffStatus();
      if (PB_DL_ON == u16downloadOnOffStatus)
      {

         m_pDbus_Cca_ApiList->vPhoneBook_DnlStart(DevDetails->getConnectionID(), pbDownlaodType, DevDetails->getDeviceHandle());
      }
      else
      {
         if ( (FC_Device_Details::DOWNLOAD_ST_IDLE != DevDetails->GetDownloadState()) && (FC_Device_Details::DOWNLOAD_ST_NOT_STARTED != DevDetails->GetDownloadState()))
         {
            ETG_TRACE_USR4(("Pbdl aborted as the downloadonoff state is not ON"));
            DevDetails->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
         }
      }
   }
   else
   {
      ETG_TRACE_USR2(("Not Valid Download Type pbDownlaodType =%x",pbDownlaodType));

      DevDetails->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);

      ETG_TRACE_USR3(("Download State changed - DOWNLOAD_ST_IDLE"));
      DevDetails->PBDLCommand = 0;
   }

}


/*******************************************************************************
*
* FUNCTION: PhoneBook_findDownloadType
*
* DESCRIPTION:  Update the gobal list of download type * Arguments:
* PARAMETER: [IN] PBDL_command_local
*            [OUT] NONE
*
* RETURNVALUE: None.
*
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_findDownloadType(
   FC_Device_Details *deviceObject,guchar pbDownlaodType)
{

   ETG_TRACE_USR4((" PhoneBook_findDownloadType entered and FIND DOWNLOAD TYPE 0x%X ",pbDownlaodType));
   unsigned char index_dl = 0;

   if(deviceObject)
   {
      for(index_dl = 0; index_dl <  MAX_PBDL_TYPE_SUPPORTED; index_dl++)
      {
         deviceObject->u8DownloadTypeList[index_dl] = 0xFF;
      }
      index_dl = 0;

      if(deviceObject->blIsItFirstTimeDownload())
      {
         if((pbDownlaodType & PBInLocal) == PBInLocal)
         {
            //CMG3G-11867 Download FAV/SPD from connected device if device supports the download of these lists.
            if ((deviceObject->m_u8FAVSPDDwnldSupport
                  & PBAP_FAV_DWNLD_SUPPORTED) == PBAP_FAV_DWNLD_SUPPORTED)
            {
               deviceObject->u8DownloadTypeList[index_dl++]
                     = DownloadPBFavorites;
            }
            if ((deviceObject->m_u8FAVSPDDwnldSupport
                  & PBAP_SPD_DWNLD_SUPPORTED) == PBAP_SPD_DWNLD_SUPPORTED)
            {
               deviceObject->u8DownloadTypeList[index_dl++]
                     = DownloadPBSpeedDial;
            }
            //End of fix

            deviceObject->u8DownloadTypeList[index_dl++] = DownLoadPBLocal;
         }
         if((pbDownlaodType & PBInSIM) == PBInSIM)
         {
            deviceObject->u8DownloadTypeList[index_dl++] = DownLoadPBSim;
         }

         PhoneBook_FillCallHistoryDnlTypes(deviceObject, pbDownlaodType, index_dl);
      }
      else
      {
         PhoneBook_FillCallHistoryDnlTypes(deviceObject, pbDownlaodType, index_dl);

         if (TRUE == m_bContactDlNeeded && FALSE
               == deviceObject->bIsCancelPBRequested()) // If connection or reconnection had happened, m_bContactDlNeeded will be updated.
         {
            if((pbDownlaodType & PBInLocal) == PBInLocal)
            {
               //CMG3G-11867 Download FAV/SPD from connected device if device supports the download of these lists.
               if ((deviceObject->m_u8FAVSPDDwnldSupport
                     & PBAP_FAV_DWNLD_SUPPORTED) == PBAP_FAV_DWNLD_SUPPORTED)
               {
                  deviceObject->u8DownloadTypeList[index_dl++]
                        = DownloadPBFavorites;
               }
               if ((deviceObject->m_u8FAVSPDDwnldSupport
                     & PBAP_SPD_DWNLD_SUPPORTED) == PBAP_SPD_DWNLD_SUPPORTED)
               {
                  deviceObject->u8DownloadTypeList[index_dl++]
                        = DownloadPBSpeedDial;
               }
               //End of fix

               deviceObject->u8DownloadTypeList[index_dl++] = DownLoadPBLocal;
            }
            if((pbDownlaodType & PBInSIM) == PBInSIM)
            {
               deviceObject->u8DownloadTypeList[index_dl++] = DownLoadPBSim;
            }
         }
      }
      m_bContactDlNeeded = TRUE; // Resetting the value to TRUE for handling normal scenarios (Other than Connection/Reconnection).
      PhoneBook_PrintAllDownloadTypes(deviceObject);
   }
   else
   {
      ETG_TRACE_ERR(("Device object is NULL"));
   }
}

void FC_PhoneBook_Dnl_Manager::vFillPendingManualPBCommandList(FC_Device_Details *deviceObject, tU8 u8DownloadType)
{
   ETG_TRACE_USR4((" vFillPendingManualPBCommandList entered with u8DownloadType: %d", u8DownloadType));
   
   if(deviceObject)
   {
      deviceObject->m_u8PendingManualPBCommandList.clear();
      if((u8DownloadType & PBInLocal) == PBInLocal)
      {
         //CMG3G-11867 Download FAV/SPD from connected device if device supports the download of these lists.
         if ((deviceObject->m_u8FAVSPDDwnldSupport & PBAP_FAV_DWNLD_SUPPORTED)
               == PBAP_FAV_DWNLD_SUPPORTED)
         {
            deviceObject->m_u8PendingManualPBCommandList.append(DownloadPBFavorites);
         }
         if ((deviceObject->m_u8FAVSPDDwnldSupport & PBAP_SPD_DWNLD_SUPPORTED)
               == PBAP_SPD_DWNLD_SUPPORTED)
         {
            deviceObject->m_u8PendingManualPBCommandList.append(DownloadPBSpeedDial);
         }
         //End of fix

         deviceObject->m_u8PendingManualPBCommandList.append(DownLoadPBLocal);
      }
   }
   else
   {
      ETG_TRACE_ERR(("Device object is NULL"));
   }
}

/*******************************************************************************
*
* FUNCTION: PhoneBook_DlStart_Failed
*
* DESCRIPTION: Function triggered when start callback fails
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_DlStart_Failed(FC_Device_Details* DevDetails,PhBk_Loopback_Data* LBData)
{
   ETG_TRACE_COMP((" -PBDL.S- Start Download Failed entered"));


   if((DevDetails) &&  m_poPhonebookService)
   {
      quint8 l_u8ConnectionID= DevDetails->getConnectionID();

      // -bn-PBDL.S: The following 'resetting' of the 'DownloadStatus' to 'false' is not necessary as the 'DownloadStatus' 
      //      is set to 'true' only, when the related download has succeeded.
      //      Furthermore we have now 2 'DownloadStatus' (one for ME and a new one for SM list),
      //      i.e. the following code part would have to be reworked for the correct list.
      //      All this is not necessary as explained above and therefore we just deactivate the following code part.
      //
      //quint8 u8phonebookType=
      //        FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(l_u8ConnectionID);
      //
      //if(u8phonebookType == DownLoadPBLocal)
      //{
      //    DevDetails->m_bMEDownloadStatus = false;
      //}

      switch(DevDetails->GetDownloadState())
      {

      default:
      case FC_Device_Details::DOWNLOAD_ST_CH_UPDATE:
      case FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS:
         if(DownloadCombinedCallHistorySimLocal ==
            getCurDownloadTypeForConnectionID(l_u8ConnectionID))
         {
            ConstructDownloadArrayForSeparateCallList(
               DevDetails, DevDetails->PBDLCommand);   // -bn: TODO: Check if 'DevDetails->PBDLCommand' or maybe 'DeviceDetails->getDeviceCapabilities()' is needed here

            // Send non CCH support to client i.e is HMI
            m_poPhonebookService->vUpdatePhoneBookFeatureSupport(DevDetails);
         }

         // Start the next download
         vDataCompleteCallback(LBData);

         break;

      case FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS:
         // Start the next download
         vDataCompleteCallback(LBData);
         break;
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL"));
   }

}


/*******************************************************************************
*
* FUNCTION: vSetPhonebookServiceInstance
* DESCRIPTION:
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::vSetPhonebookServiceInstance(fc_phonebook_tclService_PhoneBook* pbk_service)
{
   m_poPhonebookService = pbk_service;
}


/*******************************************************************************
*
* FUNCTION: void vDnlConnectionLostSignal(PhBk_Loopback_Data* LBData);

* DESCRIPTION:  Function triggered on download connnection lost callback from dbus daemon
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::vDnlConnectionLostSignal(PhBk_Loopback_Data* LBData)
{
   ETG_TRACE_COMP(( " -PBDL.S- vDnlConnectionLostSignal entered" ));

   FC_Device_Details *DevDetails=
      FC_Device_Details::deviceAddressToObjectInstance(LBData->ucDevAddr);

   if((DevDetails != NULL) && (m_poPhonebookService != NULL))
   {
      FC_Device_Details::enDownloadStates CurDnlState=
         DevDetails->GetDownloadState();
      if((FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS == CurDnlState))
      {
         ETG_TRACE_COMP((" -PBDL.S- Now calling DownloadAborted "));

         FC_PhoneBook_List* poFC_PhoneBook_List =
               FC_PhoneBook_List::pGetInstance();
         if (poFC_PhoneBook_List != NULL)
         {
            poFC_PhoneBook_List->DownloadAborted(DevDetails->getDeviceHandle());
         }
         m_poPhonebookService->vUpdatePhoneBookFeatureSupport(DevDetails);
      }
      else if(FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS == CurDnlState)
      {
         ETG_TRACE_COMP(( " -PBDL.S- Now calling: m_poPhonebookService->vPhoneBook_PBDownloadComplete(.) " ));

         m_poPhonebookService->vPhoneBook_PBDownloadComplete(DevDetails->getDeviceHandle());
         m_poPhonebookService->vUpdatePhoneBookFeatureSupport(DevDetails);
      }
   }
   else
   {
      ETG_TRACE_ERR(( " PhonebookService or DevDetails is NULL (device not found for "
         "address= '%s') ", LBData->ucDevAddr ));
   }

   //Process new device connections if any
   if (m_poPhonebookService && m_poPhonebookService->m_poBTSettings)
   {
      m_poPhonebookService->m_poBTSettings->vCheckForNewlyConnectedDevicesOrRetries();
   }
}


/*******************************************************************************
*
* FUNCTION: vIncomingCall
*
* DESCRIPTION: Pops up when Incoming call comes
* Doesnt pop up here for call waiting
*
* PARAMETER: CallStatus,CallTotal
CallHandle -	The call descriptor
CallTotal - No of active calls
CallState:
0x00 - Idle
0x01 - Active
0x02 - Held
0x03 - Dialing(outgoing calls only)
0x04 - Alerting (outgoing calls only)
0x05 - Incoming (incoming calls only)
0x06 - Waiting (incoming calls only)
0x07 - Response and Hold
0x08 - Unknown
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vIncomingCall(PhBk_Loopback_Data* LB_Data)
{

   ETG_TRACE_USR4(("fc_phonebook_tclService_PhoneBook::vIncomingCall entered"));
   (tVoid)LB_Data;
   if(LB_Data == NULL)
   {
      return;
   }

   if(LB_Data->blRingtone)
   {
      m_NoOfActiveCalls = 1;
   }

}


/*******************************************************************************
*
* FUNCTION: vCallStatus (Entry Thread context after Loopback message has been received)
*
* DESCRIPTION: Shows Call Status for different Call Instances
* PARAMETER: CallStatus, HW Address, Call Total, CallHandle to trace out the missed call
CallHandle  -  The call descriptor
CallTotal   -  Number of active calls
CallState:
0x00 - Idle
0x01 - Active
0x02 - Held
0x03 - Dialing  (outgoing calls only)
0x04 - Alerting (outgoing calls only)
0x05 - Incoming (incoming calls only)
0x06 - Waiting  (incoming calls only)
0x07 - Response and Hold
0x08 - Unknown
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_Dnl_Manager::vCallStatus(PhBk_Loopback_Data* LB_Data)
{
   ETG_TRACE_USR4(("vCallStatusl entered"));
   tU8 l_u8RecentCallDownloadCommand;

   if(LB_Data == NULL)
   {
      return;
   }
   ETG_TRACE_USR4(("m_NoOfActiveCalls %d",m_NoOfActiveCalls ));

   FC_Device_Details *deviceObject = 
      FC_Device_Details::deviceAddressToObjectInstance(LB_Data->ucDevAddr);

   if ((m_poPhonebookService != NULL) && (deviceObject != NULL))
   {
      ETG_TRACE_COMP(( " -PBDL.S.C- Entry-Thread-Context(LB): CallStatus: State= %u, Direction= %u, PhoneNumber= %s ",
         ETG_ENUM(TR_CALLSTATE,LB_Data->ucCallState), ETG_ENUM(TR_CALLDIRECTION,LB_Data->ucCallDirection), 
         LB_Data->aucPhoneNumber ));  

      if(LB_Data->ucCallHandle != 255)
      {
         RecentCallStatus::setCurrentState(LB_Data);
      }

      //Fix for GMMY16-6072:In multiple calls scenario both call were ended or one was ended then Updated the recent call list.
      if ((m_NoOfActiveCalls == NOOFCALLINSTANCE && LB_Data->ucCallTotal < m_NoOfActiveCalls) || (LB_Data->ucCallState == CALLSTATUS_IDLE))
      {
         if (deviceObject->blIsCCHSupported() && LB_Data->ucCallHandle != 255)
         {
            ETG_TRACE_USR1(("Combined call is supported"));

            RecentCallStatus::bDownloadMissedCall   = false;
            RecentCallStatus::bDownloadIncomingCall = false;
            RecentCallStatus::bDownloadOutgoingCall = false;

            // reset the any pending call
            deviceObject->m_u8PendingRecentCallCommandList.clear();

            //If already a CH update or PBK or CH download is in progress, then save the current CH download request to pending array.
            if(   (deviceObject->GetDownloadState() != FC_Device_Details::DOWNLOAD_ST_IDLE)
               && (deviceObject->GetDownloadState() != FC_Device_Details::DOWNLOAD_ST_NOT_STARTED))
            {
               ETG_TRACE_USR1(("Saving pending recent call"));
               l_u8RecentCallDownloadCommand = DownloadCombinedCallHistorySimLocal;

               deviceObject->m_u8PendingRecentCallCommandList.append(l_u8RecentCallDownloadCommand);
            }
            else
            {
               ETG_TRACE_USR1(("Start download sequence for 'CombinedCallHistory'"));
               deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
               m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(deviceObject,
                  CombinedCallHistory
                  | MissedCallHistory
                  | IncomingCallHistory
                  | OutgoingCallHistory);
            }
         }
         else
         {
            ETG_TRACE_USR1(("Combined call is not supported"));
            tU8 u8Operation = 255;

            if (RecentCallStatus::bDownloadMissedCall == true)
            {
               u8Operation = FC_PhoneBook_SQLite::callhisttype_missed;
            }
            else if (RecentCallStatus::bDownloadIncomingCall == true)
            {
               u8Operation = FC_PhoneBook_SQLite::callhisttype_incoming;
            }
            else if (RecentCallStatus::bDownloadOutgoingCall == true)
            {
               u8Operation = FC_PhoneBook_SQLite::callhisttype_outgoing;
            }

            switch (u8Operation)
            {
            case FC_PhoneBook_SQLite::callhisttype_missed:

               RecentCallStatus::bDownloadMissedCall = false;

               if(   (deviceObject->GetDownloadState() == FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS)
                  || (deviceObject->GetDownloadState() == FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS) )
               {
                  ETG_TRACE_USR1(("Saving pending recent call"));
                  l_u8RecentCallDownloadCommand = DownloadMissedCallHistorySimLocal;

                  deviceObject->m_u8PendingRecentCallCommandList.append(l_u8RecentCallDownloadCommand);
               }
               else
               {
                  ETG_TRACE_USR1(("Start download sequence for 'MissedCallHistory'"));
                  deviceObject->SetDownloadState(
                     FC_Device_Details::DOWNLOAD_ST_IDLE);
                  m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(deviceObject,
                     MissedCallHistory);
               }
               break;

            case FC_PhoneBook_SQLite::callhisttype_incoming:

               RecentCallStatus::bDownloadIncomingCall = false;

               if(   (deviceObject->GetDownloadState() == FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS)
                  || (deviceObject->GetDownloadState() == FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS) )
               {
                  ETG_TRACE_USR1(("Saving pending recent call"));
                  l_u8RecentCallDownloadCommand = DownloadIncomingCallHistorySimLocal;

                  deviceObject->m_u8PendingRecentCallCommandList.append(l_u8RecentCallDownloadCommand);
               }
               else
               {
                  ETG_TRACE_USR1(("Start download sequence for 'IncomingCallHistory'"));
                  deviceObject->SetDownloadState(
                     FC_Device_Details::DOWNLOAD_ST_IDLE);

                  m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(deviceObject,
                     IncomingCallHistory);
               }
               break;

            case FC_PhoneBook_SQLite::callhisttype_outgoing:

               RecentCallStatus::bDownloadOutgoingCall = false;

               if(   (deviceObject->GetDownloadState() == FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS)
                  || (deviceObject->GetDownloadState() == FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS) )
               {
                  ETG_TRACE_USR1(("Saving pending recent call"));
                  l_u8RecentCallDownloadCommand = DownloadOutGoingCallHistorySimLocal;

                  deviceObject->m_u8PendingRecentCallCommandList.append(l_u8RecentCallDownloadCommand);
               }
               else
               {
                  ETG_TRACE_USR1(("Start download sequence for 'OutgoingCallHistory'"));
                  deviceObject->SetDownloadState(
                     FC_Device_Details::DOWNLOAD_ST_IDLE);

                  m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(deviceObject,
                     OutgoingCallHistory);
               }
               break;

            default:
               ETG_TRACE_USR4(("No dedicated call history type found, therefore no CH Download to append or start "));
               break;
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL"));
   }
   m_NoOfActiveCalls = LB_Data->ucCallTotal;
}


/*******************************************************************************
*
* FUNCTION: void ConstructDownloadArrayForSeparateCallList
* DESCRIPTION:  This function constructs the array to types for downloading separate
* call history list
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::ConstructDownloadArrayForSeparateCallList(
   FC_Device_Details *deviceObject,guchar pbDownlaodType)
{
   ETG_TRACE_USR4(("Entered function 'ConstructDownloadArrayForSeparateCallList' entered with pbDownlaodType - 0x%X", pbDownlaodType));
   unsigned char index_dl = 0;
   unsigned char con_index;
   quint8 DnlListBackup[MAX_PBDL_TYPE_SUPPORTED];
   quint8 u8BackupCount = 0;

   quint8 DnlListToBeAdded[MAX_PBDL_TYPE_SUPPORTED];
   quint8 u8ToBeAddedCount = 0;
   for(quint8 u8ArrIndex= 0; u8ArrIndex < MAX_PBDL_TYPE_SUPPORTED; u8ArrIndex++)
   {
      DnlListBackup[u8ArrIndex] = 0xFF;   // Initialize DnlListBackup with 0xFF values  LINT Fix CMG3G-5150 -Info 785
      DnlListToBeAdded[u8ArrIndex] = 0xFF;   // Initialize DnlListToBeAdded with 0xFF values   LINT Fix CMG3G-5150 -Info 785
   }

   if(deviceObject)
   {
      con_index = (unsigned char)((deviceObject->getConnectionID())-1);

      while(deviceObject->u8DownloadTypeList[g_s32_download_index[con_index]] != 0xFF)
      {
         // -bn: Rework (correction) of GMNGA-49421 - SHUTDOWN_QUEUE_FULL - Issue was that the 'while loop' was not left,
         //      it was looping "forever" and blocked processes from dispatching their message queues --> SHUTDOWN_QUEUE_FULL.
         //      Now the 'emergency exit condition' has been corrected to: 'g_s32_download_index[con_index] >= MAX_PBDL_TYPE_SUPPORTED'
         //      instead of MV's original fix 'con_index == MAX_DEVICE_CONNECTION_SUPPORTED', which is "uselesss".
         //      Note: The root cause why the 'while loop' was not left via 
         //            'deviceObject->u8DownloadTypeList[g_s32_download_index[con_index]] != 0xFF' was not found by MV and is still open.
         if( g_s32_download_index[con_index] >= MAX_PBDL_TYPE_SUPPORTED )
         {
            // When we come here, the while loop end condition 'deviceObject->u8DownloadTypeList[g_s32_download_index[con_index]] != 0xFF'
            // violates the maximum lengths of 'deviceObject->u8DownloadTypeList', so that we break here ("emergency exit").
            ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED to come here: g_s32_download_index[con_index] >= MAX_PBDL_TYPE_SUPPORTED " ));

            ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED to come here: g_s32_download_index[con_index] >= MAX_PBDL_TYPE_SUPPORTED " ));
            ETG_TRACE_ERRMEM(( "  PHBK g_s32_download_index[con_index]= %d ", g_s32_download_index[con_index] ));
            ETG_TRACE_ERRMEM(( "  PHBK 'deviceObject->u8DownloadTypeList[.]'= %x %x %x %x %x %x %x %x ",
               deviceObject->u8DownloadTypeList[0], deviceObject->u8DownloadTypeList[1],
               deviceObject->u8DownloadTypeList[2], deviceObject->u8DownloadTypeList[3],
               deviceObject->u8DownloadTypeList[4], deviceObject->u8DownloadTypeList[5],
               deviceObject->u8DownloadTypeList[6], deviceObject->u8DownloadTypeList[7] ));
            break;
         }

         ETG_TRACE_USR2(( "  Backing up 'deviceObject->u8DownloadTypeList': g_s32_download_index= %d, download_type= %d  ",
            g_s32_download_index[con_index], deviceObject->u8DownloadTypeList[g_s32_download_index[con_index]] ));			

         DnlListBackup[u8BackupCount++] =
            deviceObject->u8DownloadTypeList[g_s32_download_index[con_index]++];

         //if(con_index == MAX_DEVICE_CONNECTION_SUPPORTED)  // This code line belongs to MV's "wrong" fix for GMNGA-49421
         //break;  // This code line belongs to MV's "useless" fix for GMNGA-49421
      }

      //Once copied to back up array clear the u8DownloadTypeList
      for(quint8 u8ArrIndex= 0; u8ArrIndex < MAX_PBDL_TYPE_SUPPORTED; u8ArrIndex++)
      {
         deviceObject->u8DownloadTypeList[u8ArrIndex] = 0xFF;
      }

      g_s32_download_index[con_index] = 0;


      // Fill the supported Call History types first to a temporary array.
      if((pbDownlaodType & OutgoingCallHistory) == OutgoingCallHistory)
      {
         DnlListToBeAdded[u8ToBeAddedCount++] = DownloadOutGoingCallHistorySimLocal;
      }
      if((pbDownlaodType & MissedCallHistory) == MissedCallHistory)
      {
         DnlListToBeAdded[u8ToBeAddedCount++] = DownloadMissedCallHistorySimLocal;
      }
      if((pbDownlaodType & IncomingCallHistory) == IncomingCallHistory)
      {
         DnlListToBeAdded[u8ToBeAddedCount++] = DownloadIncomingCallHistorySimLocal;
      }


      index_dl = 0;

      // Check if the combination of both lists, (i.e. DnlListToBeAdded[.] and DnlListBackup[.]) 
      // would fit into the the main list 'deviceObject->u8DownloadTypeList[.]'
      if( (u8BackupCount + u8ToBeAddedCount) < MAX_PBDL_TYPE_SUPPORTED )
      {  // OK: Size of combined list is in range, lists can be combined
         ETG_TRACE_USR2(( "  'deviceObject->u8DownloadTypeList[.]' get's refilled: u8ToBeAddedCount= %d, u8BackupCount= %d  ",
            u8ToBeAddedCount, u8BackupCount ));	

         // Insert the new Call history list types at the begin of the list
         for(quint8 u8ArrIndex= 0; u8ArrIndex < u8ToBeAddedCount; u8ArrIndex++)
         {
            deviceObject->u8DownloadTypeList[index_dl++] = DnlListToBeAdded[u8ArrIndex];
         }
         // Insert the Backup download list types afterwards
         for(quint8 u8ArrIndex= 0; u8ArrIndex < u8BackupCount; u8ArrIndex++)
         {
            deviceObject->u8DownloadTypeList[index_dl++] = DnlListBackup[u8ArrIndex];
         }
      }
      else
      {  // ERROR: Size of combined list would be to big!! 

         // Fail Safe Strategy: Just bring back the Backup download list types 
         // (and ignore the new ones in DnlListToBeAdded[.])
         for(quint8 u8ArrIndex= 0; u8ArrIndex < u8BackupCount; u8ArrIndex++)
         {
            deviceObject->u8DownloadTypeList[index_dl++] = DnlListBackup[u8ArrIndex];
         }
         ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED to come here: (u8BackupCount + u8ToBeAddedCount) >= MAX_PBDL_TYPE_SUPPORTED " ));

         ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED to come here: (u8BackupCount + u8ToBeAddedCount) >= MAX_PBDL_TYPE_SUPPORTED " ));
         ETG_TRACE_ERRMEM(( "  PHBK u8BackupCount= %d, u8ToBeAddedCount= %d ",
            u8BackupCount, u8ToBeAddedCount ));
      }

      PhoneBook_PrintAllDownloadTypes(deviceObject);
   }
   else
   {
      ETG_TRACE_ERR(("Device object is NULL"));
   }
}


/*******************************************************************************
*
* FUNCTION: PhoneBook_FetchingCapabilitiesFailed
* DESCRIPTION: Function triggered when start callback fials
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_FetchingCapabilitiesFailed(
   FC_Device_Details* DevDetails)
{
   ETG_TRACE_USR4(("Fetching of device capabilities failed entered"));

   if( (DevDetails) &&  m_poPhonebookService)
   {
      FC_PhoneBook_List* poFC_PhoneBook_List =
            FC_PhoneBook_List::pGetInstance();
      switch (DevDetails->GetDownloadState())
      {
         case FC_Device_Details::DOWNLOAD_ST_CH_UPDATE:
            DevDetails->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
            ETG_TRACE_USR4((" Download State changed - DOWNLOAD_ST_IDLE"));
         break;

         case FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS:
         case FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS:
            DevDetails->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);

            ETG_TRACE_USR4((" Download State changed - DOWNLOAD_ST_IDLE"));
            ETG_TRACE_COMP((" -PBDL.S- Now calling: DownloadAborted "));

            if (poFC_PhoneBook_List != NULL)
            {
               poFC_PhoneBook_List->DownloadAborted(DevDetails->getDeviceHandle());
            }
            m_poPhonebookService->vUpdatePhoneBookFeatureSupport(DevDetails);
         break;

         default:
            ETG_TRACE_USR2(("No action for download state - %d", (quint16) DevDetails->GetDownloadState()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Device object is NULL"));
   }

   //Process new device connections if any
   if ( m_poPhonebookService && m_poPhonebookService->m_poBTSettings)
   {
      m_poPhonebookService->m_poBTSettings->vCheckForNewlyConnectedDevicesOrRetries();
   }
}


void FC_PhoneBook_Dnl_Manager::PhoneBook_FetchingCapabilitiesSucess(FC_Device_Details* DevDetails)
{
   ETG_TRACE_USR4(("Fetching of device capabilities Success entered"));

   /*FIX CMG3GB-696 Phonebook support value is received as 1 even if user has given cancel for access to contact and call list (after call operation)
   If PbDlGetCapabilities is success, we hold the update and we shall update it in the PbDlPrepare success/failure response.
   If PbDlGetCapabilities fails, update will be sent then itself.
   Also sending updates of Phone book support during call history update.*/
   if((DevDetails) && m_poPhonebookService )
   {
      m_poPhonebookService->vUpdatePhoneBookFeatureSupport(DevDetails,FALSE);
      FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
      if (poFC_PhoneBook_SQLite)
      {
         tU8 u8DeviceHandle = DevDetails->getDeviceHandle();
         //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
         if (PB_DL_ON == poFC_PhoneBook_SQLite->u16GetDownloadOnOffStateForDevicehandle(u8DeviceHandle))
         {
            if (((FALSE == poFC_PhoneBook_SQLite->bGetAutomaticContactDLValue(u8DeviceHandle)) && (FALSE == poFC_PhoneBook_SQLite->bGetDlCompletedOnceValue(u8DeviceHandle))) || (TRUE == poFC_PhoneBook_SQLite->bGetAutomaticContactDLValue(u8DeviceHandle)))
            {
               m_bContactDlNeeded = TRUE;
            }
            else
            {
               m_bContactDlNeeded = FALSE;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR((" FC_PhoneBook_SQLite instance is NULL "));
      }
   }
   else
   {
      ETG_TRACE_USR4(("Device object is NULL"));
   }
   //End of fix CMG3GB-696
}


/*******************************************************************************
*
* FUNCTION: PhoneBook_FillCallHistoryDnlTypes
* DESCRIPTION:
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_FillCallHistoryDnlTypes(
   FC_Device_Details* DevDetails, quint8 pbDownloaddType, quint8& u8ArrIndex)
{
   ETG_TRACE_USR2((" PhoneBook_FillCallHistoryDnlTypes entered and Download type - %d", pbDownloaddType));
   ETG_TRACE_USR2((" Starting Index - %d", u8ArrIndex));

   if((pbDownloaddType & CombinedCallHistory) == CombinedCallHistory)
   {
      DevDetails->u8DownloadTypeList[u8ArrIndex++] = DownloadCombinedCallHistorySimLocal;
   }
   else
   {
      if((pbDownloaddType & OutgoingCallHistory) == OutgoingCallHistory)
      {
         DevDetails->u8DownloadTypeList[u8ArrIndex++] = DownloadOutGoingCallHistorySimLocal;
      }
      if((pbDownloaddType & MissedCallHistory) == MissedCallHistory)
      {
         DevDetails->u8DownloadTypeList[u8ArrIndex++] = DownloadMissedCallHistorySimLocal;
      }
      if((pbDownloaddType & IncomingCallHistory) == IncomingCallHistory)
      {
         DevDetails->u8DownloadTypeList[u8ArrIndex++] = DownloadIncomingCallHistorySimLocal;
      }
   }

}


/*******************************************************************************
*
* FUNCTION: PhoneBook_PrintAllDownloadTypes
* DESCRIPTION:
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_PrintAllDownloadTypes(
   FC_Device_Details *deviceObject)
{
   ETG_TRACE_USR2(("PhoneBook Dowload Types = %x %x %x %x %x %x %x %x ",
      deviceObject->u8DownloadTypeList[0],deviceObject->u8DownloadTypeList[1],
      deviceObject->u8DownloadTypeList[2],deviceObject->u8DownloadTypeList[3],
      deviceObject->u8DownloadTypeList[4], deviceObject->u8DownloadTypeList[5],
      deviceObject->u8DownloadTypeList[6], deviceObject->u8DownloadTypeList[7]));

}


/*******************************************************************************
*
* FUNCTION: PhoneBook_vStartRetry
* DESCRIPTION: This function starts retry for the given message and connection id.
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_vStartRetry(
   FC_PhoneBook_RetryDetails::enRetry_Messages RetryMsg,
   quint8 l_u8ConnectionID, FC_PhoneBook_RetryDetails::enRetry_Type RetryType)
{
   ETG_TRACE_USR4(("Function PhoneBook_vStartRetry is entered"));

   quint16
         u16TimerID =
               (l_u8ConnectionID == 0x01) ? (PHONEBOOK_TIMER_ID_1) : (PHONEBOOK_TIMER_ID_2);
   quint8 u8ArrIndex = (quint8)(l_u8ConnectionID - 1);

   //Check if the device is connected
   if (FC_Device_Details::getDeviceObject((INDEX) (l_u8ConnectionID - 1)))
   {
      //Check if the message is already retried for the device
      if (m_oRetryDetails[u8ArrIndex].blRetryActive && (RetryMsg
            == m_oRetryDetails[u8ArrIndex].enMsgRetried))
      {
         ETG_TRACE_USR2((" Timer already active for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg)));

         quint32 u32CurrRetryDur =
               m_oRetryDetails[u8ArrIndex].u32CurrentRetryDur;

         //Check if incremental retry
         if (m_oRetryDetails[u8ArrIndex].blIncrementalRetry)
         {
            if ((RETRY_MAX_RETRY_TIME * 1000) <= u32CurrRetryDur)
            {
               u32CurrRetryDur = (quint32)(RETRY_MAX_RETRY_TIME * 1000);

               ETG_TRACE_USR2((" Max retry time reached. So retrying at the constant"
                  "interval of %d sec", (u32CurrRetryDur / 1000)));
            }
            else
            {
               u32CurrRetryDur += RETRY_TIMER_INC_STEP;

               ETG_TRACE_USR2((" Retry time incremented. New retry time - %d ms", u32CurrRetryDur));
               m_oRetryDetails[u8ArrIndex].u32CurrentRetryDur = u32CurrRetryDur;
            }
         }
         else
         {
            ETG_TRACE_USR2((" Retrying at the constant time - %d ms", u32CurrRetryDur));
         }

         if (m_poPhonebookService)
         {
            if (m_poPhonebookService->bStartTimer(u16TimerID, u32CurrRetryDur, 0x0))
            {
               ETG_TRACE_USR2((" Retry timer started for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) m_oRetryDetails[u8ArrIndex].enMsgRetried)));
            }
            else
            {
               ETG_TRACE_USR2((" Failed to start timer for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) m_oRetryDetails[u8ArrIndex].enMsgRetried)));
               m_oRetryDetails[u8ArrIndex].blRetryActive = FALSE;
            }
         }
      }
      else if (!m_oRetryDetails[u8ArrIndex].blRetryActive)
      {
         if (m_poPhonebookService)
         {
            //Fix : CMG3G-12880
            quint16
                  u16RetryTimer =
                        (FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE
                              == RetryMsg) ? RETRY_TIMER_IN_LOCKED_SCENARIO : RETRY_TIMER_INITIAL;
            if (m_poPhonebookService->bStartTimer(u16TimerID, u16RetryTimer, 0x0))
            {
               ETG_TRACE_USR2((" Retry timer started for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg)));

               m_oRetryDetails[u8ArrIndex].blRetryActive = TRUE;
               m_oRetryDetails[u8ArrIndex].blIncrementalRetry
                     = (RetryType
                           == FC_PhoneBook_RetryDetails::RETRY_TYPE_INCREMENTAL) ? (TRUE) : (FALSE);
               m_oRetryDetails[u8ArrIndex].u32CurrentRetryDur
                     = (quint32) u16RetryTimer;
               m_oRetryDetails[u8ArrIndex].enMsgRetried = RetryMsg;
            }
            else
            {
               ETG_TRACE_ERR((" Failed to start timer for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR((" ERROR: Trying to start retry timer for message - %d, but"
            "already retry in progress for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg), ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) m_oRetryDetails[u8ArrIndex].enMsgRetried)));
      }
   }
   else
   {
      ETG_TRACE_USR4((" Device object is NULL. Therefore stopping Timer "));
      m_oRetryDetails[u8ArrIndex].blRetryActive = FALSE;
   }

   if (FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE
         != m_oRetryDetails[u8ArrIndex].enMsgRetried)
   {
      m_u8Retry_Count++;
   }

}
/*******************************************************************************
*
* FUNCTION: CallHistory_vStartRetry
* DESCRIPTION: This function starts retry for the given message and connection id.
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::CallHistory_vStartRetry(
   FC_PhoneBook_RetryDetails::enRetry_Messages RetryMsg,
   quint8 l_u8ConnectionID, FC_PhoneBook_RetryDetails::enRetry_Type RetryType)
{
   ETG_TRACE_USR4(("Function CallHistory_vStartRetry is entered"));

   quint16  u16TimerID = (l_u8ConnectionID == 0x01) ? (PHONEBOOK_TIMER_ID_1) : (PHONEBOOK_TIMER_ID_2);
   quint8 u8ArrIndex = (quint8)(l_u8ConnectionID - 1);

   //Check if the device is connected
   if (FC_Device_Details::getDeviceObject((INDEX) (l_u8ConnectionID - 1)))
   {
      //Check if the message is already retried for the device
      if (!m_oRetryDetails[u8ArrIndex].blCHRetryActive)
      {
         if (m_poPhonebookService)
         {
            //Fix : CMG3G-12880
            //quint16  u16RetryTimer =  (FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE == RetryMsg) ? RETRY_TIMER_IN_LOCKED_SCENARIO : RETRY_TIMER_INITIAL;
            quint16  u16RetryTimer = RETRY_TIMER_INITIAL;
            if (m_poPhonebookService->bStartTimer(u16TimerID, u16RetryTimer, 0x0))
            {
               ETG_TRACE_USR2((" Retry timer started for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg)));

               m_oRetryDetails[u8ArrIndex].blCHRetryActive = TRUE;
               m_oRetryDetails[u8ArrIndex].blIncrementalRetry
                     = (RetryType == FC_PhoneBook_RetryDetails::RETRY_TYPE_INCREMENTAL) ? (TRUE) : (FALSE);
               m_oRetryDetails[u8ArrIndex].u32CurrentRetryDur
                     = (quint32) u16RetryTimer;
               m_oRetryDetails[u8ArrIndex].enMsgRetried = RetryMsg;
            }
            else
            {
               ETG_TRACE_ERR((" Failed to start timer for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR((" ERROR: Trying to start retry timer for message - %d, but"
            "already retry in progress for message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg), ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) m_oRetryDetails[u8ArrIndex].enMsgRetried)));
      }
   }
   else
   {
      ETG_TRACE_USR4((" Device object is NULL "));
   }

   if (FC_PhoneBook_RetryDetails::RETRY_CH_DOWNLOAD == RetryMsg)
   {
      m_u8GetCHRetryCount++;
      ETG_TRACE_USR4(("CH retry count = %d", m_u8GetCHRetryCount));
   }

}

/*******************************************************************************
*
* FUNCTION: PhoneBook_vStopRetry
* DESCRIPTION: This function stops retry for the given message and connection id
* only if the retry is active
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE: None.
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_vStopRetry(
   FC_PhoneBook_RetryDetails::enRetry_Messages RetryMsg, quint8 l_u8ConnectionID)
{
   ETG_TRACE_USR4(("Function PhoneBook_vStopRetry is entered"));

   m_u8Retry_Count = 0;

   quint8 u8ArrIndex= (quint8)(l_u8ConnectionID-1);
   quint16 u16TimerID= (l_u8ConnectionID == 0x01)?(PHONEBOOK_TIMER_ID_1):
      (PHONEBOOK_TIMER_ID_2);

   if(m_oRetryDetails[u8ArrIndex].blRetryActive &&
      ((RetryMsg == m_oRetryDetails[u8ArrIndex].enMsgRetried) ||
      (FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE == RetryMsg)))
   {
      m_oRetryDetails[u8ArrIndex].blRetryActive= FALSE;
      if(m_poPhonebookService)
      {
         m_poPhonebookService->bStopTimer(u16TimerID);
      }
   }
   else
   {
      ETG_TRACE_USR2(("Retry is not active for the message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg)))
   }
}
/*******************************************************************************
*
* FUNCTION: CallHistory_vStopRetry
* DESCRIPTION: This function stops retry for the given message and connection id
* only if the retry is active
* PARAMETER: [IN]
*            [OUT]
*
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::CallHistory_vStopRetry(
   FC_PhoneBook_RetryDetails::enRetry_Messages RetryMsg, quint8 l_u8ConnectionID)
{
   ETG_TRACE_USR4(("Function CallHistory_vStopRetry is entered"));

   m_u8GetCHRetryCount = 0;

   quint8 u8ArrIndex= (quint8)(l_u8ConnectionID-1);
   quint16 u16TimerID= (l_u8ConnectionID == 0x01)?(PHONEBOOK_TIMER_ID_1):(PHONEBOOK_TIMER_ID_2);

   if(m_oRetryDetails[u8ArrIndex].blCHRetryActive && (FC_PhoneBook_RetryDetails::RETRY_CH_DOWNLOAD == RetryMsg))
   {
      m_oRetryDetails[u8ArrIndex].blCHRetryActive= FALSE;
      if(m_poPhonebookService)
      {
         m_poPhonebookService->bStopTimer(u16TimerID);
      }
   }
   else
   {
      ETG_TRACE_USR2(("Retry is not active for the message - %d", ETG_CENUM(FC_PhoneBook_RetryDetails::enRetry_Messages, (quint8) RetryMsg)))
   }
}
/*******************************************************************************
*
* FUNCTION: PhoneBook_blIsRetryActive
* DESCRIPTION: This function returns if the given message and connection id is active
* PARAMETER: [IN]
*            [OUT]
*
* RETURNVALUE:
********************************************************************************/
bool FC_PhoneBook_Dnl_Manager::PhoneBook_blIsRetryActive(quint8 l_u8ConnectionID,
                                                         FC_PhoneBook_RetryDetails::enRetry_Messages RetryMsg)
{
   ETG_TRACE_USR4(("Function PhoneBook_blIsRetryActive is entered"));


   quint8 u8ArrIndex= (quint8)(l_u8ConnectionID-1);
   bool blRet;

   if(m_oRetryDetails[u8ArrIndex].blRetryActive && (FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE
         != m_oRetryDetails[u8ArrIndex].enMsgRetried) && ((RetryMsg
         == m_oRetryDetails[u8ArrIndex].enMsgRetried)
         || (FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE == RetryMsg)))
   {
      blRet= TRUE;
   }
   else
   {
      blRet= FALSE;
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: PhoneBook_vTimerExpired
* DESCRIPTION: This function is triggered on expiry of the timer
* PARAMETER: [IN]
*            [OUT]
*            [OUT]
*
* RETURNVALUE:
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_vTimerExpired(quint16 u16TimerID)
{
   ETG_TRACE_USR4(("Function PhoneBook_vTimerExpired is entered"));


   quint8 l_u8ConnectionID;

   if(PHONEBOOK_TIMER_ID_1 == u16TimerID)
   {
      l_u8ConnectionID= 0x01;
   }
   else if(PHONEBOOK_TIMER_ID_2 == u16TimerID)
   {
      l_u8ConnectionID= 0x02;
   }
   else
   {
      ETG_TRACE_USR4(("Invalid Timer ID"));

      return;
   }

   quint8 u8ArrIndex= (quint8)(l_u8ConnectionID-1);

   if(FC_Device_Details::getDeviceObject((INDEX)(l_u8ConnectionID-1)))
   {
      PhoneBook_vRetryMsg(l_u8ConnectionID, m_oRetryDetails[u8ArrIndex].enMsgRetried);
   }
   else
   {
      ETG_TRACE_USR4(("Device object is NULL. Therefore stopping Timer"));


      m_oRetryDetails[u8ArrIndex].blRetryActive= FALSE;
   }
}


/*******************************************************************************
*
* FUNCTION: PhoneBook_vRetryMsg
* DESCRIPTION: This function calls the dbus function depending on the message
* passed
* PARAMETER: [IN]
*            [OUT]
*            [OUT]
*
* RETURNVALUE:
********************************************************************************/
void FC_PhoneBook_Dnl_Manager::PhoneBook_vRetryMsg(
   quint8 l_u8ConnectionID, FC_PhoneBook_RetryDetails::enRetry_Messages RetryMsg)
{
   ETG_TRACE_USR4(("Function PhoneBook_vRetryMsg entered"));

   FC_Device_Details* DeviceObject = FC_Device_Details::getDeviceObject((INDEX) (l_u8ConnectionID - 1));

   if (DeviceObject)
   {
      //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
      tU16 u16downloadOnOffStatus = DeviceObject->getDownloadOnOffStatus();
      if (PB_DL_ON == u16downloadOnOffStatus)
      {
         switch (RetryMsg)
         {
            case FC_PhoneBook_RetryDetails::RETRY_GET_CAPABILITIES:

               PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE, l_u8ConnectionID);

               m_pDbus_Cca_ApiList->vPhoneBook_DnlGetCapabilities(l_u8ConnectionID);
            break;
            case FC_PhoneBook_RetryDetails::RETRY_PREPARE:
               m_pDbus_Cca_ApiList->vPhoneBook_DnlPrepare(l_u8ConnectionID);
            break;
            case FC_PhoneBook_RetryDetails::RETRY_START_PBDL:
            case FC_PhoneBook_RetryDetails::RETRY_CH_DOWNLOAD:
            {
               quint8 u8DnlType = getCurDownloadTypeForConnectionID(l_u8ConnectionID);
               if(0xFF != u8DnlType)
               {
                  m_pDbus_Cca_ApiList->vPhoneBook_DnlStart(l_u8ConnectionID, u8DnlType, DeviceObject->getDeviceHandle());
               }
               else
               {
                  //During retry, cancel for PB download was requested by user. Hence download array has got cleared.
                  //PBAP session can be ended here.
                  ETG_TRACE_USR4((" No pending download on RETRY_START_PBDL callback. Hence closing session..."));
                  m_pDbus_Cca_ApiList->vPhoneBook_DnlComplete(l_u8ConnectionID);
               }
            }
            break;
            case FC_PhoneBook_RetryDetails::RETRY_DNL_COMPLETE:
               m_pDbus_Cca_ApiList->vPhoneBook_DnlComplete(l_u8ConnectionID);
            break;
            case FC_PhoneBook_RetryDetails::RETRY_STOP_PBDL:
               m_pDbus_Cca_ApiList->vPhoneBook_DnlStop(l_u8ConnectionID);
            break;
            case FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICECONNECT:
               m_pDbus_Cca_ApiList->vPhoneBook_onDeviceConnect(DeviceObject);
            break;
            case FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICEDISCONNECT:
               m_pDbus_Cca_ApiList->vPhoneBook_onDeviceDisconnect(l_u8ConnectionID);
            break;
            case FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE:
            {
               //Fix CMG3G -12880
               tU16 u16dwnldOnOffStatus =
                     DeviceObject->getDownloadOnOffStatus();
               if (PB_DL_ON == u16dwnldOnOffStatus)
               {
                  DeviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS);
                  DeviceObject->PBDLCommand = DownloadAll;

                  ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_PBK_PROGRESS"));

                  m_poPhonebookService->vPhoneBook_DownloadInitiated(DeviceObject);
                  m_pDbus_Cca_ApiList->vPhoneBook_StartDlPB(DeviceObject, DeviceObject->PBDLCommand);

               }
               else
               {
                  if ((FC_Device_Details::DOWNLOAD_ST_IDLE
                        != DeviceObject->GetDownloadState())
                        && (FC_Device_Details::DOWNLOAD_ST_NOT_STARTED
                              != DeviceObject->GetDownloadState()))
                  {
                     ETG_TRACE_USR4(("Pbdl aborted as the downloadonoff state is not ON"));
                     DeviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
                  }
               }
               //End Fix : CMG3G-12880
            }
            break;
            case FC_PhoneBook_RetryDetails::RETRY_PBAP_CONNECT_VIA_BM_APPL:
               if (m_poPhonebookService && m_poPhonebookService->m_poBTSettings)
               {
                  //Connect PBAP profile via BM APPL
                  m_poPhonebookService->m_poBTSettings->vTrigger_ConnectBluetoothProfile_MethodStart(DeviceObject->getDeviceHandle());
               }
            break;
            default:
               ETG_TRACE_ERR(("ERROR: Unknown message"));
         }
      }
      else
      {
         if ((FC_Device_Details::DOWNLOAD_ST_IDLE != DeviceObject->GetDownloadState()) && (FC_Device_Details::DOWNLOAD_ST_NOT_STARTED != DeviceObject->GetDownloadState()))
         {
            ETG_TRACE_USR4(("Pbdl aborted as the downloadonoff state is not ON"));
            DeviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
         }
      }
   }
   else
   {

      ETG_TRACE_USR2(("Device object is NULL"));
   }
}


void FC_PhoneBook_Dnl_Manager::vListDownloadComplete(FC_Device_Details *l_poFC_Device_Details)
{
   ETG_TRACE_COMP((" -PBDL.S- 'vListDownloadComplete' entered. "));

   if((l_poFC_Device_Details != NULL) && (m_poPhonebookService != NULL))
   {
      tU8 l_u8ConnectionID;
      tU8 l_u8CurrentDownloadType;

      l_u8ConnectionID = l_poFC_Device_Details->getConnectionID();
      l_u8CurrentDownloadType = l_poFC_Device_Details->GetDownloadState();

      if(   (l_u8CurrentDownloadType == FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS)
         || (l_u8CurrentDownloadType == FC_Device_Details::DOWNLOAD_ST_CH_UPDATE) )  // -bn-PBDL.S: added the "CH_UPDATE" to if clause
      {
         ETG_TRACE_COMP((" -PBDL.S-    Refresh callhistory"));

         m_poPhonebookService->vCallHistoryUpdated(l_u8ConnectionID, DownloadCombinedCallHistorySimLocal);
      }
      else if(l_u8CurrentDownloadType == FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS)
      {
         ETG_TRACE_COMP((" -PBDL.S-    Refresh phonebook contacts  [NOT EXPECTED to come here.] "));

         const QHash< quint16, FC_PhoneBook_List_Data* >* l_poHashTable;   // Return type made const to avoid LINT Warning 1536 (CMG3G-4848)
         l_poHashTable = fc_phonebook_tclService_PhoneBook::pGetInstance()->m_pPhoneBookDB->poGetPhonebookList()->
            poGetHashTable();

         ListChange_Handler l_oListChange_Handler;
         l_oListChange_Handler.UpdateList(l_poHashTable, enAction_Content_Refreshed,
            l_poFC_Device_Details->getDeviceHandle(), 0, FC_PhoneBook_SQLite::ListType_PHONEBOOK);
      }
      else 
      {
         ETG_TRACE_COMP((" -PBDL.S-    CurrentDownloadType= %d  [NOT EXPECTED to come here.] ",
            ETG_CENUM(enDownloadStates,l_u8CurrentDownloadType) ));
      }
   }
}


// -bn-PBDL.S: method added
bool FC_PhoneBook_Dnl_Manager::bIs_PBSim_inDownloadArray(FC_Device_Details *deviceObject)
{
   bool bResult = false;

   if(deviceObject != NULL)
   {
      tU8 u8ConnectionID = deviceObject->getConnectionID();

      for(tU8 u8Index = g_s32_download_index[u8ConnectionID-1]; u8Index < MAX_PBDL_TYPE_SUPPORTED; u8Index++)
      {
         if(deviceObject->u8DownloadTypeList[u8Index] == DownLoadPBSim)
         {
            bResult = true;
            break;
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Device object is NULL "));
   }

   return(bResult);
}
/*******************************************************************************
* FUNCTION: bIsCHDownloadComplete
* DESCRIPTION: This function checks if there is any pending CH(CCH,IC,MC or OC) to be downloaded
* PARAMETER: [IN]  deviceObject = Details of the device whose download status is to be checked.
*            [OUT]
* RETURNVALUE: Whether CH download is complete or not.
********************************************************************************/
bool FC_PhoneBook_Dnl_Manager::bIsCHDownloadComplete(FC_Device_Details *deviceObject)
{
   bool bResult = true;

   if(deviceObject != NULL)
   {
      tU8 u8ConnectionID = deviceObject->getConnectionID();

      for(tU8 u8Index = g_s32_download_index[u8ConnectionID-1]; u8Index < MAX_PBDL_TYPE_SUPPORTED; u8Index++)
      {
         if(deviceObject->u8DownloadTypeList[u8Index] == DownloadOutGoingCallHistorySimLocal
            || deviceObject->u8DownloadTypeList[u8Index] == DownloadMissedCallHistorySimLocal
            || deviceObject->u8DownloadTypeList[u8Index] == DownloadIncomingCallHistorySimLocal
            || deviceObject->u8DownloadTypeList[u8Index] == DownloadCombinedCallHistorySimLocal)
         {
            bResult = false;
            break;
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Device object is NULL "));
   }
   return(bResult);
}
//End of fix CMG3G-3262


/*******************************************************************************
 * FUNCTION: vDownloadCallHistory
 * DESCRIPTION: This function downloads the call history
 * PARAMETER: [IN]  deviceObject = Details of the device whose download is to be started.
 *            [OUT]
 * RETURNVALUE:
 ********************************************************************************/
void FC_PhoneBook_Dnl_Manager::vDownloadCallHistory(FC_Device_Details *deviceObject, tU8 u8CallDirection, tBool bDownloadMCH)
{
   ETG_TRACE_USR4((" -PBDL.S- vDownloadCallHistory entered"));
   tU8 l_u8RecentCallDownloadCommand = 0;

   if (deviceObject)
   {
      if (TRUE == deviceObject->blIsCCHSupported())
      {
         ETG_TRACE_USR1(("Combined call is supported"));

         // reset the any pending call
         deviceObject->m_u8PendingRecentCallCommandList.clear();

         //If already a CH update or PBK or CH download is in progress, then save the current CH download request to pending array.
         if(INVALID_VALUE != bCheckIfAnyDownloadIsInProgress())
         {
            ETG_TRACE_USR1(("Saving pending recent call"));
            l_u8RecentCallDownloadCommand = DownloadCombinedCallHistorySimLocal;

            deviceObject->m_u8PendingRecentCallCommandList.append(l_u8RecentCallDownloadCommand);
         }
         else
         {
            //Fix : CMG3G-12880
#ifdef HANDLE_PROBABLY_LOCKED_SCENARIO
            FC_Device_Details *pDeviceObject = NULL;
            for(tU8 u8Index= 0; u8Index < NO_OF_CONNECTED_DEVICE; u8Index++)
            {
               pDeviceObject = FC_Device_Details::getDeviceObject((INDEX)u8Index);
               if(pDeviceObject && TRUE == pDeviceObject->bIsDeviceProbablyLocked())
               {
                  ETG_TRACE_USR4((" Stopping retry for download sequence as a new hfp device got connected. "));
                  PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE, pDeviceObject->getConnectionID());
                  break;
               }
            }
#endif

            ETG_TRACE_USR1(("Start download sequence for 'CombinedCallHistory'"));
            deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
            m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(deviceObject, CombinedCallHistory
                  | MissedCallHistory | IncomingCallHistory
                  | OutgoingCallHistory);
         }
      }
      else
      {
         ETG_TRACE_USR1(("Combined call is not supported"));

         if (TRUE == bDownloadMCH)
         {
            if ((deviceObject->GetDownloadState()
                  == FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS)
                  || (deviceObject->GetDownloadState()
                        == FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS))
            {
               ETG_TRACE_USR1(("Saving pending recent call"));
               l_u8RecentCallDownloadCommand
                     = DownloadMissedCallHistorySimLocal;

               deviceObject->m_u8PendingRecentCallCommandList.append(l_u8RecentCallDownloadCommand);
            }
            else
            {
               ETG_TRACE_USR1(("Start download sequence for 'MissedCallHistory'"));
               deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);

               m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(deviceObject, MissedCallHistory);
            }
         }
         else
         {
            switch (u8CallDirection)
            {
               case CALLDIRECTION_INCOMING:

                  if ((deviceObject->GetDownloadState()
                        == FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS)
                        || (deviceObject->GetDownloadState()
                              == FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS))
                  {
                     ETG_TRACE_USR1(("Saving pending recent call"));
                     l_u8RecentCallDownloadCommand
                           = DownloadIncomingCallHistorySimLocal;

                     deviceObject->m_u8PendingRecentCallCommandList.append(l_u8RecentCallDownloadCommand);
                  }
                  else
                  {
                     ETG_TRACE_USR1(("Start download sequence for 'IncomingCallHistory'"));
                     deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);

                     m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(deviceObject, IncomingCallHistory);
                  }
               break;

               case CALLDIRECTION_OUTGOING:

                  if ((deviceObject->GetDownloadState()
                        == FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS)
                        || (deviceObject->GetDownloadState()
                              == FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS))
                  {
                     ETG_TRACE_USR1(("Saving pending recent call"));
                     l_u8RecentCallDownloadCommand
                           = DownloadOutGoingCallHistorySimLocal;

                     deviceObject->m_u8PendingRecentCallCommandList.append(l_u8RecentCallDownloadCommand);
                  }
                  else
                  {
                     ETG_TRACE_USR1(("Start download sequence for 'OutgoingCallHistory'"));
                     deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);

                     m_pDbus_Cca_ApiList->PhoneBook_InitiateDownload(deviceObject, OutgoingCallHistory);
                  }

               break;

               default:
                  ETG_TRACE_ERR((" u8CallDirection is INVALID!!!"));
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL!!!"));
   }
}

/*******************************************************************************
* FUNCTION: bCheckIfAnyDownloadIsInProgress
* DESCRIPTION:
* PARAMETER: [IN]
*            [OUT] 0xFF - No device download is in progress
* RETURNVALUE:
********************************************************************************/
tU8 FC_PhoneBook_Dnl_Manager::bCheckIfAnyDownloadIsInProgress()
{
   ETG_TRACE_USR4((" bCheckIfAnyDownloadIsInProgress entered "));

   tU8 u8DHDownloadInProgress = INVALID_VALUE;

   FC_Device_Details *pDeviceObject = NULL;
   for(tU8 u8Index= 0; u8Index < NO_OF_CONNECTED_DEVICE; u8Index++)
   {
      pDeviceObject = FC_Device_Details::getDeviceObject((INDEX)u8Index);
      if ((pDeviceObject) && (pDeviceObject->GetDownloadState()
            != FC_Device_Details::DOWNLOAD_ST_IDLE)
            && (pDeviceObject->GetDownloadState()
                  != FC_Device_Details::DOWNLOAD_ST_NOT_STARTED))
      {
         u8DHDownloadInProgress = pDeviceObject->getDeviceHandle();
         break;
      }
   }

   return u8DHDownloadInProgress;
}


/*******************************************************************************
 * FUNCTION: u8GetRetryCount
 * DESCRIPTION: This function returns the current retry count
 * PARAMETER: [IN]  
 *            [OUT] Current retry count
 * RETURNVALUE:
 ********************************************************************************/
tU8 FC_PhoneBook_Dnl_Manager::u8GetRetryCount()
{
   ETG_TRACE_USR4((" Current value of m_u8Retry_Count: %d ", m_u8Retry_Count));
   return m_u8Retry_Count;
}
/*******************************************************************************
 * FUNCTION: u8GetCHRetryCount
 * DESCRIPTION: This function returns the current retry count
 * PARAMETER: [IN]
 *            [OUT] Current retry count
 * RETURNVALUE:
 ********************************************************************************/
tU8 FC_PhoneBook_Dnl_Manager::u8GetCHRetryCount()
{
   ETG_TRACE_USR4((" Current value of m_u8GetCHRetryCount: %d ", m_u8GetCHRetryCount));
   return m_u8GetCHRetryCount;
}
