/*******************************************************************************
 *
 * FILE:          FC_PhoneBook_EvoDbusClient.cpp
 *
 * SW-COMPONENT:  FC_PhoneBook application
 *
 * PROJECT:
 *
 * DESCRIPTION:  Dbus Client handler API's
 *
 * AUTHOR:
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/
#include "FC_PhoneBook_EvoDbusClient.h"
#include "../FC_PhoneBook_service_PhoneBook.h"
#include "../Database/FC_PhoneBook_Database.h"
//CMG3G-14314
#include "../FC_PhoneBook_main.h"
#include <errno.h>
#include <fcntl.h>
#include <time.h>
//end of fix

#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_DBUS
#include "trcGenProj/Header/FC_PhoneBook_EvoDbusClient.cpp.trc.h"
#endif

//CMG3G-14314
QMutex g_Phonebook_DeviceMutex;
//end of fix

static PhoneBookDBusProxies_Evo
      ProxyConnMapping[MAX_DEVICE_CONNECTION_SUPPORTED];
static ObjectPath_Evo ObjectPathMapping[MAX_DEVICE_CONNECTION_SUPPORTED];
static SignalHandlerIDs_Evo SignalHandlerIDs;

static guint16 Pull_Iteration = 1;
static gboolean bFirstPull = FALSE;
static gboolean bRePullOnceOnError = FALSE;
static guint8 u8TransferErrorCount = 0;
static gboolean bRePullOnceOnPBError = FALSE;

static guint8 downLoadFilter = EVO_ENABLE_V_CARD_PICTURE_ICON;
static const ::std::string PBAPInfoFormat = "version:0x%x,support_repositories:0x%x";

//Evolution Callback instance.
_PhBK_CCA_Dbus_If *PhBkCCA_callbacks_Evo;

//Pointers for GDBUS connections - Session and System bus
static GDBusConnection* pBConnection_bus_Session_Evo = NULL;
static GDBusConnection* pBConnection_bus_System_Evo = NULL;

static OrgBluezObexClient* l_ProxyClient = NULL;
static OrgFreedesktopDBusObjectManager* l_ProxyObjectManager = NULL;

static _CCA_dbus_interfaces_pb
      DCCA_dbus_interfaces_pb_Evo =
            { EvoPhoneBook_Load_MaxContactCount,
              EvoPhoneBook_InitiateDownload,
              EvoPhoneBook_DlStop,
              EvoPhoneBook_onDeviceConnect,
              EvoPhoneBook_onDeviceDisconnect,
              EvoPhoneBook_DlPrepare,
              EvoPhoneBook_DlGetCapabilities,
              EvoPhoneBook_DnlStart,
              EvoPhoneBook_DlComplete,
              EvoPhoneBook_StartDlPB,
              EvoPhoneBook_DnlGetSize};

/******************************************************************
 Function Name: get_cca_dbus_interface_pb_Evo
 Arguments: NONE
 In:  NONE
 Out: *CCA_dbus_interfaces - pointer to interface table.
 *******************************************************************/
_CCA_dbus_interfaces_pb * get_cca_dbus_interface_pb_Evo(void)
{
   ETG_TRACE_USR4(("Get_cca_dbus_interface_pb_Evo entered"));
   return &DCCA_dbus_interfaces_pb_Evo;
}

/******************************************************************
 Function Name: register_PhBkcca_callbacks_Evo
 Description: Called by Phone Book CCA component to register its
 callback to PhoneBook dbus client.
 Arguments:
 In:
 Out:
 *******************************************************************/
unsigned char register_PhBkcca_callbacks_Evo(_PhBK_CCA_Dbus_If *vCCA_callbacks)
{
   PhBkCCA_callbacks_Evo = vCCA_callbacks;
   return E_SUCCESS;
}

/******************************************************************
 Function Name: updateMaxContactCount
 Description: This method is used to Load and Update the Total contact count
              in MaxContactCount.txt file
 Arguments:
 In:
 Out:
 *******************************************************************/
bool updateMaxContactCount(unsigned int maxContactCount)
{
   ETG_TRACE_USR4(("updateMaxContactCount entered with count :%d",maxContactCount));

   GKeyFile *keyfile = NULL;
   GError *error = NULL;
   gboolean ret = FALSE;

   //Create a new KeyFile object
   keyfile = g_key_file_new ();

   //Look for file in the specified path.
   ret = g_key_file_load_from_file(keyfile,
         FC_PB_DB_FILE_PATH "/" FC_PB_G_KEY_FILE_NAME,
         G_KEY_FILE_NONE,
         &error);

   g_key_file_set_integer(keyfile, "MaxContactCount", "Count", maxContactCount);

   ret = g_key_file_save_to_file(keyfile, FC_PB_DB_FILE_PATH "/" FC_PB_G_KEY_FILE_NAME, &error);
   if (FALSE == ret)
   {
      if(NULL != error)
      {
         ETG_TRACE_ERR(("Failed to set default contact count : %s", error->message));
         g_error_free (error);
      }
      g_key_file_free (keyfile);
      keyfile = NULLPTR;
      return false;
   }

   g_key_file_free(keyfile);
   keyfile = NULLPTR;

   //Update the MaxContactCount property
   fc_phonebook_tclService_PhoneBook::pGetInstance()->vPhoneBook_UpdateMaxContactCount(maxContactCount);

   return true;
}

/******************************************************************
 Function Name: getMaxContactCount
 Description: This method is used to Get the Total contact count
              in MaxContactCount.txt file
 Arguments:
 In:
 Out:
 *******************************************************************/
int getMaxContactCount()
{
   ETG_TRACE_USR4(("getMaxContactCount entered"));

   GKeyFile *keyfile = NULL;
   GError *error = NULL;
   gboolean ret = FALSE;

   //Create a new KeyFile object
   keyfile = g_key_file_new ();

   //Look for old file in the specified path.
   ret = g_key_file_load_from_file(keyfile,
         FC_PB_DB_FILE_PATH "/" FC_PB_G_KEY_FILE_NAME,
         G_KEY_FILE_NONE,
         &error);

   if(FALSE == ret)
   {
      if(NULL != error)
      {
         ETG_TRACE_ERR(("Failed to load max contact count : %s", error->message));
         g_error_free(error);
      }
      g_key_file_free(keyfile);
      keyfile = NULLPTR;
      return E_ERROR;
   }

   int maxContactCount  = g_key_file_get_integer(keyfile,"MaxContactCount","Count",NULL);
   ETG_TRACE_USR4(("MaxContactCount present in is :%d",maxContactCount));

   g_key_file_free(keyfile);
   keyfile = NULLPTR;

   return maxContactCount;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Load_MaxContactCount
 * DESCRIPTION: It Loads MaxContactCount file
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
int EvoPhoneBook_Load_MaxContactCount(void)
{
   ETG_TRACE_USR4(("EvoPhoneBook_Load_MaxContactCount entered"));

   //Default content of the MaxContactCount.txt file
   // [MaxContactCount]
   // Count=0

   int iResult=0;
   struct stat oStatBuf;
   if(E_SUCCESS == stat(FC_PB_DB_FILE_PATH "/" FC_PB_G_KEY_FILE_NAME, &oStatBuf))
   {
      ETG_TRACE_USR4 (("MaxContactCount.txt is present"));
      int maxContactCount = getMaxContactCount();
      if(E_ERROR == maxContactCount)
      {
         ETG_TRACE_ERR (("ERROR in reading MaxContactCount.txt file"));
         return E_FAILURE;
      }
      else
      {
         //Update the MaxContactCount property
         fc_phonebook_tclService_PhoneBook::pGetInstance()->vPhoneBook_UpdateMaxContactCount(maxContactCount);
      }
   }
   else
   {
      ETG_TRACE_ERR (("MaxContactCount.txt is NOT present, so Virgin startup"));
      //We need to create MaxContactCount.txt file.

      int fd = 0;
      struct passwd *poWantedPasswd = NULL;
      struct group  *poWantedGroup  = NULL;

      mode_t mode = S_IRUSR | S_IWUSR;
      fd = creat(FC_PB_DB_FILE_PATH "/" FC_PB_G_KEY_FILE_NAME, mode);
      if(fd == E_ERROR)
      {
         ETG_TRACE_ERR (( "ERROR: creat has FAILED with error=%d, [file='%s'] ",
               fd, FC_PB_DB_FILE_PATH "/" FC_PB_G_KEY_FILE_NAME ));
         return E_FAILURE;
      }
      else
      {
         close(fd);
         ETG_TRACE_USR4(("New MaxContactCount.txt file is created successfully"));
         // adapt current "owner" and "group" settings
         poWantedPasswd = getpwnam(FC_PB_USER_NAME);
         poWantedGroup  = getgrnam(FC_PB_GROUP_NAME);

         if((poWantedPasswd) && (poWantedGroup))
         {
            iResult = chown(FC_PB_DB_FILE_PATH "/" FC_PB_G_KEY_FILE_NAME, poWantedPasswd->pw_uid, poWantedGroup->gr_gid);
            if( iResult != E_SUCCESS )
            {
               ETG_TRACE_ERR (( "ERROR: 'chown(file, %d, %d)' has FAILED with error=%d, [file='%s'] ",
                     poWantedPasswd->pw_uid, poWantedGroup->gr_gid, iResult, FC_PB_DB_FILE_PATH "/" FC_PB_G_KEY_FILE_NAME ));
               return E_FAILURE;
            }
            else
            {
               //New MaxContactCount.txt file is created with desired permissions.
               //So setting default value.
               ETG_TRACE_USR4(("Default value is set in MaxContactCount.txt file"));
               // Zero is considered as a default value.
               updateMaxContactCount(DEFAULT_CONTACT_COUNT);
            }
         }
         else
         {
            ETG_TRACE_ERR (( "ERROR: Cannot resolve the 'wanted' user-name= '%s' ", FC_PB_USER_NAME ));
            ETG_TRACE_ERR (( "ERROR: Cannot resolve the 'wanted' group-name= '%s' ", FC_PB_GROUP_NAME ));
            return E_FAILURE;
         }
      }
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_InitiateDownload
 * DESCRIPTION:Initiating the download of phone book/call list download.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
int EvoPhoneBook_InitiateDownload(FC_Device_Details * deviceObject, unsigned char cmd)
{
   ETG_TRACE_USR4(("EvoPhoneBook_InitiateDownload entered"));

   if (deviceObject != NULL)
   {
      if (FC_Device_Details::DOWNLOAD_ST_IDLE
            == deviceObject->GetDownloadState())
      {
         if (cmd & (OutgoingCallHistory | MissedCallHistory
               | IncomingCallHistory | CombinedCallHistory))
         { // Currently CCH download is not supported
            deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_CH_UPDATE);

            ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_CH_UPDATE"));
            //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
            tU16 u16downloadOnOffStatus =
                  deviceObject->getDownloadOnOffStatus();
            if (PB_DL_ON == u16downloadOnOffStatus)
               return EvoPhoneBook_StartDlPB(deviceObject, cmd);
            else
            {
               if ((FC_Device_Details::DOWNLOAD_ST_IDLE
                     != deviceObject->GetDownloadState())
                     && (FC_Device_Details::DOWNLOAD_ST_NOT_STARTED
                           != deviceObject->GetDownloadState()))
               {
                  ETG_TRACE_USR4(("Pbdl aborted as the downloadonoff state is not ON"));
                  deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
               }
               return E_FAILURE;
            }
         }
         else
         {
            ETG_TRACE_ERR(("Download command not supported"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("Download state is not idle. Call history download not started"));
      }
      return E_SUCCESS;
   }
   else
   {
      ETG_TRACE_ERR(("Device object is NULL"));
      return E_FAILURE;
   }
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_StartDlPB
 * DESCRIPTION:Start the download of phone book/call list download.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
int EvoPhoneBook_StartDlPB(FC_Device_Details * deviceObject, unsigned char cmd)
{
   ETG_TRACE_USR4(("EvoPhoneBook_StartDlPB entered"));

   if (deviceObject != NULL)
   {
      deviceObject->RetryCnt = 0;
      deviceObject->PBDLCommand = cmd;
      ETG_TRACE_USR1(("Entering %s", __FUNCTION__));
      ETG_TRACE_USR1(("Entering[%d] ", __LINE__));
      ETG_TRACE_USR1(("Entering Modified Dev_ADDR:%s", deviceObject->getDeviceAddress()));
      ETG_TRACE_USR1(("PBDL_command =%d", deviceObject->PBDLCommand));
#ifdef ALPS_V_0_6_X_0
      //Fix CMG3G-12876
      // EvoPhoneBook_DnlGetSize()is used to fill all pb types(FAV/SPD/PB/SIM) count(reported by device) into device object variables,
      // by calling stack with Select and Get_Size methods. Since, Select and Get_Size are stack methods and are expensive
      // we are filling variables before download initiates, without making multiple calls.
      EvoPhoneBook_DnlGetSize(deviceObject);
      //End Fix CMG3G-12876
      if (deviceObject->getCapInProgress() == FALSE)
      {
         deviceObject->setCapInProgress(TRUE);

         EvoPhoneBook_DlGetCapabilities(deviceObject->getConnectionID());
      }
      return E_SUCCESS;
#else
      EvoPhoneBook_DlPrepare(deviceObject);
      return E_SUCCESS;
#endif
   }
   else
   {
      ETG_TRACE_USR4(("Device Object is NULL"));
      return E_FAILURE;
   }
}

/*******************************************************************************
 *
 * FUNCTION: EvoPbDlData_Signalhandler
 * DESCRIPTION:  Signal handler for PbDlDataSignal
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
static void EvoPbDlData_Signalhandler(gboolean bCompleted, guint8 u8DeviceHandle)
{
   ETG_TRACE_COMP((" -PBDL.EVO.S- EvoPbDlData_Signalhandler entered with device handle: %d", u8DeviceHandle));

   guint8 u8ConnectionID = 0;

   FC_Device_Details *deviceObject =
         FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if (deviceObject != NULL)
   {
      u8ConnectionID = deviceObject->getConnectionID();

      QTime ParseNInsertTime;
      ParseNInsertTime.start();

      //Handling for file received from ALPS EVO stack
      //find target file based on phonebook type
      const gchar* targetFile = NULL;
      guint8
            u8phonebookType =
                  FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(u8ConnectionID);
      switch (u8phonebookType)
      {
         case DownLoadPBSim:
         case DownLoadPBLocal:
         case DownloadPBSpeedDial:
         case DownloadPBFavorites:
            targetFile = TARGET_FILE_PHONE_BOOK;
         break;
         case DownloadOutGoingCallHistorySimLocal:
         case DownloadMissedCallHistorySimLocal:
         case DownloadIncomingCallHistorySimLocal:
         case DownloadCombinedCallHistorySimLocal:
            targetFile = TARGET_FILE_CALL_HISTORY;
         break;
         default:
         {
            ETG_TRACE_USR4(("Not Valid u8phonebookType "));
         }
      }
      if (!targetFile)
      {
         ETG_TRACE_ERR((" Target file is invalid!!!"));
      }
      else
      {
         //open file in stream
         std::ifstream infile(targetFile);

         if (!infile.good())
         {
            ETG_TRACE_ERR((" Unable to open file!!!"));
         }
         else
         {
            // get length of file:
            infile.seekg((long long int) 0, infile.end);
            int length = (int)(infile.tellg());
            ETG_TRACE_USR4(("Length of file: %d", length));
            infile.seekg((long long int) 0, infile.beg);

            if (0 != length)
            {
               //copy contents to char pointer
               char *str = new char[length];
               infile.read(str, length);

               GByteArray *gbarray;
               gbarray = g_byte_array_new_take((quint8*) str, (gsize)length);

               //Parse the v-cards
               FC_PhoneBook_Database::Instance()->ParseVCard(gbarray, u8DeviceHandle, u8ConnectionID);

               delete[] str;
            }
            else
            {
               ETG_TRACE_USR4((" Empty file!!!"));
            }

            infile.close();

            //FIX NCG3D-41871 Phonebook stores unencrypted
            //Delete the vcf file from target as parsing is complete and the file is no longer required.
            if (0 != remove((char *) targetFile))
            {
               ETG_TRACE_ERR((" Failed to remove the vcf file : %s ", targetFile));
            }
            //End of fix

            ETG_TRACE_USR1(("ParseVCard completed and Total time elapsed in parse and insert of data - %d ms", ParseNInsertTime.elapsed()));

            if (bCompleted == TRUE)
            {
               vPostDataCompleteLoopBack(deviceObject);
            }
            else
            {
               ETG_TRACE_USR4(("More data has to come"));
               //Pull again
               bFirstPull = FALSE;
               EvoPhonebook_Start_PullAll(deviceObject);
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL!!! "));
   }
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_onDeviceConnect
 * DESCRIPTION:
 * PARAMETER: [IN] deviceObject -Device object of remote device.
 *            [OUT] NONE
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
int EvoPhoneBook_onDeviceConnect(FC_Device_Details *deviceObject)
{
   ETG_TRACE_USR4(("EvoPhoneBook_onDeviceConnect entered"));
   tU8 ConnID = 0;

   if (deviceObject == NULL)
   {
      ETG_TRACE_ERR((" Device object not valid!!! "));
      return E_FAILURE;
   }
   else
   {
      ConnID = (tU8)((deviceObject->getConnectionID()) - 1);
      ETG_TRACE_USR1(("ConnID = %d", ConnID));

#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
      //Unref the available proxies.
      vClearEvoProxyConnMapping(ConnID);
      //Remove the object paths.
      vClearEvoObjectPathMapping(ConnID);

      //Creating PB session
      int ret = EvoPhoneBook_Create_Session(deviceObject);

      if (ret != E_SUCCESS)
      {
         ETG_TRACE_ERR((" Create Session Failed"));

         //Unref the available proxies.
         vClearEvoProxyConnMapping(ConnID);
         //Remove the object paths.
         vClearEvoObjectPathMapping(ConnID);

         return E_FAILURE;
      }
#endif

      //Creating the PB access proxy
      ProxyConnMapping[ConnID].m_poProxyPbAccess
            = EvoPhoneBook_Create_RemoteProxy_PhoneBookAccess(deviceObject);

      //Creating the Device proxy
      ProxyConnMapping[ConnID].m_poProxyDevice
            = EvoPhoneBook_Create_RemoteProxy_Device(deviceObject);

      if ((ProxyConnMapping[ConnID].m_poProxyPbAccess == NULL)
            || (ProxyConnMapping[ConnID].m_poProxyDevice == NULL))
      {
         ETG_TRACE_ERR(("Failed to create any one of the proxy object"));

         //Unref the available proxies.
         vClearEvoProxyConnMapping(ConnID);
         //Remove the object paths.
         vClearEvoObjectPathMapping(ConnID);

         return E_FAILURE;
      }
      else
      {
         ETG_TRACE_USR4((" All Proxy objects created successfully"));
         ETG_TRACE_USR1((" ObjectPathMapping[ConnID].SessionPath: %s", ObjectPathMapping[ConnID].SessionPath));

         //Storing device handle
         ObjectPathMapping[ConnID].DeviceHandle
               = deviceObject->getDeviceHandle();

         //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
         tU16 u16downloadOnOffStatus = deviceObject->getDownloadOnOffStatus();
         if (PB_DL_ON == u16downloadOnOffStatus)
         {
            deviceObject->PBDLCommand = DownloadAll;

            deviceObject->vSetSuspiciousGetSize(TRUE); //Fix : CMG3G-12880

            //Fix NCG3D-65982
            if (fc_phonebook_tclService_PhoneBook::pGetInstance())
            {
               tU8 u8IsAnyDownloadIsInProgress = fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.bCheckIfAnyDownloadIsInProgress();

               //Check if any PB or CH download is in progress for any other device
               //0xFF - No device download is in progress
               if (INVALID_VALUE != u8IsAnyDownloadIsInProgress)
               {
                  ETG_TRACE_COMP((" -PBDL.S- PB or CH download is in progress for another device"));
                  //If any device download is in progrees, check whether it is in probably lokced state or not. If the other device is in probably locked
                  //Continue with current device download or put the current device in download suspended list.
                  FC_Device_Details* ptrDeviceObject =
                        FC_Device_Details::deviceHandleToObjectInstance(u8IsAnyDownloadIsInProgress);
                  if((ptrDeviceObject) && (FALSE == ptrDeviceObject->bIsDeviceProbablyLocked()))
                  {
                     FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite =
                           FC_PhoneBook_SQLite::GetSQLiteInstance();
                     if (poFC_PhoneBook_SQLite)
                     {
                        //Phoebook feature support should be updated true in reconnection scenario only, since FC_Phonebook has data in its database.
                        //So that HMI can trigger CreateContactList() and pb data will be visible on HMI untill the verification download starts
                        if (TRUE
                              == poFC_PhoneBook_SQLite->bGetDlCompletedOnceValue(deviceObject->getDeviceHandle()))
                        {
                           vSetCapabilitiesForSuspendedDownloadDevice(deviceObject);
                        }
                     }
                     else
                     {
                        ETG_TRACE_USR4(("poFC_PhoneBook_SQLite is NULL"));
                     }
                     fc_phonebook_tclService_PhoneBook::pGetInstance()->vAddToSuspendedDownloadList(deviceObject->getDeviceHandle());
                     return E_SUCCESS;
                  }
                  else
                  {
                     ETG_TRACE_COMP((" Device :%u is in probably locked state, continue with pb download for current device", u8IsAnyDownloadIsInProgress));
                  }
               }
            }
            PhBkCCA_callbacks_Evo->vPhoneBook_DownloadInitiated(deviceObject);
            //End fix NCG3D-65982
            deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS);
            ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_PBK_PROGRESS"));
            EvoPhoneBook_StartDlPB(deviceObject, deviceObject->PBDLCommand);

         }
         else
         {
            if ((FC_Device_Details::DOWNLOAD_ST_IDLE
                  != deviceObject->GetDownloadState())
                  && (FC_Device_Details::DOWNLOAD_ST_NOT_STARTED
                        != deviceObject->GetDownloadState()))
            {
               ETG_TRACE_USR4(("Pbdl aborted as the downloadonoff state is not ON"));
               deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
            }
         }
      }

      return E_SUCCESS;
   }
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_onDeviceDisconnect
 * DESCRIPTION:Once the device is disconnected, signals are disconnected and
 *              DBus proxy is dereferenced.
 * PARAMETER: [IN] u8ConnectionID - ConnectionID of the device disconnected.
 *            [OUT] NONE
 *
 * RETURNVALUE: SUCCESS or FAILURE of the function.
 *
 ********************************************************************************/
int EvoPhoneBook_onDeviceDisconnect(guint8 u8ConnectionID)
{
   ETG_TRACE_USR1(("EvoPhoneBook_onDeviceDisconnect entered with ConnectionID :%d", u8ConnectionID));

   int res = E_FAILURE;
   guint8 l_ConnID = (guint8)(u8ConnectionID - 1);
   ETG_TRACE_USR1(("ConnID = %d", l_ConnID));

   //disconnect signal handler
   if ( ProxyConnMapping[l_ConnID].m_poFreeDesktopProxy && (0
         != ObjectPathMapping[l_ConnID].u1PropertyChangedID) )
   {
      g_signal_handler_disconnect(ProxyConnMapping[l_ConnID].m_poFreeDesktopProxy, ObjectPathMapping[l_ConnID].u1PropertyChangedID);
      ObjectPathMapping[l_ConnID].u1PropertyChangedID = 0;
   }

#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
   // Remove session and clear proxies
   gchar *l_session_path = ObjectPathMapping[l_ConnID].SessionPath;
   gboolean bStopRetry = false;
   GError *error = NULL;
   gboolean ret;

   if (l_session_path && l_ProxyClient)
   {
      ret
            = org_bluez_obex_client_call_remove_session_sync(l_ProxyClient, l_session_path, NULLPTR , &error);

      if ((error != NULL) || (ret == FALSE))
      {
         printErrorMessage(error, "EvoPhoneBook_onDeviceDisconnect");

         //Retry for failed remove session
         if (fc_phonebook_tclService_PhoneBook::pGetInstance())
         {
            if ((fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_blIsRetryActive(u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE))
                  && (!fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_blIsRetryActive(u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICEDISCONNECT)))
            {
               ETG_TRACE_USR4(("Retrying already active for another message"));
            }
            else
            {
               if (fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.u8GetRetryCount()
                     < 1) //Keeping the number of retries as 1 so that it will not block the thread on ignition off.
               {
                  ETG_TRACE_USR4((" Starting retry for Device Disconnection sequence. "));
                  fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICEDISCONNECT, u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_TYPE_CONSTANT);
               }
               else
               {
                  ETG_TRACE_USR4((" Max No. of retries reached. Hence stopping retry. "));
                  bStopRetry = TRUE;
               }
            }
         }
      }
      else
      {
         ETG_TRACE_USR1(("EvoPhoneBook_onDeviceDisconnect: Session Removed Successfully"));
         bStopRetry = TRUE;
         res = E_SUCCESS;
      }
   }

   if (TRUE == bStopRetry)
   {
      //Stop retry if it is active for remove session.
      if (fc_phonebook_tclService_PhoneBook::pGetInstance())
      {
         fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICEDISCONNECT, u8ConnectionID);
      }
      //Unref the available proxies.
      vClearEvoProxyConnMapping(l_ConnID);
      //Remove the object paths.
      vClearEvoObjectPathMapping(l_ConnID);
   }

#else
   //Session would have been removed by BM APPL, clear the proxies.
   //Unref the available proxies.
   vClearEvoProxyConnMapping(l_ConnID);
   //Remove the object paths.
   vClearEvoObjectPathMapping(l_ConnID);
   res = E_SUCCESS;

#endif

   return res;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_DlPrepare
 * DESCRIPTION:This Function Prepare for PHoneBookDownLoad process.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
int EvoPhoneBook_DlPrepare(guint8 u8ConnectionID)
{
   ETG_TRACE_USR1(("EvoPhoneBook_DlPrepare for connectionID- %d", u8ConnectionID));

   //Note: There is no Prepare method as such available in EVO stack.
   //We assume that all types of downloads are supported and proceed here.
   ETG_TRACE_COMP((" -PBDL.EVO.S- PREPARE mock for EVO stack."));
   FC_Device_Details *deviceObject =
         FC_Device_Details::getDeviceObject((INDEX) (u8ConnectionID - 1));
   if (deviceObject)
   {
      PhBk_Loopback_Data PrepareCB_LB_Data;
      memset((void*) &PrepareCB_LB_Data, 0, sizeof(PhBk_Loopback_Data));
      //CMG3G-14313-Coverity Fix-Using method vStringCopy instead of strncpy
	  vStringCopy(PrepareCB_LB_Data.ucDevAddr, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
      //end of Fix

      PrepareCB_LB_Data.u16FunctionID = (guint16)FC_PB_LB_FID_DNL_PREPARE_CB;
      PrepareCB_LB_Data.blFailed = FALSE;

      ETG_TRACE_USR1(("ObjectPathMapping[ConnID].SessionPath = %s", ObjectPathMapping[u8ConnectionID - 1].SessionPath));
      if (ObjectPathMapping[u8ConnectionID - 1].SessionPath)
      {
         PrepareCB_LB_Data.ucDnlType = 0x3F; //All download types supported[CCH, ICH, OCH, MCH, ME, SM]
      }
      else
      {
         //CreateSession will fail in Evo stack if user rejects/ignores the pop up for Phonebook/CallHistory download.
         //We assign ucDnlType as ZERO so that the related handling will be taken care in FC_Phonebook. (Like deletion of call history from DB.)
         PrepareCB_LB_Data.ucDnlType = 0x00;
      }

      //This function is called from entry thread. Hence there is no need of Loopback. Directly invoke the callback.
      if ( fc_phonebook_tclService_PhoneBook::pGetInstance() )
      {
         fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.vPrepareCallback(&PrepareCB_LB_Data);
      }
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_DlGetCapabilities
 * DESCRIPTION:Get the capabilities of phone book/call list download.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
int EvoPhoneBook_DlGetCapabilities(guint8 u8ConnectionID)
{
   ETG_TRACE_USR1(("EvoPhoneBook_DlGetCapabilities with ConnectionID: %d", u8ConnectionID));

   //Note: There is no GET capabilities method as such available in EVO stack.
   //We assume that all types of downloads are supported and proceed here.
   ETG_TRACE_COMP((" -PBDL.EVO.S- GETCAP mock for EVO stack."));

   FC_Device_Details *deviceObject =
         FC_Device_Details::getDeviceObject((INDEX) (u8ConnectionID - 1));
   if (deviceObject)
   {
      deviceObject->setCapInProgress(FALSE);

      PhBk_Loopback_Data GetCapabilitiesCB_LB_Data;
      memset((void*) &GetCapabilitiesCB_LB_Data, 0, sizeof(PhBk_Loopback_Data));
	  //CMG3G-14313-Coverity Fix-Using method vStringCopy instead of strncpy
      vStringCopy(GetCapabilitiesCB_LB_Data.ucDevAddr, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
      //end of fix
      GetCapabilitiesCB_LB_Data.u16FunctionID
            = (guint16)FC_PB_LB_FID_GET_CAPABILITIES_CB;
      GetCapabilitiesCB_LB_Data.blFailed = FALSE;
      GetCapabilitiesCB_LB_Data.ucDnlType = 0x3F; //All download types supported[CCH, ICH, OCH, MCH, ME, SM]

      //This function is called from entry thread. Hence there is no need of Loopback. Directly invoke the callback.
      if (fc_phonebook_tclService_PhoneBook::pGetInstance())
      {
         fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.vGetCapabilitiesCallback(&GetCapabilitiesCB_LB_Data);
      }
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION:  EvoPhoneBook_DnlStart
 * DESCRIPTION:Start phone book or call history list download
 * PARAMETER: [IN]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
int EvoPhoneBook_DnlStart(guint8 u8ConnectionID, guchar pbDownlaodType, tU8 f_u8DeviceHandle)
{
   ETG_TRACE_USR1(("EvoPhoneBook_DnlStart entered with ConnectionID :%d and DeviceHandle: %d", u8ConnectionID, f_u8DeviceHandle));

   const gchar* arg_location = VCARD_LOCATION_INT;
   const gchar* arg_phonebook = getEvoPhonebookType(pbDownlaodType);

   if (VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_INVALID, arg_phonebook))
   {
      ETG_TRACE_ERR((" arg_phonebook is invalid!!!"));
      return E_FAILURE;
   }

   FC_Device_Details* deviceObject =
         FC_Device_Details::deviceHandleToObjectInstance(f_u8DeviceHandle);
   if (deviceObject)
   {
      deviceObject->downloadType = pbDownlaodType;
      ETG_TRACE_USR1(("Updated downloadType in device object :%d", pbDownlaodType));

      if (DownLoadPBSim == (FC_PhoneBookDwldCallType) pbDownlaodType)
      {
         arg_location = VCARD_LOCATION_SIM;
      }

      ETG_TRACE_COMP((" -PBDL.EVO.S- SELECT - Location: %s ", arg_location));
      ETG_TRACE_COMP((" -PBDL.EVO.S- SELECT - Phonebook: %s ", arg_phonebook));
      bFirstPull = TRUE;
      EvoPhoneBook_Select_PhoneBook(deviceObject, arg_location, arg_phonebook);
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL!!!"));
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Create_RemoteProxy_Client
 * DESCRIPTION:  Create Remote Object proxy for client
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE:
 ********************************************************************************/
int EvoPhoneBook_Create_RemoteProxy_Client()
{
   ETG_TRACE_USR1(("EvoPhoneBook_Create_RemoteProxy_Client entered "));

   GError *error = NULL;

   l_ProxyClient
         = org_bluez_obex_client_proxy_new_sync(pBConnection_bus_Session_Evo, G_DBUS_PROXY_FLAGS_NONE, BASE_SERVICE_NAME_EVO, BASE_OBJECT_PATH_EVO, NULLPTR, &error);

   if (error  || (!l_ProxyClient) )
   {
      printErrorMessage(error, "EvoPhoneBook_Create_RemoteProxy_Client");
      return E_FAILURE;
   }
   else
   {
      ETG_TRACE_USR1(("EvoPhoneBook_Create_RemoteProxy_Client Cient Proxy creation success "));

      //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
      // this removes the lint warning for this block only
      OrgBluezObexClientProxy *clientproxy =
            ORG_BLUEZ_OBEX_CLIENT_PROXY(l_ProxyClient);
      if (clientproxy != NULL)
      {
         // The default timeout value for DBus call in G3g is set to 300s - CMG3GB-1364
         g_dbus_proxy_set_default_timeout(&clientproxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
      }
      else
      {
         ETG_TRACE_ERR(("bEvoPhoneBookCreateRemoteProxy : set timeout failed for client proxy"));
      }
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Create_RemoteProxy_ObjectManager
 * DESCRIPTION:  Create Remote Object proxy for object manager
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
int EvoPhoneBook_Create_RemoteProxy_ObjectManager()
{
   ETG_TRACE_USR1(("EvoPhoneBook_Create_RemoteProxy_ObjectManager entered "));

   GError *error = NULL;

   l_ProxyObjectManager
         = org_freedesktop_dbus_object_manager_proxy_new_sync(pBConnection_bus_Session_Evo, G_DBUS_PROXY_FLAGS_NONE, BASE_SERVICE_NAME_EVO, "/", NULLPTR, &error);

   if ( error  || (!l_ProxyObjectManager) )
   {
      printErrorMessage(error, "EvoPhoneBook_Create_RemoteProxy_ObjectManager");
      return E_FAILURE;
   }
   else
   {
      ETG_TRACE_USR1(("EvoPhoneBook_Create_RemoteProxy_ObjectManager Proxy creation success "));

      //Register to signals "interfaces-added" and "interfaces-removed"
      SignalHandlerIDs.ulInterfacesAddedID
            = g_signal_connect(l_ProxyObjectManager, "interfaces-added", G_CALLBACK(interfacesAddedCallback), NULLPTR);
      SignalHandlerIDs.ulInterfacesRemovedID
            = g_signal_connect(l_ProxyObjectManager, "interfaces-removed", G_CALLBACK(interfacesRemovedCallback), NULLPTR);

      //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
      // this removes the lint warning for this block only
      OrgFreedesktopDBusObjectManagerProxy *objectManagerProxy =
            ORG_FREEDESKTOP_DBUS_OBJECT_MANAGER_PROXY(l_ProxyObjectManager);
      if (objectManagerProxy != NULL)
      {
         // The default timeout value for DBus call in G3g is set to 300s - CMG3GB-1364
         g_dbus_proxy_set_default_timeout(&objectManagerProxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
      }
      else
      {
         ETG_TRACE_ERR((" EvoPhoneBook_Create_RemoteProxy_ObjectManager : set timeout failed for object manager proxy"));
      }
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Create_Session
 * DESCRIPTION:  Create Session with Remote Device
 * PARAMETER: [IN] DeviceObject - Device object of the remote device.
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
int EvoPhoneBook_Create_Session(FC_Device_Details *deviceObject)
{
   ETG_TRACE_USR1(("EvoPhoneBook_Create_Session entered  with deviceAddress  ::%s", deviceObject->getDeviceAddress()));

   GVariant *value;
   GVariantBuilder* builder;
   gchar *l_session_path = NULL;
   gboolean ret = FALSE;
   GError *error = NULL;

   char localAddress[LENGTH_OF_DEVICE_ADDRESS];
   memset(localAddress, '\0', LENGTH_OF_EVO_DEVICE_ADDRESS);

   if (fc_phonebook_tclService_PhoneBook::pGetInstance() != NULL)
   {
      fc_phonebook_tclService_PhoneBook::pGetInstance()->vGetVehicleBtAddress(localAddress);
   }
   guint8 l_ConnID = (guint8)((deviceObject->getConnectionID()) - 1);
   char* deviceAddress = deviceObject->getDeviceAddress();

   ETG_TRACE_USR1((" ConnID = %d", l_ConnID));
   ETG_TRACE_USR1((" Local BT Address = %s", localAddress));
   ETG_TRACE_USR1((" Device Address = %s", deviceAddress));

   //Evo stack accepts device address as aa:bb:cc:dd:ee:ff
   //Adapt dev address from dev_aa_bb_cc_dd_ee_ff to aa:bb:cc:dd:ee:ff
   char achEvoDeviceAddr[LENGTH_OF_EVO_DEVICE_ADDRESS];
   memset(achEvoDeviceAddr, 0, LENGTH_OF_EVO_DEVICE_ADDRESS);

   for (tU8 u8Index = 0; u8Index < 6; u8Index++)
   {
      achEvoDeviceAddr[u8Index * 3] = deviceAddress[4 + u8Index * 3];
      achEvoDeviceAddr[1 + u8Index * 3] = deviceAddress[5 + u8Index * 3];
      if (u8Index != 5)//: not required at end
      {
         achEvoDeviceAddr[2 + u8Index * 3] = ':';
      }
   }

   //Adapt local address from aabbccddeeff to aa:bb:cc:dd:ee:ff
   char achEvoLocalAddr[LENGTH_OF_EVO_DEVICE_ADDRESS];
   memset(achEvoLocalAddr, 0, LENGTH_OF_EVO_DEVICE_ADDRESS);

   for (tU8 u8Index1 = 0; u8Index1 < 6; u8Index1++)
   {
      achEvoLocalAddr[u8Index1 * 3] = localAddress[0 + u8Index1 * 2];
      achEvoLocalAddr[1 + u8Index1 * 3] = localAddress[1 + u8Index1 * 2];
      if (u8Index1 != 5)//: not required at end
      {
         achEvoLocalAddr[2 + u8Index1 * 3] = ':';
      }
   }

   ETG_TRACE_USR1((" EVO Local BT Address = %s", achEvoLocalAddr));
   ETG_TRACE_USR1((" EVO Device Address = %s", achEvoDeviceAddr));

   if (l_ProxyClient == NULL)
   {
      ETG_TRACE_ERR((" l_ProxyClient is NULL!!! "));
      return E_FAILURE;
   }

   builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
   g_variant_builder_add(builder, "{sv}", "Target", g_variant_new_string("PBAP"));
   g_variant_builder_add(builder, "{sv}", "Source", g_variant_new_string(achEvoLocalAddr));

   value = g_variant_new("a{sv}", builder);

   ret
         = org_bluez_obex_client_call_create_session_sync(l_ProxyClient, achEvoDeviceAddr, value, &l_session_path, NULLPTR, &error);

   if (ret == FALSE)
   {
      printErrorMessage(error, "EvoPhoneBook_Create_Session");

      //Retry for failed create session
      if (fc_phonebook_tclService_PhoneBook::pGetInstance())
      {
         if ((fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_blIsRetryActive(deviceObject->getConnectionID(), FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE))
               && (!fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_blIsRetryActive(deviceObject->getConnectionID(), FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICECONNECT)))
         {
            ETG_TRACE_USR4(("Retrying already active for another message"));
         }
         else
         {
            if (fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.u8GetRetryCount()
                  < NO_OF_RETRIES_EVO)
            {
               ETG_TRACE_USR4((" Starting retry for Device Connection sequence. "));
               fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICECONNECT, deviceObject->getConnectionID(), FC_PhoneBook_RetryDetails::RETRY_TYPE_CONSTANT);
            }
            else
            {
               ETG_TRACE_USR4((" Max No. of retries reached. Hence stopping retry. "));
               fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICECONNECT, deviceObject->getConnectionID());
            }
         }
      }

      return E_FAILURE;
   }

   ETG_TRACE_USR1(("EvoPhoneBook_Create_Session: Session created Successfully"));
   //Storing the Session path
   g_free(ObjectPathMapping[l_ConnID].SessionPath);
   ObjectPathMapping[l_ConnID].SessionPath = g_strdup(l_session_path);

   //Stop retry if it is active for create session.
   if (fc_phonebook_tclService_PhoneBook::pGetInstance())
   {
      fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_EVO_DEVICECONNECT, deviceObject->getConnectionID());
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Create_RemoteProxy_PhoneBookAccess
 * DESCRIPTION:  Create Remote Object proxy for PhoneBook Access.
 * PARAMETER: [IN] DeviceObject - Device object of the remote device.
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
OrgBluezObexPhonebookAccess* EvoPhoneBook_Create_RemoteProxy_PhoneBookAccess(FC_Device_Details *deviceObject)
{
   ETG_TRACE_USR1(("EvoPhoneBook_Create_RemoteProxy_PhoneBookAccess entered  with deviceAddress  ::%s", deviceObject->getDeviceAddress()));

   OrgBluezObexPhonebookAccess* l_ProxyPbAccess = NULL;
   GError *error = NULL;

   guint8 l_ConnID = (guint8)((deviceObject->getConnectionID()) - 1);
   gchar* l_SessionPath = ObjectPathMapping[l_ConnID].SessionPath;

   ETG_TRACE_USR1(("ConnID = %d", l_ConnID));

   if (l_SessionPath)
   {
      l_ProxyPbAccess
            = org_bluez_obex_phonebook_access_proxy_new_sync(pBConnection_bus_Session_Evo, G_DBUS_PROXY_FLAGS_NONE, BASE_SERVICE_NAME_EVO, l_SessionPath, NULLPTR, &error);

      if ( (error) || (!l_ProxyPbAccess))
      {
         printErrorMessage(error, "EvoPhoneBook_Create_RemoteProxy_PhoneBookAccess");
      }
      else
      {
         ETG_TRACE_USR4(("bPhoneBookCreateRemoteProxy : PB access proxy creation SUCCESS"));

         //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
         // this removes the lint warning for this block only
         OrgBluezObexPhonebookAccessProxy* PbAccessProxy =
               ORG_BLUEZ_OBEX_PHONEBOOK_ACCESS_PROXY(l_ProxyPbAccess);
         if (PbAccessProxy != NULL)
         {
            // The default timeout value for DBus call in G3g is set to 300s - CMG3GB-1364
            g_dbus_proxy_set_default_timeout(&PbAccessProxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
         }
         else
         {
            ETG_TRACE_ERR(("bPhoneBookCreateRemoteProxy : set timeout failed for PB access proxy"));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" l_SessionPath is NULL!!!"));
   }

   return l_ProxyPbAccess;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Create_RemoteProxy_Device
 * DESCRIPTION:  Create Remote Object proxy for interface Device.
 * PARAMETER: [IN] DeviceObject - Device object of the remote device.
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
OrgBluezDevice1* EvoPhoneBook_Create_RemoteProxy_Device(FC_Device_Details *deviceObject)
{
   ETG_TRACE_USR1((" EvoPhoneBook_Create_RemoteProxy_Device entered."));


   OrgBluezDevice1* l_ProxyDevice = NULL;
   if (!deviceObject)
   {
      ETG_TRACE_ERR((" deviceObject is NULL!!!"));
   }
   else
   {
      GError *error = NULL;
      guint8 l_ConnID = (guint8)((deviceObject->getConnectionID()) - 1);//CMG3G-14313 Coverity Warnings Fix

      tChar *obj_path = NULL;
      tU8 pathSize = (tU8)(strlen(PHBK_DEVICE_OBJ_PATH)
            + strlen(deviceObject->getDeviceAddress()) + 1);

      obj_path = (char*) malloc(pathSize);
      if (obj_path == NULL)
      {
         ETG_TRACE_ERR((" malloc failed!!!"));
      }
      else
      {
         memset(obj_path, 0, pathSize);
         strncpy(obj_path, PHBK_DEVICE_OBJ_PATH, pathSize);

         //Note: Address in Object path is to be in Upper case. For use covert to Upper case.
         tChar *address = NULL;
         address = (char*) malloc(LENGTH_OF_DEVICE_ADDRESS + 1);
         if (address)
         {
            memset(address, 0, LENGTH_OF_DEVICE_ADDRESS);
            strncpy(address, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
            for (tU8 i = 4; i < strlen(address); i++)
            {
               address[i] = (tChar)toupper(address[i]);
            }

            strncat(obj_path, (const char*) address, pathSize
                  - strlen(PHBK_DEVICE_OBJ_PATH));
            ETG_TRACE_USR4((" obj_path: %s", obj_path));
            free(address);

            l_ProxyDevice
                  = org_bluez_device1_proxy_new_sync(pBConnection_bus_System_Evo, G_DBUS_PROXY_FLAGS_NONE, "org.bluez", obj_path, NULLPTR, &error);

            if ( (!error) && (l_ProxyDevice) )
            {
               ETG_TRACE_USR1((" bPhoneBookCreateRemoteProxy : Device proxy creation SUCCESS"));

               //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
               // this removes the lint warning for this block only
               OrgBluezDevice1Proxy* DeviceProxy =
                     ORG_BLUEZ_DEVICE1_PROXY(l_ProxyDevice);
               if (DeviceProxy != NULL)
               {
                  // The default timeout value for DBus call in G3g is set to 300s - CMG3GB-1364
                  g_dbus_proxy_set_default_timeout(&DeviceProxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
               }
               else
               {
                  ETG_TRACE_ERR((" DeviceProxy : set timeout failed for device proxy"));
               }

               //Storing the Device path-
               ETG_TRACE_USR1((" ConnID: %d", l_ConnID));

               g_free(ObjectPathMapping[l_ConnID].DevicePath);
               ObjectPathMapping[l_ConnID].DevicePath = g_strdup(obj_path);//CMG3G-14313-Coverity Fix
               ETG_TRACE_USR4((" DevicePath: %s", ObjectPathMapping[l_ConnID].DevicePath));

               //Storing the PBAP Info
               gchar* PBAPInfo = org_bluez_device1_dup_pbapinfo(l_ProxyDevice);
               ETG_TRACE_USR4((" PBAPInfo retrieved: %s", PBAPInfo));

               guint16 PBAPVersion = 0xFFFF;
               if(PBAPInfo)
               {
                  sscanf(PBAPInfo, PBAPInfoFormat.c_str(), &PBAPVersion, &deviceObject->m_u8FAVSPDDwnldSupport);
#ifdef PBDL_SKIP_SIM_LIST
                  //Based on the project configurtion, sim download will be skipped
                  deviceObject->m_u8FAVSPDDwnldSupport = deviceObject->m_u8FAVSPDDwnldSupport & 0xFD;
#endif
                  ETG_TRACE_USR4((" PBAPVersion retrieved: 0x%04x", PBAPVersion));
                  ETG_TRACE_COMP((" -PBDL.S- PBAPSupportRep retrieved: 0x%02x", deviceObject->m_u8FAVSPDDwnldSupport));
               }
               else
               {
                  ETG_TRACE_ERR((" PBAPInfo is null!"));
               }

               g_free(PBAPInfo);
            }

            tVoid* pCustodialLose = obj_path; //To remove LINT warning : Custodial Pointer
            (tVoid) pCustodialLose;

            if ( (error) || (!l_ProxyDevice))
            {
               printErrorMessage(error, "EvoPhoneBook_Create_RemoteProxy_Device");
            }

         }
      }
	  //CMG3G-14313-Coverity Fix
	  if(obj_path)
	  {
		  free(obj_path);
		  ETG_TRACE_USR4(("freeing obj_path"));
	  }
	  ETG_TRACE_USR4((" DevicePath: %s", ObjectPathMapping[l_ConnID].DevicePath));
	  //end of Coverity Fix
   }
   return l_ProxyDevice;
}
/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Create_RemoteProxy_Transfer
 * DESCRIPTION:  Create Remote Object proxy for Transfer
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
OrgBluezObexTransfer* EvoPhoneBook_Create_RemoteProxy_Transfer(gchar* transferPath)
{
   ETG_TRACE_USR1(("EvoPhoneBook_Create_RemoteProxy_Transfer entered"));

   OrgBluezObexTransfer* l_ProxyTransfer = NULL;
   GError *error = NULL;

   //Creating the transfer Proxy.
   l_ProxyTransfer
         = org_bluez_obex_transfer_proxy_new_sync(pBConnection_bus_Session_Evo, G_DBUS_PROXY_FLAGS_NONE, BASE_SERVICE_NAME_EVO, transferPath, NULLPTR, &error);

   if ( (!l_ProxyTransfer) || (error) )
   {
      printErrorMessage(error, "EvoPhoneBook_Create_RemoteProxy_Transfer");
      return NULL;
   }

   //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
   // this removes the lint warning for this block only
   OrgBluezObexTransferProxy* TransferProxy =
         ORG_BLUEZ_OBEX_TRANSFER_PROXY(l_ProxyTransfer);
   if (TransferProxy != NULL)
   {
      // The default timeout value for DBus call in G3g is set to 300s - CMG3GB-1364
      g_dbus_proxy_set_default_timeout(&TransferProxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
   }
   else
   {
      ETG_TRACE_ERR(("EvoPhoneBook_Create_RemoteProxy_Transfer : set timeout failed for client proxy"));
   }

   return l_ProxyTransfer;
}

/*******************************************************************************
 *
 * FUNCTION: AssignSessionBusConnection
 * DESCRIPTION: gets GDBUS session bus connection.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void AssignSessionBusConnection(GDBusConnection *sessionBusConnection)
{
   ETG_TRACE_USR4((" AssignSessionBusConnection entered"));

   pBConnection_bus_Session_Evo = sessionBusConnection;

   //Create the Client proxy
   EvoPhoneBook_Create_RemoteProxy_Client();

   //Create the object Manager proxy
   EvoPhoneBook_Create_RemoteProxy_ObjectManager();

}

/*******************************************************************************
 *
 * FUNCTION: AssignSystemBusConnection
 * DESCRIPTION: gets GDBUS System bus connection.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void AssignSystemBusConnection(GDBusConnection *systemBusConnection)
{
   ETG_TRACE_FATAL((" AssignSystemBusConnection entered"));

   pBConnection_bus_System_Evo = systemBusConnection;

}

/*******************************************************************************
 *
 * FUNCTION: RemoveSystemBusConnection
 * DESCRIPTION: removes GDBUS System bus connection.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void RemoveSystemBusConnection()
{
   ETG_TRACE_FATAL((" RemoveSystemBusConnection entered"));
   pBConnection_bus_System_Evo = NULLPTR;
}
/*******************************************************************************
 *
 * FUNCTION: RemoveSessionBusConnection
 * DESCRIPTION: removes GDBUS System bus connection.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void RemoveSessionBusConnection()
{
   ETG_TRACE_USR1((" RemoveSessionBusConnection entered"));

   pBConnection_bus_Session_Evo = NULLPTR;

   vUnRegisterClientAndObjectManagerSignals();
}


/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Select_PhoneBook
 * DESCRIPTION:  Select the PhoneBook object for other operations
 * PARAMETER: [IN] DeviceObject - Device object of the remote device.
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
int EvoPhoneBook_Select_PhoneBook(FC_Device_Details *deviceObject, const gchar* arg_location, const gchar* arg_phonebook)
{
   ETG_TRACE_USR1(("EvoPhoneBook_Select_PhoneBook entered  with deviceAddress  ::%s", deviceObject->getDeviceAddress()));

   OrgBluezObexPhonebookAccess* l_poProxyPbAccess = NULL;
   GError *error = NULL;
   gboolean ret = FALSE;

   if (deviceObject)
   {
      guint8 l_ConnID = (guint8)((deviceObject->getConnectionID()) - 1);
      ETG_TRACE_USR4(("ConnID = %d", l_ConnID));

      l_poProxyPbAccess = ProxyConnMapping[l_ConnID].m_poProxyPbAccess;

      if (l_poProxyPbAccess == NULL)
      {
         ETG_TRACE_ERR((" l_poProxyPbAccess is NULL!!!"));
         return E_FAILURE;
      }

      ret
            = org_bluez_obex_phonebook_access_call_select_sync(l_poProxyPbAccess, arg_location, arg_phonebook, NULLPTR, &error);
      if (ret == FALSE)
      {
         printErrorMessage(error, "EvoPhoneBook_Select_PhoneBook");
      }
      else
      {
         ETG_TRACE_USR1(("EvoPhoneBook_Select_PhoneBook success"));

         //Get size of selected phone book - 528902- Quick dial and call history is not getting downloaded after IGN cycle
         int RET_VALUE = EvoPhoneBook_Get_Size(deviceObject);
         if(E_FAILURE == RET_VALUE)
         {
            if(  (VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_CCH, arg_phonebook) ) ||
                 (VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_ICH, arg_phonebook) ) ||
                 (VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_OCH, arg_phonebook) ) ||
                 (VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_MCH, arg_phonebook) )  )
            {
               updateCHErrorStatusToClients();
            }/* We are already updating the errorstate to clients through printerrormessage function call for pb,spd,sim,fav
            else if(  (VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_PB, arg_phonebook)) ||
                      (VALUE_ZERO == u8StringCompare(VCARD_LOCATION_SIM, arg_location)) ||
                      (VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_SPD, arg_phonebook)) ||
                      (VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_FAV, arg_phonebook)) )
            {
               updatePBErrorStatusToClients();
            }*/
         }
         else
         {
            ETG_TRACE_USR1(("EvoPhoneBook_Get_Size success"));
         }
         //end of fix- 528902
      }
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_DnlGetSize
 * DESCRIPTION: This method is used to fill count for each pb type into device object variables
 * PARAMETER: [IN] deviceObject - deviceObject corresponding to device handle
 *            [OUT]
 *
 * RETURNVALUE:
 ********************************************************************************/
void EvoPhoneBook_DnlGetSize(FC_Device_Details *deviceObject)
{
   ETG_TRACE_USR1(("EvoPhoneBook_DnlGetSize entered"));

   //Fix 246286
   FC_PhoneBook_MemoryUsage *poPbMemUsage = fc_phonebook_tclService_PhoneBook::pGetInstance()->m_pPhoneBookMemoryUsage;
   // FIX GMMY15-13121 Nullpointerexception in "phonebook.out" in thread phonebook.out
   // Probably 'm_pPhoneBookMemoryUsage' (member of 'fc_phonebook_tclService_PhoneBook') was not existent, so
   // we add a check here. (-bn: Comment: I don't think that this explanation fits to the reset... To check this
   // I have added the else clause with FATAL and ERRMEM traces.. Let's see if we see such traces ..)
   if(!poPbMemUsage)
   {
      ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED:  poPbMemUsage == NULL " ));
      ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED:  poPbMemUsage == NULL " ));
      return;
   }
   else
   {
      poPbMemUsage->CalculateMemoryUsageFromDB();
   }
   //End fix  246286

   if(deviceObject)
   {
      OrgBluezObexPhonebookAccess* l_poProxyPbAccess = NULL;
      guint8 l_ConnID = (guint8)((deviceObject->getConnectionID()) - 1);
      ETG_TRACE_USR4(("ConnID = %d", l_ConnID));
      l_poProxyPbAccess = ProxyConnMapping[l_ConnID].m_poProxyPbAccess;

      if (l_poProxyPbAccess == NULL)
      {
         ETG_TRACE_ERR((" l_poProxyPbAccess is NULL!!!"));
         return;
      }
      if ((deviceObject->m_u8FAVSPDDwnldSupport & PBAP_FAV_DWNLD_SUPPORTED)
            == PBAP_FAV_DWNLD_SUPPORTED)
      {
         deviceObject->m_u32DeviceFavContactsCount
               = EvoPhoneBook_Select_GetSize(deviceObject, l_poProxyPbAccess, VCARD_LOCATION_INT, VCARD_PHONEBOOK_FAV);
      }
      if ((deviceObject->m_u8FAVSPDDwnldSupport & PBAP_SPD_DWNLD_SUPPORTED)
            == PBAP_SPD_DWNLD_SUPPORTED)
      {
         deviceObject->m_u32DeviceSpdContactsCount
               = EvoPhoneBook_Select_GetSize(deviceObject, l_poProxyPbAccess, VCARD_LOCATION_INT, VCARD_PHONEBOOK_SPD);
      }
      deviceObject->m_u32DeviceIntPbContactsCount
            = EvoPhoneBook_Select_GetSize(deviceObject, l_poProxyPbAccess, VCARD_LOCATION_INT, VCARD_PHONEBOOK_PB);

      if ((deviceObject->m_u8FAVSPDDwnldSupport & PBAP_SIM_DWNLD_SUPPORTED)
            == PBAP_SIM_DWNLD_SUPPORTED)
      {
         deviceObject->m_u32DeviceSimContactsCount
               = EvoPhoneBook_Select_GetSize(deviceObject, l_poProxyPbAccess, VCARD_LOCATION_SIM, VCARD_PHONEBOOK_PB);
      }

      deviceObject->m_u32DeviceContactsCount
            = (deviceObject->m_u32DeviceIntPbContactsCount
                  + deviceObject->m_u32DeviceSimContactsCount);
      ETG_TRACE_COMP((" Total number of Contacts available in the connected device is (Both PB and SIM) is %u", deviceObject->m_u32DeviceContactsCount));

      if((unsigned int)getMaxContactCount() < deviceObject->m_u32DeviceContactsCount)
      {
         if (true == updateMaxContactCount(deviceObject->m_u32DeviceContactsCount))
         {
            ETG_TRACE_USR1(("EvoPhoneBook_DnlGetSize TotalContact count updated to clients"));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Deviceobject is NULL"));
   }

   ETG_TRACE_USR1(("EvoPhoneBook_DnlGetSize exited"));
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Select_GetSize
 * DESCRIPTION: This method is used for select and retrieving the get_size of different phonebook types
 * PARAMETER: [IN]  deviceObject - deviceObject corresponding to device handle
 *            [IN] l_poProxyPbAccess - Phonebook access proxy
 *            [IN] arg_location - Either INT/SIM
 *            [IN] arg_phonebook - Either SPD/FAV/PB/ICH/OCH/MCH/CCH
 *            [OUT]
 *
 * RETURNVALUE: Out_Size - Size for respective phonebook type
 ********************************************************************************/
quint32 EvoPhoneBook_Select_GetSize(FC_Device_Details *deviceObject, OrgBluezObexPhonebookAccess *l_poProxyPbAccess, const gchar* arg_location, const gchar* arg_phonebook)
{
   ETG_TRACE_USR1(("EvoPhoneBook_Select_GetSize entered"));

   GError *error = NULL;
   gboolean ret = FALSE;
   ETG_TRACE_COMP((" -PBDL.EVO.S- EvoPhoneBook_Select_GetSize - Location: %s ", arg_location));
   ETG_TRACE_COMP((" -PBDL.EVO.S- EvoPhoneBook_Select_GetSize - Phonebook: %s ", arg_phonebook));
   //TO DO : Have to check whether we can make asynchronous call for get_size
   ret
         = org_bluez_obex_phonebook_access_call_select_sync(l_poProxyPbAccess, arg_location, arg_phonebook, NULLPTR, &error);
   guint16 out_size = 0;
   if (ret == TRUE)
   {
         ret
             = org_bluez_obex_phonebook_access_call_get_size_sync(l_poProxyPbAccess, &out_size, NULLPTR, &error);
         if (FALSE == ret)
         {
            ETG_TRACE_USR1(("Error during getsize"));
            printErrorMessage(error, arg_phonebook);
            out_size = 0;
         }
   }
   else
   {
      ETG_TRACE_USR1(("Error during select"));
      printErrorMessage(error, arg_phonebook);
   }

   //Fix RTC 176167 , 239517, 240713 & NCG3D-101820
   //Though, some devices support SIM, FAV & SPD contacts download (as per PBAPInfo), While requesting for Get_Size()
   //device responded with an error as below. If this scenario detects, FC_Phonebook prepares a download array in such a way that
   //SIM/SPD/FAV  contacts download will be ignored.
   //"error_name=org.bluez.obex.Error.Failed" and "string = GDBus.Error:org.bluez.obex.Error.Failed: operation fail" (from session logs)
   if((FALSE == ret) && (error))
   {
      deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
      ETG_TRACE_USR1(("Error message is :%s", error->message));
      if (  (VALUE_ZERO == u8StringCompare(error->message, DBUS_ERROR_OBEX_OPERATION_FAILED)) ||
            (VALUE_ZERO == u8StringCompare(error->message, DBUS_ERROR_OPERATION_FAILED))  )
      {
         if ((VALUE_ZERO == u8StringCompare(VCARD_LOCATION_SIM, arg_location)))
         {
            //Masking SIM contacts
            ETG_TRACE_USR1(("Masking SIM contacts"));
            deviceObject->m_u8FAVSPDDwnldSupport = deviceObject->m_u8FAVSPDDwnldSupport & 0xFD;
         }
         else if ((VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_SPD, arg_phonebook)))
         {
            //Masking Speed dial contacts
            ETG_TRACE_USR1(("Masking Speed dial contacts"));
            deviceObject->m_u8FAVSPDDwnldSupport = deviceObject->m_u8FAVSPDDwnldSupport & 0xFB;
         }
         else if ((VALUE_ZERO == u8StringCompare(VCARD_PHONEBOOK_FAV, arg_phonebook)))
         {
            //Masking Favorites contacts
            ETG_TRACE_USR1(("Masking Favorites contacts"));
            deviceObject->m_u8FAVSPDDwnldSupport = deviceObject->m_u8FAVSPDDwnldSupport & 0xF7;
         }
         else
         {
            ETG_TRACE_USR1(("There is an error in INT,PB get_size"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("Error different from org.bluez.Error.Failed or org.bluez.obex.Error.Failed"));
      }
   }
   //End Fix RTC 176167 , 239517 , 240713 & NCG3D-101820

   //Solve coverity warnings 104034 & 104035
   quint32 u32tmpOutSize = VALUE_ZERO;
   if (0 != out_size)
   {
      u32tmpOutSize = (quint32)(out_size);
   }

   ETG_TRACE_USR1(("EvoPhoneBook_Select_GetSize exited"));
   return u32tmpOutSize;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Pull_All
 * DESCRIPTION: Pulling all contacts from the remote device.
 * PARAMETER: [IN] DeviceObject - Device object of the remote device.
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
void EvoPhoneBook_Pull_All(FC_Device_Details *deviceObject, const gchar* targetFile, guint16 offset, guint maxcount, guint8 filter)
{
   ETG_TRACE_USR1(("EvoPhoneBook_Pull_All entered"));

   if (deviceObject)
   {
      if (targetFile)
      {
         OrgBluezObexPhonebookAccess* l_poProxyPbAccess = NULL;

         guint8 l_ConnID = (guint8)((deviceObject->getConnectionID()) - 1);
         l_poProxyPbAccess = ProxyConnMapping[l_ConnID].m_poProxyPbAccess;
         ETG_TRACE_USR1(("ConnID = %d", l_ConnID));

         if (l_poProxyPbAccess != NULL)
         {
            GVariantBuilder *pAttributesBuilder =
                  g_variant_builder_new(G_VARIANT_TYPE("as"));
            g_variant_builder_add(pAttributesBuilder, "s", "VERSION");
            g_variant_builder_add(pAttributesBuilder, "s", "FN");
            g_variant_builder_add(pAttributesBuilder, "s", "N");
            g_variant_builder_add(pAttributesBuilder, "s", "TEL");
            g_variant_builder_add(pAttributesBuilder, "s", "ADR");
            g_variant_builder_add(pAttributesBuilder, "s", "EMAIL");
            g_variant_builder_add(pAttributesBuilder, "s", "ORG");
            //292459-Phonetics column added to recognize (YOMI field Phonetics-vCard attribute for SDS) to differentiate the phonetic transcription either Hiragana or Katakana
            //  Incase of iPhone, we use the "PRIVATE_PROPERTY"(from the org.bluez.obex.PhonebookAccess.ListFilterFields supported by ALPS stack) to represent "X-PHONETIC-FIRST-NAME" & "X-PHONETIC-LAST-NAME"
            //  proprietary fields. Incase of Android phone device "SORT-STRING" attribute is directly indicated using the same string name in ListFilterFields.
            //  array [
            //          string "VERSION"
            //          string "FN"
            //          ........
            //          string "SORT-STRING"
            //          string "PRIVATE_PROPERTY"
            //        ]
            //  We cannot directly add the attributes in the below manner as these are not supported by the ALPS stack
            //  g_variant_builder_add(pAttributesBuilder, "s", "X-PHONETIC-FIRST-NAME");
            //  g_variant_builder_add(pAttributesBuilder, "s", "X-PHONETIC-LAST-NAME");
            //
            //end of fix
            g_variant_builder_add(pAttributesBuilder, "s", "SORT-STRING");
            g_variant_builder_add(pAttributesBuilder, "s", "SOUND");
            g_variant_builder_add(pAttributesBuilder, "s", "PRIVATE_PROPERTY");
            g_variant_builder_add(pAttributesBuilder, "s", "X-IRMC-CALL-DATETIME");

            if (EVO_ENABLE_V_CARD_PICTURE_ICON == filter)
            {
               g_variant_builder_add(pAttributesBuilder, "s", "PHOTO");
            }

            GVariantBuilder *pArgFilterBuilder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
            g_variant_builder_add(pArgFilterBuilder, "{sv}", "Format", g_variant_new_string("vcard30"));
            g_variant_builder_add(pArgFilterBuilder, "{sv}", "Order", g_variant_new_string("indexed"));
            g_variant_builder_add(pArgFilterBuilder, "{sv}", "Offset", g_variant_new_uint16(offset));
            g_variant_builder_add(pArgFilterBuilder, "{sv}", "MaxCount", g_variant_new_uint16((guint16)maxcount));
            g_variant_builder_add(pArgFilterBuilder, "{sv}", "Fields", g_variant_new("as", pAttributesBuilder));

            GVariant *pvArgFilter = g_variant_builder_end(pArgFilterBuilder);

            ETG_TRACE_COMP((" -PBDL.EVO.S- PULL_ALL - Offset: %u", offset));
            ETG_TRACE_COMP((" -PBDL.EVO.S- PULL_ALL - Maxcount: %u", maxcount));
            gchar* p_dupAddress = g_strdup(deviceObject->getDeviceAddress());
            if (NULLPTR != p_dupAddress)
            {
            	ETG_TRACE_USR4(("Calling pull all for address %s", p_dupAddress ));
            	org_bluez_obex_phonebook_access_call_pull_all(l_poProxyPbAccess, targetFile, pvArgFilter, NULLPTR, pullAllPhoneBookCallback, p_dupAddress);
            }
            else
            {
            	ETG_TRACE_FATAL((" Unable to allocate memory for device address!!"));
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL!!! "));
   }
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Pull_All
 * DESCRIPTION: Pulling all contacts from the remote device.
 * PARAMETER: [IN] GObject
 *            [IN] AsyncResult
 *            [OUT] UserData which is a device Object
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
void pullAllPhoneBookCallback(GObject* pGObject, GAsyncResult* pGAsyncResult, gpointer pUserData)
{
   GError* error = NULL;
   gchar* out_transfer = NULLPTR;
   gboolean bRet = FALSE;
   FC_Device_Details *deviceObject = NULLPTR;
   tChar l_devAddr[LENGTH_OF_DEVICE_ADDRESS];
   memset(l_devAddr, 0, LENGTH_OF_DEVICE_ADDRESS);
   if (NULL != pUserData)
   {
	   vStringCopy(l_devAddr, (tChar*)pUserData, LENGTH_OF_DEVICE_ADDRESS);
	   g_free (pUserData);
   }
   else
   {
	   ETG_TRACE_FATAL((" pullAllPhoneBookCallback user data NULL"));
	   return;
   }
   bRet = org_bluez_obex_phonebook_access_call_pull_all_finish((OrgBluezObexPhonebookAccess*) pGObject, &out_transfer, pGAsyncResult, &error);
   QMutexLocker lock(&g_Phonebook_DeviceMutex);
   deviceObject = FC_Device_Details::deviceAddressToObjectInstance (l_devAddr);
   if (NULL == deviceObject)
   {
	   ETG_TRACE_ERR((" Device object is NULL!!! "));
	   return;
   }
   guint8 l_ConnID = (guint8)((deviceObject->getConnectionID()) - 1);
   if (bRet == FALSE)
   {
      //When EvoPhoneBook_Pull_All for CallHistory download is not successful, the download state is set to most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::FI_EN_E8RCDS_ERROR
      //When the error received is "GDBus.Error:org.bluez.Error.Failed: Session busy" we go for a CallHistory_vStartRetry by posting a loopback message from the EvodbusClient to the Phonebook service.
      //NOTE:Upon receiving the Session busy error from Stack, we go for a SINGLE retry.
      if((deviceObject->downloadType == DownloadOutGoingCallHistorySimLocal) ||
            (deviceObject->downloadType == DownloadMissedCallHistorySimLocal) ||
            (deviceObject->downloadType == DownloadIncomingCallHistorySimLocal) ||
            (deviceObject->downloadType == DownloadCombinedCallHistorySimLocal ) )
      {
         ETG_TRACE_FATAL((" EvoPhoneBook_Pull_All_Callback Error during CH download"));
         updateCHErrorStatusToClients();
         if (FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS == deviceObject->GetDownloadState())
         {
            deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
         }

         if(NULLPTR == error)
         {
            ETG_TRACE_FATAL((" Error is NULL"));
         }
         else
         {
            ETG_TRACE_FATAL((" EvoPhoneBook_Pull_All_Callback Error during CH download: %s", g_dbus_error_get_remote_error(error)));
            ETG_TRACE_FATAL(("Error->message: %s", error->message));

            if( (VALUE_ZERO == u8StringCompare(error->message, DBUS_ERROR_SESSION_BUSY) ) &&
                  (fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.u8GetCHRetryCount() < NO_OF_CH_RETRIES )  )
            {
               PhBk_Loopback_Data DnlStartCB_LB_Data;
               memset((void*) &DnlStartCB_LB_Data, 0, sizeof(PhBk_Loopback_Data));
               DnlStartCB_LB_Data.u16FunctionID = (guint16) FC_PB_LB_FID_DNL_START_CB;
               vStringCopy(DnlStartCB_LB_Data.ucDevAddr, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
               ETG_TRACE_USR1(("DnlStopCB_LB_Data.ucDevAddr: %s", DnlStartCB_LB_Data.ucDevAddr));
               DnlStartCB_LB_Data.blFailed = TRUE;
               ETG_TRACE_USR4(("Posting Loopback message for FID- FC_PB_LB_FID_DNL_START_CB"));
               EvopostLoopBackMessage((guint32) FC_PB_LB_FID_DNL_START_CB, &DnlStartCB_LB_Data);
            }
            else
            {
               ETG_TRACE_USR4(("Either reached Max.retries for CH download or Error message is not Session Busy" ));
            }
         }
      }
      else if (deviceObject->downloadType ==  DownLoadPBSim ||
            deviceObject->downloadType == DownLoadPBLocal ||
            deviceObject->downloadType == DownloadPBSpeedDial ||
            deviceObject->downloadType == DownloadPBFavorites)
      {
         // Hack: Sleep should be avoided (BG: This is implemented as a hack for customer ticket NCG3D-127094)
         EvoPhonebook_PullAll_Delay();
         printErrorMessage(error, "EvoPhoneBook_Pull_All_Callback");
		 // Note: In printErrorMessage, also the message updatePBErrorStatusToClients is called for dedicated errors. 
         // printErrorMessage and updatePBErrorStatusToClients should be separated
         if (FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS == deviceObject->GetDownloadState())
         {
            deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
         }

         if(NULLPTR == error)
         {
            ETG_TRACE_FATAL((" Error is NULL"));
         }
         else
         {
            ETG_TRACE_FATAL((" EvoPhoneBook_Pull_All_Callback Error during PB download: %s", g_dbus_error_get_remote_error(error)));
            ETG_TRACE_FATAL(("Error->message: %s", error->message));

            if (  (VALUE_ZERO == u8StringCompare(error->message, DBUS_ERROR_SESSION_BUSY) ) &&(FALSE == bRePullOnceOnPBError )  )
            {
               EvoPhonebook_Start_PullAll(deviceObject);
               bRePullOnceOnPBError = TRUE;
            }
         }
      }
   }
   else
   {
      bRePullOnceOnPBError = FALSE;
	  if (NULLPTR != out_transfer)
	  {
	      ObjectPathMapping[l_ConnID].TransferPath = g_strdup(out_transfer);
	      ETG_TRACE_USR1(("ObjectPathMapping[ConnID].TransferPath = %s", ObjectPathMapping[l_ConnID].TransferPath));
	      ProxyConnMapping[l_ConnID].m_poProxyTransfer = EvoPhoneBook_Create_RemoteProxy_Transfer(ObjectPathMapping[l_ConnID].TransferPath);
	      ETG_TRACE_USR1((" Registering for property-changed... "));
	      RegisterForPropertiesChanged(deviceObject->getConnectionID());
	  }
	  else
	  {
	      ETG_TRACE_FATAL((" out_transfer transfer path is NULL"));
	  }
   }
}
/*******************************************************************************
 *
 * FUNCTION: EvoPhonebook_PullAll_Delay
 * DESCRIPTION: Injecting NanoSleep through this function
 * PARAMETER: [IN] .
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
void EvoPhonebook_PullAll_Delay()
{
   // Do a sleep which, if it gets interrupted by any 'signal', gets restarted with the needed "remaining"
   // sleep time.
   // Note: We use "nanosleep(..)" for this purpose as it is based on the CLOCK_MONOTONIC clock in Linux
   //       and therefore is not affected by any changes of the "system time", see also:
   //       http://man7.org/linux/man-pages/man2/nanosleep.2.html

   // Define reqested sleep time in ms
   long milisec = 100L;

   // Declaration of needed "timespec" variables
   struct timespec req = {0,0};   // req|ested time interval   [input to nanosleep(..)]
   struct timespec rem = {0,0};   // rem|aining time interval  [returned by nanosleep(..), if it gets interrupted]

   // Set requested time interval to "req|ested" sleep time
   req.tv_sec =  milisec / 1000;                // type is "time_t"
   req.tv_nsec = (milisec % 1000) * 1000000L;   // type is "long"

   ETG_TRACE_COMP(("pullAllPhoneBookCallback::EvoPhonebook_PullAll_Delay ErrorInjection: Sleep: req.tv_sec= %d, req.tv_nsec= %d. ", req.tv_sec, req.tv_nsec ));

   while(1)
   {
      if(nanosleep(&req, &rem) == -1)
      {
         if(errno == EINTR)
         {
            ETG_TRACE_COMP(("pullAllPhoneBookCallback::EvoPhonebook_PullAll_Delay ErrorInjection: Sleep: got interrupted." ));
            // Use the "rem|aining" time as new "req|ested" time for a next call of "nanosleep(..)"
            req.tv_sec =  rem.tv_sec;
            req.tv_nsec = rem.tv_nsec;
         }
         else
         {
            ETG_TRACE_COMP(("pullAllPhoneBookCallback::EvoPhonebook_PullAll_Delay ErrorInjection: Sleep: returned with error 'EFAULT' or 'EINVAL' (NOT expected.. " ));
            // The other possible errors, i.e. EFAULT and EINVAL, are ignored here as they are not expected.
            // For "paranoia" reasons we could add an assert here.. if wanted.
            break;
         }
      }
      else
      {
         // nanosleep(..) has elapsed without error.
         break;
      }
   }
}
/****************************************************************************************
 *
 * FUNCTION: EvoPhonebook_Start_PullAll
 * DESCRIPTION: Handling before initiating Pull all
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 *******************************************************************************************/
void EvoPhonebook_Start_PullAll(FC_Device_Details* deviceObject)
{
   ETG_TRACE_USR4((" EvoPhonebook_Start_PullAll entered"));

   guint8 filter = downLoadFilter;
   guint16 offset = 0;
   const gchar* targetFile = NULL;
   guint MaxEntriesToBeDownloaded = 0;
   guchar pbDownlaodType = 0;

   if (deviceObject)
   {
      pbDownlaodType = (guchar)(deviceObject->downloadType);

      if (INVALID_VALUE != (deviceObject->downloadType))
      {
         // Update filter for download
#ifdef CONNECTIVITY_PHOTODOWNLOAD_ENABLE
#ifdef TWO_PASS_DOWNLOAD
         if(deviceObject->blIsItFirstTimeDownload())
         {
            ETG_TRACE_USR4(("First time download with disable pic for two pass"));
            filter = EVO_DISABLE_V_CARD_PICTURE_ICON;
         }
         else
         {
            ETG_TRACE_USR4(("download with enable pic for two pass"));
            filter = EVO_ENABLE_V_CARD_PICTURE_ICON;
         }
#else
         ETG_TRACE_USR4(("download with enable pic for single pass"));
         filter = EVO_ENABLE_V_CARD_PICTURE_ICON;
#endif //TWO_PASS_DOWNLOAD
#else
         ETG_TRACE_USR4(("Download with photo disabled"));
         filter = EVO_DISABLE_V_CARD_PICTURE_ICON;
#endif //CONNECTIVITY_PHOTODOWNLOAD_DISABLE
         //FIX CMG3G-3262 Download State property implementation
         if (fc_phonebook_tclService_PhoneBook::pGetInstance() != NULL
               && (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance() != NULL)
         {
            (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetDeviceHandle(deviceObject->getDeviceHandle());
            switch ((FC_PhoneBookDwldCallType) pbDownlaodType)
            {
               case DownLoadPBSim:
               case DownLoadPBLocal:
               case DownloadPBSpeedDial:
               case DownloadPBFavorites:

                  //Pull-All iteration
                  if (TRUE == bFirstPull)
                  {
                     ETG_TRACE_USR4((" PULL_ALL - Setting for first iteration"));
                     offset = 0;
                     Pull_Iteration = 1;

                     //Update download state during the first pull
#ifdef CONNECTIVITY_PHOTODOWNLOAD_ENABLE
#ifdef  TWO_PASS_DOWNLOAD
                     if (filter == EVO_DISABLE_V_CARD_PICTURE_ICON)
                     {
                        ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO"));
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO);
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
                     }
                     else if ((filter == EVO_ENABLE_V_CARD_PICTURE_ICON))
                     {
                        ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO"));
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO);
                        (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
                     }
#else
                     ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO"));
                     (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO);
                     (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
#endif //TWO_PASS_DOWNLOAD
#else
                     ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO"));
                     (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO);
                     (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
#endif //CONNECTIVITY_PHOTODOWNLOAD_ENABLE
                  }
                  else if (TRUE == bRePullOnceOnError)
                  {
                     ETG_TRACE_USR4((" PULL_ALL - Setting for RePull on ERROR during download "));
                     offset = (guint16)(PULL_ITERATION_CONTACTS_COUNT * (Pull_Iteration - 1));
                  }
                  else
                  {
                     ETG_TRACE_USR4((" PULL_ALL - Setting for next iteration"));
                     offset = (guint16)(PULL_ITERATION_CONTACTS_COUNT * Pull_Iteration);
                     Pull_Iteration++;
                  }

                  MaxEntriesToBeDownloaded = (guint)(PULL_ITERATION_CONTACTS_COUNT);

                  targetFile = TARGET_FILE_PHONE_BOOK;
               break;

               case DownloadOutGoingCallHistorySimLocal:
               case DownloadMissedCallHistorySimLocal:
               case DownloadIncomingCallHistorySimLocal:
                  MaxEntriesToBeDownloaded = (guint) FC_PB_MAX_CALL_HISTORY_ENTRIES_PER_TYPE;
                  filter = EVO_DISABLE_V_CARD_PICTURE_ICON;

                  deviceObject->m_countDownload = 0;
                  deviceObject->downloadType = (FC_PhoneBookDwldCallType) pbDownlaodType;

                  ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS"));
                  (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS);
                  (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();

                  targetFile = TARGET_FILE_CALL_HISTORY;
               break;

               case DownloadCombinedCallHistorySimLocal:
                  MaxEntriesToBeDownloaded = (guint) MAX_NO_OF_CCH_ENTRIES;
                  filter = EVO_DISABLE_V_CARD_PICTURE_ICON;

                  deviceObject->m_countDownload = 0;
                  deviceObject->downloadType = (FC_PhoneBookDwldCallType) pbDownlaodType;

                  //On end of every CH download clear Datetime related variables so that they can be used in next download.
                  deviceObject->Device_DateTime_Support.VcardsHaveDatetime = FALSE;
                  deviceObject->Device_DateTime_Support.VcardsHaveDatetimeupdated = FALSE;
                  deviceObject->Device_DateTime_Support.CHDateTimeSupport = TRUE;
                  deviceObject->Device_DateTime_Support.DateTimesupportUpdated = FALSE;

                  ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS"));
                  (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS);
                  (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();

                  targetFile = TARGET_FILE_CALL_HISTORY;
               break;
            }
         }
         else
         {
            ETG_TRACE_ERR(("fc_phonebook_tclService_PhoneBook::pGetInstance() is NULL "));
         }
         //End of fix CMG3G-3262 Download State property implementation

         //EVO stack Pull All invocation
         EvoPhoneBook_Pull_All(deviceObject, targetFile, offset, MaxEntriesToBeDownloaded, filter);
      }
      else
      {
         //Bug 277564 - If download type is 0xff , we should send download state as an Error. This scenario can happen in following case
         //While pb download is on going, user had triggered cancel pb download "after pull is successfull and before next pull starts".
         //So, transfer path will be cleared after pull is successful and yet to create transfer path for next pull.
         //Hence, in cancel pb download proxy will be null and download array will be prepared with 0xff in all elements.
         //We should ignore 0xff if it is in download array, and should send appropriate download state updates to the clients.
         deviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
         updatePBErrorStatusToClients();
      }
   }
   else
   {
      ETG_TRACE_ERR(("Device object is NULL!!!"));
   }
}

/****************************************************************************************
 *
 * FUNCTION: getEvoPhonebookType
 * DESCRIPTION: Get the phonebook type parameter to be passed to Select method
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 *******************************************************************************************/
const gchar* getEvoPhonebookType(guchar pbDownlaodType)
{
   switch ((FC_PhoneBookDwldCallType) pbDownlaodType)
   {
      case DownLoadPBSim:
      case DownLoadPBLocal:
         return VCARD_PHONEBOOK_PB;

      case DownloadOutGoingCallHistorySimLocal:
         return VCARD_PHONEBOOK_OCH;

      case DownloadMissedCallHistorySimLocal:
         return VCARD_PHONEBOOK_MCH;

      case DownloadIncomingCallHistorySimLocal:
         return VCARD_PHONEBOOK_ICH;

      case DownloadCombinedCallHistorySimLocal:
         return VCARD_PHONEBOOK_CCH;

      case DownloadPBSpeedDial:
         return VCARD_PHONEBOOK_SPD;

      case DownloadPBFavorites:
         return VCARD_PHONEBOOK_FAV;

      default:
         return VCARD_PHONEBOOK_INVALID;
   }
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_DlComplete
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
int EvoPhoneBook_DlComplete(guint8 u8ConnectionID)
{
   ETG_TRACE_USR4((" EvoPhoneBook_DlComplete entered with ConnectionID:%d", u8ConnectionID));
   ETG_TRACE_COMP((" -PBDL.EVO.S- COMPLETE mock for EVO stack."));

   FC_Device_Details *deviceObject =
         FC_Device_Details::getDeviceObject((INDEX) (u8ConnectionID - 1));
   if (deviceObject)
   {
      PhBk_Loopback_Data DnlCompleteCB_LB_Data;
      memset((void*) &DnlCompleteCB_LB_Data, 0, sizeof(PhBk_Loopback_Data));
	  //CMG3G-14313-Coverity Fix-Using method vStringCopy instead of strncpy
      vStringCopy(DnlCompleteCB_LB_Data.ucDevAddr, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
	  //end of fix
      ETG_TRACE_USR1((" DnlCompleteCB_LB_Data.ucDevAddr: %s", DnlCompleteCB_LB_Data.ucDevAddr));

      DnlCompleteCB_LB_Data.u16FunctionID
            = (guint16)FC_PB_LB_FID_DNL_COMPLETE_CB;
      DnlCompleteCB_LB_Data.blFailed = FALSE;
      DnlCompleteCB_LB_Data.ucStatus = PB_DOWNLOAD_COMPLETE_SUCCESS;

      //This function is called from entry thread. Hence there is no need of Loopback. Directly invoke the callback.
      if (fc_phonebook_tclService_PhoneBook::pGetInstance())
      {
         fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.vDnlCompleteCallback(&DnlCompleteCB_LB_Data);
      }
   }

   return E_SUCCESS;
}

/*******************************************************************************
 *
 * FUNCTION: propertyChangedCallback
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE:
 ********************************************************************************/
void propertyChangedCallback(GObject *object, const gchar *interfaceName, GVariant *changedProperties, const gchar* const *arg_invalidated_properties)
{
   ETG_TRACE_USR1(("inside propertyChangedCallback: property is %s ", interfaceName));

   (tVoid) arg_invalidated_properties;//To solve lint

   if (!object)
   {
      ETG_TRACE_ERR((" object is NULL!!!"));
      return;
   }

   QMutexLocker lock(&g_Phonebook_DeviceMutex);

   GVariantIter iter;
   const gchar *key;
   GVariant *value;
   tU8 deviceHandle = 0;
   gboolean bCompleted = FALSE;
   int itr = 0;

   //Getting transfer path from Free Desktop proxy.

   //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
   // this removes the lint warning for this block only
   const char* obj_path = g_dbus_proxy_get_object_path((GDBusProxy*) object);
   ETG_TRACE_USR4(("obj_path: %s", obj_path));

   //Iterating through available transferPath and get corresponding deviceHandle.
   for (itr = 0; itr < MAX_DEVICE_CONNECTION_SUPPORTED; itr++)
   {
      if ( (ObjectPathMapping[itr].TransferPath) &&  obj_path)
      {
         ETG_TRACE_USR4(("ObjectPathMapping[iter].TransferPath: %s", ObjectPathMapping[itr].TransferPath));
         if (VALUE_ZERO == u8StringCompare(ObjectPathMapping[itr].TransferPath, obj_path))
         {
            deviceHandle = ObjectPathMapping[itr].DeviceHandle;
            break;
         }
      }
   }

   //The following check has been added to solve the coverity issues 64954 & 64955
   if(itr == MAX_DEVICE_CONNECTION_SUPPORTED)
   {
      return;
   }

   ETG_TRACE_USR4(("deviceHandle: %d ", deviceHandle));

   FC_Device_Details *deviceObject =
         FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);
   if (deviceObject)
   {
      g_variant_iter_init(&iter, changedProperties);

      if ((interfaceName)
            && (VALUE_ZERO == u8StringCompare(PHBK_TRANSFER_INT_NAME, interfaceName)))
      {
         while (g_variant_iter_loop(&iter, "{&sv}", &key, &value))
         {
            if (VALUE_ZERO == u8StringCompare(key, "Status"))
            {
               const gchar * TransferStatus;
               g_variant_get(value, "s", &TransferStatus);

               ETG_TRACE_USR4(("Key(Property)- %s ", key));
               ETG_TRACE_USR4(("Value(status)- %s ", TransferStatus));


               if ((TransferStatus)
                     && (VALUE_ZERO == u8StringCompare(TransferStatus, "Complete"))) //Note: "complete" changed to "Complete" based on Evo_C254.
               {
                  ETG_TRACE_COMP((" -PBDL.EVO.S- Pull completed successfully"));

                  //Delete the existing transfer path so that the same is not reconsidered in interfaces-removed.
                  //TO DO: The following clearing could be moved to entry thread for better synchronization between the entry and dbus threads.
                  ETG_TRACE_USR4((" Clearing transfer path "));
                  if (ProxyConnMapping[itr].m_poProxyTransfer)
                  {
                     g_object_unref(ProxyConnMapping[itr].m_poProxyTransfer);
                     ProxyConnMapping[itr].m_poProxyTransfer = NULLPTR;
                  }
                  g_free(ObjectPathMapping[itr].TransferPath);
                  ObjectPathMapping[itr].TransferPath = NULLPTR;

                  if (ProxyConnMapping[itr].m_poFreeDesktopProxy)
                  {
                     g_object_unref(ProxyConnMapping[itr].m_poFreeDesktopProxy);
                     ProxyConnMapping[itr].m_poFreeDesktopProxy = NULLPTR;
                  }

                  ETG_TRACE_USR4(("Pull_Iteration * PULL_ITERATION_CONTACTS_COUNT : %d", Pull_Iteration
                        * PULL_ITERATION_CONTACTS_COUNT));

                  bRePullOnceOnError = FALSE; //If the 're-pull on error' is successful, reset on success.
                  u8TransferErrorCount = 0;

                  bCompleted = bCheckIfDownloadIsComplete(deviceObject->getDeviceHandle());

                  //Parse the downloaded v-cards
                  EvoPbDlData_Signalhandler(bCompleted, deviceHandle);

               }
               else if ((TransferStatus)
                     && (VALUE_ZERO == u8StringCompare(TransferStatus, "Error")))
               {
                  ETG_TRACE_COMP((" -PBDL.EVO.S- ERROR during PULL... "));

                  //Delete the existing transfer path so that the same is not reconsidered in interfaces-removed.
                  ETG_TRACE_USR4((" Clearing transfer path "));
                  if (ProxyConnMapping[itr].m_poProxyTransfer)
                  {
                     g_object_unref(ProxyConnMapping[itr].m_poProxyTransfer);
                     ProxyConnMapping[itr].m_poProxyTransfer = NULLPTR;
                  }
                  g_free(ObjectPathMapping[itr].TransferPath);
                  ObjectPathMapping[itr].TransferPath = NULLPTR;

                  if (ProxyConnMapping[itr].m_poFreeDesktopProxy)
                  {
                     g_object_unref(ProxyConnMapping[itr].m_poFreeDesktopProxy);
                     ProxyConnMapping[itr].m_poFreeDesktopProxy = NULLPTR;
                  }

                  u8TransferErrorCount++;
                  if (u8TransferErrorCount < NO_OF_RETRIES_EVO)
                  {
                     // Proceed further with current download
                     bFirstPull = FALSE;

                     if (TRUE == bRePullOnceOnError)
                     {
                        ETG_TRACE_COMP((" -PBDL.EVO.S- Continuing with further pulls as Re-pull done once... "));
                        bRePullOnceOnError = FALSE;
                     }
                     else
                     {
                        ETG_TRACE_COMP((" -PBDL.EVO.S- Re-pulling once... "));
                        bRePullOnceOnError = TRUE;
                     }

                     quint8
                           u8PhonebookType =
                                 FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(deviceObject->getConnectionID());
                     if (DownLoadPBSim == u8PhonebookType || DownLoadPBLocal
                           == u8PhonebookType)
                     {
                        ETG_TRACE_USR4(("deviceObject->m_u32InidvidualTypeContactsCount : %d", deviceObject->m_u32InidvidualTypeContactsCount));
                        if (Pull_Iteration * PULL_ITERATION_CONTACTS_COUNT
                              < deviceObject->m_u32InidvidualTypeContactsCount || TRUE
                              == bRePullOnceOnError)
                        {
                           EvoPhonebook_Start_PullAll(deviceObject);
                        }
                        else
                        {
                           vPostDataCompleteLoopBack(deviceObject);
                        }
                     }
                     else if (DownloadPBSpeedDial == u8PhonebookType
                           || DownloadPBFavorites == u8PhonebookType)
                     {
                        ETG_TRACE_USR4(("deviceObject->m_u32VCardsAvailableForDwnld : %d", deviceObject->m_u32VCardsAvailableForDwnld));
                        if (Pull_Iteration * PULL_ITERATION_CONTACTS_COUNT
                              < deviceObject->m_u32VCardsAvailableForDwnld
                              || TRUE == bRePullOnceOnError)
                        {
                           EvoPhonebook_Start_PullAll(deviceObject);
                        }
                        else
                        {
                           vPostDataCompleteLoopBack(deviceObject);
                        }
                     }
                     else
                     {
                        // Error during Call history download
                        if (TRUE == bRePullOnceOnError)
                        {
                           EvoPhonebook_Start_PullAll(deviceObject);
                        }
                        else
                        {
                           vPostDataCompleteLoopBack(deviceObject);
                        }
                     }
                  }
                  else
                  {
                     //Stop current download - As continuously 4 pulls have resulted in failure.
                     vPostDataCompleteLoopBack(deviceObject);
                  }
               }
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL!!!"));
   }
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_Get_Size
 * DESCRIPTION:  Get the number of contacts.
 * PARAMETER: [IN] DeviceObject - Device object of the remote device.
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: proxy
 ********************************************************************************/
int EvoPhoneBook_Get_Size(FC_Device_Details *deviceObject)
{
   ETG_TRACE_USR1(("EvoPhoneBook_Get_Size entered  with deviceAddress  ::%s", deviceObject->getDeviceAddress()));
   int iRetValue = E_FAILURE;
   OrgBluezObexPhonebookAccess* l_poProxyPbAccess = NULL;

   GError *error = NULL;

   guint8 l_ConnID = (guint8)((deviceObject->getConnectionID()) - 1);
   ETG_TRACE_USR1(("ConnID = %d", l_ConnID));

   l_poProxyPbAccess = ProxyConnMapping[l_ConnID].m_poProxyPbAccess;
   if (l_poProxyPbAccess != NULL)
   {
      guint16 out_size = 0;

         //TO DO : Have to check whether we can make asynchronous call for get_size
         org_bluez_obex_phonebook_access_call_get_size_sync(l_poProxyPbAccess, &out_size, NULLPTR, &error);

         ETG_TRACE_COMP((" -PBDL.EVO.S- GET_SIZE - Total count: %d", out_size));

         PhBk_Loopback_Data DnlStartCB_LB_Data;
         memset((void*) &DnlStartCB_LB_Data, 0, sizeof(PhBk_Loopback_Data));
         DnlStartCB_LB_Data.u16FunctionID = (guint16)FC_PB_LB_FID_DNL_START_CB;
         //CMG3G-14313-Coverity Fix-Using method vStringCopy instead of strncpy
         vStringCopy(DnlStartCB_LB_Data.ucDevAddr, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
		 //end of fix
         ETG_TRACE_USR1(("DnlStartCB_LB_Data.ucDevAddr: %s", DnlStartCB_LB_Data.ucDevAddr));

         if (error != NULL)
         {
            printErrorMessage(error, "EvoPhoneBook_Get_Size");
            DnlStartCB_LB_Data.blFailed = TRUE;
            DnlStartCB_LB_Data.ucStatus = PBDL_STOPPED; //Setting status other than PBDL_SUCCESS
            DnlStartCB_LB_Data.ucErrorCode = (guchar)(error->code);
            iRetValue = E_FAILURE;
         }
         else
         {
            DnlStartCB_LB_Data.blFailed = FALSE;
            DnlStartCB_LB_Data.ucStatus = PBDL_SUCCESS;
            DnlStartCB_LB_Data.ucMissedCalls = 0; //Not required as per internal discussion.
            DnlStartCB_LB_Data.u32PhoneNumberCount = (quint32)out_size;
            iRetValue = E_SUCCESS;
         }

         //This function is called from entry thread. Hence there is no need of Loopback. Directly invoke the callback.
         if (fc_phonebook_tclService_PhoneBook::pGetInstance())
         {
            fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.vDnlStartCallback(&DnlStartCB_LB_Data);
         }
   }
   else
   {
      ETG_TRACE_ERR((" l_poProxyPbAccess is NULL "));
   }
   return iRetValue;
}

/*******************************************************************************
 *
 * FUNCTION: EvoPhoneBook_DlStop
 * DESCRIPTION:Stop phone book or call history list download.
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
int EvoPhoneBook_DlStop(guint8 u8ConnectionID)
{
   ETG_TRACE_USR4((" EvoPhoneBook_DlStop entered with ConnectionID %d", u8ConnectionID));

   OrgBluezObexTransfer* l_poProxyTransfer;
   l_poProxyTransfer = ProxyConnMapping[u8ConnectionID - 1].m_poProxyTransfer;

   if (l_poProxyTransfer == NULL)
   {
      ETG_TRACE_ERR(("EvoPhoneBook_DlStop Invalid Proxy"));
      //FIX for NCG3D-161771
      tU8 u8DeviceHandle = ObjectPathMapping[u8ConnectionID - 1].DeviceHandle;
      FC_Device_Details *deviceObject =
            FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
      if (deviceObject)
      {
         PhBk_Loopback_Data DnlStopCB_LB_Data;
         memset((void*) &DnlStopCB_LB_Data, 0, sizeof(PhBk_Loopback_Data));
         DnlStopCB_LB_Data.u16FunctionID = (guint16) FC_PB_LB_FID_DNL_STOP_CB;
         //CMG3G-14313-Coverity Fix-Using method vStringCopy instead of strncpy
         vStringCopy(DnlStopCB_LB_Data.ucDevAddr, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
         //end of fix
         DnlStopCB_LB_Data.blFailed = FALSE;
         ETG_TRACE_USR1(("DnlStopCB_LB_Data.ucDevAddr: %s", DnlStopCB_LB_Data.ucDevAddr));
         ETG_TRACE_USR4(("Posting Loopback message for FID- FC_PB_LB_FID_DNL_STOP_CB"));
         EvopostLoopBackMessage((guint32) FC_PB_LB_FID_DNL_STOP_CB, &DnlStopCB_LB_Data);
         return E_FAILURE;
      }
   }

   guint8 *pDeviceHandle = (guint8*) malloc(sizeof(guint8));

   if (!pDeviceHandle)
   {
      ETG_TRACE_ERR((" malloc failed!!!"));
      return E_FAILURE;
   }
   *pDeviceHandle = ObjectPathMapping[u8ConnectionID - 1].DeviceHandle;

   //cancel PB or CH Download.
   ETG_TRACE_COMP((" -PBDL.EVO.S- CANCEL "));
   org_bluez_obex_transfer_call_cancel(l_poProxyTransfer, NULLPTR, EvoPhoneBook_DlStop_Callback, pDeviceHandle);

   tVoid* pCustodialLose = pDeviceHandle; //To remove LINT warning : Custodial Pointer
   (tVoid) pCustodialLose;

   return E_SUCCESS;
}

/****************************************************************************************
 *
 * FUNCTION: EvoPhoneBook_DlStop_Callback
 * DESCRIPTION:Callback function is used to return the result of PhoneBook_DlStop
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 *******************************************************************************************/
void EvoPhoneBook_DlStop_Callback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4((" EvoPhoneBook_DlStop_Callback entered with device handle: %d", *((guint8*) user_data)));

   gboolean ret = FALSE;
   GError *error = NULL;

   guint8 u8DeviceHandle = *((guint8*) user_data);
   g_free(user_data);
   ETG_TRACE_USR1((" u8DeviceHandle: %d", u8DeviceHandle));

   FC_Device_Details *deviceObject =
         FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if (deviceObject)
   {
      PhBk_Loopback_Data DnlStopCB_LB_Data;
      memset((void*) &DnlStopCB_LB_Data, 0, sizeof(PhBk_Loopback_Data));
      DnlStopCB_LB_Data.u16FunctionID = (guint16) FC_PB_LB_FID_DNL_STOP_CB;
      //CMG3G-14313-Coverity Fix-Using method vStringCopy instead of strncpy
      vStringCopy(DnlStopCB_LB_Data.ucDevAddr, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
	  //end of fix
      ETG_TRACE_USR1(("DnlStopCB_LB_Data.ucDevAddr: %s", DnlStopCB_LB_Data.ucDevAddr));

      ret
            = org_bluez_obex_transfer_call_cancel_finish((OrgBluezObexTransfer*) source_object, res, &error);
      if (ret == FALSE)
      {
         DnlStopCB_LB_Data.blFailed = TRUE;
         printErrorMessage(error, "EvoPhoneBook_DlStop_Callback");
      }
      else
      {
         ETG_TRACE_USR1((" PhoneBook_DlStop_Callback Finished successfully"));
         DnlStopCB_LB_Data.blFailed = FALSE;
      }

      ETG_TRACE_USR4(("Posting Loopback message for FID- FC_PB_LB_FID_DNL_STOP_CB"));
      EvopostLoopBackMessage((guint32) FC_PB_LB_FID_DNL_STOP_CB, &DnlStopCB_LB_Data);
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL!!!"));
   }
}

/*******************************************************************************
 *
 * FUNCTION: EvopostLoopBackMessage
 * DESCRIPTION: Posts a loop back message and the related data into the data queue
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void EvopostLoopBackMessage(guint32 u32FunctionID, PhBk_Loopback_Data* LoopbackData)
{
   ETG_TRACE_USR4(("EvopostLoopBackMessage entered"));

   PhBkCCA_callbacks_Evo->vPhoneBook_PushLoopbackData(LoopbackData);
   PhBkCCA_callbacks_Evo->vPhoneBook_PostLoopback(u32FunctionID);
}

/*******************************************************************************
 *
 * FUNCTION: RegisterForPropertiesChanged
 * DESCRIPTION: Register to properties changed callback of session bus
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void RegisterForPropertiesChanged(guint8 u8ConnectionID)
{
   ETG_TRACE_USR4(("RegisterForPropertiesChanged ENTERED with u8ConnectionID: %d", u8ConnectionID));

   GError* error = NULL;
   guint8 l_ConnID = (guint8)(u8ConnectionID - 1);
   ETG_TRACE_USR1(("ConnID = %d", l_ConnID));

   if((!ObjectPathMapping[l_ConnID].TransferPath) || (!pBConnection_bus_Session_Evo))
   {
      ETG_TRACE_ERR((" ObjectPathMapping[l_ConnID].TransferPath OR pBConnection_bus_Session_Evo is NULL"));
      return;
   }

   //To receive obex transfer properties
   OrgFreedesktopDBusProperties
         * freedesktopProxy =
               org_freedesktop_dbus_properties_proxy_new_sync(pBConnection_bus_Session_Evo, G_DBUS_PROXY_FLAGS_NONE, BASE_SERVICE_NAME_EVO, ObjectPathMapping[l_ConnID].TransferPath, NULLPTR, &error);

   if ( error || (!freedesktopProxy) )
   {
      printErrorMessage(error, "freedesktopProxyCB");
      return;
   }

   ProxyConnMapping[l_ConnID].m_poFreeDesktopProxy = freedesktopProxy;

   ETG_TRACE_USR4(("Connect to signal properties-changed... "));
   ObjectPathMapping[l_ConnID].u1PropertyChangedID
         = g_signal_connect(freedesktopProxy, "properties-changed", G_CALLBACK(propertyChangedCallback), NULLPTR);
}

/*******************************************************************************
 *
 * FUNCTION: vClearEvoProxyConnMapping
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT].
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void vClearEvoProxyConnMapping(tU8 l_ConnID)
{
   ETG_TRACE_USR4((" vClearEvoProxyConnMapping entered."));

   //Unref the available proxies.
   if (ProxyConnMapping[l_ConnID].m_poProxyPbAccess != NULL)
   {
      ETG_TRACE_USR4((" Dereferencing PB Access proxy."));
      g_object_unref(ProxyConnMapping[l_ConnID].m_poProxyPbAccess );
      ProxyConnMapping[l_ConnID].m_poProxyPbAccess = NULLPTR;
   }
   if (ProxyConnMapping[l_ConnID].m_poProxyDevice != NULL)
   {
      ETG_TRACE_USR4((" Dereferencing Device proxy."));
      g_object_unref(ProxyConnMapping[l_ConnID].m_poProxyDevice );
      ProxyConnMapping[l_ConnID].m_poProxyDevice = NULLPTR;
   }
   if (ProxyConnMapping[l_ConnID].m_poFreeDesktopProxy != NULL)
   {
      ETG_TRACE_USR4((" Dereferencing Free Desktop proxy."));
      g_object_unref(ProxyConnMapping[l_ConnID].m_poFreeDesktopProxy );
      ProxyConnMapping[l_ConnID].m_poFreeDesktopProxy = NULLPTR;
   }
   if (ProxyConnMapping[l_ConnID].m_poProxyTransfer != NULL)
   {
      ETG_TRACE_USR4((" Dereferencing Transfer proxy."));
      g_object_unref(ProxyConnMapping[l_ConnID].m_poProxyTransfer );
      ProxyConnMapping[l_ConnID].m_poProxyTransfer = NULLPTR;
   }
}

/*******************************************************************************
 *
 * FUNCTION: vClearEvoObjectPathMapping
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void vClearEvoObjectPathMapping(tU8 l_ConnID)
{
   ETG_TRACE_USR4((" vClearEvoObjectPathMapping entered."));

   //Removing the session path.
   ETG_TRACE_USR4((" Clearing SessionPath."));
   g_free(ObjectPathMapping[l_ConnID].SessionPath);
   ObjectPathMapping[l_ConnID].SessionPath = NULLPTR;

   //Removing the Transfer path.
   ETG_TRACE_USR4((" Clearing TransferPath."));
   g_free(ObjectPathMapping[l_ConnID].TransferPath);
   ObjectPathMapping[l_ConnID].TransferPath = NULLPTR;

   //Removing the Device path.
   ETG_TRACE_USR4((" Clearing DevicePath."));
   g_free(ObjectPathMapping[l_ConnID].DevicePath);
   ObjectPathMapping[l_ConnID].DevicePath = NULLPTR;

   ETG_TRACE_USR4((" Clearing DeviceHandle."));
   ObjectPathMapping[l_ConnID].DeviceHandle = 0;

   ETG_TRACE_USR4((" Clearing PropertyChangedID."));
   ObjectPathMapping[l_ConnID].u1PropertyChangedID = 0;
}

/*******************************************************************************
 *
 * FUNCTION: postDataCompleteLoopBack
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void vPostDataCompleteLoopBack(FC_Device_Details * deviceObject)
{
   ETG_TRACE_USR4((" vPostDataCompleteLoopBack entered."));

   u8TransferErrorCount = 0; //Reset the error count as current download is over.

   if (deviceObject)
   {
      tU8 ConnID = (tU8)((deviceObject->getConnectionID()) - 1);
      //disconnect signal handler
      if ( (ProxyConnMapping[ConnID].m_poFreeDesktopProxy) && 0
            != ObjectPathMapping[ConnID].u1PropertyChangedID)
      {
         g_signal_handler_disconnect(ProxyConnMapping[ConnID].m_poFreeDesktopProxy, ObjectPathMapping[ConnID].u1PropertyChangedID);
      }

      // Post loop back message, BEGIN
      PhBk_Loopback_Data DataCompleteCB_LB_Data;
      memset((void*) &DataCompleteCB_LB_Data, 0, sizeof(PhBk_Loopback_Data));
      DataCompleteCB_LB_Data.u16FunctionID
            = (guint16)FC_PB_LB_FID_DATA_COMPLETE_SIG;
      memcpy(DataCompleteCB_LB_Data.ucDevAddr, deviceObject->getDeviceAddress(), LENGTH_OF_DEVICE_ADDRESS);
      DataCompleteCB_LB_Data.ucStatus = 0x00; //Status = COMPLETED

      ETG_TRACE_USR1(("PhBk_Loopback_Data.ucDevAddr: %s", DataCompleteCB_LB_Data.ucDevAddr));
      ETG_TRACE_USR4(("Posting Loopback message for FID- FC_PB_LB_FID_DATA_COMPLETE_SIG"));

      EvopostLoopBackMessage((guint32)FC_PB_LB_FID_DATA_COMPLETE_SIG, &DataCompleteCB_LB_Data);
      // Post loop back message, END
   }
   else
   {
      ETG_TRACE_ERR((" Device object is NULL!!!"));
   }
}

/*******************************************************************************
 * FUNCTION: printErrorMessage
 * DESCRIPTION: Function to print the error messages
 *
 * PARAMETER: [IN] error,       Type - GError*
 *            [IN] description, Type - const char *
 *            [OUT] None
 *
 * RETURNVALUE: None.
 ********************************************************************************/
void printErrorMessage(GError* error, const char* description)
{
   const gchar* printmsg = "unknown error message";
   gchar* errmsg = NULLPTR;
   if (error)
   {
      errmsg = g_dbus_error_get_remote_error(error);
      if (errmsg)
      {
         printmsg = errmsg;
      }
   }

   ETG_TRACE_FATAL(("Error message for: %s", (description ? description : "")));
   ETG_TRACE_FATAL(("Error message: %s", printmsg));

   //Fix RTC 176167 , 239517 & 240713, NCG3D-103980
   //Fix RTC 219996 - Stack sends "UnknownMethod" error sometimes, when FC_Phonebook made a stack call if interface removed

   if ( (VALUE_ZERO == u8StringCompare(printmsg, "org.bluez.obex.Error.Failed")) ||
        (VALUE_ZERO == u8StringCompare(printmsg, "org.bluez.Error.Failed")) ||
        (VALUE_ZERO == u8StringCompare(printmsg, "org.freedesktop.DBus.Error.UnknownMethod")) )
   {
      updatePBErrorStatusToClients();
   }
   //End Fix RTC 176167 , 239517 & 240713, NCG3D-103980
   g_free(errmsg); // if errmsg is NULL g_free simply returns
}
//NCG3D-103980
/*********************************************************************************
 *
 * FUNCTION: updateCHErrorStatusToClients
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE:
 ********************************************************************************/
void updateCHErrorStatusToClients()
{
   if ( ( fc_phonebook_tclService_PhoneBook::pGetInstance() ) && ( (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()))
   {
      ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::FI_EN_E8RCDS_ERROR"));
      (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::FI_EN_E8RCDS_ERROR);
      (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
   }
}

/*******************************************************************************
 *
 * FUNCTION: updatePBErrorStatusToClients
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE:
 ********************************************************************************/
void updatePBErrorStatusToClients()
{
   if ( ( fc_phonebook_tclService_PhoneBook::pGetInstance() ) && ( (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()))
   {
      ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_ERROR"));
      (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_ERROR);
      (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
   }
}
//end of fix NCG3D-103980
/*******************************************************************************
 *
 * FUNCTION: interfacesAddedCallback
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE:
 ********************************************************************************/
void interfacesAddedCallback(OrgFreedesktopDBusObjectManager *object, const gchar *arg_object, GVariant *arg_interfaces)
{
   ETG_TRACE_USR4((" interfacesAddedCallback entered with arg_object: %s", arg_object));
   (void)arg_interfaces;

   if (!object)
   {
      ETG_TRACE_ERR((" object is NULL!!!"));
      return;
   }

#ifdef CONNECT_PBAP_MAP_VIA_BM_APPL
   gboolean bIsSession = FALSE;
   gboolean bIsPBAPSession = FALSE;

   GVariantIter iter;
   GVariantIter iter2;
   gchar *interface_name = NULL;
   GVariant *interface, *properties;
   const gchar *key = NULL;
   GVariant *value = NULL;
   gchar *dest = NULL;
   gchar* achDevAddr = NULL;

   char achDeviceAddr[LENGTH_OF_DEVICE_ADDRESS];
   memset(achDeviceAddr, 0, LENGTH_OF_DEVICE_ADDRESS);
   memcpy(achDeviceAddr, "dev", 3);

   // Check if the added interface is a PBAP session
   if (g_str_has_prefix(arg_object, PHBK_SESSION_PATH_PREFIX))
   {
      // The prefix has been found, now we "skip" it and then analyze the rest of the string
      const gchar *temp = arg_object + strlen(PHBK_SESSION_PATH_PREFIX);

      // Now after the prefix "/org/bluez/obex/client/session" has been skipped, we expect only a number (session number) and no further path extension.
      if (!strchr(temp, '/'))
      {
         // The object_path is a (new) "session", either be PBAP or MAP. The details has to be found out.
         bIsSession = TRUE;
         ETG_TRACE_USR4((" The object_path is a session!"));
      }
   }

   if (TRUE == bIsSession)
   {
      g_variant_iter_init(&iter, arg_interfaces);
      //Solving gen4 lint warning
      interface = g_variant_iter_next_value(&iter);
      while (interface)
      {
         g_variant_get(interface, "{s*}", &interface_name, &properties);
         if (g_strcmp0(interface_name, "org.bluez.obex.PhonebookAccess") == 0)
         {
            ETG_TRACE_USR4((" The object_path is a PBAP session!"));
            bIsPBAPSession = TRUE;
         }
         else if (g_strcmp0(interface_name, "org.bluez.obex.Session") == 0)
         {
            //Find out for which device this property update belongs to
            if (g_strcmp0(g_variant_get_type_string(properties), "a{sv}") == 0)
            {
               g_variant_iter_init(&iter2, properties);
               while (g_variant_iter_next(&iter2, "{sv}", &key, &value))
               {
                  if (g_strcmp0(key, "Destination") == 0)
                  {
                     if (g_strcmp0(g_variant_get_type_string(value), "s") == 0)
                     {
                        g_variant_get(value, "s", &dest);

                        //Convert address from format aa:bb:cc:dd:ee:ff to dev_aa_bb_cc_dd_ee_ff
                        sscanf(dest, "%c%c:%c%c:%c%c:%c%c:%c%c:%c%c", &achDeviceAddr[4], &achDeviceAddr[5], &achDeviceAddr[7], &achDeviceAddr[8], &achDeviceAddr[10], &achDeviceAddr[11], &achDeviceAddr[13], &achDeviceAddr[14], &achDeviceAddr[16], &achDeviceAddr[17], &achDeviceAddr[19], &achDeviceAddr[20]);

                        for (guint i = 3; i <= 18; i = i + 3)
                        {
                           achDeviceAddr[i] = '_';
                        }
                        achDeviceAddr[21] = '\0';

                        //Note: Destination address received is in Upper case. For use covert to Lower case.
                        achDevAddr = g_ascii_strdown(achDeviceAddr, -1);
                        ETG_TRACE_USR1((" Address retrieved: %s", achDevAddr));
                     }
                  }
               }
            }
         }
         interface = g_variant_iter_next_value(&iter);
      }

      if (TRUE == bIsPBAPSession)
      {
         QMutexLocker lock(&g_Phonebook_DeviceMutex);
         FC_Device_Details
         *deviceObject =
         FC_Device_Details::deviceAddressToObjectInstance(achDevAddr);
         if (deviceObject)
         {
            guint8 l_ConnID = static_cast<guint8>(deviceObject->getConnectionID() - 1);
            ObjectPathMapping[l_ConnID].SessionPath = g_strdup(arg_object);
            ETG_TRACE_COMP((" -PBDL.S- SessionPath retrieved from interfacesAddedCallback: %s", ObjectPathMapping[l_ConnID].SessionPath));

            //Fix NCG3D-65982
            //Once session is successfully created, reset the value of "any device connection is in progress or not". If it success
            //FC_Phonebook can request for another PBAP session.
            if (fc_phonebook_tclService_PhoneBook::pGetInstance())
            {
               fc_phonebook_tclService_PhoneBook::pGetInstance()->m_poBTSettings->vSetDeviceConnectInProgress(0);
               //NCG3D-78910
               //Here we are checking if any device is trying to connect while other device session connection is in progress
               fc_phonebook_tclService_PhoneBook::pGetInstance()->m_poBTSettings->vCheckForNewlyConnectedDevicesOrRetries();
               //End fix NCG3D-78910
            }
            //End of fix NCG3D-65982

            //Proceed with device connection via loopback.
            PhBk_Loopback_Data EvoConnect_LB_Data;
            memset((void*) &EvoConnect_LB_Data, 0, sizeof(PhBk_Loopback_Data));
            EvoConnect_LB_Data.u16FunctionID = (guint16)FC_PB_LB_FID_EVO_CONNECT;
            EvoConnect_LB_Data.ucDeviceHandle = deviceObject->getDeviceHandle();

            ETG_TRACE_USR4((" Posting Loopback message for FID- FC_PB_LB_FID_EVO_CONNECT"));
            EvopostLoopBackMessage((guint32)FC_PB_LB_FID_EVO_CONNECT, &EvoConnect_LB_Data);
         }
         else
         {
            ETG_TRACE_ERR((" Device object is NULL! Device currently not connected... "));
         }
      }
   }
#endif
}

/*******************************************************************************
 *
 * FUNCTION: interfacesRemovedCallback
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE:
 ********************************************************************************/
void interfacesRemovedCallback(OrgFreedesktopDBusObjectManager *object, const gchar *arg_object, const gchar * const *arg_interfaces)
{
   ETG_TRACE_USR4((" interfacesRemovedCallback entered with arg_object: %s", arg_object));


   (tVoid) arg_interfaces;

   if (!object)
   {
      ETG_TRACE_ERR((" object is NULL!!!"));
      return;
   }

   QMutexLocker lock(&g_Phonebook_DeviceMutex);

   if (arg_object)
   {
      for (int itr = 0; itr < MAX_DEVICE_CONNECTION_SUPPORTED; itr++)
      {
         if (ObjectPathMapping[itr].SessionPath)
         {
            //Iterating through available session path and getting corresponding deviceHandle.
            ETG_TRACE_USR4(("ObjectPathMapping[iter].SessionPath: %s", ObjectPathMapping[itr].SessionPath));
            if (VALUE_ZERO == u8StringCompare(ObjectPathMapping[itr].SessionPath, arg_object))
            {
               ETG_TRACE_USR4(("ObjectPathMapping[itr].DeviceHandle: %d ", ObjectPathMapping[itr].DeviceHandle));

#ifdef CONNECT_PBAP_MAP_VIA_BM_APPL
               FC_Device_Details
               *deviceObject =
               FC_Device_Details::deviceHandleToObjectInstance(ObjectPathMapping[itr].DeviceHandle);
               if (deviceObject)
               {
                  ETG_TRACE_COMP((" -PBDL.S- PBAP DISCONNECTED for DeviceHandle: %d", ObjectPathMapping[itr].DeviceHandle));

                  //Proceed with device disconnection via loopback.
                  PhBk_Loopback_Data EvoDisconnect_LB_Data;
                  memset((void*) &EvoDisconnect_LB_Data, 0, sizeof(PhBk_Loopback_Data));
                  EvoDisconnect_LB_Data.u16FunctionID = (guint16)FC_PB_LB_FID_EVO_DISCONNECT;
                  EvoDisconnect_LB_Data.ucDeviceHandle = deviceObject->getDeviceHandle();


                  //Removing the session path- NCG3D-89499.
                  ETG_TRACE_USR4((" Clearing SessionPath."));
                  g_free(ObjectPathMapping[itr].SessionPath);
                  ObjectPathMapping[itr].SessionPath = NULLPTR;
                  //Removing the PhonebookProxyAccess path- NCG3D-89499.

                  for (int n = 0; arg_interfaces != NULL && arg_interfaces[n] != NULL; n++)
                  {
                     ETG_TRACE_COMP(("PH_Alps:Interfaces to be removed =%s",arg_interfaces[n]));
                     if(VALUE_ZERO == g_strcmp0(arg_interfaces[n], "org.bluez.obex.PhonebookAccess") )
                      {
                        ETG_TRACE_USR4((" org.bluez.obex.PhonebookAccess is removed"));
                        if (ProxyConnMapping[itr].m_poProxyPbAccess != NULL)
                        {
                           ETG_TRACE_USR4((" Dereferencing PB Access proxy."));
                           g_object_unref(ProxyConnMapping[itr].m_poProxyPbAccess);
                           ProxyConnMapping[itr].m_poProxyPbAccess = NULLPTR;
                        }
                      }
                  }

                  ETG_TRACE_USR4((" Posting Loopback message for FID- FC_PB_LB_FID_EVO_DISCONNECT"));
                  EvopostLoopBackMessage((guint32)FC_PB_LB_FID_EVO_DISCONNECT, &EvoDisconnect_LB_Data);
               }
               else
               {
                  ETG_TRACE_ERR((" Device object is NULL! Device currently not connected... "));
               }
#endif
               break;
            }
         }

         if (ObjectPathMapping[itr].TransferPath)
         {
            //Iterating through available transfer path and getting corresponding deviceHandle.
            ETG_TRACE_USR4(("ObjectPathMapping[iter].TransferPath: %s", ObjectPathMapping[itr].TransferPath));
            if (VALUE_ZERO == u8StringCompare(ObjectPathMapping[itr].TransferPath, arg_object))
            {
               ETG_TRACE_USR4(("ObjectPathMapping[itr].DeviceHandle: %d ", ObjectPathMapping[itr].DeviceHandle));

               FC_Device_Details
                     * deviceObject =
                           FC_Device_Details::deviceHandleToObjectInstance(ObjectPathMapping[itr].DeviceHandle);
               if (deviceObject)
               {
                  // FIX NCG3D-11306 "Loading Phonebook" pop-up is displayed forever
                  // Property-changed update for transfer "Complete", reaches FC_Phonebook before the proxy for the transfer path is created and Property-changed is registered for the same.
                  // Hence the update goes without corresponding handling in FC_Phonebook.
                  // To handle this special case, check for interfaces-removed for the stored transfer path if any.
                  // If transfer "Complete" was not handled in Property-changed update, then transfer path will not be null here, then handle it in interfaces-removed.
                  // Wrt the special case, the current transfer is always considered as complete, as there is no status available in interfaces-removed.

                  ETG_TRACE_COMP((" -PBDL.EVO.S- Pull completed via interfacesRemovedCallback "));

                  //Delete the transfer path as processing is complete.
                  //TO DO: The following clearing could be moved to entry thread for better synchronization between the entry and dbus threads.
                  ETG_TRACE_USR4((" Clearing transfer path "));
                  if (ProxyConnMapping[itr].m_poProxyTransfer)
                  {
                     g_object_unref(ProxyConnMapping[itr].m_poProxyTransfer);
                     ProxyConnMapping[itr].m_poProxyTransfer = NULLPTR;
                  }
                  g_free(ObjectPathMapping[itr].TransferPath);
                  ObjectPathMapping[itr].TransferPath = NULLPTR;

                  ETG_TRACE_USR4(("Pull_Iteration * PULL_ITERATION_CONTACTS_COUNT : %d", Pull_Iteration
                        * PULL_ITERATION_CONTACTS_COUNT));

                  bRePullOnceOnError = FALSE; //If the 're-pull on error' is successful, reset on success.
                  u8TransferErrorCount = 0;

                  //Parse the downloaded v-cards, if any.
                  gboolean bCompleted
                        = bCheckIfDownloadIsComplete(deviceObject->getDeviceHandle());
                  EvoPbDlData_Signalhandler(bCompleted, ObjectPathMapping[itr].DeviceHandle);
               }
               else
               {
                  ETG_TRACE_USR4((" Device object is NULL!!!"));
               }
            }
         }
      }
   }
}


/*******************************************************************************
 *
 * FUNCTION: vUnRegisterObjectManagerSignals
 * DESCRIPTION:
 * PARAMETER: [IN]
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE:
 ********************************************************************************/
void vUnRegisterClientAndObjectManagerSignals()
{
   ETG_TRACE_USR4((" vUnRegisterClientAndObjectManagerSignals entered. "));

   if (l_ProxyObjectManager)
   {
      ETG_TRACE_USR4((" Disconnecting Object manager signals."));
      if (0 != SignalHandlerIDs.ulInterfacesAddedID)
      {
         g_signal_handler_disconnect(l_ProxyObjectManager, SignalHandlerIDs.ulInterfacesAddedID);
         SignalHandlerIDs.ulInterfacesAddedID = 0;
      }
      if (0 != SignalHandlerIDs.ulInterfacesRemovedID)
      {
         g_signal_handler_disconnect(l_ProxyObjectManager, SignalHandlerIDs.ulInterfacesRemovedID);
         SignalHandlerIDs.ulInterfacesRemovedID = 0;
      }

      ETG_TRACE_USR4((" Dereferencing Object manager proxy."));
      g_object_unref(l_ProxyObjectManager );
      l_ProxyObjectManager = NULLPTR;
   }

   if (l_ProxyClient)
   {
      ETG_TRACE_USR4((" Dereferencing Client proxy."));
      g_object_unref(l_ProxyClient);
      l_ProxyClient = NULLPTR;
   }
}


/*******************************************************************************
 *
 * FUNCTION: bCheckIfDownloadIsComplete
 * DESCRIPTION:
 * PARAMETER: [IN] u8DeviceHandle
 *            [OUT]
 *            [OUT]
 *
 * RETURNVALUE: gbooolean - Denotes whether download is complete or not.
 ********************************************************************************/
gboolean bCheckIfDownloadIsComplete(guint8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" bCheckIfDownloadIsComplete entered with device handle: %d ", u8DeviceHandle));

   gboolean bCompleted = FALSE;
   FC_Device_Details *deviceObject =
         FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

   if (deviceObject)
   {
      quint8
            u8PhonebookType =
                  FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(deviceObject->getConnectionID());
      if( (DownLoadPBSim == u8PhonebookType) || (DownLoadPBLocal == u8PhonebookType) )
      {
         ETG_TRACE_USR4(("u8PhonebookType= %d;  deviceObject->m_u32InidvidualTypeContactsCount= %d;  deviceObject->m_u32ContributingVCardCount= %d ", u8PhonebookType, deviceObject->m_u32InidvidualTypeContactsCount, deviceObject->m_u32ContributingVCardCount));
         if( (Pull_Iteration * PULL_ITERATION_CONTACTS_COUNT < deviceObject->m_u32InidvidualTypeContactsCount)  // TRUE means: Phone has still contacts to delvier
			 && (deviceObject->m_u32ContributingVCardCount < PBDL_MAX_CONTACTS_PER_PHONE_TO_STORE)  // TRUE means: We still have not reached the maximum number of contacts per phone, which we could store in the DB
		   )
         {
            bCompleted = FALSE;
         }
         else
         {
            bCompleted = TRUE;
         }
      }
      else if( (DownloadPBSpeedDial == u8PhonebookType) || (DownloadPBFavorites == u8PhonebookType) )
      {
         ETG_TRACE_USR4(("u8PhonebookType= %d;  deviceObject->m_u32VCardsAvailableForDwnld= %d;  deviceObject->m_u32ContributingVCardCount= %d ", u8PhonebookType, deviceObject->m_u32VCardsAvailableForDwnld, deviceObject->m_u32ContributingVCardCount));
         if( (Pull_Iteration * PULL_ITERATION_CONTACTS_COUNT < deviceObject->m_u32VCardsAvailableForDwnld)   // TRUE means: Phone has still contacts to delvier
			 && (deviceObject->m_u32ContributingVCardCount < PBDL_MAX_CONTACTS_PER_PHONE_TO_STORE)  // TRUE means: We still have not reached the maximum number of contacts per phone, which we could store in the DB
		   )
         {
            bCompleted = FALSE;
         }
         else
         {
            bCompleted = TRUE;
         }
      }
      else
      {
         //For call history download, there is no pull iteration concept currently. So download will be bCompleted in a single pull.
         //Hence there is no need to depend on pull iteration variables to decide if download is bCompleted or not.
         bCompleted = TRUE;
      }
   }
   else
   {
      ETG_TRACE_USR4((" Device object is NULL!!! "));
      bCompleted = TRUE; //Complete the download sequence if device already disconnected
   }

   ETG_TRACE_USR4((" bCompleted: %d ", bCompleted));
   return bCompleted;
}

/*******************************************************************************
 *
 * FUNCTION: vSetCapabilitiesForSuspendedDownloadDevice
 * DESCRIPTION:
 * PARAMETER: [IN] deviceObject
 *
 * RETURNVALUE: gbooolean - Denotes whether download is complete or not.
 ********************************************************************************/
void vSetCapabilitiesForSuspendedDownloadDevice(FC_Device_Details * deviceObject)
{
   ETG_TRACE_USR4((" vSetCapabilitiesForSuspendedDownloadDevice entered for the device handle :%u ", deviceObject->getDeviceHandle()));
   if(deviceObject)
   {
      quint8 u8DnlType = deviceObject->PBDLCommand;
      //Checking SIM support
      if((deviceObject->m_u8FAVSPDDwnldSupport & PBAP_SIM_DWNLD_SUPPORTED)
            != PBAP_SIM_DWNLD_SUPPORTED)
      {
         u8DnlType = u8DnlType & 0xFE;
      }
      deviceObject->setDeviceCapabilities(u8DnlType);
      fc_phonebook_tclService_PhoneBook::pGetInstance()->vUpdatePhoneBookFeatureSupport(deviceObject, TRUE);
   }
   else
   {
      ETG_TRACE_USR4((" Device object is NULL!!! "));
   }
   ETG_TRACE_USR4((" vSetCapabilitiesForSuspendedDownloadDevice exited "));
}

/*******************************************************************************
 *
 * FUNCTION: bCheckSessionPathAvailable
 * DESCRIPTION: Check whether PBAP session is created with respect to device handle
 * PARAMETER: [IN] DeviceObject
 *            [OUT] bool (session available or not)
 *
 * RETURNVALUE: None.
 ********************************************************************************/
bool bCheckSessionPathAvailable(FC_Device_Details *pDeviceObject)
{
   ETG_TRACE_USR4((" bCheckSessionPathAvailable entered."));
   bool bSessionPath = false;
   tU8 u8ConnectionID = (tU8)(pDeviceObject->getConnectionID());
   if((u8ConnectionID <= MAX_DEVICE_CONNECTION_SUPPORTED) && (VALUE_ZERO != u8ConnectionID))
   {
      ETG_TRACE_USR4(("ObjectPathMapping[u8ConnectionID - 1].SessionPath is : %s ", ObjectPathMapping[u8ConnectionID - 1].SessionPath));
      if (ObjectPathMapping[u8ConnectionID - 1].SessionPath)
      {
         bSessionPath = true;
      }
   }
   return bSessionPath;
}
