/*******************************************************************************
*
* FILE:          FC_PhoneBook_ListChange_Handler.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:   File contains list change handling related functionality
*
* AUTHOR:		 Balasubramani.V
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include <QtCore/QtCore>

#include "../FC_PhoneBook_Debug.h"
#include "FC_PhoneBook_SQLite.h"
#include "../FC_PhoneBook_service_PhoneBook.h"
#include "FC_PhoneBook_ListChange_Handler.h"
#include "FC_PhoneBook_List.h"
#include <malloc.h>  // -bn: Added for 'malloc_trim()'

#include "FC_PhoneBook_CallHistory_List.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC

#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_SQLITE
#include "trcGenProj/Header/FC_PhoneBook_ListChange_Handler.cpp.trc.h"
#endif


/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::ListChange_Handler
*
* DESCRIPTION:
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
ListChange_Handler::ListChange_Handler()
{
   ETG_TRACE_USR4(("ListChange_Handler() entered."));

   enList_Change_Type = enInvalid_List_Change_Type;
   enList_Type = enInvalid_List_Type;

   m_poSqlite= FC_PhoneBook_SQLite::GetSQLiteInstance();
   m_poservice_phonebook= fc_phonebook_tclService_PhoneBook::pGetInstance();
}


/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::~ListChange_Handler
*
* DESCRIPTION:
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
ListChange_Handler::~ListChange_Handler()
{
   m_poSqlite = NULLPTR;
   m_poservice_phonebook = NULLPTR;
}

/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::ListUpdated
*
* DESCRIPTION: This function should be triggered on every action like add, edit or
*              delete contact to check if any list to be updated
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
void ListChange_Handler::UpdateList(const QHash< quint16, FC_PhoneBook_List_Data* >* phashPbList,
                                    quint8 u8ActionDone, quint8 u8DeviceHandle, quint32 ContactHandle, FC_PhoneBook_SQLite::enListType u8PBListType)   // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{

   ETG_TRACE_USR2((" Updating the list on action - %d", u8ActionDone));

   switch(u8ActionDone)
   {
   case enAction_Contact_Added:
      NewContactAdded(phashPbList, u8DeviceHandle, ContactHandle, u8PBListType);
      break;
   case enAction_Contact_Edited:
      SingleContactEdited(phashPbList, u8DeviceHandle, ContactHandle, u8PBListType);
      break;
   case enAction_Contact_Deleted:
      SingleContactDeleted(phashPbList, u8DeviceHandle, ContactHandle, u8PBListType);
      break;
   case enAction_Content_Cleared:
      ContentCleared(phashPbList, u8DeviceHandle, u8PBListType);
      break;
   case enAction_Content_Refreshed:
      ContentRefreshed(phashPbList, u8DeviceHandle, u8PBListType);
      break;
   case enAction_Dummy:
      DummyListChange(phashPbList, u8DeviceHandle);
      break;
   default:

      ETG_TRACE_USR2((" ListChange_Handler: Wrong Parameter passed"));
   }

}

/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::ListUpdated
*
* DESCRIPTION: Overloaded function for call history
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
void ListChange_Handler::UpdateList(const QHash< quint16, FC_CallHistory_List_Data* >* phashChList,
                                    quint8 u8ActionDone, quint8 u8ConnectionID, quint32 u32NewContactHandle,
                                    quint32 u32OldContactHandle, quint8 callHistoryTpye)
{
   ETG_TRACE_USR4((" UpdateList entered"));
   switch(u8ActionDone)
   {
   case enAction_Contact_Edited:
      SingleContactEdited(phashChList, u8ConnectionID, u32NewContactHandle,
         u32OldContactHandle);
      break;
   case enAction_Content_Refreshed:
      ContentRefreshed(phashChList, u8ConnectionID, callHistoryTpye);
      break;
   case enAction_Content_Cleared:
      ContentCleared(phashChList, u8ConnectionID);
      break;
   default: 
      ETG_TRACE_USR2((" ListChange_Handler: Wrong Parameter passed"));
   }
}

/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::NewContactAdded
*
* DESCRIPTION: This function checks if any list is active corresponding to the
* device handle to which a new a new contact is added. If exists then a new list
* will be created and List change property update will be triggered
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
void ListChange_Handler::NewContactAdded(const QHash< quint16, FC_PhoneBook_List_Data* >* phashPbList,
                                         quint8 u8DeviceHandle,	quint32 u32ContactHandle, quint8 u8ListType)   // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{
   ETG_TRACE_USR2(
      (" ListChangeHandler:- New contact added to devicehandle- %u", u8DeviceHandle));
   ETG_TRACE_USR2(
      (" ListChangeHandler:- New contact added to contacthandle- %u", u32ContactHandle));

   QList<quint16> ListHandles;
   GetListofMatchingListHandles(phashPbList, u8DeviceHandle, u8ListType, ListHandles);
   quint32 u32ChangedIndex;
   quint16 u16ListSize;

   u16ListSize= (quint16)(ListHandles.size());
   if(0 == u16ListSize)
   {
      ETG_TRACE_USR4((" ListChangeHandler:- No matching list handle found "));
      return;
   }

   enList_Change_Type= enItems_Added;
   enList_Type= enPhoneBook_list;
   QList<FC_PhoneBook_Contact_Detail_Short> PBookList;

   for(quint16 u16Index=0; u16Index < u16ListSize; u16Index++)
   {
      PBookList.clear();
      quint16 u16ListHandle= ListHandles.at(u16Index);
      FC_PhoneBook_List_Data* PBookListData= phashPbList->value(u16ListHandle);
      if(CreateNewList(PBookListData->listType, u8DeviceHandle, PBookListData->sortType, PBookList))
      {
         UpdateNewContactList(PBookListData, PBookList);

         ETG_TRACE_USR4(("  ListChange_Handler:- New list created for listhandle- %u ",
            u16ListHandle));

         if(FindIndexOfContactHandle(PBookList, u32ContactHandle, u32ChangedIndex))
         {

            ETG_TRACE_USR4(("  ListChange_Handler:- New index for the contact handle is %u ",
                  u32ChangedIndex));
            SendListChangeToClients(u16ListHandle, (quint16)PBookList.size(), u32ChangedIndex);

         }
         else
         {

            ETG_TRACE_ERR(("  ListChange_Handler:- Failed to fetch the index of the new "
               "contact handle "));
         }
      }
      else
      {

         ETG_TRACE_ERR(("  ListChange_Handler:- Creation of new list failed for listhandle- "
            "%u ",	u16ListHandle));
      }
   }
}


/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::SingleContactEdited
*
* DESCRIPTION:This function checks if any list is active corresponding to the
* device handle in which contact was edited. If exists then the list will be modified
*  and List change property update will be triggered
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
***************************************************************************************************/
void ListChange_Handler::SingleContactEdited(const QHash< quint16, FC_PhoneBook_List_Data* >* phashPbList,
                                             quint8 u8DeviceHandle, quint32 u32ContactHandle, quint8 u8ListType)   // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{
   QList<quint16> ListHandles;
   GetListofMatchingListHandles(phashPbList, u8DeviceHandle, u8ListType, ListHandles);
   quint32 u32ChangedIndex;
   quint16 u16ListSize;

   ETG_TRACE_USR2((" SingleContactEdited entered and ListChangeHandler:- New contact added to devicehandle- %u"
      , u8DeviceHandle));
   ETG_TRACE_USR2((" ListChangeHandler:- New contact added to contacthandle- %u"
      , u32ContactHandle));

   u16ListSize= (quint16)(ListHandles.size());
   if(0 == u16ListSize)
   {
      ETG_TRACE_USR4((" ListChangeHandler:- No matching list handle found "));
      return;
   }

   //Fix - NCG3D-13408
   //PhonebookMW should send "enList_Change_Type" as"enItems_changed" instead of "enContent_Changed" when a quick dial entry is edited
   enList_Change_Type = enItems_changed;  
   //End of fix NCG3D-13408
   enList_Type= enPhoneBook_list;
   QList<FC_PhoneBook_Contact_Detail_Short> PBookList;

   for(quint16 u16Index=0; u16Index < u16ListSize; u16Index++)
   {
      PBookList.clear();
      quint16 u16ListHandle= ListHandles.at(u16Index);
      FC_PhoneBook_List_Data* PBookListData = phashPbList->value(u16ListHandle);
      if(CreateNewList(PBookListData->listType ,u8DeviceHandle, PBookListData->sortType, PBookList))
      {
         UpdateNewContactList(PBookListData, PBookList);

         ETG_TRACE_USR4(("  ListChange_Handler:- New list created for listhandle- %u ",
            u16ListHandle));

         if(FindIndexOfContactHandle(PBookList, u32ContactHandle, u32ChangedIndex))
         {
            ETG_TRACE_USR4(("  ListChange_Handler:- New index for the contact handle is %u ",
                  u32ChangedIndex));
            SendListChangeToClients(u16ListHandle, (quint16)PBookList.size(), u32ChangedIndex);
         }
         else
         {
            ETG_TRACE_ERR(("  ListChange_Handler:- Failed to fetch the index of the new "
               "contact handle "));
         }
      }
      else
      {
         ETG_TRACE_ERR(("  ListChange_Handler:- Creation of new list failed for listhandle- "
            "%u ",	u16ListHandle));
      }
   }
}

/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::SingleContactEdited
*
* DESCRIPTION: This function finds the call history entry in the list having old contact handle
* and updates it with the details of the call history entry in the database having the new contact
* handle. Also triggers a list change property update
*              .

* PARAMETER: [IN] -
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
***************************************************************************************************/
void ListChange_Handler::SingleContactEdited(const QHash< quint16, FC_CallHistory_List_Data* >* phashChList,
                                             quint8 u8ConnectionID, quint32 u32NewContactHandle, quint32 u32OldContactHandle)
{
   ETG_TRACE_USR2(
      (" SingleContactEdited entered and ListChangeHandler:- Call history record edited from u8ConnectionID- %u", u8ConnectionID));
   ETG_TRACE_USR2(
      (" ListChangeHandler:- Call history record edited from contacthandle- %u", u32NewContactHandle));

   QList<quint16> l_ListHandles;
   GetListofMatchingListHandles(phashChList, u8ConnectionID, l_ListHandles);

   quint16 u16ListHandleSize;
   u16ListHandleSize= (quint16)(l_ListHandles.size());
   if(0 == u16ListHandleSize)
   {
      ETG_TRACE_USR4((" ListChangeHandler:- No matching list handle found "));
      return;
   }

   enList_Change_Type= enItems_changed;
   enList_Type= enCallHistory_list;

   quint16 u16Index;
   quint16 u16ChangedIndex;
   quint16 u16IndexDBCallHistoryDetail;
   quint16 u16IndexPhbkCallHistoryDetail;
   quint16 u16ListHandle;
   quint16 u16PhbkChDetailListSize;
   quint16 u16DBChListSize;

   FC_CallHistory_List_Data* l_poChListData;
   QList<FC_PhoneBook_Callhistory_Detail> l_DBCallHistoryDetailList; //to store call history list from database
   FC_PhoneBook_Callhistory_Detail l_DBCallHistoryDetail; //for call history detail item in Db Callhistory list
   FC_PhoneBook_Callhistory_Detail l_PhbkCallHistoryDetail; //for call history detail item in FC_Phonebook Callhistory list
   FC_PhoneBook_Callhistory_Detail l_CallHistoryDetail; //temporary call history detail

   for(u16Index=0; u16Index < u16ListHandleSize; u16Index++)
   {
      u16ListHandle = l_ListHandles.at(u16Index);
      l_poChListData = phashChList->value(u16ListHandle);

      //Call history list maintained by FC_Phonebook apart from Database
      QList<FC_PhoneBook_Callhistory_Detail> &l_PhbkChDetailList = l_poChListData->callhistoryList;

      if (m_poSqlite->GetCallHistoryRecordDetails(u32NewContactHandle,
         u8ConnectionID, l_DBCallHistoryDetailList))
      {
         u16DBChListSize = (quint16)(l_DBCallHistoryDetailList.size());
         u16PhbkChDetailListSize = (quint16)(l_PhbkChDetailList.size());

         for (u16IndexPhbkCallHistoryDetail = 0;
            u16IndexPhbkCallHistoryDetail < u16PhbkChDetailListSize;
            u16IndexPhbkCallHistoryDetail++)
         {
            //compare contact handle and phone number of each item in local call history list
            //with all the items retreived with new contact handle from the database

            l_PhbkCallHistoryDetail = l_PhbkChDetailList.at(u16IndexPhbkCallHistoryDetail);

            if (l_PhbkCallHistoryDetail.contactHandle	== u32OldContactHandle)
            {
               for (u16IndexDBCallHistoryDetail = 0;
                  u16IndexDBCallHistoryDetail < u16DBChListSize;
                  u16IndexDBCallHistoryDetail++)
               {
                  l_DBCallHistoryDetail = l_DBCallHistoryDetailList.at(u16IndexDBCallHistoryDetail);

                  if (QString::compare(l_PhbkCallHistoryDetail.phoneNumber,
                     l_DBCallHistoryDetail.phoneNumber,
                     Qt::CaseInsensitive) == 0)
                  {
                     u16ChangedIndex = u16IndexPhbkCallHistoryDetail;

                     l_CallHistoryDetail.contactHandle = u32NewContactHandle;
                     l_CallHistoryDetail.firstName = l_DBCallHistoryDetail.firstName;
                     l_CallHistoryDetail.lastName = l_DBCallHistoryDetail.lastName;
                     l_CallHistoryDetail.callType = l_PhbkCallHistoryDetail.callType;
                     l_CallHistoryDetail.phoneNumber = l_PhbkCallHistoryDetail.phoneNumber;
                     l_CallHistoryDetail.numberType = l_PhbkCallHistoryDetail.numberType;
                     l_CallHistoryDetail.dateStamp = l_PhbkCallHistoryDetail.dateStamp;
                     l_CallHistoryDetail.timeStamp = l_PhbkCallHistoryDetail.timeStamp;

                     l_PhbkChDetailList.replace(u16IndexPhbkCallHistoryDetail, l_CallHistoryDetail);

                     ETG_TRACE_USR4((" ListChange_Handler:- List modified is - %u", u16ListHandle));
                     ETG_TRACE_USR4((" ListChange_Handler:- Index of the changed contact is - %u", u16ChangedIndex));

                     SendListChangeToClients(u16ListHandle, u16PhbkChDetailListSize, u16ChangedIndex);
                     break;
                  }
               }
            }
         }
      }
      else
      {
         ETG_TRACE_USR2(
            (" ListChange_Handler: No entry found for the contact handle - %u in the database", u32NewContactHandle));
      }
   }
}
/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::SingleContactDeleted
*
* DESCRIPTION: This function finds the contact having the passed contact handle in the list and
* removes the contact from the list and triggers a list change property update
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
***************************************************************************************************/
void ListChange_Handler::SingleContactDeleted(const QHash< quint16, FC_PhoneBook_List_Data* >* phashPbList,
                                              quint8 u8DeviceHandle, quint32 u32ContactHandle, quint8 u8ListType)   // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{
   QList<quint16> ListHandles;
   GetListofMatchingListHandles(phashPbList, u8DeviceHandle, u8ListType, ListHandles);
   quint16 u16ChangedIndex = 0;
   quint16 u16ListHandleSize = 0;
   bool blContactFound = false;

   ETG_TRACE_USR2((" SingleContactDeleted entered and ListChangeHandler:- Contact to be deleted from devicehandle- %u"
      , u8DeviceHandle));
   ETG_TRACE_USR2((" ListChangeHandler:- Contact to be deleted with contact handle- %u"
      , u32ContactHandle));

   u16ListHandleSize= (quint16)(ListHandles.size());
   if(0 == u16ListHandleSize)
   {
      ETG_TRACE_USR4((" ListChangeHandler:- No matching list handle found "));
      return;
   }

   enList_Change_Type= enItems_removed;
   enList_Type= enPhoneBook_list;

   for(quint16 u16Index=0; u16Index < u16ListHandleSize; u16Index++)
   {
      quint16 u16ContactListSize;
      blContactFound= false;
      quint16 u16ListHandle= ListHandles.at(u16Index);
      FC_PhoneBook_List_Data* PBookListData= phashPbList->value(u16ListHandle);
      QList<FC_PhoneBook_Contact_Detail_Short>& ContactList= PBookListData->contactList;
      u16ContactListSize= (quint16)(ContactList.size());
      for(quint16 u16IndexOfDeletedContact= 0; u16IndexOfDeletedContact < u16ContactListSize; u16IndexOfDeletedContact++)
      {
         if(ContactList.at(u16IndexOfDeletedContact).contactHandle == u32ContactHandle)
         {
            u16ChangedIndex= u16IndexOfDeletedContact;
            blContactFound= true;
            break;
         }
      }

      if(blContactFound)
      {
         ContactList.removeAt(u16ChangedIndex);

         ETG_TRACE_USR4((" ListChange_Handler:- List modified is - %u"
            , u16ListHandle));
         ETG_TRACE_USR4((" ListChange_Handler:- Index of the deleted contact is - %u"
            , u16ChangedIndex));

         SendListChangeToClients(u16ListHandle, (quint16)ContactList.size(), u16ChangedIndex);
      }
   }
}

/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::ContentRefreshed
*
* DESCRIPTION: This functions triggers a list change property update to all the lists corresponding
* to the passed connection ID.

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: 
*
***************************************************************************************************/
void ListChange_Handler::ContentRefreshed(const QHash< quint16, FC_CallHistory_List_Data* >* phashChList,
                                          quint8 u8ConnectionID, quint8 callHistoryTpye)
{
   ETG_TRACE_COMP((" -PBDL.S- Refreshing Call History List(s) and SendListChange(s) for u8ConnectionID= %u "
      , u8ConnectionID ));

   QList<quint16> ListHandles;
   quint16 u16ListHandle;
   FC_CallHistory_List_Data* ChListData = NULL;

   GetListofMatchingListHandles(phashChList, u8ConnectionID, ListHandles);
   quint16 u16ListHandleSize= (quint16)(ListHandles.size());

   if(0 == u16ListHandleSize)
   {
      ETG_TRACE_USR4(("ListChangeHandler:- No matching list handle found "));
      return;
   }

   enList_Change_Type= enContent_Changed;
   enList_Type= enCallHistory_list;

   for(quint16 u16Index=0; u16Index < u16ListHandleSize; u16Index++)
   {
      u16ListHandle= ListHandles.at(u16Index);
      ChListData= phashChList->value(u16ListHandle,NULLPTR);

      //Fix GMMY15-2582
      //System reset observed since check ChListData!= NULL was missing. If it is NULL, its contents should not be accessed.
      if( (ChListData != NULL) && ((callHistoryTpye == ChListData->callHistoryTpye)||
         (FC_PhoneBook_SQLite::callhisttype_combined == ChListData->callHistoryTpye) ||
         (FC_PhoneBook_SQLite::callhisttype_combined == callHistoryTpye)) )
      {
         FC_Device_Details* pDevDetails = FC_Device_Details::getDeviceObject((INDEX )(u8ConnectionID-1));
         if(pDevDetails)
         {
            //FIX JAC-5053 In ListChange property update for MCH, ICH and OCH lists, list size value is incorrect
            //Size passed for list change update was of old list (ChListData->callhistoryList), not of new one (CallHistoryList) retrieved from DB.
            ChListData->callhistoryList.clear();
            m_poSqlite->GetCallHistoryRecords((FC_PhoneBook_SQLite::SortType)ChListData->sortType,
               ChListData->callHistoryTpye, u8ConnectionID, ChListData->callhistoryList, pDevDetails->Device_DateTime_Support.CHDateTimeSupport);
            SendListChangeToClients(u16ListHandle, (quint16)ChListData->callhistoryList.size(), 0);
            //End of fix
         }         
      }
      else
      {
         ETG_TRACE_USR2(("CallHistoryType doesn't match for listhandle- %d", u16ListHandle));
      }
   }
}
/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::GetListofMatchingListHandles
*
* DESCRIPTION: This function returns the list of listhandles matching the given
*              device handle
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void ListChange_Handler::GetListofMatchingListHandles(const QHash< quint16,
                                                      FC_PhoneBook_List_Data* >* pPbHash, quint8 u8DeviceHandle,
                                                      quint8 u8PBListType,
                                                      QList<quint16>& FilteredListHandles) // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{
   ETG_TRACE_USR2((" GetListofMatchingListHandles Entered for Device: %d ",u8DeviceHandle));
   QList<quint16> AllListHandles= pPbHash->keys();
   quint16 u16ListHandlesSize= (quint16)(AllListHandles.size());
   ETG_TRACE_USR2((" Total no: of list handles available in hash: %d ",u16ListHandlesSize));

   for(quint16 listHandleIndex=0; listHandleIndex < u16ListHandlesSize; listHandleIndex++ )
   {
      quint16 u16ListHandle= AllListHandles.at(listHandleIndex);

      //Fix GMMY15-2582
      //If PbListData pointer is NULL, its contents should not be accessed.
      FC_PhoneBook_List_Data* PbListData= pPbHash->value(u16ListHandle, NULLPTR);
      ETG_TRACE_USR2((" GetListofMatchingListHandles ..In For Loop for List Handle: %d ",u16ListHandle));
      if(PbListData  != NULL && PbListData->u8DeviceHandle == u8DeviceHandle)
      {
         if(FC_PhoneBook_SQLite::ListType_ALL == u8PBListType)
         {
            ETG_TRACE_USR2((" Appending List Handle: %d",u16ListHandle));
            FilteredListHandles.append(u16ListHandle);
         }
         else
         {
            if(PbListData->listType == u8PBListType)
            {
               ETG_TRACE_USR4((" Appending List Handle: %d with list type: %d", u16ListHandle, ETG_CENUM(FC_PhoneBook_SQLite::enListType, u8PBListType)));
               FilteredListHandles.append(u16ListHandle);
            }
         }
      }
   }
}

/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::GetListofMatchingListHandles
*
* DESCRIPTION: This function returns the list of listhandles matching the given
*              connection ID
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void ListChange_Handler::GetListofMatchingListHandles(const QHash< quint16, FC_CallHistory_List_Data* >* pChHash,
                                                      quint8 u8ConnectionID,	QList<quint16>& FilteredListHandles)
{
   ETG_TRACE_USR4(("GetListofMatchingListHandles entered"));
   QList<quint16> AllListHandles= pChHash->keys();
   quint16 u16ListHandlesSize= (quint16)(AllListHandles.size());

   for(quint16 listHandleIndex=0; listHandleIndex < u16ListHandlesSize; listHandleIndex++ )
   {
      quint16 u16ListHandle= AllListHandles.at(listHandleIndex);

      //Fix GMMY15-2582
      //If PbListData pointer is NULL, its contents should not be accessed.
      FC_CallHistory_List_Data* PbListData= pChHash->value(u16ListHandle, NULLPTR);
      ETG_TRACE_USR2(("GetListofMatchingListHandles ..u16ListHandle =%x ",u16ListHandle));
      if(PbListData  != NULL && PbListData->ConnectionID == u8ConnectionID)
      {
         FilteredListHandles.append(u16ListHandle);
         ETG_TRACE_USR2(("MatchingListHandles ..u16ListHandle =%x ",u16ListHandle));
      }
   }
}

/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::CreateNewList
*
* DESCRIPTION: creates a new list for the given parameters
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool ListChange_Handler::CreateNewList(tU8 u8ListType, quint8 u8DeviceHandle, quint8 u8SortType,
                                       QList<FC_PhoneBook_Contact_Detail_Short>& PBook_List)
{
   ETG_TRACE_USR4(("CreateNewList entered"));
   m_poSqlite->GetRecords(u8ListType, (FC_PhoneBook_SQLite::SortType) u8SortType, u8DeviceHandle, PBook_List); //List is passed as reference to reduce the amount of memory used
   return true;
}

/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::UpdateNewContactList
*
* DESCRIPTION: Updates the List data with the new Phonebook list
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void ListChange_Handler::UpdateNewContactList(FC_PhoneBook_List_Data* pPBookListData,
                                              QList<FC_PhoneBook_Contact_Detail_Short>& PBookList)
{
   ETG_TRACE_USR4(("UpdateNewContactList entered"));
   pPBookListData->contactList.clear();
   pPBookListData->contactList= PBookList;
}

/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::FindIndexOfContactHandle
*
* DESCRIPTION: Finds the index of the contact handle in the given phonebook list
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool ListChange_Handler::FindIndexOfContactHandle(QList<FC_PhoneBook_Contact_Detail_Short>& PBookList,
                                                  quint32 u32ContactHandle, quint32& u32Index)
{
   ETG_TRACE_USR4(("FindIndexOfContactHandle entered"));
   bool blFound= false;
   quint32 PBookListSize= PBookList.size();

   for(u32Index= 0; u32Index< PBookListSize; u32Index++)
   {
      if(PBookList.at(u32Index).contactHandle == u32ContactHandle)
      {
         blFound= true;
         break;
      }
   }

   return blFound;
}

/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::SendListChangeToClients
*
* DESCRIPTION: This function triggers a list change property update to clients
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void ListChange_Handler::SendListChangeToClients(quint16 u16ListHandle, quint16 u16ListLength,
                                                 quint32 u32ChangedItem)
{
   ETG_TRACE_USR4(("SendListChangeToClients entered"));
   most_phonbkfi_tclMsgListChangeStatus ListChangeStatus;

   ListChangeStatus.u16ListHandle= u16ListHandle;

   //Coverity Fix- CMG3G-14313
   MapPhonBkListType(ListChangeStatus.e8ListType);
   MapPhonBkListChangeType(ListChangeStatus.e8ListChangeType);
   //end of fix

   ListChangeStatus.u16ListLength= u16ListLength;
   if(enList_Change_Type != enContent_Changed)
   {
      ListChangeStatus.u32NumItems= 1; //Only one item changed
      ListChangeStatus.oChangeItems.u32Items.push_back(u32ChangedItem);
   }
   else
   {
      ListChangeStatus.u32NumItems= 0;
      ListChangeStatus.oChangeItems.u32Items.clear();
   }

   m_poservice_phonebook->vPhoneBook_UpdateListChange(ListChangeStatus);

   ETG_TRACE_USR1((" List Type:: %u",ETG_ENUM(TR_LISTTYPE,ListChangeStatus.e8ListType.enType)));
   ETG_TRACE_USR1((" List Change Type:: %u",ETG_ENUM(TR_LISTCHANGETYPE,ListChangeStatus.e8ListChangeType.enType)));
   ETG_TRACE_USR2((" ListChange_Handler:- List change update triggered for "
      "ListHandle        - %u " , u16ListHandle));
   ETG_TRACE_USR2((" ListChange_Handler:- List change update triggered for "
      "ListLength        - %u " , u16ListLength));
   ETG_TRACE_USR2((" ListChange_Handler:- List change update triggered for "
      "NumOfItemsChanged        - 1 " ));
   ETG_TRACE_USR2((" ListChange_Handler:- List change update triggered for "
      "ChangedIndex        - %u " "",u32ChangedItem));
}

//Coverity Fix-CMG3G-14313
/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::MapPhonBkListType
*
* DESCRIPTION: Mapping of List Type to response
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void ListChange_Handler::MapPhonBkListType(most_fi_tcl_e8_PhonBkListType &enListType)
{

   ETG_TRACE_USR4(( "FC_PhoneBook_ListChangeHandler::MapPhonBkListType entered with enList_Type= %d",enList_Type));

   switch(enList_Type)
      {
      case enPhoneBook_list:
         enListType.enType = most_fi_tcl_e8_PhonBkListType::FI_EN_E8PHONEBOOK_LIST;
         break;

      case enCallHistory_list:
         enListType.enType = most_fi_tcl_e8_PhonBkListType::FI_EN_E8CALLHISTORY_LIST;
         break;

      case enInvalid_List_Type:
         ETG_TRACE_ERR(( "FC_PhoneBook_ListChangeHandler::Invalid ListType "));
         break;

      default:
         ETG_TRACE_ERR(( "MapPhonBkListType default case Entered "));
      }

}
/*******************************************************************************
*
* FUNCTION:  ListChange_Handler::MapPhonBkListChangeType
*
* DESCRIPTION: Mapping of List ChangeType to response
*              .

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void ListChange_Handler::MapPhonBkListChangeType(most_fi_tcl_e8_PhonBkListChangeType &enListChangeType)
{

   ETG_TRACE_USR4(( "FC_PhoneBook_ListChangeHandler::MapPhonBkListChangeType entered with enList_Change_Type= %d",enList_Change_Type));

   switch(enList_Change_Type)
      {
      case enContent_Changed:
         enListChangeType.enType = most_fi_tcl_e8_PhonBkListChangeType::FI_EN_E8LCH_CONTENT_CHANGED;
         break;

      case enItems_Added:
         enListChangeType.enType = most_fi_tcl_e8_PhonBkListChangeType::FI_EN_E8LCH_ITEMS_ADDED;
         break;

      case enItems_removed:
         enListChangeType.enType = most_fi_tcl_e8_PhonBkListChangeType::FI_EN_E8LCH_ITEMS_REMOVED;
         break;

      case enItems_changed:
         enListChangeType.enType = most_fi_tcl_e8_PhonBkListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
         break;

      case enInvalid_List_Change_Type:
         ETG_TRACE_ERR(( "FC_PhoneBook_ListChangeHandler::Invalid List Change Type"));
         break;

      default:
         ETG_TRACE_ERR(( "MapPhonBkListChangeType default case Entered "));
      }

}
//end of Fix-CMG3G-14313
/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::ContentCleared
*
* DESCRIPTION: This functions clears all the lists corresponding to a given device and triggers a list
* change property update. Used when a device is disconnected

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: 
*
***************************************************************************************************/
void ListChange_Handler::ContentCleared(const QHash< quint16,
                                        FC_PhoneBook_List_Data* >* pPbHash, quint8 u8DeviceHandle, quint8 u8PBListType)   // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{
   ETG_TRACE_USR2((" ContentCleared entered and Phonebook list to be cleared for DeviceHandle= %u ", u8DeviceHandle));

   QList<quint16> ListHandles;
   if(pPbHash)
   {
      GetListofMatchingListHandles(pPbHash, u8DeviceHandle, u8PBListType, ListHandles);

      quint16 u16ListHandleSize;
      u16ListHandleSize= (quint16)(ListHandles.size());
      if(0 == u16ListHandleSize)
      {
         ETG_TRACE_USR1((" ListChangeHandler:- No matching list handle found "));
         return;
      }

      enList_Change_Type= enContent_Changed;
      enList_Type= enPhoneBook_list;

      for(quint16 u16Index=0; u16Index < u16ListHandleSize; u16Index++)
      {
         quint16 u16ListHandle= ListHandles.at(u16Index);
         //Clearing list and corresponding entries from hash lists.
         FC_PhoneBook_List::pGetInstance()->vClearPBListAndHashListEntriesForListHandle(u16ListHandle, u8DeviceHandle);
         //Sending list change update to clients.
         SendListChangeToClients(u16ListHandle, 0, 0);
         ETG_TRACE_USR2((" ListChangeHandler:- Cleared list and sent list change update for ListHandle= %u ", u16ListHandle));
      }

      malloc_trim(0);  // -bn: This is a trigger to return freed memory to the OS
   }
}


/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::ContentRefreshed
*
* DESCRIPTION: This functions triggers a list change property update to all the lists corresponding
* to the passed connection ID.

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: 
*
***************************************************************************************************/
void ListChange_Handler::ContentRefreshed(
   const QHash< quint16,	FC_PhoneBook_List_Data* >* pPbHash,	quint8 u8DeviceHandle, quint8 u8PBListType)   // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{
   QList<quint16> ListHandles;
   GetListofMatchingListHandles(pPbHash, u8DeviceHandle, u8PBListType, ListHandles);
   quint16 u16ListHandleSize;

   ETG_TRACE_COMP((" -PBDL.S- Refreshing Contact List(s) and SendListChange(s) for DeviceHandle= %u ",
      u8DeviceHandle ));

   u16ListHandleSize= (quint16)(ListHandles.size());
   if(0 == u16ListHandleSize)
   {
      ETG_TRACE_USR1((" ListChangeHandler:- No matching list handle found "));

      return;
   }

   enList_Change_Type= enContent_Changed;
   enList_Type= enPhoneBook_list;

   QList<FC_PhoneBook_Contact_Detail_Short> PBookList;
   for(quint16 u16Index=0; u16Index < u16ListHandleSize; u16Index++)
   {
      quint16 u16ListHandle= ListHandles.at(u16Index);

      // -bn: Stability Improvement BEGIN - Added default return value NULL to call of QHash->value(..) 
      // so that it is clear that it returns NULL when the key is not found. 
      // Added a related NULL pointer check with an em-Trace entry in case of NULL.
      FC_PhoneBook_List_Data* PBListData = pPbHash->value(u16ListHandle, NULLPTR);

      if(PBListData != NULL)
      {
         PBookList.clear();
         if(CreateNewList(PBListData->listType,u8DeviceHandle, PBListData->sortType, PBookList))
         {
            UpdateNewContactList(PBListData, PBookList);

            ETG_TRACE_USR2(("ListChange_Handler:- New list created for listhandle- %u ",u16ListHandle));
         }
         else
         {
            ETG_TRACE_ERR((" ListChange_Handler:- Creation of new list failed for listhandle- ""%u ",	u16ListHandle));
         }

         SendListChangeToClients(u16ListHandle, (quint16)PBListData->contactList.size(), 0);
      }
      else
      {
         ETG_TRACE_FATAL(( "  PHBK_NEVER_EXPECTED to come here: ListHandle is not in 'pPbHash' table " ));
         ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED to come here: ListHandle is not in 'pPbHash' table " ));
      }
      // --: Stability Improvement END		
   }

   PBookList.clear();  // -bn: We clear the list, so that the following 'malloc_trim(0)' will include the content of this list.
   malloc_trim(0);  // -bn: This is a trigger to return freed memory to the OS
}


/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::ContentCleared
*
* DESCRIPTION: This functions clears all the lists corresponding to a given connection ID and triggers a list
* change property update. Used when a device is disconnected

* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
***************************************************************************************************/
void ListChange_Handler::ContentCleared(const QHash< quint16, FC_CallHistory_List_Data* >* pChHash,
                                        quint8 u8ConnectionID)
{
   ETG_TRACE_USR2((" ContentCleared entered and call history list to be cleared for u8ConnectionID- %u"
      , u8ConnectionID));

   QList<quint16> ListHandles;
   if(pChHash)
   {
      GetListofMatchingListHandles(pChHash, u8ConnectionID, ListHandles);

      quint16 u16ListHandleSize = (quint16)(ListHandles.size());
      if(0 == u16ListHandleSize)
      {
         ETG_TRACE_USR1(("ListChangeHandler:- No matching list handle found "));
         return;
      }

      enList_Change_Type= enContent_Changed;
      enList_Type= enCallHistory_list;

      for(quint16 u16Index=0; u16Index < u16ListHandleSize; u16Index++)
      {
         quint16 u16ListHandle= ListHandles.at(u16Index);
         //Clearing list and corresponding entries from hash lists.
         FC_PhoneBook_CallHistory_List::GetCurrentInstance()->vClearCHListAndHashListEntriesForListHandle(u16ListHandle);
         //Sending list change update to clients.
         SendListChangeToClients(u16ListHandle, 0, 0);

         ETG_TRACE_USR2((" ListChangeHandler:- Cleared list and sent list change update for ListHandle= %u ", u16ListHandle));
      }
      malloc_trim(0);  // -bn: This is a trigger to return freed memory to the OS
   }
}

/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::DummyListChange
*
* DESCRIPTION: This functions triggers a dummy list change

* PARAMETER: [IN]- PhoneBook Hash table, DeviceHandle
*            [OUT]
* RETURNVALUE:
*
***************************************************************************************************/
void ListChange_Handler::DummyListChange(
   const QHash< quint16, FC_PhoneBook_List_Data* >* phashPbList, quint8 u8DeviceHandle)   // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{
   QList<quint16> ListHandles;
   GetListofMatchingListHandles(phashPbList, u8DeviceHandle, FC_PhoneBook_SQLite::ListType_PHONEBOOK, ListHandles);
   quint16 u16ListHandleSize;

   ETG_TRACE_USR2((" DummyListChange entered and ListChangeHandler:- Dummy list change for device- %u", u8DeviceHandle));

   u16ListHandleSize= (quint16)(ListHandles.size());
   if(0 == u16ListHandleSize)
   {
      ETG_TRACE_USR1(("ListChangeHandler:- No matching list handle found"));
      return;
   }

   ETG_TRACE_USR2(("Triggering dummy list change to %d list handles", u16ListHandleSize));

   for(quint16 u16Index=0; u16Index < u16ListHandleSize; u16Index++)
   {
      TriggerDummyListChange(phashPbList, ListHandles.at(u16Index));
   }
}

/***********************************************************************************************
*
* FUNCTION:  ListChange_Handler::TriggerDummyListChange
*
* DESCRIPTION: This functions triggers a dummy list change for the given list handle

* PARAMETER: [IN] - ListHandle
*            [OUT]
* RETURNVALUE:
*
***************************************************************************************************/
void ListChange_Handler::TriggerDummyListChange(
   const QHash< quint16, FC_PhoneBook_List_Data* >* phashPbList, quint16 u16ListHandle)   // Return type of QHash made const to avoid LINT Warning 1536 (CMG3G-4848)
{


   ETG_TRACE_USR2((" Dummy list change triggered for ListHandle- %d", u16ListHandle));

   most_phonbkfi_tclMsgListChangeStatus ListChangeStatus;

   FC_PhoneBook_List_Data* PBListData= phashPbList->value(u16ListHandle);

   ListChangeStatus.u16ListHandle= u16ListHandle;
   ListChangeStatus.e8ListType.enType=
      (most_fi_tcl_e8_PhonBkListType::tenType)enPhoneBook_list;
   ListChangeStatus.e8ListChangeType.enType=
      (most_fi_tcl_e8_PhonBkListChangeType::tenType)enContent_Changed;
   ListChangeStatus.u16ListLength= (quint16)PBListData->contactList.size();
   ListChangeStatus.u32NumItems= 0xFFFFFFFF;
   ListChangeStatus.oChangeItems.u32Items.clear();

   m_poservice_phonebook->vPhoneBook_UpdateListChange(ListChangeStatus);
}
