/******************************************************************************
 *
 * FILE:          FC_Phone_WaitingMode.cpp
 *
 * SW-COMPONENT:  FC_Phone application
 *
 * PROJECT:       PSA
 *
 * DESCRIPTION:   Implements class FC_Phone_WaitingMode
 *
 * AUTHOR:        Sandeep Arjun
 *
 * COPYRIGHT:    (c) 2015 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

#include "../FC_Phone_service_Telephone.h"
#include "../StateMachines/FC_Phone_SMEvents.h"
#include "../StateMachines/FC_Phone_SM.h"
#include "../StateMachines/FC_Phone_SMManager.h"
#include "../FC_Phone_clienthandler_BTSettings.h"
#include "../HelperClasses/FC_Phone_AudioManager.h"
#include "../Interface/FC_Phone_DBusInterface.h"

#include "FC_Phone_WaitingMode.h"
#include "../Audio/FC_Phone_AudioPlayer.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
#include "trcGenProj/Header/FC_Phone_WaitingMode.cpp.trc.h"
#endif

#define NARROW_BAND_SPEECH 1
#define WIDE_BAND_SPEECH 2

class fc_phone_tclService_Telephone;

fc_phone_tclWaitingMode* fc_phone_tclWaitingMode::s_poInstance = NULLPTR;

/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclWaitingMode
 *
 * DESCRIPTION: Constructor for Waiting Mode. Initialize default values, mutex
 *                object creation
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
fc_phone_tclWaitingMode::fc_phone_tclWaitingMode(tVoid)
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::fc_phone_tclWaitingMode"));

   m_enWaitingModeState = FC_PHONE_WAITINGMODE_IDLE;
   m_bWaitModeTonePlaying = FALSE;
   m_bPhoneWaitSrcAvailable = FALSE;
   m_bEventConsumed = FALSE;
   m_u8SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_STATUS_UNKNOWN;
   m_u8Codec = NARROW_BAND_SPEECH;

   // LINT warning - Fix 
   m_sFilePathNB = NULLPTR;
   m_sFilePathWB = NULLPTR;

   //Mutex to be initialised
   g_mutex_init(&m_Mutex_WaitMode);
}

/*******************************************************************************
 *
 * FUNCTION: ~fc_phone_tclWaitingMode
 *
 * DESCRIPTION: Destructor. Stops the waiting mode tone, performs
 * memory deallocation, audio source release
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/

fc_phone_tclWaitingMode::~fc_phone_tclWaitingMode()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::~fc_phone_tclWaitingMode"));

   vStopWaitingModeTone();
   vClearFilePath();
   if (TRUE == m_bPhoneWaitSrcAvailable)
   {
      tBool ret_val = (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioDeAllocation(FC_PHONE_AUDIOCHANNEL_PHONEWAIT);
      (tVoid)ret_val;
   }

   // Mutex should be cleared
   g_mutex_clear(&m_Mutex_WaitMode);
}

/*******************************************************************************
 *
 * FUNCTION: poGetInstance
 *
 * DESCRIPTION: Returns the instance of WaitingMode class
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
fc_phone_tclWaitingMode* fc_phone_tclWaitingMode::poGetInstance()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::poGetInstance"));

   if (!s_poInstance)
   {
      s_poInstance = OSAL_NEW fc_phone_tclWaitingMode;
   }

   return s_poInstance;
}

/*******************************************************************************
 *
 * FUNCTION: vDestroyWaitingModeInstance
 *
 * DESCRIPTION: Destroys the instance of WaitingMode class
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vDestroyWaitingModeInstance()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vDestroyWaitingModeInstance"));

   if (s_poInstance)
   {
      OSAL_DELETE s_poInstance;
      s_poInstance = NULLPTR;
   }
}

/*******************************************************************************
 *
 * FUNCTION: vSetFilePath
 *
 * DESCRIPTION: Sets the file path for narrow band and wideband files, one of
 *                the files will be be played during WaitingMode
 *
 * PARAMETER: sFilePathNB - File path for Narrow band file
 *            sFilePathWB - File path for Wideband file
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vSetFilePath(tString sFilePathNB, tString sFilePathWB)
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vSetFilePath"));

   vClearFilePath();

   tU16 u16StrLen = static_cast<tU16>(strlen(sFilePathNB) + 1);
   m_sFilePathNB = new tChar[u16StrLen];
   if (m_sFilePathNB)
   {
      memset(m_sFilePathNB, OSAL_NULL, u16StrLen);
      vStringCopy(m_sFilePathNB, (tCString) sFilePathNB, u16StrLen);
   }
   else
   {
      ETG_TRACE_USR4(("m_sFilePathNB is NULL"));
   }

   u16StrLen = static_cast<tU16>(strlen(sFilePathWB) + 1);
   m_sFilePathWB = new tChar[u16StrLen];
   if ( m_sFilePathWB)
   {
      memset(m_sFilePathWB, OSAL_NULL, u16StrLen);
      vStringCopy(m_sFilePathWB, (tCString) sFilePathWB, u16StrLen);
   }
   else
   {
      ETG_TRACE_USR4(("m_sFilePathWB is NULL"));
   }
}

/*******************************************************************************
 *
 * FUNCTION: enGetWaitingModeState
 *
 * DESCRIPTION: Returns the current WaitingMode state
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tenFC_Phone_WaitingModeState fc_phone_tclWaitingMode::enGetWaitingModeState()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::enGetWaitingModeState : '%u'",
         ETG_ENUM(TR_PHON_WAITINGMODESTATE, m_enWaitingModeState)));

   tenFC_Phone_WaitingModeState enFC_Phone_WaitingModeState;

      g_mutex_lock(&m_Mutex_WaitMode);
      ETG_TRACE_USR4(("m_Mutex_WaitMode lock acquired"));

   enFC_Phone_WaitingModeState = m_enWaitingModeState;

      g_mutex_unlock(&m_Mutex_WaitMode);
      ETG_TRACE_USR4(("m_Mutex_WaitMode lock released"));

   return enFC_Phone_WaitingModeState;
}

/*******************************************************************************
 *
 * FUNCTION: vSetWaitingModeState
 *
 * DESCRIPTION: Sets the current WaitingMode state
 *
 * PARAMETER: enWaitingModeState - New Waiting mode state
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vSetWaitingModeState(
      tenFC_Phone_WaitingModeState enWaitingModeState)
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vSetWaitingModeState : '%u'",
         ETG_ENUM(TR_PHON_WAITINGMODESTATE, enWaitingModeState)));

      g_mutex_lock(&m_Mutex_WaitMode);
      ETG_TRACE_USR4(("m_Mutex_WaitMode lock acquired"));

   if (m_enWaitingModeState != enWaitingModeState)
   {

      m_enWaitingModeState = enWaitingModeState;

      fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();
      if (m_enWaitingModeState == FC_PHONE_WAITINGMODE_IDLE)
      {
         poTelService->vSendWaitingModeState(most_fi_tcl_e8_WaitingModeState::FI_EN_E8WAITING_MODE_IDLE);
      }
      else if (m_enWaitingModeState == FC_PHONE_WAITINGMODE_ACTIVE)
      {
         poTelService->vSendWaitingModeState(most_fi_tcl_e8_WaitingModeState::FI_EN_E8WAITING_MODE_ACTIVE);
      }
      else if (m_enWaitingModeState == FC_PHONE_WAITINGMODE_START_IN_PROGRESS)
      {
         poTelService->vSendWaitingModeState(most_fi_tcl_e8_WaitingModeState::FI_EN_E8WAITING_MODE_STARTING);
      }
      else if (m_enWaitingModeState == FC_PHONE_WAITINGMODE_STOP_IN_PROGRESS)
      {
         poTelService->vSendWaitingModeState(most_fi_tcl_e8_WaitingModeState::FI_EN_E8WAITING_MODE_STOPPING);
      }
   }

      g_mutex_unlock(&m_Mutex_WaitMode);
      ETG_TRACE_USR4(("m_Mutex_WaitMode lock released"));
}

/*******************************************************************************
 *
 * FUNCTION: bProcessEvent
 *
 * DESCRIPTION: Handles all the external events received during all the states
 *                of the Waiting mode
 *
 * PARAMETER: enWaitingModeEvent - Event received
 *            pvArg - Arguments of the event
 *
 * RETURNVALUE: Returns TRUE if event is consumed.
 *                Otherwise FALSE is returned.
 *
 ********************************************************************************/
tBool fc_phone_tclWaitingMode::bProcessEvent(
      tenFC_Phone_WaitingModeEvent enWaitingModeEvent, tVoid *pvArg)
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::bProcessEvent : '%u'",
         ETG_ENUM(TR_PHON_WAITINGMODEEVENT, enWaitingModeEvent)));

   m_bEventConsumed = FALSE;

   switch (m_enWaitingModeState)
   {
      case FC_PHONE_WAITINGMODE_IDLE:
      {
         vHandleEventInIdle(enWaitingModeEvent, pvArg);
      }
      break;

      case FC_PHONE_WAITINGMODE_START_IN_PROGRESS:
      {
         vHandleEventInStartInProgress(enWaitingModeEvent, pvArg);
      }
      break;

      case FC_PHONE_WAITINGMODE_ACTIVE:
      {
         vHandleEventInActive(enWaitingModeEvent, pvArg);
      }
      break;

      case FC_PHONE_WAITINGMODE_STOP_IN_PROGRESS:
      {
         vHandleEventInStopInProgress(enWaitingModeEvent);
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Invalid Waiting Mode State : '%u'!!", m_enWaitingModeState));
      }
   }

   return m_bEventConsumed;
}

/*******************************************************************************
 *
 * FUNCTION: vHandleEventInIdle
 *
 * DESCRIPTION: Handles all the events received during Idle Waiting
 *                mode state
 *
 * PARAMETER: enWaitingModeEvent - Event received
 *            pvArg - Arguments of the event
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vHandleEventInIdle(
      tenFC_Phone_WaitingModeEvent enWaitingModeEvent, tVoid* pvArg)
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vHandleEventInIdle entered"));

   switch (enWaitingModeEvent)
   {
      case FC_PHONE_WAITING_MODE_EVENT_START:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_START"));

         if ((m_sFilePathNB) && (strlen(m_sFilePathNB))
               && (m_sFilePathWB) && (strlen(m_sFilePathWB)))
         {
            vStartWaitingMode();
         }
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_SCO_CONNECTION_STATUS:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_SCO_CONNECTION_STATUS"));

         trFC_Phone_SCOConnStatus* prProcessEventArg = (trFC_Phone_SCOConnStatus*) pvArg;

         if (prProcessEventArg)
         {
            m_u8SCOConnectionStatus = prProcessEventArg->u8SCOConnectionStatus;
            m_u8Codec = prProcessEventArg->u8Codec;
         }
         else
         {
            ETG_TRACE_ERR(("ProcessEvent Argument received is NULL"));
         }
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_ON:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_ON"));

         if (FALSE == ((fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioDeAllocation(
               FC_PHONE_AUDIOCHANNEL_PHONEWAIT)))
         {
            ETG_TRACE_ERR(("Audio Release request posting failed!"));
         }
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("vHandleEventInIdle default case event ID: '%u'", enWaitingModeEvent));
      }
   }
}

/*******************************************************************************
 *
 * FUNCTION: vHandleEventInStartInProgress
 *
 * DESCRIPTION: Handles all the events received during StartInProgress Waiting
 *                mode state
 *
 * PARAMETER: enWaitingModeEvent - Event received
 *            pvArg - Arguments of the event
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vHandleEventInStartInProgress(
      tenFC_Phone_WaitingModeEvent enWaitingModeEvent, tVoid* pvArg)
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vHandleEventInStartInProgress entered"));

   switch (enWaitingModeEvent)
   {
      case FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_ON:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_ON"));

         m_bPhoneWaitSrcAvailable = TRUE;

         if (FC_PHONE_DBUS_SCO_CONN_ESTABLISHED == m_u8SCOConnectionStatus)
         {
            vHandleSCOConnAndPhoneWaitSrcOn();
         }
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_SCO_CONNECTION_STATUS:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_SCO_CONNECTION_STATUS"));

         trFC_Phone_SCOConnStatus* prProcessEventArg = (trFC_Phone_SCOConnStatus*) pvArg;

         if (prProcessEventArg)
         {
            m_u8SCOConnectionStatus = prProcessEventArg->u8SCOConnectionStatus;
            m_u8Codec = prProcessEventArg->u8Codec;
         }
         else
         {
            ETG_TRACE_ERR(("ProcessEvent Argument received is NULL"));
         }

         if ((TRUE == m_bPhoneWaitSrcAvailable)
               && (FC_PHONE_DBUS_SCO_CONN_ESTABLISHED == m_u8SCOConnectionStatus))
         {
            vHandleSCOConnAndPhoneWaitSrcOn();
         }
         else if (FC_PHONE_DBUS_SCO_CONN_DISCONNECTED == m_u8SCOConnectionStatus)
         {
            vHandleSCODiscSLCOffIdleCallStatus();

            fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();

            poTelService->vSendStartStopWaitingModeMethodResult(
                  most_fi_tcl_e8_TelWaitingModeStatus::FI_EN_E8WAITINGMODE_FAILURE);
         }
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_END_OF_ACTIVE_CALLS:
      case FC_PHONE_WAITING_MODE_EVENT_SLC_OFF:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_END_OF_ACTIVE_CALLS"
               "|FC_PHONE_WAITING_MODE_EVENT_SLC_OFF"));

         vHandleSCODiscSLCOffIdleCallStatus();

         fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();

         poTelService->vSendStartStopWaitingModeMethodResult(
               most_fi_tcl_e8_TelWaitingModeStatus::FI_EN_E8WAITINGMODE_FAILURE);
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_ROUTING_FAILED:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_ROUTING_FAILED"));

         vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);

         fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();

         poTelService->vSendStartStopWaitingModeMethodResult(
               most_fi_tcl_e8_TelWaitingModeStatus::FI_EN_E8WAITINGMODE_FAILURE);
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_ACCEPTCALL_ERROR:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_ACCEPTCALL_ERROR"));

         vStopWaitingModeTone();
         if (FALSE == ((fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioDeAllocation(
               FC_PHONE_AUDIOCHANNEL_PHONEWAIT)))
         {
            ETG_TRACE_ERR(("Audio Release request posting failed!"));
         }

         m_bPhoneWaitSrcAvailable = FALSE;
         vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);
         (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vResetAllChannelAttributes();

         fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();
         poTelService->vHandle_StartStopWaitingMode_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("vHandleEventInStartInProgress default case event ID: '%u'",
               enWaitingModeEvent));
      }
   }
}

/*******************************************************************************
 *
 * FUNCTION: vHandleEventInActive
 *
 * DESCRIPTION: Handles all the events received during Active Waiting
 *                mode state
 *
 * PARAMETER: enWaitingModeEvent - Event received
 *            pvArg - Arguments of the event
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vHandleEventInActive(
      tenFC_Phone_WaitingModeEvent enWaitingModeEvent, tVoid* pvArg)
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vHandleEventInActive entered"));

   switch (enWaitingModeEvent)
   {
      case FC_PHONE_WAITING_MODE_EVENT_STOP:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_STOP"));

         vStopWaitingMode();
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_SCO_CONNECTION_STATUS:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_SCO_CONNECTION_STATUS"));

         trFC_Phone_SCOConnStatus* prProcessEventArg = (trFC_Phone_SCOConnStatus*) pvArg;

         if (prProcessEventArg)
         {
            m_u8SCOConnectionStatus = prProcessEventArg->u8SCOConnectionStatus;
            m_u8Codec = prProcessEventArg->u8Codec;
         }
         else
         {
            ETG_TRACE_ERR(("ProcessEvent Argument received is NULL"));
         }

         if (FC_PHONE_DBUS_SCO_CONN_DISCONNECTED == m_u8SCOConnectionStatus)
         {
            vHandleSCODiscSLCOffIdleCallStatus();
         }
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_OFF:
      {
         //Audio source switch by ARL (external trigger)
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_OFF"));

         vStopWaitingModeTone();
         m_bPhoneWaitSrcAvailable = FALSE;

         vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_END_OF_ACTIVE_CALLS:
      case FC_PHONE_WAITING_MODE_EVENT_SLC_OFF:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_END_OF_ACTIVE_CALLS"
               "|FC_PHONE_WAITING_MODE_EVENT_SLC_OFF"));

         vHandleSCODiscSLCOffIdleCallStatus();
      }
      break;
	  
      default:
      {
         ETG_TRACE_USR4(("vHandleEventInActive default case event ID: '%u'", enWaitingModeEvent));
      }
   }

}

/*******************************************************************************
 *
 * FUNCTION: vHandleEventInStopInProgress
 *
 * DESCRIPTION: Handles all the events received during StopInProgress Waiting
 *                mode state
 *
 * PARAMETER: enWaitingModeEvent - Event received
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vHandleEventInStopInProgress(
      tenFC_Phone_WaitingModeEvent enWaitingModeEvent)
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vHandleEventInStopInProgress entered"));

   switch (enWaitingModeEvent)
   {
      case FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_OFF:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_OFF"));

         m_bPhoneWaitSrcAvailable = FALSE;

         fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();

         poTelService->vSendStartStopWaitingModeMethodResult(
               most_fi_tcl_e8_TelWaitingModeStatus::FI_EN_E8WAITINGMODE_SUCCESS);

         vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);
      }
      break;

      case FC_PHONE_WAITING_MODE_EVENT_END_OF_ACTIVE_CALLS:
      case FC_PHONE_WAITING_MODE_EVENT_SLC_OFF:
      {
         ETG_TRACE_USR4(("Event :: FC_PHONE_WAITING_MODE_EVENT_END_OF_ACTIVE_CALLS"
               "|FC_PHONE_WAITING_MODE_EVENT_SLC_OFF"));

         fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();

         poTelService->vSendStartStopWaitingModeMethodResult(
               most_fi_tcl_e8_TelWaitingModeStatus::FI_EN_E8WAITINGMODE_FAILURE);

         vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("vHandleEventInStopInProgress default case event ID: '%u'",
               enWaitingModeEvent));
      }
   }
}

/*******************************************************************************
 *
 * FUNCTION: vStartWaitingMode
 *
 * DESCRIPTION: Posts the Start waiting mode event to respective StateMachine
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vStartWaitingMode()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vStartWaitingMode entered"));

   vSetWaitingModeState(FC_PHONE_WAITINGMODE_START_IN_PROGRESS);

   tU16 u16CallInstanceId;
   fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();
   tBool bWaitingCallFound = poTelService->bGetCallInstanceIdForWaitingCall(&u16CallInstanceId);

   //fill process Event Arguments and send processEvent
   FcPhone_processEventArg tProcessEventArg;
   tProcessEventArg.bIsItDbusAck = FALSE;
   tProcessEventArg.u16CallInstanceID = u16CallInstanceId;

   if (TRUE == bWaitingCallFound)
   {
      m_bEventConsumed = fc_phone_tclSM::m_poSMManager->bProcessEvent(
            FC_PHONE_SM_EVENT_START_WAITING_MODE, FC_PHONE_EVENTTYPE_TARGATED,
            &tProcessEventArg);
   }
   else
   {
      /* Fixed as a part of CMG3GB-3184.
       * With the Broadcast type as FC_PHONE_EVENTTYPE_BROADCAST, the event
       * FC_PHONE_SM_EVENT_START_WAITING_MODE is broadcasted to both the
       * SMs in case of three way call scenario. And thus the
       * vTriggerPhoneAudioAllocation for FC_PHONE_AUDIOCHANNEL_PHONEWAIT is posted twice.
       * Hence the Broadcast type should be FC_PHONE_EVENTTYPE_BROADCAST_CONSUMED
       */
      m_bEventConsumed = fc_phone_tclSM::m_poSMManager->bProcessEvent(
            FC_PHONE_SM_EVENT_START_WAITING_MODE, FC_PHONE_EVENTTYPE_BROADCAST_CONSUMED,
            &tProcessEventArg);
   }

   ETG_TRACE_USR4(("vStartWaitingMode EventConsumed: %s", (
         m_bEventConsumed ? "TRUE" : "FALSE")));
   if(false == m_bEventConsumed)
   {
      vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);
   }
}

/*******************************************************************************
 *
 * FUNCTION: vStopWaitingMode
 *
 * DESCRIPTION: Initiates Stop Waiting mode
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vStopWaitingMode()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vStopWaitingMode entered"));

   vStopWaitingModeTone();

   if (FALSE == ((fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioAllocation(
         FC_PHONE_AUDIOCHANNEL_PHONE)))
   {
      ETG_TRACE_ERR(("Audio Route request posting failed!"));

      vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);

      m_bEventConsumed = FALSE;
      m_bPhoneWaitSrcAvailable = FALSE;
      //If this ever happens, then update channel attributes and Transfer call to handset?
   }
   else
   {
      ETG_TRACE_USR4(("Audio Route request posting SUCCESS..."));
      m_bEventConsumed = TRUE;
      vSetWaitingModeState( FC_PHONE_WAITINGMODE_STOP_IN_PROGRESS);

      //FIX PSARCCB-4286 Phone Audio source not deallocated after resuming from waiting mode and ending call
      ETG_TRACE_USR4((" Updating acquisition counter"));
      (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vUpdateAcquisitionCounter();
      //End of fix PSARCCB-4286
   }
}

/*******************************************************************************
 *
 * FUNCTION: bPlayWaitingModeTone
 *
 * DESCRIPTION: Plays the WaitingMode tone file based on the codec info
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tBool fc_phone_tclWaitingMode::bPlayWaitingModeTone()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::bPlayWaitingModeTone"));

   if (FALSE == m_bWaitModeTonePlaying)
   {
      FC_Phone_AudioPlayer* poAudioPlayer = FC_Phone_AudioPlayer::getInstance(); //gstreamer

      if (poAudioPlayer)
      {
         tBool bRet = false;
         if (WIDE_BAND_SPEECH == m_u8Codec)
         {
            bRet = poAudioPlayer->playFile(m_sFilePathWB);
         }
         else
         {
            bRet = poAudioPlayer->playFile(m_sFilePathNB);
         }

         if(bRet)
         {
            ETG_TRACE_USR4(("WaitingMode tone is playing successfully"));
            m_bWaitModeTonePlaying = true;
            ETG_TRACE_USR4(("m_bWaitModeTonePlaying = %d", m_bWaitModeTonePlaying));

            return true;
         }
         else
         {
            ETG_TRACE_USR4(("WaitingMode tone is not played"));
         }

      }
      else
      {
         ETG_TRACE_USR4((" poAudioPlayer is NULL "));
      }
   }
   return false;
}

/*******************************************************************************
 *
 * FUNCTION: vStopWaitingModeTone
 *
 * DESCRIPTION: Stops the WaitingMode tone file if playing.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vStopWaitingModeTone()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vStopWaitingModeTone"));

   if (TRUE == m_bWaitModeTonePlaying)
   {
      FC_Phone_AudioPlayer* poAudioPlayer = FC_Phone_AudioPlayer::getInstance(); //gstreamer
      poAudioPlayer->stopPlayback();

      m_bWaitModeTonePlaying = FALSE;
      ETG_TRACE_USR4(("m_bWaitModeTonePlaying: %d", m_bWaitModeTonePlaying));
   }
}

/*******************************************************************************
 *
 * FUNCTION: vHandleSCOConnAndPhoneWaitSrcOn
 *
 * DESCRIPTION: Plays the WaitingMode tone file and sends respective MR.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vHandleSCOConnAndPhoneWaitSrcOn()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vHandleSCOConnAndPhoneWaitSrcOn"));

   if(bPlayWaitingModeTone())
   {
      vSetWaitingModeState(FC_PHONE_WAITINGMODE_ACTIVE);

      fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();
      poTelService->vSendStartStopWaitingModeMethodResult(
            most_fi_tcl_e8_TelWaitingModeStatus::FI_EN_E8WAITINGMODE_SUCCESS);
   }
   else
   {
      vHandleAudioPlayerErrorResponse();
   }
}

/*******************************************************************************
 *
 * FUNCTION: vHandleSCODiscSLCOffIdleCallStatus
 *
 * DESCRIPTION: Stops the WaitingMode tone file and release the PhoneWait
 *               audio source
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vHandleSCODiscSLCOffIdleCallStatus()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vHandleSCODiscSLCOffIdleCallStatus"));

   if (true == (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->bIsPhoneOrRingtoneChannelAcquistionIdle())
   {
      vStopWaitingModeTone();

      if (FALSE == ((fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioDeAllocation(
            FC_PHONE_AUDIOCHANNEL_PHONEWAIT)))
      {
         ETG_TRACE_ERR(("Audio Release request posting failed!"));
      }

      m_bPhoneWaitSrcAvailable = FALSE;


      //FIX PSARCCB-5431 Not able to accept Incoming call if Previous call ended during waiting mode
      //When call is in waiting mode, once all calls end or SLC OFF occurs or SCO disconnection occurs,
      //all channel attributes needs to be reset.
      // Reset all the channel attributes.

      (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vResetAllChannelAttributes();
      //End of fix PSARCCB-5431
   }
   vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);
}

/*******************************************************************************
 *
 * FUNCTION: vClearFilePath
 *
 * DESCRIPTION: Deletes the dynamically allocated memory for File paths
 *             (Both Narrow band and WideBand filepaths)
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vClearFilePath()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vClearFilePath"));

   if ( m_sFilePathNB)
   {
      OSAL_DELETE[] m_sFilePathNB;
      m_sFilePathNB = NULLPTR;
   }

   if (m_sFilePathWB)
   {
      OSAL_DELETE[] m_sFilePathWB;
      m_sFilePathWB = NULLPTR;
   }
}

/*******************************************************************************
 *
 * FUNCTION: vHandleAudioPlayerErrorResponse
 *
 * DESCRIPTION: Handles the error response for playing the Audio File
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclWaitingMode::vHandleAudioPlayerErrorResponse()
{
   ETG_TRACE_USR4(("fc_phone_tclWaitingMode::vHandleAudioPlayerErrorResponse"));

   if (FALSE == ((fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioAllocation(
         FC_PHONE_AUDIOCHANNEL_PHONE)))
   {
      ETG_TRACE_ERR(("Audio Release request posting failed!"));
   }

   m_bPhoneWaitSrcAvailable = FALSE;
   vSetWaitingModeState(FC_PHONE_WAITINGMODE_IDLE);

   fc_phone_tclService_Telephone* poTelService = fc_phone_tclService_Telephone::pGetInstance();
   poTelService->vSendStartStopWaitingModeMethodResult(
         most_fi_tcl_e8_TelWaitingModeStatus::FI_EN_E8WAITINGMODE_FAILURE);
}
