/******************************************************************************
 *
 * FILE:          FC_Phone_Timer.cpp
 *
 * SW-COMPONENT:  FC_Phone application
 *
 * PROJECT:
 *
 * DESCRIPTION:   Abstracts Timer service provided by PhoneService class
 *
 * AUTHOR:        Guruprasad G R
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/
#include "../FC_Phone_main.h"
#include "../FC_Phone_service_Telephone.h"
#include "FC_Phone_Timer.h"
#include "../Interface/FC_Phone_DBusInterface.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
   #include "trcGenProj/Header/FC_Phone_Timer.cpp.trc.h"
#endif //VARIANT_S_FTR_ENABLE_TRC_GEN

fc_phone_tclTimer* fc_phone_tclTimer::m_poInstance = NULLPTR;

/*******************************************************************************
 ** FUNCTION:   vTimerCallbackDummy(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   Internal method to handle a dummy callback.
 *  PARAMETERS:
 *     IN:   u16TimerId - Timer Id for the timer being expired.
 *
 *  RETURNVALUE: NONE
 ********************************************************************************/
void vTimerCallbackDummy(tU16 u16TimerId)
{
   ETG_TRACE_USR4(("vTimerCallbackDummy"));
   /* Print Error */
   (tVoid) u16TimerId;
}


/*******************************************************************************
 ** FUNCTION:   fc_phone_tclTimer(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   Constructor
 *  PARAMETERS:
 *     IN:   rfpTelService - Reference to service which provides timer service
 *
 *  RETURNVALUE: None
 ********************************************************************************/
fc_phone_tclTimer::fc_phone_tclTimer(fc_phone_tclApp*& rfpApp):m_oOsalTimer()
{
   ETG_TRACE_USR4(("fc_phone_tclTimer::fc_phone_tclTimer"));

   if (rfpApp)
   {
      m_poApp = rfpApp;
   }

   m_poInstance = this;
   m_u8RefCount = 0;

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_TIMER_LIMIT; u16Index++)
   {
      m_rTimerCallBackMap[u16Index].u16TimerId = FC_PHONE_TIMER_ID_UNUSED;
      m_rTimerCallBackMap[u16Index].FpCallBack = vTimerCallbackDummy;
   }
}


/*******************************************************************************
 ** FUNCTION:   ~fc_phone_tclTimer(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   Destructor
 *  PARAMETERS:
 *     IN:   None
 *
 *  RETURNVALUE: None
 ********************************************************************************/
/* Destructor */
fc_phone_tclTimer::~fc_phone_tclTimer()
{
   ETG_TRACE_USR4(("fc_phone_tclTimer::~fc_phone_tclTimer"));
   m_poApp = NULLPTR;
   m_u8RefCount = 0;
}


/*******************************************************************************
 ** FUNCTION:   s32Start
 *******************************************************************************/
/* DESCRIPTION:
 *   Start a timer with correponding Id
 *  PARAMETERS:
 *     IN:   u16TimerId - Timer Id to be started.
 *     IN:   u16TimeInMilliSeconds - Time after which fpCallBack should be called.
 *     IN:   fpCallBack - Call back function to be called once timer is expired.
 *
 *  RETURNVALUE: s32Error - FC_PHONE_TIMER_OK or ERROR Code
 ********************************************************************************/
tS32 fc_phone_tclTimer::s32Start(tU16 u16TimerId, tU32 u32TimeInMilliSeconds,
      TimerCallBack fpCallBack)
{
   ETG_TRACE_USR3(("fc_phone_tclTimer::s32Start TimerId = %d", u16TimerId));
   tU32 u32TimerId = static_cast< tU32>( u16TimerId);
   tS32 s32Error = FC_PHONE_TIMER_OK;
   tU16 u16SlotId = 0x00;

   /* Find the empty slot */
   s32Error = s32GetNextTimerSlot(u16SlotId, u16TimerId);

   if (FC_PHONE_TIMER_OK == s32Error)
   {
      if (m_oOsalTimer[u16SlotId].bInit(this, &fc_phone_tclTimer::OsalTimerCallBack, reinterpret_cast<tVoid*>(static_cast<intptr_t>(u32TimerId)))== TRUE)
      {
         ETG_TRACE_USR3(("bInit success %d ",u16TimerId));
         if (m_oOsalTimer[u16SlotId].bStart(u32TimeInMilliSeconds, u32TimeInMilliSeconds) == TRUE)
         {
            ETG_TRACE_USR3(("bStart success %d",u16TimerId));
            m_rTimerCallBackMap[u16SlotId].u16TimerId = u16TimerId;
            m_rTimerCallBackMap[u16SlotId].FpCallBack = fpCallBack;
            return (s32Error);
         }
      }
      ETG_TRACE_ERR(("s32Start timer start failed as bInit/bStart failed for timerId: %d", u16TimerId));
      s32Error = FC_PHONE_TIMER_APPLICATION_SERVICE_FAILED;
   }

   return (s32Error);
}



/*******************************************************************************
 ** FUNCTION:   s32Stop
 *******************************************************************************/
/* DESCRIPTION:
 *   Start a timer with correponding Id
 *  PARAMETERS:
 *     IN:   u16TimerId - Timer Id to be stopped.
 *
 *  RETURNVALUE: s32Error - FC_PHONE_TIMER_OK or ERROR Code
 ********************************************************************************/
tS32 fc_phone_tclTimer::s32Stop(tU16 u16TimerId)
{
   ETG_TRACE_USR3(("fc_phone_tclTimer::s32Stop TimerId = %d", u16TimerId));

   /* Find the timer in the table */
   tS32 s32Error = FC_PHONE_TIMER_OK;
   tU16 u16SlotId = 0x00;

   s32Error = s32GetSlotIdFromTimerId(u16TimerId, u16SlotId);

   if (FC_PHONE_TIMER_OK == s32Error)
   {
      if (u16TimerId == FC_PHONE_CALLINSTANCE_TIMER_ID && (m_u8RefCount != 0))
      {
         m_u8RefCount--;
         ETG_TRACE_USR4(("s32Stop RefCount: %d", m_u8RefCount));
      }

      if (u16TimerId != FC_PHONE_CALLINSTANCE_TIMER_ID || m_u8RefCount == 0)
      {
         m_rTimerCallBackMap[u16SlotId].u16TimerId = FC_PHONE_TIMER_ID_UNUSED;
         m_rTimerCallBackMap[u16SlotId].FpCallBack = vTimerCallbackDummy;
         //GMMY16-24623
         if (m_oOsalTimer[u16SlotId].bStop() == TRUE)
         {
            ETG_TRACE_USR4(("Timer stopped success %d ", u16TimerId));
         }
         else
         {
            ETG_TRACE_USR4(("Timer coudnot be stopped %d ",u16TimerId));
         }
      }
   }

   return (s32Error);
}


/*******************************************************************************
 ** FUNCTION:   s32GetNextTimerSlot
 *******************************************************************************/
/* DESCRIPTION:
 *   Fills next empty timer slot
 *  PARAMETERS:
 *     OUT:   rfSlotId - Filled with Slot Id that is free.
 *
 *  RETURNVALUE: s32Error - FC_PHONE_TIMER_OK or ERROR Code
 ********************************************************************************/
tS32 fc_phone_tclTimer::s32GetNextTimerSlot(tU16& rfSlotId, tU16 u16TimerId)
{
   ETG_TRACE_USR4(("fc_phone_tclTimer::s32GetNextTimerSlot"));

   tS32 s32Error = FC_PHONE_TIMER_LIMIT_EXCEEDED;

   // Bugfix for GMMY16-17241
   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_TIMER_LIMIT; u16Index++)
   {
      if (m_rTimerCallBackMap[u16Index].u16TimerId == u16TimerId)
      {
         if (FC_PHONE_CALLINSTANCE_TIMER_ID == u16TimerId)
         {
            m_u8RefCount++;
            ETG_TRACE_USR4(("s32GetNextTimerSlot RefCount: %d", m_u8RefCount));
         }
         ETG_TRACE_USR4(("s32GetNextTimerSlot Timer already running"));
         return FC_PHONE_TIMER_RUNNING;
      }
   }

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_TIMER_LIMIT; u16Index++)
   {
      if (FC_PHONE_TIMER_ID_UNUSED == m_rTimerCallBackMap[u16Index].u16TimerId)
      {
         if (FC_PHONE_CALLINSTANCE_TIMER_ID == u16TimerId)
         {
            m_u8RefCount++;
            ETG_TRACE_USR4(("s32GetNextTimerSlot RefCount: %d", m_u8RefCount));
         }
         rfSlotId = u16Index;
         s32Error = FC_PHONE_TIMER_OK;
         break;
      }
   }

   ETG_TRACE_USR4(("s32GetNextTimerSlot Slot: %d, Error: %d", rfSlotId, s32Error));
   return (s32Error);
}

/*******************************************************************************
 ** FUNCTION:   s32GetSlotIdFromTimerId
 *******************************************************************************/
/* DESCRIPTION:
 *   Fills next empty timer slot
 *  PARAMETERS:
 *     OUT:   rfSlotId - Filled with Slot Id that is free.
 *
 *  RETURNVALUE: s32Error - FC_PHONE_TIMER_OK or ERROR Code
 ********************************************************************************/
tS32 fc_phone_tclTimer::s32GetSlotIdFromTimerId(tU16 u16TimerId, tU16& rfSlotId)
{
   ETG_TRACE_USR3(("fc_phone_tclTimer::s32GetSlotIdFromTimerId TimerId: %d", u16TimerId));

   tS32 s32Error = FC_PHONE_TIMER_ID_INVALID;

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_TIMER_LIMIT; u16Index++)
   {
      if (u16TimerId == m_rTimerCallBackMap[u16Index].u16TimerId)
      {
         ETG_TRACE_USR4(("s32GetSlotIdFromTimerId: Timer Slot found"));
         rfSlotId = u16Index;
         s32Error = FC_PHONE_TIMER_OK;
         break;
      }
   }

   return (s32Error);
}

/*******************************************************************************
 ** FUNCTION:   vTimerExpired
 *******************************************************************************/
/* DESCRIPTION:
 *   Fills next empty timer slot
 *  PARAMETERS:
 *     OUT:
 *
 *  RETURNVALUE: s32Error - FC_PHONE_TIMER_OK or ERROR Code
 ********************************************************************************/
tVoid fc_phone_tclTimer::vTimerExpired(tU16 u16TimerId)
{
   ETG_TRACE_USR3(("fc_phone_tclTimer::vTimerExpired TimerId: %d", u16TimerId));

   tU16 u16SlotId = 0x00;
   tS32 s32Error = FC_PHONE_TIMER_OK;

   s32Error = s32GetSlotIdFromTimerId(u16TimerId, u16SlotId);

   if (FC_PHONE_TIMER_OK == s32Error)
   {
      m_rTimerCallBackMap[u16SlotId].FpCallBack(u16TimerId);
   }
}

/*******************************************************************************
 ** FUNCTION:   TimerCallBack
 *******************************************************************************/
 /* DESCRIPTION:
 *   Fills next empty timer slot
 *  PARAMETERS:
 *     OUT:   
 *
 *  RETURNVALUE: s32Error - FC_PHONE_TIMER_OK or ERROR Code
 ********************************************************************************/

//GMMY16-24623 loop back , as Osal timer runs in different thread so thread sync needed.

tVoid fc_phone_tclTimer::OsalTimerCallBack(tVoid* vTimerId)
{
   tU16 u16TimerId = static_cast<tU16>(reinterpret_cast<intptr_t>(vTimerId));
   ETG_TRACE_USR3(("fc_phone_tclTimer::vTimerExpired TimerId: %d", u16TimerId));

   fc_phone_tclService_Telephone *m_TelService_instance = (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone;
   LoopBackData LB_data;
   LB_data.u16FunctionID = FC_PHONE_CB_FID_TIMER;
   LB_data.Timer_CB.u16TimerId = u16TimerId;
   m_TelService_instance->push_back_LB_data(LB_data);
   m_TelService_instance->vPrepareLoopBackMsg(FC_PHONE_CB_FID_TIMER);

}



/*******************************************************************************
 ** FUNCTION:   pGetInstance
 *******************************************************************************/
/* DESCRIPTION:
 *   Gets the current Instance
 *  PARAMETERS:None
 *
 *
 *  RETURNVALUE: fc_phone_tclTimer* - Instance to singleton.
 ********************************************************************************/
fc_phone_tclTimer* fc_phone_tclTimer::pGetInstance(tVoid)
{
   ETG_TRACE_USR4(("fc_phone_tclTimer::pGetInstance"));
   return m_poInstance;
}


/*******************************************************************************
 ** FUNCTION:   vStopAll
 *******************************************************************************/
/* DESCRIPTION:
 *   Stops all the timers
 *  PARAMETERS:
 *     OUT:   None
 *
 *  RETURNVALUE: None
 ********************************************************************************/
tVoid fc_phone_tclTimer::vStopAll(tVoid)
{
   ETG_TRACE_USR4(("fc_phone_tclTimer::vStopAll"));

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_TIMER_LIMIT; u16Index++)
   {
      if (FC_PHONE_TIMER_ID_UNUSED != m_rTimerCallBackMap[u16Index].u16TimerId)
      {
         //GMMY16-24623
         //if(FALSE == m_poApp->bStopTimer(m_rTimerCallBackMap[u16Index].u16TimerId))
         //should we clear table values here? as of now its invoked only in onsavesettings
         if(m_oOsalTimer[u16Index].bStop() == FALSE)
         {
            ETG_TRACE_ERR(("vStopAll: Stop timer failed for TimerId: %d", m_rTimerCallBackMap[u16Index].u16TimerId));
         }
      }
   }
}


/*******************************************************************************
 ** FUNCTION:   s32GetAliveTimerStatus
 *******************************************************************************/
/* DESCRIPTION:
 *   Fills next empty timer slot
 *  PARAMETERS:
 *     OUT:   rfSlotId - Filled with Slot Id that is free.
 *
 *  RETURNVALUE: s32Error - FC_PHONE_TIMER_OK or ERROR Code
 ********************************************************************************/
tBool fc_phone_tclTimer::s32GetAliveTimerId(tU16 u16TimerId)
{
   ETG_TRACE_USR4(("fc_phone_tclTimer::s32GetAliveTimerId"));

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_TIMER_LIMIT; u16Index++)
   {
      if (u16TimerId == m_rTimerCallBackMap[u16Index].u16TimerId)
      {
         return TRUE;
      }
   }
   return FALSE;
}
