/********************************************************************************
 *
 * FILE:          FC_Phone_service_Telephone.cpp
 *
 * SW-COMPONENT:  FC_Phone application
 *
 * DESCRIPTION:   CCA service Telephone.
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

/********** INCLUDES    ***************************/

#include "FC_Phone_service_Telephone.h"
#include "FC_Phone_clienthandler_BTSettings.h"
#include "WorkQueue/FC_Phone_WorkQueue.h"
#include "WorkQueue/FC_Phone_WorkQueueScheduler.h"
#include "Audio/FC_Phone_tclAudioChannelWatchdog.h"
#include "Audio/FC_Phone_AudioHelper.h"
#include "Audio/FC_Phone_AudioPlayer.h"
#include "DBusClient/FC_Phone_DBusClientEvo.h"

#define UTFUTIL_S_IMPORT_INTERFACE_GENERIC
#include "utf_if.h"

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_SERVICE_TELEPHONE
#include "trcGenProj/Header/FC_Phone_service_Telephone.cpp.trc.h"
#endif

#define FC_PHONE_S_IMPORT_INTERFACE_MSG
#include "FC_Phone_if.h"

#include "stdio.h"
#include "StateMachines/FC_Phone_SMIncludes.h"
#include "HelperClasses/FC_Phone_PhoneData.h"
#include "time.h"

#include "midw_fi_if.h"

#include "FC_Phone_clienthandler_SystemState.h"
#include "FC_Phone_clienthandler_PhoneBook.h"
#include "HelperClasses/FC_Phone_CallManager.h"


fc_phone_tclService_Telephone* fc_phone_tclService_Telephone::m_poTelService = NULLPTR;
static void convert_raw_btaddr_to_formatted(char *node_name,const char *BTSetting_addr,tU32 dest_size);
static void convert_formatted_btaddr_to_raw(const char* BTSetting_addr, char* node_name, tU32 dest_size);
#ifdef PHONE_SIRI
tVoid vPhoneTimerCallBackService(tU16 u16TimerId);
#endif //PHONE_SIRI
// NCG3D-3221- Hold for handset-handsfree mode
tVoid vSCOChannelReleaseTimerCallBack(tU16 u16TimerID);
tVoid vHeldtoActiveSwapTimerCallBack(tU16 u16TimerId);

#define NARROW_BAND_SPEECH 1
#define WIDE_BAND_SPEECH 2

/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/
BEGIN_MSG_MAP(fc_phone_tclService_Telephone, ahl_tclBaseWork)

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_TRANSFERCALLTOHANDSET,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_TransferCallToHandSet_MethodStart)

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_TRANSFERCALLTOVEHICLE,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_TransferCallToVehicle_MethodStart)

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_ACCEPTCALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_AcceptCall_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_REJECTCALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_RejectCall_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_HANGUPCALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_HangUpCall_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_IGNORECALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_IgnoreCall_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_HOLDCALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_HoldCall_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_RESUMECALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ResumeCall_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_DIAL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_Dial_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_DIALEXTENDED,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_DialExtended_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_SPEEDDIAL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SpeedDial_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_CANCELOUTGOINGCALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_CancelOutgoingCall_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_QUERYTELEPHONENUMBER,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_QueryTelephoneNumber_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_VOICEMAIL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_VoiceMail_MethodStart )

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_SENDDTMF,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SendDTMF_MethodStart)

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_SWAPCALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SwapCall_MethodStart)

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_CONFERENCEIN,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ConferenceCall_MethodStart)

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_SPLITCALL,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SplitCall_MethodStart)

   ON_MESSAGE_SVCDATA(  MOST_TELFI_C_U16_ACQUIRETELEPHONENUMBER,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_AcquireTelephoneNumber_MethodStart)

   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_STARTSTOPWAITINGMODE,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_StartStopWaitingMode_MethodStart)

   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_SETACTIVEPHONE,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SetActivePhone_MethodStart)

   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_ACTIVATEEXTERNALVR,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ActivateExternalVR_MethodStart)

   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_DEACTIVATEEXTERNALVR,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_DeActivateExternalVR_MethodStart)

END_MSG_MAP()


/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/


/*******************************************************************************
*
* FUNCTION:    fc_phone_tclService_Telephone
*
* DESCRIPTION: Constructor.
*
*              Create an object of the base class ahl_tclBaseOneThreadService
*              with a pointer to this application, the service identifier,
*              the application identifier and the service version as parameters.
*
* PARAMETER:   None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phone_tclService_Telephone::fc_phone_tclService_Telephone(fc_phone_tclApp* poMainAppl)
: ahl_tclBaseOneThreadService(
                              /* Application Pointer              */ poMainAppl,
                              /* ID of offered Service            */ CCA_C_U16_SRV_FB_TELEPHONE,
                              /* MajorVersion of offered Service  */ FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION,
                              /* MinoreVersion of offered Service */ FC_PHONE_SERVICE_TELEPHONE_FI_MINOR_VERSION,
                              /* PatchVersion of offered Service  */ FC_PHONE_SERVICE_TELEPHONE_FI_PATCH_VERSION)
{
   ETG_TRACE_USR4(( "fc_phone_tclService_Telephone constructor entered"));

   T_CCA_Dbus_If *pCCA_DBusIf = NULLPTR;
   m_bServerAvailable = FALSE;
   m_bDiagnosticModeFlag = FALSE;
   m_poSMObserver =  NULLPTR;
   m_poSMManager =  NULLPTR;
   m_poCallManager =  NULLPTR;
   m_poSMVRSession =  NULLPTR;
   m_poSMExtVRSession =  NULLPTR;
   fc_phone_tclService_Telephone::m_poTelService = this;
   m_bDialMethodStartFlag = FALSE;
   m_bAcceptMethodStartFlag = FALSE ;
   m_bIsPnpRxd = FALSE;
   m_bIsConnectionCompRxd = FALSE;
   m_bHeldtoActiveSwapTimerRunning = FALSE;
   for(tU16 u16index = 0 ;u16index < FC_PHONE_MAX_CALLINSTANCES ;u16index++)
   {
      m_sDialInfo[u16index].m_sDialName = NULLPTR;
      m_sDialInfo[u16index].m_sDialNumber = NULLPTR;
   }

   //2HFP
   m_u8ActiveDeviceHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN;
   m_u8StoredDeviceHandletoSetActive = FC_PHONE_DEVICEHANDLE_UNKNOWN;
   m_u8ActiveDeviceHandletobeRetained = FC_PHONE_DEVICEHANDLE_UNKNOWN;

   m_u8HFProfDevHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN;

   /* Initialise DBus Client Layer stuffs */

   pCCA_DBusIf = pGetCCAPhoneDBusIf();

   ETG_TRACE_USR4(("get_cca_evo_dbus_interface Evolution entered "));
   m_pDBus_CCA_If =  get_cca_evo_dbus_interface();//Evolution instance got assigned

   register_cca_evo_callbacks(pCCA_DBusIf);

   m_pDBus_CCA_If->vInit();
   m_u8TransferCallCause = 0;
   isConfSMExecutedFromHmi = FALSE;
   isConfSMExecutedFromCallStatus = FALSE;
   m_bDtmfProcessing = FALSE;    
   u16Counter = 0;
   m_u8CurrentCallIndex = 0;
   m_u8CurrentIncomingCallIndex = 0;
   m_u8CurrentBDAddressIDforUpdate = INVALID_BDADDRESS_ID;
   m_u8BDAddressIDforAcquiringTelNo = INVALID_BDADDRESS_ID;
   m_u8AcquireTelephoneNumberRetryCnt = 0;
   m_bHoldOrResumeAllFlag = FALSE;
   m_u8EcnrConfigurationDataset = 0;

   // m_u8ConsolidatedCSCountprocessed should be FC_PHONE_MAX_CALLINSTANCES-1 so that the condition will get satisfied in
   // vUpdateCallStatus and the CS will be updated whenever called.
   // If m_u8ConsolidatedCSCountprocessed is less than FC_PHONE_MAX_CALLINSTANCES-1, the update of CS will be ignored
   // and only upon reaching FC_PHONE_MAX_CALLINSTANCES-1, the CS will be updated.
   m_u8ConsolidatedCSCountprocessed = FC_PHONE_MAX_CALLINSTANCES-1;

   m_u8DeviceHandletoSetActiveToStartVR = FC_PHONE_DEVICEHANDLE_UNKNOWN;

   for (tU8 ind = 0; ind < FC_PHONE_MAX_CONNECTED_DEVICE; ind++)
   {
#ifdef PHONE_SIRI
      m_u8SiriReqRetryCount[ind] = 0;
      m_bSiriRetryTimerRunning[ind] = FALSE;
      m_bSiriNrRetryTimerRunning[ind] = FALSE;
#endif //PHONE_SIRI
      m_u16PnpInfoVendorID[ind] = 0;
      m_u16PnpInfoVendorSource[ind] =0;
   }


 // NCG3D-3221- Hold for handset-handsfree mode
   m_bScoChannelReleaseTimerRunning = FALSE;
   m_bEndCallforPassiveDev = FALSE;

   m_sIncomingEventInfo.FirstName = NULLPTR;
   m_sIncomingEventInfo.LastName = NULLPTR;
   vResetIncomingEventInfoVariable();

   m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM;
   m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM;
   m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM;
   m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM;
   m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM;

   m_u8BandSelection = NARROW_BAND_SPEECH;
   m_bStartAudioStatus = FALSE;
   m_bSpmSubState = CALLSTATUS_INACTIVE;

   for(tU8 index = 0; index<FC_PHONE_DIAL_NUMBER_MAX_LEN_IN_CHARS +1; index++)
   {
      m_sValidPhoneNumber[index] = 0;  // 'LEN' +1 for terminating \0
   }

   T_CCA_ECNR_If *pCCA_EcnrIf = NULLPTR;
   pCCA_EcnrIf = pGetCCAEcnrIf();
   register_ecnr_callbacks(pCCA_EcnrIf); 

   m_oAudioWatchdog = OSAL_NEW fc_phone_tclAudioChannelWatchdog(*this);
   if (!m_oAudioWatchdog)
   {
      ETG_TRACE_ERR(( "Creation m_oAudioWatchdog failed"));
   }

   /* Initialise state machine stuffs  and start IDLE State Machine*/
   m_poSMObserver = OSAL_NEW fc_phone_tclSMObserver(m_poTelService);
   if (!m_poSMObserver)
   {
      ETG_TRACE_ERR(( "Creation m_poSMObserver failed"));
   }

   m_poSMManager =  OSAL_NEW fc_phone_tclSMManager(m_poSMObserver);
   if (!m_poSMManager)
   {
      ETG_TRACE_ERR(( "Creation m_poSMManager failed"));
   }

   m_poCallManager = OSAL_NEW fc_phone_tclCallManager(m_poSMManager);
   if (!m_poCallManager)
   {
      ETG_TRACE_ERR(( "Creation m_poCallManager failed"));
   }

   m_poSMVRSession =  OSAL_NEW fc_phone_tclSMVRSession();
   if (!m_poSMVRSession)
   {
      ETG_TRACE_ERR(( "Creation m_poSMVRSession failed"));
   }

   m_poSMExtVRSession =  OSAL_NEW fc_phone_tclSMExtVRSession();
   if (!m_poSMExtVRSession)
   {
      ETG_TRACE_ERR(( "Creation m_poSMExtVRSession failed"));
   }

   fc_phone_tclSM::m_poAudioManager = fc_phone_tclAudioManager::pGetInstance();
   fc_phone_tclSM::m_poTimer = fc_phone_tclTimer::pGetInstance();
   fc_phone_tclSM::m_poSMManager = m_poSMManager;
   fc_phone_tclSM::m_pDbusIf = m_pDBus_CCA_If;

   m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecActive = FC_PHONE_BT_VR_INACTIVE ;

   // CMG3G-7861 - Siri availability status
   m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive = FC_PHONE_BT_VR_INACTIVE ;
   m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = FC_PHONE_BT_DEVICE_VR_NOT_SUPPORTED;
   m_oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState.enType = (most_fi_tcl_e8_TelSiriAvailabilityState::tenType) NOT_AVAILABLE;
   m_oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN;

   for(tU16 u16index = 0 ;u16index < FC_PHONE_MAX_CALLINSTANCES ;u16index++)
   {
      fc_phone_tclCallInstance* pCallInstance = &m_oCallInstances[u16index];
      pCallInstance->m_u16Id = u16index ;

      if (m_poSMManager)
      {
         m_poSMManager->vStartSM(FC_PHONE_SMMANAGER_SMID_IDLE,pCallInstance);
      }
   }

   //Initialize a new mutex object and create work queue
   fc_phone_tclPhoneMutex::PhoneMutexInit();
   // CMG3G-8447. Lint fix.
   // fc_phone_tclWorkQueue::getWorkQueueInstance()->createWorkQueue();

   FC_Phone_AudioPlayer::getInstance(); //gstreamer

   // Fix for NCG3D-103060
   m_bExternalTriggerStatusforPhoneChannel = FALSE;
#ifdef PHONE_WAITINGMODE
   fc_phone_tclWaitingMode::poGetInstance();
#endif
   m_pobtTelClient = NULLPTR;
   m_u16SCOChannelReleaseTimerId = FC_PHONE_TIMER_ID_UNUSED;
   OutputDevStatus = 0;
   InputDevStatus = 0;
   memset(m_SCOChannelConnectionStatus.BD_Address, '\0', PHONE_BDADRESS_LENGTH);
   m_SCOChannelConnectionStatus.Codec = FC_PHONE_DBUS_SCO_CONN_CODE;
   m_SCOChannelConnectionStatus.SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_STATUS_UNKNOWN;
   m_oExternalVRStateStatus.e8ExternalVRState.enType = most_fi_tcl_e8_ExternalVRState::FI_EN_E8IDLE;
   m_oExternalVRStateStatus.e8VRErrorCode.enType = most_fi_tcl_e8_VRErrorCode::FI_EN_E8NO_ERROR;

   u16ConferenceCall1 = INVALID_INDEX;
   u16ConferenceCall2 = INVALID_INDEX;

   //Mutex to be initialised
   g_mutex_init(&p_mutex_loopback);
}

/*******************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::~fc_phone_tclService_Telephone(tVoid)
*
* DESCRIPTION: Destructor.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phone_tclService_Telephone::~fc_phone_tclService_Telephone()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::~fc_phone_tclService_Telephone entered"));

   m_oSLCStatusStatus.vDestroy();
   m_oSourceAvailableStatus.vDestroy();
   m_oBTDeviceVoiceRecStatus.vDestroy();
   m_oWaitingModeFilePathStatus.vDestroy();
   // CMG3G-7861 - Siri availability status
   m_oBTDeviceVoiceRecExtendedStatus.vDestroy();
   m_oWaitingModeStateStatus.vDestroy();

   if (m_poSMObserver)
   {
      OSAL_DELETE m_poSMObserver;
      m_poSMObserver = NULLPTR;
   }

   if (m_poSMManager)
   {
      OSAL_DELETE m_poSMManager;
      m_poSMManager = NULLPTR;
   }

   if (m_poCallManager)
   {
      OSAL_DELETE m_poCallManager;
      m_poCallManager = NULLPTR;
   }

   if (m_poSMVRSession)
   {
      OSAL_DELETE m_poSMVRSession;
      m_poSMVRSession = NULLPTR;
   }

   if (m_poSMExtVRSession)
   {
      OSAL_DELETE m_poSMExtVRSession;
      m_poSMExtVRSession = NULLPTR;
   }

   if (m_oAudioWatchdog)
   {
      OSAL_DELETE m_oAudioWatchdog;
      m_oAudioWatchdog = NULLPTR;
   }

	//lint fix
   if (m_pDBus_CCA_If)
   {
      m_pDBus_CCA_If = NULLPTR;
   }

   if (m_pobtTelClient)
   {
      m_pobtTelClient = NULLPTR;
   }

   //Destroy the work queue and mutex object created
   fc_phone_tclWorkQueue::destroyWorkQueue();
   fc_phone_tclPhoneMutex::PhoneMutexDestroy();

   FC_Phone_AudioPlayer::deleteInstance(); //gstreamer

#ifdef PHONE_WAITINGMODE
   fc_phone_tclWaitingMode::vDestroyWaitingModeInstance();
#endif

   for(tU8 u8Index =0 ;u8Index<FC_PHONE_MAX_CALLINSTANCES ;u8Index++)
   {
      vResetDiallerInfoForCallInstance(u8Index);
   }

   vResetIncomingEventInfoVariable();

   // Mutex should be cleared
   g_mutex_clear(&p_mutex_loopback);
   LB_data_queue.clear();
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_phone_tclService_Telephone::vOnServiceAvailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              which is offered by this server has become available.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadService::vOnServiceAvailable().
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vOnServiceAvailable()
{
   ETG_TRACE_USR4((" ENTER :fc_phone_tclClientHandler_Telephone::vOnServiceAvailable entered"));
   m_bServerAvailable = TRUE;
   /* Send that phone is initialised and ready */
   ETG_TRACE_USR4((" Sending vOnServiceAvailable Status "));
   m_oSourceAvailableStatus.bSourceAvailable = TRUE;

   //"SetSourceAvailable() is not needed here"- Confirmed by AUDIO team.->Fixing this as part of NCG3D-56473
   //(fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->SetSourceAvailable(FALSE);

   //Make the ARL_SRC_PHONE and ARL_SRC_PHONE_VR arl sources available for playing, 1 - indicates subsource
   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vSetSourceAvailability(ARL_EN_SRC_PLAYABLE, ARL_EN_REASON_NEWMEDIA, ARL_SRC_PHONE, 1);
   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vSetSourceAvailability(ARL_EN_SRC_PLAYABLE, ARL_EN_REASON_NEWMEDIA, ARL_SRC_PHONE_VR, 1);
   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vSetSourceAvailability(ARL_EN_SRC_PLAYABLE, ARL_EN_REASON_NEWMEDIA, ARL_SRC_PHONE_WAIT, 1);

   vSendPropertyUpdate(MOST_TELFI_C_U16_SOURCEAVAILABLE, NULLPTR);
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phone_tclService_Telephone::vOnServiceUnavailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              which is offered by this server has become unavailable.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadService::vOnServiceUnavailable().
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vOnServiceUnavailable()
{
   ETG_TRACE_USR4(("vOnServiceUnavailable entered."));
   m_bServerAvailable = FALSE;
   m_oSourceAvailableStatus.bSourceAvailable = FALSE;
   vSendPropertyUpdate(MOST_TELFI_C_U16_SOURCEAVAILABLE, NULLPTR);
}

/******************************************************************************
*
* FUNCTION: vOnHeldtoActiveSwapTimerExpired
*
* DESCRIPTION: This function will get called to stop timer and update the call status
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vOnHeldtoActiveSwapTimerExpired(tU16 u16TimerId)
{
   ETG_TRACE_USR3((" vOnHeldtoActiveSwapTimerExpired "));

   // Stop the timer
   (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(u16TimerId);

   ETG_TRACE_USR3((" m_bHeldtoActiveSwapTimerRunning = FALSE"));
   m_bHeldtoActiveSwapTimerRunning = FALSE;

   //updating CALLSTATUSNOTICE and CALLSTATUSNOTICEEXTENDED
   vUpdateCallStatus();
}

/*******************************************************************************
*
* FUNCTION: vHeldtoActiveSwapTimerCallBack
*
* DESCRIPTION: Called when HeldtoActiveSwapTimer expired.
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid vHeldtoActiveSwapTimerCallBack(tU16 u16TimerId)
{
   ETG_TRACE_USR3((" vHeldtoActiveSwapTimerCallBack ENTERED with u16TimerId %d ",u16TimerId));

   fc_phone_tclService_Telephone *ptelService = fc_phone_tclService_Telephone::pGetInstance();
   ptelService->vOnHeldtoActiveSwapTimerExpired(u16TimerId);

}

/*******************************************************************************
*
* FUNCTION: bStatusMessageFactory
*
* DESCRIPTION: This function is called by the CCA framework to request ANY
*              property which is offered by this service. For each property
*              accessed via parameter 'u16FunctionId' the user has to prepare
*              the corresponding FI data object which is then copied to the
*              referenced parameter 'roOutMsg'.
*
* PARAMETER: [IN] u16FunctionId = Function ID of the requested property.
*            [OUT] roOutMsg = Reference to the service data object to which the
*                             content of the prepared FI data object should be
*                             copied to.
*            [IN] poInMsg = Selector message which is used to select dedicated
*                           content to be copied to 'roOutMsg' instead of
*                           updating the entire FI data object.
*
* RETURNVALUE: TRUE = Status message for property successfully generated.
*              FALSE = Failed to generate status message for property.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadService::bStatusMessageFactory().
*******************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageFactory(tU16 u16FunctionId, amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg)
{
   tBool bSuccess = FALSE;

   ETG_TRACE_USR3(("bStatusMessageFactory entered. FID = 0x%4x.", u16FunctionId));

   switch(u16FunctionId)
   {
   case MOST_TELFI_C_U16_INCOMINGCALLEVENT:
      {
         bSuccess = bStatusMessageIncomingCallEvent(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_CALLSTATUSNOTICE:
      {
         bSuccess = bStatusMessageCallStatusNotice(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED:
      {
         bSuccess = bStatusMessageCallStatusNoticeExtended(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_ACTIVECALLINHANDSET:
      {
         bSuccess = bStatusMessageActiveCallInHandset(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_CELLDEVICENAME:
      {
         bSuccess = bStatusMessageCellDeviceName(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_CELLNETWORKSTATUS:
      {
         bSuccess = bStatusMessageCellNetworkStatus(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_CELLSTATEOFCHARGE:
      {
         bSuccess = bStatusMessageCellStateOfChargeStatus(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_CELLSIGNALSTRENGTH:
      {
         bSuccess = bStatusMessageCellSignalStrength(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_SLCSTATUS:
      {
         bSuccess = bStatusMessageSLCStatus(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_SOURCEAVAILABLE:
      {
         bSuccess = bStatusMessageSourceAvailable(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_NETWORKOPERATOR:
      {
         bSuccess = bStatusMessageNetworkOperator(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_VEHICLEAUDIO:
      {
         bSuccess = bStatusMessageVehicleAudio(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_MICROPHONEMUTESTATE:
      {
         bSuccess = bStatusMessageMicrophoneMuteState(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_HFSERVICE:
      {
         bSuccess = bStatusMessageHFService(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITION:
      {
         bSuccess = bStatusMessageBTVoiceRecognition(roOutMsg, poInMsg);
      }
      break;

   // CMG3G-7861 - Siri availability status
   case MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITIONEXTENDED:
      {
         bSuccess = bStatusMessageBTVoiceRecognitionExtended(roOutMsg, poInMsg);
      }
      break;

   case MOST_TELFI_C_U16_RINGTONEMUTESTATE:
      {
         bSuccess = bStatusMessageRingtoneMuteState(roOutMsg, poInMsg);
      }
      break;

#ifdef PHONE_HFENHANCED
   case MOST_TELFI_C_U16_HANDSFREEENHANCEDCALLCONTROL:
      {
         bSuccess = bStatusMessageHFEnhancedControl(roOutMsg, poInMsg);
      }
      break;
#endif /* PHONE_HFENHANCED */

   case MOST_TELFI_C_U16_WAITINGMODEFILEPATH:
   {
      bSuccess = bStatusMessageWaitingModeFilePath(roOutMsg, poInMsg);
   }
   break;

   case MOST_TELFI_C_U16_ACTIVEPHONEDEVICE:
   {
      bSuccess = bStatusMessageActivePhoneDevice(roOutMsg, poInMsg);
   }
   break;
   case MOST_TELFI_C_U16_OUTGOINGCALLFAILEDEVENT:
   {
      ETG_TRACE_USR3(("bStatusMessageOutgoingCallFailedEvent MOST_TELFI_C_U16_OUTGOINGCALLFAILEDEVENT"));
      bSuccess = bStatusMessageOutgoingCallFailedEvent(roOutMsg, poInMsg);
   }
   break;
   case MOST_TELFI_C_U16_INCOMINGCALLREJECTEDEVENT:
   {
      ETG_TRACE_USR3(("bStatusMessageIncomingCallRejectEvent MOST_TELFI_C_U16_INCOMINGCALLREJECTEDEVENT"));
      bSuccess = bStatusMessageIncomingCallRejectEvent(roOutMsg, poInMsg);
   }
   break;

   case MOST_TELFI_C_U16_WAITINGMODESTATE:
   {
      ETG_TRACE_USR3(("bStatusMessageWaitingModeStatus MOST_TELFI_C_U16_WAITINGMODESTATE"));
      bSuccess = bStatusMessageWaitingModeStatus(roOutMsg, poInMsg);
   }
   break;

   case MOST_TELFI_C_U16_EXTERNALVRSTATE:
   {
      ETG_TRACE_USR3(("bStatusMessageExternalVRStatus MOST_TELFI_C_U16_EXTERNALVRSTATE"));
      bSuccess = bStatusMessageExternalVRStatus(roOutMsg, poInMsg);
   }
   break;

   default:
      {
         ETG_TRACE_ERR(("bStatusMessageFactory.::Invalid Property Id: Creation of message with 'FID = %u' failed.", u16FunctionId));
      }
      break;
   }

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION: bProcessSet
*
* DESCRIPTION: This function is called by the CCA framework when it has
*              received a message for a property with Opcode 'SET' or 'PURESET'
*              and there is no dedicated handler function defined in the
*              message map for this pair of FID and opcode. The user has to
*              set the application specific property to the requested value
*              and the CCA framework then cares about informing the requesting
*              client as well as other registered clients.
*
* PARAMETER: [IN]    poMessage = Property to be set.
*            [OUT]   bPropertyChanged = Property changed flag to be set to TRUE
*                                     if property has changed. Otherwise to be
*                                     set to FALSE (default).
*            [OUT]   u16ErrorCode = Error code to be set if a CCA error occurs,
*                                 otherwise don't touch.
*
* RETURNVALUE: TRUE = Send 'STATUS' message to the requesting client if opcode
*                     was 'SET'. Send 'STATUS' message to other registered
*                     clients as well if [OUT] parameter 'bPropertyChanged'
*                     is set to TRUE.
*              FALSE = Send an error message to the requesting client.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadService::bProcessSet().
*******************************************************************************/
tBool fc_phone_tclService_Telephone::bProcessSet(amt_tclServiceData* poMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::bProcessSet entered "));
   tBool bSuccess = FALSE;
   bPropertyChanged = FALSE;

   tU16 u16FunctionId = poMessage->u16GetFunctionID();

   ETG_TRACE_USR1((" FID = 0x%4x.", u16FunctionId));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   switch(u16FunctionId)
   {
      case MOST_TELFI_C_U16_RINGTONEMUTESTATE:
      {
         ETG_TRACE_USR4((" bProcessSet::MOST_TELFI_C_U16_RINGTONEMUTESTATE ENTERED"));
         most_telfi_tclMsgRingtoneMuteStatePureSet oRingtoneMuteStatePureSet;

         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.

         if ( OSAL_ERROR != oVisitorMsg.s32GetData(oRingtoneMuteStatePureSet, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION) )
         {
            ETG_TRACE_USR3((" propertySet:oRingtoneMuteStatePureSet  bEnable :: %d",oRingtoneMuteStatePureSet.bEnable));

            tU16 u16EventId ;
            tBool bEventConsumed = FALSE ;

            if(TRUE == oRingtoneMuteStatePureSet.bEnable)
            {
               u16EventId = FC_PHONE_SM_EVENT_RINGTONEMUTE ;
            }
            else
            {
               u16EventId = FC_PHONE_SM_EVENT_RINGTONEUNMUTE ;
            }

            //fill process Event Arguments and send processEvent
            FcPhone_processEventArg   tProcessEventArg;
            tProcessEventArg.bIsItDbusAck =  FALSE;
            if (m_poSMManager)
            {
               bEventConsumed = m_poSMManager->bProcessEvent(u16EventId, FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
            }

            if(TRUE == bEventConsumed)
            {

               bPropertyChanged = TRUE;
               ETG_TRACE_USR3((" Ringtone Mute Processed.... "));
            }
            else
            {
               ETG_TRACE_ERR(( "RingtoneMuteState_Get: Received message invalid." ));
               u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
            }
         }
         else
         {
            ETG_TRACE_ERR(( "RingtoneMuteState_Get: Received message invalid." ));
            u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }

         bSuccess = TRUE;
         oRingtoneMuteStatePureSet.vDestroy();
      }
      break;
      case MOST_TELFI_C_U16_MICROPHONEMUTESTATE:
      {
         ETG_TRACE_USR4((" bProcessSet::MOST_TELFI_C_U16_MICROPHONEMUTESTATE ENTERED"));


         most_telfi_tclMsgMicrophoneMuteStatePureSet oMicrophoneMuteStatePureSet;

         if ( OSAL_ERROR != oVisitorMsg.s32GetData(oMicrophoneMuteStatePureSet, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION) )
         {
            /* Get the coresponding Call Instance */
            tU16 u16Id = 0xFF;
            tU16 u16Index;
            ETG_TRACE_USR1((" propertySet:oMicrophoneMuteStatePureSet bEnable :: %d ",oMicrophoneMuteStatePureSet.bEnable));

            //TODO : process the mute/unmute to SM

            // check for the active Object
            for(u16Index =0 ;u16Index<FC_PHONE_MAX_CALLINSTANCES ;u16Index++)
            {
               // TODO : have to take care in FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE implementation
               if((FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE == m_oCallInstances[u16Index].m_u8ObjectState) &&((FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE ==  m_oCallInstances[u16Index].m_u16CallStatusNotice) || (FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE ==  m_oCallInstances[u16Index].m_u16CallStatusNotice)|| (FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD  ==  m_oCallInstances[u16Index].m_u16CallStatusNotice)))
               {
                  u16Id = u16Index;
                  break;
               }
            }

            if (0xFF != u16Id) //got the active object and active call status
            {
               if(REQ_SUCCESS == vMuteMicroPhone(oMicrophoneMuteStatePureSet.bEnable))
               {
                  ETG_TRACE_USR3((" In fc_phone_tclService_Telephone::MicrophoneMute action succeeded"));
                  bPropertyChanged = TRUE;
               }
            }
            else
            {
               /* Invalid CallInstance Requested, So send an error message */
               ETG_TRACE_ERR(( "vHandle_MicroPhoneMute_MethodStart: Received message invalid: NO call object/state active ." ));
               u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
            }
         }
         else
         {
            ETG_TRACE_ERR(( "vHandle_MicroPhoneMute_MethodStart: Received message invalid." ));
            u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }

         bSuccess = TRUE;
         oMicrophoneMuteStatePureSet.vDestroy();
      }
      break;

      case MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITION:
      {
         ETG_TRACE_USR3((" In fc_phone_tclService_Telephone::vHandle_BtDeviceVoiceRecognition_PureSet ENTERED"));

         most_telfi_tclMsgBTDeviceVoiceRecognitionPureSet oBTDeviceVoiceRecognitionPureSet;


         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.

         if ( OSAL_ERROR != oVisitorMsg.s32GetData(oBTDeviceVoiceRecognitionPureSet, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION) )
         {
            ETG_TRACE_USR1((" propertySet:oBTDeviceVoiceRecognitionPureSet BTDeviceVoiceRecActive :: %d ",oBTDeviceVoiceRecognitionPureSet.bBTDeviceVoiceRecActive));

            m_oBTDeviceVoiceRecMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
            m_oBTDeviceVoiceRecMessage.u16RegisterID = poMessage->u16GetRegisterID();
            m_oBTDeviceVoiceRecMessage.u16CmdCounter = poMessage->u16GetCmdCounter();
            m_oBTDeviceVoiceRecMessage.u16FunctionID = poMessage->u16GetFunctionID();

            tU8 u8ActiveBDAddressId = INVALID_BDADDRESS_ID;
            tBool bGetActiveDeviceAddress = b_get_ActiveDevice_AddressId_FromBT(&u8ActiveBDAddressId);
            if (FALSE == bGetActiveDeviceAddress)
            {
               ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
               return false;
            }

            if (m_poSMManager && (m_oPhoneData[u8ActiveBDAddressId].m_bBtVoiceRecSupport))
            {
               if (TRUE == oBTDeviceVoiceRecognitionPureSet.bBTDeviceVoiceRecActive)
               {
                  if((fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->bIsPhoneOrRingtoneChannelAcquistionIdle())
                  {
                     m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_CCX_BT_DEVICE_VR_ACTIVE);
                  }
                  else
                  {  // NCG3D-13611 VR session is activated, during the process of phone channel deallocation. Hence the BT_VR channel request got denied.
                     u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
                     ETG_TRACE_USR4(("Error: Could not process vProcessVREvent, as Phone/Ringtone channel is not released yet "));
                     vHandle_BtDeviceVoiceRecognition_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
                  }
               }
               else
               {
                  m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_CCX_BT_DEVICE_VR_INACTIVE);
               }
            }
            else
            {
               m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecActive = FC_PHONE_BT_VR_INACTIVE;
               m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecSupported = FC_PHONE_BT_DEVICE_VR_NOT_SUPPORTED;
               bPropertyChanged = TRUE;
            }
         }
         else
         {
            ETG_TRACE_ERR(( "RingtoneMuteState_Get: Received message invalid." ));
            u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }

         bSuccess = TRUE;
         oBTDeviceVoiceRecognitionPureSet.vDestroy();
      }
      break;

      case MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITIONEXTENDED:
      {
         ETG_TRACE_USR3((" In fc_phone_tclService_Telephone::vHandle_BtDeviceVoiceRecognitionExtended_PureSet ENTERED"));

         most_telfi_tclMsgBTDeviceVoiceRecognitionExtendedPureSet oBTDeviceVoiceRecognitionExtendedPureSet;

         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.

         if ( OSAL_ERROR != oVisitorMsg.s32GetData(oBTDeviceVoiceRecognitionExtendedPureSet, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION) )
         {
            ETG_TRACE_USR1((" oBTDeviceVoiceRecognitionExtendedPureSet BTDeviceVoiceRecActive: %d", oBTDeviceVoiceRecognitionExtendedPureSet.bBTDeviceVoiceRecActive));

            tU8 u8Received_VRDevHandle = oBTDeviceVoiceRecognitionExtendedPureSet.u8DeviceHandle;
            ETG_TRACE_USR1((" oBTDeviceVoiceRecognitionExtendedPureSet u8DeviceHandle: %u", u8Received_VRDevHandle));

            m_oBTDeviceVoiceRecExtendedMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
            m_oBTDeviceVoiceRecExtendedMessage.u16RegisterID = poMessage->u16GetRegisterID();
            m_oBTDeviceVoiceRecExtendedMessage.u16CmdCounter = poMessage->u16GetCmdCounter();
            m_oBTDeviceVoiceRecExtendedMessage.u16FunctionID = poMessage->u16GetFunctionID();

            tU8 u8BDAddressId =(tU8) m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(u8Received_VRDevHandle);

            if (INVALID_BDADDRESS_ID == u8BDAddressId)
            {
               ETG_TRACE_ERR((" Received Device Handle is INVALID or the device is currently not connected"));
               u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
               vHandle_BtDeviceVoiceRecognitionExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
            else
            {
               if (m_poSMManager && m_oPhoneData[u8BDAddressId].m_bBtVoiceRecSupport)
               {
                  if (oBTDeviceVoiceRecognitionExtendedPureSet.bBTDeviceVoiceRecActive)
                  {
                     if((fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->bIsPhoneOrRingtoneChannelAcquistionIdle())
                     {
                        if (u8Received_VRDevHandle == m_u8ActiveDeviceHandle)
                        {
                           if(fc_phone_tclSM::m_poSMManager->bIsExternalVRSessionIdle())
                           {
                              m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_CCX_BT_DEVICE_VR_ACTIVE);
                           }
                           else
                           {
                              ETG_TRACE_USR3((" External VR is currently active"));
                           }
                        }
                        else
                        {
                           ETG_TRACE_USR3((" VR request received for Passive device"));
                           m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_CCX_BT_DEVICE_VR_ACTIVE_FOR_PASSIVE_DEVICE);
                           m_u8DeviceHandletoSetActiveToStartVR = u8Received_VRDevHandle;
                        }
                     }
                     else
                     {
                        // NCG3D-13611 VR session is activated, during the process of phone channel deallocation. Hence the BT_VR channel request got denied.
                        u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
                        ETG_TRACE_USR4(("Error: Could not process vProcessVREvent, as Phone/Ringtone channel is not released yet "));
                        vHandle_BtDeviceVoiceRecognitionExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
                     }
                  }
                  else
                  {
                     m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_CCX_BT_DEVICE_VR_INACTIVE);
                  }
               }
               else
               {
                  m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive = FC_PHONE_BT_VR_INACTIVE;
                  m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = FC_PHONE_BT_DEVICE_VR_NOT_SUPPORTED;
                  m_oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle = u8Received_VRDevHandle;
#ifdef PHONE_SIRI
                  m_oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState.enType = (most_fi_tcl_e8_TelSiriAvailabilityState::tenType)\
                        ((fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->m_oSiriData[u8BDAddressId].eGetAplSiriState());
#endif
                  bPropertyChanged = TRUE;
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(( "BTVRVoiceRecActive_Set: Received message invalid." ));
            u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }

         bSuccess = TRUE;
         oBTDeviceVoiceRecognitionExtendedPureSet.vDestroy();
      }
      break;

      case MOST_TELFI_C_U16_WAITINGMODEFILEPATH:
      {
         ETG_TRACE_USR4((" bProcessSet::MOST_TELFI_C_U16_WAITINGMODEFILEPATH ENTERED"));

         most_telfi_tclMsgWaitingModeFilePathPureSet oWaitingModeFilePathPureSet;

         if (OSAL_ERROR != oVisitorMsg.s32GetData(oWaitingModeFilePathPureSet, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION))
         {
            tString sFilePath = oWaitingModeFilePathPureSet.sFilePathNB.szValue;

            if (sFilePath)
               ETG_TRACE_USR3(("oWaitingModeFilePathPureSet  FilePathNB :: '%s'", sFilePath));

            tString sConcatPath = checkFileNameAndConcatFilePath(sFilePath);

            if (sConcatPath)
            {
               ETG_TRACE_USR3(("Concatenated FilePathNB :: %s", sConcatPath));

               m_oWaitingModeFilePathStatus.sFilePathNB.bSet(sConcatPath);

               if (sConcatPath != sFilePath)
                  delete [] sConcatPath;
            }
            else
            {
               ETG_TRACE_ERR(("Concatenation path is NULL for NB"));
               // TODO: Later set the default waiting mode tone path here for NB

               // Do nothing but set it as an empty string
               m_oWaitingModeFilePathStatus.sFilePathNB.bSet("");
            }

            sFilePath = oWaitingModeFilePathPureSet.sFilePathWB.szValue;

            if (sFilePath)
               ETG_TRACE_USR3(("oWaitingModeFilePathPureSet  FilePathWB :: '%s'", sFilePath));

            sConcatPath = checkFileNameAndConcatFilePath(sFilePath);

            if (sConcatPath)
            {
               ETG_TRACE_USR3(("Concatenated FilePathWB :: '%s'", sConcatPath));
               m_oWaitingModeFilePathStatus.sFilePathWB.bSet(sConcatPath);

               if (sConcatPath != sFilePath)
                  delete [] sConcatPath;
            }
            else
            {
               ETG_TRACE_ERR(("Concatenation path is NULL for WB"));
               // TODO: Later set the default waiting mode tone path here for WB

               // Do nothing but set it as an empty string
               m_oWaitingModeFilePathStatus.sFilePathWB.bSet("");
            }

            // This Macro has to be checked and removed
#ifdef PHONE_WAITINGMODE
            fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
            poWaitingMode->vSetFilePath(
                  m_oWaitingModeFilePathStatus.sFilePathNB.szValue,
                  m_oWaitingModeFilePathStatus.sFilePathWB.szValue);
#endif

         }
         else
         {
            ETG_TRACE_ERR(("WaitingModeFilePathPureSet: Received message invalid."));
            u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }

         bSuccess = TRUE;
         oWaitingModeFilePathPureSet.vDestroy();
      }
      break;

      default:
      { // Added to solve lint warnings.
         ETG_TRACE_USR3(("bProcessSet. Following FID not found: '%u'", u16FunctionId));
      }
      break;
   }

   return bSuccess;
}

/*******************************************************************************
*FUNCTION:    checkFileNameAndConcatFilePath
*
*DESCRIPTION:
*Check whether the tone received is with full path or not.
*  If NOT, then append the filename with /var/opt/bosch/static/connectivity/tones/waitingmodetones/, before set.
*  If YES, then set it directly.
*
*RETURN VALUE:
*  NULL if error  :empty string, file does not exist
*  else           :pointer to the string which has to be set
*                    if this pointer is different to sFilePath paramter then the calling function has to delete this pointer
*
* PARAMETER:
*  sFilePath -NB/WB File name to be concatenated with FilePath
********************************************************************************/

tString fc_phone_tclService_Telephone::checkFileNameAndConcatFilePath(tString sFilePath)
{
   ETG_TRACE_USR3(("Entered checkFileNameAndConcatFilePath"));

   if( (!sFilePath) || (strlen(sFilePath) == 0) )
      return NULLPTR;

   ETG_TRACE_USR3(("checkFileNameAndConcatFilePath :: sFilePath: %s", sFilePath));

   tString retString = NULLPTR;
   tString searchFilePath = strchr(sFilePath, '/');

   if (!searchFilePath)
   {
      tU16 u16StrLen = static_cast<tU16>(strlen(sFilePath) + strlen(WAITINGMODE_TONE_PATH) + 1);

      tString sCompletePath = new tChar[u16StrLen];

      if (sCompletePath)
      {
         memset(sCompletePath, OSAL_NULL, u16StrLen);
         strcpy(sCompletePath, WAITINGMODE_TONE_PATH);
         strcat(sCompletePath, (tCString) sFilePath);

         ETG_TRACE_USR3(("concatWaitingModeTonePath Concatenated FilePath:: %s", sCompletePath));

         retString = sCompletePath;
      }
      else
      {
         ETG_TRACE_ERR(("allocation error"));
         return NULLPTR;
      }
   }
   else
   {
      ETG_TRACE_USR3(("The complete file path is provided by the client"));
      retString = sFilePath;
   }

   if((access(retString, F_OK) != -1))
   {
      ETG_TRACE_USR3(("File is present in the waiting mode tone path"));
   }
   else
   {
      ETG_TRACE_ERR(("File path does not exist!"));

      if (retString != sFilePath)
         delete [] retString;

      return NULLPTR;
   }

   return retString;
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageIncomingCallEvent
*
* DESCRIPTION: Prepares the status message regarding Incoming call event
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageIncomingCallEvent(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;

   most_telfi_tclMsgInComingCallEventStatus oIncomingCallEvent_Status;

   oIncomingCallEvent_Status.e8IncomingCallEventEnum.enType = (most_fi_tcl_e8_TelIncomingCallEventEnum::tenType)m_sIncomingEventInfo.bIsRinging;

   oIncomingCallEvent_Status.u16CallInstance = m_sIncomingEventInfo.CallInstance;
   oIncomingCallEvent_Status.u8DeviceHandle = m_sIncomingEventInfo.DeviceHandle;
   oIncomingCallEvent_Status.sTelephoneNumber.bSet(m_sIncomingEventInfo.TelephoneNumber);
   oIncomingCallEvent_Status.bCallWait = m_sIncomingEventInfo.CallWait;
   oIncomingCallEvent_Status.sFirstName.bSet(m_sIncomingEventInfo.FirstName);
   oIncomingCallEvent_Status.sLastName.bSet(m_sIncomingEventInfo.LastName);


   ETG_TRACE_USR1((" Incoming Call Event:: %u",ETG_ENUM(TR_INCOMINGCALLEVENT,oIncomingCallEvent_Status.e8IncomingCallEventEnum.enType)));
   ETG_TRACE_USR1(("oIncomingCallEvent_Status.u16CallInstance:%d",oIncomingCallEvent_Status.u16CallInstance));
   ETG_TRACE_USR1(("oIncomingCallEvent_Status.u8DeviceHandle:%d",oIncomingCallEvent_Status.u8DeviceHandle));
   ETG_TRACE_USR1(("oIncomingCallEvent_Status.sTelephoneNumber:%s", oIncomingCallEvent_Status.sTelephoneNumber.szGet()));
   ETG_TRACE_USR1(("oIncomingCallEvent_Status.bCallWait:%d",oIncomingCallEvent_Status.bCallWait));
   ETG_TRACE_USR1(("oIncomingCallEvent_Status.sLastName:%s", oIncomingCallEvent_Status.sLastName.szGet()));
   ETG_TRACE_USR1(("oIncomingCallEvent_Status.sFirstName:%s", oIncomingCallEvent_Status.sFirstName.szGet()));

   bSuccess =  fi_tclVisitorMessage(oIncomingCallEvent_Status, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oIncomingCallEvent_Status.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageCallStatusNotice
*
* DESCRIPTION: Prepares the status message regarding call status
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageCallStatusNotice(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;

   tU8 u8SMDeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(m_oCallInstances[0].m_u8BDAddressId);
   ETG_TRACE_USR3(("u8SMDeviceHandle: %u", u8SMDeviceHandle));

   if (FC_PHONE_DEVICEHANDLE_UNKNOWN == u8SMDeviceHandle)
   {
      u8SMDeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(m_oCallInstances[1].m_u8BDAddressId);
      ETG_TRACE_USR3(("u8SMDeviceHandle: %u", u8SMDeviceHandle));
   }

   if ((FC_PHONE_DEVICEHANDLE_UNKNOWN != u8SMDeviceHandle) && (m_u8ActiveDeviceHandle != u8SMDeviceHandle))
   {
      // This case will occur while accepting the incoming call in the second phone when there is an Active call in the Active device
      // This intermediate update is ignored since the call instances correspond to the Passive device
      // The CS will be updated after the successful Active Passive switching
      ETG_TRACE_USR3(("Intermediate update is avoided here"));
   }
   else
   {
      most_telfi_tclMsgCallStatusNoticeStatus    oCallStatusNotice;

      tU16 u16CallInstanceID = 0;

      tU8 u8DeviceHandletobeConsidered = m_u8ActiveDeviceHandle;
      ETG_TRACE_USR4(("u8DeviceHandletobeConsidered :%d", u8DeviceHandletobeConsidered));

      if(FC_PHONE_DEVICEHANDLE_UNKNOWN != u8DeviceHandletobeConsidered)
      {
         for(tU8 CIId = 0; CIId < CallStatusList.size(); CIId++)
         {
            if(CallStatusList[CIId].DeviceHandle == u8DeviceHandletobeConsidered)
            {
               u16CallInstanceID = CIId;
               break;
            }
         }
      }
      ETG_TRACE_USR1(("u16CallInstanceID :%d", u16CallInstanceID));

      for (tU8 u8Index = 0x00; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
      {
         most_fi_tcl_TelCallStatusNoticeStreamItem oStreamItem;

         oStreamItem.u16CallInstance = u16CallInstanceID++;
         oStreamItem.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)m_oCallInstances[u8Index].m_u16CallStatusNotice;
         oStreamItem.bUsingVehicleAudio = (m_oCallInstances[0].m_bUsingVehicleAudio ||  m_oCallInstances[1].m_bUsingVehicleAudio);
         oStreamItem.u8CallDurationHr = m_oCallInstances[u8Index].m_rCallDuration.u8Hours;
         oStreamItem.u8CallDurationMin= m_oCallInstances[u8Index].m_rCallDuration.u8Minutes;
         oStreamItem.u8CallDurationSec= m_oCallInstances[u8Index].m_rCallDuration.u8Seconds;

         oStreamItem.u8DeviceHandle = u8DeviceHandletobeConsidered;

         if( oCallStatusNotice.oCallStatusNoticeStream.oItems.size() < FC_PHONE_MAX_CALLINSTANCES)
         {
            oCallStatusNotice.oCallStatusNoticeStream.oItems.push_back(oStreamItem);
         }
         else
         {
            oCallStatusNotice.oCallStatusNoticeStream.oItems[u8Index]=oStreamItem;
         }

         ETG_TRACE_USR1(("bStatusMessageCallStatusNotice:oStreamItem.u16CallInstance:%d",oStreamItem.u16CallInstance));
         ETG_TRACE_USR1(("bStatusMessageCallStatusNotice:oStreamItem.bUsingVehicleAudio:%d",oStreamItem.bUsingVehicleAudio));
         ETG_TRACE_USR1(("bStatusMessageCallStatusNotice:oStreamItem.u8CallDurationHr:%d",oStreamItem.u8CallDurationHr));
         ETG_TRACE_USR1(("bStatusMessageCallStatusNotice:oStreamItem.u8CallDurationMin:%d",oStreamItem.u8CallDurationMin));
         ETG_TRACE_USR1(("bStatusMessageCallStatusNotice:oStreamItem.u8CallDurationSec:%d",oStreamItem.u8CallDurationSec));
         ETG_TRACE_USR1(("bStatusMessageCallStatusNotice:oStreamItem.u8DeviceHandle:%d",oStreamItem.u8DeviceHandle));
         ETG_TRACE_USR1(("bStatusMessageCallStatusNotice:oStreamItem.e8CallStatus.enType:: %d",oStreamItem.e8CallStatus.enType));
         ETG_TRACE_USR3(("bStatusMessageCallStatusNotice: Tel server vector %d ", (int)oCallStatusNotice.oCallStatusNoticeStream.oItems.size()));
      }
      ETG_TRACE_USR4(("StatusMessageFactory:MOST_TELFI_C_U16_CALLSTATUSNOTICE"));
      bSuccess = fi_tclVisitorMessage(oCallStatusNotice, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);

      oCallStatusNotice.vDestroy();
   }
   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageCallStatusNoticeExtended
*
* DESCRIPTION: Prepares the status message regarding call status extended for the addition of name parameter
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageCallStatusNoticeExtended(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR4(("Entered bStatusMessageCallStatusNoticeExtended"));

   tBool bSuccess = FALSE;
   most_telfi_tclMsgCallStatusNoticeExtendedStatus    oCallStatusNoticeExtended;

   tU8 u8TotCalls = 2;
   if (CallStatusList.size() != 0)
   {
      u8TotCalls = static_cast<tU8>(CallStatusList.size());
   }

   oCallStatusNoticeExtended.u8TotalCallInstances = u8TotCalls;
   ETG_TRACE_USR4(("oCallStatusNoticeExtended.u8TotalCallInstances = %d", u8TotCalls));

   tU8 u8SMDeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(m_oCallInstances[0].m_u8BDAddressId);
   ETG_TRACE_USR3(("u8SMDeviceHandle: %u", u8SMDeviceHandle));

   if (FC_PHONE_DEVICEHANDLE_UNKNOWN == u8SMDeviceHandle)
   {
      u8SMDeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(m_oCallInstances[1].m_u8BDAddressId);
      ETG_TRACE_USR3(("u8SMDeviceHandle: %u", u8SMDeviceHandle));
   }

   for (tU8 u8Index = 0x00; u8Index < u8TotCalls; u8Index++)
   {
      most_fi_tcl_TelCallStatusNoticeExtendedStreamItem oStreamItem;
      oStreamItem.u16CallInstance= u8Index;
      ETG_TRACE_USR4(("u8Index = %d", u8Index));

      if(CallStatusList.size() == 0)
      {
         oStreamItem.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE;
         oStreamItem.bUsingVehicleAudio = 0;
         oStreamItem.u8CallDurationHr = 0;
         oStreamItem.u8CallDurationMin= 0;
         oStreamItem.u8CallDurationSec= 0;
         oStreamItem.sName.bSet("");
         oStreamItem.sTelephoneNumber.bSet("");

         oStreamItem.u8DeviceHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN;
      }
      else
      {
         if (CallStatusList[u8Index].DeviceHandle == u8SMDeviceHandle)
         {
            tU8 ind = (u8Index%2);
            oStreamItem.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)m_oCallInstances[ind].m_u16CallStatusNotice;
            oStreamItem.bUsingVehicleAudio = (m_oCallInstances[0].m_bUsingVehicleAudio ||  m_oCallInstances[1].m_bUsingVehicleAudio);
            oStreamItem.u8CallDurationHr = m_oCallInstances[ind].m_rCallDuration.u8Hours;
            oStreamItem.u8CallDurationMin= m_oCallInstances[ind].m_rCallDuration.u8Minutes;
            oStreamItem.u8CallDurationSec= m_oCallInstances[ind].m_rCallDuration.u8Seconds;
            oStreamItem.sName.bSet(m_sDialInfo[ind].m_sDialName);
            if (m_sDialInfo[ind].m_sDialNumber)
            {
               oStreamItem.sTelephoneNumber.bSet(m_sDialInfo[ind].m_sDialNumber);
            }
            else
            {
               oStreamItem.sTelephoneNumber.bSet(CallStatusList[u8Index].CallStatus.achCallNumber);
            }
         }
         else
         {
            //FC_Phone usually updates FC_PHONE_CALLINSTANCE_CALLSTATUS_BUSY also as FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE
            if (FC_PHONE_CALLINSTANCE_CALLSTATUS_BUSY == CallStatusList[u8Index].CallStatus.u8CallState)
            {
               oStreamItem.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE;
            }
            else
            {
               oStreamItem.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)CallStatusList[u8Index].CallStatus.u8CallState;
            }
            oStreamItem.bUsingVehicleAudio = 0;
            oStreamItem.u8CallDurationHr = 0;
            oStreamItem.u8CallDurationMin= 0;
            oStreamItem.u8CallDurationSec= 0;
            oStreamItem.sName.bSet("");
            oStreamItem.sTelephoneNumber.bSet(CallStatusList[u8Index].CallStatus.achCallNumber);
         }
         oStreamItem.u8DeviceHandle = CallStatusList[u8Index].DeviceHandle;
      }

      if( oCallStatusNoticeExtended.oCallStatusNoticeExtendedStream.oItems.size() < u8TotCalls)
      {
         oCallStatusNoticeExtended.oCallStatusNoticeExtendedStream.oItems.push_back(oStreamItem);
      }
      else
      {
         oCallStatusNoticeExtended.oCallStatusNoticeExtendedStream.oItems[u8Index]=oStreamItem;
      }
      ETG_TRACE_USR1(("bStatusMessageCallStatusNoticeExtended:oStreamItem.bUsingVehicleAudio:%d",oStreamItem.bUsingVehicleAudio));
      ETG_TRACE_USR1(("bStatusMessageCallStatusNoticeExtended:oStreamItem.u8CallDurationHr:%d",oStreamItem.u8CallDurationHr));
      ETG_TRACE_USR1(("bStatusMessageCallStatusNoticeExtended:oStreamItem.u8CallDurationMin:%d",oStreamItem.u8CallDurationMin));
      ETG_TRACE_USR1(("bStatusMessageCallStatusNoticeExtended:oStreamItem.u8CallDurationSec:%d",oStreamItem.u8CallDurationSec));
      ETG_TRACE_USR1(("bStatusMessageCallStatusNoticeExtended:oStreamItem.u8DeviceHandle:%d",oStreamItem.u8DeviceHandle));
      ETG_TRACE_USR1(("bStatusMessageCallStatusNoticeExtended:oStreamItem.e8CallStatus.enType::%d",oStreamItem.e8CallStatus.enType));
      ETG_TRACE_USR1(("bStatusMessageCallStatusNoticeExtended:oStreamItem.sName::%s",oStreamItem.sName.szGet()));
      ETG_TRACE_USR1(("bStatusMessageCallStatusNoticeExtended:oStreamItem.sTelephoneNumber::%s",oStreamItem.sTelephoneNumber.szGet()));
      ETG_TRACE_USR3(("bStatusMessageCallStatusNoticeExtended: Tel server vector-%d ", (int)oCallStatusNoticeExtended.oCallStatusNoticeExtendedStream.oItems.size()));

   }
   ETG_TRACE_USR4(("StatusMessageFactory:MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED"));
   bSuccess = fi_tclVisitorMessage(oCallStatusNoticeExtended, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);

   oCallStatusNoticeExtended.vDestroy();

   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageCallStatusNotice
*
* DESCRIPTION: Prepares the status message regarding call status
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageActiveCallInHandset(amt_tclServiceData& roOutMsg, amt_tclServiceData* /*poInMsg*/)
{
   ETG_TRACE_USR3(("fc_phone_tclService_Telephone::bHandleActiveCallInHandset entered"));

   tBool bSuccess;
   tU8 u8NoofActiveCalls = 0;
   tU8 u8NoofHoldCalls = 0;
   most_telfi_tclMsgActiveCallinHandsetStatus oActiveCallInHandset_Status;
   for (tU8 u8Index = 0x00; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      if (FALSE == m_oCallInstances[u8Index].m_bUsingVehicleAudio)
      {
         if (FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE == m_oCallInstances[u8Index].m_u16CallStatusNotice)
         {
            u8NoofActiveCalls ++;
         }
         else if (FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD == m_oCallInstances[u8Index].m_u16CallStatusNotice)
         {
             u8NoofHoldCalls ++;
         }
      }
   }
   for (tU8 u8Index = 0x00; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      most_fi_tcl_TelActiveCallinHandsetStreamItem oActiveCallinHandsetStreamItem;

      //update the device handle
      oActiveCallinHandsetStreamItem.u8DeviceHandle = m_u8ActiveDeviceHandle;

      if (FALSE == m_oCallInstances[u8Index].m_bUsingVehicleAudio && u8Index == 0x00)
      {
         //update the HandSet Status
         if( u8NoofActiveCalls == 2 || FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE == m_oCallInstances[u8Index].m_u16CallStatusNotice)
         {
            oActiveCallinHandsetStreamItem.e8HandsetState.enType = most_fi_tcl_e8_TelHandsetState::FI_EN_E8MULTIPARTY_CALL;
            ETG_TRACE_USR4(("oActiveCallinHandsetStreamItem.e8HandsetState.enType::most_fi_tcl_e8_TelHandsetState::FI_EN_E8MULTIPARTY_CALL"));
         }
         else if(u8NoofActiveCalls >0 && u8NoofHoldCalls >0)
         {
            oActiveCallinHandsetStreamItem.e8HandsetState.enType = most_fi_tcl_e8_TelHandsetState::FI_EN_E8ACTIVEANDHOLD_CALL;
            ETG_TRACE_USR4(("oActiveCallinHandsetStreamItem.e8HandsetState.enType::most_fi_tcl_e8_TelHandsetState::FI_EN_E8ACTIVEANDHOLD_CALL"));
         }
         else if(u8NoofActiveCalls >0 && u8NoofHoldCalls == 0)
         {
            oActiveCallinHandsetStreamItem.e8HandsetState.enType = most_fi_tcl_e8_TelHandsetState::FI_EN_E8ACTIVE_CALL;
            ETG_TRACE_USR4(("oActiveCallinHandsetStreamItem.e8HandsetState.enType::most_fi_tcl_e8_TelHandsetState::FI_EN_E8ACTIVE_CALL"));
         }
         else if(u8NoofActiveCalls == 0 && u8NoofHoldCalls > 0)
         {
            oActiveCallinHandsetStreamItem.e8HandsetState.enType = most_fi_tcl_e8_TelHandsetState::FI_EN_E8HOLD_CALL;
            ETG_TRACE_USR4(("oActiveCallinHandsetStreamItem.e8HandsetState.enType::most_fi_tcl_e8_TelHandsetState::FI_EN_E8HOLD_CALL"));
         }
         else
         {
            /* Do nothing */
            ETG_TRACE_USR4((" Do Nothing "));
            oActiveCallinHandsetStreamItem.e8HandsetState.enType = most_fi_tcl_e8_TelHandsetState::FI_EN_E8NO_CALL;
            ETG_TRACE_USR4(("oActiveCallinHandsetStreamItem.e8HandsetState.enType::most_fi_tcl_e8_TelHandsetState::FI_EN_E8NO_CALL"));
         }
      }
      else
      {
         oActiveCallinHandsetStreamItem.e8HandsetState.enType = most_fi_tcl_e8_TelHandsetState::FI_EN_E8NO_CALL;
         ETG_TRACE_USR4(("oActiveCallinHandsetStreamItem.e8HandsetState.enType::most_fi_tcl_e8_TelHandsetState::FI_EN_E8NO_CALL"));
      }

      if (oActiveCallInHandset_Status.oActiveCallinHandsetStream.oItems.size() < FC_PHONE_MAX_CALLINSTANCES)
      {
         oActiveCallInHandset_Status.oActiveCallinHandsetStream.oItems.push_back(oActiveCallinHandsetStreamItem);
      }
      else
      {
         oActiveCallInHandset_Status.oActiveCallinHandsetStream.oItems[u8Index] = oActiveCallinHandsetStreamItem;
      }

      ETG_TRACE_USR3((" oActiveCallinHandsetStream size %d ", (int)oActiveCallInHandset_Status.oActiveCallinHandsetStream.oItems.size()));
   }

   bSuccess = fi_tclVisitorMessage(oActiveCallInHandset_Status, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oActiveCallInHandset_Status.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageCellDeviceName
*
* DESCRIPTION: Prepares the status message regarding cell device name
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageCellDeviceName(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR1(("Entered bStatusMessageCellDeviceName"));
   tBool bSuccess = FALSE;
   most_telfi_tclMsgCellDeviceNameStatus      oCellDeviceNameStatus;
   tChar sDeviceNameToHMI[FC_PHONE_DEVICENAME_MAX_LEN_BYTE] = { 0 };

   tU8 u8BDAddressIDforUpdate = u8GetBDAddressIDforStatusUpdate();

   // String Copy with max length restriction in chars and in bytes for destination, end of destination string will be UTF-8 aligned, \0 termination is included (within the max length bytes)
   (void) UTF8_szSaveStringNCopy( (tString) sDeviceNameToHMI, (tString) m_oPhoneData[u8BDAddressIDforUpdate].m_achCellDeviceName, FC_PHONE_DEVICENAME_MAX_LEN_BYTE, FC_PHONE_DEVICEYNAME_MAX_LEN_CHAR_COUNT);

   ETG_TRACE_USR3((" sDeviceNameToHMI : %s",sDeviceNameToHMI));

   oCellDeviceNameStatus.sDeviceName.bSet(sDeviceNameToHMI);
   oCellDeviceNameStatus.u8DeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8BDAddressIDforUpdate);

   ETG_TRACE_USR1(("oCellDeviceNameStatus.sDeviceName:%s", oCellDeviceNameStatus.sDeviceName.szGet()));
   ETG_TRACE_USR1(("oCellDeviceNameStatus.u8DeviceHandle:%d",oCellDeviceNameStatus.u8DeviceHandle));

   bSuccess = fi_tclVisitorMessage(oCellDeviceNameStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oCellDeviceNameStatus.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageCellNetworkStatus
*
* DESCRIPTION: Prepares the status message regarding cell Network status
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageCellNetworkStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR1(("Entered bStatusMessageCellNetworkStatus"));
   tBool bSuccess = FALSE;
   most_telfi_tclMsgCellNetworkStatusStatus   oCellNetworkStatusStatus;

   tU8 u8BDAddressIDforUpdate = u8GetBDAddressIDforStatusUpdate();

   oCellNetworkStatusStatus.u8DeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8BDAddressIDforUpdate);
   ETG_TRACE_USR1(("oCellNetworkStatusStatus.u8DeviceHandle:%d",oCellNetworkStatusStatus.u8DeviceHandle));

   if(FC_PHONE_PHONEDATA_NWSTATUS_AVAILABLE == m_oPhoneData[u8BDAddressIDforUpdate].m_u8NetworkStatus)
   {
      oCellNetworkStatusStatus.e8CellService.enType = most_fi_tcl_e8_TelCellService::FI_EN_E8SERVICE;
   }
   else
   {
      oCellNetworkStatusStatus.e8CellService.enType = most_fi_tcl_e8_TelCellService::FI_EN_E8NO_SERVICE;
   }
   ETG_TRACE_USR1((" Cell Service:: %u",ETG_ENUM(TR_SERVICE,oCellNetworkStatusStatus.e8CellService.enType)));

   if(FC_PHONE_PHONEDATA_ROAMINGSTATUS_TRUE == m_oPhoneData[u8BDAddressIDforUpdate].m_u8RoamingStatus)
   {
      oCellNetworkStatusStatus.bRoamingStatus = TRUE;
    }
   else
   {
      oCellNetworkStatusStatus.bRoamingStatus = FALSE;
   }
   ETG_TRACE_USR1(("oCellNetworkStatusStatus.bRoamingStatus:%d",oCellNetworkStatusStatus.bRoamingStatus));

   bSuccess = fi_tclVisitorMessage(oCellNetworkStatusStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oCellNetworkStatusStatus.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageCellStateOfChargeStatus
*
* DESCRIPTION: Prepares the status message regarding cell state of charge
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageCellStateOfChargeStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR1(("Entered bStatusMessageCellStateOfChargeStatus"));
   tBool bSuccess = FALSE;
   most_telfi_tclMsgCellStateofChargeStatus oCellStateOfChargeStatus;

   tU8 u8BDAddressIDforUpdate = u8GetBDAddressIDforStatusUpdate();

   oCellStateOfChargeStatus.u8StateofCharge = m_oPhoneData[u8BDAddressIDforUpdate].m_u8Charge;
   oCellStateOfChargeStatus.u8DeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8BDAddressIDforUpdate);

   ETG_TRACE_USR1(("oCellStateOfChargeStatus:%d", oCellStateOfChargeStatus.u8StateofCharge));
   ETG_TRACE_USR1(("oCellStateOfChargeStatus.u8DeviceHandle:%d", oCellStateOfChargeStatus.u8DeviceHandle));

   bSuccess = fi_tclVisitorMessage(oCellStateOfChargeStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oCellStateOfChargeStatus.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageCellSignalStrength
*
* DESCRIPTION: Prepares the status message regarding cell signal strength
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageCellSignalStrength(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR1(("Entered bStatusMessageCellSignalStrength"));
   tBool bSuccess = FALSE;
   most_telfi_tclMsgCellSignalStrengthStatus oCellSignalStrengthStatus;

   tU8 u8BDAddressIDforUpdate = u8GetBDAddressIDforStatusUpdate();

   oCellSignalStrengthStatus.u8SignalSrength = m_oPhoneData[u8BDAddressIDforUpdate].m_u8SignalStrength;
   ETG_TRACE_USR1(("oCellSignalStrengthStatus:%d", oCellSignalStrengthStatus.u8SignalSrength));
   oCellSignalStrengthStatus.u8DeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8BDAddressIDforUpdate);
   ETG_TRACE_USR1(("oCellSignalStrengthStatus.u8DeviceHandle:%d", oCellSignalStrengthStatus.u8DeviceHandle));
   bSuccess = fi_tclVisitorMessage(oCellSignalStrengthStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oCellSignalStrengthStatus.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageSLCStatus
*
* DESCRIPTION: Prepares the status message regarding slc status
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageSLCStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   return fi_tclVisitorMessage(m_oSLCStatusStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageSourceAvailable
*
* DESCRIPTION: Prepares the status message regarding source available
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageSourceAvailable(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   return fi_tclVisitorMessage(m_oSourceAvailableStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageNetworkOperator
*
* DESCRIPTION: Prepares the status message regarding network operator
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageNetworkOperator(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;

   most_telfi_tclMsgNetworkOperatorStatus oNetworkOperatorStatus;

   tU8 u8BDAddressIDforUpdate = u8GetBDAddressIDforStatusUpdate();

   oNetworkOperatorStatus.sNetworkOperator.bSet(m_oPhoneData[u8BDAddressIDforUpdate].m_NetworkOperatorName);
   ETG_TRACE_USR1(("Operator:%s", oNetworkOperatorStatus.sNetworkOperator.szGet()));
   bSuccess = fi_tclVisitorMessage(oNetworkOperatorStatus,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oNetworkOperatorStatus.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageVehicleAudio
*
* DESCRIPTION: Prepares the status message regarding vehicle audio
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageVehicleAudio(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;
   most_telfi_tclMsgVehicleAudioStatus oVehicleAudioStatus;
   oVehicleAudioStatus.bUsingVehicleAudio = (m_oCallInstances[0].m_bUsingVehicleAudio || m_oCallInstances[1].m_bUsingVehicleAudio);
   ETG_TRACE_USR1(("VehicleAudioStatus:%d", oVehicleAudioStatus.bUsingVehicleAudio));
   bSuccess = fi_tclVisitorMessage(oVehicleAudioStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oVehicleAudioStatus.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageMicrophoneMuteState
*
* DESCRIPTION: Prepares the status message regarding microphone mute state
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageMicrophoneMuteState(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;
   most_telfi_tclMsgMicrophoneMuteStateStatus oMicrophoneMuteStateStatus;
   oMicrophoneMuteStateStatus.bEnable = m_oCallInstances[m_u8CurrentCallIndex].m_bMicroPhoneMuteStatus;
   ETG_TRACE_USR1(("oMicrophoneMuteStateStatus:%d", oMicrophoneMuteStateStatus.bEnable));
   bSuccess = fi_tclVisitorMessage(oMicrophoneMuteStateStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oMicrophoneMuteStateStatus.vDestroy();
   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageHFService
*
* DESCRIPTION: Prepares the status message regarding Handsfree service
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageHFService(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;
   most_telfi_tclMsgHFServiceStatus oHFServiceStatus;

   tU8 u8BDAddressIDforUpdate = u8GetBDAddressIDforStatusUpdate();

   oHFServiceStatus.bCallWaitAvail = m_oPhoneData[u8BDAddressIDforUpdate].m_bCallWaitingSupport;
   ETG_TRACE_USR1(("oHFServiceStatus::bCallWaitAvail:%d", oHFServiceStatus.bCallWaitAvail));
   oHFServiceStatus.bThreeWayCallAvail = m_oPhoneData[u8BDAddressIDforUpdate].m_bThreeWayCallSupport;
   ETG_TRACE_USR1(("oHFServiceStatus::bThreeWayCallAvail:%d", oHFServiceStatus.bThreeWayCallAvail));
   bSuccess = fi_tclVisitorMessage(oHFServiceStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oHFServiceStatus.vDestroy();

   return bSuccess;
}


/*******************************************************************************
*
* FUNCTION:    bStatusMessageBTVoiceRecognition
*
* DESCRIPTION: Prepares the status message regarding bt voice recognition
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageBTVoiceRecognition(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR3(( "m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecSupported= '%d'", m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecSupported));
   ETG_TRACE_USR3(( "m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecActive= '%d'", m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecActive));
   return fi_tclVisitorMessage(m_oBTDeviceVoiceRecStatus,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
}

// CMG3G-7861 - Siri availability status
/***********************************************************************************
*
* FUNCTION:    bStatusMessageBTVoiceRecognitionExtended
*
* DESCRIPTION: Prepares the status message regarding bt voice recognition Extended
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
************************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageBTVoiceRecognitionExtended(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR3(( "m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported= '%d'", m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported));
   ETG_TRACE_USR3(( "m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive= '%d'", m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive));
   ETG_TRACE_USR3(("m_oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState: '%u'", (m_oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState.enType)));
   ETG_TRACE_USR3(( "m_oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle= '%d'", m_oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle));

   return fi_tclVisitorMessage(m_oBTDeviceVoiceRecExtendedStatus,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
}
/*******************************************************************************
*
* FUNCTION:    bStatusMessageRingtoneMuteState
*
* DESCRIPTION: Prepares the status message regarding ringtone mutestate
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageRingtoneMuteState(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;
   most_telfi_tclMsgRingtoneMuteStateStatus oRingtoneMuteStateStatus;
   oRingtoneMuteStateStatus.bEnable = m_oCallInstances[m_u8CurrentCallIndex].m_bRingtoneMuteStatus;
   ETG_TRACE_USR1(("oRingtoneMuteStateStatus.bEnable:%d", oRingtoneMuteStateStatus.bEnable));
   bSuccess = fi_tclVisitorMessage(oRingtoneMuteStateStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oRingtoneMuteStateStatus.vDestroy();
   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageWaitingModeFilePath
*
* DESCRIPTION: Prepares the status message regarding waiting mode filepath
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageWaitingModeFilePath(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   return fi_tclVisitorMessage(m_oWaitingModeFilePathStatus,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageActivePhoneDevice
*
* DESCRIPTION: Prepares the status message regarding Active Phone device
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageActivePhoneDevice(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;
   most_telfi_tclMsgActivePhoneDeviceStatus oActivePhoneDeviceStatus;
   oActivePhoneDeviceStatus.u8DeviceHandle = m_u8ActiveDeviceHandle;
   ETG_TRACE_USR1(("oActivePhoneDeviceStatus.u8DeviceHandle :%d", oActivePhoneDeviceStatus.u8DeviceHandle));
   bSuccess = fi_tclVisitorMessage(oActivePhoneDeviceStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oActivePhoneDeviceStatus.vDestroy();
   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageOutgoingCallFailedEvent
*
* DESCRIPTION: Prepares the status message regarding OutgoingCall Failed Event
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageOutgoingCallFailedEvent(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR3(("bStatusMessageOutgoingCallFailedEvent entered"));
   tBool bSuccess = FALSE;

   most_telfi_tclMsgOutgoingCallFailedEventStatus oOutgoingCallFailedEventStatus;
   oOutgoingCallFailedEventStatus.oOutgoingFailed.u8DeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(m_u8CurrentCallIndex);
   oOutgoingCallFailedEventStatus.oOutgoingFailed.sTelephoneNumber.bSet((const char*) m_oCallInstances[m_u8CurrentCallIndex].m_strCellNumber);
   if(m_sDialInfo[m_u8CurrentCallIndex].m_sDialName)
   {
      oOutgoingCallFailedEventStatus.oOutgoingFailed.sName.bSet((const char*)m_sDialInfo[m_u8CurrentCallIndex].m_sDialName);
   }
   else
   {
      oOutgoingCallFailedEventStatus.oOutgoingFailed.sName.bSet((const char*)"");
   }
   oOutgoingCallFailedEventStatus.oOutgoingFailed.sOutgoingCallFailedReason.bSet((const char*)m_oCallInstances[m_u8CurrentCallIndex].m_strDialCallFailedReason);


   ETG_TRACE_USR1(("oOutgoingCallFailedEventStatus.u8DeviceHandle :%d", oOutgoingCallFailedEventStatus.oOutgoingFailed.u8DeviceHandle));
   ETG_TRACE_USR1(("oOutgoingCallFailedEventStatus.sTelephoneNumber.szValue :%s", oOutgoingCallFailedEventStatus.oOutgoingFailed.sTelephoneNumber.szValue));
   ETG_TRACE_USR1(("oOutgoingCallFailedEventStatus.sName.szValue :%s", oOutgoingCallFailedEventStatus.oOutgoingFailed.sName.szValue));
   ETG_TRACE_USR1(("oOutgoingCallFailedEventStatus.sOutgoingCallFailedReason.szValue :%s", oOutgoingCallFailedEventStatus.oOutgoingFailed.sOutgoingCallFailedReason.szValue));

   bSuccess = fi_tclVisitorMessage(oOutgoingCallFailedEventStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);

   oOutgoingCallFailedEventStatus.vDestroy();
   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageIncomingCallRejectEvent
*
* DESCRIPTION: Prepares the status message regarding IncomingCall Reject Event
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageIncomingCallRejectEvent(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR3(("bStatusMessageIncomingCallRejectEvent entered"));
   tBool bSuccess = FALSE;

   most_telfi_tclMsgIncomingCallRejectedEventStatus oCallRejectedEventStatus;
   oCallRejectedEventStatus.oIncomingRejected.u8DeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(m_u8CurrentCallIndex);
   oCallRejectedEventStatus.oIncomingRejected.sTelephoneNumber.bSet((const char*) m_oCallInstances[m_u8CurrentCallIndex].m_strCellNumber);
   oCallRejectedEventStatus.oIncomingRejected.sIncomingCallRejectedReason.bSet((const char*)m_oCallInstances[m_u8CurrentCallIndex].m_strIncomingCallRejectReason);

   ETG_TRACE_USR1(("oCallRejectedEventStatus.oIncomingRejected.u8DeviceHandle :%d", oCallRejectedEventStatus.oIncomingRejected.u8DeviceHandle));
   ETG_TRACE_USR1(("oCallRejectedEventStatus.oIncomingRejected.sTelephoneNumber.szValue :%s", oCallRejectedEventStatus.oIncomingRejected.sTelephoneNumber.szValue));
   ETG_TRACE_USR1(("oCallRejectedEventStatus.oIncomingRejected.sIncomingCallRejectedReason.szValue :%s", oCallRejectedEventStatus.oIncomingRejected.sIncomingCallRejectedReason.szValue));

   bSuccess = fi_tclVisitorMessage(oCallRejectedEventStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oCallRejectedEventStatus.vDestroy();
   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageWaitingModeStatus
*
* DESCRIPTION: Prepares the status message regarding Waiting mode status
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageWaitingModeStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR1(("m_oWaitingModeStateStatus.e8WaitingModeState.enType :%d", m_oWaitingModeStateStatus.e8WaitingModeState.enType));
   return fi_tclVisitorMessage(m_oWaitingModeStateStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
}

/*******************************************************************************
*
* FUNCTION:    bStatusMessageExternalVRStatus
*
* DESCRIPTION: Prepares the status message regarding Waiting mode status
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageExternalVRStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   ETG_TRACE_USR1(("m_oExternalVRStateStatus.sBdAddress :%s", m_oExternalVRStateStatus.sBdAddress.szValue));
   ETG_TRACE_USR1(("m_oExternalVRStateStatus.e8ExternalVRState :%d", m_oExternalVRStateStatus.e8ExternalVRState.enType));
   ETG_TRACE_USR1(("m_oExternalVRStateStatus.e8VRErrorCode :%d", m_oExternalVRStateStatus.e8VRErrorCode.enType));
   return fi_tclVisitorMessage(m_oExternalVRStateStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
}

#ifdef PHONE_HFENHANCED
/*******************************************************************************
*
* FUNCTION:    bStatusMessageHFEnhancedControl
*
* DESCRIPTION: Prepares the status message regarding enhanced control
*
* PARAMETER:   roOutMsg       service to handover reply message
*              poInMsg        incoming parameters
*
* RETURNVALUE: Boolean indicating handover was a success or not
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bStatusMessageHFEnhancedControl(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
{
   tBool bSuccess = FALSE;
   most_telfi_tclMsgHandsFreeEnhancedCallControlStatus oEnhancedCallControlStatus;
   most_fi_tcl_TelHFEnhancedCallControlStreamItem oActivePhoneItem;

   tU8 u8BDAddressIDforUpdate = u8GetBDAddressIDforStatusUpdate();

   oActivePhoneItem.u8DeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8BDAddressIDforUpdate);
   ETG_TRACE_USR1(("oActivePhoneItem.u8DeviceHandle:%d",oActivePhoneItem.u8DeviceHandle));
   oActivePhoneItem.bEndTargetCall = (bool)(m_oPhoneData[u8BDAddressIDforUpdate].m_u8AgThreeWayCallSuppFeatures & FC_PHONE_ENHANCECALLCTRL_ENDTARGETCALL_MASK);
   ETG_TRACE_USR1(("oActivePhoneItem.bEndTargetCall:%d",oActivePhoneItem.bEndTargetCall));
   oActivePhoneItem.bSplitMergedCalls = (bool)(m_oPhoneData[u8BDAddressIDforUpdate].m_u8AgThreeWayCallSuppFeatures & FC_PHONE_ENHANCECALLCTRL_SPLITMERGECALLS_MASK);
   ETG_TRACE_USR1(("oActivePhoneItem.bSplitMergedCalls:%d",oActivePhoneItem.bSplitMergedCalls));

   //Currently it supports only one active phone.
   oEnhancedCallControlStatus.oHFEnhancedCallControlStream.oItems.clear();
   oEnhancedCallControlStatus.oHFEnhancedCallControlStream.oItems.push_back(oActivePhoneItem);

   bSuccess = fi_tclVisitorMessage(oEnhancedCallControlStatus, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION).bHandOver(&roOutMsg);
   oEnhancedCallControlStatus.vDestroy();
   return bSuccess;
}
#endif

/*******************************************************************************
*
* FUNCTION: b_get_ActiveDevice_AddressId_FromBT
*
* DESCRIPTION: get ActiveDevice Adderss From BTclient
*              and convert it to desire format
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::b_get_ActiveDevice_AddressId_FromBT(tU8 *pu8_BDAddressId)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::b_get_ActiveDevice_AddressId_FromBT entered "));
   tU8 u8_btActiveBDAddressId= m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
   tBool b_SuccessFlag = FALSE ;

   if(INVALID_BDADDRESS_ID != u8_btActiveBDAddressId)
   {
      *pu8_BDAddressId = u8_btActiveBDAddressId;
      b_SuccessFlag = TRUE ;
   }
   else
   {
      ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
   }

   return b_SuccessFlag;
}

/*******************************************************************************
*
* FUNCTION: b_get_ConnectedDevice_AddressId_FromBT
*
* DESCRIPTION: get ActiveDevice Adderss From BTclient
*              and convert it to desire format
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::b_get_ConnectedDevice_AddressId_FromBT(tU8 *pu8_BDAddressId, const tChar *pDeviceAddress)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::b_get_ConnectedDevice_AddressId_FromBT entered "));
   tU8 u8BDId= fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(pDeviceAddress);
   tBool b_SuccessFlag = FALSE ;

   if(INVALID_BDADDRESS_ID != u8BDId)
   {
      *pu8_BDAddressId = u8BDId;
      b_SuccessFlag = TRUE;
   }
   else
   {
      ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
   }

   return b_SuccessFlag;
}

/*******************************************************************************
*
* FUNCTION: copy_from_loopback_queue
*
* DESCRIPTION: copy data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
void fc_phone_tclService_Telephone::copy_from_loopback_queue(T_CallData* prCallData, LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_from_loopback_queue  entered "));
   vStringCopy(prCallData->achCellDeviceName, p_LB_data->Call_data.achCellDeviceName, PHONE_CELLDEVICENAME_LENGTH);
   vStringCopy(prCallData->BD_Address,p_LB_data->Call_data.BD_Address, PHONE_BDADRESS_LENGTH);

   //rCallStatus
   prCallData->rCallStatus.u8CallHandle =p_LB_data->Call_data.rCallStatus.u8CallHandle;
   prCallData->rCallStatus.u8CallTotal =p_LB_data->Call_data.rCallStatus.u8CallTotal;
   prCallData->rCallStatus.u8CallDirection =p_LB_data->Call_data.rCallStatus.u8CallDirection;
   prCallData->rCallStatus.u8CallState  =p_LB_data->Call_data.rCallStatus.u8CallState;
   prCallData->rCallStatus.u8CallMode =p_LB_data->Call_data.rCallStatus.u8CallMode;
   vStringCopy(prCallData->rCallStatus.achCallNumber, p_LB_data->Call_data.rCallStatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);
   prCallData->rCallStatus.u8CallType =p_LB_data->Call_data.rCallStatus.u8CallType;

   //MR_Data
   prCallData->MR_Data.Status =p_LB_data->Call_data.MR_Data.Status;
   prCallData->MR_Data.Code =  p_LB_data->Call_data.MR_Data.Code;
   prCallData->MR_Data.State =  p_LB_data->Call_data.MR_Data.State;
   vStringCopy(prCallData->MR_Data.device_name,p_LB_data->Call_data.MR_Data.device_name, PHONE_CELLDEVICENAME_LENGTH);
   vStringCopy(prCallData->MR_Data.voiceMailNumber,p_LB_data->Call_data.MR_Data.voiceMailNumber, PHONE_VOICEMAILNUMBER_LENGTH);
   vStringCopy(prCallData->MR_Data.TelephoneNumber,p_LB_data->Call_data.MR_Data.TelephoneNumber, PHONE_TELEPHONENUMBER_LENGTH);
   vStringCopy(prCallData->MR_Data.SubcriberNumber,p_LB_data->Call_data.MR_Data.SubcriberNumber, PHONE_TELEPHONENUMBER_LENGTH);

   prCallData->Service = p_LB_data->Call_data.Service;
   prCallData->Status_ConnComp = p_LB_data->Call_data.Status_ConnComp;
   prCallData->ConnLosstype = p_LB_data->Call_data.ConnLosstype;
   prCallData->u8RingTone =p_LB_data->Call_data.u8RingTone;
   prCallData->b_SLC_status = p_LB_data->Call_data.b_SLC_status;
}

/*******************************************************************************
*
* FUNCTION: copy_Consolidated_CallData_from_loopback_queue
*
* DESCRIPTION: copy consolidated call data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
void fc_phone_tclService_Telephone::copy_Consolidated_CallData_from_loopback_queue(T_CallData prCallData[], LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_Consolidated_CallData_from_loopback_queue  entered "));

   for(tU8 ind = 0; ind < FC_PHONE_MAX_CALLINSTANCES; ++ind)
   {
      vStringCopy(prCallData[ind].achCellDeviceName, p_LB_data->Call_data_Consolidated[ind].achCellDeviceName, PHONE_CELLDEVICENAME_LENGTH);
      vStringCopy(prCallData[ind].BD_Address,p_LB_data->Call_data_Consolidated[ind].BD_Address, PHONE_BDADRESS_LENGTH);

      ETG_TRACE_USR4(("prCallData[%d].BD_Address :%s",ind,prCallData[ind].BD_Address));

      prCallData[ind].rCallStatus.u8CallHandle =p_LB_data->Call_data_Consolidated[ind].rCallStatus.u8CallHandle;
      prCallData[ind].rCallStatus.u8CallTotal =p_LB_data->Call_data_Consolidated[ind].rCallStatus.u8CallTotal;
      prCallData[ind].rCallStatus.u8CallDirection =p_LB_data->Call_data_Consolidated[ind].rCallStatus.u8CallDirection;
      prCallData[ind].rCallStatus.u8CallState  =p_LB_data->Call_data_Consolidated[ind].rCallStatus.u8CallState;
      prCallData[ind].rCallStatus.u8CallMode =p_LB_data->Call_data_Consolidated[ind].rCallStatus.u8CallMode;
      vStringCopy(prCallData[ind].rCallStatus.achCallNumber, p_LB_data->Call_data_Consolidated[ind].rCallStatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);
      prCallData[ind].rCallStatus.u8CallType =p_LB_data->Call_data_Consolidated[ind].rCallStatus.u8CallType;
   }
}

/*******************************************************************************
*
* FUNCTION: copy_PhoneStatus_from_loopback_queue
*
* DESCRIPTION:copy PhoneStatus data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::copy_PhoneStatus_from_loopback_queue( T_PhoneStatus* pPhoneStatus, LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_PhoneStatus_from_loopback_queue  entered"));
   pPhoneStatus->u8PhoneService = p_LB_data->phone_status.u8PhoneService;
   pPhoneStatus->u8PhoneSignal   = p_LB_data->phone_status.u8PhoneSignal ;
   pPhoneStatus->u8PhoneRoaming = p_LB_data->phone_status.u8PhoneRoaming;
   pPhoneStatus->u8PhoneBattery = p_LB_data->phone_status.u8PhoneBattery;
   pPhoneStatus->u8PhoneAudiopath =p_LB_data->phone_status.u8PhoneAudiopath ;
   pPhoneStatus->u8AgThreeWayCallSuppFeatures = p_LB_data->phone_status.u8AgThreeWayCallSuppFeatures;
   pPhoneStatus->u32PhoneSuppFeature =p_LB_data->phone_status.u32PhoneSuppFeature;

   memset(pPhoneStatus->BD_Address, '\0', PHONE_CONNMAP_DEV_ADDRESS);
   memset(pPhoneStatus->achPhoneOperatorName, '\0', PHONE_NWOPERATORNAME_LENGTH);

   vStringCopy((char *) pPhoneStatus->achPhoneOperatorName,(const char*)p_LB_data->phone_status.achPhoneOperatorName, PHONE_NWOPERATORNAME_LENGTH);
   vStringCopy((char *) pPhoneStatus->BD_Address,(const char*)p_LB_data->phone_status.BD_Address, PHONE_BDADRESS_LENGTH);
}


/*******************************************************************************
*
* FUNCTION: copy_SCOConnectionStatus_from_loopback_queue
*
* DESCRIPTION:copy SCO Connection Status data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::copy_SCOConnectionStatus_from_loopback_queue( T_SCOConnectionStatus*  pSCOConnectionStatus,LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_SCOConnectionStatus_from_loopback_queue entered"));
   pSCOConnectionStatus->SCOConnectionStatus = p_LB_data->SCOConnectionStatus.SCOConnectionStatus;
   pSCOConnectionStatus->Codec = p_LB_data->SCOConnectionStatus.Codec;
   vStringCopy((char *)pSCOConnectionStatus->BD_Address, (const char*)p_LB_data->SCOConnectionStatus.BD_Address,PHONE_BDADRESS_LENGTH);
}

/*******************************************************************************
*
* FUNCTION: copy_VoiceRecInd_from_loopback_queue
*
* DESCRIPTION: copy Voice Recognition indication data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::copy_VoiceRecInd_from_loopback_queue( T_VoiceRecInd* pVoiceRecInd, LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_VoiceRecInd_from_loopback_queue entered"));
   pVoiceRecInd->Activation = p_LB_data->VoiceRecInd.Activation;
   vStringCopy((char *)pVoiceRecInd->BD_Address, (const char*)p_LB_data->VoiceRecInd.BD_Address,PHONE_BDADRESS_LENGTH);
}


#ifdef PHONE_SIRI
/*******************************************************************************
*
* FUNCTION: copy_SiriNrInd_from_loopback_queue
*
* DESCRIPTION: copy Noise Reduction indication data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::copy_SiriNrInd_from_loopback_queue( T_SiriNRStatusInd* pNoiseReducInd, LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_NoiseReducInd_from_loopback_queue entered"));
   pNoiseReducInd->m_u8SiriNrStatus = p_LB_data->NRStatusInd.m_u8SiriNrStatus;
   vStringCopy((char *) pNoiseReducInd->BD_Address,(const char*)p_LB_data->NRStatusInd.BD_Address, PHONE_BDADRESS_LENGTH);
}

/*******************************************************************************
*
* FUNCTION: copy_SiriStatus_from_loopback_queue
*
* DESCRIPTION:copy SiriStatus data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::copy_SiriStatus_from_loopback_queue( T_SiriStatus* pSiriStatus, LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_SiriStatus_from_loopback_queue  entered"));
   pSiriStatus->m_u8SiriStatus = p_LB_data->siri_status.m_u8SiriStatus;
   vStringCopy((char *) pSiriStatus->BD_Address,(const char*)p_LB_data->siri_status.BD_Address, PHONE_BDADRESS_LENGTH);
}
#endif //SIRI_FEATURE_ENABLE

/*******************************************************************************
*
* FUNCTION: copy_method_data_from_loopback_queue
*
* DESCRIPTION: opy Method Result data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::copy_method_data_from_loopback_queue(Async_method_data_t* p_MR_Data, LoopBackData *p_LB_data)
{
   //MR_Data
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_method_data_from_loopback_queue entered"));
   p_MR_Data->Status = p_LB_data->Call_data.MR_Data.Status;
   p_MR_Data->Code = p_LB_data->Call_data.MR_Data.Code;
   p_MR_Data->State = p_LB_data->Call_data.MR_Data.State;
   vStringCopy(p_MR_Data->device_name, p_LB_data->Call_data.MR_Data.device_name, PHONE_CELLDEVICENAME_LENGTH);
   vStringCopy(p_MR_Data->voiceMailNumber, p_LB_data->Call_data.MR_Data.voiceMailNumber, PHONE_VOICEMAILNUMBER_LENGTH);
   vStringCopy(p_MR_Data->TelephoneNumber, p_LB_data->Call_data.MR_Data.TelephoneNumber, PHONE_TELEPHONENUMBER_LENGTH);
   vStringCopy(p_MR_Data->SubcriberNumber, p_LB_data->Call_data.MR_Data.SubcriberNumber, PHONE_TELEPHONENUMBER_LENGTH);
}
/*******************************************************************************
*
* FUNCTION: copy_WQJob_from_loopback_queue
*
* DESCRIPTION: copy WQ job data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
// FIX GMMY16-7888 	[CT][MY16 D1BL Telephone] HMI reset observed when calls were swapped and then end call selected immediately.
tVoid fc_phone_tclService_Telephone::copy_WQJob_from_loopback_queue( Job* job, LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_WQJob_from_loopback_queue entered"));
   *job = p_LB_data->job;

}
//End of fix


/*******************************************************************************
*
* FUNCTION: push_back_LB_data
*
* DESCRIPTION: push data to the loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::push_back_LB_data(LoopBackData &LBData)
{
   g_mutex_lock(&p_mutex_loopback);

   ETG_TRACE_USR3((" INSIDE push_back_LB_data  "));
   LB_data_queue.push_back(LBData);
   g_mutex_unlock(&p_mutex_loopback);
}

/*******************************************************************************
*
* FUNCTION: erase_LB_data
*
* DESCRIPTION: erase data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::erase_LB_data(tU16 position)
{
   ETG_TRACE_USR4((" pop_back_LB_data::fc_phone_tclService_Telephone::erase_LB_data entered "));
   g_mutex_lock(&p_mutex_loopback);

   ETG_TRACE_USR3(("\n INSIDE pop_back_LB_data  "));
   LB_data_queue.erase(LB_data_queue.begin()+position);
   g_mutex_unlock(&p_mutex_loopback);
}


/*******************************************************************************
*
* FUNCTION: get_data_position
*
* DESCRIPTION: get the data position in loopback data queue , corresponding to CB
*              function id
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tU16 fc_phone_tclService_Telephone::get_data_position(tU32 CB_functionID)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::get_data_position entered "));
   tU16 index1;
   tU16 position = FC_PHONE_LOOPBACK_VECTOR_INVALID_POSITION ;

   g_mutex_lock(&p_mutex_loopback);
   for(index1=0;index1< LB_data_queue.size();index1++)
   {
      if(CB_functionID == LB_data_queue[index1].u16FunctionID )
      {
         position = index1 ;
         break;
      }
   }
   g_mutex_unlock(&p_mutex_loopback);
   ETG_TRACE_USR3((" found position : %d",position));

   return position;
}


/*******************************************************************************
 *
 * FUNCTION: vOnLoopbackService
 *
 * DESCRIPTION: This function is called from the vOnLoopback() function of this
 *              CCA application if a message from one of this applications
 *              services is received (so called self or loopback messages).
 *
 * PARAMETER: [IN] u16ServiceID = Identifier of the service from where the
 *                                message was sent.
 *            [IN] poMessage = Message object.
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid fc_phone_tclService_Telephone::vOnLoopbackService(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vOnLoopbackService entered "));

   gm_tclU32Message oVisitorMsg(poMessage);
   tU16 position ;

   tU16 u16FunctionId = poMessage->u16GetFunctionID();
   tU32 u32MsgContent = oVisitorMsg.u32GetDWord();

   ETG_TRACE_USR1((" u16FunctionId:: 0x%x u32MsgContent :: %d LB_data_queue  size::%d",u16FunctionId,u32MsgContent,(int)LB_data_queue.size()));

   if ( (u32MsgContent < FC_PHONE_CB_FID_BASE+1) || (u32MsgContent > FC_PHONE_CB_FID_END-1) )
   {
      ETG_TRACE_USR3((" ERROR :: u32MsgContent is out of range"));
      return;
   }

   position = get_data_position(u32MsgContent);
   if(FC_PHONE_LOOPBACK_VECTOR_INVALID_POSITION == position)
   {
      ETG_TRACE_ERR((" CALLBACK FUNCTION_ID DATA IS NOT PRESENT IN VECTOR QUEUE  "));
      return;
   }

   switch(u16FunctionId)
   {
      case TELEPHONE_C_U16_IFID_EXAMPLE_1 :
      {
         switch(u32MsgContent)
         {
            case FC_PHONE_CB_FID_REJECTCALL :
            {
               ETG_TRACE_USR3(("  interface loop back FC_PHONE_CB_FID_REJECTCALL"));

               //copy data from loopback
               FcPhone_processEventArg tProcessEventArg;
               tProcessEventArg.bIsItDbusAck = TRUE;
               tProcessEventArg.u16DbusAckType = FC_PHONE_CB_FID_REJECTCALL;
               tProcessEventArg.Code = LB_data_queue[position].Call_data.MR_Data.Code;
               tProcessEventArg.State = LB_data_queue[position].Call_data.MR_Data.State;
               tProcessEventArg.Status = LB_data_queue[position].Call_data.MR_Data.Status;

               //erase loop back data
               erase_LB_data(position);
               vAcknowledgeMethodStartForCallInstance(&tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_INCOMINGCALL :
            {
               T_CallData  CallData ;
               //tBool b_getActiveDeviceAddress = FALSE;
               //tU8 u8_ActiveBDAddressId;
               //tBool b_CurrObjState = FALSE;
               ETG_TRACE_USR3(("  interface loop back FC_PHONE_CB_FID_INCOMINGCALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_INCOMINGCALL;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;
               copy_from_loopback_queue(&CallData,&LB_data_queue[position]);
               erase_LB_data( position);

               /*
               b_getActiveDeviceAddress = b_get_ActiveDevice_AddressId_FromBT(&u8_ActiveBDAddressId);
               if(FALSE == b_getActiveDeviceAddress)
               {
                  ETG_TRACE_ERR(("  ACTIVE DEVICE ADDRESS IS NULL"));
                  return ;
               }
               */

               //TODO how to handle if call is there in source changed device and if another incoming call comes in other device which is temporarily not an outgoing source device
               //Literally work out.Need to cross check
               //if(u8_ActiveBDAddressId == fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(CallData.BD_Address))
               //{
               tU8 devicehandle = u8GetDeviceHandlefromDevAddrinCSList(CallData.BD_Address);
               if(devicehandle != FC_PHONE_DEVICEHANDLE_UNKNOWN)
               {
                  if (m_u8ActiveDeviceHandle == devicehandle)
                  {
                     ETG_TRACE_USR3((" INCOMING CALL BELONGS TO ACTIVE DEVICE"));
                     ETG_TRACE_USR2((" updating ringtone device addrsID"));
                     tU8 u8ActiveDeviceBDAddressID = fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(CallData.BD_Address);
                     if(u8ActiveDeviceBDAddressID!= INVALID_BDADDRESS_ID)
                     {
                        m_pobtTelClient->vUpdateRingToneDeviceAddrId(u8ActiveDeviceBDAddressID);
                        ETG_TRACE_USR2((" m_oPhoneData[%d].m_bInBandRingtoneSupport = %d",u8ActiveDeviceBDAddressID, m_oPhoneData[u8ActiveDeviceBDAddressID].m_bInBandRingtoneSupport));
                        m_pobtTelClient->vUpdateInBandRingTone(m_oPhoneData[u8ActiveDeviceBDAddressID].m_bInBandRingtoneSupport);
                     }
                     else
                     {
                        ETG_TRACE_USR2((" Invalid BTDeviceID"));
                     }

                     /* End Voice based VR */
                     if (m_poSMVRSession->u16GetVRSessionState() == (tU16)STATE_ACTIVE)
                     {
                        ETG_TRACE_USR4((" Getting an incoming Bluetooth call- End Siri"));

                        m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_CCX_BT_DEVICE_VR_INACTIVE);
                     }

                     //erase loop back data
                     m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_INCOMINGCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST_CONSUMED,&tProcessEventArg);
                  }
                  else
                  {
                     ETG_TRACE_USR3(("   INCOMING CALL BELONGS TO PASSIVE DEVICE  "));

                     /*
                     for(tU8 u8Index = 0x00; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
                     {

                        if(m_oCallInstances[u8Index].m_u8ObjectState == FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE)
                        {
                           ETG_TRACE_USR3(("!!!!!!!!!! CALL IS ALREADY ACTIVE !!!!!!!!!!!!!!!"));
                           b_CurrObjState = TRUE;
                           break;
                        }
                     }

                     if(b_CurrObjState == TRUE)
                        break;

                     m_pobtTelClient->SetOutGoingSourceChangeStatus(TRUE);
                     */
                  }
               }

               else
               {
                  ETG_TRACE_USR3(("No device connected with the given address: %s", CallData.BD_Address));
                  return;
               }
            }
            break;

            case FC_PHONE_CB_FID_ACCEPTCALL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_ACCEPTCALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_ACCEPTCALL;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_ACCEPTCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_CALLSTATUS :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_CALLSTATUS"));
               T_CallData  CallData ;

               //get data here from vector if required,
               //and than erase from the vector corresponding position data
               copy_from_loopback_queue(&CallData,&LB_data_queue[position]);
               erase_LB_data( position);

               tU8 devicehandle = u8GetDeviceHandlefromDevAddrinCSList(CallData.BD_Address);
               ETG_TRACE_USR3(("devicehandle: %d m_u8ActiveDeviceHandle: %d", devicehandle,m_u8ActiveDeviceHandle));

               if(devicehandle != FC_PHONE_DEVICEHANDLE_UNKNOWN)
               {
                  ETG_TRACE_USR3((" The device with Address: %s is connected", CallData.BD_Address));
                  ETG_TRACE_USR3((" CallData.rCallStatus.u8CallTotal :%d", CallData.rCallStatus.u8CallTotal));
                  ETG_TRACE_USR3((" CallData.rCallStatus.u8CallState :%d", CallData.rCallStatus.u8CallState));

                  // Fix for NCG3D-63471
                  // Before posting Dial_MR/ME, if IDLE CS is received and handled, it destroys the Dial_SM.
                  // And thus Dial_MR/ME will be missed out posting to the client.
                  // Hence, while waiting for Dial response from stack, the IDLE CS is not handled in SM.
                  // The logic is done here and not below to avoid the update in the CallStatusList table from "vUpdateCSinCallStatusList"
                  // Fix for RTC-441792: condition check for External VR is added. During Active External VR session, no calls will handled.
                  if (((0 == CallData.rCallStatus.u8CallHandle) && (m_bDialMethodStartFlag) && (m_u8ActiveDeviceHandle == devicehandle)) || (!m_poSMExtVRSession->isExtVRSessionIdle()))
                  {
                     if(!m_poSMExtVRSession->isExtVRSessionIdle())
                     {
                        switch(CallData.rCallStatus.u8CallState)
                        {
                           case PHONE_CALLSTATE_IDLE:
                           {
                              ETG_TRACE_USR3(("IDLE call status received for External VR Session"));
                              m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_DBUS_CALLSTATUS_IDLE);
                           }
                           break;

                           case PHONE_CALLSTATE_ACTIVE:
                           {
                              ETG_TRACE_USR3(("Active call status received for External VR Session"));
                              m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_DBUS_CALLSTATUS_ACTIVE);
                           }
                           break;

                           case PHONE_CALLSTATE_INCOMING:
                           {
                              ETG_TRACE_USR3(("Incoming call status received during External VR Session"));
                              m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_DBUS_CALLSTATUS_INCOMING);
                           }
                           break;

                           default:
                           {
                              ETG_TRACE_USR3(("Other call status received during External VR Session"));
                           }
                           break;
                        }
                     }
                     break;
                  }

                  vUpdateCSinCallStatusList(CallData.BD_Address, CallData.rCallStatus);
                  vPrintCallStatusListVector();

                  if (devicehandle != m_u8ActiveDeviceHandle)
                  {
                     ETG_TRACE_USR2(("Call Status belongs to Passive device"));

                     if ((TRUE == bIsActiveDeviceIdle()) && ((PHONE_CALLSTATE_INCOMING == CallData.rCallStatus.u8CallState) \
                           || (PHONE_CALLSTATE_DIALING == CallData.rCallStatus.u8CallState) \
                           || (PHONE_CALLSTATE_ALERTING == CallData.rCallStatus.u8CallState)))
                     {
                        ETG_TRACE_USR3(("Active device is IDLE. Hence Switch Active/Passive and then process Incoming call status"));

                        if (PHONE_CALLSTATE_INCOMING == CallData.rCallStatus.u8CallState)
                        {
                           m_u8ActiveDeviceHandletobeRetained = m_u8ActiveDeviceHandle;
                           ETG_TRACE_USR2(("m_u8ActiveDeviceHandletobeRetained: %u", m_u8ActiveDeviceHandletobeRetained));
                        }
                        vSwitchActivePassive(devicehandle, FALSE);
                     }
                  }

                  if (m_u8ActiveDeviceHandle == devicehandle)
                  {
                     ETG_TRACE_USR3(("!!!!!!!!!! CALL STATUS BELONGS TO ACTIVE DEVICE !!!!!!!!!!!!!!!"));
                     if ((m_poSMVRSession->u16GetVRSessionState() == (tU16)STATE_ACTIVE) \
                           && (PHONE_CALLSTATE_INCOMING == CallData.rCallStatus.u8CallState))
                     {
                        ETG_TRACE_USR4((" Getting an incoming Bluetooth call- End Siri"));
                        m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_DEACT_INCOMINGCALL);
                     }
                     // NCG3D-3221- Hold for handset-handsfree mode
                     if(CallData.rCallStatus.u8CallHandle == 0 && m_bScoChannelReleaseTimerRunning == TRUE)
                     {
                        ETG_TRACE_USR3((" Timer interrupted and stopped because all are calls idle"));
                        vStopTimerAndProcessSCODisconnection(m_u16SCOChannelReleaseTimerId);
                     }

                     if ((PHONE_CALLSTATE_HELD == CallData.rCallStatus.u8CallState)
                           && (FC_PHONE_CALL_TOTAL_ONE == CallData.rCallStatus.u8CallTotal)
                           && (FALSE == m_bHeldtoActiveSwapTimerRunning))
                     {
                        //This is to avoid ReleaseAndSwap flickering
                        ETG_TRACE_USR3(("Single call on HOLD. May change to ACTIVE later. Hence wait for 1500 ms"));

                        (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Start(FC_PHONE_HELD_TO_ACTIVE_SWAP_TIMER_ID,
                              FC_PHONE_THREEHUNDRED_MILLISECOND * 5, vHeldtoActiveSwapTimerCallBack);

                        ETG_TRACE_USR3((" m_bHeldtoActiveSwapTimerRunning = TRUE"));
                        m_bHeldtoActiveSwapTimerRunning = TRUE;

                     }
                     else if (m_bHeldtoActiveSwapTimerRunning)
                     {
                        (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_HELD_TO_ACTIVE_SWAP_TIMER_ID);

                        ETG_TRACE_USR3((" m_bHeldtoActiveSwapTimerRunning = FALSE"));
                        m_bHeldtoActiveSwapTimerRunning = FALSE;
                     }

                     m_poCallManager->vNewCallStatus(&CallData);

                     if (0 != CallData.rCallStatus.u8CallHandle)
                     {
                        tU8 u8CallsActive = 0;
                        for(tU8 u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; ++u8Index)
                        {
                           if (FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE == m_oCallInstances[u8Index].m_u16CallStatusNotice)
                           {
                              u8CallsActive++;
                           }
                        }
                        vCheckConference(u8CallsActive);
                     }
                     else
                     {
                        // Calls are IDLE in the Current Active device
                        if (FC_PHONE_DEVICEHANDLE_UNKNOWN != m_u8ActiveDeviceHandletobeRetained)
                        {
                           vSwitchActivePassive(m_u8ActiveDeviceHandletobeRetained);
                        }
                     }

                     vUpdateSpmSubState();
                  }
                  else
                  {
                     ETG_TRACE_USR3(("!!!!!!!!!! CALL STATUS BELONGS TO PASSIVE DEVICE !!!!!!!!!!!!!!!"));
                     vHandleCallStatusofPassiveDevice(&CallData, devicehandle);
                     vSendPropertyUpdate(MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED, NULLPTR);
                  }
               }
               else
               {
                  ETG_TRACE_USR3(("No device connected with the given address: %s", CallData.BD_Address));
               }
            }
            break;

            case FC_PHONE_CB_FID_CALLSTATUS_CONSOLIDATED :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_CALLSTATUS_CONSOLIDATED"));
               T_CallData CallData[FC_PHONE_MAX_CALLINSTANCES] ;

               //get data here from vector if required,
               //and than erase from the vector corresponding position data
               copy_Consolidated_CallData_from_loopback_queue(CallData,&LB_data_queue[position]);
               erase_LB_data( position);

               tU8 devicehandle = u8GetDeviceHandlefromDevAddrinCSList(CallData[0].BD_Address);
               ETG_TRACE_USR3(("devicehandle: %d", devicehandle));

               if(devicehandle != FC_PHONE_DEVICEHANDLE_UNKNOWN)
               {
                  ETG_TRACE_USR3((" The device with Address: %s is connected", CallData[0].BD_Address));

                  for (tU8 ind = 0 ; ind < FC_PHONE_MAX_CALLINSTANCES; ++ind)
                  {
                     vUpdateCSinCallStatusList(CallData[ind].BD_Address, CallData[ind].rCallStatus);
                  }
                  vPrintCallStatusListVector();

                  if (m_u8ActiveDeviceHandle == devicehandle)
                  {
                     ETG_TRACE_USR3(("!!!!!!!!!! CALL STATUS BELONGS TO ACTIVE DEVICE !!!!!!!!!!!!!!!"));

                     if (!m_poSMExtVRSession->isExtVRSessionIdle())
                     {
                        if((PHONE_CALLSTATE_ACTIVE == CallData[0].rCallStatus.u8CallState)
                              && (PHONE_CALLSTATE_INCOMING == CallData[1].rCallStatus.u8CallState)
                              && (FC_PHONE_CALL_TOTAL_TWO == CallData[0].rCallStatus.u8CallTotal))
                        {
                           ETG_TRACE_USR3(("Incoming call status received during External VR Session"));
                           m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_DBUS_CALLSTATUS_INCOMING);
                        }
                        break;
                     }

                     tU8 u8CallsActive = 0;

                     // NCG3D-3221- Hold for handset-handsfree mode
                     if(CallData[0].rCallStatus.u8CallHandle == 0 && m_bScoChannelReleaseTimerRunning == TRUE)
                     {
                        ETG_TRACE_USR3((" Timer interrupted and stopped because all are calls idle"));
                        vStopTimerAndProcessSCODisconnection(m_u16SCOChannelReleaseTimerId);
                     }

                     if ((PHONE_CALLSTATE_HELD == CallData[0].rCallStatus.u8CallState) && ((PHONE_CALLSTATE_WAITING == CallData[1].rCallStatus.u8CallState))
                           && (FC_PHONE_CALL_TOTAL_TWO == CallData[0].rCallStatus.u8CallTotal) && (FALSE == m_bHeldtoActiveSwapTimerRunning))
                     {
                        //This is to avoid ReleaseAndSwap flickering
                        ETG_TRACE_USR3(("Single call on HOLD. May change to ACTIVE later. Hence wait for 1500 ms"));

                        (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Start(FC_PHONE_HELD_TO_ACTIVE_SWAP_TIMER_ID, FC_PHONE_THREEHUNDRED_MILLISECOND
                              * 5, vHeldtoActiveSwapTimerCallBack);

                        ETG_TRACE_USR3((" m_bHeldtoActiveSwapTimerRunning = TRUE"));
                        m_bHeldtoActiveSwapTimerRunning = TRUE;

                     }
                     else if (m_bHeldtoActiveSwapTimerRunning)
                     {
                        (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_HELD_TO_ACTIVE_SWAP_TIMER_ID);

                        ETG_TRACE_USR3((" m_bHeldtoActiveSwapTimerRunning = FALSE"));
                        m_bHeldtoActiveSwapTimerRunning = FALSE;
                     }
                     //Fix for NCG3D-39444.
                     //On receiving the consolidated CS, only on processing last received CS, the update shall be given to the clients
                     //and the remaining CS shall be ignored in vSendPropertyUpdate(MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED) update from SMs.

                     m_u8ConsolidatedCSCountprocessed = 0;
                     ETG_TRACE_USR3((" m_u8ConsolidatedCSCountprocessed :%d", m_u8ConsolidatedCSCountprocessed));

                     for ( ; m_u8ConsolidatedCSCountprocessed < FC_PHONE_MAX_CALLINSTANCES; ++m_u8ConsolidatedCSCountprocessed)
                     {
                        m_poCallManager->vNewCallStatus(&CallData[m_u8ConsolidatedCSCountprocessed]);
                     }

                     // While coming out from the for loop, the m_u8ConsolidatedCSCountprocessed = FC_PHONE_MAX_CALLINSTANCES.
                     // Now it shall be made FC_PHONE_MAX_CALLINSTANCES-1 so that the condition will get satisfied in
                     // vUpdateCallStatus and the CS will be updated whenever called.
                     m_u8ConsolidatedCSCountprocessed = FC_PHONE_MAX_CALLINSTANCES-1;
                     ETG_TRACE_USR3((" m_u8ConsolidatedCSCountprocessed :%d", m_u8ConsolidatedCSCountprocessed));

                     for(tU8 u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
                     {
                        if (FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE == m_oCallInstances[u8Index].m_u16CallStatusNotice)
                        {
                           u8CallsActive++;
                        }
                     }

                     if ((TWO_CALLS_ACTIVE == u8CallsActive) && (!vGetConfSMStatus()) && (!isConfSMExecutedFromCallStatus))
                     {
                        vStartConference();
                     }

                     vUpdateSpmSubState();
                  }
                  else
                  {
                     ETG_TRACE_USR3(("!!!!!!!!!! CALL STATUS BELONGS TO PASSIVE DEVICE !!!!!!!!!!!!!!!"));

                     for (tU8 ind = 0 ; ind < FC_PHONE_MAX_CALLINSTANCES; ind++)
                     {
                        vHandleCallStatusofPassiveDevice(&CallData[ind], devicehandle);
                     }
                     vSendPropertyUpdate(MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED, NULLPTR);
                  }
               }
               else
               {
                  ETG_TRACE_USR3(("No device connected with the given address: %s", CallData[0].BD_Address));
               }
            }
            break;

            case FC_PHONE_CB_FID_PHONESTATUS :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_PHONESTATUS"));
               T_PhoneStatus phone_status;

               //get data here from vector if required,
               //and than erase from the vector corresponding position data
               copy_PhoneStatus_from_loopback_queue(&phone_status,&LB_data_queue[position]);
               tU8 u8_ConnectedBDAddressId=fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(phone_status.BD_Address);
               erase_LB_data( position);

               //if phone status belong to Active device , than only process it
               //otherwise drop the processing  phone status data
               ETG_TRACE_USR2((" u8_ConnectedBDAddressId:: %d   ",u8_ConnectedBDAddressId));
               if(INVALID_BDADDRESS_ID != u8_ConnectedBDAddressId)
               {
                  ETG_TRACE_USR3((" Sending Phone Status to client"));
                  vPhoneStatus(&phone_status, u8_ConnectedBDAddressId);
               }
               else
               {
                  ETG_TRACE_USR3((" DROPPED processing phone status, because PhoneStatus data do not belong to Connected Device "));
               }
            }
            break;

            case FC_PHONE_CB_FID_HANGUPCALL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_HANGUPCALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_HANGUPCALL;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               //erase loop back data
               erase_LB_data( position);
               vAcknowledgeMethodStartForCallInstance(&tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_HOLDCALL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_HOLDCALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_HOLDCALL;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_HOLDCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_HOLDINCOMINGCALL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_HOLDINCOMINGCALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_HOLDINCOMINGCALL;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_HOLDINCOMINGCALL_ACK_OK,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_RESUMECALL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_RESUMECALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_RESUMECALL;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_RESUMECALL_ACK_OK,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_SWAPCALL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_SWAPCALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_SWAPCALL;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_SWAPCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);

               //If a call exists and another incoming call received, swap is called to accept it.
               //Then it would have resulted in queuing of other messages which came after that.Hence unlock here.
               if(TRUE == m_bAcceptMethodStartFlag)
               {
                  fc_phone_tclPhoneMutex::PhoneMutexUnlock();//DBUS ack received for swap call. Hence unlock.
               }
            }
            break;

            case FC_PHONE_CB_FID_CONFCALL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_CONFCALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;
               tProcessEventArg.u16DbusAckType =  FC_PHONE_CB_FID_CONFCALL;

               //erase loop back data
               erase_LB_data( position);

               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_CONFCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST, &tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_SPLITCALL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_SPLITCALL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;
               tProcessEventArg.u16DbusAckType =  FC_PHONE_CB_FID_SPLITCALL;

               //erase loop back data
               erase_LB_data( position);

               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_SPLITCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST, &tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_VOICERECOGNITION_CFM :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_VOICERECOGNITION_CFM"));

               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;
               tProcessEventArg.u16DbusAckType =  FC_PHONE_CB_FID_VOICERECOGNITION_CFM;

               //erase loop back data
               erase_LB_data( position);

               tBool bVRActive = (tBool)tProcessEventArg.Status;

               if (FC_PHONE_VOICE_RECOGNITION_ACTIVATE == bVRActive)
               {
                  ETG_TRACE_USR3((" FC_PHONE_VOICE_RECOGNITION_ACTIVATE"));
                  m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_ACTIVATION_ACK_OK, &tProcessEventArg);
               }
               else if (FC_PHONE_VOICE_RECOGNITION_DEACTIVATE == bVRActive)
               {
                  ETG_TRACE_USR3((" FC_PHONE_VOICE_RECOGNITION_DEACTIVATE"));
                  m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_DEACTIVATION_ACK_OK, &tProcessEventArg);
               }
            }
            break;

            case FC_PHONE_CB_FID_VOICERECOGNITION_IND :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_VOICERECOGNITION_IND"));
               //copy data from loopback
               T_VoiceRecInd voiceRecIndication;
               copy_VoiceRecInd_from_loopback_queue(&voiceRecIndication,&LB_data_queue[position]);
               erase_LB_data( position);

               ETG_TRACE_USR2((" voiceRecIndication.BD_Address :: %s",voiceRecIndication.BD_Address));

               tU8 u8DeviceHandle = u8GetDeviceHandlefromDevAddrinCSList(voiceRecIndication.BD_Address);
               ETG_TRACE_USR2(("u8DeviceHandle: %d", u8DeviceHandle));

               if ((u8DeviceHandle != m_u8ActiveDeviceHandle) && (FC_PHONE_DEVICEHANDLE_UNKNOWN != u8DeviceHandle))
               {
                  ETG_TRACE_USR2(("VOICERECOGNITION IND belongs to Passive device"));

                  if ((TRUE == bIsActiveDeviceIdle()) && (FC_PHONE_BT_VR_ACTIVE == voiceRecIndication.Activation))
                  {
                     ETG_TRACE_USR3(("Active device is IDLE. Hence Switch Active/Passive and then process VOICERECOGNITION_IND"));
                     vSwitchActivePassive(u8DeviceHandle);
                  }
               }

               if(u8DeviceHandle == m_u8ActiveDeviceHandle)
               {
                  ETG_TRACE_USR2(("VOICERECOGNITION IND belongs to Active device, m_u8ActiveDeviceHandle: %d", m_u8ActiveDeviceHandle));
                  if (FC_PHONE_BT_VR_ACTIVE == voiceRecIndication.Activation)
                  {
                     ETG_TRACE_USR3((" FC_PHONE_SM_EVENT_DBUS_VOICEREC_ACTIVATION_IND"));
                     m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_ACTIVATION_IND);
                  }
                  else if (FC_PHONE_BT_VR_INACTIVE == voiceRecIndication.Activation)
                  {
                     ETG_TRACE_USR3((" FC_PHONE_SM_EVENT_DBUS_VOICEREC_DEACTIVATION_IND"));
                     m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_DEACTIVATION_IND);
                  }
                  else
                  {
                     ETG_TRACE_USR3((" Activation type unknown"));
                  }
               }
               else
               {
                  ETG_TRACE_USR2(("VOICERECOGNITION IND belongs to Passive device"));
               }
            }
            break;

            case FC_PHONE_CB_FID_DIAL :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_DIAL"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_DIAL;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_DIAL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST, &tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_TERMINATE :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_TERMINATE"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_TERMINATE;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_TERMINATE_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);

               m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM;
               ETG_TRACE_USR4((" m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM "));
               fc_phone_tclPhoneMutex::PhoneMutexUnlock();//DBus ack is received for Terminate. Hence unlock mutex.
            }
            break;

            case FC_PHONE_CB_FID_QUERYTELEPHONENUMBER :
            {
               ETG_TRACE_USR3(("interface loop back FC_PHONE_CB_FID_QUERYTELEPHONENUMBER"));
               Async_method_data_t vAsync_method_data;
               //get data here from vector if required,
               //and than erase from the vector corresponding position data
               copy_method_data_from_loopback_queue(&vAsync_method_data,&LB_data_queue[position]);
               erase_LB_data( position);

               vQueryTelephoneNumberCB(&vAsync_method_data);
            }
            break;

            case FC_PHONE_CB_FID_SLCSTATUS :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_SLCSTATUS"));

               T_CallData  CallData ;
               //get data here from vector if required,
               //and than erase from the vector corresponding position data
               copy_from_loopback_queue(&CallData,&LB_data_queue[position]);

               //update the phoneData
               ETG_TRACE_USR2((" LB_data_queue[position].bSLCStatus :: %d  CallData.BD_Address :: %s",CallData.b_SLC_status,CallData.BD_Address));

               erase_LB_data( position);
               vSendSLCStatus(&CallData);
            }
            break;

            case FC_PHONE_CB_FID_DEVICENAME :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_DEVICENAME"));
               //get data here from vector if required,
               //and than erase from the vector corresponding position data

               tU8 u8_ConnectedBDAddressId=fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(LB_data_queue[position].Call_data.BD_Address);

               //get data here from vector if required,
               //and than erase from the vector corresponding position data


               //if phone status belong to Active device , than only process it
               //otherwise drop the processing  phone status data
               if(INVALID_BDADDRESS_ID != u8_ConnectedBDAddressId)
               {
                  ETG_TRACE_USR3((" Sending CellDeviceName to client"));

                  if (0 != u8StringCompare(m_oPhoneData[u8_ConnectedBDAddressId].m_achCellDeviceName, (const char*)LB_data_queue[position].Call_data.MR_Data.device_name))
                  {
                     vStringCopy(m_oPhoneData[u8_ConnectedBDAddressId].m_achCellDeviceName, (const char*) LB_data_queue[position].Call_data.MR_Data.device_name, FC_PHONE_STRING_LENGTH_MAX);
                     vSendCellDeviceName(u8_ConnectedBDAddressId);
                  }
               }
               else
               {
                  ETG_TRACE_USR3((" DROPPED processing CellDeviceName status, because CellDeviceName dta do not belong to Connected Device List "));
               }

               erase_LB_data( position);
            }
            break;

            case FC_PHONE_CB_FID_AUDIOTRANSFER :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_AUDIOTRANSFER"));
               //copy data from loopback
               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_AUDIOTRANSFER;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               if(LB_data_queue[position].Call_data.MR_Data.Status != FC_PHONE_DBUS_SCO_CONN_FAILED)
               {
                  tU8 ActiveBDAddressId =(tU8) m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
                  if(INVALID_BDADDRESS_ID != ActiveBDAddressId)
                  {
                     m_oPhoneData[ActiveBDAddressId].AudioOutStatus = LB_data_queue[position].Call_data.MR_Data.Status;
                  }
                  else
                  {
                     ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
                  }
               }

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_TRANSFERCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_SENDDTMF :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_SENDDTMF"));
               //copy data from loopback

               tU16 u16Id = 0x00;
               tS32 s32Error = OSAL_ERROR;

               FcPhone_processEventArg   tProcessEventArg ;
               tProcessEventArg.bIsItDbusAck =  TRUE ;
               tProcessEventArg.u16DbusAckType   =  FC_PHONE_CB_FID_SENDDTMF;
               tProcessEventArg.Code         =  LB_data_queue[position].Call_data.MR_Data.Code ;
               tProcessEventArg.State        =  LB_data_queue[position].Call_data.MR_Data.State ;
               tProcessEventArg.Status       =  LB_data_queue[position].Call_data.MR_Data.Status ;

               // get active call instance
               s32Error = s32GetActiveCallInstanceIndex(u16Id);

               if (OSAL_ERROR == s32Error)
               {
                  ETG_TRACE_USR4((" No calls are active"));
               }
               else
               {
                  tProcessEventArg.u16CallInstanceID = u16Id;
               }

               //erase loop back data
               erase_LB_data( position);
               m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_DTMF_ACK_OK, FC_PHONE_EVENTTYPE_TARGATED,&tProcessEventArg);
            }
            break;

            case FC_PHONE_CB_FID_SCOCONNECTIONSTATUS:
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_SCOCONNECTIONSTATUS"));
               T_SCOConnectionStatus SCOConnexnStatus;
               //get data here from vector if required,
               //and than erase from the vector corresponding position data
               copy_SCOConnectionStatus_from_loopback_queue(&SCOConnexnStatus,&LB_data_queue[position]);
               erase_LB_data( position);

               tU8 u8_ConnectedBDAddressId=fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(SCOConnexnStatus.BD_Address);
               ETG_TRACE_USR3((" u8_ConnectedBDAddressId : %d", u8_ConnectedBDAddressId));

               if(INVALID_BDADDRESS_ID != u8_ConnectedBDAddressId)
               {
                  fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
                  NULL_CHECK(p_clientHndler_bluetooth)

                  tU8 u8DevHandle = p_clientHndler_bluetooth->u8GetDeviceHandleFromBtAddressId(u8_ConnectedBDAddressId);
                  ETG_TRACE_USR3((" u8DevHandle : %d", u8DevHandle));

                  if(u8DevHandle != m_u8ActiveDeviceHandle)
                  {
                     ETG_TRACE_USR3(("FC_PHONE_CB_FID_SCOCONNECTIONSTATUS received for PASSIVE device"));
                     vUpdateSCOChannelConnectionStatus(SCOConnexnStatus);
                     vProcessSCODisconnection();
                     break;
                  }
               }
               else
               {
                  ETG_TRACE_USR3(("u8_ConnectedBDAddressId is INVALID => Device is not connected. Hence not processing further"));
                  break;
               }

               // NCG3D-3221- Hold for handset-handsfree mode
               vUpdateSCOChannelConnectionStatus(SCOConnexnStatus);

               if (SCOConnexnStatus.SCOConnectionStatus == FC_PHONE_DBUS_SCO_CONN_DISCONNECTED)
               {
                  // NCG3D-131177:
                  // Since there is an incoming call in passive device, it would try to establish the SCO connection.
                  // Hence the SCO disconnection of active device will be processed immediately without any timer dealy.
                  tU16 u16Id = 0x00;
                  tS32 s32CIStatusActiveOrPassive = s32GetCallInstanceIndexFromFiCallInstance((tU16)m_u16StoredCallInstanceForAccept, u16Id);
                  bool passiveDeviceAcceptCallInProgress = false;
                  if (s32CIStatusActiveOrPassive == true)
                  {
                     // SCO disconnection received for a current Active device when Accept call is in progress for the
                     // Passive device. Hence this SCO disconnection should be handled immediately.
                     ETG_TRACE_USR4(("Accept call in progress for Passive device"));
                     passiveDeviceAcceptCallInProgress = true;
                  }

                  if(!(m_poSMExtVRSession->isExtVRSessionIdle()))
                  {
                     vProcessSCODisconnection();
                     m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_SCO_RELEASE);
                  }
                  //Fix for PSARCCB-5362. HfReleaseAudio has to be processed immediately if ExtendVRSessionTimerRunning is RUNNING
                  else if ((m_poSMVRSession->u16GetVRSessionState() == (tU16)STATE_ACTIVE) || (TRUE == bCheckIdleCallStatus()) \
                        || (m_poSMVRSession->bIsExtendVRSessionTimerRunning()) || passiveDeviceAcceptCallInProgress)
                  {
                     vProcessSCODisconnection();
                     m_poSMManager->vProcessVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_SCO_RELEASE);
                  }
                  else
                  {
                     ETG_TRACE_USR3((" Audio release : FC_PHONE_CB_FID_SCOCONNECTIONSTATUS"));
                     // NCG3D-3221- Hold for handset-handsfree mode
                     m_u16SCOChannelReleaseTimerId = FC_PHONE_SCO_CHANNEL_RELEASE_TIMER_ID;

                     /* In case of suspending(put into Wait mode) the first incoming call in IPhone, if HfReleaseAudio and HfSetupAudio
                      * were received sequentially with more than 500ms time gap it will result in the acquisition of the LC_Phone channel.
                      * Hence the wait duration to process the HfReleaseAudio is raised to 1 sec if WaitMode is active.
                      * Thus the HfReleaseAudio and HfSetupAudio will be ignored in the above mentioned scenario, resulting in
                      * holding the acquired WaitMode Channel.
                      */
                     tU16 SCOChannelReleaseDuration = bIsTimerDurationExtendedForWaitingMode()? FC_PHONE_SCOCHANNEL_RELEASE_TIMEOUT_DURATION_WAITINGMODE_FIRSTCALL : FC_PHONE_SCOCHANNEL_RELEASE_TIMEOUT_DURATION;
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Start(m_u16SCOChannelReleaseTimerId, SCOChannelReleaseDuration, vSCOChannelReleaseTimerCallBack);
                     //Fix for NCG3D-142658,  ecnrStopAudio/ecnrStartAudio would be done during SCO Relinquishment, except during WaitingMode
#ifdef PHONE_WAITINGMODE
                     fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
                     tenFC_Phone_WaitingModeState m_enWaitingModeState = poWaitingMode->enGetWaitingModeState();
                     if(FC_PHONE_WAITINGMODE_IDLE == m_enWaitingModeState)
                     {
                        (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vResetSaScoEcnrInit(EN_SCO_EST);
                        ecnrStopAudio();
                     }
#endif
                     ETG_TRACE_USR3((" Timer started before releasing request is processed"));
                     m_bScoChannelReleaseTimerRunning = TRUE;
                     ETG_TRACE_USR3(("m_bScoChannelReleaseTimerRunning = TRUE"));
                  }
               }
               else
               {
                  if(TRUE == m_bScoChannelReleaseTimerRunning)
                  {
                     ETG_TRACE_USR3(("HfReleaseAudio is not yet processed, hence ignoring this HfSetupAudio"));
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_SCO_CHANNEL_RELEASE_TIMER_ID);
                     m_bScoChannelReleaseTimerRunning = FALSE;
                     ETG_TRACE_USR3(("m_bScoChannelReleaseTimerRunning = FALSE"));
                     //Fix for NCG3D-142658,  ecnrStopAudio/ecnrStartAudio would be done during SCO Relinquishment, except during WaitingMode
#ifdef PHONE_WAITINGMODE
                     fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
                     tenFC_Phone_WaitingModeState m_enWaitingModeState = poWaitingMode->enGetWaitingModeState();
                     if(FC_PHONE_WAITINGMODE_IDLE == m_enWaitingModeState)
                     {
                        (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetSaScoEcnrInit(EN_SCO_EST);
                        ecnrStartAudio();
                     }
#endif
                     return;
                  }

                  //Fix for PSARCCB-5362. On establishment of the SCO channel, the ExtendVRSessionTimer has to be ended if running.
                  vStopExtendedVRSession();

                  if( (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->u8SpeechChannelStatus() == EN_SPEECH_CHNL_NOTACTIVE )
                  {
                     m_u8EcnrConfigurationDataset = PHONE_DATASET_1; /*NarrowBand*/
                  }
                  else
                  {
#ifdef PHONE_SEPARATE_ECNR_SET_SIRI //Task CMG3G-7981
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->vGetVRDataSet(
                           m_u8EcnrConfigurationDataset, NARROW_BAND_SPEECH);
#else
                     m_u8EcnrConfigurationDataset = PHONE_DATASET_3; /*NarrowBand VR*/
#endif
                  }
                  (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetDataSet(m_u8EcnrConfigurationDataset);
                  if(!(m_poSMExtVRSession->isExtVRSessionIdle()))
                  {
                     m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_SCO_ACQUISITION);
                  }
                  // NCG3D-3221- Hold for handset-handsfree mode
                  vProcessChannelAndUpdateSCOConnectionstatus();
               }
            }
            break;
            case FC_PHONE_CB_FID_DELAYED_PROCESS_AUDIOSIGNAL:
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_DELAYED_PROCESS_AUDIOSIGNAL"));
               //erase loop back data
               erase_LB_data(position);
               m_poSMManager->vProcessAudioSignal();
            }
            break;

            case FC_PHONE_CB_FID_PHONE_DEVICE_CALL_STATUS :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_PHONE_DEVICE_CALL_STATUS"));

               tChar u8_ConnectedDeviceAddress[PHONE_BDADRESS_LENGTH] = {0};

               vStringCopy(u8_ConnectedDeviceAddress, LB_data_queue[position].Call_data.BD_Address, PHONE_BDADRESS_LENGTH);

               //erase loop back data
               erase_LB_data( position);

               if (m_pobtTelClient)
               {
                  if (FALSE == m_pobtTelClient->bIsOutGoingDevicePresent())
                  {
                     m_oSLCStatusStatus.bCommandStatus = FALSE;

                     vInitPhoneData(0xFF);

                     ETG_TRACE_USR3((" FC_PHONE_CB_FID_PHONE_DEVICE_CALL_STATUS: Send SLC OFF"));

                     ail_tenCommunicationError enResult =  eUpdateClients(MOST_TELFI_C_U16_SLCSTATUS);

                     if (AIL_EN_N_NO_ERROR != enResult)
                     {
                        /* Throw up error */
                        ETG_TRACE_ERR(( "vSendSLCStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                     }

                     return;
                  }
               }

               vRequestPhoneStatus(u8_ConnectedDeviceAddress);
            }
            break;

            case FC_PHONE_CB_FID_AUDIOMANAGER:
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_AUDIOMANAGER"));
               //copy data from loopback
               tU16 u16AudioChannelId = 0;
               tU16 u16RequestState = 0;
               tU16 u16AudioReqID = 0;
               u16AudioChannelId = LB_data_queue[position].AudioMgr_Data.u16AudioChannelId;
               u16AudioReqID = LB_data_queue[position].AudioMgr_Data.u16AudioReqID;
               u16RequestState = LB_data_queue[position].AudioMgr_Data.u16RequestState;
               //NCG3D-151011 Apple Certification fix. When VoiceRecognition is ongoing and the STACK sends the
               //             CallStatus as ACTIVE. This update is unexpected.
               //             But when this update is received, the PHONE channel request will be denied and the call
               //             would be transferred to AG.
               //             This fix would block the transfer of the call to AG, in the above mentioned issue.
               if ((FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId) && (FC_PHONE_AUDIOCHANNEL_DENIED == u16RequestState)
                       && ((fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->u8SpeechChannelStatus() != EN_SPEECH_CHNL_NOTACTIVE))
               {
                  LB_data_queue[position].AudioMgr_Data.AudioCallBack(u16AudioReqID,u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_GRANTED);
               }
               else
               {
                  LB_data_queue[position].AudioMgr_Data.AudioCallBack(u16AudioReqID,u16AudioChannelId,u16RequestState);
               }

               //Fix for NCG3D-10893. Unmute status should be updated on fresh and successful LC_Phone channel establishment.
               if ((FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId) && (FC_PHONE_AUDIOCHANNEL_GRANTED == u16RequestState))
               {
                  vMuteMicroPhone(FALSE, FALSE);
               }

               //erase loop back data
               erase_LB_data( position);
            }
            break;

            case FC_PHONE_CB_FID_ACQUIRETELEPHONENUMBER:
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_ACQUIRETELEPHONENUMBER"));
               Async_method_data_t vAsync_method_data;
               //get data here from vector if required,
               //and than erase from the vector corresponding position data
               copy_method_data_from_loopback_queue(&vAsync_method_data,&LB_data_queue[position]);
               erase_LB_data( position);
               vAcquireTelephoneNumberStatusCheck(&vAsync_method_data);
            }
            break;

            case FC_PHONE_CB_FID_VOICERECOGNITION_STATUS:
            {
               ETG_TRACE_USR3(("interface loop back FC_PHONE_CB_FID_VOICERECOGNITION_STATUS"));
               //erase loop back data
               erase_LB_data( position);
               vSendBtVRStatus();
            }
            break;

            //DIP
            case FC_PHONE_CB_FID_PNPINFO_IND:
            {
               ETG_TRACE_USR3(("interface loop back FC_PHONE_CB_FID_PNPINFO_IND"));
               T_PnpInfoInd PnpInfoInd_data;
               copy_PnpInfoInd_from_loopback_queue(&PnpInfoInd_data,&LB_data_queue[position]);
               erase_LB_data( position);

               tU8 u8BDId= fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(PnpInfoInd_data.BD_Address);
               if(u8BDId != INVALID_BDADDRESS_ID)
               {
                  m_u16PnpInfoVendorID[u8BDId] = PnpInfoInd_data.m_usVendorID;
                  m_u16PnpInfoVendorSource[u8BDId] = PnpInfoInd_data.m_usVendorIDSource;
                  ETG_TRACE_USR3((" PnpInfoInd_data[%d].m_usVendorID %d",u8BDId,PnpInfoInd_data.m_usVendorID));
                  ETG_TRACE_USR3((" PnpInfoInd_data[%d].m_usVendorID source %d",u8BDId,PnpInfoInd_data.m_usVendorIDSource));
                  ETG_TRACE_USR3((" PnpInfoInd_data[%d].BD_Address %s",u8BDId,PnpInfoInd_data.BD_Address));
               }
               else
               {
                  ETG_TRACE_USR3((" Invalid BDAddressID in PNPInfo"));
               }


            }
            break;

#ifdef PHONE_SIRI
            case FC_PHONE_CB_FID_SIRI_NR_IND :
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_SIRI_NR_IND"));
               //copy data from loopback
               T_SiriNRStatusInd siriNrIndication;
               copy_SiriNrInd_from_loopback_queue(&siriNrIndication,&LB_data_queue[position]);
               erase_LB_data( position);

               tU8 u8ConnectedDeviceBDAddressId = INVALID_BDADDRESS_ID;
               tBool bGetConnectedDeviceAddress = b_get_ConnectedDevice_AddressId_FromBT(&u8ConnectedDeviceBDAddressId, siriNrIndication.BD_Address);
               if (FALSE == bGetConnectedDeviceAddress)
               {
                  ETG_TRACE_ERR((" CONNECTED DEVICE ADDRESS IS NULL"));
                  return;
               }

               ETG_TRACE_USR3((" siriNrIndication.m_u8SiriNrStatus %d",siriNrIndication.m_u8SiriNrStatus));
               m_oSiriData[u8ConnectedDeviceBDAddressId].vSetSiriNrStatus(siriNrIndication.m_u8SiriNrStatus);
            }
            break;

            case FC_PHONE_CB_FID_SIRISTATUS_IND:
            {
               T_SiriStatus  siri_status;
               copy_SiriStatus_from_loopback_queue(&siri_status,&LB_data_queue[position]);
               erase_LB_data( position);

               tU8 u8ConnectedDeviceBDAddressId = INVALID_BDADDRESS_ID;
               tBool bGetConnectedDeviceAddress = b_get_ConnectedDevice_AddressId_FromBT(&u8ConnectedDeviceBDAddressId, siri_status.BD_Address);
               if (FALSE == bGetConnectedDeviceAddress)
               {
                  ETG_TRACE_ERR((" CONNECTED DEVICE ADDRESS IS NULL"));
                  return;
               }

               ETG_TRACE_USR3((" siri_status.m_u8SiriStatus %d",siri_status.m_u8SiriStatus));
               m_oSiriData[u8ConnectedDeviceBDAddressId].vSetSiriStatus(siri_status.m_u8SiriStatus);
               m_u8SiriReqRetryCount[u8ConnectedDeviceBDAddressId] = 0;
            }
            break;

            case FC_PHONE_CB_FID_NRSTATUS:
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_NRSTATUS"));

               T_SiriNRStatusInd siriNrIndication;
               copy_SiriNrInd_from_loopback_queue(&siriNrIndication, &LB_data_queue[position]);
               erase_LB_data( position);

               tU8 u8ConnectedDeviceBDAddressId = INVALID_BDADDRESS_ID;
               tBool bGetConnectedDeviceAddress = b_get_ConnectedDevice_AddressId_FromBT(&u8ConnectedDeviceBDAddressId, siriNrIndication.BD_Address);
               if (FALSE == bGetConnectedDeviceAddress)
               {
                  ETG_TRACE_ERR((" CONNECTED DEVICE ADDRESS IS NULL"));
                  return;
               }
               m_oSiriData[u8ConnectedDeviceBDAddressId].vSiriNRCallback();
               m_u8SiriReqRetryCount[u8ConnectedDeviceBDAddressId] = 0;
            }
            break;

            case FC_PHONE_CB_FID_SIRISTATUSREQ_RETRY:
            {
               T_SiriStatus  siri_status;
               copy_SiriStatus_from_loopback_queue(&siri_status,&LB_data_queue[position]);
               erase_LB_data( position);

               tU8 u8ConnectedDeviceBDAddressId = INVALID_BDADDRESS_ID;
               tBool bGetConnectedDeviceAddress = b_get_ConnectedDevice_AddressId_FromBT(&u8ConnectedDeviceBDAddressId, siri_status.BD_Address);
               if (FALSE == bGetConnectedDeviceAddress)
               {
                  ETG_TRACE_ERR((" CONNECTED DEVICE ADDRESS IS NULL"));
                  return;
               }

               if(FC_PHONE_SIRISTATUS_MAXRETRY_COUNT > m_u8SiriReqRetryCount[u8ConnectedDeviceBDAddressId])
               {
                  tBool bIsSiriStatusRetryTimerRunning = false;
                  //Start the retry timer
                  for(tU8 ind = 0; ind < FC_PHONE_MAX_CONNECTED_DEVICE; ind++)
                  {
                     if(m_bSiriRetryTimerRunning[ind])
                     {
                        bIsSiriStatusRetryTimerRunning = true;
                        break;
                     }
                  }

                  ETG_TRACE_USR3((" bIsSiriStatusRetryTimerRunning: %d", bIsSiriStatusRetryTimerRunning));

                  if(!bIsSiriStatusRetryTimerRunning)
                  {
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Start(FC_PHONE_SIRISTATUS_RETRY_TIMER_ID,
                           FC_PHONE_TWOHUNDRED_MILLISECOND, vPhoneTimerCallBackService);
                  }
                  m_u8SiriReqRetryCount[u8ConnectedDeviceBDAddressId]++;
                  m_bSiriRetryTimerRunning[u8ConnectedDeviceBDAddressId] = TRUE;
               }
               else
               {
                  m_u8SiriReqRetryCount[u8ConnectedDeviceBDAddressId] = 0;
                  ETG_TRACE_ERR(("Max Siri Request retry attempts reached!!"));
               }
            }
            break;

            case FC_PHONE_CB_FID_SIRINRSTATUSREQ_RETRY:
            {
               T_SiriNRStatusInd siriNrIndication;
               copy_SiriNrInd_from_loopback_queue(&siriNrIndication,&LB_data_queue[position]);
               erase_LB_data( position);

               tU8 u8ConnectedDeviceBDAddressId = INVALID_BDADDRESS_ID;
               tBool bGetConnectedDeviceAddress = b_get_ConnectedDevice_AddressId_FromBT(&u8ConnectedDeviceBDAddressId, siriNrIndication.BD_Address);
               if (FALSE == bGetConnectedDeviceAddress)
               {
                  ETG_TRACE_ERR((" CONNECTED DEVICE ADDRESS IS NULL"));
                  return;
               }

               if(FC_PHONE_SIRISTATUS_MAXRETRY_COUNT > m_u8SiriReqRetryCount[u8ConnectedDeviceBDAddressId])
               {
                  tBool bIsSiriNRStatusRetryTimerRunning = false;
                  //Start the retry timer
                  for(tU8 ind = 0; ind < FC_PHONE_MAX_CONNECTED_DEVICE; ind++)
                  {
                     if(m_bSiriNrRetryTimerRunning[ind])
                     {
                        bIsSiriNRStatusRetryTimerRunning = true;
                        break;
                     }
                  }

                  ETG_TRACE_USR3((" bIsSiriNRStatusRetryTimerRunning: %d", bIsSiriNRStatusRetryTimerRunning));

                  if(!bIsSiriNRStatusRetryTimerRunning)
                  {

                     //Start the retry timer
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Start(FC_PHONE_SIRINRSTATUS_RETRY_TIMER_ID,
                           FC_PHONE_TWOHUNDRED_MILLISECOND, vPhoneTimerCallBackService);
                  }
                  m_u8SiriReqRetryCount[u8ConnectedDeviceBDAddressId]++;
                  m_bSiriNrRetryTimerRunning[u8ConnectedDeviceBDAddressId] = TRUE;
               }
               else
               {
                  m_u8SiriReqRetryCount[u8ConnectedDeviceBDAddressId] = 0;
                  ETG_TRACE_ERR(("Max SiriNr Request retry attempts reached!!"));
               }
            }
            break;

#endif//PHONE_SIRI
            case FC_PHONE_LB_FID_SCO_CONNECT_REQUEST:
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_LB_FID_SCO_CONNECT_REQUEST"));

               T_CallData CallData;

               //BD_Address is enough for processing SCO Connect Request
               vStringCopy(CallData.BD_Address,LB_data_queue[position].Call_data.BD_Address, PHONE_BDADRESS_LENGTH);
               erase_LB_data(position);

               ETG_TRACE_USR2(("SCO Connect Request LB, Device Address :: %s", CallData.BD_Address));

               tU8 u8DeviceHandle = u8GetDeviceHandlefromDevAddrinCSList(CallData.BD_Address);
               ETG_TRACE_USR2(("SCO Connect Request for u8DeviceHandle : %d ", u8DeviceHandle));

               if (FC_PHONE_DEVICEHANDLE_UNKNOWN == m_u8ActiveDeviceHandle)
               {
                  // m_u8ActiveDeviceHandle is INVALID.
                  // The Connection is in progress with no device being connected via HFP.
                  // The received SCO connect Request will be accepted directly.
                  // Since the prior HFP connecting device will be the Active device.

                  ETG_TRACE_USR3(("First HFP device connecting. Accepting SCO connect request directly"));
                  acceptSCOConnectRequest(CallData.BD_Address);
               }
               else if(u8DeviceHandle == m_u8ActiveDeviceHandle)
               {
                  ETG_TRACE_USR2(("SCO Connect Request is for the ACTIVE device, m_u8ActiveDeviceHandle : %d ", m_u8ActiveDeviceHandle));
                  acceptSCOConnectRequest(CallData.BD_Address);
               }
               else if (TRUE == bIsActiveDeviceIdle())
               {
                  // Active device is IDLE. And SCOConnectRequest is received for Passive device.
                  // Hence Switching the Passive to Active and accepting the SCO connect request.
                  m_u8ActiveDeviceHandletobeRetained = m_u8ActiveDeviceHandle;
                  ETG_TRACE_USR2(("Switch Active Passive and Accept SCO request with m_u8ActiveDeviceHandletobeRetained : %d", m_u8ActiveDeviceHandletobeRetained));
                  vSwitchActivePassive(u8DeviceHandle,false);
                  acceptSCOConnectRequest(CallData.BD_Address);
               }
            }
            break;

            case FC_PHONE_CB_FID_WIDEBANDSPEECHSETUP:
            {
               ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_WIDEBANDSPEECHSETUP"));

               // NCG3D-3221- Hold for handset-handsfree mode
               if(TRUE == m_bScoChannelReleaseTimerRunning)
               {
                  ETG_TRACE_USR3(("HfReleaseAudio is not yet processed, hence ignoring this HfSetupAudio"));
                  (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_SCO_CHANNEL_RELEASE_TIMER_ID);
                  m_bScoChannelReleaseTimerRunning = FALSE;
                  ETG_TRACE_USR3(("m_bScoChannelReleaseTimerRunning = FALSE"));
                  //Fix for NCG3D-142658,  ecnrStopAudio/ecnrStartAudio would be done during SCO Relinquishment, except during WaitingMode
#ifdef PHONE_WAITINGMODE
                  fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
                  tenFC_Phone_WaitingModeState m_enWaitingModeState = poWaitingMode->enGetWaitingModeState();
                  if(FC_PHONE_WAITINGMODE_IDLE == m_enWaitingModeState)
                  {
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetSaScoEcnrInit(EN_SCO_EST);
                     ecnrStartAudio();
                  }
#endif
                  erase_LB_data( position);
                  return;
               }

               T_SCOConnectionStatus SCOConnexnStatus;
               copy_SCOConnectionStatus_from_loopback_queue(&SCOConnexnStatus,&LB_data_queue[position]);
               erase_LB_data(position);

               ETG_TRACE_USR2((" SCOConnectionStatus :: %d Device Address :: %s",SCOConnexnStatus.SCOConnectionStatus,SCOConnexnStatus.BD_Address));
               ETG_TRACE_USR2((" m_u8BandSelection :: %d Codec :: %d",m_u8BandSelection,SCOConnexnStatus.Codec ));

               //Fix for PSARCCB-5362. On establishment of the SCO channel, the ExtendVRSessionTimer has to be ended if running.
               vStopExtendedVRSession();

               if( (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->u8SpeechChannelStatus() == EN_SPEECH_CHNL_NOTACTIVE )
               {
                  // PHONE channel is active.
                  if( SCOConnexnStatus.Codec == WIDE_BAND_SPEECH )
                  {
                     // For wide band speech ecnr configuration required.
                     m_u8EcnrConfigurationDataset = PHONE_DATASET_2;
                  }
                  else
                  {
                     // For narrow band speech ecnr configuration not required.
                     m_u8EcnrConfigurationDataset = PHONE_DATASET_1;
					 // audio not heard issue was observed when ecnr is not configured for narrow band
                     //(fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetSaScoEcnrInit(EN_ECNR_CONFIGURATION);
                  }
               }
               else
               {
                  if(m_poSMExtVRSession->isExtVRSessionIdle())
                  {
                     // SPEECH channel is active.
                     if( SCOConnexnStatus.Codec == WIDE_BAND_SPEECH )
                     {
#ifdef PHONE_SEPARATE_ECNR_SET_SIRI //Task CMG3G-7981
                        (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->vGetVRDataSet(
                              m_u8EcnrConfigurationDataset, WIDE_BAND_SPEECH);
#else
                        // For wide band speech ecnr configuration required.
                        m_u8EcnrConfigurationDataset = PHONE_DATASET_4;
#endif
                     }
                     else
                     {
#ifdef PHONE_SEPARATE_ECNR_SET_SIRI
                        (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->vGetVRDataSet(
                              m_u8EcnrConfigurationDataset, NARROW_BAND_SPEECH);
#else
                        // For wide band speech ecnr configuration not required.
                        m_u8EcnrConfigurationDataset = PHONE_DATASET_3;
                        //(fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetSaScoEcnrInit(EN_ECNR_CONFIGURATION);
#endif
                     }
                  }
                  else
                  {
                     // External VR is active.
                     if( SCOConnexnStatus.Codec == WIDE_BAND_SPEECH )
                     {
                        m_u8EcnrConfigurationDataset = PHONE_DATASET_4;
                     }
                     else
                     {
                        m_u8EcnrConfigurationDataset = PHONE_DATASET_3;
                     }
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetDataSet(m_u8EcnrConfigurationDataset);
                     if(!(m_poSMExtVRSession->isExtVRSessionIdle()))
                     {
                        m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_DBUS_VOICEREC_SCO_ACQUISITION);
                     }
                  }
               }
               (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetDataSet(m_u8EcnrConfigurationDataset);
               vHandleOffState(&SCOConnexnStatus);
               m_pobtTelClient->vUpdateSCOConnectionStatus(&SCOConnexnStatus);

               if (m_poSMVRSession->u16GetVRSessionState() != (tU16)STATE_ACTIVE)
               {
#ifdef PHONE_WAITINGMODE
                  ETG_TRACE_USR4((" Handling SCO connection in WaitingMode"));
                  trFC_Phone_SCOConnStatus rSCOConnStatus;
                  rSCOConnStatus.u8SCOConnectionStatus = SCOConnexnStatus.SCOConnectionStatus;
                  rSCOConnStatus.u8Codec = SCOConnexnStatus.Codec;
                  fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
                  poWaitingMode->bProcessEvent(FC_PHONE_WAITING_MODE_EVENT_SCO_CONNECTION_STATUS, &rSCOConnStatus);
#endif
                  m_poSMManager->vProcessAudioSignal() ;
               }

               vUpdateSpmSubState(); // Update spm substate when vehicle audio changes.
            }
            break;
            // FIX GMMY16-7888 	[CT][MY16 D1BL Telephone] HMI reset observed when calls were swapped and then end call selected immediately.
            case FC_PHONE_LB_FROM_WQ_FOR_QUEUED_METHODS:
            {
               ETG_TRACE_USR4((" interface loop back FC_PHONE_LB_FROM_WQ_FOR_QUEUED_METHODS"));
               Job *job = new Job;
               if (job)
               {
                  copy_WQJob_from_loopback_queue(job,&LB_data_queue[position]);
                  //gmmy16-10681
                  ETG_TRACE_USR3((" Calling using function pointer"));
                  (fc_phone_tclService_Telephone:: pGetInstance()->*(job->m_pfMethod))(NULLPTR,job);
                  //End of fix
               }
               erase_LB_data( position);
            }
            break;

            //Fix for task CMG3G-5839
            case FC_PHONE_LB_FROM_WQ_FOR_FAILED_METHODS:
            {
               ETG_TRACE_USR4((" interface loop back FC_PHONE_LB_FROM_WQ_FOR_FAILED_METHODS"));

               tU16 u16FunctionID = LB_data_queue[position].job.u16FunctionID;
               erase_LB_data(position);
               (fc_phone_tclService_Telephone::pGetInstance())->vPostMethodError(u16FunctionID);

               // In case, the MS is for Passive device, m_bEndCallforPassiveDev has to be made FALSE
               m_bEndCallforPassiveDev = false;
            }
            break;

            case FC_PHONE_CB_FID_ECNR_STARTAUDIO:
            {
               ETG_TRACE_USR3(( " interface loop back FC_PHONE_CB_FID_ECNR_STARTAUDIO " ));
               T_CallbackStatus tCallBackStatus = LB_data_queue[position].ecnrCallBackStatus;
               erase_LB_data( position);
               if( tCallBackStatus.bCallBackStatus == FALSE )
               {
                  // if start audio fails, transfer call to handset.
                  fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
                  tU8 u8SCOConnectionStatus = p_clientHndler_bluetooth->vGetSCOConnectionStatus(const_cast<tChar *>(sGetDeviceAddressfromCSList(m_u8ActiveDeviceHandle)));

                  if (FC_PHONE_DBUS_SCO_CONN_ESTABLISHED == u8SCOConnectionStatus)
                  {
                     ETG_TRACE_USR3(( " SCO connection already established. Hence transferring the call " ));
                     vTransferCallToHandset(FC_PHONE_TRANSFERCALL_EXT_TRIGGER_RELEASE_CHANNEL);
                  }
                  else
                  {
                     ETG_TRACE_USR3(( " SCO connection got released already" ));
                  }
               }
               else
               {
                  if(!(m_poSMExtVRSession->isExtVRSessionIdle()))
                  {
                     m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_CCX_EXT_VR_ECNRSTARTAUDIO);
                  }
               }
            }
            break;

            case FC_PHONE_CB_FID_ECNR_STOPAUDIO:
            {
               ETG_TRACE_USR3(( " interface loop back FC_PHONE_CB_FID_ECNR_STOPAUDIO " ));
               T_CallbackStatus tCallBackStatus = LB_data_queue[position].ecnrCallBackStatus;
               erase_LB_data( position);
               if( tCallBackStatus.bCallBackStatus == TRUE )
               {
                  if(!(m_poSMExtVRSession->isExtVRSessionIdle()))
                  {
                     m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_CCX_EXT_VR_ECNRSTOPAUDIO);
                  }
               }
            }
            break;
            case FC_PHONE_CB_FID_ECNR_MUTESWITCH:
            {
               ETG_TRACE_USR3(( " interface loop back FC_PHONE_CB_FID_ECNR_MUTESWITCH " ));
               erase_LB_data( position);
            }
            break;
      case FC_PHONE_LB_FROM_ARL_FOR_SA_ON:
         {
            ETG_TRACE_USR3(( " interface loop back FC_PHONE_LB_FROM_ARL_FOR_SA_ON " ));
                  erase_LB_data(position);

                  if (m_poSMManager)
                  {
                     tBool bEventConsumed = FALSE;
                     FcPhone_processEventArg   tProcessEventArg;
                     tProcessEventArg.bIsItDbusAck = FALSE;
                     bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_ARL_SRCACTIVITY_ON,FC_PHONE_EVENTTYPE_BROADCAST_CONSUMED,&tProcessEventArg);

                     if(FALSE == bEventConsumed)
                     {
                        ETG_TRACE_USR3(( "Event not consumed in SM so post SA_MR to ARL"));
                        (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vPostSrcActivityMRToARL();
                     }
                  }
                  // Fix for NCG3D-103060
                  // Resetting here because it confirms here that ARL is responding for PHONE channel request
                  m_bExternalTriggerStatusforPhoneChannel = FALSE;
                  ETG_TRACE_USR3(("m_bExternalTriggerStatusforPhoneChannel : %u", m_bExternalTriggerStatusforPhoneChannel));
               }
               break;

            case FC_PHONE_LB_FID_AUDIOTRANSFER_TO_HS:
               {
                  ETG_TRACE_USR3(( " interface loop back FC_PHONE_LB_FID_AUDIOTRANSFER_TO_HS " ));
                  erase_LB_data(position);
                  // Fix for NCG3D-103060
                  m_bExternalTriggerStatusforPhoneChannel = TRUE;
                  ETG_TRACE_USR3(("m_bExternalTriggerStatusforPhoneChannel : %u", m_bExternalTriggerStatusforPhoneChannel));
                  vTransferCallToHandset();
               }
               break;

            case FC_PHONE_LB_FID_AUDIOTRANSFER_TO_HF:
               {
                  ETG_TRACE_USR3(( " interface loop back FC_PHONE_LB_FID_AUDIOTRANSFER_TO_HF " ));
                  erase_LB_data(position);
                  vTransferCallToHandsfree();
               }
               break;
            case FC_PHONE_LB_FROM_ARL_FOR_SA_PAUSE:
               {
                  ETG_TRACE_USR3(( " interface loop back FC_PHONE_LB_FROM_ARL_FOR_SA_PAUSE " ));
                  erase_LB_data(position);
                  vMuteMicroPhone(FALSE, FALSE);
               }
               break;
//GMMY16-24623 
            case FC_PHONE_CB_FID_TIMER:
               {
                  ETG_TRACE_USR3(( " interface loop back FC_PHONE_CB_FID_TIMER " ));
                  tU16 u16TimerId =  LB_data_queue[position].Timer_CB.u16TimerId;
                  (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->vTimerExpired(u16TimerId);
                  erase_LB_data( position);
               }
               break;

            case FC_PHONE_CB_FID_NAME_APPEARED:
               {
                  ETG_TRACE_USR3((" interface loop back FC_PHONE_CB_FID_NAME_APPEARED "));
                  char activeDeviceAddressEvo[PHONE_OBJECT_PATH] = {'\0'};
                  vStringCopy(activeDeviceAddressEvo, LB_data_queue[position].Call_data.BD_Address, PHONE_BDADRESS_LENGTH);
                  erase_LB_data(position);

                  //Calling Evolution instance bCreateProxyAndRegisterSignals.
                  ETG_TRACE_USR3(("calling EvoCreateProxyAndRegisterSignals.."));
                  m_pDBus_CCA_If->bCreateProxyAndRegisterSignals(activeDeviceAddressEvo);
               }
               break;


#ifdef PHONE_WAITINGMODE
            case FC_PHONE_CB_FID_PHONEWAIT_SRC_ON:
            {
               ETG_TRACE_USR3(("Interface loop back FC_PHONE_CB_FID_PHONEWAIT_SRC_ON"));

               erase_LB_data( position);

               //If there is a waiting incoming call then post to SM, other post to the WaitingMode

               tU16 u16CallInstanceId;
               tBool bWaitingCallFound = bGetCallInstanceIdForWaitingCall(&u16CallInstanceId);

               //fill process Event Arguments and send processEvent
               FcPhone_processEventArg tProcessEventArg;
               tProcessEventArg.bIsItDbusAck = FALSE;
               tProcessEventArg.u16CallInstanceID = u16CallInstanceId;

               if (TRUE == bWaitingCallFound)
               {
                  tBool bEventConsumed = FALSE;
                  bEventConsumed = m_poSMManager->bProcessEvent(
                        FC_PHONE_SM_EVENT_ARL_PHONEWAIT_SRC_ON,
                        FC_PHONE_EVENTTYPE_TARGATED,
                        &tProcessEventArg);

                  if(FALSE == bEventConsumed)
                  {
                     ETG_TRACE_USR4(("Event not Consumed"));
                  }
               }
               else
               {
                  fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
                  poWaitingMode->bProcessEvent(FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_ON);
               }
               // Fix for NCG3D-103060
               // Resetting here because it confirms here that ARL is responding for PHONE/WAITMODE channel request
               m_bExternalTriggerStatusforPhoneChannel = FALSE;
               ETG_TRACE_USR3(("m_bExternalTriggerStatusforPhoneChannel : %u", m_bExternalTriggerStatusforPhoneChannel));
            }
            break;

            case FC_PHONE_CB_FID_PHONEWAIT_SRC_OFF:
            {
               ETG_TRACE_USR3(("Interface loop back FC_PHONE_CB_FID_PHONEWAIT_SRC_OFF"));

               erase_LB_data( position);

               fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
               poWaitingMode->bProcessEvent(FC_PHONE_WAITING_MODE_EVENT_PHONEWAIT_SRC_OFF);
            }
            break;
#endif

            default:
               {
                  ETG_TRACE_USR3((" vOnLoopbackService Unknown u32MsgContent %u  IN DEFAULT::WRONG interface loop back function ID FC_PHONE_CB_FID ",u32MsgContent));
                  erase_LB_data( position);
               }
               break;
         }
      }
      break;
      default:
      {
         ETG_TRACE_ERR(("vOnLoopbackService. Unknown iFID %u.", u16FunctionId));
      }
      break;
   }
}


/*******************************************************************************
** FUNCTION:   vHandle_AcceptCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for AccepCall method
*   Thie function talks to dbus_client to accept call
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_AcceptCall_MethodStart(amt_tclServiceData* poMessage, struct Job* f_poJob)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vHandle_AcceptCall_MethodStart entered"));

   // Create the (message related) FI data object
   most_telfi_tclMsgAcceptCallMethodStart *poAcceptCall_MS = NULLPTR;

   //FIX GMMY15-10101 [CT]Next Gen HMI - Issue with unable to answer the call, when the user tries to make a call at the same time
   //Second request sent to BT Daemon from FC_Phone before receiving response of former request is responded with error by BT Daemon for some method requests.
   //For those requests for which API restrictions are defined in ALPS document, FC_Phone tries to acquire a mutex lock.
   //If acquired then only request is forwarded to DBUS. Else it is added to a work queue.
   struct Job* l_poJob = NULLPTR;

   if(poMessage)
   {
      /* Extract the Call instance and other required info from poMessage */
      // Create a FI visitor message for the received CCA message
      fi_tclVisitorMessage oVisitorMsg( poMessage );
      poAcceptCall_MS = new most_telfi_tclMsgAcceptCallMethodStart;

      if(poAcceptCall_MS)
      {
         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.
         if (( oVisitorMsg.s32GetData(*poAcceptCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION) != OSAL_ERROR) && (FALSE == m_bDiagnosticModeFlag))
         {
            ETG_TRACE_USR4((" m_u16StoredCallInstanceForAccept : %d",m_u16StoredCallInstanceForAccept));
            //If same request comes again to FC_Phone, it is not processed. Accept request returns giving error.
            if( m_u16StoredCallInstanceForAccept == poAcceptCall_MS->u16CallInstance )
            {
               ETG_TRACE_ERR(( "vHandle_AcceptCall_MethodStart: Already received message under process. Rejecting duplicates." ));
               vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

               poAcceptCall_MS->vDestroy();
               delete poAcceptCall_MS;
               return;
            }
            else
            {
               //If same request comes again to FC_Phone we dont push them to queue. Accept request returns giving error.
               //Store current parameter to check duplicate requests which may come afterwards.

               m_u16StoredCallInstanceForAccept = poAcceptCall_MS->u16CallInstance;
               ETG_TRACE_USR4(("m_u16StoredCallInstanceForAccept %d",m_u16StoredCallInstanceForAccept ));
            }
            //Create a new job
            l_poJob = new Job;
            if(l_poJob)//If creation of job is successful, assign parameters
            {
               l_poJob->m_pfMethod  = &fc_phone_tclService_Telephone::vHandle_AcceptCall_MethodStart;
               l_poJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               l_poJob->u16FunctionID = poMessage->u16GetFunctionID();
               l_poJob->u16RegisterID = poMessage->u16GetRegisterID();
               l_poJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               l_poJob->m_poMethodStart = poAcceptCall_MS;
            }
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_AcceptCall_MethodStart: Received message invalid."));
            // We never expect to come here.
            vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            poAcceptCall_MS->vDestroy();
            delete poAcceptCall_MS;
            return;
         }
      }
   }
   else if(f_poJob)
   {
      l_poJob = f_poJob;
   }
   else
   {
      return; // Should not come here
   }

   if(!l_poJob)
   {
      ETG_TRACE_ERR((" l_poJob is NULL!!"));
      vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob);

      if(poAcceptCall_MS)
      {
         poAcceptCall_MS->vDestroy();
         delete poAcceptCall_MS;
      }
      return;
   }

   if (fc_phone_tclPhoneMutex::PhoneMutexTrylock(l_poJob) != 0)//Try to acquire lock
   {
      tVoid*  pCustodialLose = l_poJob ;  //To remove LINT warning
      (tVoid) pCustodialLose ;
      return; // Didn't get the lock :(
   }

   //No need to clear the previously stored values. If value cleared, next request from HMI(duplicate) would enter the queue (which is not recommended)
   // and the subsequent requests after that only will be rejected .

   //m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM ;

   //If lock is acquired process the request
   m_oAcceptCallMessage.u16SourceAppID = l_poJob->u16SourceAppID;
   m_oAcceptCallMessage.u16RegisterID = l_poJob->u16RegisterID;
   m_oAcceptCallMessage.u16CmdCounter = l_poJob->u16CmdCounter;
   m_oAcceptCallMessage.u16FunctionID = l_poJob->u16FunctionID;


   poAcceptCall_MS = (most_telfi_tclMsgAcceptCallMethodStart*)   l_poJob->m_poMethodStart;
   NULL_CHECK(poAcceptCall_MS);

   /* Get the coresponding Call Instance */
   tU16 u16Id = 0x00;
   tS32 s32CIStatusActiveOrPassive = s32GetCallInstanceIndexFromFiCallInstance(poAcceptCall_MS->u16CallInstance, u16Id);
   if((OSAL_OK == s32CIStatusActiveOrPassive) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
   {
      ETG_TRACE_USR1((" ********MethodStart:AcceptCall*********"));
      ETG_TRACE_USR1((" u16CallInstance: %u, u16Id: %d ", poAcceptCall_MS->u16CallInstance, u16Id));

      //FIX GMMY16-7888 HMI reset observed when calls were swapped and then end call selected immediately.
      NULL_CHECK(m_poSMManager)
      //End of fix

      fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];
      if(m_poSMManager->bIsSMRunning(FC_PHONE_SMMANAGER_SMID_ACCEPTCALL, pCallInstance->m_u16Id))
      {
         ETG_TRACE_USR3((" vHandle_AcceptCall_MethodStart::Sending Method Result for Accept Call Method "));
         pCallInstance->m_AckError = FC_PHONE_CALLINSTANCE_ACKERROR_OK;
         m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_METHOD_STATUS_ACCEPTCALL,pCallInstance);
         //unlock while posting method result, else other request will not get signal
         fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error
      }
      else
      {
         //FIX GMMY15-9417 [CT]Next Gen HMI - Issue with HMI requesting AV activation LC phone under no call
         //When there is no call and still a client requests for Accept call, then Error should be sent.
         //Only when the incoming call is in 'Ringing' state, we should process accept requests.
         if( (FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE == pCallInstance->m_u8ObjectState) && (FC_PHONE_CALLINSTANCE_INCOMINGCALLEVENT_RINGING ==  pCallInstance->m_u16IncomingCallEvent) )
         {
            ETG_TRACE_USR3((" vHandle_AcceptCall_MethodStart::vStartSM "));
            m_bAcceptMethodStartFlag = TRUE;
            m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_ACCEPTCALL , pCallInstance);
         }
         else
         {
            /* Requested instance is not in ringing state, So send an error message */
            ETG_TRACE_USR4(( "vHandle_AcceptCall_MethodStart: Requested instance is not in ringing state" ));
            vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
         }
         //End of FIX
      }
   }

   else if (true == s32CIStatusActiveOrPassive)
   {
      ETG_TRACE_USR4(("Call Instance belongs to the Passive Device"));
      m_bAcceptMethodStartFlag = TRUE;

      tU8 u8SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_FAILED;
      fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
      u8SCOConnectionStatus = p_clientHndler_bluetooth->vGetSCOConnectionStatus(const_cast<tChar *>(sGetDeviceAddressfromCSList(m_u8ActiveDeviceHandle)));

      ETG_TRACE_USR4(("m_u8ActiveDeviceHandle = %u and m_u8ActiveDeviceHandletobeRetained = %u",m_u8ActiveDeviceHandle,m_u8ActiveDeviceHandletobeRetained));
      if(m_u8ActiveDeviceHandletobeRetained == FC_PHONE_DEVICEHANDLE_UNKNOWN)
      {
         m_u8ActiveDeviceHandletobeRetained = u8GetActiveDeviceHandle();
         ETG_TRACE_USR4(("m_u8ActiveDeviceHandletobeRetained: %u", m_u8ActiveDeviceHandletobeRetained));
      }
      if (FC_PHONE_DBUS_SCO_CONN_ESTABLISHED == u8SCOConnectionStatus)
      {
         ETG_TRACE_USR4(("SCO channel is established for the Active Device. Hence transferring the call to HS"));
         vTransferCallToHandset(FC_PHONE_TRANSFERCALL_EXT_TRIGGER_DNT_RELEASE_CHANNEL);
      }
      else
      {
         ETG_TRACE_USR4(("SCO channel is NOT established for the Active Device"));

         (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vReleaseUsedAudioChannels();

         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(m_oCallInstances[0].m_u16TimerId);
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(m_oCallInstances[1].m_u16TimerId);

         ETG_TRACE_USR3((" Reset StateMachines and Call Instances"));
         for(tU16 u16index = 0 ;u16index < FC_PHONE_MAX_CALLINSTANCES ;u16index++)
         {
            fc_phone_tclCallInstance* pCallInstance = &m_oCallInstances[u16index];
            m_poSMManager->vStartSM(FC_PHONE_SMMANAGER_SMID_IDLE,pCallInstance);
            pCallInstance->vInit();
         }
         ETG_TRACE_USR3((" Reset VR StateMachine"));
         fc_phone_tclSMVRSession* poVRInstance = fc_phone_tclSMVRSession::poGetInstance();
         if(poVRInstance)
         {
            poVRInstance->vCancel();
         }

         NULL_CHECK(m_poCallManager)
         m_poCallManager->vResetCallHandlerMaptable();

         //Create Call Status CallData so that it shall be given to CallHandlerMaptable and shall be handled in SM
         T_CallData CallData;
         vCreateCallDatafromCSListIndex(&CallData, m_u16StoredCallInstanceForAccept);
         m_poCallManager->vAddEntryToCallHandlerMaptable(&CallData);

         // Call Instances are initialized. Hence Call Instance Index 0 is used directly to create AcceptCall_SM
         tU16 u16CIIndex = 0x00;

         /*Device will be made Active and the vRequestPhoneStatus() shall be called on sending the AcceptCall_MR or ME
          * This is to reduce the delay in processing AcceptCall and posting the MR or ME
          */
         fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16CIIndex];
         fc_phone_tclSM::vCopyCallData(&CallData, pCallInstance);
         pCallInstance->m_u8ObjectState = FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE;

         ETG_TRACE_USR3((" vHandle_AcceptCall_MethodStart::vStartSM "));
         m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_ACCEPTCALL , pCallInstance);
      }
   }

   else if (OSAL_ERROR == s32CIStatusActiveOrPassive)
   {
      /* Invalid CallInstance Requested, So send an error message */
      ETG_TRACE_USR4(( "vHandle_AcceptCall_MethodStart: Invalid CallInstance Requested" ));
      vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   if(poAcceptCall_MS)
   {
      poAcceptCall_MS->vDestroy();
      delete poAcceptCall_MS;
   }
   delete l_poJob;//delete job pointer after use
   //End of fix
}

/*******************************************************************************
** FUNCTION:   vSwapHoldCallStatus(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a function for swapping the call states for the hold call
*
*  PARAMETERS: None
*
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vSwapHoldCallStatus()
{
   ETG_TRACE_USR4((" ENTER :fc_phone_tclService_Telephone::vSwapHoldCallStatus entered"));
   if (fc_phone_tclCallManager::m_u16Count > 1)
   {
      for (tU8 u8Index = 0x00; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
      {
         if(FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD == m_oCallInstances[u8Index].m_u16CallStatusNotice)
         {
            m_oCallInstances[u8Index].m_u16CallStatusNotice = FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE;
         }
         else if(FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE == m_oCallInstances[u8Index].m_u16CallStatusNotice)
         {
            m_oCallInstances[u8Index].m_u16CallStatusNotice = FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD;
         }
      }
   }
}

/*******************************************************************************
** FUNCTION:   vHandle_RejectCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for AccepCall method
*   Thie function talks to dbus_client to accept call
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_RejectCall_MethodStart(amt_tclServiceData* poMessage)
{
   /* Extract the Call instance and other required info from poMessage */
   ETG_TRACE_USR4((" ENTER :fc_phone_tclService_Telephone::vHandle_RejectCall_MethodStart "));
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_telfi_tclMsgRejectCallMethodStart oRejectCall_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( (OSAL_ERROR != oVisitorMsg.s32GetData(oRejectCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) &&(FALSE == m_bDiagnosticModeFlag) )
   {
      /* Get the coresponding Call Instance */
      tU16 u16Id = 0x00;
      tS32 s32CIStatusActiveOrPassive = s32GetCallInstanceIndexFromFiCallInstance(oRejectCall_MS.u16CallInstance, u16Id);

      m_u16StoredCallInstanceForReject = oRejectCall_MS.u16CallInstance;

      if ((OSAL_OK == s32CIStatusActiveOrPassive) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
      {
         ETG_TRACE_USR1(("*****MethodStart:RejectCall*****"));
         ETG_TRACE_USR1((" u16CallInstance: %u",oRejectCall_MS.u16CallInstance));

         // fc_phone_tclSMManager.vStartSM(FC_PHONE_SMMANAGER_SMID_ACCEPTCALL , m_oCallInstances[u16Id]);
         m_oRejectCallMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
         m_oRejectCallMessage.u16RegisterID  = poMessage->u16GetRegisterID();
         m_oRejectCallMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
         m_oRejectCallMessage.u16FunctionID  = poMessage->u16GetFunctionID();

         fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];
         m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_REJECTCALL , pCallInstance);
      }

      else if (true == s32CIStatusActiveOrPassive)
      {
         ETG_TRACE_USR4(("Call Instance belongs to the Passive Device"));
         ETG_TRACE_USR2(("Received FICallInstance = %d",oRejectCall_MS.u16CallInstance)) ;
         ETG_TRACE_USR2(("Device Address Corressponding to the received call instance : %s", CallStatusList[oRejectCall_MS.u16CallInstance].BD_Address));

         vRequestEndCallBasedOnCallInstance(oRejectCall_MS.u16CallInstance);
      }

      else
      {
         /* Invalid CallInstance Requested, So send an error message */
         ETG_TRACE_ERR(( "vHandle_RejectCall_MethodStart: Invalid CallInstance Requested" ));
         vHandle_RejectCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_RejectCall_MethodStart: Received message invalid." ));
      // We never expect to come here.
      vHandle_RejectCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oRejectCall_MS.vDestroy();
}


/*******************************************************************************
** FUNCTION:   vHandle_HangUpCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for AccepCall method
*   Thie function talks to dbus_client to accept call
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_HangUpCall_MethodStart(amt_tclServiceData* poMessage, struct Job* f_poJob)
{
   ETG_TRACE_USR4((" ENTER :fc_phone_tclService_Telephone::vHandle_HangUpCall_MethodStart "));
   /* Extract the Call instance and other required info from poMessage */
   tBool bVehicleAudioStatus = FALSE;

   // Create the (message related) FI data object
   most_telfi_tclMsgHangUpCallMethodStart *poHangUpCall_MS = NULLPTR;

   //FIX GMMY15-9676 [CT][MY15 GMT166 M2 Telephone] When END/MUTE button is pressed on the SWC while in a Multiple Active Calls (merged), only the 1st call is disconnected while the 2nd call was set to handset.
   //Second request sent to BT Daemon from FC_Phone before receiving response of former request is responded with error by BT Daemon for some method requests.
   //For those requests for which API restrictions are defined in ALPS document, FC_Phone tries to acquire a mutex lock. If acquired then only request is forwarded to DBUS. Else it is added to a work queue.
   struct Job* l_poJob = NULLPTR;

   if(poMessage)//Direct call from client, not from work queue
   {
      // Create a FI visitor message for the received CCA message
      fi_tclVisitorMessage oVisitorMsg( poMessage );
      poHangUpCall_MS = new most_telfi_tclMsgHangUpCallMethodStart;

      if(poHangUpCall_MS)
      {
         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data object.
         if ( (OSAL_ERROR != oVisitorMsg.s32GetData(*poHangUpCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION) ) && (FALSE == m_bDiagnosticModeFlag) )
         {
            ETG_TRACE_USR4((" m_u16StoredCallInstanceForHangUp: %d",m_u16StoredCallInstanceForHangUp));
            ETG_TRACE_USR4((" poHangUpCall_MS->u16CallInstance: %d",poHangUpCall_MS->u16CallInstance));

            //If same request comes again to FC_Phone, it is not processed. HangUp request returns giving error.
            if( m_u16StoredCallInstanceForHangUp == poHangUpCall_MS->u16CallInstance )
            {
               ETG_TRACE_ERR(( "vHandle_HangUpCall_MethodStart: Already received message under process. Rejecting duplicates." ));
               vHandle_HangUpCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

               poHangUpCall_MS->vDestroy();
               delete poHangUpCall_MS;
               return;
            }
            else
            {
               //If same request comes again to FC_Phone we dont push them to queue. Hangup request returns giving error.
               //Store current parameter to check duplicate requests which may come afterwards.
               m_u16StoredCallInstanceForHangUp = poHangUpCall_MS->u16CallInstance;
               ETG_TRACE_USR4((" m_u16StoredCallInstanceForHangUp: %d",m_u16StoredCallInstanceForHangUp));
            }

            //Create a new job
            ETG_TRACE_USR4((" Creating new job"));
            l_poJob = new Job;
            if(l_poJob)//If creation of job is successful, assign parameters
            {
               l_poJob->m_pfMethod  = &fc_phone_tclService_Telephone::vHandle_HangUpCall_MethodStart;
               l_poJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               l_poJob->u16FunctionID = poMessage->u16GetFunctionID();
               l_poJob->u16RegisterID = poMessage->u16GetRegisterID();
               l_poJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               l_poJob->m_poMethodStart = poHangUpCall_MS;
            }
            //GMMY16-14585
            /* When Conference Call is ended from cluster, 2 Hangupcall method start is received from the cluster continuously.
            The Hangup call for CL 0 is processed while the other is pushed in to work queue with CI value 255.
            Since unlock of the queue is done when callstatus is received as IDLE instead of DBUS-ACK of hangup call,
            the second Hangup call method start should be pushed into the queue with callinstance value 1 instead of 255.*/

         }
         else
         {
            ETG_TRACE_ERR(("vHandle_HangUpCall_MethodStart: Received message invalid."));
            vHandle_HangUpCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            poHangUpCall_MS->vDestroy();
            delete poHangUpCall_MS;
            return;
         }
      }
      else
      {
         ETG_TRACE_ERR(( "poHangUpCall_MS is NULL" ));
      }
   }
   else if(f_poJob)//Received from WorkQueue, not directly from client
   {
      ETG_TRACE_USR4((" Received from queue"));
      l_poJob = f_poJob;
   }
   else
   {
      ETG_TRACE_USR4((" Should not come here!!"));
      return; // Should not come here
   }

   if(!l_poJob)
   {
      ETG_TRACE_ERR(( "l_poJob is NULL" ));
      // We never expect to come here.
      vHandle_HangUpCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

      if(poHangUpCall_MS)
      {
         poHangUpCall_MS->vDestroy();
         delete poHangUpCall_MS;
      }
      return;
   }

   if (fc_phone_tclPhoneMutex::PhoneMutexTrylock(l_poJob) != 0)//Try to acquire lock
   {
      tVoid*  pCustodialLose = l_poJob ;  //To remove LINT warning
      (tVoid) pCustodialLose ;
      return; // Didn't get the lock :(
   }

   //No need to clear the previously stored values. If value cleared, next request from HMI(duplicate) would enter the queue (which is not recommended)
   // and the subsequent requests after that only will be rejected .
   //m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM ;

   //If lock is acquired process the request
   m_oHangUpCallMessage.u16SourceAppID = l_poJob->u16SourceAppID;
   m_oHangUpCallMessage.u16RegisterID = l_poJob->u16RegisterID;
   m_oHangUpCallMessage.u16CmdCounter = l_poJob->u16CmdCounter;
   m_oHangUpCallMessage.u16FunctionID = l_poJob->u16FunctionID;


   poHangUpCall_MS = (most_telfi_tclMsgHangUpCallMethodStart*)   l_poJob->m_poMethodStart;
   NULL_CHECK(poHangUpCall_MS);
   /* Get the coresponding Call Instance */
   ETG_TRACE_USR1((" ***** MethodStart:HangUpCall *****"));
   ETG_TRACE_USR1(("u16CallInstance: %u",poHangUpCall_MS->u16CallInstance));

   tU16 u16Id = 0x00;
   tS32 s32CIStatusActiveOrPassive = s32GetCallInstanceIndexFromFiCallInstance(poHangUpCall_MS->u16CallInstance, u16Id);

   tU16 u16HangUpCItoCheckActiveCall = FC_PHONE_HANGUP_ALL;

   if(FC_PHONE_HANGUP_ALL == poHangUpCall_MS->u16CallInstance)
   {
      fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[0];
      if(pCallInstance->m_bUsingVehicleAudio == TRUE)
      {
         bVehicleAudioStatus = TRUE;
      }
   }
   else if (OSAL_OK == s32CIStatusActiveOrPassive)
   {
      fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];
      if(pCallInstance->m_bUsingVehicleAudio == TRUE)
      {
         bVehicleAudioStatus = TRUE;
      }
      u16HangUpCItoCheckActiveCall = (poHangUpCall_MS->u16CallInstance)%2;
   }

   else if (true == s32CIStatusActiveOrPassive)
   {
      ETG_TRACE_USR4(("Call Instance belongs to the Passive Device"));
      ETG_TRACE_USR2(("Received FICallInstance = %d",poHangUpCall_MS->u16CallInstance)) ;
      ETG_TRACE_USR2(("Device Address Corressponding to the received call instance : %s", CallStatusList[poHangUpCall_MS->u16CallInstance].BD_Address));

      vRequestEndCallBasedOnCallInstance(poHangUpCall_MS->u16CallInstance);

      if(poHangUpCall_MS)
      {
         poHangUpCall_MS->vDestroy();
         delete poHangUpCall_MS;
      }
      delete l_poJob;
      return;
   }

   ETG_TRACE_USR2(("bVehicleAudioStatus = %d",bVehicleAudioStatus)) ;
   ETG_TRACE_USR2(("(poHangUpCall_MS->u16CallInstance)mod(2) = %d",((poHangUpCall_MS->u16CallInstance)%2))) ;

   NULL_CHECK(m_poCallManager)
      if(m_poCallManager->bCheckCallActive(u16HangUpCItoCheckActiveCall))
      {
         //for hangupALL
         if(FC_PHONE_HANGUP_ALL == poHangUpCall_MS->u16CallInstance )
         {
            tBool bEventConsumed = FALSE ;

            fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[0];
            for (tU8 ind = 0; ind < FC_PHONE_MAX_CALLINSTANCES; ++ind)
            {
               if(m_oCallInstances[ind].m_u8ObjectState == FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE)
               {
                  pCallInstance =  &m_oCallInstances[ind];
                  break;
               }
            }
            pCallInstance->m_HangupALL_flag = TRUE ;

            //fill process Event Arguments and send processEvent
            FcPhone_processEventArg   tProcessEventArg;
            tProcessEventArg.bIsItDbusAck =  FALSE;
            // when two calls exist, if event broadcasted, it will be consumed by both the statemachine and hf_terminateall will be posted twice.
            NULL_CHECK(m_poSMManager)
               bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_HANGUPCALLALL_OK,FC_PHONE_EVENTTYPE_BROADCAST_CONSUMED,&tProcessEventArg);

            if (!bEventConsumed)
            {
               m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_HANGUPCALL , pCallInstance);
            }
         }
         //for one callInstance Hangup
         else if (OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(poHangUpCall_MS->u16CallInstance, u16Id))
         {
            tBool bEventConsumed = FALSE ;
            fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];
            pCallInstance->m_HangupALL_flag = FALSE ;

            //fill process Event Arguments and send processEvent
            FcPhone_processEventArg   tProcessEventArg;
            tProcessEventArg.bIsItDbusAck =  FALSE;
            tProcessEventArg.u16CallInstanceID = u16Id ;

            NULL_CHECK(m_poSMManager)
               bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_HANGUPCALL_OK,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);

            if (!bEventConsumed)
            {
               m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_HANGUPCALL , pCallInstance);
            }
         }
      }
      else
      {
         //Sending Method Error for Hangup since Call instance doesn't exist
         ETG_TRACE_ERR(( "vHandle_HangUpCall_MethodStart: Call instance doesn't exist" ));
         vHandle_HangUpCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
      }

      // Always destroy the (message related) FI data object (before leaving
      // its creation scope)
      if(poHangUpCall_MS)
      {
         poHangUpCall_MS->vDestroy();
         delete poHangUpCall_MS;
      }
      delete l_poJob;//delete job pointer after use
      //End of fix
}


/*******************************************************************************
** FUNCTION:   vHandle_IgnoreCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for Ignorecall method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_IgnoreCall_MethodStart(amt_tclServiceData* poMessage,
      struct Job* f_poJob)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vHandle_IgnoreCall_MethodStart entered"));

   // Create the (message related) FI data object
   most_telfi_tclMsgIgnoreCallMethodStart *oIgnoreCall_MS = NULLPTR;

   struct Job* poJob = NULLPTR;

   if (poMessage)
   {
      /* Extract the Call instance and other required info from poMessage */
      // Create a FI visitor message for the received CCA message
      fi_tclVisitorMessage oVisitorMsg(poMessage);
      oIgnoreCall_MS = new most_telfi_tclMsgIgnoreCallMethodStart;

      if (oIgnoreCall_MS)
      {
         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.
         if ((OSAL_ERROR != oVisitorMsg.s32GetData(*oIgnoreCall_MS,
               FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
         {
            ETG_TRACE_USR4((" m_u16StoredCallInstanceForIgnore : %d", m_u16StoredCallInstanceForIgnore));
            // If same request comes again to FC_Phone, it is not processed. Accept request returns giving error.
            if (m_u16StoredCallInstanceForIgnore == oIgnoreCall_MS->u16CallInstance)
            {
               ETG_TRACE_ERR(("vHandle_IgnoreCall_MethodStart: Already received message under process. "
                     "Rejecting duplicates."));
               vHandle_IgnoreCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
               oIgnoreCall_MS->vDestroy();
               delete oIgnoreCall_MS;
               return;
            }
            else
            {
               /* If same request comes again to FC_Phone we dont push them to queue. Ignore request returns
                * giving error. Store current parameter to check duplicate requests which may come afterwards. */
               m_u16StoredCallInstanceForIgnore = oIgnoreCall_MS->u16CallInstance;
               ETG_TRACE_USR4(("m_u16StoredCallInstanceForIgnore %d", m_u16StoredCallInstanceForIgnore));
            }

            //Create a new job
            poJob = new Job;
            if (poJob) //If creation of job is successful, assign parameters
            {
               poJob->m_pfMethod = &fc_phone_tclService_Telephone::vHandle_IgnoreCall_MethodStart;
               poJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               poJob->u16FunctionID = poMessage->u16GetFunctionID();
               poJob->u16RegisterID = poMessage->u16GetRegisterID();
               poJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               poJob->m_poMethodStart = oIgnoreCall_MS;
            }
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_IgnoreCall_MethodStart: Received message invalid."));
            // We never expect to come here.
            vHandle_IgnoreCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            oIgnoreCall_MS->vDestroy();
            delete oIgnoreCall_MS;
            return;
         }
      }
   }
   else if (f_poJob)
   {
      poJob = f_poJob;
   }
   else
   {
      return; // Should not come here
   }

   if (!poJob)
   {
      ETG_TRACE_ERR(("poJob is NULL!!"));
      vHandle_IgnoreCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob);

      if (oIgnoreCall_MS)
      {
         oIgnoreCall_MS->vDestroy();
         delete oIgnoreCall_MS;
      }
      return;
   }

   if (fc_phone_tclPhoneMutex::PhoneMutexTrylock(poJob) != 0) //Try to acquire lock
   {
      tVoid* pCustodialLose = poJob; //To remove LINT warning
      (tVoid) pCustodialLose;
      return; // Didn't get the lock :(
   }

   //If lock is acquired process the request
   m_oIgnoreCallMessage.u16SourceAppID = poJob->u16SourceAppID;
   m_oIgnoreCallMessage.u16RegisterID = poJob->u16RegisterID;
   m_oIgnoreCallMessage.u16CmdCounter = poJob->u16CmdCounter;
   m_oIgnoreCallMessage.u16FunctionID = poJob->u16FunctionID;


   oIgnoreCall_MS = (most_telfi_tclMsgIgnoreCallMethodStart*) poJob->m_poMethodStart;
   NULL_CHECK(oIgnoreCall_MS);
   /* Get the coresponding Call Instance */
   tU16 u16Id = 0x00;
   tS32 s32CIStatusActiveOrPassive = s32GetCallInstanceIndexFromFiCallInstance(oIgnoreCall_MS->u16CallInstance, u16Id);
   if ((OSAL_OK == s32CIStatusActiveOrPassive) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
   {
      ETG_TRACE_USR1((" ***** MethodStart:oIgnoreCall ***** "));
      ETG_TRACE_USR1((" u16CallInstance: %u u16Id: %d", oIgnoreCall_MS->u16CallInstance, u16Id));

      NULL_CHECK(m_poSMManager)

      //start state machine for ignore call
      fc_phone_tclCallInstance *pCallInstance = &m_oCallInstances[u16Id];

      if ((FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE == pCallInstance->m_u8ObjectState) &&
            (FC_PHONE_CALLINSTANCE_INCOMINGCALLEVENT_RINGING == pCallInstance->m_u16IncomingCallEvent))
      {
         ETG_TRACE_USR3(("Creating Ignore SM"));
         m_poSMManager->vStartSM(FC_PHONE_SMMANAGER_SMID_IGNORECALL, pCallInstance);
      }
      else
      {
         ETG_TRACE_USR4(("Sending IgnoreCall METHOD_ERROR to HMI."));
         /* Invalid CallInstance Requested, So send an error message */
         ETG_TRACE_USR4(("vHandle_IgnoreCall_MethodStart: NOT Creating Ignore SM, As Input parameter "
               "is wrong, for this callInstanceId no Incoming Event found.."));
         vHandle_IgnoreCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
      }
   }

   else if (true == s32CIStatusActiveOrPassive)
   {
      ETG_TRACE_USR4(("Call Instance belongs to the Passive Device"));
      ETG_TRACE_USR2(("Received FICallInstance = %d",oIgnoreCall_MS->u16CallInstance)) ;
      ETG_TRACE_USR2(("Device Address Corressponding to the received call instance : %s", CallStatusList[oIgnoreCall_MS->u16CallInstance].BD_Address));

      vRequestEndCallBasedOnCallInstance(oIgnoreCall_MS->u16CallInstance);
   }

   else
   {
      /* Invalid CallInstance Requested, So send an error message */
      ETG_TRACE_USR4(("vHandle_IgnoreCall_MethodStart: Invalid CallInstance Requested"));
      vHandle_IgnoreCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   if (oIgnoreCall_MS)
   {
      oIgnoreCall_MS->vDestroy();
      delete oIgnoreCall_MS;
   }
   delete poJob;
}


/*******************************************************************************
** FUNCTION:   vHandle_HoldCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_HoldCall_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_HoldCall_MethodStart(amt_tclServiceData* poMessage) //
{
   /* Extract the Call instance and other required info from poMessage */
   ETG_TRACE_USR4((" ENTER :fc_phone_tclService_Telephone::vHandle_HoldCall_MethodStart "));
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_telfi_tclMsgHoldCallMethodStart oHoldCall_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if (( OSAL_ERROR != oVisitorMsg.s32GetData(oHoldCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
   {
      //FIX GMMY15-8315 CT]Cannot place conference call on hold
      //When Hold call method start is received in FC_Phone with call instance value as FF, then the conference call should be put on hold.
      if( FC_PHONE_HOLD_ALL == oHoldCall_MS.u16CallInstance)
      {
         oHoldCall_MS.u16CallInstance = 0;//If request is to hold all calls then assign call instance as 0 or 1 (0 assigned here) as the same is required for further execution in FC_Phone
         m_bHoldOrResumeAllFlag = TRUE;
         ETG_TRACE_USR4((" Updated oHoldCall_MS.u16CallInstance: %d ", oHoldCall_MS.u16CallInstance));
      }
      //End of fix

      /* Get the coresponding Call Instance */
      tU16 u16Id = 0x00;
      if (OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oHoldCall_MS.u16CallInstance, u16Id) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
      {
         ETG_TRACE_USR1((" ***** MethodStart:oHoldCall_MS ***** "));
         ETG_TRACE_USR1(("   u16CallInstance: %u", oHoldCall_MS.u16CallInstance));
         //start state machine for ignore call
         fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];

         if((FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE == pCallInstance->m_u8ObjectState)
            &&((FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE ==  pCallInstance->m_u16CallStatusNotice)
            || (FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE ==  pCallInstance->m_u16CallStatusNotice)
            || (FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE == pCallInstance->m_u16CallStatusNotice)) // TODO : have to take care in FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE implementation
            )
         {
            m_oHoldCallMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
            m_oHoldCallMessage.u16RegisterID  = poMessage->u16GetRegisterID();
            m_oHoldCallMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
            m_oHoldCallMessage.u16FunctionID  = poMessage->u16GetFunctionID();

            //fill process Event Arguments and send processEvent
            FcPhone_processEventArg   tProcessEventArg ;
            tProcessEventArg.bIsItDbusAck =  FALSE ;
            tProcessEventArg.u16CallInstanceID = u16Id ;
            m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_HOLDCALL_REQUEST , FC_PHONE_EVENTTYPE_TARGATED ,&tProcessEventArg);
         }
         else
         {
            //send error message : no call is active
            ETG_TRACE_ERR(( "vHandle_HoldCall_MethodStart: no call is active." ));
            vHandle_HoldCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_HoldCall_MethodStart: Invalid CallInstance Requested." ));
         vHandle_HoldCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_HoldCall_MethodStart: Received message invalid." ));
      // We never expect to come here.
      vHandle_HoldCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oHoldCall_MS.vDestroy();
}

/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vRequestEndCallBasedOnCallInstance
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vRequestEndCallBasedOnCallInstance(tU16 CallInstanceID)
{
   ETG_TRACE_USR2((" vRequestEndCallBasedOnCallInstance entered with CallInstanceID:: %d", CallInstanceID));
   if(CallInstanceID >= CallStatusList.size())
   {
      ETG_TRACE_USR4((" CallInstanceID is INVALID. Should be less than CallStatusList Vector size "));
      return;
   }
   ETG_TRACE_USR2((" call status: %d", CallStatusList[CallInstanceID].CallStatus.u8CallState));

   // As per the current requirement, there can only be 2 calls even with 2 HFP devices connected to the target.
   // Hangupcall for the Passive device will be called by the HMI, only when the AcceptCall is called for the Passive device.
   // After which the Passive device is switched to Active, and now the recently Passive device's call should be ended by HMI

   // As of now, if Hangup call is called for the Passive device, FC_Phone calls HangupAll interface for the Passive device
   fc_phone_tclSM::m_pDbusIf->bHangUpCall(CallStatusList[CallInstanceID].BD_Address, FC_PHONE_DBUS_TERMINATETYPE_ALL);
   m_bEndCallforPassiveDev = true;
}

/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSendCellDeviceName
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vSendCellDeviceName(tU8 u8BDAddressID)
{
   ETG_TRACE_USR3((" Entered vSendCellDeviceName and u8BDAddressID : %d", u8BDAddressID));

   m_u8CurrentBDAddressIDforUpdate = u8BDAddressID;
   ETG_TRACE_USR3((" vSendCellDeviceName::CellDeviceName : %s", m_oPhoneData[u8BDAddressID].m_achCellDeviceName));

   ail_tenCommunicationError enResult =  eUpdateClients(MOST_TELFI_C_U16_CELLDEVICENAME);
   if(AIL_EN_N_NO_ERROR != enResult)
   {
      ETG_TRACE_ERR(("vSendCellDeviceName: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u",(tU32)enResult));
   }
   m_u8CurrentBDAddressIDforUpdate = INVALID_BDADDRESS_ID;

}


/*******************************************************************************
** FUNCTION:   vHandle_ResumeCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_ResumeCall_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_ResumeCall_MethodStart(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("  ENTERING :fc_phone_tclService_Telephone::vHandle_ResumeCall_MethodStart "));
   /* Extract the Call instance and other required info from poMessage */

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_telfi_tclMsgResumeCallMethodStart oResumeCall_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ((OSAL_ERROR != oVisitorMsg.s32GetData(oResumeCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
   {
      //FIX GMMY15-8315 CT]Cannot place conference call on hold
      //When Hold call method start is received in FC_Phone with call instance value as FF, then the conference call should be put on hold.
      if( FC_PHONE_RESUME_ALL == oResumeCall_MS.u16CallInstance)
      {
         oResumeCall_MS.u16CallInstance = 0;//If request is to resume all calls then assign call instance as 0 or 1 (0 assigned here) as the same is required for further execution in FC_Phone
         m_bHoldOrResumeAllFlag = TRUE;
         ETG_TRACE_USR4((" Updated oHoldCall_MS.u16CallInstance: %d ", oResumeCall_MS.u16CallInstance));
      }
      //End of fix

      /* Get the coresponding Call Instance */
      tU16 u16Id = 0x00;

      if (OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oResumeCall_MS.u16CallInstance, u16Id) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
      {
         ETG_TRACE_USR1(("*****  MethodStart:oResumeCall_MS u16CallInstance: %u ", oResumeCall_MS.u16CallInstance));

         //start state machine for ignore call
         fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];

         if(FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD ==  pCallInstance->m_u16CallStatusNotice)
         {
            m_oResumeCallMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
            m_oResumeCallMessage.u16RegisterID  = poMessage->u16GetRegisterID();
            m_oResumeCallMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
            m_oResumeCallMessage.u16FunctionID  = poMessage->u16GetFunctionID();

            //fill process Event Arguments and send processEvent
            FcPhone_processEventArg   tProcessEventArg ;
            tProcessEventArg.bIsItDbusAck =  FALSE ;
            tProcessEventArg.u16CallInstanceID = u16Id ;
            m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_RESUMECALL_REQUEST ,FC_PHONE_EVENTTYPE_TARGATED, &tProcessEventArg);
         }
         // Bugfix GMMY15-6638 - If the call is already active do not resume the call again
         else if(FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE ==  pCallInstance->m_u16CallStatusNotice)
         {
            vSendMethodAck(MOST_TELFI_C_U16_RESUMECALL, pCallInstance);
         }
         else
         {
            ETG_TRACE_ERR((" Sending Error To Client: this callInstance is not on hold "));
            // We never expect to come here.

            vHandle_ResumeCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERNOTAVAILABLE, poMessage);
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_ResumeCall_MethodStart: Invalid CallInstance Requested." ));
         vHandle_ResumeCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_ResumeCall_MethodStart: Received message invalid." ));
      vHandle_ResumeCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oResumeCall_MS.vDestroy();

   ETG_TRACE_USR4((" EXITING :fc_phone_tclService_Telephone::vHandle_ResumeCall_MethodStart "));
}


/*******************************************************************************
** FUNCTION:   vHandle_Dial_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for Dial_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_Dial_MethodStart(amt_tclServiceData* poMessage, struct Job* f_poJob)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vHandle_Dial_MethodStart entered"));

   /* Extract the Call instance and other required info from poMessage */
   // Create the (message related) FI data object
   most_telfi_tclMsgDialMethodStart *poDial_MS = NULLPTR;

   //FIX GMMY15-10013 [CT]Call is dialed out even when end is selected
   //FIX GMMY15-10101 [CT]Next Gen HMI - Issue with unable to answer the call, when the user tries to make a call at the same time
   //Second request sent to BT Daemon from FC_Phone before receiving response of former request is responded with error by BT Daemon for some method requests.
   //For those requests for which API restrictions are defined in ALPS document, FC_Phone tries to acquire a mutex lock. If acquired then only request is forwarded to DBUS. Else it is added to a work queue.

   struct Job* l_poJob = NULLPTR;
   if(poMessage)//Direct call from client, not from work queue
   {
      // Create a FI visitor message for the received CCA message
      fi_tclVisitorMessage oVisitorMsg( poMessage );

      poDial_MS = new most_telfi_tclMsgDialMethodStart;

      if(poDial_MS)
      {
         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.

         if (( OSAL_ERROR != oVisitorMsg.s32GetData(*poDial_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
         {

            tBool isTelephoneNumberEmpty = FALSE;
            validateTelephoneNumber(poDial_MS->sTelephoneNumber.szValue,isTelephoneNumberEmpty);

            if (TRUE == isTelephoneNumberEmpty)
            {
               ETG_TRACE_ERR(( "vHandle_Dial_MethodStart: Telephone number empty" ));
               vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
               poDial_MS->vDestroy();
               delete poDial_MS;
               return;
            }

            //If telephone number is valid process the request
            ETG_TRACE_USR4((" Creating new job"));
            //Create a new job
            l_poJob = new Job;

            if(l_poJob)//If creation of job is successful, assign parameters
            {
               l_poJob->m_pfMethod  = &fc_phone_tclService_Telephone::vHandle_Dial_MethodStart;
               l_poJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               l_poJob->u16FunctionID = poMessage->u16GetFunctionID();
               l_poJob->u16RegisterID = poMessage->u16GetRegisterID();
               l_poJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               l_poJob->m_poMethodStart = poDial_MS;
            }
         }
         else
         {
            ETG_TRACE_ERR(( "vHandle_Dial_MethodStart: Received message invalid." ));
            // We never expect to come here.
            // NORMAL_M_ASSERT_ALWAYS();
            vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

            poDial_MS->vDestroy();
            delete poDial_MS;
            return;
         }
      }
      else
      {
         ETG_TRACE_ERR(( "poDial_MS is NULL" ));
         vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         return;
      }
   }
   else if(f_poJob)//Received from WorkQueue, not directly from client
   {
      ETG_TRACE_USR4((" Received from queue"));
      l_poJob = f_poJob;
   }
   else
   {
      ETG_TRACE_USR4((" Should not come here!!"));
      return; // Should not come here
   }

   if(!l_poJob)
   {
      ETG_TRACE_ERR(( "l_poJob is NULL" ));
      vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      if(poDial_MS)
      {
         poDial_MS->vDestroy();
         delete poDial_MS;
      }
      return;
   }

   if (fc_phone_tclPhoneMutex::PhoneMutexTrylock(l_poJob, FC_PHONE_WORK_QUEUE_TIMER_VALUE_FOR_DIAL) != 0)//Try to acquire lock
   {
      tVoid*  pCustodialLose = l_poJob ;  //To remove LINT warning
      (tVoid) pCustodialLose ;
      return; // Didn't get the lock :(
   }

   //If lock is acquired process the job
   m_oDialMessage.u16SourceAppID = l_poJob->u16SourceAppID;
   m_oDialMessage.u16RegisterID  = l_poJob->u16RegisterID;
   m_oDialMessage.u16CmdCounter  = l_poJob->u16CmdCounter;
   m_oDialMessage.u16FunctionID  = l_poJob->u16FunctionID;


   poDial_MS = (most_telfi_tclMsgDialMethodStart*)   l_poJob->m_poMethodStart;
   NULL_CHECK(poDial_MS);
   tU16 u16Id = 0x00;
   tS32 s32Error = OSAL_ERROR;

   tU8 u8_ActiveBDAddressId=INVALID_BDADDRESS_ID ;
   tBool b_getAdressSuccess = FALSE ;
   FcPhoneDial_InputParamStruct st_dialInParam ;

   ETG_TRACE_USR1(("*****MethodStart:m_oDialMessage*****"));

   // CMG3GB-867 - hf-dial should be triggered even without sim for emergency calls.

   //get Free call instance
   s32Error = s32GetIdleCallInstanceIndex(u16Id);

   if (OSAL_ERROR == s32Error)
   {
      ETG_TRACE_ERR(( "vHandle_Dial_MethodStart: Both call instances occupied." ));
      vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
   }
   else
   {
      ETG_TRACE_USR2((" got idle callInstance: %d ",u16Id));

      //Check for server voice recognition mode
      ecnrServerVoiceMode(poDial_MS->e8EchoCancellationNoiseReductionSetting.enType);

      //start state machine for ignore call
      fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];

      b_getAdressSuccess =  b_get_ActiveDevice_AddressId_FromBT(&u8_ActiveBDAddressId);

      if ((TRUE == b_getAdressSuccess) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
      {
         pCallInstance->vCreate(u16Id,u8_ActiveBDAddressId) ;

         //Start SM for Dial/SpeedDial
         st_dialInParam.en_dialType = FC_PHONE_DIALTYPE_DIAL ;
         st_dialInParam.pu8_phoneNumber = (tU8*)m_sValidPhoneNumber;
         m_bDialMethodStartFlag = TRUE;
         m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_DIAL , pCallInstance,&st_dialInParam);
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_Dial_MethodStart: Either m_oSLCStatusStatus is false or Telephone number empty" ));
         vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
      }
   }

   for(tU8 index = 0; index< (FC_PHONE_DIAL_NUMBER_MAX_LEN_IN_CHARS+1); index++)
   {
      m_sValidPhoneNumber[index] = 0;//Reset to default after use
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   if(poDial_MS)
   {
      poDial_MS->vDestroy();
      delete poDial_MS;
   }
   delete l_poJob;//delete job pointer after use
   //End of fix

   ETG_TRACE_USR4((" EXITTINGGGG :  fc_phone_tclService_Telephone::vHandle_Dial_MethodStart "));
}


/*******************************************************************************
** FUNCTION:   vHandle_DialExtended_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for DialExtended_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_DialExtended_MethodStart(amt_tclServiceData* poMessage, struct Job* f_poJob)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vHandle_DialExtended_MethodStart entered"));

   /* Extract the Call instance and other required info from poMessage */
   // Create the (message related) FI data object
   most_telfi_tclMsgDialExtendedMethodStart *poDiaExtended_MS = NULLPTR;

   //FIX GMMY15-10013 [CT]Call is dialed out even when end is selected
   //FIX GMMY15-10101 [CT]Next Gen HMI - Issue with unable to answer the call, when the user tries to make a call at the same time
   //Second request sent to BT Daemon from FC_Phone before receiving response of former request is responded with error by BT Daemon for some method requests.
   //For those requests for which API restrictions are defined in ALPS document, FC_Phone tries to acquire a mutex lock. If acquired then only request is forwarded to DBUS. Else it is added to a work queue.

   struct Job* l_poJob = NULLPTR;
   if(poMessage)//Direct call from client, not from work queue
   {
      // Create a FI visitor message for the received CCA message
      fi_tclVisitorMessage oVisitorMsg( poMessage );

      poDiaExtended_MS = new(std::nothrow) most_telfi_tclMsgDialExtendedMethodStart;

      if(!poDiaExtended_MS)
      {
         ETG_TRACE_ERR(( "poDiaExtended_MS is NULL" ));
         vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         return;
      }

         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.

      if (( OSAL_ERROR != oVisitorMsg.s32GetData(*poDiaExtended_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
      {

         tBool isTelephoneNumberEmpty = FALSE;
         validateTelephoneNumber(poDiaExtended_MS->sTelephoneNumber.szValue,isTelephoneNumberEmpty);

         if (TRUE == isTelephoneNumberEmpty)
         {
            ETG_TRACE_ERR(( "vHandle_DialExtended_MethodStart: Telephone number empty" ));
            vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            poDiaExtended_MS->vDestroy();
            delete poDiaExtended_MS;
            return;
         }

         //If telephone number is valid process the request
         ETG_TRACE_USR4((" Creating new job"));
         //Create a new job
         l_poJob = new Job;

         if(l_poJob)//If creation of job is successful, assign parameters
         {
            l_poJob->m_pfMethod  = &fc_phone_tclService_Telephone::vHandle_DialExtended_MethodStart;
            l_poJob->u16CmdCounter = poMessage->u16GetCmdCounter();
            l_poJob->u16FunctionID = poMessage->u16GetFunctionID();
            l_poJob->u16RegisterID = poMessage->u16GetRegisterID();
            l_poJob->u16SourceAppID = poMessage->u16GetSourceAppID();
            l_poJob->m_poMethodStart = poDiaExtended_MS;
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_DialExtended_MethodStart: Received message invalid." ));
         // We never expect to come here.
         // NORMAL_M_ASSERT_ALWAYS();
         vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

         poDiaExtended_MS->vDestroy();
         delete poDiaExtended_MS;
         return;
      }

   }
   else if(f_poJob)//Received from WorkQueue, not directly from client
   {
      ETG_TRACE_USR4((" Received from queue"));
      l_poJob = f_poJob;
   }
   else
   {
      ETG_TRACE_USR4((" Should not come here!!"));
      return; // Should not come here
   }

   if(!l_poJob)
   {
      ETG_TRACE_ERR(( "l_poJob is NULL" ));
      vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      if (poDiaExtended_MS)
      {
         poDiaExtended_MS->vDestroy();
         delete poDiaExtended_MS;
      }
      return;
   }

   if (fc_phone_tclPhoneMutex::PhoneMutexTrylock(l_poJob, FC_PHONE_WORK_QUEUE_TIMER_VALUE_FOR_DIAL) != 0)//Try to acquire lock
   {
      tVoid*  pCustodialLose = l_poJob ;  //To remove LINT warning
      (tVoid) pCustodialLose ;
      return; // Didn't get the lock :(
   }

   //If lock is acquired process the job
   m_oDialMessage.u16SourceAppID = l_poJob->u16SourceAppID;
   m_oDialMessage.u16RegisterID  = l_poJob->u16RegisterID;
   m_oDialMessage.u16CmdCounter  = l_poJob->u16CmdCounter;
   m_oDialMessage.u16FunctionID  = l_poJob->u16FunctionID;



   poDiaExtended_MS = (most_telfi_tclMsgDialExtendedMethodStart*)   l_poJob->m_poMethodStart;
   NULL_CHECK(poDiaExtended_MS);
   tU16 u16Id = 0x00;
   tS32 s32Error = OSAL_ERROR;

   tU8 u8_ActiveBDAddressId=INVALID_BDADDRESS_ID ;
   tBool b_getAdressSuccess = FALSE ;
   FcPhoneDial_InputParamStruct st_dialInParam ;

   ETG_TRACE_USR1(("*****MethodStart:m_oDialExtendedMessage*****"));

   // CMG3GB-867 - hf-dial should be triggered even without sim for emergency calls.

   //get Free call instance
   s32Error = s32GetIdleCallInstanceIndex(u16Id);

   if (OSAL_ERROR == s32Error)
   {
      ETG_TRACE_ERR(( "vHandle_DialExtended_MethodStart: Both call instances occupied." ));
      vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
   }
   else
   {
      ETG_TRACE_USR2((" got idle callInstance: %d ",u16Id));

      if(poDiaExtended_MS)
      {
         vResetDiallerInfoForCallInstance(static_cast<tU8>(u16Id));
         if (poDiaExtended_MS->sName.szValue)
         {
            ETG_TRACE_USR1((" Caller name is ::%s", poDiaExtended_MS->sName.szValue));
            tU32 u32ContactNameLength = static_cast<tU32> (strlen(poDiaExtended_MS->sName.szValue) + 1);
            m_sDialInfo[u16Id].m_sDialName = new char[u32ContactNameLength];
            if (m_sDialInfo[u16Id].m_sDialName)
            {
               memset(m_sDialInfo[u16Id].m_sDialName, 0, u32ContactNameLength);
               vStringCopy(m_sDialInfo[u16Id].m_sDialName, poDiaExtended_MS->sName.szValue, u32ContactNameLength);
               ETG_TRACE_USR1((" m_sDialName: %s",m_sDialInfo[u16Id].m_sDialName));
            }
            else
            {
               ETG_TRACE_ERR(( "Cannot create memory for m_sDialName" ));
            }
         }
         else
         {
            ETG_TRACE_USR1((" Caller name is NULL"));
         }

         m_sDialInfo[u16Id].m_sDialNumber = new char[FC_PHONE_DIAL_NUMBER_MAX_LEN_IN_CHARS];
         if (m_sDialInfo[u16Id].m_sDialNumber)
         {
            memset(m_sDialInfo[u16Id].m_sDialNumber, 0, FC_PHONE_DIAL_NUMBER_MAX_LEN_IN_CHARS);
            vStringCopy(m_sDialInfo[u16Id].m_sDialNumber, m_sValidPhoneNumber, FC_PHONE_DIAL_NUMBER_MAX_LEN_IN_CHARS + 1);
            ETG_TRACE_USR1((" m_sDialNumber: %s",m_sDialInfo[u16Id].m_sDialNumber));
         }
         else
         {
            ETG_TRACE_ERR(( "Cannot create memory for m_sDialNumber" ));
         }
         //Check for server voice recognition mode
         ecnrServerVoiceMode(poDiaExtended_MS->e8EchoCancellationNoiseReductionSetting.enType);
      }

      //start state machine for ignore call
      fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];

      b_getAdressSuccess =  b_get_ActiveDevice_AddressId_FromBT(&u8_ActiveBDAddressId);

      if ((TRUE == b_getAdressSuccess) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
      {
         pCallInstance->vCreate(u16Id,u8_ActiveBDAddressId) ;

         //Start SM for Dial/SpeedDial
         st_dialInParam.en_dialType = FC_PHONE_DIALTYPE_DIALEXTENDED ;
         st_dialInParam.pu8_phoneNumber = (tU8*)m_sValidPhoneNumber;
         m_bDialMethodStartFlag = TRUE;
         m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_DIAL , pCallInstance,&st_dialInParam);
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_DialExtended_MethodStart: Either m_oSLCStatusStatus is false or Telephone number empty" ));
         vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
         vResetDiallerInfoForCallInstance(static_cast<tU8>(u16Id));
      }
   }

   for(tU8 index = 0; index< (FC_PHONE_DIAL_NUMBER_MAX_LEN_IN_CHARS+1); index++)
   {
      m_sValidPhoneNumber[index] = 0;//Reset to default after use
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   if(poDiaExtended_MS)
   {
      poDiaExtended_MS->vDestroy();
      delete poDiaExtended_MS;
   }
   delete l_poJob;//delete job pointer after use
   //End of fix

   ETG_TRACE_USR4((" EXITTINGGGG :  fc_phone_tclService_Telephone::vHandle_DialExtended_MethodStart "));
}

/*******************************************************************************
** FUNCTION:   vHandle_ConferenceCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_ConferenceCall_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_ConferenceCall_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone:: vHandle_ConferenceCall_MethodStart ENTERED "));
   /* Extract the Call instance and other required info from poMessage */
   tU8 u8Index = 0;
   fc_phone_tclCallInstance *pCallInstance[FC_PHONE_MAX_CALLINSTANCES];

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_telfi_tclMsgConferenceInMethodStart oConferenceCall_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( ( OSAL_ERROR != oVisitorMsg.s32GetData(oConferenceCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag) )
   {

      if ( isConfSMExecutedFromHmi )
      {
         // Already conference call is started.Avoid duplicate Events.
         vHandle_ConferenceCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         oConferenceCall_MS.vDestroy();
         return;
      }
      /* Get the coresponding Call Instance */
      tU16 u16IdCall1 = 0x00;
      tU16 u16IdCall2 = 0x00;
      tU8 u8AudioAcquiredCnt = 0;

      NULL_CHECK(m_pobtTelClient)

      tU8 u8_btActiveBDAddressId= m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
      if(INVALID_BDADDRESS_ID != u8_btActiveBDAddressId)
      {
         if (m_oPhoneData[u8_btActiveBDAddressId].m_bThreeWayCallSupport)
         {
            if ( (OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oConferenceCall_MS.oStreamOfCallInstances.u16Items[0], u16IdCall1) ) && (OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oConferenceCall_MS.oStreamOfCallInstances.u16Items[1], u16IdCall2)) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
            {
               ETG_TRACE_USR1(("***** MethodStart:oConferenceCall_MS "));
               ETG_TRACE_USR1((" u16CallInstance1: %u", oConferenceCall_MS.oStreamOfCallInstances.u16Items[0]));
               ETG_TRACE_USR1(("u16CallInstance2: %u", oConferenceCall_MS.oStreamOfCallInstances.u16Items[1]));

               u16ConferenceCall1 = oConferenceCall_MS.oStreamOfCallInstances.u16Items[0];
               u16ConferenceCall2 = oConferenceCall_MS.oStreamOfCallInstances.u16Items[1];

               m_oConferenceCallMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
               m_oConferenceCallMessage.u16RegisterID  = poMessage->u16GetRegisterID();
               m_oConferenceCallMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
               m_oConferenceCallMessage.u16FunctionID  = poMessage->u16GetFunctionID();

               for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
               {
                  pCallInstance[u8Index] = NULLPTR;
                  if ((m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE) || (m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD))
                  {
                     pCallInstance[u8Index] = &m_oCallInstances[u8Index];
                  }

                  if (m_oCallInstances[u8Index].m_bUsingVehicleAudio == TRUE)
                  {
                     u8AudioAcquiredCnt++;
                  }
               }

               if (AUDIO_ACQUIRED_TWO_CALLS == u8AudioAcquiredCnt)
               {
                  ETG_TRACE_USR3((" u8AudioAcquiredCnt: %d ",u8AudioAcquiredCnt));
                  isConfSMExecutedFromHmi = TRUE;
                  m_poSMManager->vStartSM(FC_PHONE_SMMANAGER_SMID_CONFERENCECALL, pCallInstance);
               }
               else if (AUDIO_ACQUIRED_NONE == u8AudioAcquiredCnt)
               {
                  ETG_TRACE_USR3((" u8AudioAcquiredCnt: %d ",u8AudioAcquiredCnt));
                  FCPhoneConf_InputParamStruct st_ConfInParam ;
                  st_ConfInParam.isConfFromCallStatus = FALSE;
                  isConfSMExecutedFromHmi = TRUE;
                  m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_CONFERENCECALL , pCallInstance, &st_ConfInParam);

               }
               else
               {
                  ETG_TRACE_USR3((" u8AudioAcquiredCnt: %d ",u8AudioAcquiredCnt));
                  /* Invalid case - internal error, So send an error message */

                  ETG_TRACE_ERR(( "vHandle_ConferenceCall_MethodStart: Invalid case - " ));
                  vHandle_ConferenceCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
               }
            }
            else
            {
               ETG_TRACE_ERR(( "vHandle_ConferenceCall_MethodStart: Invalid CallInstance Requested." ));
               vHandle_ConferenceCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_ConferenceCall_MethodStart: Three way calling not supported." ));
         vHandle_ConferenceCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_ConferenceCall_MethodStart: Received message invalid." ));
      // We never expect to come here.
      vHandle_ConferenceCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oConferenceCall_MS.vDestroy();
}


/*******************************************************************************
** FUNCTION:   vHandle_SplitCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_SplitCall_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SplitCall_MethodStart(amt_tclServiceData* poMessage)
{
   /* Extract the Call instance and other required info from poMessage */
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vHandle_SplitCall_MethodStart ENTERED "));
   tU8 u8Index = 0;

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_telfi_tclMsgSplitCallMethodStart oSplitCall_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( (OSAL_ERROR != oVisitorMsg.s32GetData(oSplitCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag) )
   {
      /* Get the coresponding Call Instance */
      tU16 u16Id = 0x00;
      tU8 u8numberOfCallsActive = 0;

      if ((OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oSplitCall_MS.u16CallInstance, u16Id)) && (TRUE == m_oSLCStatusStatus.bCommandStatus) )
      {
         ETG_TRACE_USR1((" ***** MethodStart:oSplitCall_MS, u16CallInstance: %u", oSplitCall_MS.u16CallInstance));

         m_oSplitCallMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
         m_oSplitCallMessage.u16RegisterID  = poMessage->u16GetRegisterID();
         m_oSplitCallMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
         m_oSplitCallMessage.u16FunctionID  = poMessage->u16GetFunctionID();
         m_oSplitCallMessage.u16InstanceID  = u16Id ;

         for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
         {
            if ((m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE) || (m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE))
            {
               u8numberOfCallsActive++;
            }
         }

         if (AUDIO_ACQUIRED_TWO_CALLS == u8numberOfCallsActive)
         {
            ETG_TRACE_USR3((" u8numberOfCallsActive: %d ",u8numberOfCallsActive));

            //fill process Event Arguments and send processEvent
            FcPhone_processEventArg   tProcessEventArg ;
            tProcessEventArg.bIsItDbusAck =  FALSE ;
            tProcessEventArg.u16CallInstanceID = u16Id ;

            m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_SPLITCALL_OK,FC_PHONE_EVENTTYPE_TARGATED,&tProcessEventArg);
         }
         else
         {
            ETG_TRACE_ERR(( "vHandle_SplitCall_MethodStart: Cannot unmerge calls as two calls are not active. u8numberOfCallsActive: %d",u8numberOfCallsActive));
            vHandle_SplitCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         }
      }
      else
      {
         //send error message : no call is active
         ETG_TRACE_ERR(( "vHandle_SplitCall_MethodStart: No call is active." ));
         vHandle_SplitCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_SplitCall_MethodStart: Received message invalid." ));
      // We never expect to come here.
      vHandle_SplitCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oSplitCall_MS.vDestroy();
}


/*******************************************************************************
** FUNCTION:   vHandle_SpeedDial_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for SpeedDial_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SpeedDial_MethodStart(amt_tclServiceData* poMessage, struct Job* f_poJob)
{
   //FIX GMMY15-10440 If speed dialing is performed to a number , call "End" from centerstack doesnot work if selected before call gets connected
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::vHandle_SpeedDial_MethodStart "));

   /* Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   // Create the (message related) FI data object
   most_telfi_tclMsgSpeedDialMethodStart *poSpeedDial_MS = NULLPTR;
   struct Job* l_poJob = NULLPTR;

   if(poMessage)//Direct call from client, not from work queue
   {
      // Create a FI visitor message for the received CCA message
      fi_tclVisitorMessage oVisitorMsg( poMessage );
      poSpeedDial_MS = new most_telfi_tclMsgSpeedDialMethodStart;

      if(poSpeedDial_MS)
      {
         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data object.
         if ((OSAL_ERROR != oVisitorMsg.s32GetData(*poSpeedDial_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
         {
            //Create a new job
            _BP_TRY_BEGIN
            {
               ETG_TRACE_USR4((" Creating new job"));
               l_poJob = new Job;
            }
            _BP_CATCH_ALL
            {
               poSpeedDial_MS->vDestroy();
               delete poSpeedDial_MS;
               return;
            }
            if(l_poJob)//If creation of job is successful, assign parameters
            {
               l_poJob->m_pfMethod  = &fc_phone_tclService_Telephone::vHandle_SpeedDial_MethodStart;
               l_poJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               l_poJob->u16FunctionID = poMessage->u16GetFunctionID();
               l_poJob->u16RegisterID = poMessage->u16GetRegisterID();
               l_poJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               l_poJob->m_poMethodStart = poSpeedDial_MS;
            }
         }
         else
         {
            poSpeedDial_MS->vDestroy();
            delete poSpeedDial_MS;
            ETG_TRACE_ERR(( "vHandle_SpeedDial_MethodStart: Received message invalid." ));
            vHandle_SpeedDial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            return;
         }
      }
      else
      {
         ETG_TRACE_ERR(( "poSpeedDial_MS is NULL" ));
         vHandle_SpeedDial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         return;
      }
   }
   else if(f_poJob)//Received from WorkQueue, not directly from client
   {
      ETG_TRACE_USR4((" Received from queue"));
      l_poJob = f_poJob;
   }
   else
   {
      ETG_TRACE_USR4((" Should not come here!!"));
      return;
   }

   if(!l_poJob)
   {
      ETG_TRACE_ERR(( "l_poJob is NULL" ));
      vHandle_SpeedDial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

      if(poSpeedDial_MS)
      {
         poSpeedDial_MS->vDestroy();
         delete poSpeedDial_MS;
      }
      return;
   }

   if (fc_phone_tclPhoneMutex::PhoneMutexTrylock(l_poJob, FC_PHONE_WORK_QUEUE_TIMER_VALUE_FOR_DIAL) != 0)//Try to acquire lock
   {
      tVoid*  pCustodialLose = l_poJob ;  //To remove LINT warning
      (tVoid) pCustodialLose ;
      return; // Didn't get the lock :(
   }

   m_oSpeedDialMessage.u16SourceAppID = l_poJob->u16SourceAppID;
   m_oSpeedDialMessage.u16RegisterID  = l_poJob->u16RegisterID;
   m_oSpeedDialMessage.u16CmdCounter  = l_poJob->u16CmdCounter;
   m_oSpeedDialMessage.u16FunctionID  = l_poJob->u16FunctionID;


   poSpeedDial_MS = (most_telfi_tclMsgSpeedDialMethodStart*)   l_poJob->m_poMethodStart;
   NULL_CHECK(poSpeedDial_MS);
   tU16 u16Id = 0x00;
   tS32 s32Error = OSAL_ERROR;
   tU8 ActiveBDAddressId=INVALID_BDADDRESS_ID;

   FcPhoneDial_InputParamStruct st_dialInParam ;
   ETG_TRACE_USR1(("***** MethodStart:poSpeedDial_MS :: s8MemoryDialParam: %d ",poSpeedDial_MS->s8MemoryDialParam));
   //Before dialing, check for network availability
   //If network is not available , send MError (Do not send dial request to the phone)

   NULL_CHECK(m_pobtTelClient)

   tU8 u8_btActiveBDAddressId= m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
   if(INVALID_BDADDRESS_ID != u8_btActiveBDAddressId)
   {
      if(FC_PHONE_PHONEDATA_NWSTATUS_NOTAVAILABLE == m_oPhoneData[u8_btActiveBDAddressId].m_u8NetworkStatus)
      {
         ETG_TRACE_USR4(( "vHandle_SpeedDial_MethodStart: m_u8NetworkStatus(0), Network not available , Send SpeedDial method Error" ));
         vHandle_SpeedDial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
         if(poSpeedDial_MS)
         {
            poSpeedDial_MS->vDestroy();
            delete poSpeedDial_MS;
         }
         return ;
      }
   }
   else
   {
      ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
   }

   //get Free call instance
   s32Error = s32GetIdleCallInstanceIndex(u16Id);

   if (OSAL_ERROR == s32Error)
   {
      ETG_TRACE_ERR(( "vHandle_SpeedDial_MethodStart: Both call instances occupied." ));
      vHandle_SpeedDial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
   }
   else
   {
      ETG_TRACE_USR3((" got idle callInstance: %d ",u16Id));
      //start state machine for ignore call
      fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16Id];

      ActiveBDAddressId =(tU8) m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);

      if((INVALID_BDADDRESS_ID != ActiveBDAddressId)  && (m_oSLCStatusStatus.bCommandStatus == TRUE))
      {
         ETG_TRACE_USR2((" Dial_methodStart : ActiveBDAddressId :: %d",ActiveBDAddressId));
         //update device address to callInstance
         pCallInstance->vCreate(u16Id, ActiveBDAddressId) ;

         //Start SM for Dial/SpeedDial
         st_dialInParam.en_dialType = FC_PHONE_DIALTYPE_SPEEDDIAL ;
         st_dialInParam.u16_memID = (tU16)poSpeedDial_MS->s8MemoryDialParam;
         m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_DIAL , pCallInstance,&st_dialInParam);
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_SpeedDial_MethodStart: m_oSLCStatusStatus is false or device address is null." ));
         vHandle_SpeedDial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, f_poJob, true);
      }
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   if(poSpeedDial_MS)
   {
      poSpeedDial_MS->vDestroy();
      delete poSpeedDial_MS;
   }
   delete l_poJob;//delete job pointer after use
   //End of fix

   ETG_TRACE_USR4((" EXITTINGGGG :  fc_phone_tclService_Telephone::vHandle_SpeedDial_MethodStart "));
}


/*******************************************************************************
** FUNCTION:   vHandle_CancelOutgoingCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for CancelOutgoingCall_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_MethodStart(amt_tclServiceData* poMessage, struct Job* f_poJob)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_MethodStart "));
   /* Extract the Call instance and other required info from poMessage */
   // Create the (message related) FI data object
   most_telfi_tclMsgCancelOutgoingCallMethodStart *poCancelOutgoingCall_MS = NULLPTR;

   //FIX GMMY15-10013 [CT]Call is dialed out even when end is selected
   //Second request sent to BT Daemon from FC_Phone before receiving response of former request is responded with error by BT Daemon for some method requests.
   //For those requests for which API restrictions are defined in ALPS document, FC_Phone tries to acquire a mutex lock. If acquired then only request is forwarded to DBUS. Else it is added to a work queue.

   struct Job* l_poJob = NULLPTR;

   if(poMessage)//Direct call from client, not from work queue
   {
      // Create a FI visitor message for the received CCA message
      fi_tclVisitorMessage oVisitorMsg( poMessage );
      poCancelOutgoingCall_MS = new most_telfi_tclMsgCancelOutgoingCallMethodStart;

      if(poCancelOutgoingCall_MS)
      {
         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data object.

         if ((OSAL_ERROR != oVisitorMsg.s32GetData(*poCancelOutgoingCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
         {
            ETG_TRACE_USR4((" m_u16StoredCallInstanceForCancel : %d ",m_u16StoredCallInstanceForCancel));
            //If same request comes again to FC_Phone, it is not processed. Cancel outgoing call request returns giving error.
            if( m_u16StoredCallInstanceForCancel == poCancelOutgoingCall_MS->u16CallInstance )
            {
               ETG_TRACE_ERR(( "vHandle_CancelOutgoingCall_MethodStart: Already received message under process. Rejecting duplicates." ));
               vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

               poCancelOutgoingCall_MS->vDestroy();
               delete poCancelOutgoingCall_MS;
               return;
            }
            else
            {
               //If same request comes again to FC_Phone we dont push them to queue. Cancel outgoing call request returns giving error.
               //Store current parameter to check duplicate requests which may come afterwards.
               m_u16StoredCallInstanceForCancel = poCancelOutgoingCall_MS->u16CallInstance;
            }

            //Create a new job
            l_poJob = new Job;
            if(l_poJob)//If creation of job is successful, assign parameters
            {
               l_poJob->m_pfMethod  = &fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_MethodStart;
               l_poJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               l_poJob->u16FunctionID = poMessage->u16GetFunctionID();
               l_poJob->u16RegisterID = poMessage->u16GetRegisterID();
               l_poJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               l_poJob->m_poMethodStart = poCancelOutgoingCall_MS;
            }
         }
         else
         {
            ETG_TRACE_ERR(( "poCancelOutgoingCall_MS: Received message invalid." ));
            // We never expect to come here.
            vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            poCancelOutgoingCall_MS->vDestroy();
            delete poCancelOutgoingCall_MS;
            return;
         }
      }
      else
      {
         ETG_TRACE_ERR(( "poCancelOutgoingCall_MS is NULL" ));
         vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         return;
      }
   }
   else if(f_poJob)//Received from WorkQueue, not directly from client
   {
      ETG_TRACE_USR4((" Received from queue"));
      l_poJob = f_poJob;
   }
   else
   {
      ETG_TRACE_USR4((" Should not come here!!"));
      return; // Should not come here
   }

   if(!l_poJob)
   {
      ETG_TRACE_ERR(( "l_poJob is NULL" ));
      // We never expect to come here.
      vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      if(poCancelOutgoingCall_MS)
      {
         poCancelOutgoingCall_MS->vDestroy();
         delete poCancelOutgoingCall_MS;
      }
      return;
   }

   if (fc_phone_tclPhoneMutex::PhoneMutexTrylock(l_poJob) != 0)//Try to acquire lock
   {
      tVoid*  pCustodialLose = l_poJob ;  //To remove LINT warning
      (tVoid) pCustodialLose ;
      return; // Didn't get the lock :(
   }

   //No need to clear the previously stored values. If value cleared, next request from HMI(duplicate) would enter the queue (which is not recommended)
   // and the subsequent requests after that only will be rejected .
   m_oCancelOutgoingCallMessage.u16SourceAppID = l_poJob->u16SourceAppID;
   m_oCancelOutgoingCallMessage.u16RegisterID  = l_poJob->u16RegisterID;
   m_oCancelOutgoingCallMessage.u16CmdCounter  = l_poJob->u16CmdCounter;
   m_oCancelOutgoingCallMessage.u16FunctionID  = l_poJob->u16FunctionID;



   poCancelOutgoingCall_MS = (most_telfi_tclMsgCancelOutgoingCallMethodStart*)   l_poJob->m_poMethodStart;
   NULL_CHECK(poCancelOutgoingCall_MS);

   ETG_TRACE_USR1(("poCancelOutgoingCall_MS->u16CallInstance %d ",poCancelOutgoingCall_MS->u16CallInstance));
   /* Get the coresponding Call Instance */
   tU16 u16Id = 0x00;
   tBool bEventConsumed = FALSE;
   if (OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(poCancelOutgoingCall_MS->u16CallInstance, u16Id) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
   {
      ETG_TRACE_USR1((" ***** MethodStart:oCancelOutgoingCall_MS "));
      //SUZUKI-13411 -  cancel outgoing call is processed only for call status dialing
      if ( (m_oCallInstances[u16Id].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_DIALING)
         || (m_oCallInstances[u16Id].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE))
      {
         //fill process Event Arguments and send processEvent
         FcPhone_processEventArg   tProcessEventArg ;
         tProcessEventArg.bIsItDbusAck =  FALSE ;
         tProcessEventArg.u16CallInstanceID = u16Id ;
         bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_CANCELOUTGOINGCALL_OK,FC_PHONE_EVENTTYPE_TARGATED,&tProcessEventArg);
      }
      else
      {
         ETG_TRACE_ERR(( "oCancelOutgoingCall_MS: Invalid Request - Call Status Active" ));
      }
   }
   else
   {
      ETG_TRACE_ERR(( "oCancelOutgoingCall_MS: Invalid CallInstance Requested." ));
   }

   if (!bEventConsumed)
   {
      vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, l_poJob, true);
      m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM;
   }


   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   if(poCancelOutgoingCall_MS)
   {
      poCancelOutgoingCall_MS->vDestroy();
      delete poCancelOutgoingCall_MS;
   }
   delete l_poJob;//delete job pointer after use
   //End of fix

   ETG_TRACE_USR3((" EXITTINGGGG :  fc_phone_tclService_Telephone::oCancelOutgoingCall_MS "));
}


/*******************************************************************************
** FUNCTION:   vHandle_AcquireTelephoneNumber_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for acquiring telephone number method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_AcquireTelephoneNumber_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::vHandle_AcquireTelephoneNumber_MethodStart "));
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );
   // Create the (message related) FI data object
   IncomingMessage oAcquireTelephoneNumber;
   most_telfi_tclMsgAcquireTelephoneNumberMethodStart oAcquireTelephoneNumber_MS;
   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.
   if (OSAL_ERROR != (oVisitorMsg.s32GetData(oAcquireTelephoneNumber_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
   {
      ETG_TRACE_USR1(("***** MethodStart:oAcquireTelephoneNumber_MS "));
      ETG_TRACE_USR1(("  oAcquireTelephoneNumber_MS.u8DeviceHandle = %d",oAcquireTelephoneNumber_MS.u8DeviceHandle));
      oAcquireTelephoneNumber.u16SourceAppID = poMessage->u16GetSourceAppID();
      oAcquireTelephoneNumber.u16RegisterID  = poMessage->u16GetRegisterID();
      oAcquireTelephoneNumber.u16CmdCounter  = poMessage->u16GetCmdCounter();
      oAcquireTelephoneNumber.u16FunctionID  = poMessage->u16GetFunctionID();
      m_oAcquireTelephoneNumberMessage.push_back(oAcquireTelephoneNumber);
      const tChar *pu8_btActiveDeviceAddress = NULLPTR;
      tU8 pu8_btActiveDeviceAddressId = INVALID_BDADDRESS_ID;

      if (m_pobtTelClient)
      {
         pu8_btActiveDeviceAddressId = m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(oAcquireTelephoneNumber_MS.u8DeviceHandle);
         pu8_btActiveDeviceAddress =  fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(pu8_btActiveDeviceAddressId);
      }

      if(pu8_btActiveDeviceAddress)
      {
         m_u8BDAddressIDforAcquiringTelNo = pu8_btActiveDeviceAddressId;
         tBool b_isProxyObjNull = static_cast<tBool>(m_pDBus_CCA_If->bAcquireTelephoneNumber((const char*)pu8_btActiveDeviceAddress));
         if(TRUE == b_isProxyObjNull)
         {
            ETG_TRACE_ERR((" Proxy object is NULL "));
            vHandle_AcquireTelephoneNumber_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
         }
      }
      else
      {
         ETG_TRACE_ERR(("  No device with outgoing source hence telephone # cannot be acquired "));
         vHandle_AcquireTelephoneNumber_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "oAcquireTelephoneNumber_MS: Received message invalid." ));
      // We never expect to come here.
      vHandle_AcquireTelephoneNumber_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
   }
   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oAcquireTelephoneNumber_MS.vDestroy();
   ETG_TRACE_USR4((" EXITTINGGGG :  fc_phone_tclService_Telephone::oAcquireTelephoneNumber_MS "));
}


/*******************************************************************************
** FUNCTION:   vHandle_QueryTelephoneNumber_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_QueryTelephoneNumber_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_QueryTelephoneNumber_MethodStart(amt_tclServiceData* poMessage)
{
   /* Extract the Call instance and other required info from poMessage */

   ETG_TRACE_USR3((" ENTERING :fc_phone_tclService_Telephone::vHandle_QueryTelephoneNumber_MethodStart "));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_telfi_tclMsgQueryTelephoneNumberMethodStart oQueryTelephoneNumber_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if (OSAL_ERROR != (oVisitorMsg.s32GetData(oQueryTelephoneNumber_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag))
   {
      /* Get the coresponding Call Instance */
      tU16 u16Id = 0x00;
      tS32 s32CIStatusActiveOrPassive = s32GetCallInstanceIndexFromFiCallInstance(oQueryTelephoneNumber_MS.u16CallInstance, u16Id);
      if (((OSAL_OK == s32CIStatusActiveOrPassive) || (true == s32CIStatusActiveOrPassive)) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
      {
         ETG_TRACE_USR1((" ***** MethodStart:oQueryTelephoneNumber_MS  "));
         ETG_TRACE_USR1(("     oQueryTelephoneNumber_MS.u16CallInstance : %d     ",oQueryTelephoneNumber_MS.u16CallInstance));
         ETG_TRACE_USR1((" u16Id : %d ", u16Id));

         m_oQueryTelephoneNumberMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
         m_oQueryTelephoneNumberMessage.u16RegisterID  = poMessage->u16GetRegisterID();
         m_oQueryTelephoneNumberMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
         m_oQueryTelephoneNumberMessage.u16FunctionID  = poMessage->u16GetFunctionID();
         m_oQueryTelephoneNumberMessage.u16InstanceID  = oQueryTelephoneNumber_MS.u16CallInstance ;

         Async_method_data_t vAsync_method_data;

         vStringCopy(vAsync_method_data.TelephoneNumber, CallStatusList[oQueryTelephoneNumber_MS.u16CallInstance].CallStatus.achCallNumber, \
               PHONE_TELEPHONENUMBER_LENGTH);

         vQueryTelephoneNumberCB(&vAsync_method_data);
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_QueryTelephoneNumber_MethodStart: Received message invalid, wrong call instance." ));
         vHandle_QueryTelephoneNumber_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_QueryTelephoneNumber_MethodStart: Received message invalid" ));
      vHandle_QueryTelephoneNumber_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oQueryTelephoneNumber_MS.vDestroy();

   ETG_TRACE_USR4((" EXITTINGGGG :  fc_phone_tclService_Telephone::vHandle_QueryTelephoneNumber_MethodStart "));
}


/*******************************************************************************
** FUNCTION:   vHandle_VoiceMail_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_VoiceMail_MethodStart method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_VoiceMail_MethodStart(amt_tclServiceData* poMessage)
{
   /* Extract the Call instance and other required info from poMessage */
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::vHandle_VoiceMail_MethodStart "));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_telfi_tclMsgVoiceMailMethodStart oVoiceMail_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if (OSAL_ERROR != (oVisitorMsg.s32GetData(oVoiceMail_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (m_bDiagnosticModeFlag == FALSE) )
   {
      ETG_TRACE_USR1(("***** MethodStart:oVoiceMail_MS DeviceName: %s",oVoiceMail_MS.sDeviceName.szValue));

      NULL_CHECK(m_pobtTelClient)

         const tChar *psVoiceMailNumber = m_pobtTelClient->pu8GetVoiceMailNumberFromDeviceName(oVoiceMail_MS.sDeviceName.szValue);
      if((psVoiceMailNumber) && (m_oSLCStatusStatus.bCommandStatus == TRUE))
      {
         //sending VoiceMail Method result
         ETG_TRACE_USR3((" sending VoiceMail Method result "));

         most_telfi_tclMsgVoiceMailMethodResult oVoiceMail_MR;

         if(u8StringCompare(psVoiceMailNumber,"NO_VOICEMAIL") != 0)
         {
            oVoiceMail_MR.sVoiceMailNumber.bSet(psVoiceMailNumber);
         }
         else
         {
            ETG_TRACE_ERR((" NO_VOICEMAIL : present in voicemail number , so not setting oVoiceMail_MR.sVoiceMailNumber"));
         }

         ETG_TRACE_USR3(("Sending voicemail number :: %s",oVoiceMail_MR.sVoiceMailNumber.szValue));

         fi_tclVisitorMessage oMRMsg(oVoiceMail_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

         // Here the message related header information is added and by this the
         // creation of the message is completed.
         vInitServiceData( oMRMsg,                          // ServiceDataMsg
            poMessage->u16GetSourceAppID(),               // Target AppID
            poMessage->u16GetRegisterID() ,                // RegisterID
            poMessage->u16GetCmdCounter() ,                // CmdCounter,
            poMessage->u16GetFunctionID() ,                // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
            );

         if (oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(( "oVoiceMail_MR: 'MethodResult', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
            }
         }
         else
         {
            ETG_TRACE_ERR(("  oVoiceMail_MR : INVALID MESSAGE "));
         }

         oVoiceMail_MR.vDestroy();
         ETG_TRACE_USR4((" In fc_phone_tclService_Telephone::vVoiceMailCB EXIT voicemail MR sending finished"));
      }
      else
      {
         //Send Method Error for Voice Mail
         //update status , sent method result
         ETG_TRACE_ERR(( "vHandle_VoiceMail_MethodStart: VOICE MAIL REQUEST IS NOT SERVICED." ));
         vHandle_VoiceMail_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         ETG_TRACE_USR4((" In fc_phone_tclService_Telephone::vVoiceMailCB EXIT voicemail ME sending finished"));
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_VoiceMail_MethodStart: Received message invalid." ));
      // We never expect to come here.

      vHandle_VoiceMail_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oVoiceMail_MS.vDestroy();

   ETG_TRACE_USR4((" EXIT :  fc_phone_tclService_Telephone::vHandle_VoiceMail_MethodStart "));
}


/*******************************************************************************
**	FUNCTION:   vHandle_SendDTMF_MethodStart(..)
*******************************************************************************/
/*	DESCRIPTION:
*   This is a handler function for SendDTMF method
*   This function goes through State Machine and then to DBus where it sends DTMF Digit to Phone thereafter played by phone
*	PARAMETERS:
*	IN:   poMessage        : This is the received message from a client
*	RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SendDTMF_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::vHandle_SendDTMF_MethodStart "));
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_telfi_tclMsgSendDTMFMethodStart oSendDTMF_MS;

   if(OSAL_ERROR!= oVisitorMsg.s32GetData(oSendDTMF_MS,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION))
   {
      /* Get the corresponding Call Instance */
      tU16 u16Id = 0x00;
      ETG_TRACE_USR1(("***** MethodStart:SendDTMF  "));
      ETG_TRACE_USR1(("      oSendDTMF_MS.u16CallInstance :: %d      ",oSendDTMF_MS.u16CallInstance));
      ETG_TRACE_USR1(("      oSendDTMF_MS.sDTMFChar :: %s       ",oSendDTMF_MS.sDTMFChar.szValue));

      if (TRUE == m_oSLCStatusStatus.bCommandStatus)
      {
         ETG_TRACE_USR3(("     u16Id = %x   ",u16Id));

         m_oSendDTMFMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
         m_oSendDTMFMessage.u16RegisterID  = poMessage->u16GetRegisterID();
         m_oSendDTMFMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
         m_oSendDTMFMessage.u16FunctionID  = poMessage->u16GetFunctionID();

         if ((0xFF == oSendDTMF_MS.u16CallInstance ) ||(OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oSendDTMF_MS.u16CallInstance, u16Id)))
         {
            fc_phone_tclCallInstance *pCallInstance =  NULLPTR;

            if (0xFF == oSendDTMF_MS.u16CallInstance)
            {
               pCallInstance =  &m_oCallInstances[0];
            }
            else if (OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oSendDTMF_MS.u16CallInstance, u16Id))
            {
               if ((m_oCallInstances[u16Id].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE) || (m_oCallInstances[u16Id].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE))
               {
                  pCallInstance =  &m_oCallInstances[u16Id];
               }
            }

            NULL_CHECK(pCallInstance)
            if ((m_oCallInstances[u16Id].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE) || (m_oCallInstances[u16Id].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE))
            {
               if (m_bDtmfProcessing)
               {
                  ETG_TRACE_USR4((" Dtmf is in process"));
                  vAppendDtmfString(pCallInstance, oSendDTMF_MS.sDTMFChar.szValue, u16Id);
               }
               else
               {
                  ETG_TRACE_USR4((" No Dtmf is in process"));
                  vStringCopy(pCallInstance->DTMFChar,oSendDTMF_MS.sDTMFChar.szValue, DTMF_MAXSTR_LEN);

                  //fill process Event Arguments and send processEvent
                  FcPhone_processEventArg   tProcessEventArg ;
                  tProcessEventArg.bIsItDbusAck =  FALSE ;
                  tProcessEventArg.u16CallInstanceID = u16Id ;

                  if (m_poSMManager)
                  {
                     m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_SENDDTMF, FC_PHONE_EVENTTYPE_TARGATED ,&tProcessEventArg);
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(( "vHandle_SendDTMF_MethodStart: No Active call." ));
               vHandle_SendDTMF_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            }
         }
         else
         {
            ETG_TRACE_ERR(( "vHandle_SendDTMF_MethodStart: No Active call instance." ));
            vHandle_SendDTMF_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_SendDTMF_MethodStart: No Active call instance." ));
         vHandle_SendDTMF_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }

      ETG_TRACE_USR3(("			u16Id = %x			",u16Id));
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_SendDTMF_MethodStart: Received message invalid." ));
      vHandle_SendDTMF_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   oSendDTMF_MS.vDestroy();
}


/*******************************************************************************
**	FUNCTION:   vHandle_SwapCall_MethodStart(..)
*******************************************************************************/
/*	DESCRIPTION:
*   This is a handler function for vHandle_SwapCall_MethodStart
*
*	PARAMETERS:
*	IN:   poMessage        : This is the received message from a client
*	RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SwapCall_MethodStart(amt_tclServiceData* poMessage)
{
   /* Extract the Call instance and other required info from poMessage */

   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::vHandle_SwapCall_MethodStart "));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_telfi_tclMsgSwapCallMethodStart oSwapCall_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if (OSAL_ERROR != (oVisitorMsg.s32GetData(oSwapCall_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION)) && (FALSE == m_bDiagnosticModeFlag) )
   {
      ETG_TRACE_USR1(("***** MethodStart:oSwapCall_MS "));
      ETG_TRACE_USR1(("      total instances :: %d",(int)oSwapCall_MS.oStreamOfCallInstances.u16Items.size()));

      for(tU16 u16Index = 0; u16Index < oSwapCall_MS.oStreamOfCallInstances.u16Items.size(); u16Index++)
      {
         ETG_TRACE_USR1(("      CallInstances[%d]:: %d",u16Index,oSwapCall_MS.oStreamOfCallInstances.u16Items[u16Index]));
      }

      m_oSwapCallMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oSwapCallMessage.u16RegisterID  = poMessage->u16GetRegisterID();
      m_oSwapCallMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
      m_oSwapCallMessage.u16FunctionID  = poMessage->u16GetFunctionID();

      /* Get the coresponding Call Instance */
      tU16 u16Id1 = 0xFF;
      tU16 u16Id2 = 0xFF;
      if ((OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oSwapCall_MS.oStreamOfCallInstances.u16Items[0], u16Id1))
         &&(OSAL_OK == s32GetCallInstanceIndexFromFiCallInstance(oSwapCall_MS.oStreamOfCallInstances.u16Items[1], u16Id2)
         && (m_oSLCStatusStatus.bCommandStatus == TRUE))
         )
      {
         //fill process Event Arguments and send processEvent
         FcPhone_processEventArg   tProcessEventArg ;
         tProcessEventArg.bIsItDbusAck =  FALSE ;

         //FIX GMMY16-7888 HMI reset observed when calls were swapped and then end call selected immediately.
         NULL_CHECK(m_poSMManager)
         //End of fix

        tBool bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_SWAPCALL_OK,FC_PHONE_EVENTTYPE_BROADCAST_CONSUMED,&tProcessEventArg);
        if(!bEventConsumed)
        {
           ETG_TRACE_USR4(( "oSwapCall_MS: not processed." ));
           vHandle_SwapCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
        }
      }
      else
      {
         ETG_TRACE_USR4(( "oSwapCall_MS: Invalid CallInstance Requested." ));
         vHandle_SwapCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "oSwapCall_MS: Received message invalid." ));
      // We never expect to come here.
      vHandle_SwapCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oSwapCall_MS.vDestroy();
   ETG_TRACE_USR4((" EXITTING :  fc_phone_tclService_Telephone::vHandle_SwapCall_MethodStart "));
}


/*******************************************************************************
** FUNCTION:   s32GetCallInstanceIndexFromFiCallInstance(..)
*******************************************************************************/
/* DESCRIPTION:
*  This function gets the location of the required callinstnce within the vector
*  PARAMETERS:
*     IN :   u16FICallInstance     : CallInstance
*     OUT:  rfId                   :  Position within the vector
*  RETURNVALUE: OSAL_ERROR, OSAL_OK
*******************************************************************************/
tS32 fc_phone_tclService_Telephone::s32GetCallInstanceIndexFromFiCallInstance(tU16 u16FICallInstance, tU16& rfIndex)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::s32GetCallInstanceIndexFromFiCallInstance with u16FICallInstance: %d", u16FICallInstance));

   if(u16FICallInstance >= CallStatusList.size())
   {
      ETG_TRACE_USR4((" u16FICallInstance is INVALID. Should be less than CallStatusList Vector size "));
      return OSAL_ERROR;
   }

   tU8 deviceHandle = CallStatusList[u16FICallInstance].DeviceHandle;

   if (deviceHandle == m_u8ActiveDeviceHandle)
   {
      ETG_TRACE_USR4((" Received CallInstance value is mapped to the Active device"));
      //The received FiCallInstance value is made to be 0 or 1 so that it shall be handled in SMs
      u16FICallInstance = u16FICallInstance%2;
      for(tU16 u16Index = 0x00; u16Index < FC_PHONE_MAX_CALLINSTANCES;u16Index++)
      {
         if (u16FICallInstance == m_oCallInstances[u16Index].m_u16Id)
         {
            rfIndex = u16Index;
            return OSAL_OK;
         }
      }
   }

   ETG_TRACE_USR4((" Received CallInstance value is mapped to the Passive device with Device Handle: %d", deviceHandle));
   rfIndex = u16FICallInstance%2;
   return true;
}


/*******************************************************************************
** FUNCTION:   s32GetIdleCallInstanceIndex()
*******************************************************************************/
/* DESCRIPTION:
*  This function gets the location of the first inactive call instance
*  PARAMETERS:
*     IN :   None
*     OUT:  rfId                   :  Position within the vector
*  RETURNVALUE: OSAL_ERROR, OSAL_OK
**************************************************************************/
tS32 fc_phone_tclService_Telephone::s32GetIdleCallInstanceIndex(tU16& rfIndex)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::s32GetIdleCallInstanceIndex"));
   tS32 s32Error = OSAL_ERROR;
   tU16 u16Index = 0x00;

   for(u16Index = 0x00; u16Index < FC_PHONE_MAX_CALLINSTANCES;u16Index++)
   {
      if (FC_PHONE_CALLINSTANCE_OBJECTSTATE_IDLE == m_oCallInstances[u16Index].m_u8ObjectState)
      {
         s32Error = OSAL_OK;
         break;
      }
   }

   if(OSAL_OK == s32Error)
   {
      rfIndex = u16Index;
   }

   return(s32Error);
}


/*******************************************************************************
** FUNCTION:   s32GetActiveCallInstanceIndex()
*******************************************************************************/
/* DESCRIPTION:
*  Gives the first active call instance
*  PARAMETERS:
*     IN :   None
*     OUT:  rfId                   :  Position within the vector
*  RETURNVALUE: OSAL_ERROR, OSAL_OK
**************************************************************************/
tS32 fc_phone_tclService_Telephone::s32GetActiveCallInstanceIndex(tU16& rfIndex)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::s32GetActiveCallInstanceIndex "));
   tS32 s32Error = OSAL_ERROR;
   tU16 u16Index = 0x00;

   for(u16Index = 0x00; u16Index < FC_PHONE_MAX_CALLINSTANCES;u16Index++)
   {
      if ((FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE == m_oCallInstances[u16Index].m_u16CallStatusNotice) || (FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE == m_oCallInstances[u16Index].m_u16CallStatusNotice))
      {
         s32Error = OSAL_OK;
         break;
      }
   }

   if(OSAL_OK == s32Error)
   {
      rfIndex = u16Index;
   }

   return(s32Error);
}


/*******************************************************************************
** FUNCTION:   DiagnosticModeConfig()
*******************************************************************************/
/* DESCRIPTION:
*  This function set the diagnostic mode on
*  PARAMETERS:

*  RETURNVALUE:
**************************************************************************/
void fc_phone_tclService_Telephone::DiagnosticModeConfig(tS32 /* eComponent */)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::DiagnosticModeConfig "));
   m_pDBus_CCA_If->vdiagnostic_mode_config(0,m_bDiagnosticModeFlag);
}


/*******************************************************************************
** FUNCTION:   pGetInstance(..)
*******************************************************************************/
/* DESCRIPTION:
*   Get the instance of Tel service
*  PARAMETERS:
*     IN:   None
*
*  RETURNVALUE: fc_phone_tclService_Telephone* : Pointer to TEL service
********************************************************************************/
fc_phone_tclService_Telephone* fc_phone_tclService_Telephone:: pGetInstance()
{
   ETG_TRACE_USR4(( "fc_phone_tclService_Telephone* fc_phone_tclService_Telephone:: pGetInstance ENTERED" ));
   return(m_poTelService);
}


/*******************************************************************************
** FUNCTION:   vSendBtVRStatus(..)
*******************************************************************************/
/* DESCRIPTION:
*   Internal method to send property update for Bt VR Status
*  PARAMETERS:
*     IN:   None
*
*  RETURNVALUE:None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vSendBtVRStatus (tBool bIsPhoneStatusInd, tU8 u8BDAddressID)
{
   ETG_TRACE_USR4(("vSendBtVRStatus ENTERED with u8BDAddressID: %d, bIsPhoneStatusInd = %d", u8BDAddressID, bIsPhoneStatusInd));

   //CMG3G-8447- Lint warnings
   (tVoid) bIsPhoneStatusInd;

   NULL_CHECK(m_pobtTelClient)

   tU8 u8_ActiveBDAddressId = m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
   ETG_TRACE_USR4(("u8_ActiveBDAddressId = %d", u8_ActiveBDAddressId));

   if ((INVALID_BDADDRESS_ID == u8BDAddressID) || (u8_ActiveBDAddressId == u8BDAddressID))
   {
      m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecSupported = FC_PHONE_BT_DEVICE_VR_NOT_SUPPORTED;

      if (INVALID_BDADDRESS_ID != u8_ActiveBDAddressId)
      {
         if (m_oPhoneData[u8_ActiveBDAddressId].m_bBtVoiceRecSupport)
         {
            m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecSupported = FC_PHONE_BT_DEVICE_VR_SUPPORTED;
         }
      }

      u8BDAddressID = u8_ActiveBDAddressId;

      ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITION);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         /* Throw up error */
         ETG_TRACE_ERR(( "vSendBtVRStatus: 'ERROR', UpdateClients failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
// CMG3G-7861 - Siri availability status
// For Apple devices update the BTVR Extended Status after receiving the SIRI event handler
// This is to avoid multiple updates of BTVR Extended

#ifdef PHONE_SIRI
      fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
      if(p_clientHndler_bluetooth && (!((p_clientHndler_bluetooth->bGetAppleDeviceStatusfromBDAddressID(u8BDAddressID)) && bIsPhoneStatusInd)))
      {
#endif
          vSendBtVRExtendedStatus(u8BDAddressID);
#ifdef PHONE_SIRI
      }
#endif
}

// CMG3G-7861 - Siri availability status
/*******************************************************************************
** FUNCTION:   vSendBtVRExtendedStatus(..)
*******************************************************************************/
/* DESCRIPTION:
*   Internal method to send property update for Bt VR Extended Status which
*   includes Siri Availability status in addition to Bt VR Status.
*  PARAMETERS:
*     IN:   None
*
*  RETURNVALUE:None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vSendBtVRExtendedStatus (tU8 u8BDAddressID)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSendBtVRExtendedStatus ENTERED with tU8 u8BDAddressID: %d", u8BDAddressID));

   tU8 u8_ActiveBDAddressId = m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
   ETG_TRACE_USR4(("u8_ActiveBDAddressId = %d", u8_ActiveBDAddressId));

   if ((INVALID_BDADDRESS_ID == u8BDAddressID) || (u8_ActiveBDAddressId == u8BDAddressID))
   {
      m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecSupported;
      m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive = m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecActive;
      m_oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle = m_u8ActiveDeviceHandle;
      u8BDAddressID = u8_ActiveBDAddressId;
   }
   else
   {
      m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = FC_PHONE_BT_DEVICE_VR_NOT_SUPPORTED;
      if (m_oPhoneData[u8BDAddressID].m_bBtVoiceRecSupport)
      {
         m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = FC_PHONE_BT_DEVICE_VR_SUPPORTED;
      }

      m_oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive = FC_PHONE_BT_VR_INACTIVE;
      tU8 devHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8BDAddressID);
      m_oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle = devHandle;
   }

#ifdef PHONE_SIRI

   if (INVALID_BDADDRESS_ID == u8BDAddressID)
   {
      ETG_TRACE_ERR(("INVALID BD Address ID"));
      m_oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState.enType = (most_fi_tcl_e8_TelSiriAvailabilityState::tenType) NOT_AVAILABLE;
   }
   else
   {
      m_oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState.enType = (most_fi_tcl_e8_TelSiriAvailabilityState::tenType)\
            ((fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->m_oSiriData[u8BDAddressID].eGetAplSiriState());
   }
#endif
   ETG_TRACE_USR4((" Updating clients about the BTVR Extended status  "));
   ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITIONEXTENDED);

   if (AIL_EN_N_NO_ERROR != enResult)
   {
      /* Throw up error */
      ETG_TRACE_ERR(( "vSendBtVRExtendedStatus: 'ERROR', UpdateClients failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
   }
}
/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSendSLCStatus
*
* DESCRIPTION: common function for SLC On from BTClient and dbus connection Complete ACk
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vSendSLCStatus(T_CallData* const prCallData)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vSendSLCStatus ENTERED "));

   if(TRUE == prCallData->b_SLC_status)
   {
      ETG_TRACE_USR3((" fc_phone_tclService_Telephone::vSendSLCStatus ON "));

      //The dbus proxies for this device should already have been created and
      //the dbus signal handlers should already have been installed in the glib
      //main loop thread.
      //Add device address to SLC device List
      tU8 u8BDId= fc_phone_tclDeviceList::getInstance()->u8AddDeviceToDeviceList(prCallData->BD_Address);
      if(u8BDId == INVALID_BDADDRESS_ID)
      {
         ETG_TRACE_USR3((" vSendSLCStatus::Invalid BDAddress "));
         return;
      }
      vSetSLCStatus(prCallData->b_SLC_status);

      //GMMY15-6449(HFP-OFF,HFP-ON)
      vRequestPhoneStatus(prCallData->BD_Address);

      ETG_TRACE_USR3(( "fc_phone_tclService_Telephone::vSendSLCStatus : m_u16PnpInfoVendorID= %u",m_u16PnpInfoVendorID[u8BDId]));
      ETG_TRACE_USR3(( "fc_phone_tclService_Telephone::vSendSLCStatus : m_u16PnpInfoVendorSource= %u",m_u16PnpInfoVendorSource[u8BDId]));

#ifdef PHONE_SIRI
      fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
      if(p_clientHndler_bluetooth && p_clientHndler_bluetooth->bGetAppleDeviceStatusfromBDAddressID(u8BDId))
      {
         //Enable special SIRI functions
         m_oSiriData[u8BDId].vEnableSiri(prCallData->BD_Address);
      }
#endif //PHONE_SIRI

      // Fix for PSARCCB-10813 - Update the SPM substate when phone gets disconnected/connected
      vUpdateSpmSubState();
   }

}


/*******************************************************************************
** FUNCTION:   vUpdateSearchPhoneBookMR_ResponseToSMs(..)
*******************************************************************************/
/* DESCRIPTION:
*   Forward phonebook MR response to SMs
*  PARAMETERS:
*
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vUpdateSearchPhoneBookMR_ResponseToSMs()
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vUpdateSearchPhoneBookMR_ResponseToSMs ENTERED "));

   NULL_CHECK(m_poSMManager)

      //fill process Event Arguments and send processEvent
      FcPhone_processEventArg   tProcessEventArg ;
   tProcessEventArg.bIsItDbusAck =  FALSE ;

   //send update to the SM , that phonebook response
   m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_PHONEBOOK_RESPONSE_OK ,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
}

/*******************************************************************************
** FUNCTION:   vUpdateRingingIncomingCallEvent(..)
*******************************************************************************/
/* DESCRIPTION:
*   Update IncomingCallEvent for the Incoming call
*  PARAMETERS:
*
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vUpdateRingingIncomingCallEvent(tU8 DevHandle)
{
   ETG_TRACE_USR4(("vUpdateRingingIncomingCallEvent ENTERED with DevHandle: %d", DevHandle));
   ETG_TRACE_USR4(("m_u8ActiveDeviceHandle: %d", m_u8ActiveDeviceHandle));

   for(tU8 i=0; i<CallStatusList.size(); ++i)
   {
      if (CallStatusList[i].DeviceHandle == DevHandle)
      {
         if((CallStatusList[i].CallStatus.u8CallState == FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE) \
               || (CallStatusList[i].CallStatus.u8CallState == FC_PHONE_CALLINSTANCE_CALLSTATUS_BUSY))
         {
            // Set incoming call instance. So that it shall be used to send IDLE incoming call event
            fc_phone_tclClientHandler_PhoneBook *p_clientHndler_phonebook = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerPhoneBook;
            p_clientHndler_phonebook->vSetIncomingCallInstanceinPBList(DevHandle, i);
            vSendIncomingCallEvent(i, true);
            return;
         }
      }
   }
   ETG_TRACE_USR4(("No incoming call as of now for the received device handle. Hence no need to update IncomingCallEvent"));
}

/*******************************************************************************
** FUNCTION:   vSendPropertyUpdate(..)
*******************************************************************************/
/* DESCRIPTION:
*   EXported  method to send property updates
*  PARAMETERS:
*     IN:   u16PropertyId :  PropertyId whose update needs to be sent.
*     IN:   rfpCallInstance: Reference to current callinstance
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vSendPropertyUpdate(tU16 u16PropertyId, tVoid* varg)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vSendPropertyUpdate ENTERED with u16PropertyId:%d",u16PropertyId));
   switch(u16PropertyId)
   {
   case MOST_TELFI_C_U16_RINGTONEMUTESTATE:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_RINGTONEMUTESTATE"));
         vUpdateCurrentCallIndexAndProperty(u16PropertyId,varg);
      }
      break;
   case MOST_TELFI_C_U16_SLCSTATUS:
      {
         tS32 s32_connectedDeviceCount ;
         s32_connectedDeviceCount = fc_phone_tclDeviceList::getInstance()->s32GetConnectedDevNumFromDeviceList();

         if(s32_connectedDeviceCount == 0)
         {
            ETG_TRACE_USR3((" Sending SLC OFF to Client "));
            m_oSLCStatusStatus.bCommandStatus = FALSE;
            ail_tenCommunicationError enResult = eUpdateClients(u16PropertyId);


            if (AIL_EN_N_NO_ERROR != enResult)
            {
               ETG_TRACE_ERR(("  MOST_TELFI_C_U16_SLCSTATUS Failed "));
            }
         }
         else
         {
            ETG_TRACE_USR3((" Dropped sending SLC OFF to client AS %d device is still connected ",s32_connectedDeviceCount));
         }
      }
      break;
   case MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITION :
      {
         ETG_TRACE_USR3((" Update MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITION"));

         FCPhone_VRData *ptProcessEventArg = (FCPhone_VRData*)varg;

         m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecActive = ptProcessEventArg->bBTDeviceVoiceRecActive;
         m_oBTDeviceVoiceRecStatus.bBTDeviceVoiceRecSupported = ptProcessEventArg->bBTDeviceVoiceRecSupported;

         ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITION);

         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR(("Voice Recognition Status: 'ERROR', UpdateClients failed, 'ail_tenCommunicationError' = %u",(tU32)enResult));
         }
         // CMG3G-7861 - Siri availability status
         vSendBtVRExtendedStatus();
      }
      break;

   case MOST_TELFI_C_U16_CALLSTATUSNOTICE:      /* Intentional fallthrough */
   case MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED:
   {
      ETG_TRACE_USR3((" Update MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED"));
      if (FALSE == m_bHeldtoActiveSwapTimerRunning)
      {
         vUpdateCallStatus();
      }
   }
   break;

   case MOST_TELFI_C_U16_ACTIVECALLINHANDSET:   /* Intentional fallthrough */
   case MOST_TELFI_C_U16_CELLSIGNALSTRENGTH:    /* Intentional fallthrough */
   case MOST_TELFI_C_U16_SOURCEAVAILABLE:       /* Intentional fallthrough */
   {
      ail_tenCommunicationError enResult = eUpdateClients(u16PropertyId);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("Update with %d; has Failed ", u16PropertyId));
      }
   }
   break;
   case MOST_TELFI_C_U16_VEHICLEAUDIO:
   {
      ail_tenCommunicationError enResult = eUpdateClients(u16PropertyId);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("Update with %d; has Failed ", u16PropertyId));
      }
   }
   break;
   case MOST_TELFI_C_U16_MICROPHONEMUTESTATE:
   {
      ETG_TRACE_USR3(("MOST_TELFI_C_U16_MICROPHONEMUTESTATE"));
      vUpdateCurrentCallIndexAndProperty(u16PropertyId,varg);
   }
   break;
   case MOST_TELFI_C_U16_INCOMINGCALLEVENT:
   {
      ETG_TRACE_USR3(("MOST_TELFI_C_U16_INCOMINGCALLEVENT"));
      vUpdateCurrentCallIndexAndProperty(u16PropertyId,varg);

      // Incoming call event update when device is connected
      if(CallStatusList.size() == 0)
      {
         ETG_TRACE_USR3((" No device connected"));
         vResetIncomingEventInfoVariable();
         ail_tenCommunicationError enResult = eUpdateClients(u16PropertyId);

         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR(("Update with %d; has Failed ", u16PropertyId));
         }
         break;
      }

      // Incoming call event update when there is no active call
      fc_phone_tclClientHandler_PhoneBook *p_clientHndler_phonebook = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerPhoneBook;
      tU8 devHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(m_oCallInstances[m_u8CurrentCallIndex].m_u8BDAddressId);
      if(FC_PHONE_DEVICEHANDLE_UNKNOWN == devHandle)
      {
         ETG_TRACE_USR3(("CallInstance might be RESET. Hence sending IDLE INCOMINGCALLEVENT if any ACTIVE INCOMINGCALLEVENT"));

         for(tU8 ind = 0; ind < CallStatusList.size(); ++ind)
         {
            if((CallStatusList[ind].DeviceHandle == m_u8ActiveDeviceHandle) && (m_sIncomingEventInfo.CallInstance == ind))
            {
               vSendIncomingCallEvent(ind, false);
               p_clientHndler_phonebook->vSetIncomingCallInstanceinPBList(m_u8ActiveDeviceHandle, 0xFF);
               break;
            }
         }
         break;
      }

      // Incoming call event update from the SM
      tU8 u8IncomingCallIndex = p_clientHndler_phonebook->u8GetIncomingCallInstancefromPBList(devHandle);

      if(FC_PHONE_CALLINSTANCE_INCOMINGCALLEVENT_RINGING == m_oCallInstances[m_u8CurrentCallIndex].m_u16IncomingCallEvent)
      {
         ETG_TRACE_USR3((" Ringing INCOMINGCALLEVENT for ACTIVE device"));
         vUpdateRingingIncomingCallEvent(devHandle);
      }
      else
      {
         ETG_TRACE_USR3((" IDLE INCOMINGCALLEVENT for ACTIVE device"));
         vSendIncomingCallEvent(u8IncomingCallIndex, false);
         p_clientHndler_phonebook->vSetIncomingCallInstanceinPBList(devHandle, 0xFF);
      }
   }
   break;
   case MOST_TELFI_C_U16_ACTIVEPHONEDEVICE:
   {
      ail_tenCommunicationError enResult = eUpdateClients(u16PropertyId);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("Update with %d; has Failed ", u16PropertyId));
      }
   }
   break;
   case MOST_TELFI_C_U16_OUTGOINGCALLFAILEDEVENT:
   {
      ETG_TRACE_USR3(("MOST_TELFI_C_U16_OUTGOINGCALLFAILEDEVENT"));
      vUpdateCurrentCallIndexAndProperty(u16PropertyId,varg);
   }
   break;
   case MOST_TELFI_C_U16_INCOMINGCALLREJECTEDEVENT:
   {
      ETG_TRACE_USR3(("MOST_TELFI_C_U16_INCOMINGCALLREJECTEDEVENT"));
      vUpdateCurrentCallIndexAndProperty(u16PropertyId,varg);
   }
   break;
   case MOST_TELFI_C_U16_EXTERNALVRSTATE:
   {

      ETG_TRACE_USR3((" Update MOST_TELFI_C_U16_EXTERNALVRSTATE"));

      ExtVRStateParams *ptProcessEventArg = (ExtVRStateParams*)varg;

      tU8 activeDeviceHandle = u8GetActiveDeviceHandle();
      fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
      tU8 btaddressID = p_clientHndler_bluetooth->u8GetBTAddressIdFromDeviceHandle(activeDeviceHandle);

      const tChar * pu8_btActiveDeviceAddress =  NULLPTR;
      pu8_btActiveDeviceAddress = fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(btaddressID);
      if(pu8_btActiveDeviceAddress)
      {
         char btDeviceAddress[PHONE_SM_BDADDRESS];
         convert_formatted_btaddr_to_raw(pu8_btActiveDeviceAddress,btDeviceAddress,sizeof(btDeviceAddress));
         m_oExternalVRStateStatus.sBdAddress.bSet(btDeviceAddress);
         m_oExternalVRStateStatus.e8ExternalVRState.enType = (most_fi_tcl_e8_ExternalVRState::tenType)ptProcessEventArg->m_u8ExtVRState;
         m_oExternalVRStateStatus.e8VRErrorCode.enType = (most_fi_tcl_e8_VRErrorCode::tenType)ptProcessEventArg->m_u8ErrorCode;

         ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_EXTERNALVRSTATE);
         // NCG3D-136500 : GetCalls would be requested if the External VR state is IDLE.
         // The SCO connection received during the External VR is always treated as the SCO for VR.
         // If an incoming call is received during the Active External VR, then the SCO received for Phone was not taken care. But the SCO would get updated.
         // Hence after the deactivation of External VR, GetCallStatus is requested to confirm, if there is any calls currently available.
         if(FC_PHONE_EXTERNAL_VR_STATE_IDLE == (most_fi_tcl_e8_ExternalVRState::tenType)ptProcessEventArg->m_u8ExtVRState)
         {
            m_pDBus_CCA_If->bGetCallStatus(pu8_btActiveDeviceAddress);
         }

         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR(("Voice Recognition Status: 'ERROR', UpdateClients failed, 'ail_tenCommunicationError' = %u",(tU32)enResult));
         }
      }
      else
      {
         ETG_TRACE_USR3((" pu8_btActiveDeviceAddress is not valid."));
      }
   }
   break;
   default:
   {
      ETG_TRACE_ERR(("Property id: %d; not found", u16PropertyId));
   }
   break;
   }
}


/*******************************************************************************
** FUNCTION:   vPhoneStatus(..)
*******************************************************************************/
/* DESCRIPTION:
*  Notification from D-Bus for phone status
*  PARAMETERS:
*     IN:   u16PropertyId :  PropertyId whose update needs to be sent.
*     IN:   rfpCallInstance: Reference to current callinstance
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vPhoneStatus(const T_PhoneStatus* const prPhoneData, tU8 BDAddressID)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vPhoneStatus ENTERED with BDAddressID: %d", BDAddressID));
   /* Update the phone data and if there is any change start notifying with property */
   fc_phone_tclPhoneData oPhoneData(prPhoneData);

   /* Get Notifications to be sent */
   tU16 u16Notifications = oPhoneData.u16Compare(m_oPhoneData[BDAddressID]);

   ETG_TRACE_USR3(("vPhoneStatus: u16Notifications %d", u16Notifications));

   m_oPhoneData[BDAddressID] = oPhoneData;
   m_u8CurrentBDAddressIDforUpdate = BDAddressID;
   // Fix for RTC- 671770, RTC-681288(SMART)
   tU8* pu8_btDeviceName = (tU8*)const_cast<tChar*>(m_pobtTelClient->sGetDeviceNameFromBtAddressId(BDAddressID));
   if(pu8_btDeviceName)
   {
      if (0 != u8StringCompare(m_oPhoneData[BDAddressID].m_achCellDeviceName, (const char*)pu8_btDeviceName))
      {
         vStringCopy(m_oPhoneData[BDAddressID].m_achCellDeviceName, (const char*) pu8_btDeviceName, FC_PHONE_STRING_LENGTH_MAX);
      }
   }
   else
   {
      ETG_TRACE_ERR((" DeviceName is NULL"));
   }

   if(   ((u16Notifications & FC_PHONE_PHONEDATA_MASK_NWSTATUS) == FC_PHONE_PHONEDATA_MASK_NWSTATUS )
      || ((u16Notifications & FC_PHONE_PHONEDATA_MASK_ROAMINGSTATUS) == FC_PHONE_PHONEDATA_MASK_ROAMINGSTATUS)
      )
   {
      ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_CELLNETWORKSTATUS);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         /* Throw up error */
         ETG_TRACE_ERR(( "CellNetworkStatus_Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }

   if((u16Notifications & FC_PHONE_PHONEDATA_MASK_CHARGE) == FC_PHONE_PHONEDATA_MASK_CHARGE )
   {
      ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_CELLSTATEOFCHARGE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         /* Throw up error */
         ETG_TRACE_ERR(( "CellStateOfChargeStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }

   if((u16Notifications & FC_PHONE_PHONEDATA_MASK_SIGNALSTRENGTH) == FC_PHONE_PHONEDATA_MASK_SIGNALSTRENGTH )
   {
      ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_CELLSIGNALSTRENGTH);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         /* Throw up error */
         ETG_TRACE_ERR(( "CellSignalStrengthStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }

   if((u16Notifications & FC_PHONE_PHONEDATA_MASK_NWOPNAME) == FC_PHONE_PHONEDATA_MASK_NWOPNAME)
   {
      ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_NETWORKOPERATOR);
      ETG_TRACE_USR3(("Updating Network operator"));
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vSendNetworkOperatorName: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u",(tU32)enResult));
      }
   }

   if(   ((u16Notifications & FC_PHONE_PHONEDATA_MASK_CALLWAITING) == FC_PHONE_PHONEDATA_MASK_CALLWAITING )
      || ((u16Notifications & FC_PHONE_PHONEDATA_MASK_THREEWAYCALL) == FC_PHONE_PHONEDATA_MASK_THREEWAYCALL)
      )
   {
      ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_HFSERVICE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         /* Throw up error */
         ETG_TRACE_ERR(( "vSendHFServiceStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }

   if((u16Notifications & FC_PHONE_PHONEDATA_MASK_VR) == FC_PHONE_PHONEDATA_MASK_VR)
   {
      // When "Change Phone" is clicked from HMI, HMI will be waiting for response from FC-Phone
      // either as TRUE or FALSE for the VR session in the current Active Phone device.
      // But during Device switching, the Phone status and the VR status is updated for the recently switched Active device.
      // And VR is updated as FALSE. The below condition check is used to avoid this intermediate update from FC_Phone during "Change Phone"
      // Anyways the SMVR is waiting for response from Stack
      // and it will be updated once received(may be TRUE or FALSE(if VR Activation is failed)).
      if (FALSE == (m_poSMVRSession->bIsVRActivationRequestPendingFromBTStack()))
      {
         // CMG3G-7861 - Siri availability status
         vSendBtVRStatus(true, BDAddressID);
      }
   }

#ifdef PHONE_HFENHANCED
   /*TODO: HFENHANCED feature related Most-fi is not available
   because of old version of di_most used in JAC project
   Once avaialble, it has to be removed. */
   if((u16Notifications & FC_PHONE_PHONEDATA_MASK_ENHANCECALLCTRL) == FC_PHONE_PHONEDATA_MASK_ENHANCECALLCTRL)
   {
      ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_HANDSFREEENHANCEDCALLCONTROL);


      if (AIL_EN_N_NO_ERROR != enResult)
      {
         /* Throw up error */
         ETG_TRACE_ERR(( "vSendEnhancedCallControlStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
#endif /* PHONE_HFENHANCED */

   //Fix for GMMY17-3170.
   //SLC status update is done after the DeviceList property update is received. This is to assure the SLC status updation only for the Connected devices.
   ail_tenCommunicationError enResult =  eUpdateClients(MOST_TELFI_C_U16_SLCSTATUS);

   if (AIL_EN_N_NO_ERROR != enResult)
   {
      /* Throw up error */
      ETG_TRACE_ERR(( "vSendSLCStatus: 'ERROR', UpdateClients failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
   }

   m_u8CurrentBDAddressIDforUpdate = INVALID_BDADDRESS_ID;
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSendMethodAck
*
* DESCRIPTION: This function will send method Results
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vSendMethodAck(tU16 u16MethodId, fc_phone_tclCallInstance*& rfpCallInstance)
{
   //send corresponding Method results
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSendMethodAck ENTERED "));
   if(!rfpCallInstance)
   {
      ETG_TRACE_USR4(("rfpCallInstance is NULL. Hence returning"));
      return;
   }
   switch(u16MethodId)
   {
   case MOST_TELFI_C_U16_ACCEPTCALL:
   {
      ETG_TRACE_USR3(("MOST_TELFI_C_U16_ACCEPTCALL"));

      //m_u16StoredCallInstanceForAccept is used in sending the MR. Hence it's resetted after sending MR or ME
      if(rfpCallInstance->m_AckError == FC_PHONE_CALLINSTANCE_ACKERROR_OK)
      {
         ETG_TRACE_USR3((" if(rfpCallInstance->m_AckError == FC_PHONE_CALLINSTANCE_ACKERROR_OK) "));
         //create the method result and send
         most_telfi_tclMsgAcceptCallMethodResult oAcceptCall_MR;

         oAcceptCall_MR.u16CallInstance = static_cast<tU16>(m_u16StoredCallInstanceForAccept);
         oAcceptCall_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
         oAcceptCall_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;
         ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oAcceptCall_MR.e8CallStatus.enType)));

         /* Get the coresponding Call Instance */
         ETG_TRACE_USR2(( "fc_phone_tclService_Telephone::vHandle_AcceptCall_MethodResult : u16CallInstance= %u", rfpCallInstance->m_u16Id ));


         fi_tclVisitorMessage oMRMsg(oAcceptCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

         // Here the message related header information is added and by this the
         // creation of the message is completed.
         vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oAcceptCallMessage.u16SourceAppID,               // Target AppID
               m_oAcceptCallMessage.u16RegisterID ,                // RegisterID
               m_oAcceptCallMessage.u16CmdCounter ,                // CmdCounter,
               m_oAcceptCallMessage.u16FunctionID ,                // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
         );

         fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //DBUS ack received for accept call. Hence unlock.

         if (oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(( "vHandle_AcceptCall_MethodStart: 'MethodResult', enPostMessage() failed, "
                     "'ail_tenCommunicationError' = %u", (tU32)enResult ));
               // We can come here e.g. if the client has changed his application
               // state to OFF in the meantime. Therefore we don?t throw an assert.
            }
            else
            {
               m_bAcceptMethodStartFlag = FALSE;
            }
         }
         oAcceptCall_MR.vDestroy();
      }
      else
      {
         ETG_TRACE_ERR(("   Sending Method Error ::oAcceptCall_MError...... "));

         most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

         e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
         vHandle_AcceptCall_Error(e8ErrorCode);
      }

      tU8 devHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN;
      devHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(rfpCallInstance->m_u8BDAddressId);
      ETG_TRACE_USR4(("devHandle = %u and m_u8ActiveDeviceHandle = %u", devHandle, u8GetActiveDeviceHandle()));
      if((FC_PHONE_DEVICEHANDLE_UNKNOWN != devHandle) && (devHandle != u8GetActiveDeviceHandle()))
      {
         ETG_TRACE_USR4(("The received AcceptCall_MR or ME is for the Passive Device"));
         if (m_u8TransferCallCause == FC_PHONE_TRANSFERCALL_EXT_TRIGGER_DNT_RELEASE_CHANNEL)
         {
         /* Since TransferCalltoHandset for the Active device failed,
          * AcceptCall_ME is sent for the Passive Device and hence
          * m_u8TransferCallCause == FC_PHONE_TRANSFERCALL_EXT_TRIGGER_DNT_RELEASE_CHANNEL,
          * therefore no need to switch the Passive device to ACTIVE
          */

            ETG_TRACE_USR4((" m_u8TransferCallCause == FC_PHONE_TRANSFERCALL_EXT_TRIGGER_DNT_RELEASE_CHANNEL, hence no need switch it to ACTIVE now"));
         }
         else
         {
            ETG_TRACE_USR4(("If the received AcceptCall_MR or ME is for the Passive Device, switch it to ACTIVE now"));

            ETG_TRACE_USR4(("m_u8ActiveDeviceHandletobeRetained = %u", m_u8ActiveDeviceHandletobeRetained));
            if (devHandle == m_u8ActiveDeviceHandletobeRetained)
            {
               m_u8ActiveDeviceHandletobeRetained = FC_PHONE_DEVICEHANDLE_UNKNOWN;
            }
            else
            {
               m_u8ActiveDeviceHandletobeRetained = u8GetActiveDeviceHandle();
            }
            ETG_TRACE_USR4(("m_u8ActiveDeviceHandletobeRetained = %u", m_u8ActiveDeviceHandletobeRetained));
            vSwitchActivePassive(devHandle, false);
         }
      }
      ETG_TRACE_USR4(("m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM"));
      m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM ;
   }
      break;

   case MOST_TELFI_C_U16_REJECTCALL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_REJECTCALL"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            //create the method result and send
            most_telfi_tclMsgRejectCallMethodResult oRejectCall_MR;

            m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM;

            oRejectCall_MR.u16CallInstance = rfpCallInstance->m_u16Id;
            oRejectCall_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
            oRejectCall_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oRejectCall_MR.e8CallStatus.enType)));
            /* Get the coresponding Call Instance */
            ETG_TRACE_USR2(( "fc_phone_tclService_Telephone::vHandle_RejectCall_MethodResult : u16CallInstance= %u", rfpCallInstance->m_u16Id ));


            fi_tclVisitorMessage oMRMsg(oRejectCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oRejectCallMessage.u16SourceAppID,        // Target AppID
               m_oRejectCallMessage.u16RegisterID ,         // RegisterID
               m_oRejectCallMessage.u16CmdCounter ,         // CmdCounter,
               m_oRejectCallMessage.u16FunctionID ,         // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT         // Opcode
               );


            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (AIL_EN_N_NO_ERROR != enResult)
               {
                  ETG_TRACE_ERR(( "vHandle_RejectCall_MethodStart: 'MethodResult', enPostMessage failed, "
                        "'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            oRejectCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::oRejectCall_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_RejectCall_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_HANGUPCALL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_HANGUPCALL"));

         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            //create the method result and send
            most_telfi_tclMsgHangUpCallMethodResult oHangUpCall_MR;
            oHangUpCall_MR.u16CallInstance = rfpCallInstance->m_u16Id;
            oHangUpCall_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
            oHangUpCall_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oHangUpCall_MR.e8CallStatus.enType)));
            /* Get the coresponding Call Instance */

            ETG_TRACE_USR2(( "fc_phone_tclService_Telephone::vHandle_HangUpCall_MethodResult : u16CallInstance= %u", rfpCallInstance->m_u16Id ));

            fi_tclVisitorMessage oMRMsg(oHangUpCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oHangUpCallMessage.u16SourceAppID,      // Target AppID
               m_oHangUpCallMessage.u16RegisterID ,       // RegisterID
               m_oHangUpCallMessage.u16CmdCounter ,       // CmdCounter,
               m_oHangUpCallMessage.u16FunctionID ,       // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT       // Opcode
               );


            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (AIL_EN_N_NO_ERROR != enResult)
               {
                  ETG_TRACE_ERR(( "vHandle_HangupCall_MethodStart: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            oHangUpCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::oHangUpCall_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_HangUpCall_Error(e8ErrorCode);
         }

         //CMG3GB-2551.
         //The parameter "m_u16StoredCallInstanceForHangUp" must be set to default on successful D-Bus acknowledgement.
         m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM;
         ETG_TRACE_USR4(("m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM"));

      }
      break;

   case MOST_TELFI_C_U16_IGNORECALL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_IGNORECALL"));
         ETG_TRACE_USR4(("m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM"));
         m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM;

         if (FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            //create the method result and send
            most_telfi_tclMsgIgnoreCallMethodResult oIgnoreCall_MR;

            oIgnoreCall_MR.u16CallInstance = rfpCallInstance->m_u16Id;
            oIgnoreCall_MR.e8CallStatus.enType =
                  (most_fi_tcl_e8_TelCallStatus::tenType) rfpCallInstance->m_u16CallStatusNotice;
            oIgnoreCall_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            ETG_TRACE_USR1((" Call Status:: %u", ETG_ENUM(TR_CALLSTATUS, oIgnoreCall_MR.e8CallStatus.enType)));
            /* Get the coresponding Call Instance */
            ETG_TRACE_USR2(("fc_phone_tclService_Telephone::vHandle_IgnoreCall_MethodResult : u16CallInstance= %u",
                  rfpCallInstance->m_u16Id));

            fi_tclVisitorMessage oMRMsg(oIgnoreCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData(oMRMsg, // ServiceDataMsg
            m_oIgnoreCallMessage.u16SourceAppID, // Target AppID
            m_oIgnoreCallMessage.u16RegisterID, // RegisterID
            m_oIgnoreCallMessage.u16CmdCounter, // CmdCounter,
            m_oIgnoreCallMessage.u16FunctionID, // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
            );

            fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //DBUS ack received for ignore call. Hence unlock.

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_ERR(("vHandle_IgnoreCall_MethodStart: 'MethodResult', enPostMessage failed, "
                        "'ail_tenCommunicationError' = %u", (tU32) enResult));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            oIgnoreCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::oIgnoreCall_MError...... "));
            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;
            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_IgnoreCall_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_HOLDCALL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_HOLDCALL"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            //create the method result and send
            most_telfi_tclMsgHoldCallMethodResult oHoldCall_MR;

            //When Hold call method start is received in FC_Phone with call instance value as FF, then the conference call should be put on hold.
            //Assign call instance value in method result as FF if Hold received is for all calls
            if(TRUE == m_bHoldOrResumeAllFlag)
            {
               oHoldCall_MR.u16CallInstance = FC_PHONE_HOLD_ALL;
               m_bHoldOrResumeAllFlag = FALSE;
            }
            else
            {
               oHoldCall_MR.u16CallInstance = rfpCallInstance->m_u16Id;
            }

            oHoldCall_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
            oHoldCall_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oHoldCall_MR.e8CallStatus.enType)));
            /* Get the coresponding Call Instance */
            ETG_TRACE_USR2(( "fc_phone_tclService_Telephone::vHandle_HoldCall_MethodResult : u16CallInstance= %u", rfpCallInstance->m_u16Id ));


            fi_tclVisitorMessage oMRMsg(oHoldCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oHoldCallMessage.u16SourceAppID,       // Target AppID
               m_oHoldCallMessage.u16RegisterID ,        // RegisterID
               m_oHoldCallMessage.u16CmdCounter ,        // CmdCounter,
               m_oHoldCallMessage.u16FunctionID ,        // Function ID,
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            if (oMRMsg.bIsValid())
            {

               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (AIL_EN_N_NO_ERROR != enResult)
               {
                  ETG_TRACE_ERR(( "HoldcallMethodResult called      HoldcallMethodResult: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we dont throw an assert.
               }
            }
            oHoldCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::oHoldCall_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_HoldCall_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_DIAL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_DIAL"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3((" if(rfpCallInstance->m_AckError == FC_PHONE_CALLINSTANCE_ACKERROR_OK) "));
            //create the method result and send
            most_telfi_tclMsgDialMethodResult oDial_MR;

            oDial_MR.u16CallInstance = rfpCallInstance->m_u16Id;
            oDial_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
            oDial_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oDial_MR.e8CallStatus.enType)));

            fi_tclVisitorMessage oMRMsg(oDial_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oDialMessage.u16SourceAppID,               // Target AppID
               m_oDialMessage.u16RegisterID ,                // RegisterID
               m_oDialMessage.u16CmdCounter ,                // CmdCounter,
               m_oDialMessage.u16FunctionID ,                // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //DBus ack is received for Dial dial. Hence unlock mutex.

            if (oMRMsg.bIsValid())
            {

               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_ERR(( "DialMethodResult called      DialMethodResult: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
               else
               {
                  m_bDialMethodStartFlag = FALSE;
               }
            }
            else
            {
               ETG_TRACE_ERR((" INVALID MESSAGE "));
            }
            oDial_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("  Sending Method Error ::oDial_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_Dial_Error(e8ErrorCode);

            // Fix for NCG3D-63471
            // In case, if IDLE CS is received while waiting for Dial response from stack to initiate second call,
            // that IDLE CS is not handled. Now in this scenario, if Dial receives ERROR response, then both the SMs should be deleted.
            // But only the Dial_SM will be deleted based on DialError response.
            // In order to delete the first SM, GetCalls is posted.
            m_pDBus_CCA_If->bGetCallStatus(fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(rfpCallInstance->m_u8BDAddressId));
         }
      }
      break;

   case MOST_TELFI_C_U16_DIALEXTENDED:
   {
      ETG_TRACE_USR3(("MOST_TELFI_C_U16_DIALEXTENDED"));
      if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
      {
         ETG_TRACE_USR3((" if(rfpCallInstance->m_AckError == FC_PHONE_CALLINSTANCE_ACKERROR_OK) "));
         //create the method result and send
         most_telfi_tclMsgDialExtendedMethodResult oDialExtended_MR;

         oDialExtended_MR.u16CallInstance = rfpCallInstance->m_u16Id;
         oDialExtended_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
         oDialExtended_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

         ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oDialExtended_MR.e8CallStatus.enType)));

         fi_tclVisitorMessage oMRMsg(oDialExtended_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

         // Here the message related header information is added and by this the
         // creation of the message is completed.
         vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oDialMessage.u16SourceAppID,               // Target AppID
               m_oDialMessage.u16RegisterID ,                // RegisterID
               m_oDialMessage.u16CmdCounter ,                // CmdCounter,
               m_oDialMessage.u16FunctionID ,                // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
         );

         fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //DBus ack is received for Dial dial. Hence unlock mutex.

         if (oMRMsg.bIsValid())
         {

            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(( "DialExtendedMethodResult called      DialExtendedMethodResult: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
               // We can come here e.g. if the client has changed his application
               // state to OFF in the meantime. Therefore we don?t throw an assert.
            }
            else
            {
               m_bDialMethodStartFlag = FALSE;
            }
         }
         else
         {
            ETG_TRACE_ERR((" INVALID MESSAGE "));
         }
         oDialExtended_MR.vDestroy();
      }
      else
      {
         ETG_TRACE_ERR(("  Sending Method Error ::oDial_MError...... "));

         most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

         e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
         vHandle_DialExtended_Error(e8ErrorCode);

         // Fix for NCG3D-63471
         // In case, if IDLE CS is received while waiting for Dial response from stack to initiate second call,
         // that IDLE CS is not handled. Now in this scenario, if Dial receives ERROR response, then both the SMs should be deleted.
         // But only the Dial_SM will be deleted based on DialError response.
         // In order to delete the first SM, GetCalls is posted.
         m_pDBus_CCA_If->bGetCallStatus(fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(rfpCallInstance->m_u8BDAddressId));
      }
   }
   break;

   case MOST_TELFI_C_U16_SPEEDDIAL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_SPEEDDIAL"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3((" if(rfpCallInstance->m_AckError == FC_PHONE_CALLINSTANCE_ACKERROR_OK) "));
            //create the method result and send
            most_telfi_tclMsgSpeedDialMethodResult oSpeedDial_MR;

            oSpeedDial_MR.u16CallInstance = rfpCallInstance->m_u16Id;
            oSpeedDial_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
            oSpeedDial_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oSpeedDial_MR.e8CallStatus.enType)));
            fi_tclVisitorMessage oMRMsg(oSpeedDial_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oSpeedDialMessage.u16SourceAppID,               // Target AppID
               m_oSpeedDialMessage.u16RegisterID ,                // RegisterID
               m_oSpeedDialMessage.u16CmdCounter ,                // CmdCounter,
               m_oSpeedDialMessage.u16FunctionID ,                // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            //GMMY17-4565
            fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //DBus ack is received for SpeedDial. Hence unlock mutex.

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_ERR(( "SpeedDialMethodResult called      SpeedDialMethodResult: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            else
            {
               ETG_TRACE_ERR((" INVALID MESSAGE "));
            }
            oSpeedDial_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("  Sending Method Error ::oSpeedDial_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_SpeedDial_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_CANCELOUTGOINGCALL :
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_CANCELOUTGOINGCALL"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3((" if(rfpCallInstance->m_AckError == FC_PHONE_CALLINSTANCE_ACKERROR_OK) "));
            //create the method result and send
            most_telfi_tclMsgCancelOutgoingCallMethodResult oCancelOutgoingCall_MR;

            oCancelOutgoingCall_MR.u16CallInstance = rfpCallInstance->m_u16Id;
            oCancelOutgoingCall_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
            oCancelOutgoingCall_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS, oCancelOutgoingCall_MR.e8CallStatus.enType)));

            fi_tclVisitorMessage oMRMsg(oCancelOutgoingCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oCancelOutgoingCallMessage.u16SourceAppID,               // Target AppID
               m_oCancelOutgoingCallMessage.u16RegisterID ,                // RegisterID
               m_oCancelOutgoingCallMessage.u16CmdCounter ,                // CmdCounter,
               m_oCancelOutgoingCallMessage.u16FunctionID ,                // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //DBus ack is received for Canceloutgoing call. Hence unlock mutex.

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (AIL_EN_N_NO_ERROR != enResult)
               {
                  ETG_TRACE_ERR(( "CancelOutgoingCall_MR called      oCancelOutgoingCall_MR: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            else
            {
               ETG_TRACE_ERR(("oCancelOutgoingCall_MR : INVALID MESSAGE      "));
            }

            oCancelOutgoingCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR((" Sending Method Error ::oCancelOutgoingCall_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_CancelOutgoingCall_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_SENDDTMF:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_SENDDTMF"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3((" SENDING MOST_TELFI_C_U16_SENDDTMF METHOD RESULT "));
            //Create the method result and send
            most_telfi_tclMsgSendDTMFMethodResult oSendDTMF_MR;

            oSendDTMF_MR.u16CallInstance = rfpCallInstance->m_u16Id ;
            oSendDTMF_MR.bCommandStatus  = TRUE ;

            fi_tclVisitorMessage oMRMsg(oSendDTMF_MR,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData(oMRMsg,                          // ServiceDataMsg
               m_oSendDTMFMessage.u16SourceAppID,       // Target AppID
               m_oSendDTMFMessage.u16RegisterID ,        // RegisterID
               m_oSendDTMFMessage.u16CmdCounter ,        // CmdCounter,
               m_oSendDTMFMessage.u16FunctionID ,        // Function ID,
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            if(oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
               if(AIL_EN_N_NO_ERROR != enResult)
               {
                  ETG_TRACE_ERR(("SendDTMFMethodResult: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            oSendDTMF_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::oSendDTMF_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_SendDTMF_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_RESUMECALL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_RESUMECALL"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3((" SENDING MOST_TELFI_C_U16_RESUMECALL METHOD RESULT "));

            //create the method result and send
            most_telfi_tclMsgResumeCallMethodResult oResumeCall_MR;

            //When Hold call method start is received in FC_Phone with call instance value as FF, then the conference call should be put on hold.
            //Assign call instance value in method result as FF if Hold received is for all calls
            if(TRUE == m_bHoldOrResumeAllFlag)
            {
               oResumeCall_MR.u16CallInstance = FC_PHONE_RESUME_ALL;
               m_bHoldOrResumeAllFlag = FALSE;
            }
            else
            {
               oResumeCall_MR.u16CallInstance = rfpCallInstance->m_u16Id ;
            }

            oResumeCall_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
            oResumeCall_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oResumeCall_MR.e8CallStatus.enType)));

            fi_tclVisitorMessage oMRMsg(oResumeCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oResumeCallMessage.u16SourceAppID,       // Target AppID
               m_oResumeCallMessage.u16RegisterID ,        // RegisterID
               m_oResumeCallMessage.u16CmdCounter ,        // CmdCounter,
               m_oResumeCallMessage.u16FunctionID ,        // Function ID,
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_ERR(( "ResumeCallMethodResult: 'MethodResult', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            oResumeCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::oResumeCallCall_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_ResumeCall_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_SWAPCALL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_SWAPCALL"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3((" SENDING MOST_TELFI_C_U16_SWAPCALL METHOD RESULT "));

            //create the method result and send
            most_telfi_tclMsgSwapCallMethodResult oSwapCall_MR;

            oSwapCall_MR.u16CallInstance = rfpCallInstance->m_u16Id ;
            oSwapCall_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;

            /* Get the coresponding Call Instance */
            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oSwapCall_MR.e8CallStatus.enType)));

            fi_tclVisitorMessage oMRMsg(oSwapCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oSwapCallMessage.u16SourceAppID,       // Target AppID
               m_oSwapCallMessage.u16RegisterID ,        // RegisterID
               m_oSwapCallMessage.u16CmdCounter ,        // CmdCounter,
               m_oSwapCallMessage.u16FunctionID ,        // Function ID,
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (AIL_EN_N_NO_ERROR != enResult)
               {
                  ETG_TRACE_ERR(( "oSwapCall_MR: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }

            }
            oSwapCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::oSwapCall_MError...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_SwapCall_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_TRANSFERCALLTOHANDSET:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_TRANSFERCALLTOHANDSET"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3(("SENDING MOST_TELFI_C_U16_TRANSFERCALL TO HANDSET METHOD RESULT "));

            m_u8TransferCallCause = 0;

            for(tU16 Index = 0; Index < m_oTransferCallIncomingMsgQueue.size(); Index++)
            {
               ETG_TRACE_USR3(("   Index= %d ",Index));

               //Create the method result and send
               most_telfi_tclMsgTransferCallToHandsetMethodResult oTransferCallHandSet_MR;
               /* Get the coresponding Call Instance */
               fi_tclVisitorMessage oMRMsg(oTransferCallHandSet_MR,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
               ETG_TRACE_USR2(("   m_oTransferCallIncomingMsgQueue.size = %d ",(int)m_oTransferCallIncomingMsgQueue.size()));

               // Here the message related header information is added and by this the
               // creation of the message is completed.
               vInitServiceData(oMRMsg,                          // ServiceDataMsg
                  m_oTransferCallIncomingMsgQueue[Index].u16SourceAppID,       // Target AppID
                  m_oTransferCallIncomingMsgQueue[Index].u16RegisterID ,        // RegisterID
                  m_oTransferCallIncomingMsgQueue[Index].u16CmdCounter ,        // CmdCounter,
                  m_oTransferCallIncomingMsgQueue[Index].u16FunctionID ,        // Function ID,
                  AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
                  );

               if(oMRMsg.bIsValid())
               {
                  // Post message
                  ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
                  if(enResult != AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_ERR(("TransferCall ToHandsetMethodResult: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult));
                     // We can come here e.g. if the client has changed his application
                     // state to OFF in the meantime. Therefore we don?t throw an assert.
                  }
               }

               oTransferCallHandSet_MR.vDestroy();
            }
            m_oTransferCallIncomingMsgQueue.clear();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::TransferCall ToHandset...... "));
            m_u8TransferCallCause = 0;
            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_TransferCallToHandSet_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_TRANSFERCALLTOVEHICLE:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_TRANSFERCALLTOVEHICLE"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3(("SENDING MOST_TELFI_C_U16_TRANSFERCALL TO VEHICLE METHOD RESULT "));
            //Create the method result and send
            most_telfi_tclMsgTransferCallToVehicleMethodResult oTransferCallVehicle_MR;
            oTransferCallVehicle_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;
            //TODO Need to get Device Handle in proper way for 2 call instances. Temp work around
            oTransferCallVehicle_MR.u8DeviceHandle = m_u8ActiveDeviceHandle;

            /* Get the coresponding Call Instance */
            fi_tclVisitorMessage oMRMsg(oTransferCallVehicle_MR,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData(oMRMsg,                          // ServiceDataMsg
               m_oTransferCallToVehicle.u16SourceAppID,       // Target AppID
               m_oTransferCallToVehicle.u16RegisterID ,        // RegisterID
               m_oTransferCallToVehicle.u16CmdCounter ,        // CmdCounter,
               m_oTransferCallToVehicle.u16FunctionID ,        // Function ID,
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            if(oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
               if(enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_ERR(("TransferCall ToVehicleMethodResult: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            oTransferCallVehicle_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::TransferCall ToVehicle...... "));

            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_TransferCallToVehicle_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_CONFERENCEIN:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_CONFERENCEIN"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3(("if(rfpCallInstance->m_AckError == FC_PHONE_CALLINSTANCE_ACKERROR_OK) "));
            //create the method result and send
            most_telfi_tclMsgConferenceInMethodResult oConferenceCall_MR;

            oConferenceCall_MR.oStreamOfCallInstances.u16Items.push_back(u16ConferenceCall1);
            oConferenceCall_MR.oStreamOfCallInstances.u16Items.push_back(u16ConferenceCall2);

            fi_tclVisitorMessage oMRMsg(oConferenceCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oConferenceCallMessage.u16SourceAppID,               // Target AppID
               m_oConferenceCallMessage.u16RegisterID ,                // RegisterID
               m_oConferenceCallMessage.u16CmdCounter ,                // CmdCounter,
               m_oConferenceCallMessage.u16FunctionID ,                // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (AIL_EN_N_NO_ERROR != enResult)
               {
                  ETG_TRACE_ERR(( "  ConferencecallMethodResult called vHandle_ConferenceCall_MethodStart: 'MethodResult', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }

            oConferenceCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::ConferenceCall ...... "));
            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_ConferenceCall_Error(e8ErrorCode);
         }
      }
      break;

   case MOST_TELFI_C_U16_SPLITCALL:
      {
         ETG_TRACE_USR3(("MOST_TELFI_C_U16_SPLITCALL"));
         if(FC_PHONE_CALLINSTANCE_ACKERROR_OK == rfpCallInstance->m_AckError)
         {
            ETG_TRACE_USR3(("if(rfpCallInstance->m_AckError == FC_PHONE_CALLINSTANCE_ACKERROR_OK) "));
            //create the method result and send
            most_telfi_tclMsgSplitCallMethodResult oSplitCall_MR;

            oSplitCall_MR.u16CallInstance = rfpCallInstance->m_u16Id;
            oSplitCall_MR.e8CallStatus.enType = (most_fi_tcl_e8_TelCallStatus::tenType)rfpCallInstance->m_u16CallStatusNotice;
            oSplitCall_MR.bUsingVehicleAudio = rfpCallInstance->m_bUsingVehicleAudio;

            fi_tclVisitorMessage oMRMsg(oSplitCall_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
            ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oSplitCall_MR.e8CallStatus.enType)));

            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,                          // ServiceDataMsg
               m_oSplitCallMessage.u16SourceAppID,               // Target AppID
               m_oSplitCallMessage.u16RegisterID ,                // RegisterID
               m_oSplitCallMessage.u16CmdCounter ,                // CmdCounter,
               m_oSplitCallMessage.u16FunctionID ,                // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
               );

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (AIL_EN_N_NO_ERROR != enResult)
               {
                  ETG_TRACE_ERR(( "vHandle_SplitCall_MethodStart: 'MethodResult', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we don?t throw an assert.
               }
            }
            oSplitCall_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("   Sending Method Error ::SplitCall...... "));
            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

            e8ErrorCode = vGetMostErrorCode(rfpCallInstance);
            vHandle_SplitCall_Error(e8ErrorCode);
         }
      }
      break;

   default:
      {
         ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vSendMethodAck IN DEFAULT "));
      }
      break;
   }

   ETG_TRACE_USR4(("    Exiting vSendMethodAck    "));
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSendPropertyError
*
* DESCRIPTION: This function will send method Results
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vSendPropertyError(tU16 u16PropertyId, tVoid *varg)
{
   ETG_TRACE_USR4(("      Entering vSendPropertyError  ENTERED  "));
   switch(u16PropertyId)
   {
   case MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITION:
      {
         ETG_TRACE_USR4(("   Sending Method Error ::oBTDeviceVoiceRecognition_ME...... "));

         FCPhone_VRData *ptProcessEventArg = (FCPhone_VRData*)varg;
         most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

         if (FC_PHONE_CALLINSTANCE_ACK_STATE_ERROR == ptProcessEventArg->u8ErrorAck)
         {
            ETG_TRACE_USR3((" most_fi_tcl_e8_ErrorCode::FI_EN_BUSY"));
            e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_BUSY;
         }
         else if (FC_PHONE_CALLINSTANCE_ACKERROR_NOTAVAILABLE == ptProcessEventArg->u8ErrorAck)
         {
            ETG_TRACE_USR3((" most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE"));
            e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }
         else
         {
            ETG_TRACE_USR3((" Defalut : most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE"));
            e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }

         vHandle_BtDeviceVoiceRecognition_Error(e8ErrorCode);
         vHandle_BtDeviceVoiceRecognitionExtended_Error(e8ErrorCode);
      }
      break;

   default:
      {
         ETG_TRACE_USR4(("      Entering vSendPropertyError  IN DEFAULT   "));
      }
      break;
   }
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vPrepareLoopBackMsg
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vPrepareLoopBackMsg(tU32 internal_functionID)
{
   // Create a 'generic message' (with tU32 content), intended to be sent
   // to our own server (loopback message)

   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vPrepareLoopBackMsg :: prepare and send loopback msg ENTERED "));

   gm_tclU32Message oServiceReadyMsg(
      _poMainAppl->u16GetAppId(),      // Source AppID
      _poMainAppl->u16GetAppId(),      // Target AppID
      0,                               // RegisterID
      0,                               // CmdCounter
      CCA_C_U16_SRV_FB_TELEPHONE,      // ServiceID
      TELEPHONE_C_U16_IFID_EXAMPLE_1,    // Function ID
      AMT_C_U8_CCAMSG_OPCODE_STATUS    // Opcode
      );

   oServiceReadyMsg.vSetDWord(internal_functionID);

   // Post message
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oServiceReadyMsg, TRUE);

   if (AIL_EN_N_NO_ERROR != enResult)
   {
      ETG_TRACE_USR3(( "vProcessService: 'ServiceReady' Opcode STATUS: enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      // If this message is not posted successfully, we would never make
      // our service available.
      // We never expect to come here.
   }
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vQueryTelephoneNumberCB
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vQueryTelephoneNumberCB(const Async_method_data_t* const vAsync_method_data)
{
   ETG_TRACE_USR3((" In fc_phone_tclService_Telephone::vQueryTelephoneNumberCB ::ENTERED  Telephonenumber %s ",vAsync_method_data->TelephoneNumber));
   //prepare the method result and send to client
   most_telfi_tclMsgQueryTelephoneNumberMethodResult oQueryTelephoneNumber_MR;

   if(vAsync_method_data->TelephoneNumber[0] == 0)
   {
      //Send Method Error for Query Telephone Number
      ETG_TRACE_USR4(("Telephone number is empty"));

      oQueryTelephoneNumber_MR.sTelephoneNumber.bSet("");
   }
   else
   {
      oQueryTelephoneNumber_MR.sTelephoneNumber.bSet(vAsync_method_data->TelephoneNumber);
      ETG_TRACE_USR3(("oQueryTelephoneNumber_MR.sTelephoneNumber.szValue : %s",oQueryTelephoneNumber_MR.sTelephoneNumber.szValue));
   }

   //fill the call instance
   oQueryTelephoneNumber_MR.u16CallInstance = m_oQueryTelephoneNumberMessage.u16InstanceID;
   fi_tclVisitorMessage oMRMsg(oQueryTelephoneNumber_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                          // ServiceDataMsg
      m_oQueryTelephoneNumberMessage.u16SourceAppID,               // Target AppID
      m_oQueryTelephoneNumberMessage.u16RegisterID ,                // RegisterID
      m_oQueryTelephoneNumberMessage.u16CmdCounter ,                // CmdCounter,
      m_oQueryTelephoneNumberMessage.u16FunctionID ,                // Function ID
      AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(( "vQueryTelephoneNumber MR: 'MethodResult', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
         // We can come here e.g. if the client has changed his application
         // state to OFF in the meantime. Therefore we don?t throw an assert.
      }
   }
   else
   {
      ETG_TRACE_ERR(("  oVoiceMail_MR : INVALID MESSAGE "));
   }

   oQueryTelephoneNumber_MR.vDestroy();
   ETG_TRACE_USR4((" In fc_phone_tclService_Telephone::oQueryTelephoneNumber EXIT oQueryTelephoneNumber_MR MR sending finished"));
}


/*******************************************************************************
*
* FUNCTION: vSetBtClient
*
* DESCRIPTION: set BTClient instance
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vSetBtClient(fc_phone_tclClientHandler_BTSettings*& rfpbtTelClient)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vSetBtClient ENTERED "));
   m_pobtTelClient = rfpbtTelClient ;
}

/*******************************************************************************
*
* FUNCTION: sGetDeviceNameFromPhoneData()
*
* DESCRIPTION: function returns CellDevice name
*
* PARAMETER: None.
*
* RETURNVALUE: cell device name pointer.
*
********************************************************************************/
const tChar* fc_phone_tclService_Telephone::sGetDeviceNameFromPhoneData()
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::sGetDeviceNameFromPhoneData ENTERED"));
   if(m_pobtTelClient)
   {
      tU8 u8_btActiveBDAddressId= m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
      if(INVALID_BDADDRESS_ID != u8_btActiveBDAddressId)
      {
         return m_oPhoneData[u8_btActiveBDAddressId].m_achCellDeviceName ;
      }
      else
      {
         ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
      }
   }
   return NULLPTR;
}

/*******************************************************************************
** FUNCTION:   vInitPhoneData
*******************************************************************************/
/* DESCRIPTION:
*  Initialise Phone data
*
*  PARAMETERS:
*          IN : tU8 BDAddressID
*  RETURNVALUE: none
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vInitPhoneData(tU8 BDAddressID)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vInitPhoneData ENTERED with BDAddressID: %d", BDAddressID));
   if(INVALID_BDADDRESS_ID == BDAddressID)
   {
      for(tU8 ind=0; ind < FC_PHONE_MAX_CONNECTED_DEVICE; ind++)
      {
         m_oPhoneData[ind].vInit();
      }
   }
   else
   {
      m_oPhoneData[BDAddressID].vInit();
   }
}


/*******************************************************************************
**	FUNCTION:   vHandle_TransferCallToHandSet_MethodStart(..)
*******************************************************************************/
/*	DESCRIPTION:
*   This is a handler function for TransferCalltoHandSet method
*   Under ACTIVE or ONHOLD CallInstance this function transfers the call to Handset(phone) by traversing State Machine and DBus.
*	PARAMETERS:
*	IN:   poMessage        : This is the received message from a client
*	RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_TransferCallToHandSet_MethodStart(amt_tclServiceData* poMessage)
{
   // BOSCH TRANSFER_AUDIO_HOTFIX
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vHandle_TransferCallToHandSet_MethodStart ENTERED"));
   tU8 u8Index = 0;
   IncomingMessage oTransferCallToHandSet;
   fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;//Fix for CMG3GB-1774

   NULL_CHECK(m_poSMManager)

      if(m_poSMManager->bIsSMRunning(FC_PHONE_SMMANAGER_SMID_TRANSFERCALL, u8Index))
      {
         ETG_TRACE_USR4((" Transfer Call SM is running"));
         tU8 u8SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_FAILED;
         
         for(tU16 Index = 0; Index < FC_PHONE_MAX_CALLINSTANCES; Index++)
         {
            fc_phone_tclCallInstance* loCallInstance = &m_oCallInstances[Index];
            if(FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE != loCallInstance->m_u16CallStatusNotice)
            {
               ETG_TRACE_USR4(("Call Status is active"));
               oTransferCallToHandSet.u16SourceAppID = poMessage->u16GetSourceAppID();
               oTransferCallToHandSet.u16RegisterID  = poMessage->u16GetRegisterID();
               oTransferCallToHandSet.u16CmdCounter  = poMessage->u16GetCmdCounter();
               oTransferCallToHandSet.u16FunctionID  = poMessage->u16GetFunctionID();
               ETG_TRACE_USR1(("      m_oTransferCallToHandSet.u16SourceAppID = %d",oTransferCallToHandSet.u16SourceAppID));
               ETG_TRACE_USR1(("      m_oTransferCallToHandSet.u16RegisterID = %d",oTransferCallToHandSet.u16RegisterID));
               ETG_TRACE_USR1(("      m_oTransferCallToHandSet.u16CmdCounter = %d",oTransferCallToHandSet.u16CmdCounter));
               ETG_TRACE_USR1(("      m_oTransferCallToHandSet.u16FunctionID = %d",oTransferCallToHandSet.u16FunctionID));
               m_oTransferCallIncomingMsgQueue.push_back(oTransferCallToHandSet);
               u8SCOConnectionStatus = p_clientHndler_bluetooth->vGetSCOConnectionStatus(const_cast<tChar *>(fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(loCallInstance->m_u8BDAddressId))); //CMG3G-4847 - To solve Lint

               if(FC_PHONE_DBUS_SCO_CONN_DISCONNECTED == u8SCOConnectionStatus)
               {
                  ETG_TRACE_USR4(("      Audio is already disconnected"));
                  ETG_TRACE_USR4(( "vHandle_TransferCallToHandset_MethodStart: Audio is with Handset hence sending Method Result" ));
                  loCallInstance->m_AckError = FC_PHONE_CALLINSTANCE_ACKERROR_OK;
                  m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_METHOD_STATUS_TRANSFERCALLTOHANDSET,loCallInstance);
                  break;
               }
               else
               {
                  ETG_TRACE_USR4((" Audio is not yet  disconnected"));
                  break;
               }
            }
         }
         return;
      }

      tU8 u8CallInstance = 0;
      tBool bCallTransferToHandSetStatus = FALSE;
      tBool bEventConsumed = FALSE ;
      FcPhone_processEventArg  tProcessEventArg;
      fc_phone_tclCallInstance *pCallInstance[FC_PHONE_MAX_CALLINSTANCES];
      fi_tclVisitorMessage oVisitorMsg(poMessage);
      most_telfi_tclMsgTransferCallToHandsetMethodStart oTransferCallToHandSet_MS;
      fc_phone_tclClientHandler_SystemState *p_ClientHdlerSystemState = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerSystemState;

      if(OSAL_ERROR != oVisitorMsg.s32GetData(oTransferCallToHandSet_MS,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION))
      {
         if(TRUE == p_ClientHdlerSystemState->SystemStateAlertOnLocalPhone)
         {
            p_ClientHdlerSystemState->vHandleSystemStateOnTransferCall();
         }
         else if(TRUE == (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32GetAliveTimerId((LOCAL_PHONE_BT_RAPTIMER_ID)))
         {
            (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(LOCAL_PHONE_BT_RAPTIMER_ID);
         }
         /* Get the corresponding Call Instance */

         ETG_TRACE_USR1((" *****MethodStart:TransferCallToHandSet "));

         oTransferCallToHandSet.u16SourceAppID = poMessage->u16GetSourceAppID();
         oTransferCallToHandSet.u16RegisterID  = poMessage->u16GetRegisterID();
         oTransferCallToHandSet.u16CmdCounter  = poMessage->u16GetCmdCounter();
         oTransferCallToHandSet.u16FunctionID  = poMessage->u16GetFunctionID();
         m_u8TransferCallCause = FC_PHONE_TRANSFERCALL_HMI_TRIGGER;
         m_oTransferCallIncomingMsgQueue.push_back(oTransferCallToHandSet);
         for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
         {
            pCallInstance[u8Index] = NULLPTR;
            u8CallInstance = u8Index;
            if(TRUE == m_oCallInstances[u8Index].m_bUsingVehicleAudio)
            {
               if(0 == u8CallInstance)
               {
                  if(((m_oCallInstances[u8CallInstance].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE) &&
                     (m_oCallInstances[u8CallInstance+1].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE))
                     || ((m_oCallInstances[u8CallInstance].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE) &&
                     (m_oCallInstances[u8CallInstance+1].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE))
                     )
                  {
                     ETG_TRACE_USR3(("/Call In conference/"));
                     tProcessEventArg.bIsItDbusAck =  FALSE;
                     bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_TRANSFERCALLHANDSET_REQUEST_OK,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);

                     if(!bEventConsumed)
                     {
                        ETG_TRACE_ERR(( "vHandle_TransferCallToHandset_MethodStart: FC_PHONE_SM_EVENT_TRANSFERCALLHANDSET_REQUEST_OK event not consumed." ));
                        vHandle_TransferCallToHandSet_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
                     }
                     else
                     {
                        ETG_TRACE_USR3((" Transferring conference call to handset is  Processed"));
                     }
                     return;
                  }
               }
               if((m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE) ||
                  (m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD) || 
				  ((m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_DIALING) && 
				  (p_clientHndler_bluetooth->vGetSCOConnectionStatus(const_cast<tChar *>(fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(m_oCallInstances[u8Index].m_u8BDAddressId))) == FC_PHONE_DBUS_SCO_CONN_ESTABLISHED)))//Fix for CMG3GB-1774
               {
                  bCallTransferToHandSetStatus = TRUE;
                  pCallInstance[u8Index] = &m_oCallInstances[u8Index];
               }
            }
         }

         if((TRUE == bCallTransferToHandSetStatus) && (TRUE == m_oSLCStatusStatus.bCommandStatus))
         {
            m_poSMManager->vStartSM(FC_PHONE_SMMANAGER_SMID_TRANSFERCALL,pCallInstance);
         }
         else
         {
            bCallTransferToHandSetStatus = FALSE;

            ETG_TRACE_ERR(( "vHandle_TransferCallToHandSet_MethodStart: Received message invalid: No ACTIVE CALL INSTANCE." ));
            vHandle_TransferCallToHandSet_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_TransferCallToHandSet_MethodStart: Received message invalid." ));
         vHandle_TransferCallToHandSet_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }

      oTransferCallToHandSet_MS.vDestroy();
}


/*******************************************************************************
**	FUNCTION:   vHandle_TransferCallToVehicle_MethodStart(..)
*******************************************************************************/
/*	DESCRIPTION:
*   This is a handler function for TransferCalltoHandsFree method
*   Under ACTIVE or ONHOLD CallInstance this function transfers the call to Handfree Device by traversing State Machine and DBus.
*	PARAMETERS:
*	IN:   poMessage        : This is the received message from a client
*	RETURNVALUE: None
*******************************************************************************/

tVoid fc_phone_tclService_Telephone::vHandle_TransferCallToVehicle_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vHandle_TransferCallToVehicle_MethodStart ENTERED"));

   tBool CallTransferStatus = FALSE;
   tBool bEventConsumed = FALSE ;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   tU8 ActiveBTDeviceAddressId;
   FcPhone_processEventArg  tProcessEventArg;
   most_telfi_tclMsgTransferCallToVehicleMethodStart oTransferCallToVehicle_MS;

   if(OSAL_ERROR != oVisitorMsg.s32GetData(oTransferCallToVehicle_MS,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION))
   {
      /* Get the corresponding Call Instance */
      ETG_TRACE_USR1((" DeviceHandle = %d ",oTransferCallToVehicle_MS.u8DeviceHandle));

      m_oTransferCallToVehicle.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oTransferCallToVehicle.u16RegisterID  = poMessage->u16GetRegisterID();
      m_oTransferCallToVehicle.u16CmdCounter  = poMessage->u16GetCmdCounter();
      m_oTransferCallToVehicle.u16FunctionID  = poMessage->u16GetFunctionID();
      for(tU8 u8Index = 0x00; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
      {
         if(FALSE == m_oCallInstances[u8Index].m_bUsingVehicleAudio)
         {
            //TODO Need to get Device Handle when two devices are connected by passing call instance ID.
            ActiveBTDeviceAddressId = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetBTAddressIdFromDeviceHandle(oTransferCallToVehicle_MS.u8DeviceHandle);
            ETG_TRACE_USR2(("ActiveBTDeviceAddressId = %u     ",ActiveBTDeviceAddressId));
            if(INVALID_BDADDRESS_ID != ActiveBTDeviceAddressId)
            {
               if(ActiveBTDeviceAddressId == m_oCallInstances[u8Index].m_u8BDAddressId)
               {
                  if((FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE == m_oCallInstances[u8Index].m_u16CallStatusNotice) || (FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD == m_oCallInstances[u8Index].m_u16CallStatusNotice) || (FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE == m_oCallInstances[u8Index].m_u16CallStatusNotice)
					 || (FC_PHONE_CALLINSTANCE_CALLSTATUS_DIALING == m_oCallInstances[u8Index].m_u16CallStatusNotice)) //Fix for CMG3GB-1774
                  {
                     //If call in HandSet State Already and if 2nd call comes
                     CallTransferStatus = TRUE;
                     break;
                  }
               }
            }
         }
      }

      if((TRUE == CallTransferStatus) && (TRUE == m_oSLCStatusStatus.bCommandStatus))
      {
         tProcessEventArg.bIsItDbusAck =  FALSE;

         NULL_CHECK(m_poSMManager)

            bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_TRANSFERCALLVEHICLE_REQUEST_OK,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
         if(!bEventConsumed)
         {
            bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_IF_HANDSETCALL_TRANSFER_TOVEHICLE,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
            if(!bEventConsumed)
            {
               ETG_TRACE_USR4(( "vHandle_TransferCallToVehicle_MethodStart: FC_PHONE_SM_EVENT_IF_HANDSETCALL_TRANSFER_TOVEHICLE event not consumed." ));
               vHandle_TransferCallToVehicle_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_TransferCallToVehicle_MethodStart: Received message invalid: No ACTIVE CALL INSTANCE." ));
         vHandle_TransferCallToVehicle_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_TransferCallToVehicle_MethodStart: Received message invalid." ));
      vHandle_TransferCallToVehicle_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   oTransferCallToVehicle_MS.vDestroy();
}


/*******************************************************************************
** FUNCTION:   ChangeDeviceStateResponse
*******************************************************************************/
/* DESCRIPTION:
*  Initialise Phone data
*
*  PARAMETERS:
*          IN : none
*  RETURNVALUE: none
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::ChangeDeviceStateResponse()
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::ChangeDeviceStateResponse: ENTERED "));
   NULL_CHECK(m_poSMManager)
      FcPhone_processEventArg tProcessEventArg;
   tProcessEventArg.bIsItDbusAck =  FALSE;
   //Is it advisable to pass tProcessEventArg as NULL
   m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_CHANGEDEVICESTATE_RESPONSE_OK,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);
}


/*******************************************************************************
*
* FUNCTION: sGetCallStatus()
*
* DESCRIPTION: function returns Active Call exists or not
*
* PARAMETER: None.
*
* RETURNVALUE: TRUE -> Active Call Exists FALSE ->No Active Calls.
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::sGetCallStatus()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::sGetCallStatus ENTERED "));
   tU8 u8Index = 0;
   for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      //TODO IF call is in Hand set already should i hv to consider that case or not
      if((m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE) || (m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD) || (m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE))
      {
         return CALLSTATUS_ACTIVE;
      }
   }
   return CALLSTATUS_INACTIVE;
}


/*******************************************************************************
*
* FUNCTION: vTransferCallToHandset()
*
* DESCRIPTION: function transfers active/hold Calls to handset
*
* PARAMETER: 	//FIX GMMY15-8577 Unable to transfer the call from hands free to handset mode after navigation speech session

*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vTransferCallToHandset(tU8 u8ExtTrigger)
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vTransferCallToHandset "));
   tU8 u8Index = 0;
   tBool bOngoingCall = FALSE;
   NULL_CHECK(m_poSMManager)
   fc_phone_tclCallInstance *pCallInstance[FC_PHONE_MAX_CALLINSTANCES];
   fc_phone_tclClientHandler_SystemState *p_ClientHdlerSystemState = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerSystemState;
   if(p_ClientHdlerSystemState)
   {
      if(TRUE == p_ClientHdlerSystemState->SystemStateAlertOnLocalPhone)
      {
         p_ClientHdlerSystemState->vHandleSystemStateOnTransferCall();
      }
   }

   for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      pCallInstance[u8Index] = NULLPTR;


      if(TRUE == m_oCallInstances[u8Index].m_bUsingVehicleAudio)
      {
         //gmmy15-9887- if in dialing state and onstar session initiated the call is not transfered  to handset as dialing was not checked here.
         if(FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE != m_oCallInstances[u8Index].m_u16CallStatusNotice)
         {
            bOngoingCall = TRUE;
            pCallInstance[u8Index] = &m_oCallInstances[u8Index];
         }
      }
   }

   if(TRUE == bOngoingCall)
   {
      //FIX GMMY15-8577 Unable to transfer the call from hands free to handset mode after navigation speech session
      //If TransfertoHandset() is called from any state machine, it means that Phonechannel has been deactivated by AV Manager because of activation of some other channel.
      //In this case, we release the channel on transfer to handset and acquire it on transferring it back to handsfree.

      //gmmy15-9887

      m_poSMManager->vStartSM(FC_PHONE_SMMANAGER_SMID_TRANSFERCALL,pCallInstance);
      if(FC_PHONE_TRANSFERCALL_EXT_TRIGGER == u8ExtTrigger)
      {
         ETG_TRACE_USR4((" m_u8TransferCallCause = FC_PHONE_TRANSFERCALL_EXT_TRIGGER"));
         m_u8TransferCallCause = FC_PHONE_TRANSFERCALL_EXT_TRIGGER;
      }
      else if(FC_PHONE_TRANSFERCALL_EXT_TRIGGER_RELEASE_CHANNEL == u8ExtTrigger)
      {
         ETG_TRACE_USR4((" m_u8TransferCallCause = FC_PHONE_TRANSFERCALL_EXT_TRIGGER_RELEASE_CHANNEL"));
         m_u8TransferCallCause = FC_PHONE_TRANSFERCALL_EXT_TRIGGER_RELEASE_CHANNEL;
      }
      else if(FC_PHONE_TRANSFERCALL_EXT_TRIGGER_DNT_RELEASE_CHANNEL == u8ExtTrigger)
      {
         ETG_TRACE_USR4((" m_u8TransferCallCause = FC_PHONE_TRANSFERCALL_EXT_TRIGGER_DNT_RELEASE_CHANNEL"));
         m_u8TransferCallCause = FC_PHONE_TRANSFERCALL_EXT_TRIGGER_DNT_RELEASE_CHANNEL;
      }
      bOngoingCall = FALSE;
   }
   else
   {
      ETG_TRACE_USR4((" Call not transfered to handset"));
   }

   ETG_TRACE_USR4((" EXIT: vTransferCallToHandset"));
}


/*******************************************************************************
*
* FUNCTION: vTransferCallToHandsfree()
*
* DESCRIPTION: function transfers active/hold Calls to handsfree
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vTransferCallToHandsfree()
{
   ETG_TRACE_USR3(("ENTER: vTransferCallToHandsfree"));

   tU8 u8Index = 0;
   tU8 u8CallIndex = FC_PHONE_CALLMANAGER_CALLINSTANCE_UNKNOWN;
   tBool bOngoingCall = FALSE;
   FcPhone_processEventArg  tProcessEventArg;

   NULL_CHECK(m_poSMManager)

      for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
      {
         if(FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE == m_oCallInstances[u8Index].m_u8ObjectState)
         {
            //as if its an incoming call no need to call sync of LC_Phone channel , as ringtone channel need to be in sync
            if(FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE != m_oCallInstances[u8Index].m_u16CallStatusNotice)
            {
               m_oCallInstances[u8Index].m_u8AudioChannel = FC_PHONE_AUDIOCHANNEL_PHONE;

               (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->s32SyncAudioMangerOnChannelGranted(m_oCallInstances[u8Index].m_u16Id, FC_PHONE_AUDIOCHANNEL_PHONE);
               bOngoingCall = TRUE;
               u8CallIndex = u8Index;
            }
         }
      }

      if (TRUE == bOngoingCall)
      {
         tProcessEventArg.bIsItDbusAck= FALSE;
         tBool bEventConsumed = FALSE ;

         bEventConsumed = m_poSMManager->bProcessEvent(FC_PHONE_SM_EVENT_IF_HANDSETCALL_TRANSFER_TOVEHICLE,FC_PHONE_EVENTTYPE_BROADCAST,&tProcessEventArg);

         if (bEventConsumed)
         {
            if(u8CallIndex != FC_PHONE_CALLMANAGER_CALLINSTANCE_UNKNOWN)
            {
               fc_phone_tclCallInstance* pCallInstance  = NULLPTR;
               pCallInstance = &m_oCallInstances[u8CallIndex];
               pCallInstance->m_bMicroPhoneMuteStatus = FC_PHONE_MICROPHONE_UNMUTED;
               m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_MICROPHONEMUTESTATE,pCallInstance);
            }
            ETG_TRACE_USR3(("Call transfered to handsfree"));
         }

         bOngoingCall = FALSE;
      }
      else
      {
         ETG_TRACE_USR3(("Call not transfered to handsfree"));
      }

      ETG_TRACE_USR3(("EXIT: vTransferCallToHandset"));
}

/*******************************************************************************
*
* FUNCTION: vGetAudioTransferStatus()
*
* DESCRIPTION: Function to Send AudioTransfer Status to StateMachines
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tU8 fc_phone_tclService_Telephone::vGetAudioTransferStatus()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vGetAudioTransferStatus ENTERED"));
   if(m_pobtTelClient)
   {
      tU8 ActiveBDAddressId =(tU8) m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
      if(INVALID_BDADDRESS_ID != ActiveBDAddressId)
      {
         ETG_TRACE_USR3((" m_oPhoneData[%d].AudioOutStatus = %d",ActiveBDAddressId, m_oPhoneData[ActiveBDAddressId].AudioOutStatus));
         return m_oPhoneData[ActiveBDAddressId].AudioOutStatus;
      }
      else
      {
         ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
      }
   }
   return FC_PHONE_DBUS_SCO_CONN_STATUS_UNKNOWN;
}


/*******************************************************************************
*
* FUNCTION: vResetStateMachines()
*
* DESCRIPTION: Function to Send AudioTransfer Status to StateMachines
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vResetStateMachines(tVoid)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vResetStateMachines ENTERED"));
   NULL_CHECK(m_poSMManager)

      tU8 u8SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_FAILED;
   fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
   for(tU16 u16index = 0 ;u16index < FC_PHONE_MAX_CALLINSTANCES ;u16index++)
   {
      fc_phone_tclCallInstance* pCallInstance = &m_oCallInstances[u16index];
      m_poSMManager->vStartSM(FC_PHONE_SMMANAGER_SMID_IDLE,pCallInstance);
      pCallInstance->vInit();
      /*Send Idle call status to HMI*/
      m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_INCOMINGCALL,pCallInstance);
      m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_CALLSTATUS,pCallInstance);
      m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_VEHICLEAUDIO,pCallInstance);
      u8SCOConnectionStatus = p_clientHndler_bluetooth->vGetSCOConnectionStatus(const_cast<tChar *> (fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(pCallInstance->m_u8BDAddressId))); //CMG3G-4847 - To solve Lint

      if(FC_PHONE_DBUS_SCO_CONN_ESTABLISHED == u8SCOConnectionStatus)
      {
         ETG_TRACE_USR3((" Calling Audio Transfer"));
         m_pDBus_CCA_If->bAudioTransfer(fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(pCallInstance->m_u8BDAddressId), FC_PHONE_CALLTRANSFER_FROMHANSFREE_TOHANDSET);
      }
   }

   ETG_TRACE_USR3((" Reset VR StateMachine"));
   fc_phone_tclSMVRSession* poVRInstance = fc_phone_tclSMVRSession::poGetInstance();
   if(poVRInstance)
   {
      poVRInstance->vCancel();
   }
}


/*******************************************************************************
*
* FUNCTION: vResetSMManager()
*
* DESCRIPTION: Start Idle SM on call end
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vResetSMManager(tBool isDeviceDisconnected)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vResetSMManager ENTERED"));
   NULL_CHECK(m_poSMManager)

   fc_phone_tclCallInstance* pCallInstance = NULLPTR;
   for(tU16 u16index = 0 ;u16index < FC_PHONE_MAX_CALLINSTANCES ;u16index++)
   {
      pCallInstance = &m_oCallInstances[u16index];
      pCallInstance->vInit();
      m_poSMManager->vStartSM(FC_PHONE_SMMANAGER_SMID_IDLE,pCallInstance);

      /*Send Idle call status to HMI*/
      if(FALSE == isDeviceDisconnected) //GMMY15-3655:Not Able To Delete The Device From BT DeviceList After Make An Outgoing Call And Then Disconnect The Phone From The Target
      {
         m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_INCOMINGCALL,pCallInstance);
      }

      //GMMY15-6330:Audio Is Not Changed To HandsFree Mode,After Reconnecting The Phone From The Target
      m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_VEHICLEAUDIO,pCallInstance);
      m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_CALLSTATUS,pCallInstance);
   }

   pCallInstance = &m_oCallInstances[0];
   if(FALSE == isDeviceDisconnected)
   {
      m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_ACTIVECALLINHANDSET,pCallInstance);
   }
}


/*******************************************************************************
*
* FUNCTION: vResetAudioTranferStatusOnEndCall()
*
* DESCRIPTION: Rest AudioOut Status flag to handsfree on end call
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vResetAudioTranferStatusOnEndCall()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vResetAudioTranferStatusOnEndCall ENTERED"));
   if(m_pobtTelClient)
   {
      tU8 ActiveBDAddressId =(tU8) m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
      if(INVALID_BDADDRESS_ID != ActiveBDAddressId)
      {
         m_oPhoneData[ActiveBDAddressId].AudioOutStatus = FC_PHONE_DBUS_SCO_CONN_ESTABLISHED;
      }
      else
      {
         ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
      }
   }
}


/*******************************************************************************
*
* FUNCTION: MuteMicroPhone on User Request
*
* DESCRIPTION: Mute MicroPhone on user request
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::vMuteMicroPhone(tBool bMute, tBool bSendtoEcnr )
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vMuteMicroPhone ENTERED"));
   fc_phone_tclCallInstance* pCallInstance  = NULLPTR;
   tU8 u8Index = 0;
   tBool bReqStatus = REQ_FAILURE;

   if ( bSendtoEcnr )
   {
      // Dont send to ECNR when SCO connection disconnected.
      tU8 u8SwitchEcnrMuteEnable = bMute ? 1 : 0; // Mute off (default)
      ecnrMuteSwitch( u8SwitchEcnrMuteEnable );
   }

   for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      if(TRUE == m_oCallInstances[u8Index].m_bUsingVehicleAudio && m_poSMManager)
      {
         pCallInstance = &m_oCallInstances[u8Index];
         if(bMute)
         {
            ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vMuteMicroPhone is MUTED"));
            pCallInstance->m_bAudioStatus = FC_PHONE_CALLINSTANCE_AUDIOMUTED;
            pCallInstance->m_bMicroPhoneMuteStatus = FC_PHONE_MICROPHONE_MUTED;
            m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_AUDIOMUTE,pCallInstance);
            m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_MICROPHONEMUTESTATE,pCallInstance);
         }
         else
         {
            ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vMuteMicroPhone is UNMUTED"));
            pCallInstance->m_bAudioStatus = FC_PHONE_CALLINSTANCE_AUDIOUNMUTED;
            pCallInstance->m_bMicroPhoneMuteStatus = FC_PHONE_MICROPHONE_UNMUTED;
            m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_AUDIOMUTE,pCallInstance);
            m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_MICROPHONEMUTESTATE,pCallInstance);
         }
         bReqStatus = REQ_SUCCESS;
      }
   }

   return bReqStatus;
}


/*******************************************************************************
*
* FUNCTION: vSetConfSMStatus()
*
* DESCRIPTION:Setter
*
* PARAMETER: True/False
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vSetConfSMStatus(tBool blValue)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSetConfSMStatus ENTERED"));
   isConfSMExecutedFromHmi = blValue;
   isConfSMExecutedFromCallStatus = blValue;
}


/*******************************************************************************
*
* FUNCTION: vGetConfSMStatus()
*
* DESCRIPTION:Getter
*
* PARAMETER: True/False
*
* RETURNVALUE: None.
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::vGetConfSMStatus()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vGetConfSMStatus ENTERED"));
   return isConfSMExecutedFromHmi;
}


/*******************************************************************************
*
* FUNCTION: vTimerCallBackConference
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid vTimerCallBackConference(tU16 u16TimerId)
{
   ETG_TRACE_USR3((" vTimerCallBackConference ENTERED with u16TimerId %d ",u16TimerId));

   if (FC_PHONE_CONFERENCE_TIMER_ID == u16TimerId)
   {
      // Increase the counter
      (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->u16Counter++;

      (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->vCheckCallStatus();

      if ((fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->u16Counter > FC_PHONE_MAX_COUNT_CONFERENCE)
      {
         // Stop the timer
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(u16TimerId);

         // Reset the counter
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->u16Counter = 0;

         // Start Conference SM
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->vStartConference();
      }
   }
   else
   {
      ETG_TRACE_ERR((" vTimerCallBackConference : Invalid timer id"));
   }
}

/******************************************************************************
*
* FUNCTION: vUpdateCallStatus
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vUpdateCallStatus()
{
   ETG_TRACE_USR3((" vUpdateCallStatus "));

   ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_CALLSTATUSNOTICE);
   if (AIL_EN_N_NO_ERROR != enResult)
   {
      ETG_TRACE_ERR(("Update Failed for MOST_TELFI_C_U16_CALLSTATUSNOTICE"));
   }

   if((FC_PHONE_MAX_CALLINSTANCES-1) == m_u8ConsolidatedCSCountprocessed)
   {
      enResult = eUpdateClients(MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED);
      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("Update Failed for MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED"));
      }
   }

}

/*******************************************************************************
*
* FUNCTION: vCheckConference()
*
* DESCRIPTION:Start conference if both calls are active and conf SM is not running
*
* PARAMETER: True/False
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vCheckConference(tU8 u8CallsActive)
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vCheckConference"));

   if ((!vGetConfSMStatus()) && (!isConfSMExecutedFromCallStatus))
   {
      if (TWO_CALLS_ACTIVE == u8CallsActive)
      {
         // Start the timer
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Start(FC_PHONE_CONFERENCE_TIMER_ID,
            FC_PHONE_ONE_SECOND, vTimerCallBackConference);
      }
      else
      {
         ETG_TRACE_USR4(("Two calls are not active"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("Conference SM already running"));
   }
   ETG_TRACE_USR4((" EXIT: fc_phone_tclService_Telephone::vCheckConference"));
}


/*******************************************************************************
** FUNCTION:   vGetVehicleAudioStatus()
*******************************************************************************/
/* DESCRIPTION:
*   Internal method to get VehicleAudio Status of call
*  PARAMETERS:
*     IN:   None
*
*  RETURNVALUE: fc_phone_tclService_Telephone* : Pointer to TEL service
********************************************************************************/
tBool fc_phone_tclService_Telephone::vGetVehicleAudioStatus(tVoid)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vGetVehicleAudioStatus ENTERED...  "));
   if(((m_oCallInstances[0].m_bUsingVehicleAudio) ||  (m_oCallInstances[1].m_bUsingVehicleAudio)) == TRUE)
   {
      ETG_TRACE_USR3(("Calls Use Vehicle Audio"));
      return TRUE;
   }
   else
   {
      ETG_TRACE_USR3(("Calls Doesnt Use Vehicle Audio"));
      return FALSE;
   }
}


/*******************************************************************************
** FUNCTION:   vGetCallInstanceOfOtherSM()
*******************************************************************************/
/* DESCRIPTION:
*   Internal method to get Call Instance of Another SM Status of call
*  PARAMETERS:
*     IN:   None
*
*  RETURNVALUE: fc_phone_tclService_Telephone* : Pointer to TEL service
********************************************************************************/
fc_phone_tclCallInstance* fc_phone_tclService_Telephone::vGetCallInstanceOfOtherSM(fc_phone_tclCallInstance*& rfpCallInstance)
{
   ETG_TRACE_USR4(("  fc_phone_tclCallInstance* fc_phone_tclService_Telephone::vGetCallInstanceOfOtherSM ENTERED...  "));
   for(tU16 u16index = 0 ;u16index < FC_PHONE_MAX_CALLINSTANCES ;u16index++)
   {
      fc_phone_tclCallInstance* pCallInstance = &m_oCallInstances[u16index];
      if(rfpCallInstance->m_u16Id != pCallInstance->m_u16Id)
      {
         ETG_TRACE_USR3((" rfpCallInstance->m_u16Id = %d",rfpCallInstance->m_u16Id));
         return pCallInstance;
      }
   }
   return NULLPTR;
}


/*******************************************************************************
*
* FUNCTION: vStartConference()
*
* DESCRIPTION:Start conference if both calls are active and conf SM is not running
*
* PARAMETER: True/False
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vStartConference()
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vStartConference "));

   tU8 u8Index = 0;
   tU8 u8AudioAcquiredCnt = 0;
   fc_phone_tclCallInstance *pCallInstance[FC_PHONE_MAX_CALLINSTANCES];

   NULL_CHECK(m_poSMManager)

      for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
      {
         pCallInstance[u8Index] = NULLPTR;
         if (FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE == m_oCallInstances[u8Index].m_u16CallStatusNotice)
         {
            pCallInstance[u8Index] = &m_oCallInstances[u8Index];
            u8AudioAcquiredCnt++;
         }
      }

      if (AUDIO_ACQUIRED_TWO_CALLS == u8AudioAcquiredCnt)
      {
         ETG_TRACE_USR4(("Start Conference SM"));

         FCPhoneConf_InputParamStruct st_ConfInParam ;
         st_ConfInParam.isConfFromCallStatus = TRUE;
         isConfSMExecutedFromCallStatus = TRUE;
         m_poSMManager->vStartSM (FC_PHONE_SMMANAGER_SMID_CONFERENCECALL , pCallInstance, &st_ConfInParam);
      }
      else
      {
         ETG_TRACE_USR4(("Two calls are not active"));
      }

      ETG_TRACE_USR4((" EXIT: fc_phone_tclService_Telephone::vStartConference"));
}


/*******************************************************************************
*
* FUNCTION: vCheckCallStatus()
*
* DESCRIPTION: Check call status and stop timer if either one call is on hold
*
* PARAMETER: True/False
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vCheckCallStatus()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vCheckCallStatus ENTERED...  "));
   tU8 u8Index = 0;
   tU8 u8AudioAcquiredCnt = 0;

   for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      if ((m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE))
      {
         u8AudioAcquiredCnt++;
      }
   }

   if (AUDIO_ACQUIRED_TWO_CALLS != u8AudioAcquiredCnt)
   {
      ETG_TRACE_USR4(("Two calls are not active: stop the timer"));

      // Stop the timer
      (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_CONFERENCE_TIMER_ID);

      // Reset the counter
      u16Counter = 0;
   }
}


/*******************************************************************************
*
* FUNCTION: vOnSaveSettings()
*
* DESCRIPTION: Transfer call to handset
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vOnSaveSettings()
{
   ETG_TRACE_USR3((" ENTER: fc_phone_tclService_Telephone::vOnSaveSettings"));
   tU8 u8Index = 0;
   for(u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      if(FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE != m_oCallInstances[u8Index].m_u16CallStatusNotice)
      {
         (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vHandleError();
         break;
      }
   }
}


/*******************************************************************************
*
* FUNCTION: vOnLoadSettings()
*
* DESCRIPTION: On load settings get call status
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vOnLoadSettings()
{
   ETG_TRACE_USR3((" ENTER: fc_phone_tclService_Telephone::vOnLoadSettings"));
   m_bHeldtoActiveSwapTimerRunning = FALSE;
   // Fix for NCG3D-103060
   // Resetting here during vOnLoadSettings
   m_bExternalTriggerStatusforPhoneChannel = FALSE;
   ETG_TRACE_USR3(("m_bExternalTriggerStatusforPhoneChannel : %u", m_bExternalTriggerStatusforPhoneChannel));
}


/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_AcceptCall_Error
*
* DESCRIPTION: Sends Accept call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                              amt_tclServiceData* poMessage, struct Job* f_poJob, tBool bIsPhoneMutexLockAcquired)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vHandle_AcceptCall_Error ENTERED"));
   most_telfi_tclMsgAcceptCallError oAcceptCall_MError ;
   m_bAcceptMethodStartFlag = FALSE;
   oAcceptCall_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oAcceptCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   if(poMessage)
   {
      vInitServiceData( oMRMsg,                                       // ServiceDataMsg
         poMessage->u16GetSourceAppID(),               // Target AppID
         poMessage->u16GetRegisterID() ,               // RegisterID
         poMessage->u16GetCmdCounter() ,               // CmdCounter,
         poMessage->u16GetFunctionID() ,               // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
         );
      ETG_TRACE_USR4(("bIsPhoneMutexLockAcquired: %d", bIsPhoneMutexLockAcquired));
      if (bIsPhoneMutexLockAcquired)
      {
         // In case (NULL != poMessage) ie., direct message from CCA, the acquired lock should be released in vHandle_AcceptCall_Error()
         fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //Error occurred while processing. Hence unlock.
         ETG_TRACE_USR4((" AcceptCall  m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM"));
         m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM;
      }
   }
   else if(f_poJob) //Error while processing message from work queue
   {
      vInitServiceData( oMRMsg,                                // ServiceDataMsg
         f_poJob->u16SourceAppID,               // Target AppID
         f_poJob->u16RegisterID ,               // RegisterID
         f_poJob->u16CmdCounter ,               // CmdCounter,
         f_poJob->u16FunctionID ,               // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR           // Opcode
         );

      ETG_TRACE_USR4(("m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM"));
      m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM ;

      fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
   }

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oAcceptCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oAcceptCall_MError.vDestroy();
}


/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_AcceptCall_Error
*
* DESCRIPTION: Sends Accept call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER: fc_phone_tclService_Telephone::vHandle_AcceptCall_Error"));
   // NCG3D-131177
   // If the Accept call error is sent for a Passive device,
   // accept call request "m_u8TransferCallCause" should be reset.
   tU16 u16Id = 0x00;
   tS32 s32CIStatusActiveOrPassive = s32GetCallInstanceIndexFromFiCallInstance((tU16)m_u16StoredCallInstanceForAccept, u16Id);
   if (s32CIStatusActiveOrPassive == true)
   {
     m_u8TransferCallCause = 0;
   }

   most_telfi_tclMsgAcceptCallError oAcceptCall_MError ;

   oAcceptCall_MError.e8ErrorCode.enType = e8ErrorCode;
   m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM ;
   ETG_TRACE_USR4(("m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM"));
   m_bAcceptMethodStartFlag = FALSE;
   fi_tclVisitorMessage oMRMsg(oAcceptCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                       // ServiceDataMsg
      m_oAcceptCallMessage.u16SourceAppID,          // Target AppID
      m_oAcceptCallMessage.u16RegisterID ,          // RegisterID
      m_oAcceptCallMessage.u16CmdCounter ,          // CmdCounter,
      m_oAcceptCallMessage.u16FunctionID ,          // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
      );

   fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //Error occurred while processing. Hence unlock.

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oAcceptCall_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oAcceptCall_MError.vDestroy();
}


/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_RejectCall_Error
*
* DESCRIPTION: Sends Reject call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_RejectCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                              amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER: fc_phone_tclService_Telephone::vHandle_RejectCall_Error"));
   most_telfi_tclMsgRejectCallError oRejectCall_MError ;
   oRejectCall_MError.e8ErrorCode.enType = e8ErrorCode;

   m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM;
   ETG_TRACE_USR4(("m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM"));

   fi_tclVisitorMessage oMRMsg(oRejectCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                       // ServiceDataMsg
      poMessage->u16GetSourceAppID(),               // Target AppID
      poMessage->u16GetRegisterID() ,               // RegisterID
      poMessage->u16GetCmdCounter() ,               // CmdCounter,
      poMessage->u16GetFunctionID() ,               // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oRejectCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("       Message is not Valid"));
   }

   oRejectCall_MError.vDestroy();
}


/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_RejectCall_Error
*
* DESCRIPTION: Sends Reject call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_RejectCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER: fc_phone_tclService_Telephone::vHandle_RejectCall_Error"));
   most_telfi_tclMsgRejectCallError oRejectCall_MError ;
   oRejectCall_MError.e8ErrorCode.enType = e8ErrorCode;

   m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM;
   ETG_TRACE_USR4(("m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM"));

   fi_tclVisitorMessage oMRMsg(oRejectCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                       // ServiceDataMsg
      m_oRejectCallMessage.u16SourceAppID,          // Target AppID
      m_oRejectCallMessage.u16RegisterID ,          // RegisterID
      m_oRejectCallMessage.u16CmdCounter ,          // CmdCounter,
      m_oRejectCallMessage.u16FunctionID ,          // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oRejectCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oRejectCall_MError.vDestroy();
}


/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_HangUpCall_Error
*
* DESCRIPTION: Sends Hangup call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_HangUpCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                              amt_tclServiceData* poMessage, struct Job* f_poJob, tBool bIsPhoneMutexLockAcquired)
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vHandle_HangUpCall_Error"));
   most_telfi_tclMsgHangUpCallError oHangUpCall_MError;

   oHangUpCall_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oHangUpCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   if(poMessage)
   {
      vInitServiceData( oMRMsg,                                       // ServiceDataMsg
         poMessage->u16GetSourceAppID(),               // Target AppID
         poMessage->u16GetRegisterID() ,               // RegisterID
         poMessage->u16GetCmdCounter() ,               // CmdCounter,
         poMessage->u16GetFunctionID() ,               // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
         );

      ETG_TRACE_USR4(("bIsPhoneMutexLockAcquired: %d", bIsPhoneMutexLockAcquired));
      if (bIsPhoneMutexLockAcquired)
      {
         m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM;
         ETG_TRACE_USR4(("m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM"));
         fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
      }
   }
   else if(f_poJob) //Error while processing message from work queue
   {
      vInitServiceData( oMRMsg,                                // ServiceDataMsg
         f_poJob->u16SourceAppID,               // Target AppID
         f_poJob->u16RegisterID ,               // RegisterID
         f_poJob->u16CmdCounter ,               // CmdCounter,
         f_poJob->u16FunctionID ,               // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR           // Opcode
         );
      m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM;
      ETG_TRACE_USR4(("m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM"));
      fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
   }

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(( "oHangUpCall_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oHangUpCall_MError.vDestroy();
}


/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_HangUpCall_Error
*
* DESCRIPTION: Sends Hangup call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_HangUpCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER: fc_phone_tclService_Telephone::vHandle_HangUpCall_Error"));
   most_telfi_tclMsgHangUpCallError oHangUpCall_MError;

   oHangUpCall_MError.e8ErrorCode.enType = e8ErrorCode;
   m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM;
   ETG_TRACE_USR4(("m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM"));

   fi_tclVisitorMessage oMRMsg(oHangUpCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                       // ServiceDataMsg
      m_oHangUpCallMessage.u16SourceAppID,          // Target AppID
      m_oHangUpCallMessage.u16RegisterID ,          // RegisterID
      m_oHangUpCallMessage.u16CmdCounter ,          // CmdCounter,
      m_oHangUpCallMessage.u16FunctionID ,          // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
      );

   fc_phone_tclPhoneMutex::PhoneMutexUnlock();  //Error occurred while processing. Hence unlock.

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(( "oHangUpCall_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));

      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oHangUpCall_MError.vDestroy();
}


/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_IgnoreCall_Error
*
* DESCRIPTION: Sends Ignore call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_IgnoreCall_Error(
      most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage, struct Job* f_poJob, tBool bIsPhoneMutexLockAcquired)
{
   ETG_TRACE_USR4(("ENTER: fc_phone_tclService_Telephone::vHandle_IgnoreCall_Error"));
   most_telfi_tclMsgIgnoreCallError oIgnoreCall_MError;

   oIgnoreCall_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oIgnoreCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   if (poMessage)
   {
      vInitServiceData(oMRMsg, // ServiceDataMsg
      poMessage->u16GetSourceAppID(), // Target AppID
      poMessage->u16GetRegisterID(), // RegisterID
      poMessage->u16GetCmdCounter(), // CmdCounter,
      poMessage->u16GetFunctionID(), // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
      );

      ETG_TRACE_USR4(("bIsPhoneMutexLockAcquired: %d", bIsPhoneMutexLockAcquired));
      if (bIsPhoneMutexLockAcquired)
      {
         ETG_TRACE_USR4(("m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM"));
         m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM;
         fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //Unlock in case of error in processing message from queue
      }
   }
   else if (f_poJob) //Error while processing message from work queue
   {
      vInitServiceData(oMRMsg, // ServiceDataMsg
      f_poJob->u16SourceAppID, // Target AppID
      f_poJob->u16RegisterID, // RegisterID
      f_poJob->u16CmdCounter, // CmdCounter,
      f_poJob->u16FunctionID, // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
      );

      ETG_TRACE_USR4(("m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM"));
      m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM;
      fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //Unlock in case of error in processing message from queue
   }

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("oIgnoreCall_MError: 'MethodError', enPostMessage failed, "
               "'ail_tenCommunicationError' = %u", (tU32) enResult));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Message is not Valid"));
   }

   oIgnoreCall_MError.vDestroy();
}


/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_IgnoreCall_Error
*
* DESCRIPTION: Sends Ignore call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_IgnoreCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER: fc_phone_tclService_Telephone::vHandle_IgnoreCall_Error"));

   most_telfi_tclMsgIgnoreCallError oIgnoreCall_MError;

   oIgnoreCall_MError.e8ErrorCode.enType = e8ErrorCode;
   m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM;
   ETG_TRACE_USR4(("m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM"));

   fi_tclVisitorMessage oMRMsg(oIgnoreCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData(oMRMsg, // ServiceDataMsg
   m_oIgnoreCallMessage.u16SourceAppID, // Target AppID
   m_oIgnoreCallMessage.u16RegisterID, // RegisterID
   m_oIgnoreCallMessage.u16CmdCounter, // CmdCounter,
   m_oIgnoreCallMessage.u16FunctionID, // Function ID
   AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
   );

   fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //Error occurred while processing. Hence unlock.

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("oIgnoreCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32) enResult));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Message is not Valid"));
   }

   oIgnoreCall_MError.vDestroy();
}


/****************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_HoldCall_Error
*
* DESCRIPTION: Sends Hold call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_HoldCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_HoldCall_Error"));
   most_telfi_tclMsgHoldCallError oHoldCall_MError;
   oHoldCall_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oHoldCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                   // ServiceDataMsg
      poMessage->u16GetSourceAppID(),           // Target AppID
      poMessage->u16GetRegisterID() ,           // RegisterID
      poMessage->u16GetCmdCounter() ,           // CmdCounter,
      poMessage->u16GetFunctionID() ,           // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR              // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(( "oHoldCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oHoldCall_MError.vDestroy();
}


/****************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_HoldCall_Error
*
* DESCRIPTION: Sends Hold call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_HoldCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4((" ENTER  fc_phone_tclService_Telephone::vHandle_HoldCall_Error"));
   most_telfi_tclMsgHoldCallError oHoldCall_MError;

   oHoldCall_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oHoldCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                   // ServiceDataMsg
      m_oHoldCallMessage.u16SourceAppID,        // Target AppID
      m_oHoldCallMessage.u16RegisterID ,        // RegisterID
      m_oHoldCallMessage.u16CmdCounter ,        // CmdCounter,
      m_oHoldCallMessage.u16FunctionID ,        // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR              // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(( "oHoldCall_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oHoldCall_MError.vDestroy();
}


/************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_Dial_Error
*
* DESCRIPTION: Sends Dial Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
**************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage, struct Job* f_poJob, tBool bIsPhoneMutexLockAcquired)
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vHandle_Dial_Error"));
   most_telfi_tclMsgDialError oDial_MError ;
   oDial_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oDial_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   if(poMessage)
   {
      vInitServiceData( oMRMsg,                               // ServiceDataMsg
            poMessage->u16GetSourceAppID(),       // Target AppID
            poMessage->u16GetRegisterID() ,       // RegisterID
            poMessage->u16GetCmdCounter() ,       // CmdCounter,
            poMessage->u16GetFunctionID() ,       // Function ID
            AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

      ETG_TRACE_USR4(("bIsPhoneMutexLockAcquired: %d", bIsPhoneMutexLockAcquired));
      // In case (NULL != poMessage) ie., direct message from CCA, the acquired lock should be released in vHandle_Dial_Error()
      if (bIsPhoneMutexLockAcquired)
      {
         fc_phone_tclPhoneMutex::PhoneMutexUnlock();
      }
   }
   else if(f_poJob) //Error while processing message from work queue
   {
      vInitServiceData( oMRMsg,         	             // ServiceDataMsg
         f_poJob->u16SourceAppID,           // Target AppID
         f_poJob->u16RegisterID ,           // RegisterID
         f_poJob->u16CmdCounter ,           // CmdCounter,
         f_poJob->u16FunctionID ,           // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR       // Opcode
         );

      fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
   }

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oDial_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("       Message is not Valid"));
   }

   oDial_MError.vDestroy();
}


/************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_Dial_Error
*
* DESCRIPTION: Sends Dial Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
**************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4((" ENTER fc_phone_tclService_Telephone::vHandle_Dial_Error"));
   most_telfi_tclMsgDialError oDial_MError ;

   oDial_MError.e8ErrorCode.enType = e8ErrorCode;
   m_bDialMethodStartFlag = FALSE;
   fi_tclVisitorMessage oMRMsg(oDial_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                               // ServiceDataMsg
      m_oDialMessage.u16SourceAppID,        // Target AppID
      m_oDialMessage.u16RegisterID ,        // RegisterID
      m_oDialMessage.u16CmdCounter ,        // CmdCounter,
      m_oDialMessage.u16FunctionID ,        // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

   fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //Error occurred while processing. Hence unlock.

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oDial_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("       Message is not Valid"));
   }

   oDial_MError.vDestroy();
}


/************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_DialExtended_Error
*
* DESCRIPTION: Sends DialExtended Method  error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
**************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage, struct Job* f_poJob, tBool bIsPhoneMutexLockAcquired)
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vHandle_DialExtended_Error"));
   most_telfi_tclMsgDialExtendedError oDialExtended_MError ;
   oDialExtended_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oDialExtended_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   if(poMessage)
   {
      vInitServiceData( oMRMsg,                // ServiceDataMsg
            poMessage->u16GetSourceAppID(),       // Target AppID
            poMessage->u16GetRegisterID() ,       // RegisterID
            poMessage->u16GetCmdCounter() ,       // CmdCounter,
            poMessage->u16GetFunctionID() ,       // Function ID
            AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

      ETG_TRACE_USR4(("bIsPhoneMutexLockAcquired: %d", bIsPhoneMutexLockAcquired));
      // In case (NULL != poMessage) ie., direct message from CCA, the acquired lock should be released in vHandle_DialExtended_Error()
      if (bIsPhoneMutexLockAcquired)
      {
         fc_phone_tclPhoneMutex::PhoneMutexUnlock();
      }
   }
   else if(f_poJob) //Error while processing message from work queue
   {
      vInitServiceData( oMRMsg,         	// ServiceDataMsg
            f_poJob->u16SourceAppID,           // Target AppID
            f_poJob->u16RegisterID ,           // RegisterID
            f_poJob->u16CmdCounter ,           // CmdCounter,
            f_poJob->u16FunctionID ,           // Function ID
            AMT_C_U8_CCAMSG_OPCODE_ERROR       // Opcode
      );

      fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
   }

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oDialExtended_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("       Message is not Valid"));
   }

   oDialExtended_MError.vDestroy();
}


/************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_DialExtended_Error
*
* DESCRIPTION: Sends Dial Method Extended error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
**************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4((" ENTER fc_phone_tclService_Telephone::vHandle_DialExtended_Error"));
   most_telfi_tclMsgDialExtendedError oDialExteded_MError ;

   oDialExteded_MError.e8ErrorCode.enType = e8ErrorCode;
   m_bDialMethodStartFlag = FALSE;
   fi_tclVisitorMessage oMRMsg(oDialExteded_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                // ServiceDataMsg
         m_oDialMessage.u16SourceAppID,        // Target AppID
         m_oDialMessage.u16RegisterID ,        // RegisterID
         m_oDialMessage.u16CmdCounter ,        // CmdCounter,
         m_oDialMessage.u16FunctionID ,        // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
   );

   fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //Error occurred while processing. Hence unlock.

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oDialExteded_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("       Message is not Valid"));
   }

   oDialExteded_MError.vDestroy();
}

/*****************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SpeedDial_Error
*
* DESCRIPTION: Sends Speed Dial Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SpeedDial_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage, struct Job* f_poJob, tBool bIsPhoneMutexLockAcquired)
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vHandle_SpeedDial_Error"));
   most_telfi_tclMsgSpeedDialError oSpeedDial_MError;
   oSpeedDial_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oSpeedDial_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   if(poMessage)
   {
      vInitServiceData( oMRMsg,							// ServiceDataMsg
            poMessage->u16GetSourceAppID(),       // Target AppID
            poMessage->u16GetRegisterID() ,       // RegisterID
            poMessage->u16GetCmdCounter() ,       // CmdCounter,
            poMessage->u16GetFunctionID() ,       // Function ID
            AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

      ETG_TRACE_USR4(("bIsPhoneMutexLockAcquired: %d", bIsPhoneMutexLockAcquired));
      if (bIsPhoneMutexLockAcquired)
      {
         fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
      }
   }
   else if(f_poJob) //Error while processing message from work queue
   {
      vInitServiceData( oMRMsg,         	             // ServiceDataMsg
         f_poJob->u16SourceAppID,           // Target AppID
         f_poJob->u16RegisterID ,           // RegisterID
         f_poJob->u16CmdCounter ,           // CmdCounter,
         f_poJob->u16FunctionID ,           // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR       // Opcode
         );

      fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
   }

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oSpeedDial_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oSpeedDial_MError.vDestroy();
}


/*****************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SpeedDial_Error
*
* DESCRIPTION: Sends Speed Dial Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SpeedDial_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_SpeedDial_Error"));
   most_telfi_tclMsgSpeedDialError oSpeedDial_MError;

   oSpeedDial_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oSpeedDial_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                   // ServiceDataMsg
      m_oSpeedDialMessage.u16SourceAppID,       // Target AppID
      m_oSpeedDialMessage.u16RegisterID ,       // RegisterID
      m_oSpeedDialMessage.u16CmdCounter ,       // CmdCounter,
      m_oSpeedDialMessage.u16FunctionID ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR              // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(( "oSpeedDial_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oSpeedDial_MError.vDestroy();
}


/**************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_Error
*
* DESCRIPTION: Sends Cancel Outgoing call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                                      amt_tclServiceData* poMessage, struct Job* f_poJob, tBool bIsPhoneMutexLockAcquired)
{
   ETG_TRACE_USR4((" ENTER fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_Error"));
   most_telfi_tclMsgCancelOutgoingCallError oCancelOutgoingCall_MError;

   oCancelOutgoingCall_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oCancelOutgoingCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   if(poMessage)
   {
      vInitServiceData( oMRMsg,                             // ServiceDataMsg
         poMessage->u16GetSourceAppID(),       // Target AppID
         poMessage->u16GetRegisterID() ,       // RegisterID
         poMessage->u16GetCmdCounter() ,   // CmdCounter,
         poMessage->u16GetFunctionID() ,   // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR      // Opcode
         );

      ETG_TRACE_USR4(("bIsPhoneMutexLockAcquired: %d", bIsPhoneMutexLockAcquired));
      if (bIsPhoneMutexLockAcquired)
      {
         m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM;
         ETG_TRACE_USR4((" m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM "));

         fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
      }
   }
   else if(f_poJob) //Error while processing message from work queue
   {
      vInitServiceData( oMRMsg,                              // ServiceDataMsg
         f_poJob->u16SourceAppID,               // Target AppID
         f_poJob->u16RegisterID ,               // RegisterID
         f_poJob->u16CmdCounter ,               // CmdCounter,
         f_poJob->u16FunctionID ,               // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR           // Opcode
         );

      m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM;
      ETG_TRACE_USR4((" m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM "));

      fc_phone_tclPhoneMutex::PhoneMutexUnlock();//Unlock in case of error in processing message from queue
   }

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oCancelOutgoingCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oCancelOutgoingCall_MError.vDestroy();
}


/**************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_Error
*
* DESCRIPTION: Sends Cancel Outgoing call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vHandle_CancelOutgoingCall_Error"));
   most_telfi_tclMsgCancelOutgoingCallError oCancelOutgoingCall_MError;
   oCancelOutgoingCall_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oCancelOutgoingCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                     // ServiceDataMsg
      m_oCancelOutgoingCallMessage.u16SourceAppID,         // Target AppID
      m_oCancelOutgoingCallMessage.u16RegisterID ,         // RegisterID
      m_oCancelOutgoingCallMessage.u16CmdCounter ,         // CmdCounter,
      m_oCancelOutgoingCallMessage.u16FunctionID ,         // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                         // Opcode
      );

   fc_phone_tclPhoneMutex::PhoneMutexUnlock(); //Error occurred while processing. Hence unlock.

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(( "oCancelOutgoingCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("       Message is not Valid"));
   }

   oCancelOutgoingCall_MError.vDestroy();
}


/**************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_AcquireTelephoneNumber_Error
*
* DESCRIPTION: Sends Acquire Telephone Number Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_AcquireTelephoneNumber_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4((" ENTER fc_phone_tclService_Telephone::vHandle_AcquireTelephoneNumber_Error"));
   m_u8BDAddressIDforAcquiringTelNo = INVALID_BDADDRESS_ID;

   for(tU16 Index = 0; Index < m_oAcquireTelephoneNumberMessage.size(); Index++)
   {
      most_telfi_tclMsgAcquireTelephoneNumberError oAcquireTelephoneNumber_MError;
      oAcquireTelephoneNumber_MError.e8ErrorCode.enType = e8ErrorCode;

      fi_tclVisitorMessage oMRMsg(oAcquireTelephoneNumber_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
      // Here the message related header information is added and by this the
      // creation of the message is completed.
      vInitServiceData( oMRMsg,                               // ServiceDataMsg
         m_oAcquireTelephoneNumberMessage[Index].u16SourceAppID,       // Target AppID
         m_oAcquireTelephoneNumberMessage[Index].u16RegisterID ,       // RegisterID
         m_oAcquireTelephoneNumberMessage[Index].u16CmdCounter ,       // CmdCounter,
         m_oAcquireTelephoneNumberMessage[Index].u16FunctionID ,       // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
         );

      if (oMRMsg.bIsValid())
      {
         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR(( "oAcquireTelephoneNumber_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
         }
      }
      else
      {
         ETG_TRACE_ERR(("  Message is not Valid"));
      }
      oAcquireTelephoneNumber_MError.vDestroy();
   }
   m_oAcquireTelephoneNumberMessage.clear();
}


/**************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_AcquireTelephoneNumber_MethodResult
*
* DESCRIPTION: Sends Acquire Telephone Number Method Result
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_AcquireTelephoneNumber_MethodResult()
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vHandle_AcquireTelephoneNumber_MethodResult"));

   for(tU16 Index = 0; Index < m_oAcquireTelephoneNumberMessage.size(); Index++)
   {
      most_telfi_tclMsgAcquireTelephoneNumberMethodResult oAcquireTelephoneNumber_MR;
      //create the method result and send
      m_u8AcquireTelephoneNumberRetryCnt = 0;

      if(FC_PHONE_MAX_CONNECTED_DEVICE > m_u8BDAddressIDforAcquiringTelNo)
      {
         oAcquireTelephoneNumber_MR.sTelephoneNumber.bSet((const char*)m_oPhoneData[m_u8BDAddressIDforAcquiringTelNo].m_SubscriberMSISDN);
         ETG_TRACE_USR1(("oAcquireTelephoneNumber_MR.sTelephoneNumber.szValue = %s",oAcquireTelephoneNumber_MR.sTelephoneNumber.szValue));
      }
      fi_tclVisitorMessage oMRMsg(oAcquireTelephoneNumber_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
      // Here the message related header information is added and by this the
      // creation of the message is completed.
      vInitServiceData( oMRMsg,                          // ServiceDataMsg
         m_oAcquireTelephoneNumberMessage[Index].u16SourceAppID,               // Target AppID
         m_oAcquireTelephoneNumberMessage[Index].u16RegisterID ,                // RegisterID
         m_oAcquireTelephoneNumberMessage[Index].u16CmdCounter ,                // CmdCounter,
         m_oAcquireTelephoneNumberMessage[Index].u16FunctionID ,                // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT                // Opcode
         );
      if (oMRMsg.bIsValid())
      {
         // Post message
         ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMRMsg, TRUE);
         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR(( "vHandle_AcquireTelephonenumber_MethodResult, enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
         }
      }
      oAcquireTelephoneNumber_MR.vDestroy();
   }
   m_u8BDAddressIDforAcquiringTelNo = INVALID_BDADDRESS_ID;
   m_oAcquireTelephoneNumberMessage.clear();
}


/*******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SendDTMF_Error
*
* DESCRIPTION: Sends Send DTMF Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*********************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SendDTMF_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                            amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" ENTER: fc_phone_tclService_Telephone::vHandle_SendDTMF_Error"));
   most_telfi_tclMsgSendDTMFError oSendDTMF_MError;

   oSendDTMF_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oSendDTMF_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                               // ServiceDataMsg
      poMessage->u16GetSourceAppID(),       // Target AppID
      poMessage->u16GetRegisterID() ,       // RegisterID
      poMessage->u16GetCmdCounter() ,       // CmdCounter,
      poMessage->u16GetFunctionID() ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(( "oSendDTMF_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oSendDTMF_MError.vDestroy();
}


/*******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SendDTMF_Error
*
* DESCRIPTION: Sends Send DTMF Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
*********************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SendDTMF_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4((" ENTER fc_phone_tclService_Telephone::vHandle_SendDTMF_Error"));
   most_telfi_tclMsgSendDTMFError oSendDTMF_MError;
   oSendDTMF_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oSendDTMF_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                           // ServiceDataMsg
      m_oSendDTMFMessage.u16SourceAppID,         // Target AppID
      m_oSendDTMFMessage.u16RegisterID ,         // RegisterID
      m_oSendDTMFMessage.u16CmdCounter ,         // CmdCounter,
      m_oSendDTMFMessage.u16FunctionID ,         // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR               // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oSendDTMF_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR((" Message is not Valid"));
   }

   oSendDTMF_MError.vDestroy();
}


/*******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_ResumeCall_Error
*
* DESCRIPTION: Sends Resume Call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*********************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_ResumeCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_ResumeCall_Error"));
   most_telfi_tclMsgResumeCallError oResumeCallCall_MError;
   oResumeCallCall_MError.e8ErrorCode.enType = e8ErrorCode;
   fi_tclVisitorMessage oMRMsg(oResumeCallCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                               // ServiceDataMsg
      poMessage->u16GetSourceAppID(),       // Target AppID
      poMessage->u16GetRegisterID() ,       // RegisterID
      poMessage->u16GetCmdCounter() ,       // CmdCounter,
      poMessage->u16GetFunctionID() ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oResumeCallCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oResumeCallCall_MError.vDestroy();
}


/*******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_ResumeCall_Error
*
* DESCRIPTION: Sends Resume Call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
*********************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_ResumeCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_ResumeCall_Error"));
   most_telfi_tclMsgResumeCallError oResumeCallCall_MError;
   oResumeCallCall_MError.e8ErrorCode.enType = e8ErrorCode;
   fi_tclVisitorMessage oMRMsg(oResumeCallCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                                               // ServiceDataMsg
      m_oResumeCallMessage.u16SourceAppID,        // Target AppID
      m_oResumeCallMessage.u16RegisterID ,        // RegisterID
      m_oResumeCallMessage.u16CmdCounter ,        // CmdCounter,
      m_oResumeCallMessage.u16FunctionID ,        // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oResumeCallCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oResumeCallCall_MError.vDestroy();
}


/****************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SwapCall_Error
*
* DESCRIPTION: Sends Swap Call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SwapCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER :fc_phone_tclService_Telephone::vHandle_SwapCall_Error"));
   most_telfi_tclMsgSwapCallError oSwapCall_MError;
   oSwapCall_MError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oSwapCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                               // ServiceDataMsg
      poMessage->u16GetSourceAppID(),       // Target AppID
      poMessage->u16GetRegisterID() ,       // RegisterID
      poMessage->u16GetCmdCounter() ,       // CmdCounter,
      poMessage->u16GetFunctionID() ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oSwapCall_MError: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oSwapCall_MError.vDestroy();
}


/****************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SwapCall_Error
*
* DESCRIPTION: Sends Swap Call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SwapCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_SwapCall_Error"));
   most_telfi_tclMsgSwapCallError oSwapCall_MError;
   oSwapCall_MError.e8ErrorCode.enType = e8ErrorCode;
   fi_tclVisitorMessage oMRMsg(oSwapCall_MError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData( oMRMsg,                                   // ServiceDataMsg
      m_oSwapCallMessage.u16SourceAppID,        // Target AppID
      m_oSwapCallMessage.u16RegisterID ,        // RegisterID
      m_oSwapCallMessage.u16CmdCounter ,        // CmdCounter,
      m_oSwapCallMessage.u16FunctionID ,        // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR              // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oSwapCall_MError: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oSwapCall_MError.vDestroy();
}


/*****************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_TransferCallToHandSet_Error
*
* DESCRIPTION: Sends Transfer Call to Handset Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_TransferCallToHandSet_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                                         amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_TransferCallToHandSet_Error"));
   most_telfi_tclMsgTransferCallToHandsetError oTransferCallHandSet_ME;
   oTransferCallHandSet_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oTransferCallHandSet_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMRMsg,                               // ServiceDataMsg
      poMessage->u16GetSourceAppID(),       // Target AppID
      poMessage->u16GetRegisterID() ,       // RegisterID
      poMessage->u16GetCmdCounter() ,       // CmdCounter,
      poMessage->u16GetFunctionID() ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

   if(oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMRMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oTransferCallHandSet_ME: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oTransferCallHandSet_ME.vDestroy();
}


/*****************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_TransferCallToHandSet_Error
*
* DESCRIPTION: Sends Transfer Call to Handset Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_TransferCallToHandSet_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_TransferCallToHandSet_Error"));
   for(tU16 Index = 0; Index < m_oTransferCallIncomingMsgQueue.size(); Index++)
   {
      most_telfi_tclMsgTransferCallToHandsetError oTransferCallHandSet_ME;
      oTransferCallHandSet_ME.e8ErrorCode.enType = e8ErrorCode;
      fi_tclVisitorMessage oMRMsg(oTransferCallHandSet_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

      vInitServiceData(oMRMsg,                          // ServiceDataMsg
         m_oTransferCallIncomingMsgQueue[Index].u16SourceAppID,       // Target AppID
         m_oTransferCallIncomingMsgQueue[Index].u16RegisterID ,        // RegisterID
         m_oTransferCallIncomingMsgQueue[Index].u16CmdCounter ,        // CmdCounter,
         m_oTransferCallIncomingMsgQueue[Index].u16FunctionID ,        // Function ID,
         AMT_C_U8_CCAMSG_OPCODE_ERROR                      // Opcode
         );

      if(oMRMsg.bIsValid())
      {
         // Post message
         ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMRMsg, TRUE);
         if(AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR(( "oTransferCallHandSet_ME: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
         }
      }
      else
      {
         ETG_TRACE_ERR(("  Message is not Valid"));
      }

      oTransferCallHandSet_ME.vDestroy();
   }
   m_oTransferCallIncomingMsgQueue.clear();
}


/*****************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_TransferCallToVehicle_Error
*
* DESCRIPTION: Sends Transfer Call to Vehicle Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_TransferCallToVehicle_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                                         amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_TransferCallToVehicle_Error"));
   most_telfi_tclMsgTransferCallToVehicleError oTransferCallVehicle_ME;
   oTransferCallVehicle_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oTransferCallVehicle_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMRMsg,                               // ServiceDataMsg
      poMessage->u16GetSourceAppID(),       // Target AppID
      poMessage->u16GetRegisterID() ,       // RegisterID
      poMessage->u16GetCmdCounter() ,       // CmdCounter,
      poMessage->u16GetFunctionID() ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
      );

   if(oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMRMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oTransferCallVehicle_ME: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oTransferCallVehicle_ME.vDestroy();
}


/*****************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_TransferCallToVehicle_Error
*
* DESCRIPTION: Sends Transfer Call to Vehicle Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_TransferCallToVehicle_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_TransferCallToVehicle_Error"));
   most_telfi_tclMsgTransferCallToVehicleError oTransferCallVehicle_ME;
   oTransferCallVehicle_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oTransferCallVehicle_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMRMsg,                                       // ServiceDataMsg
      m_oTransferCallToVehicle.u16SourceAppID,        // Target AppID
      m_oTransferCallToVehicle.u16RegisterID ,        // RegisterID
      m_oTransferCallToVehicle.u16CmdCounter ,        // CmdCounter,
      m_oTransferCallToVehicle.u16FunctionID ,        // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                          // Opcode
      );

   if(oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMRMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oTransferCallVehicle_ME: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oTransferCallVehicle_ME.vDestroy();
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_ConferenceCall_Error
*
* DESCRIPTION: Sends Conference call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_ConferenceCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_ConferenceCall_Error"));
   most_telfi_tclMsgConferenceInError oConferenceCall_ME;
   oConferenceCall_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMEMsg(oConferenceCall_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMEMsg,                               // ServiceDataMsg
      poMessage->u16GetSourceAppID(),       // Target AppID
      poMessage->u16GetRegisterID() ,       // RegisterID
      poMessage->u16GetCmdCounter() ,       // CmdCounter,
      poMessage->u16GetFunctionID() ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

   if(oMEMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMEMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "m_oConferenceCall Message: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oConferenceCall_ME.vDestroy();
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_ConferenceCall_Error
*
* DESCRIPTION: Sends Conference call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_ConferenceCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_ConferenceCall_Error"));
   most_telfi_tclMsgConferenceInError oConferenceCall_ME;
   oConferenceCall_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMEMsg(oConferenceCall_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMEMsg,                                           // ServiceDataMsg
      m_oConferenceCallMessage.u16SourceAppID,          // Target AppID
      m_oConferenceCallMessage.u16RegisterID ,          // RegisterID
      m_oConferenceCallMessage.u16CmdCounter ,          // CmdCounter,
      m_oConferenceCallMessage.u16FunctionID ,          // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                      // Opcode
      );

   if(oMEMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMEMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "m_oConferenceCall Message: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oConferenceCall_ME.vDestroy();
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SplitCall_Error
*
* DESCRIPTION: Sends split call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SplitCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_SplitCall_Error"));
   most_telfi_tclMsgSplitCallError oSplitCall_ME;
   oSplitCall_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMEMsg(oSplitCall_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMEMsg,                               // ServiceDataMsg
      poMessage->u16GetSourceAppID(),       // Target AppID
      poMessage->u16GetRegisterID() ,       // RegisterID
      poMessage->u16GetCmdCounter() ,       // CmdCounter,
      poMessage->u16GetFunctionID() ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

   if(oMEMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMEMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oSplitCall_ME: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR((" Message is not Valid"));
   }

   oSplitCall_ME.vDestroy();
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SplitCall_Error
*
* DESCRIPTION: Sends split call Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SplitCall_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_SplitCall_Error"));
   most_telfi_tclMsgSplitCallError oSplitCall_ME;
   oSplitCall_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMEMsg(oSplitCall_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMEMsg,                                       // ServiceDataMsg
      m_oSplitCallMessage.u16SourceAppID,           // Target AppID
      m_oSplitCallMessage.u16RegisterID ,           // RegisterID
      m_oSplitCallMessage.u16CmdCounter ,           // CmdCounter,
      m_oSplitCallMessage.u16FunctionID ,           // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
      );

   if(oMEMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMEMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oSplitCall_ME: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR((" Message is not Valid"));
   }

   oSplitCall_ME.vDestroy();
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_QueryTelephoneNumber_Error
*
* DESCRIPTION: Sends Query telephone number Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_QueryTelephoneNumber_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                                        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_QueryTelephoneNumber_Error"));
   most_telfi_tclMsgQueryTelephoneNumberError oQueryTelephoneNumber_ME;
   oQueryTelephoneNumber_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMEMsg(oQueryTelephoneNumber_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMEMsg,                               // ServiceDataMsg
      poMessage->u16GetSourceAppID(),       // Target AppID
      poMessage->u16GetRegisterID() ,       // RegisterID
      poMessage->u16GetCmdCounter() ,       // CmdCounter,
      poMessage->u16GetFunctionID() ,       // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
      );

   if(oMEMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMEMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oQueryTelephoneNumber_ME: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oQueryTelephoneNumber_ME.vDestroy();
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_VoiceMail_Error
*
* DESCRIPTION: Sends Voice mail Method error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_VoiceMail_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandle_VoiceMail_Error"));
   most_telfi_tclMsgVoiceMailError oVoiceMail_ME;
   oVoiceMail_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMEMsg(oVoiceMail_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMEMsg,                                       // ServiceDataMsg
      poMessage->u16GetSourceAppID(),               // Target AppID
      poMessage->u16GetRegisterID() ,               // RegisterID
      poMessage->u16GetCmdCounter() ,               // CmdCounter,
      poMessage->u16GetFunctionID() ,               // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                  // Opcode
      );

   if(oMEMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMEMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oVoiceMail_ME: 'MethodError', enPostMessage failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oVoiceMail_ME.vDestroy();
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_BtDeviceVoiceRecognition_Error
*
* DESCRIPTION: Sends BT VR Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_BtDeviceVoiceRecognition_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER  fc_phone_tclService_Telephone::vHandle_BtDeviceVoiceRecognition_Error"));
   most_telfi_tclMsgBTDeviceVoiceRecognitionError oBTDeviceVoiceRecognition_ME;
   oBTDeviceVoiceRecognition_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMEMsg(oBTDeviceVoiceRecognition_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMEMsg,                                              // ServiceDataMsg
      m_oBTDeviceVoiceRecMessage.u16SourceAppID,     // Target AppID
      m_oBTDeviceVoiceRecMessage.u16RegisterID,      // RegisterID
      m_oBTDeviceVoiceRecMessage.u16CmdCounter,      // CmdCounter,
      m_oBTDeviceVoiceRecMessage.u16FunctionID,      // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                         // Opcode
      );

   if(oMEMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMEMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oBTDeviceVoiceRecognition_ME: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oBTDeviceVoiceRecognition_ME.vDestroy();
}

/****************************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_BtDeviceVoiceRecognitionExtended_Error
*
* DESCRIPTION: Sends BT VR Method error
*
* PARAMETER: e8ErrorCode - Most error code
*
* RETURNVALUE: None
*
******************************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_BtDeviceVoiceRecognitionExtended_Error(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode)
{
   ETG_TRACE_USR4(("ENTER  fc_phone_tclService_Telephone::vHandle_BtDeviceVoiceRecognitionExtended_Error"));
   most_telfi_tclMsgBTDeviceVoiceRecognitionExtendedError oBTDeviceVoiceRecognitionExt_ME;
   oBTDeviceVoiceRecognitionExt_ME.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMEMsg(oBTDeviceVoiceRecognitionExt_ME, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);
   vInitServiceData( oMEMsg,                                              // ServiceDataMsg
      m_oBTDeviceVoiceRecExtendedMessage.u16SourceAppID,     // Target AppID
      m_oBTDeviceVoiceRecExtendedMessage.u16RegisterID,      // RegisterID
      m_oBTDeviceVoiceRecExtendedMessage.u16CmdCounter,      // CmdCounter,
      m_oBTDeviceVoiceRecExtendedMessage.u16FunctionID,      // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR                         // Opcode
      );

   if(oMEMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =_poMainAppl->enPostMessage(&oMEMsg, TRUE);
      if(AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(( "oBTDeviceVoiceRecognitionExt_ME: 'MethodError', enPostMessage() failed, 'ail_tenCommunicationError' = %u",(tU32)enResult ));
      }
   }
   else
   {
      ETG_TRACE_ERR(("  Message is not Valid"));
   }

   oBTDeviceVoiceRecognitionExt_ME.vDestroy();
}

/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vGetMostErrorCode
*
* DESCRIPTION: This returns equivalent most error code of dbus errors
*
* PARAMETER: rfpCallInstance - call instance
*
* RETURNVALUE: most error code
*
***************************************************************************/
most_fi_tcl_e8_ErrorCode::tenType fc_phone_tclService_Telephone::vGetMostErrorCode(fc_phone_tclCallInstance*& rfpCallInstance)
{
   ETG_TRACE_USR4((" ENTER fc_phone_tclService_Telephone::vGetMostErrorCode"));

   if(!rfpCallInstance)
   {
      ETG_TRACE_USR4((" rfpCallInstance is NULL. Hence returning"));
      return most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
   }

   if (FC_PHONE_CALLINSTANCE_ACK_STATE_ERROR == rfpCallInstance->m_AckError)
   {
      ETG_TRACE_USR3((" most_fi_tcl_e8_ErrorCode::FI_EN_BUSY"));
      return most_fi_tcl_e8_ErrorCode::FI_EN_BUSY;
   }
   else if (FC_PHONE_CALLINSTANCE_ACKERROR_NOTAVAILABLE == rfpCallInstance->m_AckError)
   {
      ETG_TRACE_USR3((" most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE"));
      return most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
   }
   else if(FC_PHONE_CALLINSTANCE_ACK_NO_AUDIO_CHANNEL == rfpCallInstance->m_AckError)
   {
      ETG_TRACE_USR3((" most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE"));
      return most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
   }
   else
   {
      ETG_TRACE_USR3((" Defalut : most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE"));
      return most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
   }
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandleOffState
*
* DESCRIPTION: Handles Audio signals when system goes to off state
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandleOffState(T_SCOConnectionStatus* SCOConexnStatus)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vHandleOffState "));
   if((fc_phone_tclApp::m_poMainAppInstance)->m_u8FBlockState == FC_PHONE_STATE_OFF)
   {
      if(SCOConexnStatus->SCOConnectionStatus == FC_PHONE_DBUS_SCO_CONN_ESTABLISHED)
      {
         ETG_TRACE_USR3((" System Goes to OFF state and hence Transferring Audio to handset"));
         m_pDBus_CCA_If->bAudioTransfer(SCOConexnStatus->BD_Address,FC_PHONE_CALLTRANSFER_FROMHANSFREE_TOHANDSET);
      }
   }
}


/*******************************************************************************
*
* FUNCTION: vSetDtmfStatus()
*
* DESCRIPTION:Setter
*
* PARAMETER: True/False
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vSetDtmfStatus(tBool blValue)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclService_Telephone::vSetDtmfStatus"));
   m_bDtmfProcessing = blValue;
}


/*******************************************************************************
*
* FUNCTION: vSetDtmfStatus()
*
* DESCRIPTION:Setter
*
* PARAMETER: True/False
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vAppendDtmfString(fc_phone_tclCallInstance*& rfpCallInstance, tChar* pu8DTMFChar, tU16 u16CallInstanceID)
{
   ETG_TRACE_USR4((" ENTER : fc_phone_tclService_Telephone::vAppendDtmfString"));
   tU16 u16Size  = static_cast<tU16>(OSAL_u32StringLength(pu8DTMFChar));
   tU16 uDtmfLen = static_cast<tU16>(OSAL_u32StringLength(rfpCallInstance->DTMFChar));
   tU16 uIndex   = 0;

   NULL_CHECK(rfpCallInstance)


      if ((uDtmfLen + u16Size) > DTMF_MAXSTR_LEN)
      {
         ETG_TRACE_USR3((" (uDtmfLen + u16Size) > DTMF_MAXSTR_LEN"));
         vHandle_SendDTMF_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
         return;
      }

      for (uIndex = 0; uIndex < u16Size; uIndex++)
      {
         rfpCallInstance->DTMFChar[uDtmfLen + uIndex] = pu8DTMFChar[uIndex];
      }

      uDtmfLen = static_cast<tU16>(OSAL_u32StringLength(rfpCallInstance->DTMFChar));
      ETG_TRACE_USR3((" uDtmfLen =  %d      ", uDtmfLen));

      if (m_poSMManager)
      {
         m_poSMManager->vSetDtmfLen(uDtmfLen, u16CallInstanceID);
      }
}


/*******************************************************************************
*
* FUNCTION: vTimerCallBackAcquireTelephoneNumber
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid vTimerCallBackAcquireTelephoneNumber(tU16 u16TimerId)
{
   ETG_TRACE_USR4(("ENTER: vTimerCallBackAcquireTelephoneNumber u16TimerId %d",u16TimerId));
   tU8 pu8_btActiveBDAddressId;
   char *pu8_btActiveDeviceAddress = NULLPTR;
   tU8 actDeviceHandle = fc_phone_tclService_Telephone::pGetInstance()->u8GetActiveDeviceHandle();
   pu8_btActiveBDAddressId =(tU8)(fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetBTAddressIdFromDeviceHandle(actDeviceHandle);
   if(INVALID_BDADDRESS_ID != pu8_btActiveBDAddressId)
   {
      pu8_btActiveDeviceAddress = const_cast<tChar *> (fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(pu8_btActiveBDAddressId)); //CMG3G-4847 - To solve Lint
   }
   else
   {
      ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
   }

   if(FC_PHONE_ACQUIRETELEPHONENUMBER_TIMER_ID == u16TimerId)
   {
      (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(u16TimerId);
      if(pu8_btActiveDeviceAddress)
      {
         ETG_TRACE_USR4((" On time out sending request to acquire telephone # again"));
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->m_pDBus_CCA_If->bAcquireTelephoneNumber((const char*)pu8_btActiveDeviceAddress);
      }
      else
      {
         ETG_TRACE_USR4(("Fetching Device address"));
         ETG_TRACE_USR4((" On time out sending request to acquire telephone # again"));
      }
   }
   else
   {
      ETG_TRACE_ERR((" vTimerCallBackAcquireTelephoneNumber : Invalid timer id"));
   }
}


/*******************************************************************************
*
* FUNCTION: vAcquireTelephoneNumberStatusCheck
*
* DESCRIPTION:Acquire Telephone Number Status
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vAcquireTelephoneNumberStatusCheck(const Async_method_data_t* const vAsync_method_data)
{
   ETG_TRACE_USR3((" ENTER : fc_phone_tclService_Telephone::vAcquireTelephoneNumberStatusCheck"));

   if(FC_PHONE_DBUS_STATUS_SUCCESS == vAsync_method_data->Status)
   {
      (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_ACQUIRETELEPHONENUMBER_TIMER_ID);
      m_u8AcquireTelephoneNumberRetryCnt = 0;
      tU32 u32_size = static_cast<tU32>(OSAL_u32StringLength(vAsync_method_data->SubcriberNumber));

      if(FC_PHONE_MAX_CONNECTED_DEVICE > m_u8BDAddressIDforAcquiringTelNo)
      {
         vStringCopy(m_oPhoneData[m_u8BDAddressIDforAcquiringTelNo].m_SubscriberMSISDN,(const char*)vAsync_method_data->SubcriberNumber,u32_size +1);
         ETG_TRACE_USR3((" m_oPhoneData[%d].m_SubscriberMSISDN = %s ",m_u8BDAddressIDforAcquiringTelNo, m_oPhoneData[m_u8BDAddressIDforAcquiringTelNo].m_SubscriberMSISDN));
         vHandle_AcquireTelephoneNumber_MethodResult();
      }
   }
   else
   {
      if(m_u8AcquireTelephoneNumberRetryCnt >= FC_PHONE_RETRY_CNT_ACQUIRETELEPHONENUMBER)
      {
         ETG_TRACE_ERR(("   Sending Method Error ::oAcquireTelephoneNumber_MError as Retry Count exceeds "));
         most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;
         if(FC_PHONE_DBUS_ERRORCODE_STATE_ERROR == vAsync_method_data->Code)
         {
            e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }
         else if(FC_PHONE_DBUS_ERRORCODE_ERR_BUSY == vAsync_method_data->Code)
         {
            e8ErrorCode =  most_fi_tcl_e8_ErrorCode::FI_EN_BUSY;
         }
         else
         {
            e8ErrorCode= most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }
         vHandle_AcquireTelephoneNumber_Error(e8ErrorCode);
         m_u8AcquireTelephoneNumberRetryCnt = 0;
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_ACQUIRETELEPHONENUMBER_TIMER_ID);
      }
      else
      {
         if((0 != vAsync_method_data->Code) || (FC_PHONE_DBUS_FAIL == vAsync_method_data->Status))
         {
            // Retry acquiring Telephone Number
            ETG_TRACE_USR3(("   Retry AcquireTelephoneNumber"));
            // Start the timer and retry after 3S
            (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Start(FC_PHONE_ACQUIRETELEPHONENUMBER_TIMER_ID, FC_PHONE_ONE_SECOND * 3, vTimerCallBackAcquireTelephoneNumber);
            m_u8AcquireTelephoneNumberRetryCnt++;
            ETG_TRACE_USR3(("   m_u8AcquireTelephoneNumberRetryCnt = %d",m_u8AcquireTelephoneNumberRetryCnt));
         }
         else
         {
            m_u8AcquireTelephoneNumberRetryCnt = 0;
            (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_ACQUIRETELEPHONENUMBER_TIMER_ID);
            //Send ME for Acquire Telephone #
            ETG_TRACE_ERR(("   Sending Method Error ::oAcquireTelephoneNumber_MError...... "));
            most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;
            if(FC_PHONE_DBUS_ERRORCODE_STATE_ERROR == vAsync_method_data->Code)
            {
               e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
            }
            else if(FC_PHONE_DBUS_ERRORCODE_ERR_BUSY == vAsync_method_data->Code)
            {
               e8ErrorCode =  most_fi_tcl_e8_ErrorCode::FI_EN_BUSY;
            }
            else
            {
               e8ErrorCode= most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
            }
            vHandle_AcquireTelephoneNumber_Error(e8ErrorCode);
         }
      }
   }
}


/*******************************************************************************
*
* FUNCTION: vResetAudioChannel
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vResetAudioChannel()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vResetAudioChannel() entered"));
   for(tU16 u16index = 0 ;u16index < FC_PHONE_MAX_CALLINSTANCES ;u16index++)
   {
      fc_phone_tclCallInstance* pCallInstance = &m_oCallInstances[u16index];

      //Since reference is taken from a member function, pCallInstance cannot be NULL, lint fix
      pCallInstance->m_u8AudioChannel = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
      pCallInstance->m_bUsingVehicleAudio = FALSE;
      if (m_poSMManager)
         m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_VEHICLEAUDIO,pCallInstance);
   }
}

/*******************************************************************************
*
* FUNCTION: vSetPhoneChannelforRingingCallInstance
*
* DESCRIPTION: Set the Channel ID as Phone channel to Call Instance with call status as Ringing
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vSetPhoneChannelforRingingCallInstance()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSetPhoneChannelforRingingCallInstance() entered"));
   for(tU8 u8Index = 0 ;u8Index < FC_PHONE_MAX_CALLINSTANCES ;u8Index++)
   {
      if(FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE == m_oCallInstances[u8Index].m_u16CallStatusNotice)
      {
         m_oCallInstances[u8Index].m_u8AudioChannel = FC_PHONE_AUDIOCHANNEL_PHONE;
         ETG_TRACE_USR3((" Setting AudioChannel = FC_PHONE_AUDIOCHANNEL_PHONE for m_poCallInstance u8Index: %d", u8Index));
         break;
      }
   }
}

/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vRequestPhoneStatus
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vRequestPhoneStatus(const tChar *pDeviceAddress)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vRequestPhoneStatus ENTERED with Address: %s", pDeviceAddress));

   tU8 u8_BDAddressId = fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(pDeviceAddress);
   ETG_TRACE_USR4(("u8_BDAddressId: %d", u8_BDAddressId));

   if(INVALID_BDADDRESS_ID != u8_BDAddressId)
   {
      tU8 *pu8_btDeviceName;

      m_pDBus_CCA_If->bGetPhoneStatus(pDeviceAddress);
      m_pDBus_CCA_If->bGetSCOConnectionStatus(pDeviceAddress);
      m_pDBus_CCA_If->bGetCallStatus(pDeviceAddress);

      if(m_pobtTelClient)
      {
         pu8_btDeviceName = (tU8*)const_cast<tChar*>(m_pobtTelClient->sGetDeviceNameFromBtAddressId(u8_BDAddressId)); // Lint fix
         if(pu8_btDeviceName)
         {
            if (0 != u8StringCompare(m_oPhoneData[u8_BDAddressId].m_achCellDeviceName, (const char*)pu8_btDeviceName))
            {
               vStringCopy(m_oPhoneData[u8_BDAddressId].m_achCellDeviceName, (const char*) pu8_btDeviceName, FC_PHONE_STRING_LENGTH_MAX);
               vSendCellDeviceName(u8_BDAddressId);
            }
         }
         else
         {
            ETG_TRACE_ERR((" DeviceName is NULL"));
         }
      }

   }
}


/*************************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vUpdatePhoneStatusSLCoff
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phone_tclService_Telephone::vUpdatePhoneStatusSLCoff(tU8 u8BDId)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vUpdatePhoneStatusSLCoff ENTERED with u8BDId: %d", u8BDId));

   m_u8CurrentBDAddressIDforUpdate = u8BDId;

   ail_tenCommunicationError enResultStatus = eUpdateClients(MOST_TELFI_C_U16_CELLNETWORKSTATUS);

   if (AIL_EN_N_NO_ERROR != enResultStatus)
   {
      /* Throw up error */
      ETG_TRACE_ERR(( "CellNetworkStatus_Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResultStatus ));
   }

   ail_tenCommunicationError enResultState = eUpdateClients(MOST_TELFI_C_U16_CELLSTATEOFCHARGE);

   if (AIL_EN_N_NO_ERROR != enResultState)
   {
      /* Throw up error */
      ETG_TRACE_ERR(( "CellStateOfChargeStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResultState ));
   }


   ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_CELLSIGNALSTRENGTH);

   if (AIL_EN_N_NO_ERROR != enResult)
   {
      /* Throw up error */
      ETG_TRACE_ERR(( "CellSignalStrengthStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
   }

   vSendCellDeviceName(u8BDId);

   ETG_TRACE_USR3(("Updating Network operator"));
   ail_tenCommunicationError enResultNetwork = eUpdateClients(MOST_TELFI_C_U16_NETWORKOPERATOR);

   if(AIL_EN_N_NO_ERROR != enResultNetwork)
   {
      ETG_TRACE_ERR(("vSendNetworkOperatorName: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u",(tU32)enResultNetwork));
   }

   ail_tenCommunicationError enResultHFService = eUpdateClients(MOST_TELFI_C_U16_HFSERVICE);

   if (AIL_EN_N_NO_ERROR != enResultHFService)
   {
      /* Throw up error */
      ETG_TRACE_ERR(( "vSendHFServiceStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResultHFService ));
   }

   vSendBtVRStatus(false, u8BDId);

   m_u8CurrentBDAddressIDforUpdate = INVALID_BDADDRESS_ID;
}


/*******************************************************************************
*
* FUNCTION: getAgThreeWayCallSuppFeatures()
*
* DESCRIPTION: This function returns AgThreeWayCallSuppFeatures
*
* PARAMETER: None.
*
* RETURNVALUE: unsigned integer
*
********************************************************************************/
tU8 fc_phone_tclService_Telephone::getAgThreeWayCallSuppFeatures() const
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::getAgThreeWayCallSuppFeatures ENTERED"));
   if(m_pobtTelClient)
   {
      tU8 u8ActiveBDAddressId =(tU8) m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
      if(INVALID_BDADDRESS_ID != u8ActiveBDAddressId)
      {
         ETG_TRACE_USR4(("value of m_u8AgThreeWayCallSuppFeatures is %d", m_oPhoneData[u8ActiveBDAddressId].m_u8AgThreeWayCallSuppFeatures));
         return m_oPhoneData[u8ActiveBDAddressId].m_u8AgThreeWayCallSuppFeatures;
      }
      else
      {
         ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
      }
   }
   return FC_PHONE_ENHANCECALLCTRL_UNKNOWN;
}

//GMMY16-12628- According to the feedback from ALPS, the HfTerminateAll request shall only be sent if the phone supports three way call.
//Therefore you have to replace the HfTerminateAll request by HfTerminateCall (only if the phone reports three way call as "not supported"). 

/*******************************************************************************
*
* FUNCTION: getAgIsThreeWayCallSuppd()
*
* DESCRIPTION: This function returns whether the phone support threeway call or not.
*
* PARAMETER: None.
*
* RETURNVALUE: unsigned integer
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bIsAgThreeWayCallSupported() const
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::getAgIsThreeWayCallSuppd ENTERED"));
   if(m_pobtTelClient)
   {
      tU8 u8ActiveBDAddressId =(tU8) m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(m_u8ActiveDeviceHandle);
      if(INVALID_BDADDRESS_ID != u8ActiveBDAddressId)
      {
         ETG_TRACE_USR4(("value of m_u8AgThreeWayCallSuppFeatures is %d", m_oPhoneData[u8ActiveBDAddressId].m_bThreeWayCallSupport));
         return m_oPhoneData[u8ActiveBDAddressId].m_bThreeWayCallSupport;
      }
      else
      {
         ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
      }
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION: u8getDeviceHandle()
*
* DESCRIPTION:
*
* PARAMETER: None.
*
* RETURNVALUE: unsigned integer
*
********************************************************************************/
tU8 fc_phone_tclService_Telephone::u8getDeviceHandle(tU8 BDAddrID)
{
   tU8 devHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN;
   if( INVALID_BDADDRESS_ID != BDAddrID)
   {
      devHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(BDAddrID);
   }
   else
   {
      devHandle = m_u8ActiveDeviceHandle;
   }

   return devHandle;
}


/*******************************************************************************
*
* FUNCTION: copy_PnpInfoInd_from_loopback_queue()
*
* DESCRIPTION:
*
* PARAMETER: None.
*
* RETURNVALUE: unsigned integer
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::copy_PnpInfoInd_from_loopback_queue( T_PnpInfoInd*  pPnpInfoInd,LoopBackData *p_LB_data)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::copy_PnpInfoInd_from_loopback_queue entered"));
   pPnpInfoInd->m_usSpecificationID = p_LB_data->PnpInfoInd.m_usSpecificationID;
   pPnpInfoInd->m_usProductID = p_LB_data->PnpInfoInd.m_usProductID;
   pPnpInfoInd->m_usVendorID = p_LB_data->PnpInfoInd.m_usVendorID;
   pPnpInfoInd->m_ucPrimaryRecord = p_LB_data->PnpInfoInd.m_ucPrimaryRecord;
   pPnpInfoInd->m_usVendorIDSource = p_LB_data->PnpInfoInd.m_usVendorIDSource;
   pPnpInfoInd->m_usVersion = p_LB_data->PnpInfoInd.m_usVersion;
   vStringCopy(pPnpInfoInd->BD_Address,p_LB_data->PnpInfoInd.BD_Address,PHONE_BDADRESS_LENGTH);
}

//gmmy15-11360
/*******************************************************************************
*
* FUNCTION: vResetStoredCallInstance()
*
* DESCRIPTION:  Resets the stored call instance value to default while unlocking from work queue.
*
* PARAMETER: None.
*
* RETURNVALUE: unsigned integer
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vResetStoredCallInstance(tU16 u16FunId)
{
   ETG_TRACE_USR4((" vResetStoredCallInstance Entered  %d ", u16FunId));
   switch (u16FunId)
   {
      case MOST_TELFI_C_U16_HANGUPCALL:
      {
         ETG_TRACE_USR4((" HangUpCall  m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM"));
         m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM;
      }
      break;

      case MOST_TELFI_C_U16_ACCEPTCALL:
      {
         ETG_TRACE_USR4((" AcceptCall  m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM"));
         m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM;
      }
      break;

      case MOST_TELFI_C_U16_CANCELOUTGOINGCALL:
      {
         ETG_TRACE_USR4((" CancelOutgoingCall m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM"));
         m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM;
      }
      break;

      case MOST_TELFI_C_U16_IGNORECALL:
      {
         ETG_TRACE_USR4((" IgnoreCall m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM"));
         m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM;
      }
      break;

      case MOST_TELFI_C_U16_REJECTCALL:
      {
         ETG_TRACE_USR4((" IgnoreCall m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM"));
         m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM;
      }
      break;

      default:
      {
         ETG_TRACE_USR4((" In default Resetting all the variables "));
         m_u16StoredCallInstanceForHangUp = FC_PHONE_DEFAULT_STORED_PARAM;
         m_u16StoredCallInstanceForAccept = FC_PHONE_DEFAULT_STORED_PARAM;
         m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM;
         m_u16StoredCallInstanceForIgnore = FC_PHONE_DEFAULT_STORED_PARAM;
         m_u16StoredCallInstanceForReject = FC_PHONE_DEFAULT_STORED_PARAM;
      }
      break;
   }
}


/*******************************************************************************
*
* FUNCTION: oGetSLCStatus()
*
* DESCRIPTION:  Getter function for SLC status
*
* PARAMETER: None.
*
* RETURNVALUE: SLCStatus
*
********************************************************************************/
most_telfi_tclMsgSLCStatusStatus fc_phone_tclService_Telephone::oGetSLCStatus()
{
   return m_oSLCStatusStatus;
}


/*******************************************************************************
*
* FUNCTION: oGetCallInstances()
*
* DESCRIPTION:  Getter function for call instances
*
* PARAMETER: None.
*
* RETURNVALUE: call instance
*
********************************************************************************/
const fc_phone_tclCallInstance* fc_phone_tclService_Telephone::oGetCallInstances() //CMG3G-4847 - To solve Lint
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::oGetCallInstances entered "));
   return m_oCallInstances;
}

/*******************************************************************************
*
* FUNCTION: isAppleDevice()
*
* DESCRIPTION:  Check whether the connected device is apple
*
* PARAMETER: None.
*
* RETURNVALUE: AV manager client handler
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bIsAppleDevice(tU8 u8BDAddressID)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::bIsAppleDevice entered with u8BDAddressID: %d", u8BDAddressID));
   if (((FC_PHONE_APPLE_USB_IMPLEMENTER_VENDOR_ID == m_u16PnpInfoVendorID[u8BDAddressID])
      && (m_u16PnpInfoVendorSource[u8BDAddressID] == FC_PHONE_USB_IMPLEMENTER_FORUM_ID_SOURCE))||
      ((FC_PHONE_APPLE_BLUETOOTH_SIG_VENDOR_ID == m_u16PnpInfoVendorID[u8BDAddressID])
      &&(m_u16PnpInfoVendorSource[u8BDAddressID] == FC_PHONE_BLUETOOTH_SIG_ID_SOURCE)))
   {
      return TRUE;
   }
   else
   {
      return FALSE;
   }
}
/*******************************************************************************
*
* FUNCTION: bGetSpmSubState()
*
* DESCRIPTION:
*
* PARAMETER: None.
*
* RETURNVALUE: Boolean
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bGetSpmSubState()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::bGetSpmSubState ENTERED "));

   tU8 u8ActiveBDAddressId = INVALID_BDADDRESS_ID;
   tBool bGetActiveDeviceAddress = b_get_ActiveDevice_AddressId_FromBT(&u8ActiveBDAddressId);
   if (FALSE == bGetActiveDeviceAddress)
   {
      ETG_TRACE_ERR((" ACTIVE DEVICE ADDRESS IS NULL"));
      return CALLSTATUS_INACTIVE;
   }

   for(tU8 u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      ETG_TRACE_USR4(("m_oCallInstances[u8Index].m_u16CallStatusNotice: %d", m_oCallInstances[u8Index].m_u16CallStatusNotice));

      // CRQ 90674: PHONE substate to be set to TRUE, even when phone is in Handset (privacy) mode
      if( m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE
            || m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD
            || m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_CONFERENCE
            || m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_DIALING )
      {
         fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
         tU8 u8SCOConnectionStatus = p_clientHndler_bluetooth->vGetSCOConnectionStatus(const_cast<tChar *> (fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(u8ActiveBDAddressId))); //CMG3G-4847 - To solve Lint
         ETG_TRACE_USR4(("u8SCOConnectionStatus: %u", u8SCOConnectionStatus));
         if(FC_PHONE_DBUS_SCO_CONN_ESTABLISHED == u8SCOConnectionStatus)
            return CALLSTATUS_ACTIVE;
      }
      else if ( m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE
            ||  m_oCallInstances[u8Index].m_u16CallStatusNotice == FC_PHONE_CALLINSTANCE_CALLSTATUS_BUSY )
      {
         return CALLSTATUS_ACTIVE;
      }
   }
   return CALLSTATUS_INACTIVE;
}

/*******************************************************************************
*
* FUNCTION: vUpdateSpmSubState()
*
* DESCRIPTION:
*
* PARAMETER: None.
*
* RETURNVALUE: NA
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vUpdateSpmSubState()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vUpdateSpmSubState ENTERED"));

   tBool bSpmState = bGetSpmSubState();
   if ( m_bSpmSubState != bSpmState )
   {
      // Update SPM substate when changes.
      m_bSpmSubState = bSpmState;

      fc_phone_tclClientHandler_SystemState *p_ClientHdlerSystemState = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerSystemState;
      if (p_ClientHdlerSystemState)
      {
         p_ClientHdlerSystemState->vSendSpmSubState(m_bSpmSubState);
      }
   }
   else
   {
      ETG_TRACE_USR4(("SPM substate is not changed!!!!"));
   }

   vCheckAndUpdateHFProfileUsage();
}

/*******************************************************************************
*
* FUNCTION: vCheckAndUpdateHFProfileUsage()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: NA
*
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vCheckAndUpdateHFProfileUsage()
{
   ETG_TRACE_USR4(("vCheckAndUpdateHFProfileUsage ENTERED"));

   fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;

   if (!p_clientHndler_bluetooth)
   {
      ETG_TRACE_USR4(("p_clientHndler_bluetooth is NULL"));
      return;
   }

   tBool bHFProfileUsage = FALSE;

   for (tU8 index = 0; index < FC_PHONE_MAX_CALLINSTANCES; ++index)
   {
      if (FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE != m_oCallInstances[index].m_u16CallStatusNotice)
      {
         bHFProfileUsage = TRUE;
         break;
      }
   }

   ETG_TRACE_USR4(("bHFProfileUsage : %d", bHFProfileUsage));

   ETG_TRACE_USR4(("m_u8HFProfDevHandle: %d and m_u8ActiveDeviceHandle = %d", m_u8HFProfDevHandle, m_u8ActiveDeviceHandle));

   if(bHFProfileUsage)
   {
      if (FC_PHONE_DEVICEHANDLE_UNKNOWN == m_u8HFProfDevHandle)
      {
         m_u8HFProfDevHandle = m_u8ActiveDeviceHandle;
         p_clientHndler_bluetooth->vSetBTProfilesUsage(m_u8HFProfDevHandle, true);
      }
      else if (m_u8ActiveDeviceHandle == m_u8HFProfDevHandle)
      {
         ETG_TRACE_USR4(("The value would be already set"));
      }
      else
      {
         p_clientHndler_bluetooth->vSetBTProfilesUsage(m_u8HFProfDevHandle, false);
         m_u8HFProfDevHandle = m_u8ActiveDeviceHandle;
         p_clientHndler_bluetooth->vSetBTProfilesUsage(m_u8HFProfDevHandle, true);
      }
   }

   else
   {
      if (FC_PHONE_DEVICEHANDLE_UNKNOWN == m_u8HFProfDevHandle)
      {
         ETG_TRACE_USR4(("The value would be already 0"));
      }
      else
      {
         p_clientHndler_bluetooth->vSetBTProfilesUsage(m_u8HFProfDevHandle, bHFProfileUsage);
         m_u8HFProfDevHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN;
      }
   }
}

/*GMMY16-7759 Not Possible to Switch to second call, if first ongoing call is ended from Handset
When any call is ended, waiting state of the other call instance should be set to FALSE as it is not waiting any more.*/
/*******************************************************************************
** FUNCTION:   vCheckAndUpdateCallWaitingState(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a function for checking if call status is IDLE.
*   If IDLE, sets callwaiting of other instance to false.
*  PARAMETERS: None
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vCheckAndUpdateCallWaitingState()
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vCheckAndUpdateCallWaitingState ENTERED with m_oCallInstances[0].m_bcallWaitingState: %d and m_oCallInstances[1].m_bcallWaitingState: %d", m_oCallInstances[0].m_bcallWaitingState,m_oCallInstances[1].m_bcallWaitingState));

   if (FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE == m_oCallInstances[0].m_u16CallStatusNotice)
   {
      m_oCallInstances[1].m_bcallWaitingState = FALSE;
      ETG_TRACE_USR4((" Updated m_oCallInstances[1].m_bcallWaitingState to: %d", m_oCallInstances[1].m_bcallWaitingState));
   }
   if (FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE == m_oCallInstances[1].m_u16CallStatusNotice)
   {
      m_oCallInstances[0].m_bcallWaitingState = FALSE;
      ETG_TRACE_USR4((" Updated m_oCallInstances[0].m_bcallWaitingState to: %d", m_oCallInstances[0].m_bcallWaitingState));
   }
}
//End of fix GMMY16-7759

//Bugfix for GMMY16-12538: [CT]Inconsistencies with Siri
#ifdef PHONE_SIRI
tVoid vPhoneTimerCallBackService(tU16 u16TimerId)
{
   ETG_TRACE_USR4((" vPhoneTimerCallBackService "));

   if(fc_phone_tclApp::m_poMainAppInstance && fc_phone_tclApp::m_poMainAppInstance->m_poTelephone)
   {
      ETG_TRACE_USR4(("calling vTimerCallBack"));
      fc_phone_tclApp::m_poMainAppInstance->m_poTelephone->vTimerCallBack(u16TimerId);
   }
}

tVoid fc_phone_tclService_Telephone::vTimerCallBack(tU16 u16TimerId)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vTimerCallBack ENTERED u16TimerId %d ", u16TimerId));

   tChar u8ConnectedDeviceAddress[PHONE_BDADRESS_LENGTH] =
   {  0};

   if (FC_PHONE_SIRISTATUS_RETRY_TIMER_ID == u16TimerId)
   {
      (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_SIRISTATUS_RETRY_TIMER_ID);
      for (tU8 ind = 0; ind < FC_PHONE_MAX_CONNECTED_DEVICE; ind++)
      {
         if(m_bSiriRetryTimerRunning[ind])
         {
            m_bSiriRetryTimerRunning[ind] = FALSE;
            vStringCopy(u8ConnectedDeviceAddress, fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(ind), PHONE_BDADRESS_LENGTH);

            fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
            if(p_clientHndler_bluetooth && p_clientHndler_bluetooth->bGetAppleDeviceStatusfromBDAddressID(ind) && strlen(u8ConnectedDeviceAddress) > 0)
            {
               m_oSiriData[ind].vEnableSiri(u8ConnectedDeviceAddress);
            }
         }
      }
   }
   else if(FC_PHONE_SIRINRSTATUS_RETRY_TIMER_ID == u16TimerId)
   {
      (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_SIRINRSTATUS_RETRY_TIMER_ID);
      for (tU8 ind = 0; ind < FC_PHONE_MAX_CONNECTED_DEVICE; ind++)
      {
         if(m_bSiriNrRetryTimerRunning[ind])
         {
            m_bSiriNrRetryTimerRunning[ind] = FALSE;
            vStringCopy(u8ConnectedDeviceAddress, fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(ind), PHONE_BDADRESS_LENGTH);
            if(fc_phone_tclSM::m_pDbusIf && strlen(u8ConnectedDeviceAddress) > 0)
            {
               fc_phone_tclSM::m_pDbusIf->bSetSiriNrStatus(u8ConnectedDeviceAddress,(const char)FC_PHONE_ENABLED);
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" vTimerCallBack : Invalid timer id"));
   }
}
#endif //PHONE_SIRI

/*******************************************************************************
*
* FUNCTION: bCheckIdleCallStatus()
*
* DESCRIPTION: Function returns true if both the call instances are IDLE
*
* PARAMETER: None.
*
* RETURNVALUE: TRUE  -> If both call instances are IDLE &
*              FALSE -> If both the call instances are not IDLE
*
********************************************************************************/
tBool fc_phone_tclService_Telephone::bCheckIdleCallStatus()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::bCheckIdleCallStatus ENTERED "));
   for(tU8 u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; u8Index++)
   {
      if(FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE != m_oCallInstances[u8Index].m_u16CallStatusNotice)
      {
         return FALSE;
      }
   }
   return TRUE;
}

//Fix for task CMG3G-5839
tVoid fc_phone_tclService_Telephone::vPostMethodError(tU16 u16FunctionID)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vPostMethodError ENTERED "));

   switch (u16FunctionID)
   {
      case MOST_TELFI_C_U16_ACCEPTCALL:
      {
         ETG_TRACE_USR4(("AcceptCall"));
         vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
      break;

      case MOST_TELFI_C_U16_IGNORECALL:
      {
         ETG_TRACE_USR4(("IgnoreCall"));
         vHandle_IgnoreCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
      break;

      case MOST_TELFI_C_U16_HANGUPCALL:
      {
         ETG_TRACE_USR4(("HangUpCall"));
         vHandle_HangUpCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);

      }
      break;

      case MOST_TELFI_C_U16_DIAL:
      {
         ETG_TRACE_USR4(("DialCall"));
         vHandle_Dial_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
      break;

      case MOST_TELFI_C_U16_DIALEXTENDED:
      {
         ETG_TRACE_USR4(("DialCallExtended"));
         vHandle_DialExtended_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
      break;

      case MOST_TELFI_C_U16_CANCELOUTGOINGCALL:
      {
         ETG_TRACE_USR4(("CancelOutgoingCall"));
         //Check if the below line can be moved to vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode)
         m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM; //Reset stored call instance for Cancel
         vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
      break;

      case MOST_TELFI_C_U16_STARTSTOPWAITINGMODE:
      {
         ETG_TRACE_USR4(("StartStopWaitingMode"));
         vHandle_StartStopWaitingMode_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
      break;

      case MOST_TELFI_C_U16_SETACTIVEPHONE:
      {
         ETG_TRACE_USR4(("SetActivePhone"));
         vHandle_SetActivePhone_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("In default, Invalid function ID:: %x received", u16FunctionID));
         //Since Method Error is posted. release the Mutex lock.
         fc_phone_tclPhoneMutex::PhoneMutexUnlock();
      }
   }
}


/*******************************************************************************
 * FUNCTION:   vHandle_StartStopWaitingMode_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *    This is a handler function for StartStopWaitingMode method to start a WAITING mode
 *    or to stop an already active WAITING MODE. It can also be triggerred to accept an
 *    incoming call and start a WAITING mode.
 *
 *  PARAMETERS:
 *  IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_StartStopWaitingMode_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vHandle_StartStopWaitingMode_MethodStart entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_telfi_tclMsgStartStopWaitingModeMethodStart oStartStopWaitingMode_MS;

   if (OSAL_ERROR != oVisitorMsg.s32GetData(oStartStopWaitingMode_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION))
   {
      ETG_TRACE_USR4(("vHandle_StartStopWaitingMode_MethodStart WaitingModeAction : %u",
            ETG_ENUM(TR_PHON_WAITINGMODEACTION, oStartStopWaitingMode_MS.e8WaitingModeAction.enType)));

#ifdef PHONE_WAITINGMODE
      fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();

      tBool bEventConsumed = FALSE;

      if (most_fi_tcl_e8_TelWaitingModeAction::FI_EN_E8START == oStartStopWaitingMode_MS.e8WaitingModeAction.enType)
      {
         bEventConsumed = poWaitingMode->bProcessEvent(FC_PHONE_WAITING_MODE_EVENT_START);
      }
      else
      {
         bEventConsumed = poWaitingMode->bProcessEvent(FC_PHONE_WAITING_MODE_EVENT_STOP);
      }

      if (FALSE == bEventConsumed)
      {
         ETG_TRACE_ERR(("vHandle_StartStopWaitingMode_MethodStart: Event not consumed"));
         vHandle_StartStopWaitingMode_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
      else
      {
         m_oStartStopWaitingMode.u16SourceAppID = poMessage->u16GetSourceAppID();
         m_oStartStopWaitingMode.u16RegisterID = poMessage->u16GetRegisterID();
         m_oStartStopWaitingMode.u16CmdCounter = poMessage->u16GetCmdCounter();
         m_oStartStopWaitingMode.u16FunctionID = poMessage->u16GetFunctionID();
      }
#else
      vHandle_StartStopWaitingMode_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
#endif
   }
   else
   {
      ETG_TRACE_ERR(("vHandle_StartStopWaitingMode_MethodStart: Received message is invalid!"));
      vHandle_StartStopWaitingMode_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }

   oStartStopWaitingMode_MS.vDestroy();
}



/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_StartStopWaitingMode_Error
*
* DESCRIPTION: Sends  StartStopWaitingMode Method Error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_StartStopWaitingMode_Error(
      most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vHandle_StartStopWaitingMode_Error entered"));

   most_telfi_tclMsgStartStopWaitingModeError oStartStopWaitingModeError;

   oStartStopWaitingModeError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oStartStopWaitingModeError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   if (poMessage)
   {
      vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(),
            poMessage->u16GetRegisterID(),
            poMessage->u16GetCmdCounter(),
            poMessage->u16GetFunctionID(),
            AMT_C_U8_CCAMSG_OPCODE_ERROR);
   }
   else
   {
      vInitServiceData(oMRMsg,
            m_oStartStopWaitingMode.u16SourceAppID,
            m_oStartStopWaitingMode.u16RegisterID,
            m_oStartStopWaitingMode.u16CmdCounter,
            m_oStartStopWaitingMode.u16FunctionID,
            AMT_C_U8_CCAMSG_OPCODE_ERROR);
   }

   if (oMRMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vHandle_StartStopWaitingMode_Error: 'MethodError', "
               "enPostMessage failed, 'ail_tenCommunicationError' = %u",
               (tU32) enResult));
      }
   }
   else
   {
      ETG_TRACE_ERR(("VisitorMessage is not Valid!"));
   }

   oStartStopWaitingModeError.vDestroy();
}

#ifdef PHONE_WAITINGMODE
tVoid fc_phone_tclService_Telephone::vSendStartStopWaitingModeMethodResult(
      most_fi_tcl_e8_TelWaitingModeStatus::tenType enWaitingModeStatus)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSendStartStopWaitingModeMethodResult entered"));

   most_telfi_tclMsgStartStopWaitingModeMethodResult oStartStopWaitingMode_MR;

   oStartStopWaitingMode_MR.e8WaitingModeStatus.enType = enWaitingModeStatus;

   fi_tclVisitorMessage oMRMsg(oStartStopWaitingMode_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   ETG_TRACE_USR4(("WaitingModeStatus : '%u'", ETG_ENUM(TR_PHON_WAITINGMODESTATUS, oStartStopWaitingMode_MR.e8WaitingModeStatus.enType)));

   vInitServiceData(oMRMsg, m_oStartStopWaitingMode.u16SourceAppID,
         m_oStartStopWaitingMode.u16RegisterID, m_oStartStopWaitingMode.u16CmdCounter,
         m_oStartStopWaitingMode.u16FunctionID, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);

   if (oMRMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vHandle_StartStopWaitingMode_MethodStart: 'MethodResult', "
               "enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32) enResult));
      }
   }
   oStartStopWaitingMode_MR.vDestroy();
}


tBool fc_phone_tclService_Telephone::bGetCallInstanceIdForWaitingCall(tU16* pu16CallInstanceID)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::bGetCallInstanceIdForWaitingCall entered"));

   tBool bWaitingCallFound = FALSE;

   for (tU16 u16index = 0; u16index < FC_PHONE_MAX_CALLINSTANCES; u16index++)
   {
      fc_phone_tclCallInstance* pCallInstance = &m_oCallInstances[u16index];

      if (FC_PHONE_CALLINSTANCE_INCOMINGCALLEVENT_RINGING == pCallInstance->m_u16IncomingCallEvent)
      {
         *pu16CallInstanceID = pCallInstance->m_u16Id;
         bWaitingCallFound = TRUE;
         break;
      }
   }
   return bWaitingCallFound;
}

tVoid fc_phone_tclService_Telephone::vSendWaitingModeState(most_fi_tcl_e8_WaitingModeState::tenType enWaitingMode)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSendWaitingModeState entered"));

   m_oWaitingModeStateStatus.e8WaitingModeState.enType = enWaitingMode;

   ETG_TRACE_USR4(("m_oWaitingModeStateStatus.e8WaitingModeState.enType:%d enWaitingMode:%d", m_oWaitingModeStateStatus.e8WaitingModeState.enType, enWaitingMode));

   ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_WAITINGMODESTATE);
   if (AIL_EN_N_NO_ERROR != enResult)
   {
      /* Throw up error */
      ETG_TRACE_ERR(( "WaitingModeStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
   }
}
#endif //PHONE_WAITINGMODE

// CMG3G-7982- Extend VR session
/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vStopExtendedVRSession
*
* DESCRIPTION: Interface to stop Extended VR Session
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vStopExtendedVRSession()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vStopExtendedVRSession entered"));
   if(m_poSMVRSession)
   {
       m_poSMVRSession->vStopExtendedVRSession();
   }
}



/***********************************************************************************************
 *
 * FUNCTION: fc_phone_tclService_Telephone::vSetVehicleBtAddress
 *
 * DESCRIPTION: Stores the Vehicle BT address.
 *
 * PARAMETER: IN:
 *
 * RETURNVALUE:
 *
 ***********************************************************************************************/
tVoid fc_phone_tclService_Telephone::vSetVehicleBtAddress(char* Address)
{
   ETG_TRACE_USR2(("vSetVehicleBtAddress - %s", Address));
   SetVehicleBtAddress(Address);
}

// NCG3D-3221- Hold for handset-handsfree mode

/*******************************************************************************
 *
 * FUNCTION: vSCOChannelReleaseTimerCallBack
 *
 * DESCRIPTION: Timer callback on expiry of the vSCOChannelReleaseTimerCallBack Timer
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid vSCOChannelReleaseTimerCallBack(tU16 m_u16SCOChannelReleaseTimerId)
{
   ETG_TRACE_USR4(("ENTER vSCOChannelReleaseTimerCallBack"));
   fc_phone_tclService_Telephone *ptelService = fc_phone_tclService_Telephone::pGetInstance();
   ptelService->vStopTimerAndProcessSCODisconnection(m_u16SCOChannelReleaseTimerId);
}


/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclService_Telephone::vStopTimerAndProcessSCODisconnection
 *
 * DESCRIPTION: This function is used to Stop the vStopTimerAndProcessSCODisconnection Timer
 * to release the channel and call the SCO disconnection processing function.
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclService_Telephone::vStopTimerAndProcessSCODisconnection(tU16 u16SCOChannelReleaseTimerId)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vStopTimerAndProcessSCODisconnection "));
   if(FC_PHONE_SCO_CHANNEL_RELEASE_TIMER_ID == u16SCOChannelReleaseTimerId && TRUE == m_bScoChannelReleaseTimerRunning)
   {
      (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(u16SCOChannelReleaseTimerId);
      m_bScoChannelReleaseTimerRunning = FALSE;
      ETG_TRACE_USR3(("m_bScoChannelReleaseTimerRunning = FALSE"));
      vProcessSCODisconnection();
   }
}

/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclService_Telephone::vUpdateSCOChannelConnectionStatus
 *
 * DESCRIPTION: This function is used to copy the SCOConnectionStatus  to local variable.
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclService_Telephone::vUpdateSCOChannelConnectionStatus(T_SCOConnectionStatus &SCOConnexnStatus)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vUpdateSCOChannelConnectionStatus "));
   vStringCopy(m_SCOChannelConnectionStatus.BD_Address, SCOConnexnStatus.BD_Address, PHONE_BDADRESS_LENGTH);
   m_SCOChannelConnectionStatus.SCOConnectionStatus = SCOConnexnStatus.SCOConnectionStatus;
   m_SCOChannelConnectionStatus.Codec = SCOConnexnStatus.Codec;
}

/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclService_Telephone::vProcessSCODisconnection
 *
 * DESCRIPTION: This function is used to process the SCo disconnection.
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclService_Telephone::vProcessSCODisconnection()
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vProcessSCODisconnection "));
   vMuteMicroPhone(FALSE, FALSE);
   if(!(m_poSMExtVRSession->isExtVRSessionIdle()) || (m_poSMVRSession->u16GetVRSessionState() == (tU16)STATE_ACTIVE) || (TRUE == bCheckIdleCallStatus()))
   {
      (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vResetSaScoEcnrInit(EN_SCO_EST);
      ecnrStopAudio();
   }
   m_u8EcnrConfigurationDataset = 0;
   vProcessChannelAndUpdateSCOConnectionstatus();
}

/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclService_Telephone::vProcessChannelAndUpdateSCOConnectionstatus
 *
 * DESCRIPTION: This function is used to update SCOconnection status and process audio signal.
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclService_Telephone::vProcessChannelAndUpdateSCOConnectionstatus()
{
   vHandleOffState(&m_SCOChannelConnectionStatus);
   ETG_TRACE_USR2((" SCOConnectionStatus :: %d Device Address :: %s",m_SCOChannelConnectionStatus.SCOConnectionStatus,m_SCOChannelConnectionStatus.BD_Address));
   m_pobtTelClient->vUpdateSCOConnectionStatus(&m_SCOChannelConnectionStatus);

   // gmmy15-5984
   // check - if BTVR session is Active, then no need to process it
  if (m_poSMVRSession->u16GetVRSessionState() != (tU16)STATE_ACTIVE)
   {
#ifdef PHONE_WAITINGMODE
     fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
     //RTC-268032 : The SCO connection for NarrowBand was treated as disconnection. Hence it is not handled in the StateMachines.
     if ((m_SCOChannelConnectionStatus.SCOConnectionStatus == FC_PHONE_DBUS_SCO_CONN_DISCONNECTED) && (FC_PHONE_WAITINGMODE_START_IN_PROGRESS == poWaitingMode->enGetWaitingModeState())\
           && ((FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE == m_oCallInstances[0].m_u16CallStatusNotice) || (FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE == m_oCallInstances[1].m_u16CallStatusNotice)))
     {
        ETG_TRACE_USR4(("SCO disconnection not handled in SM_WaitingMode and SM_Acceptcall"));
     }
     else
     {
        ETG_TRACE_USR4((" Processing the received SCO"));
        trFC_Phone_SCOConnStatus rSCOConnStatus;
        rSCOConnStatus.u8SCOConnectionStatus = m_SCOChannelConnectionStatus.SCOConnectionStatus;
        rSCOConnStatus.u8Codec = m_SCOChannelConnectionStatus.Codec;

        poWaitingMode->bProcessEvent(FC_PHONE_WAITING_MODE_EVENT_SCO_CONNECTION_STATUS, &rSCOConnStatus);
#endif
        m_poSMManager->vProcessAudioSignal() ;
#ifdef PHONE_WAITINGMODE
     }
#endif
   }
   vUpdateSpmSubState(); // Update spm substate when vehicle audio changes.
}

/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclService_Telephone::validateTelephoneNumber
 *
 * DESCRIPTION: This function is used to validate the telephone number.
 *
 * PARAMETER: phoneNumber - Dialled telephone number
 *            isTelNumberEmpty - A boolean to update whether the number is empty or not
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
void fc_phone_tclService_Telephone::validateTelephoneNumber(tString phoneNumber,tBool &isTelNumberEmpty)
{
    if(!phoneNumber)
    {
       ETG_TRACE_ERR(("  NULL phoneNumber for Client"));
       isTelNumberEmpty = TRUE;
    }
    else
    {
       ETG_TRACE_USR3(("  phoneNumber = '%s'  ", phoneNumber ));

       // Phone number check:
       // Extract only "valid" characters of given phone number (see GIS-327-3259 Dialing String Characters),
       // valid characters are: 1 2 3 4 5 6 7 8 9 0 * # + A B C and W; (Note: All are ASCII characters)
       tU32 u32SrcIdx  = 0;
       tU32 u32DestIdx = 0;
       tU32 u32SrcLen  = static_cast<tU32>(OSAL_u32StringLength(phoneNumber));
       tChar cPhoneNumChar = ' ';

       for( u32SrcIdx = 0; u32SrcIdx < u32SrcLen; u32SrcIdx++)
       {
          cPhoneNumChar = phoneNumber[u32SrcIdx];

          if(u32DestIdx < FC_PHONE_DIAL_NUMBER_MAX_LEN_IN_CHARS)  // Copying is stopped if FC_PHONE_DIAL_NUMBER_MAX_LEN_IN_CHARS characters are available in m_sValidPhoneNumber
          {
             // First we check if highest bit is NOT set to 1. If so, we have an single ASCII character (and not an UTF-8 multi-byte character).
             if( (cPhoneNumChar & 0x80) != 0x80 )
             {
                // We have an single ASCII character and can continue with more detailed checks:
                if(  ((cPhoneNumChar >= 0x30) && (cPhoneNumChar <= 0x39))  // Check for digit, i.e. range: 0x30 ('0') to 0x39 ('9')
                   ||
                   ((cPhoneNumChar == 0x2A) || (cPhoneNumChar == 0x23) || (cPhoneNumChar == 0x2B) || (cPhoneNumChar == 0x2C)) // Check for: '*' (0x2A), '#' (0x23), '+' (0x2B)and ',' (0x2C)
                   ||
                   ((cPhoneNumChar >= 0x41) && (cPhoneNumChar <= 0x43)) // Check for 'A', 'B', 'C', i.e range: 0x41 to 0x43
                   ||
                   (cPhoneNumChar == 0x57)  // Check for 'W' (0x57)
                   )
                {
                   // Copy 'cPhoneNumChar' to valid phone number
                   m_sValidPhoneNumber[u32DestIdx++] = cPhoneNumChar;
                }
             }
          }  // end of: if(u32DestIdx < ..)
          else
          {
             break;  // Maximum number of digits in destination 'm_sValidPhoneNumber' is reached. No need to run the for loop any longer.
          }
       }
    }

    ETG_TRACE_USR3(( "  m_sValidPhoneNumber= '%s'  ", m_sValidPhoneNumber ));

    if (0x00 == m_sValidPhoneNumber[0])
    {
       ETG_TRACE_ERR(("  ERROR: Extracted 'm_sValidPhoneNumber' is Empty  "));
       isTelNumberEmpty = TRUE;
    }
    else
    {
       ETG_TRACE_USR3(("  Extracted 'm_sValidPhoneNumber' has 'valid' characters  "));
    }
}

/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclService_Telephone::ecnrServerVoiceMode
 *
 * DESCRIPTION: This function is used to set the ECNR server mode.
 *
 * PARAMETER: enType - This is received ECNR type from client
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
void fc_phone_tclService_Telephone::ecnrServerVoiceMode(tU8 ecnrType)
{
    switch (ecnrType)
      {
      case most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting::FI_EN_E8ECNR_NOCHANGE:
         {
            ETG_TRACE_USR3((" Echo Cancellation NoiseReduction Setting - ECNR_NOCHANGE "));
         }
         break;

      case most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting::FI_EN_E8ECNR_VOICE:
         {
            ETG_TRACE_USR3((" Echo Cancellation NoiseReduction Setting - ECNR_VOICE "));
         }
         break;

      case most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting::FI_EN_E8ECNR_SERVER:
         {
            ETG_TRACE_USR3((" Echo Cancellation NoiseReduction Setting - ECNR_SERVER "));
         }
         break;

      default:
         break;
      }
}

/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclService_Telephone::vResetDiallerInfoForCallInstance
 *
 * DESCRIPTION: This function is used to reset the dialer name and number for a particular call instance.
 *
 * PARAMETER: callInstance - call instance
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
void fc_phone_tclService_Telephone::vResetDiallerInfoForCallInstance(tU8 callInstance)
{
   ETG_TRACE_USR3((" Reset of dialler info for the call instance %d ",callInstance));
   if(callInstance < FC_PHONE_MAX_CALLINSTANCES)
   {
      if(m_sDialInfo[callInstance].m_sDialName)
      {
         delete []m_sDialInfo[callInstance].m_sDialName;
         m_sDialInfo[callInstance].m_sDialName = NULLPTR;
      }
      if(m_sDialInfo[callInstance].m_sDialNumber)
      {
         delete []m_sDialInfo[callInstance].m_sDialNumber;
         m_sDialInfo[callInstance].m_sDialNumber = NULLPTR;
      }
   }
}

/*******************************************************************************
 * FUNCTION:   vHandle_SetActivePhone_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *    This is a handler function for SetActivePhone method to set the given deviceHandle
 *    as the ACTIVE phone device.
 *
 *  PARAMETERS:
 *  IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SetActivePhone_MethodStart(amt_tclServiceData* poMessage, struct Job* f_poJob)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vHandle_SetActivePhone_MethodStart entered"));

   most_telfi_tclMsgSetActivePhoneMethodStart *oSetActivePhone_MS = NULLPTR;
   struct Job* poJob = NULLPTR;

   if (poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);
      oSetActivePhone_MS = new(std::nothrow) most_telfi_tclMsgSetActivePhoneMethodStart;

      if (oSetActivePhone_MS)
      {
         if (OSAL_ERROR != oVisitorMsg.s32GetData(*oSetActivePhone_MS, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION))
         {
            ETG_TRACE_USR4(("vHandle_SetActivePhone_MethodStart:oSetActivePhone_MS->u8DeviceHandle = %d", oSetActivePhone_MS->u8DeviceHandle));

            if (FC_PHONE_DEVICEHANDLE_UNKNOWN == m_u8StoredDeviceHandletoSetActive)
            {
               m_u8StoredDeviceHandletoSetActive = oSetActivePhone_MS->u8DeviceHandle;
               if ((m_u8StoredDeviceHandletoSetActive >= MINIMUM_DEVICE_HANDLE) && (m_u8StoredDeviceHandletoSetActive <= MAXIMUM_DEVICE_HANDLE))
               {
                  if (m_u8StoredDeviceHandletoSetActive != m_u8ActiveDeviceHandle)
                  {
                     //Create a new job
                     poJob = new Job;
                     if (poJob) //If creation of job is successful, assign parameters
                     {
                        poJob->m_pfMethod = &fc_phone_tclService_Telephone::vHandle_SetActivePhone_MethodStart;
                        poJob->u16CmdCounter = poMessage->u16GetCmdCounter();
                        poJob->u16FunctionID = poMessage->u16GetFunctionID();
                        poJob->u16RegisterID = poMessage->u16GetRegisterID();
                        poJob->u16SourceAppID = poMessage->u16GetSourceAppID();
                        poJob->m_poMethodStart = oSetActivePhone_MS;
                     }

                  }
                  else
                  {
                     ETG_TRACE_USR4(("The received DeviceHandle is already the ACTIVE device. Hence sending Method Result."));
                     vSend_SetActivePhone_MethodResult();
                     oSetActivePhone_MS->vDestroy();
                     delete oSetActivePhone_MS;
                     return;
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("vHandle_SetActivePhone_MethodStart: Received DeviceHandle is invalid!"));
                  vHandle_SetActivePhone_Error(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE, poMessage);
                  oSetActivePhone_MS->vDestroy();
                  delete oSetActivePhone_MS;
                  return;
               }
            }
            else
            {
               ETG_TRACE_USR4(("m_u8StoredDeviceHandletoSetActive = %d. The previous request is in progress. Hence sending ME", m_u8StoredDeviceHandletoSetActive));
               vHandle_SetActivePhone_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, false);
               oSetActivePhone_MS->vDestroy();
               delete oSetActivePhone_MS;
               return;
            }
         }
         else
         {
            ETG_TRACE_ERR(("vHandle_SetActivePhone_MethodStart: Received message is invalid!"));
            vHandle_SetActivePhone_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage, false);
            oSetActivePhone_MS->vDestroy();
            delete oSetActivePhone_MS;
            return;
         }
      }
   }
   else if (f_poJob)
   {
      poJob = f_poJob;
   }
   else
   {
      ETG_TRACE_ERR(("Should not come here"));
      return;
   }

   if (!poJob)
   {
      ETG_TRACE_ERR(("poJob is NULL!!"));
      vHandle_SetActivePhone_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

      if (oSetActivePhone_MS)
      {
         oSetActivePhone_MS->vDestroy();
         delete oSetActivePhone_MS;
      }
      return;
   }

   if (fc_phone_tclPhoneMutex::PhoneMutexTrylock(poJob) != 0) //Try to acquire lock
   {
      tVoid* pCustodialLose = poJob; //To remove LINT warning
      (tVoid) pCustodialLose;
      return; // Didn't get the lock :(
   }

   //If lock is acquired process the request
   m_oSetActivePhone.u16SourceAppID = poJob->u16SourceAppID;
   m_oSetActivePhone.u16RegisterID = poJob->u16RegisterID;
   m_oSetActivePhone.u16CmdCounter = poJob->u16CmdCounter;
   m_oSetActivePhone.u16FunctionID = poJob->u16FunctionID;

   oSetActivePhone_MS = (most_telfi_tclMsgSetActivePhoneMethodStart*) poJob->m_poMethodStart;
   NULL_CHECK(oSetActivePhone_MS);
   tU8 u8SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_FAILED;
   fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
   u8SCOConnectionStatus = p_clientHndler_bluetooth->vGetSCOConnectionStatus(const_cast<tChar *>(sGetDeviceAddressfromCSList(m_u8ActiveDeviceHandle)));
   ETG_TRACE_USR4(("u8SCOConnectionStatus = %d",u8SCOConnectionStatus));

   //***NOTE: VR session Extend Timer
   if ((TRUE == bIsActiveDeviceIdle()))
   {
      ETG_TRACE_USR4(("No Active calls in Active Device"));

      vSwitchActivePassive(m_u8StoredDeviceHandletoSetActive);
      vSend_SetActivePhone_MethodResult();
   }
   else
   {
      ETG_TRACE_USR4(("Active calls in Active Device. Hence sending ME"));
      vHandle_SetActivePhone_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);

      // The acquired PhoneMutex Lock is not released in "vHandle_SetActivePhone_Error()" if (NULL != poMessage).
      // Hence it should be released here.
      if (poMessage)
      {
         fc_phone_tclPhoneMutex::PhoneMutexUnlock();
      }
   }

   if (oSetActivePhone_MS)
   {
      oSetActivePhone_MS->vDestroy();
      delete oSetActivePhone_MS;
   }
   delete poJob;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSendSetActivePhone_MethodResult
*
* DESCRIPTION: Sends SetActivePhone Method Result
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSend_SetActivePhone_MethodResult()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSend_SetActivePhone_MethodResult entered"));

   m_u8StoredDeviceHandletoSetActive = FC_PHONE_DEVICEHANDLE_UNKNOWN;
   most_telfi_tclMsgSetActivePhoneMethodResult oSetActivePhone_MR;

   fi_tclVisitorMessage oMRMsg(oSetActivePhone_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   vInitServiceData(oMRMsg,
         m_oSetActivePhone.u16SourceAppID,
         m_oSetActivePhone.u16RegisterID,
         m_oSetActivePhone.u16CmdCounter,
         m_oSetActivePhone.u16FunctionID,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);

   fc_phone_tclPhoneMutex::PhoneMutexUnlock();

   if (oMRMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vSend_SetActivePhone_MethodResult: 'MethodResult', "
               "enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32) enResult));
      }
   }
   oSetActivePhone_MR.vDestroy();
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_SetActivePhone_Error
*
* DESCRIPTION: Sends  SetActivePhone Method Error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_SetActivePhone_Error(
      most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage, tBool bResetStoredDeviceHandletoSetActive)
{
   ETG_TRACE_USR4(("vHandle_SetActivePhone_Error entered with bResetStoredDeviceHandletoSetActive: %d", bResetStoredDeviceHandletoSetActive));

   if(bResetStoredDeviceHandletoSetActive)
   {
      m_u8StoredDeviceHandletoSetActive = FC_PHONE_DEVICEHANDLE_UNKNOWN;
   }

   most_telfi_tclMsgSetActivePhoneError oSetActivePhoneError;

   oSetActivePhoneError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oSetActivePhoneError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   if (poMessage)
   {
      vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(),
            poMessage->u16GetRegisterID(),
            poMessage->u16GetCmdCounter(),
            poMessage->u16GetFunctionID(),
            AMT_C_U8_CCAMSG_OPCODE_ERROR);
   }
   else
   {
      vInitServiceData(oMRMsg,
            m_oSetActivePhone.u16SourceAppID,
            m_oSetActivePhone.u16RegisterID,
            m_oSetActivePhone.u16CmdCounter,
            m_oSetActivePhone.u16FunctionID,
            AMT_C_U8_CCAMSG_OPCODE_ERROR);

      fc_phone_tclPhoneMutex::PhoneMutexUnlock();
   }

   if (oMRMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vHandle_SetActivePhone_Error: 'MethodError', "
               "enPostMessage failed, 'ail_tenCommunicationError' = %u",
               (tU32) enResult));
      }
   }
   else
   {
      ETG_TRACE_ERR(("VisitorMessage is not Valid!"));
   }

   ETG_TRACE_USR4(("vHandle_SetActivePhone_Error exited with m_u8StoredDeviceHandletoSetActive: %d", m_u8StoredDeviceHandletoSetActive));
   oSetActivePhoneError.vDestroy();
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_ActivateExternalVR_Error
*
* DESCRIPTION: Sends  ActivateExternalVR Method Error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_ActivateExternalVR_Error(
      most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ActivateExternalVR_Error entered"));

   most_telfi_tclMsgActivateExternalVRError oActivateExternalVRError;
   oActivateExternalVRError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oActivateExternalVRError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   if (poMessage)
   {
      vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(),
            poMessage->u16GetRegisterID(),
            poMessage->u16GetCmdCounter(),
            poMessage->u16GetFunctionID(),
            AMT_C_U8_CCAMSG_OPCODE_ERROR);
   }
   else
   {
      ETG_TRACE_USR4(("poMessage is not valid"));
   }

   if (oMRMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vHandle_ActivateExternalVR_Error: 'MethodError', "
               "enPostMessage failed, 'ail_tenCommunicationError' = %u",
               (tU32) enResult));
      }
   }
   else
   {
      ETG_TRACE_ERR(("VisitorMessage is not Valid!"));
   }

   ExtVRStateParams extVRStateParams;
   extVRStateParams.m_u8ExtVRState = EXT_VR_IDLE;
   extVRStateParams.m_u8ErrorCode = EXT_VR_ERROR_IN_STREAMING;
   m_poSMManager->vSendStatus(FC_PHONE_SMMANAGER_STATUS_EXTERNALVR_STATUS,&extVRStateParams);

   ETG_TRACE_USR4(("vHandle_ActivateExternalVR_Error exited"));
   oActivateExternalVRError.vDestroy();
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandle_DeActivateExternalVR_Error
*
* DESCRIPTION: Sends  DeActivateExternalVR Method Error
*
* PARAMETER: e8ErrorCode - Most error code
*            poMessage - Message object.
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_DeActivateExternalVR_Error(
      most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_DeActivateExternalVR_Error entered"));

   most_telfi_tclMsgDeActivateExternalVRError oDeActivateExternalVRError;
   oDeActivateExternalVRError.e8ErrorCode.enType = e8ErrorCode;

   fi_tclVisitorMessage oMRMsg(oDeActivateExternalVRError, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   if (poMessage)
   {
      vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(),
            poMessage->u16GetRegisterID(),
            poMessage->u16GetCmdCounter(),
            poMessage->u16GetFunctionID(),
            AMT_C_U8_CCAMSG_OPCODE_ERROR);
   }
   else
   {
      ETG_TRACE_USR4(("poMessage is not valid"));
   }

   if (oMRMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vHandle_DeActivateExternalVR_Error: 'MethodError', "
               "enPostMessage failed, 'ail_tenCommunicationError' = %u",
               (tU32) enResult));
      }
   }
   else
   {
      ETG_TRACE_ERR(("VisitorMessage is not Valid!"));
   }
   ETG_TRACE_USR4(("vHandle_DeActivateExternalVR_Error exited"));
   oDeActivateExternalVRError.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_ActivateExternalVR_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for ActivateExternalVR method
*   This function goes through State Machine and then request allocation for the VR channel to ARL
*  PARAMETERS:
*  IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_ActivateExternalVR_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::vHandle_ActivateExternalVR_MethodStart "));
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_telfi_tclMsgActivateExternalVRMethodStart oActivateExternalVR_MS;

   bool sendMethodError = true;

   if(OSAL_ERROR!= oVisitorMsg.s32GetData(oActivateExternalVR_MS,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION))
   {
      if((oActivateExternalVR_MS.sBdAddress.szValue == NULLPTR) || !(strlen(oActivateExternalVR_MS.sBdAddress.szValue)))
      {
         ETG_TRACE_USR4(("The BdAddress received for ExternalVR activation is not valid"));
      }
      else
      {
         char format_dev_addr[PHONE_BDADRESS_LENGTH] = {'\0'};
         convert_raw_btaddr_to_formatted(format_dev_addr,oActivateExternalVR_MS.sBdAddress.szValue,PHONE_BDADRESS_LENGTH);
         ETG_TRACE_USR4(("vHandle_ActivateExternalVR_MethodStart:oActivateExternalVR_MS.sBdAddress = %s", oActivateExternalVR_MS.sBdAddress.szValue));
         ETG_TRACE_USR4(("vHandle_ActivateExternalVR_MethodStart:format_dev_addr = %s", format_dev_addr));

         tU8 u8ActiveDeviceHandle = u8GetActiveDeviceHandle();
         tU8 u8_BDAddressId = fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(format_dev_addr);

         if(INVALID_BDADDRESS_ID != u8_BDAddressId)
         {
            tU8 u8SMDeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8_BDAddressId);
            if(u8ActiveDeviceHandle == u8SMDeviceHandle)
            {
               ETG_TRACE_USR4(("External VR request is from Active Device"));

               if (bIsActiveDeviceIdle() && (m_poSMManager->bIsExternalVRSessionIdle()))
               {
                  vSend_ActivateExternalVR_MethodResult(poMessage);
                  ETG_TRACE_USR4(("Active Device is in IDLE state"));
                  m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_CCX_EXT_VR_ACTIVE);
                  sendMethodError = false;
               }
            }
         }
      }
   }

   if (sendMethodError)
   {
      ETG_TRACE_ERR(( "vHandle_ActivateExternalVR_MethodStart: Send Method Error" ));
      vHandle_ActivateExternalVR_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }
   oActivateExternalVR_MS.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_DeActivateExternalVR_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for DeActivateExternalVR method
*   This function goes through State Machine and then request Deallocation for the VR channel to ARL
*  PARAMETERS:
*  IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandle_DeActivateExternalVR_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" ENTERING :fc_phone_tclService_Telephone::vHandle_DeActivateExternalVR_MethodStart "));
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_telfi_tclMsgDeActivateExternalVRMethodStart oDeActivateExternalVR_MS;
   if(OSAL_ERROR!= oVisitorMsg.s32GetData(oDeActivateExternalVR_MS,FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION))
   {
      if((oDeActivateExternalVR_MS.sBdAddress.szValue == NULLPTR) || (!(strlen(oDeActivateExternalVR_MS.sBdAddress.szValue))))
      {
         ETG_TRACE_USR4(("The BdAddress received for ExternalVR de-activation is not valid"));
         vHandle_DeActivateExternalVR_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         oDeActivateExternalVR_MS.vDestroy();
         return;
      }
      char format_dev_addr[PHONE_BDADRESS_LENGTH] = {'\0'};
      convert_raw_btaddr_to_formatted(format_dev_addr,oDeActivateExternalVR_MS.sBdAddress.szValue,PHONE_BDADRESS_LENGTH);
      ETG_TRACE_USR4(("vHandle_DeActivateExternalVR_MethodStart:oDeActivateExternalVR_MS.sBdAddress = %s", oDeActivateExternalVR_MS.sBdAddress.szValue));
      ETG_TRACE_USR4(("vHandle_DeActivateExternalVR_MethodStart:format_dev_addr = %s", format_dev_addr));

      tU8 u8ActiveDeviceHandle = u8GetActiveDeviceHandle();
      tU8 u8_BDAddressId = fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(format_dev_addr);
      if(INVALID_BDADDRESS_ID != u8_BDAddressId)
      {
         tU8 u8SMDeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8_BDAddressId);
         if(u8ActiveDeviceHandle == u8SMDeviceHandle)
         {
            vSend_DeActivateExternalVR_MethodResult(poMessage);
            m_poSMManager->vProcessExtVREvent(FC_PHONE_SM_EVENT_CCX_EXT_VR_INACTIVE);
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_DeActivateExternalVR_MethodStart: Not an active device. Hence not processed"));
            vHandle_DeActivateExternalVR_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vHandle_DeActivateExternalVR_MethodStart: Received INVALID_BDADDRESS_ID." ));
         vHandle_DeActivateExternalVR_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_DeActivateExternalVR_MethodStart: Received message invalid." ));
      vHandle_DeActivateExternalVR_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, poMessage);
   }
   oDeActivateExternalVR_MS.vDestroy();
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSend_ActivateExternalVR_MethodResult
*
* DESCRIPTION: Sends ActivateExternalVR Method Result
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSend_ActivateExternalVR_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSend_ActivateExternalVR_MethodResult entered"));

   most_telfi_tclMsgActivateExternalVRMethodResult oActivateExternalVR_MR;

   fi_tclVisitorMessage oMRMsg(oActivateExternalVR_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   if(poMessage)
   {
      vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(),
            poMessage->u16GetRegisterID(),
            poMessage->u16GetCmdCounter(),
            poMessage->u16GetFunctionID(),
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
   }
   if (oMRMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vSend_ActivateExternalVR_MethodResult: 'MethodResult', "
               "enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32) enResult));
      }
   }
   oActivateExternalVR_MR.vDestroy();
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSend_DeActivateExternalVR_MethodResult
*
* DESCRIPTION: Sends DeActivateExternalVR Method Result
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSend_DeActivateExternalVR_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSend_DeActivateExternalVR_MethodResult entered"));

   most_telfi_tclMsgDeActivateExternalVRMethodResult oDeActivateExternalVR_MR;

   fi_tclVisitorMessage oMRMsg(oDeActivateExternalVR_MR, FC_PHONE_SERVICE_TELEPHONE_FI_MAJOR_VERSION);

   if(poMessage)
   {
      vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(),
            poMessage->u16GetRegisterID(),
            poMessage->u16GetCmdCounter(),
            poMessage->u16GetFunctionID(),
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
   }
   if (oMRMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("vSend_DeActivateExternalVR_MethodResult: 'MethodResult', "
               "enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32) enResult));
      }
   }
   oDeActivateExternalVR_MR.vDestroy();
}
/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSwitchActivePassive
*
* DESCRIPTION: Sets the ActDeviceHandle device as Active device
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSwitchActivePassive(tU8 ActDeviceHandle, tBool bResetActiveDeviceHandletobeRetained)
{
   ETG_TRACE_USR4(("vSwitchActivePassive entered with ActDeviceHandle: %u and bResetActiveDeviceHandletobeRetained: %u", ActDeviceHandle, bResetActiveDeviceHandletobeRetained));

   if (m_u8ActiveDeviceHandle != ActDeviceHandle)
   {
      tU8 u8RecentActDeviceHandle = m_u8ActiveDeviceHandle;
      vSetasActiveDevice(ActDeviceHandle);
      tU8 u8BDAddressId =(tU8) m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(ActDeviceHandle);
      // Fix for NCG3D-85047
      // BTVRExtended status is required for SDS during device switch.
      // But this status update is not required for the normal device switch in VR Screen.
      if (false == (m_poSMVRSession->bIsVRActivationRequestPendingFromBTStack()))
      {
         vSendBtVRExtendedStatus(u8BDAddressId);
      }

      // Init of PhoneData is done before requesting for Phone status from the Stack.
      // This is to ensure that the Phone status is updated on Device Switching (Active/Passive)

      vInitPhoneData(u8BDAddressId);

      vRequestPhoneStatus(sGetDeviceAddressfromCSList(m_u8ActiveDeviceHandle));

      if (TRUE == bResetActiveDeviceHandletobeRetained)
      {
         m_u8ActiveDeviceHandletobeRetained = FC_PHONE_DEVICEHANDLE_UNKNOWN;
         ETG_TRACE_USR4(("m_u8ActiveDeviceHandletobeRetained: %u", m_u8ActiveDeviceHandletobeRetained));
      }

      if (TRUE == m_bScoChannelReleaseTimerRunning)
      {
         ETG_TRACE_USR4(("SCOChannelReleaseTimer should be stopped and the event need not be handled now since the ActivePassive switching is happening"));

         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_SCO_CHANNEL_RELEASE_TIMER_ID);
         m_bScoChannelReleaseTimerRunning = FALSE;
         ETG_TRACE_USR3(("m_bScoChannelReleaseTimerRunning = FALSE"));
         // NCG3D-131177:
         // During device switch, the SCO connection status is not updated.
         // This sometimes lead to retain the previous value(of the active device).
         T_SCOConnectionStatus SCOConnexnStatus;
         tU8 u8BDAddressId = m_pobtTelClient->u8GetBTAddressIdFromDeviceHandle(u8RecentActDeviceHandle);
         vStringCopy(SCOConnexnStatus.BD_Address,fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(u8BDAddressId),PHONE_BDADRESS_LENGTH);
         SCOConnexnStatus.SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_DISCONNECTED;
         SCOConnexnStatus.Codec = NARROW_BAND_SPEECH;
         m_pobtTelClient->vUpdateSCOConnectionStatus(&SCOConnexnStatus);
      }
   }
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSwitchActivePassiveFromVR
*
* DESCRIPTION: Sets m_u8DeviceHandletoSetActiveToStartVR as Active device
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSwitchActivePassiveFromVR()
{
   ETG_TRACE_USR4(("vSwitchActivePassiveFromVR entered and m_u8DeviceHandletoSetActiveToStartVR: %d", m_u8DeviceHandletoSetActiveToStartVR));
   vSwitchActivePassive(m_u8DeviceHandletoSetActiveToStartVR);
   m_u8DeviceHandletoSetActiveToStartVR = FC_PHONE_DEVICEHANDLE_UNKNOWN;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vInsertCallStatusinCSList
*
* DESCRIPTION: Used to create 2 entries of CS in the CallStatusList vector for the connected device
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vInsertCallStatusinCSList(tU8 u8BDId)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vInsertCallStatusinCSList entered with BDAddressID= %d", u8BDId));

   T_CallStatusList CSListItem;

   CSListItem.DeviceHandle = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8BDId);

   vStringCopy(CSListItem.BD_Address, fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(u8BDId), PHONE_BDADRESS_LENGTH);

   CSListItem.CallStatus.u8CallDirection = 0;
   CSListItem.CallStatus.u8CallHandle = 0;
   CSListItem.CallStatus.u8CallMode = 0;
   CSListItem.CallStatus.u8CallState = PHONE_CALLSTATE_IDLE;
   CSListItem.CallStatus.u8CallTotal = 0;
   CSListItem.CallStatus.u8CallType = 0;
   CSListItem.CallStatus.achCallNumber[0] = '\0';

   //Currently Two call instances are maintained for each Device. Hence two IDLE call status are allocated in the vector.
   for (tU8 ind = 0; ind < FC_PHONE_MAX_CALLINSTANCES; ++ind)
   {
      CallStatusList.push_back(CSListItem);
   }

   vPrintCallStatusListVector();
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vDeleteCSfromCallStatusList
*
* DESCRIPTION: Used to delete 2 entries of CS from the CallStatusList vector
*              for given device handle which is disconnected
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vDeleteCSfromCallStatusList(const tChar *pDeviceAddress)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vDeleteCSfromCallStatusList entered with DeviceAddress : %s", pDeviceAddress));

   for(tU8 i =0; i<CallStatusList.size(); ++i)
   {
      if (0 == u8StringCompare(CallStatusList[i].BD_Address, pDeviceAddress))
      {
         ETG_TRACE_USR4(("CallStatusList[%d].BD_Address = %s", i, CallStatusList[i].BD_Address));
         CallStatusList.erase(CallStatusList.begin()+i, CallStatusList.begin()+i+FC_PHONE_MAX_CALLINSTANCES);
         break;
      }
   }

   vPrintCallStatusListVector();
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vUpdateCSinCallStatusList
*
* DESCRIPTION: Used to Update call Status of any connected devices in CallStatusList vector
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vUpdateCSinCallStatusList(const tChar *pDeviceAddress, T_CallStatus &CallStatus)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vUpdateCSinCallStatusList entered with DeviceAddress : %s",pDeviceAddress));

   if(CallStatus.u8CallHandle > 2)
   {
      ETG_TRACE_USR4(("Ignore CallHandle greater than 2 directly"));
      return;
   }

   tU8 CSIndinCSList = 0xFF;

   ETG_TRACE_USR4(("Printing Items of CallStatusList before update"));
   vPrintCallStatusListVector();

   for(tU8 i =0; i<CallStatusList.size(); ++i)
   {
      if (0 == u8StringCompare(CallStatusList[i].BD_Address, pDeviceAddress))
      {
         CSIndinCSList = i;
         break;
      }
   }

   ETG_TRACE_USR4(("CSIndinCSList: %d", CSIndinCSList));
   if (0xFF == CSIndinCSList)
   {
      ETG_TRACE_USR4(("Address is not present in the CallStatusList vector"));
      return;
   }

   // If received IDLE CS, reset both the call instances
   if(CallStatus.u8CallHandle == 0)
   {
      for(tU8 ind =0; ind < FC_PHONE_MAX_CALLINSTANCES; ++ind)
      {
         vResetCallStatusinCallStatusList(static_cast<tU8>(CSIndinCSList+ind));
      }
      return;
   }

   ETG_TRACE_USR4(("Call Total : %d", CallStatus.u8CallTotal));
   // If phone number matches, simply update the corresponding Call instance
   tU8 ind = 0;
   for (ind = CSIndinCSList; ind < CSIndinCSList+FC_PHONE_MAX_CALLINSTANCES ; ++ind)
   {
      //RTC-652704 Added callhandle, as the stack supports and update the callstate for same number, as a new call.
      if ((0 == u8StringCompare(CallStatusList[ind].CallStatus.achCallNumber, CallStatus.achCallNumber)) &&
            (CallStatusList[ind].CallStatus.u8CallHandle == CallStatus.u8CallHandle))
      {
         if(CallStatus.u8CallTotal == 1 || (CallStatusList[ind].CallStatus.u8CallHandle != CallStatus.u8CallHandle))
         {
            if(ind%2 == 0)
               vResetCallStatusinCallStatusList(static_cast<tU8>(ind+1));
            else
               vResetCallStatusinCallStatusList(static_cast<tU8>(ind-1));
         }
         vCopyCallStatustoCSList(CallStatus, ind);
         return;
      }
   }

   // A new callstatus received. Hence update in the fresh slot
   for (ind = CSIndinCSList; ind < CSIndinCSList+FC_PHONE_MAX_CALLINSTANCES ; ++ind)
   {
      if (CallStatusList[ind].CallStatus.u8CallHandle == 0)
      {
         vCopyCallStatustoCSList(CallStatus, ind);
         return;
      }
   }

   // A new callstatus received.
   //when there were more than 2 calls, CH 2 got ended and the CH 3 is now CH 2
   for (ind = CSIndinCSList; ind < CSIndinCSList+FC_PHONE_MAX_CALLINSTANCES ; ++ind)
   {
      if (CallStatusList[ind].CallStatus.u8CallHandle == 2)
      {
         vCopyCallStatustoCSList(CallStatus, ind);
         return;
      }
   }
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vCopyCallStatustoCSList
*
* DESCRIPTION:
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vCopyCallStatustoCSList(T_CallStatus &CallStatus, tU8 ind)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vCopyCallStatustoCSList entered with ind : %d", ind));

   vStringCopy(CallStatusList[ind].CallStatus.achCallNumber , CallStatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);
   CallStatusList[ind].CallStatus.u8CallDirection = CallStatus.u8CallDirection;
   CallStatusList[ind].CallStatus.u8CallHandle = CallStatus.u8CallHandle;
   CallStatusList[ind].CallStatus.u8CallMode = CallStatus.u8CallMode;
   vSetCallStatefromDBUSCS(CallStatus.u8CallState, ind);
   CallStatusList[ind].CallStatus.u8CallTotal = CallStatus.u8CallTotal;
   CallStatusList[ind].CallStatus.u8CallType = CallStatus.u8CallType;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vResetCallStatusinCallStatusList
*
* DESCRIPTION: Used to Reset Call Status of any connected devices in CallStatusList vector
*              at the given iterator value
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vResetCallStatusinCallStatusList(tU8 iterator)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vResetCallStatusinCallStatusList entered with iterator: %d", iterator));
   CallStatusList[iterator].CallStatus.u8CallDirection = 0;
   CallStatusList[iterator].CallStatus.u8CallHandle = 0;
   CallStatusList[iterator].CallStatus.u8CallMode = 0;
   CallStatusList[iterator].CallStatus.u8CallState = PHONE_CALLSTATE_IDLE;
   CallStatusList[iterator].CallStatus.u8CallTotal = 0;
   CallStatusList[iterator].CallStatus.u8CallType = 0;
   CallStatusList[iterator].CallStatus.achCallNumber[0] = '\0';
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::u8GetDeviceHandlefromDevAddrinCSList
*
* DESCRIPTION: Used to get the Device Handle for the given Address from the CS list
*
* PARAMETER: None
*
* RETURNVALUE: DeviceHandle- If DeviceHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN, Device is not connected
*
*******************************************************************/
tU8 fc_phone_tclService_Telephone::u8GetDeviceHandlefromDevAddrinCSList(const tChar *pDeviceAddress)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::u8GetDeviceHandlefromDevAddrinCSList entered with pDeviceAddress: %s", pDeviceAddress));

   for(tU8 i=0; i<CallStatusList.size(); i=static_cast<tU8>(i+FC_PHONE_MAX_CALLINSTANCES))
   {
      if (0 == (u8StringCompare(CallStatusList[i].BD_Address, pDeviceAddress)))
      {
         ETG_TRACE_USR3(("Address is present at index= %d and Device Handle= %d ", i, CallStatusList[i].DeviceHandle));
         return CallStatusList[i].DeviceHandle;
      }
   }
   return FC_PHONE_DEVICEHANDLE_UNKNOWN;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::sGetDeviceAddressfromCSList
*
* DESCRIPTION: Used to get the Device Address for the given Device Handle from the CS list
*
* PARAMETER: None
*
* RETURNVALUE: Address = NULL - Device is not in the CS list
*
*******************************************************************/
const tChar* fc_phone_tclService_Telephone::sGetDeviceAddressfromCSList(tU8 DeviceHandle)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::sGetDeviceAddressfromCSList entered with DeviceHandle: %d", DeviceHandle));

   for(tU8 i=0; i<CallStatusList.size(); ++i)
   {
      if (CallStatusList[i].DeviceHandle == DeviceHandle)
      {
         ETG_TRACE_USR3(("Device Handle is present at index= %d and Device Address= %s ", i, CallStatusList[i].BD_Address));
         return CallStatusList[i].BD_Address;
      }
   }
   return NULLPTR;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSetCallStatefromDBUSCS
*
* DESCRIPTION: Used to set the Call State based on MOST_FI from DBUS call state
*
* PARAMETER: u8DBUSCallState- Call state received from DBus
*              index- Index of the CallStatusList vector which should be modified
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSetCallStatefromDBUSCS(tU8 u8DBUSCallState, tU8 index)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSetCallStatefromDBUSCS entered with u8DBUSCallState: %d and ind: %d", u8DBUSCallState, index));
   switch(u8DBUSCallState)
   {
      case PHONE_CALLSTATE_IDLE:
      {
         CallStatusList[index].CallStatus.u8CallState = \
               FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE;

      }
      break;
      case PHONE_CALLSTATE_ACTIVE:
      {
         CallStatusList[index].CallStatus.u8CallState = \
               FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE;

      }
      break;
      case PHONE_CALLSTATE_HELD:
      {
         CallStatusList[index].CallStatus.u8CallState = \
               FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD;

      }
      break;
      case PHONE_CALLSTATE_DIALING:
      {
         CallStatusList[index].CallStatus.u8CallState = \
               FC_PHONE_CALLINSTANCE_CALLSTATUS_DIALING;

      }
      break;
      case PHONE_CALLSTATE_ALERTING:
      {
         // TO DO: At the moment there is no corresponding state in most_TelFi
         //m_oCallInstances[u16CallInstanceIndex].m_u16CallStatusNotice =
         //    FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE;
         CallStatusList[index].CallStatus.u8CallState = \
               FC_PHONE_CALLINSTANCE_CALLSTATUS_DIALING;
      }
      break;
      case PHONE_CALLSTATE_INCOMING:
      {
         CallStatusList[index].CallStatus.u8CallState = \
               FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE;

      }
      break;
      case PHONE_CALLSTATE_WAITING:
      {
         CallStatusList[index].CallStatus.u8CallState = \
               FC_PHONE_CALLINSTANCE_CALLSTATUS_BUSY;

      }
      break;
      case PHONE_CALLSTATE_RESPONSE_AND_HOLD:
      {
         // TO DO: At the moment there is no corresponding state in most_TelFi
         // m_oCallInstances[u16CallInstanceIndex].m_u16CallStatusNotice =
         //     FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE;
      }
      break;
      case PHONE_CALLSTATE_UNKNOWN:
      {
         // TO DO: At the moment there is no corresponding state in most_TelFi
         // Stay in the same state
      }
      break;

      default:
      {
         ETG_TRACE_USR4((" In default"));
      }
      break;
   }
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSetCallStatebasedonDBusCS
*
* DESCRIPTION: Used to set the Call State based on DBUS call state from MOST FI Call state
*
* PARAMETER: u8DBUSCallState- Call state received from DBus
*              index- Index of the CallStatusList vector which should be modified
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSetCallStatebasedonDBusCS(tU8 u8MostFICallState, tU8 &u8DBUSCallState)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSetCallStatebasedonDBusCS entered with u8MostFICallState: %d ", u8MostFICallState));

   switch(u8MostFICallState)
   {
      case FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE :
      {
         u8DBUSCallState = PHONE_CALLSTATE_IDLE;
      }
      break;
      case FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE:
      {
         u8DBUSCallState = PHONE_CALLSTATE_ACTIVE;
      }
      break;
      case FC_PHONE_CALLINSTANCE_CALLSTATUS_ONHOLD:
      {
         u8DBUSCallState = PHONE_CALLSTATE_HELD;
      }
      break;
      case FC_PHONE_CALLINSTANCE_CALLSTATUS_DIALING:
      {
         u8DBUSCallState = PHONE_CALLSTATE_DIALING;
      }
      break;
      case FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE:
      {
         u8DBUSCallState = PHONE_CALLSTATE_INCOMING;
      }
      break;
      case FC_PHONE_CALLINSTANCE_CALLSTATUS_BUSY:
      {
         u8DBUSCallState = PHONE_CALLSTATE_WAITING;
      }
      break;
      default:
      {
         ETG_TRACE_USR4((" In default"));
      }
      break;
   }
   ETG_TRACE_USR4(("u8DBUSCallState: %d ", ETG_ENUM(TR_CALLSTATE,u8DBUSCallState)));
   ETG_TRACE_USR4(("u8DBUSCallState: %d ", u8DBUSCallState));
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSetasActiveDevice
*
* DESCRIPTION: Used to set the given Device Handle as Active
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSetasActiveDevice(tU8 DeviceHandle)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSetasActiveDevice entered with DeviceHandle: %d", DeviceHandle));
   m_u8ActiveDeviceHandle = DeviceHandle;

   fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;

   if(p_clientHndler_bluetooth)
   {
      p_clientHndler_bluetooth->vSetPrimaryDevice(DeviceHandle);
   }

   vSendPropertyUpdate(MOST_TELFI_C_U16_ACTIVEPHONEDEVICE, NULLPTR);
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::u8GetActiveDeviceHandle
*
* DESCRIPTION: Used to get the Active Device Handle
*
* PARAMETER: None
*
* RETURNVALUE: tU8 - Active Device Handle
*
*******************************************************************/
tU8 fc_phone_tclService_Telephone::u8GetActiveDeviceHandle()
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::u8GetActiveDeviceHandle and ActiveDeviceHandle: %d", m_u8ActiveDeviceHandle));
   return m_u8ActiveDeviceHandle;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSetSLCStatus
*
* DESCRIPTION: Used to set the SLC status
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSetSLCStatus(tBool SLCStatus)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSetSLCStatus entered with SLCStatus: %d", SLCStatus));
   if (!SLCStatus)
   {
      tS32 s32_connectedDeviceCount ;
      s32_connectedDeviceCount = fc_phone_tclDeviceList::getInstance()->s32GetConnectedDevNumFromDeviceList();
      ETG_TRACE_USR3((" Number of device(s) still connected = %d ",s32_connectedDeviceCount));
      if(s32_connectedDeviceCount == 0)
      {
         ETG_TRACE_USR3((" Setting SLC as OFF"));
         m_oSLCStatusStatus.bCommandStatus = FALSE;
      }
   }
   else
   {
      m_oSLCStatusStatus.bCommandStatus = TRUE;
   }
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vHandleWaitingMethodsOnDisconnection
*
* DESCRIPTION: Used to check for the waiting Methods and
*              send ME since the device is disconnected
*
* PARAMETER: Address of the Disconnected device
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vHandleWaitingMethodsOnDisconnection(const tChar *pDeviceAddress)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vHandleWaitingMethodsOnDisconnection entered with Address: %s", pDeviceAddress));

   for(tU8 i =0; i<CallStatusList.size(); i=static_cast<tU8>(i+FC_PHONE_MAX_CALLINSTANCES))
   {
      if (0 == u8StringCompare(CallStatusList[i].BD_Address, pDeviceAddress))
      {
         for (tU8 index = i; index < i+FC_PHONE_MAX_CALLINSTANCES; ++index)
         {
            if(m_u16StoredCallInstanceForAccept == index)
            {
               ETG_TRACE_USR4(("AcceptCall"));
               vHandle_AcceptCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
            else if(m_u16StoredCallInstanceForHangUp == index)
            {
               ETG_TRACE_USR4(("HangUpCall"));
               vHandle_HangUpCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
            else if(m_u16StoredCallInstanceForCancel == index)
            {
               ETG_TRACE_USR4(("CancelOutgoingCall"));
               //Check if the below line can be moved to vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode)
               m_u16StoredCallInstanceForCancel = FC_PHONE_DEFAULT_STORED_PARAM; //Reset stored call instance for Cancel
               vHandle_CancelOutgoingCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
            else if(m_u16StoredCallInstanceForIgnore == index)
            {
               ETG_TRACE_USR4(("IgnoreCall"));
               vHandle_IgnoreCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
            else if(m_u16StoredCallInstanceForReject == index)
            {
               ETG_TRACE_USR4(("RejectCall"));
               vHandle_RejectCall_Error(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
         }
         break;
      }
   }
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vCreateCallDatafromCSListIndex
*
* DESCRIPTION: Used to Create T_CallData variable
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vCreateCallDatafromCSListIndex(T_CallData* prCallData, int index)
{
   if(index >= static_cast<int>(CallStatusList.size()) || index < 0)
   {
      ETG_TRACE_USR4(("vCreateCallDatafromCSListIndex :: Index out of bound: %d", index));
      return;
   }
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vCreateCallDatafromCSListIndex entered with index: %d", index));
   prCallData->rCallStatus.u8CallDirection = CallStatusList[index].CallStatus.u8CallDirection;
   prCallData->rCallStatus.u8CallHandle = CallStatusList[index].CallStatus.u8CallHandle;
   prCallData->rCallStatus.u8CallMode = CallStatusList[index].CallStatus.u8CallMode;
   prCallData->rCallStatus.u8CallTotal = CallStatusList[index].CallStatus.u8CallTotal;
   prCallData->rCallStatus.u8CallType = CallStatusList[index].CallStatus.u8CallType;
   vStringCopy(prCallData->rCallStatus.achCallNumber , CallStatusList[index].CallStatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);

   tU8 u8DBusCallState = PHONE_CALLSTATE_IDLE;

   vSetCallStatebasedonDBusCS(CallStatusList[index].CallStatus.u8CallState, u8DBusCallState);
   prCallData->rCallStatus.u8CallState = u8DBusCallState;

   vStringCopy(prCallData->BD_Address, CallStatusList[index].BD_Address, PHONE_BDADRESS_LENGTH);

   //Populate CellDeviceName
   memset(prCallData->achCellDeviceName, 0, PHONE_CELLDEVICENAME_LENGTH);
   if(m_pobtTelClient)
   {
      tU8 u8_ConnectedBDAddressId=fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(CallStatusList[index].BD_Address);
      tU8 *pu8_btDeviceName = (tU8*)const_cast<tChar*>(m_pobtTelClient->sGetDeviceNameFromBtAddressId(u8_ConnectedBDAddressId)); // Lint fix
      if(pu8_btDeviceName)
      {
         vStringCopy(prCallData->achCellDeviceName, (const char*) pu8_btDeviceName, PHONE_CELLDEVICENAME_LENGTH);
      }
      else
      {
         ETG_TRACE_ERR(("DeviceName is NULL"));
      }
   }
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSwitchtoActive
*
* DESCRIPTION: Used to switch to Active
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSwitchtoActive(tU16 &u16ID)
{
   ETG_TRACE_USR4(("fc_phone_tclService_Telephone::vSwitchtoActive entered"));

   // Call Instances will be initialized. Hence Call Instance Index 0 is used directly to create AcceptCall_SM
   u16ID = 0x00;

   tBool bIsUsingVehicleAudio = (m_oCallInstances[0].m_bUsingVehicleAudio || m_oCallInstances[1].m_bUsingVehicleAudio);
   tU8 u8AudioChannelofReceivedCIId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
   tBool bMicroPhoneMuteStatus = m_oCallInstances[0].m_bMicroPhoneMuteStatus || m_oCallInstances[1].m_bMicroPhoneMuteStatus;

   if(((fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->bIsPhoneChannelGet()) &&
         ((FC_PHONE_AUDIOCHANNEL_PHONE == m_oCallInstances[0].m_u8AudioChannel)||
               (FC_PHONE_AUDIOCHANNEL_PHONE == m_oCallInstances[1].m_u8AudioChannel)))
   {
      ETG_TRACE_USR4(("LC_Phone channel acquired already or the acquisition is in progress"));
      u8AudioChannelofReceivedCIId = FC_PHONE_AUDIOCHANNEL_PHONE;
   }

   for(tU8 u8index = 0 ;u8index < FC_PHONE_MAX_CALLINSTANCES; ++u8index)
   {
      fc_phone_tclCallInstance* pCallInstance = &m_oCallInstances[u8index];
      pCallInstance->vInit();
      vResetDiallerInfoForCallInstance(u8index);
   }

   NULL_CHECK(m_poCallManager)
   m_poCallManager->vResetCallHandlerMaptable();
   T_CallData CallData;
   //m_u16StoredCallInstanceForAccept added to CH MapTable
   vCreateCallDatafromCSListIndex(&CallData, m_u16StoredCallInstanceForAccept);

   //vAddEntryToCallHandlerMaptable is not used since the function uses the SM mapped Call instances to check the Object state.
   //And it fails to check for the second call Instance and issue occures if only the second call instance is active
   m_poCallManager->vAddEntryToCallHandlerMaptablewithCIId(&CallData, u16ID);

   fc_phone_tclCallInstance *pCallInstance =  &m_oCallInstances[u16ID];
   pCallInstance->m_bUsingVehicleAudio = bIsUsingVehicleAudio;
   pCallInstance->m_u8AudioChannel = u8AudioChannelofReceivedCIId;
   pCallInstance->m_bMicroPhoneMuteStatus = bMicroPhoneMuteStatus;
   pCallInstance->m_u16IncomingCallEvent = FC_PHONE_CALLINSTANCE_INCOMINGCALLEVENT_RINGING;
   pCallInstance->m_enCallStatusHdlr_MapState = FC_PHONE_CALLSTATUS_MAPSTATE_SM_USING_MAPPED ;

   if (FC_PHONE_CALLINSTANCE_CALLSTATUS_BUSY == CallStatusList[(tU8)m_u16StoredCallInstanceForAccept].CallStatus.u8CallState)
   {
      pCallInstance->m_bcallWaitingState = true;
      ETG_TRACE_USR4((" Updated m_oCallInstances[%d].m_bcallWaitingState to: %d", u16ID, pCallInstance->m_bcallWaitingState));
   }
   else
   {
      pCallInstance->m_bcallWaitingState = false;
      ETG_TRACE_USR4((" Updated m_oCallInstances[%d].m_bcallWaitingState to: %d", u16ID, pCallInstance->m_bcallWaitingState));
   }

   fc_phone_tclSM::vCopyCallData(&CallData, pCallInstance);
   pCallInstance->m_u8ObjectState = FC_PHONE_CALLINSTANCE_OBJECTSTATE_ACTIVE;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::u8GetBDAddressIDforStatusUpdate
*
* DESCRIPTION: Manipulates m_u8CurrentBDAddressIDforUpdate and
*              returns the apt BDAddressID for updating the Status
*
* PARAMETER: None
*
* RETURNVALUE: BDAddressID
*
*******************************************************************/
tU8 fc_phone_tclService_Telephone::u8GetBDAddressIDforStatusUpdate()
{
   ETG_TRACE_USR4(("u8GetBDAddressIDforStatusUpdate entered and m_u8CurrentBDAddressIDforUpdate: %d", m_u8CurrentBDAddressIDforUpdate));
   tU8 u8BDAddressIDforUpdate = INVALID_BDADDRESS_ID;
   if(INVALID_BDADDRESS_ID == m_u8CurrentBDAddressIDforUpdate)
   {
      tU8 u8ActiveBDAddressId = INVALID_BDADDRESS_ID;
      tBool bGetActiveDeviceAddress = b_get_ActiveDevice_AddressId_FromBT(&u8ActiveBDAddressId);
      if (FALSE == bGetActiveDeviceAddress)
      {
         u8BDAddressIDforUpdate = 0;
      }
      else
      {
         u8BDAddressIDforUpdate = u8ActiveBDAddressId;
      }

      m_u8CurrentBDAddressIDforUpdate = u8BDAddressIDforUpdate;
   }
   else
   {
      u8BDAddressIDforUpdate = m_u8CurrentBDAddressIDforUpdate;
   }
   ETG_TRACE_USR4(("u8BDAddressIDforUpdate: %d", u8BDAddressIDforUpdate));
   return u8BDAddressIDforUpdate;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSendSLCOFF
*
* DESCRIPTION: Manipulates m_u8CurrentBDAddressIDforUpdate and
*              returns the apt BDAddressID for updating the Status
*
* PARAMETER: BDAddressID, DeviceHandle
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSendSLCOFF(tU8 u8BDId, tU8 u8DeviceHandle)
{
   ETG_TRACE_USR3((" fc_phone_tclService_Telephone::vSendSLCOFF with u8BDId :; %d and u8DeviceHandle :: %d", u8BDId, u8DeviceHandle));
   fc_phone_tclClientHandler_PhoneBook *p_clientHndler_phonebook = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerPhoneBook;

   m_u16PnpInfoVendorID[u8BDId] = 0;
   m_u16PnpInfoVendorSource[u8BDId] = 0;

   tChar pBDAddress[PHONE_BDADRESS_LENGTH] = {0};
   vStringCopy(pBDAddress, fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(u8BDId), PHONE_BDADRESS_LENGTH);
   ETG_TRACE_USR3(("pBDAddress :: %s", pBDAddress));

   //clear phone status of disconnecting device
#ifdef PHONE_SIRI

   fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
   if(p_clientHndler_bluetooth && p_clientHndler_bluetooth->bGetAppleDeviceStatusfromBDAddressID(u8BDId))
   {
      m_oSiriData[u8BDId].vDisableSiri();
   }

   m_u8SiriReqRetryCount[u8BDId] = 0;
   //For both the below timers, if one of the device timer is running then we should not stop the timer.
   if(m_bSiriRetryTimerRunning[u8BDId])
   {
      m_bSiriRetryTimerRunning[u8BDId] = FALSE;
      tBool bIsSiriRetryTimerRunning = false;
      for(tU8 ind = 0; ind < FC_PHONE_MAX_CONNECTED_DEVICE; ++ind)
      {
         if(m_bSiriRetryTimerRunning[ind])
         {
            bIsSiriRetryTimerRunning = true;
            break;
         }
      }
      if(!bIsSiriRetryTimerRunning)
      {
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_SIRISTATUS_RETRY_TIMER_ID);
      }

   }
   if(m_bSiriNrRetryTimerRunning[u8BDId])
   {
      m_bSiriNrRetryTimerRunning[u8BDId] = FALSE;

      tBool bIsSiriNRRetryTimerRunning = false;
      for(tU8 ind = 0; ind < FC_PHONE_MAX_CONNECTED_DEVICE; ++ind)
      {
         if(m_bSiriNrRetryTimerRunning[ind])
         {
            bIsSiriNRRetryTimerRunning = true;
            break;
         }
      }
      if(!bIsSiriNRRetryTimerRunning)
      {
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_SIRINRSTATUS_RETRY_TIMER_ID);
      }
   }
#endif //PHONE_SIRI

   //Update the IncomingCallEvent idle to the clients on device disconnection if there was any ACTIVE INCOMINGCALL event update earlier
   // And this has to be done before updating the Call Status List vector
   for(tU8 i=0; i<CallStatusList.size(); ++i)
   {
      if (CallStatusList[i].DeviceHandle == u8DeviceHandle)
      {
         ETG_TRACE_USR3(("Device Handle is present at index= %d and Device Address= %s ", i, CallStatusList[i].BD_Address));

         for(tU8 ind =0; ind < FC_PHONE_MAX_CALLINSTANCES; ++ind)
         {
            vResetCallStatusinCallStatusList(static_cast<tU8>(ind+i));
         }
         break;
      }
   }

   if(u8DeviceHandle == m_u8ActiveDeviceHandle)
   {
      // NCG3D-3221- Hold for handset-handsfree mode
      if(m_bScoChannelReleaseTimerRunning)
      {
         (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(FC_PHONE_SCO_CHANNEL_RELEASE_TIMER_ID);
         m_bScoChannelReleaseTimerRunning = FALSE;
         ETG_TRACE_USR3(("m_bScoChannelReleaseTimerRunning = FALSE"));
      }
      //start
      for(tU8 u8Index = 0; u8Index < FC_PHONE_MAX_CALLINSTANCES; ++u8Index)
      {
         if(TRUE == (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32GetAliveTimerId(static_cast<tU16>(FC_PHONE_DIAL_CALLSTATUS_FAILURE_TIMER_ID + m_oCallInstances[u8Index].m_u16Id)))
         {
            (fc_phone_tclApp::m_poMainAppInstance)->m_poTimer->s32Stop(static_cast<tU16>(FC_PHONE_DIAL_CALLSTATUS_FAILURE_TIMER_ID + m_oCallInstances[u8Index].m_u16Id));
         }
      }
      //end

      //The dbus proxies for this device should have been deleted and
      //the dbus signal handlers should have been disconnected in the glib
      //main loop thread.

      //If SLC OFF : release audio channels if holding and come to idle state
      //GMMY15-7789
      NULL_CHECK(m_poSMManager)
      m_poSMManager->vProcessOnDynamicDisconnect(FC_PHONE_SM_EVENT_SLC_OFF,u8BDId);

#ifdef PHONE_WAITINGMODE
      //Post SLC off event to Waiting mode handler
      fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
      poWaitingMode->bProcessEvent(FC_PHONE_WAITING_MODE_EVENT_SLC_OFF);
#endif

      //GMMY15-3655:Not Able To Delete The Device From BT DeviceList After Make An Outgoing Call And Then Disconnect The Phone From The Target
      //ETG_TRACE_USR4((" Reset the StateMachine and call the CallStatus when device is disconnected and deleted from the target. "));
      //vResetSMManager(TRUE);

      //Fix for PSARCC30-2846. HfReleaseAudio was missing. And hence the SPM sub state update was missing from FC_Phone.
      vUpdateSpmSubState();
   }
   else
   {
      tU8 IncomingCallInstance = p_clientHndler_phonebook->u8GetIncomingCallInstancefromPBList(u8DeviceHandle);
      if(0xFF == IncomingCallInstance)
      {
         ETG_TRACE_USR4(("No need to send IDLE IncomingCallEvent"));
      }
      else
      {
         vSendIncomingCallEvent(IncomingCallInstance, false);
      }

      if (u8DeviceHandle == m_u8ActiveDeviceHandletobeRetained)
      {
         m_u8ActiveDeviceHandletobeRetained = FC_PHONE_DEVICEHANDLE_UNKNOWN;
      }
   }
   m_oPhoneData[u8BDId].vInit();
   vUpdatePhoneStatusSLCoff(u8BDId);

#ifdef PHONE_HFENHANCED
   /*TODO: HFENHANCED feature related Most-fi is not available
   because of old version of di_most used in JAC project
   Once avaialble, it has to be removed. */

   m_u8CurrentBDAddressIDforUpdate = u8BDId;
   ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_HANDSFREEENHANCEDCALLCONTROL);


   if (AIL_EN_N_NO_ERROR != enResult)
   {
      /* Throw up error */
      ETG_TRACE_ERR(( "vSendEnhancedCallControlStatus: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
   }
   m_u8CurrentBDAddressIDforUpdate = INVALID_BDADDRESS_ID;
#endif /* PHONE_HFENHANCED */

   vHandleWaitingMethodsOnDisconnection(pBDAddress);

   vDeleteCSfromCallStatusList(pBDAddress);
   if (u8DeviceHandle == m_u8ActiveDeviceHandle)
   {
      if (0 == CallStatusList.size())
      {
         vSetasActiveDevice(FC_PHONE_DEVICEHANDLE_UNKNOWN);
      }

      // For 2- HFP
      else
      {
         vSwitchActivePassive(CallStatusList[0].DeviceHandle);
      }
   }

   vSendPropertyUpdate(MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED, NULLPTR);

   /*Fix for PSARCCB-8471. u8DeleteDeviceFromDeviceList() is called only on receiving the DeviceListStatus from FC_Bluetooth.
    *And not on receiving the SLC OFF(_DisconnectionEvent_handler for HFP).
    */
   //Fix for GMMY15-7789
   fc_phone_tclDeviceList::getInstance()->u8DeleteDeviceFromDeviceList(u8BDId);
   vSetSLCStatus(false);

   p_clientHndler_phonebook->vRemovePBListItem(u8DeviceHandle);
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vPrintCallStatusListVector
*
* DESCRIPTION: Prints the CallStatusListVector
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vPrintCallStatusListVector()
{
   ETG_TRACE_USR3((" fc_phone_tclService_Telephone::vPrintCallStatusListVector "));
   for(tU8 i =0; i<CallStatusList.size(); ++i)
   {
      ETG_TRACE_USR4(("CallStatusList[%d].DeviceHandle = %d", i, CallStatusList[i].DeviceHandle));
      ETG_TRACE_USR4(("CallStatusList[%d].BD_Address: %s   ", i, CallStatusList[i].BD_Address));

      ETG_TRACE_USR4(("CallStatusList[%d].CallHandle: %d   ", i, CallStatusList[i].CallStatus.u8CallHandle));
      ETG_TRACE_USR4(("CallStatusList[%d].CallTotal: %d   ", i, CallStatusList[i].CallStatus.u8CallTotal));
      ETG_TRACE_USR4(("CallStatusList[%d].CallDirection: %u", i, CallStatusList[i].CallStatus.u8CallDirection));
      ETG_TRACE_USR4(("CallStatusList[%d].CallState: %u", i, CallStatusList[i].CallStatus.u8CallState));
      ETG_TRACE_USR4(("CallStatusList[%d].CallMode: %u", i, CallStatusList[i].CallStatus.u8CallMode));
      ETG_TRACE_USR4(("CallStatusList[%d].CallNumber: %s   ", i, CallStatusList[i].CallStatus.achCallNumber));
      ETG_TRACE_USR4(("CallStatusList[%d].CallType: %u", i, CallStatusList[i].CallStatus.u8CallType));
   }
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vResetIncomingEventInfoVariable
*
* DESCRIPTION:
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vResetIncomingEventInfoVariable()
{
   ETG_TRACE_USR3((" fc_phone_tclService_Telephone::vResetIncomingEventInfoVariable "));
   m_sIncomingEventInfo.CallInstance = 0xFF;
   m_sIncomingEventInfo.CallWait = 0;
   m_sIncomingEventInfo.DeviceHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN;

   if (m_sIncomingEventInfo.FirstName)
   {
      delete []m_sIncomingEventInfo.FirstName;
      m_sIncomingEventInfo.FirstName = NULLPTR;
   }
   if (m_sIncomingEventInfo.LastName)
   {
      delete []m_sIncomingEventInfo.LastName;
      m_sIncomingEventInfo.LastName = NULLPTR;
   }

   memset(m_sIncomingEventInfo.TelephoneNumber, 0, PHONE_CALLNUMBER_LENGTH);
   m_sIncomingEventInfo.bIsRinging = 0;
}

/******************************************************************
*
* FUNCTION: fc_phone_tclService_Telephone::vSendIncomingCallEvent
*
* DESCRIPTION:
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************/
tVoid fc_phone_tclService_Telephone::vSendIncomingCallEvent(tU8 CallInstance, tBool bIsRinging)
{
   ETG_TRACE_USR3((" fc_phone_tclService_Telephone::vSendIncomingCallEvent entered with CallInstance: %d and bIsRinging: %d", CallInstance, bIsRinging));
   if(CallInstance >= CallStatusList.size())
   {
      ETG_TRACE_USR3((" Not a valid CallInstance "));
      return;
   }

   m_sIncomingEventInfo.CallInstance = CallInstance;
   m_sIncomingEventInfo.CallWait = 0;
   m_sIncomingEventInfo.DeviceHandle = CallStatusList[CallInstance].DeviceHandle;
   m_sIncomingEventInfo.bIsRinging = bIsRinging;

   if (m_sIncomingEventInfo.FirstName)
   {
      delete []m_sIncomingEventInfo.FirstName;
      m_sIncomingEventInfo.FirstName = NULLPTR;
   }
   if (m_sIncomingEventInfo.LastName)
   {
      delete []m_sIncomingEventInfo.LastName;
      m_sIncomingEventInfo.LastName = NULLPTR;
   }

   memset(m_sIncomingEventInfo.TelephoneNumber, 0, PHONE_CALLNUMBER_LENGTH);

   if (bIsRinging)
   {
      fc_phone_tclClientHandler_PhoneBook *p_clientHndler_phonebook = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerPhoneBook;
      p_clientHndler_phonebook->vGetFirstName(m_sIncomingEventInfo.FirstName, CallStatusList[CallInstance].DeviceHandle);
      p_clientHndler_phonebook->vGetLastName(m_sIncomingEventInfo.LastName, CallStatusList[CallInstance].DeviceHandle);
      vStringCopy(m_sIncomingEventInfo.TelephoneNumber, CallStatusList[CallInstance].CallStatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);

      if (FC_PHONE_CALLINSTANCE_CALLSTATUS_BUSY == CallStatusList[CallInstance].CallStatus.u8CallState)
         m_sIncomingEventInfo.CallWait = 1;
   }

   ail_tenCommunicationError enResult = eUpdateClients(MOST_TELFI_C_U16_INCOMINGCALLEVENT);

   if (AIL_EN_N_NO_ERROR != enResult)
   {
      ETG_TRACE_ERR(("Update with %d; has Failed ", MOST_TELFI_C_U16_INCOMINGCALLEVENT));
   }

   // Fix for NCG3D-40884
   // After sending IDLE INCOMING CALL event, the Call instance parameter should be made 0xFF.
   if (!bIsRinging)
   {
      vResetIncomingEventInfoVariable();
   }
}

/*******************************************************************************
** FUNCTION:   vUpdateCurrentCallIndexAndProperty(..)
*******************************************************************************/
/* DESCRIPTION:
*   EXported  method to send property updates
*  PARAMETERS:
*     IN:   u16PropertyId :  PropertyId whose update needs to be sent.
*     IN:   rfpCallInstance: Reference to current callinstance
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vUpdateCurrentCallIndexAndProperty(tU16 u16PropertyId, tVoid* varg)
{
   ETG_TRACE_USR4((" fc_phone_tclService_Telephone::vUpdateCurrentCallIndexAndProperty ENTERED with u16PropertyId:%d",u16PropertyId));

   fc_phone_tclCallInstance*& rfpCallInstance = (fc_phone_tclCallInstance*&) varg;

   m_u8CurrentCallIndex = 0;
   for (tU8 iCount = 0; iCount < FC_PHONE_MAX_CALLINSTANCES; iCount++)
   {
      if (&m_oCallInstances[iCount] == rfpCallInstance)
      {
         m_u8CurrentCallIndex = iCount;
         break;
      }
   }

   if( MOST_TELFI_C_U16_INCOMINGCALLEVENT != u16PropertyId )
   {
      ail_tenCommunicationError enResult = eUpdateClients(u16PropertyId);
      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("Update with %d; has Failed ", u16PropertyId));
      }
   }
}

/*******************************************************************************
** FUNCTION:   vHandleCallStatusofPassiveDevice(..)
*******************************************************************************/
/* DESCRIPTION:
*   EXported  method to send property updates
*  PARAMETERS:
*     IN:   prCallData :  Call status received
*     IN:   devHandle: Device handle of the Passive device
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vHandleCallStatusofPassiveDevice(T_CallData* prCallData, tU8 devicehandle)
{
   ETG_TRACE_USR4((" vHandleCallStatusofPassiveDevice ENTERED with devicehandle :%d",devicehandle));

   ETG_TRACE_USR3(("DBUS CallStatus: prCallData->rCallStatus.u8CallState: %d", prCallData->rCallStatus.u8CallState));
   ETG_TRACE_USR3(("prCallData->rCallStatus.achCallNumber: %s", prCallData->rCallStatus.achCallNumber));

   if((PHONE_CALLSTATE_INCOMING == prCallData->rCallStatus.u8CallState) || (PHONE_CALLSTATE_WAITING == prCallData->rCallStatus.u8CallState))
   {
      (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerPhoneBook->vSearchPhoneBook(devicehandle, prCallData->rCallStatus.achCallNumber, 0);
   }
   else if((PHONE_CALLSTATE_IDLE == prCallData->rCallStatus.u8CallState) || (PHONE_CALLSTATE_ACTIVE == prCallData->rCallStatus.u8CallState) \
         || (PHONE_CALLSTATE_HELD == prCallData->rCallStatus.u8CallState))
   {
      fc_phone_tclClientHandler_PhoneBook *p_clientHndler_phonebook = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerPhoneBook;
      tU8 IncomingCallInstance = p_clientHndler_phonebook->u8GetIncomingCallInstancefromPBList(devicehandle);
      if(0xFF == IncomingCallInstance)
      {
         ETG_TRACE_USR4(("No need to send IDLE IncomingCallEvent"));
      }
      else
      {
         vSendIncomingCallEvent(IncomingCallInstance, false);
         p_clientHndler_phonebook->vSetIncomingCallInstanceinPBList(devicehandle, 0xFF);
      }
   }
}

/*******************************************************************************
** FUNCTION:   vAcknowledgeMethodStartForCallInstance(..)
*******************************************************************************/
/* DESCRIPTION:
*   Send Method result/error for Hangup/reject/ignore calls
*  PARAMETERS:
*     IN:   varg :  ProcessEvent details for the call instance.
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phone_tclService_Telephone::vAcknowledgeMethodStartForCallInstance(tVoid *varg)
{
   (tVoid)varg ;
   FcPhone_processEventArg   *ptProcessEventArg = (FcPhone_processEventArg *) varg;
   ETG_TRACE_USR3(("vAcknowledgeMethodStartForCallInstance:: m_bEndCallforPassiveDev : %d",m_bEndCallforPassiveDev));
   if(m_bEndCallforPassiveDev == false)
   {
      if( ptProcessEventArg->u16DbusAckType == FC_PHONE_CB_FID_REJECTCALL)
      {
         m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_REJECTCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST, ptProcessEventArg);
      }
      else
      {
         m_poSMManager->vProcessEvent(FC_PHONE_SM_EVENT_DBUS_HANGUPCALL_ACK_OK, FC_PHONE_EVENTTYPE_BROADCAST, ptProcessEventArg);
      }
   }
   else
   {
      ETG_TRACE_USR3(("m_bEndCallforPassiveDev == true"));
      if(FC_PHONE_DBUS_STATUS_SUCCESS != ptProcessEventArg->Status)
      {
         ETG_TRACE_USR4((" DBUS Call has failed, hence sending ME"));
         most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

         switch(ptProcessEventArg->Code)
         {
            case FC_PHONE_DBUS_ERRORCODE_STATE_ERROR:
            {
               e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_BUSY;
            }
            break;
            case FC_PHONE_DBUS_ERRORCODE_OPERATION_NOT_ALLOWED:
            case FC_PHONE_DBUS_ERRORCODE_OPERATION_NOT_SUPPORTED:
            {
               e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
            }
            break;
            default:
            {
               ETG_TRACE_USR4((" IN DEFAULT This Error Code [0x%x] is not supported yet",ptProcessEventArg->Code));
               e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
            }
            break;
         }
         if(FC_PHONE_DEFAULT_STORED_PARAM != m_u16StoredCallInstanceForIgnore)
            vHandle_IgnoreCall_Error(e8ErrorCode);
         else if(FC_PHONE_DEFAULT_STORED_PARAM != m_u16StoredCallInstanceForHangUp)
            vHandle_HangUpCall_Error(e8ErrorCode);
         else if(FC_PHONE_DEFAULT_STORED_PARAM != m_u16StoredCallInstanceForReject)
            vHandle_RejectCall_Error(e8ErrorCode);
      }
      else
      {
         ETG_TRACE_USR4(("DBUS Call is succeeded, hence sending MR"));

         fc_phone_tclCallInstance *HangUpCallInstance = new(std::nothrow) fc_phone_tclCallInstance;

         if(HangUpCallInstance)
         {
            HangUpCallInstance->m_AckError = FC_PHONE_CALLINSTANCE_ACKERROR_OK;
            HangUpCallInstance->m_u16CallStatusNotice = FC_PHONE_CALLINSTANCE_CALLSTATUS_DISCONNECTING;
            HangUpCallInstance->m_bUsingVehicleAudio = false;
            if(FC_PHONE_DEFAULT_STORED_PARAM != m_u16StoredCallInstanceForHangUp)
            {
               ETG_TRACE_USR4(("m_u16StoredCallInstanceForHangUp: %d", m_u16StoredCallInstanceForHangUp));
               HangUpCallInstance->m_u16Id = static_cast<tU16>(m_u16StoredCallInstanceForHangUp);
               vSendMethodAck(MOST_TELFI_C_U16_HANGUPCALL, HangUpCallInstance);
               fc_phone_tclPhoneMutex::PhoneMutexUnlock();
            }
            else if(FC_PHONE_DEFAULT_STORED_PARAM != m_u16StoredCallInstanceForIgnore)
            {
               ETG_TRACE_USR4(("m_u16StoredCallInstanceForIgnore: %d", m_u16StoredCallInstanceForIgnore));
               HangUpCallInstance->m_u16Id = static_cast<tU16>(m_u16StoredCallInstanceForIgnore);
               vSendMethodAck(MOST_TELFI_C_U16_IGNORECALL, HangUpCallInstance);
            }
            else if(FC_PHONE_DEFAULT_STORED_PARAM != m_u16StoredCallInstanceForReject)
            {
               ETG_TRACE_USR4(("m_u16StoredCallInstanceForReject: %d", m_u16StoredCallInstanceForReject));
               HangUpCallInstance->m_u16Id = static_cast<tU16>(m_u16StoredCallInstanceForReject);
               vSendMethodAck(MOST_TELFI_C_U16_REJECTCALL, HangUpCallInstance);
            }
            delete HangUpCallInstance;
         }
         else
         {
            // TBD: ME or MR-> ??
            ETG_TRACE_USR4(("HangUpCallInstance is NULL"));
         }
      }
      m_bEndCallforPassiveDev = false;
   }
}

/*******************************************************************************
** FUNCTION:   u16GetCallStateFromCallInstID(..)
*******************************************************************************/
/* DESCRIPTION:
*   EXported  method to send property updates
*  PARAMETERS:
*     IN:   u16CallInstID_in :  Call instance ID
*
*  RETURNVALUE:
*     OUT:   u8CallState : Current Call state of the corresponding Call instance ID
********************************************************************************/
tU16 fc_phone_tclService_Telephone::u16GetCallStateFromCallInstID(tU16 u16CallInstID_in)
{
   ETG_TRACE_USR4((" u16GetCallStateFromCallInstID ENTERED with u16CallInstID_in :%d", u16CallInstID_in));

   tU16 CallInstanceIdIter = 0;
   tU16 u16CallState = FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE;

   ETG_TRACE_USR1(("m_u8ActiveDeviceHandle :%d", m_u8ActiveDeviceHandle));
   vPrintCallStatusListVector();

   for(; CallInstanceIdIter < CallStatusList.size(); CallInstanceIdIter = static_cast<tU16>(CallInstanceIdIter+2))
   {
      if(CallStatusList[CallInstanceIdIter].DeviceHandle == m_u8ActiveDeviceHandle)
      {
         u16CallState = CallStatusList[CallInstanceIdIter+u16CallInstID_in].CallStatus.u8CallState;
         break;
      }
   }

   ETG_TRACE_USR1(("CallInstanceIdIter :%d", CallInstanceIdIter));
   ETG_TRACE_USR3(("u16CallState: %u", ETG_ENUM(TR_CALLSTATUS,u16CallState)));

   return u16CallState;
}

/*******************************************************************************
** FUNCTION:   bIsActiveDeviceIdle(..)
*******************************************************************************/
/* DESCRIPTION:
*   This function is used to know whether there are any calls or VR session is triggered in the Active device
*  PARAMETERS:
*  RETURNVALUE:
*     OUT:     TRUE  - No calls in Active device and VR session is IDLE
*              FALSE - There is a call in the Active device or VR session is not IDLE
********************************************************************************/
tBool fc_phone_tclService_Telephone::bIsActiveDeviceIdle()
{
   ETG_TRACE_USR4((" bIsActiveDeviceIdle ENTERED "));

   tBool bIsCallStatusIdle = bCheckIdleCallStatus();
   ETG_TRACE_USR1(("bIsCallStatusIdle :%d", bIsCallStatusIdle));

   if(((EN_SPEECH_CHNL_NOTACTIVE == (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->u8SpeechChannelStatus())) \
         && (TRUE == bIsCallStatusIdle))
   {
      return TRUE;
   }
   return FALSE;
}

// Fix for NCG3D-103060
/*******************************************************************************
 *
 * FUNCTION:    bIsExternalTriggerReceivedForPhoneChannel
 *
 * DESCRIPTION: Used to get the External trigger status from ARL
 *
 * PARAMETER:
 *
 * RETURNVALUE:   TRUE  - EXT trigger was received
 *                FALSE - No EXT trigger was received
 *
 *******************************************************************************/
tBool fc_phone_tclService_Telephone::bIsExternalTriggerReceivedForPhoneChannel()
{
   ETG_TRACE_USR4(("bIsExternalTriggerReceivedForPhoneChannel called"));
   ETG_TRACE_USR4(("m_bExternalTriggerStatusforPhoneChannel : %u", m_bExternalTriggerStatusforPhoneChannel));
   return m_bExternalTriggerStatusforPhoneChannel;
}

/*******************************************************************************
 *
 * FUNCTION:    bIsFirstIncomingCallPresent
 *
 * DESCRIPTION: Used to get the Incoming call status
 *
 * PARAMETER:
 *
 * RETURNVALUE:   TRUE  - First incoming call, else FALSE
 *
 *******************************************************************************/
tBool fc_phone_tclService_Telephone::bIsFirstIncomingCallPresent()
{
   ETG_TRACE_USR4(("bIsFirstIncomingCallPresent called"));

   if ((FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE == m_oCallInstances[0].m_u16CallStatusNotice) || (FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE == m_oCallInstances[1].m_u16CallStatusNotice))
   {
      if (fc_phone_tclCallManager::m_u16Count == 1)
         return true;
   }

   return false;
}

/*******************************************************************************
 *
 * FUNCTION:    bIsTimerDurationExtendedForWaitingMode
 *
 * DESCRIPTION: Used to confirm and set the timer to hold the SCO channel for Waiting mode.
 *              This is because, in IPhone V11.4, there is some extra(~2secs) delay in SCO disconnection/connection.
 *
 * PARAMETER:
 *
 * RETURNVALUE:   True - If Timer is extended ; False - if default timer is set.
 *
 *******************************************************************************/
tBool fc_phone_tclService_Telephone::bIsTimerDurationExtendedForWaitingMode()
{
   ETG_TRACE_USR4(("Entered bIsTimerDurationExtendedForWaitingMode"));
#ifdef PHONE_WAITINGMODE
   fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
   tenFC_Phone_WaitingModeState m_enWaitingModeState = poWaitingMode->enGetWaitingModeState();
   tU16 nonIdleCallInstance = 0;
   if(FC_PHONE_CALLINSTANCE_CALLSTATUS_IDLE == m_oCallInstances[0].m_u16CallStatusNotice)
   {
      nonIdleCallInstance = 1;
   }

   ETG_TRACE_USR4(("m_enWaitingModeState : %d callstatus = %d",m_enWaitingModeState,m_oCallInstances[nonIdleCallInstance].m_u16CallStatusNotice));

   if((FC_PHONE_WAITINGMODE_ACTIVE == m_enWaitingModeState) || (FC_PHONE_WAITINGMODE_START_IN_PROGRESS == m_enWaitingModeState))
   {
      if((FC_PHONE_CALLINSTANCE_CALLSTATUS_RINGTONE == m_oCallInstances[nonIdleCallInstance].m_u16CallStatusNotice) || FC_PHONE_CALLINSTANCE_CALLSTATUS_ACTIVE == m_oCallInstances[nonIdleCallInstance].m_u16CallStatusNotice)
      {
         ETG_TRACE_USR4(("5secs timer will be set"));
         return true;
      }
   }
#endif
   ETG_TRACE_USR4(("500ms timer will be set"));
   return false;
}
/*******************************************************************************
 *
 * FUNCTION:    convert_raw_btaddr_to_formatted
 *
 * DESCRIPTION: Used to confirm and set the timer to hold the SCO channel for Waiting mode.
 *              This is because, in IPhone V11.4, there is some extra(~2secs) delay in SCO disconnection/connection.
 *
 * PARAMETER:
 *
 * RETURNVALUE:   True - If Timer is extended ; False - if default timer is set.
 *
 *******************************************************************************/
static void convert_raw_btaddr_to_formatted(char *node_name,const char *BTSetting_addr,
      tU32 dest_size)
{

   ETG_TRACE_USR4(("convert_raw_btaddr_to_formatted entered "));
   if(BTSetting_addr)
   {
      ETG_TRACE_USR4(("BTSetting_addr = %s",BTSetting_addr));
      if(strstr(BTSetting_addr,"dev_"))
      {
         vStringCopy(node_name, BTSetting_addr, dest_size);
      }
      else
      {
         tChar tempBD_Address[PHONE_TEMP_DEV_ADDRESS] = {'\0'};

         sscanf(BTSetting_addr, "%02x%02x%02x%02x%02x%02x",
               (tUInt *) &tempBD_Address[0], (tUInt *) &tempBD_Address[1],
               (tUInt *) &tempBD_Address[2], (tUInt *) &tempBD_Address[3],
               (tUInt *) &tempBD_Address[4], (tUInt *) &tempBD_Address[5]);

         snprintf(node_name, dest_size, "dev_%02x_%02x_%02x_%02x_%02x_%02x",
               (unsigned char)tempBD_Address[0],
               (unsigned char)tempBD_Address[1],
               (unsigned char)tempBD_Address[2],
               (unsigned char)tempBD_Address[3],
               (unsigned char)tempBD_Address[4],
               (unsigned char)tempBD_Address[5]
         );
      }
      ETG_TRACE_USR4(("node_name = %s",node_name));
   }
}


void convert_formatted_btaddr_to_raw(const char* BTSetting_addr, char* node_name, tU32 dest_size)
{
   ETG_TRACE_USR4(("convert_formatted_btaddr_to_raw: Active Device Address entry :%s", BTSetting_addr));
   char addresstoken[PHONE_TEMP_DEV_ADDRESS];
   const char * deviceAddressWithUnderscore;
   std::string tempOutput;

   char acTempActiveDeviceBD_Address[PHONE_FORMAT_BDADDRESS];

   deviceAddressWithUnderscore = strstr(BTSetting_addr,"dev_");

   if(deviceAddressWithUnderscore)
   {
      vStringCopy( addresstoken, deviceAddressWithUnderscore+4, sizeof(addresstoken) );
      deviceAddressWithUnderscore = strtok (addresstoken ,"_");

      while( deviceAddressWithUnderscore)
      {
         tempOutput.append(deviceAddressWithUnderscore);
         deviceAddressWithUnderscore = strtok (NULLPTR, "_");
      }
      vStringCopy(node_name,tempOutput.c_str(),PHONE_TEMP_DEV_ADDRESS);

      ETG_TRACE_USR4((" sscanf: Entry Remote device address :%s", node_name));

      sscanf((const char*)node_name, "%02X%02X%02X%02X%02X%02X",
            (unsigned int *) &acTempActiveDeviceBD_Address[0], (unsigned int *) &acTempActiveDeviceBD_Address[1],
            (unsigned int *) &acTempActiveDeviceBD_Address[2], (unsigned int *) &acTempActiveDeviceBD_Address[3],
            (unsigned int *) &acTempActiveDeviceBD_Address[4], (unsigned int *) &acTempActiveDeviceBD_Address[5]);
      acTempActiveDeviceBD_Address[6] = 0;

      ETG_TRACE_USR4((" sprintf: Entry Remote device address :%s", node_name));

      snprintf(node_name, dest_size, "%02X%02X%02X%02X%02X%02X",
            (unsigned char)acTempActiveDeviceBD_Address[0], (unsigned char)acTempActiveDeviceBD_Address[1], (unsigned char)acTempActiveDeviceBD_Address[2],
            (unsigned char)acTempActiveDeviceBD_Address[3], (unsigned char)acTempActiveDeviceBD_Address[4], (unsigned char)acTempActiveDeviceBD_Address[5]);

      ETG_TRACE_USR4((" Active Device Address after conversion node_name :%s", node_name));
   }
   else
   {
      vStringCopy(node_name, BTSetting_addr, dest_size);
   }
}
