/*******************************************************************************
 *
 * FILE:          FC_Phone_DBusClientEvo.h
 *
 * SW-COMPONENT:
 *
 * PROJECT:
 *
 * DESCRIPTION:   FC_Phone Evolution DBUS client
 *
 * AUTHOR:        Elumalai Narasimman
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *  
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "FC_Phone_DBusClientEvo.h"
#include "FC_Phone_DbusClient_bindings_evolution_handsfree.h"
#include "FC_Phone_DbusClient_bindings_evolution_manager.h"
#include "FC_Phone_DbusClient_bindings_evolution_modem.h"
#include "FC_Phone_DbusClient_bindings_evolution_siri.h"
#include "FC_Phone_DbusClient_bindings_evolution_networkRegistration.h"
#include "FC_Phone_DbusClient_bindings_evolution_networkOperator.h"
#include "FC_Phone_DbusClient_bindings_evolution_callvolume.h"
#include "FC_Phone_DbusClient_bindings_evolution_voicecall.h"
#include "FC_Phone_DbusClient_bindings_evolution_voicecallmanager.h"
#include "FC_Phone_DbusClient_bindings_object_manager.h"

#include "../FC_Phone_main.h"
#include "../HelperClasses/FC_Phone_PhoneData.h"
#include "../HelperClasses/FC_Phone_Timer.h"
#include "../Interface/FC_Phone_DBusInterface.h"
#include "../Defines/TelephoneService_Defines.h"
#include "../StateMachines/FC_Phone_SMEvents.h"

#ifdef PHONE_WAITINGMODE
#include "../WaitingMode/FC_Phone_WaitingMode.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
#include "trcGenProj/Header/FC_Phone_DBusClientEvo.cpp.trc.h"
#endif

#define NARROW_BAND_SPEECH 1
#define WIDE_BAND_SPEECH 2

static GDBusConnection* gdbus = NULLPTR;
static char* tempDeviceAddress = NULLPTR;
const char* vehicleBtAddress = NULLPTR;
const char* gnetworkOperatorPath = NULLPTR;

std::vector <const gchar*> featureList;
std::vector <const gchar*> interfaceList;
std::vector <const gchar*> emergencyNumberList;
std::vector <const gchar*> VoiceCallPathList;
std::vector <const gchar*> ChldFeatureList;

static OrgOfonoManager* l_ProxyModemManager = NULLPTR;

static int FC_Phone_Diagnostic_flag_evo;
static T_CallData EvoCallbackdata;
static T_CallData EvoCallbackdataArray[FC_PHONE_MAX_CALLINSTANCES];

//static T_PhoneStatus PhoneStatus_data;
T_SCOConnectionStatus SCOConnectionStatus_data_evo;
T_VoiceRecInd VoiceRec_data_evo;
_T_CCA_Dbus_If *DCCA_callbacks_evo;
gchar setDTMFChar = '\0';

#ifdef PHONE_SIRI
T_SiriNRStatusInd SiriNRStatus_data_evo;
T_SiriStatus SiriStatus_value_evo;
#endif

//Connected devices Table
ConnMapTableSTruct ConnMapTableST[FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT];

//Modem Proxy table.
std::vector<ModemProxyTableSTruct> ModemProxyTable;

//function Declarations
static void FB_Evodbus_init(void);
static gboolean EvoCreateProxyAndRegisterSignals(const char* DevAddress);
static gboolean EvoDeleteProxyAndDeRegisterSignals(const char* DevAddress);
static gboolean EvoAcceptCall(const char* DevAddress, FCPhoneDbusClient_AcceptType_t enAcceptType);
static gboolean EvoHoldCall(const char* DevAddress);
static gboolean EvoHangupCall(const char* DevAddress, FCPhoneDbusClient_TerminateType_t enTerminateType);
static gboolean EvoRejectCall(const char* DevAddress, FCPhoneDbusClient_IgnoreType_t enIgnoreType);
static gboolean EvoCellDeviceName(const char* DevAddress);
static gboolean EvoDial(const char* DevAddress,const char* u8_phone_number);
static gboolean EvoTerminateCall(const char* DevAddress,FCPhoneDbusClient_TerminateType_t enTerminateType);
static void EvoDiagnosticModeConfig(int eComponent,int bool_value);
static gboolean EvoAudioTransfer(const char* DevAddress, gboolean direction);
static gboolean EvoQueryTelephoneNumber(const char* DevAddress);
static gboolean EvoGetPhoneStatus(const char* DevAddress);
static gboolean EvoSendDTMF(const char* DevAddress,guchar DTMFChar);
static gboolean EvoSpeedDial(const char* DevAddress,guint mem_index);
static gboolean EvoResumeHeldCall(const char* DevAddress);
static gboolean EvoHoldIncomingCall(const char* DevAddress);
static gboolean EvoSwapCall(const char* DevAddress);
static gboolean EvoConfCall(const char* DevAddress);
static gboolean EvoSplitCall(const char* DevAddress, const guchar index);
static gboolean EvoReleaseSpecCall(const char* DevAddress,FCPhoneDbusClient_TerminateType_t enTerminateType, const guchar index);
static gboolean EvoGetCallStatus(const char* DevAddress);
static gboolean EvoGetSCOConnectionStatus(const char* DevAddress);
static gboolean EvoActivateVoiceRecognition(const char* DevAddress, const guchar activationIndex);
static gboolean EvoAcquireTelephoneNumber(const char* DevAddress);
#ifdef PHONE_SIRI
static gboolean EvoEnableSiri(const char* DevAddress, VehicleInformation DevInfo);
static gboolean EvoSetSiriNrStatus(const char* DevAddress, const guchar activationIndex);
static gboolean EvoSetSiriEfm(const char* DevAddress, const guchar activationIndex);
#endif

//Dbus functions linking with CCA
static _CCA_dbus_interfaces CCADbusEvoInterfaces =
      {
            FB_Evodbus_init,
            EvoCreateProxyAndRegisterSignals,
            EvoDeleteProxyAndDeRegisterSignals,
            EvoAcceptCall,
            EvoHoldCall,
            EvoHangupCall,
            EvoRejectCall,
            EvoCellDeviceName,
            EvoDial,
            EvoTerminateCall,
            EvoDiagnosticModeConfig,
            EvoAudioTransfer,
            EvoQueryTelephoneNumber,
            EvoGetPhoneStatus,
            EvoSendDTMF,
            EvoSpeedDial,
            EvoResumeHeldCall,
            EvoHoldIncomingCall,
            EvoSwapCall,
            EvoConfCall,
            EvoSplitCall,
            EvoReleaseSpecCall,
            EvoGetCallStatus,
            EvoGetSCOConnectionStatus,
            EvoActivateVoiceRecognition,
            #ifndef PHONE_SIRI
            EvoAcquireTelephoneNumber
            #else
            EvoAcquireTelephoneNumber,
            EvoEnableSiri,
            EvoSetSiriNrStatus,
            EvoSetSiriEfm
            #endif
      };

GMutex ConnMapTableMutex;
GMutex TempDeviceAddMutex;
GMutex ModemProxyTableMutex;

void AssignBusConnection(void);
GDBusConnection* getEvoBusConnection(tVoid);
void CreateVoiceCallProxy(const char* voiceCallObjpath);
void CreateVoiceCallManagerProxy(const char* objpath);
void CreateCallVolumeProxy(const char* objpath);
void CreateSiriProxy(const char* objpath);
void CreateModemProxy(const char* objpath);
void CreateNetworkRegistrationProxy(const char* objpath);
void CreateNetworkOperatorProxy(const char* gnetworkOperatorPath);
void registerSiganlHandler(int connIndex);
void getVoiceCallProperties(VoiceCall* voiceCallProxy);
void getVoiceCallManagerProperties(int connIndex);
void _printSlotStatusEvo(void);
void PrintCallstatus(void);
void PrintCallstatusConsolidated(void);
void PrintCallMapTable(int connIndex);
void SendAllCallstatus(int connIndex);
void removeCallInstance(int connIndex, int index,int removedCallHandle);
void voiceCallHoldRequest(int connIndex, int callHandle = FC_PHONE_VOICECALL_INDEX_ONE);
void voiceCallAnswerRequest(int connIndex, int callHandle = FC_PHONE_VOICECALL_INDEX_ONE);
gboolean voiceCallHangupRequest(int connIndex, const char* user_data, int callHandle = FC_PHONE_VOICECALL_INDEX_ONE);
gboolean voiceCallTerminateRequest(int connIndex,const char* user_data,int callHandle = FC_PHONE_VOICECALL_INDEX_ONE);
gboolean voiceCallRejectRequest(int connIndex, int callHandle = FC_PHONE_VOICECALL_INDEX_ONE);
void voiceCallHoldIncomingRequest(int connIndex, int callHandle = FC_PHONE_VOICECALL_INDEX_ONE);
//gboolean voiceCallManagerCreateMultiparty(const char*);
void InitSCOConnectionStatusEvo(void);
void GetCallStatusFromCallMapTable(T_CallStatus &callMapTable_CallStatus);
void CopyCallStatusToCallMapTable(T_CallStatus &EvoCallbackdata_callStatus,int connIndex, int Index);
void vFillCallStatus_evo(GObject* source_object, guchar Status, guchar Code);
tChar ErrorMessageToErrorCode(char* errMsg);
void UpdateSLCStatus(const char* devicePath,tBool status);
void AssignVoiceCallProxy(GObject* voiceCallProxy,int connIndex, int callIndex);
GObject* GetVoiceCallProxy(int connIndex, int callHandle);
//void setSiriEnableRequest(const char* deviceaddress, guint16 vendorId, guint16 productId, guint16 version, guint16 supportType);
void getDeviceAddressFromHfpPath(char* remoteDeviceaddress, const char* hfpObjpath);
void getRemoteDeviceAddressFromDevicePath(const char* deviceObjpath, char* acDeviceAddresEvo);
void UpdateScoConnectionStatus(tU8 scotype, GObject* object);
tBool CheckVoiceCallProxyDuplication(int callIndex,int connIndex);
tBool CreateProxies(const char* handsfreeObjpath);
void GetPhoneStatus(int connIndex,T_PhoneStatus *PhoneStatus_data_temp);
void StorePhoneStatus(int connIndex,T_PhoneStatus *PhoneStatus_data_temp);
int getFreeSlotFromCallMapTable(int connIndex);
int getCallSlotFromCallMapTable(int connIndex, int callIndex);
gboolean removeVoiceCallProxy(int connIndex, int removedindex);
void PrintPhoneStatus(int connIndex);
void ResetPhoneStatus(int connIndex);
void ResetCallMapTable(int connIndex);
static void _Evo_Hangup_Release_And_Swap_completion_callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData);
static void _Evo_Hangup_All_completion_callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData);
gboolean voiceCallManagerReleaseAndSwap(int connIndex, const char* user_data);
gboolean voiceCallManagerHangupAll(int connIndex,const char* user_data);
int getCallTotalForConnectedDevice(int connIndex);
void updateAndInsertNewCall(int connIndex, int callIndex,int free_slot, const gchar* CallPath);
gboolean VoiceCallTerminateOrHangup(int connIndex, FCPhoneDbusClient_TerminateType_t enHangupType, const char* user_data);
gboolean IsIntermediateCallState(int connIndex);
gboolean voiceCallManagerGetCalls(int connIndex);

// Modem Proxy Table
void vAddModemtoModemProxyTable(const gchar* objPath);
void vRemoveModemFromModemProxyTable(const gchar* objPath);
char* GetDevAddressFromModemProxyTable(Modem *modemProxy);
Modem* GetModemProxyFromModemProxyTable(tString pDevAddress);
void vPrintModemProxyTable();
void vClearModemProxyTable();
void vGetModems();

#ifdef PHONE_SIRI
void vFill_Siri_evo(GObject* source_object);
#endif
//function Definitions

/******************************************************************
 Function Name : register_cca_evo_dbus_interface

 Description   : Called by Telephony CCA component to get the telephony
                 dbus client interface table.

 Arguments     : NONE
 In            : NONE
 Out           : *CCA_dbus_interfaces - pointer to interface table.
 *******************************************************************/
_CCA_dbus_interfaces * get_cca_evo_dbus_interface(void)
{
   ETG_TRACE_USR4(("get_cca_evo_dbus_interface Fusion entered "));
   return &CCADbusEvoInterfaces;
}

/******************************************************************
 Function Name : SetVehicleBtAddress

 Description   : Setting Vehicle BT Address.

 Arguments     :
 In            : vehicleAddress - Vehicle BT Address
 Out           : NONE
 *******************************************************************/
void SetVehicleBtAddress(char* vehicleAddress)
{
   ETG_TRACE_USR2(("vSetVehicleBtAddress lower :%s", vehicleAddress));
   char acTempActiveDeviceBD_Address[12];
   char vehicleAddressUpper[PHONE_OBJECT_PATH];

   sscanf((const char*) vehicleAddress, "%02X%02X%02X%02X%02X%02X",
         (unsigned int *) &acTempActiveDeviceBD_Address[0], (unsigned int *) &acTempActiveDeviceBD_Address[1],
         (unsigned int *) &acTempActiveDeviceBD_Address[2], (unsigned int *) &acTempActiveDeviceBD_Address[3],
         (unsigned int *) &acTempActiveDeviceBD_Address[4], (unsigned int *) &acTempActiveDeviceBD_Address[5]);

   acTempActiveDeviceBD_Address[6] = 0;

   snprintf(vehicleAddressUpper, 25, "%02X%02X%02X%02X%02X%02X",
         (unsigned char) acTempActiveDeviceBD_Address[0], (unsigned char) acTempActiveDeviceBD_Address[1],
         (unsigned char) acTempActiveDeviceBD_Address[2], (unsigned char) acTempActiveDeviceBD_Address[3],
         (unsigned char) acTempActiveDeviceBD_Address[4], (unsigned char) acTempActiveDeviceBD_Address[5]);

   ETG_TRACE_USR4((" Active Device Address after conversion vehicleAddressUpper :%s", vehicleAddressUpper));

   if (!vehicleBtAddress)
   {
      //Setting Vehicle BT address.
      vehicleBtAddress = strdup(vehicleAddressUpper);
      ETG_TRACE_USR2(("vehicleBtAddress :%s", vehicleBtAddress));
   }

}

/******************************************************************
 Function Name : _checkfor_Registration_repetition

 Description   : check for connection complete callback repetition
 for same address
 NOTE          : The caller must hold a lock on the ConnMapTableMutex when
                 calling this routine.

 Arguments     :
 In            : pu8TempDevAddress - Device Address
 Out           :
 *******************************************************************/
static tBool _checkfor_Registration_repetition(char *pu8TempDevAddress)
{
   ETG_TRACE_USR4(("_checkfor_Registration_repetition entered "));
   tBool bRetVal = FALSE;

   for (int index1 = 0; index1 < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index1++)
   {
      if (FC_PHONE_DEVICE_CONNECTED == ConnMapTableST[index1].en_conn_status
            || FC_PHONE_DEVICE_CONNECTION_IN_PROGRESS == ConnMapTableST[index1].en_conn_status)
      {
         if (0 == u8StringCompare(pu8TempDevAddress, ConnMapTableST[index1].DevAddr))
         {
            bRetVal = TRUE;
            break;
         }
      }
   }
   return bRetVal;
}

/******************************************************************
Function Name : _free_proxy_slot_evo

Description   : free the slot corresponding the slot

NOTE          : The caller must hold a lock on the ConnMapTableMutex when
                calling this routine.

Arguments:
In            : proxy - Proxy
                vproxy_id - Proxy Id
Out: 
 *******************************************************************/
int _free_proxy_slot_evo(GObject *proxy, FC_PHONE_DBUS_ProxyIdEnum_evo vproxy_id)
{
   ETG_TRACE_USR4(("_free_proxy_slot_evo entered "));
   ETG_TRACE_USR4(("Proxy is :%p with Proxy Index:%d ", proxy, vproxy_id));
   int index1, index2;
   int slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;

   for (index1 = 0; index1 < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index1++)
   {
      if ((FC_PHONE_DEVICE_CONNECTED == ConnMapTableST[index1].en_conn_status)
            && (proxy == ConnMapTableST[index1].proxy_common[vproxy_id]))
      {
         for (index2 = 0; index2 < FC_PHONE_DBUS_PROXY_INTERFACE_COUNT_EVO; index2++)
         {
            if(ConnMapTableST[index1].proxy_common[index2])
            {
               ETG_TRACE_USR4(("Removing proxy for Connected device"));

               if (G_IS_OBJECT(ConnMapTableST[index1].proxy_common[index2]))
               {
                  ETG_TRACE_USR4(("g_object_unref for valid GObject Proxy"));
                  g_object_unref(ConnMapTableST[index1].proxy_common[index2]);
               }
               else
               {
                  ETG_TRACE_USR4(("GObject Proxy is NOT valid"));
               }

               ConnMapTableST[index1].proxy_common[index2] = NULLPTR;
            }
         }
         ConnMapTableST[index1].en_conn_status = FC_PHONE_DEVICE_NOT_CONNECTED;
         ConnMapTableST[index1].gcallTotal = FC_PHONE_CALL_TOTAL_ZERO;
         ConnMapTableST[index1].DevAddr[0] = '\0';
         ResetPhoneStatus(index1);
         ResetCallMapTable(index1);
         ConnMapTableST[index1].AllCallStatus = FC_PHONE_EVO_DBUS_UNKNOWN;
         ConnMapTableST[index1].isGetcallsInProgress = FALSE;

         ConnMapTableST[index1].bAudioTfer_direction = FALSE;
         ConnMapTableST[index1].u8VRAction = 0;

         slot = index1;
         break;
      }
   }
   return slot;
}

/******************************************************************
Function Name : Get_DeviceAddress_from_proxy

Description   : Get Device address from the proxy

NOTE          : This routine locks the ConnMapTableMutex.
                Nevertheless, it is not thread-safe because it returns a pointer
                to a device address string stored within ConnMapTableST.  Once
                our lock on the table is dropped, there is nothing preventing
                this entry in the table from being deleted or reused.  We need some
                sort of reference counting here.

Arguments     :
In            : proxy - Proxy
                vproxy_id - Proxy Id
Out           :
 *******************************************************************/
static char* Get_DeviceAddress_from_proxy_evo(GObject *proxy, FC_PHONE_DBUS_ProxyIdEnum_evo vproxy_id)
{
   ETG_TRACE_USR4(("Get_DeviceAddress_from_proxy entered "));
   ETG_TRACE_USR4(("Given proxy :%p with Proxy Index:%d ", proxy, vproxy_id));
   int index1;
   char *dev_addr = NULLPTR;

   g_mutex_lock(&ConnMapTableMutex);

   //_printSlotStatusEvo();
   for (index1 = 0; index1 < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index1++)
   {
      if (FC_PHONE_DEVICE_CONNECTED == ConnMapTableST[index1].en_conn_status)
      {
         if (proxy == ConnMapTableST[index1].proxy_common[vproxy_id])
         {
            dev_addr = (ConnMapTableST[index1].DevAddr);
            ETG_TRACE_USR4(("Device Address found for the given proxy:%s", dev_addr));
            break;
         }
      }
   }
   g_mutex_unlock(&ConnMapTableMutex);

   if (!dev_addr)
   {
      ETG_TRACE_ERR(("ERROR ::No Device Address matched from the LIST"));
   }

   return dev_addr;
}


/******************************************************************
 Function Name : Get_proxy_from_ConnIndex_Evo

 Description   : Get  proxy From the connIndex

 NOTE          : This routine locks the ConnMapTableMutex.

 Arguments     :
 In            : dev_address - Device Address
                 vproxy_id - Proxy Id.
 Out           :
 *******************************************************************/
static GObject* Get_proxy_from_ConnIndex_Evo(int connIndex, FC_PHONE_DBUS_ProxyIdEnum_evo vproxy_id)
{
   ETG_TRACE_USR4(("Get_proxy_from_ConnIndex_Evo entered "));
   ETG_TRACE_USR4(("Requested proxy id :%d with connIndex :%d ", vproxy_id, connIndex));
   GObject* proxy = NULLPTR;

   if (connIndex >= FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT)
   {
      ETG_TRACE_ERR(("ERROR :: Greater than Number of device Connection Supported"));
      return proxy;
   }

   g_mutex_lock(&ConnMapTableMutex);

   for (int index1 = 0; index1 < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index1++)
   {
      if ((FC_PHONE_DEVICE_CONNECTED == ConnMapTableST[index1].en_conn_status) && (index1 == connIndex))
      {
            ETG_TRACE_USR1(("ConnMapTableST[%d].DevAddr:: %s  ", index1, ConnMapTableST[index1].DevAddr));
            proxy = ConnMapTableST[index1].proxy_common[vproxy_id];
            ETG_TRACE_USR4(("connIndex Matched:%d ", index1));
            break;
       }
   }
   g_mutex_unlock(&ConnMapTableMutex);

   if (!proxy)
   {
      ETG_TRACE_ERR(("ERROR ::No Proxy matched from the LIST"));
   }

   return proxy;
}

/******************************************************************
 Function Name : _getfree_slot

 Description   : get free slot for new connection

 NOTE          : The caller must hold a lock on the ConnMapTableMutex when
 calling this routine.

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
static int _getfree_slot(void)
{
   ETG_TRACE_USR1(("_getfree_slot entered "));
   int index1;
   int free_slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;

   for (index1 = 0; index1 < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index1++)
   {
      if (FC_PHONE_DEVICE_NOT_CONNECTED == ConnMapTableST[index1].en_conn_status)
      {
         free_slot = index1;
         ETG_TRACE_USR4(("Free slot found at :%d", free_slot));
         break;
      }
   }
   if (free_slot == FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::No Free slot found"));
   }
   return free_slot;
}

/*****************************************************************
 Function Name : _getconnected_slot

 Description   : get a connected slot.

 NOTE          : The caller must hold a lock on the ConnMapTableMutex when
                 calling this routine.

 Arguments     :
 In            :
 Out           :
 ******************************************************************/
int _getconnected_slot(GObject* source_object)
{
   ETG_TRACE_USR4(("_getconnected_slot entered "));
   int connected_slot = FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT;
   char activeDeviceAddressEvo[PHONE_OBJECT_PATH];
   int index = 0;
   const gchar *hfpObjpath = g_dbus_proxy_get_object_path((GDBusProxy*) source_object); //lint !e826

   getDeviceAddressFromHfpPath(activeDeviceAddressEvo,hfpObjpath);

   ETG_TRACE_USR4(( "activeDeviceAddress :%s", activeDeviceAddressEvo));

   g_mutex_lock(&ConnMapTableMutex);

   for (index = 0; index < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index++)
   {
      ETG_TRACE_USR4(( "ConnMapTableST[%d].DevAddr:%s",index,ConnMapTableST[index].DevAddr));
      if (FC_PHONE_DEVICE_CONNECTED == ConnMapTableST[index].en_conn_status
            || FC_PHONE_DEVICE_CONNECTION_IN_PROGRESS == ConnMapTableST[index].en_conn_status)
      {
         if(u8StringCompare(activeDeviceAddressEvo,ConnMapTableST[index].DevAddr) == 0)
         {
            connected_slot = index;
            ETG_TRACE_USR4(("Connected slot found at :%d", connected_slot));
            break;
         }
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   return connected_slot;
}

/*****************************************************************
 Function Name : getConnIndexFromDeviceAdd

 Description   : get a connected slot using device address.

 NOTE          : The caller must hold a lock on the ConnMapTableMutex when
                 calling this routine.

 Arguments     :
 In            :
 Out           :
 ******************************************************************/
int getConnIndexFromDeviceAdd(const char* DevAddress)
{
   ETG_TRACE_USR1((" getConnIndexFromDeviceAdd entered"));
   ETG_TRACE_USR1((" getConnIndexFromDeviceAdd DevAddress: %s",DevAddress));

   int connected_slot = FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT;
   char activeDeviceAddressEvo[PHONE_OBJECT_PATH];

   getRemoteDeviceAddressFromDevicePath(DevAddress, activeDeviceAddressEvo);
   ETG_TRACE_USR4((" getConnIndexFromDeviceAdd device address %s", activeDeviceAddressEvo));

   g_mutex_lock(&ConnMapTableMutex);

   for (int index = 0; index < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index++)
   {
      if (FC_PHONE_DEVICE_CONNECTED == ConnMapTableST[index].en_conn_status)
      {
         if (u8StringCompare(activeDeviceAddressEvo, ConnMapTableST[index].DevAddr) == 0)
         {
            connected_slot = index;
            ETG_TRACE_USR4(("Connected slot found at :%d", connected_slot));
            break;
         }
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   if (connected_slot == FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::No Connected slot found"));
   }
   return connected_slot;
}

/******************************************************************
 Function Name : _printSlotStatusEvo

 Description   : Print all the slot status

 NOTE          : The caller must hold a lock on the ConnMapTableMutex when
                 calling this routine.

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void _printSlotStatusEvo(void)
{
   ETG_TRACE_USR4((" _printSlotStatusEvo entered"));
   int index1, index2;
   for (index1 = 0; index1 < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index1++)
   {
      for (index2 = 0; index2 < FC_PHONE_DBUS_PROXY_INTERFACE_COUNT_EVO; index2++)
      {
         ETG_TRACE_USR3(("ConnMapTableST[%d].proxy_common[%d] :: %p", index1, index2, ConnMapTableST[index1].proxy_common[index2]));
      }
      ETG_TRACE_USR3(("ConnMapTableST[%d].en_conn_status :: %d", index1, ConnMapTableST[index1].en_conn_status));
      ETG_TRACE_USR3(("ConnMapTableST[%d].DevAddr        :: %s", index1, ConnMapTableST[index1].DevAddr));

   }
}

/******************************************************************
 Function Name : register_cca_evo_callbacks

 Description   : Called by Telephony CCA component to register its
                 callback to telephony dbus client.

 Arguments     :
 In            : vCCA_callbacks - Callback instance
 Out           :
 *******************************************************************/

tVoid register_cca_evo_callbacks(_T_CCA_Dbus_If *vCCA_callbacks)
{
   ETG_TRACE_USR4(("register_cca_evo_callbacks entered "));
   DCCA_callbacks_evo = vCCA_callbacks;
}

/******************************************************************
 Function Name : FB_Evodbus_init

 Description   : Initializes the DBUS Communication interface

 NOTE          : This function should be called before any other API calls
 Arguments     : NONE
 In            : NONE
 Out           : NONE
 *******************************************************************/

void FB_Evodbus_init(void)
{
   ETG_TRACE_USR4(("  In FB_Evodbus_init  entered"));
   int index1,index2;

   g_mutex_lock(&ConnMapTableMutex);

   for (index1 = 0; index1 < FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT; index1++)
   {
      for (index2 = 0; index2 < FC_PHONE_DBUS_PROXY_INTERFACE_COUNT_EVO; index2++)
      {
         ConnMapTableST[index1].proxy_common[index2] = NULLPTR;
      }

      ConnMapTableST[index1].en_conn_status = FC_PHONE_DEVICE_NOT_CONNECTED;
      ConnMapTableST[index1].DevAddr[0] = '\0';
      ConnMapTableST[index1].gcallTotal = FC_PHONE_CALL_TOTAL_ZERO;
      ConnMapTableST[index1].AllCallStatus = FC_PHONE_EVO_DBUS_UNKNOWN;
      ConnMapTableST[index1].isGetcallsInProgress = FALSE;

      ConnMapTableST[index1].bAudioTfer_direction = FALSE;
      ConnMapTableST[index1].u8VRAction = 0;

      ResetPhoneStatus(index1);
      ResetCallMapTable(index1);
   }
   g_mutex_unlock(&ConnMapTableMutex);
}

/*############################# METHOD CALL APIs ################################# */

/******************************************************************
 Function Name : getDeviceAddressFromHfpPath

 Description   : Gives the remote device address  from hfp object path

 Arguments     : NONE
 In            : remoteDeviceaddress - Device address
                 deviceObjpath - Device path
 Out           : NONE
 *******************************************************************/
void getDeviceAddressFromHfpPath(char* remoteDeviceaddress, const gchar* hfpObjpath)
{
   ETG_TRACE_USR4(( "getDeviceAddressFromHfpPath entered"));
   ETG_TRACE_USR4(( "Device object path :%s", hfpObjpath));
   char node_name[25] = {0};
   const char * deviceAddressWithVehicleaddress;


   if(hfpObjpath)
   {
      //Find out the "/hfp/" from the object path.
      deviceAddressWithVehicleaddress = strstr(hfpObjpath,"/hfp/");
      if(deviceAddressWithVehicleaddress)
      {
         //Copy the device address to node_name as "_deviceaddress'. to ignore /hfp/vehicleaddress add +18;
         vStringCopy ( node_name, deviceAddressWithVehicleaddress+18, PHONE_TEMP_DEV_ADDRESS);
         ETG_TRACE_USR4(( "node_name :%s", node_name));

         //Remove the '/' from node_name and returns token by token.
         deviceAddressWithVehicleaddress = strtok (node_name,"/");

         if(deviceAddressWithVehicleaddress)
         {
            ETG_TRACE_USR4(( "This is a VoiceCall object path"));
            vStringCopy(remoteDeviceaddress,deviceAddressWithVehicleaddress,PHONE_TEMP_DEV_ADDRESS);
         }
         else
         {
            ETG_TRACE_USR4(( "This is a HFP Object Path"));
            vStringCopy(remoteDeviceaddress,node_name,PHONE_TEMP_DEV_ADDRESS);
         }


      }
      else
      {
         ETG_TRACE_FATAL(( "FATAL_ERROR ::Invalid object path"));
         return;
      }
   }
   ETG_TRACE_USR4(( "Remote device address :%s", remoteDeviceaddress));
}


/******************************************************************
 Function Name : getremoteDeviceaddress

 Description   : Gives the remote device address  from device path

 Arguments     : NONE
 In            : remoteDeviceaddress - Device address
                 deviceObjpath - Device path
 Out           : NONE
 *******************************************************************/
void getremoteDeviceaddress(char* remoteDeviceaddress, const char* deviceObjpath)
{
   ETG_TRACE_USR4(( "getremoteDeviceaddress entered"));
   ETG_TRACE_USR4(( "Device object path :%s", deviceObjpath));
   char node_name[25] = {0};
   const char * deviceAddressWithUnderscore;
   std::string tempOutput;

   if(deviceObjpath)
   {
      //Find out the "dev_" from the object path.
      deviceAddressWithUnderscore = strstr(deviceObjpath,"dev_");
      if(deviceAddressWithUnderscore)
      {
         //Copy the device address to node_name with "_'. to ignore dev_ add +4;
         vStringCopy( node_name, deviceAddressWithUnderscore+4, sizeof(node_name) );

         //Remove the underscore from node_name and returns token by token.
         deviceAddressWithUnderscore = strtok (node_name,"_");

         //Constructs the tokens to string.
         while( deviceAddressWithUnderscore)
         {
            tempOutput.append(deviceAddressWithUnderscore);
            deviceAddressWithUnderscore = strtok (NULLPTR, "_");
         }
         vStringCopy(remoteDeviceaddress,tempOutput.c_str(),PHONE_TEMP_DEV_ADDRESS);
      }
      else
      {
         ETG_TRACE_FATAL(( "FATAL_ERROR ::Invalid object path"));
         return;
      }
   }
   ETG_TRACE_USR4(( "Remote device address :%s", remoteDeviceaddress));
}

/******************************************************************
 Function Name : getRemoteDeviceAddressFromDevicePath

 Description   : Gives the remote device address  from device path

 Arguments     : NONE
 In            : deviceObjpath - Device path
                 acDeviceAddresEvo - Device address
 Out           : NONE
 *******************************************************************/
//Converting Device address from 'dev_AA_BB_CC_DD_EE' to 'AABBCCDDEE'
void getRemoteDeviceAddressFromDevicePath(const char* deviceObjpath, char* acDeviceAddresEvo)
{
   ETG_TRACE_USR4(("getRemoteDeviceAddressFromDevicePath entered"));
   ETG_TRACE_USR4(("getRemoteDeviceAddressFromDevicePath: Active Device Address entry :%s", deviceObjpath));
   char acTempActiveDeviceBD_Address[12];
   char remoteDeviceAddress[25] = {0};

   getremoteDeviceaddress(remoteDeviceAddress,deviceObjpath);

   ETG_TRACE_USR4((" sscanf: Entry Remote device address :%s", remoteDeviceAddress));

   sscanf((const char*)remoteDeviceAddress, "%02X%02X%02X%02X%02X%02X",
         (unsigned int *) &acTempActiveDeviceBD_Address[0], (unsigned int *) &acTempActiveDeviceBD_Address[1],
         (unsigned int *) &acTempActiveDeviceBD_Address[2], (unsigned int *) &acTempActiveDeviceBD_Address[3],
         (unsigned int *) &acTempActiveDeviceBD_Address[4], (unsigned int *) &acTempActiveDeviceBD_Address[5]);
   acTempActiveDeviceBD_Address[6] = 0;

   ETG_TRACE_USR4((" sprintf: Entry Remote device address :%s", remoteDeviceAddress));

   snprintf(acDeviceAddresEvo, 25, "%02X%02X%02X%02X%02X%02X",
         (unsigned char)acTempActiveDeviceBD_Address[0], (unsigned char)acTempActiveDeviceBD_Address[1], (unsigned char)acTempActiveDeviceBD_Address[2],
         (unsigned char)acTempActiveDeviceBD_Address[3], (unsigned char)acTempActiveDeviceBD_Address[4], (unsigned char)acTempActiveDeviceBD_Address[5]);

   ETG_TRACE_USR4((" Active Device Address after conversion acDeviceAddresEvo :%s", acDeviceAddresEvo));

}

/******************************************************************
 Function Name : getDevicePathFromRemoteDeviceAddress

 Description   : Gives the remote device path  from device address

 Arguments     : NONE
 In            : deviceObjpath - Device path
                 acDeviceAddresEvo - Device address
 Out           : NONE
 *******************************************************************/
//Converting Device address from 'AABBCCDDEE' to 'dev_AA_BB_CC_DD_EE'
void getDevicePathFromRemoteDeviceAddress(const char* acDeviceAddresEvo,char* remoteDevicePath)
{
   ETG_TRACE_USR4(("getDevicePathFromRemoteDeviceAddress entered"));
   ETG_TRACE_USR4(("Active Device Address entry :%s", acDeviceAddresEvo));
   tChar acTempActiveDeviceBD_Address[PHONE_TEMP_DEV_ADDRESS] = {'\0'};

   sscanf(acDeviceAddresEvo, "%02x%02x%02x%02x%02x%02x",
         (tUInt *) &acTempActiveDeviceBD_Address[0], (tUInt *) &acTempActiveDeviceBD_Address[1],
         (tUInt *) &acTempActiveDeviceBD_Address[2], (tUInt *) &acTempActiveDeviceBD_Address[3],
         (tUInt *) &acTempActiveDeviceBD_Address[4], (tUInt *) &acTempActiveDeviceBD_Address[5]);

   snprintf(remoteDevicePath, PHONE_TEMP_DEV_ADDRESS, "dev_%02x_%02x_%02x_%02x_%02x_%02x",
         (unsigned char)acTempActiveDeviceBD_Address[0], (unsigned char)acTempActiveDeviceBD_Address[1], (unsigned char)acTempActiveDeviceBD_Address[2],
         (unsigned char)acTempActiveDeviceBD_Address[3], (unsigned char)acTempActiveDeviceBD_Address[4], (unsigned char)acTempActiveDeviceBD_Address[5]);

   ETG_TRACE_USR4(("getDevicePathFromRemoteDeviceAddress: devicepath :%s",remoteDevicePath));
}

/******************************************************************
 Function Name : getHandsfreeObjpath

 Description   : Gives the hands free object path from remote device and vehicle address.

 Arguments     : NONE
 In            : handsfreeObpath - Hands free object path
                 deviceAddress - Device address
 Out           : NONE
 *******************************************************************/
void getHandsfreeObjpath(char* handsfreeObpath, const char* deviceAddress)
{
   ETG_TRACE_USR4(("getHandsfreeObjpath entered"));

   if (!deviceAddress)
   {
      ETG_TRACE_USR4(("Device address is null"));
      handsfreeObpath = NULLPTR;
      return;
   }
   if (!vehicleBtAddress)
   {
      ETG_TRACE_USR4(("Adapter address is null"));
      handsfreeObpath = NULLPTR;
      return;
   }
   snprintf(handsfreeObpath, PHONE_OBJECT_PATH, BASE_HANDSFREE_OBJECT_PATH_EVO"%s""_""%s", vehicleBtAddress, deviceAddress);
   ETG_TRACE_USR4(("handsfree obj path :%s", handsfreeObpath));
}
/******************************************************************
Function Name : EvoCreateProxyAndRegisterSignals

Description   : Create remote proxies and Register signals

NOTE          : Since this routine makes calls to the gdbus API for proxy
                creation and dbus signal handler registration, it should only be
                called in the context of the glib main loop thread where the
                gdbus dispatcher runs.

Arguments     :
In            : devicePath - Handsfree object path
Out           :
 *******************************************************************/
static gboolean EvoCreateProxyAndRegisterSignals(const char* devicePath)
{
   ETG_TRACE_USR4(("EvoCreateProxyAndRegisterSignals Entered"));
   ETG_TRACE_USR4((" Device path :%s", devicePath));
   char Obj_path[PHONE_OBJECT_PATH] = { 0 };
   int b_repetition = TRUE;
   char* handsfreeObjpath = NULLPTR;
   char activeDeviceAddressEvo[PHONE_OBJECT_PATH] = {0};


   getRemoteDeviceAddressFromDevicePath(devicePath, activeDeviceAddressEvo);
   ETG_TRACE_USR4((" Remote device address :%s", activeDeviceAddressEvo));

   g_mutex_lock(&ConnMapTableMutex);
   //Check for repetition
   b_repetition = _checkfor_Registration_repetition(activeDeviceAddressEvo);

   g_mutex_unlock(&ConnMapTableMutex);
   if (TRUE == b_repetition) //if Repeated Do nothing
   {
      ETG_TRACE_ERR(("ERROR ::This Device Address :: %s is Already Registered", activeDeviceAddressEvo));
      return FC_PHONE_DBUS_FAIL;
   }

   //constructs handsfree obj path with remote device address
   getHandsfreeObjpath(Obj_path, (const char *) activeDeviceAddressEvo);

   ETG_TRACE_USR4((" object path :%s", Obj_path));

   handsfreeObjpath = Obj_path;

   ETG_TRACE_USR4((" Handsfree object path %s", handsfreeObjpath));

   if (gdbus)
   {
      tBool retVal = CreateProxies(handsfreeObjpath);
      if (retVal == FC_PHONE_DBUS_FAIL)
      {
         ETG_TRACE_ERR(("ERROR :: Handsfree proxy creation failed"));
         return FC_PHONE_DBUS_FAIL;
      }
   }
   else
   {
      ETG_TRACE_ERR(("ERROR :: gdbus is NULL"));
      g_mutex_lock(&TempDeviceAddMutex);

      tempDeviceAddress = strdup(activeDeviceAddressEvo);

      g_mutex_unlock(&TempDeviceAddMutex);
      return FC_PHONE_DBUS_FAIL;
   }

   return FC_PHONE_DBUS_SUCCESS;
}


/******************************************************************
Function Name : CreateProxies

Description   : Create remote proxies and Register signals

NOTE          : Since this routine makes calls to the gdbus API for proxy
                creation and dbus signal handler registration, it should only be
                called in the context of the glib main loop thread where the
                gdbus dispatcher runs.

Arguments     :
In            : devicePath - Handsfree object path
Out           :
 *******************************************************************/
tBool CreateProxies(const char* handsfreeObjpath)
{
   ETG_TRACE_USR4(("CreateProxies Entered"));
   ETG_TRACE_USR4((" handsfreeObjpath :%s", handsfreeObjpath));
   int connIndex = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;
   GError *error = NULLPTR;
   GObject* handsfreeProxy = NULLPTR;
   char activeDeviceAddressEvo[PHONE_OBJECT_PATH] = {'\0'};
   char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};


   getDeviceAddressFromHfpPath(activeDeviceAddressEvo,handsfreeObjpath);
   ETG_TRACE_USR4((" Remote device address :%s", activeDeviceAddressEvo));


   //Create handsfreeProxy Proxy
   handsfreeProxy
         = (GObject*) handsfree_proxy_new_sync(gdbus, G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
         BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
         handsfreeObjpath,//OBJECT PATH
         NULLPTR,//GCancellable
         &error);
   if (error)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::Couldn't create the handsfree proxy object %s", error->message));
      g_error_free(error);
      return FC_PHONE_DBUS_FAIL;
   }

   if (!handsfreeProxy)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::Couldn't create the handsfree proxy object"));
      return FC_PHONE_DBUS_FAIL;
   }
   else
   {
      ETG_TRACE_USR4(("Handsfree proxy created successfully"));
   }

   g_mutex_lock(&ConnMapTableMutex);

   connIndex = _getfree_slot();
   if (FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT == connIndex)
   {
      g_mutex_unlock(&ConnMapTableMutex);
      ETG_TRACE_FATAL(("FATAL ERROR ::No connection slot is free all used"));
      g_object_unref(handsfreeProxy);
      return FC_PHONE_DBUS_FAIL;
   }
   //set the device address in connection slot
   ConnMapTableST[connIndex].en_conn_status = FC_PHONE_DEVICE_CONNECTION_IN_PROGRESS;

   vStringCopy(ConnMapTableST[connIndex].DevAddr, activeDeviceAddressEvo, PHONE_CONNMAP_DEV_ADDRESS);
   ETG_TRACE_USR3(("Connected Device Address :%s", ConnMapTableST[connIndex].DevAddr));
   ConnMapTableST[connIndex].proxy_common[FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE] = handsfreeProxy;

   getDevicePathFromRemoteDeviceAddress(activeDeviceAddressEvo, remoteDevicePath);
   vStringCopy(ConnMapTableST[connIndex].PhoneStatus_data.BD_Address, remoteDevicePath, PHONE_CONNMAP_DEV_ADDRESS);

   //_printSlotStatusEvo();
   g_mutex_unlock(&ConnMapTableMutex);

   //Create Handsfree related pxoxies.
   CreateVoiceCallManagerProxy(handsfreeObjpath);
   CreateNetworkRegistrationProxy(handsfreeObjpath);
   CreateCallVolumeProxy(handsfreeObjpath);
   CreateSiriProxy(handsfreeObjpath);
   CreateModemProxy(handsfreeObjpath);

   InitSCOConnectionStatusEvo();

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
Function Name : UpdateSLCStatus

Description   : Updating SLC status.

NOTE          : This method will update SLCstatus.
                This must be called when DBUS client is ready with all proxy creation and property registration.
                Should be placed in last proxy callback.
Arguments     :
In            :
Out           :
 *******************************************************************/
void UpdateSLCStatus(const char* devicePath,tBool status)
{
   ETG_TRACE_USR1(("UpdateSLCStatus entered"));

   //HFP is already connected, so Update SLC status.
   vStringCopy((char *) EvoCallbackdata.BD_Address, devicePath, PHONE_OBJECT_PATH);

   ETG_TRACE_USR4(("Sending vSLCStatus to CCA for Device :%s", EvoCallbackdata.BD_Address));
   EvoCallbackdata.Service = FC_PHONE_DBUS_SERVICE_TYPE_HFP;
   EvoCallbackdata.Status_ConnComp = FC_PHONE_DBUS_SUCCESS;
   EvoCallbackdata.b_SLC_status = status;
   DCCA_callbacks_evo->vSLCStatus(&EvoCallbackdata);
}

/******************************************************************
Function Name : InitSCOConnectionStatusEvo

Description   : Initialising SCOConnectionStatus_data_evo

Arguments     :
In            :
Out           :
 *******************************************************************/
void InitSCOConnectionStatusEvo(void)
{
   ETG_TRACE_USR4(("InitSCOConnectionStatusEvo entered"));
   memset(SCOConnectionStatus_data_evo.BD_Address,0,sizeof(SCOConnectionStatus_data_evo.BD_Address));
   SCOConnectionStatus_data_evo.SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_STATUS_UNKNOWN;
   SCOConnectionStatus_data_evo.Codec = NARROW_BAND_SPEECH;
}

/******************************************************************
 Function Name : handsfreePropertyChangedSignalHandler

 Description   : Callback function for handsfree property changed.

 Arguments     :
 In            : object - Handsfree object
                 propertyName - Property name
                 arg_value - variable list of properties.
 Out           :
 *******************************************************************/
void handsfreePropertyChangedSignalHandler(Handsfree *object, const gchar *propertyName, GVariant *arg_value)
{
   ETG_TRACE_USR4(("handsfreePropertyChangedSignalHandler entered"));

   if (!propertyName)
   {
      ETG_TRACE_USR4(("Received propertyName is NULL. Hence returning."));
      return;
   }

   ETG_TRACE_USR4(("handsfreePropertyChangedSignalHandler propertyName is :%s ", propertyName));
   const GVariantType *type;
   type = g_variant_get_type(arg_value);

   if (g_variant_type_is_variant(type))
   {
      int connIndex = _getconnected_slot((GObject*)object);
      if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
      {
         ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
         return;
      }

      T_PhoneStatus PhoneStatus_data_temp;
      GetPhoneStatus(connIndex, &PhoneStatus_data_temp);

      GVariant *child;
      child = g_variant_get_child_value(arg_value, 0);

      if (strcmp("InbandRinging", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BOOLEAN))
         {
            gboolean inbandRingtone = g_variant_get_boolean(child);
            ETG_TRACE_USR4(("Value of propertyChanged inbandRingtone :%d ", inbandRingtone));
         }
      }
      else if (strcmp("VoiceRecognition", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BOOLEAN))
         {
            gboolean voiceRecognition = g_variant_get_boolean(child);
            ETG_TRACE_USR4(("Value of propertyChanged voiceRecognition :%d ", voiceRecognition));

            char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};
            char* p_devAdd = Get_DeviceAddress_from_proxy_evo((GObject*) object, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
            if (p_devAdd)
            {
               getDevicePathFromRemoteDeviceAddress(p_devAdd, remoteDevicePath);
               vStringCopy(VoiceRec_data_evo.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
               ETG_TRACE_USR3(("VoiceRecognitionInd VoiceRec_data_evo.BD_Address :%s", VoiceRec_data_evo.BD_Address));

               VoiceRec_data_evo.Activation = static_cast<tU8>(voiceRecognition);

               ETG_TRACE_USR3((" Sending vVoiceRecognitionInd to CCA "));
               DCCA_callbacks_evo->vVoiceRecognitionInd(&VoiceRec_data_evo);
            }
         }
      }
      else if (strcmp("EchoCancelingNoiseReduction", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BOOLEAN))
         {
            gboolean isECNREnabled = g_variant_get_boolean(child);
            ETG_TRACE_USR4(("Value of propertyChanged isecnrEnabled :%d ", isECNREnabled));
         }
      }
      else if (strcmp("BatteryChargeLevel", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            tU8 batterylevel = (tU8) g_variant_get_byte(child);
            ETG_TRACE_USR4(("Value of propertyChanged battery level :%d ", batterylevel));
            PhoneStatus_data_temp.u8PhoneBattery = batterylevel;
         }
      }
      else if (strcmp("SCOType", propertyName) == 0)
      {
         tU8 scotype = FC_PHONE_EVO_DBUS_NOT_CONNECTED;
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            scotype = (tU8) g_variant_get_byte(child);
            ETG_TRACE_USR4(("Value of propertyChanged SCOType :%d ", scotype));
            UpdateScoConnectionStatus(scotype, (GObject*) object);
         }
      }
      else if (strcmp("AudioPath", propertyName) == 0)
      {
         tU8 audioPath = FC_PHONE_EVO_DBUS_AUDIO_PATH_UNKNOWN;
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            audioPath = (tU8) g_variant_get_byte(child);
            ETG_TRACE_USR4(("Value of propertyChanged audio path :%d ", audioPath));
         }

         switch (audioPath)
         {
            case FC_PHONE_EVO_DBUS_AUDIO_PATH_IDLE:
            {
               audioPath = 0x00;
               EvoCallbackdata.u8RingTone = FC_PHONE_EVO_SERVICE_DISABLED;
            }
            break;
            case FC_PHONE_EVO_DBUS_AUDIO_PATH_CALL:
            {
               audioPath = 0x01;
               //No need to send vIncomingCall when call is active
            }
            break;
            case FC_PHONE_EVO_DBUS_AUIOD_PATH_INBAND_RINGING:
            {
               audioPath = 0x02;
            }
            break;
            case FC_PHONE_EVO_DBUS_AUIOD_PATH_MODULE_INT_RINGING:
            {
               audioPath = 0x03;
            }
            break;
            case FC_PHONE_EVO_DBUS_AUDIO_PATH_VOICE_REC:
            {
               audioPath = 0x05;//value 0x04 is missing or might be intentional.
            }
            break;
            case FC_PHONE_EVO_DBUS_AUDIO_PATH_PENDING_FOR_INBAND_RINGTONE:
            {
               audioPath = 0x06;
            }
            break;
            case FC_PHONE_EVO_DBUS_AUDIO_PATH_UNKNOWN:
            {
               audioPath = 0xFF;
            }
            break;
            default:
            break;
         }
      }
      else if (strcmp("AllCallStatus", propertyName) == 0)
      {
         g_mutex_lock(&ConnMapTableMutex);
         ConnMapTableST[connIndex].AllCallStatus = FC_PHONE_EVO_DBUS_UNKNOWN;
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            ConnMapTableST[connIndex].AllCallStatus = (tU8) g_variant_get_byte(child);
            ETG_TRACE_USR4(("Value of propertyChanged AllCallStatus :%d ", ConnMapTableST[connIndex].AllCallStatus));
         }
         g_mutex_unlock(&ConnMapTableMutex);
      }
      else if (strcmp("Audio", propertyName) == 0)
      {
         const gchar* audio = "";
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            audio = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged Audio status :%s ", audio));
         }
         if (strcmp(audio, "AG") == 0)
         {
            //EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_SCO_CONN_DISCONNECTED;
         }
         else if (strcmp(audio, "HF") == 0)
         {
            //EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_SCO_CONN_ESTABLISHED;
         }
      }
      else if (strcmp(propertyName, "ChldFeature") == 0)
      {
         //Clear old data from the vector.
         if (ChldFeatureList.size() > 0)
         {
            ChldFeatureList.clear();
         }

         GVariantIter iter2;
         gchar * list;
         tU8 enhancedSupport = 0;
         g_variant_iter_init(&iter2, child);
         while (g_variant_iter_loop(&iter2, "s", &list))
         {
            ETG_TRACE_USR4(("ChldFeature feature :%s", list));
            //TODO: Memory should be allocated while push_back
            ChldFeatureList.push_back(list);

            if (strcmp(list, "CHLD=0") == 0)
            {
               enhancedSupport |= 1 << 0;
            }
            if (strcmp(list, "CHLD=1") == 0)
            {
               enhancedSupport |= 1 << 1;
            }
            if (strcmp(list, "CHLD=1x") == 0)
            {
               enhancedSupport |= 1 << 2;
            }
            if (strcmp(list, "CHLD=2") == 0)
            {
               enhancedSupport |= 1 << 3;
            }
            if (strcmp(list, "CHLD=2x") == 0)
            {
               enhancedSupport |= 1 << 4;
            }
            if (strcmp(list, "CHLD=3") == 0)
            {
               enhancedSupport |= 1 << 5;
            }
            if (strcmp(list, "CHLD=4") == 0)
            {
               enhancedSupport |= 1 << 6;
            }
         }

         PhoneStatus_data_temp.u8AgThreeWayCallSuppFeatures = enhancedSupport;
         ETG_TRACE_USR1((" u8AgThreeWayCallSuppFeatures:%d", PhoneStatus_data_temp.u8AgThreeWayCallSuppFeatures));

      }
      else if (strcmp(propertyName, "Features") == 0)
      {

         GVariantIter iter2;
         gchar * list;
         tU32 supportFeatures = 0;
         g_variant_iter_init(&iter2, child);
         while (g_variant_iter_loop(&iter2, "s", &list))
         {
            ETG_TRACE_USR4(("Supported feature :%s", list));

            if (strcmp(list, "3way") == 0)
            {
               supportFeatures |= 1 << 0;
            }
            if (strcmp(list, "ECNR") == 0)
            {
               supportFeatures |= 1 << 1;
            }
            if (strcmp(list, "Voice-recognition") == 0)
            {
               supportFeatures |= 1 << 2;
            }
            if (strcmp(list, "Inband_ring") == 0)
            {
               supportFeatures |= 1 << 3;
            }
            if (strcmp(list, "Attach-voice-tag") == 0)
            {
               supportFeatures |= 1 << 4;
            }
            if (strcmp(list, "Reject_incoming_call") == 0)
            {
               supportFeatures |= 1 << 5;
            }
            if (strcmp(list, "Enhanced_call_status") == 0)
            {
               supportFeatures |= 1 << 6;
            }
            if (strcmp(list, "Enhanced_call_control") == 0)
            {
               supportFeatures |= 1 << 7;
            }
            if (strcmp(list, "Extended_error_codes") == 0)
            {
               supportFeatures |= 1 << 8;
            }
            if (strcmp(list, "Codec_negotiation") == 0)
            {
               supportFeatures |= 1 << 9;
            }
         }

         PhoneStatus_data_temp.u32PhoneSuppFeature = supportFeatures;
         ETG_TRACE_USR1((" u32PhoneSuppFeature:%d", PhoneStatus_data_temp.u32PhoneSuppFeature));

      }

      StorePhoneStatus(connIndex, &PhoneStatus_data_temp);
      PrintPhoneStatus(connIndex);
      ETG_TRACE_USR3((" Sending vPhoneStatus to CCA "));
      DCCA_callbacks_evo->vPhoneStatus(&PhoneStatus_data_temp);
   }
}

/******************************************************************
 Function Name : getHandsfreePropertiesCB

 Description   : Callback function for get handsfree properties method.

 Arguments     :
 In            : source_object - Handsfree object
                 res -
                 user_data - user data if any
 Out           :
 *******************************************************************/
void getHandsfreePropertiesCB(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("getHandsfreePropertiesCB entered"));
   (tVoid) user_data;
   GError *error = NULLPTR;
   GVariant *out_data;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;

   retval = handsfree_call_get_properties_finish((Handsfree*) source_object, &out_data, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      GVariantIter iter;
      const gchar *propertyName;
      GVariant *value;
      g_variant_iter_init(&iter, out_data);


      int connIndex = _getconnected_slot((GObject*)source_object);
      if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
      {
         ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
         return;
      }

      T_PhoneStatus PhoneStatus_data_temp;
      GetPhoneStatus(connIndex, &PhoneStatus_data_temp);

      while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
      {
         ETG_TRACE_USR4(("PrpertyName :%s", propertyName));

         if (strcmp(propertyName, "InbandRinging") == 0)
         {
            gboolean inbandRinging;
            g_variant_get(value, "b", &inbandRinging);
            ETG_TRACE_USR4(("inbandRinging :%d", inbandRinging));
         }
         if (strcmp(propertyName, "VoiceRecognition") == 0)
         {
            gboolean voiceRecognition;
            g_variant_get(value, "b", &voiceRecognition);
            ETG_TRACE_USR4(("voiceRecognition :%d", voiceRecognition));
         }
         if (strcmp(propertyName, "BatteryChargeLevel") == 0)
         {
            tU8 batteryChargeLevel;
            batteryChargeLevel = g_variant_get_byte(value);
            PhoneStatus_data_temp.u8PhoneBattery = batteryChargeLevel;
            ETG_TRACE_USR4(("batteryChargeLevel :%d", PhoneStatus_data_temp.u8PhoneBattery));
         }
         if (strcmp(propertyName, "Audio") == 0)
         {
            const gchar* audio = "";
            g_variant_get(value, "s", &audio);
            ETG_TRACE_USR4(("audio :%s", audio));

            if (strcmp(audio, "AG") == 0)
            {
               EvoCallbackdata.MR_Data.Status = 0x00; //Successfully transfered to AG
            }
            else if (strcmp(audio, "HF") == 0)
            {
               EvoCallbackdata.MR_Data.Status = 0x01; //Successfully transfered to HF
            }
         }
         if (strcmp("AllCallStatus", propertyName) == 0)
         {
            g_mutex_lock(&ConnMapTableMutex);
            ConnMapTableST[connIndex].AllCallStatus = FC_PHONE_EVO_DBUS_UNKNOWN;
            if (g_variant_is_of_type(value, G_VARIANT_TYPE_BYTE))
            {
               ConnMapTableST[connIndex].AllCallStatus = (tU8) g_variant_get_byte(value);
               ETG_TRACE_USR4(("Value of propertyChanged AllCallStatus :%d ", ConnMapTableST[connIndex].AllCallStatus));
            }
            g_mutex_unlock(&ConnMapTableMutex);
         }
         if (strcmp(propertyName, "SCOType") == 0)
         {
            tU8 SCOType;
            g_variant_get(value, "y", &SCOType);
            ETG_TRACE_USR4(("SCOType :%d", SCOType));
            if(SCOType != FC_PHONE_EVO_SERVICE_DISABLED)
            {
               ETG_TRACE_USR4(("Calling UpdateScoConnectionStatus in getHandsfreePropertiesCB"));
               UpdateScoConnectionStatus(SCOType, (GObject*) source_object);
            }
         }
         if (strcmp(propertyName, "EchoCancelingNoiseReduction") == 0)
         {
            gboolean ecnr;
            g_variant_get(value, "b", &ecnr);
            ETG_TRACE_USR4(("ecnr :%d", ecnr));
         }
         if (strcmp(propertyName, "ChldFeature") == 0)
         {
            //Clear old data from the vector.
            if (ChldFeatureList.size() > 0)
            {
               ChldFeatureList.clear();
            }

            GVariantIter iter2;
            gchar * list;
            tU8 enhancedSupport = 0;
            g_variant_iter_init(&iter2, value);
            while (g_variant_iter_loop(&iter2, "s", &list))
            {
               ETG_TRACE_USR4(("ChldFeature feature :%s", list));
               //TODO: Memory should be allocated for all push_backs
               ChldFeatureList.push_back(list);

               if (strcmp(list, "CHLD=0") == 0)
               {
                  enhancedSupport |= 1 << 0;
               }
               if (strcmp(list, "CHLD=1") == 0)
               {
                  enhancedSupport |= 1 << 1;
               }
               if (strcmp(list, "CHLD=1x") == 0)
               {
                  enhancedSupport |= 1 << 2;
               }
               if (strcmp(list, "CHLD=2") == 0)
               {
                  enhancedSupport |= 1 << 3;
               }
               if (strcmp(list, "CHLD=2x") == 0)
               {
                  enhancedSupport |= 1 << 4;
               }
               if (strcmp(list, "CHLD=3") == 0)
               {
                  enhancedSupport |= 1 << 5;
               }
               if (strcmp(list, "CHLD=4") == 0)
               {
                  enhancedSupport |= 1 << 6;
               }
            }

            PhoneStatus_data_temp.u8AgThreeWayCallSuppFeatures = enhancedSupport;
            ETG_TRACE_USR1((" u8AgThreeWayCallSuppFeatures:%d", PhoneStatus_data_temp.u8AgThreeWayCallSuppFeatures));
         }
         if (strcmp(propertyName, "Features") == 0)
         {
            GVariantIter iter2;
            gchar * list;
            tU32 supportFeatures = 0;
            g_variant_iter_init(&iter2, value);
            while (g_variant_iter_loop(&iter2, "s", &list))
            {
               ETG_TRACE_USR4(("Supported feature :%s", list));

               if (strcmp(list, "3way") == 0)
               {
                  supportFeatures |= 1 << 0;
               }
               if (strcmp(list, "ECNR") == 0)
               {
                  supportFeatures |= 1 << 1;
               }
               if (strcmp(list, "Voice-recognition") == 0)
               {
                  supportFeatures |= 1 << 2;
               }
               if (strcmp(list, "Inband_ring") == 0)
               {
                  supportFeatures |= 1 << 3;
               }
               if (strcmp(list, "Attach-voice-tag") == 0)
               {
                  supportFeatures |= 1 << 4;
               }
               if (strcmp(list, "Reject_incoming_call") == 0)
               {
                  supportFeatures |= 1 << 5;
               }
               if (strcmp(list, "Enhanced_call_status") == 0)
               {
                  supportFeatures |= 1 << 6;
               }
               if (strcmp(list, "Enhanced_call_control") == 0)
               {
                  supportFeatures |= 1 << 7;
               }
               if (strcmp(list, "Extended_error_codes") == 0)
               {
                  supportFeatures |= 1 << 8;
               }
               if (strcmp(list, "Codec_negotiation") == 0)
               {
                  supportFeatures |= 1 << 9;
               }
            }
            PhoneStatus_data_temp.u32PhoneSuppFeature = supportFeatures;
            ETG_TRACE_USR1((" u32PhoneSuppFeature:%d", PhoneStatus_data_temp.u32PhoneSuppFeature));
         }
      }

      StorePhoneStatus(connIndex, &PhoneStatus_data_temp);
      PrintPhoneStatus(connIndex);
      ETG_TRACE_USR3((" Sending vPhoneStatus to CCA "));
      DCCA_callbacks_evo->vPhoneStatus(&PhoneStatus_data_temp);
   }
}

/******************************************************************
 Function Name : getHandsfreeProperties

 Description   : Function to get Hands free properties

 Arguments     :
 In            : deviceaddress - Device address
 Out           :
 *******************************************************************/
gboolean getHandsfreeProperties(int connIndex)
{
   ETG_TRACE_USR1(("getHandsfreeProperties entered"));
   ETG_TRACE_USR1(("getHandsfreeProperties connIndex :%d ", connIndex));
   GObject* handsfreeProxy = NULLPTR;

   handsfreeProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
   if (!handsfreeProxy)
   {
      ETG_TRACE_ERR(("ERROR ::HandsfreeProxy is null"));
      return FC_PHONE_DBUS_FAIL;
   }

   handsfree_call_get_properties((Handsfree*) handsfreeProxy, NULLPTR, getHandsfreePropertiesCB, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : vGetModemsCallback

 Description   : Callback function for GetModems

 Arguments     :
 In            : source_object - Manager proxy object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void vGetModemsCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("vGetModemsCallback entered"));
   (tVoid) user_data;
   GError *error = NULLPTR;
   GVariant *out_data = NULLPTR;
   gchar *errmsg = NULLPTR;
   gboolean retval = org_ofono_manager_call_get_modems_finish((OrgOfonoManager*) source_object, &out_data, res, &error);
   if (retval == FALSE || !out_data)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      GVariantIter iter;
      GVariant *obj;
      GVariant *property;
      g_variant_iter_init(&iter, out_data);

      while (g_variant_iter_loop(&iter, "(@o@a{sv})", &obj, &property))
      {
         const gchar *objpath = NULLPTR;
         g_variant_get(obj, "o", &objpath);

         if (!objpath)
         {
            ETG_TRACE_USR4(("objpath is NULL. Hence check for the next object in the loop if present"));
         }
         else
         {
            ETG_TRACE_USR4(("object path :%s ", objpath));

            vAddModemtoModemProxyTable(objpath);

            GVariantIter iter1;
            const gchar *propertyName;
            GVariant *value;
            g_variant_iter_init(&iter1, property);
            while (g_variant_iter_loop(&iter1, "{&sv}", &propertyName, &value))
            {
               ETG_TRACE_USR4(("Propertyname :%s", propertyName));
               if (strcmp(propertyName, "Powered") == 0)
               {
                  gboolean isPowered;
                  g_variant_get(value, "b", &isPowered);
                  ETG_TRACE_USR4(("isPowered :%d", isPowered));
                  if (isPowered)
                  {
                     ETG_TRACE_USR4(("Modem is powered ON"));
                  }
                  else
                  {
                     ETG_TRACE_USR4(("Modem is powered OFF"));
                  }
               }
               if (strcmp(propertyName, "Name") == 0)
               {
                  const gchar* modemName;
                  g_variant_get(value, "s", &modemName);
                  ETG_TRACE_USR4(("ModemName :%s ", modemName));
               }
               if (strcmp(propertyName, "Type") == 0)
               {
                  const gchar* modemType;
                  g_variant_get(value, "s", &modemType);
                  ETG_TRACE_USR4(("ModemType :%s ", modemType));

               }
               else if (strcmp("Interfaces", propertyName) == 0)
               {
                  ETG_TRACE_USR4(("Extracting interface - vGetModemsCallback"));
                  //Clear old data from the vector.
                  if (interfaceList.size() > 0)
                  {
                     interfaceList.clear();
                  }

                  GVariantIter iter2;
                  gchar* interface;
                  g_variant_iter_init(&iter2, value);
                  while (g_variant_iter_loop(&iter2, "s", &interface))
                  {
                     ETG_TRACE_USR4(("Supported interface Name(Property changed)- :%s", interface));
                     //Update new values to vector.
                     interfaceList.push_back(interface);
                  }
               }
            }
         }
      }

      vPrintModemProxyTable();
      ETG_TRACE_USR4(("Exit"));
   }
}

/******************************************************************
 Function Name : vGetModems

 Description   : gets the available modem devices.

 Arguments     :
 In            : managerProxy - Manager proxy object
 Out           :
 *******************************************************************/
void vGetModems()
{
   ETG_TRACE_USR4(("vGetModems entered"));

   if (l_ProxyModemManager)
   {
      org_ofono_manager_call_get_modems(l_ProxyModemManager, NULLPTR, vGetModemsCallback, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::Manager proxy is NULL"));
   }
}

/******************************************************************
 Function Name : networkRegPropertyChangedSignalHandler

 Description   : callback for Network registeration interface property changed signal

 Arguments     :
 In            : object - Network registration proxy object
                 propertyName - property name
                 arg_value - Variable argument value.
 Out           :
 *******************************************************************/
void networkRegPropertyChangedSignalHandler(NetworkRegistration* object, const gchar *propertyName, GVariant *arg_value)
{
   ETG_TRACE_USR4(("networkRegPropertyChangedSignalHandler entered "));

   if (!propertyName)
   {
      ETG_TRACE_USR4(("Received propertyName is NULL. Hence returning."));
      return;
   }

   ETG_TRACE_USR4(("NetworkRegistration - property name %s", propertyName));
   const GVariantType *type;

   int connIndex = _getconnected_slot((GObject*)object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return;
   }

    T_PhoneStatus PhoneStatus_data_temp;
    GetPhoneStatus(connIndex, &PhoneStatus_data_temp);

   type = g_variant_get_type(arg_value);
   if (g_variant_type_is_variant(type))
   {
      GVariant *child;
      child = g_variant_get_child_value(arg_value, 0);

      if (strcmp("Status", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            PhoneStatus_data_temp.u8PhoneService = FC_PHONE_EVO_SERVICE_DISABLED;
            PhoneStatus_data_temp.u8PhoneRoaming = FC_PHONE_EVO_SERVICE_DISABLED;

            //The current registration status of a modem.
            const gchar* modemStatus = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged modemStatus :%s ", modemStatus));
            //Map to phone status
            if (0 == strcmp(modemStatus, "Registered"))
            {
               PhoneStatus_data_temp.u8PhoneService = FC_PHONE_EVO_SERVICE_ENABLED;
            }
            else if( (0 == strcmp(modemStatus, "Unregistered")) || (0 == strcmp(modemStatus, "Searching"))
                  || (0 == strcmp(modemStatus, "Unknown")) || (0 == strcmp(modemStatus, "Denied")))
            {
               PhoneStatus_data_temp.u8PhoneService = FC_PHONE_EVO_SERVICE_DISABLED;
            }
            else if (0 == strcmp(modemStatus, "Roaming"))
            {
               PhoneStatus_data_temp.u8PhoneService = FC_PHONE_EVO_SERVICE_ENABLED; //Phone service is registered.
               PhoneStatus_data_temp.u8PhoneRoaming = FC_PHONE_EVO_SERVICE_ENABLED; //Roaming is enabled.
            }
         }
      }
      else if (strcmp("MobileCountryCode", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* mobileCountryCode = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged mobileCountryCode :%s ", mobileCountryCode));
         }
      }
      else if (strcmp("Name", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* operatorName = g_variant_get_string(child, NULLPTR);
            vStringCopy(PhoneStatus_data_temp.achPhoneOperatorName, operatorName, PHONE_NWOPERATORNAME_LENGTH);
            ETG_TRACE_USR4(("Value of propertyChanged operatorName:%s ", PhoneStatus_data_temp.achPhoneOperatorName));

         }
      }
      else if (strcmp("Strength", propertyName) == 0)
      {
         tU8 signalStrength;
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            signalStrength = g_variant_get_byte(child);
            PhoneStatus_data_temp.u8PhoneSignal = signalStrength;
            ETG_TRACE_USR4(("Value of propertyChanged signalStrength :%d ", PhoneStatus_data_temp.u8PhoneSignal));
         }
      }
      else if (strcmp("CurrentOperator", propertyName) == 0)
      {
         //Use this Object path to create NetworkOperator proxy.
         const gchar* networkOperatorPath;
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            networkOperatorPath = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged modemRevision :%s ", networkOperatorPath));

            if (0 == strcmp(gnetworkOperatorPath, networkOperatorPath))
            {
               //Same Networkobj path is returned. So no need to create proxy for Networkoperator path.
            }
            else
            {
               //Update the value.
               gnetworkOperatorPath = networkOperatorPath;
               //Create Network operator path proxy.
               CreateNetworkOperatorProxy(gnetworkOperatorPath);
            }
         }
      }
   }

   StorePhoneStatus(connIndex, &PhoneStatus_data_temp);
   PrintPhoneStatus(connIndex);
   ETG_TRACE_USR3((" Sending vPhoneStatus to CCA "));
   DCCA_callbacks_evo->vPhoneStatus(&PhoneStatus_data_temp);
}

/******************************************************************
 Function Name : getNetworkRegistrationPropertiesCB

 Description   : callback for Network registeration interface's get property

 Arguments     :
 In            : source_object - Manager proxy object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void getNetworkRegistrationPropertiesCB(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("getNetworkRegistrationPropertiesCB"));
   (tVoid) user_data;
   GError *error = NULLPTR;
   GVariant *out_data;
   gchar *errmsg = NULLPTR;

   gboolean retval = network_registration_call_get_properties_finish((NetworkRegistration*) source_object, &out_data, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      GVariantIter iter;
      const gchar *propertyName;
      GVariant *value;
      g_variant_iter_init(&iter, out_data);

      int connIndex = _getconnected_slot((GObject*)source_object);
      if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
      {
         ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
         return;
      }

       T_PhoneStatus PhoneStatus_data_temp;
       GetPhoneStatus(connIndex, &PhoneStatus_data_temp);

      while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
      {
         ETG_TRACE_USR4(("NetworkRegistration Property name :%s ", propertyName));

         if (strcmp(propertyName, "Status") == 0)
         {
            const gchar * modemStatus;
            PhoneStatus_data_temp.u8PhoneService = FC_PHONE_EVO_SERVICE_DISABLED;
            PhoneStatus_data_temp.u8PhoneRoaming = FC_PHONE_EVO_SERVICE_DISABLED;

            //The current registration status of a modem.
            g_variant_get(value, "s", &modemStatus);
            ETG_TRACE_USR4(("modemStatus :%s ", modemStatus));

            //Map to phone status
            if (0 == strcmp(modemStatus, "Registered"))
            {
               PhoneStatus_data_temp.u8PhoneService = FC_PHONE_EVO_SERVICE_ENABLED;
            }
            else if ((0 == strcmp(modemStatus, "Unregistered")) || (0 == strcmp(modemStatus, "Searching"))
                  || (0 == strcmp(modemStatus, "Unknown")) || (0 == strcmp(modemStatus, "Denied")))
            {
               PhoneStatus_data_temp.u8PhoneService = FC_PHONE_EVO_SERVICE_DISABLED;
            }
            else if (0 == strcmp(modemStatus, "Roaming"))
            {
               PhoneStatus_data_temp.u8PhoneService = FC_PHONE_EVO_SERVICE_ENABLED; //Phone service is registered.
               PhoneStatus_data_temp.u8PhoneRoaming = FC_PHONE_EVO_SERVICE_ENABLED; //Roaming is enabled.
            }
         }
         if (strcmp(propertyName, "MobileCountryCode") == 0)
         {
            const gchar* mobileCountryCode;
            g_variant_get(value, "s", &mobileCountryCode);
            ETG_TRACE_USR4(("mobileCountryCode :%s ", mobileCountryCode));
         }
         if (strcmp(propertyName, "Name") == 0)
         {
            const gchar* operatorName;
            g_variant_get(value, "s", &operatorName);
            vStringCopy(PhoneStatus_data_temp.achPhoneOperatorName, operatorName, PHONE_NWOPERATORNAME_LENGTH);
            ETG_TRACE_USR4(("operatorName :%s ", PhoneStatus_data_temp.achPhoneOperatorName));
         }
         if (strcmp(propertyName, "Strength") == 0)
         {
            tU8 signalStrength;
            //g_variant_get (value, "y", &signalStrength);
            signalStrength = g_variant_get_byte(value);
            PhoneStatus_data_temp.u8PhoneSignal = signalStrength;
            ETG_TRACE_USR4(("signalStrength :%d ", PhoneStatus_data_temp.u8PhoneSignal));
         }
         if (strcmp(propertyName, "CurrentOperator") == 0)
         {
            //Use this Object path to create NetworkOperator proxy.
            const gchar* networkOperatorPath;
            if (g_variant_is_of_type(value, G_VARIANT_TYPE_STRING))
            {
               networkOperatorPath = g_variant_get_string(value, NULLPTR);
               ETG_TRACE_USR4(("Value of propertyChanged modemRevision :%s ", networkOperatorPath));

               if (0 == strcmp(gnetworkOperatorPath, networkOperatorPath))
               {
                  //Same Networkobj path is returned.So no need to create proxy for Networkoperator path.
               }
               else
               {
                  //Update the value.
                  gnetworkOperatorPath = networkOperatorPath;
                  //Create Network operator path proxy.
                  CreateNetworkOperatorProxy(gnetworkOperatorPath);
               }
            }
         }
      }

      StorePhoneStatus(connIndex,&PhoneStatus_data_temp);
      PrintPhoneStatus(connIndex);
      ETG_TRACE_USR3((" Sending vPhoneStatus to CCA "));
      DCCA_callbacks_evo->vPhoneStatus(&PhoneStatus_data_temp);
   }
}

/******************************************************************
 Function Name : getNetworkRegistrationProperties

 Description   : gets network registration properties.

 Arguments     :
 In            : deviceaddress - Device address
 Out           :
 *******************************************************************/
gboolean getNetworkRegistrationProperties(int connIndex)
{
   ETG_TRACE_USR1(("getNetworkRegistrationProperties entered"));
   ETG_TRACE_USR4(("getNetworkRegistrationProperties connIndex :%d", connIndex));

   GObject* networkRegistrationProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_NETWORK_REGISTRATION_INTERFACE);

   if (networkRegistrationProxy)
   {
      ETG_TRACE_USR4(("calling network_registration_call_get_properties"));
      network_registration_call_get_properties((NetworkRegistration*) networkRegistrationProxy, NULLPTR, getNetworkRegistrationPropertiesCB, NULLPTR);
      return FC_PHONE_DBUS_SUCCESS;
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::NetworkRegistrationProxy is null"));
      return FC_PHONE_DBUS_FAIL;
   }
}

/******************************************************************
 Function Name : NetworkRegistrationProxyCB

 Description   : Call back for Network registration proxy new

 Arguments     :
 In            : GObject - network registration object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void NetworkRegistrationProxyCB(GObject* source_object, GAsyncResult* res, gpointer user_data)
{
   ETG_TRACE_USR1(("NetworkRegistrationProxyCB entered"));

   (tVoid)user_data;
   int connected_slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;
   GError* error = NULLPTR;
   GObject* networkRegistrartionProxy = NULLPTR;

   networkRegistrartionProxy = (GObject*) network_registration_proxy_new_finish(res, &error);

   if (error)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::Couldn't create the networkRegistrartion proxy object %s", error->message));
      g_error_free(error);
      return;
   }
   if (!networkRegistrartionProxy)
   {
      ETG_TRACE_ERR(("ERROR ::NetworkRegistrartionProxy is NULL"));
      return;
   }
   else
   {
      ETG_TRACE_USR4(("CreateNetworkRegistration proxy created successfully"));
   }

   connected_slot = _getconnected_slot(source_object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connected_slot)
   {
      ETG_TRACE_USR4(("connected_slot is INVALID. Hence returning"));
      g_object_unref(networkRegistrartionProxy);
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);
   //update the status as slot used
   ConnMapTableST[connected_slot].proxy_common[FC_PHONE_DBUS_PROXY_EVO_NETWORK_REGISTRATION_INTERFACE] = networkRegistrartionProxy;
   //_printSlotStatusEvo();
   g_mutex_unlock(&ConnMapTableMutex);

}

/******************************************************************
 Function Name : CreateNetworkRegistrationProxy

 Description   : Creates network registration proxy.

 Arguments     :
 In            : handsfreeObjPath - Device address
 Out           :
 *******************************************************************/
void CreateNetworkRegistrationProxy(const char* handsfreeObjPath)
{
   ETG_TRACE_USR1(("CreateNetworkRegistrationProxy entered"));
   ETG_TRACE_USR4(("HandsfreeObjpath :%s", handsfreeObjPath));

   //Create networkRegistrartion proxy
   network_registration_proxy_new(
         gdbus,
         G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
         BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
         handsfreeObjPath,//OBJECT PATH
         NULLPTR,//GCancellable
         NetworkRegistrationProxyCB,
         NULLPTR);
}

/******************************************************************
 Function Name : networkOperatorPropertyChangedSignalHandler

 Description   : Callback for network operator property changed.

 Arguments     :
 In            : object - Network operator object
                 propertyName - name of the property
                 arg_value - variable argument value
 Out           :
 *******************************************************************/
void networkOperatorPropertyChangedSignalHandler(NetworkOperator *object, const gchar *propertyName, GVariant *arg_value)
{
   ETG_TRACE_USR1(("networkOperatorPropertyChangedSignalHandler entered"));
   ETG_TRACE_USR4(("networkOperatorPropertyChangedSignalHandler - property name :%s", propertyName));
   (tVoid) object;
   const GVariantType *type;
   type = g_variant_get_type(arg_value);

   if (g_variant_type_is_variant(type))
   {
      GVariant *child;
      child = g_variant_get_child_value(arg_value, 0);

      if (strcmp("Status", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            //The current registration status of a modem.
            const gchar* operatorStatus = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged operatorStatus :%s ", operatorStatus));
         }
      }
      else if (strcmp("Name", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            //use the Name property of the NetworkRegistration interface instead.
            const gchar* operaterName = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged operaterName level :%s ", operaterName));
         }
      }
   }
}

/******************************************************************
 Function Name : getNetworkOperatorPropertiesCB

 Description   : Callback for get network operator properties.

 Arguments     :
 In            : source_object - Network operator object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void getNetworkOperatorPropertiesCB(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("getNetworkOperatorPropertiesCB entered"));
   (tVoid) user_data;
   GError *error = NULLPTR;
   GVariant *out_data;
   gchar *errmsg = NULLPTR;
   gboolean retval = network_operator_call_get_properties_finish((NetworkOperator*) source_object, &out_data, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      GVariantIter iter;
      const gchar *propertyName;
      GVariant *value;
      g_variant_iter_init(&iter, out_data);
      while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
      {
         ETG_TRACE_USR4(("NetworkOperator Property name :%s ", propertyName));

         if (strcmp(propertyName, "Status") == 0)
         {
            const gchar * modemStatus;
            g_variant_get(value, "s", &modemStatus);
            ETG_TRACE_USR4(("modemStatus :%s ", modemStatus));
         }
         //use the Name property of the NetworkRegistration interface instead.
         if (strcmp(propertyName, "Name") == 0)
         {
            const gchar* operatorName;
            g_variant_get(value, "s", &operatorName);
            ETG_TRACE_USR4(("operatorName :%s ", operatorName));
         }
      }
   }
}

/******************************************************************
 Function Name : getNetworkOperatorProperties

 Description   : gets network operator properties.

 Arguments     :
 In            : deviceaddress - Device address
 Out           :
 *******************************************************************/
void getNetworkOperatorProperties(int connIndex)
{
   ETG_TRACE_USR1(("getNetworkOperatorProperties entered"));

   GObject* networkOperatorProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_NETWORK_OPERATOR_INTERFACE);
   if (networkOperatorProxy)
   {
      network_operator_call_get_properties((NetworkOperator*) networkOperatorProxy, NULLPTR, getNetworkOperatorPropertiesCB, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::NetworkOperatorProxy is not null"));
   }
}

/******************************************************************
 Function Name : NetworkOperatorProxyCB

 Description   : Call back for Network operator proxy new

 Arguments     :
 In            : GObject - Network operator object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void NetworkOperatorProxyCB(GObject* source_object, GAsyncResult* res, gpointer user_data)
{
   ETG_TRACE_USR1(("NetworkOperatorProxyCB entered"));

   (tVoid)user_data;
   int connected_slot = FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT;
   GError *error = NULLPTR;
   GObject* networkOperatorProxy = NULLPTR;

   networkOperatorProxy = (GObject*) network_operator_proxy_new_finish(res,&error);

   if (error)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::Couldn't create the networkOperatorProxy proxy object %s", error->message));
      g_error_free(error);
      return;
   }
   if (!networkOperatorProxy)
    {
       ETG_TRACE_FATAL(("FATAL_ERROR ::networkOperatorProxy: Proxy creation failed"));
       return;
    }
    else
    {
       ETG_TRACE_USR4(("networkOperatorProxy created successfully"));
    }

   connected_slot = _getconnected_slot(source_object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connected_slot)
   {
      ETG_TRACE_USR4(("connected_slot is INVALID. Hence returning"));
      g_object_unref(networkOperatorProxy);
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);
   //update the status as slot used
   ConnMapTableST[connected_slot].proxy_common[FC_PHONE_DBUS_PROXY_EVO_NETWORK_OPERATOR_INTERFACE] = networkOperatorProxy;
   g_mutex_unlock(&ConnMapTableMutex);

   g_signal_connect(networkOperatorProxy,"property-changed", G_CALLBACK(networkOperatorPropertyChangedSignalHandler), NULLPTR);
}

/******************************************************************
 Function Name : CreateNetworkOperatorProxy

 Description   : Creates network operator proxy.

 Arguments     :
 In            : networkOperatorObjpath - Operator path of the network.
 Out           :
 *******************************************************************/
void CreateNetworkOperatorProxy(const char* networkOperatorObjpath)
{
   ETG_TRACE_USR1(("CreateNetworkOperatorProxy entered"));

   //Create networkOperatorProxy proxy
   network_operator_proxy_new(
         gdbus,
         G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
         BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
         networkOperatorObjpath,//OBJECT PATH
         NULLPTR,//GCancellable
         NetworkOperatorProxyCB,
         NULLPTR);
}

/******************************************************************
 Function Name : callVolumePropertyChangedSignalHandler

 Description   : Callback for call volume property changed

 Arguments     :
 In            : object - Call volume object.
                 propertyName - name of the property
                 arg_value - variable argument values.
 Out           :
 *******************************************************************/
void callVolumePropertyChangedSignalHandler(CallVolume *object, const gchar *propertyName, GVariant *arg_value)
{
   ETG_TRACE_USR1(("callVolumePropertyChangedSignalHandler"));
   ETG_TRACE_USR4(("CallVolume - property name :%s", propertyName));
   (tVoid) object;
   const GVariantType *type;
   type = g_variant_get_type(arg_value);

   if (g_variant_type_is_variant(type))
   {
      GVariant *child;
      child = g_variant_get_child_value(arg_value, 0);

      if (strcmp("SpeakerVolume", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            //The current registration status of a modem.
            tU8 speakerVolume = g_variant_get_byte(child);
            ETG_TRACE_USR4(("Value of propertyChanged speakerVolume - : %d", speakerVolume));
         }
      }
      else if (strcmp("MicrophoneVolume", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            tU8 microphoneVolume = g_variant_get_byte(child);
            ETG_TRACE_USR4(("Value of propertyChanged microphoneVolume level - : %d", microphoneVolume));
         }
      }
   }
}

/******************************************************************
 Function Name : getCallVolumePropertiesCB

 Description   : Callback for call volume property changed

 Arguments     :
 In            : source_object - Call volume object.
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void getCallVolumePropertiesCB(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("getCallVolumePropertiesCB entered"));
   (tVoid) user_data;
   GError *error = NULLPTR;
   GVariant *out_data;
   gchar *errmsg = NULLPTR;

   gboolean retval = call_volume_call_get_properties_finish((CallVolume*) source_object, &out_data, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      GVariantIter iter;
      const gchar *propertyName;
      GVariant *value;
      g_variant_iter_init(&iter, out_data);
      while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
      {
         ETG_TRACE_USR4(("CallVolume Property name :%s", propertyName));

         if (strcmp(propertyName, "SpeakerVolume") == 0)
         {
            tU8 speakerVolume;
            g_variant_get(value, "y", &speakerVolume);
            ETG_TRACE_USR4(("speakerVolume :%d", speakerVolume));
         }
         if (strcmp(propertyName, "MicrophoneVolume") == 0)
         {
            tU8 microphoneVolume;
            g_variant_get(value, "y", &microphoneVolume);
            ETG_TRACE_USR4(("microphoneVolume :%d", microphoneVolume));
         }
      }
   }
}

/******************************************************************
 Function Name : getCallVolumeProperties

 Description   : gets the call volume properties.

 Arguments     :
 In            : deviceaddress - Device Address.
 Out           :
 *******************************************************************/
void getCallVolumeProperties(int connIndex)
{
   ETG_TRACE_USR1(("getCallVolumeProperties entered"));

   GObject* callVolumeProxy = Get_proxy_from_ConnIndex_Evo(connIndex,FC_PHONE_DBUS_PROXY_EVO_CALL_VOLUME_INTERFACE);
   if(callVolumeProxy)
   {
      call_volume_call_get_properties((CallVolume*)callVolumeProxy,NULLPTR,getCallVolumePropertiesCB,NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR( ("ERROR ::CallVolume proxy is null"));
   }
}

/******************************************************************
Function Name : _EvoCallVolumeSetSpeakerVolumeCompletionCallback

Description   :   Call back of EvoAudoioTransfer

Arguments     :
In            : source_object - Call volume object
                res -
                user_data - user data if any
Out           :
 *******************************************************************/
static void _EvoCallVolumeSetSpeakerVolumeCompletionCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoCallVolumeSetSpeakerVolumeCompletionCallback Entered "));
   (tVoid) user_data;
   GError *error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = call_volume_call_set_property_finish((CallVolume*) source_object, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_EvoCallVolumeSetSpeakerVolumeCompletionCallback Exit "));
   }
}

/******************************************************************
Function Name : setPropertycallVolume

Description   : sets the call volume property

Arguments     :
In            : deviceaddress - Device address
                speakerVolume - Speaker volume
Out           :
 *******************************************************************/
void setPropertycallVolume(int connIndex, int speakerVolume)
{
   ETG_TRACE_USR1(("setPropertycallVolume entered "));
   GObject* callVolumeProxy = Get_proxy_from_ConnIndex_Evo(connIndex,FC_PHONE_DBUS_PROXY_EVO_CALL_VOLUME_INTERFACE);
   if(callVolumeProxy)
   {
      GVariant *varspeakerVolume;
      varspeakerVolume = g_variant_new ("y", speakerVolume);//Type is string in introspection file.

      call_volume_call_set_property((CallVolume*)callVolumeProxy,(const gchar*)"SpeakerVolume",g_variant_new("v", varspeakerVolume), NULLPTR,_EvoCallVolumeSetSpeakerVolumeCompletionCallback,NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::CallVolume proxy is NULL"));
   }
}

/******************************************************************
Function Name :_EvoCallVolumeSetMicVolumeCompletionCallback

Description   : Call back of set Mic volume

Arguments     :
In            : source_object - call volume object
                res -
                user_data - User data if any
Out           :
 *******************************************************************/
static void _EvoCallVolumeSetMicVolumeCompletionCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoCallVolumeSetMicVolumeCompletionCallback entered "));
   (tVoid) user_data ;
   GError *error = NULLPTR;

   gboolean retval = call_volume_call_set_property_finish((CallVolume*)source_object,res,&error);
   if (retval==FALSE)
   {
      if(NULLPTR != error)
      {
         gchar *errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
         g_free(errmsg);
      }
   }
   else
   {
      ETG_TRACE_USR1(("_EvoCallVolumeSetMicVolumeCompletionCallback Exit"));
   }
}

/******************************************************************
Function Name :setPropertyMiceVolume

Description   : Sets Mic volume property

Arguments     :
In            : deviceaddress - device address
                speakerVolume - speaker volume value
Out           :
 *******************************************************************/
void setPropertyMiceVolume(int connIndex, int speakerVolume)
{
   ETG_TRACE_USR1(("setPropertyMiceVolume entered "));

   GObject* callVolumeProxy = Get_proxy_from_ConnIndex_Evo(connIndex,FC_PHONE_DBUS_PROXY_EVO_CALL_VOLUME_INTERFACE);
   if(callVolumeProxy)
   {
      GVariant *varMicVolume = NULLPTR;
      varMicVolume = g_variant_new ("y", speakerVolume);//Type is string in introspection file.

      call_volume_call_set_property((CallVolume*)callVolumeProxy,(const gchar*)"MicrophoneVolume",g_variant_new("v", varMicVolume), NULLPTR,_EvoCallVolumeSetMicVolumeCompletionCallback,NULLPTR);
   }
   else
   {
      ETG_TRACE_USR4( ("CallVolume proxy is NULL"));
   }
}

/******************************************************************
 Function Name : CallVolumeProxyCB

 Description   : Call back for call volume proxy new

 Arguments     :
 In            : GObject - Call volume object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void CallVolumeProxyCB(GObject* source_object, GAsyncResult* res, gpointer user_data)
{
   ETG_TRACE_USR1(("CallVolumeProxyCB entered"));

   (tVoid)user_data;
   int connected_slot = FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT;
   GError *error = NULLPTR;
   GObject* callVolumeProxy = NULLPTR;

   callVolumeProxy = (GObject*) call_volume_proxy_new_finish(res,&error);

   if (error)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::Couldn't create the CreateCallVolumeProxy proxy object %s", error->message));
      g_error_free(error);
   }
   if (!callVolumeProxy)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::CallVolumeProxy: Proxy creation failed"));
      return;
   }
   else
   {
      ETG_TRACE_USR4(("callVolume proxy created successfully"));
   }


   connected_slot = _getconnected_slot(source_object);

   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connected_slot)
   {
      ETG_TRACE_USR4(("connected_slot is INVALID. Hence returning"));
      g_object_unref(callVolumeProxy);
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);
   //update the status as slot used
   ConnMapTableST[connected_slot].proxy_common[FC_PHONE_DBUS_PROXY_EVO_CALL_VOLUME_INTERFACE] = callVolumeProxy;
   //_printSlotStatusEvo();
   g_mutex_unlock(&ConnMapTableMutex);

}

/******************************************************************
 Function Name :CreateCallVolumeProxy

 Description   : creates call volume proxy

 Arguments     :
 In            : handsfreeObjPath - hands free object path.
 Out           :
 *******************************************************************/
void CreateCallVolumeProxy(const char* handsfreeObjPath)
{
   ETG_TRACE_USR1(("CreateCallVolumeProxy entered"));
   ETG_TRACE_USR4(("Create CreateCallVolumeProxy proxy %s", handsfreeObjPath));

   //Create networkOperatorProxy proxy
   call_volume_proxy_new(
         gdbus,
         G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
         BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
         handsfreeObjPath,//OBJECT PATH
         NULLPTR,//GCancellable
         CallVolumeProxyCB,
         NULLPTR);
}

/******************************************************************
 Function Name : _EvoModemSetPoweredCompletionCallback

 Description   : Call back for modem set powered mrthod

 Arguments     :
 In            : source_object - Modem object instance
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoModemSetPoweredCompletionCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoModemSetPoweredCompletionCallback entered "));
   (tVoid) user_data;
   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;
   gboolean retval = modem_call_set_property_finish((Modem*) source_object, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_EvoModemSetPoweredCompletionCallback Exit "));
   }
}

/******************************************************************
 Function Name : setPropertyPowered

 Description   : Sets the property powered

 Arguments     :
 In            : deviceaddress - Device address
                 bpowered - property value
 Out           :
 *******************************************************************/
void setPropertyPowered(int connIndex, bool bpowered)
{
   ETG_TRACE_USR1(("setPropertyPowered entered "));

   GObject* modemProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_MODEM_INTERFACE);
   if (modemProxy)
   {
      GVariant *varPowered;
      varPowered = g_variant_new("b", bpowered);//Type is string in introspection file.

      modem_call_set_property((Modem*) modemProxy, (const gchar*) "Powered", g_variant_new("v", varPowered), NULLPTR, _EvoModemSetPoweredCompletionCallback, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::ModemProxy proxy is NULL"));
   }
}

/******************************************************************
 Function Name : _EvoModemAcceptSCOConnectionRequestCallback

 Description   : Call back of EvoAudoioTransfer

 Arguments     :
 In            : source_object - Modem object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoModemAcceptSCOConnectionRequestCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoModemAcceptSCOConnectionRequestCallback entered "));
   (tVoid) user_data;
   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;
   gboolean retval = modem_call_accept_scoconnect_finish((Modem*) source_object, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_EvoModemAcceptSCOConnectionRequestCallback exit "));
   }
}

/******************************************************************
 Function Name : acceptSCOConnectRequest

 Description   : accepts the incoming sco request

 Arguments     :
 In            : modemProxy - Modem object
 Out           :
 *******************************************************************/
void acceptSCOConnectRequest(char* Address)
{
   ETG_TRACE_USR1(("acceptSCOConnectRequest Entered with Address: %s", Address));

   char activeDeviceAddressEvo[PHONE_OBJECT_PATH] = {'\0'};

   getRemoteDeviceAddressFromDevicePath(Address, activeDeviceAddressEvo);
   ETG_TRACE_USR4((" getConnIndexFromDeviceAdd device address %s", activeDeviceAddressEvo));

   Modem *modemProxy = GetModemProxyFromModemProxyTable(activeDeviceAddressEvo);
   if (modemProxy)
   {
      //Acknowledgement for scoConnectRequest signal.
      modem_call_accept_scoconnect(modemProxy, NULLPTR, _EvoModemAcceptSCOConnectionRequestCallback, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::modemProxy is NULL"));
   }
}

/******************************************************************
 Function Name : getModemPropertiesCB

 Description   : Call back for get modem properties.

 Arguments     :
 In            : source_object - Modem object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void getModemPropertiesCB(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("getModemPropertiesCB Entered"));
   (tVoid) user_data;
   GError *error = NULLPTR;
   GVariant *out_data;
   gchar *errmsg = NULLPTR;

   gboolean retval = modem_call_get_properties_finish((Modem*) source_object, &out_data, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);

   }
   else
   {
      GVariantIter iter;
      const gchar *propertyName;
      GVariant *value;
      g_variant_iter_init(&iter, out_data);
      while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
      {
         ETG_TRACE_USR4(("PrpertyName :%s", propertyName));
         if (strcmp(propertyName, "Powered") == 0)
         {
            gboolean powered;
            g_variant_get(value, "b", &powered);
            ETG_TRACE_USR4(("Powered :%d ", powered));
         }
         if (strcmp(propertyName, "Name") == 0)
         {
            const gchar* modemName;
            g_variant_get(value, "s", &modemName);
            ETG_TRACE_USR4(("ModemName :%s ", modemName));
         }
         if (strcmp(propertyName, "Type") == 0)
         {
            const gchar* modemType;
            g_variant_get(value, "s", &modemType);
            ETG_TRACE_USR4(("ModemType :%s ", modemType));

         }
         else if (strcmp("Interfaces", propertyName) == 0)
         {
            ETG_TRACE_USR4(("Extracting interface - getModemPropertiesCB"));
            //Clear old data from the vector.
            if (interfaceList.size() > 0)
            {
               interfaceList.clear();
            }

            GVariantIter iter2;
            gchar* interface;
            g_variant_iter_init(&iter2, value);
            while (g_variant_iter_loop(&iter2, "s", &interface))
            {
               ETG_TRACE_USR4(("Supported interface Name(Property changed)- :%s", interface));
               //Update new values to vector.
               interfaceList.push_back(interface);
            }
         }
      }
   }
}

/******************************************************************
 Function Name : getModemProperties

 Description   : Gets the modem properties

 Arguments     :
 In            : remoteproxy - Modem proxy
 Out           :
 *******************************************************************/
void getModemProperties(Modem* remoteproxy)
{
   ETG_TRACE_USR1(("getModemProperties entered"));
   if (remoteproxy)
   {
      modem_call_get_properties(remoteproxy, NULLPTR, getModemPropertiesCB, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::Modem proxy is NULL"));
   }
}

/******************************************************************
 Function Name : modemPropertyChangedSignalHandler

 Description   : Call back for modem property changed signal

 Arguments     :
 In            : object - Modem object
                 propertyName - name of the property
                 arg_value - variable argument value
 Out           :
 *******************************************************************/
static void modemPropertyChangedSignalHandler(Modem *object, const gchar *propertyName, GVariant *arg_value)
{
   ETG_TRACE_USR1(("modemPropertyChangedSignalHandler entered"));
   ETG_TRACE_USR4(("Modem - property name :%s", propertyName));
   (tVoid) object;
   const GVariantType *type;
   type = g_variant_get_type(arg_value);

   if (g_variant_type_is_variant(type))
   {
      GVariant *child;
      child = g_variant_get_child_value(arg_value, 0);

      char activeDeviceAddressEvo[PHONE_OBJECT_PATH] = {0};
      const gchar *hfpObjpath = g_dbus_proxy_get_object_path((GDBusProxy*) object);
      ETG_TRACE_USR4(( "Modem Object path :%s",hfpObjpath));

      getDeviceAddressFromHfpPath(activeDeviceAddressEvo, hfpObjpath);
      ETG_TRACE_USR4(( "activeDeviceAddress :%s", activeDeviceAddressEvo));

      if (strcmp("Powered", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BOOLEAN))
         {
            gboolean bpowered = g_variant_get_boolean(child);
            ETG_TRACE_USR4(("Value of propertyChanged powered - : %d ", bpowered));
         }
      }
      else if (strcmp("Name", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* modemFriendlyName = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged modemFriendlyName - : %s ", modemFriendlyName));
         }
      }
      else if (strcmp("Type", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* modemType = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged modemType - : %s ", modemType));
         }
      }
      else if (strcmp("Manufacturer", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* modemManufacturer = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged modemManufacturer level - : %s ", modemManufacturer));
         }
      }
      else if (strcmp("Model", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* modemModel = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged modemModel - : %s ", modemModel));
         }
      }
      else if (strcmp("Revision", propertyName) == 0)
      {
         const gchar* modemRevision;
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            modemRevision = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged modemRevision - : %s ", modemRevision));
         }
      }
      else if (strcmp("Serial", propertyName) == 0)
      {
         const gchar* modemSerialNumber;
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            modemSerialNumber = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged modemSerialNumber - : %s ", modemSerialNumber));
         }
      }
      //Array of string type
      else if (strcmp("Interfaces", propertyName) == 0)
      {
         ETG_TRACE_USR4(("Extracting interface"));

         //Clear old data from the vector.
         if (interfaceList.size() > 0)
         {
            interfaceList.clear();
         }

         GVariantIter iter;
         gchar* interface;
         g_variant_iter_init(&iter, child);
         while (g_variant_iter_loop(&iter, "s", &interface))
         {
            ETG_TRACE_USR4(("Supported interface Name(Property changed)- :%s", interface));
            //Update new values to vector.
            interfaceList.push_back(interface);
         }
      }
   }
}

/******************************************************************
 Function Name : scoConnectSignalHandler

 Description   : Call back for SCO connect request

 Arguments     :
 In            : modemProxy - Modem proxy
                 arg_type - SCO connection type
 Out           :
 *******************************************************************/
static void scoConnectSignalHandler(Modem *modemProxy, guchar arg_type)
{
   ETG_TRACE_USR1(("scoConnectSignalHandler entered"));
   (tVoid) arg_type;

   //In SCO connect request 0x00 - NBS&S1~S3, 0x01 - WBS and 0x02 - NBS&S4
   //But in SCOType property 0x01 - NBS&S1~S3, 0x02 - WBS and 0x03 - NBS&S4

   //Call acceptSCOConnectRequest to acknowledge the SCOConnection request signal.
   //This accept should be called within 1 second.

   char *pDevAdd = GetDevAddressFromModemProxyTable(modemProxy);
   if (pDevAdd)
   {
      ETG_TRACE_USR1(("pDevAdd: %s", pDevAdd));

      char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};
      getDevicePathFromRemoteDeviceAddress(pDevAdd, remoteDevicePath);
      vStringCopy(EvoCallbackdata.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
      ETG_TRACE_USR3(("EvoCallbackdata.BD_Address :%s", EvoCallbackdata.BD_Address));

      ETG_TRACE_USR3(("Sending SCO connect Requisition"));
      DCCA_callbacks_evo->vSCOConnectRequest(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : ModemProxyCB

 Description   : Call back for Modem proxy new

 Arguments     :
 In            : GObject - Modem object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void ModemProxyCB(GObject* source_object, GAsyncResult* res, gpointer user_data)
{
   ETG_TRACE_USR1(("ModemProxyCB entered"));

   (tVoid)user_data;
   int connected_slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;
   GError * error = NULLPTR;
   GObject* modemProxy = NULLPTR;
   char p_devAdd[PHONE_TEMP_DEV_ADDRESS] = {'\0'};
   char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};

   modemProxy = (GObject*) modem_proxy_new_finish(res,&error);

   if (error)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::Couldn't create the modemProxy proxy object %s", error->message));
      g_error_free(error);
      return;
   }
   if (!modemProxy)
   {
      ETG_TRACE_ERR(("ERROR ::ModemProxy proxy is NULL"));
      return;
   }
   else
   {
      ETG_TRACE_USR4(("ModemProxy proxy created successfully"));
   }



   connected_slot = _getconnected_slot(source_object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connected_slot)
   {
      ETG_TRACE_USR4(("connected_slot is INVALID. Hence returning"));
      g_object_unref(modemProxy);
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);

   ConnMapTableST[connected_slot].proxy_common[FC_PHONE_DBUS_PROXY_EVO_MODEM_INTERFACE] = modemProxy;
   vStringCopy(p_devAdd, ConnMapTableST[connected_slot].DevAddr,PHONE_CONNMAP_DEV_ADDRESS);
   ConnMapTableST[connected_slot].en_conn_status = FC_PHONE_DEVICE_CONNECTED;

   //_printSlotStatusEvo();
   g_mutex_unlock(&ConnMapTableMutex);

   //Converting dev address into dev path
   getDevicePathFromRemoteDeviceAddress(p_devAdd, remoteDevicePath);

   //Calling modem get properties
   getModemProperties((Modem*) modemProxy);

   //Register signal Handlers properties.
   registerSiganlHandler(connected_slot);

   /*This method will update SLCstatus.
   This must be called when DBUS client is ready with all proxy creation and property registration.
   Should be placed in last proxy callback.*/
   UpdateSLCStatus(remoteDevicePath,TRUE);

}

/******************************************************************
 Function Name : CreateModemProxy

 Description   : creates Modem proxy

 Arguments     :
 In            : handsfreeObjPath - Hands free object path
 Out           :
 *******************************************************************/
void CreateModemProxy(const char* handsfreeObjPath)
{
   ETG_TRACE_USR1(("CreateModemProxy entered"));
   ETG_TRACE_USR4(("CreateModemProxy = %s", handsfreeObjPath));


   //Create Modem proxy
   modem_proxy_new(
         gdbus,
         G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
         BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
         handsfreeObjPath,//OBJECT PATH
         NULLPTR,//GCancellable
         ModemProxyCB,
         NULLPTR);
}

//Siri Interface
/******************************************************************
 Function Name : siriPropertyChangedSignalHandler

 Description   : Call back for SIRI property changed

 Arguments     :
 In            : object - SIRI object
                 propertyName - Property name
                 arg_value - Variable argument value
 Out           :
 *******************************************************************/
void siriPropertyChangedSignalHandler(EvoSiri *object, const gchar *propertyName, GVariant *arg_value)
{
   ETG_TRACE_USR1(("siriPropertyChangedSignalHandler entered"));
   const GVariantType *type;
   type = g_variant_get_type(arg_value);

#ifdef PHONE_SIRI
   //Updating Device address into SIRI static call back data.
   vFill_Siri_evo((GObject *)object);
#endif
   (tVoid) object;
   ETG_TRACE_USR4(("SIRI - property name :%s", propertyName));
   if (g_variant_type_is_variant(type))
   {
      GVariant *child;
      child = g_variant_get_child_value(arg_value, 0);

      if (strcmp("Status", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            tU8 siriStatus = g_variant_get_byte(child);
#ifdef PHONE_SIRI
            SiriStatus_value_evo.m_u8SiriStatus = siriStatus;
            DCCA_callbacks_evo->vSiriStatusInd(&SiriStatus_value_evo);
#endif
            ETG_TRACE_USR4(("Value of propertyChanged siriStatus :%d ", siriStatus));
         }
      }
      else if (strcmp("NR", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            tU8 NRStatus = g_variant_get_byte(child);
            ETG_TRACE_USR4(("Value of propertyChanged NRStatus :%d ", NRStatus));
#ifdef PHONE_SIRI
            SiriNRStatus_data_evo.m_u8SiriNrStatus = NRStatus;
            //TODO: check is this update is needed. vSiriNRStatusInd
            //DCCA_callbacks_evo->vSiriNRStatusInd(&SiriNRStatus_data_evo);
#endif
         }
      }
      else if (strcmp("EyesFreeMode", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BOOLEAN))
         {
            gboolean eyesFreeMode = g_variant_get_boolean(child);
            ETG_TRACE_USR4(("Value of propertyChanged eyesFreeMode :%d ", eyesFreeMode));
         }
      }
      else if (strcmp("SupportedType", propertyName) == 0)
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_UINT16))
         {
            tS16 supportedType = g_variant_get_int16(child);
            ETG_TRACE_USR4(("Value of propertyChanged supportedType level :%d ", supportedType));
         }
      }
   }
}

/******************************************************************
 Function Name : _EvoSiriSetEyesFreeModeCompletionCallback

 Description   : Call back of Set Eyes free mode

 Arguments     :
 In            : source_object - Siri object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoSiriSetEyesFreeModeCompletionCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoSiriSetEyesFreeModeCompletionCallback Entered "));
   (tVoid) user_data;
   GError * error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = evo_siri_call_set_property_finish((EvoSiri*) source_object, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_EvoSiriSetEyesFreeModeCompletionCallback OK "));
   }
}

/******************************************************************
 Function Name : _EvoSiriSetEyesFreeModeCompletionCallback

 Description   : Sets Eyes free mode

 Arguments     :
 In            : deviceaddress - Device Address
                 eyesFreeMode - eyes free mode input
 Out           :
 *******************************************************************/
gboolean setPropertyEyesFreeMode(int connIndex, const guchar eyesFreeMode)
{
   ETG_TRACE_USR1(("setPropertyEyesFreeMode entred"));
   ETG_TRACE_USR4(("EvoSetSiriEfm.. activationIndex %d", eyesFreeMode));

   GObject* siriProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_SIRI_INTERFACE);
   if (siriProxy)
   {
      GVariant *vareyesFreeMode;
      vareyesFreeMode = g_variant_new("b",eyesFreeMode);//Type is string in introspection file.
      evo_siri_call_set_property((EvoSiri*) siriProxy, (const gchar*) "EyesFreeMode", g_variant_new("v", vareyesFreeMode), NULLPTR, _EvoSiriSetEyesFreeModeCompletionCallback, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::siriProxy is NULL"));
   }

   return FALSE;
}

/******************************************************************
 Function Name : _EvoSiriEnableRequestCallback

 Description   : Callback function for Siri enable request.

 Arguments     :
 In            : source_object - Siri Object.
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoSiriEnableRequestCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoSiriEnableRequestCallback entered "));
   (tVoid) user_data;
   GError * error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = evo_siri_call_enable_finish((EvoSiri*) source_object,res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
         g_free(errmsg);
      }

#ifdef PHONE_SIRI
      int connIndex = _getconnected_slot((GObject*) source_object);
      if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
      {
         ETG_TRACE_ERR(("connIndex is INVALID. ERROR ::No Proxy matched from the LIST"));
         return;
      }

      SiriStatus_value_evo.m_u8SiriStatus = 0;

      g_mutex_lock(&ConnMapTableMutex);
      vStringCopy(SiriStatus_value_evo.BD_Address, ConnMapTableST[connIndex].DevAddr, PHONE_BDADRESS_LENGTH);
      g_mutex_unlock(&ConnMapTableMutex);

      DCCA_callbacks_evo->vSiriStatusReqRetry(&SiriStatus_value_evo); //Retry if there is any error.
#endif
   }
   else
   {
      ETG_TRACE_USR1(("_EvoSiriEnableRequest OK"));
   }
}

/******************************************************************
 Function Name : setSiriEnableRequest

 Description   : Enables SIRI

 Arguments     :
 In            : deviceaddress - Device address
                 vendorId - Vendor ID
                 productId - Product ID
                 version - version (TODO:)
                 supportType - support type (TODO:)
 Out           :
 *******************************************************************/
void setSiriEnableRequest(int connIndex, guint16 vendorId, guint16 productId, guint16 version, guint16 supportType)
{
   ETG_TRACE_USR1(("setSiriEnableRequest entered "));

   GObject* siriProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_SIRI_INTERFACE);
   if (siriProxy)
   {
      ETG_TRACE_USR1(("calling evo_siri_call_enable... "));
      evo_siri_call_enable((EvoSiri*) siriProxy, vendorId, productId, version, supportType, NULLPTR, _EvoSiriEnableRequestCallback, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::Siri proxy is NULL"));
   }
}

/******************************************************************
 Function Name : SiriProxyCB

 Description   : Call back for SIRI proxy new

 Arguments     :
 In            : GObject - SIRI object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void SiriProxyCB(GObject* source_object, GAsyncResult* res, gpointer user_data)
{
   ETG_TRACE_USR1(("SiriProxyCB entered"));

   (tVoid)user_data;
   int connected_slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;
   GError *error = NULLPTR;
   GObject* siriProxy = NULLPTR;

   siriProxy = (GObject*)evo_siri_proxy_new_finish(res,&error);

   if (error)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::Couldn't create the evo siri proxy object %s", error->message));
      g_error_free(error);
   }
   if (!siriProxy)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::siriProxy: Proxy creation failed"));
      return;
   }
   else
   {
      ETG_TRACE_USR4(("Siri proxy created successfully"));
   }

   connected_slot = _getconnected_slot(source_object);

   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connected_slot)
   {
      ETG_TRACE_USR4(("connected_slot is INVALID. Hence returning"));
      g_object_unref(siriProxy);
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);

   //update the status as slot used
   ConnMapTableST[connected_slot].proxy_common[FC_PHONE_DBUS_PROXY_EVO_SIRI_INTERFACE] = siriProxy;
   //_printSlotStatusEvo();
   g_mutex_unlock(&ConnMapTableMutex);

}

/******************************************************************
 Function Name : CreateSiriProxy

 Description   : Creates SIRI proxy.

 Arguments     :
 In            : handsfreeObjPath - Hands free object path
 Out           :
 *******************************************************************/
void CreateSiriProxy(const char* handsfreeObjPath)
{
   ETG_TRACE_USR1(("CreateSiriProxy entered"));
   ETG_TRACE_USR4(("Create siri proxy handsfreeObjpath %s", handsfreeObjPath));


   //Create Manager proxy
   evo_siri_proxy_new(
         gdbus,
         G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
         BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
         handsfreeObjPath,//OBJECT PATH, Siri obj path is same as Handsfree obj path.
         NULLPTR,//GCancellable
         SiriProxyCB,
         NULLPTR);
}

/******************************************************************
 Function Name : getSiriPropertiesCB

 Description   : Call back function for SIRI get properties callback.

 Arguments     :
 In            : source_object - SIRI object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void getSiriPropertiesCB(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("getSiriPropertiesCB entered"));
   (tVoid) user_data;
   GError *err = NULLPTR;
   GVariant *out_data;
   evo_siri_call_get_properties_finish((EvoSiri*) source_object, &out_data, res, &err);

   GVariantIter iter;
   const gchar *propertyName;
   GVariant *value;
   g_variant_iter_init(&iter, out_data);
   while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
   {
      ETG_TRACE_USR4(("PrpertyName - %s ", propertyName));
      if (strcmp(propertyName, "Status") == 0)
      {
         tU8 siriStatus;
         g_variant_get(value, "y", &siriStatus);
         ETG_TRACE_USR4(("siriStatus %d ", siriStatus));
      }
      if (strcmp(propertyName, "NR") == 0)
      {
         tU8 NRStatus;
         g_variant_get(value, "y", &NRStatus);
         ETG_TRACE_USR4(("NRStatus %d", NRStatus));
      }
      if (strcmp(propertyName, "EyesFreeMode") == 0)
      {
         gboolean eyesFreeMode;
         g_variant_get(value, "b", &eyesFreeMode);
         ETG_TRACE_USR4(("eyesFreeMode - %d ", eyesFreeMode));
      }
      if (strcmp(propertyName, "SupportedType") == 0)
      {
         tU16 supportedType;
         g_variant_get(value, "q", &supportedType);
         ETG_TRACE_USR4(("supportedType - %d ", supportedType));
      }
   }
}

/******************************************************************
 Function Name : getEvoSiriProperties

 Description   : Gets the SIRI properties

 Arguments     :
 In            : deviceaddress - Device address
 Out           :
 *******************************************************************/
void getEvoSiriProperties(int connIndex)
{
   ETG_TRACE_USR1(("getEvoSiriProperties entered"));

   GObject* siriProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_SIRI_INTERFACE);
   if (siriProxy)
   {
      evo_siri_call_get_properties((EvoSiri*) siriProxy, NULLPTR, getSiriPropertiesCB, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::Siri proxy is NULL"));
   }
}

/******************************************************************
 Function Name : getVoiceCallIndexFromCallPath

 Description   : Gets Voice call Index from callpath

 Arguments     :
 In            : CallPath - Callpath of each voiceCall
 Out           :
 *******************************************************************/
int getVoiceCallIndexFromCallPath(const gchar* CallPath)
{
   ETG_TRACE_USR1(("getVoiceCallIndexFromCallPath entered "));
   char Index[3] ={0};

   //Copy the voiceCall Index from CallPath
   vStringCopy(Index, &CallPath[strlen(BASE_CALL_PATH_EVO)], sizeof(Index));

   return atoi(Index);
}

/******************************************************************
 Function Name : getVoiceCallIndexFromProxy

 Description   : Gets Voice call Index from Voice call proxy.

 Arguments     :
 In            : proxy - Voice call proxy.
 Out           :
 *******************************************************************/
int getVoiceCallIndexFromProxy(VoiceCall* proxy)
{
   ETG_TRACE_USR1(("getVoiceCallIndexFromProxy entered "));
   char Index[3] ={0};

   const gchar* CallPath = g_dbus_proxy_get_object_path((GDBusProxy*) proxy);
   ETG_TRACE_USR4(("getVoiceCallIndexFromProxy CallPath:%s", CallPath));

   //Copy the voiceCall Index from CallPath
   vStringCopy(Index, &CallPath[strlen(BASE_CALL_PATH_EVO)], sizeof(Index));

   return atoi(Index);
}

#ifdef PHONE_WAITINGMODE
/******************************************************************
 Function Name : _Evo_Release_Waiting_call_completion_callback

 Description   : call back function for release second waiting call

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Release_Waiting_call_completion_callback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_Evo_Release_Waiting_call_completion_callback entered"));

   (tVoid) user_data;
   GError *error = NULLPTR;
   gboolean retval = FALSE;
   retval = voice_call_call_hangup_finish((VoiceCall*) source_object,res, &error);

   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         gchar *errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
         g_free(errmsg);
      }
   }
   else
   {
      ETG_TRACE_USR1(("_Evo_Release_Waiting_call_completion_callback OK"));
   }
}
#endif

/******************************************************************
 Function Name : callAddedSignalHandler

 Description   : Callback function for call-added signal

 Arguments     :
 In            : source_object - Voice call manager proxy.
                 CallPath - Unique call path for each call
                 properties - New voice calls properties.
 Out           :
 *******************************************************************/
void callAddedSignalHandler(VoiceCallManager *source_object, const gchar* CallPath, GVariant *properties)
{
   ETG_TRACE_USR1(("callAddedSignalHandler entered "));
   ETG_TRACE_USR4(("callAddedSignalHandler CallPath %s", CallPath));

   int connIndex = _getconnected_slot((GObject*)source_object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return;
   }

   int callIndex = getVoiceCallIndexFromCallPath(CallPath);

   if (CheckVoiceCallProxyDuplication(callIndex, connIndex))
   {
      ETG_TRACE_USR4(("Voice call is already present in the CallMapTable"));
      return;
   }

   int free_slot = getFreeSlotFromCallMapTable(connIndex);

   updateAndInsertNewCall(connIndex,callIndex,free_slot,CallPath);

   //Copy initial value of the Callstaus to EvoCallbackdata.rCallStatus
   GetCallStatusFromCallMapTable(ConnMapTableST[connIndex].CallMapTable[free_slot].callstatus);

   ETG_TRACE_USR4(("Call Handle :%d::Call Total :%d", EvoCallbackdata.rCallStatus.u8CallHandle,EvoCallbackdata.rCallStatus.u8CallTotal));

   GVariantIter iter;
   GVariant *value;
   const gchar *propertyName;
   g_variant_iter_init(&iter, properties);
   while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
   {
      //ETG_TRACE_USR4(("PrpertyName - %s ", propertyName));
      if (strcmp(propertyName, "LineIdentification") == 0)
      {
         const gchar* PhoneNumber;
         g_variant_get(value, "s", &PhoneNumber);
         vStringCopy(EvoCallbackdata.rCallStatus.achCallNumber, PhoneNumber, PHONE_CALLNUMBER_LENGTH);
         ETG_TRACE_USR4(("PhoneNumber %s ", PhoneNumber));
      }
      if (strcmp(propertyName, "Name") == 0)
      {
         const gchar* IDName = "";
         g_variant_get(value, "s", &IDName);
         ETG_TRACE_USR4(("IDName %s", IDName));
      }
      if (strcmp(propertyName, "Multiparty") == 0)
      {
         gboolean isMultiparty;
         g_variant_get(value, "b", &isMultiparty);
         ETG_TRACE_USR4(("isMultiparty :%d ", isMultiparty));
      }
      if (strcmp(propertyName, "Emergency") == 0)
      {
         gboolean isEmergency;
         g_variant_get(value, "b", &isEmergency);
         ETG_TRACE_USR4(("isEmergency :%d ", isEmergency));
      }
      if (strcmp(propertyName, "State") == 0)
      {
         const gchar* callStatus;
         EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_IDLE;// IDLE call status
         g_variant_get(value, "s", &callStatus);
         ETG_TRACE_USR4(("Value of propertyChanged State :%s", callStatus));
         if (strcmp(callStatus, "Active") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_ACTIVE;
         }
         else if (strcmp(callStatus, "Held") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_HELD;
         }
         else if (strcmp(callStatus, "Dialing") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_DIALING;
         }
         else if (strcmp(callStatus, "Alerting") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_ALERTING;
         }
         else if (strcmp(callStatus, "Incoming") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_INCOMING;
         }
         else if (strcmp(callStatus, "Waiting") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_WAITING;

#ifdef PHONE_WAITINGMODE
            fc_phone_tclWaitingMode* poWaitingMode = fc_phone_tclWaitingMode::poGetInstance();
            tenFC_Phone_WaitingModeState m_enWaitingModeState = poWaitingMode->enGetWaitingModeState();

            if ((FC_PHONE_WAITINGMODE_IDLE != m_enWaitingModeState) && (EvoCallbackdata.rCallStatus.u8CallState == CALL_STATE_WAITING))
            {
               //Any incoming call during waiting mode will be rejected
               g_mutex_lock(&ConnMapTableMutex);
               ETG_TRACE_USR4(("Phone in Waiting mode, reject the incoming call"));
               voice_call_call_hangup((VoiceCall*)ConnMapTableST[connIndex].CallMapTable[free_slot].voiceCallProxy, NULLPTR, _Evo_Release_Waiting_call_completion_callback, NULLPTR);
               g_mutex_unlock(&ConnMapTableMutex);
               return;
            }
#endif

#ifdef REJECT_THIRD_INCOMING_CALL
            if( ConnMapTableST[connIndex].CallMapTable[free_slot].callstatus.u8CallHandle > FC_PHONE_CALL_TOTAL_TWO)
            {
               //NCG3D-27721:Only two calls are supported in HU. So reject third call immediately.
               g_mutex_lock(&ConnMapTableMutex);
               ETG_TRACE_USR4(("Rejecting third incoming call"));
               voice_call_call_hangup((VoiceCall*) ConnMapTableST[connIndex].CallMapTable[free_slot].voiceCallProxy, NULLPTR, _Evo_Release_Waiting_call_completion_callback, NULLPTR);
               g_mutex_unlock(&ConnMapTableMutex);
               return;
            }
#endif
         }
         else if (strcmp(callStatus, "Disconnected") == 0)
         {
            //NOTE: Call-added should never come to this part
            //Call disconnected is as same as IDLE.
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_IDLE;
         }
      }
      if (strcmp("Direction", propertyName) == 0)
      {
         const gchar* Direction;
         g_variant_get(value, "s", &Direction);
         ETG_TRACE_USR4(("Value of propertyChanged Direction :%s ", Direction));
         if (strcmp("Unknown", Direction) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallDirection = CALL_DIR_UNKNOWN;
         }
         if (strcmp("Outgoing", Direction) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallDirection = CALL_DIR_OUTGOING;
         }
         if (strcmp("Incoming", Direction) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallDirection = CALL_DIR_INCOMING;
         }
      }
      if (strcmp("Mode", propertyName) == 0)
      {
         const gchar* Mode;
         g_variant_get(value, "s", &Mode);
         ETG_TRACE_USR4(("Value of propertyChanged Mode :%s ", Mode));

         if (strcmp("Unknown", Mode) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_UNKNOWN;
         }
         if (strcmp("Voice", Mode) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_VOICE;
         }
         if (strcmp("Data", Mode) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_DATA;
         }
         if (strcmp("FAX", Mode) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_FAX;
         }
      }
      if (strcmp("Type", propertyName) == 0)
      {
         const gchar* Type;
         g_variant_get(value, "s", &Type);
         ETG_TRACE_USR4(("Value of propertyChanged Type :%s ", Type));

         if (strcmp("Unknown", Type) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_UNKNOWN;
         }
         if (strcmp("International", Type) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_INTERNATIONAL;
         }
         if (strcmp("National", Type) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_NATIONAL;
         }
         if (strcmp("Network", Type) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_NETWORK;
         }
         if (strcmp("Dedicated", Type) == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_DEDICATED;
         }
      }
   }

   vFillCallStatus_evo((GObject*) source_object, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
   CopyCallStatusToCallMapTable(EvoCallbackdata.rCallStatus,connIndex, callIndex);

   SendAllCallstatus(connIndex);

}

/******************************************************************
 Function Name : callRemovedSignalHandler

 Description   : Callback function for call-removed signal

 Arguments     :
 In            : obj - Voice call manager proxy.
                 obj_path - Unique call path for each call
 Out           :
 *******************************************************************/
void callRemovedSignalHandler(VoiceCallManager *obj, const gchar *obj_path)
{
   ETG_TRACE_USR1(("callRemovedSignalHandler entered "));
   ETG_TRACE_USR4(("callRemovedSignalHandler CallPath %s", obj_path));
   int call_slot = 0;
   int removedCallHandle =0;

   int connIndex = _getconnected_slot((GObject*)obj);
   ETG_TRACE_USR4(("connIndex :%d", connIndex));
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return;
   }

   int callIndex = getVoiceCallIndexFromCallPath(obj_path);
   ETG_TRACE_USR4(("callIndex :%d", callIndex));

   g_mutex_lock(&ConnMapTableMutex);

   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      if (ConnMapTableST[connIndex].CallMapTable[index].callIndex == callIndex)
      {
         call_slot = index;
         removedCallHandle = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle;
         break;
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   removeCallInstance(connIndex,call_slot,removedCallHandle);
}

/******************************************************************
 Function Name : removeCallInstance

 Description   : Function to removed given call instance based on Call Index

 Arguments     :
 In            : index - instance of CallMap table
 Out           :
 *******************************************************************/
void removeCallInstance(int connIndex, int removedindex,int removedCallHandle)
{
   ETG_TRACE_USR1(("removeCallInstance entered "));

   int gcallTotal = getCallTotalForConnectedDevice(connIndex);

   ETG_TRACE_USR4(("gcallTotal :%d, removedindex :%d", gcallTotal,removedindex));
   GObject* HFProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);

   if(gcallTotal == FC_PHONE_CALL_TOTAL_ZERO)
   {
      ETG_TRACE_ERR(("ERROR :: CallTotal is Zero"));
      return;
   }

   PrintCallMapTable(connIndex);

   //un-referencing the voiceCallProxy
   gboolean retVal = removeVoiceCallProxy(connIndex, removedindex);
   if(retVal == FALSE)
   {
      ETG_TRACE_ERR(("ERROR :: removedindex is not matched"));
      return;
   }

   PrintCallMapTable(connIndex);

   g_mutex_lock(&ConnMapTableMutex);

   //Arrange callHandle
   for (int call_slot = 0; call_slot < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; call_slot++)
   {
      if (ConnMapTableST[connIndex].CallMapTable[call_slot].isCallPresent == TRUE)
      {
            if(ConnMapTableST[connIndex].CallMapTable[call_slot].callstatus.u8CallHandle > removedCallHandle)
            {
               ETG_TRACE_USR1(("CallHandle and Total Call changed for call_slot :%d ", call_slot));
               ConnMapTableST[connIndex].CallMapTable[call_slot].callstatus.u8CallHandle = static_cast<tU8>(ConnMapTableST[connIndex].CallMapTable[call_slot].callstatus.u8CallHandle - 1);
               ConnMapTableST[connIndex].CallMapTable[call_slot].isPropertyChanged = TRUE;
            }
      }
   }

   ConnMapTableST[connIndex].gcallTotal = static_cast<tU8>(ConnMapTableST[connIndex].gcallTotal - 1);  // Decrement the call total by ONE as Call is removed
   ETG_TRACE_USR4(("1. gcallTotal :%d", ConnMapTableST[connIndex].gcallTotal));

   //Arrange callTotal
    for (int call_slot = 0; call_slot < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; call_slot++)
    {
       if (ConnMapTableST[connIndex].CallMapTable[call_slot].isCallPresent == TRUE)
       {
          ETG_TRACE_USR1(("Total Call changed for call_slot :%d ",call_slot));
          ConnMapTableST[connIndex].CallMapTable[call_slot].isPropertyChanged = TRUE;
          ConnMapTableST[connIndex].CallMapTable[call_slot].callstatus.u8CallTotal = ConnMapTableST[connIndex].gcallTotal;
       }
    }

   g_mutex_unlock(&ConnMapTableMutex);

   ETG_TRACE_USR4(("After Call Map table arrangement"));
   if(HFProxy)
   {
      vFillCallStatus_evo(HFProxy, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
   }
   SendAllCallstatus(connIndex);

}

/******************************************************************
 Function Name : PrintCallMapTable

 Description   : Function to Print CallMapTable.

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void PrintCallMapTable(int connIndex)
{
   ETG_TRACE_USR1(("PrintCallMapTable entered "));

   g_mutex_lock(&ConnMapTableMutex);

   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      if (ConnMapTableST[connIndex].CallMapTable[index].isCallPresent == TRUE)
      {
         ETG_TRACE_USR1(("CallMapTable[%d].callIndex                  = %d ", index, ConnMapTableST[connIndex].CallMapTable[index].callIndex));
         ETG_TRACE_USR1(("CallMapTable[%d].isCallPresent              = %d ", index, ConnMapTableST[connIndex].CallMapTable[index].isCallPresent));
         ETG_TRACE_USR1(("CallMapTable[%d].isPropertyChanged          = %d ", index, ConnMapTableST[connIndex].CallMapTable[index].isPropertyChanged));
         ETG_TRACE_USR1(("CallMapTable[%d].voiceCallProxy             = %p", index, ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy));
         ETG_TRACE_USR1(("CallMapTable[%d].achCallPath                = %s", index, ConnMapTableST[connIndex].CallMapTable[index].achCallPath));
         ETG_TRACE_USR1(("CallMapTable[%d].callstatus.u8CallHandle    = %d", index, ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle));
         ETG_TRACE_USR1(("CallMapTable[%d].callstatus.u8CallTotal     = %d", index, ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallTotal));
         ETG_TRACE_USR1(("CallMapTable[%d].callstatus.u8CallState     = %u", index, ETG_ENUM(TR_CALLSTATE, ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState)));
         ETG_TRACE_USR1(("CallMapTable[%d].callstatus.u8CallMode      = %u", index, ETG_ENUM(TR_CALLMODETYPE, ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallMode)));
         ETG_TRACE_USR1(("CallMapTable[%d].callstatus.u8CallType      = %u", index, ETG_ENUM(TR_CALLTYPE, ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallType)));
         ETG_TRACE_USR1(("CallMapTable[%d].callstatus.u8CallDirection = %u", index, ETG_ENUM(TR_CALLDIRECTION, ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallDirection)));
         ETG_TRACE_USR1(("CallMapTable[%d].callstatus.achCallNumber   = %s", index, ConnMapTableST[connIndex].CallMapTable[index].callstatus.achCallNumber));
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);
}
/******************************************************************
 Function Name : voiceCallManagerPropertyChangedSignalHandler

 Description   : Callback function for voice call manager property changed

 Arguments     :
 In            : object - Voice call manager proxy.
                 propertyName - changed property name
                 arg_value - Property value
 Out           :
 *******************************************************************/
void voiceCallManagerPropertyChangedSignalHandler(VoiceCallManager *object, const gchar *propertyName, GVariant *arg_value)
{
   ETG_TRACE_USR1(("voiceCallManagerPropertyChangedSignalHandler entered "));
   ETG_TRACE_USR4(("handle_propertyChanged : %s", propertyName));
   (tVoid) object;
   const GVariantType *type;
   type = g_variant_get_type(arg_value);

   if (g_variant_type_is_variant(type))
   {
      //GVariant *child;
      //child = g_variant_get_child_value(arg_value, 0);

      if (strcmp("EmergencyNumbers", propertyName) == 0)
      {
         //Clear old data from the vector.
         if (emergencyNumberList.size() > 0)
         {
            emergencyNumberList.clear();
         }

         GVariantIter iter2;
         gchar* emergecyPhNumbers;
         g_variant_iter_init(&iter2, arg_value);
         while (g_variant_iter_loop(&iter2, "s", &emergecyPhNumbers))
         {
            ETG_TRACE_USR4(("EmergencyNumbers :%s", emergecyPhNumbers));
            emergencyNumberList.push_back(emergecyPhNumbers);
         }
      }
   }
}

/******************************************************************
 Function Name : VoiceCallManagerProxyCB

 Description   : Call back for Voice call manger proxy new

 Arguments     :
 In            : GObject - Voice call manager object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void VoiceCallManagerProxyCB(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("VoiceCallManagerProxyCB entered "));

   (tVoid)user_data;
   int connected_slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;
   GError *error = NULLPTR;
   GObject* poVoiceCallManagerProxy = NULLPTR;

   poVoiceCallManagerProxy = (GObject*) voice_call_manager_proxy_new_finish(res, &error);

   if (error)
   {
      ETG_TRACE_ERR(("ERROR ::Couldn't create the VoiceCallManager proxy object %s", error->message));
      g_error_free(error);
      return;
   }
   if (!poVoiceCallManagerProxy)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::poVoiceCallManagerProxy is NULL"));
      return;
   }
   else
   {
      ETG_TRACE_USR4(("CreateVoiceCallManager proxy created successfully"));
   }

   connected_slot = _getconnected_slot(source_object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connected_slot)
   {
      ETG_TRACE_USR4(("connected_slot is INVALID. Hence returning"));
      g_object_unref(poVoiceCallManagerProxy);
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);

   ConnMapTableST[connected_slot].proxy_common[FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE] = poVoiceCallManagerProxy;

   //_printSlotStatusEvo();
   g_mutex_unlock(&ConnMapTableMutex);

   getVoiceCallManagerProperties(connected_slot);
}

/******************************************************************
 Function Name : CreateVoiceCallManagerProxy

 Description   : Creates Voice call manger proxy

 Arguments     :
 In            : handsfreeObjPath - hands free object path
 Out           :
 *******************************************************************/
void CreateVoiceCallManagerProxy(const char* handsfreeObjPath)
{
   ETG_TRACE_USR1(("CreateVoiceCallManagerProxy entered "));
   ETG_TRACE_USR4(("handsfreeObjpath %s", handsfreeObjPath));

   voice_call_manager_proxy_new(
         gdbus,
         G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
         BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
         handsfreeObjPath,//OBJECT PATh is same for Handsfree and VoicecallManager
         NULLPTR,//GCancellable
         VoiceCallManagerProxyCB,
         NULLPTR);
}

/******************************************************************
 Function Name : getVoiceCallManagerPropertiesCallback

 Description   : Gets voice call manager properties call back

 Arguments     :
 In            : poSourceObject - voice call manager objet
                 poResult -
                 pUserData - User data if any
 Out           :
 *******************************************************************/
void getVoiceCallManagerPropertiesCallback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("getVoiceCallManagerPropertiesCallback entered "));
   (tVoid) pUserData;
   GError *poGErr = NULLPTR;
   GVariant *out_data = NULLPTR;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;

   retval = voice_call_manager_call_get_properties_finish((VoiceCallManager*) poSourceObject, &out_data, poResult, &poGErr);
   if (retval == FALSE)
   {
      if(NULLPTR != poGErr)
      {
         errmsg = g_dbus_error_get_remote_error(poGErr);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      GVariantIter iter;
      const gchar *propertyName;
      GVariant *value;
      g_variant_iter_init(&iter, out_data);
      while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
      {
         ETG_TRACE_USR4(("PrpertyName - %s ", propertyName));

         if (strcmp("EmergencyNumbers", propertyName) == 0)
         {
            //Clear old data from the vector.
            if (emergencyNumberList.size() > 0)
            {
               emergencyNumberList.clear();
            }

            GVariantIter iter2;
            gchar* emergecyPhNumbers;
            g_variant_iter_init(&iter2, value);
            while (g_variant_iter_loop(&iter2, "s", &emergecyPhNumbers))
            {
               ETG_TRACE_USR4(("EmergencyNumbers :%s", emergecyPhNumbers));
               emergencyNumberList.push_back(emergecyPhNumbers);
            }
         }
      }
   }
}

/******************************************************************
 Function Name : getVoiceCallManagerProperties

 Description   : Gets the voice call manager properties

 Arguments     :
 In            : deviceaddress - Device address
 Out           :
 *******************************************************************/
void getVoiceCallManagerProperties(int connIndex)
{
   ETG_TRACE_USR1(("getVoiceCallManagerProperties entered "));

   GObject* voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (voiceCallManagerProxy)
   {
      voice_call_manager_call_get_properties((VoiceCallManager*) voiceCallManagerProxy, NULLPTR, getVoiceCallManagerPropertiesCallback, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallManagerProxy proxy is NULL"));
   }
}

//--------------------------------------VoiceCall--------------------------------------//
/******************************************************************
 Function Name : getVoiceCallPropertiesCallback

 Description   : Callback function for voice Call get properties

 Arguments     :
 In            : poSourceObject - Voice Call object
                 poResult -
                 pUserData - User data if any
 Out           :
 *******************************************************************/
void getVoiceCallPropertiesCallback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("getVoiceCallPropertiesCallback entered "));
   (tVoid) pUserData;
   GError *error = NULLPTR;
   GVariant *out_data;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;
   int Index = 0;
   retval = voice_call_call_get_properties_finish((VoiceCall*) poSourceObject, &out_data, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      EvoCallbackdata.rCallStatus.u8CallHandle = (tU8)Index;

      GVariantIter iter;
      const gchar *propertyName;
      GVariant *value;
      g_variant_iter_init(&iter, out_data);
      while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
      {
         ETG_TRACE_USR4(("PrpertyName :%s", propertyName));

         if (strcmp(propertyName, "LineIdentification") == 0)
         {
            const gchar* PhoneNumber;
            g_variant_get(value, "s", &PhoneNumber);
            //Copying Phone number to callstatus.
            vStringCopy(EvoCallbackdata.rCallStatus.achCallNumber, PhoneNumber, PHONE_CALLNUMBER_LENGTH);
            ETG_TRACE_USR4(("PhoneNumber :%s", PhoneNumber));
         }
         if (strcmp(propertyName, "Name") == 0)
         {
            const gchar* IDName;
            g_variant_get(value, "s", &IDName);
            ETG_TRACE_USR4(("IDName %s ", IDName));
         }
         if (strcmp(propertyName, "Multiparty") == 0)
         {
            gboolean isMultipartyCall;
            g_variant_get(value, "b", &isMultipartyCall);
            ETG_TRACE_USR4(("isMultipartyCall :%d ", isMultipartyCall));
         }
         if (strcmp(propertyName, "State") == 0)
         {
            const gchar* callStatus;
            g_variant_get(value, "s", &callStatus);
            ETG_TRACE_USR4(("Value of propertyChanged State :%s ", callStatus));

            if (strcmp(callStatus, "Active") == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_ACTIVE;
            }
            else if (strcmp(callStatus, "Held") == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_HELD;
            }
            else if (strcmp(callStatus, "Dialing") == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_DIALING;
            }
            else if (strcmp(callStatus, "Alerting") == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_ALERTING;
            }
            else if (strcmp(callStatus, "Incoming") == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_INCOMING;
            }
            else if (strcmp(callStatus, "Waiting") == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_WAITING;
            }
            else if (strcmp(callStatus, "Disconnected") == 0)
            {
               //NOTE:Call disconnected is as same as IDLE.
               EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_IDLE;
            }
         }
         if (strcmp(propertyName, "Emergency") == 0)
         {
            gboolean isEmergencyNumber;
            g_variant_get(value, "b", &isEmergencyNumber);
            ETG_TRACE_USR4(("isEmergencyNumber - %d ", isEmergencyNumber));
         }
         if (strcmp("Direction", propertyName) == 0)
         {
            const gchar* Direction;
            g_variant_get(value, "s", &Direction);
            ETG_TRACE_USR4(("Value of propertyChanged Direction - : %s ", Direction));

            if (strcmp("Outgoing", Direction) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallDirection = CALL_DIR_OUTGOING;
            }
            if (strcmp("Incoming", Direction) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallDirection = CALL_DIR_INCOMING;
            }
         }
         if (strcmp("Mode", propertyName) == 0)
         {
            const gchar* Mode;
            g_variant_get(value, "s", &Mode);
            ETG_TRACE_USR4(("Value of propertyChanged Mode :%s ", Mode));

            if (strcmp("Voice", Mode) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_VOICE;
            }
            if (strcmp("Data", Mode) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_DATA;
            }
            if (strcmp("FAX", Mode) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_FAX;
            }
         }
         if (strcmp("Type", propertyName) == 0)
         {
            const gchar* Type;
            g_variant_get(value, "s", &Type);
            ETG_TRACE_USR4(("Value of propertyChanged Type :%s ", Type));

            if (strcmp("International", Type) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_INTERNATIONAL;
            }
            if (strcmp("National", Type) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_NATIONAL;
            }
            if (strcmp("Network", Type) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_NETWORK;
            }
            if (strcmp("Dedicated", Type) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_DEDICATED;
            }
         }
      }
   }
}

/******************************************************************
 Function Name : getVoiceCallProperties

 Description   : Gets the voice call properties

 Arguments     :
 In            : voiceCallProxy - Voice call proxy
 Out           :
 *******************************************************************/
void getVoiceCallProperties(VoiceCall* voiceCallProxy)
{
   ETG_TRACE_USR1(("getVoiceCallProperties entered"));

   if(voiceCallProxy)
   {
      voice_call_call_get_properties(voiceCallProxy,NULLPTR,getVoiceCallPropertiesCallback,NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallProxy proxy is NULL"));
   }
}

/******************************************************************
 Function Name : GetCallStatusFromCallMapTable

 Description   : Copy Callstatus from callMap table to EvoCallback data.

 Arguments     :
 In            : callStatus - Call status struct
 Out           :
 *******************************************************************/
void GetCallStatusFromCallMapTable(T_CallStatus &callMapTable_CallStatus)
{
   ETG_TRACE_USR1(("GetCallStatusFromCallMapTable entered"));
   g_mutex_lock(&ConnMapTableMutex);

   //This method is called whenever a property update comes from ALPS stack.
   //Previous CallStatus values are taken into EvoCallbackdata before updating with new values.

   EvoCallbackdata.rCallStatus.u8CallHandle = callMapTable_CallStatus.u8CallHandle;
   EvoCallbackdata.rCallStatus.u8CallTotal = callMapTable_CallStatus.u8CallTotal;
   EvoCallbackdata.rCallStatus.u8CallDirection = callMapTable_CallStatus.u8CallDirection;
   EvoCallbackdata.rCallStatus.u8CallState = callMapTable_CallStatus.u8CallState;
   EvoCallbackdata.rCallStatus.u8CallMode = callMapTable_CallStatus.u8CallMode;
   EvoCallbackdata.rCallStatus.u8CallType = callMapTable_CallStatus.u8CallType;
   vStringCopy(EvoCallbackdata.rCallStatus.achCallNumber, callMapTable_CallStatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);

   g_mutex_unlock(&ConnMapTableMutex);
}

/******************************************************************
 Function Name : CopyCallStatusToCallMapTable

 Description   : Copies Callstatus from EvoCallback data to CallMap table.

 Arguments     :
 In            : callStatus - Call status struct
                 Index - Index of the voiceCall
 Out           :
 *******************************************************************/
void CopyCallStatusToCallMapTable(T_CallStatus &EvoCallbackdata_callStatus, int connIndex, int Index)
{
   ETG_TRACE_USR1(("CopyCallStatusToCallMapTable entered"));

   g_mutex_lock(&ConnMapTableMutex);

   //This method is called after new values are updated into EvoCallbackdata.rCallStatus.
   //This will store the newly updated property values back into the CallMap Table.

   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      if (ConnMapTableST[connIndex].CallMapTable[index].callIndex == Index)
      {
         //Check if any property is changed by comparing with OLD values.
         if (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle == EvoCallbackdata_callStatus.u8CallHandle
               && ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallTotal == EvoCallbackdata_callStatus.u8CallTotal
               && ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallDirection == EvoCallbackdata_callStatus.u8CallDirection
               && ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState == EvoCallbackdata_callStatus.u8CallState
               && ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallMode == EvoCallbackdata_callStatus.u8CallMode
               && ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallType == EvoCallbackdata_callStatus.u8CallType
               && (strcmp(ConnMapTableST[connIndex].CallMapTable[index].callstatus.achCallNumber, EvoCallbackdata_callStatus.achCallNumber)== 0))
         {
            ETG_TRACE_USR1(("No property changed OR Unknown value from stack"));
            ConnMapTableST[connIndex].CallMapTable[index].isPropertyChanged = FALSE;
         }
         else
         {
            ETG_TRACE_USR1(("Any one of the property changed"));
            ConnMapTableST[connIndex].CallMapTable[index].isPropertyChanged = TRUE;
            ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle= EvoCallbackdata_callStatus.u8CallHandle;
            ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallTotal = EvoCallbackdata_callStatus.u8CallTotal;
            ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallDirection= EvoCallbackdata_callStatus.u8CallDirection;
            ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState = EvoCallbackdata_callStatus.u8CallState;
            ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallMode = EvoCallbackdata_callStatus.u8CallMode;
            ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallType = EvoCallbackdata_callStatus.u8CallType;
            vStringCopy(ConnMapTableST[connIndex].CallMapTable[index].callstatus.achCallNumber, EvoCallbackdata_callStatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);
         }
      }
   }
   g_mutex_unlock(&ConnMapTableMutex);
}
/******************************************************************
 Function Name : PrintCallstatus

 Description   : Printing Call status of EvoCallbackdata.rCallStatus

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void PrintCallstatus()
{
   ETG_TRACE_USR1(("Printing EvoCallbackdata.rCallStatus....."));
   ETG_TRACE_USR1(("CallHandle: %d   ", EvoCallbackdata.rCallStatus.u8CallHandle));
   ETG_TRACE_USR1(("CallTotal: %d   ", EvoCallbackdata.rCallStatus.u8CallTotal));
   ETG_TRACE_USR1(("CallDirection:: %u", ETG_ENUM(TR_CALLDIRECTION, EvoCallbackdata.rCallStatus.u8CallDirection)));
   ETG_TRACE_USR1(("CallState:: %u", ETG_ENUM(TR_CALLSTATE, EvoCallbackdata.rCallStatus.u8CallState)));
   ETG_TRACE_USR1(("CallMode:: %u", ETG_ENUM(TR_CALLMODETYPE, EvoCallbackdata.rCallStatus.u8CallMode)));
   ETG_TRACE_USR1(("CallType:: %u", ETG_ENUM(TR_CALLTYPE, EvoCallbackdata.rCallStatus.u8CallType)));
   ETG_TRACE_USR1(("CallNumber: %s   ", EvoCallbackdata.rCallStatus.achCallNumber));
}


/******************************************************************
 Function Name : PrintCallstatusConsolidated

 Description   : Printing Call status of EvoCallbackdata.rCallStatus

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void PrintCallstatusConsolidated()
{
   ETG_TRACE_USR1(("Printing EvoCallbackdata.rCallStatus....."));
   for (int index = 0; index < FC_PHONE_MAX_CALLINSTANCES; index++)
   {
      ETG_TRACE_USR1(("CallHandle: %d   ", EvoCallbackdataArray[index].rCallStatus.u8CallHandle));
      ETG_TRACE_USR1(("CallTotal: %d   ", EvoCallbackdataArray[index].rCallStatus.u8CallTotal));
      //ETG_TRACE_USR1(("CallDirection:: %u", ETG_ENUM(TR_CALLDIRECTION, EvoCallbackdataArray[index].rCallStatus.u8CallDirection)));
      ETG_TRACE_USR1(("CallState:: %u", ETG_ENUM(TR_CALLSTATE, EvoCallbackdataArray[index].rCallStatus.u8CallState)));
      //ETG_TRACE_USR1(("CallMode:: %u", ETG_ENUM(TR_CALLMODETYPE, EvoCallbackdataArray[index].rCallStatus.u8CallMode)));
      //ETG_TRACE_USR1(("CallType:: %u", ETG_ENUM(TR_CALLTYPE, EvoCallbackdataArray[index].rCallStatus.u8CallType)));
      ETG_TRACE_USR1(("CallNumber: %s   ", EvoCallbackdataArray[index].rCallStatus.achCallNumber));
   }
}

/******************************************************************
 Function Name : voiceCallPropertyChangedSignalHandler

 Description   : Call-back function for voice call property changed

 Arguments     :
 In            : object - Voice call proxy
                 propertyName - name of the property
                 arg_value - Value of the property
 Out           :
 *******************************************************************/
void voiceCallPropertyChangedSignalHandler(VoiceCall *object, const gchar *propertyName, GVariant *arg_value)
{
   ETG_TRACE_USR1(("voiceCallPropertyChangedSignalHandler entered"));
   ETG_TRACE_USR4(("voiceCallPropertyChangedSignalHandler - propertyName is: %s ", propertyName));
   const GVariantType *type;
   type = g_variant_get_type(arg_value);

   int connIndex = _getconnected_slot((GObject*)object);
   ETG_TRACE_USR4(("connIndex :%d ", connIndex));
   if(FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR1(("INVALID Connection INDEX. Proxy is not yet created for that device. "));
      return;
   }

   int callIndex = getVoiceCallIndexFromProxy((VoiceCall*) object);
   ETG_TRACE_USR4(("callIndex :%d ", callIndex));

   int call_slot = getCallSlotFromCallMapTable(connIndex,callIndex);
   if(call_slot == FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT)
   {
      ETG_TRACE_USR4(("Invalid call_slot received!"));
      return;
   }

   //Copy initial value of the Callstaus to EvoCallbackdata.rCallStatus
   GetCallStatusFromCallMapTable(ConnMapTableST[connIndex].CallMapTable[call_slot].callstatus);

   //PrintCallMapTable(connIndex);

   ETG_TRACE_USR4(("Call Handle :%d::Call Total :%d", EvoCallbackdata.rCallStatus.u8CallHandle,EvoCallbackdata.rCallStatus.u8CallTotal));

   if (g_variant_type_is_variant(type))
   {
      GVariant *child;
      child = g_variant_get_child_value(arg_value, 0);

      if (0 == strcmp("LineIdentification", propertyName))
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* PhoneNumber = g_variant_get_string(child, NULLPTR);
            //Copying Phone number to callstatus.
            vStringCopy(EvoCallbackdata.rCallStatus.achCallNumber, PhoneNumber, PHONE_CALLNUMBER_LENGTH);
            ETG_TRACE_USR4(("PhoneNumber :%s", PhoneNumber));
         }
      }
      if (0 == strcmp("Name", propertyName))
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* IDName = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged IDName :%s ", IDName));
         }
      }
      if (0 == strcmp("Multiparty", propertyName))
      {
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BOOLEAN))
         {
            gboolean isMultyparty = g_variant_get_boolean(child);
            ETG_TRACE_USR4(("Value of propertyChanged isMultyparty :%d ", isMultyparty));
            //This property update can be ignored. This update is not used bt service.
            return;
         }
      }
      //This can also be retrieved from Call Added or Get Calls
      if (0 == strcmp("State", propertyName))
      {
         const gchar* callStatus = "";
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            callStatus = g_variant_get_string(child, NULLPTR);
            ETG_TRACE_USR4(("Value of propertyChanged State :%s ", callStatus));
         }

         if (strcmp(callStatus, "Active") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_ACTIVE;
         }
         else if (strcmp(callStatus, "Held") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_HELD;
         }
         else if (strcmp(callStatus, "Dialing") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_DIALING;
         }
         else if (strcmp(callStatus, "Alerting") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_ALERTING;
         }
         else if (strcmp(callStatus, "Incoming") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_INCOMING;
         }
         else if (strcmp(callStatus, "Waiting") == 0)
         {
            EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_WAITING;
         }
         else if (strcmp(callStatus, "Disconnected") == 0)
         {
            //Call disconnected is as same as IDLE.
            //EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_IDLE;
            //This will be handled in call-removed signal
            //This property update can be ignored. This update is not used bt service.
            return;
         }
      }
      if (strcmp("Emergency", propertyName) == 0)
      {
         gboolean isEmergencyNumber;
         if (g_variant_is_of_type(child, G_VARIANT_TYPE_BYTE))
         {
            isEmergencyNumber = g_variant_get_byte(child);
            ETG_TRACE_USR4(("Value of propertyChanged isEmergencyNumber :%d ", isEmergencyNumber));
         }
      }
      if (strcmp("Direction", propertyName) == 0)
      {
         return;
         //NCG3D-12382 - Commented to avoid random update of Mode and Type Properties.
         /*if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* Direction = g_variant_get_string(child, NULL);
            ETG_TRACE_USR4(("Value of propertyChanged Direction :%s ", Direction));

            if (strcmp("Outgoing", Direction) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallDirection = CALL_DIR_OUTGOING;
            }
            if (strcmp("Incoming", Direction) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallDirection = CALL_DIR_INCOMING;
            }
         }*/
      }
      if (strcmp("Mode", propertyName) == 0)
      {
         return;
         //NCG3D-12382 - Commented to avoid random update of Mode and Type Properties.
/*         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* Mode = g_variant_get_string(child, NULL);
            ETG_TRACE_USR4(("Value of propertyChanged Mode :%s ", Mode));

            if (strcmp("Voice", Mode) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_VOICE;
            }
            if (strcmp("Data", Mode) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_DATA;
            }
            if (strcmp("FAX", Mode) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_FAX;
            }
         }*/
      }
      if (strcmp("Type", propertyName) == 0)
      {
         return;
         //NCG3D-12382 - Commented to avoid random update of Mode and Type Properties.
/*         if (g_variant_is_of_type(child, G_VARIANT_TYPE_STRING))
         {
            const gchar* Type = g_variant_get_string(child, NULL);
            ETG_TRACE_USR4(("Value of propertyChanged Type :%s ", Type));

            if (strcmp("International", Type) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_INTERNATIONAL;
            }
            if (strcmp("National", Type) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_NATIONAL;
            }
            if (strcmp("Network", Type) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_NETWORK;
            }
            if (strcmp("Dedicated", Type) == 0)
            {
               EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_DEDICATED;
            }
         }*/
      }

      vFillCallStatus_evo((GObject*)object, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      CopyCallStatusToCallMapTable(EvoCallbackdata.rCallStatus, connIndex, callIndex);

      SendAllCallstatus(connIndex);

   }
}

/******************************************************************
 Function Name : IsIntermediateCallStats

 Description   : Checking if received property changed is intermediate update.
                 This logic will check whether we need to call 'voiceCallManagerGetCalls' method or not.

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
gboolean IsIntermediateCallState(int connIndex)
{
   ETG_TRACE_USR1(("IsIntermediateCallStats entered"));
   gboolean retVal = FALSE;

   int callTotal = getCallTotalForConnectedDevice(connIndex);

   tU8 firstCallState = CALL_STATE_IDLE;
   tU8 secondCallState = CALL_STATE_IDLE;

   g_mutex_lock(&ConnMapTableMutex);

   //Retriving Allcallstatus.
   tU8 u8AllCallStatus = ConnMapTableST[connIndex].AllCallStatus;

   for (tU8 index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      if (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle == 1)
      {
         firstCallState = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState;
      }
      else if (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle == 2)
      {
         secondCallState = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState;
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   //Check for Duplicate conference.
   //NCG3D-18725
   //If both calls are "HELD" then it Allcallstatus should be FC_PHONE_EVO_DBUS_CONF_CALL_HELD. Otherwise it must be a intermediate call status.
   //This will occur during swapcalls operation.
   if ((firstCallState == CALL_STATE_HELD && secondCallState == CALL_STATE_HELD) && (u8AllCallStatus != FC_PHONE_EVO_DBUS_CONF_CALL_HELD)
         && (callTotal <= FC_PHONE_CALL_TOTAL_TWO))
   {
      ETG_TRACE_USR1(("u8AllCallStatus != FC_PHONE_EVO_DBUS_CONF_CALL_HELD"));
      retVal = TRUE;
   }
   //Check for Duplicate conference.
   //NCG3D-18725
   //If both calls are "ACTIVE" then it Allcallstatus should be FC_PHONE_EVO_DBUS_CONF_CALL. Otherwise it must be a intermediate call status.
   //This will occur during swapcalls operation.
   else if ((firstCallState == CALL_STATE_ACTIVE && secondCallState == CALL_STATE_ACTIVE) && (u8AllCallStatus != FC_PHONE_EVO_DBUS_CONF_CALL)
         && (callTotal <= FC_PHONE_CALL_TOTAL_TWO))
   {
      ETG_TRACE_USR1(("u8AllCallStatus != FC_PHONE_EVO_DBUS_CONF_CALL"));
      retVal = TRUE;
   }
   //Check for ReleaseAndSwap call status.
   //NCG3D-39445
   //The condition of Sigle call HELD is handled in Service file with Timer implementation.

   ETG_TRACE_USR1(("IsIntermediateCallStats :%d",retVal));
   return retVal;

}
/******************************************************************
 Function Name : SendAllCallstatus

 Description   : Sending call status to Service.

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void SendAllCallstatus(int connIndex)
{
   ETG_TRACE_USR1(("SendAllCallstatus entered"));

   if(IsIntermediateCallState(connIndex))
   {
      ETG_TRACE_USR1(("calling voiceCallManagerGetCalls"));
      voiceCallManagerGetCalls(connIndex);
      return;
   }

   int gcallTotal = getCallTotalForConnectedDevice(connIndex);

   if (gcallTotal == FC_PHONE_CALL_TOTAL_ZERO)
   {
      ETG_TRACE_USR1(("SendAllCallstatus FC_PHONE_CALL_TOTAL_ZERO"));
      EvoCallbackdata.rCallStatus.u8CallHandle = 0;
      EvoCallbackdata.rCallStatus.u8CallTotal = 0;
      EvoCallbackdata.rCallStatus.u8CallDirection = 0;
      EvoCallbackdata.rCallStatus.u8CallState = 0;
      EvoCallbackdata.rCallStatus.u8CallMode = 0;
      EvoCallbackdata.rCallStatus.u8CallType = 0;
      EvoCallbackdata.rCallStatus.achCallNumber[0] = '\0';

      PrintCallstatus();
      ETG_TRACE_USR3((" Sending call status to CCA "));
      DCCA_callbacks_evo->vCallStatus(&EvoCallbackdata);

   }
   else if(gcallTotal == FC_PHONE_CALL_TOTAL_ONE)
   {
      ETG_TRACE_USR1(("SendAllCallstatus FC_PHONE_CALL_TOTAL_ONE"));
      tU8 CallHandle = 1;
      bool iSSendStatus = false;
      g_mutex_lock(&ConnMapTableMutex);

      for (tU8 index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
      {
         if (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle == CallHandle)
         {
            if ((ConnMapTableST[connIndex].CallMapTable[index].isPropertyChanged == TRUE))
            {
               ETG_TRACE_USR3((" Property is changed for currently available calls... "));
               ConnMapTableST[connIndex].CallMapTable[index].isPropertyChanged = FALSE;
               iSSendStatus = true;
               EvoCallbackdata.rCallStatus.u8CallHandle = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle;
               EvoCallbackdata.rCallStatus.u8CallTotal = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallTotal;
               EvoCallbackdata.rCallStatus.u8CallDirection = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallDirection;
               EvoCallbackdata.rCallStatus.u8CallState = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState;
               EvoCallbackdata.rCallStatus.u8CallMode = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallMode;
               EvoCallbackdata.rCallStatus.u8CallType = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallType;
               vStringCopy(EvoCallbackdata.rCallStatus.achCallNumber, ConnMapTableST[connIndex].CallMapTable[index].callstatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);

               PrintCallstatus();
               break;
            }
         }
         else
         {
            ETG_TRACE_USR3(("No Property changed for this call Handle"));
         }
      }
      g_mutex_unlock(&ConnMapTableMutex);
      if(iSSendStatus)
      {
         ETG_TRACE_USR3((" Sending call status to CCA "));
         DCCA_callbacks_evo->vCallStatus(&EvoCallbackdata);
      }
   }
   else if (gcallTotal > FC_PHONE_CALL_TOTAL_ONE)
   {
      ETG_TRACE_USR1(("SendAllCallstatus > FC_PHONE_CALL_TOTAL_ONE "));
      tU8 CallHandle = 1;

      if (ConnMapTableST[connIndex].isGetcallsInProgress == TRUE)
      {
         //Final Call status will be sent from _Evo_GetCallStatus_Completion_Callback.
         //Get_calls is already in progress.So avoiding duplicate vCallStatusConsolidated status
         ETG_TRACE_USR1(("isGetcallsInProgress == TRUE"));
         return;
      }

      g_mutex_lock(&ConnMapTableMutex);

      do
      {
         ETG_TRACE_USR3(("Iteration CallHandle:%d",CallHandle));

         for (tU8 index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
         {
            if (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle == CallHandle)
            {
               ETG_TRACE_USR3((" Property is changed for currently available calls... "));
               ConnMapTableST[connIndex].CallMapTable[index].isPropertyChanged = FALSE;

               EvoCallbackdataArray[CallHandle - 1].rCallStatus.u8CallHandle = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle;
               EvoCallbackdataArray[CallHandle - 1].rCallStatus.u8CallTotal = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallTotal;
               EvoCallbackdataArray[CallHandle - 1].rCallStatus.u8CallDirection = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallDirection;
               EvoCallbackdataArray[CallHandle - 1].rCallStatus.u8CallState = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState;
               EvoCallbackdataArray[CallHandle - 1].rCallStatus.u8CallMode = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallMode;
               EvoCallbackdataArray[CallHandle - 1].rCallStatus.u8CallType = ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallType;
               vStringCopy(EvoCallbackdataArray[CallHandle - 1].rCallStatus.achCallNumber, ConnMapTableST[connIndex].CallMapTable[index].callstatus.achCallNumber, PHONE_CALLNUMBER_LENGTH);
            }
            else
            {
               ETG_TRACE_USR3(("No Property changed for this call Handle"));
            }
         }

         CallHandle++;
      }
      while (CallHandle <= 2); //only sending first and second call status.

      PrintCallstatusConsolidated();
      g_mutex_unlock(&ConnMapTableMutex);
      ETG_TRACE_USR3((" Sending CallstatusConsolidated to CCA "));
      DCCA_callbacks_evo->vCallStatusConsolidated(EvoCallbackdataArray);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR :: This line should never get Executed"));
   }

}

/******************************************************************
 Function Name : CreateVoiceCallProxy

 Description   : Callback function for voice call property changed

 Arguments     :
 In            : voiceCallObjpath - Voice call object path
 Out           :
 *******************************************************************/
void CreateVoiceCallProxy(const char* voiceCallObjpath)
{
   ETG_TRACE_USR1(("CreateVoiceCallProxy entered "));
   GError *error = NULLPTR;
   GObject* poVoiceCallProxy = NULLPTR;

   poVoiceCallProxy = (GObject*) voice_call_proxy_new_sync(
         gdbus,
         G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
         BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
         voiceCallObjpath,//OBJECT PATh is retrieved from VoicecallManager
         NULLPTR,//GCancellable
         &error);

   if (error)
   {
      ETG_TRACE_ERR(("ERROR ::Couldn't create the VoiceCallManager proxy object %s", error->message));
      g_error_free(error);
      return;
   }
   if(!poVoiceCallProxy)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::poVoiceCallProxy is NULL"));
      return;
   }
   else
   {
      ETG_TRACE_USR4(("VoiceCallProxy is created successfully "));
   }

   int connIndex = _getconnected_slot((GObject*)poVoiceCallProxy);
   ETG_TRACE_USR4(("connIndex :%d",connIndex));
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return;
   }

   //Getting correct VoiceCall Index
   int callIndex = getVoiceCallIndexFromCallPath(voiceCallObjpath);
   ETG_TRACE_USR4(("callIndex :%d",callIndex));

   //Assigning voiceCall proxy to CallMap Table
   AssignVoiceCallProxy(poVoiceCallProxy,connIndex,callIndex);

   g_signal_connect(poVoiceCallProxy, "property-changed", G_CALLBACK(voiceCallPropertyChangedSignalHandler), NULLPTR);

   // getVoiceCallProperties((VoiceCall*)poVoiceCallProxy);
}

/******************************************************************
 Function Name : _EvoVoiceCallHangupRequestCallback

 Description   : Callback function for voice call property changed

 Arguments     :
 In            : source_object - Voice call object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoVoiceCallHangupRequestCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoVoiceCallHangupRequestCallback entered "));
   ETG_TRACE_USR1(("_EvoVoiceCallHangupRequestCallback user_data:%s ",(char*)user_data));
   (tVoid) user_data;
   GError * error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = voice_call_call_hangup_finish((VoiceCall*) source_object,res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      if(u8StringCompare((char*)user_data, "HangUp") == 0)
      {
         ETG_TRACE_USR3((" Sending vHangupCB to CCA "));
         DCCA_callbacks_evo->vHangupCB(&EvoCallbackdata);
      }
      else
      {
         ETG_TRACE_USR3((" Sending vTerminateCB to CCA "));
         DCCA_callbacks_evo->vTerminateCB(&EvoCallbackdata);
      }
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_EvoVoiceCallHangupRequestCallback OK "));

      vFillCallStatus_evo((GObject*)source_object,FC_PHONE_DBUS_SUCCESS,FC_PHONE_DBUS_SUCCESS);
      if(u8StringCompare((char*)user_data, "HangUp") == 0)
      {
         ETG_TRACE_USR3((" Sending vHangupCB to CCA "));
         DCCA_callbacks_evo->vHangupCB(&EvoCallbackdata);
      }
      else
      {
         ETG_TRACE_USR3((" Sending vTerminateCB to CCA "));
         DCCA_callbacks_evo->vTerminateCB(&EvoCallbackdata);
      }
   }
}

/******************************************************************
 Function Name : voiceCallHangupRequest

 Description   : Hungs up voice call

 Arguments     :
 In            : deviceaddress - Device address
                 Index - Index of the voice call
 Out           :
 *******************************************************************/
gboolean voiceCallHangupRequest(int connIndex, const char* user_data, int callHandle)
{
   ETG_TRACE_USR1(("voiceCallHangupRequest entered"));
   ETG_TRACE_USR1(("CallHandle :%d",callHandle));

   GObject* voiceCallProxy = NULLPTR;

   voiceCallProxy = GetVoiceCallProxy(connIndex,callHandle);

   if (voiceCallProxy)
   {
      voice_call_call_hangup((VoiceCall*) voiceCallProxy, NULLPTR, _EvoVoiceCallHangupRequestCallback, const_cast<char*>(user_data));
      return FC_PHONE_DBUS_SUCCESS;
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallProxy proxy is NULL"));
      return FC_PHONE_DBUS_FAIL;
   }
}

/******************************************************************
 Function Name : voiceCallTerminateRequest

 Description   : Terminate the voice call

 Arguments     :
 In            : deviceaddress - Device address
                 Index - Index of the voice call
 Out           :
 *******************************************************************/
gboolean voiceCallTerminateRequest(int connIndex, const char* user_data, int callHandle)
{
   ETG_TRACE_USR1(("voiceCallTerminateRequest entered"));
   ETG_TRACE_USR1(("connIndex :%d, CallHandle :%d",connIndex,callHandle));

   GObject* voiceCallProxy = NULLPTR;

   voiceCallProxy = GetVoiceCallProxy(connIndex,callHandle);

   if (voiceCallProxy)
   {
      voice_call_call_hangup((VoiceCall*) voiceCallProxy, NULLPTR, _EvoVoiceCallHangupRequestCallback, const_cast<char*>(user_data));
      return FC_PHONE_DBUS_SUCCESS;
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallProxy proxy is NULL"));
      return FC_PHONE_DBUS_FAIL;
   }
}
/******************************************************************
 Function Name : _EvoVoiceCallRejectRequestCallback

 Description   : call back function for voice call reject request.

 Arguments     :
 In            : source_object - Voice call object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoVoiceCallRejectRequestCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoVoiceCallRejectRequestCallback entered "));
   (tVoid) user_data;
   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = voice_call_call_hangup_finish((VoiceCall*) source_object,res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vRejectCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo((GObject*)source_object,FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vRejectCB to CCA "));
      DCCA_callbacks_evo->vRejectCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : voiceCallRejectRequest

 Description   : Rejecting the voice call

 Arguments     :
 In            : deviceaddress - Device address
                 Index - Voice call Index
 Out           :
 *******************************************************************/
gboolean voiceCallRejectRequest(int connIndex, int callHandle)
{
   ETG_TRACE_USR1(("voiceCallRejectRequest entered"));

   GObject* voiceCallProxy = NULLPTR;

   voiceCallProxy = GetVoiceCallProxy(connIndex,callHandle);

   if (voiceCallProxy)
   {
      voice_call_call_hangup((VoiceCall*) voiceCallProxy, NULLPTR, _EvoVoiceCallRejectRequestCallback, NULLPTR);
      return FC_PHONE_DBUS_SUCCESS;
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallRejectRequest proxy is NULL"));
      return FC_PHONE_DBUS_FAIL;
   }
}

/******************************************************************
 Function Name : _Evo_Release_Second_Waiting_call_completion_callback

 Description   : call back function for release second waiting call

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Release_Second_Waiting_call_completion_callback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_Evo_Release_Second_Waiting_call_completion_callback entered"));

   (tVoid) user_data;
   GError *error = NULLPTR;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;
   retval = voice_call_call_hangup_finish((VoiceCall*) source_object,res, &error);

   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);;
      DCCA_callbacks_evo->vRejectCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo((GObject*)source_object,FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vRejectCB to CCA "));
      DCCA_callbacks_evo->vRejectCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : releaseSecondWaitingCallRequest

 Description   : Releases the second waiting call.

 Arguments     :
 In            : deviceaddress - Device address
 Out           :
 *******************************************************************/
gboolean releaseSecondWaitingCallRequest(int connIndex, const char* user_data)
{
   ETG_TRACE_USR1(("releaseSecondWaitingCallRequest entered"));
   GObject* voiceCallProxy = NULLPTR;

   g_mutex_lock(&ConnMapTableMutex);

   //Get the VoiceCall proxy based on the CallHandle
   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      if (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState == CALL_STATE_WAITING)
      {
         if (ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy)
         {
            voiceCallProxy = ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy;
            ETG_TRACE_USR1(("Got VoiceCall proxy"));
            break;
         }
         else
         {
            ETG_TRACE_ERR(("ERROR ::Voice Call Proxy is NULL"));
         }
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   if (!voiceCallProxy)
   {
      ETG_TRACE_USR1(("releaseSecondWaitingCallRequest: voiceCallManagerProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_call_hangup((VoiceCall*) voiceCallProxy, NULLPTR, _EvoVoiceCallHangupRequestCallback, const_cast<char*>(user_data));
   return FC_PHONE_DBUS_SUCCESS;
}


/******************************************************************
 Function Name : rejectSecondWaitingCallRequest

 Description   : Reject the second waiting call.

 Arguments     :
 In            : deviceaddress - Device address
 Out           :
 *******************************************************************/
gboolean rejectSecondWaitingCallRequest(int connIndex,GAsyncReadyCallback callback)
{
   ETG_TRACE_USR1(("releaseSecondWaitingCallRequest entered"));
   GObject* voiceCallProxy = NULLPTR;

   g_mutex_lock(&ConnMapTableMutex);

   //Get the VoiceCall proxy based on the CallHandle
   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      if (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState == CALL_STATE_WAITING)
      {
         if (ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy)
         {
            voiceCallProxy = ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy;
            ETG_TRACE_USR1(("Got VoiceCall proxy"));
            break;
         }
         else
         {
            ETG_TRACE_ERR(("ERROR ::Voice Call Proxy is NULL"));
         }
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   if (!voiceCallProxy)
   {
      ETG_TRACE_USR1(("releaseSecondWaitingCallRequest: voiceCallManagerProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_call_hangup((VoiceCall*) voiceCallProxy, NULLPTR, callback, NULLPTR);
   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : voiceCallTerminateMultipleInactive

 Description   : Releases the inactive calls

 Arguments     :
 In            : deviceaddress - Device address
 Out           :
 *******************************************************************/
gboolean voiceCallTerminateMultipleInactive(int connIndex, const char* user_data)
{
   ETG_TRACE_USR1(("voiceCallTerminateMultipleInactive entered"));
   bool bISSuccess = FC_PHONE_DBUS_SUCCESS;

   g_mutex_lock(&ConnMapTableMutex);

   //Get the VoiceCall proxy based on the CallHandle
   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      if ((ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState != CALL_STATE_ACTIVE)
            && (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState != CALL_STATE_IDLE))
      {
         if (ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy)
         {
            ETG_TRACE_USR1(("Got VoiceCall proxy"));
            ETG_TRACE_USR1(("Disconnecting u8CallHandle :%d ",ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle));
            voice_call_call_hangup((VoiceCall*) ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy, NULLPTR, _EvoVoiceCallHangupRequestCallback, const_cast<char*>(user_data));
         }
         else
         {
            ETG_TRACE_ERR(("ERROR ::Voice Call Proxy is NULL"));
            bISSuccess = FC_PHONE_DBUS_FAIL;
            break;
         }
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   return bISSuccess;
}
/******************************************************************
 Function Name : _EvoVoiceCallAnswerRequestCallback

 Description   : call back function for voice call answer request.

 Arguments     :
 In            : source_object - Voice call object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoVoiceCallAnswerRequestCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoVoiceCallAnswerRequestCallback entered "));
   (tVoid) user_data ;
   GError *error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = voice_call_call_answer_finish((VoiceCall*)source_object,res,&error);
   if (retval==FALSE){
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vAcceptCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_EvoVoiceCallAnswerRequestCallback OK"));

      vFillCallStatus_evo((GObject*)source_object,FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vAcceptCB to CCA "));
      DCCA_callbacks_evo->vAcceptCB(&EvoCallbackdata);

   }
}

/******************************************************************
 Function Name : voiceCallAnswerRequest

 Description   : Answers the voice call based on the index.

 Arguments     :
 In            : DevAddress - Device address.
                 Index - index of the vocie call
 Out           :
 *******************************************************************/
void voiceCallAnswerRequest(int connIndex, int callHandle)
{
   ETG_TRACE_USR1(("voiceCallAnswerRequest"));

   GObject* voiceCallProxy = NULLPTR;
   bool isCallIncoming = FALSE;

   voiceCallProxy = GetVoiceCallProxy(connIndex,callHandle);

   if (voiceCallProxy)
   {
      for (int iter = 0; iter < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; iter++)
      {
         if (ConnMapTableST[connIndex].CallMapTable[iter].callstatus.u8CallHandle == callHandle)
         {
            if (ConnMapTableST[connIndex].CallMapTable[iter].callstatus.u8CallState == CALL_STATE_INCOMING)
            {
               isCallIncoming = TRUE;
               break;
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallProxy proxy is NULL"));
   }

   if (isCallIncoming)
   {
      //Answer only if the call is incoming
      ETG_TRACE_USR4(("Answering the incoming call..."));
      voice_call_call_answer((VoiceCall*) voiceCallProxy, NULLPTR, _EvoVoiceCallAnswerRequestCallback, NULLPTR);
   }
}


/******************************************************************
 Function Name : _EvoVoiceCallHoldIncomingRequestCallback

 Description   : call back function for voice call Hold incoming call request.

 Arguments     :
 In            : source_object - Voice call object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoVoiceCallHoldIncomingRequestCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoVoiceCallHoldIncomingRequestCallback Entered "));
   (tVoid) user_data;
   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = voice_call_call_hold_finish((VoiceCall*) source_object, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vHoldCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo((GObject*)source_object,FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);

      PrintCallstatus();
      ETG_TRACE_USR3((" Sending call status to CCA "));
      DCCA_callbacks_evo->vHoldCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : _EvoVoiceCallHoldRequestCallback

 Description   : call back function for voice call Hold request.

 Arguments     :
 In            : source_object - Voice call object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoVoiceCallHoldRequestCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoVoiceCallHoldRequestCallback Entered "));
   (tVoid) user_data;
   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = voice_call_manager_call_swap_calls_finish((VoiceCallManager*) source_object, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vHoldCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo(source_object, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);

      PrintCallstatus();
      ETG_TRACE_USR3((" Sending vHoldCB to CCA "));
      DCCA_callbacks_evo->vHoldCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : voiceCallHoldRequest

 Description   : call back function for voice call reject request.

 Arguments     :
 In            : DevAddress - Device address.
                 Index - Index of the voice call
 Out           :
 *******************************************************************/
void voiceCallHoldRequest(int connIndex, int callHandle)
{
   ETG_TRACE_USR1(("voiceCallHoldRequest entered"));
   (tVoid)callHandle;

   GObject* voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);

   if (voiceCallManagerProxy)
   {
      voice_call_manager_call_swap_calls((VoiceCallManager*) voiceCallManagerProxy, NULLPTR, _EvoVoiceCallHoldRequestCallback, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallProxy proxy is NULLL"));
   }
}

/******************************************************************
 Function Name : GetVoiceCallProxy

 Description   : Gets the Voice call proxy for corresponding Call Handle

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
GObject* GetVoiceCallProxy(int connIndex,int callHandle)
{
   ETG_TRACE_USR1(("GetVoiceCallProxy entered"));
   GObject* voiceCallProxy = NULLPTR;

   g_mutex_lock(&ConnMapTableMutex);

   //Get the VoiceCall proxy based on the CallHandle
   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      ETG_TRACE_USR1(("for loop :%d connIndex :%d callHandle :%d",index,connIndex,callHandle));
      if (ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle == (tU8) callHandle)
      {
         if (ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy)
         {
            voiceCallProxy = ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy;
            ETG_TRACE_USR1(("Got VoiceCall proxy"));
            break;
         }
         else
         {
            ETG_TRACE_ERR(("ERROR ::Voice Call Proxy is NULL"));
         }
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   return voiceCallProxy;
}

/******************************************************************
 Function Name : AssignVoiceCallProxy

 Description   : Assigns the voiceCall object to the corresponding CallMap table

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void AssignVoiceCallProxy(GObject* voiceCallProxy, int connIndex, int callIndex)
{
   ETG_TRACE_USR1(("AssignVoiceCallProxy entered"));
   ETG_TRACE_USR1(("Index :%d",callIndex));

   g_mutex_lock(&ConnMapTableMutex);

   //Assigns newly created proxy into a CallMap Table instance
   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      ETG_TRACE_USR1(("For loop index:%d, connIndex :%d, callIndex :%d",index,connIndex,callIndex));
      if(ConnMapTableST[connIndex].CallMapTable[index].callIndex == callIndex)
      {
         if (!ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy)
         {
            ETG_TRACE_USR1(("Voice Call proxy assigned"));
            ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy = voiceCallProxy;
            break;
         }
         else
         {
            ETG_TRACE_ERR(("ERROR ::Voice Call Proxy is NOT NULL"));
         }
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);
}

/******************************************************************
 Function Name : modemAddedSignalHandler

 Description   : Callback function for newly added modem.

 Arguments     :
 In            :object - modem manger proxy
                arg_path - argument path
 Out           :
 *******************************************************************/
void modemAddedSignalHandler(OrgOfonoManager *object, const gchar *arg_path, GVariant *arg_properties)
{
   ETG_TRACE_USR4((" modemAddedSignalHandler entered with arg_path: %s", arg_path));
   (tVoid) arg_properties;

   if (!object)
   {
      ETG_TRACE_ERR((" object is NULL!!!"));
      return;
   }

   vAddModemtoModemProxyTable(arg_path);
   vPrintModemProxyTable();
}

/******************************************************************
 Function Name : modemRemovedSignalHandler

 Description   : Callback function for modem-removed .

 Arguments     :
 In            :object - modem manger proxy
                arg_path - argument path
 Out           :
 *******************************************************************/
void modemRemovedSignalHandler(OrgOfonoManager *object, const gchar *arg_path)
{
   ETG_TRACE_USR4((" modemRemovedSignalHandler entered with arg_path: %s", arg_path));

   if (!object)
   {
      ETG_TRACE_ERR((" object is NULL!!!"));
      return;
   }

   vRemoveModemFromModemProxyTable(arg_path);
   vPrintModemProxyTable();
}

/******************************************************************
 Function Name : vAddModemtoModemProxyTable

 Description   : This function is called to update the Modems list in ModemTableST

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void vAddModemtoModemProxyTable(const gchar* objPath)
{
   ETG_TRACE_USR4(("vAddModemtoModemProxyTable entered with objPath: %s", objPath));

   char sDevAddress[PHONE_TEMP_DEV_ADDRESS] = {'\0'};

   getDeviceAddressFromHfpPath(sDevAddress, objPath);
   ETG_TRACE_USR4(( "sDevAddress :%s", sDevAddress));
   tU8 index1 = 0;

   g_mutex_lock(&ModemProxyTableMutex);

   for (; index1 < ModemProxyTable.size(); ++index1)
   {
      if (0 == u8StringCompare(sDevAddress, ModemProxyTable[index1].sDevAddr))
      {
         ETG_TRACE_USR4(( "Device Address found at :%d in the Modem proxy table", index1));
         break;
      }
   }

   if (index1 == ModemProxyTable.size())
   {
      ETG_TRACE_USR4(( "Device is not present in the Modem proxy table"));

      GError *error = NULLPTR;

      Modem* modemProxy = modem_proxy_new_sync(
            gdbus,
            G_DBUS_PROXY_FLAGS_NONE,//GDBusProxyFlags
            BASE_EVO_HANDSFREE_SERVICE_NAME,//SERVICE NAME
            objPath,//OBJECT PATH
            NULLPTR,//GCancellable
            &error);

      if(modemProxy && !error)
      {
         ETG_TRACE_USR4((" modemProxy Created successfully"));

         ModemProxyTableSTruct modemProxyObj;

         modemProxyObj.sDevAddr = new(std::nothrow) char[PHONE_TEMP_DEV_ADDRESS];
         if (!(modemProxyObj.sDevAddr))
         {
            ETG_TRACE_USR4(("Memory not allocated successfully. Hence returning"));
            g_mutex_unlock(&ModemProxyTableMutex);
            return;
         }

         memset(modemProxyObj.sDevAddr, '\0', PHONE_TEMP_DEV_ADDRESS);
         vStringCopy(modemProxyObj.sDevAddr, sDevAddress, PHONE_CONNMAP_DEV_ADDRESS);

         modemProxyObj.modemProxy = modemProxy;

         ETG_TRACE_USR4(("Adding the device into Modem proxy table"));
         // Adding the Modem to the "ModemProxyTable" Vector
         ModemProxyTable.push_back(modemProxyObj);

         g_signal_connect(modemProxy, "scoconnect-request", G_CALLBACK(scoConnectSignalHandler), NULLPTR);
         g_signal_connect(modemProxy, "property-changed", G_CALLBACK(modemPropertyChangedSignalHandler), NULLPTR);
      }
      else
      {
         ETG_TRACE_ERR(("ERROR :: Error in modemProxy creation"));
         gchar *errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
         g_free(errmsg);
      }
   }

   g_mutex_unlock(&ModemProxyTableMutex);
}

/******************************************************************
 Function Name : vRemoveModemFromModemProxyTable

 Description   : This function is called to update the Modems list in ModemTableST

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void vRemoveModemFromModemProxyTable(const gchar* objPath)
{
   ETG_TRACE_USR4(("vRemoveModemFromModemProxyTable entered with objPath: %s", objPath));

   char sDevAddress[PHONE_TEMP_DEV_ADDRESS] = {'\0'};

   getDeviceAddressFromHfpPath(sDevAddress, objPath);
   ETG_TRACE_USR4(("sDevAddress :%s", sDevAddress));

   g_mutex_lock(&ModemProxyTableMutex);

   for (tU8 index1 = 0; index1 < ModemProxyTable.size(); ++index1)
   {
      if (0 == u8StringCompare(sDevAddress, ModemProxyTable[index1].sDevAddr))
      {
         ETG_TRACE_USR4(( "Device Address found at :%d", index1));

         if (G_IS_OBJECT(ModemProxyTable[index1].modemProxy))
         {
            ETG_TRACE_USR4(("g_object_unref for valid GObject modemProxy"));
            g_object_unref(ModemProxyTable[index1].modemProxy);
         }

         if (ModemProxyTable[index1].sDevAddr)
         {
            delete []ModemProxyTable[index1].sDevAddr;
            ModemProxyTable[index1].sDevAddr = NULLPTR;
         }

         ModemProxyTable.erase(ModemProxyTable.begin()+index1);

         break;
      }
   }

   g_mutex_unlock(&ModemProxyTableMutex);
}

/******************************************************************
 Function Name : GetDevAddressFromModemProxyTable

 Description   : This function is called to get Device address from the Modem proxy table

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
char* GetDevAddressFromModemProxyTable(Modem *modemProxy)
{
   ETG_TRACE_USR4(("vGetDevAddressFromModemProxyTable entered"));

   char *pDevAddress = NULLPTR;

   g_mutex_lock(&ModemProxyTableMutex);

   for (tU8 index1 = 0; index1 < ModemProxyTable.size(); ++index1)
   {
      if (modemProxy == ModemProxyTable[index1].modemProxy)
      {
         ETG_TRACE_USR4(("modemProxy is at index: %u", index1));
         pDevAddress = ModemProxyTable[index1].sDevAddr;
      }
   }

   g_mutex_unlock(&ModemProxyTableMutex);

   return pDevAddress;
}

/******************************************************************
 Function Name : GetModemProxyFromModemProxyTable

 Description   : This function is called to get Modem proxy from the Modem proxy table

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
Modem* GetModemProxyFromModemProxyTable(tString sDevAddress)
{
   ETG_TRACE_USR4(("GetModemProxyFromModemProxyTable entered"));

   Modem* modemProxy = NULLPTR;

   g_mutex_lock(&ModemProxyTableMutex);

   for (tU8 index1 = 0; index1 < ModemProxyTable.size(); ++index1)
   {
      if (0 == u8StringCompare(sDevAddress, ModemProxyTable[index1].sDevAddr))
      {
         ETG_TRACE_USR4(("sDevAddr is at index: %u", index1));
         modemProxy = ModemProxyTable[index1].modemProxy;
      }
   }

   g_mutex_unlock(&ModemProxyTableMutex);

   return modemProxy;
}

/******************************************************************
 Function Name : vPrintModemProxyTable

 Description   : This function is called to print the Modem proxy table

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void vPrintModemProxyTable()
{
   ETG_TRACE_USR4(("vPrintModemProxyTable entered"));

   g_mutex_lock(&ModemProxyTableMutex);

   for (tU8 index1 = 0; index1 < ModemProxyTable.size(); ++index1)
   {
      ETG_TRACE_USR1(("ModemProxyTable[%u].sDevAddr: %s", index1, ModemProxyTable[index1].sDevAddr));
   }

   g_mutex_unlock(&ModemProxyTableMutex);

   ETG_TRACE_USR4(("EXIT"));
}

/******************************************************************
 Function Name : vClearModemProxyTable

 Description   : This function is called to clear the Modem proxy table

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void vClearModemProxyTable()
{
   ETG_TRACE_USR4(("vClearModemProxyTable entered"));

   g_mutex_lock(&ModemProxyTableMutex);

   for (tU8 index1 = 0; index1 < ModemProxyTable.size(); ++index1)
   {
      if (G_IS_OBJECT(ModemProxyTable[index1].modemProxy))
      {
         ETG_TRACE_USR4(("g_object_unref for valid GObject modemProxy"));
         g_object_unref(ModemProxyTable[index1].modemProxy);
      }

      if (ModemProxyTable[index1].sDevAddr)
      {
         delete []ModemProxyTable[index1].sDevAddr;
         ModemProxyTable[index1].sDevAddr = NULLPTR;
      }
   }

   ModemProxyTable.clear();
   
   g_mutex_unlock(&ModemProxyTableMutex);

   ETG_TRACE_USR4(("EXIT"));
}

/******************************************************************
 Function Name : AssignBusConnection

 Description   : Assigns Bus connection signal

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void AssignBusConnection(void)
{
   ETG_TRACE_USR1(("AssignBusConnection entered"));
   char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};

   gdbus = getEvoBusConnection();

   g_mutex_lock(&TempDeviceAddMutex);

   if (tempDeviceAddress)
   {
      ETG_TRACE_USR1(("calling CreateProxies inside AssignBusConnection"));
      getDevicePathFromRemoteDeviceAddress(tempDeviceAddress,remoteDevicePath);
      free(tempDeviceAddress);
      tempDeviceAddress = NULLPTR;
      g_mutex_unlock(&TempDeviceAddMutex);
      vStringCopy(EvoCallbackdata.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
      ETG_TRACE_USR1(("Sending vNameAppearedCB to CCA"));
      DCCA_callbacks_evo->vNameAppearedCB(&EvoCallbackdata);
   }
   else
   {
      ETG_TRACE_USR1(("tempDeviceAddress is NULL"));
      g_mutex_unlock(&TempDeviceAddMutex);
   }

   GError *error = NULLPTR;

   // Clearing the Modem Proxy Table whenever FC_Phone receives the "name_appeared" signal for "org.ofono"
   vClearModemProxyTable();

   l_ProxyModemManager
      = org_ofono_manager_proxy_new_sync(gdbus, G_DBUS_PROXY_FLAGS_NONE, BASE_EVO_HANDSFREE_SERVICE_NAME, BASE_EVO_MANAGER_OBJECT_PATH, NULLPTR, &error);
   if ((error) || (!l_ProxyModemManager))
   {
      ETG_TRACE_ERR(("ERROR :: Error in l_ProxyModemManager Proxy Creation"));
   }
   else
   {
      ETG_TRACE_USR1(("l_ProxyModemManager Proxy creation success "));

      //Register to signals "modem-added" and "modem-removed"
      g_signal_connect(l_ProxyModemManager, "modem-added", G_CALLBACK(modemAddedSignalHandler), NULLPTR);
      g_signal_connect(l_ProxyModemManager, "modem-removed", G_CALLBACK(modemRemovedSignalHandler), NULLPTR);

      vGetModems();
   }

   ETG_TRACE_USR4(("AssignBusConnection EXIT"));
}

/******************************************************************
 Function Name : EvoDeleteProxyAndDeRegisterSignals

 Description   : Deletes the proxies of disconnected device.

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
static gboolean EvoDeleteProxyAndDeRegisterSignals(const char* devicePath)
{
   ETG_TRACE_USR1(("EvoDeleteProxyAndDeRegisterSignals entered"));
   ETG_TRACE_USR1(("EvoDeleteProxyAndDeRegisterSignals devicePath :%s",devicePath));
   gboolean b_retVal = FC_PHONE_DBUS_SUCCESS;
   int slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;

   int connIndex = getConnIndexFromDeviceAdd(devicePath);

   if (connIndex >= FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT)
   {
      ETG_TRACE_ERR(("ERROR :: Greater than Number of device Connection Supported"));
      return FC_PHONE_DBUS_FAIL;
   }

   ETG_TRACE_USR4(("connected_slot :%d",connIndex));

   GObject* handsFreeProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);

   if (handsFreeProxy)
   {
      //Removing all Call instances.
      for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
      {
         removeCallInstance(connIndex,index,0);
      }

      ETG_TRACE_USR4(("SLOT STATUS BEFORE FREEING THE SLOT"));

      g_mutex_lock(&ConnMapTableMutex);
      _printSlotStatusEvo();

      //NOTE: If the device is connected then use Handsfree proxy to free all proxies slot.
      slot = _free_proxy_slot_evo(handsFreeProxy, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
      if (slot == FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT)
      {
         ETG_TRACE_FATAL(("FATAL_ERROR ::Cannot remove all the proxies of connected device"));
      }
      else
      {
         ETG_TRACE_USR3(("All the proxies cleared for connected device slot %d", slot));
      }

      ETG_TRACE_USR4(("SLOT STATUS AFTERRR FREEING THE SLOT"));

      _printSlotStatusEvo();

      g_mutex_unlock(&ConnMapTableMutex);

      //Updating SLC status for device disconnection
      UpdateSLCStatus(devicePath,FALSE);

   }
   else
   {
      ETG_TRACE_USR4(("This device is Not connected OR already removed"));
      b_retVal = FC_PHONE_DBUS_FAIL;
   }

   return b_retVal;
}

/******************************************************************
 Function Name : _Evo_Hold_And_Answer_completion_callback

 Description   : Call back function for Hold and answer call

 Arguments     :
 In            : poSourceObject - Voice call manager object path
                 poResult  -
                 pUserData - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Hold_And_Answer_completion_callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR4(("_Evo_Hold_And_Answer_completion_callback entered "));
   (tVoid) poSourceObject;
   (tVoid) pUserData;
   GError *error = NULLPTR;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;
   retval = voice_call_manager_call_hold_and_answer_finish((VoiceCallManager*) poSourceObject, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vAcceptCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_Evo_Hold_And_Answer_completion_callback OK"));
      vFillCallStatus_evo(poSourceObject, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vAcceptCB to CCA "));
      DCCA_callbacks_evo->vAcceptCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : voiceCallManagerHoldAndAnswer

 Description   : Calls voice call managers Hold and Answer method

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
gboolean voiceCallManagerHoldAndAnswer(int connIndex)
{
   ETG_TRACE_USR1(("voiceCallManagerHoldAndAnswer entered"));

   GObject* voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallManagerProxy)
   {
      ETG_TRACE_USR1(("HoldAndAnswer: voiceCallManagerProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_manager_call_hold_and_answer((VoiceCallManager*) voiceCallManagerProxy, NULLPTR, _Evo_Hold_And_Answer_completion_callback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _Evo_Dial_Completion_Callback

 Description   : Call back function for Dial mmethod call

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
static void _Evo_Dial_Completion_Callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_Dial_Completion_Callback entered "));
   (tVoid) pUserData;
   GError *poGErr = NULLPTR;
   gchar* objVoiceCallPath = NULLPTR;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;

   retval = voice_call_manager_call_dial_finish((VoiceCallManager*) poSourceObject, &objVoiceCallPath, poResult, &poGErr);
   if (retval == FALSE)
   {
      if(NULLPTR != poGErr)
      {
         errmsg = g_dbus_error_get_remote_error(poGErr);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vDialCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("Dialed Object Path %s", objVoiceCallPath));
      vFillCallStatus_evo(poSourceObject, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR4((" Sending vDialCB to CCA "));
      DCCA_callbacks_evo->vDialCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : voiceCallManagerDial

 Description   : Calls the voice call manager's Dial

 Arguments     :
 In            : DevAddress - Device address
                 u8_phone_number - Phone number
 Out           :
 *******************************************************************/
gboolean voiceCallManagerDial(int connIndex, const char* u8_phone_number)
{
   ETG_TRACE_USR1(("voiceCallManagerDial entered "));
   GObject* voiceCallManagerProxy;

   ETG_TRACE_USR4((" voiceCallManagerDial connIndex :%d", connIndex));

   if(strlen(u8_phone_number)+1 > PHONE_DIALING_PHONE_NUMBER_EVO)
   {
      ETG_TRACE_USR1(("EvoDial : Phone Number is exceeded stack limit"));
      ETG_TRACE_USR1(("EvoDial :: Dialing string length : %d ",(int)strlen(u8_phone_number)));
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = FC_PHONE_DBUS_ERRORCODE_DIAL_STRING_TOOLONG;
      DCCA_callbacks_evo->vDialCB(&EvoCallbackdata);
      return FC_PHONE_DBUS_FAIL;
   }

   voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex,FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallManagerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallManagerDial: remoteProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   ETG_TRACE_USR1(("voiceCallManagerDial :: Dialling number : %s ", u8_phone_number));
   voice_call_manager_call_dial((VoiceCallManager*) voiceCallManagerProxy, u8_phone_number, "Default", NULLPTR, _Evo_Dial_Completion_Callback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoAcceptCall

 Description   : Calls the Accept call method

 Arguments     :
 In            : DevAddress - Device address
                 enAcceptType - Accept type
 Out           :
 *******************************************************************/
static gboolean EvoAcceptCall(const char* DevAddress, FCPhoneDbusClient_AcceptType_t enAcceptType)
{
   ETG_TRACE_USR1(("EvoAcceptCall entered"));
   ETG_TRACE_USR4(("EvoAcceptCall enAcceptType:%d",enAcceptType));

   gboolean ret_accept = FC_PHONE_DBUS_FAIL;

   int connIndex =getConnIndexFromDeviceAdd(DevAddress);
   if(connIndex == FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT)
   {
      ETG_TRACE_USR1(("EvoAcceptCall : Invalid connIndex"));
      return ret_accept;
   }

   if (FC_PHONE_DBUS_ACCEPT_SINGLECALL == enAcceptType)
   {
      //Accept an incoming call.
      voiceCallAnswerRequest(connIndex);
      ret_accept = FC_PHONE_DBUS_SUCCESS;
   }
   else if (FC_PHONE_DBUS_ACCEPT_SECOND_WAITINGCALL == enAcceptType)
   {
      //Hold active calls and accept waiting call.
      voiceCallManagerHoldAndAnswer(connIndex);
      ret_accept = FC_PHONE_DBUS_SUCCESS;
   }
   else
   {
      ETG_TRACE_USR1(("Wrong Accept Type enAcceptType :: %d", enAcceptType));
   }
   return ret_accept;
}

/******************************************************************
 Function Name : EvoHoldCall

 Description   : Calls the Hold call method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
static gboolean EvoHoldCall(const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoHoldCall entered"));
   ETG_TRACE_USR4((" EvoHoldCall : Device Address %s",DevAddress));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   voiceCallHoldRequest(connIndex);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoHangupCall

 Description   : Calls the Hangup method

 Arguments     :
 In            : DevAddress - Device address
                 enTerminateType - Terminate type
 Out           :
 *******************************************************************/
static gboolean EvoHangupCall(const char* DevAddress, FCPhoneDbusClient_TerminateType_t enHangupType)
{
   ETG_TRACE_USR1(("EvoHangupCall entered"));
   ETG_TRACE_USR4((" EvoHangupCall : Device Address:%s", DevAddress));
   int retVal = FC_PHONE_DBUS_FAIL;

   int connIndex = getConnIndexFromDeviceAdd(DevAddress);
   if(connIndex != FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT)
   {
      retVal = VoiceCallTerminateOrHangup(connIndex,enHangupType,"HangUp");
   }
   else
   {
      ETG_TRACE_USR1(("EvoHangupCall : Invalid connIndex"));
   }

   return retVal;
}

/******************************************************************
 Function Name : EvoRejectCall

 Description   : Calls Reject call method.

 Arguments     :
 In            : DevAddress - Device address
                 enIgnoreType - Ignore type
 Out           :
 *******************************************************************/
static gboolean EvoRejectCall(const char* DevAddress, FCPhoneDbusClient_IgnoreType_t enIgnoreType)
{
   ETG_TRACE_USR1(("EvoRejectCall entered"));
   ETG_TRACE_USR4((" EvoRejectCall Remote device address %s", DevAddress));
   gboolean retVal = FALSE;

   int connIndex = getConnIndexFromDeviceAdd(DevAddress);

   if (FC_PHONE_DBUS_IGNORE_SINGLECALL == enIgnoreType)
   {
      ETG_TRACE_USR3(("Reject single call"));
      //This function again uses the voiceCallhungupRequest interface. Only callbacks differ.
      retVal = voiceCallRejectRequest(connIndex);
   }
   else if (FC_PHONE_DBUS_IGNORE_SECOND_WAITINGCALL == enIgnoreType)
   {
      ETG_TRACE_USR3(("Reject Second incoming or waiting call"));
      //hangup call of vocie call to be mapped.
      retVal = rejectSecondWaitingCallRequest(connIndex,_Evo_Release_Second_Waiting_call_completion_callback);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::Wrong enIgnoreType :: %d", enIgnoreType));
   }
   return retVal;
}

/******************************************************************
 Function Name : _Evo_CellDeviceName_completion_callback

 Description   : Call back for get cell Device name method

 Arguments     :
 In            : poSourceObject - Modem object
                 poResult -
                 pUserData - User data if any
 Out           :
 *******************************************************************/
void _Evo_CellDeviceName_completion_callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR4(("_Evo_CellDeviceName_completion_callback entered "));
   (tVoid) pUserData;
   GError *error = NULLPTR;
   gchar *errmsg = NULLPTR;
   GVariant *out_properties;
   gboolean isPowered = FALSE;

   gboolean retval = modem_call_get_properties_finish((Modem*) poSourceObject, &out_properties, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vGetDeviceNameCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      GVariantIter iter;
      const gchar *propertyName;
      GVariant *value;
      const gchar* modemName = "";

      g_variant_iter_init(&iter, out_properties);
      while (g_variant_iter_loop(&iter, "{&sv}", &propertyName, &value))
      {
         ETG_TRACE_USR4(("PrpertyName :%s", propertyName));
         if (strcmp(propertyName, "Powered") == 0)
         {
            isPowered = g_variant_get_boolean(value);
            ETG_TRACE_USR4(("isPowered :%d", isPowered));
         }
         if (strcmp(propertyName, "Name") == 0)
         {
            g_variant_get(value, "s", &modemName);
            ETG_TRACE_USR4(("ModemName :%s", modemName));

            vStringCopy(EvoCallbackdata.achCellDeviceName, modemName, PHONE_CELLDEVICENAME_LENGTH);
            ETG_TRACE_USR4(("ModemName achCellDeviceName :%s ", EvoCallbackdata.achCellDeviceName));
            EvoCallbackdata.MR_Data.Code = FC_PHONE_DBUS_SUCCESS;
            EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_SUCCESS;
            DCCA_callbacks_evo->vGetDeviceNameCB(&EvoCallbackdata);
         }
      }
   }
}

/******************************************************************
 Function Name : EvoCellDeviceName

 Description   : Calls Cell device name method

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
static gboolean EvoCellDeviceName(const char* DevAddress)
{
   ETG_TRACE_USR4(("EvoCellDeviceName entered"));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   //get the proxy from device address: from SLOT
   GObject* modemProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_MODEM_INTERFACE);
   if (!modemProxy)
   {
      ETG_TRACE_USR1(("EvoCellDeviceName is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   ETG_TRACE_USR4((" EvoCellDeviceName Calling modem_call_get_properties "));
   //NOTE: getModemProperties function is avoided to have seperate callback for EvoCellDeviceName.
   modem_call_get_properties((Modem*) modemProxy, NULLPTR, _Evo_CellDeviceName_completion_callback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoDial

 Description   : Initiates a new outgoing call. Returns the object path to the newly created call.
                 The hide_callerid variable holds the CLIR override for this call.
                 This method is usually implemented using the ATD AT command.
 NOTE          : If an active call (single or multiparty) exists,
                 then it is automatically put on hold if the dial procedure is successful.
 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
static gboolean EvoDial(const char* DevAddress, const char* u8_phone_number)
{
   ETG_TRACE_USR4(("EvoDial entered"));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   //Call dial function of voice call manager
   voiceCallManagerDial(connIndex, u8_phone_number);
   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _Evo_Release_And_Answer_completion_callback

 Description   : Call back function for Release and answer

 Arguments     :
 In            : poSourceObject - voice call manager object
                 poResult -
                 pUserData - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Release_And_Answer_completion_callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_Release_And_Answer_completion_callback entered "));
   (tVoid) pUserData;
   GError* error = NULLPTR;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;
   retval = voice_call_manager_call_release_and_answer_finish((VoiceCallManager*) poSourceObject, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vHangupCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo(poSourceObject, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vHangupCB to CCA "));
      DCCA_callbacks_evo->vHangupCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : voiceCallManagerReleaseAndAnswer

 Description   : Calls release and answer method

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
gboolean voiceCallManagerReleaseAndAnswer(int connIndex)
{
   ETG_TRACE_USR1(("voiceCallManagerReleaseAndAnswer entered "));
   ETG_TRACE_USR1(("voiceCallManagerReleaseAndAnswer connIndex :%d ",connIndex));

   GObject* voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallManagerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallManagerProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_manager_call_release_and_answer((VoiceCallManager*) voiceCallManagerProxy, NULLPTR, _Evo_Release_And_Answer_completion_callback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _Evo_Hangup_Release_And_Swap_completion_callback

 Description   : Call back function for  release and swap method with vHangupCB

 Arguments     :
 In            : poSourceObject - voice call manager object
                 poResult -
                 pUserData - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Hangup_Release_And_Swap_completion_callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR4(("_Evo_Hangup_Release_And_Swap_completion_callback entered "));
   (tVoid) pUserData;
   GError *poGErr = NULLPTR;
   gchar *errmsg = NULLPTR;
   gboolean retval = FALSE;

   retval = voice_call_manager_call_release_and_swap_finish((VoiceCallManager*) poSourceObject, poResult, &poGErr);
   if (retval == FALSE)
   {
      if(NULLPTR != poGErr)
      {
         errmsg = g_dbus_error_get_remote_error(poGErr);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      if (u8StringCompare((char*)pUserData, "HangUp") == 0)
      {
         ETG_TRACE_USR3((" Sending vHangupCB to CCA "));
         DCCA_callbacks_evo->vHangupCB(&EvoCallbackdata);
      }
      else
      {
         ETG_TRACE_USR3((" Sending vTerminateCB to CCA "));
         DCCA_callbacks_evo->vTerminateCB(&EvoCallbackdata);
      }
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_Evo_Hangup_Release_And_Swap_completion_callback OK "));

      vFillCallStatus_evo(poSourceObject,FC_PHONE_DBUS_SUCCESS,FC_PHONE_DBUS_SUCCESS);
      if (u8StringCompare((char*)pUserData, "HangUp") == 0)
      {
         ETG_TRACE_USR3((" Sending vHangupCB to CCA "));
         DCCA_callbacks_evo->vHangupCB(&EvoCallbackdata);
      }
      else
      {
         ETG_TRACE_USR3((" Sending vTerminateCB to CCA "));
         DCCA_callbacks_evo->vTerminateCB(&EvoCallbackdata);
      }
   }

}
/******************************************************************
 Function Name : voiceCallManagerReleaseAndSwap

 Description   : Calls release and swap method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
gboolean voiceCallManagerReleaseAndSwap(int connIndex, const char* user_data)
{
   ETG_TRACE_USR1(("voiceCallManagerReleaseAndSwap entered"));

   GObject* voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallManagerProxy)
   {
      ETG_TRACE_USR1(("voiceCallManagerProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_manager_call_release_and_swap((VoiceCallManager*) voiceCallManagerProxy, NULLPTR, _Evo_Hangup_Release_And_Swap_completion_callback, const_cast<char*>(user_data));

   return FC_PHONE_DBUS_SUCCESS;

}

/******************************************************************
 Function Name : _Evo_Hangup_All_completion_callback

 Description   : Call back function for hungup all

 Arguments     :
 In            : poSourceObject - voice call manager object
                 poResult -
                 pUserData - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Hangup_All_completion_callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_Hangup_All_completion_callback entered "));
   (tVoid) poSourceObject;
   (tVoid) pUserData;
   GError *error = NULLPTR;
   gchar *errmsg = NULLPTR;
   gboolean retval = FALSE;

   retval = voice_call_manager_call_hangup_all_finish((VoiceCallManager*) poSourceObject, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      if (u8StringCompare((char*)pUserData, "HangUp") == 0)
      {
         ETG_TRACE_USR3((" Sending vHangupCB to CCA "));
         DCCA_callbacks_evo->vHangupCB(&EvoCallbackdata);
      }
      else
      {
         ETG_TRACE_USR3((" Sending vTerminateCB to CCA "));
         DCCA_callbacks_evo->vTerminateCB(&EvoCallbackdata);
      }
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo(poSourceObject, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      if (u8StringCompare((char*)pUserData, "HangUp") == 0)
      {
         ETG_TRACE_USR3((" Sending vHangupCB to CCA "));
         DCCA_callbacks_evo->vHangupCB(&EvoCallbackdata);
      }
      else
      {
         ETG_TRACE_USR3((" Sending vTerminateCB to CCA "));
         DCCA_callbacks_evo->vTerminateCB(&EvoCallbackdata);
      }
   }
}

/******************************************************************
 Function Name : voiceCallManagerHangupAll

 Description   : Calls hungup all method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
gboolean voiceCallManagerHangupAll(int connIndex,const char* user_data)
{
   ETG_TRACE_USR1(("voiceCallManagerHangupAll entered "));

   GObject* voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallManagerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::remoteProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_manager_call_hangup_all((VoiceCallManager*) voiceCallManagerProxy, NULLPTR, _Evo_Hangup_All_completion_callback, const_cast<char*>(user_data));

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoTerminateCall

 Description   : Calls the terminate call method

 Arguments     :
 In            : DevAddress - Device address.
                 enTerminateType - Terminate type
 Out           :
 *******************************************************************/
static gboolean EvoTerminateCall(const char* DevAddress, FCPhoneDbusClient_TerminateType_t enTerminateType)
{
   ETG_TRACE_USR1(("EvoTerminateCall entered"));
   ETG_TRACE_USR4(("enTerminateType :: %d", enTerminateType));
   int retVal = FC_PHONE_DBUS_FAIL;

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);
   if(connIndex != FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT)
   {
      retVal = VoiceCallTerminateOrHangup(connIndex,enTerminateType,"Terminate");
   }
   else
   {
      ETG_TRACE_USR1(("EvoTerminateCall: Invalid connIndex"));
   }
   return retVal;
}

/******************************************************************
 Function Name : EvoDiagnosticModeConfig

 Description   : Sets Diag mode config flag

 Arguments     :
 In            : eComponent -
                 bool_value - Flag value
 Out           :
 *******************************************************************/
static void EvoDiagnosticModeConfig(int eComponent, int bool_value)
{
   ETG_TRACE_USR1(("EvoDiagnosticModeConfig entered"));
   (tVoid) eComponent;
   //1=on,0=off
   FC_Phone_Diagnostic_flag_evo = bool_value;
}

/******************************************************************
 Function Name : _EvoHandsfreeSetAudioTransferCompletionCallback

 Description   : Call back function for audio transfer

 Arguments     :
 In            : source_object - hands free object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
static void _EvoHandsfreeSetAudioTransferCompletionCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoHandsfreeSetAudioTransferCompletionCallback entered "));
   (tVoid) user_data;

   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;
   gboolean retval = handsfree_call_set_property_finish((Handsfree*) source_object, res, &error);
   ETG_TRACE_USR4(("retval :%d", retval));

   int connIndex = _getconnected_slot(source_object);
   ETG_TRACE_USR4(("connIndex :%d", connIndex));

   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return;
   }

   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);

      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_SCO_CONN_FAILED;

      char *p_devAdd = NULLPTR;
      p_devAdd = Get_DeviceAddress_from_proxy_evo(source_object, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
      if (!p_devAdd)
      {
         ETG_TRACE_ERR(("ERROR ::p_devAdd is NULL: device address corresponding to proxy , not found in SLOT"));
      }
      else
      {
         //Converting Device address from 'AABBCCDDEE' to 'dev_AA_BB_CC_DD_EE'
         char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};
         getDevicePathFromRemoteDeviceAddress(p_devAdd, remoteDevicePath);
         vStringCopy(EvoCallbackdata.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
         ETG_TRACE_USR4(("EvoCallbackdata.BD_Address :%s", EvoCallbackdata.BD_Address));
      }
   }
   else
   {
      ETG_TRACE_USR1(("_EvoHandsfreeSetAudioTransferCompletionCallback OK "));

      g_mutex_lock(&ConnMapTableMutex);
      gboolean bAudio_Tfer_direction = ConnMapTableST[connIndex].bAudioTfer_direction;
      g_mutex_unlock(&ConnMapTableMutex);

      if (bAudio_Tfer_direction)
      {
         vFillCallStatus_evo(source_object, FC_PHONE_DBUS_SCO_CONN_ESTABLISHED, FC_PHONE_DBUS_SUCCESS);
      }
      else
      {
         vFillCallStatus_evo(source_object, FC_PHONE_DBUS_SCO_CONN_DISCONNECTED, FC_PHONE_DBUS_SUCCESS);
      }
   }

   ETG_TRACE_USR3((" Sending vAudioTransferCB to CCA "));
   DCCA_callbacks_evo->vAudioTransferCB(&EvoCallbackdata);
   //Expect the results from SCOtype property
}

/******************************************************************
 Function Name : EvoAudioTransfer

 Description   : Calls Audio transfer method

 Arguments     :
 In            : DevAddress - Device address.
                 direction - direction of the transfer
 Out           :
 *******************************************************************/
static gboolean EvoAudioTransfer(const char* DevAddress, gboolean direction)
{
   ETG_TRACE_USR1(("EvoAudioTransfer"));
   ETG_TRACE_USR4(("Direction of transfer [0:To HS 1:To HF] : %d", direction));

   int connIndex = getConnIndexFromDeviceAdd(DevAddress);
   ETG_TRACE_USR4(("connIndex : %d", connIndex));

   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return FC_PHONE_DBUS_FAIL;
   }

   GObject* handsfreeProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
   if (!handsfreeProxy)
   {
      ETG_TRACE_ERR(("ERROR ::handsfreeProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   g_mutex_lock(&ConnMapTableMutex);
   ConnMapTableST[connIndex].bAudioTfer_direction = direction;
   g_mutex_unlock(&ConnMapTableMutex);

   GVariant *varDirection;
   //Type mismatch, convert direction(int) to string then to variant.
   //snprintf(strdirection,100,"%d", direction);
   if (direction)
   {
      //Convert to variant
      varDirection = g_variant_new("s", "HF");//Type is string in introspection file.
   }
   else
   {
      varDirection = g_variant_new("s", "AG");//Type is string in introspection file.
   }

   ETG_TRACE_USR4(("CALLED : com_alps_bt_Handsfree_hf_audio_transfer_async "));
   handsfree_call_set_property((Handsfree*) handsfreeProxy, "Audio", g_variant_new("v", varDirection), NULLPTR, _EvoHandsfreeSetAudioTransferCompletionCallback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _EvoQueryTelephoneNumberCallback

 Description   : Call back of Query telephone number

 Arguments     :
 In            : source_object - Hands free object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _EvoQueryTelephoneNumberCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoQueryTelephoneNumberCallback entered "));
   (tVoid) user_data;
   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;
   guchar* out_type = NULLPTR;
   gchar* out_number;
   char *TelephoneNumber = NULLPTR;

   gboolean retval = handsfree_call_get_subscriber_finish((Handsfree*) source_object, out_type, &out_number, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_SCO_CONN_FAILED;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vQueryTelephoneNumber(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("Type of Phone number :%s", out_type));
      TelephoneNumber = (char*) out_number;
      ETG_TRACE_USR4(("TelephoneNumber :%s", TelephoneNumber));

      if (TelephoneNumber)
      {
         vStringCopy(EvoCallbackdata.MR_Data.TelephoneNumber, TelephoneNumber, PHONE_TELEPHONENUMBER_LENGTH);
         vFillCallStatus_evo(source_object, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
         ETG_TRACE_USR3((" Sending vQueryTelephoneNumber to CCA "));
         DCCA_callbacks_evo->vQueryTelephoneNumber(&EvoCallbackdata);
      }
   }
}

/******************************************************************
 Function Name : EvoQueryTelephoneNumber

 Description   : Gets the Subscriber number

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
static gboolean EvoQueryTelephoneNumber(const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoQueryTelephone entered"));
   ETG_TRACE_USR4(("EvoQueryTelephoneNumber : %s", DevAddress));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   GObject* handsfreeProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
   if (!handsfreeProxy)
   {
      ETG_TRACE_USR1(("handsfreeProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   handsfree_call_get_subscriber((Handsfree*) handsfreeProxy, NULLPTR, _EvoQueryTelephoneNumberCallback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoGetPhoneStatus

 Description   : Gets the Phone status.

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
static gboolean EvoGetPhoneStatus(const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoGetPhoneStatus entered"));
   gboolean retNetworkVal = FALSE;
   gboolean retHandsfreeVal = FALSE;
   gboolean retVal = FC_PHONE_DBUS_FAIL;

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);


   //To get the values of Service ,Roaming, Signal Strength and Operator Name.
   retNetworkVal = getNetworkRegistrationProperties(connIndex);

   //To get the value of BatteryChargeLevel.
   retHandsfreeVal = getHandsfreeProperties(connIndex);

   ETG_TRACE_USR4(("retNetworkVal %d ", retNetworkVal));
   ETG_TRACE_USR4(("retHandsfreeVal %d ", retHandsfreeVal));

   if (retNetworkVal == FC_PHONE_DBUS_SUCCESS || retHandsfreeVal == FC_PHONE_DBUS_SUCCESS)
   {
      retVal = FC_PHONE_DBUS_SUCCESS;
   }
   return retVal;
}

/******************************************************************
 Function Name : _Evo_Send_Tones_Completion_Callback

 Description   : Call back function for Send tones

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Send_Tones_Completion_Callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_Send_Tones_Completion_Callback entered"));
   (tVoid) poSourceObject;
   (tVoid) pUserData;
   GError *error = NULLPTR;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;

   retval = voice_call_manager_call_send_tones_finish((VoiceCallManager*) poSourceObject, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vSendDTMFCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo(poSourceObject, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vSendDTMFCB to CCA "));
      DCCA_callbacks_evo->vSendDTMFCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : voiceCallManagerSendTones

 Description   : Calls Send Tones method

 Arguments     :
 In            : DevAddress - Device address.
                 DTMFChar - DTMF character
 Out           :
 *******************************************************************/
gboolean voiceCallManagerSendTones(int connIndex, guchar DTMFChar)
{
   ETG_TRACE_USR1(("voiceCallManagerSendTones entered with DTMFChar: %d", DTMFChar));

   GObject* voiceCallMangerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallMangerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallMangerProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }
   setDTMFChar = '\0';
   setDTMFChar = static_cast<gchar>(DTMFChar);
   voice_call_manager_call_send_tones((VoiceCallManager*) voiceCallMangerProxy, (const gchar*) &setDTMFChar, NULLPTR, _Evo_Send_Tones_Completion_Callback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoSendDTMF

 Description   : Call the voiceCallManagerSendTones method

 Arguments     :
 In            : DevAddress - Device address.
                 DTMFChar - DTMF character
 Out           :
 *******************************************************************/
static gboolean EvoSendDTMF(const char* DevAddress, guchar DTMFChar)
{
   ETG_TRACE_USR1(("EvoSendDTMF entered"));
   ETG_TRACE_USR4(("EvoSendDTMF :: DTMFChar : %c", DTMFChar));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   voiceCallManagerSendTones(connIndex, DTMFChar);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _EvoSpeedDialCallback

 Description   : Call back function for speed dial

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _EvoSpeedDialCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoSpeedDialCallback Entered "));
   (tVoid) user_data;
   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;
   gchar* out_path;

   gboolean retval = voice_call_manager_call_mem_dial_finish((VoiceCallManager*) source_object, &out_path,res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_SCO_CONN_FAILED;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vDialCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR1(("Speed Dial Object Path %s", out_path));
      vFillCallStatus_evo(source_object, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vDialCB to CCA "));
      DCCA_callbacks_evo->vDialCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : EvoSpeedDial

 Description   : Speed dial

 Arguments     :
 In            : DevAddress - Device address
                 mem_index - memory index.
 Out           :
 *******************************************************************/
static gboolean EvoSpeedDial(const char* DevAddress, guint mem_index)
{
   ETG_TRACE_USR1(("EvoSpeedDial entered"));
   ETG_TRACE_USR4(("DSpeedDial :: Dialling mem index : %d :%s  ", mem_index, DevAddress));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   GObject* voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallManagerProxy)
   {
      ETG_TRACE_USR1(("handsfreeProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_manager_call_mem_dial((VoiceCallManager*)voiceCallManagerProxy,static_cast<guint16>(mem_index),"Default",NULLPTR,_EvoSpeedDialCallback,NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;

}

/******************************************************************
 Function Name : _EvoVoiceCallResumeRequestCallback

 Description   : Call back for resume request

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _EvoVoiceCallResumeRequestCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoVoiceCallResumeRequestCallback Entered "));
   (tVoid) user_data;
   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = voice_call_manager_call_swap_calls_finish((VoiceCallManager*) source_object, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vResumeCallCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo(source_object, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vResumeCallCB to CCA "));
      DCCA_callbacks_evo->vResumeCallCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : voiceCallManagerResumeCall

 Description   : Calls resume call method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
gboolean voiceCallManagerResumeCall(int connIndex)
{
   ETG_TRACE_USR1(("voiceCallManagerResumeCall entered"));
   GObject* voiceCallManagerProxy;

   voiceCallManagerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallManagerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallManagerProxy is NULL corresponding to DevAdderss not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   ETG_TRACE_USR4(("calling voice_call_manager_call_swap_calls "));
   voice_call_manager_call_swap_calls((VoiceCallManager*) voiceCallManagerProxy, NULLPTR, _EvoVoiceCallResumeRequestCallback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoResumeHeldCall

 Description   : Calls the voiceCallManagerResumeCall method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
static gboolean EvoResumeHeldCall(const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoResumeHeldCall"));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   voiceCallManagerResumeCall(connIndex);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : voiceCallHoldIncomingRequest

 Description   : Function to Hold the incoming call.

 Arguments     :
 In            : DevAddress - Device address.
                 Index - Index of the voice call
 Out           :
 *******************************************************************/
void voiceCallHoldIncomingRequest(int connIndex, int callHandle)
{
   ETG_TRACE_USR1(("voiceCallHoldIncomingRequest entered"));
   (tVoid)callHandle;

   GObject* voiceCallProxy = GetVoiceCallProxy(connIndex,callHandle);

   if (voiceCallProxy)
   {
      //This inferface is used to Hold incoming call. For active call hold swap interface is used.
      voice_call_call_hold((VoiceCall*) voiceCallProxy, NULLPTR, _EvoVoiceCallHoldIncomingRequestCallback, NULLPTR);
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::voiceCallProxy proxy is NULLL"));
   }
}

/******************************************************************
 Function Name : EvoHoldIncomingCall

 Description   : Calls Hold incoming call method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
static gboolean EvoHoldIncomingCall (const char* DevAddress)
{
   ETG_TRACE_USR4(("EvoHoldIncomingCall :%s",DevAddress));
   gboolean retHoldIncoming = FC_PHONE_DBUS_FAIL;

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);
   if(FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT != connIndex)
   {
      voiceCallHoldIncomingRequest(connIndex);
      retHoldIncoming = FC_PHONE_DBUS_SUCCESS;
   }
   else
   {
      ETG_TRACE_USR4(("EvoHoldIncomingCall: Invalid connIndex"));
   }
   return retHoldIncoming;
}

/******************************************************************
 Function Name : _Evo_Swap_Calls_Completion_Callback

 Description   : Call back for swap call method

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Swap_Calls_Completion_Callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_Swap_Calls_Completion_Callback entered "));
   (tVoid) pUserData;
   GError *error = NULLPTR;
   gchar *errmsg = NULLPTR;

   gboolean retval = voice_call_manager_call_swap_calls_finish((VoiceCallManager*) poSourceObject, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vSwapCallCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      vFillCallStatus_evo(poSourceObject, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vSwapCallCB to CCA "));
      DCCA_callbacks_evo->vSwapCallCB(&EvoCallbackdata);
   }
}

/******************************************************************
 Function Name : voiceCallManagerSwapCalls

 Description   : Calls swap call method

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
gboolean voiceCallManagerSwapCalls(int connIndex)
{
   ETG_TRACE_USR1(("voiceCallManagerSwapCalls entered "));
   GObject* voiceCallMangerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallMangerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::GObject is NULL corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_manager_call_swap_calls((VoiceCallManager*) voiceCallMangerProxy, NULLPTR, _Evo_Swap_Calls_Completion_Callback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoSwapCall

 Description   : Calls voiceCallManagerSwapCalls method

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
static gboolean EvoSwapCall(const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoSwapCall entered "));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   voiceCallManagerSwapCalls(connIndex);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _Evo_Create_Multiparty_Completion_Callback

 Description   : Callback for create multiparty method

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Create_Multiparty_Completion_Callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_Create_Multiparty_Completion_Callback entered "));
   (tVoid) pUserData;
   GError *error = NULLPTR;
   gchar *errmsg = NULLPTR;
   gchar **out_CallPath;
   tU8 index;

   gboolean retval = voice_call_manager_call_create_multiparty_finish((VoiceCallManager*) poSourceObject, &out_CallPath, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vConfCallCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      if (!out_CallPath)
      {
         ETG_TRACE_USR4(("out_CallPath is NULL. Hence returning"));
         return;
      }

      int connIndex = _getconnected_slot(poSourceObject);
      if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
      {
         ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
         return;
      }

      //Clear old data from the vector.
      if (VoiceCallPathList.size() > 0)
      {
         VoiceCallPathList.clear();
      }

      ETG_TRACE_USR4(("ListOfCallPath   :"));
      for (index = 0; out_CallPath[index]; index++)
      {
         ETG_TRACE_USR4(("CallPath[%d]:%s", index, out_CallPath[index]));
         //New CallPath list is updated.
         VoiceCallPathList.push_back(out_CallPath[index]);

         g_mutex_lock(&ConnMapTableMutex);

         //Get the VoiceCall proxy based on the CallHandle
         for (int index1 = 0; index1 < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index1++)
         {
            if (u8StringCompare(ConnMapTableST[connIndex].CallMapTable[index1].achCallPath,out_CallPath[index]) == 0 )
            {
               //CallMapTable[index1].isPropertyChanged = TRUE;
            }
         }
         g_mutex_unlock(&ConnMapTableMutex);
      }

      vFillCallStatus_evo(poSourceObject, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vConfCallCB to CCA "));
      DCCA_callbacks_evo->vConfCallCB(&EvoCallbackdata);

      //Updating conference call status to top layer.
      //SendAllCallstatus();
   }
}

/******************************************************************
 Function Name : voiceCallManagerCreateMultiparty

 Description   : Calls create multiparty method

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
//TODO: create Multiparty call and split call from HMI needs to be tested via TML
gboolean voiceCallManagerCreateMultiparty(int connIndex)
{
   ETG_TRACE_USR1(("voiceCallManagerCreateMultiparty entered "));

   GObject* voiceCallMangerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallMangerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::GObject is NULL corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_manager_call_create_multiparty((VoiceCallManager*) voiceCallMangerProxy, NULLPTR, _Evo_Create_Multiparty_Completion_Callback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _Evo_Hangup_Multiparty_Completion_Callback

 Description   : Callback for hangup multiparty method

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Hangup_Multiparty_Completion_Callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_Hangup_Multiparty_Completion_Callback entered "));
   (tVoid) poSourceObject;
   (tVoid) pUserData;
   GError *poGErr = NULLPTR;
   gchar **out_objPath;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;

   retval = voice_call_manager_call_create_multiparty_finish((VoiceCallManager*) poSourceObject, &out_objPath, poResult, &poGErr);
   if (retval == FALSE)
   {
      if(NULLPTR != poGErr)
      {
         errmsg = g_dbus_error_get_remote_error(poGErr);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vConfCallCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR1(("_Evo_Hangup_Multiparty_Completion_Callback OK "));
   }
}

/******************************************************************
 Function Name : voiceCallManagerHangupMultiparty

 Description   : Calls hangup multiparty method

 Arguments     :
 In            : DevAddress -  Device address
 Out           :
 *******************************************************************/
gboolean voiceCallManagerHangupMultiparty(int connIndex)
{
   ETG_TRACE_USR1(("voiceCallManagerHangupMultiparty entered "));
   ETG_TRACE_USR1(("voiceCallManagerHangupMultiparty connIndex : %d",connIndex));

   GObject* voiceCallMangerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallMangerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::RemoteProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   voice_call_manager_call_hangup_multiparty((VoiceCallManager*) voiceCallMangerProxy, NULLPTR, _Evo_Hangup_Multiparty_Completion_Callback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoConfCall

 Description   : Calls voiceCallManagerCreateMultiparty method

 Arguments     :
 In            : DevAddress -  Device address
 Out           :
 *******************************************************************/
static gboolean EvoConfCall(const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoConfCall entered "));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   voiceCallManagerCreateMultiparty(connIndex);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _Evo_Private_Chat_Completion_Callback

 Description   : Call back for Private chat method

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _Evo_Private_Chat_Completion_Callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_Private_Chat_Completion_Callback entered "));
   (tVoid) pUserData;
   GError *error = NULLPTR;
   gchar *errmsg = NULLPTR;
   gchar **out_CallPath;
   tU8 index;

   gboolean retval = voice_call_manager_call_private_chat_finish((VoiceCallManager*) poSourceObject, &out_CallPath, poResult, &error);

   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      DCCA_callbacks_evo->vSplitCallCB(&EvoCallbackdata);
      g_free(errmsg);
   }
   else
   {
      if (!out_CallPath)
      {
         ETG_TRACE_USR4(("out_CallPath is INVALID. Hence returning"));
         return;
      }

      int connIndex = _getconnected_slot(poSourceObject);
      if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
      {
         ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
         return;
      }

      //Clear old data from the vector.
      if (VoiceCallPathList.size() > 0)
      {
         VoiceCallPathList.clear();
      }

      ETG_TRACE_USR4(("ListOfCallPath   :"));
      for (index = 0; out_CallPath[index]; index++)
      {
         ETG_TRACE_USR4(("CallPath[%d]: %s", index, out_CallPath[index]));
         VoiceCallPathList.push_back(out_CallPath[index]);

         g_mutex_lock(&ConnMapTableMutex);

         //Get the VoiceCall Proxy based on the CallHandle
         for (int index1 = 0; index1 < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index1++)
         {
            if (u8StringCompare(ConnMapTableST[connIndex].CallMapTable[index1].achCallPath, out_CallPath[index]) == 0)
            {
               //CallMapTable[index1].isPropertyChanged = TRUE;
            }
         }
         g_mutex_unlock(&ConnMapTableMutex);
      }

      vFillCallStatus_evo(poSourceObject, FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);
      ETG_TRACE_USR3((" Sending vSplitCallCB to CCA "));
      DCCA_callbacks_evo->vSplitCallCB(&EvoCallbackdata);

      //Updating Split call status to above layer.
      //SendAllCallstatus();
   }
}

/******************************************************************
 Function Name : voiceCallManagerPrivateChatRequest

 Description   : Calls privte chat method

 Arguments     :
 In            : DevAddress -  Device address
                 index - Call index
 Out           :
 *******************************************************************/
gboolean voiceCallManagerPrivateChatRequest(int connIndex, const guchar CallHandle)
{
   ETG_TRACE_USR1(("voiceCallManagerPrivateChatRequest entered"));
   ETG_TRACE_USR4(("voiceCallManagerPrivateChatRequest connIndex :%d , CallHandle :%d", connIndex,CallHandle));
   int index = 0;

   GObject* voiceCallMangerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallMangerProxy)
   {
      ETG_TRACE_ERR(("ERROR ::RemoteProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   g_mutex_lock(&ConnMapTableMutex);

   for(int iter = 0; iter < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; iter++)
   {
      if(ConnMapTableST[connIndex].CallMapTable[iter].callstatus.u8CallHandle == CallHandle)
      {
         index = iter;
         break;
      }
   }

   voice_call_manager_call_private_chat((VoiceCallManager*) voiceCallMangerProxy, ConnMapTableST[connIndex].CallMapTable[index].achCallPath, NULLPTR, _Evo_Private_Chat_Completion_Callback, NULLPTR);

   g_mutex_unlock(&ConnMapTableMutex);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoSplitCall

 Description   : Calls voiceCallManagerPrivateChatRequest method

 Arguments     :
 In            : DevAddress -  Device address
                 index - Call index
 Out           :
 *******************************************************************/
static gboolean EvoSplitCall(const char* DevAddress, const guchar CallHandle)
{
   ETG_TRACE_USR1(("EvoSplitCall entered"));
   ETG_TRACE_USR4((" EvoSplitCall:: Splitindex %d", CallHandle));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   voiceCallManagerPrivateChatRequest(connIndex, CallHandle);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoReleaseSpecCall

 Description   : Releases the specific voice call

 Arguments     :
 In            : DevAddress -  Device address
                 enTerminateType - Terminate type
                 index - Call index
 Out           :
 *******************************************************************/
static gboolean EvoReleaseSpecCall(const char* DevAddress, FCPhoneDbusClient_TerminateType_t enTerminateType, const guchar index)
{
   ETG_TRACE_USR1(("EvoReleaseSpecCall entered"));
   int Index = (int)index;
   ETG_TRACE_USR4((" EvoSplitCall:: Index %d", Index));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);
   
   if (FC_PHONE_DBUS_TERMINATETYPE_THREEWAYCALL_SPECIFIC == enTerminateType)
   {
      ETG_TRACE_USR4(("calling voice_call_call_hangup :%d",enTerminateType));
      voiceCallHangupRequest(connIndex,"HangUp",Index);
   }
   if (FC_PHONE_DBUS_TERMINATETYPE_THREEWAYCALL_SPECIFIC_CANCEL_OUTGOING == enTerminateType)
   {
      ETG_TRACE_USR4(("calling voice_call_call_hangup :%d",enTerminateType));
   }
   else
   {
      ETG_TRACE_ERR(("ERROR ::Wrong enTerminateType :: %d", enTerminateType));
   }

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _Evo_GetCallStatus_Completion_Callback

 Description   : Calls back for get all calls method

 Arguments     :
 In            : source_object - Voice call manager object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _Evo_GetCallStatus_Completion_Callback(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("_Evo_GetCallStatus_Completion_Callback entered "));
   (tVoid) pUserData;
   GError *error = NULLPTR;
   GVariant *out_data = NULLPTR;
   gchar *errmsg = NULLPTR;

   //Getting connIndex for sending Empty call status when no calls present.
   int connIndex = _getconnected_slot((GObject*) poSourceObject);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);
   ETG_TRACE_USR1(("isGetcallsInProgress = FALSE"));
   ConnMapTableST[connIndex].isGetcallsInProgress = FALSE;
   g_mutex_unlock(&ConnMapTableMutex);

   gboolean retval = voice_call_manager_call_get_calls_finish((VoiceCallManager*) poSourceObject, &out_data, poResult, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("_Evo_GetCallStatus_Completion_Callback OK "));
      GVariantIter iterObjpath;
      GVariant *getObjpath;
      GVariant *getProperties;
      g_variant_iter_init(&iterObjpath, out_data);

      ETG_TRACE_USR4(("_Evo_GetCallStatus_Completion_Callback Outside while loop "));

      while (g_variant_iter_loop(&iterObjpath, "(@o@a{sv})", &getObjpath, &getProperties))
      {
         ETG_TRACE_USR4(("_Evo_GetCallStatus_Completion_Callback Inside while loop "));
         const gchar * CallPath;
         g_variant_get(getObjpath, "o", &CallPath);
         ETG_TRACE_USR4(("object path :%s \n", CallPath));
         GVariantIter iterProperty;
         const gchar *propertyName;
         GVariant *value;
         int call_slot;

         int callIndex = getVoiceCallIndexFromCallPath(CallPath);
         ETG_TRACE_USR4(("VoiceCall callIndex :%d", callIndex));

         if (CheckVoiceCallProxyDuplication(callIndex,connIndex))
         {
            ETG_TRACE_USR4(("Voice call is already present in the CallMapTable"));
            int gcallTotal = getCallTotalForConnectedDevice(connIndex);
            ETG_TRACE_USR4(("gcallTotal :%d", gcallTotal));

            call_slot = getCallSlotFromCallMapTable(connIndex,callIndex);
            if(call_slot == FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT)
            {
               ETG_TRACE_USR4(("Invalid call_slot received!"));
               return;
            }
         }
         else
         {
            call_slot = getFreeSlotFromCallMapTable(connIndex);
            updateAndInsertNewCall(connIndex, callIndex, call_slot,CallPath);
         }

         //Copy initial value of the Callstaus to EvoCallbackdata.rCallStatus
         GetCallStatusFromCallMapTable(ConnMapTableST[connIndex].CallMapTable[call_slot].callstatus);

         ETG_TRACE_USR4(("Call Handle :%d::Call Total :%d", EvoCallbackdata.rCallStatus.u8CallHandle, EvoCallbackdata.rCallStatus.u8CallTotal));

         g_variant_iter_init(&iterProperty, getProperties);
         while (g_variant_iter_loop(&iterProperty, "{&sv}", &propertyName, &value))
         {
            ETG_TRACE_USR4(("PrpertyName :%s ", propertyName));

            if (strcmp(propertyName, "LineIdentification") == 0)
            {
               const gchar* PhoneNumber;
               g_variant_get(value, "s", &PhoneNumber);
               //Copying Phone number to callstatus.
               vStringCopy(EvoCallbackdata.rCallStatus.achCallNumber, PhoneNumber, PHONE_CALLNUMBER_LENGTH);
               ETG_TRACE_USR4(("PhoneNumber :%s", PhoneNumber));
            }
            else if (strcmp(propertyName, "Name") == 0)
            {
               const gchar* IDName;
               g_variant_get(value, "s", &IDName);
               ETG_TRACE_USR4(("IDName :%s", IDName));
            }
            else if (strcmp(propertyName, "Multiparty") == 0)
            {
               gboolean isMultiparty;
               g_variant_get(value, "b", &isMultiparty);
               ETG_TRACE_USR4(("isMultiparty :%d", isMultiparty));
            }
            else if (strcmp(propertyName, "State") == 0)
            {
               const gchar* callStatus;
               g_variant_get(value, "s", &callStatus);

               ETG_TRACE_USR4(("Value of propertyChanged State :%s ", callStatus));
               if (strcmp(callStatus, "Active") == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_ACTIVE;
               }
               else if (strcmp(callStatus, "Held") == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_HELD;
               }
               else if (strcmp(callStatus, "Dialing") == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_DIALING;
               }
               else if (strcmp(callStatus, "Alerting") == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_ALERTING;
               }
               else if (strcmp(callStatus, "Incoming") == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_INCOMING;
               }
               else if (strcmp(callStatus, "Waiting") == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_WAITING;

#ifdef REJECT_THIRD_INCOMING_CALL
                  if( ConnMapTableST[connIndex].CallMapTable[call_slot].callstatus.u8CallHandle > FC_PHONE_CALL_TOTAL_TWO)
                  {
                     //NCG3D-27721:Only two calls are supported in HU. So reject third call immediately.
                     g_mutex_lock(&ConnMapTableMutex);
                     ETG_TRACE_USR4(("Rejecting third incoming call"));
                     voice_call_call_hangup((VoiceCall*) ConnMapTableST[connIndex].CallMapTable[call_slot].voiceCallProxy, NULLPTR, _Evo_Release_Waiting_call_completion_callback, NULLPTR);
                     g_mutex_unlock(&ConnMapTableMutex);
                     return;
                  }
#endif
               }
               else if (strcmp(callStatus, "Disconnected") == 0)
               {
                  //Call disconnected is as same as IDLE.
                  EvoCallbackdata.rCallStatus.u8CallState = CALL_STATE_IDLE;
               }
            }
            else if (strcmp(propertyName, "Emergency") == 0)
            {
               gboolean isEmergencyCall;
               g_variant_get(value, "b", &isEmergencyCall);
               ETG_TRACE_USR4(("isEmergencyCall :%d", isEmergencyCall));
            }
            else if (strcmp("Direction", propertyName) == 0)
            {
               const gchar* Direction;
               g_variant_get(value, "s", &Direction);
               ETG_TRACE_USR4(("Value of propertyChanged Direction :%s ", Direction));

               if (strcmp("Outgoing", Direction) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallDirection = CALL_DIR_OUTGOING;
               }
               if (strcmp("Incoming", Direction) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallDirection =CALL_DIR_INCOMING;
               }
            }
            else if (strcmp("Mode", propertyName) == 0)
            {
               const gchar* Mode;
               g_variant_get(value, "s", &Mode);
               ETG_TRACE_USR4(("Value of propertyChanged Mode :%s ", Mode));

               if (strcmp("Voice", Mode) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_VOICE;
               }
               if (strcmp("Data", Mode) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_DATA;
               }
               if (strcmp("FAX", Mode) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallMode = CALL_MODE_FAX;
               }
            }
            else if (strcmp("Type", propertyName) == 0)
            {
               const gchar* Type;
               g_variant_get(value, "s", &Type);
               ETG_TRACE_USR4(("Value of propertyChanged Type :%s ", Type));

               if (strcmp("International", Type) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_INTERNATIONAL;
               }
               if (strcmp("National", Type) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_NATIONAL;
               }
               if (strcmp("Network", Type) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_NETWORK;
               }
               if (strcmp("Dedicated", Type) == 0)
               {
                  EvoCallbackdata.rCallStatus.u8CallType = CALL_TYPE_DEDICATED;
               }
            }
         }
         //Pushing the voiceCall properties into CallMap Table
         CopyCallStatusToCallMapTable(EvoCallbackdata.rCallStatus, connIndex,callIndex);

         g_mutex_lock(&ConnMapTableMutex);
         //Property changed TRUE for currently available calls. So that Call status wont
         //be consider as Duplicate in SendAllCallStatus
         ConnMapTableST[connIndex].CallMapTable[call_slot].isPropertyChanged = TRUE;
         g_mutex_unlock(&ConnMapTableMutex);

    }
      vFillCallStatus_evo((GObject*)poSourceObject,FC_PHONE_DBUS_SUCCESS, FC_PHONE_DBUS_SUCCESS);

      SendAllCallstatus(connIndex);
   }
}

/******************************************************************
 Function Name : voiceCallManagerGetCalls

 Description   : Calls get calls method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
gboolean voiceCallManagerGetCalls(int connIndex)
{
   ETG_TRACE_USR1(("voiceCallManagerGetcalls entered"));

   GObject* voiceCallMangerProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!voiceCallMangerProxy)
   {
      ETG_TRACE_USR1(("ERROR:EvoVoiceCallManager: remoteProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }


   g_mutex_lock(&ConnMapTableMutex);
   if(ConnMapTableST[connIndex].isGetcallsInProgress != TRUE)
   {
      ETG_TRACE_USR1(("isGetcallsInProgress = TRUE"));
      ConnMapTableST[connIndex].isGetcallsInProgress = TRUE;
      voice_call_manager_call_get_calls((VoiceCallManager*) voiceCallMangerProxy, NULLPTR, _Evo_GetCallStatus_Completion_Callback, NULLPTR);
   }
   else
   {
      ETG_TRACE_USR1(("Get calls is in progress"));
   }

   g_mutex_unlock(&ConnMapTableMutex);

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoGetCallStatus

 Description   : Calls Get call status method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
static gboolean EvoGetCallStatus (const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoGetCallStatus entered"));

   int connIndex = getConnIndexFromDeviceAdd(DevAddress);

   if(FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR1(("EvoGetCallStatus FC_PHONE_DBUS_FAIL"));
      return FC_PHONE_DBUS_FAIL;
   }
   else
   {
      voiceCallManagerGetCalls(connIndex);
      return FC_PHONE_DBUS_SUCCESS;
   }

}

/******************************************************************
 Function Name : EvoGetSCOConnectionStatus

 Description   : Gets SCO connection status

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
static gboolean EvoGetSCOConnectionStatus(const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoGetSCOConnectionStatus entered"));
   ETG_TRACE_USR1(("EvoGetSCOConnectionStatus DevAddress :%s",DevAddress));

   ETG_TRACE_USR4((" EvoGetSCOConnectionStatus SCOConnectionStatusData.BD_Address %s", SCOConnectionStatus_data_evo.BD_Address));

   if (u8StringCompare(DevAddress, SCOConnectionStatus_data_evo.BD_Address) == 0)
   {
      if (SCOConnectionStatus_data_evo.Codec == NARROW_BAND_SPEECH)
      {
         ETG_TRACE_USR3((" Sending vSCOConnectionStatus to CCA "));
         DCCA_callbacks_evo->vSCOConnectionStatus(&SCOConnectionStatus_data_evo);
      }
      else
      {
         ETG_TRACE_USR3((" Sending vWideBandAudioSetup to CCA "));
         DCCA_callbacks_evo->vWideBandAudioSetup(&SCOConnectionStatus_data_evo);
      }
   }
   else
   {
      ETG_TRACE_USR3(("Device address is Doesnt Match and hence SCO status not sent    "));
      return FC_PHONE_DBUS_FAIL;
   }

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _EvoHandsfreeActivateVoiceRecognitionCompletionCallback

 Description   : Call back for activate voice recognition method

 Arguments     :
 In            : DevAddress - Device address.
 Out           :
 *******************************************************************/
static void _EvoHandsfreeActivateVoiceRecognitionCompletionCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoHandsfreeActivateVoiceRecognitionCompletionCallback entered "));

   GError* error = NULLPTR;
   gchar *errmsg = NULLPTR;

   int connIndex = _getconnected_slot(source_object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return;
   }
   guchar u8VRActivationIndex = 0;
   if (user_data)
   {
      guchar* pVRStatusActivationIndex = ((guchar*)user_data);
      u8VRActivationIndex = *pVRStatusActivationIndex;
      ETG_TRACE_USR1(("_EvoHandsfreeActivateVoiceRecognitionCompletionCallback u8VRActivationIndex = %d ",u8VRActivationIndex));
      delete pVRStatusActivationIndex;
      pVRStatusActivationIndex = NULLPTR;
   }

   gboolean retval = handsfree_call_set_property_finish((Handsfree*) source_object, res, &error);

   ETG_TRACE_USR4(("retval :%d", retval));

   if (FALSE == retval)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);

      EvoCallbackdata.MR_Data.Status = (tChar)u8VRActivationIndex;

      char *p_devAdd = NULLPTR;
      p_devAdd = Get_DeviceAddress_from_proxy_evo(source_object, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
      if (!p_devAdd)
      {
         ETG_TRACE_ERR(("ERROR ::p_devAdd is NULL: device address corresponding to proxy , not found in SLOT"));
      }
      else
      {
         //Converting Device address from 'AABBCCDDEE' to 'dev_AA_BB_CC_DD_EE'
         char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};
         getDevicePathFromRemoteDeviceAddress(p_devAdd, remoteDevicePath);
         vStringCopy(EvoCallbackdata.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
         ETG_TRACE_USR4(("EvoCallbackdata.BD_Address :%s", EvoCallbackdata.BD_Address));
      }
   }
   else
   {
      vFillCallStatus_evo(source_object, u8VRActivationIndex, FC_PHONE_DBUS_SUCCESS);
   }

   ETG_TRACE_USR3((" Sending vVoiceRecognitionCB to CCA "));
   DCCA_callbacks_evo->vVoiceRecognitionCB(&EvoCallbackdata);
}

/******************************************************************
 Function Name : setPropertyVoiceRecognition

 Description   : Sets the Voice recognition property

 Arguments     :
 In            : DevAddress - Device address.
                 activationIndex - Activation index.
 Out           :
 *******************************************************************/
gboolean setPropertyVoiceRecognition(int connIndex, const guchar activationIndex)
{
   ETG_TRACE_USR1(("setPropertyVoiceRecognition entered with connIndex: %d and activationIndex: %d", connIndex, activationIndex));

   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connIndex)
   {
      ETG_TRACE_USR4(("connIndex is INVALID. Hence returning"));
      return FC_PHONE_DBUS_FAIL;
   }

   GObject* handsfreeProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
   if (!handsfreeProxy)
   {
      ETG_TRACE_ERR(("ERROR ::HandsfreeProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }
   guchar* pVRStatusActivationIndex = new(std::nothrow) guchar;
   if (pVRStatusActivationIndex)
   {
      GVariant *varActivationIndex;
      *pVRStatusActivationIndex = activationIndex;
      varActivationIndex = g_variant_new("b", activationIndex);
      handsfree_call_set_property((Handsfree*) handsfreeProxy, "VoiceRecognition", g_variant_new("v", varActivationIndex), NULL, _EvoHandsfreeActivateVoiceRecognitionCompletionCallback,gpointer(pVRStatusActivationIndex));
   }

   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoActivateVoiceRecognition

 Description   : Calls setPropertyVoiceRecognition method

 Arguments     :
 In            : DevAddress - Device address.
                 activationIndex - activation Index
 Out           :
 *******************************************************************/
static gboolean EvoActivateVoiceRecognition (const char* DevAddress, const guchar activationIndex)
{
   ETG_TRACE_USR1(("EvoActivateVoiceRecognition entered"));
   ETG_TRACE_USR4(("EvoActivateVoiceRecognition..activationIndex %d",activationIndex));

   int connIndex = getConnIndexFromDeviceAdd(DevAddress);

   setPropertyVoiceRecognition(connIndex,activationIndex);

   return  FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _EvoRequestPhoneNumbercallback

 Description   : Callback function for EvoAcquireTelephoneNumber

 Arguments     :
 In            : source_object - hands free object
                 res -
                 user_data - User data if any
 Out           :
 *******************************************************************/
void _EvoRequestPhoneNumbercallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoRequestPhoneNumbercallback entered"));
   (tVoid) source_object;
   (tVoid) res;
   (tVoid) user_data;
   GError *error = NULLPTR;
   gchar *phoneNumber;
   gboolean retval = FALSE;
   gchar *errmsg = NULLPTR;

   retval = handsfree_call_request_phone_number_finish((Handsfree*) source_object, &phoneNumber, res, &error);
   if (retval == FALSE)
   {
      if(NULLPTR != error)
      {
         errmsg = g_dbus_error_get_remote_error(error);
         ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
      }
      EvoCallbackdata.MR_Data.Status = FC_PHONE_DBUS_FAIL;
      EvoCallbackdata.MR_Data.Code = ErrorMessageToErrorCode(errmsg);
      g_free(errmsg);
   }
   else
   {
      ETG_TRACE_USR4(("PhoneNumber is: %s ", phoneNumber));
   }
}

/******************************************************************
 Function Name : EvoAcquireTelephoneNumber

 Description   : Get the Telephone number

 Arguments     :
 In            : DevAddress - Device address
 Out           :
 *******************************************************************/
static gboolean EvoAcquireTelephoneNumber(const char* DevAddress)
{
   ETG_TRACE_USR1(("EvoAcquireTelephoneNumber entered"));
   ETG_TRACE_USR4(("EvoAcquireTelephoneNumber : %s ", DevAddress));

   int connIndex = getConnIndexFromDeviceAdd(DevAddress);

   GObject* handsfreeProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
   if (!handsfreeProxy)
   {
      ETG_TRACE_USR1(("handsfreeProxy is NULL proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   handsfree_call_request_phone_number((Handsfree*) handsfreeProxy, NULLPTR, _EvoRequestPhoneNumbercallback, NULLPTR);

   return FC_PHONE_DBUS_SUCCESS;
}

#ifdef PHONE_SIRI
/******************************************************************
 Function Name : EvoEnableSiri

 Description   : Enables the SIRI option

 Arguments     :
 In            : DevAddress - Device address.
                 Id -
 Out           :
 *******************************************************************/
static gboolean EvoEnableSiri(const char* DevAddress, VehicleInformation devInfo)
{
   ETG_TRACE_USR1(("EvoEnableSiri entered"));
   ETG_TRACE_USR1((" EvoEnableSiri DevAddress: %s",DevAddress));

   int connIndex = getConnIndexFromDeviceAdd(DevAddress);

   guint16 vendorId = devInfo.m_u16VendorID;
   guint16 productId = devInfo.m_u16ProductID;
   guint16 version = devInfo.m_u16ProductVersion;
   guint16 supportType = SUPPORTED_FEATURES;

   setSiriEnableRequest(connIndex, vendorId, productId, version, supportType);

   return  FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : _EvoSetSiriNrStatus_completion_callback

 Description   : Callback Function to Set SIRI status.

 Arguments     :
 In            : source_object - SIRI object
                 res -
                 user_data  - User data if any
 Out           :
 *******************************************************************/
static void _EvoSetSiriNrStatus_completion_callback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR1(("_EvoSetSiriNrStatus_completion_callback entered "));

   guchar u8SiriNRStatusActivationIndex = 0;
   if (user_data)
   {
      guchar* pSiriNRStatusActivationIndex = ((guchar*)user_data);
      u8SiriNRStatusActivationIndex = *pSiriNRStatusActivationIndex;
      delete pSiriNRStatusActivationIndex;
      pSiriNRStatusActivationIndex = NULLPTR;
   }
   ETG_TRACE_USR1(("u8SiriNRStatusActivationIndex: %u", u8SiriNRStatusActivationIndex));

   GError *error = NULLPTR;
   gchar *errmsg = NULLPTR;
   gboolean retval = evo_siri_call_set_nr_finish((EvoSiri*)source_object,res, &error);
   ETG_TRACE_USR4(("retval: %d", retval));

   char remoteDevicePath[PHONE_BDADRESS_LENGTH] = {'\0'};
   char* p_devAdd = Get_DeviceAddress_from_proxy_evo(source_object, FC_PHONE_DBUS_PROXY_EVO_SIRI_INTERFACE);
   if (p_devAdd)
   {
      getDevicePathFromRemoteDeviceAddress(p_devAdd, remoteDevicePath);
      vStringCopy(SiriNRStatus_data_evo.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
      ETG_TRACE_USR3(("SiriNRStatus_data_evo.BD_Address :%s", SiriNRStatus_data_evo.BD_Address));

      if(FALSE == retval)
      {
         if(NULLPTR != error)
         {
            errmsg = g_dbus_error_get_remote_error(error);
            ETG_TRACE_ERR(("ERROR ::Callback error message is :%s ", errmsg));
            g_free(errmsg);
         }

         SiriNRStatus_data_evo.m_u8SiriNrStatus = 0x04;

         ETG_TRACE_USR3((" Sending vSiriNrStatusReqRetry to CCA "));
         DCCA_callbacks_evo->vSiriNrStatusReqRetry(&SiriNRStatus_data_evo); //Retry if there is any error.
      }
      else
      {
         ETG_TRACE_USR4(("_EvoSetSiriNrStatus_completion_callback OK"));

         SiriNRStatus_data_evo.m_u8SiriNrStatus = u8SiriNRStatusActivationIndex;

         ETG_TRACE_USR3((" Sending vSiriNrStatus to CCA "));
         DCCA_callbacks_evo->vSiriNrStatus(&SiriNRStatus_data_evo);
      }
   }

}

/******************************************************************
 Function Name : setPropertySiriNrStatus

 Description   : Sets SIRI NR status

 Arguments     :
 In            : DevAddress - Device address.
                 activationIndex - activation Index
 Out           :
 *******************************************************************/
gboolean setPropertySiriNrStatus(int connIndex, const guchar u8SiriNRStatusActivationIndex)
{
   ETG_TRACE_USR4(("setPropertySiriNrStatus entered with u8SiriNRStatusActivationIndex: %d", u8SiriNRStatusActivationIndex));

   GObject* evoSiriProxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_SIRI_INTERFACE);
   if (!evoSiriProxy)
   {
      ETG_TRACE_ERR(("ERROR ::RemoteProxy is NULL: proxy corresponding to DevAdderss , not found in SLOT"));
      return FC_PHONE_DBUS_FAIL;
   }

   guchar* pSiriNRStatusActivationIndex = new(std::nothrow) guchar;
   if (pSiriNRStatusActivationIndex)
   {
      *pSiriNRStatusActivationIndex = u8SiriNRStatusActivationIndex;
      ETG_TRACE_USR4(("CALLED : evo_siri_call_set_nr"));
      evo_siri_call_set_nr((EvoSiri*) evoSiriProxy, u8SiriNRStatusActivationIndex, NULLPTR, _EvoSetSiriNrStatus_completion_callback, (gpointer)pSiriNRStatusActivationIndex);
   }
   return FC_PHONE_DBUS_SUCCESS;
}

/******************************************************************
 Function Name : EvoSetSiriNrStatus

 Description   : Calls setPropertySiriNrStatus method

 Arguments     :
 In            : DevAddress - Device address.
                 activationIndex - activation index
 Out           :
 *******************************************************************/
static gboolean EvoSetSiriNrStatus(const char* DevAddress, const guchar activationIndex)
{
   ETG_TRACE_USR1(("EvoSetSiriNrStatus entered"));
   ETG_TRACE_USR4(("EvoSetSiriNrStatus.. activationIndex %d", activationIndex));

   int connIndex = getConnIndexFromDeviceAdd(DevAddress);

   return setPropertySiriNrStatus(connIndex, activationIndex);
}

/******************************************************************
 Function Name : EvoSetSiriEfm

 Description   : Sets SIRI Efm

 Arguments     :
 In            : DevAddress - Device address.
                 activationIndex - activation index
 Out           :
 *******************************************************************/
static gboolean EvoSetSiriEfm(const char* DevAddress, const guchar activationIndex)
{
   ETG_TRACE_USR1(("EvoSetSiriEfm entered"));
   ETG_TRACE_USR4(("EvoSetSiriEfm.. activationIndex %d", activationIndex));

   int connIndex  = getConnIndexFromDeviceAdd(DevAddress);

   return setPropertyEyesFreeMode(connIndex,activationIndex);
}
#endif //PHONE_SIRI

/******************************************************************
 Function Name : vFillCallStatus_evo

 Description   : Fills Callstatus into call back data

 Arguments     :
 In            : Object - Incoming object
                 Status - Status code
                 Code - Error code
                 vproxy_id - Proxy ID
 Out           :
 *******************************************************************/
void vFillCallStatus_evo(GObject* source_object, guchar Status, guchar Code)
{
   ETG_TRACE_USR1(("vFillCallStatus_evo entered with Status: %d,Code: %d", Status, Code));

   char p_devAdd[PHONE_TEMP_DEV_ADDRESS] = {'\0'};
   char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};

   int connected_slot = _getconnected_slot(source_object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connected_slot)
   {
      ETG_TRACE_FATAL(("FATAL ERROR ::No connection slot is free all used"));
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);
   vStringCopy(p_devAdd, ConnMapTableST[connected_slot].DevAddr, PHONE_CONNMAP_DEV_ADDRESS);
   g_mutex_unlock(&ConnMapTableMutex);

   ETG_TRACE_USR3(("Connected Device Address :%s", p_devAdd));
   //Converting Device address from 'AABBCCDDEE' to 'dev_AA_BB_CC_DD_EE'
   getDevicePathFromRemoteDeviceAddress(p_devAdd, remoteDevicePath);

   vStringCopy(EvoCallbackdata.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
   vStringCopy(EvoCallbackdataArray[0].BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
   vStringCopy(EvoCallbackdataArray[1].BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);

   ETG_TRACE_USR4((" EvoCallbackdata.BD_Address :%s", EvoCallbackdata.BD_Address));
   ETG_TRACE_USR4((" EvoCallbackdataArray[0].BD_Address :%s", EvoCallbackdataArray[0].BD_Address));
   ETG_TRACE_USR4((" EvoCallbackdataArray[1].BD_Address :%s", EvoCallbackdataArray[1].BD_Address));

   EvoCallbackdata.MR_Data.Status = (tChar)Status;
   EvoCallbackdata.MR_Data.Code = (tChar)Code;
}

#ifdef PHONE_SIRI
/******************************************************************
 Function Name : vFill_Siri_evo

 Description   : Fills Add of remote device into siri call back data

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void vFill_Siri_evo(GObject* source_object)
{
   ETG_TRACE_USR1(("vFill_Siri_evo entered"));
   char p_devAdd[PHONE_TEMP_DEV_ADDRESS] = {'\0'};
   char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};

   int connected_slot = _getconnected_slot(source_object);
   if (FC_PHONE_NUMBER_OF_DEVICE_CONNECTION_SUPPORT == connected_slot)
   {
      ETG_TRACE_FATAL(("FATAL ERROR ::No connection slot is free all used"));
      return;
   }

   g_mutex_lock(&ConnMapTableMutex);

   vStringCopy(p_devAdd, ConnMapTableST[connected_slot].DevAddr,PHONE_CONNMAP_DEV_ADDRESS);

   //_printSlotStatusEvo();
   g_mutex_unlock(&ConnMapTableMutex);

   ETG_TRACE_USR3(("Connected Device Address :%s", p_devAdd));
   //Converting Device address from 'AABBCCDDEE' to 'dev_AA_BB_CC_DD_EE'
   getDevicePathFromRemoteDeviceAddress(p_devAdd, remoteDevicePath);
   vStringCopy(SiriStatus_value_evo.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
   ETG_TRACE_USR4((" vFill_Siri_evo SiriStatus_value_evo.BD_Address :%s", EvoCallbackdata.BD_Address));
}
#endif

/******************************************************************
 Function Name : ErrorMessageToErrorCode

 Description   : Converts Error message into HFP Error code

 Arguments     :
 In            : errMsg - Error message from stack
 Out           :
 *******************************************************************/
tChar ErrorMessageToErrorCode(char* errMsg)
{
   ETG_TRACE_USR1(("ErrorMessageToErrorCode entered"));
   tChar code = static_cast<tChar>(0xFF);
   if(!errMsg)
   {
      ETG_TRACE_ERR(("ERROR ::Error Message is NULL .."));
      return FC_PHONE_DBUS_FAIL;
   }
   else
   {
      ETG_TRACE_USR1(("Incoming Error message :%s", errMsg));

      if (strstr(errMsg, "org.ofono.Error.HFP"))
      {
         ETG_TRACE_USR1(("Error Code :%s", errMsg));

         if (strcmp(errMsg, "org.ofono.Error.HFP.AG failure") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_AG_FAILURE;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.No connection to phone") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_NO_CONNECTION_TO_PHONE;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Operation not allowed") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_OPERATION_NOT_ALLOWED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Operation not supported") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_OPERATION_NOT_SUPPORTED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.PH-SIM PIN required") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_PH_SIM_PIN_REQUIRED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.SIM not inserted") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_SIM_NOT_INSERTED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.SIM PIN required") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_SIM_PIN_REQUIRED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.SIM PUK required") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_SIM_PUK_REQUIRED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.SIM failure") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_SIM_FAILURE;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.SIM busy") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_SIM_BUSY;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Incorrect password") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_INCORRECT_PASSWORD;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.SIM PIN2 required") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_SIM_PIN2_REQUIRED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.SIM PUK2 required") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_SIM_PUK2_REQUIRED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Memory full") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_MEMORY_FULL;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Invalid index") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_INVALID_INDEX;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Memory failure") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_MEMORY_FAILURE;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Text string too long") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_TEXTSTRING_TOOLONG;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Invalid characters in text string") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_INVALIDCHAR_TEXTSTRING;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Dial string too long") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_DIAL_STRING_TOOLONG;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Invalid characters in dial string") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_INVALIDCHAR_DIALSTRING;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.No network service") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_NO_NETWORK_SERVICE;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Network timeout") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_NETWORK_TIMEOUT;
         }
         else if (strcmp(errMsg, "org.ofono.Error.HFP.Network not allowed - Emergency calls only") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_EMERGENCY_CALLS_ONLY;
         }

         ETG_TRACE_USR4(("Error code :%d", code));
         return code;
      }
      else if (strstr(errMsg, "org.ofono.Error"))
      {
         ETG_TRACE_USR4(("Not a HFP Error code Other Error codes"));

         ETG_TRACE_USR1(("Error Code :%s", errMsg));
         if (strcmp(errMsg, "org.ofono.Error.InProgress") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_INPROGRESS;
         }
         else if (strcmp(errMsg, "org.ofono.Error.InvalidArguments") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_INVALID_ARGUMENTS;
         }
         else if (strcmp(errMsg, "org.ofono.Error.InvalidFormat") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_INVALID_FORMAT;
         }
         else if (strcmp(errMsg, "org.ofono.Error.NotImplemented") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_NOT_IMPLEMENTED;
         }
         else if (strcmp(errMsg, "org.ofono.Error.Failed") == 0)
         {
            code = FC_PHONE_DBUS_ERRORCODE_FAILED;
         }

         return code;
      }
      else
      {
         ETG_TRACE_USR4(("No Error Information"));
         return code;
      }
   }
}

/******************************************************************
 Function Name : UpdateScoConnectionStatus

 Description   : updating SCO connection status to CCA

 Arguments     :
 In            :
 Out           :
 *******************************************************************/
void UpdateScoConnectionStatus(tU8 scotype, GObject* object)
{
   ETG_TRACE_USR1(("UpdateScoConnectionStatus entered"));

   if (SCOConnectionStatus_data_evo.SCOConnectionStatus == FC_PHONE_DBUS_SCO_CONN_ESTABLISHED)
   {
      if (SCOConnectionStatus_data_evo.Codec == scotype)
      {
         ETG_TRACE_USR1(("SCOType does not changed from Previous value"));
         //SCO Connect request's SCOType and Property Update's  SCOType are same
         return;
      }
   }

   char remoteDevicePath[PHONE_TEMP_DEV_ADDRESS] = {'\0'};
   char* p_devAdd = Get_DeviceAddress_from_proxy_evo(object, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
   if (p_devAdd)
   {
      getDevicePathFromRemoteDeviceAddress(p_devAdd, remoteDevicePath);
      vStringCopy(SCOConnectionStatus_data_evo.BD_Address, remoteDevicePath, PHONE_BDADRESS_LENGTH);
      ETG_TRACE_USR3(("SCOType SCOConnectionStatusData.BD_Address :%s", SCOConnectionStatus_data_evo.BD_Address));
   }

   switch (scotype)
   {
      case FC_PHONE_EVO_DBUS_NOT_CONNECTED:
      {
         ETG_TRACE_USR3((" Setting SCO NOT Connected "));
         SCOConnectionStatus_data_evo.SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_DISCONNECTED;
         ETG_TRACE_USR3((" Sending vSCOConnectionStatus to CCA "));
         DCCA_callbacks_evo->vSCOConnectionStatus(&SCOConnectionStatus_data_evo);
      }
      break;
      case FC_PHONE_EVO_DBUS_NBS_AND_S4: /* Intentional fallthrough */
      case FC_PHONE_EVO_DBUS_NBS_S1_TO_S3:
      {
         SCOConnectionStatus_data_evo.SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_ESTABLISHED;
         ETG_TRACE_USR3((" Setting Narrow Band Speech "));
         SCOConnectionStatus_data_evo.Codec = NARROW_BAND_SPEECH;
         ETG_TRACE_USR3((" Sending vSCOConnectionStatus to CCA "));
         DCCA_callbacks_evo->vSCOConnectionStatus(&SCOConnectionStatus_data_evo);

      }
      break;
      case FC_PHONE_EVO_DBUS_WBS:
      {
         SCOConnectionStatus_data_evo.SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_ESTABLISHED;
         SCOConnectionStatus_data_evo.Codec = WIDE_BAND_SPEECH;
         ETG_TRACE_USR3((" Sending vWideBandAudioSetup to CCA "));
         DCCA_callbacks_evo->vWideBandAudioSetup(&SCOConnectionStatus_data_evo);
      }
      break;
      default:
      break;
   }
}

/*****************************************************************
 Function Name : GetPhoneStatus

 Description   : Gets the Phone status using Connection index.

 Arguments     :
 In            : connIndex - Connection index of a device
 Out           :
 ******************************************************************/
void GetPhoneStatus(int connIndex,T_PhoneStatus *PhoneStatus_data_temp )
{
   ETG_TRACE_USR1(("GetPhoneStatus entered"));

   memset(PhoneStatus_data_temp->BD_Address, '\0', PHONE_CONNMAP_DEV_ADDRESS);
   memset(PhoneStatus_data_temp->achPhoneOperatorName, '\0', PHONE_NWOPERATORNAME_LENGTH);

   g_mutex_lock(&ConnMapTableMutex);

   vStringCopy(PhoneStatus_data_temp->BD_Address,ConnMapTableST[connIndex].PhoneStatus_data.BD_Address,PHONE_CONNMAP_DEV_ADDRESS);
   vStringCopy(PhoneStatus_data_temp->achPhoneOperatorName,ConnMapTableST[connIndex].PhoneStatus_data.achPhoneOperatorName, PHONE_NWOPERATORNAME_LENGTH);
   PhoneStatus_data_temp->u32PhoneSuppFeature = ConnMapTableST[connIndex].PhoneStatus_data.u32PhoneSuppFeature;
   PhoneStatus_data_temp->u8AgThreeWayCallSuppFeatures = ConnMapTableST[connIndex].PhoneStatus_data.u8AgThreeWayCallSuppFeatures;
   PhoneStatus_data_temp->u8PhoneBattery = ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneBattery;
   PhoneStatus_data_temp->u8PhoneRoaming = ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneRoaming;
   PhoneStatus_data_temp->u8PhoneService = ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneService;
   PhoneStatus_data_temp->u8PhoneSignal = ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneSignal;

   g_mutex_unlock(&ConnMapTableMutex);
}

/*****************************************************************
 Function Name : StorePhoneStatus

 Description   : Stores the Phone status back into the connectionMap table using Connection Index

 Arguments     :
 In            : connIndex - Connection index of a device
 Out           :
 ******************************************************************/
void StorePhoneStatus(int connIndex, T_PhoneStatus *PhoneStatus_data_temp)
{
   ETG_TRACE_USR1(("StorePhoneStatus entered"));

   g_mutex_lock(&ConnMapTableMutex);

   vStringCopy(ConnMapTableST[connIndex].PhoneStatus_data.achPhoneOperatorName,PhoneStatus_data_temp->achPhoneOperatorName, PHONE_NWOPERATORNAME_LENGTH);
   ConnMapTableST[connIndex].PhoneStatus_data.u32PhoneSuppFeature = PhoneStatus_data_temp->u32PhoneSuppFeature;
   ConnMapTableST[connIndex].PhoneStatus_data.u8AgThreeWayCallSuppFeatures = PhoneStatus_data_temp->u8AgThreeWayCallSuppFeatures;
   ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneBattery = PhoneStatus_data_temp->u8PhoneBattery;
   ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneRoaming = PhoneStatus_data_temp->u8PhoneRoaming;
   ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneService = PhoneStatus_data_temp->u8PhoneService;
   ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneSignal = PhoneStatus_data_temp->u8PhoneSignal;

   g_mutex_unlock(&ConnMapTableMutex);
}

/*****************************************************************
 Function Name : PrintPhoneStatus

 Description   : Stores the Phone status back into the connectionMap table using Connection Index

 Arguments     :
 In            : connIndex - Connection index of a device
 Out           :
 ******************************************************************/
void PrintPhoneStatus(int connIndex)
{
   ETG_TRACE_USR1(("PrintPhoneStatus entered"));

   g_mutex_lock(&ConnMapTableMutex);

   ETG_TRACE_USR4(("PhoneStatus_data.BD_Address :%s", ConnMapTableST[connIndex].PhoneStatus_data.BD_Address));
   ETG_TRACE_USR4(("PhoneStatus_data.achPhoneOperatorName :%s", ConnMapTableST[connIndex].PhoneStatus_data.achPhoneOperatorName));
   ETG_TRACE_USR4(("PhoneStatus_data.u8PhoneBattery :%d", ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneBattery));
   ETG_TRACE_USR4(("PhoneStatus_data.u8PhoneRoaming :%d", ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneRoaming));
   ETG_TRACE_USR4(("PhoneStatus_data.u8PhoneService :%d", ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneService));
   ETG_TRACE_USR4(("PhoneStatus_data.u8PhoneSignal :%d", ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneSignal));
   ETG_TRACE_USR4(("PhoneStatus_data.u8AgThreeWayCallSuppFeatures :%d", ConnMapTableST[connIndex].PhoneStatus_data.u8AgThreeWayCallSuppFeatures));
   ETG_TRACE_USR4(("PhoneStatus_data.u32PhoneSuppFeature :%d", ConnMapTableST[connIndex].PhoneStatus_data.u32PhoneSuppFeature));

   g_mutex_unlock(&ConnMapTableMutex);
}

/*****************************************************************
 Function Name : ResetPhoneStatus

 Description   : Resets Phone status values.

 Arguments     :
 In            : connIndex - Connection index of a device
 Out           :
 ******************************************************************/
void ResetPhoneStatus(int connIndex)
{
   ETG_TRACE_USR1(("ResetPhoneStatus entered"));

   ConnMapTableST[connIndex].PhoneStatus_data.BD_Address[0] = '\0';
   ConnMapTableST[connIndex].PhoneStatus_data.achPhoneOperatorName[0] = '\0';
   ConnMapTableST[connIndex].PhoneStatus_data.u32PhoneSuppFeature = 0;
   ConnMapTableST[connIndex].PhoneStatus_data.u8AgThreeWayCallSuppFeatures = 0;
   ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneBattery = 0xFF;
   ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneRoaming = 0;
   ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneService = 0xFF;
   ConnMapTableST[connIndex].PhoneStatus_data.u8PhoneSignal = 0xFF;
}

/*****************************************************************
 Function Name : ResetCallMapTable

 Description   : Resets CallMapTable for given Connection Index

 Arguments     :
 In            : connIndex - Connection index of a device
 Out           :
 ******************************************************************/
void ResetCallMapTable(int connIndex)
{
   ETG_TRACE_USR1(("ResetCallMapTable entered"));

   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      ConnMapTableST[connIndex].CallMapTable[index].achCallPath[0] = '\0';
      ConnMapTableST[connIndex].CallMapTable[index].callIndex = 0;
      ConnMapTableST[connIndex].CallMapTable[index].isCallPresent = FALSE;
      ConnMapTableST[connIndex].CallMapTable[index].isPropertyChanged = FALSE;
      ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy = NULLPTR;
      ConnMapTableST[connIndex].CallMapTable[index].callstatus.achCallNumber[0] = '\0';
      ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallDirection = CALL_DIR_UNKNOWN;
      ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallHandle = 0;
      ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallMode = CALL_MODE_UNKNOWN;
      ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallState = CALL_STATE_IDLE;
      ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallTotal = 0;
      ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallType = CALL_TYPE_UNKNOWN;

   }
}

/*****************************************************************
 Function Name : CheckVoiceCallProxyDuplication

 Description   : Checks the voicecall proxy creation Duplication

 Arguments     :
 In            : errMsg - Error message from stack
 Out           :
 ******************************************************************/
tBool CheckVoiceCallProxyDuplication(int callIndex, int connIndex)
{
   ETG_TRACE_USR1(("CheckVoiceCallProxyDuplication entered"));
   tBool bRetVal = FALSE;

   g_mutex_lock(&ConnMapTableMutex);

   //Assigns newly created proxy into a CallMap Table instance
   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      ETG_TRACE_USR1(("For loop index:%d, connIndex :%d, callIndex :%d", index, connIndex, callIndex));

      if (ConnMapTableST[connIndex].CallMapTable[index].callIndex == callIndex)
      {
         if (ConnMapTableST[connIndex].CallMapTable[index].voiceCallProxy)
         {
            ETG_TRACE_ERR(("Already VoiceCall Proxy is available."));
            bRetVal = TRUE;
            break;
         }
         else
         {
            ETG_TRACE_USR1(("VoiceCall Proxy slot is empty - New Proxy to be created"));
         }
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   return bRetVal;
}

/*****************************************************************
 Function Name : getCallTotalForConnectedDevice

 Description   :

 Arguments     :
 In            :
 Out           :
 ******************************************************************/
int getCallTotalForConnectedDevice(int connIndex)
{
   ETG_TRACE_USR4(("getCallTotalForConnectedDevice entered "));
   int callTotal = 0;

   g_mutex_lock(&ConnMapTableMutex);

   callTotal = ConnMapTableST[connIndex].gcallTotal;

   g_mutex_unlock(&ConnMapTableMutex);

   return callTotal;
}

/*****************************************************************
 Function Name : getFreeSlotFromCallMapTable

 Description   :

 Arguments     :
 In            :
 Out           :
 ******************************************************************/
int getFreeSlotFromCallMapTable(int connIndex)
{
   ETG_TRACE_USR4(("getFreeSlotFromCallMapTable entered "));
   int free_slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;

   g_mutex_lock(&ConnMapTableMutex);

   for(int index =0;index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT;index++)
   {
      if(ConnMapTableST[connIndex].CallMapTable[index].isCallPresent != TRUE)
      {
         ETG_TRACE_USR4(("getFreeSlotFromCallMapTable got free_slot at:%d ",index));
         free_slot = index;
         break;
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   if(free_slot == FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT)
   {
      ETG_TRACE_ERR(("ERROR :: getFreeSlotFromCallMapTable No free_slot available"));
   }

   return free_slot;
}

/*****************************************************************
 Function Name : getCallSlotFromCallMapTable

 Description   :

 Arguments     :
 In            :
 Out           :
 ******************************************************************/
int getCallSlotFromCallMapTable(int connIndex, int callIndex)
{
   ETG_TRACE_USR4(("getCallSlotFromCallMapTable entered "));
   int call_slot = FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT;

   g_mutex_lock(&ConnMapTableMutex);

   for(int index =0;index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT;index++)
   {
      if((ConnMapTableST[connIndex].CallMapTable[index].isCallPresent == TRUE)
            && (ConnMapTableST[connIndex].CallMapTable[index].callIndex == callIndex))
      {
         ETG_TRACE_USR4(("getCallSlotFromCallMapTable got call_slot at:%d ",index));
         call_slot = index;
         break;
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);

   if(call_slot == FC_PHONE_DEVICE_CONNECTION_INVALID_SLOT)
   {
      ETG_TRACE_ERR(("ERROR :: getCallSlotFromCallMapTable No call_slot available"));
   }

   return call_slot;
}

/*****************************************************************
 Function Name : removeVoiceCallProxy

 Description   :

 Arguments     :
 In            :
 Out           :
 ******************************************************************/
gboolean removeVoiceCallProxy(int connIndex, int removedindex)
{
   ETG_TRACE_USR1(("removeVoiceCallProxy entered"));
   gboolean retVal = FALSE;
   g_mutex_lock(&ConnMapTableMutex);

   ETG_TRACE_USR1(("Removing call instance :%s", ConnMapTableST[connIndex].CallMapTable[removedindex].achCallPath));

   //Unref the Voice Call proxy
   if (ConnMapTableST[connIndex].CallMapTable[removedindex].voiceCallProxy)
   {
      if (G_IS_OBJECT(ConnMapTableST[connIndex].CallMapTable[removedindex].voiceCallProxy))
      {
         ETG_TRACE_USR4(("g_object_unref for valid GObject Proxy"));
         g_object_unref(ConnMapTableST[connIndex].CallMapTable[removedindex].voiceCallProxy);
      }
      else
      {
         ETG_TRACE_USR4(("GObject Proxy is NOT valid"));
      }

      ConnMapTableST[connIndex].CallMapTable[removedindex].voiceCallProxy = NULLPTR;
      ConnMapTableST[connIndex].CallMapTable[removedindex].isCallPresent = FALSE;
      ConnMapTableST[connIndex].CallMapTable[removedindex].isPropertyChanged = FALSE;
      ConnMapTableST[connIndex].CallMapTable[removedindex].achCallPath[0] = '\0';
      ConnMapTableST[connIndex].CallMapTable[removedindex].callIndex = 0;
      ConnMapTableST[connIndex].CallMapTable[removedindex].callstatus.achCallNumber[0] = '\0';
      ConnMapTableST[connIndex].CallMapTable[removedindex].callstatus.u8CallDirection = CALL_DIR_UNKNOWN;
      //CallHandle is initialized to Zero. Otherwise in calladded signal we may omit the
      //new call status as duplicate
      ConnMapTableST[connIndex].CallMapTable[removedindex].callstatus.u8CallHandle = 0;
      ConnMapTableST[connIndex].CallMapTable[removedindex].callstatus.u8CallMode = CALL_MODE_UNKNOWN;
      ConnMapTableST[connIndex].CallMapTable[removedindex].callstatus.u8CallState = CALL_STATE_IDLE;
      ConnMapTableST[connIndex].CallMapTable[removedindex].callstatus.u8CallTotal = 0;
      ConnMapTableST[connIndex].CallMapTable[removedindex].callstatus.u8CallType = CALL_TYPE_UNKNOWN;
      retVal = TRUE;
   }

   g_mutex_unlock(&ConnMapTableMutex);

   return retVal;
}

/******************************************************************
 Function Name : registerSiganlHandler

 Description   : registering for handsfree property changed signal

 Arguments     :
 In            : remoteDeviceAddress - Device address
 Out           :
 *******************************************************************/
void registerSiganlHandler(int connIndex)
{
   ETG_TRACE_USR4(("registerSiganlHandler entered"));

   GObject* Proxy = NULLPTR;
   Proxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_HANSFREE_INTERFACE);
   if (!Proxy)
   {
      ETG_TRACE_ERR(("ERROR ::Proxy is null"));
   }
   else
   {
      g_signal_connect(Proxy, "property-changed", G_CALLBACK(handsfreePropertyChangedSignalHandler), NULLPTR);
   }

   Proxy = NULLPTR;
   Proxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_SIRI_INTERFACE);
   if (!Proxy)
   {
      ETG_TRACE_ERR(("ERROR ::Proxy is null"));
   }
   else
   {
      g_signal_connect(Proxy, "property-changed", G_CALLBACK(siriPropertyChangedSignalHandler), NULLPTR);
   }


   Proxy = NULLPTR;
   Proxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_NETWORK_REGISTRATION_INTERFACE);
   if (!Proxy)
   {
      ETG_TRACE_ERR(("ERROR ::Proxy is null"));
   }
   else
   {
      g_signal_connect(Proxy, "property-changed", G_CALLBACK(networkRegPropertyChangedSignalHandler), NULLPTR);
   }

   Proxy = NULLPTR;
   Proxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_VOICECALL_MANAGER_INTERFACE);
   if (!Proxy)
   {
      ETG_TRACE_ERR(("ERROR ::Proxy is null"));
   }
   else
   {
      g_signal_connect(Proxy, "call-added", G_CALLBACK(callAddedSignalHandler), NULLPTR);
      g_signal_connect(Proxy, "call-removed", G_CALLBACK(callRemovedSignalHandler), NULLPTR);
      g_signal_connect(Proxy, "property-changed", G_CALLBACK(voiceCallManagerPropertyChangedSignalHandler), NULLPTR);
   }

   Proxy = NULLPTR;
   Proxy = Get_proxy_from_ConnIndex_Evo(connIndex, FC_PHONE_DBUS_PROXY_EVO_CALL_VOLUME_INTERFACE);
   if (!Proxy)
   {
      ETG_TRACE_ERR(("ERROR ::Proxy is null"));
   }
   else
   {
      g_signal_connect(Proxy, "property-changed", G_CALLBACK(callVolumePropertyChangedSignalHandler), NULLPTR);
   }
}


/******************************************************************
 Function Name : updateAndInsertNewCall

 Description   : registering for handsfree property changed signal

 Arguments     :
 In            : source_object - source object for voicecall
               : CallPath - Voicecall Object path
 Out           :
 *******************************************************************/
void updateAndInsertNewCall(int connIndex, int callIndex,int free_slot,const gchar* CallPath)
{
   ETG_TRACE_USR4(("updateAndInsertNewCall entered"));

   ETG_TRACE_USR4(("connIndex :%d", connIndex));
   ETG_TRACE_USR4(("callIndex :%d", callIndex));

   g_mutex_lock(&ConnMapTableMutex);
   ConnMapTableST[connIndex].CallMapTable[free_slot].callIndex = callIndex;
   vStringCopy(ConnMapTableST[connIndex].CallMapTable[free_slot].achCallPath, CallPath, FC_PHONE_CALL_PATH_LENGTH);
   ConnMapTableST[connIndex].CallMapTable[free_slot].isCallPresent = TRUE;
   ConnMapTableST[connIndex].gcallTotal = static_cast<tU8>(ConnMapTableST[connIndex].gcallTotal+ 1);
   ConnMapTableST[connIndex].CallMapTable[free_slot].callstatus.u8CallHandle = ConnMapTableST[connIndex].gcallTotal;
   ConnMapTableST[connIndex].CallMapTable[free_slot].callstatus.u8CallTotal = ConnMapTableST[connIndex].gcallTotal;
   g_mutex_unlock(&ConnMapTableMutex);

   //Creating VoiceCall Proxy for each call added.
   //This should be called after updating callIndex to the CallMapTable
   CreateVoiceCallProxy(CallPath);

   int gcallTotal = getCallTotalForConnectedDevice(connIndex);
   ETG_TRACE_USR4(("gcallTotal :%d", gcallTotal));

   g_mutex_lock(&ConnMapTableMutex);
   //Updating call count for other call instances.
   for (int index = 0; index < FC_PHONE_NUMBER_OF_CALLS_SUPPORT; index++)
   {
      if (ConnMapTableST[connIndex].CallMapTable[index].isCallPresent == TRUE)
      {
         ConnMapTableST[connIndex].CallMapTable[index].callstatus.u8CallTotal = (tU8) gcallTotal;
         ETG_TRACE_USR1(("Assigning CallTotal for All call instance :%d", index));
      }
   }

   g_mutex_unlock(&ConnMapTableMutex);
}

/******************************************************************
 Function Name : VoiceCallTerminateOrHangup

 Description   : Terminate or Hangup the desired call.

 Arguments     :
 In            : connIndex - Connection Index
               : enHangupType - Type of Terminate or Hangup
               : user_data - User data
 Out           :
 *******************************************************************/
gboolean VoiceCallTerminateOrHangup(int connIndex, FCPhoneDbusClient_TerminateType_t enHangupType, const char* user_data)
{
   ETG_TRACE_USR4(("VoiceCallTerminateOrHangup entered"));
   int retVal = FC_PHONE_DBUS_FAIL;

   ETG_TRACE_USR4(("VoiceCallTerminateOrHangup :connIndex :%d", connIndex));
   ETG_TRACE_USR4(("VoiceCallTerminateOrHangup :enHangupType :%d", enHangupType));
   ETG_TRACE_USR4(("VoiceCallTerminateOrHangup :user_data :%s", user_data));

   switch (enHangupType)
   {
      case FC_PHONE_DBUS_TERMINATETYPE_SINGLECALL:
      {
         ETG_TRACE_USR4(("FC_PHONE_DBUS_TERMINATETYPE_SINGLECALL "));
         retVal = voiceCallTerminateRequest(connIndex, user_data);
      }
      break;
      case FC_PHONE_DBUS_TERMINATETYPE_MULTIPLECALL_ACTIVE:
      {
         ETG_TRACE_USR4(("FC_PHONE_DBUS_TERMINATETYPE_MULTIPLECALL_ACTIVE "));
         retVal = voiceCallManagerReleaseAndSwap(connIndex, user_data);
      }
      break;
      case FC_PHONE_DBUS_TERMINATETYPE_MULTIPLECALL_INACTIVE:
      {
         ETG_TRACE_USR4(("FC_PHONE_DBUS_TERMINATETYPE_MULTIPLECALL_INACTIVE "));
         retVal = voiceCallTerminateMultipleInactive(connIndex, user_data);
      }
      break;
      case FC_PHONE_DBUS_TERMINATETYPE_SECONDCALL_WAITING:
      {
         ETG_TRACE_USR4(("FC_PHONE_DBUS_TERMINATETYPE_SECONDCALL_WAITING "));
         retVal = releaseSecondWaitingCallRequest(connIndex, user_data);
      }
      break;
      case FC_PHONE_DBUS_TERMINATETYPE_ALL:
      {
         ETG_TRACE_USR4(("FC_PHONE_DBUS_TERMINATETYPE_ALL "));
         retVal = voiceCallManagerHangupAll(connIndex, user_data);
      }
      break;
      case FC_PHONE_DBUS_TERMINATETYPE_ACTIVECALL_ACCEPT_WAITING:
      {
         ETG_TRACE_USR4(("FC_PHONE_DBUS_TERMINATETYPE_ACTIVECALL_ACCEPT_WAITING "));
         retVal = voiceCallManagerReleaseAndAnswer(connIndex);
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Error ::Wrong enHangupType :%d", enHangupType));
      }
      break;
   }

   return retVal;
}
