/*******************************************************************************
 *
 * FILE:          fc_phone_tclAudioChannelWatchdog.h
 *
 * SW-COMPONENT:  FC_Phone
 *
 * DESCRIPTION:   Audio channel watchdog.
 *
 * COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/
#include "FC_Phone_tclAudioChannelWatchdog.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_SERVICE_TELEPHONE
#include "trcGenProj/Header/FC_Phone_tclAudioChannelWatchdog.cpp.trc.h"
#endif


/*******************************************************************************
 *
 * Function:       fc_phone_tclAudioChannelWatchdog
 *
 * Description:    Constructor
 *
 *******************************************************************************/
fc_phone_tclAudioChannelWatchdog::fc_phone_tclAudioChannelWatchdog(fc_phone_tclService_Telephone& roParent) : m_roParent(roParent), m_oTimer(), m_u32ErrorCounter(0)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclAudioChannelWatchdog::fc_phone_tclAudioChannelWatchdog"));
   m_oTimer.bInit(this, &fc_phone_tclAudioChannelWatchdog::vCallbackLocal);
   m_oTimer.bStart( 3000, 3000);
}


/*******************************************************************************
 *
 * Function:       vCallback
 *
 * Description:    Callback function when the timer started in the constructor
 *                 expires. Checks for errors if more than 3 errors occur audio
 *                 channel will be released.
 *
 *******************************************************************************/
tVoid fc_phone_tclAudioChannelWatchdog::vCallbackLocal()
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclAudioChannelWatchdog::vCallback"));
   if (bMatchesErrorCondition())
   {
      m_u32ErrorCounter++;
   }
   else  
   {
      // Reset error counter
      m_u32ErrorCounter = 0;
   }

   if (m_u32ErrorCounter < 3)
   {
      // Action at third occurrence of error condition
      return;
   }

   // Error condition met several times: Now act and release audio channel
   ETG_TRACE_ERR(("fc_phone audio watchdog ==> ERROR condition met %d times! releasing LC_PHONE ...", m_u32ErrorCounter));
   et_vErrmemStringNormal(TR_CLASS_FC_PHONE_SERVICE_TELEPHONE, "fc_phone audio watchdog ==> ERROR condition met %d times! releasing LC_PHONE ...", m_u32ErrorCounter);

   if(fc_phone_tclSM::m_poAudioManager)
   {
      ETG_TRACE_USR4(("calling vReleasePhoneChannel "));
      (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vReleasePhoneChannel();
      m_u32ErrorCounter = 0;
   }

   //Fix for GMMY15-6691. Reset audio channel to unused when it is released on timer callback
   if((fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone)
   {
      (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->vResetAudioChannel();
   }
}

/*******************************************************************************
 *
 * Function:       bMatchesErrorCondition
 *
 * Description:    Checks to see if the callback matches any of the error
 *                 conditions.
 *
 * Return:         Boolean to indicate if an error was detected or not.
 *
 *******************************************************************************/
tBool fc_phone_tclAudioChannelWatchdog :: bMatchesErrorCondition()
{
   // helper function to determine, whether the conditions for this
   // watchdog to act are met.
   ETG_TRACE_USR4(("ENTER fc_phone_tclAudioChannelWatchdog :: bMatchesErrorCondition"));

   // no action if logical channel LC_PHONE is NOT assigned to us
   if((fc_phone_tclSM::m_poAudioManager) && (fc_phone_tclSM::m_poAudioManager->bIsPhoneChannelGet() == FALSE))
   {
      return FALSE;
   }

   // check device connection status
   tBool bSLCStatus = m_roParent.oGetSLCStatus().bCommandStatus;
   if (bSLCStatus == FALSE)
   {
      // we have LC_PHONE, but no device connected
      ETG_TRACE_ERR((" fc_phone audio watchdog . . . SLCStatus NOT connected"));
      return TRUE;
   }

   // determine number of non-idle calls
   tU32 u32NumberOfActiveCalls = 0;
   for (tU8 u8Index = 0x00; u8Index < FC_PHONE_MAX_CALLINSTANCES;u8Index++)
   {
      //GMMY16-10454 - if not dereference properly, and if first call ends, audio watch dog error will happen and audio will be transfered to handset.
      if ((most_fi_tcl_e8_TelCallStatus::tenType)m_roParent.oGetCallInstances()[u8Index].m_u16CallStatusNotice != most_fi_tcl_e8_TelCallStatus::FI_EN_E8IDLE)
      {
         // a non-idle call is found
         u32NumberOfActiveCalls++;  
      }
   }
   ETG_TRACE_USR3((" fc_phone audio watchdog . . . SLCStatus connected, %d non-idle calls", u32NumberOfActiveCalls));

   // error condition is met if we no longer have calls,
   // but still have LC_PHONE assigned to us
   if (u32NumberOfActiveCalls)
   {
      return FALSE;
   }
   else
   {
      return TRUE;
   }
}
